/*--------------------------------*-C-*---------------------------------*
 * This module is all original work by Robert Nation
 * (nation@rocket.sanders.lockheed.com)
 *
 * Copyright 1994, Robert Nation, no rights reserved.
 * The author guarantees absolutely nothing about anything, anywhere, anytime
 * Use this code at your own risk for anything at all.
 *
 * Modified by mj olesen <olesen@me.queensu.ca>
 * No additional restrictions.
 *
 * Modifications by Raul Garcia Garcia (rgg@tid.es)
 *----------------------------------------------------------------------*/
#include "rxvt.h"
#include "debug.h"

#ifndef DEBUG_MALLOC

/* malloc and check for NULL return */
void *
my_malloc (int size)
{
   void *s;
   if ((s = malloc (size)) == NULL) abort ();
   return s;
}

void
my_free (void *ptr)
{
   if (ptr != NULL) free (ptr);
}

#else	/* DEBUG_MALLOC */

/* Number of pad characters to use when checking for out-of-bounds */
# define DEBUG_PAD 4
# define MAX_MALLOCS 100
static void *all_ptrs[MAX_MALLOCS];
static char *all_names[MAX_MALLOCS];
static int alloc_init = 0;

/*
 * Malloc that checks for NULL return, and adds out-of-bounds checking if
 * DEBUG_MALLOC is set
 */
void *
safe_malloc (int size, char *identifier)
{
   void *s;
#ifdef DEBUG_MALLOC
   char *vptr;
   int *l,i;

   if (!alloc_init)
     {
	for (i = 0; i < MAX_MALLOCS; i++)
	  all_ptrs [i] = 0;
	alloc_init = 1;
     }

   if ((s = malloc (size + 2 * DEBUG_PAD * sizeof(int))) == NULL)
     abort ();
   vptr = s;
   for (i = 0; i < (DEBUG_PAD-1) * sizeof(int); i++)
     vptr [i] = 0x3E;
   for (i = size + (DEBUG_PAD * sizeof(int));
	i < (size + (2 * DEBUG_PAD) * sizeof(int)); i++)
     vptr [i] = 0x3F;
   l = (int *)s + DEBUG_PAD - 1;
   *l = size;

   fprintf (stderr, "Allocated %d bytes at 0x%08x, name %s\n",
	    size, s, identifier);

   i = 0; while ((i < MAX_MALLOCS) && (all_ptrs[i] != 0)) i++;
   if (i >= MAX_MALLOCS)
     {
	fprintf (stderr, "MAX_MALLOCS exceeded. Please increase\n");
     }
   else
     {
	all_ptrs [i] = s;
	all_names [i] = identifier;
	check_all_mem ("Malloc",identifier);
     }

   return ((void *)((int *)s+DEBUG_PAD));
#else
   if ((s = malloc (size)) == NULL)
     abort ();
   return s;
#endif
}

/*
 * Free command good for use with above malloc, checks for out-of-bounds
 * before freeing.
 */
void
safe_free (void *ptr, char *id1, char *id2)
{
#ifdef DEBUG_MALLOC
   int i;
   char *base;

   if (ptr == NULL) return;
   base = (char *)((int *)(ptr) - DEBUG_PAD);

   fprintf (stderr,"Freeing memory at 0x%08x",base);
   i = 0; while ((i<MAX_MALLOCS) && (all_ptrs[i] != (void *)(base))) i++;
   if (i < MAX_MALLOCS)
     {
	fprintf (stderr," name %s\n",all_names[i]);
	check_all_mem(id1,id2);
	all_ptrs[i] = 0;
     }

   free ((void *)(base));
#else
   if (ptr != NULL)
     free (ptr);
#endif
}

/*
 * Checks all allocated memory for out of bounds memory usage.
 */
void
check_all_mem (char *which1, char *which2)
{
#ifdef DEBUG_MALLOC
   int l, i, j, fail = 0;
   int *lptr;
   char *base;
   static char *prev_check1 = NULL, *prev_check2 = NULL;

   if (!alloc_init)
     return;

   for (i = 0; i < MAX_MALLOCS; i++)
     {
	if (all_ptrs[i] != 0)
	  {
	     /* Check each memory region */
	     base = all_ptrs[i];

	     for (j = 0; j < ((DEBUG_PAD-1)*sizeof(int)); j++)
	       {
		  if (base[j] != 0x3E)
		    {
		       fprintf (stderr, "Ouch! ptr = 0x%08x j = %d %s %s:%s\n",
				base, j, all_names[i], which1, which2);
		       if (prev_check1 != NULL)
			 fprintf (stderr, "Last successful check %s %s\n",
				  prev_check1, prev_check2);
		       fail = 1;
		    }
	       }

	     lptr = (int*) (&base[(DEBUG_PAD-1)*sizeof(int)]);
	     l = *lptr;
	     for (j = l+DEBUG_PAD*sizeof(int);
		  j < l+(2*DEBUG_PAD)*sizeof(int); j++)
	       {
		  if (base[j] != 0x3F)
		    {
		       fprintf (stderr,"Ouch2! ptr = 0x%08x k = %d name %s %s:%s\n",
				base, j, all_names[i], which1, which2);
		       if (prev_check1 != NULL)
			 fprintf (stderr,"Last successful check %s %s\n",
				  prev_check1, prev_check2);
		       fail = 1;
		    }
	       }

	  }
     }

   if (!fail)
     {
	prev_check1 = which1;
	prev_check2 = which2;
     }
#endif
}

#endif	/* DEBUG_MALLOC */
/*----------------------- end-of-file (C source) -----------------------*/
