/*
 *  File:        math_panel.C
 *  Purpose:     Mathed GUI for lyx
 *  Author:      Alejandro Aguilar Sierra <asierra@servidor.unam.mx>
 *  Created:     March 28, 1996
 * 
 *  Dependencies: Xlib, Xpm, XForms, Lyx
 *
 *  Copyright: (c) 1996, Alejandro Aguilar Sierra 
 *
 *   You are free to use and modify it under the terms of
 *   the GNU General Public Licence version 2 or later.
 */

#include "config.h"
#include <forms.h>
#include <stdio.h>
#include <stdlib.h>
#include "math_panel.h"
#include "symbol_def.h"
#include "formula.h"
#include "keybind.h"

/* Bitmaps */
#include "delim.xbm"
#include "delim0.xpm"
#include "delim.xpm"
#include "sqrt.xpm"
#include "frac.xpm"
#include "matrix.xpm"
#include "equation.xpm"

static FD_panel *fd_panel;
static FD_delim *fd_delim;
static FD_matrix *fd_matrix;

extern Inset* the_locking_inset;
extern void mathed_define_delimiter(int, int);

int delim_code[] = {   
   '(', ')', LM_lceil,  LM_rceil,  LM_uparrow,  LM_Uparrow,
   '[', ']', LM_lfloor,  LM_rfloor,  LM_updownarrow, LM_Updownarrow,
   '{', '}',  '/', LM_backslash,  LM_downarrow,  LM_Downarrow,
   LM_langle,  LM_rangle, '|', LM_Vert, 0
};

static char h_align_str[80] = "ccc";

/* callbacks for form panel */
void button_cb(FL_OBJECT *ob, long data)
{   
   extern void free_symbols_form();
   switch (data)  {
    case MM_GREEK:
    case MM_VARSIZE:
    case MM_BRELATS:
    case MM_ARROW:
    case MM_BOP:
    case MM_MISC: 
      {	   
	 BitmapMenu *menu = (BitmapMenu *)ob->u_vdata;
	 menu->Show();  
	 break;
      }
    case MM_FRAC:
      if (the_locking_inset)
	((InsetFormula*)the_locking_inset)->LocalDispatch(LFUN_INSERT_MATH, "frac");
      else
	DispatchFunction(LFUN_INSERT_MATH, "frac");
      break;
    case MM_SQRT:
      if (the_locking_inset)
	((InsetFormula*)the_locking_inset)->LocalDispatch(LFUN_INSERT_MATH, "sqrt");
      else
	DispatchFunction(LFUN_INSERT_MATH, "sqrt");
      break;
    case MM_DELIM:
      fl_show_form(fd_delim->delim,FL_PLACE_CENTER,FL_FULLBORDER,"Delimiter");
      break;
    case MM_MATRIX:
      fl_show_form(fd_matrix->matrix,FL_PLACE_CENTER,FL_FULLBORDER,"Matrix");
      break;
    case MM_EQU:
      DispatchFunction(LFUN_INSERT_EQUATION, NULL);
      break;
    case 100:
      free_symbols_form();
      break;
   }
}


/* callbacks for form delim */
void delim_cb(FL_OBJECT *, long data)
{
   static int left=0, right=1;
   int side=(fl_get_button(fd_delim->right)!=0);
   Pixmap p1, p2;
   
   switch (data) {
    case 0:
      {
	 mathed_define_delimiter(delim_code[left], delim_code[right]);
	 if (the_locking_inset)
	   ((InsetFormula*)the_locking_inset)->LocalDispatch(LFUN_INSERT_MATH, "delim");
	 else
	   DispatchFunction(LFUN_INSERT_MATH, "delim");
      }
    case 1: fl_hide_form(fd_delim->delim); break;
    case 2: 
      if (side) 
	right = fl_get_bmtable(fd_delim->menu);
      else
	left = fl_get_bmtable(fd_delim->menu);
      p1 = fl_get_pixmap_pixmap(fd_delim->pix, &p1, &p2);
      fl_draw_bmtable_item(fd_delim->menu,left,p1,0,0);
      fl_draw_bmtable_item(fd_delim->menu,right,p1,16,0);
      fl_redraw_object(fd_delim->pix);
      break;
    case 3: break;
    case 4: break;
   }
}

/* callbacks for form matrix */
void matrix_cb(FL_OBJECT *, long data)
{
   int nx, ny;
   static char v_align_c[] = "tcb";
 
   switch (data) {
    case 3:
    case 0: 
      {
	 char s[80];
	 char c = v_align_c[fl_get_choice(fd_matrix->valign)-1];
	 const char *sh = fl_get_input(fd_matrix->halign);
	 nx = (int)(fl_get_slider_value(fd_matrix->columns)+0.5);
	 ny = (int)(fl_get_slider_value(fd_matrix->rows)+0.5);
	 sprintf(s, "%d %d %c%s", nx, ny, c, sh);      
	 if (data!=3) fl_hide_form(fd_matrix->matrix);
	 if (the_locking_inset)
	   ((InsetFormula*)the_locking_inset)->LocalDispatch(LFUN_INSERT_MATRIX, s);
	 else
	   DispatchFunction(LFUN_INSERT_MATRIX, s);
	 break;
      }
    case 1: fl_hide_form(fd_matrix->matrix); break;
    case 2: 
      {
	 nx = (int)(fl_get_slider_value(fd_matrix->columns)+0.5);
	 for (int i=0; i<nx; i++) h_align_str[i] = 'c';
	 //memset(h_align_str, 'c', nx);
	 h_align_str[nx] = '\0';
//	 fl_freeze_form(fd_form_main->form_main);
//	fl_addto_form(fd_form_main->form_main);

	 fl_set_input(fd_matrix->halign, h_align_str);	
	 fl_redraw_object(fd_matrix->halign); 	 
	 break;
      }
   }
}

int align_filter(FL_OBJECT *, const char *, const char *cur, int c)
{
   int n = (int)(fl_get_slider_value(fd_matrix->columns)+0.5) - strlen(cur);
   return ((c=='c'||c=='l'||c=='r') && n>=0) ? FL_VALID: FL_INVALID;
}

char** mathed_get_pixmap_from_icon(int d)
{
   switch (d) {
    case MM_FRAC: return frac;
    case MM_SQRT: return sqrt;
    case MM_DELIM: return delim;
    case MM_MATRIX: return matrix;
    case MM_EQU: return equation; 
    default: return NULL;
   }
}

FD_panel *create_math_panel( )
{
   fd_panel = create_form_panel();
   fd_delim = create_form_delim();
   fd_matrix = create_form_matrix();

   /* fill-in form initialization code */
   fl_set_button(fd_delim->left, 1);
   fl_set_pixmap_data(fd_delim->pix, delim0);
   fl_set_bmtable_data(fd_delim->menu,6,4,delim_width,delim_height,delim_bits);
   fl_set_bmtable_maxitems(fd_delim->menu, 23);
   
   fl_set_pixmap_data(fd_panel->sqrt, sqrt);
   fl_set_pixmap_data(fd_panel->frac, frac);
   fl_set_pixmap_data(fd_panel->delim, delim);
   fl_set_pixmap_data(fd_panel->matrix, matrix);
   fl_set_pixmap_data(fd_panel->equation, equation);

   fl_addto_choice(fd_matrix->valign, "Top | Center | Bottom");
   fl_set_choice(fd_matrix->valign, 2);
   fl_set_input(fd_matrix->halign, h_align_str);
   fl_set_input_filter(fd_matrix->halign, align_filter);
   
   return fd_panel;
}

/* Form definition file generated with fdesign. */

FD_panel *create_form_panel(void)
{
  FL_OBJECT *obj;
  FD_panel *fdui = (FD_panel *) fl_calloc(1, sizeof(*fdui));
  int old_bw = fl_get_border_width();

  fl_set_border_width(-2);
  fdui->panel = fl_bgn_form(FL_NO_BOX, 322, 162);
  obj = fl_add_box(FL_UP_BOX,0,0,322,162,"");
  obj = fl_add_frame(FL_ENGRAVED_FRAME,6,100,310,54,"");
  fdui->greek = obj = fl_add_button(FL_NORMAL_BUTTON,10,114,50,30,"Greek");
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lstyle(obj,15);
    fl_set_object_callback(obj,button_cb,MM_GREEK);
  fdui->arrow = obj = fl_add_button(FL_NORMAL_BUTTON,161,114,50,30," ");
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lstyle(obj,15);
    fl_set_object_callback(obj,button_cb,MM_ARROW);
  fdui->boperator = obj = fl_add_button(FL_NORMAL_BUTTON,61,114,50,30," ");
    fl_set_object_lsize(obj,FL_MEDIUM_SIZE);
    fl_set_object_lstyle(obj,15);
    fl_set_object_callback(obj,button_cb,MM_BOP);
  fdui->brelats = obj = fl_add_button(FL_NORMAL_BUTTON,111,114,50,30," @");
    fl_set_object_lsize(obj,FL_MEDIUM_SIZE);
    fl_set_object_lstyle(obj,15);
    fl_set_object_callback(obj,button_cb,MM_BRELATS);
  fdui->varsize = obj = fl_add_button(FL_NORMAL_BUTTON,211,114,50,30,"S  ");
    fl_set_object_lsize(obj,FL_MEDIUM_SIZE);
    fl_set_object_lstyle(obj,15);
    fl_set_object_callback(obj,button_cb,MM_VARSIZE);
  fdui->misc = obj = fl_add_button(FL_NORMAL_BUTTON,261,114,50,30,"Misc");
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lstyle(obj,FL_TIMESITALIC_STYLE);
    fl_set_object_callback(obj,button_cb,MM_MISC);
  obj = fl_add_button(FL_RETURN_BUTTON,246,12,63,21,"Hide ");
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE);
    fl_set_object_callback(obj,button_cb,100);
  obj = fl_add_text(FL_NORMAL_TEXT,6,8,126,30,"Math Panel");
    fl_set_object_lcol(obj,FL_BOTTOM_BCOL);
    fl_set_object_lsize(obj,FL_HUGE_SIZE);
    fl_set_object_lalign(obj,FL_ALIGN_LEFT|FL_ALIGN_INSIDE);
    fl_set_object_lstyle(obj,FL_TIMESITALIC_STYLE+FL_EMBOSSED_STYLE);
  obj = fl_add_text(FL_NORMAL_TEXT,130,92,58,16,"Symbols");
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lalign(obj,FL_ALIGN_LEFT|FL_ALIGN_INSIDE);
  fdui->equation = obj = fl_add_pixmapbutton(FL_NORMAL_BUTTON,260,52,30,30,"");
    fl_set_object_color(obj,FL_MCOL,FL_BLUE);
    fl_set_object_callback(obj,button_cb,MM_EQU);
  fdui->sqrt = obj = fl_add_pixmapbutton(FL_NORMAL_BUTTON,28,52,30,30,"");
    fl_set_object_color(obj,FL_MCOL,FL_BLUE);
    fl_set_object_callback(obj,button_cb,MM_SQRT);
  fdui->frac = obj = fl_add_pixmapbutton(FL_NORMAL_BUTTON,88,52,30,30,"");
    fl_set_object_color(obj,FL_MCOL,FL_BLUE);
    fl_set_object_lcol(obj,FL_COL1);
    fl_set_object_callback(obj,button_cb,MM_FRAC);
  fdui->delim = obj = fl_add_pixmapbutton(FL_NORMAL_BUTTON,148,52,30,30,"");
    fl_set_object_color(obj,FL_MCOL,FL_BLUE);
    fl_set_object_callback(obj,button_cb,MM_DELIM);
  fdui->matrix = obj = fl_add_pixmapbutton(FL_NORMAL_BUTTON,208,52,30,30,"");
    fl_set_object_color(obj,FL_MCOL,FL_BLUE);
    fl_set_object_callback(obj,button_cb,MM_MATRIX);
  fl_end_form();
  fl_set_border_width(old_bw);

  return fdui;
}
/*---------------------------------------*/

FD_delim *create_form_delim(void)
{
  FL_OBJECT *obj;
  FD_delim *fdui = (FD_delim *) fl_calloc(1, sizeof(*fdui));
  int old_bw = fl_get_border_width();

  fl_set_border_width(-2);
  fdui->delim = fl_bgn_form(FL_NO_BOX, 216, 300);
  obj = fl_add_box(FL_UP_BOX,0,0,216,300,"");
  fdui->menu = obj = fl_add_bmtable(1,24,108,170,140,"");
    fl_set_object_lcol(obj,FL_BLUE);
    fl_set_object_callback(obj,delim_cb,2);

  fdui->lado = fl_bgn_group();
  fdui->right = obj = fl_add_checkbutton(FL_RADIO_BUTTON,40,67,60,30,"Right");
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_callback(obj,delim_cb,4);
  fdui->left = obj = fl_add_checkbutton(FL_RADIO_BUTTON,40,42,60,30,"Left");
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_callback(obj,delim_cb,3);
  fl_end_group();

  obj = fl_add_text(FL_NORMAL_TEXT,10,10,90,30,"Delimiter");
    fl_set_object_lcol(obj,FL_INACTIVE);
    fl_set_object_lsize(obj,FL_LARGE_SIZE);
    fl_set_object_lalign(obj,FL_ALIGN_LEFT|FL_ALIGN_INSIDE);
    fl_set_object_lstyle(obj,FL_TIMESITALIC_STYLE+FL_EMBOSSED_STYLE);
  obj = fl_add_button(FL_RETURN_BUTTON,122,264,56,24,"Apply  ");
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE);
    fl_set_object_callback(obj,delim_cb,0);
  obj = fl_add_button(FL_NORMAL_BUTTON,50,264,52,24,"Cancel");
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE);
    fl_set_object_callback(obj,delim_cb,1);
  fdui->pix = obj = fl_add_pixmap(FL_NORMAL_PIXMAP,120,50,50,40,"");
    fl_set_object_boxtype(obj,FL_FRAME_BOX);
  fl_end_form();
  fl_set_border_width(old_bw);

  return fdui;
}
/*---------------------------------------*/

FD_matrix *create_form_matrix(void)
{
  FL_OBJECT *obj;
  FD_matrix *fdui = (FD_matrix *) fl_calloc(1, sizeof(*fdui));
  int old_bw = fl_get_border_width();

  fl_set_border_width(-2);
  fdui->matrix = fl_bgn_form(FL_NO_BOX, 280, 198);
  obj = fl_add_box(FL_UP_BOX,0,0,280,198,"");
  obj = fl_add_text(FL_NORMAL_TEXT,6,8,60,24,"Matrix");
    fl_set_object_lcol(obj,FL_INACTIVE);
    fl_set_object_lsize(obj,FL_LARGE_SIZE);
    fl_set_object_lalign(obj,FL_ALIGN_LEFT|FL_ALIGN_INSIDE);
    fl_set_object_lstyle(obj,FL_TIMESITALIC_STYLE+FL_EMBOSSED_STYLE);
  obj = fl_add_button(FL_RETURN_BUTTON,174,162,56,24,"OK  ");
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE);
    fl_set_object_callback(obj,matrix_cb,0);
  obj = fl_add_button(FL_NORMAL_BUTTON,42,162,56,24,"Cancel");
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE);
    fl_set_object_callback(obj,matrix_cb,1);
  fdui->rows = obj = fl_add_valslider(FL_HOR_NICE_SLIDER,14,56,140,20,"Rows");
    fl_set_object_lsize(obj,FL_DEFAULT_SIZE);
    fl_set_object_lalign(obj,FL_ALIGN_TOP);
    fl_set_object_callback(obj,matrix_cb,-1);
    fl_set_slider_precision(obj, 0);
    fl_set_slider_bounds(obj, 1, 20);
    fl_set_slider_value(obj, 3);
     fl_set_slider_return(obj, FL_RETURN_END_CHANGED);
  fdui->columns = obj = fl_add_valslider(FL_HOR_NICE_SLIDER,14,114,140,20,"Columns");
    fl_set_object_lsize(obj,FL_DEFAULT_SIZE);
    fl_set_object_lalign(obj,FL_ALIGN_TOP);
    fl_set_object_callback(obj,matrix_cb,2);
    fl_set_slider_precision(obj, 0);
    fl_set_slider_bounds(obj, 1, 20);
    fl_set_slider_value(obj, 3);
     fl_set_slider_return(obj, FL_RETURN_END_CHANGED);
  fdui->valign = obj = fl_add_choice(FL_NORMAL_CHOICE,192,118,56,24,"Vertical align");
    fl_set_object_boxtype(obj,FL_UP_BOX);
    fl_set_object_lalign(obj,FL_ALIGN_TOP);
    fl_set_object_callback(obj,matrix_cb,-1);
  fdui->halign = obj = fl_add_input(FL_NORMAL_INPUT,174,56,94,24,"Horizontal align");
    fl_set_object_lalign(obj,FL_ALIGN_TOP);
    fl_set_object_callback(obj,matrix_cb,2);
  obj = fl_add_button(FL_NORMAL_BUTTON,108,162,56,24,"Apply");
    fl_set_button_shortcut(obj,"^M",1);
    fl_set_object_lsize(obj,FL_NORMAL_SIZE);
    fl_set_object_lstyle(obj,FL_BOLD_STYLE);
    fl_set_object_callback(obj,matrix_cb,3);
  fl_end_form();
  fl_set_border_width(old_bw);

  return fdui;
}
/*---------------------------------------*/

