/*************************************************************/
/* Original version was an example in the kernel source tree */
/*                                                           */
/* Rest was written by me, Michael Meskes                    */
/* meskes@informatik.rwth-aachen.de                          */
/*                                                           */
/*************************************************************/
#include "version.h"
#include <stdio.h>
#include <unistd.h>
#include <fcntl.h>
#include <signal.h>
#include <stdlib.h>
#include <strings.h>
#include <unistd.h>
#include <sys/stat.h>
#if defined(USE_SYSLOG)
#include <syslog.h>
#endif				/* USE_SYSLOG */
#include <sys/types.h>
#include <sys/wait.h>

#if !defined(PIDFILE)
#define PIDFILE "/var/run/watchdog.pid"
#endif				/* !PIDFILE */

#if !defined(DEVNAME)
#define DEVNAME "/dev/watchdog"
#endif				/* !DEVNAME */

#if !defined(TIMER_MARGIN)
#define TIMER_MARGIN 60
#endif				/* !defined TIMER_MARGIN */

#define TRUE 1
#define FALSE 0

int softdog;
char *devname = DEVNAME;
char *progname;

/* write a log entry on exit */
void log_end()
{
#if defined(USE_SYSLOG)
    /* Log the closinging message */
    syslog(LOG_INFO, "stopping daemon (%d.%d)", MAJOR_VERSION, MINOR_VERSION);
    closelog();
#endif				/* USE_SYSLOG */
    return;
}

/* close the device and check for error */
void close_dev()
{
    int err;

    err = close(softdog);
    if (err == -1) {
#if defined(USE_SYSLOG)
	syslog(LOG_ALERT, "cannot close %s", devname);
#else				/* USE_SYSLOG */
	perror(progname);
#endif				/* USE_SYSLOG */
    }
    return;
}

/* on exit we close the device and log that we stop */
void terminate(int arg)
{
    close_dev();
    log_end();
    exit(0);
}

void usage(void)
{
    fprintf(stderr, "%s version %d.%d, usage:\n", progname, MAJOR_VERSION, MINOR_VERSION);
#if defined(USE_SYSLOG)
    fprintf(stderr, "%s [-i <interval> [-f]] [-n <file-name>] [-d <dev-name>] [-v] [-s]\n", progname);
#else				/* USE_SYSLOG */
    fprintf(stderr, "%s [-i <interval> [-f]] [-n <file-name>] [-d <dev-name>] [-s]\n", progname);
#endif				/* USE_SYSLOG */
    exit(1);
}

/* get the basename of file <name> */
char *basename(char *name)
{
    char *tmp = strrchr(name, '/');

    return (tmp ? tmp + 1 : name);
}

/* panic: we're still alive but shoudln't */
void panic(void)
{
    /* if we are still alive, we just exit */
    close_dev();
    fprintf(stderr, "WATCHDOG PANIC: Still alive after sleeping %d seconds!\n", 4*TIMER_MARGIN);
#if defined(USE_SYSLOG)
    openlog(progname, LOG_PID, LOG_DAEMON);
    syslog(LOG_ALERT, "still alive after sleeping %d seconds", 4*TIMER_MARGIN);
    closelog();
#endif
    exit(1);
}

/* shut down the system in 2 minutes */
void do_shutdown(void)
{
    char *args[8];
    int i = 0;

    /* tell them we're leaving */
    log_end();

    args[i++] = "shutdown";
    args[i++] = "-r";
    args[i++] = "+2";
    args[i++] = NULL;

    execv("/sbin/shutdown", args);
    execv("/etc/shutdown", args);
    execv("/bin/shutdown", args);

    /* we cannot start shutdown */
    /* so we try causing a hard reset */
#if defined(USE_SYSLOG)
    openlog(progname, LOG_PID, LOG_DAEMON);
    syslog(LOG_ALERT, "cannot find a shutdown program, try a hard reset");
    closelog();
#else				/* USE_SYSLOG */
    perror("shutdown");
#endif				/* USE_SYSLOG */
    sync();
    sync();
    /* re-open the device */
    softdog = open(devname, O_WRONLY);
    sleep(4 * TIMER_MARGIN);	/* this should cause the reboot */
    panic();
}

/* Try to sync */
void sync_system(void)
{
    sync();
    sync();
}

int main(int argc, char *const argv[])
{
    FILE *fp;
    int c, tint = 10, force = FALSE, sync_it = FALSE;
    int softboot = FALSE, boot_now = FALSE;
    char *filename = NULL;
    char log[128];
#if defined(USE_SYSLOG)
    char *opts = "d:i:n:fsvb";
    int verbose = FALSE;
    long count = 0L;
#else				/* USE_SYSLOG */
    char *opts = "d:i:n:fsb";
#endif				/* USE_SYSLOG */

    progname = basename(argv[0]);
    /* check the options */
    while ((c = getopt(argc, argv, opts)) != EOF) {
	switch (c) {
	case 'n':
	    filename = optarg;
	    break;
	case 'd':
	    devname = optarg;
	    break;
	case 'i':
	    tint = atoi(optarg);
	    break;
	case 'f':
	    force = TRUE;
	    break;
	case 's':
	    sync_it = TRUE;
	    break;
	case 'b':
	    softboot = TRUE;
	    break;
#if defined(USE_SYSLOG)
	case 'v':
	    verbose = TRUE;
	    break;
#endif				/* USE_SYSLOG */
	default:
	    usage();
	}
    }

    if (tint < 0)
	usage();
    if (tint >= TIMER_MARGIN && !force) {
	fprintf(stderr, "%s warning:\n", progname);
	fprintf(stderr, "This interval length might reboot the system while the process sleeps!\n");
	fprintf(stderr, "To force this interval length use the -f option.\n");
	exit(1);
    }
    if (fork())
	exit(0);

    /* Okay, we're a daemon     */
#if defined(USE_SYSLOG)
    /* Log the starting message */
    openlog(progname, LOG_PID, LOG_DAEMON);
    sprintf(log, "starting daemon (%d.%d): ", MAJOR_VERSION, MINOR_VERSION);
    if (filename != NULL)
	sprintf(log + strlen(log), "%s ", filename);
    sprintf(log + strlen(log), "%ds%s%s", tint, sync_it ? " sync" : "", softboot ? " soft" : "");
    syslog(LOG_INFO, log);
#endif				/* USE_SYSLOG */

    /* open the device */
    softdog = open(devname, O_WRONLY);
    if (softdog == -1) {
#if defined(USE_SYSLOG)
	syslog(LOG_ERR, "cannot open %s", devname);
	log_end();
#else				/* USE_SYSLOG */
	perror(progname);
#endif				/* USE_SYSLOG */
	exit(1);
    }
    /* tuck my process id away */
    fp = fopen(PIDFILE, "w");
    if (fp != NULL) {
	fprintf(fp, "%d\n", getpid());
	(void) fclose(fp);
    }
    /* set signal term to call terminate() */
    /* to make sure softdog device is closed */
    signal(SIGTERM, terminate);

    /* main loop: update after <tint> seconds */
    while (1) {
	pid_t forkresult;

	/* write to the softdof device */
	write(softdog, "\0", 1);

	/* sync system if we have to */
	if (sync_it)
	    sync_system();

	/* check if process table is full */
	forkresult = fork();
	if (!forkresult)
	    exit(0);		/* child, exit immediately */
	else if (forkresult < 0) {	/* fork failed */
#if defined(USE_SYSLOG)
	    syslog(LOG_ERR, "process table is full!");
#endif				/* USE_SYSLOG */
	    closelog();
	    sync_system();
	    sleep(4 * TIMER_MARGIN);	/* should be enough to reboot */
	    panic();
	}
	/* now wait for child to stop */
	if (waitpid(forkresult, NULL, 0) < 0) {
#if defined(USE_SYSLOG)
	    syslog(LOG_ERR, "child %d does not exist", forkresult);
#else				/* USE_SYSLOG */
	    perror(progname);
#endif				/* USE_SYSLOG */
	    boot_now = TRUE;
	}
	/* do verbose  logging */
#if defined(USE_SYSLOG)
	if (verbose) {
	    count++;
	    syslog(LOG_INFO, "write (#%ld) to %s", count, devname);
	}
#endif				/* USE _SYSLOG */

	/* in filemode stat file */
	if (filename != NULL) {
	    struct stat buf;

	    if (stat(filename, &buf) == -1) {
#if defined(USE_SYSLOG)
		syslog(LOG_ERR, "cannot stat %s", filename);
#else				/* USE_SYSLOG */
		perror(progname);
#endif				/* USE_SYSLOG */
		boot_now = TRUE;
	    }
	}
	/* do we have to do a softboot? */
	if (softboot && boot_now) {
	    /* soft-boot the system */
	    /* first close the device */
	    close_dev();
#if defined(USE_SYSLOG)
	    /* now tell syslog what's happening */
	    syslog(LOG_INFO, "calling shutdown to reboot the system");
#endif				/* USE_SYSLOG */
	    /* finally bring the system down */
	    /* we cannot close the log since we have to */
	    /* log shutdown problems, too */
	    do_shutdown();
	}
	/* finally sleep some seconds */
	sleep(tint);
    }
}
