/*
 **************************************************************************
 *
 * Utility program to create the configuration shell script.
 *
 * Module:  config.c
 * Purpose: Create configuration shell script for bootrom
 * Entries: main
 *
 **************************************************************************
 *
 * Copyright (C) 1995,1996 Gero Kuhlmann <gero@gkminix.han.de>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <stdio.h>
#include <stdlib.h>
#include <fcntl.h>
#include <string.h>
#include <ctype.h>
#ifdef MSDOS
#include <io.h>
#include <dir.h>
#endif
#ifdef UNIX
#include <dirent.h>
#endif
#include "../../headers/general.h"
#include "../../headers/version.h"


#ifndef ROOTDIR
#define ROOTDIR		"."
#endif
#ifdef MSDOS
#define KERNELDIR	ROOTDIR "\\binaries"
#define PKTDRVDIR	ROOTDIR "\\pktdrvr"
#define UTILSDIR	ROOTDIR "\\utils"
#else
#define KERNELDIR	ROOTDIR "/binaries"
#define PKTDRVDIR	ROOTDIR "/pktdrvr"
#define UTILSDIR	ROOTDIR "/utils"
#endif
#define DEFAULTINT	0x62
#define PROGNUM		8

#ifdef UNIX
#define DEFAULT_SCRIPT	"mkimage.sh"
#endif
#ifdef MSDOS
#define DEFAULT_SCRIPT	"mkimage.bat"
#endif
#define DEFAULT_CONFIG	"mkimage.cfg"



/* Public variables */
static char *cfgname = DEFAULT_CONFIG;
static char *scriptname = DEFAULT_SCRIPT;
static char *kernelname;
static char *pktdrvname;
static char *pktdrvoptions;
static char *progname[PROGNUM];
static char *progoptions[PROGNUM];
static int num_progs = 0;
static int is86 = 0;
static int isdefint = 0;



/* Directory entry name list */
static struct direntry {
	struct direntry *next;
	char             fname[1];
} *dirlist = NULL;



/* List of packet driver options for selected Crynwr packet drivers */
#define HW_IRQ		0x01
#define IO_ADDR		0x02
#define BASE_MEM	0x04
#define AUI_TYPE	0x08
#define DMA_NUM		0x10

static struct pktopt {
	char *pktname;
	int   pktopt;
} pktlist[] = {
	{ "3c501",	HW_IRQ | IO_ADDR },
	{ "3c503",	HW_IRQ | IO_ADDR | AUI_TYPE },
	{ "3c509",	0 },
	{ "arcether",	HW_IRQ | IO_ADDR | BASE_MEM },
	{ "depca",	HW_IRQ | IO_ADDR | BASE_MEM },
	{ "ne1000",	HW_IRQ | IO_ADDR },
	{ "ne2000",	HW_IRQ | IO_ADDR },
	{ "ne2100",	HW_IRQ | IO_ADDR | DMA_NUM },
	{ "smc_wd",	HW_IRQ | IO_ADDR | BASE_MEM },
	{ NULL,		0 }};



/**************************************************************************

		MSDOS dependent functions

 **************************************************************************/


#ifdef MSDOS
/*
 * Convert a unix path to a DOS path with backslashes
 */
static void convpath(char *str)
{
  while (*str) {
	if (*str == '/')
		*str = '\\';
	str++;
  }
}



/*
 * Get a list of directory entries with a specified name pattern.
 */
static int getdirlist(char *pattern)
{
  struct ffblk ff;
  struct direntry *dp;
  int len, count;
  char *src, *dest;

  convpath(pattern);
  if (findfirst(pattern, &ff, FILENAME | EXTENSION | WILDCARDS) < 0)
	return(0);
  count = 0;
  while (TRUE) {
	len = strlen(ff.ff_name) + sizeof(struct direntry);
	if ((dp = (struct direntry *)malloc(len)) == NULL) {
		fprintf(stderr, "not enough memory for directory list\n");
		exit(1);
	}
	for (src = ff.ff_name, dest = dp->fname; *src; src++, dest++) {
		if (*src == '.') break;
		*dest = tolower(*src);
	}
	*dest = '\0';
	dp->next = dirlist;
	dirlist = dp;
	count++;
	if (findnext(&ff) < 0)
		break;
  }
  return(count);
}



/*
 * Create the configuration shell script
 */
static void buildconfig(void)
{
  FILE *outfile;
  int i;

  /* First write script file */
  if ((outfile = fopen(scriptname, "wt")) == NULL) {
	fprintf(stderr, "cannot create script file %s\n", scriptname);
	exit(1);
  }
  fprintf(outfile, "@echo off\n");
  fprintf(outfile, "%s\\pass1 -k %s -c %s -o image.bin\n",
					UTILSDIR, kernelname, cfgname);
  fprintf(outfile, "%s\\compress -b15 image.bin\n", UTILSDIR);
  fprintf(outfile, "%s\\pass2 -i image.biz -l %s\\%s -o image.flo\n",
	UTILSDIR, KERNELDIR, is86 ? "floppy86.bin" : "floppy.bin");
  fprintf(outfile, "%s\\pass2 -i image.biz -l %s\\%s -o image.rom\n",
	UTILSDIR, KERNELDIR, is86 ? "rom86.bin" : "rom.bin");
  fprintf(outfile, "if exist image.biz del image.biz\n");
  fprintf(outfile, "if exist image.bin del image.bin\n");
  fprintf(outfile, "if exist %s del %s\n", cfgname, cfgname);
  fclose(outfile);

  /* Write the pass1 configuration file */
  if ((outfile = fopen(cfgname, "wt")) == NULL) {
	fprintf(stderr, "cannot create config file %s\n", cfgname);
	exit(1);
  }
  fprintf(outfile, "%s %s\n", pktdrvname, pktdrvoptions);
  for (i = 0; i < num_progs; i++)
	fprintf(outfile, "%s %s\n", progname[i], progoptions[i]);
  fclose(outfile);
}
#endif



/**************************************************************************

		UNIX dependent functions

 **************************************************************************/

#ifdef UNIX
/*
 * Compare a directory entry against a pattern. This covers wildcards
 * in the same simple way as MSDOS does.
 */
static int dircmp(char *pattern, char *name)
{
  char *cp;

  while (*pattern && *name) {
	if (*pattern == '?' || *pattern == *name) {
		pattern++;
		name++;
	} else {
		if (*pattern != '*')
			break;
		pattern++;
		while (*name && *name != *pattern)
			name++;
	}
  }
  return(!*pattern && !*name);
}



/*
 * Get a list of directory entries with a specified name pattern.
 */
static int getdirlist(char *pattern)
{
  DIR *dir;
  struct dirent *ent;
  struct direntry *dp;
  int len, count;
  char *dirname;
  char *src, *dest;

  /* First open the directory */
  if ((dirname = strdup(pattern)) == NULL) {
	fprintf(stderr, "not enough memory to save directory pattern\n");
	exit(1);
  }
  if ((src = strrchr(dirname, '/')) == NULL) {
	pattern = dirname;
	dirname = ".";
  } else {
	*src++ = '\0';
	pattern = src;
  }
  if ((dir = opendir(dirname)) == NULL) {
	fprintf(stderr, "cannot open directory %s\n", dirname);
	exit(1);
  }

  /* Next step through all directory entries */
  count = 0;
  while ((ent = readdir(dir)) != NULL) {
	if (!dircmp(pattern, ent->d_name))
		continue;
	len = strlen(ent->d_name) + sizeof(struct direntry);
	if ((dp = (struct direntry *)malloc(len)) == NULL) {
		fprintf(stderr, "not enough memory for directory list\n");
		exit(1);
	}
	for (src = ent->d_name, dest = dp->fname; *src; src++, dest++) {
		if (*src == '.') break;
		*dest = tolower(*src);
	}
	*dest = '\0';
	dp->next = dirlist;
	dirlist = dp;
	count++;
  }
  closedir(dir);
  return(count);
}



/*
 * Create the configuration shell script
 */
static void buildconfig(void)
{
  FILE *outfile;
  int i;

  /* First write script file */
  if ((outfile = fopen(scriptname, "w")) == NULL) {
	fprintf(stderr, "cannot create script file %s\n", scriptname);
	exit(1);
  }
  fprintf(outfile, "#!/bin/sh\n");
  fprintf(outfile, "%s/pass1 -k %s -c %s -o image.bin\n",
					UTILSDIR, kernelname, cfgname);
  fprintf(outfile, "compress -b15 image.bin\n");
  fprintf(outfile, "%s/pass2 -i image.bin.Z -l %s/%s -o image.flo\n",
	UTILSDIR, KERNELDIR, is86 ? "floppy86.bin" : "floppy.bin");
  fprintf(outfile, "%s/pass2 -i image.bin.Z -l %s/%s -o image.rom\n",
	UTILSDIR, KERNELDIR, is86 ? "rom86.bin" : "rom.bin");
  fprintf(outfile, "rm -f image.bin image.bin.Z %s\n", cfgname);
  fclose(outfile);

  /* Write the pass1 configuration file */
  if ((outfile = fopen(cfgname, "w")) == NULL) {
	fprintf(stderr, "cannot create config file %s\n", cfgname);
	exit(1);
  }
  fprintf(outfile, "%s %s\n", pktdrvname, pktdrvoptions);
  for (i = 0; i < num_progs; i++)
	fprintf(outfile, "%s %s\n", progname[i], progoptions[i]);
  fclose(outfile);
}
#endif



/**************************************************************************

			OS independent functions

 **************************************************************************/


/*
 * Delete directory list
 */
static void deldirlist(void)
{
  struct direntry *dp1, *dp2;

  dp1 = dirlist;
  while (dp1 != NULL) {
	dp2 = dp1;
	dp1 = dp1->next;
	free(dp2);
  }
  dirlist = NULL;
}



/*
 * Get a string response from the user
 */
static char *getstring(void)
{
  static char buf[1024];
  char *cp;

  fgets(buf, sizeof(buf) - 1, stdin);
  for (cp = buf; *cp && *cp != '\n'; cp++) ;
  *cp = '\0';
  return(buf);
}



/*
 * Get a numeric response from the user
 */
static long getnum(char *prompt, long min, long max, int hex)
{
  long j;
  char *cp;

  j = min - 1;
  while (j < min || j > max) {
	printf("  %s (%s): ", prompt, hex ? "hex" : "decimal");
	cp = getstring();
	if ((hex ? sscanf(cp, "%lx", &j) : sscanf(cp, "%ld", &j)) != 1)
		j = min - 1;
	if (j < min || j > max)
		if (hex)
			printf("  Only allowed values are [0x%lx..0x%lx]\n",
								min, max);
		else
			printf("  Only allowed values are [%ld..%ld]\n",
								min, max);
  }
  return(j);
}



/*
 * Get packet driver options
 */
static void getpktopt(char *pktname)
{
  int i;
  long j;
  char *cp;

  i = 0;
  while (pktlist[i].pktname != NULL) {
	if (!strcmp(pktlist[i].pktname, pktname))
		break;
	i++;
  }
  if (pktlist[i].pktname == NULL)
	return;
  if ((pktdrvoptions = malloc(1024)) == NULL) {
	fprintf(stderr, "cannot allocate memory for pktdrv options\n");
	exit(1);
  }
  sprintf(pktdrvoptions, "0x%x", DEFAULTINT);
  isdefint++;
  if (pktlist[i].pktopt) {
	cp = pktdrvoptions + strlen(pktdrvoptions);
	*cp++ = ' ';
	printf("Enter packet driver command line options:\n");
	if (pktlist[i].pktopt & HW_IRQ) {
		j = getnum("Hardware IRQ number", 2, 15, FALSE);
		sprintf(cp, "%ld", j);
		cp = pktdrvoptions + strlen(pktdrvoptions);
		if (pktlist[i].pktopt &= ~HW_IRQ)
			*cp++ = ' ';
	}
	if (pktlist[i].pktopt & IO_ADDR) {
		j = getnum("I/O address", 0, 0x1fff, TRUE);
		sprintf(cp, "0x%lx", j);
		cp = pktdrvoptions + strlen(pktdrvoptions);
		if (pktlist[i].pktopt &= ~IO_ADDR)
			*cp++ = ' ';
	}
	if (pktlist[i].pktopt & BASE_MEM) {
		j = getnum("Shared memory address", 0, 0xffff, TRUE);
		sprintf(cp, "0x%lx", j);
		cp = pktdrvoptions + strlen(pktdrvoptions);
		if (pktlist[i].pktopt &= ~BASE_MEM)
			*cp++ = ' ';
	}
	if (pktlist[i].pktopt & DMA_NUM) {
		j = getnum("DMA number", 0, 7, FALSE);
		sprintf(cp, "%ld", j);
		cp = pktdrvoptions + strlen(pktdrvoptions);
		if (pktlist[i].pktopt &= ~DMA_NUM)
			*cp++ = ' ';
	}
	if (pktlist[i].pktopt & AUI_TYPE) {
		j = getnum("AUI type (-1 for default)", -1, 1, FALSE);
		if (j > -1) {
			sprintf(cp, "%ld", j);
			cp = pktdrvoptions + strlen(pktdrvoptions);
		}
	}
  }
  return;
}



/*
 * Let the user select a packet driver.
 */
static void getpktdrv(void)
{
  int num, sel, i;
  struct direntry *dp;
  char *cp;

  if ((num = getdirlist(PKTDRVDIR "/*.com")) == 0) {
	fprintf(stderr, "Oops, no packet drivers found!\n");
	exit(1);
  }
  printf("The following %d packet drivers are available:\n", num);
  printf("(0)    user defined packet driver\n");
  for (dp = dirlist, i = 1; dp != NULL; dp = dp->next, i++)
	printf("(%d)    %s\n", i, dp->fname);
  sel = -1;
  while (sel < 0 || sel > num) {
	printf("Select the packet driver you wish to use: ");
	cp = getstring();
	sscanf(cp, "%d", &sel);
  }
  if (sel == 0) {
	printf("Enter the full path name of the packet driver: ");
	cp = getstring();
	if ((i = strcspn(cp, " \t")) < strlen(cp))
		cp[i] = '\0';
	if ((pktdrvname = malloc(strlen(cp) + 2)) == NULL) {
		fprintf(stderr, "not enough memory for packet driver path\n");
		exit(1);
	}
	strcpy(pktdrvname, cp);
  } else {
	for (dp = dirlist, i = 1; dp != NULL && sel != i; dp = dp->next, i++) ;
	if (dp != NULL) {
		i = strlen(dp->fname) + strlen(PKTDRVDIR) + 6;
		if ((pktdrvname = malloc(i)) == NULL) {
			fprintf(stderr, "not enough memory for packet driver path name\n");
			exit(1);
		}
		sprintf(pktdrvname, PKTDRVDIR "/%s.com", dp->fname);
		getpktopt(dp->fname);
	}
  }
  if (pktdrvoptions == NULL) {
	printf("Enter command line arguments for packet driver: ");
	cp = getstring();
	if ((pktdrvoptions = malloc(strlen(cp) + 2)) == NULL) {
		fprintf(stderr, "not enough memory for packet driver options\n");
		exit(1);
	}
	strcpy(pktdrvoptions, cp);
  }
#ifdef MSDOS
  convpath(pktdrvname);
#endif
  deldirlist();
  printf("\n\n\n");
}


/*
 * Let the user select the name of an additional program.
 */
static void getprogram(void)
{
  char *cp;
  long intnum = DEFAULTINT;
  long pkttype = 0;


  if (getdirlist(UTILSDIR "/ansidrv.com") == 1) {
	printf("Do you want to use the ANSI display driver (y/n) ? ");
	cp = getstring();
	if (*cp == 'y' || *cp == 'Y') {
		if ((cp = malloc(1024)) == NULL) {
			fprintf(stderr, "not enough memory for temp buf\n");
			exit(1);
		}
		progoptions[num_progs] = "";
		sprintf(cp, UTILSDIR "/%s.com", dirlist->fname);
		if ((progname[num_progs] = malloc(strlen(cp) + 2)) == NULL) {
			fprintf(stderr, "not enough memory for path name\n");
			exit(1);
		}
		strcpy(progname[num_progs], cp);
#ifdef MSDOS
		convpath(progname[num_progs]);
#endif
		num_progs++;
		free(cp);
	}
  }
  deldirlist();

  if (getdirlist(UTILSDIR "/pktwatch.com") == 1) {
	printf("Do you want to use the packet driver debug program (y/n) ? ");
	cp = getstring();
	if (*cp == 'y' || *cp == 'Y') {
		if ((cp = malloc(1024)) == NULL) {
			fprintf(stderr, "not enough memory for temp buf\n");
			exit(1);
		}
		if (!isdefint)
			intnum = getnum("packet driver interrupt number", 0x60, 0x7f, TRUE);
		pkttype = getnum("packet type (0 for default)", 0, 0xffff, TRUE);
		if (pkttype == 0)
			sprintf(cp, "%lx", intnum);
		else
			sprintf(cp, "%lx %lx", intnum, pkttype);
		if ((progoptions[num_progs] = malloc(strlen(cp) + 2)) == NULL) {
			fprintf(stderr, "not enough memory for options\n");
			exit(1);
		}
		strcpy(progoptions[num_progs], cp);
		sprintf(cp, UTILSDIR "/%s.com", dirlist->fname);
		if ((progname[num_progs] = malloc(strlen(cp) + 2)) == NULL) {
			fprintf(stderr, "not enough memory for path name\n");
			exit(1);
		}
		strcpy(progname[num_progs], cp);
#ifdef MSDOS
		convpath(progname[num_progs]);
#endif
		num_progs++;
		free(cp);
	}
  }
  deldirlist();

  while(1) {
	printf("Do you want to specify an additional program (y/n) ? ");
	cp = getstring();
	if (*cp == 'y' || *cp == 'Y') {
		printf("Enter the full path name of the program: ");
		cp = getstring();
		if ((progname[num_progs] = malloc(strlen(cp) + 2)) == NULL) {
			fprintf(stderr, "not enough memory for path name\n");
			exit(1);
		}
		strcpy(progname[num_progs], cp);
		printf("Enter command line arguments for program: ");
		cp = getstring();
		if ((progoptions[num_progs] = malloc(strlen(cp) + 2)) == NULL) {
			fprintf(stderr, "not enough memory for options\n");
			exit(1);
		}
		strcpy(progoptions[num_progs], cp);
#ifdef MSDOS
		convpath(progname[num_progs]);
#endif
		num_progs++;
	} else break;
  }
  printf("\n\n");
}


/*
 * Let the user select a bootrom kernel.
 */
static void getkernel(void)
{
  int num, sel, i;
  struct direntry *dp;
  char *cp;

  if ((num = getdirlist(KERNELDIR "/kernel*.bin")) == 0) {
	fprintf(stderr, "Oops, no kernel binaries found!\n");
	exit(1);
  }
  printf("The following %d kernels are available:\n", num);
  for (dp = dirlist, i = 1; dp != NULL; dp = dp->next, i++) {
	printf("(%d)    %s\t", i, dp->fname);
	if (!strcmp(dp->fname, "kernel"))
		printf("-  standard kernel (recommended)\n");
	else if (!strcmp(dp->fname, "kernelm"))
		printf("-  minimal kernel for small bootroms\n");
	else if (!strcmp(dp->fname, "kernel86"))
		printf("-  kernel for 16-Bit x86 processors\n");
	else
		printf("\n");
  }
  sel = -1;
  while (sel < 1 || sel > num) {
	printf("Select the kernel you wish to use: ");
	cp = getstring();
	sscanf(cp, "%d", &sel);
  }
  for (dp = dirlist, i = 1; dp != NULL && sel != i; dp = dp->next, i++) ;
  if (dp != NULL) {
	i = strlen(dp->fname) + strlen(KERNELDIR) + 6;
	if ((kernelname = malloc(i)) == NULL) {
		fprintf(stderr, "not enough memory for kernel path name\n");
		exit(1);
	}
	sprintf(kernelname, KERNELDIR "/%s.bin", dp->fname);
	if (!strcmp(dp->fname, "kernel86"))
		is86++;
  }
#ifdef MSDOS
  convpath(kernelname);
#endif
  deldirlist();
  printf("\n\n\n");
}


void usage(void)
{
  fprintf(stderr, "usage: config [-s<script file>] [-c<config file>]\n");
  exit(1);
}


void main(int argc, char **argv)
{
  char *cp;
  int i;

  /* Isolate the different command line options */
  for (i = 1; i < argc; i++) {
	if (*argv[i] != '-' || strlen(argv[i]) < 2)
		usage();
	 switch (argv[i][1]) {
		case 'c':  if (strlen(argv[i]) > 2) {
				cp = &argv[i][2];
		           } else {
				if (++i >= argc) usage();
				cp = argv[i];
		           }
		           cfgname = strdup(cp);
		           break;
		case 's':  if (strlen(argv[i]) > 2) {
				cp = &argv[i][2];
		           } else {
				if (++i >= argc) usage();
				cp = argv[i];
		           }
		           scriptname = strdup(cp);
		           break;
		default:   usage();
	}
  }

  /* Check that all necessary command line options are given */
  if (scriptname == NULL || cfgname == NULL)
	usage();

  printf("\nConfiguration program for bootrom, version %d.%d\n\n\n",
							MAJOR_VER, MINOR_VER);
  getkernel();
  getpktdrv();
  getprogram();
  printf("Building configuration script for the following parameters:\n");
  printf("Kernel:        \"%s\"\n", kernelname);
  printf("Packet driver: \"%s %s\"\n", pktdrvname, pktdrvoptions);
  for (i = 0; i < num_progs; i++)
	printf("Program %d:     \"%s %s\"\n", i, progname[i], progoptions[i]);
  buildconfig();
}

