;**************************************************************************
;*
;* PKTWATCH - Log packets going through a packet driver
;*
;* Module:  pktwatch.asm
;* Purpose: Log packets going through a packet driver
;* Entries: start
;*
;**************************************************************************
;*
;* Copyright (C) 1995,1996 Gero Kuhlmann <gero@gkminix.han.de>
;*
;*  This program is free software; you can redistribute it and/or modify
;*  it under the terms of the GNU General Public License as published by
;*  the Free Software Foundation; either version 2 of the License, or
;*  any later version.
;*
;*  This program is distributed in the hope that it will be useful,
;*  but WITHOUT ANY WARRANTY; without even the implied warranty of
;*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;*  GNU General Public License for more details.
;*
;*  You should have received a copy of the GNU General Public License
;*  along with this program; if not, write to the Free Software
;*  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
;*
;**************************************************************************
;*

;
;**************************************************************************
;
; The purpose of this program is to log all packets onto the screen, which
; go through a packet driver. The packet driver interrupt has to be specified
; on the command line in hex. It then splits the screen into two halves by
; using the 43x80 text resolution on a EGA/VGA card. The upper 25 lines
; receive the unmodified output from the original screen, and the lower
; lines receive the packets in hex dump.
; We don't use any of the packet driver assembler include modules here, as
; this is a completely independent program. No fancy 386 optimization here.
; Therefore we can use the standard MASM shortcuts for defining segments
; etc., so we let MASM assume a small model (this is going to become a COM
; program!) and pascal calling convention.
;
.model	small,pascal


;
;**************************************************************************
;
; Definition of BIOS data area:
;
biosseg		segment	at 0040h

		org	0049h

b_vidmode	db	?		; current video mode
b_columns	dw	?		; number of screen columns
b_vidsize	dw	?		; size of video buffer in bytes
b_vidaddr	dw	?		; address of video buffer
b_curpos	dw	8 dup (?)	; cursor positions for 8 screen pages
b_curtype	dw	?		; cursor type
b_vidpage	db	?		; current screen page
b_vidbase	dw	?		; base address of video controller

		org	0080h

b_kbdbegin	dw	?		; offset to beginning of keyboard buf
b_kbdend	dw	?		; offset to end of keyboard buffer
b_lines		db	?		; number of screen lines - 1
b_charsize	dw	?		; number of scan lines per character

biosseg		ends


;
;**************************************************************************
;
; Miscellaneous defines:
;
CMDLINE		equ	00080h		; offset of command line into PSP
VIDEO_SEG	equ	0B800h		; segment of video buffer
DEACTIVATE	equ	0CF45h		; magic key for deactivation

MAX_BYTES	equ	36		; max num of buffer bytes to display
IP_TYPE		equ	0008h		; IP network type in network order
ADDR_SIZE	equ	6		; size of hardware address
DEST_OFS	equ	6		; offset to dest addr in MAC header
TYPE_OFS	equ	12		; offset to packet type in MAC header
DATA_OFS	equ	14		; offset to data area in ethernet pkt


;
;**************************************************************************
;
; There is no other than a code segment. All data will be placed in it
; as well. The resident section of the program starts here.
;
.code

	org	0100h			; we have to start at 0100h

	assume	cs:_text
	assume	ds:_text		; code and data are in the same segment
	assume	es:nothing

start:	jmp	start1


;
;**************************************************************************
;
; Data segment:
;

scnlines	dw	0		; number of screen lines
scncolumns	dw	0		; number of screen columns
scnbytes	dw	0		; number of bytes per screen line
vidbufsize	dw	0		; video buffer size
vidoffset	dw	0		; offset into second half of screen
curoffset	dw	0		; current character offset

pkttype		dw	IP_TYPE		; packet type to display

pktint		dw	0		; packet driver interrupt number
old2f_vect	dd	0		; old 2Fh interrupt vector
oldpkt_vect	dd	0		; old packet driver interrupt vector
oldreceiver	dd	0		; pointer to old receiver routine


;
;**************************************************************************
;
; Interrupt routine to terminate pktwatch services. This will wait for
; a key press, and then restore the screen.
; Input:  AX  -  Magic key 0CF45h
; Output: none
; Registers used: none
;
new_int2f	proc	far

	assume	ds:nothing
	cmp	ax,DEACTIVATE		; check if we have to deactivate
	je	short int2f1
	jmp	old2f_vect

int2f1:	cli
	push	ax
	push	si
	push	ds
	cli
	xor	ax,ax			; first restore all interrupt vectors
	mov	ds,ax
	mov	si,pktint
	shl	si,1
	shl	si,1
	mov	ax,word ptr [oldpkt_vect + 0]
	mov	ds:[si + 0],ax
	mov	ax,word ptr [oldpkt_vect + 2]
	mov	ds:[si + 2],ax
	mov	ax,word ptr [old2f_vect + 0]
	mov	ds:[2Fh * 4 + 0],ax
	mov	ax,word ptr [old2f_vect + 2]
	mov	ds:[2Fh * 4 + 2],ax
	pushf
	call	old2f_vect		; daisy chain old interrupt
	sti

	mov	ax,cs			; let DS point to data segment
	mov	ds,ax
	assume	ds:_text
	call	scrollup		; scroll screen up one line
	mov	si,offset waitmsg
	call	printstr		; print waiting message
	xor	ah,ah
	int	16h			; wait for key press
	mov	ax,0003h		; restore screen to mode 3
	int	10h
	pop	ds
	pop	si
	pop	ax
	iret

waitmsg	db	'Terminating pktwatch. Press a key to continue...', 0

new_int2f	endp


;
;**************************************************************************
;
; New packet driver interrupt.
; Input:  none
; Output: none
; Registers changed: none
;
new_pkt		proc	far

	assume	ds:nothing
	assume	es:nothing

	jmp	short newpk0

	db	0
	db	'PKT DRVR'		; packet driver signature

newpk0:	cmp	ah,4			; send packet?
	je	short newpk2
	cmp	ah,2			; initialize access type?
	je	short newpk4
newpk1:	jmp	oldpkt_vect		; call old packet vector

; Requested to send a packet.

newpk2:	push	ax
	mov	ax,pkttype
	cmp	word ptr ds:[si + TYPE_OFS],ax
	jne	short newpk3
	mov	al,'S'			; character indicating 'send'
	call	printbuf
newpk3:	pop	ax
	jmp	short newpk1

; Requested to set new packet access type. We only handle packets with
; a specified packet type, e.g. 'typelen' has to be exactly 2 which is
; the size of word. The receiver routine is then replaced by a new one
; which prints every received packet.

newpk4: push	ax
	cmp	cx,size pkttype		; check for correct packet type size
	jne	short newpk3
	mov	ax,word ptr ds:[si]	; check for correct packet type
	cmp	ax,pkttype
	jne	short newpk3
	mov	word ptr [oldreceiver + 0],di
	mov	word ptr [oldreceiver + 2],es
	mov	ax,cs
	mov	es,ax
	mov	di,offset new_receiver	; replace old receiver routine with
	jmp	short newpk3		; new one

	assume	ds:_text

new_pkt		endp


;
;**************************************************************************
;
; New receiver routine. This routine gets called by the packet driver
; whenever a new packet arrives.
; Input:  BX     -  handle
;         AX     -  flag indicating first or second call
;         CX     -  buffer length
;         DS:SI  -  pointer to receive buffer (only if AX = 1)
; Output: none
; Registers changed: none
;
new_receiver	proc	far

	assume	ds:nothing
	assume	es:nothing

	or	ax,ax			; check if this is the correct call
	jz	newrc9
	push	ax
	mov	al,'R'			; character indicating 'receive'
	call	printbuf		; print buffer contents
	pop	ax
newrc9:	jmp	oldreceiver

	assume	ds:_text

new_receiver	endp


;
;**************************************************************************
;
; Print a packet buffer into the last screen line
; Input:  DS:SI  -  pointer to buffer
;         CX     -  size of buffer
;         AL     -  character in first column
; Output: none
; Registers changed: AX
;
printbuf	proc	near

	push	es
	push	ds
	pop	es			; point ES:SI to buffer
	push	cs
	pop	ds
	push	bx
	xor	bx,bx
	cmp	al,'S'
	jne	short prnb0		; select source or destination address
	mov	bx,DEST_OFS		; to print
prnb0:	call	scrollup		; scroll screen up one line
	call	printchr
	mov	al,' '			; first print the character indicating
	call	printchr		; direction of packet

prnb1:	mov	al,es:[si + bx]
	call	printbyte		; print next byte
	cmp	bx,DEST_OFS - 1
	jne	short prnb4
	add	bx,ADDR_SIZE		; skip destination addr if source addr
prnb4:	cmp	bx,TYPE_OFS - 1		; printed
	jne	short prnb5
	add	bx,DATA_OFS - TYPE_OFS	; skip packet type
prnb5:	cmp	bx,DATA_OFS - 1
	jb	short prnb3
prnb2:	mov	al,' '			; print blanks between data bytes
	call	printchr
prnb3:	inc	bx
	cmp	bx,cx			; check if at the end of the buffer
	jae	short prnb9
	cmp	bx,MAX_BYTES		; can't write more than one line
	jb	short prnb1

prnb9:	pop	bx
	push	es
	pop	ds			; restore DS:SI
	pop	es
	ret

printbuf	endp


;
;**************************************************************************
;
; Print a byte in hex form onto screen
; Input:  AL  -  byte value
; Output: none
; Registers changed: AX
;
printbyte	proc	near

	push	ax
	shr	al,1
	shr	al,1
	shr	al,1			; first print upper nibble
	shr	al,1
	call	printnib
	pop	ax

printnib	label	near

	and	al,0Fh
	add	al,'0'
	cmp	al,'9'			; convert nibble and print it
	jbe	short prnn1
	add	al,7
prnn1:	call	printchr
	ret

printbyte	endp


;
;**************************************************************************
;
; Scroll the lower half of the screen up one line.
; Input:  none
; Output: none
; Registers used: none
;
scrollup	proc	near

	cld
	push	es
	push	di
	push	ds
	push	si
	push	cx
	mov	cx,VIDEO_SEG
	mov	es,cx			; set video buffer segment
	mov	ds,cx
	assume	ds:nothing
	mov	si,scnbytes		; compute start of source and
	mov	di,vidoffset		; destination area
	add	si,di
	mov	cx,vidbufsize		; compute number of words to move
	sub	cx,si
	shr	cx,1
	rep	movsw			; now scroll the screen
	mov	si,ax			; save AX
	mov	ax,0720h		; fill last line with blanks
	mov	cx,scncolumns
	rep	stosw
	mov	ax,si			; restore AX
	pop	cx
	pop	si
	pop	ds
	assume	ds:_text
	sub	di,scnbytes
	mov	curoffset,di		; set current character position to
	pop	di			; first column in last line
	pop	es
	ret

scrollup	endp


;
;**************************************************************************
;
; Print a string into the last screen line.
; Input:  DS:SI  -  pointer to zero terminated string
; Output: none
; Registers used: AX, SI
;
printstr	proc	near

	cld
	push	es
	push	di
	mov	di,VIDEO_SEG		; set video buffer segment
	mov	es,di
	mov	di,curoffset
	mov	ah,07h			; white on black
prnst1:	lodsb				; now print the string
	or	al,al			; check for end of string
	jz	short prnst2
	stosw				; store character into video buffer
	jmp	short prnst1
prnst2:	pop	di
	pop	es
	ret

printstr	endp


;
;**************************************************************************
;
; Print a character into the last screen line.
; Input:  AL  -  character to print
; Output: none
; Registers used: AX
;
printchr	proc	near

	cld
	push	es
	push	di
	mov	di,VIDEO_SEG		; set video buffer segment
	mov	es,di
	mov	di,curoffset
	mov	ah,07h			; white on black
	stosw				; write character onto screen
	mov	curoffset,di		; save new character position
	pop	di
	pop	es
	ret

printchr	endp


;
;**************************************************************************
;
; Start the non-resident section here:
;
start1:	mov	ax,cs
	mov	ds,ax
	assume	ds:_text
	assume	es:nothing

; First analyze the command line. The only arguments for this program
; are the interrupt vector and the packet type in hex. Then initialize
; the screen.

	mov	si,CMDLINE + 1
	call	skipblank
	cmp	al,0Dh			; must be hex number
	je	short start5
	call	gethex			; get hex number
	cmp	bx,0060h		; should be in the range of 60h
	jb	short start2		; and 7Fh
	cmp	bx,007Fh
	jbe	short start4
start2:	mov	dx,offset interr	; print error message
start3:	mov	ah,09h
	int	21h
	mov	ax,4C01h		; terminate program
	int	21h

start4:	mov	pktint,bx		; save packet driver interrupt number
	call	skipblank
	cmp	al,0Dh
	je	short start6		; packet type is optional
	call	gethex
	xchg	al,ah			; convert from host into network order
	mov	pkttype,ax
	call	skipblank		; nothing should follow anymore
	cmp	al,0Dh
	je	short start6
start5:	mov	dx,offset useerr	; print usage message
	jmp	short start3

start6:	push	es
	mov	ax,pktint
	mov	ah,35h			; determine packet driver vector
	int	21h
	mov	word ptr [oldpkt_vect + 0],bx
	mov	word ptr [oldpkt_vect + 2],es
	pop	es
	call	checkpkt		; check if there really is a packet drvr
	jnc	short start7
	mov	dx,offset pkterr	; print error message
	jmp	short start3

start7:	call	setvideo		; setup new video mode
	jnc	short start8
	mov	dx,offset viderr	; print error message
	jmp	short start3

start8:	mov	dx,offset cpymsg	; print copyright
	mov	ah,09h
	int	21h

; From this point on we are going to manipulate interrupt vectors, so
; there is no way back now.

	mov	dx,offset new_pkt
	mov	ax,pktint
	mov	ah,25h			; set new packet driver interrupt
	int	21h
	push	es
	mov	ax,352Fh		; determine 2Fh interrupt vector
	int	21h
	mov	word ptr [old2f_vect + 0],bx
	mov	word ptr [old2f_vect + 2],es
	pop	es
	mov	dx,offset new_int2f
	mov	ax,252Fh		; set new interrupt 2Fh vector
	int	21h

	mov	dx,offset start1
	mov	cl,4			; determine number of resident
	shr	dx,cl			; paragraphs
	inc	dx
	mov	ax,3100h		; terminate and stay resident
	int	21h
	ret

cpymsg	db	0Dh,0Ah
	db	'pktwatch - Packet driver logging utility',0Dh,0Ah
	db	'Copyright (C) G. Kuhlmann, 1996',0Dh,0Ah
	db	0Dh,0Ah,'$'

interr	db	'Illegal interrupt number',0Dh,0Ah,'$'
useerr	db	'usage: pktwatch <pkt-int-no> [<pkt type>]',0Dh,0Ah,'$'
pkterr	db	'No packet driver at given interrupt number',0Dh,0Ah,'$'
viderr	db	'Cannot switch video mode',0Dh,0Ah,'$'


;
;**************************************************************************
;
; Skip blanks on command line.
; Input:  DS:SI  -  pointer to command line
; Output: DS:SI  -  pointer to first non-blank
;         AL     -  first non-blank character
; Registers changed: AX, SI
;
skipblank	proc	near

	cld
skip1:	lodsb
	cmp	al,' '			; skip blanks
	je	short skip1
	cmp	al,09h			; skip tabs
	je	short skip1
	dec	si			; point SI to first non-blank
	ret

skipblank	endp


;
;**************************************************************************
;
; Return hex number from string pointed to by DS:SI.
; Input:  DS:SI  -  pointer to string
; Output: DS:SI  -  pointer to first character after hex number
;         BX     -  hex number
; Registers changed: AX, BX, CX, SI
;
gethex		proc	near

	xor	bx,bx
geth1:	lodsb
	cmp	al,'0'
	jb	short geth9		; check for valid hex digit
	cmp	al,'9'
	jbe	short geth3
	cmp	al,'A'
	jb	short geth9
	cmp	al,'a'
	jb	short geth2
	sub	al,20h			; convert character to upper case
geth2:	cmp	al,'F'
	ja	short geth9
	sub	al,7
geth3:	sub	al,'0'			; convert character to digit
	and	al,0Fh
	mov	cl,4
	shl	bx,cl			; include digit into result
	or	bl,al
	jmp	short geth1		; get next character

geth9:	dec	si			; point SI to last non-digit
	ret

gethex		endp


;
;**************************************************************************
;
; Check if there really is a packet driver at the given interrupt.
; Input:  none
; Output: Carry set, if no packet driver
; Registers changed: AX, CX, SI, DI
;
checkpkt	proc	near

	push	es
	les	di,oldpkt_vect
	mov	ax,es			; check if the interrupt vector
	or	ax,di			; points to somewhere
	jz	short chkp8
	add	di,3			; DI points to start of signature
	mov	si,offset pktsig
	mov	cx,8
	repe	cmpsb			; check for packet driver
	clc
	jz	short chkp9		; found packet driver
chkp8:	stc
chkp9:	pop	es
	ret

pktsig	db	'PKT DRVR'		; packet driver signature

checkpkt	endp


;
;**************************************************************************
;
; Setup video mode to 43x80 (or 50x80 on VGA). This is done by using the
; 8x8 character set.
; Input:  none
; Output: Carry flag set if error
; Registers changed: AX, BX, CX, DX, DI
;
setvideo	proc	near

	mov	ah,0Fh
	int	10h			; get current video mode
	cmp	al,7			; monochrome not supported
	je	short setv8
	cmp	al,3			; if it's mode 3 we don't have
	je	short setv1		; to change the mode
	mov	ax,0003h		; set video mode to 3
	int	10h

setv1:	mov	ah,12h			; determine current EGA configuration
	mov	bx,0110h		; this is used to check if we really
	int	10h			; have an EGA or VGA card
	or	bh,bh			; BH has to be set to 0 by BIOS
	jnz	short setv8

	push	es
	mov	ax,seg biosseg
	mov	es,ax
	assume	es:biosseg
	mov	al,b_lines
	xor	ah,ah
	mov	scnlines,ax		; save current screen size
	mov	bx,b_vidsize
	mov	vidbufsize,bx
	mov	bx,b_columns
	mov	scncolumns,bx
	inc	ax
	shl	ax,1
	mul	bx			; compute address of first character
	or	dx,dx			; after first half of screen
	jz	short setv2		; it should not be larger than 64kB
	pop	es
setv8:	stc				; return with error
	jmp	short setv9

setv2:	mov	vidoffset,ax		; save offset to second half
	mov	ax,1112h		; load 8x8 character set into
	xor	bl,bl			; block no. 0 and reset screen
	int	10h
	mov	al,b_lines
	xor	ah,ah			; restore original screen size
	xchg	scnlines,ax		; this will let the BIOS output
	mov	b_lines,al		; routines just access the upper 25
	inc	scnlines		; scnlines is number of lines - 1
	mov	ax,b_vidsize
	xchg	vidbufsize,ax
	mov	b_vidsize,ax
	mov	ax,b_columns
	mov	cx,ax			; save number of columns for later
	xchg	scncolumns,ax
	mov	b_columns,ax
	pop	es
	assume	es:nothing

	shl	cx,1			; compute number of bytes per line
	mov	scnbytes,cx
	mov	ax,scnlines
	mul	cx			; compute real size of video buffer
	or	dx,dx
	jz	short setv3
	mov	ax,0FFFFh		; if screen buffer too large, limit it
setv3:	mov	vidbufsize,ax		; to 64kB

	cld
	push	es
	shr	cx,1
	mov	ax,VIDEO_SEG
	mov	es,ax
	mov	di,vidoffset
	mov	ax,70C4h		; print horizontal line
	rep	stosw
	pop	es
	mov	vidoffset,di
	clc				; return without error
setv9:	ret

setvideo	endp

	end	start

