;**************************************************************************
;*
;* ANSIDRV - Display ANSI control sequences
;*
;* Module:  ansidrv.asm
;* Purpose: Display ANSI control sequences onto the console
;* Entries: start
;*
;**************************************************************************
;*
;* Copyright (C) 1995,1996 Gero Kuhlmann <gero@gkminix.han.de>
;*
;*  This program is free software; you can redistribute it and/or modify
;*  it under the terms of the GNU General Public License as published by
;*  the Free Software Foundation; either version 2 of the License, or
;*  any later version.
;*
;*  This program is distributed in the hope that it will be useful,
;*  but WITHOUT ANY WARRANTY; without even the implied warranty of
;*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;*  GNU General Public License for more details.
;*
;*  You should have received a copy of the GNU General Public License
;*  along with this program; if not, write to the Free Software
;*  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
;*
;**************************************************************************
;*

;
;**************************************************************************
;
; This program links its resident section into interrupt 29h and provides
; ANSI control sequence interpretation. It can be used optionally for the
; bootrom.
;
; ANSI control sequences supported by this program:
;
;                       Display attributes
;
;    Code          Effect
;    <esc>[0m      normal text
;    <esc>[1m      high-intensity on
;    <esc>[21m     high-intensity off
;    <esc>[5m      blinking on
;    <esc>[25m     blinking off
;    <esc>[7m      reverse video on
;    <esc>[27m     reverse video off
;    <esc>[3xm     set foreground color:
;    <esc>[4xm     set background color. x can be:
;
;                       0 - black  4 - blue
;                       1 - red    5 - magenta
;                       2 - green  6 - cyan
;                       3 - yellow 7 - white
;
;
;                         Cursor control
;
;    Code          Effect
;    <esc>[r;cH    move cursor to row r col c (r and c are both numbers)
;    <esc>[rA      move cursor up r rows
;    <esc>[rB      move cursor down r rows
;    <esc>[cC      move cursor right c columns
;    <esc>[cD      move cursor left c columns
;    <esc>[?7l     turn off line wrap
;    <esc>[?7h     turn on line wrap
;    <esc>[J       clear screen and home cursor
;    <esc>[s       save the cursor position
;    <esc>[u       return cursor to saved position
;
;
.model	small,pascal


;
;**************************************************************************
;
; Miscellaneous defines:
;
DEACTIVATE	equ	0CF45h		; magic key for deactivation

ARGMAX		equ	2		; maximum number of arguments
MAXLINES	equ	25		; default number of screen lines

CHR_BELL	equ	07h		; bell character
CHR_BS		equ	08h		; backspace character
CHR_TAB		equ	09h		; tab character
CHR_LF		equ	0Ah		; linefeed character
CHR_CR		equ	0Dh		; carriage return character
CHR_ESC		equ	1Bh		; escape character
CHR_BRACKET	equ	'['		; left bracket
CHR_COLON	equ	';'		; colon character

FLAG_NOWRAP	equ	00000001b	; flag indicating not to wrap
FLAG_INVERS	equ	00000010b	; flag for invers video
FLAG_BLINK	equ	00000100b	; flag for blinking
FLAG_INTENSE	equ	00001000b	; flag for high intensity
FLAG_NORMAL	equ	11110001b	; mask to set flags to normal mode

NORMAL_STATE	equ	0		; normal display state
ESC_STATE	equ	1		; found escape in input stream
BRACKET_STATE	equ	2		; found bracket in input stream
NUMBER_STATE	equ	3		; found numbers in input stream


;
;
;**************************************************************************
;
; There is no other than a code segment. All data will be placed in it
; as well. The resident section of the program starts here.
;
.code

	org	0100h			; we have to start at 0100h

	assume	cs:_text
	assume	ds:_text		; code and data are in the same segment
	assume	es:nothing
	assume	ss:nothing

start:	jmp	start1


;
;**************************************************************************
;
; Data segment:
;
scnmode		db	0		; current screen mode
scnpage		db	0		; current screen page
scnattr		db	07h		; attributes for characters to print
scncolumns	db	0		; number of screen columns
curpos		dw	0		; current cursor position
savedpos	dw	0		; saved cursor position
flags		db	0		; display flags
fg_color	db	07h		; foreground color
bg_color	db	0		; background color
dispstate	db	NORMAL_STATE	; current display state
argbuf		db	ARGMAX dup (0)	; argument number buffer
argnum		db	0		; current argument number

old29_vect	dd	0		; old 29h interrupt vector
old2f_vect	dd	0		; old 2Fh interrupt vector


;
;**************************************************************************
;
; Interrupt routine to terminate ansidrv services. It will just restore the
; old interrupt vectors. Note that the interrupt vectors have to be restored
; first before calling the old interrupt handler, so that the deactivation
; routines in other such programs don't get confused.
; Input:  AX  -  Magic key 0CF45h
; Output: none
; Registers used: none
;
new_int2f	proc	far

	assume	ds:nothing
	assume	es:nothing

	cmp	ax,DEACTIVATE		; check if we have to deactivate
	je	short int2f1
int2f9:	jmp	old2f_vect

int2f1:	push	ds
	push	ax
	xor	ax,ax			; restore all old interrupt vectors
	mov	ds,ax
	cli
	mov	ax,word ptr [old29_vect + 0]
	mov	ds:[29h * 4 + 0],ax
	mov	ax,word ptr [old29_vect + 2]
	mov	ds:[29h * 4 + 2],ax
	mov	ax,word ptr [old2f_vect + 0]
	mov	ds:[2Fh * 4 + 0],ax
	mov	ax,word ptr [old2f_vect + 2]
	mov	ds:[2Fh * 4 + 2],ax
	pop	ax
	pop	ds
	jmp	short int2f9

new_int2f	endp


;
;**************************************************************************
;
; Command table
;
cmdtab	label	byte

; Define commands for uppercase letters

	irpc	cmdletter,<ABCDHJ>

	db	'&cmdletter'			;; put letter into table
	dw	offset _text:cmd_u&cmdletter	;; put offset to cmd routine

	endm

; Define commands for lowercase letters

	irpc	cmdletter,<hlmsu>

	db	'&cmdletter'			;; put letter into table
	dw	offset _text:cmd_l&cmdletter	;; put offset to cmd routine

	endm

; Mark end of table

	db	0

TABSIZE	equ	3				; size of each table entry


;
;**************************************************************************
;
; Interrupt routine to display a character. It also interprets the ANSI
; control sequences mentioned above.
; Input:  AL  -  character to display
; Output: none
; Registers used: none
;
new_int29	proc	far

	assume	ds:nothing
	assume	es:nothing

	sti
	push	ax
	push	bx
	push	cx
	push	dx
	push	ds
	mov	bx,cs
	mov	ds,bx
	assume	ds:_text

; Determine the current cursor position.

	push	ax
	mov	ah,03h
	mov	bh,scnpage
	int	10h
	mov	curpos,dx
	pop	ax

; State switcher. Depending on the character received with this interrupt
; routine the state will be switched and the corresponding action executed.

	mov	ah,dispstate
	cmp	ah,NORMAL_STATE		; first handle normal state
	jne	short int292

	cmp	al,CHR_ESC		; is it an escape character?
	jne	short int291
	mov	dispstate,ESC_STATE	; yes, just switch the state and
	jmp	short int29F		; terminate
int291:	call	putchar			; no, we have to print the character
	jmp	short int293

int292:	cmp	ah,ESC_STATE
	jne	short int294		; are we in escape state?
	cmp	al,CHR_BRACKET
	jne	short int293		; a bracket has to follow
	mov	dispstate,BRACKET_STATE	; simply switch the state and terminate
	jmp	short int29F
int293:	mov	dispstate,NORMAL_STATE	; if no bracket, this is an error, so
int29F:	jmp	short int299		; switch back to normal mode

int294:	cmp	ah,BRACKET_STATE	; check for bracket state
	jne	short int295
	cmp	al,'?'			; skip any question marks
	je	short int29F
	cmp	al,'='			; skip any equal signs
	je	short int29F
	mov	dispstate,NUMBER_STATE	; continue with number state
	mov	argnum,0
if ARGMAX eq 2
	mov	word ptr argbuf,0	; zero out argument buffer
else
	mov	bx,ARGMAX
@@:	mov	argbuf[bx],0		; zero out argument buffer
	dec	bx
	jnz	short @B
endif
	jmp	short int296

int295:	cmp	ah,NUMBER_STATE		; if we are not in number state it's an
	jne	short int293		; error, as we don't have anymore states
int296:	cmp	al,CHR_COLON
	jne	short int297		; a semicolon sign advances the argument
	mov	al,argnum		; counter
	inc	al
	cmp	al,ARGMAX		; no more than ARGMAX arguments allowed,
	jae	short int299		; simply skip the rest
	mov	argnum,al
	jmp	short int299

int297:	cmp	al,'0'			; check if we got a number here
	jb	short int298
	cmp	al,'9'
	ja	short int298
	sub	al,'0'			; convert character into number
	mov	cl,al
	mov	bl,argnum
	xor	bh,bh
	mov	al,argbuf[bx]		; multiply the number in the buffer
	mov	ah,10			; by 10
	mul	ah
	add	al,cl			; and add the new value
	mov	argbuf[bx],al		; set new value in argument buffer
	jmp	short int299		; that's it

; Now execute the command associated with the last character of the escape
; sequence.

int298:	xor	bx,bx
int29A:	mov	ah,cmdtab[bx]
	or	ah,ah			; end of table reached
	je	short int29C
	cmp	al,ah			; found character
	je	short int29B
	add	bx,TABSIZE		; continue with next table entry
	jmp	short int29A

int29B:	mov	ax,word ptr cmdtab[bx+1]
	mov	dx,curpos
	call	ax			; call command handling routine
int29C:	mov	dispstate,NORMAL_STATE	; restore state counter

; Terminate the interrupt routine by restoring all registers

int299:	pop	ds
	pop	dx
	pop	cx
	pop	bx
	pop	ax
	iret

new_int29	endp


;
;**************************************************************************
;
; Actually display a character onto the console
; Input:  AL  -  character to display
;         DX  -  current cursor position
; Output: none
; Registers used: AX, BX, CX, DX
;
putchar		proc	near

	mov	bh,scnpage		; used throughout this routine

; The bell character is best handled by the BIOS

	cmp	al,CHR_BELL
	jne	short putch2
putch1:	mov	ah,0Eh			; print special character using the
	int	10h			; BIOS
	jmp	short putch9

; TAB's are not handled by the BIOS teletype driver, so we have to do it
; ourselves.

putch2:	cmp	al,CHR_TAB		; check for tab character
	jne	short putch4
	mov	cl,dl			; compute number of spaces to print
	or	cl,0F8h
	neg	cl
	xor	ch,ch
putch3:	push	cx
	mov	al,' '			; print spaces
	call	putchar			; recursive call
	pop	cx
	loop	short putch3
	jmp	short putch9

; Carriage return simply involves setting the cursor position to zero

putch4:	cmp	al,CHR_CR		; check for carriage return
	jne	short putch5
	xor	dl,dl			; move cursor into column zero
	jmp	short putcursor

; Linefeed involves incrementing the line number and scrolling the screen if
; necessary

putch5:	cmp	al,CHR_LF		; check for linefeed
	je	short putch8		; the linefeed code is below

; Backspace simply involves decrementing the current cursor position

	cmp	al,CHR_BS		; check for backspace
	jne	short putch6
	or	dl,dl			; don't decrement below zero
	jz	short putcursor
	dec	dl			; decrement column number
	jmp	short putcursor

; All other characters are printed at the cursor position

putch6:	mov	ah,09h
	mov	bl,scnattr		; print the character using BIOS
	mov	cx,1
	int	10h

	inc	dl			; increment cursor position
	cmp	dl,scncolumns		; check if at right end of line
	jb	short putcursor
	dec	dl			; restore cursor to rightmost column
	test	flags,FLAG_NOWRAP	; check if we should wrap at end of line
	jnz	short putcursor
	xor	dl,dl			; put cursor into column 0
putch8:	inc	dh
	cmp	dh,MAXLINES		; check if at bottom of screen
	jb	short putcursor
	dec	dh			; restore cursor to lowest screen line
	push	dx
	mov	ax,0601h
	xor	cx,cx			; define upper left corner
	mov	dl,scncolumns		; define lower right corner
	dec	dl
	mov	bh,scnattr
	int	10h			; use the BIOS to scroll the screen
	pop	dx

; Put cursor to position inidicated by the curpos variable

putcursor	label	near

	mov	curpos,dx		; save new cursor position
	mov	ah,02h
	mov	bh,scnpage		; set cursor using BIOS
	int	10h
putch9:	ret

putchar		endp


;
;**************************************************************************
;
; Handle escape sequence 'A': move cursor up
; Input:  DX  -  current cursor position
; Output: none
; Registers used: AX, BX, CX, DX
;
cmd_uA		proc	near

	sub	dh,argbuf[0]		; decrement line number
	jae	short putcursor		; set new cursor position
	ret

cmd_uA		endp


;
;**************************************************************************
;
; Handle escape sequence 'B': move cursor down
; Input:  DX  -  current cursor position
; Output: none
; Registers used: AX, BX, CX, DX
;
cmd_uB		proc	near

	add	dh,argbuf[0]		; increment line number
	cmp	dh,MAXLINES
	jb	short putcursor		; set new cursor position
	ret

cmd_uB		endp


;
;**************************************************************************
;
; Handle escape sequence 'C': move cursor right
; Input:  DX  -  current cursor position
; Output: none
; Registers used: AX, BX, CX, DX
;
cmd_uC		proc	near

	add	dl,argbuf[0]		; increment column number
	cmp	dl,scncolumns
	jb	short putcursor		; set new cursor position
	ret

cmd_uC		endp


;
;**************************************************************************
;
; Handle escape sequence 'D': move cursor left
; Input:  DX  -  current cursor position
; Output: none
; Registers used: AX, BX, CX, DX
;
cmd_uD		proc	near

	sub	dh,argbuf[0]		; decrement column number
	jae	short putcursor		; set new cursor position
	ret

cmd_uD		endp


;
;**************************************************************************
;
; Handle escape sequence 'H': move cursor to specified position
; Input:  none
; Output: none
; Registers used: AX, BX, CX, DX
;
cmd_uH		proc	near

	mov	dx,word ptr argbuf[0]
	xchg	dh,dl			; get new cursor position
	jmp	short putcursor

cmd_uH		endp


;
;**************************************************************************
;
; Handle escape sequence 'J': clear screen and move cursor into upper
; left corner of the screen
; Input:  none
; Output: none
; Registers used: AX, BX, CX, DX
;
cmd_uJ		proc	near

	mov	ax,0600h
	mov	bh,scnattr
	xor	cx,cx
	mov	dh,MAXLINES-1		; use the BIOS scrolling function
	mov	dl,scncolumns		; to clear the screen
	dec	dl
	int	10h
	xor	dx,dx			; put cursor into upper left corner
	jmp	short putcursor

cmd_uJ		endp


;
;**************************************************************************
;
; Handle escape sequence 'h': set display option, currently only option 7
; is supported, which turns line wrapping on.
; Input:  none
; Output: none
; Registers used: AX, BX, CX, DX
;
cmd_lh		proc	near

	mov	al,argbuf[0]
	cmp	al,7			; check for option 7
	jne	short cmdlh9
	and	flags,not FLAG_NOWRAP	; turn wrapping off
cmdlh9:	ret

cmd_lh		endp


;
;**************************************************************************
;
; Handle escape sequence 'l': reset display option, currently only option 7
; is supported, which turns line wrapping off.
; Input:  none
; Output: none
; Registers used: AX, BX, CX, DX
;
cmd_ll		proc	near

	mov	al,argbuf[0]
	cmp	al,7			; check for option 7
	jne	short cmdll9
	or	flags,FLAG_NOWRAP	; turn wrapping off
cmdll9:	ret

cmd_ll		endp


;
;**************************************************************************
;
; Handle escape sequence 'm': set screen attributes according to the
; following table:
;
;    <esc>[0m      normal text
;    <esc>[1m      high-intensity on
;    <esc>[21m     high-intensity off
;    <esc>[5m      blinking on
;    <esc>[25m     blinking off
;    <esc>[7m      reverse video on
;    <esc>[27m     reverse video off
;    <esc>[3xm     set foreground color
;    <esc>[4xm     set background color
;
; Input:  none
; Output: none
; Registers used: AX, BX, CX, DX
;
cmd_lm		proc	near

; First set the global variables according to the first parameter of the
; escape sequence

	mov	ah,flags
	mov	al,argbuf[0]
	or	al,al
	jnz	short cmdlm1
	and	ah,FLAG_NORMAL		; set normal mode
	jmp	short cmdlm9

cmdlm1:	mov	cl,al
	cmp	cl,20			; check for reset command
	jb	short cmdlm2
	sub	cl,20
cmdlm2:	cmp	cl,1
	jne	short cmdlm3
	mov	ch,FLAG_INTENSE		; set high-intensity mode
	jmp	short cmdlm5
cmdlm3:	cmp	cl,5
	jne	short cmdlm4
	mov	ch,FLAG_BLINK		; set blinking mode
	jmp	short cmdlm5
cmdlm4:	cmp	cl,7
	jne	short cmdlm7
	mov	ch,FLAG_INVERS		; set inverse mode
cmdlm5:	cmp	al,20
	jae	short cmdlm6
	or	ah,ch			; set the flag
	jmp	short cmdlm9
cmdlm6:	not	ch
	and	ah,ch			; reset the flag
	jmp	short cmdlm9

cmdlm7:	sub	al,30
	jb	short cmdlmF		; check for foreground color
	cmp	al,7
	ja	short cmdlm8
	mov	fg_color,al
	jmp	short cmdlm9

cmdlm8:	sub	al,10
	jb	short cmdlmF		; check for background color
	cmp	al,7
	ja	short cmdlmF
	mov	bg_color,al

; Now set the actual attribute value according to the flags set previously

cmdlm9:	mov	flags,ah
	xor	ch,ch
	test	ah,FLAG_BLINK
	jz	cmdlmA
	or	ch,10000000b		; set blink attribute
cmdlmA:	test	ah,FLAG_INTENSE
	jz	cmdlmB
	or	ch,00001000b		; set high-intensity attribute
cmdlmB:	mov	cl,4
	mov	bl,fg_color
	mov	bh,bg_color		; set colors
	test	ah,FLAG_INVERS
	jz	cmdlmC
	shl	bl,cl			; shift foreground color into background
	jmp	short cmdlmD		; for inverse display
cmdlmC:	shl	bh,cl			; shift background color
cmdlmD:	or	bl,bh
	and	bl,01110111b		; compute color attribute value
	or	ch,bl			; and merge it into the final value
	mov	scnattr,ch		; save final screen attributes
cmdlmF:	ret

cmd_lm		endp


;
;**************************************************************************
;
; Handle escape sequence 's': save current cursor position
; Input:  none
; Output: none
; Registers used: AX, BX, CX, DX
;
cmd_ls		proc	near

	mov	ax,curpos
	mov	savedpos,ax		; save the current cursor position
	ret

cmd_ls		endp


;
;**************************************************************************
;
; Handle escape sequence 'u': restore cursor position
; Input:  none
; Output: none
; Registers used: AX, BX, CX, DX
;
cmd_lu		proc	near

	mov	dx,savedpos
	jmp	putcursor		; place cursor to new position

cmd_lu		endp


;
;**************************************************************************
;
; Start the non-resident section here:
;
start1:	mov	ax,cs
	mov	ds,ax
	assume	ds:_text
	assume	es:nothing

	mov	dx,offset cpymsg	; print copyright
	mov	ah,09h
	int	21h

	mov	ah,0Fh			; determine current screen mode
	int	10h
	mov	scncolumns,ah		; set number of columns on screen
	mov	scnpage,bh		; set screen page
	mov	al,bh
	mov	ah,05h			; select screen page
	int	10h

ifdef ANSITEST
	mov	ax,3560h
else
	mov	ax,3529h		; determine interrupt 29h vector
endif
	int	21h
	mov	word ptr [old29_vect + 0],bx
	mov	word ptr [old29_vect + 2],es
	mov	dx,offset new_int29
ifdef ANSITEST
	mov	ax,2560h
else
	mov	ax,2529h		; set new interrupt 29h interrupt
endif
	int	21h
ifndef ANSITEST
	mov	ax,352Fh		; determine 2Fh interrupt vector
	int	21h
	mov	word ptr [old2f_vect + 0],bx
	mov	word ptr [old2f_vect + 2],es
	mov	dx,offset new_int2f
	mov	ax,252Fh		; set new interrupt 2Fh vector
	int	21h
endif

	mov	dx,offset start1
	mov	cl,4			; determine number of resident
	shr	dx,cl			; paragraphs
	inc	dx
	mov	ax,3100h		; terminate and stay resident
	int	21h
	ret

cpymsg	db	0Dh,0Ah
	db	'ansidrv - ANSI display driver',0Dh,0Ah
	db	'Copyright (C) G. Kuhlmann, 1996',0Dh,0Ah
	db	0Dh,0Ah,'$'

	end	start

