;;;
;;; mu-cite.el --- yet another citation tool for GNU Emacs
;;;
;;; Copyright (C) 1995 Free Software Foundation, Inc.
;;; Copyright (C) 1995,1996 MORIOKA Tomohiko
;;; Copyright (C) 1995 MINOURA Makoto
;;;
;;; Author: MORIOKA Tomohiko <morioka@jaist.ac.jp>
;;;         MINOURA Makoto <minoura@@leo.bekkoame.or.jp>
;;; modified by KOBAYASHI Shuhei <shuhei@cmpt01.phys.tohoku.ac.jp>
;;; Version: $Revision: 7.11 $
;;; Keywords: mail, news, citation
;;;
;;; This file is part of tl (Tiny Library).
;;;
;;; This program is free software; you can redistribute it and/or
;;; modify it under the terms of the GNU General Public License as
;;; published by the Free Software Foundation; either version 2, or
;;; (at your option) any later version.
;;;
;;; This program is distributed in the hope that it will be useful,
;;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
;;; General Public License for more details.
;;;
;;; You should have received a copy of the GNU General Public License
;;; along with This program.  If not, write to the Free Software
;;; Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.
;;;
;;; Commentary:
;;;
;;; - How to install.
;;;   1. bytecompile this file and copy it to the apropriate directory.
;;;   2. put the following lines to your ~/.emacs:
;;;      for Emacs 19 or later
;;;		(autoload 'mu-cite/cite-original "mu-cite" nil t)
;;;		(add-hook 'mail-citation-hook 'mu-cite/cite-original)
;;;      for Emacs 18
;;;		;; for all but mh-e
;;;		(add-hook 'mail-yank-hooks (function mu-cite/cite-original))
;;;		;; for mh-e only
;;;		(add-hook 'mh-yank-hooks (function mu-cite/cite-original))
;;;
;;; Code:

(require 'tl-822)
(require 'tl-str)
(require 'tl-list)


;;; @ version
;;;

(defconst mu-cite/RCS-ID
  "$Id: mu-cite.el,v 7.11 1996/01/31 00:07:57 morioka Exp $")
(defconst mu-cite/version (get-version-string mu-cite/RCS-ID))


;;; @ formats
;;;

(defvar cited-prefix-regexp "^[^ \t>]*[>|]+[ \t#]*")
(defvar mu-cite/cited-prefix-regexp "\\(^[^ \t\n>]+>+[ \t]*\\|^[ \t]*$\\)")

(defvar mu-cite/prefix-format '(prefix-register-verbose "> ")
  "*List to represent citation prefix.
Each elements must be string or method name.")
(defvar mu-cite/top-format '(in-id
			     ">>>>>	" from " wrote:\n")
  "*List to represent top string of citation.
Each elements must be string or method name.")


;;; @ hooks
;;;

(defvar mu-cite/pre-cite-hook nil
  "*List of functions called before citing a region of text.")
(defvar mu-cite/post-cite-hook nil
  "*List of functions called after citing a region of text.")


;;; @ field
;;;

(defvar mu-cite/get-field-value-methoda-list
  (list (cons 'mh-letter-mode
	      (function
	       (lambda (name)
		 (if (and (stringp mh-sent-from-folder)
			  (numberp mh-sent-from-msg))
		     (save-excursion
		       (set-buffer mh-sent-from-folder)
		       (set-buffer mh-show-buffer)
		       (and (boundp 'mime::preview/article-buffer)
			    (bufferp mime::preview/article-buffer)
			    (set-buffer mime::preview/article-buffer))
		       (rfc822/get-field-body name)
		       ))
		 )))))

(defun mu-cite/get-field-value (name)
  (or (rfc822/get-field-body name)
      (let ((method (assq major-mode mu-cite/get-field-value-methoda-list)))
	(if method
	    (funcall (cdr method) name)
	  ))))


;;; @ prefix registration
;;;

(defvar mu-cite/registration-file
  (expand-file-name "~/.mu-cite.el")
  "*The name of the user environment file for mu-cite.")

(defvar mu-cite/registration-symbol 'mu-cite/citation-name-alist)

(defvar mu-cite/citation-name-alist nil)
(load mu-cite/registration-file t t t)
(or (eq 'mu-cite/citation-name-alist mu-cite/registration-symbol)
    (setq mu-cite/citation-name-alist
	  (symbol-value mu-cite/registration-symbol))
    )
(defvar mu-cite/minibuffer-history nil)

;; get citation-name from the database
(defmacro mu-cite/get-citation-name (from)
  (` (cdr (assoc (, from) mu-cite/citation-name-alist))))

;; register citation-name to the database
(defun mu-cite/add-citation-name (name from)
  (let* ((elt (assoc from mu-cite/citation-name-alist)))
    (if elt
	(setq mu-cite/citation-name-alist
	      (delq elt mu-cite/citation-name-alist)))
    (setq elt (cons from name))
    (setq mu-cite/citation-name-alist
	  (cons elt
		mu-cite/citation-name-alist))
    (mu-cite/save-to-file)
    ))

;; save to file
(defun mu-cite/save-to-file ()
  (let* ((filename mu-cite/registration-file)
	 (buffer (get-buffer-create " *mu-register*")))
    (save-excursion
      (set-buffer buffer)
      (setq buffer-file-name filename)
      (erase-buffer)
      (insert (format ";; generated automatically by mu-cite %s.\n"
		      mu-cite/version))
      (insert (format "(setq %s
 '(" mu-cite/registration-symbol))
      (insert (mapconcat
	       (function prin1-to-string)
	       mu-cite/citation-name-alist "\n   "))
      (insert "\n   ))\n")
      (save-buffer))
    (kill-buffer buffer)))


;;; @ item methods
;;;

;;; @@ ML count
;;;

(defvar mu-cite/ml-count-field-list
  '("X-Ml-Count" "X-Mail-Count" "X-Seqno" "X-Sequence" "Mailinglist-Id"))

(defun mu-cite/get-ml-count-method ()
  (let ((field-list mu-cite/ml-count-field-list))
    (catch 'tag
      (while field-list
        (let* ((field (car field-list))
               (ml-count (mu-cite/get-field-value field)))
          (if (and ml-count (string-match "[0-9]+" ml-count))
              (throw 'tag
                     (substring ml-count
                                (match-beginning 0)(match-end 0))
                     ))
          (setq field-list (cdr field-list))
          )))))


;;; @@ prefix and registration
;;;

(defun mu-cite/get-prefix-method ()
  (or (mu-cite/get-citation-name (mu-cite/get-value 'address))
      ">")
  )

(defun mu-cite/get-prefix-register-method ()
  (let ((addr (mu-cite/get-value 'address)))
    (or (mu-cite/get-citation-name addr)
	(let ((return
	       (tl:read-string "Citation name? "
			       (mu-cite/get-value 'full-name)
			       'mu-cite/minibuffer-history)
	       ))
	  (if (y-or-n-p (format "Register \"%s\"? " return))
	      (mu-cite/add-citation-name return addr)
	    )
	  return))))

(defun mu-cite/get-prefix-register-verbose-method ()
  (let* ((addr (mu-cite/get-value 'address))
	 (return1 (or (mu-cite/get-citation-name addr)
		      (mu-cite/get-value 'full-name)))
	 return)
    (setq return
	  (tl:read-string "Citation name? "
			  return1
			  'mu-cite/minibuffer-history))
    (if (and (not (string-equal return return1))
	     (y-or-n-p (format "Register \"%s\"? " return))
	     )
	(mu-cite/add-citation-name return addr)
      )
    return))


;;; @@ set up
;;;

(defvar mu-cite/default-methods-alist
  (list (cons 'from
	      (function
	       (lambda ()
		 (mu-cite/get-field-value "From")
		 )))
	(cons 'date
	      (function
	       (lambda ()
		 (mu-cite/get-field-value "Date")
		 )))
	(cons 'message-id
	      (function
	       (lambda ()
		 (mu-cite/get-field-value "Message-Id")
		 )))
	(cons 'subject
	      (function
	       (lambda ()
		 (mu-cite/get-field-value "Subject")
		 )))
	(cons 'ml-name
	      (function
	       (lambda ()
		 (mu-cite/get-field-value "X-Ml-Name")
		 )))
	(cons 'ml-count (function mu-cite/get-ml-count-method))
	(cons 'address-structure
	      (function
	       (lambda ()
		 (car (rfc822/parse-address
		       (rfc822/lexical-analyze
			(mu-cite/get-value 'from)
			)))
		 )))
	(cons 'full-name
	      (function
	       (lambda ()
		 (rfc822/full-name-string
		  (mu-cite/get-value 'address-structure))
		 )))
	(cons 'address
	      (function
	       (lambda ()
		 (rfc822/address-string
		  (mu-cite/get-value 'address-structure))
		 )))
	(cons 'id
	      (function
	       (lambda ()
		 (let ((ml-name (mu-cite/get-value 'ml-name)))
		   (if ml-name
		       (concat "["
			       ml-name
			       " : No."
			       (mu-cite/get-value 'ml-count)
			       "]")
		     (mu-cite/get-value 'message-id)
		     )))))
	(cons 'in-id
	      (function
	       (lambda ()
		 (let ((id (mu-cite/get-value 'id)))
		   (if id
		       (format ">>>>> In %s \n" id)
		     "")))))
	(cons 'prefix (function mu-cite/get-prefix-method))
	(cons 'prefix-register
	      (function mu-cite/get-prefix-register-method))
	(cons 'prefix-register-verbose
	      (function mu-cite/get-prefix-register-verbose-method))
	))


;;; @ fundamentals
;;;

(defvar mu-cite/methods-alist nil)

(defun mu-cite/make-methods ()
  (setq mu-cite/methods-alist
	(copy-alist mu-cite/default-methods-alist))
  )

(defun mu-cite/get-value (item)
  (let ((ret (assoc-value item mu-cite/methods-alist)))
    (if (or (byte-code-function-p ret)
	    (and (symbolp ret)(fboundp ret))
	    (and (consp ret)(eq (car ret) 'lambda))
	    )
	(prog1
	    (setq ret (funcall ret))
	  (set-alist 'mu-cite/methods-alist item ret)
	  )
      ret)))

(defun mu-cite/eval-format (list)
  (mapconcat (function
	      (lambda (elt)
		(cond ((stringp elt) elt)
		      ((symbolp elt) (mu-cite/get-value elt))
		      )))
	     list "")
  )


;;; @ main function
;;;

(defun mu-cite/cite-original ()
  "Citing filter function.
This is callable from the various mail and news readers' reply
function according to the agreed upon standard."
  (interactive)
  (mu-cite/make-methods)
  (save-restriction
    (narrow-to-region (point)(point-max))
    (run-hooks 'mu-cite/pre-cite-hook)
    (let ((last-point (point))
	  (top (mu-cite/eval-format mu-cite/top-format))
	  (prefix (mu-cite/eval-format mu-cite/prefix-format))
	  )
      (if (re-search-forward "^$\\|^-+$" nil nil)
	  (forward-line 1)
	)
      (widen)
      (delete-region last-point (point))
      (insert top)
      (setq last-point (point))
      (while (< (point)(mark))
	(or (looking-at mu-cite/cited-prefix-regexp)
	    (insert prefix))
	(forward-line 1))
      (goto-char last-point)
      )
    (run-hooks 'mu-cite/post-cite-hook)
    ))


;;; @ message editing utilities
;;;

(defun fill-cited-region (beg end)
  (interactive "*r")
  (save-excursion
    (save-restriction
      (goto-char end)
      (while (not (eolp))
	(backward-char)
	)
      (setq end (point))
      (narrow-to-region beg end)
      (goto-char (point-min))
      (let* ((fill-prefix
	      (and (re-search-forward cited-prefix-regexp nil t)
		   (or (re-search-forward cited-prefix-regexp nil t)
		       t)
		   (buffer-substring (match-beginning 0)
				     (match-end 0)
				     )))
	     (pat (concat "\n" fill-prefix))
	     )
	(goto-char (point-min))
	(while (search-forward pat nil t)
	  (if (and (> (match-beginning 0) (point-min))
		   (member (char-category
			    (char-before (match-beginning 0)))
			   '("a" "l"))
		   )
	      (replace-match " ")
	    (replace-match "")
	    )
	  )
	(goto-char (point-min))
	(fill-region (point-min) (point-max))
	))))

(defun replace-top-string (&optional old new)
  (interactive)
  (if (null old)
      (setq old (read-string "old string is ? "))
    )
  (if (null new)
      (setq new (read-string "new string is ? "))
    )
  (while (re-search-forward (concat "^" (regexp-quote old)) nil t)
    (replace-match new)
    ))


;;; @ end
;;;

(provide 'mu-cite)

(run-hooks 'mu-cite-load-hook)

;;; mu-cite.el ends here
