/* -*- Mode: C; tab-width: 2; indent-tabs-mode: nil; c-basic-offset: 2 -*- */
/* vim:set sw=2 sts=2 et cin: */
/*
 * This file is part of the MUSE Instrument Pipeline
 * Copyright (C) 2005-2014 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 */

#ifndef MUSE_PIXGRID_H
#define MUSE_PIXGRID_H

/*----------------------------------------------------------------------------*
 *                              Includes                                      *
 *----------------------------------------------------------------------------*/
#include "muse_pixtable.h"

/*----------------------------------------------------------------------------*
 *                              Defines                                       *
 *----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*
 *                          Special variable types                            *
 *----------------------------------------------------------------------------*/

/** @addtogroup muse_pixgrid */
/**@{*/

/*----------------------------------------------------------------------------*/
/**
  @brief   The pixel extension map.
 */
/*----------------------------------------------------------------------------*/
typedef struct {
  int npix;  /* number of pixels in this grid point */
  cpl_size *pix; /* the row number(s) in the pixel table */
} muse_pixels_ext;

/*----------------------------------------------------------------------------*/
/**
  @brief   The pixel grid.
 */
/*----------------------------------------------------------------------------*/
typedef struct {
  cpl_size *pix; /* The pixel grid array, elements can be       *
                  * 0:        empty                             *
                  * positive: row_number in the pixel table     *
                  * negative: -(i_ext+1) in the extension array */
  cpl_size size_x; /* horizontal spatial size */
  cpl_size size_y; /* vertical spatial size */
  cpl_size size_z; /* size in dispersion direction */
  cpl_size n_ext; /* number of filled pixels in the extension map */
  cpl_size n_alloc; /* number of allocated pixels in the extension map */
  muse_pixels_ext *ext; /* the extension map */
} muse_pixgrid;

/*----------------------------------------------------------------------------*
 *                           Inline functions                                 *
 *----------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------*/
/**
  @brief   Get the grid index determined from all three coordinates.
  @param   aPixels         Pointer to pixgrid.
  @param   aX              x index
  @param   aY              y index
  @param   aZ              z index
  @param   aAllowOutside   if true, return a positive value for pixels nominally
                           outside the grid
  @return  Index to be used in the other pixgrid functions, or -1 on error.

  @note This function is defined as static inline, i.e. should get integrated
        into the caller by the compiler, to maximise execution speed.
*/
/*---------------------------------------------------------------------------*/
static inline cpl_size
muse_pixgrid_get_index(muse_pixgrid *aPixels, cpl_size aX, cpl_size aY,
                       cpl_size aZ, cpl_boolean aAllowOutside)
{
  if (!aAllowOutside &&
      (aX < 0 || aX >= aPixels->size_x || aY < 0 || aY >= aPixels->size_y ||
       aZ < 0 || aZ >= aPixels->size_z)) {
    return -1;
  }
  if (aX < 0) {
    aX = 0;
  }
  if (aX >= aPixels->size_x) {
    aX = aPixels->size_x - 1;
  }
  if (aY < 0) {
    aY = 0;
  }
  if (aY >= aPixels->size_y) {
    aY = aPixels->size_y - 1;
  }
  if (aZ < 0) {
    aZ = 0;
  }
  if (aZ >= aPixels->size_z) {
    aZ = aPixels->size_z - 1;
  }
  return aX + aPixels->size_x * (aY + aPixels->size_y * aZ);
} /* muse_pixgrid_get_index() */

/*---------------------------------------------------------------------------*/
/**
  @brief   Return the number of rows stored in one pixel.
  @param   aPixels   Pointer to pixgrid.
  @param   aIndex    Pixel index, as computed by muse_pixgrid_get_index().

  @note This function is defined as static inline, i.e. should get integrated
        into the caller by the compiler, to maximise execution speed.
*/
/*---------------------------------------------------------------------------*/
static inline cpl_size
muse_pixgrid_get_count(muse_pixgrid *aPixels, cpl_size aIndex)
{
  if (aIndex < 0) {
    return 0;
  }
  cpl_size p = aPixels->pix[aIndex];
  return (p == 0) ? 0
                  : (p > 0) ? 1 : aPixels->ext[-p-1].npix;
} /* muse_pixgrid_get_count() */

/*---------------------------------------------------------------------------*/
/**
  @brief   Return a pointer to the rows stored in one pixel.
  @param   aPixels   Pointer to pixgrid.
  @param   aIndex    Pixel index, as computed by muse_pixgrid_get_index().

  @note This function is defined as static inline, i.e. should get integrated
        into the caller by the compiler, to maximise execution speed.
*/
/*---------------------------------------------------------------------------*/
static inline const cpl_size *
muse_pixgrid_get_rows(muse_pixgrid *aPixels, cpl_size aIndex)
{
  if (aIndex < 0) {
    return 0;
  }
  cpl_size p = aPixels->pix[aIndex];
  return (p == 0) ? NULL
                  : (p > 0) ? aPixels->pix + aIndex : aPixels->ext[-p-1].pix;
} /* muse_pixgrid_get_rows() */

/**@}*/

/*----------------------------------------------------------------------------*
 *                           Function prototypes                              *
 *----------------------------------------------------------------------------*/
muse_pixgrid *muse_pixgrid_create(muse_pixtable *, cpl_propertylist *, cpl_size, cpl_size, cpl_size);
muse_pixgrid *muse_pixgrid_2d_create(cpl_table *, double, double, double, double, float *);
void muse_pixgrid_delete(muse_pixgrid *);

#endif /* MUSE_PIXGRID_H */
