const char net_box_lua[] =
"-- net_box.lua (internal file)\n"
"local log      = require('log')\n"
"local ffi      = require('ffi')\n"
"local fiber    = require('fiber')\n"
"local msgpack  = require('msgpack')\n"
"local urilib   = require('uri')\n"
"local internal = require('net.box.lib')\n"
"local trigger  = require('internal.trigger')\n"
"local utils    = require('internal.utils')\n"
"\n"
"local this_module\n"
"\n"
"local max               = math.max\n"
"local fiber_clock       = fiber.clock\n"
"\n"
"local check_select_opts   = box.internal.check_select_opts\n"
"local check_index_arg     = box.internal.check_index_arg\n"
"local check_space_arg     = box.internal.check_space_arg\n"
"local check_primary_index = box.internal.check_primary_index\n"
"local check_param_table   = utils.check_param_table\n"
"\n"
"local ibuf_t = ffi.typeof('struct ibuf')\n"
"local is_tuple = box.tuple.is\n"
"\n"
"local TIMEOUT_INFINITY = 500 * 365 * 86400\n"
"\n"
"-- select errors from box.error\n"
"local E_NO_CONNECTION        = box.error.NO_CONNECTION\n"
"local E_PROC_LUA             = box.error.PROC_LUA\n"
"\n"
"-- Method types used internally by net.box.\n"
"local M_PING        = 0\n"
"local M_CALL_16     = 1\n"
"local M_CALL_17     = 2\n"
"local M_EVAL        = 3\n"
"local M_INSERT      = 4\n"
"local M_REPLACE     = 5\n"
"local M_DELETE      = 6\n"
"local M_UPDATE      = 7\n"
"local M_UPSERT      = 8\n"
"local M_SELECT      = 9\n"
"local M_EXECUTE     = 10\n"
"local M_PREPARE     = 11\n"
"local M_UNPREPARE   = 12\n"
"local M_GET         = 13\n"
"local M_MIN         = 14\n"
"local M_MAX         = 15\n"
"local M_COUNT       = 16\n"
"local M_BEGIN       = 17\n"
"local M_COMMIT      = 18\n"
"local M_ROLLBACK    = 19\n"
"local M_SELECT_FETCH_POS  = 20\n"
"-- Injects raw data into connection. Used by tests.\n"
"local M_INJECT      = 21\n"
"\n"
"local REQUEST_OPTION_TYPES = {\n"
"    is_async    = \"boolean\",\n"
"    iterator    = \"string\",\n"
"    limit       = \"number\",\n"
"    offset      = \"number\",\n"
"    on_push     = \"function\",\n"
"    on_push_ctx = \"any\",\n"
"    return_raw  = \"boolean\",\n"
"    skip_header = \"boolean\",\n"
"    timeout     = \"number\",\n"
"    fetch_pos   = \"boolean\",\n"
"    after = function(after)\n"
"        if after ~= nil and type(after) ~= \"string\" and type(after) ~= \"table\"\n"
"                and not is_tuple(after) then\n"
"            return false, \"string, table, tuple\"\n"
"        end\n"
"        return true\n"
"    end,\n"
"    buffer = function(buf)\n"
"       if not ffi.istype(ibuf_t, buf) then\n"
"           return false, \"struct ibuf\"\n"
"       end\n"
"       return true\n"
"    end,\n"
"}\n"
"\n"
"local CONNECT_OPTION_TYPES = {\n"
"    user                        = \"string\",\n"
"    password                    = \"string\",\n"
"    wait_connected              = \"number, boolean\",\n"
"    reconnect_after             = \"number\",\n"
"    call_16                     = \"boolean\",\n"
"    console                     = \"boolean\",\n"
"    connect_timeout             = \"number\",\n"
"    fetch_schema                = \"boolean\",\n"
"    auth_type                   = \"string\",\n"
"    required_protocol_version   = \"number\",\n"
"    required_protocol_features  = \"table\",\n"
"    _disable_graceful_shutdown  = \"boolean\",\n"
"}\n"
"\n"
"-- Given an array of IPROTO feature ids, returns a map {feature_name: bool}.\n"
"local function iproto_features_resolve(feature_ids)\n"
"    local features = {}\n"
"    local feature_names = {}\n"
"    for name, id in pairs(box.iproto.feature) do\n"
"        features[name] = false\n"
"        feature_names[id] = name\n"
"    end\n"
"    for _, id in ipairs(feature_ids) do\n"
"        local name = feature_names[id]\n"
"        assert(name ~= nil)\n"
"        features[name] = true\n"
"    end\n"
"    return features\n"
"end\n"
"\n"
"-- Check if all required features (array) are set in the features map.\n"
"-- Returns the status and an array of missing features.\n"
"local function iproto_features_check(features, required)\n"
"    local ok = true\n"
"    local missing = {}\n"
"    for _, feature_name in ipairs(required) do\n"
"        if not features[feature_name] then\n"
"            table.insert(missing, feature_name)\n"
"            ok = false\n"
"        end\n"
"    end\n"
"    return ok, missing\n"
"end\n"
"\n"
"--\n"
"-- Default action on push during a synchronous request -\n"
"-- ignore.\n"
"--\n"
"local function on_push_sync_default() end\n"
"\n"
"local function parse_connect_params(host_or_uri, ...) -- self\? host_or_uri port\? opts\?\n"
"    local port, opts = ...\n"
"    if host_or_uri == this_module then host_or_uri, port, opts = ... end\n"
"    if type(port) == 'table' then opts = port; port = nil end\n"
"    if opts == nil then opts = {} else\n"
"        local copy = {}\n"
"        for k, v in pairs(opts) do copy[k] = v end\n"
"        opts = copy\n"
"    end\n"
"    local uri\n"
"    if port == nil and (type(host_or_uri) == 'string' or\n"
"                        type(host_or_uri) == 'number' or\n"
"                        type(host_or_uri) == 'table') then\n"
"        uri = host_or_uri\n"
"    elseif (type(host_or_uri) == 'string' or host_or_uri == nil) and\n"
"            (type(port) == 'string' or type(port) == 'number') then\n"
"        uri = urilib.format({host = host_or_uri, service = tostring(port)})\n"
"    else\n"
"        box.error(E_PROC_LUA,\n"
"                  \"usage: connect(uri[, opts] | host, port[, opts])\")\n"
"    end\n"
"    return uri, opts\n"
"end\n"
"\n"
"local function remote_serialize(self)\n"
"    return {\n"
"        host = self.host,\n"
"        port = self.port,\n"
"        opts = next(self.opts) and self.opts,\n"
"        state = self.state,\n"
"        error = self.error,\n"
"        protocol = self.protocol,\n"
"        schema_version = self.schema_version,\n"
"        peer_uuid = self.peer_uuid,\n"
"        peer_version_id = self.peer_version_id,\n"
"        peer_protocol_version = self.peer_protocol_version,\n"
"        peer_protocol_features = self.peer_protocol_features,\n"
"    }\n"
"end\n"
"\n"
"local function stream_serialize(self)\n"
"    local t = remote_serialize(self._conn)\n"
"    t.stream_id = self._stream_id\n"
"    return t\n"
"end\n"
"\n"
"local function stream_spaces_serialize(self)\n"
"    return self._stream._conn.space\n"
"end\n"
"\n"
"local function stream_space_serialize(self)\n"
"    return self._src\n"
"end\n"
"\n"
"local function stream_indexes_serialize(self)\n"
"    return self._space._src.index\n"
"end\n"
"\n"
"local function stream_index_serialize(self)\n"
"    return self._src\n"
"end\n"
"\n"
"local remote_methods = {}\n"
"local remote_mt = {\n"
"    __index = remote_methods, __serialize = remote_serialize,\n"
"    __metatable = false\n"
"}\n"
"\n"
"-- Create stream space index, which is same as connection space\n"
"-- index, but have non zero stream ID.\n"
"local function stream_wrap_index(stream_id, src)\n"
"    return setmetatable({\n"
"        _stream_id = stream_id,\n"
"        _src = src,\n"
"    }, {\n"
"        __index = src,\n"
"        __serialize = stream_index_serialize,\n"
"        __autocomplete = stream_index_serialize\n"
"    })\n"
"end\n"
"\n"
"-- Metatable for stream space indexes. When stream space being\n"
"-- created there are no indexes in it. When accessing the space\n"
"-- index, we look for corresponding space index in corresponding\n"
"-- connection space. If it is found we create same index for the\n"
"-- stream space but with corresponding stream ID. We do not need\n"
"-- to compare stream _schema_version and connection schema_version,\n"
"-- because all access to index  is carried out through it's space.\n"
"-- So we update schema_version when we access space.\n"
"local stream_indexes_mt = {\n"
"    __index = function(self, key)\n"
"        local _space = self._space\n"
"        local src = _space._src.index[key]\n"
"        if not src then\n"
"            return nil\n"
"        end\n"
"        local res = stream_wrap_index(_space._stream_id, src)\n"
"        self[key] = res\n"
"        return res\n"
"    end,\n"
"    __serialize = stream_indexes_serialize,\n"
"    __autocomplete = stream_indexes_serialize\n"
"}\n"
"\n"
"-- Create stream space, which is same as connection space,\n"
"-- but have non zero stream ID.\n"
"local function stream_wrap_space(stream, src)\n"
"    local res = setmetatable({\n"
"        _stream_id = stream._stream_id,\n"
"        _src = src,\n"
"        index = setmetatable({\n"
"            _space = nil,\n"
"        }, stream_indexes_mt)\n"
"    }, {\n"
"        __index = src,\n"
"        __serialize = stream_space_serialize,\n"
"        __autocomplete = stream_space_serialize\n"
"    })\n"
"    res.index._space = res\n"
"    return res\n"
"end\n"
"\n"
"-- Metatable for stream spaces. When stream being created there\n"
"-- are no spaces in it. When user try to access some space in\n"
"-- stream, we first of all compare _schema_version of stream with\n"
"-- schema_version from connection and if they are not equal, we\n"
"-- clear stream space cache and update it's schema_version. Then\n"
"-- we look for corresponding space in the connection. If it is\n"
"-- found we create same space for the stream but with corresponding\n"
"-- stream ID.\n"
"local stream_spaces_mt = {\n"
"    __index = function(self, key)\n"
"        local stream = self._stream\n"
"        if stream._schema_version ~= stream._conn.schema_version then\n"
"            stream._schema_version = stream._conn.schema_version\n"
"            self._stream_space_cache = {}\n"
"        end\n"
"        if self._stream_space_cache[key] then\n"
"            return self._stream_space_cache[key]\n"
"        end\n"
"        local src = stream._conn.space[key]\n"
"        if not src then\n"
"            return nil\n"
"        end\n"
"        local res = stream_wrap_space(stream, src)\n"
"        self._stream_space_cache[key] = res\n"
"        return res\n"
"    end,\n"
"    __serialize = stream_spaces_serialize,\n"
"    __autocomplete = stream_spaces_serialize\n"
"}\n"
"\n"
"-- This callback is invoked in a new fiber upon receiving 'box.shutdown' event\n"
"-- from a remote host. It runs on_shutdown triggers and then gracefully\n"
"-- terminates the connection.\n"
"local function graceful_shutdown(remote)\n"
"    remote._shutdown_pending = true\n"
"    local ok, err = pcall(remote._on_shutdown.run, remote._on_shutdown, remote)\n"
"    if not ok then\n"
"        log.error(err)\n"
"    end\n"
"    -- While the triggers were running, the connection could have been closed\n"
"    -- and even reestablished (if reconnect_after is set), in which case we\n"
"    -- must not initiate a graceful shutdown.\n"
"    if remote._shutdown_pending then\n"
"        remote._transport:graceful_shutdown()\n"
"        remote._shutdown_pending = nil\n"
"    end\n"
"end\n"
"\n"
"local space_metatable, index_metatable\n"
"\n"
"local function new_sm(uri, opts)\n"
"    local parsed_uri, err = urilib.parse(uri)\n"
"    if not parsed_uri then\n"
"        error(err)\n"
"    end\n"
"    if opts.user == nil and opts.password == nil then\n"
"        opts.user, opts.password = parsed_uri.login, parsed_uri.password\n"
"    end\n"
"    if opts.auth_type == nil and parsed_uri.params ~= nil and\n"
"       parsed_uri.params.auth_type ~= nil then\n"
"        opts.auth_type = parsed_uri.params.auth_type[1]\n"
"    end\n"
"    local host, port = parsed_uri.host, parsed_uri.service\n"
"    local user, password = opts.user, opts.password; opts.password = nil\n"
"    local last_reconnect_error\n"
"    local remote = {host = host, port = port, opts = opts, state = 'initial'}\n"
"    local function callback(what, ...)\n"
"        if remote._fiber == nil then\n"
"            remote._fiber = fiber.self()\n"
"        end\n"
"        if what == 'state_changed' then\n"
"            local state, err = ...\n"
"            remote.state, remote.error = state, err\n"
"            local was_connected = remote._is_connected\n"
"            if state == 'active' then\n"
"                if not was_connected then\n"
"                    remote._is_connected = true\n"
"                    remote._on_connect:run(remote)\n"
"                end\n"
"            elseif state == 'error' or state == 'error_reconnect' or\n"
"                   state == 'closed' then\n"
"                if was_connected then\n"
"                    remote._is_connected = false\n"
"                    local ok, trigger_err = pcall(remote._on_disconnect.run,\n"
"                                                  remote._on_disconnect, remote)\n"
"                    if not ok then\n"
"                        log.error(trigger_err)\n"
"                    end\n"
"                end\n"
"                -- A server may exit after initiating a graceful shutdown but\n"
"                -- before all clients close their connections (for example, on\n"
"                -- timeout). In this case it's important that we don't initiate\n"
"                -- a graceful shutdown on our side, because a connection can\n"
"                -- already be reestablished (if reconnect_after is set) by the\n"
"                -- time we finish running on_shutdown triggers.\n"
"                remote._shutdown_pending = nil\n"
"            end\n"
"            if state == 'error_reconnect' then\n"
"                -- Repeat the same error in verbose log only.\n"
"                -- Else the error clogs the log. See gh-3175.\n"
"                if err ~= last_reconnect_error then\n"
"                    log.warn('%s:%s: %s', host or \"\", port or \"\", err)\n"
"                    last_reconnect_error = err\n"
"                else\n"
"                    log.verbose('%s:%s: %s', host or \"\", port or \"\", err)\n"
"                end\n"
"            end\n"
"            remote._state_cond:broadcast()\n"
"        elseif what == 'handshake' then\n"
"            local greeting, version, features = ...\n"
"            remote.protocol = greeting.protocol\n"
"            remote.peer_uuid = greeting.uuid\n"
"            remote.peer_version_id = greeting.version_id\n"
"            features = iproto_features_resolve(features)\n"
"            remote.peer_protocol_version = version\n"
"            remote.peer_protocol_features = features\n"
"            if opts.required_protocol_version and\n"
"               opts.required_protocol_version > version then\n"
"                box.error({\n"
"                    code = E_NO_CONNECTION,\n"
"                    reason = string.format(\n"
"                        'Protocol version (%d) < required (%d)',\n"
"                         version, opts.required_protocol_version),\n"
"                })\n"
"            end\n"
"            if opts.required_protocol_features then\n"
"                local ok, missing = iproto_features_check(\n"
"                    features, opts.required_protocol_features)\n"
"                if not ok then\n"
"                    box.error({\n"
"                        code = E_NO_CONNECTION,\n"
"                        reason = 'Missing required protocol features: ' ..\n"
"                                 table.concat(missing, ', '),\n"
"                    })\n"
"                end\n"
"            end\n"
"        elseif what == 'did_fetch_schema' then\n"
"            remote:_install_schema(...)\n"
"        elseif what == 'event' then\n"
"            local key, value = ...\n"
"            local state = remote._watchers and remote._watchers[key]\n"
"            if state then\n"
"                state.value = value\n"
"                state.has_value = true\n"
"                state.version = state.version + 1\n"
"                state.is_acknowledged = false\n"
"                while state.idle do\n"
"                    local watcher = state.idle\n"
"                    state.idle = watcher._next\n"
"                    watcher:_run_async()\n"
"                end\n"
"            end\n"
"        end\n"
"    end\n"
"    if opts.console then\n"
"        box.error(box.error.ILLEGAL_PARAMS,\n"
"                  \"Netbox text protocol support was dropped, \" ..\n"
"                  \"please use require('console').connect() instead\")\n"
"    else\n"
"        setmetatable(remote, remote_mt)\n"
"        remote._space_mt = space_metatable(remote)\n"
"        remote._index_mt = index_metatable(remote)\n"
"        if opts.call_16 then\n"
"            remote.call = remote.call_16\n"
"            remote.eval = remote.eval_16\n"
"        end\n"
"    end\n"
"    remote._on_schema_reload = trigger.new(\"on_schema_reload\")\n"
"    remote._on_shutdown = trigger.new(\"on_shutdown\")\n"
"    remote._on_disconnect = trigger.new(\"on_disconnect\")\n"
"    remote._on_connect = trigger.new(\"on_connect\")\n"
"    remote._is_connected = false\n"
"    -- Signaled when the state changes.\n"
"    remote._state_cond = fiber.cond()\n"
"    -- Last stream ID used for this connection.\n"
"    remote._last_stream_id = 0\n"
"    local weak_refs = setmetatable({callback = callback}, {__mode = 'v'})\n"
"    -- Create a transport, adding auto-stop-on-GC feature.\n"
"    -- The tricky part is the callback:\n"
"    --  * callback references the transport (indirectly);\n"
"    --  * worker fiber references the callback;\n"
"    --  * fibers are GC roots - i.e. transport is never GC-ed!\n"
"    -- We solve the issue by making the worker->callback ref weak.\n"
"    -- Now it is necessary to have a strong ref to callback somewhere or\n"
"    -- it is GC-ed prematurely. We store a strong reference in the remote\n"
"    -- connection object.\n"
"    local function weak_callback(...)\n"
"        local callback = weak_refs.callback\n"
"        if callback then return callback(...) end\n"
"        -- The callback is responsible for handling graceful shutdown.\n"
"        -- If it's garbage collected, the connection won't be closed on\n"
"        -- receiving 'box.shutdown' event and so the server won't exit\n"
"        -- until the connection object is garbage collected, which may\n"
"        -- take forever. To avoid that, let's break the worker loop if\n"
"        -- we see that the callback is unavailable.\n"
"        weak_refs.transport:stop()\n"
"    end\n"
"    remote._callback = callback\n"
"    local transport = internal.new_transport(\n"
"            uri, user, password, weak_callback,\n"
"            opts.connect_timeout, opts.reconnect_after,\n"
"            opts.fetch_schema, opts.auth_type)\n"
"    weak_refs.transport = transport\n"
"    remote._transport = transport\n"
"    remote._gc_hook = ffi.gc(ffi.new('char[1]'), function()\n"
"        pcall(transport.stop, transport);\n"
"    end)\n"
"    if not opts._disable_graceful_shutdown then\n"
"        remote:watch('box.shutdown', function(_, value)\n"
"            if value then\n"
"                graceful_shutdown(remote)\n"
"            end\n"
"        end)\n"
"    end\n"
"    transport:start()\n"
"    if opts.wait_connected ~= false then\n"
"        remote:wait_state('active', tonumber(opts.wait_connected))\n"
"    end\n"
"    return remote\n"
"end\n"
"\n"
"--\n"
"-- Connect to a remote server.\n"
"-- @param uri OR host and port. URI is a string like\n"
"--        hostname:port@login:password. Host and port can be\n"
"--        passed separately with login and password in the next\n"
"--        parameter.\n"
"-- @param opts Options like reconnect_after, connect_timeout,\n"
"--        wait_connected, login, password, ...\n"
"--\n"
"-- @retval Net.box object.\n"
"--\n"
"local function connect(...)\n"
"    local uri, opts = parse_connect_params(...)\n"
"    check_param_table(opts, CONNECT_OPTION_TYPES)\n"
"    return new_sm(uri, opts)\n"
"end\n"
"\n"
"local function check_remote_arg(remote, method)\n"
"    if type(remote) ~= 'table' then\n"
"        local fmt = 'Use remote:%s(...) instead of remote.%s(...):'\n"
"        box.error(E_PROC_LUA, string.format(fmt, method, method))\n"
"    end\n"
"end\n"
"\n"
"local function check_call_args(args)\n"
"    if args ~= nil and type(args) ~= 'table' and\n"
"       not msgpack.is_object(args) then\n"
"        error(\"Use remote:call(func_name, {arg1, arg2, ...}, opts) \"..\n"
"              \"instead of remote:call(func_name, arg1, arg2, ...)\")\n"
"    end\n"
"end\n"
"\n"
"local function check_eval_args(args)\n"
"    if args ~= nil and type(args) ~= 'table' and\n"
"       not msgpack.is_object(args) then\n"
"        error(\"Use remote:eval(expression, {arg1, arg2, ...}, opts) \"..\n"
"              \"instead of remote:eval(expression, arg1, arg2, ...)\")\n"
"    end\n"
"end\n"
"\n"
"local function stream_new_stream(stream)\n"
"    check_remote_arg(stream, 'new_stream')\n"
"    return stream._conn:new_stream()\n"
"end\n"
"\n"
"local function stream_begin(stream, txn_opts, netbox_opts)\n"
"    check_remote_arg(stream, 'begin')\n"
"    check_param_table(netbox_opts, REQUEST_OPTION_TYPES)\n"
"    local timeout\n"
"    local txn_isolation\n"
"    if txn_opts then\n"
"        if type(txn_opts) ~= 'table' then\n"
"            error(\"txn_opts should be a table\")\n"
"        end\n"
"        timeout = txn_opts.timeout\n"
"        if timeout and (type(timeout) ~= \"number\" or timeout <= 0) then\n"
"            error(\"timeout must be a number greater than 0\")\n"
"        end\n"
"        txn_isolation = txn_opts.txn_isolation\n"
"        if txn_isolation ~= nil then\n"
"            txn_isolation =\n"
"                box.internal.normalize_txn_isolation_level(txn_isolation)\n"
"        end\n"
"    end\n"
"    local res = stream:_request(M_BEGIN, netbox_opts, nil,\n"
"                                stream._stream_id, timeout, txn_isolation)\n"
"    if netbox_opts and netbox_opts.is_async then\n"
"        return res\n"
"    end\n"
"end\n"
"\n"
"local function stream_commit(stream, opts)\n"
"    check_remote_arg(stream, 'commit')\n"
"    check_param_table(opts, REQUEST_OPTION_TYPES)\n"
"    local res = stream:_request(M_COMMIT, opts, nil, stream._stream_id)\n"
"    if opts and opts.is_async then\n"
"        return res\n"
"    end\n"
"end\n"
"\n"
"local function stream_rollback(stream, opts)\n"
"    check_remote_arg(stream, 'rollback')\n"
"    check_param_table(opts, REQUEST_OPTION_TYPES)\n"
"    local res = stream:_request(M_ROLLBACK, opts, nil, stream._stream_id)\n"
"    if opts and opts.is_async then\n"
"        return res\n"
"    end\n"
"end\n"
"\n"
"function remote_methods:new_stream()\n"
"    check_remote_arg(self, 'new_stream')\n"
"    self._last_stream_id = self._last_stream_id + 1\n"
"    local stream = setmetatable({\n"
"        new_stream = stream_new_stream,\n"
"        begin = stream_begin,\n"
"        commit = stream_commit,\n"
"        rollback = stream_rollback,\n"
"        _stream_id = self._last_stream_id,\n"
"        space = setmetatable({\n"
"            _stream_space_cache = {},\n"
"            _stream = nil,\n"
"        }, stream_spaces_mt),\n"
"        _conn = self,\n"
"        _schema_version = self.schema_version,\n"
"    }, { __index = self, __serialize = stream_serialize })\n"
"    stream.space._stream = stream\n"
"    return stream\n"
"end\n"
"\n"
"local watcher_methods = {}\n"
"local watcher_mt = {\n"
"    __index = watcher_methods,\n"
"    __tostring = function()\n"
"        return 'net.box.watcher'\n"
"    end,\n"
"}\n"
"watcher_mt.__serialize = watcher_mt.__tostring\n"
"\n"
"function watcher_methods:_run()\n"
"    local state = self._state\n"
"    assert(state ~= nil)\n"
"    assert(state.has_value)\n"
"    self._version = state.version\n"
"    local status, err = pcall(self._func, state.key, state.value)\n"
"    if not status then\n"
"        log.error(err)\n"
"    end\n"
"    if not self._state then\n"
"        -- The watcher was unregistered while the callback was running.\n"
"        return\n"
"    end\n"
"    assert(state == self._state)\n"
"    if self._version == state.version then\n"
"        -- The value was not updated while this watcher was running.\n"
"        -- Append it to the list of ready watchers and send an ack to\n"
"        -- the server unless we've already sent it.\n"
"        self._next = state.idle\n"
"        state.idle = self\n"
"        if not state.is_acknowledged then\n"
"            self._conn._transport:watch(state.key)\n"
"            state.is_acknowledged = true\n"
"        end\n"
"    else\n"
"        -- The value was updated while this watcher was running.\n"
"        -- Rerun it with the new value.\n"
"        assert(self._version < state.version)\n"
"        return self:_run()\n"
"    end\n"
"end\n"
"\n"
"function watcher_methods:_run_async()\n"
"    fiber.new(self._run, self)\n"
"end\n"
"\n"
"function watcher_methods:unregister()\n"
"    if type(self) ~= 'table' then\n"
"        box.error(E_PROC_LUA,\n"
"                  'Use watcher:unregister() instead of watcher.unregister()')\n"
"    end\n"
"    local state = self._state\n"
"    if not self._state then\n"
"        box.error(E_PROC_LUA, 'Watcher is already unregistered')\n"
"    end\n"
"    self._state = nil\n"
"    local conn = self._conn\n"
"    assert(conn._watchers ~= nil)\n"
"    assert(conn._watchers[state.key] == state)\n"
"    if state.idle then\n"
"        -- Remove the watcher from the idle list.\n"
"        if state.idle == self then\n"
"            state.idle = self._next\n"
"        else\n"
"            local watcher = state.idle\n"
"            while watcher._next do\n"
"                if watcher._next == self then\n"
"                    watcher._next = self._next\n"
"                    break\n"
"                end\n"
"                watcher = watcher._next\n"
"            end\n"
"        end\n"
"    end\n"
"    assert(state.watcher_count > 0)\n"
"    state.watcher_count = state.watcher_count - 1\n"
"    if state.watcher_count == 0 then\n"
"        -- No watchers left. Unsubscribe and drop the state.\n"
"        conn._transport:unwatch(state.key)\n"
"        conn._watchers[state.key] = nil\n"
"    end\n"
"end\n"
"\n"
"function remote_methods:watch(key, func)\n"
"    check_remote_arg(self, 'watch')\n"
"    if type(key) ~= 'string' then\n"
"        box.error(E_PROC_LUA, 'key must be a string')\n"
"    end\n"
"    if type(func) ~= 'function' then\n"
"        box.error(E_PROC_LUA, 'func must be a function')\n"
"    end\n"
"    if not self._watchers then\n"
"        self._watchers = {}\n"
"        -- Install a trigger to resubscribe registered watchers on reconnect.\n"
"        self._on_connect(function(conn)\n"
"            for key, _ in pairs(conn._watchers) do\n"
"                conn._transport:watch(key)\n"
"            end\n"
"        end)\n"
"    end\n"
"    local state = self._watchers[key]\n"
"    if not state then\n"
"        state = {}\n"
"        state.key = key\n"
"        state.value = nil\n"
"        -- Set when the first value is received for the state. We can't rely\n"
"        -- on the value being non-nil, because a state can actually have nil\n"
"        -- value so we need a separate flag.\n"
"        state.has_value = false\n"
"        -- Incremented every time a new value is received for this state.\n"
"        -- We use to reschedule a watcher that was already running when a new\n"
"        -- value was received.\n"
"        state.version = 0\n"
"        -- Set to true if the last received value has been acknowledged\n"
"        -- (that is we sent a WATCH packet after receiving it).\n"
"        state.is_acknowledged = false\n"
"        -- Number of watchers registered for this key. We delete a state when\n"
"        -- nobody uses it.\n"
"        state.watcher_count = 0\n"
"        -- Singly-linked (by ._next) list of watchers ready to be notified\n"
"        -- (i.e. registered and not currently running).\n"
"        state.idle = nil\n"
"        -- We don't care whether the connection is active or not, because if\n"
"        -- the connection fails, we will resubscribe all registered watchers\n"
"        -- from the on_connect trigger.\n"
"        self._transport:watch(key)\n"
"        self._watchers[key] = state\n"
"    end\n"
"    local watcher = setmetatable({\n"
"        _conn = self,\n"
"        _func = func,\n"
"        _state = state,\n"
"        _version = 0,\n"
"        _next = nil,\n"
"    }, watcher_mt)\n"
"    state.watcher_count = state.watcher_count + 1\n"
"    if state.has_value then\n"
"        watcher:_run_async()\n"
"    else\n"
"        watcher._next = state.idle\n"
"        state.idle = watcher\n"
"    end\n"
"    return watcher\n"
"end\n"
"\n"
"function remote_methods:close()\n"
"    check_remote_arg(self, 'close')\n"
"    self._transport:stop(true)\n"
"end\n"
"\n"
"function remote_methods:on_schema_reload(...)\n"
"    check_remote_arg(self, 'on_schema_reload')\n"
"    return self._on_schema_reload(...)\n"
"end\n"
"\n"
"function remote_methods:on_shutdown(...)\n"
"    check_remote_arg(self, 'on_shutdown')\n"
"    return self._on_shutdown(...)\n"
"end\n"
"\n"
"function remote_methods:on_disconnect(...)\n"
"    check_remote_arg(self, 'on_disconnect')\n"
"    return self._on_disconnect(...)\n"
"end\n"
"\n"
"function remote_methods:on_connect(...)\n"
"    check_remote_arg(self, 'on_connect')\n"
"    return self._on_connect(...)\n"
"end\n"
"\n"
"function remote_methods:is_connected()\n"
"    check_remote_arg(self, 'is_connected')\n"
"    return self.state == 'active' or self.state == 'fetch_schema'\n"
"end\n"
"\n"
"function remote_methods:wait_connected(timeout)\n"
"    check_remote_arg(self, 'wait_connected')\n"
"    return self:wait_state('active', timeout)\n"
"end\n"
"\n"
"--\n"
"-- Make a request, which throws an exception in case of critical errors\n"
"-- (e.g. wrong API usage) and returns nil,err if there's connection related\n"
"-- issues.\n"
"--\n"
"function remote_methods:_request_impl(method, opts, format, stream_id, ...)\n"
"    local transport = self._transport\n"
"    local on_push, on_push_ctx, buffer, skip_header, return_raw, deadline\n"
"    -- Extract options, set defaults, check if the request is\n"
"    -- async.\n"
"    if opts then\n"
"        buffer = opts.buffer\n"
"        skip_header = opts.skip_header\n"
"        return_raw = opts.return_raw\n"
"        if opts.is_async then\n"
"            if opts.on_push or opts.on_push_ctx then\n"
"                error('To handle pushes in an async request use future:pairs()')\n"
"            end\n"
"            return transport:perform_async_request(self, buffer, skip_header,\n"
"                                                   return_raw, table.insert,\n"
"                                                   {}, format, stream_id,\n"
"                                                   method, ...)\n"
"        end\n"
"        if opts.timeout then\n"
"            deadline = fiber_clock() + opts.timeout\n"
"        end\n"
"        on_push = opts.on_push or on_push_sync_default\n"
"        on_push_ctx = opts.on_push_ctx\n"
"    else\n"
"        on_push = on_push_sync_default\n"
"    end\n"
"    -- Execute synchronous request.\n"
"    if self._fiber == fiber.self() then\n"
"        error('Synchronous requests are not allowed in net.box trigger')\n"
"    end\n"
"    local timeout = deadline and max(0, deadline - fiber_clock())\n"
"    if self.state ~= 'active' then\n"
"        self:wait_state('active', timeout)\n"
"        timeout = deadline and max(0, deadline - fiber_clock())\n"
"    end\n"
"    local res, err = transport:perform_request(timeout, buffer, skip_header,\n"
"                                               return_raw, on_push, on_push_ctx,\n"
"                                               format, stream_id, method, ...)\n"
"    -- Try to wait until a schema is reloaded if needed.\n"
"    -- Regardless of reloading result, the main response is\n"
"    -- returned, since it does not depend on any schema things.\n"
"    if self.state == 'fetch_schema' then\n"
"        timeout = deadline and max(0, deadline - fiber_clock())\n"
"        self:wait_state('active', timeout)\n"
"    end\n"
"    return res, err\n"
"end\n"
"\n"
"function remote_methods:_request(method, opts, format, stream_id, ...)\n"
"    local res, err = self:_request_impl(method, opts, format, stream_id, ...)\n"
"    if err then\n"
"        box.error(err)\n"
"    end\n"
"    return res\n"
"end\n"
"\n"
"function remote_methods:_inject(str, opts)\n"
"    check_param_table(opts, REQUEST_OPTION_TYPES)\n"
"    return self:_request(M_INJECT, opts, nil, nil, str)\n"
"end\n"
"\n"
"function remote_methods:_next_sync()\n"
"    return self._transport:next_sync()\n"
"end\n"
"\n"
"function remote_methods:ping(opts)\n"
"    check_remote_arg(self, 'ping')\n"
"    check_param_table(opts, REQUEST_OPTION_TYPES)\n"
"    if opts and opts.is_async then\n"
"        error(\"conn:ping() doesn't support `is_async` argument\")\n"
"    end\n"
"    local _, err = self:_request_impl(M_PING, opts, nil, self._stream_id)\n"
"    return err == nil\n"
"end\n"
"\n"
"function remote_methods:reload_schema()\n"
"    check_remote_arg(self, 'reload_schema')\n"
"    self:ping()\n"
"end\n"
"\n"
"-- @deprecated since 1.7.4\n"
"function remote_methods:call_16(func_name, ...)\n"
"    check_remote_arg(self, 'call')\n"
"    return (self:_request(M_CALL_16, nil, nil, self._stream_id,\n"
"                          tostring(func_name), {...}))\n"
"end\n"
"\n"
"function remote_methods:call(func_name, args, opts)\n"
"    check_remote_arg(self, 'call')\n"
"    check_call_args(args)\n"
"    check_param_table(opts, REQUEST_OPTION_TYPES)\n"
"    args = args or {}\n"
"    local res = self:_request(M_CALL_17, opts, nil, self._stream_id,\n"
"                              tostring(func_name), args)\n"
"    if type(res) ~= 'table' or opts and opts.is_async then\n"
"        return res\n"
"    end\n"
"    return unpack(res)\n"
"end\n"
"\n"
"-- @deprecated since 1.7.4\n"
"function remote_methods:eval_16(code, ...)\n"
"    check_remote_arg(self, 'eval')\n"
"    return unpack((self:_request(M_EVAL, nil, nil, self._stream_id,\n"
"                                 code, {...})))\n"
"end\n"
"\n"
"function remote_methods:eval(code, args, opts)\n"
"    check_remote_arg(self, 'eval')\n"
"    check_eval_args(args)\n"
"    check_param_table(opts, REQUEST_OPTION_TYPES)\n"
"    args = args or {}\n"
"    local res = self:_request(M_EVAL, opts, nil, self._stream_id, code, args)\n"
"    if type(res) ~= 'table' or opts and opts.is_async then\n"
"        return res\n"
"    end\n"
"    return unpack(res)\n"
"end\n"
"\n"
"function remote_methods:execute(query, parameters, sql_opts, netbox_opts)\n"
"    check_remote_arg(self, \"execute\")\n"
"    if sql_opts ~= nil then\n"
"        box.error(box.error.UNSUPPORTED, \"execute\", \"options\")\n"
"    end\n"
"    check_param_table(netbox_opts, REQUEST_OPTION_TYPES)\n"
"    return self:_request(M_EXECUTE, netbox_opts, nil, self._stream_id,\n"
"                         query, parameters or {}, sql_opts or {})\n"
"end\n"
"\n"
"function remote_methods:prepare(query, parameters, sql_opts, netbox_opts) -- luacheck: no unused args\n"
"    check_remote_arg(self, \"prepare\")\n"
"    if type(query) ~= \"string\" then\n"
"        box.error(box.error.SQL_PREPARE, \"expected string as SQL statement\")\n"
"    end\n"
"    if sql_opts ~= nil then\n"
"        box.error(box.error.UNSUPPORTED, \"prepare\", \"options\")\n"
"    end\n"
"    check_param_table(netbox_opts, REQUEST_OPTION_TYPES)\n"
"    return self:_request(M_PREPARE, netbox_opts, nil, self._stream_id, query)\n"
"end\n"
"\n"
"function remote_methods:unprepare(query, parameters, sql_opts, netbox_opts)\n"
"    check_remote_arg(self, \"unprepare\")\n"
"    if type(query) ~= \"number\" then\n"
"        box.error(box.error.ILLEGAL_PARAMS,\n"
"                  \"query id is expected to be numeric\")\n"
"    end\n"
"    if sql_opts ~= nil then\n"
"        box.error(box.error.UNSUPPORTED, \"unprepare\", \"options\")\n"
"    end\n"
"    check_param_table(netbox_opts, REQUEST_OPTION_TYPES)\n"
"    return self:_request(M_UNPREPARE, netbox_opts, nil, self._stream_id,\n"
"                         query, parameters or {}, sql_opts or {})\n"
"end\n"
"\n"
"function remote_methods:wait_state(state, timeout)\n"
"    check_remote_arg(self, 'wait_state')\n"
"    local deadline = fiber_clock() + (timeout or TIMEOUT_INFINITY)\n"
"    -- FYI: [] on a string is valid\n"
"    repeat until self.state == state or state[self.state] or\n"
"                 self.state == 'closed' or self.state == 'error' or\n"
"                 (not self.opts.reconnect_after and\n"
"                  self.state == 'graceful_shutdown') or\n"
"                 not self._state_cond:wait(max(0, deadline - fiber_clock()))\n"
"    return self.state == state or state[self.state] or false\n"
"end\n"
"\n"
"function remote_methods:_install_schema(schema_version, spaces, indices,\n"
"                                        collations, space_sequences)\n"
"    local sl, space_mt, index_mt = {}, self._space_mt, self._index_mt\n"
"    for _, space in pairs(spaces) do\n"
"        local name = space[3]\n"
"        local id = space[1]\n"
"        local engine = space[4]\n"
"        local field_count = space[5]\n"
"        local format = space[7] or {}\n"
"        local s = {}\n"
"        if self.space ~= nil and self.space[id] ~= nil then\n"
"            s = self.space[id]\n"
"        else\n"
"            setmetatable(s, space_mt)\n"
"        end\n"
"        s.id = id\n"
"        s.name = name\n"
"        s.engine = engine\n"
"        s.field_count = field_count\n"
"        s.enabled = true\n"
"        s.index = {}\n"
"        s.temporary = false\n"
"        s.is_sync = false\n"
"        s._format = format\n"
"        s._format_cdata = box.internal.new_tuple_format(format)\n"
"        s.connection = self\n"
"        if #space > 5 then\n"
"            local opts = space[6]\n"
"            if type(opts) == 'table' then\n"
"                -- Tarantool >= 1.7.0\n"
"                s.temporary = not not opts.temporary\n"
"                s.is_sync = not not opts.is_sync\n"
"            elseif type(opts) == 'string' then\n"
"                -- Tarantool < 1.7.0\n"
"                s.temporary = string.match(opts, 'temporary') ~= nil\n"
"            end\n"
"        end\n"
"\n"
"        sl[id] = s\n"
"        sl[name] = s\n"
"    end\n"
"\n"
"    local seql = {}\n"
"    if space_sequences ~= nil then\n"
"        for _, seq in ipairs(space_sequences) do\n"
"            seql[seq[1]] = seq\n"
"        end\n"
"    end\n"
"\n"
"    for _, index in pairs(indices) do\n"
"        local idx = {\n"
"            space   = index[1],\n"
"            id      = index[2],\n"
"            name    = index[3],\n"
"            type    = string.upper(index[4]),\n"
"            parts   = {},\n"
"        }\n"
"        local OPTS = 5\n"
"        local PARTS = 6\n"
"\n"
"        if type(index[OPTS]) == 'number' then\n"
"            idx.unique = index[OPTS] == 1\n"
"\n"
"            for k = 0, index[PARTS] - 1 do\n"
"                local pktype = index[7 + k * 2 + 1]\n"
"                local pkfield = index[7 + k * 2]\n"
"\n"
"                local pk = {\n"
"                    type = pktype,\n"
"                    fieldno = pkfield + 1\n"
"                }\n"
"                idx.parts[k + 1] = pk\n"
"            end\n"
"        else\n"
"            for k = 1, #index[PARTS] do\n"
"                local pknullable = index[PARTS][k].is_nullable or false\n"
"                local pkexcludenull = index[PARTS][k].exclude_null or false\n"
"                local pkcollationid = index[PARTS][k].collation\n"
"                local pkpath = index[PARTS][k].path\n"
"                local pktype = index[PARTS][k][2] or index[PARTS][k].type\n"
"                local pkfield = index[PARTS][k][1] or index[PARTS][k].field\n"
"                -- resolve a collation name if a peer has\n"
"                -- _vcollation view\n"
"                local pkcollation = nil\n"
"                if pkcollationid ~= nil and collations ~= nil then\n"
"                    pkcollation = collations[pkcollationid + 1][2]\n"
"                end\n"
"\n"
"                local pk = {\n"
"                    type = pktype,\n"
"                    fieldno = pkfield + 1,\n"
"                    is_nullable = pknullable,\n"
"                    exclude_null = pkexcludenull,\n"
"                    path = pkpath,\n"
"                }\n"
"                if collations == nil then\n"
"                    pk.collation_id = pkcollationid\n"
"                else\n"
"                    pk.collation = pkcollation\n"
"                end\n"
"                idx.parts[k] = pk\n"
"            end\n"
"            idx.unique = not not index[OPTS].unique\n"
"        end\n"
"\n"
"        if idx.id == 0 then\n"
"            local seq = seql[idx.space]\n"
"            if seq ~= nil then\n"
"                idx.sequence_id = seq[2]\n"
"                idx.sequence_fieldno = seq[4] + 1\n"
"                if seq[5] ~= '' then\n"
"                    idx.sequence_path = seq[5]\n"
"                end\n"
"            end\n"
"        end\n"
"\n"
"        if sl[idx.space] ~= nil then\n"
"            sl[idx.space].index[idx.id] = idx\n"
"            sl[idx.space].index[idx.name] = idx\n"
"            idx.space = sl[idx.space]\n"
"            setmetatable(idx, index_mt)\n"
"        end\n"
"    end\n"
"\n"
"    self.schema_version = schema_version\n"
"    self.space = sl\n"
"    local ok, err = pcall(self._on_schema_reload.run, self._on_schema_reload,\n"
"                          self)\n"
"    if not ok then\n"
"        log.error(err)\n"
"    end\n"
"end\n"
"\n"
"local function nothing_or_data(value)\n"
"    if value ~= nil then\n"
"        return value\n"
"    end\n"
"end\n"
"\n"
"space_metatable = function(remote)\n"
"    local methods = {}\n"
"\n"
"    function methods:insert(tuple, opts)\n"
"        check_space_arg(self, 'insert')\n"
"        check_param_table(opts, REQUEST_OPTION_TYPES)\n"
"        return remote:_request(M_INSERT, opts, self._format_cdata,\n"
"                               self._stream_id, self.id, tuple)\n"
"    end\n"
"\n"
"    function methods:replace(tuple, opts)\n"
"        check_space_arg(self, 'replace')\n"
"        check_param_table(opts, REQUEST_OPTION_TYPES)\n"
"        return remote:_request(M_REPLACE, opts, self._format_cdata,\n"
"                               self._stream_id, self.id, tuple)\n"
"    end\n"
"\n"
"    function methods:select(key, opts)\n"
"        check_space_arg(self, 'select')\n"
"        return check_primary_index(self):select(key, opts)\n"
"    end\n"
"\n"
"    function methods:delete(key, opts)\n"
"        check_space_arg(self, 'delete')\n"
"        return check_primary_index(self):delete(key, opts)\n"
"    end\n"
"\n"
"    function methods:update(key, oplist, opts)\n"
"        check_space_arg(self, 'update')\n"
"        return check_primary_index(self):update(key, oplist, opts)\n"
"    end\n"
"\n"
"    function methods:upsert(key, oplist, opts)\n"
"        check_space_arg(self, 'upsert')\n"
"        check_param_table(opts, REQUEST_OPTION_TYPES)\n"
"        return nothing_or_data(remote:_request(M_UPSERT, opts, nil,\n"
"                                               self._stream_id, self.id,\n"
"                                               key, oplist))\n"
"    end\n"
"\n"
"    function methods:get(key, opts)\n"
"        check_space_arg(self, 'get')\n"
"        return check_primary_index(self):get(key, opts)\n"
"    end\n"
"\n"
"    function methods:format(format)\n"
"        if format == nil then\n"
"            return self._format\n"
"        else\n"
"            box.error(box.error.UNSUPPORTED, \"net.box\", \"setting space format\")\n"
"        end\n"
"    end\n"
"\n"
"    return { __index = methods, __metatable = false }\n"
"end\n"
"\n"
"index_metatable = function(remote)\n"
"    local methods = {}\n"
"\n"
"    function methods:select(key, opts)\n"
"        check_index_arg(self, 'select')\n"
"        check_param_table(opts, REQUEST_OPTION_TYPES)\n"
"        local key_is_nil = (key == nil or\n"
"                            (type(key) == 'table' and #key == 0))\n"
"        local iterator, offset, limit, _, after, fetch_pos =\n"
"            check_select_opts(opts, key_is_nil)\n"
"        if (after ~= nil or fetch_pos)\n"
"                and not remote.peer_protocol_features.pagination then\n"
"            return box.error(box.error.UNSUPPORTED, \"Remote server\",\n"
"                \"pagination\")\n"
"        end\n"
"\n"
"        local res\n"
"        local method = fetch_pos and M_SELECT_FETCH_POS or M_SELECT\n"
"        res = (remote:_request(method, opts, self.space._format_cdata,\n"
"                               self._stream_id, self.space.id, self.id,\n"
"                               iterator, offset, limit, key, after, fetch_pos))\n"
"        if type(res) ~= 'table' or not fetch_pos or opts and opts.is_async then\n"
"            return res\n"
"        end\n"
"        return unpack(res)\n"
"    end\n"
"\n"
"    function methods:get(key, opts)\n"
"        check_index_arg(self, 'get')\n"
"        check_param_table(opts, REQUEST_OPTION_TYPES)\n"
"        if opts and opts.buffer then\n"
"            error(\"index:get() doesn't support `buffer` argument\")\n"
"        end\n"
"        return nothing_or_data(remote:_request(M_GET, opts,\n"
"                                               self.space._format_cdata,\n"
"                                               self._stream_id,\n"
"                                               self.space.id, self.id,\n"
"                                               box.index.EQ, 0, 2, key,\n"
"                                               nil, false))\n"
"    end\n"
"\n"
"    function methods:min(key, opts)\n"
"        check_index_arg(self, 'min')\n"
"        check_param_table(opts, REQUEST_OPTION_TYPES)\n"
"        if opts and opts.buffer then\n"
"            error(\"index:min() doesn't support `buffer` argument\")\n"
"        end\n"
"        return nothing_or_data(remote:_request(M_MIN, opts,\n"
"                                               self.space._format_cdata,\n"
"                                               self._stream_id,\n"
"                                               self.space.id, self.id,\n"
"                                               box.index.GE, 0, 1, key,\n"
"                                               nil, false))\n"
"    end\n"
"\n"
"    function methods:max(key, opts)\n"
"        check_index_arg(self, 'max')\n"
"        check_param_table(opts, REQUEST_OPTION_TYPES)\n"
"        if opts and opts.buffer then\n"
"            error(\"index:max() doesn't support `buffer` argument\")\n"
"        end\n"
"        return nothing_or_data(remote:_request(M_MAX, opts,\n"
"                                               self.space._format_cdata,\n"
"                                               self._stream_id,\n"
"                                               self.space.id, self.id,\n"
"                                               box.index.LE, 0, 1, key,\n"
"                                               nil, false))\n"
"    end\n"
"\n"
"    function methods:count(key, opts)\n"
"        check_index_arg(self, 'count')\n"
"        check_param_table(opts, REQUEST_OPTION_TYPES)\n"
"        if opts and opts.buffer then\n"
"            error(\"index:count() doesn't support `buffer` argument\")\n"
"        end\n"
"        local code = string.format('box.space.%s.index.%s:count',\n"
"                                   self.space.name, self.name)\n"
"        return remote:_request(M_COUNT, opts, nil, self._stream_id,\n"
"                               code, { key, opts })\n"
"    end\n"
"\n"
"    function methods:delete(key, opts)\n"
"        check_index_arg(self, 'delete')\n"
"        check_param_table(opts, REQUEST_OPTION_TYPES)\n"
"        return nothing_or_data(remote:_request(M_DELETE, opts,\n"
"                                               self.space._format_cdata,\n"
"                                               self._stream_id, self.space.id,\n"
"                                               self.id, key))\n"
"    end\n"
"\n"
"    function methods:update(key, oplist, opts)\n"
"        check_index_arg(self, 'update')\n"
"        check_param_table(opts, REQUEST_OPTION_TYPES)\n"
"        return nothing_or_data(remote:_request(M_UPDATE, opts,\n"
"                                               self.space._format_cdata,\n"
"                                               self._stream_id, self.space.id,\n"
"                                               self.id, key, oplist))\n"
"    end\n"
"\n"
"    return { __index = methods, __metatable = false }\n"
"end\n"
"\n"
"this_module = {\n"
"    connect = connect,\n"
"    new = connect, -- Tarantool < 1.7.1 compatibility,\n"
"    _method = { -- for tests\n"
"        ping        = M_PING,\n"
"        call_16     = M_CALL_16,\n"
"        call_17     = M_CALL_17,\n"
"        eval        = M_EVAL,\n"
"        insert      = M_INSERT,\n"
"        replace     = M_REPLACE,\n"
"        delete      = M_DELETE,\n"
"        update      = M_UPDATE,\n"
"        upsert      = M_UPSERT,\n"
"        select      = M_SELECT,\n"
"        execute     = M_EXECUTE,\n"
"        prepare     = M_PREPARE,\n"
"        unprepare   = M_UNPREPARE,\n"
"        get         = M_GET,\n"
"        min         = M_MIN,\n"
"        max         = M_MAX,\n"
"        count       = M_COUNT,\n"
"        begin       = M_BEGIN,\n"
"        commit      = M_COMMIT,\n"
"        rollback    = M_ROLLBACK,\n"
"        inject      = M_INJECT,\n"
"    }\n"
"}\n"
"\n"
"function this_module.timeout(timeout, ...)\n"
"    if type(timeout) == 'table' then timeout = ... end\n"
"    if not timeout then return this_module end\n"
"    local function timed_connect(...)\n"
"        local uri, opts = parse_connect_params(...)\n"
"        if opts.wait_connected ~= false then opts.wait_connected = timeout end\n"
"        return connect(uri, opts)\n"
"    end\n"
"    return setmetatable({\n"
"        connect = timed_connect, new = timed_connect\n"
"    }, {__index = this_module})\n"
"end\n"
"\n"
"local function rollback()\n"
"    if rawget(box, 'rollback') ~= nil then\n"
"        -- roll back local transaction on error\n"
"        box.rollback()\n"
"    end\n"
"end\n"
"\n"
"local function handle_eval_result(status, ...)\n"
"    if not status then\n"
"        rollback()\n"
"        return box.error(E_PROC_LUA, (...))\n"
"    end\n"
"    local results = {...}\n"
"    for i = 1, select('#', ...) do\n"
"        if type(results[i]) == 'cdata' then\n"
"            results[i] = msgpack.decode(msgpack.encode(results[i]))\n"
"        end\n"
"    end\n"
"    return unpack(results)\n"
"end\n"
"\n"
"this_module.self = {\n"
"    ping = function() return true end,\n"
"    reload_schema = function() end,\n"
"    close = function() end,\n"
"    timeout = function(self) return self end,\n"
"    wait_connected = function(self) return true end,\n"
"    is_connected = function(self) return true end,\n"
"    call = function(__box, proc_name, args)\n"
"        check_remote_arg(__box, 'call')\n"
"        check_call_args(args)\n"
"        args = args or {}\n"
"        proc_name = tostring(proc_name)\n"
"        if box.ctl.is_recovery_finished() then\n"
"            local f = box.func[proc_name]\n"
"            if f ~= nil then\n"
"                return handle_eval_result(pcall(f.call, f, args))\n"
"            end\n"
"        end\n"
"        local status, proc, obj = pcall(box.internal.call_loadproc, proc_name)\n"
"        if not status then\n"
"            rollback()\n"
"            return error(proc) -- re-throw\n"
"        end\n"
"        if obj ~= nil then\n"
"            return handle_eval_result(pcall(proc, obj, unpack(args)))\n"
"        else\n"
"            return handle_eval_result(pcall(proc, unpack(args)))\n"
"        end\n"
"    end,\n"
"    eval = function(__box, expr, args)\n"
"        check_remote_arg(__box, 'eval')\n"
"        check_eval_args(args)\n"
"        args = args or {}\n"
"        local proc, errmsg = loadstring(expr)\n"
"        if not proc then\n"
"            proc, errmsg = loadstring(\"return \"..expr)\n"
"        end\n"
"        if not proc then\n"
"            rollback()\n"
"            return box.error(box.error.PROC_LUA, errmsg)\n"
"        end\n"
"        return handle_eval_result(pcall(proc, unpack(args)))\n"
"    end\n"
"}\n"
"\n"
"setmetatable(this_module.self, {\n"
"    __index = function(self, key)\n"
"        if key == 'space' then\n"
"            -- proxy self.space to box.space\n"
"            return require('box').space\n"
"        end\n"
"        return nil\n"
"    end\n"
"})\n"
"\n"
"return this_module\n"
""
;
