// Copyright © 2023-2025 Advanced Micro Devices, Inc.
// SPDX-License-Identifier: MIT

#include "../shim.bwd_kernel_dq.h"
#include <aotriton/_internal/triton_kernel.h>
#include <aotriton/_internal/kernel_cluster.h>
#include <aotriton/cpp_tune.h>
#include <string_view>
#ifndef NDEBUG
#include <iostream>
#endif

#define CURRENT_ENTRY_PUBLIC Autotune_bwd_kernel_dq__A10__F156

#define ARRAY_SIZE(array)  (sizeof(array) / sizeof(array[0]))

namespace { // Anonymous namespace

using namespace std::literals::string_view_literals;

#if AOTRITON_BUILD_FOR_TUNING
// PSels and Copts in JSON String
static const char* kernel_psels[] = {
  R"xyzw({"BLOCK_M": 16, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 16, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 16, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 16, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 32, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 32, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 32, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 64, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 64, "BLOCK_N": 64})xyzw"
};
static const char* kernel_copts[] = {
  R"xyzw({"waves_per_eu": 1, "num_warps": 4, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 2, "num_warps": 4, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 3, "num_warps": 4, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 4, "num_warps": 4, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 1, "num_warps": 4, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 2, "num_warps": 4, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 4, "num_warps": 4, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 3, "num_warps": 4, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 1, "num_warps": 4, "num_stages": 1})xyzw"
};
#endif

struct PerfFields {
  int16_t BLOCK_M;
    int16_t BLOCK_N;
};

static PerfFields image_perf_list [] = {
    { .BLOCK_M = 16, .BLOCK_N = 16 },
    { .BLOCK_M = 16, .BLOCK_N = 16 },
    { .BLOCK_M = 16, .BLOCK_N = 16 },
    { .BLOCK_M = 16, .BLOCK_N = 16 },
    { .BLOCK_M = 32, .BLOCK_N = 16 },
    { .BLOCK_M = 32, .BLOCK_N = 16 },
    { .BLOCK_M = 32, .BLOCK_N = 16 },
    { .BLOCK_M = 64, .BLOCK_N = 16 },
    { .BLOCK_M = 64, .BLOCK_N = 64 }
};

// u8R generates char8_t which is poorly supported almost everywhere.
constexpr pstring_view PACKAGE_PATH
#if defined(_WIN32)
{ LR"xyzw(amd-gfx120x/flash/bwd_kernel_dq/FONLY__＊fp16@16_224_3_T_F_0___gfx120x)xyzw" };
#else
{ R"xyzw(amd-gfx120x/flash/bwd_kernel_dq/FONLY__＊fp16@16_224_3_T_F_0___gfx120x)xyzw" };
#endif
constexpr std::string_view FUNC_NAME { R"xyzw(＊fp16@16_224_3_T_F_0)xyzw" };
constexpr std::string_view ARCH_NAME { R"xyzw(gfx1200)xyzw" };

// Checksum can be confirmed with `echo -n '<string>' | b2sum -l 64`
// For example:
//   $ echo -n 'amd-gfx110x/flash/attn_fwd/FONLY__^bf16@16,128,False,False,0,0___gfx1100__P__32_32_0_2_False__CO__wave3_warp2_stg1-Gpu-gfx1100' | b2sum -l 64
//   c4b51ee645d79580  -
static AOTRITON_NS::TritonKernelCompactMeta meta_list[] = {
    { 0x90fbc168u, 0x60c60de4u, 0, 138 }, // 90fbc16860c60de4 = b2sum -l 64 <<< amd-gfx120x/flash/bwd_kernel_dq/FONLY__＊fp16@16_224_3_T_F_0___gfx120x__P__16_16__CO__wave1_warp4_stg1--Arch_gfx1200
    { 0xda2a1002u, 0xd91aeae0u, 0, 80 }, // da2a1002d91aeae0 = b2sum -l 64 <<< amd-gfx120x/flash/bwd_kernel_dq/FONLY__＊fp16@16_224_3_T_F_0___gfx120x__P__16_16__CO__wave2_warp4_stg1--Arch_gfx1200
    { 0xb088eb74u, 0xda430864u, 0, 155 }, // b088eb74da430864 = b2sum -l 64 <<< amd-gfx120x/flash/bwd_kernel_dq/FONLY__＊fp16@16_224_3_T_F_0___gfx120x__P__16_16__CO__wave3_warp4_stg1--Arch_gfx1200
    { 0x2a3a3925u, 0x288fff72u, 0, 103 }, // 2a3a3925288fff72 = b2sum -l 64 <<< amd-gfx120x/flash/bwd_kernel_dq/FONLY__＊fp16@16_224_3_T_F_0___gfx120x__P__16_16__CO__wave4_warp4_stg1--Arch_gfx1200
    { 0x8e942e22u, 0xd891cb0cu, 40, 138 }, // 8e942e22d891cb0c = b2sum -l 64 <<< amd-gfx120x/flash/bwd_kernel_dq/FONLY__＊fp16@16_224_3_T_F_0___gfx120x__P__32_16__CO__wave1_warp4_stg1--Arch_gfx1200
    { 0x42164113u, 0x33acead2u, 40, 80 }, // 4216411333acead2 = b2sum -l 64 <<< amd-gfx120x/flash/bwd_kernel_dq/FONLY__＊fp16@16_224_3_T_F_0___gfx120x__P__32_16__CO__wave2_warp4_stg1--Arch_gfx1200
    { 0x08393bb7u, 0x057732f3u, 40, 103 }, // 08393bb7057732f3 = b2sum -l 64 <<< amd-gfx120x/flash/bwd_kernel_dq/FONLY__＊fp16@16_224_3_T_F_0___gfx120x__P__32_16__CO__wave4_warp4_stg1--Arch_gfx1200
    { 0x4247ae3au, 0x19659a2au, 120, 155 }, // 4247ae3a19659a2a = b2sum -l 64 <<< amd-gfx120x/flash/bwd_kernel_dq/FONLY__＊fp16@16_224_3_T_F_0___gfx120x__P__64_16__CO__wave3_warp4_stg1--Arch_gfx1200
    { 0x679d3398u, 0x660824bcu, 132, 138 }, // 679d3398660824bc = b2sum -l 64 <<< amd-gfx120x/flash/bwd_kernel_dq/FONLY__＊fp16@16_224_3_T_F_0___gfx120x__P__64_64__CO__wave1_warp4_stg1--Arch_gfx1200
};

static constexpr int kTotalNumKernels = ARRAY_SIZE(meta_list);

using AOTRITON_NS::v3::flash::autotune::bwd_kernel_dq_packed_string;

static AOTRITON_NS::TritonKernelCluster<kTotalNumKernels>
kernel_cluster(meta_list,
               bwd_kernel_dq_packed_string);

static int8_t lut[1][10][10] =
{
// GPU gfx1200_mod0
{{0,2,2,1,0,1,1,1,2,2},
 {6,6,6,6,6,6,6,6,6,6},
 {3,3,3,3,3,3,3,3,3,3},
 {3,5,6,6,3,3,3,3,3,3},
 {5,6,4,5,4,6,5,6,3,3},
 {5,4,5,4,5,5,5,3,3,3},
 {3,5,4,7,7,7,5,5,5,5},
 {3,4,4,7,7,4,4,5,5,5},
 {3,5,4,7,4,5,5,8,8,8},
 {3,7,7,4,4,5,8,8,8,8}}
// End of GPU gfx1200_mod0
}
;

}; // End of anonymous namespace

namespace AOTRITON_NS::v3::flash::autotune {

// using AOTRITON_NS::v2::flash::BwdKernelDqContext;

void CURRENT_ENTRY_PUBLIC(BwdKernelDqContext& context, int mod_number) {
#if AOTRITON_BUILD_FOR_TUNING
    int preferred_index = context._has_preferred_kernel;
    context._total_number_of_kernels = kTotalNumKernels;
#ifndef NDEBUG
    std::cerr << "Autotune_bwd_kernel_dq__A10__F156 "
              << "kTotalNumKernels = " << kTotalNumKernels << " "
              << "_has_preferred_kernel = " << preferred_index << " "
              << std::endl;
#endif
    if (preferred_index != -1) {
        if (preferred_index >= kTotalNumKernels)
            return ;
        context.kernel_on_device = kernel_cluster.get(preferred_index);
        context.pp_args_index = 5;
        context.package_path = PACKAGE_PATH;
        context.func_name = FUNC_NAME;
        context.arch_name = ARCH_NAME;
        context._preferred_kernel_psels = kernel_psels[preferred_index];
        context._preferred_kernel_copts = kernel_copts[preferred_index];
        const auto& perf = image_perf_list[preferred_index];
        context.BLOCK_M = perf.BLOCK_M;
    context.BLOCK_N = perf.BLOCK_N;
        return ;
    }
#endif
    auto kernel_index = bwd_kernel_dq__lut_lambda__0(*context.params, mod_number, lut);
    if (kernel_index < 0) {
      return ;
    }
    context.kernel_on_device = kernel_cluster.get(kernel_index);
    context.pp_args_index = 5;
    context.package_path = PACKAGE_PATH;
    context.func_name = FUNC_NAME;
    context.arch_name = ARCH_NAME;
#ifndef NDEBUG
    std::cerr << __FILE__ << " kernel_index = " << int(kernel_index) << std::endl;
#endif
    const auto& perf = image_perf_list[kernel_index];
    context.BLOCK_M = perf.BLOCK_M;
    context.BLOCK_N = perf.BLOCK_N;
}

#undef CURRENT_ENTRY_PUBLIC
}

// Human-readable Signature 
// Q = "*fp16:16"
// sm_scale = "fp32"
// L = "*fp32:16"
// D = "*fp32:16"
// stride_bz = 0
// stride_dbz = 0
// num_head_q = "i32"
// cu_seqlens_q = "*i32:16"
// num_seqlens = "i32"
// dropout_p = "fp32"
// philox_seed_ptr = "*u64"
// philox_offset1 = "*u64"
// philox_offset2 = "u64"
// Window_left = "i32"
// BLOCK_DMODEL = 224
// CAUSAL_TYPE = 3
// ENABLE_DROPOUT = True
// PADDED_HEAD = False
// BIAS_TYPE = 0

// SELECT * FROM FLASH$bwd_kernel_dq WHERE gpu IN ('gfx1201_mod0') AND inputs$Q_dtype = 'torch.float16' AND inputs$BLOCK_DMODEL = 224 AND inputs$CAUSAL_TYPE = 3 AND inputs$ENABLE_DROPOUT = True AND inputs$PADDED_HEAD = False AND inputs$BIAS_TYPE = 0

// vim: set fileencoding=utf-8

