// Copyright © 2023-2025 Advanced Micro Devices, Inc.
// SPDX-License-Identifier: MIT

#include "../shim.bwd_kernel_dk_dv.h"
#include <aotriton/_internal/triton_kernel.h>
#include <aotriton/_internal/kernel_cluster.h>
#include <aotriton/cpp_tune.h>
#include <string_view>
#ifndef NDEBUG
#include <iostream>
#endif

#define CURRENT_ENTRY_PUBLIC Autotune_bwd_kernel_dk_dv__A4__F102

#define ARRAY_SIZE(array)  (sizeof(array) / sizeof(array[0]))

namespace { // Anonymous namespace

using namespace std::literals::string_view_literals;

#if AOTRITON_BUILD_FOR_TUNING
// PSels and Copts in JSON String
static const char* kernel_psels[] = {
  R"xyzw({"BLOCK_M": 16, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 16, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 16, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 16, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 16, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 32, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 32, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 32, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 64, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 64, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 64, "BLOCK_N": 16})xyzw"
};
static const char* kernel_copts[] = {
  R"xyzw({"waves_per_eu": 1, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 1, "num_warps": 4, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 3, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 4, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 4, "num_warps": 4, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 1, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 2, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 4, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 1, "num_warps": 4, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 2, "num_warps": 4, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 4, "num_warps": 4, "num_stages": 1})xyzw"
};
#endif

struct PerfFields {
  int16_t BLOCK_M;
    int16_t BLOCK_N;
};

static PerfFields image_perf_list [] = {
    { .BLOCK_M = 16, .BLOCK_N = 16 },
    { .BLOCK_M = 16, .BLOCK_N = 16 },
    { .BLOCK_M = 16, .BLOCK_N = 16 },
    { .BLOCK_M = 16, .BLOCK_N = 16 },
    { .BLOCK_M = 16, .BLOCK_N = 16 },
    { .BLOCK_M = 32, .BLOCK_N = 16 },
    { .BLOCK_M = 32, .BLOCK_N = 16 },
    { .BLOCK_M = 32, .BLOCK_N = 16 },
    { .BLOCK_M = 64, .BLOCK_N = 16 },
    { .BLOCK_M = 64, .BLOCK_N = 16 },
    { .BLOCK_M = 64, .BLOCK_N = 16 }
};

// u8R generates char8_t which is poorly supported almost everywhere.
constexpr pstring_view PACKAGE_PATH
#if defined(_WIN32)
{ LR"xyzw(amd-gfx11xx/flash/bwd_kernel_dk_dv/FONLY__＊fp16@16_128_0_T_T_0___gfx11xx)xyzw" };
#else
{ R"xyzw(amd-gfx11xx/flash/bwd_kernel_dk_dv/FONLY__＊fp16@16_128_0_T_T_0___gfx11xx)xyzw" };
#endif
constexpr std::string_view FUNC_NAME { R"xyzw(＊fp16@16_128_0_T_T_0)xyzw" };
constexpr std::string_view ARCH_NAME { R"xyzw(gfx1103)xyzw" };

// Checksum can be confirmed with `echo -n '<string>' | b2sum -l 64`
// For example:
//   $ echo -n 'amd-gfx110x/flash/attn_fwd/FONLY__^bf16@16,128,False,False,0,0___gfx1100__P__32_32_0_2_False__CO__wave3_warp2_stg1-Gpu-gfx1100' | b2sum -l 64
//   c4b51ee645d79580  -
static AOTRITON_NS::TritonKernelCompactMeta meta_list[] = {
    { 0x29b26585u, 0xd592d870u, 0, 29 }, // 29b26585d592d870 = b2sum -l 64 <<< amd-gfx11xx/flash/bwd_kernel_dk_dv/FONLY__＊fp16@16_128_0_T_T_0___gfx11xx__P__16_16__CO__wave1_warp2_stg1--Arch_gfx1103
    { 0x1e07058au, 0xe0c8217au, 0, 109 }, // 1e07058ae0c8217a = b2sum -l 64 <<< amd-gfx11xx/flash/bwd_kernel_dk_dv/FONLY__＊fp16@16_128_0_T_T_0___gfx11xx__P__16_16__CO__wave1_warp4_stg1--Arch_gfx1103
    { 0x17865fd5u, 0xdd92b17au, 0, 46 }, // 17865fd5dd92b17a = b2sum -l 64 <<< amd-gfx11xx/flash/bwd_kernel_dk_dv/FONLY__＊fp16@16_128_0_T_T_0___gfx11xx__P__16_16__CO__wave3_warp2_stg1--Arch_gfx1103
    { 0xb2ba5c77u, 0xdbc12d66u, 0, 63 }, // b2ba5c77dbc12d66 = b2sum -l 64 <<< amd-gfx11xx/flash/bwd_kernel_dk_dv/FONLY__＊fp16@16_128_0_T_T_0___gfx11xx__P__16_16__CO__wave4_warp2_stg1--Arch_gfx1103
    { 0x9dae45a9u, 0x3d9c9333u, 0, 143 }, // 9dae45a93d9c9333 = b2sum -l 64 <<< amd-gfx11xx/flash/bwd_kernel_dk_dv/FONLY__＊fp16@16_128_0_T_T_0___gfx11xx__P__16_16__CO__wave4_warp4_stg1--Arch_gfx1103
    { 0x78670547u, 0x55a62eb7u, 23, 29 }, // 7867054755a62eb7 = b2sum -l 64 <<< amd-gfx11xx/flash/bwd_kernel_dk_dv/FONLY__＊fp16@16_128_0_T_T_0___gfx11xx__P__32_16__CO__wave1_warp2_stg1--Arch_gfx1103
    { 0x3419ffcdu, 0xfa22f542u, 23, 6 }, // 3419ffcdfa22f542 = b2sum -l 64 <<< amd-gfx11xx/flash/bwd_kernel_dk_dv/FONLY__＊fp16@16_128_0_T_T_0___gfx11xx__P__32_16__CO__wave2_warp2_stg1--Arch_gfx1103
    { 0x809e9288u, 0x624028c2u, 23, 63 }, // 809e9288624028c2 = b2sum -l 64 <<< amd-gfx11xx/flash/bwd_kernel_dk_dv/FONLY__＊fp16@16_128_0_T_T_0___gfx11xx__P__32_16__CO__wave4_warp2_stg1--Arch_gfx1103
    { 0x6feb8217u, 0x2cdce6fcu, 103, 109 }, // 6feb82172cdce6fc = b2sum -l 64 <<< amd-gfx11xx/flash/bwd_kernel_dk_dv/FONLY__＊fp16@16_128_0_T_T_0___gfx11xx__P__64_16__CO__wave1_warp4_stg1--Arch_gfx1103
    { 0x64d151b2u, 0x55c04263u, 103, 126 }, // 64d151b255c04263 = b2sum -l 64 <<< amd-gfx11xx/flash/bwd_kernel_dk_dv/FONLY__＊fp16@16_128_0_T_T_0___gfx11xx__P__64_16__CO__wave2_warp4_stg1--Arch_gfx1103
    { 0xc89423ffu, 0x28df61fdu, 103, 143 }, // c89423ff28df61fd = b2sum -l 64 <<< amd-gfx11xx/flash/bwd_kernel_dk_dv/FONLY__＊fp16@16_128_0_T_T_0___gfx11xx__P__64_16__CO__wave4_warp4_stg1--Arch_gfx1103
};

static constexpr int kTotalNumKernels = ARRAY_SIZE(meta_list);

using AOTRITON_NS::v3::flash::autotune::bwd_kernel_dk_dv_packed_string;

static AOTRITON_NS::TritonKernelCluster<kTotalNumKernels>
kernel_cluster(meta_list,
               bwd_kernel_dk_dv_packed_string);

static int8_t lut[1][10][10] =
{
// GPU gfx1103_mod0
{{ 4, 3, 4, 7, 7, 7, 7, 5, 0, 0},
 { 4, 2, 4, 1, 2, 3, 0, 0, 0, 0},
 { 1, 2, 1, 1, 0, 0, 0, 0, 0, 3},
 { 1, 3, 1, 1, 5, 0, 0, 6, 6, 0},
 { 1, 6,10,10, 5, 6, 6, 6, 6, 6},
 { 1, 5,10,10,10,10,10,10, 6, 6},
 {10,10,10,10,10,10,10,10,10,10},
 {10,10,10,10,10, 9,10,10,10,10},
 {10,10,10, 9,10,10,10,10,10,10},
 {10,10,10,10, 9,10, 8, 9, 8, 8}}
// End of GPU gfx1103_mod0
}
;

}; // End of anonymous namespace

namespace AOTRITON_NS::v3::flash::autotune {

// using AOTRITON_NS::v2::flash::BwdKernelDkDvContext;

void CURRENT_ENTRY_PUBLIC(BwdKernelDkDvContext& context, int mod_number) {
#if AOTRITON_BUILD_FOR_TUNING
    int preferred_index = context._has_preferred_kernel;
    context._total_number_of_kernels = kTotalNumKernels;
#ifndef NDEBUG
    std::cerr << "Autotune_bwd_kernel_dk_dv__A4__F102 "
              << "kTotalNumKernels = " << kTotalNumKernels << " "
              << "_has_preferred_kernel = " << preferred_index << " "
              << std::endl;
#endif
    if (preferred_index != -1) {
        if (preferred_index >= kTotalNumKernels)
            return ;
        context.kernel_on_device = kernel_cluster.get(preferred_index);
        context.pp_args_index = 2;
        context.package_path = PACKAGE_PATH;
        context.func_name = FUNC_NAME;
        context.arch_name = ARCH_NAME;
        context._preferred_kernel_psels = kernel_psels[preferred_index];
        context._preferred_kernel_copts = kernel_copts[preferred_index];
        const auto& perf = image_perf_list[preferred_index];
        context.BLOCK_M = perf.BLOCK_M;
    context.BLOCK_N = perf.BLOCK_N;
        return ;
    }
#endif
    auto kernel_index = bwd_kernel_dk_dv__lut_lambda__0(*context.params, mod_number, lut);
    if (kernel_index < 0) {
      return ;
    }
    context.kernel_on_device = kernel_cluster.get(kernel_index);
    context.pp_args_index = 2;
    context.package_path = PACKAGE_PATH;
    context.func_name = FUNC_NAME;
    context.arch_name = ARCH_NAME;
#ifndef NDEBUG
    std::cerr << __FILE__ << " kernel_index = " << int(kernel_index) << std::endl;
#endif
    const auto& perf = image_perf_list[kernel_index];
    context.BLOCK_M = perf.BLOCK_M;
    context.BLOCK_N = perf.BLOCK_N;
}

#undef CURRENT_ENTRY_PUBLIC
}

// Human-readable Signature 
// Q = "*fp16:16"
// sm_scale = "fp32"
// L = "*fp32:16"
// D = "*fp32:16"
// stride_bz = 0
// num_head_q = "i32"
// cu_seqlens_q = "*i32:16"
// num_seqlens = "i32"
// dropout_p = "fp32"
// philox_seed_ptr = "*u64"
// philox_offset1 = "*u64"
// philox_offset2 = "u64"
// Window_left = 0
// BLOCK_DMODEL = 128
// CAUSAL_TYPE = 0
// ENABLE_DROPOUT = True
// PADDED_HEAD = True
// BIAS_TYPE = 0

// SELECT * FROM FLASH$bwd_kernel_dk_dv WHERE gpu IN ('gfx1100_mod0') AND inputs$Q_dtype = 'torch.float16' AND inputs$BLOCK_DMODEL = 128 AND inputs$CAUSAL_TYPE = 0 AND inputs$ENABLE_DROPOUT = True AND inputs$PADDED_HEAD = False AND inputs$BIAS_TYPE = 0

// vim: set fileencoding=utf-8

