use crate::App;
use crate::gobject_models::GSidebarSelection;
use gdk4::Rectangle;
use glib::{Enum, Properties, SignalHandlerId, subclass::*};
use gtk4::{
    Accessible, Buildable, CompositeTemplate, ConstraintTarget, ListBox, ListBoxRow, PopoverMenu, Widget, prelude::*,
    subclass::prelude::*,
};
use libadwaita::{Bin, subclass::prelude::*};
use once_cell::sync::Lazy;
use std::cell::Cell;

#[derive(Clone, Default, Debug, Eq, PartialEq, Copy, Enum)]
#[repr(u32)]
#[enum_type(name = "GSidebarSpecialRowType")]
pub enum SidebarSpecialRowType {
    #[default]
    All,
    Today,
}

mod imp {
    use super::*;

    #[derive(Debug, CompositeTemplate, Default, Properties)]
    #[properties(wrapper_type = super::SpecialSidebarRows)]
    #[template(file = "data/resources/ui_templates/sidebar/special_sidebar_rows.blp")]
    pub struct SpecialSidebarRows {
        #[template_child]
        pub list: TemplateChild<ListBox>,
        #[template_child]
        pub all_row: TemplateChild<ListBoxRow>,
        #[template_child]
        pub today_row: TemplateChild<ListBoxRow>,
        #[template_child]
        pub all_popover: TemplateChild<PopoverMenu>,
        #[template_child]
        pub today_popover: TemplateChild<PopoverMenu>,

        #[property(get, set, name = "all-item-count")]
        pub all_item_count: Cell<u32>,

        #[property(get, set, name = "today-item-count")]
        pub today_item_count: Cell<u32>,
    }

    #[glib::object_subclass]
    impl ObjectSubclass for SpecialSidebarRows {
        const NAME: &'static str = "SpecialSidebarRows";
        type ParentType = Bin;
        type Type = super::SpecialSidebarRows;

        fn class_init(klass: &mut Self::Class) {
            klass.bind_template();
            klass.bind_template_callbacks();
        }

        fn instance_init(obj: &InitializingObject<Self>) {
            obj.init_template();
        }
    }

    #[glib::derived_properties]
    impl ObjectImpl for SpecialSidebarRows {
        fn signals() -> &'static [Signal] {
            static SIGNALS: Lazy<Vec<Signal>> = Lazy::new(|| {
                vec![
                    Signal::builder("activated")
                        .param_types([
                            super::SpecialSidebarRows::static_type(),
                            SidebarSpecialRowType::static_type(),
                        ])
                        .build(),
                    Signal::builder("selection-changed")
                        .param_types([
                            super::SpecialSidebarRows::static_type(),
                            GSidebarSelection::static_type(),
                        ])
                        .build(),
                ]
            });

            SIGNALS.as_ref()
        }
    }

    impl WidgetImpl for SpecialSidebarRows {}

    impl BinImpl for SpecialSidebarRows {}

    #[gtk4::template_callbacks]
    impl SpecialSidebarRows {
        #[template_callback]
        fn on_all_clicked(&self, times: i32, x: f64, y: f64) {
            if times != 1 {
                return;
            }

            if App::default().is_offline() {
                return;
            }

            let position = Rectangle::new(x as i32, y as i32, 0, 0);
            self.all_popover.set_pointing_to(Some(&position));
            self.all_popover.popup();
        }

        #[template_callback]
        fn on_all_pressed(&self, x: f64, y: f64) {
            if App::default().is_offline() {
                return;
            }

            let position = Rectangle::new(x as i32, y as i32, 0, 0);
            self.all_popover.set_pointing_to(Some(&position));
            self.all_popover.popup();
        }

        #[template_callback]
        fn on_today_clicked(&self, times: i32, x: f64, y: f64) {
            if times != 1 {
                return;
            }

            if App::default().is_offline() {
                return;
            }

            let position = Rectangle::new(x as i32, y as i32, 0, 0);
            self.today_popover.set_pointing_to(Some(&position));
            self.today_popover.popup();
        }

        #[template_callback]
        fn on_today_pressed(&self, x: f64, y: f64) {
            if App::default().is_offline() {
                return;
            }

            let position = Rectangle::new(x as i32, y as i32, 0, 0);
            self.today_popover.set_pointing_to(Some(&position));
            self.today_popover.popup();
        }

        #[template_callback]
        fn on_row_selected(&self, row: Option<&ListBoxRow>) {
            let obj = self.obj();

            if let Some(row) = row {
                let selection = if row == &*self.all_row {
                    GSidebarSelection::all(self.all_item_count.get())
                } else {
                    GSidebarSelection::today(self.today_item_count.get())
                };
                obj.emit_by_name::<()>("selection-changed", &[&*obj, &selection]);
            }
        }

        #[template_callback]
        fn format_item_count(&self, count: u32) -> String {
            count.to_string()
        }

        #[template_callback]
        fn is_greater_than_zero(&self, count: u32) -> bool {
            count > 0
        }
    }
}

glib::wrapper! {
    pub struct SpecialSidebarRows(ObjectSubclass<imp::SpecialSidebarRows>)
        @extends Widget, Bin,
        @implements Accessible, Buildable, ConstraintTarget;
}

impl Default for SpecialSidebarRows {
    fn default() -> Self {
        glib::Object::new::<Self>()
    }
}

impl SpecialSidebarRows {
    pub fn clear_selection(&self) {
        self.imp().list.unselect_all();
    }

    pub fn select(&self, _type: SidebarSpecialRowType) {
        let imp = self.imp();
        let row = match _type {
            SidebarSpecialRowType::All => &*imp.all_row,
            SidebarSpecialRowType::Today => &*imp.today_row,
        };
        imp.list.select_row(Some(row));
    }

    pub fn activate(&self, _type: SidebarSpecialRowType) {
        self.emit_by_name::<()>("activated", &[self, &_type]);
    }

    pub fn connect_activated<F: Fn(&Self, SidebarSpecialRowType) + 'static>(&self, f: F) -> SignalHandlerId {
        self.connect_local("activated", false, move |args| {
            let all_articles = args[1]
                .get::<Self>()
                .expect("The value needs to be of type `SpecialSidebarRows`");
            let _type = args[2]
                .get::<SidebarSpecialRowType>()
                .expect("The value needs to be of type `SidebarSpecialRowType`");
            f(&all_articles, _type);
            None
        })
    }
}
