use crate::i18n::i18n;
use crate::themes::NewsflashTheme;
use chrono::TimeDelta;
use glib::{Enum, Properties, prelude::*, subclass::prelude::*};
use serde::{Deserialize, Serialize};
use std::cell::Cell;
use std::fmt;

#[derive(Clone, Default, Debug, Serialize, Deserialize, Eq, PartialEq, Copy, Enum)]
#[repr(u32)]
#[enum_type(name = "NewsFlashSyncTime")]
pub enum SyncIntervalType {
    Never,
    #[default]
    Predefined,
    Custom, // Seconds
}

#[derive(Default, Debug, Clone, Copy, Serialize, Deserialize, Eq, PartialEq, Enum)]
#[repr(u32)]
#[enum_type(name = "SyncInterval")]
pub enum PredefinedSyncInterval {
    #[default]
    QuaterHour,
    HalfHour,
    Hour,
    TwoHour,
}

impl fmt::Display for PredefinedSyncInterval {
    fn fmt(&self, f: &mut fmt::Formatter<'_>) -> fmt::Result {
        match self {
            Self::QuaterHour => write!(f, "{}", i18n("15 Minutes")),
            Self::HalfHour => write!(f, "{}", i18n("30 Minutes")),
            Self::Hour => write!(f, "{}", i18n("1 Hour")),
            Self::TwoHour => write!(f, "{}", i18n("2 Hours")),
        }
    }
}

impl PredefinedSyncInterval {
    pub fn as_minutes(&self) -> Option<u32> {
        match self {
            Self::QuaterHour => Some(15),
            Self::HalfHour => Some(30),
            Self::Hour => Some(60),
            Self::TwoHour => Some(120),
        }
    }

    pub fn as_seconds(&self) -> Option<u32> {
        self.as_minutes().map(|m| m * 60)
    }

    pub fn from_u32(v: u32) -> Self {
        match v {
            0 => Self::QuaterHour,
            1 => Self::HalfHour,
            2 => Self::Hour,
            3 => Self::TwoHour,
            _ => Self::QuaterHour,
        }
    }

    pub fn as_u32(&self) -> u32 {
        match self {
            Self::QuaterHour => 0,
            Self::HalfHour => 1,
            Self::Hour => 2,
            Self::TwoHour => 3,
        }
    }
}

#[derive(Debug, Clone, Copy, Serialize, Deserialize, Eq, PartialEq, Enum)]
#[repr(u32)]
#[enum_type(name = "KeepArticlesDuration")]
pub enum KeepArticlesDuration {
    Forever,
    OneYear,
    SixMonths,
    OneMonth,
    OneWeek,
}

impl fmt::Display for KeepArticlesDuration {
    fn fmt(&self, f: &mut fmt::Formatter<'_>) -> fmt::Result {
        match self {
            Self::Forever => write!(f, "{}", i18n("Forever")),
            Self::OneYear => write!(f, "{}", i18n("One Year")),
            Self::SixMonths => write!(f, "{}", i18n("6 Months")),
            Self::OneMonth => write!(f, "{}", i18n("One Month")),
            Self::OneWeek => write!(f, "{}", i18n("One Week")),
        }
    }
}

impl KeepArticlesDuration {
    pub fn as_duration(&self) -> Option<TimeDelta> {
        match self {
            Self::Forever => None,
            Self::OneYear => Some(TimeDelta::try_days(365).unwrap()),
            Self::SixMonths => Some(TimeDelta::try_days(182).unwrap()),
            Self::OneMonth => Some(TimeDelta::try_days(30).unwrap()),
            Self::OneWeek => Some(TimeDelta::try_days(7).unwrap()),
        }
    }

    pub fn from_duration(duration: Option<TimeDelta>) -> Self {
        if let Some(duration) = duration {
            match duration.num_days() {
                365 => Self::OneYear,
                182 => Self::SixMonths,
                30 => Self::OneMonth,
                7 => Self::OneWeek,

                _ => Self::Forever,
            }
        } else {
            KeepArticlesDuration::Forever
        }
    }

    pub fn as_u32(&self) -> u32 {
        match self {
            Self::Forever => 0,
            Self::OneYear => 1,
            Self::SixMonths => 2,
            Self::OneMonth => 3,
            Self::OneWeek => 4,
        }
    }

    pub fn from_u32(i: u32) -> Self {
        match i {
            0 => Self::Forever,
            1 => Self::OneYear,
            2 => Self::SixMonths,
            3 => Self::OneMonth,
            4 => Self::OneWeek,

            _ => Self::Forever,
        }
    }
}

mod imp {
    use super::*;

    const fn _default_true() -> Cell<bool> {
        Cell::new(true)
    }

    #[derive(Default, Debug, Serialize, Deserialize, Properties)]
    #[properties(wrapper_type = super::GeneralSettings)]
    pub struct GeneralSettings {
        #[property(get, set, name = "keep-running-in-background", default_value = false)]
        #[serde(default)]
        pub keep_running_in_background: Cell<bool>,

        #[property(get, set, default_value = false)]
        #[serde(default)]
        pub autostart: Cell<bool>,

        #[property(get, set, name = "sync-on-startup", default_value = false)]
        #[serde(default)]
        pub sync_on_startup: Cell<bool>,

        #[property(get, set, name = "sync-on-metered", default_value = true)]
        #[serde(default = "_default_true")]
        pub sync_on_metered: Cell<bool>,

        #[property(get, set, name = "sync-type", builder(SyncIntervalType::Predefined))]
        #[serde(default)]
        pub sync_type: Cell<SyncIntervalType>,

        #[property(
            get,
            set,
            name = "predefined-sync-interval",
            builder(PredefinedSyncInterval::QuaterHour)
        )]
        #[serde(default)]
        pub predefined_interval: Cell<PredefinedSyncInterval>,

        #[property(get, set, name = "custom-sync-interval", default = 80)]
        #[serde(default)]
        pub custom_interval: Cell<u32>,

        #[property(get, set, builder(NewsflashTheme::Default))]
        #[serde(default)]
        pub theme: Cell<NewsflashTheme>,
    }

    #[glib::object_subclass]
    impl ObjectSubclass for GeneralSettings {
        const NAME: &'static str = "GeneralSettings";
        type Type = super::GeneralSettings;
    }

    #[glib::derived_properties]
    impl ObjectImpl for GeneralSettings {}
}

glib::wrapper! {
    pub struct GeneralSettings(ObjectSubclass<imp::GeneralSettings>);
}

impl Serialize for GeneralSettings {
    fn serialize<S>(&self, serializer: S) -> Result<S::Ok, S::Error>
    where
        S: serde::Serializer,
    {
        self.imp().serialize(serializer)
    }
}

impl<'de> Deserialize<'de> for GeneralSettings {
    fn deserialize<D>(deserializer: D) -> Result<Self, D::Error>
    where
        D: serde::Deserializer<'de>,
    {
        let inner = imp::GeneralSettings::deserialize(deserializer)?;
        Ok(inner.into())
    }
}

impl From<imp::GeneralSettings> for GeneralSettings {
    fn from(inner: imp::GeneralSettings) -> Self {
        glib::Object::builder()
            .property("keep-running-in-background", inner.keep_running_in_background.get())
            .property("autostart", inner.autostart.get())
            .property("sync-on-startup", inner.sync_on_startup.get())
            .property("sync-on-metered", inner.sync_on_metered.get())
            .property("sync-type", inner.sync_type.get())
            .property("predefined-sync-interval", inner.predefined_interval.get())
            .property("custom-sync-interval", inner.custom_interval.get())
            .property("theme", inner.theme.get())
            .build()
    }
}

impl Default for GeneralSettings {
    fn default() -> Self {
        imp::GeneralSettings::default().into()
    }
}
