// Copyright 2019 Drone IO, Inc.
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//      http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

package license

import (
	"context"
	"time"

	"github.com/drone/drone/core"
)

// NewService returns a new License service.
func NewService(
	users core.UserStore,
	repos core.RepositoryStore,
	builds core.BuildStore,
	license *core.License,
) core.LicenseService {
	return &service{
		users:   users,
		repos:   repos,
		builds:  builds,
		license: license,
	}
}

type service struct {
	users   core.UserStore
	repos   core.RepositoryStore
	builds  core.BuildStore
	license *core.License
}

func (s *service) Exceeded(ctx context.Context) (bool, error) {
	if limit := s.license.Builds; limit > 0 {
		count, _ := s.builds.Count(ctx)
		if count > limit {
			return true, core.ErrBuildLimit
		}
	}
	if limit := s.license.Users; limit > 0 {
		count, _ := s.users.Count(ctx)
		if count > limit {
			return true, core.ErrUserLimit
		}
	}
	if limit := s.license.Repos; limit > 0 {
		count, _ := s.repos.Count(ctx)
		if count > limit {
			return true, core.ErrRepoLimit
		}
	}
	return false, nil
}

func (s *service) Expired(ctx context.Context) bool {
	return s.license.Expired()
}

func (s *service) Expires(ctx context.Context) time.Time {
	return s.license.Expires
}
