/*
 * Copyright (C) 2014-2026 CZ.NIC
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * In addition, as a special exception, the copyright holders give
 * permission to link the code of portions of this program with the
 * OpenSSL library under certain conditions as described in each
 * individual source file, and distribute linked combinations including
 * the two.
 */

#pragma once

#include <QObject>

/*
 * wsdl2h -s -o dmBaseTypes.h dmBaseTypes.xsd
 * wsdl2h -s -o dbTypes.h dbTypes.xsd
 */

namespace Isds {

	/*!
	 * @brief Provides enumeration types based on dbTypes.xsd .
	 */
	class Type : public QObject {
		Q_OBJECT

	private:
		/*!
		 * @brief Private constructor.
		 */
		Type(QObject *parent = Q_NULLPTR);

	public:
		/*!
		 * @brief Nullable bool.
		 *
		 * @note Should NULL value be treated as false?
		 */
		enum NilBool {
			BOOL_NULL = -1, /*!< Convenience value, converted from/to NULL. */
			BOOL_FALSE = 0,
			BOOL_TRUE = 1
		};

		/*!
		 * @brief Data box type.
		 *
		 * @note Defined in dbTypes.xsd. Described in
		 *     pril_3/WS_ISDS_Sprava_datovych_schranek.pdf
		 *     (section 2.1 CreateDataBox).
		 */
		enum DbType {
			BT_NULL = -1, /*!< Convenience value, converted from/to NULL. */
			BT_SYSTEM = 0, /*!<
			                * This value is not listed in dbTypes.xsd but is mentioned in
			                * pril_2/WS_ISDS_Manipulace_s_datovymi_zpravami.pdf
			                * (section 2.4.1 MessageDownload).
			                *
			                * When used in full-text search then it is converted to "ALL".
			                */
			BT_OVM = 10, /*!< Public authority. */
			BT_OVM_NOTAR = 11, /* This type has been replaced with OVM_PFO. */
			BT_OVM_EXEKUT = 12, /* This type has been replaced with OVM_PFO. */
			BT_OVM_REQ = 13,
			BT_OVM_FO = 14,
			BT_OVM_PFO = 15,
			BT_OVM_PO = 16,
			BT_PO = 20,
			BT_PO_ZAK = 21, /* This type has been replaced with PO. */
			BT_PO_REQ = 22,
			BT_PFO = 30,
			BT_PFO_ADVOK = 31,
			BT_PFO_DANPOR = 32,
			BT_PFO_INSSPR = 33,
			BT_PFO_AUDITOR = 34,
			BT_PFO_ZNALEC = 35,
			BT_PFO_TLUMOCNIK = 36,
			BT_PFO_ARCH = 37,
			BT_PFO_AIAT = 38,
			BT_PFO_AZI = 39,
			BT_PFO_REQ = 50,
			BT_FO = 40,

			BT_OVM_MAIN /* Special value for full-text data box search "OVM_MAIN". */
		};

		/*!
		 * @brief Data box accessibility state.
		 *
		 * @note Described in pril_3/WS_ISDS_Sprava_datovych_schranek.pdf
		 *     (appendix 4).
		 */
		enum DbState {
			BS_ERROR = 0, /* Error value, see documentation. */ /* Also converted ftom/to NULL. */
			BS_ACCESSIBLE = 1,
			BS_TEMP_INACCESSIBLE = 2,
			BS_NOT_YET_ACCESSIBLE = 3,
			BS_PERM_INACCESSIBLE = 4,
			BS_REMOVED = 5,
			BS_TEMP_UNACCESSIBLE_LAW = 6
		};

		/*!
		 * @brief User permissions.
		 *
		 * @note Described in
		 *     pril_2/WS_ISDS_Manipulace_s_datovymi_zpravami.pdf (section 1.4),
		 *     pril_3/WS_ISDS_Sprava_datovych_schranek.pdf (section 2.4 AddDataBoxUser)
		 */
		enum Privilege {
			PRIVIL_NONE = 0x00, /* Added for convenience. */
			PRIVIL_READ_NON_PERSONAL = 0x01,
			PRIVIL_READ_ALL = 0x02,
			PRIVIL_CREATE_DM = 0x04,
			PRIVIL_VIEW_INFO = 0x08,
			PRIVIL_SEARCH_DB = 0x10,
			PRIVIL_OWNER_ADM = 0x20,
			PRIVIL_READ_VAULT = 0x40, /* zrušeno od července 2012 */
			PRIVIL_ERASE_VAULT = 0x80
		};
		Q_DECLARE_FLAGS(Privileges, Privilege)
#if (QT_VERSION >= QT_VERSION_CHECK(5, 5, 0))
		Q_FLAG(Privileges)
#else /* < Qt-5.5 */
		Q_FLAGS(Privileges)
#endif /* >= Qt-5.5 */

		/*
		 * @brief Describes the message cycle.
		 *
		 * @note Described in
		 *     pril_2/WS_ISDS_Manipulace_s_datovymi_zpravami.pdf (section 1.5).
		 */
		enum DmState {
			MS_NULL = -1, /*!< Convenience value, converted from/to NULL. */
			MS_POSTED = 1,
			MS_STAMPED = 2,
			MS_INFECTED = 3,
			MS_DELIVERED = 4,
			MS_ACCEPTED_FICT = 5,
			MS_ACCEPTED = 6,
			MS_READ = 7,
			MS_UNDELIVERABLE = 8,
			MS_REMOVED = 9,
			MS_IN_VAULT = 10
		};

		/*!
		 * @brief Message filter status as used by GetListOfSentMessages and
		 *     GetListOfReceivedMessages.
		 *
		 * @note Usage described in
		 *     pril_2/WS_ISDS_Manipulace_s_datovymi_zpravami.pdf
		 *     (section 2.8.2).
		 */
		enum DmFiltState {
			MFS_POSTED = 0x02,
			MFS_STAMPED = 0x04,
			MFS_INFECTED = 0x08,
			MFS_DELIVERED = 0x10,
			MFS_ACCEPTED_FICT = 0x20,
			MFS_ACCEPTED = 0x40,
			MFS_READ = 0x80,
			MFS_UNDELIVERABLE = 0x0100,
			MFS_REMOVED = 0x0200,
			MFS_IN_VAULT = 0x0400,
			MFS_ANY = 0x07fe /* Convenience value. */
		};
		Q_DECLARE_FLAGS(DmFiltStates, DmFiltState)
#if (QT_VERSION >= QT_VERSION_CHECK(5, 5, 0))
		Q_FLAG(DmFiltStates)
#else /* < Qt-5.5 */
		Q_FLAGS(DmFiltStates)
#endif /* >= Qt-5.5 */

		/*!
		 * @brief Describes message type.
		 *
		 * @note See pril_2/WS_ISDS_Manipulace_s_datovymi_zpravami.pdf.
		 */
		enum DmType {
			MT_UNKNOWN = -1, /*!< Convenience value, converted from/to NULL. */
			MT_I = 'I', /* Initiatory. */
			MT_K = 'K', /* Commercial. */
			MT_O = 'O', /* Commercial response paid by sender of initiatory. */
			MT_V = 'V', /* Non-commercial government message. */
			MT_A = 'A', /* Subsidised initiatory commercial, can pay a response. */
			MT_B = 'B', /* Subsidised initiatory commercial, has already paid the response. */
			MT_C = 'C', /* Subsidised initiatory commercial, response offer expired. */
			MT_D = 'D', /* Externally subsidised initiatory commercial. */
			MT_E = 'E', /* Stamp-prepaid commercial. */
			MT_G = 'G', /* Sponsor-prepaid commercial. */
			MT_X = 'X', /* Initiatory commercial, response offer expired. */
			MT_Y = 'Y', /* Initiatory commercial, has already paid the response. */
			MT_Z = 'Z'
		};

		/*!
		 * @brief Describes commercial (PDZ) payment type.
		 *
		 * @note Mentioned in pril_2/WS_ISDS_Vyhledavani_datovych_schranek.pdf
		 *     (section 2.5)
		 */
		enum PdzPaymentType {
			PPT_UNKNOWN = -1, /*!< Convenience value, converted from/to NULL. */
			PPT_K = 'K', /* Commercial paid by contract. */
			PPT_E = 'E', /* Commercial paid by credit. */
			PPT_G = 'G', /* Sponsor-prepaid commercial. */
			PPT_O = 'O', /* Commercial response paid by sender of initiatory. */
			PPT_Z = 'Z', /* Undocumented, based on libdatovka. */
			PPT_D = 'D' /* Undocumented, based on libdatovka. */
		};

		/*!
		 * Commercial message (PDZ) type. Convenience value, taken from libdatovka.
		 */
		enum PdzMessageType {
			PMT_NORMAL = 0, /*!< Normal commercial message. */
			PMT_INIT /*!< Initiatory commercial message. */
		};

		/*!
		 * @brief User types tUserType (dbTypes.xsd).
		 *
		 * @note Described in
		 *     pril_3/WS_ISDS_Sprava_datovych_schranek.pdf (section 2.4).
		 */
		enum UserType {
			UT_NULL = -1, /*!< Convenience value, converted from/to NULL. */
			UT_PRIMARY,
			UT_ENTRUSTED,
			UT_ADMINISTRATOR,
			UT_OFFICIAL,
			UT_OFFICIAL_CERT,
			UT_LIQUIDATOR,
			UT_RECEIVER,
			UT_GUARDIAN
		};

		/*!
		 * @brief Sender type as mentioned in response description of
		 *     GetMessageAuthor.
		 *
		 * @note Described in pril_2/WS_ISDS_Manipulace_s_datovymi_zpravami.pdf
		 *     (section 2.9).
		 *     The values differ slightly from enum UserType.
		 */
		enum SenderType {
			ST_NULL = -1, /*!< Convenience value, converted from/to NULL. */
			ST_PRIMARY,
			ST_ENTRUSTED,
			ST_ADMINISTRATOR,
			ST_OFFICIAL,
			ST_VIRTUAL,
			ST_OFFICIAL_CERT,
			ST_LIQUIDATOR,
			ST_RECEIVER,
			ST_GUARDIAN
		};

		/*!
		 * @brief Hash algorithm type.
		 *
		 * @todo Find definition in documentation.
		 */
		enum HashAlg {
			HA_UNKNOWN = -1, /* Convenience value. */
			HA_MD5,
			HA_SHA_1,
			HA_SHA_224,
			HA_SHA_256,
			HA_SHA_384,
			HA_SHA_512,
			HA_SHA3_256
		};

		/*!
		 * @brief Message event.
		 *
		 * @note Described in pril_2/WS_ISDS_Manipulace_s_datovymi_zpravami.pdf,
		 *     section 2.7.1.
		 */
		enum Event {
			EV_UNKNOWN = -1, /* Convenience value. */
			EV_ENTERED = 0, /* Message came into being. */
			EV_DELIVERED = 5, /* Message was delivered into recipient's box. */
			EV_ACCEPTED_LOGIN = 1, /* Before 27.10.210, message was accepted by recipient logging in. */
			EV_PRIMARY_LOGIN = 11, /* Primary/competent (Czech: opravneny) user with capability to read has logged in. */
			EV_ENTRUSTED_LOGIN = 12, /* Entrusted (Czech: povereny) user with capability to read has logged in. */
			EV_SYSCERT_LOGIN = 13, /* Application authenticated using a system certificate has logged in. */
			EV_ACCEPTED_FICTION = 2, /* Message has been accepted by fiction. */
			EV_ACCEPTED_FICTION_NO_USER = 6, /* Message accepted by fiction because there was no user able to accept the message. */
			EV_UNDELIVERABLE = 3, /* Recipient box made inaccessible, message is undeliverable. */
			EV_ACCEPTED_BY_RECIPIENT = 4, /* Before 11.2011, message has been delivered and accepted by recipient action. */
			EV_UNDELIVERED_AV_CHECK = 8 /* Message didn't pass antivirus check, message has been rejected. */
		};

		/*!
		 * @brief Attachment type.
		 *
		 * @note Mentioned in pril_2/WS_ISDS_Manipulace_s_datovymi_zpravami.pdf
		 *     section 2.1.
		 */
		enum FileMetaType {
			FMT_UNKNOWN = -1, /* Convenience value. */
			FMT_MAIN, /* First in list of documents. */
			FMT_ENCLOSURE, /* Attachment. */
			FMT_SIGNATURE, /* Digital signature of another document. */
			FMT_META /* Special XML data for ESS (records management service). */
		};

		/*!
		 * @brief Raw type. Convenience value, taken from libdatovka.
		 */
		enum RawType {
			RT_UNKNOWN = -1, /* Convenience value. */
			RT_INCOMING_MESSAGE,
			RT_PLAIN_SIGNED_INCOMING_MESSAGE,
			RT_CMS_SIGNED_INCOMING_MESSAGE,
			RT_PLAIN_SIGNED_OUTGOING_MESSAGE,
			RT_CMS_SIGNED_OUTGOING_MESSAGE,
			RT_DELIVERYINFO,
			RT_PLAIN_SIGNED_DELIVERYINFO,
			RT_CMS_SIGNED_DELIVERYINFO
		};

		/*!
		 * @brief Full-text search type.
		 *
		 * @note Mentioned in pril_2/WS_ISDS_Vyhledavani_datovych_schranek.pdf
		 *     section 2.2.
		 */
		enum FulltextSearchType {
			FST_GENERAL, /*!< Search in all fields. */
			FST_ADDRESS, /*!< Search in address. */
			FST_IC, /*!< Search in organisation identifier. */
			FST_BOX_ID /*!< Search in data-box identifier. */
		};

		/*!
		 * @brief One-time password authentication method.
		 */
		enum OtpMethod {
			OM_UNKNOWN = -1, /*!< Convenience value. */
			OM_HMAC, /*!< HMAC-based method. */
			OM_TIME /*!< Time-base method. */
		};

		/*!
		 * @brief One-time password resolution status.
		 *
		 * @note Described in pril_2/ISDS_OTP_autentizace.pdf.
		 */
		enum OtpResolution {
			OR_SUCCESS = 0, /*!< Authentication succeeded. */
			OR_UNKNOWN, /*!< Unknown status. */
			OR_BAD_AUTH, /*!< Bad log-in, try again. */
			OR_BLOCKED, /*!< Access blocked for 60 minutes, (brute force attack detected). */
			OR_PWD_EXPIRED, /*!< Password expired. (OTP or regular password expired?) */
			OR_TOO_FAST, /*!< OTP cannot be sent repeatedly at this rate (minimal delay depends on TOTP window setting). */
			OR_UNAUTHORIZED, /*!< User name is not allowed to access requested URI. */
			OR_TOTP_SENT, /*!< OTP has been generated and sent by ISDS. */
			OR_TOTP_NOT_SENT, /*!< OTP could not bee sent. Retry later. */
		};

		/*!
		 * @brief MEP resolution status.
		 *
		 * @note Described in pril_2/MobilniKlic_autentizace.pdf.
		 */
		enum MepResolution {
			MR_SUCCESS = 0, /*!< Authentication succeeded. */
			MR_UNKNOWN, /*!< Unknown status. */
			MR_UNRECOGNISED, /*!< Authentication request not recognised. */
			MR_ACK_REQUESTED, /*!< Waiting for acknowledgement. */
			MR_ACK, /*!< Acknowledged. */
			MR_ACK_EXPIRED /*!< Acknowledgement request expired.*/
		};

		/*!
		 * @brief Credit event type.
		 */
		enum CreditEventType {
			CET_UNKNOWN = -1, /*!< Convenience values. */
			CET_CHARGED = 1, /*!< Credit has been charged. */
			CET_DISCHARGED = 2, /*!< Credit has been discharged. */
			CET_MESSAGE_SENT = 3, /*!< Credit has been spent for sending a commerical message. */
			CET_STORAGE_SET = 4, /*!< Credit has been spent for setting a long-term storage. */
			CET_EXPIRED = 5, /*!< Credit has expired. */
			CET_DELETED_MESSAGE_RECOVERED = 7 /*< Message previously deleted from long-term storage has been recovered. */
		};

		/*!
		 * @brief Long term storage type.
		 *
		 * @note Described in pril_2/WS_ISDS_vyhledavani_datovych_schranek.pdf
		 *     section 2.8.
		 */
		enum DTType {
			DTT_UNKNOWN = -1, /*!< Convenience value. */
			DTT_INACTIVE = 0, /*!< Long term storage is inactive. */
			DTT_PREPAID = 1, /*!< Long term storage is active using a prepaid service. */
			DTT_UNUSED_2 = 2, /*<! This type was used until 2013. No more used since then. */
			DTT_CONTRACTUAL = 3, /*!< Long term storage is active based on a contract. */
			DTT_TRIAL = 4, /*!< Long term storage is active to try it out. */
			DTT_UNUSED_5 = 5, /*< No more used. */
			DTT_SPECIAL_OFFER = 6 /*!< Long term storage is active because of an special offer. */
		};

		/*!
		 * @brief Long term storage payment state.
		 *
		 * @note Described in pril_2/WS_ISDS_vyhledavani_datovych_schranek.pdf
		 *     section 2.8.
		 */
		enum FutDTPaidState {
			FDTPS_UNKNOWN = -1, /*!< Convenience value, converted from/to NULL. */
			FDTPS_NOT_PAID_YET = 0, /*!< Long term storage has not been paid yet. */
			FDTPS_PAID_ALREADY = 1 /*!< Long term storage has already been paid. */
		};

		/*!
		 * @brief Allows the sender to publish specified information about his person when sending a message.
		 *
		 * @note Mentioned in pril_2/WS_ISDS_Manipulace_s_datovymi_zpravami.pdf
		 *       section 2.1.
		 */
		enum IdLevelValue {
			IDLEV_NULL = -1, /*!< Convenience value, converted from/to NULL. */
			IDLEV_PUBLISH_USERTYPE = 0x00, /*!<  The sender user type isds_UserType is always enabled. */
			IDLEV_PUBLISH_PERSONNAME = 0x01, /*!<  Publish name. Comprises pnGivenNames and pnLastName. */
			IDLEV_PUBLISH_BIDATE = 0x02, /*!<  Publish biDate. */
			IDLEV_PUBLISH_BICITY = 0x04, /*!<  Publish biCity - only when sender is ST_ENTRUSTED of FO or PFO box. */
			IDLEV_PUBLISH_BICOUNTY = 0x08, /*!<  Publish biCounty - only when sender is ST_ENTRUSTED of FO or PFO box. */
			IDLEV_PUBLISH_ADCODE = 0x10, /*!<  Publish adCode (RUIAN address code). */
			IDLEV_PUBLISH_FULLADDRESS = 0x20, /*!<  Publish fullAddress. */
			IDLEV_PUBLISH_ROBIDENT = 0x40 /*!<  Publish robIdent - flag whether the person is identifiers within the ROB (Registr obyvatel/citizen registry). */
		};

		/*!
		 * @brief Response status type.
		 *
		 * @note Is actually not a type but may be used to distinguish
		 *     between dbStatus and dmStatus.
		 */
		enum ResponseStatusType {
			RST_UNKNOWN = -1, /*!< Convenience value. */
			RST_DB, /*!< Status obtained from dbStatus. */
			RST_DM /*!< Status obtained from dmStatus. */
		};

		/*!
		 * @brief Error value. Taken from libdatovka for compatibility.
		 */
		enum Error {
			ERR_SUCCESS = 0, /* No error. */
			ERR_ERROR, /* Unspecified error. */
			ERR_NOTSUP,
			ERR_INVAL,
			ERR_INVALID_CONTEXT,
			ERR_NOT_LOGGED_IN,
			ERR_CONNECTION_CLOSED,
			ERR_TIMED_OUT,
			ERR_NONEXIST,
			ERR_NOMEM,
			ERR_NETWORK,
			ERR_HTTP,
			ERR_SOAP,
			ERR_XML,
			ERR_ISDS,
			ERR_ENUM,
			ERR_DATE,
			ERR_TOO_BIG,
			ERR_TOO_SMALL,
			ERR_NOTUNIQ,
			ERR_NOTEQUAL,
			ERR_PARTIAL_SUCCESS,
			ERR_ABORTED,
			ERR_SECURITY
		};
	};

	Q_DECLARE_OPERATORS_FOR_FLAGS(Type::Privileges)
	Q_DECLARE_OPERATORS_FOR_FLAGS(Type::DmFiltStates)

}
