// SPDX-License-Identifier: Apache-2.0
// Copyright Authors of Cilium

package node

import (
	"context"
	"fmt"
	"log/slog"
	"net"

	"github.com/cilium/cilium/pkg/cidr"
	"github.com/cilium/cilium/pkg/datapath/tunnel"
	"github.com/cilium/cilium/pkg/defaults"
	"github.com/cilium/cilium/pkg/lock"
	"github.com/cilium/cilium/pkg/logging"
	"github.com/cilium/cilium/pkg/logging/logfields"
	"github.com/cilium/cilium/pkg/option"
	wgTypes "github.com/cilium/cilium/pkg/wireguard/types"
)

var (
	addrs addresses

	// localNode holds the current state of the local "types.Node".
	// This is defined here until all uses of the getters and
	// setters in this file have been migrated to use LocalNodeStore
	// directly.
	// Initialized to proper instance via an invoke function in LocalNodeStoreCell,
	// or temporarily in tests with 'WithTestLocalNodeStore'.
	localNode *LocalNodeStore
)

func getLocalNode(logger *slog.Logger) LocalNode {
	// Only expecting errors if we're called after LocalNodeStore has stopped, e.g.
	// we have a component that uses the legacy getters and setters here and does
	// not depend on LocalNodeStore.
	if localNode == nil {
		logging.Fatal(logger, "getLocalNode called for nil localNode")
	}
	n, err := localNode.Get(context.TODO())
	if err != nil {
		logging.Fatal(logger, "getLocalNode: unexpected error", logfields.Error, err)
	}
	return n
}

type addresses struct {
	mu         lock.RWMutex
	routerInfo RouterInfo
}

type RouterInfo interface {
	GetCIDRs() []net.IPNet
}

func makeIPv6HostIP(logger *slog.Logger) net.IP {
	ipstr := "fc00::10CA:1"
	ip := net.ParseIP(ipstr)
	if ip == nil {
		logging.Fatal(logger, "Unable to parse IP", logfields.IPAddr, ipstr)
	}

	return ip
}

// initDefaultPrefix initializes the node address and allocation prefixes with
// default values derived from the system. device can be set to the primary
// network device of the system in which case the first address with global
// scope will be regarded as the system's node address.
func initDefaultPrefix(logger *slog.Logger, device string) {
	localNode.Update(func(n *LocalNode) {
		setDefaultPrefix(logger, option.Config, device, n)
	})
}

func setDefaultPrefix(logger *slog.Logger, cfg *option.DaemonConfig, device string, node *LocalNode) {
	if cfg.EnableIPv4 {
		isIPv6 := false

		ip, err := firstGlobalV4Addr(device, node.GetCiliumInternalIP(isIPv6))
		if err != nil {
			return
		}

		if node.GetNodeIP(isIPv6) == nil {
			node.SetNodeInternalIP(ip)
		}

		ipv4range := node.IPv4AllocCIDR
		ipv6range := node.IPv6AllocCIDR

		if ipv4range == nil {
			// If the IPv6AllocRange is not nil then the IPv4 allocation should be
			// derived from the IPv6AllocRange.
			//                     vvvv vvvv
			// FD00:0000:0000:0000:0000:0000:0000:0000
			if ipv6range != nil {
				ip = net.IPv4(
					ipv6range.IP[8],
					ipv6range.IP[9],
					ipv6range.IP[10],
					ipv6range.IP[11])
			}
			v4range := fmt.Sprintf(defaults.DefaultIPv4Prefix+"/%d",
				ip.To4()[3], defaults.DefaultIPv4PrefixLen)
			_, ip4net, err := net.ParseCIDR(v4range)
			if err != nil {
				logging.Panic(logger, "BUG: Invalid default IPv4 prefix",
					logfields.Error, err,
					logfields.V4Prefix, v4range,
				)
			}

			node.IPv4AllocCIDR = cidr.NewCIDR(ip4net)
			logger.Debug(
				"Using autogenerated IPv4 allocation range",
				logfields.V4Prefix, node.IPv4AllocCIDR,
			)
		}
	}

	if cfg.EnableIPv6 {
		isIPv6 := true
		ipv4range := node.IPv4AllocCIDR
		ipv6range := node.IPv6AllocCIDR

		if node.GetNodeIP(isIPv6) == nil {
			// Find a IPv6 node address first
			addr, _ := firstGlobalV6Addr(device, node.GetCiliumInternalIP(isIPv6))
			if addr == nil {
				addr = makeIPv6HostIP(logger)
			}
			node.SetNodeInternalIP(addr)
		}

		if ipv6range == nil {
			var v6range string
			var logMessage string
			if ipv4range != nil {
				// The IPv6 allocation should be derived from the IPv4 allocation.
				ip := node.IPv4AllocCIDR.IP
				v6range = fmt.Sprintf("%s%02x%02x:%02x%02x:0:0/%d",
					cfg.IPv6ClusterAllocCIDRBase, ip[0], ip[1], ip[2], ip[3], 96)
				logMessage = "Using autogenerated IPv6 allocation range from IPv4 allocation"
			} else {
				// The IPv6 allocation is derived from the node's IPv6 address.
				ip := node.GetNodeIP(isIPv6)
				if ip == nil {
					// This should not happen, as we set the node IP above.
					logging.Panic(logger, "BUG: Node IPv6 address is not available to derive IPv6 pod CIDR")
				}

				// We use the last 4 bytes of the node's IPv6 address to build the pod CIDR.
				// This makes the allocation logic independent of IPv4.
				v6range = fmt.Sprintf("%s%02x%02x:%02x%02x:0:0/%d",
					cfg.IPv6ClusterAllocCIDRBase, ip[12], ip[13], ip[14], ip[15], 96)
				logMessage = "Using autogenerated IPv6 allocation range from node IPv6"
			}

			_, ip6net, err := net.ParseCIDR(v6range)
			if err != nil {
				logging.Panic(logger, "BUG: Invalid default IPv6 prefix",
					logfields.Error, err,
					logfields.V6Prefix, v6range,
				)
			}

			node.IPv6AllocCIDR = cidr.NewCIDR(ip6net)
			logger.Debug(
				logMessage,
				logfields.V6Prefix, node.IPv6AllocCIDR,
			)
		}
	}
}

// GetCiliumEndpointNodeIP is the node IP that will be referenced by CiliumEndpoints with endpoints
// running on this node.
func GetCiliumEndpointNodeIP(logger *slog.Logger) string {
	n := getLocalNode(logger)
	if option.Config.EnableIPv4 && n.Local.UnderlayProtocol == tunnel.IPv4 {
		return n.GetNodeIP(false).String()
	}
	return n.GetNodeIP(true).String()
}

// GetRouterInfo returns additional information for the router, the cilium_host interface.
func GetRouterInfo() RouterInfo {
	addrs.mu.RLock()
	defer addrs.mu.RUnlock()
	return addrs.routerInfo
}

// SetRouterInfo sets additional information for the router, the cilium_host interface.
func SetRouterInfo(info RouterInfo) {
	addrs.mu.Lock()
	addrs.routerInfo = info
	addrs.mu.Unlock()
}

// SetIPv4AllocRange sets the IPv4 address pool to use when allocating
// addresses for local endpoints
func SetIPv4AllocRange(net *cidr.CIDR) {
	localNode.Update(func(n *LocalNode) {
		n.IPv4AllocCIDR = net
	})
}

// SetIPv6NodeRange sets the IPv6 address pool to be used on this node
func SetIPv6NodeRange(net *cidr.CIDR) {
	localNode.Update(func(n *LocalNode) {
		n.IPv6AllocCIDR = net
	})
}

// AutoComplete completes the parts of addressing that can be auto derived
func AutoComplete(logger *slog.Logger, directRoutingDevice string) error {
	initDefaultPrefix(logger, directRoutingDevice)

	ln := getLocalNode(logger)

	if option.Config.EnableIPv6 && ln.IPv6AllocCIDR == nil {
		return fmt.Errorf("IPv6 allocation CIDR is not configured. Please specify --%s", option.IPv6Range)
	}

	if option.Config.EnableIPv4 && ln.IPv4AllocCIDR == nil {
		return fmt.Errorf("IPv4 allocation CIDR is not configured. Please specify --%s", option.IPv4Range)
	}

	return nil
}

// ValidatePostInit validates the entire addressing setup and completes it as
// required
func ValidatePostInit(logger *slog.Logger) error {
	ln := getLocalNode(logger)

	if option.Config.EnableIPv4 {
		if ln.GetNodeIP(false) == nil {
			return fmt.Errorf("external IPv4 node address could not be derived, please configure via --ipv4-node")
		}
	}

	if option.Config.TunnelingEnabled() && ln.GetNodeIP(false) == nil && ln.GetNodeIP(true) == nil {
		return fmt.Errorf("external node address could not be derived, please configure via --ipv4-node or --ipv6-node")
	}

	if option.Config.EnableIPv4 && ln.GetCiliumInternalIP(false) == nil {
		return fmt.Errorf("BUG: Internal IPv4 node address was not configured")
	}

	return nil
}

// GetEndpointEncryptKeyIndex returns the encryption key value for an endpoint
// owned by the given local node.
// With IPSec encryption, this is the ID of the currently loaded key.
// With WireGuard, this returns a non-zero static value.
// Note that the key index returned by this function is only valid for _endpoints_
// of the local node. If you want to obtain the key index of the local node itself,
// access the `EncryptionKey` field via the LocalNodeStore.
func GetEndpointEncryptKeyIndex(localNode LocalNode, wgEnabled, ipsecEnabled bool) uint8 {
	switch {
	case ipsecEnabled:
		return localNode.EncryptionKey
	case wgEnabled:
		return wgTypes.StaticEncryptKey

	}
	return 0
}

func SetTestLocalNodeStore() {
	if localNode != nil {
		panic("localNode already set")
	}

	// Set the localNode global variable temporarily so that the legacy getters
	// and setters can access it.
	localNode = NewTestLocalNodeStore(LocalNode{})
}

func UnsetTestLocalNodeStore() {
	localNode = nil
}
