pub mod constants;
mod date_util;
mod desktop_settings;
mod error;
mod gtk_util;
mod stopwatch;

pub use date_util::DateUtil;
pub use desktop_settings::{ClockFormat, DesktopSettings};
pub use error::UtilError;
use futures::future;
use gio::{NetworkConnectivity, Settings as GSettings, prelude::*};
use glib::{GString, Object};
pub use gtk_util::GTK_RESOURCE_FILE_ERROR;
pub use gtk_util::GtkUtil;
use news_flash::{
    NewsFlash,
    error::{FeedApiError, NewsFlashError},
};
use reqwest::{Client, ClientBuilder, NoProxy, Proxy};
use serde::Deserialize;
use serde::Deserializer;
use serde::de;
use url::Url;

use crate::config::VERSION;
use crate::settings::{ProxyModel, ProxyProtocoll, Settings};
use std::fmt;
use std::future::Future;
use std::time::Duration;

pub const CHANNEL_ERROR: &str = "Error sending message via glib channel";

pub struct Util;

impl Util {
    pub fn deserialize_string_or_vec<'de, D>(deserializer: D) -> Result<Vec<String>, D::Error>
    where
        D: Deserializer<'de>,
    {
        struct StringOrVec;

        impl<'de> de::Visitor<'de> for StringOrVec {
            type Value = Vec<String>;

            fn expecting(&self, formatter: &mut fmt::Formatter) -> fmt::Result {
                formatter.write_str("string or list of strings")
            }

            fn visit_str<E>(self, s: &str) -> Result<Self::Value, E>
            where
                E: de::Error,
            {
                Ok(vec![s.to_owned()])
            }

            fn visit_seq<S>(self, seq: S) -> Result<Self::Value, S::Error>
            where
                S: de::SeqAccess<'de>,
            {
                Deserialize::deserialize(de::value::SeqAccessDeserializer::new(seq))
            }
        }

        deserializer.deserialize_any(StringOrVec)
    }

    pub fn glib_spawn_future<F: Future<Output = ()> + 'static>(future: F) {
        glib::MainContext::default().spawn_local(future);
    }

    pub async fn is_online(
        connectivity: NetworkConnectivity,
        network_available: bool,
        news_flash: &NewsFlash,
        client: &Client,
        ping_urls: &[String],
    ) -> bool {
        log::info!("Networkmonitor connectivity: {connectivity:?} (available {network_available})");

        if network_available && connectivity == NetworkConnectivity::Full {
            return true;
        }

        let is_reachable = news_flash.is_reachable(client).await;
        log::debug!("backend reachable: {is_reachable:?}");
        match is_reachable {
            // if not supported (e.g. local RSS) try pinging configured urls
            Err(NewsFlashError::API(FeedApiError::Unsupported)) => {}

            Ok(reachable) => return reachable,
            Err(_) => return false,
        }

        let mut tasks = Vec::new();

        for ping_url in ping_urls {
            let ping_url = ping_url.clone();
            let client = client.clone();

            let task = tokio::spawn(async move {
                let res = client.head(&ping_url).timeout(Duration::from_secs(2)).send().await;

                match res {
                    Ok(res) => {
                        let status = res.status();
                        if status.is_success() {
                            log::info!("pinging '{ping_url}' succeeded");
                        } else {
                            log::info!("pinging '{ping_url}' failed: {status:?}");
                        }

                        status.is_success()
                    }
                    Err(error) => {
                        log::info!("pinging '{ping_url}' failed: {error}");
                        false
                    }
                }
            });

            tasks.push(task);
        }

        future::join_all(tasks)
            .await
            .into_iter()
            .filter_map(|res| res.ok())
            .any(|res| res)
    }

    pub fn discover_gnome_proxy() -> Vec<ProxyModel> {
        let mut proxy_vec = Vec::new();

        let https_settings = GSettings::new("org.gnome.system.proxy.https");
        let https_url = https_settings.string("host");
        let https_port = https_settings.int("port") as u16;

        if !https_url.is_empty()
            && let Ok(mut https_url) = Url::parse(&https_url)
        {
            _ = https_url.set_port(Some(https_port));
            proxy_vec.push(
                Object::builder::<ProxyModel>()
                    .property("protocoll", ProxyProtocoll::Https)
                    .property("url", https_url.to_string())
                    .build(),
            );
        }

        let http_settings = GSettings::new("org.gnome.system.proxy.http");
        let http_url = http_settings.string("host");
        let http_port = http_settings.int("port") as u16;
        let http_user = http_settings.string("authentication-user");
        let http_pw = http_settings.string("authentication-password");

        if let Ok(mut http_url) = Url::parse(&http_url) {
            _ = http_url.set_port(Some(http_port));
            let user = if http_user.is_empty() {
                None
            } else {
                Some(http_user.to_string())
            };
            let password = if http_pw.is_empty() {
                None
            } else {
                Some(http_pw.to_string())
            };

            proxy_vec.push(
                Object::builder::<ProxyModel>()
                    .property("protocoll", ProxyProtocoll::Http)
                    .property("url", https_url.to_string())
                    .property("user", user)
                    .property("password", password)
                    .build(),
            );
        }

        proxy_vec
    }

    pub fn symbolic_icon_set_color(data: &[u8], hex_color: &str) -> Result<Vec<u8>, UtilError> {
        let svg_string = std::str::from_utf8(data).map_err(|_| UtilError::Svg)?;
        let colored_svg_string = svg_string.replace("fill:#bebebe", &format!("fill:{hex_color}"));
        Ok(colored_svg_string.as_bytes().into())
    }

    pub fn format_data_size(bytes: u64) -> String {
        let step = 1024.0;
        let kbytes = bytes as f64 / step;
        let mbytes = kbytes / step;
        let gbytes = mbytes / step;

        if bytes < 1024 {
            format!("{bytes} bytes")
        } else if kbytes < step {
            format!("{kbytes:.1} kb")
        } else if mbytes < step {
            format!("{mbytes:.1} mb")
        } else {
            format!("{gbytes:.1} gb")
        }
    }

    pub fn build_client(settings: Settings) -> Client {
        let user_agent =
            format!("Newsflash/{VERSION} (RSS reader; +https://flathub.org/apps/io.gitlab.news_flash.NewsFlash)");

        let mut builder = ClientBuilder::new()
            .user_agent(user_agent)
            .use_native_tls()
            .hickory_dns(false)
            .gzip(true)
            .brotli(true)
            .timeout(Duration::from_secs(60))
            .danger_accept_invalid_certs(settings.advanced().accept_invalid_certs)
            .danger_accept_invalid_hostnames(settings.advanced().accept_invalid_hostnames);

        let ignore_hosts: Vec<GString> = GSettings::new("org.gnome.system.proxy").strv("ignore-hosts").into();
        let ignore_hosts = ignore_hosts.join(",");
        log::debug!("ignored hosts: {ignore_hosts}");
        let no_proxy = NoProxy::from_string(&ignore_hosts);
        log::debug!("no proxy: {no_proxy:?}");

        let mut proxys = settings.advanced().proxy.clone();
        proxys.append(&mut Util::discover_gnome_proxy());

        for proxy_model in proxys {
            if proxy_model.url().starts_with("socks4") {
                continue;
            }

            let mut proxy = match &proxy_model.protocoll() {
                ProxyProtocoll::All => Proxy::all(proxy_model.url()),
                ProxyProtocoll::Http => Proxy::http(proxy_model.url()),
                ProxyProtocoll::Https => Proxy::https(proxy_model.url()),
            }
            .unwrap_or_else(|_| panic!("Failed to build proxy: {}", proxy_model.url()));

            if let Some(proxy_user) = &proxy_model.user()
                && let Some(proxy_password) = &proxy_model.password()
            {
                proxy = proxy.basic_auth(proxy_user, proxy_password);
            }

            let proxy = proxy.no_proxy(no_proxy.clone());
            log::debug!("proxy {proxy:?}");

            builder = builder.proxy(proxy);
        }

        builder.build().expect("Failed to build reqwest client")
    }
}
