use crate::app::App;
use crate::config::APP_ID;
use crate::content_page::ArticleViewColumn;
use crate::gobject_models::GEnclosure;
use crate::i18n::i18n;
use crate::infrastructure::TokioRuntime;
use crate::main_window::MainWindow;
use crate::util::constants;
use chrono::TimeDelta;
use clapper::{MediaItem, Mpris, Player, PlayerState};
use glib::{Object, Properties, clone, subclass::prelude::*, subclass::*};
use gstreamer::{
    ElementFactory, TagList, TagMergeMode, TagScope,
    tags::{Artist, Title},
};
use gtk4::{CompositeTemplate, Scale, ScrollType, ToggleButton, Widget, prelude::*, subclass::prelude::*};
use libadwaita::{Bin, subclass::prelude::*};
use news_flash::models::Enclosure;
use std::cell::RefCell;

mod imp {
    use super::*;

    #[derive(Debug, Default, CompositeTemplate, Properties)]
    #[properties(wrapper_type = super::AudioWidget)]
    #[template(file = "data/resources/ui_templates/enclosures/audio.blp")]
    pub struct AudioWidget {
        #[template_child]
        pub scale: TemplateChild<Scale>,

        #[property(get, set, construct_only)]
        pub author: RefCell<String>,

        #[property(get, set, construct_only)]
        pub title: RefCell<String>,

        #[property(get, set, construct_only)]
        pub enclosure: RefCell<GEnclosure>,

        #[property(get, set, construct_only)]
        pub item: RefCell<MediaItem>,

        #[property(get, set, name = "volume-icon")]
        pub volume_icon: RefCell<String>,

        #[property(get)]
        pub player: Player,
    }

    #[glib::object_subclass]
    impl ObjectSubclass for AudioWidget {
        const NAME: &'static str = "AudioWidget";
        type Type = super::AudioWidget;
        type ParentType = Bin;

        fn class_init(klass: &mut Self::Class) {
            klass.bind_template();
            klass.bind_template_callbacks();
        }

        fn instance_init(obj: &InitializingObject<Self>) {
            obj.init_template();
        }
    }

    #[glib::derived_properties]
    impl ObjectImpl for AudioWidget {
        fn constructed(&self) {
            App::default()
                .bind_property("volume", &self.player, "volume")
                .bidirectional()
                .sync_create()
                .build();

            let tag_inject = ElementFactory::make("taginject")
                .property("scope", TagScope::Global)
                .build()
                .expect("failed to build taginject element");

            let title = self.title.borrow().clone();
            let author = self.author.borrow().clone();
            let mut tags = TagList::new();
            tags.make_mut().add::<Title>(&title.as_str(), TagMergeMode::Replace);
            tags.make_mut().add::<Artist>(&author.as_str(), TagMergeMode::Replace);
            let tag_string = tags.to_string();
            let tag_string = tag_string
                .strip_prefix("taglist, ")
                .expect("serialized GstTagList should start with `taglist, `")
                .to_string();
            log::debug!("tags: {tag_string}");
            tag_inject.set_property("tags", &tag_string);
            self.player.set_audio_filter(Some(&tag_inject));

            let mpris = Mpris::new("org.mpris.MediaPlayer2.Newsflash", "Newsflash", Some(APP_ID));
            mpris.set_fallback_art_url(self.enclosure.borrow().thumbnail_url().as_deref());
            self.player.add_feature(&mpris);

            if let Some(queue) = self.player.queue() {
                queue.add_item(&self.item.borrow());
                queue.select_item(Some(&*self.item.borrow()));
            }

            self.player.connect_mute_notify(clone!(
                #[weak(rename_to = imp)]
                self,
                move |player| imp.set_volume_icon(player)
            ));

            self.player.connect_volume_notify(clone!(
                #[weak(rename_to = imp)]
                self,
                move |player| imp.set_volume_icon(player)
            ));

            self.obj().set_volume_icon("audio-volume-high-symbolic");
        }

        fn dispose(&self) {
            log::debug!("dispose audio widget");

            let position = self.player.position();
            let duration = self.item.borrow().duration();
            let is_finished = position / duration >= constants::VIDEO_END_THRESHOLD_PERCENT;
            log::debug!("position {position} duration {duration} (finished {is_finished})");

            let position = if is_finished { 0.0 } else { position };
            self.player.stop();

            self.set_enclosure_position(position);
        }
    }

    impl WidgetImpl for AudioWidget {}

    impl BinImpl for AudioWidget {}

    #[gtk4::template_callbacks]
    impl AudioWidget {
        #[template_callback]
        fn format_position(&self, position: f64) -> String {
            let duration = TimeDelta::seconds(position as i64);

            if duration.num_hours() > 0 {
                format!(
                    "{:02}:{:02}:{:02}",
                    duration.num_hours(),
                    duration.num_minutes() % 60,
                    duration.num_seconds() % 60
                )
            } else {
                format!("{:02}:{:02}", duration.num_minutes() % 60, duration.num_seconds() % 60)
            }
        }

        #[template_callback]
        fn play_pause(&self) {
            let state = self.player.state();

            match state {
                PlayerState::Buffering | PlayerState::Paused => self.player.play(),
                PlayerState::Playing => self.player.pause(),
                PlayerState::Stopped => {
                    self.player.play();

                    let position = self.enclosure.borrow().position();
                    if position > 0.0 {
                        self.player.seek(position);
                    }
                }
                _ => unreachable!(),
            }
        }

        #[template_callback]
        fn play_icon(&self, state: PlayerState) -> &'static str {
            if state == PlayerState::Playing {
                "media-playback-pause-symbolic"
            } else {
                "media-playback-start-symbolic"
            }
        }

        #[template_callback]
        fn clamp_duraiton(&self, duration: f64) -> f32 {
            if duration == 0.0 { -1.0 } else { duration as f32 }
        }

        #[template_callback]
        fn on_position_changed(&self, _scroll_type: ScrollType, new_value: f64) -> bool {
            if new_value <= 0.0 {
                return true;
            }

            self.player.seek(new_value);
            true
        }

        #[template_callback]
        fn on_volume_changed(&self, _scroll_type: ScrollType, new_value: f64) -> bool {
            self.player.set_volume(new_value);
            true
        }

        #[template_callback]
        fn on_toggle_mute(&self, toggle_button: ToggleButton) {
            self.player.set_mute(toggle_button.is_active());
        }

        #[template_callback]
        fn on_copy_to_clipboard(&self) {
            let url = self.enclosure.borrow().url();
            MainWindow::instance().clipboard().set_text(url.as_str());
        }

        fn set_volume_icon(&self, player: &Player) {
            let icon_name = if player.is_muted() || player.volume() == 0.0 {
                "audio-volume-muted-symbolic"
            } else if player.volume() <= 0.25 {
                "audio-volume-low-symbolic"
            } else if player.volume() <= 0.75 {
                "audio-volume-medium-symbolic"
            } else {
                "audio-volume-high-symbolic"
            };
            self.obj().set_volume_icon(icon_name);
        }

        fn set_enclosure_position(&self, position: f64) {
            let enclosure = self.enclosure.borrow().clone();
            enclosure.set_position(position);

            if let Some(article) = ArticleViewColumn::instance().article() {
                let mut enclosures = article.enclosures();

                for e in enclosures.as_mut() {
                    if e.url() == enclosure.url() {
                        e.set_position(enclosure.position());
                        break;
                    }
                }
            }

            let enclosure = Enclosure::from(enclosure);

            TokioRuntime::instance().spawn(async move {
                if let Some(news_flash) = App::news_flash().read().await.as_ref() {
                    _ = news_flash.update_enclosure(&enclosure);
                }
            });
        }
    }
}

glib::wrapper! {
    pub struct AudioWidget(ObjectSubclass<imp::AudioWidget>)
        @extends Widget, Bin;
}

impl From<&GEnclosure> for AudioWidget {
    fn from(value: &GEnclosure) -> Self {
        let author = value
            .author()
            .unwrap_or_else(|| value.feed_title().unwrap_or_else(|| i18n("Unkown Author")));
        let title = value
            .title()
            .unwrap_or_else(|| value.article_title().unwrap_or_else(|| i18n("Unknown Enclosure")));
        let item = MediaItem::new(&value.url());

        Object::builder()
            .property("author", author)
            .property("title", title)
            .property("enclosure", value)
            .property("item", item)
            .property("volume-icon", "audio-volume-high-symbolic")
            .build()
    }
}
