use super::article_tags::ArticleRowTags;
use crate::app::App;
use crate::article_list::{MarkUpdate, OpenArticleInBrowser, ReadUpdate};
use crate::gobject_models::{GArticle, GDateTime, GMarked, GOptionalDateTime};
use crate::i18n::i18n;
use crate::infrastructure::{FaviconCache, TokioRuntime};
use crate::main_window::MainWindow;
use crate::settings::GOrderBy;
use crate::util::DateUtil;
use gdk4::{Rectangle, Texture};
use gio::{Menu, MenuItem};
use glib::{Object, Properties, SignalHandlerId, clone, closure, subclass::*};
use gtk4::{
    CompositeTemplate, ConstraintGuide, ConstraintLayout, EventControllerMotion, Label, PopoverMenu, Widget,
    prelude::*, subclass::prelude::*,
};
use once_cell::sync::Lazy;
use std::cell::{Cell, RefCell};

mod imp {
    use super::*;

    #[derive(Default, Debug, CompositeTemplate, Properties)]
    #[properties(wrapper_type = super::ArticleRow)]
    #[template(file = "data/resources/ui_templates/article_list/row.blp")]
    pub struct ArticleRow {
        #[template_child]
        pub thumbnail_layout: TemplateChild<ConstraintLayout>,
        #[template_child]
        pub context_popover: TemplateChild<PopoverMenu>,
        #[template_child]
        pub title_label: TemplateChild<Label>,
        #[template_child]
        pub motion: TemplateChild<EventControllerMotion>,

        #[property(get, set = Self::set_model)]
        pub model: RefCell<GArticle>,

        #[property(get, set = Self::set_title, nullable)]
        pub title: RefCell<Option<String>>,

        #[property(get, set)]
        pub date: RefCell<String>,

        #[property(get, set, name = "is-star-visible")]
        pub star_visible: Cell<bool>,

        #[property(get, set, name = "is-hovered")]
        pub hovered: Cell<bool>,

        #[property(get, set, name = "is-starred")]
        pub starred: Cell<bool>,

        #[property(get, set, name = "favicon-texture", nullable)]
        pub favicon_texture: RefCell<Option<Texture>>,
        #[property(get, set, name = "thumbnail-texture", nullable)]
        pub thumbnail_texture: RefCell<Option<Texture>>,
    }

    #[glib::object_subclass]
    impl ObjectSubclass for ArticleRow {
        const NAME: &'static str = "ArticleRow";
        type ParentType = gtk4::Box;
        type Type = super::ArticleRow;

        fn class_init(klass: &mut Self::Class) {
            ArticleRowTags::ensure_type();

            klass.bind_template();
            klass.bind_template_callbacks();
        }

        fn instance_init(obj: &InitializingObject<Self>) {
            obj.init_template();
        }
    }

    #[glib::derived_properties]
    impl ObjectImpl for ArticleRow {
        fn constructed(&self) {
            let guide = ConstraintGuide::builder().max_width(64).build();
            self.thumbnail_layout.add_guide(guide);
            let obj = self.obj();

            obj.property_expression("model")
                .chain_property::<GArticle>("title")
                .bind(&*obj, "title", Widget::NONE);

            obj.property_expression("model")
                .chain_property::<GArticle>("marked")
                .chain_closure::<bool>(closure!(|_: Option<Object>, marked: GMarked| {
                    marked == GMarked::Marked
                }))
                .bind(&*obj, "is-starred", Widget::NONE);

            obj.property_expression("model")
                .chain_property::<GArticle>("thumbnail")
                .watch(
                    Object::NONE,
                    clone!(
                        #[weak(rename_to = imp)]
                        self,
                        move || {
                            imp.load_thumbnail();
                        }
                    ),
                );

            obj.property_expression("model")
                .chain_property::<GArticle>("date")
                .chain_closure::<String>(closure!(|this: super::ArticleRow, date: GDateTime| {
                    if App::default().settings().article_list().order_by() == GOrderBy::Published {
                        DateUtil::format_time(date.as_ref())
                    } else {
                        let updated = this.model().updated();
                        let updated = updated.as_ref();
                        let date = updated.as_ref().unwrap_or(date.as_ref());
                        DateUtil::format_time(date)
                    }
                }))
                .bind(&*obj, "date", Some(&*obj));

            obj.property_expression("model")
                .chain_property::<GArticle>("updated")
                .chain_closure::<String>(closure!(|this: super::ArticleRow, updated: GOptionalDateTime| {
                    if App::default().settings().article_list().order_by() == GOrderBy::Published {
                        DateUtil::format_time(this.model().date().as_ref())
                    } else {
                        let updated = updated.as_ref();
                        let date = this.model().date();
                        let date = updated.as_ref().unwrap_or(date.as_ref());
                        DateUtil::format_time(date)
                    }
                }))
                .bind(&*obj, "date", Some(&*obj));

            obj.connect_is_starred_notify(super::ArticleRow::update_is_start_visible);
            obj.connect_is_hovered_notify(super::ArticleRow::update_is_start_visible);
        }

        fn signals() -> &'static [Signal] {
            static SIGNALS: Lazy<Vec<Signal>> = Lazy::new(|| {
                vec![
                    Signal::builder("activated")
                        .param_types([super::ArticleRow::static_type()])
                        .build(),
                ]
            });

            SIGNALS.as_ref()
        }
    }

    impl WidgetImpl for ArticleRow {}

    impl BoxImpl for ArticleRow {}

    #[gtk4::template_callbacks]
    impl ArticleRow {
        #[template_callback]
        pub fn is_texture_some(&self, texture: Option<Texture>) -> bool {
            texture.is_some()
        }

        #[template_callback]
        pub fn is_texture_none(&self, texture: Option<Texture>) -> bool {
            texture.is_none()
        }

        #[template_callback]
        pub fn is_string_some(&self, string: Option<String>) -> bool {
            string.is_some()
        }

        #[template_callback]
        pub fn thumbnail_size(&self, texture: Option<Texture>) -> i32 {
            if texture.is_some() { 64 } else { -1 }
        }

        #[template_callback]
        pub fn star_icon_name(&self, is_starred: bool) -> &'static str {
            if is_starred {
                "marked-symbolic"
            } else {
                "unmarked-symbolic"
            }
        }

        #[template_callback]
        pub fn summary_lines(&self, title: Option<String>) -> i32 {
            if title.is_some() { 2 } else { 3 }
        }

        #[template_callback]
        pub fn on_left_click(&self, press_count: i32, _x: f64, _y: f64) {
            if press_count != 1 {
                return;
            }

            self.obj().activate();
        }

        #[template_callback]
        pub fn on_right_click(&self, press_count: i32, x: f64, y: f64) {
            if press_count != 1 {
                return;
            }

            if App::default().is_offline() {
                return;
            }

            let rect = Rectangle::new(x as i32, y as i32, 0, 0);
            self.update_context_menu();
            self.context_popover.set_pointing_to(Some(&rect));
            self.context_popover.popup();
        }

        #[template_callback]
        pub fn on_long_press(&self, x: f64, y: f64) {
            if App::default().is_offline() {
                return;
            }

            let rect = Rectangle::new(x as i32, y as i32, 0, 0);
            self.update_context_menu();
            self.context_popover.set_pointing_to(Some(&rect));
            self.context_popover.popup();
        }

        #[template_callback]
        pub fn on_middle_press(&self, press_count: i32, _x: f64, _y: f64) {
            if press_count != 1 {
                return;
            }

            let Some(url) = self.model.borrow().url() else {
                return;
            };

            let msg = OpenArticleInBrowser {
                article_id: self.model.borrow().article_id(),
                url,
                read: self.model.borrow().read(),
            };

            MainWindow::activate_action("open-article-in-browser", Some(&msg.to_variant()));
        }

        #[template_callback]
        pub fn on_star_clicked(&self) {
            let marked_target = MarkUpdate {
                article_id: self.model.borrow().article_id(),
                marked: self.model.borrow().marked().invert(),
            };

            MainWindow::activate_action("set-article-marked", Some(&marked_target.to_variant()));
        }

        #[template_callback]
        fn on_enter(&self, _x: f64, _y: f64) {
            if !self.motion.contains_pointer() {
                return;
            }

            self.obj().set_is_hovered(true);
        }

        #[template_callback]
        fn on_leave(&self) {
            self.obj().set_is_hovered(false);
        }

        pub fn set_title(&self, title: Option<String>) {
            if self.model.borrow().title_is_markup() {
                self.title_label.set_tooltip_markup(title.as_deref());
            } else {
                self.title_label.set_tooltip_text(title.as_deref());
            }
        }

        pub fn set_model(&self, article: GArticle) {
            let article_id = article.article_id();

            if article_id == self.model.borrow().article_id() {
                return;
            }

            self.model.replace(article.clone());
            self.load_favicon();

            if article.thumbnail().is_some() {
                self.load_thumbnail();
            } else {
                self.obj().set_thumbnail_texture(Texture::NONE);
            }
        }

        fn update_context_menu(&self) {
            let model = Menu::new();
            let article = self.model.borrow();

            let read_item = MenuItem::new(Some(&i18n("Toggle Read")), None);
            let star_item = MenuItem::new(Some(&i18n("Toggle Star")), None);
            let open_item = MenuItem::new(Some(&i18n("Open in Browser")), None);

            let read_target = ReadUpdate {
                article_id: article.article_id(),
                read: article.read().invert(),
            }
            .to_variant();
            read_item.set_action_and_target_value(Some("win.set-article-read"), Some(&read_target));

            let marked_target = MarkUpdate {
                article_id: article.article_id(),
                marked: article.marked().invert(),
            }
            .to_variant();
            star_item.set_action_and_target_value(Some("win.set-article-marked"), Some(&marked_target));

            if let Some(url) = article.url() {
                let msg = OpenArticleInBrowser {
                    article_id: article.article_id(),
                    url,
                    read: article.read(),
                };

                open_item.set_action_and_target_value(Some("win.open-article-in-browser"), Some(&msg.to_variant()));
            }

            model.append_item(&read_item);
            model.append_item(&star_item);
            model.append_item(&open_item);

            self.context_popover.set_menu_model(Some(&model));
        }

        fn load_favicon(&self) {
            let feed_id = self.model.borrow().feed_id();

            TokioRuntime::execute_with_callback(
                || async move { FaviconCache::get_icon(feed_id.as_ref(), App::news_flash(), App::client()).await },
                clone!(
                    #[weak(rename_to = row)]
                    self.obj(),
                    move |texture| row.set_favicon_texture(texture)
                ),
            );
        }

        fn load_thumbnail(&self) {
            if !App::default().settings().article_list().show_thumbnails() {
                return;
            }

            let article_id = self.model.borrow().article_id();

            TokioRuntime::execute_with_callback(
                move || async move {
                    let news_flash = App::news_flash();
                    let news_flash_guad = news_flash.read().await;
                    let news_flash = news_flash_guad.as_ref()?;

                    let thumbnail = news_flash
                        .get_article_thumbnail(article_id.as_ref(), &App::client())
                        .await
                        .ok()??;
                    let bytes = thumbnail.data.map(glib::Bytes::from_owned)?;
                    gdk4::Texture::from_bytes(&bytes).ok()
                },
                clone!(
                    #[weak(rename_to = row)]
                    self.obj(),
                    move |texture| row.set_thumbnail_texture(texture)
                ),
            );
        }
    }
}

glib::wrapper! {
    pub struct ArticleRow(ObjectSubclass<imp::ArticleRow>)
        @extends gtk4::Widget, gtk4::Box;
}

impl Default for ArticleRow {
    fn default() -> Self {
        glib::Object::new::<Self>()
    }
}

impl ArticleRow {
    pub fn new() -> Self {
        Self::default()
    }

    pub fn activate(&self) {
        self.emit_by_name::<()>("activated", &[&self.clone()])
    }

    pub fn connect_activated<F: Fn(&Self) + 'static>(&self, f: F) -> SignalHandlerId {
        self.connect_local("activated", false, move |args| {
            let row = args[1]
                .get::<Self>()
                .expect("The value needs to be of type `ArticleRow`");
            f(&row);
            None
        })
    }

    fn update_is_start_visible(&self) {
        self.set_is_star_visible(self.is_starred() || self.is_hovered());
    }
}
