use std::collections::{HashMap, HashSet};

use chrono::{DateTime, Utc};
use gio::{ActionEntry, prelude::*};
use news_flash::error::NewsFlashError;
use news_flash::models::{FeedID, LoginData};

use crate::app::{App, NotificationCounts};
use crate::content_page::{ArticleListColumn, ContentPage};
use crate::i18n::i18n;
use crate::infrastructure::TokioRuntime;
use crate::main_window::MainWindow;
use crate::util::constants;

pub struct SyncActions;

impl SyncActions {
    pub fn setup() {
        // -------------------------
        // sync
        // -------------------------
        let sync = ActionEntry::builder("sync")
            .activate(|_app, _action, _parameter| Self::sync())
            .build();

        // -------------------------
        // initial sync
        // -------------------------
        let init_sync = ActionEntry::builder("init-sync")
            .activate(|_app, _action, _parameter| Self::init_sync())
            .build();

        // -------------------------
        // reset account
        // -------------------------
        let reset = ActionEntry::builder("show-reset-page")
            .activate(|_app, _action, _parameter| MainWindow::instance().show_reset_page())
            .build();

        // -------------------------
        // update account login
        // -------------------------
        let update_login = ActionEntry::builder("update-login")
            .activate(|_app, _action, _parameter| Self::update_login())
            .build();

        App::default().add_action_entries([sync, init_sync, reset, update_login]);
    }

    fn sync() {
        if App::default().is_syncing() {
            log::warn!("sync already in progress");
        }

        App::default().set_is_syncing(true);
        App::set_background_status(constants::BACKGROUND_SYNC);

        let header_maps = App::default().settings().get_feed_header_maps();

        TokioRuntime::execute_with_callback(
            || async move {
                let news_flash = App::news_flash();
                let news_flash_guad = news_flash.read().await;
                let news_flash = news_flash_guad.as_ref().ok_or(NewsFlashError::NotLoggedIn)?;
                let last_sync_datetime = news_flash.last_sync().await;

                let new_article_count_map = news_flash.sync(&App::client(), header_maps).await?;
                let unread_map = news_flash
                    .unread_count_feed_map(App::default().settings().article_list().hide_future_articles())?;
                let (feeds, _feed_mappings) = news_flash.get_feeds()?;
                Ok((new_article_count_map, unread_map, feeds, last_sync_datetime))
            },
            |res| {
                App::default().set_is_syncing(false);

                match res {
                    Ok((new_article_count_map, unread_map, feeds, last_sync_datetime)) => {
                        let feed_ids = feeds.iter().map(|f| f.feed_id.clone()).collect::<HashSet<FeedID>>();
                        let feed_names = feeds
                            .iter()
                            .map(|f| (f.feed_id.clone(), f.label.clone()))
                            .collect::<HashMap<FeedID, String>>();

                        _ = App::default().settings().delete_old_feed_settings(&feed_ids);

                        ContentPage::instance().update_sidebar();
                        ArticleListColumn::instance().update_list();
                        let counts = NotificationCounts {
                            new: new_article_count_map,
                            unread: unread_map,
                            names: feed_names,
                        };
                        App::show_notification(counts);
                        Self::scrap_content_feeds(last_sync_datetime, feed_ids);
                    }
                    Err(error) => {
                        ContentPage::instance().newsflash_error(&i18n("Failed to sync"), error);

                        App::set_background_status(constants::BACKGROUND_IDLE);
                    }
                }
            },
        );
    }

    fn init_sync() {
        App::default().set_is_syncing(false);

        let header_maps = App::default().settings().get_feed_header_maps();

        TokioRuntime::execute_with_callback(
            || async move {
                let news_flash = App::news_flash();
                let news_flash_guad = news_flash.read().await;
                let news_flash = news_flash_guad.as_ref().ok_or(NewsFlashError::NotLoggedIn)?;
                let new_article_map = news_flash.initial_sync(&App::client(), header_maps).await?;
                let unread_map = news_flash
                    .unread_count_feed_map(App::default().settings().article_list().hide_future_articles())?;
                let (feeds, _feed_mappings) = news_flash.get_feeds()?;
                Ok((new_article_map, unread_map, feeds))
            },
            |res| {
                App::default().set_is_syncing(false);

                match res {
                    Ok((new_article_map, unread_map, feeds)) => {
                        let feed_names = feeds
                            .iter()
                            .map(|f| (f.feed_id.clone(), f.label.clone()))
                            .collect::<HashMap<FeedID, String>>();

                        ContentPage::instance().update_sidebar();
                        ArticleListColumn::instance().update_list();
                        let counts = NotificationCounts {
                            new: new_article_map,
                            unread: unread_map,
                            names: feed_names,
                        };
                        App::show_notification(counts);
                    }
                    Err(error) => ContentPage::instance().newsflash_error(&i18n("Failed to sync"), error),
                }
            },
        );
    }

    fn scrap_content_feeds(last_sync: DateTime<Utc>, feed_ids: HashSet<FeedID>) {
        let scrap_feed_ids = feed_ids
            .into_iter()
            .filter(|id| {
                App::default()
                    .settings()
                    .get_feed_settings(id)
                    .map(|settings| settings.scrap_content)
                    .unwrap_or(false)
            })
            .collect::<Vec<_>>();

        if scrap_feed_ids.is_empty() {
            App::set_background_status(constants::BACKGROUND_IDLE);
            return;
        }

        App::default().set_is_scraping_content(true);
        App::default().set_is_syncing(false);

        TokioRuntime::execute_with_callback(
            move || async move {
                let news_flash = App::news_flash();
                let news_flash_guad = news_flash.read().await;
                let news_flash = news_flash_guad.as_ref().ok_or(NewsFlashError::NotLoggedIn)?;
                news_flash
                    .scrap_content_feeds(last_sync, &scrap_feed_ids, &App::client())
                    .await
            },
            |res| {
                App::default().set_is_scraping_content(false);
                ArticleListColumn::instance().update_list();

                App::set_background_status(constants::BACKGROUND_IDLE);

                if let Err(error) = res {
                    log::warn!("Internal scraper failed: {error}");
                    ContentPage::instance().newsflash_error(&i18n("Scraper failed to extract content"), error);
                }
            },
        );
    }

    fn update_login() {
        ContentPage::instance().dismiss_notification();
        TokioRuntime::execute_with_callback(
            || async move {
                let news_flash = App::news_flash();
                let news_flash_guad = news_flash.read().await;
                let news_flash = news_flash_guad.as_ref()?;
                news_flash.get_login_data().await
            },
            |res| {
                if let Some(login_data) = res {
                    let plugin_id = login_data.id();
                    match login_data {
                        LoginData::None(_id) => log::error!("updating login for local should never happen!"),
                        LoginData::Direct(_) | LoginData::OAuth(_) => {
                            MainWindow::instance().show_login_page(&plugin_id, Some(login_data))
                        }
                    }
                }
            },
        );
    }
}
