mod article_list_column;
mod article_list_mode;
mod article_view_column;
mod sidebar_column;

pub use self::article_list_column::ArticleListColumn;
pub use self::article_list_mode::ArticleListMode;
pub use self::article_view_column::ArticleViewColumn;
pub use self::sidebar_column::SidebarColumn;

use crate::app::App;
use crate::article_list::ArticleList;
use crate::article_view::ArticleView;
use crate::error::NewsFlashGtkError;
use crate::i18n::{i18n, i18n_f};
use crate::infrastructure::TokioRuntime;
use crate::main_window::MainWindow;
use crate::sidebar::{FeedListTree, SideBar, SidebarLoader, TagListModel};
use crate::undo_action::UndoAction;
use gio::prelude::ActionGroupExt;
use glib::variant::ToVariant;
use glib::{clone, subclass};
use gtk4::{CompositeTemplate, subclass::prelude::*};
use libadwaita::{NavigationSplitView, OverlaySplitView, Toast, ToastOverlay};
use news_flash::error::NewsFlashError;
use std::cell::RefCell;

mod imp {
    use crate::{settings::SyncIntervalType, themes::StyleManager};

    use super::*;

    #[derive(Debug, Default, CompositeTemplate)]
    #[template(file = "data/resources/ui_templates/content_page.blp")]
    pub struct ContentPage {
        #[template_child]
        pub content_overlay: TemplateChild<ToastOverlay>,
        #[template_child]
        pub outer: TemplateChild<OverlaySplitView>,
        #[template_child]
        pub inner: TemplateChild<NavigationSplitView>,
        #[template_child]
        pub article_list_column: TemplateChild<ArticleListColumn>,
        #[template_child]
        pub sidebar_column: TemplateChild<SidebarColumn>,
        #[template_child]
        pub articleview_column: TemplateChild<ArticleViewColumn>,

        pub toast: RefCell<Option<Toast>>,
        pub undo_acton: RefCell<Option<UndoAction>>,
    }

    #[glib::object_subclass]
    impl ObjectSubclass for ContentPage {
        const NAME: &'static str = "ContentPage";
        type ParentType = gtk4::Box;
        type Type = super::ContentPage;

        fn class_init(klass: &mut Self::Class) {
            klass.bind_template();
        }

        fn instance_init(obj: &subclass::InitializingObject<Self>) {
            obj.init_template();
        }
    }

    impl ObjectImpl for ContentPage {
        fn constructed(&self) {
            App::default()
                .settings()
                .feed_list()
                .connect_only_show_relevant_notify(|_settings| {
                    super::ContentPage::instance().update_sidebar();
                });

            App::default().settings().feed_list().connect_order_notify(|settings| {
                SideBar::set_order(settings.order());
            });

            App::default()
                .settings()
                .article_list()
                .connect_order_notify(|_settings| {
                    ArticleListColumn::instance().new_list();
                });

            App::default()
                .settings()
                .article_list()
                .connect_order_by_notify(|_settings| {
                    ArticleListColumn::instance().new_list();
                });

            App::default()
                .settings()
                .article_list()
                .connect_show_thumbnails_notify(|_settings| {
                    ArticleListColumn::instance().new_list();
                });

            App::default()
                .settings()
                .article_list()
                .connect_hide_future_articles_notify(|_settings| {
                    super::ContentPage::instance().update_sidebar();
                    ArticleListColumn::instance().update_list();
                });

            App::default().settings().general().connect_theme_notify(|settings| {
                StyleManager::apply_theme(&settings.theme());
            });

            App::default()
                .settings()
                .article_view()
                .connect_content_width_notify(|_settings| {
                    ArticleView::instance().reload_user_data();
                });

            App::default()
                .settings()
                .article_view()
                .connect_line_height_notify(|_settings| {
                    ArticleView::instance().reload_user_data();
                });

            App::default()
                .settings()
                .article_view()
                .connect_use_custom_font_notify(|_settings| {
                    ArticleView::instance().redraw_article();
                });

            App::default()
                .settings()
                .article_view()
                .connect_font_notify(|_settings| {
                    ArticleView::instance().redraw_article();
                });

            App::default()
                .settings()
                .general()
                .connect_keep_running_in_background_notify(|settings| {
                    if !settings.keep_running_in_background() {
                        settings.set_autostart(false);
                    }
                });

            App::default()
                .settings()
                .general()
                .connect_autostart_notify(|settings| {
                    App::request_background_permission(settings.autostart());
                });

            App::default()
                .settings()
                .general()
                .connect_custom_sync_interval_notify(|settings| {
                    if settings.sync_type() == SyncIntervalType::Custom {
                        App::default().schedule_sync();
                    }
                });

            App::default()
                .settings()
                .general()
                .connect_predefined_sync_interval_notify(|settings| {
                    if settings.sync_type() == SyncIntervalType::Predefined {
                        App::default().schedule_sync();
                    }
                });

            App::default()
                .settings()
                .general()
                .connect_sync_type_notify(|_settings| {
                    App::default().schedule_sync();
                });
        }
    }

    impl WidgetImpl for ContentPage {}

    impl BoxImpl for ContentPage {}
}

glib::wrapper! {
    pub struct ContentPage(ObjectSubclass<imp::ContentPage>)
        @extends gtk4::Widget, gtk4::Box;
}

impl Default for ContentPage {
    fn default() -> Self {
        glib::Object::new::<Self>()
    }
}

impl ContentPage {
    pub fn instance() -> Self {
        MainWindow::instance().imp().content_page.get()
    }

    pub fn simple_message(&self, message: &str) {
        let imp = self.imp();
        let toast = Toast::new(message);
        toast.connect_dismissed(clone!(
            #[weak]
            imp,
            #[upgrade_or_panic]
            move |toast| {
                let remove = imp.toast.borrow().as_ref() == Some(toast);
                if remove {
                    _ = imp.toast.take();
                }
            }
        ));
        imp.content_overlay.add_toast(toast.clone());
        imp.toast.replace(Some(toast));
    }

    pub fn newsflash_error(&self, message: &str, error: NewsFlashError) {
        let imp = self.imp();
        let toast = Toast::new(message);
        toast.set_button_label(Some(&i18n("details")));
        App::default().set_newsflash_error(NewsFlashGtkError::NewsFlash {
            source: error,
            context: message.into(),
        });
        toast.set_action_name(Some("win.show-error-dialog"));
        toast.connect_dismissed(clone!(
            #[weak]
            imp,
            #[upgrade_or_panic]
            move |toast| {
                let remove = imp.toast.borrow().as_ref() == Some(toast);
                if remove {
                    _ = imp.toast.take();
                }
            }
        ));
        imp.content_overlay.add_toast(toast.clone());
        imp.toast.replace(Some(toast));
    }

    pub fn dismiss_notification(&self) {
        if let Some(toast) = self.imp().toast.take() {
            toast.dismiss();
        }
    }

    pub fn add_undo_notification(&self, action: UndoAction) {
        let imp = self.imp();

        self.dismiss_notification();

        let message = match &action {
            UndoAction::MarkRead(_ids) => i18n("Mark all read"),
            UndoAction::DeleteCategory(_id, label) => i18n_f("Deleted Category '{}'", &[label]),
            UndoAction::DeleteFeed(_id, label) => i18n_f("Deleted Feed '{}'", &[label]),
            UndoAction::DeleteTag(_id, label) => i18n_f("Deleted Tag '{}'", &[label]),
        };

        let toast = Toast::new(&message);
        toast.set_button_label(Some(&i18n("Undo")));

        match &action {
            UndoAction::MarkRead(ids) => {
                let ids = ids.clone();
                toast.connect_button_clicked(move |_toast| {
                    let ids: Vec<String> = ids.iter().map(|id| id.as_str().to_string()).collect();
                    MainWindow::activate_action("set-articles-unread", Some(&ids.to_variant()));
                });
            }
            _ => {
                toast.connect_button_clicked(clone!(
                    #[weak(rename_to = obj)]
                    self,
                    move |_toast| {
                        let imp = obj.imp();

                        imp.undo_acton.take();
                        if let Some(toast) = imp.toast.take() {
                            toast.dismiss();
                        }

                        obj.update_sidebar();
                        ArticleListColumn::instance().update_list();
                    }
                ));
            }
        }

        toast.connect_dismissed(clone!(
            #[weak]
            imp,
            #[upgrade_or_panic]
            move |toast| {
                let remove = imp.toast.borrow().as_ref() == Some(toast);
                if remove {
                    _ = imp.toast.take();
                }

                let Some(action) = imp.undo_acton.take() else {
                    return;
                };

                log::debug!("remove current action: {action}");

                match &action {
                    UndoAction::MarkRead(_ids) => {}
                    UndoAction::DeleteFeed(feed_id, _label) => {
                        MainWindow::instance().activate_action("delete-feed", Some(&feed_id.as_str().to_variant()))
                    }
                    UndoAction::DeleteCategory(category_id, _label) => MainWindow::instance()
                        .activate_action("delete-category", Some(&category_id.as_str().to_variant())),
                    UndoAction::DeleteTag(tag_id, _label) => {
                        MainWindow::instance().activate_action("delete-tag", Some(&tag_id.as_str().to_variant()))
                    }
                }
            }
        ));

        imp.content_overlay.add_toast(toast.clone());
        imp.toast.replace(Some(toast));
        imp.undo_acton.replace(Some(action));
    }

    pub fn get_current_undo_action(&self) -> Option<UndoAction> {
        self.imp().undo_acton.borrow().clone()
    }

    pub fn outer(&self) -> &OverlaySplitView {
        let imp = self.imp();
        &imp.outer
    }

    pub fn inner(&self) -> &NavigationSplitView {
        let imp = self.imp();
        &imp.inner
    }

    pub fn load_branding(&self) {
        TokioRuntime::execute_with_callback(
            || async move {
                if let Some(news_flash) = App::news_flash().read().await.as_ref() {
                    (news_flash.id().await, news_flash.user_name().await)
                } else {
                    (None, None)
                }
            },
            |(id, user)| {
                if let Some(id) = id {
                    _ = SidebarColumn::instance().set_account(&id, user.as_deref());
                } else {
                    // in case of failure show 'welcome page'
                    MainWindow::instance().show_welcome_page();
                }
            },
        );
    }

    pub fn clear(&self) {
        ArticleList::instance().clear();

        let feed_tree_model = FeedListTree::new();
        let tag_list_model = TagListModel::new();

        SideBar::instance().update(0, 0, feed_tree_model, tag_list_model);
    }

    pub fn update_sidebar(&self) {
        let loader = SidebarLoader::default()
            .hide_future_articles(App::default().settings().article_list().hide_future_articles())
            .article_list_mode(ArticleListColumn::instance().mode())
            .undo_action(self.get_current_undo_action());

        TokioRuntime::execute_with_callback(
            || async move {
                if let Some(news_flash) = App::news_flash().read().await.as_ref() {
                    loader.load(news_flash)
                } else {
                    Err(NewsFlashError::NotLoggedIn)
                }
            },
            |res| match res {
                Ok(load_result) => {
                    let (total_count, today_count, feed_list_model, tag_list_model) = load_result.build_list_models();

                    SideBar::instance().update(total_count, today_count, feed_list_model, tag_list_model);
                }
                Err(error) => {
                    Self::instance().simple_message(&i18n_f("Failed to update sidebar: '{}'", &[&error.to_string()]))
                }
            },
        );
    }
}
