/* src/services/systemd/mod.rs
 *
 * Copyright 2025 Mission Center Developers
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * SPDX-License-Identifier: GPL-3.0-or-later
 */

use std::collections::{HashMap, HashSet};
use std::sync::atomic::{AtomicBool, Ordering};
use std::sync::Arc;

use futures_lite::StreamExt;
use thiserror::Error;
use zbus::zvariant::OwnedObjectPath;
use zbus::Proxy;

use magpie_platform::services::Service;

use crate::{async_runtime, sync, system_bus};

pub use manager::ServiceManager;
use service::{ActiveState, LoadState};
use systemd_manager::SystemDManagerProxy;

mod manager;
mod service;
mod systemd_manager;

#[derive(Debug, Error)]
pub enum SystemDError {
    #[error("Failed to connect to DBus system bus")]
    DBusConnectionError,
    #[error("Failed to open `libsystemd.so.0`")]
    LibSystemDNotFound,
    #[error("DBus error: {0}")]
    DBusError(#[from] zbus::Error),
    #[error("Library loading error: {0}")]
    LibLoadingError(#[from] libloading::Error),
    #[error("IO error: {0}")]
    IoError(#[from] std::io::Error),
    #[error("Failed to open journal: {0}")]
    JournalOpenError(String),
    #[error("Seek failed: {0}")]
    JournalSeekError(String),
    #[error("Failed to add match: {0}")]
    JournalAddMatchError(String),
    #[error("Failed to add disjunction: {0}")]
    JournalAddDisjunctionError(String),
    #[error("Failed to add conjunction: {0}")]
    JournalAddConjunctionError(String),
    #[error("Failed to iterate journal entries: {0}")]
    JournalIterateError(String),
}

fn list_unit_files() -> HashSet<String> {
    fn add_from_dir(path: &str, result: &mut HashSet<String>) {
        if let Ok(dir) = std::fs::read_dir(path) {
            for entry in dir.filter_map(|e| e.ok()) {
                let name = entry.file_name();
                let name = name.to_string_lossy();
                if name.ends_with(".service")
                    || name.ends_with(".mount")
                    || name.ends_with(".socket")
                {
                    result.insert(name.replace("@", ""));
                }
            }
        }
    }

    let mut result = HashSet::new();
    add_from_dir("/lib/systemd/system", &mut result);
    add_from_dir("/usr/lib/systemd/system", &mut result);
    add_from_dir("/etc/systemd/system", &mut result);

    result
}

pub struct SystemD {
    systemd1: SystemDManagerProxy<'static>,
    service_cache: HashMap<String, (Service, (Proxy<'static>, Proxy<'static>))>,

    subscribe_ok: bool,
    full_refresh_required: Arc<AtomicBool>,

    to_remove: Vec<String>,
    new_services: Vec<(
        String,
        String,
        String,
        String,
        String,
        String,
        OwnedObjectPath,
        u32,
        String,
        OwnedObjectPath,
    )>,
}

impl SystemD {
    pub fn new() -> Result<Self, SystemDError> {
        let bus = match system_bus() {
            Some(bus) => bus,
            None => {
                return Err(SystemDError::DBusConnectionError);
            }
        };

        let rt = async_runtime();
        let systemd1 = sync!(rt, SystemDManagerProxy::new(&bus))?;

        let full_refresh_required = Arc::new(AtomicBool::new(true));

        let subscribe_ok = if let Err(err) = sync!(rt, systemd1.subscribe()) {
            log::error!("Failed to subscribe to SystemD notifications, service refresh will be significantly more expensive: {err}");
            false
        } else {
            rt.spawn({
                let systemd1 = systemd1.clone();
                let full_refresh_required = full_refresh_required.clone();
                async move {
                    let mut unit_files_changed = match systemd1.receive_unit_files_changed().await {
                        Ok(unit_files_changed) => unit_files_changed,
                        Err(e) => {
                            log::error!("Failed to receive unit files changed: {e}");
                            return;
                        }
                    };

                    let mut reloading = match systemd1.receive_reloading().await {
                        Ok(reloading) => reloading,
                        Err(e) => {
                            log::error!("Failed to receive reloading: {e}");
                            return;
                        }
                    };

                    loop {
                        tokio::select! {
                            _ = unit_files_changed.next() => {
                                full_refresh_required.store(true, Ordering::Release);
                            }
                            _ = reloading.next() => {
                                full_refresh_required.store(true, Ordering::Release);
                            }
                        }
                    }
                }
            });

            true
        };

        Ok(Self {
            systemd1,
            service_cache: HashMap::new(),

            subscribe_ok,
            full_refresh_required,

            to_remove: Vec::new(),
            new_services: Vec::new(),
        })
    }

    pub fn list_services(&mut self) -> Result<Vec<Service>, SystemDError> {
        let rt = async_runtime();

        if self
            .full_refresh_required
            .fetch_and(!self.subscribe_ok, Ordering::Acquire)
        {
            let mut unit_files = list_unit_files();

            let mut services = Vec::new();
            if !unit_files.is_empty() {
                let unit_files = unit_files.drain().collect::<Vec<_>>();
                services =
                    sync!(rt, self.systemd1.list_units_by_names(&unit_files)).unwrap_or_else(|e| {
                        log::warn!("Failed to list unit files: {e}");
                        Vec::new()
                    })
            }

            if services.is_empty() {
                services = sync!(rt, self.systemd1.list_units())?
                    .drain(..)
                    .filter(|s| {
                        s.0.ends_with(".service")
                            || s.0.ends_with(".mount")
                            || s.0.ends_with(".socket")
                    })
                    .collect()
            }

            for service in self.service_cache.keys() {
                if !services.iter().any(|(name, ..)| name == service) {
                    self.to_remove.push(service.clone());
                }
            }

            for name in self.to_remove.drain(..) {
                self.service_cache.remove(&name);
            }

            for service in services.drain(..) {
                let name = &service.0;
                if self.service_cache.contains_key(name) {
                    continue;
                }

                self.new_services.push(service);
            }

            let mut new_services = self
                .new_services
                .drain(..)
                .filter_map(
                    |(
                        name,
                        description,
                        load_state,
                        active_state,
                        _sub_state,
                        _following,
                        unit_obj_path,
                        _job_id,
                        _job_type,
                        _job_obj_path,
                    )| {
                        let load_state: LoadState = load_state.as_str().into();
                        if load_state == LoadState::NotFound {
                            return None;
                        }

                        let active_state: ActiveState = active_state.as_str().into();

                        Some((
                            unit_obj_path,
                            Service {
                                id: name,
                                description: if description.is_empty() {
                                    None
                                } else {
                                    Some(description)
                                },
                                enabled: false,
                                running: active_state == ActiveState::Active,
                                failed: active_state == ActiveState::Failed,
                                pid: None,
                                user: None,
                                group: None,
                            },
                        ))
                    },
                )
                .collect::<Vec<_>>();

            let bus = match system_bus() {
                Some(bus) => bus,
                None => {
                    return Err(SystemDError::DBusConnectionError);
                }
            };

            for (unit_path, service) in new_services.drain(..) {
                let Ok(unit_proxy) = sync!(
                    rt,
                    Proxy::new(
                        bus,
                        "org.freedesktop.systemd1",
                        unit_path.clone(),
                        "org.freedesktop.systemd1.Unit",
                    )
                ) else {
                    continue;
                };

                let Ok(service_proxy) = sync!(
                    rt,
                    Proxy::new(
                        bus,
                        "org.freedesktop.systemd1",
                        unit_path,
                        "org.freedesktop.systemd1.Service",
                    )
                ) else {
                    continue;
                };

                self.service_cache
                    .insert(service.id.clone(), (service, (unit_proxy, service_proxy)));
            }
        }

        for (service, (unit_proxy, service_proxy)) in self.service_cache.values_mut() {
            let _ = sync!(
                rt,
                Self::update_service_properties(service, unit_proxy.clone(), service_proxy.clone())
            );
        }

        Ok(self
            .service_cache
            .values()
            .map(|(s, _)| s.clone())
            .collect())
    }

    #[inline]
    async fn update_service_properties(
        service: &mut Service,
        unit_proxy: Proxy<'static>,
        service_proxy: Proxy<'static>,
    ) -> Result<(), SystemDError> {
        let enabled = unit_proxy.get_property::<String>("UnitFileState").await?;
        service.enabled = enabled.to_ascii_lowercase() == "enabled";

        let pid = service_proxy.get_property::<u32>("MainPID").await?;
        let user = service_proxy.get_property::<String>("User").await?;
        let group = service_proxy.get_property::<String>("Group").await?;

        service.pid = if pid == 0 { None } else { Some(pid) };
        service.user = if user.is_empty() { None } else { Some(user) };
        service.group = if group.is_empty() { None } else { Some(group) };

        Ok(())
    }
}

#[cfg(test)]
mod tests {
    use super::*;
    use test_log::test;

    #[test]
    fn test_list_services() -> Result<(), anyhow::Error> {
        let mut systemd = SystemD::new()?;
        let services = systemd.list_services()?;
        assert!(!services.is_empty());
        dbg!(services);

        Ok(())
    }
}
