// Based on the C implementation from Marc Stevens and Dan Shumow.
// https://github.com/cr-marcstevens/sha1collisiondetection

package ubc

const (
	CheckSize = 80

	DV_I_43_0_bit  = (uint32)(1 << 0)
	DV_I_44_0_bit  = (uint32)(1 << 1)
	DV_I_45_0_bit  = (uint32)(1 << 2)
	DV_I_46_0_bit  = (uint32)(1 << 3)
	DV_I_46_2_bit  = (uint32)(1 << 4)
	DV_I_47_0_bit  = (uint32)(1 << 5)
	DV_I_47_2_bit  = (uint32)(1 << 6)
	DV_I_48_0_bit  = (uint32)(1 << 7)
	DV_I_48_2_bit  = (uint32)(1 << 8)
	DV_I_49_0_bit  = (uint32)(1 << 9)
	DV_I_49_2_bit  = (uint32)(1 << 10)
	DV_I_50_0_bit  = (uint32)(1 << 11)
	DV_I_50_2_bit  = (uint32)(1 << 12)
	DV_I_51_0_bit  = (uint32)(1 << 13)
	DV_I_51_2_bit  = (uint32)(1 << 14)
	DV_I_52_0_bit  = (uint32)(1 << 15)
	DV_II_45_0_bit = (uint32)(1 << 16)
	DV_II_46_0_bit = (uint32)(1 << 17)
	DV_II_46_2_bit = (uint32)(1 << 18)
	DV_II_47_0_bit = (uint32)(1 << 19)
	DV_II_48_0_bit = (uint32)(1 << 20)
	DV_II_49_0_bit = (uint32)(1 << 21)
	DV_II_49_2_bit = (uint32)(1 << 22)
	DV_II_50_0_bit = (uint32)(1 << 23)
	DV_II_50_2_bit = (uint32)(1 << 24)
	DV_II_51_0_bit = (uint32)(1 << 25)
	DV_II_51_2_bit = (uint32)(1 << 26)
	DV_II_52_0_bit = (uint32)(1 << 27)
	DV_II_53_0_bit = (uint32)(1 << 28)
	DV_II_54_0_bit = (uint32)(1 << 29)
	DV_II_55_0_bit = (uint32)(1 << 30)
	DV_II_56_0_bit = (uint32)(1 << 31)
)

// sha1_dvs contains a list of SHA-1 Disturbance Vectors (DV) which defines the
// unavoidable bit conditions when a collision attack is in progress.
var sha1_dvs = []DvInfo{
	{
		DvType: 1, DvK: 43, DvB: 0, TestT: 58, MaskI: 0, MaskB: 0,
		Dm: [CheckSize]uint32{
			0x08000000, 0x9800000c, 0xd8000010, 0x08000010, 0xb8000010, 0x98000000, 0x60000000,
			0x00000008, 0xc0000000, 0x90000014, 0x10000010, 0xb8000014, 0x28000000, 0x20000010,
			0x48000000, 0x08000018, 0x60000000, 0x90000010, 0xf0000010, 0x90000008, 0xc0000000,
			0x90000010, 0xf0000010, 0xb0000008, 0x40000000, 0x90000000, 0xf0000010, 0x90000018,
			0x60000000, 0x90000010, 0x90000010, 0x90000000, 0x80000000, 0x00000010, 0xa0000000,
			0x20000000, 0xa0000000, 0x20000010, 0x00000000, 0x20000010, 0x20000000, 0x00000010,
			0x20000000, 0x00000010, 0xa0000000, 0x00000000, 0x20000000, 0x20000000, 0x00000000,
			0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
			0x00000000, 0x00000000, 0x00000001, 0x00000020, 0x00000001, 0x40000002, 0x40000040,
			0x40000002, 0x80000004, 0x80000080, 0x80000006, 0x00000049, 0x00000103, 0x80000009,
			0x80000012, 0x80000202, 0x00000018, 0x00000164, 0x00000408, 0x800000e6, 0x8000004c,
			0x00000803, 0x80000161, 0x80000599},
	}, {
		DvType: 1, DvK: 44, DvB: 0, TestT: 58, MaskI: 0, MaskB: 1,
		Dm: [CheckSize]uint32{
			0xb4000008, 0x08000000, 0x9800000c, 0xd8000010, 0x08000010, 0xb8000010, 0x98000000,
			0x60000000, 0x00000008, 0xc0000000, 0x90000014, 0x10000010, 0xb8000014, 0x28000000,
			0x20000010, 0x48000000, 0x08000018, 0x60000000, 0x90000010, 0xf0000010, 0x90000008,
			0xc0000000, 0x90000010, 0xf0000010, 0xb0000008, 0x40000000, 0x90000000, 0xf0000010,
			0x90000018, 0x60000000, 0x90000010, 0x90000010, 0x90000000, 0x80000000, 0x00000010,
			0xa0000000, 0x20000000, 0xa0000000, 0x20000010, 0x00000000, 0x20000010, 0x20000000,
			0x00000010, 0x20000000, 0x00000010, 0xa0000000, 0x00000000, 0x20000000, 0x20000000,
			0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
			0x00000000, 0x00000000, 0x00000000, 0x00000001, 0x00000020, 0x00000001, 0x40000002,
			0x40000040, 0x40000002, 0x80000004, 0x80000080, 0x80000006, 0x00000049, 0x00000103,
			0x80000009, 0x80000012, 0x80000202, 0x00000018, 0x00000164, 0x00000408, 0x800000e6,
			0x8000004c, 0x00000803, 0x80000161},
	},
	{
		DvType: 1, DvK: 45, DvB: 0, TestT: 58, MaskI: 0, MaskB: 2,
		Dm: [CheckSize]uint32{
			0xf4000014, 0xb4000008, 0x08000000, 0x9800000c, 0xd8000010, 0x08000010, 0xb8000010,
			0x98000000, 0x60000000, 0x00000008, 0xc0000000, 0x90000014, 0x10000010, 0xb8000014,
			0x28000000, 0x20000010, 0x48000000, 0x08000018, 0x60000000, 0x90000010, 0xf0000010,
			0x90000008, 0xc0000000, 0x90000010, 0xf0000010, 0xb0000008, 0x40000000, 0x90000000,
			0xf0000010, 0x90000018, 0x60000000, 0x90000010, 0x90000010, 0x90000000, 0x80000000,
			0x00000010, 0xa0000000, 0x20000000, 0xa0000000, 0x20000010, 0x00000000, 0x20000010,
			0x20000000, 0x00000010, 0x20000000, 0x00000010, 0xa0000000, 0x00000000, 0x20000000,
			0x20000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
			0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000001, 0x00000020, 0x00000001,
			0x40000002, 0x40000040, 0x40000002, 0x80000004, 0x80000080, 0x80000006, 0x00000049,
			0x00000103, 0x80000009, 0x80000012, 0x80000202, 0x00000018, 0x00000164, 0x00000408,
			0x800000e6, 0x8000004c, 0x00000803},
	},
	{
		DvType: 1, DvK: 46, DvB: 0, TestT: 58, MaskI: 0, MaskB: 3,
		Dm: [CheckSize]uint32{
			0x2c000010, 0xf4000014, 0xb4000008, 0x08000000, 0x9800000c, 0xd8000010, 0x08000010,
			0xb8000010, 0x98000000, 0x60000000, 0x00000008, 0xc0000000, 0x90000014, 0x10000010,
			0xb8000014, 0x28000000, 0x20000010, 0x48000000, 0x08000018, 0x60000000, 0x90000010,
			0xf0000010, 0x90000008, 0xc0000000, 0x90000010, 0xf0000010, 0xb0000008, 0x40000000,
			0x90000000, 0xf0000010, 0x90000018, 0x60000000, 0x90000010, 0x90000010, 0x90000000,
			0x80000000, 0x00000010, 0xa0000000, 0x20000000, 0xa0000000, 0x20000010, 0x00000000,
			0x20000010, 0x20000000, 0x00000010, 0x20000000, 0x00000010, 0xa0000000, 0x00000000,
			0x20000000, 0x20000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
			0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000001, 0x00000020,
			0x00000001, 0x40000002, 0x40000040, 0x40000002, 0x80000004, 0x80000080, 0x80000006,
			0x00000049, 0x00000103, 0x80000009, 0x80000012, 0x80000202, 0x00000018, 0x00000164,
			0x00000408, 0x800000e6, 0x8000004c},
	},
	{
		DvType: 1, DvK: 46, DvB: 2, TestT: 58, MaskI: 0, MaskB: 4,
		Dm: [CheckSize]uint32{
			0xb0000040, 0xd0000053, 0xd0000022, 0x20000000, 0x60000032, 0x60000043,
			0x20000040, 0xe0000042, 0x60000002, 0x80000001, 0x00000020, 0x00000003,
			0x40000052, 0x40000040, 0xe0000052, 0xa0000000, 0x80000040, 0x20000001,
			0x20000060, 0x80000001, 0x40000042, 0xc0000043, 0x40000022, 0x00000003,
			0x40000042, 0xc0000043, 0xc0000022, 0x00000001, 0x40000002, 0xc0000043,
			0x40000062, 0x80000001, 0x40000042, 0x40000042, 0x40000002, 0x00000002,
			0x00000040, 0x80000002, 0x80000000, 0x80000002, 0x80000040, 0x00000000,
			0x80000040, 0x80000000, 0x00000040, 0x80000000, 0x00000040, 0x80000002,
			0x00000000, 0x80000000, 0x80000000, 0x00000000, 0x00000000, 0x00000000,
			0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
			0x00000000, 0x00000004, 0x00000080, 0x00000004, 0x00000009, 0x00000101,
			0x00000009, 0x00000012, 0x00000202, 0x0000001a, 0x00000124, 0x0000040c,
			0x00000026, 0x0000004a, 0x0000080a, 0x00000060, 0x00000590, 0x00001020,
			0x0000039a, 0x00000132},
	},
	{
		DvType: 1, DvK: 47, DvB: 0, TestT: 58, MaskI: 0, MaskB: 5,
		Dm: [CheckSize]uint32{
			0xc8000010, 0x2c000010, 0xf4000014, 0xb4000008, 0x08000000, 0x9800000c,
			0xd8000010, 0x08000010, 0xb8000010, 0x98000000, 0x60000000, 0x00000008,
			0xc0000000, 0x90000014, 0x10000010, 0xb8000014, 0x28000000, 0x20000010,
			0x48000000, 0x08000018, 0x60000000, 0x90000010, 0xf0000010, 0x90000008,
			0xc0000000, 0x90000010, 0xf0000010, 0xb0000008, 0x40000000, 0x90000000,
			0xf0000010, 0x90000018, 0x60000000, 0x90000010, 0x90000010, 0x90000000,
			0x80000000, 0x00000010, 0xa0000000, 0x20000000, 0xa0000000, 0x20000010,
			0x00000000, 0x20000010, 0x20000000, 0x00000010, 0x20000000, 0x00000010,
			0xa0000000, 0x00000000, 0x20000000, 0x20000000, 0x00000000, 0x00000000,
			0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
			0x00000000, 0x00000000, 0x00000001, 0x00000020, 0x00000001, 0x40000002,
			0x40000040, 0x40000002, 0x80000004, 0x80000080, 0x80000006, 0x00000049,
			0x00000103, 0x80000009, 0x80000012, 0x80000202, 0x00000018, 0x00000164,
			0x00000408, 0x800000e6},
	},
	{
		DvType: 1, DvK: 47, DvB: 2, TestT: 58, MaskI: 0, MaskB: 6,
		Dm: [CheckSize]uint32{
			0x20000043, 0xb0000040, 0xd0000053, 0xd0000022, 0x20000000, 0x60000032,
			0x60000043, 0x20000040, 0xe0000042, 0x60000002, 0x80000001, 0x00000020,
			0x00000003, 0x40000052, 0x40000040, 0xe0000052, 0xa0000000, 0x80000040,
			0x20000001, 0x20000060, 0x80000001, 0x40000042, 0xc0000043, 0x40000022,
			0x00000003, 0x40000042, 0xc0000043, 0xc0000022, 0x00000001, 0x40000002,
			0xc0000043, 0x40000062, 0x80000001, 0x40000042, 0x40000042, 0x40000002,
			0x00000002, 0x00000040, 0x80000002, 0x80000000, 0x80000002, 0x80000040,
			0x00000000, 0x80000040, 0x80000000, 0x00000040, 0x80000000, 0x00000040,
			0x80000002, 0x00000000, 0x80000000, 0x80000000, 0x00000000, 0x00000000,
			0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
			0x00000000, 0x00000000, 0x00000004, 0x00000080, 0x00000004, 0x00000009,
			0x00000101, 0x00000009, 0x00000012, 0x00000202, 0x0000001a, 0x00000124,
			0x0000040c, 0x00000026, 0x0000004a, 0x0000080a, 0x00000060, 0x00000590,
			0x00001020, 0x0000039a,
		},
	},
	{
		DvType: 1, DvK: 48, DvB: 0, TestT: 58, MaskI: 0, MaskB: 7,
		Dm: [CheckSize]uint32{
			0xb800000a, 0xc8000010, 0x2c000010, 0xf4000014, 0xb4000008, 0x08000000,
			0x9800000c, 0xd8000010, 0x08000010, 0xb8000010, 0x98000000, 0x60000000,
			0x00000008, 0xc0000000, 0x90000014, 0x10000010, 0xb8000014, 0x28000000,
			0x20000010, 0x48000000, 0x08000018, 0x60000000, 0x90000010, 0xf0000010,
			0x90000008, 0xc0000000, 0x90000010, 0xf0000010, 0xb0000008, 0x40000000,
			0x90000000, 0xf0000010, 0x90000018, 0x60000000, 0x90000010, 0x90000010,
			0x90000000, 0x80000000, 0x00000010, 0xa0000000, 0x20000000, 0xa0000000,
			0x20000010, 0x00000000, 0x20000010, 0x20000000, 0x00000010, 0x20000000,
			0x00000010, 0xa0000000, 0x00000000, 0x20000000, 0x20000000, 0x00000000,
			0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
			0x00000000, 0x00000000, 0x00000000, 0x00000001, 0x00000020, 0x00000001,
			0x40000002, 0x40000040, 0x40000002, 0x80000004, 0x80000080, 0x80000006,
			0x00000049, 0x00000103, 0x80000009, 0x80000012, 0x80000202, 0x00000018,
			0x00000164, 0x00000408,
		},
	},
	{
		DvType: 1, DvK: 48, DvB: 2, TestT: 58, MaskI: 0, MaskB: 8,
		Dm: [CheckSize]uint32{
			0xe000002a, 0x20000043, 0xb0000040, 0xd0000053, 0xd0000022, 0x20000000,
			0x60000032, 0x60000043, 0x20000040, 0xe0000042, 0x60000002, 0x80000001,
			0x00000020, 0x00000003, 0x40000052, 0x40000040, 0xe0000052, 0xa0000000,
			0x80000040, 0x20000001, 0x20000060, 0x80000001, 0x40000042, 0xc0000043,
			0x40000022, 0x00000003, 0x40000042, 0xc0000043, 0xc0000022, 0x00000001,
			0x40000002, 0xc0000043, 0x40000062, 0x80000001, 0x40000042, 0x40000042,
			0x40000002, 0x00000002, 0x00000040, 0x80000002, 0x80000000, 0x80000002,
			0x80000040, 0x00000000, 0x80000040, 0x80000000, 0x00000040, 0x80000000,
			0x00000040, 0x80000002, 0x00000000, 0x80000000, 0x80000000, 0x00000000,
			0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
			0x00000000, 0x00000000, 0x00000000, 0x00000004, 0x00000080, 0x00000004,
			0x00000009, 0x00000101, 0x00000009, 0x00000012, 0x00000202, 0x0000001a,
			0x00000124, 0x0000040c, 0x00000026, 0x0000004a, 0x0000080a, 0x00000060,
			0x00000590, 0x00001020},
	},
	{
		DvType: 1, DvK: 49, DvB: 0, TestT: 58, MaskI: 0, MaskB: 9,
		Dm: [CheckSize]uint32{
			0x18000000, 0xb800000a, 0xc8000010, 0x2c000010, 0xf4000014, 0xb4000008,
			0x08000000, 0x9800000c, 0xd8000010, 0x08000010, 0xb8000010, 0x98000000,
			0x60000000, 0x00000008, 0xc0000000, 0x90000014, 0x10000010, 0xb8000014,
			0x28000000, 0x20000010, 0x48000000, 0x08000018, 0x60000000, 0x90000010,
			0xf0000010, 0x90000008, 0xc0000000, 0x90000010, 0xf0000010, 0xb0000008,
			0x40000000, 0x90000000, 0xf0000010, 0x90000018, 0x60000000, 0x90000010,
			0x90000010, 0x90000000, 0x80000000, 0x00000010, 0xa0000000, 0x20000000,
			0xa0000000, 0x20000010, 0x00000000, 0x20000010, 0x20000000, 0x00000010,
			0x20000000, 0x00000010, 0xa0000000, 0x00000000, 0x20000000, 0x20000000,
			0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
			0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000001, 0x00000020,
			0x00000001, 0x40000002, 0x40000040, 0x40000002, 0x80000004, 0x80000080,
			0x80000006, 0x00000049, 0x00000103, 0x80000009, 0x80000012, 0x80000202,
			0x00000018, 0x00000164},
	},
	{
		DvType: 1, DvK: 49, DvB: 2, TestT: 58, MaskI: 0, MaskB: 10,
		Dm: [CheckSize]uint32{
			0x60000000, 0xe000002a, 0x20000043, 0xb0000040, 0xd0000053, 0xd0000022,
			0x20000000, 0x60000032, 0x60000043, 0x20000040, 0xe0000042, 0x60000002,
			0x80000001, 0x00000020, 0x00000003, 0x40000052, 0x40000040, 0xe0000052,
			0xa0000000, 0x80000040, 0x20000001, 0x20000060, 0x80000001, 0x40000042,
			0xc0000043, 0x40000022, 0x00000003, 0x40000042, 0xc0000043, 0xc0000022,
			0x00000001, 0x40000002, 0xc0000043, 0x40000062, 0x80000001, 0x40000042,
			0x40000042, 0x40000002, 0x00000002, 0x00000040, 0x80000002, 0x80000000,
			0x80000002, 0x80000040, 0x00000000, 0x80000040, 0x80000000, 0x00000040,
			0x80000000, 0x00000040, 0x80000002, 0x00000000, 0x80000000, 0x80000000,
			0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
			0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000004, 0x00000080,
			0x00000004, 0x00000009, 0x00000101, 0x00000009, 0x00000012, 0x00000202,
			0x0000001a, 0x00000124, 0x0000040c, 0x00000026, 0x0000004a, 0x0000080a,
			0x00000060, 0x00000590},
	},
	{
		DvType: 1, DvK: 50, DvB: 0, TestT: 65, MaskI: 0, MaskB: 11,
		Dm: [CheckSize]uint32{
			0x0800000c, 0x18000000, 0xb800000a, 0xc8000010, 0x2c000010, 0xf4000014,
			0xb4000008, 0x08000000, 0x9800000c, 0xd8000010, 0x08000010, 0xb8000010,
			0x98000000, 0x60000000, 0x00000008, 0xc0000000, 0x90000014, 0x10000010,
			0xb8000014, 0x28000000, 0x20000010, 0x48000000, 0x08000018, 0x60000000,
			0x90000010, 0xf0000010, 0x90000008, 0xc0000000, 0x90000010, 0xf0000010,
			0xb0000008, 0x40000000, 0x90000000, 0xf0000010, 0x90000018, 0x60000000,
			0x90000010, 0x90000010, 0x90000000, 0x80000000, 0x00000010, 0xa0000000,
			0x20000000, 0xa0000000, 0x20000010, 0x00000000, 0x20000010, 0x20000000,
			0x00000010, 0x20000000, 0x00000010, 0xa0000000, 0x00000000, 0x20000000,
			0x20000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
			0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000001,
			0x00000020, 0x00000001, 0x40000002, 0x40000040, 0x40000002, 0x80000004,
			0x80000080, 0x80000006, 0x00000049, 0x00000103, 0x80000009, 0x80000012,
			0x80000202, 0x00000018,
		},
	},
	{
		DvType: 1, DvK: 50, DvB: 2, TestT: 65, MaskI: 0, MaskB: 12,
		Dm: [CheckSize]uint32{
			0x20000030, 0x60000000, 0xe000002a, 0x20000043, 0xb0000040, 0xd0000053,
			0xd0000022, 0x20000000, 0x60000032, 0x60000043, 0x20000040, 0xe0000042,
			0x60000002, 0x80000001, 0x00000020, 0x00000003, 0x40000052, 0x40000040,
			0xe0000052, 0xa0000000, 0x80000040, 0x20000001, 0x20000060, 0x80000001,
			0x40000042, 0xc0000043, 0x40000022, 0x00000003, 0x40000042, 0xc0000043,
			0xc0000022, 0x00000001, 0x40000002, 0xc0000043, 0x40000062, 0x80000001,
			0x40000042, 0x40000042, 0x40000002, 0x00000002, 0x00000040, 0x80000002,
			0x80000000, 0x80000002, 0x80000040, 0x00000000, 0x80000040, 0x80000000,
			0x00000040, 0x80000000, 0x00000040, 0x80000002, 0x00000000, 0x80000000,
			0x80000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
			0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000004,
			0x00000080, 0x00000004, 0x00000009, 0x00000101, 0x00000009, 0x00000012,
			0x00000202, 0x0000001a, 0x00000124, 0x0000040c, 0x00000026, 0x0000004a,
			0x0000080a, 0x00000060},
	},
	{
		DvType: 1, DvK: 51, DvB: 0, TestT: 65, MaskI: 0, MaskB: 13,
		Dm: [CheckSize]uint32{
			0xe8000000, 0x0800000c, 0x18000000, 0xb800000a, 0xc8000010, 0x2c000010,
			0xf4000014, 0xb4000008, 0x08000000, 0x9800000c, 0xd8000010, 0x08000010,
			0xb8000010, 0x98000000, 0x60000000, 0x00000008, 0xc0000000, 0x90000014,
			0x10000010, 0xb8000014, 0x28000000, 0x20000010, 0x48000000, 0x08000018,
			0x60000000, 0x90000010, 0xf0000010, 0x90000008, 0xc0000000, 0x90000010,
			0xf0000010, 0xb0000008, 0x40000000, 0x90000000, 0xf0000010, 0x90000018,
			0x60000000, 0x90000010, 0x90000010, 0x90000000, 0x80000000, 0x00000010,
			0xa0000000, 0x20000000, 0xa0000000, 0x20000010, 0x00000000, 0x20000010,
			0x20000000, 0x00000010, 0x20000000, 0x00000010, 0xa0000000, 0x00000000,
			0x20000000, 0x20000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
			0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
			0x00000001, 0x00000020, 0x00000001, 0x40000002, 0x40000040, 0x40000002,
			0x80000004, 0x80000080, 0x80000006, 0x00000049, 0x00000103, 0x80000009,
			0x80000012, 0x80000202},
	},
	{
		DvType: 1, DvK: 51, DvB: 2, TestT: 65, MaskI: 0, MaskB: 14,
		Dm: [CheckSize]uint32{
			0xa0000003, 0x20000030, 0x60000000, 0xe000002a, 0x20000043, 0xb0000040,
			0xd0000053, 0xd0000022, 0x20000000, 0x60000032, 0x60000043, 0x20000040,
			0xe0000042, 0x60000002, 0x80000001, 0x00000020, 0x00000003, 0x40000052,
			0x40000040, 0xe0000052, 0xa0000000, 0x80000040, 0x20000001, 0x20000060,
			0x80000001, 0x40000042, 0xc0000043, 0x40000022, 0x00000003, 0x40000042,
			0xc0000043, 0xc0000022, 0x00000001, 0x40000002, 0xc0000043, 0x40000062,
			0x80000001, 0x40000042, 0x40000042, 0x40000002, 0x00000002, 0x00000040,
			0x80000002, 0x80000000, 0x80000002, 0x80000040, 0x00000000, 0x80000040,
			0x80000000, 0x00000040, 0x80000000, 0x00000040, 0x80000002, 0x00000000,
			0x80000000, 0x80000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
			0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
			0x00000004, 0x00000080, 0x00000004, 0x00000009, 0x00000101, 0x00000009,
			0x00000012, 0x00000202, 0x0000001a, 0x00000124, 0x0000040c, 0x00000026,
			0x0000004a, 0x0000080a},
	},
	{
		DvType: 1, DvK: 52, DvB: 0, TestT: 65, MaskI: 0, MaskB: 15,
		Dm: [CheckSize]uint32{
			0x04000010, 0xe8000000, 0x0800000c, 0x18000000, 0xb800000a, 0xc8000010,
			0x2c000010, 0xf4000014, 0xb4000008, 0x08000000, 0x9800000c, 0xd8000010,
			0x08000010, 0xb8000010, 0x98000000, 0x60000000, 0x00000008, 0xc0000000,
			0x90000014, 0x10000010, 0xb8000014, 0x28000000, 0x20000010, 0x48000000,
			0x08000018, 0x60000000, 0x90000010, 0xf0000010, 0x90000008, 0xc0000000,
			0x90000010, 0xf0000010, 0xb0000008, 0x40000000, 0x90000000, 0xf0000010,
			0x90000018, 0x60000000, 0x90000010, 0x90000010, 0x90000000, 0x80000000,
			0x00000010, 0xa0000000, 0x20000000, 0xa0000000, 0x20000010, 0x00000000,
			0x20000010, 0x20000000, 0x00000010, 0x20000000, 0x00000010, 0xa0000000,
			0x00000000, 0x20000000, 0x20000000, 0x00000000, 0x00000000, 0x00000000,
			0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
			0x00000000, 0x00000001, 0x00000020, 0x00000001, 0x40000002, 0x40000040,
			0x40000002, 0x80000004, 0x80000080, 0x80000006, 0x00000049, 0x00000103,
			0x80000009, 0x80000012},
	},
	{
		DvType: 2, DvK: 45, DvB: 0, TestT: 58, MaskI: 0, MaskB: 16,
		Dm: [CheckSize]uint32{
			0xec000014, 0x0c000002, 0xc0000010, 0xb400001c, 0x2c000004, 0xbc000018,
			0xb0000010, 0x0000000c, 0xb8000010, 0x08000018, 0x78000010, 0x08000014,
			0x70000010, 0xb800001c, 0xe8000000, 0xb0000004, 0x58000010, 0xb000000c,
			0x48000000, 0xb0000000, 0xb8000010, 0x98000010, 0xa0000000, 0x00000000,
			0x00000000, 0x20000000, 0x80000000, 0x00000010, 0x00000000, 0x20000010,
			0x20000000, 0x00000010, 0x60000000, 0x00000018, 0xe0000000, 0x90000000,
			0x30000010, 0xb0000000, 0x20000000, 0x20000000, 0xa0000000, 0x00000010,
			0x80000000, 0x20000000, 0x20000000, 0x20000000, 0x80000000, 0x00000010,
			0x00000000, 0x20000010, 0xa0000000, 0x00000000, 0x20000000, 0x20000000,
			0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
			0x00000001, 0x00000020, 0x00000001, 0x40000002, 0x40000041, 0x40000022,
			0x80000005, 0xc0000082, 0xc0000046, 0x4000004b, 0x80000107, 0x00000089,
			0x00000014, 0x8000024b, 0x0000011b, 0x8000016d, 0x8000041a, 0x000002e4,
			0x80000054, 0x00000967},
	},
	{
		DvType: 2, DvK: 46, DvB: 0, TestT: 58, MaskI: 0, MaskB: 17,
		Dm: [CheckSize]uint32{
			0x2400001c, 0xec000014, 0x0c000002, 0xc0000010, 0xb400001c, 0x2c000004,
			0xbc000018, 0xb0000010, 0x0000000c, 0xb8000010, 0x08000018, 0x78000010,
			0x08000014, 0x70000010, 0xb800001c, 0xe8000000, 0xb0000004, 0x58000010,
			0xb000000c, 0x48000000, 0xb0000000, 0xb8000010, 0x98000010, 0xa0000000,
			0x00000000, 0x00000000, 0x20000000, 0x80000000, 0x00000010, 0x00000000,
			0x20000010, 0x20000000, 0x00000010, 0x60000000, 0x00000018, 0xe0000000,
			0x90000000, 0x30000010, 0xb0000000, 0x20000000, 0x20000000, 0xa0000000,
			0x00000010, 0x80000000, 0x20000000, 0x20000000, 0x20000000, 0x80000000,
			0x00000010, 0x00000000, 0x20000010, 0xa0000000, 0x00000000, 0x20000000,
			0x20000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
			0x00000000, 0x00000001, 0x00000020, 0x00000001, 0x40000002, 0x40000041,
			0x40000022, 0x80000005, 0xc0000082, 0xc0000046, 0x4000004b, 0x80000107,
			0x00000089, 0x00000014, 0x8000024b, 0x0000011b, 0x8000016d, 0x8000041a,
			0x000002e4, 0x80000054},
	},
	{
		DvType: 2, DvK: 46, DvB: 2, TestT: 58, MaskI: 0, MaskB: 18,
		Dm: [CheckSize]uint32{
			0x90000070, 0xb0000053, 0x30000008, 0x00000043, 0xd0000072, 0xb0000010,
			0xf0000062, 0xc0000042, 0x00000030, 0xe0000042, 0x20000060, 0xe0000041,
			0x20000050, 0xc0000041, 0xe0000072, 0xa0000003, 0xc0000012, 0x60000041,
			0xc0000032, 0x20000001, 0xc0000002, 0xe0000042, 0x60000042, 0x80000002,
			0x00000000, 0x00000000, 0x80000000, 0x00000002, 0x00000040, 0x00000000,
			0x80000040, 0x80000000, 0x00000040, 0x80000001, 0x00000060, 0x80000003,
			0x40000002, 0xc0000040, 0xc0000002, 0x80000000, 0x80000000, 0x80000002,
			0x00000040, 0x00000002, 0x80000000, 0x80000000, 0x80000000, 0x00000002,
			0x00000040, 0x00000000, 0x80000040, 0x80000002, 0x00000000, 0x80000000,
			0x80000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
			0x00000000, 0x00000004, 0x00000080, 0x00000004, 0x00000009, 0x00000105,
			0x00000089, 0x00000016, 0x0000020b, 0x0000011b, 0x0000012d, 0x0000041e,
			0x00000224, 0x00000050, 0x0000092e, 0x0000046c, 0x000005b6, 0x0000106a,
			0x00000b90, 0x00000152},
	},
	{
		DvType: 2, DvK: 47, DvB: 0, TestT: 58, MaskI: 0, MaskB: 19,
		Dm: [CheckSize]uint32{
			0x20000010, 0x2400001c, 0xec000014, 0x0c000002, 0xc0000010, 0xb400001c,
			0x2c000004, 0xbc000018, 0xb0000010, 0x0000000c, 0xb8000010, 0x08000018,
			0x78000010, 0x08000014, 0x70000010, 0xb800001c, 0xe8000000, 0xb0000004,
			0x58000010, 0xb000000c, 0x48000000, 0xb0000000, 0xb8000010, 0x98000010,
			0xa0000000, 0x00000000, 0x00000000, 0x20000000, 0x80000000, 0x00000010,
			0x00000000, 0x20000010, 0x20000000, 0x00000010, 0x60000000, 0x00000018,
			0xe0000000, 0x90000000, 0x30000010, 0xb0000000, 0x20000000, 0x20000000,
			0xa0000000, 0x00000010, 0x80000000, 0x20000000, 0x20000000, 0x20000000,
			0x80000000, 0x00000010, 0x00000000, 0x20000010, 0xa0000000, 0x00000000,
			0x20000000, 0x20000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
			0x00000000, 0x00000000, 0x00000001, 0x00000020, 0x00000001, 0x40000002,
			0x40000041, 0x40000022, 0x80000005, 0xc0000082, 0xc0000046, 0x4000004b,
			0x80000107, 0x00000089, 0x00000014, 0x8000024b, 0x0000011b, 0x8000016d,
			0x8000041a, 0x000002e4},
	},
	{
		DvType: 2, DvK: 48, DvB: 0, TestT: 58, MaskI: 0, MaskB: 20,
		Dm: [CheckSize]uint32{
			0xbc00001a, 0x20000010, 0x2400001c, 0xec000014, 0x0c000002, 0xc0000010,
			0xb400001c, 0x2c000004, 0xbc000018, 0xb0000010, 0x0000000c, 0xb8000010,
			0x08000018, 0x78000010, 0x08000014, 0x70000010, 0xb800001c, 0xe8000000,
			0xb0000004, 0x58000010, 0xb000000c, 0x48000000, 0xb0000000, 0xb8000010,
			0x98000010, 0xa0000000, 0x00000000, 0x00000000, 0x20000000, 0x80000000,
			0x00000010, 0x00000000, 0x20000010, 0x20000000, 0x00000010, 0x60000000,
			0x00000018, 0xe0000000, 0x90000000, 0x30000010, 0xb0000000, 0x20000000,
			0x20000000, 0xa0000000, 0x00000010, 0x80000000, 0x20000000, 0x20000000,
			0x20000000, 0x80000000, 0x00000010, 0x00000000, 0x20000010, 0xa0000000,
			0x00000000, 0x20000000, 0x20000000, 0x00000000, 0x00000000, 0x00000000,
			0x00000000, 0x00000000, 0x00000000, 0x00000001, 0x00000020, 0x00000001,
			0x40000002, 0x40000041, 0x40000022, 0x80000005, 0xc0000082, 0xc0000046,
			0x4000004b, 0x80000107, 0x00000089, 0x00000014, 0x8000024b, 0x0000011b,
			0x8000016d, 0x8000041a},
	},
	{
		DvType: 2, DvK: 49, DvB: 0, TestT: 58, MaskI: 0, MaskB: 21,
		Dm: [CheckSize]uint32{
			0x3c000004, 0xbc00001a, 0x20000010, 0x2400001c, 0xec000014, 0x0c000002,
			0xc0000010, 0xb400001c, 0x2c000004, 0xbc000018, 0xb0000010, 0x0000000c,
			0xb8000010, 0x08000018, 0x78000010, 0x08000014, 0x70000010, 0xb800001c,
			0xe8000000, 0xb0000004, 0x58000010, 0xb000000c, 0x48000000, 0xb0000000,
			0xb8000010, 0x98000010, 0xa0000000, 0x00000000, 0x00000000, 0x20000000,
			0x80000000, 0x00000010, 0x00000000, 0x20000010, 0x20000000, 0x00000010,
			0x60000000, 0x00000018, 0xe0000000, 0x90000000, 0x30000010, 0xb0000000,
			0x20000000, 0x20000000, 0xa0000000, 0x00000010, 0x80000000, 0x20000000,
			0x20000000, 0x20000000, 0x80000000, 0x00000010, 0x00000000, 0x20000010,
			0xa0000000, 0x00000000, 0x20000000, 0x20000000, 0x00000000, 0x00000000,
			0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000001, 0x00000020,
			0x00000001, 0x40000002, 0x40000041, 0x40000022, 0x80000005, 0xc0000082,
			0xc0000046, 0x4000004b, 0x80000107, 0x00000089, 0x00000014, 0x8000024b,
			0x0000011b, 0x8000016d},
	},
	{
		DvType: 2, DvK: 49, DvB: 2, TestT: 58, MaskI: 0, MaskB: 22,
		Dm: [CheckSize]uint32{
			0xf0000010, 0xf000006a, 0x80000040, 0x90000070, 0xb0000053, 0x30000008,
			0x00000043, 0xd0000072, 0xb0000010, 0xf0000062, 0xc0000042, 0x00000030,
			0xe0000042, 0x20000060, 0xe0000041, 0x20000050, 0xc0000041, 0xe0000072,
			0xa0000003, 0xc0000012, 0x60000041, 0xc0000032, 0x20000001, 0xc0000002,
			0xe0000042, 0x60000042, 0x80000002, 0x00000000, 0x00000000, 0x80000000,
			0x00000002, 0x00000040, 0x00000000, 0x80000040, 0x80000000, 0x00000040,
			0x80000001, 0x00000060, 0x80000003, 0x40000002, 0xc0000040, 0xc0000002,
			0x80000000, 0x80000000, 0x80000002, 0x00000040, 0x00000002, 0x80000000,
			0x80000000, 0x80000000, 0x00000002, 0x00000040, 0x00000000, 0x80000040,
			0x80000002, 0x00000000, 0x80000000, 0x80000000, 0x00000000, 0x00000000,
			0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000004, 0x00000080,
			0x00000004, 0x00000009, 0x00000105, 0x00000089, 0x00000016, 0x0000020b,
			0x0000011b, 0x0000012d, 0x0000041e, 0x00000224, 0x00000050, 0x0000092e,
			0x0000046c, 0x000005b6},
	},
	{
		DvType: 2, DvK: 50, DvB: 0, TestT: 65, MaskI: 0, MaskB: 23,
		Dm: [CheckSize]uint32{
			0xb400001c, 0x3c000004, 0xbc00001a, 0x20000010, 0x2400001c, 0xec000014,
			0x0c000002, 0xc0000010, 0xb400001c, 0x2c000004, 0xbc000018, 0xb0000010,
			0x0000000c, 0xb8000010, 0x08000018, 0x78000010, 0x08000014, 0x70000010,
			0xb800001c, 0xe8000000, 0xb0000004, 0x58000010, 0xb000000c, 0x48000000,
			0xb0000000, 0xb8000010, 0x98000010, 0xa0000000, 0x00000000, 0x00000000,
			0x20000000, 0x80000000, 0x00000010, 0x00000000, 0x20000010, 0x20000000,
			0x00000010, 0x60000000, 0x00000018, 0xe0000000, 0x90000000, 0x30000010,
			0xb0000000, 0x20000000, 0x20000000, 0xa0000000, 0x00000010, 0x80000000,
			0x20000000, 0x20000000, 0x20000000, 0x80000000, 0x00000010, 0x00000000,
			0x20000010, 0xa0000000, 0x00000000, 0x20000000, 0x20000000, 0x00000000,
			0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000001,
			0x00000020, 0x00000001, 0x40000002, 0x40000041, 0x40000022, 0x80000005,
			0xc0000082, 0xc0000046, 0x4000004b, 0x80000107, 0x00000089, 0x00000014,
			0x8000024b, 0x0000011b},
	},
	{
		DvType: 2, DvK: 50, DvB: 2, TestT: 65, MaskI: 0, MaskB: 24,
		Dm: [CheckSize]uint32{
			0xd0000072, 0xf0000010, 0xf000006a, 0x80000040, 0x90000070, 0xb0000053,
			0x30000008, 0x00000043, 0xd0000072, 0xb0000010, 0xf0000062, 0xc0000042,
			0x00000030, 0xe0000042, 0x20000060, 0xe0000041, 0x20000050, 0xc0000041,
			0xe0000072, 0xa0000003, 0xc0000012, 0x60000041, 0xc0000032, 0x20000001,
			0xc0000002, 0xe0000042, 0x60000042, 0x80000002, 0x00000000, 0x00000000,
			0x80000000, 0x00000002, 0x00000040, 0x00000000, 0x80000040, 0x80000000,
			0x00000040, 0x80000001, 0x00000060, 0x80000003, 0x40000002, 0xc0000040,
			0xc0000002, 0x80000000, 0x80000000, 0x80000002, 0x00000040, 0x00000002,
			0x80000000, 0x80000000, 0x80000000, 0x00000002, 0x00000040, 0x00000000,
			0x80000040, 0x80000002, 0x00000000, 0x80000000, 0x80000000, 0x00000000,
			0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000004,
			0x00000080, 0x00000004, 0x00000009, 0x00000105, 0x00000089, 0x00000016,
			0x0000020b, 0x0000011b, 0x0000012d, 0x0000041e, 0x00000224, 0x00000050,
			0x0000092e, 0x0000046c},
	},
	{
		DvType: 2, DvK: 51, DvB: 0, TestT: 65, MaskI: 0, MaskB: 25,
		Dm: [CheckSize]uint32{
			0xc0000010, 0xb400001c, 0x3c000004, 0xbc00001a, 0x20000010, 0x2400001c,
			0xec000014, 0x0c000002, 0xc0000010, 0xb400001c, 0x2c000004, 0xbc000018,
			0xb0000010, 0x0000000c, 0xb8000010, 0x08000018, 0x78000010, 0x08000014,
			0x70000010, 0xb800001c, 0xe8000000, 0xb0000004, 0x58000010, 0xb000000c,
			0x48000000, 0xb0000000, 0xb8000010, 0x98000010, 0xa0000000, 0x00000000,
			0x00000000, 0x20000000, 0x80000000, 0x00000010, 0x00000000, 0x20000010,
			0x20000000, 0x00000010, 0x60000000, 0x00000018, 0xe0000000, 0x90000000,
			0x30000010, 0xb0000000, 0x20000000, 0x20000000, 0xa0000000, 0x00000010,
			0x80000000, 0x20000000, 0x20000000, 0x20000000, 0x80000000, 0x00000010,
			0x00000000, 0x20000010, 0xa0000000, 0x00000000, 0x20000000, 0x20000000,
			0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
			0x00000001, 0x00000020, 0x00000001, 0x40000002, 0x40000041, 0x40000022,
			0x80000005, 0xc0000082, 0xc0000046, 0x4000004b, 0x80000107, 0x00000089,
			0x00000014, 0x8000024b},
	},
	{
		DvType: 2, DvK: 51, DvB: 2, TestT: 65, MaskI: 0, MaskB: 26,
		Dm: [CheckSize]uint32{
			0x00000043, 0xd0000072, 0xf0000010, 0xf000006a, 0x80000040, 0x90000070,
			0xb0000053, 0x30000008, 0x00000043, 0xd0000072, 0xb0000010, 0xf0000062,
			0xc0000042, 0x00000030, 0xe0000042, 0x20000060, 0xe0000041, 0x20000050,
			0xc0000041, 0xe0000072, 0xa0000003, 0xc0000012, 0x60000041, 0xc0000032,
			0x20000001, 0xc0000002, 0xe0000042, 0x60000042, 0x80000002, 0x00000000,
			0x00000000, 0x80000000, 0x00000002, 0x00000040, 0x00000000, 0x80000040,
			0x80000000, 0x00000040, 0x80000001, 0x00000060, 0x80000003, 0x40000002,
			0xc0000040, 0xc0000002, 0x80000000, 0x80000000, 0x80000002, 0x00000040,
			0x00000002, 0x80000000, 0x80000000, 0x80000000, 0x00000002, 0x00000040,
			0x00000000, 0x80000040, 0x80000002, 0x00000000, 0x80000000, 0x80000000,
			0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
			0x00000004, 0x00000080, 0x00000004, 0x00000009, 0x00000105, 0x00000089,
			0x00000016, 0x0000020b, 0x0000011b, 0x0000012d, 0x0000041e, 0x00000224,
			0x00000050, 0x0000092e},
	},
	{
		DvType: 2, DvK: 52, DvB: 0, TestT: 65, MaskI: 0, MaskB: 27,
		Dm: [CheckSize]uint32{
			0x0c000002, 0xc0000010, 0xb400001c, 0x3c000004, 0xbc00001a, 0x20000010,
			0x2400001c, 0xec000014, 0x0c000002, 0xc0000010, 0xb400001c, 0x2c000004,
			0xbc000018, 0xb0000010, 0x0000000c, 0xb8000010, 0x08000018, 0x78000010,
			0x08000014, 0x70000010, 0xb800001c, 0xe8000000, 0xb0000004, 0x58000010,
			0xb000000c, 0x48000000, 0xb0000000, 0xb8000010, 0x98000010, 0xa0000000,
			0x00000000, 0x00000000, 0x20000000, 0x80000000, 0x00000010, 0x00000000,
			0x20000010, 0x20000000, 0x00000010, 0x60000000, 0x00000018, 0xe0000000,
			0x90000000, 0x30000010, 0xb0000000, 0x20000000, 0x20000000, 0xa0000000,
			0x00000010, 0x80000000, 0x20000000, 0x20000000, 0x20000000, 0x80000000,
			0x00000010, 0x00000000, 0x20000010, 0xa0000000, 0x00000000, 0x20000000,
			0x20000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
			0x00000000, 0x00000001, 0x00000020, 0x00000001, 0x40000002, 0x40000041,
			0x40000022, 0x80000005, 0xc0000082, 0xc0000046, 0x4000004b, 0x80000107,
			0x00000089, 0x00000014},
	},
	{
		DvType: 2, DvK: 53, DvB: 0, TestT: 65, MaskI: 0, MaskB: 28,
		Dm: [CheckSize]uint32{
			0xcc000014, 0x0c000002, 0xc0000010, 0xb400001c, 0x3c000004, 0xbc00001a,
			0x20000010, 0x2400001c, 0xec000014, 0x0c000002, 0xc0000010, 0xb400001c,
			0x2c000004, 0xbc000018, 0xb0000010, 0x0000000c, 0xb8000010, 0x08000018,
			0x78000010, 0x08000014, 0x70000010, 0xb800001c, 0xe8000000, 0xb0000004,
			0x58000010, 0xb000000c, 0x48000000, 0xb0000000, 0xb8000010, 0x98000010,
			0xa0000000, 0x00000000, 0x00000000, 0x20000000, 0x80000000, 0x00000010,
			0x00000000, 0x20000010, 0x20000000, 0x00000010, 0x60000000, 0x00000018,
			0xe0000000, 0x90000000, 0x30000010, 0xb0000000, 0x20000000, 0x20000000,
			0xa0000000, 0x00000010, 0x80000000, 0x20000000, 0x20000000, 0x20000000,
			0x80000000, 0x00000010, 0x00000000, 0x20000010, 0xa0000000, 0x00000000,
			0x20000000, 0x20000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
			0x00000000, 0x00000000, 0x00000001, 0x00000020, 0x00000001, 0x40000002,
			0x40000041, 0x40000022, 0x80000005, 0xc0000082, 0xc0000046, 0x4000004b,
			0x80000107, 0x00000089},
	},
	{
		DvType: 2, DvK: 54, DvB: 0, TestT: 65, MaskI: 0, MaskB: 29,
		Dm: [CheckSize]uint32{
			0x0400001c, 0xcc000014, 0x0c000002, 0xc0000010, 0xb400001c, 0x3c000004,
			0xbc00001a, 0x20000010, 0x2400001c, 0xec000014, 0x0c000002, 0xc0000010,
			0xb400001c, 0x2c000004, 0xbc000018, 0xb0000010, 0x0000000c, 0xb8000010,
			0x08000018, 0x78000010, 0x08000014, 0x70000010, 0xb800001c, 0xe8000000,
			0xb0000004, 0x58000010, 0xb000000c, 0x48000000, 0xb0000000, 0xb8000010,
			0x98000010, 0xa0000000, 0x00000000, 0x00000000, 0x20000000, 0x80000000,
			0x00000010, 0x00000000, 0x20000010, 0x20000000, 0x00000010, 0x60000000,
			0x00000018, 0xe0000000, 0x90000000, 0x30000010, 0xb0000000, 0x20000000,
			0x20000000, 0xa0000000, 0x00000010, 0x80000000, 0x20000000, 0x20000000,
			0x20000000, 0x80000000, 0x00000010, 0x00000000, 0x20000010, 0xa0000000,
			0x00000000, 0x20000000, 0x20000000, 0x00000000, 0x00000000, 0x00000000,
			0x00000000, 0x00000000, 0x00000000, 0x00000001, 0x00000020, 0x00000001,
			0x40000002, 0x40000041, 0x40000022, 0x80000005, 0xc0000082, 0xc0000046,
			0x4000004b, 0x80000107},
	},
	{
		DvType: 2, DvK: 55, DvB: 0, TestT: 65, MaskI: 0, MaskB: 30,
		Dm: [CheckSize]uint32{
			0x00000010, 0x0400001c, 0xcc000014, 0x0c000002, 0xc0000010, 0xb400001c,
			0x3c000004, 0xbc00001a, 0x20000010, 0x2400001c, 0xec000014, 0x0c000002,
			0xc0000010, 0xb400001c, 0x2c000004, 0xbc000018, 0xb0000010, 0x0000000c,
			0xb8000010, 0x08000018, 0x78000010, 0x08000014, 0x70000010, 0xb800001c,
			0xe8000000, 0xb0000004, 0x58000010, 0xb000000c, 0x48000000, 0xb0000000,
			0xb8000010, 0x98000010, 0xa0000000, 0x00000000, 0x00000000, 0x20000000,
			0x80000000, 0x00000010, 0x00000000, 0x20000010, 0x20000000, 0x00000010,
			0x60000000, 0x00000018, 0xe0000000, 0x90000000, 0x30000010, 0xb0000000,
			0x20000000, 0x20000000, 0xa0000000, 0x00000010, 0x80000000, 0x20000000,
			0x20000000, 0x20000000, 0x80000000, 0x00000010, 0x00000000, 0x20000010,
			0xa0000000, 0x00000000, 0x20000000, 0x20000000, 0x00000000, 0x00000000,
			0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000001, 0x00000020,
			0x00000001, 0x40000002, 0x40000041, 0x40000022, 0x80000005, 0xc0000082,
			0xc0000046, 0x4000004b},
	},
	{
		DvType: 2, DvK: 56, DvB: 0, TestT: 65, MaskI: 0, MaskB: 31,
		Dm: [CheckSize]uint32{
			0x2600001a, 0x00000010, 0x0400001c, 0xcc000014, 0x0c000002, 0xc0000010,
			0xb400001c, 0x3c000004, 0xbc00001a, 0x20000010, 0x2400001c, 0xec000014,
			0x0c000002, 0xc0000010, 0xb400001c, 0x2c000004, 0xbc000018, 0xb0000010,
			0x0000000c, 0xb8000010, 0x08000018, 0x78000010, 0x08000014, 0x70000010,
			0xb800001c, 0xe8000000, 0xb0000004, 0x58000010, 0xb000000c, 0x48000000,
			0xb0000000, 0xb8000010, 0x98000010, 0xa0000000, 0x00000000, 0x00000000,
			0x20000000, 0x80000000, 0x00000010, 0x00000000, 0x20000010, 0x20000000,
			0x00000010, 0x60000000, 0x00000018, 0xe0000000, 0x90000000, 0x30000010,
			0xb0000000, 0x20000000, 0x20000000, 0xa0000000, 0x00000010, 0x80000000,
			0x20000000, 0x20000000, 0x20000000, 0x80000000, 0x00000010, 0x00000000,
			0x20000010, 0xa0000000, 0x00000000, 0x20000000, 0x20000000, 0x00000000,
			0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000001,
			0x00000020, 0x00000001, 0x40000002, 0x40000041, 0x40000022, 0x80000005,
			0xc0000082, 0xc0000046},
	},
	{
		DvType: 0, DvK: 0, DvB: 0, TestT: 0, MaskI: 0, MaskB: 0,
		Dm: [CheckSize]uint32{
			0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
			0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
			0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
			0, 0, 0, 0, 0, 0, 0, 0},
	},
}
