/*
 * // Copyright (c) Radzivon Bartoshyk 7/2025. All rights reserved.
 * //
 * // Redistribution and use in source and binary forms, with or without modification,
 * // are permitted provided that the following conditions are met:
 * //
 * // 1.  Redistributions of source code must retain the above copyright notice, this
 * // list of conditions and the following disclaimer.
 * //
 * // 2.  Redistributions in binary form must reproduce the above copyright notice,
 * // this list of conditions and the following disclaimer in the documentation
 * // and/or other materials provided with the distribution.
 * //
 * // 3.  Neither the name of the copyright holder nor the names of its
 * // contributors may be used to endorse or promote products derived from
 * // this software without specific prior written permission.
 * //
 * // THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * // AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * // IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * // DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE
 * // FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * // DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * // SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 * // CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * // OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * // OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

/**
Y0 Remez poly coeffs at zeros and extremums
Generated by SageMath:
```python
def compute_intervals(zeros):
    intervals = []
    for i in range(0, len(zeros)):
        if i == 0:
            a = 1.35 - zeros[i]
            b = (zeros[i] + zeros[i + 1]) / 2 + 0.03 - zeros[i]
            intervals.append((RealField(18)(a), RealField(18)(b), RealField(110)(zeros[i])))
        elif i + 1 > len(zeros) - 1:
            a = (zeros[i - 1] + zeros[i]) / 2 - 0.03 - zeros[i]
            b = (zeros[i]) + 0.83 + 0.03 - zeros[i]
            intervals.append((RealField(18)(a), RealField(18)(b), RealField(110)(zeros[i])))
        else:
            a = (zeros[i - 1] + zeros[i]) / 2 - zeros[i] - 0.03
            b = (zeros[i] + zeros[i + 1]) / 2 + 0.03  - zeros[i]
            intervals.append((RealField(18)(a), RealField(18)(b), RealField(110)(zeros[i])))
    return intervals

intervals = compute_intervals(y0_zeros)
# print(intervals)

def build_sollya_script(a, b, zero, deg):
    return f"""
prec = 200;
bessel_y0 = library("./notes/bessel_sollya/cmake-build-release/libbessel_sollya.dylib");
f = bessel_y0(x + {zero});
d = [{a}, {b}];
pf = remez(f, {deg}, d);
for i from 0 to degree(pf) do {{
    write(coeff(pf, i)) >> "coefficients.txt";
    write("\\n") >> "coefficients.txt";
}};
"""

def load_coefficients(filename):
    with open(filename, "r") as f:
        return [RealField(500)(line.strip()) for line in f if line.strip()]

def call_sollya_on_interval(a, b, zero, degree=12):
    sollya_script = build_sollya_script(a, b, zero, degree)
    with open("tmp_interval.sollya", "w") as f:
        f.write(sollya_script)
    import subprocess
    if os.path.exists("coefficients.txt"):
        os.remove("coefficients.txt")
    try:
        result = subprocess.run(
            ["sollya", "tmp_interval.sollya"],
            check=True,
            capture_output=True,
            text=True
        )
    except subprocess.CalledProcessError as e:
        return

def print_remez_coeffs(poly):
    print("[")
    for i in range(len(poly)):
        coeff = poly[i]
        print_double_double("", coeff)
    print("],")

degree = 27

print(f"pub(crate) static Y0_COEFFS: [[(u64, u64); {degree + 1}]; {len(intervals)}] = [")
for i in range(0, len(intervals)):
    interval = intervals[i]
    call_sollya_on_interval(interval[0], interval[1], interval[2], degree)
    coeffs = load_coefficients(f"coefficients.txt")
    print_remez_coeffs(coeffs)
print("];")
```
**/
pub(crate) static Y0_COEFFS: [[(u64, u64); 28]; 47] = [
    [
        (0xbc7492c1ad707fcd, 0x3fe0aa48442f014b),
        (0x3889cbe7f6454fdf, 0x3becdccc60399188),
        (0x3c6201923a596bb0, 0xbfd0aa48442f014b),
        (0x3c24b04e2557df22, 0x3fa439fac16525f5),
        (0xbc059b63c3765131, 0x3f80d2af4e9323bc),
        (0xbbdcec718054f585, 0x3f4f71646bea8f8b),
        (0x3bf856364cecf8b8, 0xbf5444bda8b69d4e),
        (0x3bda1cb8be57cc8f, 0x3f384c2206701bfd),
        (0xbb8ae41d36dcce8d, 0xbf217ab4af8d0bf0),
        (0xbba9cb2ac1552425, 0x3f0dafb9b99a70f2),
        (0xbb951efcd9701bd5, 0xbef8bb9639d60207),
        (0x3b8a10d871292a77, 0x3ee48b6751827819),
        (0xbb709569bb189616, 0xbed1364cc5887f8d),
        (0x3b248bd9aba7d332, 0x3ebd06afc4e97689),
        (0x3b44874f3eb42a7d, 0xbea899b75531d45b),
        (0xbb036dc9313b0b72, 0x3e94f157466889b4),
        (0xbb15fb0fa7789259, 0xbe81e537d57e5872),
        (0xbae227bb9fde16a8, 0x3e6ebca03e74f872),
        (0x3ae23bb7c6812f3e, 0xbe5a95bfc7177df2),
        (0x3ae9256929650559, 0x3e46b9b8baa2a7e6),
        (0x3ac02e410dd7de2b, 0xbe32d4cd9df0cdc3),
        (0x3ac47f4d926b3967, 0x3e2169d04e3c7866),
        (0x3ab6f6c4b513d045, 0xbe12dac47ffdcb13),
        (0x3a6fa8ccc9968ba6, 0x3dfa16bd9d2454ff),
        (0xba171664de1838ee, 0xbda18f57d00cd0c8),
        (0x3a636003e2d9973c, 0x3dcac3134b14f54c),
        (0x3a5cb30d6a3e73d7, 0xbdd88bfc936d3a46),
        (0x3a600c3863532a47, 0x3dc074a2bfd935a7),
    ],
    [
        (0xb6985ea12b68dbc6, 0xba0ce46bb97e3b37),
        (0xbc5a4026e448075b, 0xbfd9c34256a12a0c),
        (0xbc49ded6e42ffb00, 0x3faa09c9290367ef),
        (0xbc253b026c745733, 0x3fadf6d59bf50ebd),
        (0xbc18dbfc7b944e60, 0xbf7c116fdc598096),
        (0x3bf40cf995f4f110, 0xbf61e32bc4ef8a41),
        (0xbbcb82d9e30fbb8a, 0x3f29982765166254),
        (0xbb7a0c04b5d6f2fd, 0x3f0ab2c1fecdcfc4),
        (0x3b7cda0c15fabc6d, 0xbed486371e6a6640),
        (0xbb079525b0987a4e, 0xbe93b21684089b98),
        (0xbaf3ba730f1c40e9, 0xbe5777058741ab00),
        (0xbaf583debf7de8c9, 0x3e5452ea009488fe),
        (0x3ab98e298e66ceb6, 0xbe2f6938829aeacd),
        (0x3aa9d6dc6976f3c9, 0x3e0a5a1f3a03d35d),
        (0x3a7994a2d0f56a03, 0xbde994e7358cedf6),
        (0x3a5a354bba4fc371, 0x3dc88ae5d31af5c1),
        (0xba357cd4a8a94073, 0xbda75329787f3d8c),
        (0xba2d0839c51ea186, 0x3d8643c578f084cf),
        (0xba0526a2696c015c, 0xbd6554780d3aba1a),
        (0xb9ec9b6ca0c2368e, 0x3d447c69a36927e9),
        (0xb9b175a5c0e46062, 0xbd23b88e0a626f34),
        (0x3994f94bb00b0bd0, 0x3d02fdee82cca7a8),
        (0xb98eac61aa84bafd, 0xbce25983eb139aeb),
        (0xb9685d6dfdbe7bfb, 0x3cc2440cd8848b87),
        (0x3922fa28d497dc91, 0xbca164c6877c570d),
        (0xb917682fb55945f7, 0x3c78a082a7dc3992),
        (0xb90f407a50193eb2, 0xbc621ea0651d2198),
        (0xb8c56e242114685f, 0x3c523596b99eb3ff),
    ],
    [
        (0x3c7b8d2a1c496808, 0xbfd5c7c556f0c19a),
        (0x36068b8c3a0eb972, 0x396e4544b8708e19),
        (0xbc6b8d2a1c496844, 0x3fc5c7c556f0c19a),
        (0xbc2b4d44aaaf080d, 0xbf8564d4b1ed0d7e),
        (0x3c2e4289c3ae4c4e, 0xbf8a15d92dfe3e27),
        (0xbbedbea922f2213b, 0x3f4b438430469ff7),
        (0x3bdc176048714c11, 0x3f37a8924cc2f914),
        (0xbb7021dbf4219b23, 0xbef5f69b4a8a3c05),
        (0xbb7729647a7cf1b3, 0xbed85b940eb607f9),
        (0x3b3d99d888f48f01, 0x3e955ac0f5c3162c),
        (0x3b0f25a287448fcc, 0x3e6d563e18b47eb8),
        (0x3ac452983ff83fe5, 0xbe23eac3a45e06f4),
        (0xbaa55a4be8b255ef, 0xbe0007671d55a8af),
        (0x3a6681c171d8b7ac, 0x3dc04e13e8d2d63b),
        (0xb9cb304019dc95a9, 0xbd46e2b306d18fe2),
        (0xb9e569987d3c61bc, 0x3d54d18f7c808f5a),
        (0xb9d65bf264af4caa, 0xbd349808958007f4),
        (0x39af1658e9591f8b, 0x3d0a5b43a3168022),
        (0xb98b94f19368caef, 0xbce1c9c4d5c7e2e7),
        (0xb91d94e29089b67f, 0x3cb933cacbb762ef),
        (0xb931aa0bcf9e385e, 0xbc91c66b5e59c10e),
        (0xb907d452cfe5d11e, 0x3c690530621ffb6d),
        (0x38e2fc94e9e2e560, 0xbc41a7142b188ece),
        (0x389fe83adc9ec098, 0x3c18ea69b49fa0ec),
        (0x386baf69f710f067, 0xbbf186b62773cb8b),
        (0xb830ab83fea718cc, 0x3bc97ee300661ddb),
        (0xb841e9ca9ef3f273, 0xbba4203d2c71dee3),
        (0x381487ed7b4c5c12, 0x3b7779cb668f4f30),
    ],
    [
        (0x35f13ef25b4edb08, 0x3952841995e45c48),
        (0xbc78d4484b7cd2a7, 0x3fd334cca0697a5b),
        (0xbc341c66c86d3ee2, 0xbf95aef611fc4d57),
        (0xbc44929b43aab7e3, 0xbfa8969c64cbf452),
        (0x3bfab353487eb8be, 0x3f6b2f14a95527cb),
        (0x3c0aa67a62fa8841, 0x3f61d35e85fde2a3),
        (0xbbb270ba54528c94, 0xbf226dd71e391c8b),
        (0xbb61863ae7da7a16, 0xbf08177e4fe52324),
        (0x3b4949552d9354dc, 0x3ec6a9227352f96a),
        (0xbb44a5b99cf95459, 0x3ea34aa7573ef14e),
        (0xbb0698b7b84362ae, 0xbe60a2814d7ac448),
        (0xbad900bb94c56b91, 0xbe3442a3d3359ad9),
        (0x3a8ac6ca8212a487, 0x3defa6c509566c72),
        (0x3a56a6cae4d17a25, 0x3dbf14e5584823d7),
        (0xb9f7410034885564, 0xbd7714c292be3962),
        (0x39d1bcc695f4f231, 0xbd40679cd58549e2),
        (0x3997214169109d52, 0x3cf3365d5a792113),
        (0xb95497cd3660e90c, 0x3cc38cf06b3134d3),
        (0xb92ac63766c35618, 0xbc82bd5a84bf3644),
        (0x38dd10b30698d66a, 0x3c326e932d44d957),
        (0xb8bf280f5863dcdc, 0xbc1271150465312d),
        (0xb878f6f4bf335d0d, 0x3be8df3e61da29e5),
        (0x384a254e3a1a9970, 0xbbb9b1263c861574),
        (0x381db06e04e97e27, 0x3b8b3ef4b052ee69),
        (0xb7f2c3a3c46974ad, 0xbb5da7aeb72ad1b2),
        (0xb7c4810afcc72e8d, 0x3b2fdbbcc787affa),
        (0x37a36df7ebbbb4ed, 0xbb031287031f9ece),
        (0xb75db740f07d1867, 0x3ad85ce53417a3c0),
    ],
    [
        (0x3c61dc672a53c590, 0x3fd15f993fceab5c),
        (0x35ea878b9ff55f5c, 0xb946a81d8aec1515),
        (0xbc51dc672a53c58f, 0xbfc15f993fceab5c),
        (0xbc0d1c390189caf7, 0x3f758ef6efbed797),
        (0xbc243ede64b782d9, 0x3f86395dfe49fcd4),
        (0xbbbb4484582c217c, 0xbf3fb15104a41c00),
        (0xbbd4410e752beb78, 0xbf35f88a11d5564f),
        (0xbb8b2d9b83a5b450, 0x3eef37d226a824b7),
        (0xbb61fb1782379734, 0x3ed6f7bab104f355),
        (0x3b2da2cb873ba95e, 0xbe8f0c45a3824d70),
        (0xbac11338801bec18, 0xbe6dfe11df12c715),
        (0xbac52938286abb6d, 0x3e2311adc2e753ee),
        (0xba9200612bccbf82, 0x3dfad34e18504e16),
        (0xba4c6eb5b9fade2d, 0xbdafdc8061ae8b15),
        (0xba251747464962f8, 0xbd819498ca6cef9b),
        (0x39d9f9efcc54f4de, 0x3d33a291d5991f6b),
        (0x39af3f93679996f3, 0x3d016f7ae80cadff),
        (0xb9422e176854fc2b, 0xbcb1f8d80decb659),
        (0xb8ef665f3b08f8e4, 0xbc7c27b5350d7ce7),
        (0x38af81adc69c8ae0, 0x3c2d08034e0e0d4c),
        (0xb88ed31bb5296f3b, 0x3befac201220d678),
        (0x37f6dcad77609354, 0xbb93e9257bbbec92),
        (0x37f2bad6f8609fad, 0xbb6dce086bd954ec),
        (0x37c3c466a462a963, 0x3b2c45f44eeac691),
        (0x377aca1c9e5ead22, 0xbaebf82d453f586e),
        (0x371d5ba887167511, 0x3abef01a452e6243),
        (0xb6e1a818be163885, 0xba8faba1ac8fc141),
        (0xb6f75474af743fde, 0x3a59bbaf9916fed8),
    ],
    [
        (0x360f6f90e85783b8, 0x396e539ee4be998b),
        (0x3c61e8f568f8c6b6, 0xbfcff635cc72b9f1),
        (0x3c0fa6ebe8b754aa, 0x3f89036451ff57c5),
        (0x3c46345bcf3c5b81, 0x3fa4e667a71556af),
        (0x3c065e386829f088, 0xbf60325ee41e910c),
        (0x3bfde6b6846da9c7, 0xbf5fe23914fb9128),
        (0xbbb9c28522c54b49, 0x3f17f84d7c50e4c4),
        (0xbbaeab0290c997f1, 0x3f06afdd57be1e14),
        (0xbb2bbdfb6c40ca93, 0xbec04053abf4386a),
        (0x3b3787c4b69beaba, 0xbea2aea9ec48d8fd),
        (0xbaf17a36377f9c62, 0x3e593eb9f1ddb4e9),
        (0x3ad9f8321f3ce337, 0x3e3428a3a3e30a13),
        (0xba49c9c135d6bfa4, 0xbde99d8c42f7a020),
        (0xba59203ca37ccc7f, 0xbdbec9805045aa58),
        (0xba08ceab2aadd382, 0x3d72613659ce994f),
        (0xb9e8053d9b1a7b6c, 0x3d418f302bdba980),
        (0xb96ea7068003a9dc, 0xbcf3bc9ccb704250),
        (0xb93e32e516fbc8fc, 0xbcbf0dc4067be8fe),
        (0xb91a819a6b2cfdb1, 0x3c706e491f458177),
        (0xb8b2f6e0e017d3a4, 0x3c36013bac772fd5),
        (0xb88860e3b038bbbc, 0xbbe616b4d60c99dd),
        (0x38440765a3381ed6, 0xbba93fb6fc116a65),
        (0x37f070ddd36dbc13, 0x3b57963a3ba9df68),
        (0x37b30f1b7dcdc2ea, 0x3b1937f28596352a),
        (0x3760f019da1d4947, 0xbac837f92e435c71),
        (0xb6cfd07d0b99a723, 0xba80a4be8fbdea42),
        (0xb6b4329fac42bf24, 0x3a19c0ce24bac50d),
        (0xb69a16105af87cd5, 0x39ff46fe69318a2c),
    ],
    [
        (0x3c54d14c77bc1691, 0xbfcdc14ea14e89f9),
        (0xb61ccfd68577b5e8, 0xb97a750db049a3bc),
        (0xbc44d14c77bc1688, 0x3fbdc14ea14e89f9),
        (0x3c00ff03f975831b, 0xbf6b037fe9cf2a52),
        (0xbc2f19a8f5b137a3, 0xbf8367d7d608e4ba),
        (0x3bdd892317f290f5, 0x3f34abef5636e4f7),
        (0x3bd5ba5b5b609686, 0x3f33d8a661229259),
        (0xbb700a3f0097a359, 0xbee5cfe92a1a2c45),
        (0xbb6087e84c57990c, 0xbed571814a1aa301),
        (0x3b22b6740db0f074, 0x3e87414e33c9bacd),
        (0x3afa4836a20e8280, 0x3e6ca7069e73d1d9),
        (0x3abe65fd39b0188a, 0xbe1e0b3a705ff247),
        (0x3a95774f6fa018cb, 0xbdfa15dd62cf9f10),
        (0xba399e86b7dd9f12, 0x3daa33948eada12c),
        (0x3a22398b28ff34ba, 0x3d81419b1fd428ef),
        (0x39c9ae1ea6c0045a, 0xbd30882f60b8a775),
        (0x39a31e54bbe9a33e, 0xbd015e5c0af749d4),
        (0x39454d020e06c1f7, 0x3cafb1e115e089b2),
        (0x3910a4f9eea95a23, 0x3c7b83826370860f),
        (0xb8c57c15f2ff0eb8, 0xbc27e5d539874c6c),
        (0xb87dc14618f0e43b, 0xbbf19dae391ff57d),
        (0xb82fe2e247291923, 0x3b9d2b7278390d46),
        (0x37f3825b1f8b77eb, 0x3b6298bb72cef60f),
        (0x37a7910d5f5dc998, 0xbb0d50bc222d50f5),
        (0x3776acb65e1367e0, 0xbad08ac20d33e370),
        (0x37062a8b170aa42b, 0x3a79154a0ab2b1c5),
        (0xb6d06d11501acdf8, 0x3a3884b2a5dd01b8),
        (0x36832f558f511cbf, 0xb9e1c3f19ed31df1),
    ],
    [
        (0xb5d1f4401b5b78db, 0x396e44a7ff47853b),
        (0x3c41398cacaa32cb, 0x3fcbf32a27594007),
        (0x3bfe5ce4af72439b, 0xbf80bc2d84e65214),
        (0x3c4acd5148eb7ea0, 0xbfa26cab38a8b368),
        (0x3bfab9a1103e209b, 0x3f55f03e47165d72),
        (0xbbc4598141a3f87e, 0x3f5caaa76e34992d),
        (0xbbbb7ab55e886663, 0xbf10c5f18c46d030),
        (0xbb99f58bad5e7b5f, 0xbf04f0af7d46cd48),
        (0xbb3040f00afdd8ce, 0x3eb7d1e28094e21a),
        (0xbb3f12deda9fd086, 0x3ea1ad0731228479),
        (0xbae9fe69a18b2b74, 0xbe536021c8cb3c34),
        (0x3ad92bf3eabb82c3, 0xbe3371ae6c759181),
        (0x3a630cb928debacf, 0x3de46ce077a8bed7),
        (0xba5feb436591c0b8, 0x3dbe235d3564b639),
        (0xb9d711ddaf978bf4, 0xbd6e408e259c72fb),
        (0xb9c9d325620234dd, 0xbd415f07ac87e832),
        (0x399b66d31155b6c3, 0x3cf0a44f77f6a8ee),
        (0xb927de212062774e, 0x3cbefd8ab52f8ec1),
        (0x38daebbaa8d140b4, 0xbc6c54ebb5ad3567),
        (0xb8d1c883890c6817, 0xbc360b43cbb605a1),
        (0x38851f7136cf7e49, 0x3be33d74aac5c469),
        (0xb84c8ae944909141, 0x3ba99bfbb428a4d1),
        (0xb7f50f7d69247480, 0xbb555c095b9d3de2),
        (0xb78d195ad02b9a0b, 0xbb18c29d6b596412),
        (0x37588e53fd261c0a, 0x3ac3c05d45a0cec2),
        (0xb722c1c138d71c9e, 0x3a843feaf2fed511),
        (0xb6c263b564f34be6, 0xba2ec5c49e1d5e88),
        (0x36731f7cd8008380, 0xb9ec82a1b1f74089),
    ],
    [
        (0x3c57ba12cd0fc91f, 0x3fca7022be084d99),
        (0xb5db8122d606073c, 0xb9614f36f1550bec),
        (0xbc47ba12cd0fc91d, 0xbfba7022be084d99),
        (0xbc0b3327197c9ced, 0x3f62ee079d020b12),
        (0x3c2d0f8f3671311f, 0x3f8163191c30aa62),
        (0xbbcc7f2dbb383293, 0xbf2d7806ea72fc76),
        (0x3bcf9ec581434d48, 0xbf320f95702b1d4e),
        (0x3b899e1a0f3c0abe, 0x3ee00610882294aa),
        (0x3b61505ce3b25d40, 0x3ed3e398cbc472ea),
        (0xbae050c778e973e5, 0xbe81bbe181c65162),
        (0xbb0c3fb832d7b54d, 0xbe6b0f89b7c61f3a),
        (0x3ab91b7cb5170340, 0x3e17c3f85882049b),
        (0xba976a7f60f10aab, 0x3df90236614c84dc),
        (0xba43abb3499d78c0, 0xbda564920d1387dc),
        (0x3a2b1f00d1e27499, 0xbd80be1811255cfe),
        (0x39b7396b68be0fd5, 0x3d2bb9712586bfac),
        (0x39a50316c82a22d1, 0x3d0101c31e1df222),
        (0xb93cd112332c006f, 0xbcab298cda76e792),
        (0x39181c12a03d0651, 0xbc7b22acc441d614),
        (0xb8cf36eb93d76394, 0x3c24dca37a331862),
        (0x387338c4943fcecc, 0x3bf176e94edd9a37),
        (0xb7ed97a80ea3375c, 0xbb99d4f0d7cf230e),
        (0x37c85bd213356682, 0xbb6287584fe1cda1),
        (0xb7957f660392a1e8, 0x3b0a5c024222d0c1),
        (0xb7798291e622cf2f, 0x3ad07cad512e27f5),
        (0xb712f27bce7fdd19, 0xba76918588d0c395),
        (0xb6d55b12c898a874, 0xba38d39ee7749b0b),
        (0xb680c165d28960ac, 0x39e0d21cd12bf761),
    ],
    [
        (0x360f530de8a2298c, 0xb96ccd9d9aa9b157),
        (0xbc65b9c39e42719c, 0xbfc925c35988ee29),
        (0xbbdda9c0588bab35, 0x3f7862549367591e),
        (0x3c4a142ed6a5521c, 0x3fa0a4512039d6a2),
        (0x3be94ce533c65b81, 0xbf5013b38cfb9292),
        (0xbbea2a14cf0f61a4, 0xbf5a24a1215f6684),
        (0xbbae7b4c448b426e, 0x3f08f91421377fad),
        (0x3baf72fd0f5209e6, 0x3f035d17cec0172f),
        (0x3b3cd242e4f6fc44, 0xbeb2283a93114096),
        (0x3b259dda3f9805c1, 0xbea099e71392f54e),
        (0xbad6dce1c34353e2, 0x3e4e5de01e2e6566),
        (0xbab52eec6344539e, 0x3e32885854ea8b06),
        (0x3a6795d88d7a428b, 0xbde0730c2985fd22),
        (0x3a4e069c266df28e, 0xbdbd1743cbb2e11a),
        (0x3a0121929f450b8e, 0x3d68f8728745e1e6),
        (0x39de1722c384607b, 0x3d40f166cb29a7d1),
        (0xb98c9b7f51aa7e71, 0xbcec10e67c004e39),
        (0x39306913ae262394, 0xbcbe7aec1ba33e89),
        (0x38d8c6bf0ecbf421, 0x3c68543e12c9bcc0),
        (0x38d880f40d10394c, 0x3c35d3939f6d29f1),
        (0xb885e7f0bd8af4c8, 0xbbe0c7ba1192da1c),
        (0xb84abbb402955d93, 0xbba97d91773fae0d),
        (0xb7c3bf48e2c5ddfc, 0x3b52dfb951fe4ab6),
        (0xb7ac27c13775b2fb, 0x3b18c04f68a8baaf),
        (0xb751cb733bb217c3, 0xbac1a783b2905f0d),
        (0xb72e01e5e434cb26, 0xba844df5fac82b00),
        (0x36c6a6f050600220, 0x3a2bbf5dc58ec268),
        (0xb672c2ec04596698, 0x39ec99f34bc330e3),
    ],
    [
        (0xbc63db68c567283b, 0xbfc80781c32422e7),
        (0xb60d9441230a8778, 0x397deb1fbbaf8566),
        (0x3c53db68c5672838, 0x3fb80781c32422e7),
        (0xbbfab5094c039945, 0xbf5c6923374d561f),
        (0xbc032419a9d405ca, 0xbf7fbe6df840847f),
        (0x3bc0f03bc2e75f5d, 0x3f264f4711a85f1a),
        (0x3bbd9972e6c197c9, 0x3f309ff42b0d7a68),
        (0xbb61aba5a7ce6e95, 0xbed8a60685a597d9),
        (0xbafdb153b11ed74e, 0xbed282d26a74c38e),
        (0xbb1d21d4a93dbe75, 0x3e7bdb57a889a4c1),
        (0xbb099a040b2ccf5b, 0x3e697d9c12a60cef),
        (0x3a35e0f484533648, 0xbe131aa8691738fb),
        (0x3a93c99c395b5c9a, 0xbdf7d486eee39a85),
        (0x3a223dad46ae9400, 0x3da197f258214b3d),
        (0xba1654ac01c16c13, 0x3d801dd7f92195ff),
        (0xb9cbde35eb70ebd4, 0xbd27480b9efaa860),
        (0x39a7bbce185d5c09, 0xbd00833ea1f83461),
        (0x393326656e27aad9, 0x3ca73bf7964edc39),
        (0x391e8e85dd3ba305, 0x3c7a891d704fed29),
        (0x38c6737263e77e58, 0xbc22222dcf097253),
        (0x3815c44bf33c3053, 0xbbf12db9d219b624),
        (0xb83a6ffdfedc9a0a, 0x3b96c4638eb6d0ab),
        (0xb7f870f3fdb66e9d, 0x3b624ff29cc34a5f),
        (0x3784f30eee231ab0, 0xbb0782fe0a2ed548),
        (0x376d3a1f2a7e18f9, 0xbad05b98a2d38bad),
        (0xb6fbbf0362b64b66, 0x3a745719daf0033a),
        (0xb6b5c16ada4dc5fa, 0x3a38b51735f94ad1),
        (0xb668156ec309831f, 0xb9de827eef669a9e),
    ],
    [
        (0xb5d71d9a0e2250f8, 0x3932accf9d404efd),
        (0x3c64fef53f4893e5, 0x3fc70c4f66cab47f),
        (0xbc1e032768318e73, 0xbf72c6731071e936),
        (0x3c2f806b21bc9578, 0xbf9e924b85a17361),
        (0xbbe1434be87c634b, 0x3f48d6c364d92082),
        (0x3bf5cbe0638f4cee, 0x3f58247b02d6b0f6),
        (0xbbaade8b3660fd3f, 0xbf0376125938561d),
        (0x3bae81e47dd3839a, 0xbf0206da232a2b6a),
        (0x3b424b3c317353f9, 0x3eacaf84db871510),
        (0xbb022e3bac6444e5, 0x3e9f3671177306ca),
        (0x3a9f8b17e5f17584, 0xbe48654b001006cd),
        (0x3ad272b91e39e04f, 0xbe3199ec88b18766),
        (0x3a6b9a2ed7b2626d, 0x3ddaeb1a849305fb),
        (0x3a57caf104216cf2, 0x3dbbe67c197c48c0),
        (0xb9c6b55595b8d04e, 0xbd64cdfa0bd69898),
        (0xb9dbab7dec1030e5, 0xbd4064936f97922f),
        (0xb978bade7534a976, 0x3ce7c83b5186222e),
        (0xb95c7c9291c10511, 0x3cbdb73ad7fa8bcd),
        (0x38c3fe342ee888e5, 0xbc64eecbf0b91aab),
        (0xb8ddc5bba4aa2657, 0xbc356ab697c0b36f),
        (0xb85034aab887d3c4, 0x3bdd4506fe39afb4),
        (0x383d49e7992e0ac5, 0x3ba925d81aa2ac79),
        (0x37e1c2dbc1cbe029, 0xbb50a89d8a666333),
        (0xb7b31ae47aa79118, 0xbb18875d46ae8913),
        (0xb75fe255d6973d67, 0x3abf7e2c067e0a91),
        (0x37086207483b1723, 0x3a8432812904a786),
        (0x36a9a69e34c77135, 0xba28f9cef0d7c221),
        (0x366e1739129c6d8d, 0xb9ec7ae64440285f),
    ],
    [
        (0xbc6d2f0105f3ce7c, 0x3fc62d94d97e859c),
        (0xb6196ae975958785, 0x3977cba836bb33ee),
        (0x3c5d2f0105f3ce79, 0xbfb62d94d97e859c),
        (0xbbf6c13bde837d51, 0x3f565481b55eaefb),
        (0x3c17013075a066fe, 0x3f7d5f857a2a6107),
        (0x3bc5f065aa633b82, 0xbf21a015a2ccb78f),
        (0xbbc12f349174f00d, 0xbf2eeb4eafd8614b),
        (0xbb53525d6141b63b, 0x3ed3a7b79e4d80fc),
        (0xbb662d899f08e930, 0x3ed154ed4598d2f0),
        (0xbb0444753ed61eee, 0xbe767f762ea293a1),
        (0xbafe11bb6d8cf182, 0xbe680ec1e042ee62),
        (0x3aa725a69e88f41a, 0x3e0f529652b9de0f),
        (0xba97b07721f181ff, 0x3df6ad7bd2247f55),
        (0x39e37ab8d29d6ed4, 0xbd9d4e582fb43553),
        (0x3a1f7027dbae57b9, 0xbd7eeb6efcbb6228),
        (0x39c89f73cf547041, 0x3d23b1eabb5c1c2c),
        (0x39960a6cdd6c228d, 0x3cffe92f0882b43f),
        (0xb94a0a88609a2efa, 0xbca3f20671568bda),
        (0xb917e7036f59109d, 0xbc79ce30ac166c3b),
        (0xb8b710f5a0405cb7, 0x3c1f8db6e94f5c58),
        (0x388042b39eb414de, 0x3bf0cc90c05bf217),
        (0xb83bf397bb30cb6c, 0xbb940c6bba534367),
        (0xb7f70d1d6ed27860, 0xbb61fe3820203226),
        (0xb77b9fa8fe59f5f1, 0x3b04ed91184b349c),
        (0x377e0b12ab59c923, 0x3ad023371cf7aa58),
        (0x370fad8a026e8d00, 0xba72475258dcb995),
        (0x36d50611a23601a2, 0xba3874f481897d45),
        (0xb67546ab95be8675, 0x39db9a35cb0268a5),
    ],
    [
        (0xb6160b0c9b9ffd9f, 0xb97a3923dc9c072f),
        (0x3c5fe2103f7148c0, 0xbfc5664d37c37d7b),
        (0xbc0a0fbbe8bcf31e, 0x3f6e0ee8ec84659a),
        (0xbc383a77d074bd10, 0x3f9c6c415c971b4b),
        (0xbbb06fe5233f1827, 0xbf43ec49cb941f55),
        (0x3bfe14142e172805, 0xbf56853b2d047885),
        (0xbb98cea461592a31, 0x3eff632a9396fd4e),
        (0xbba049885559b284, 0x3f00e6afa01af561),
        (0x3b452a854998ce1f, 0xbea7541247acdf59),
        (0x3b3a7242d1d8f2ff, 0xbe9d735e8beb8fcc),
        (0xbae93d2eb5241282, 0x3e440e6a7cafc096),
        (0x3acae709b26c74f3, 0x3e30b9cfae2ab337),
        (0x3a72d6e49ffa1efb, 0xbdd668087e67279b),
        (0x3a5eab7f746fd96e, 0xbdbab498e6f69c73),
        (0x3a0c7eba82778adf, 0x3d618b5674894afb),
        (0xb9dabc635cef6b79, 0x3d3f99da023c7232),
        (0x396198c6e558139c, 0xbce450b8165c264b),
        (0xb9379aa14a14f4cc, 0xbcbcd387688c5092),
        (0xb8d00bf3298263f3, 0x3c621a103bd09ea7),
        (0x38c2b31b04643f75, 0x3c34e540faeea510),
        (0x387fbfca69c0c002, 0xbbd999b3cf8b9794),
        (0xb84578c966f9c437, 0xbba8a8f1c19ab5c8),
        (0xb7d1af3943438fb6, 0x3b4d71c31555dffa),
        (0xb797b6eaed91b502, 0x3b18289e061910a0),
        (0x371f92726b30b6ae, 0xbabc18185dc0f589),
        (0x372dd6314cb1a609, 0xba83f7bf0ecd3312),
        (0xb6b22addb7be68de, 0x3a2677b40b678a01),
        (0xb6852d3eb4bf8c8d, 0x39ec35403f3feccb),
    ],
    [
        (0xbc5e9088e9ff2519, 0xbfc4b2a38f1ab9b4),
        (0xb5e56cdffbbb5799, 0xb95b36c3b9e47f10),
        (0x3c4e9088e9ff251a, 0x3fb4b2a38f1ab9b4),
        (0xbbfcc8083276a072, 0xbf5225a5c73f2233),
        (0x3c1525fe797e2125, 0xbf7b750d89a9b35f),
        (0xbbb08db59499cc78, 0x3f1cbdb4f1d5dbbd),
        (0x3bc35c23938c8103, 0x3f2cfe933fc6d0b8),
        (0x3b52d453a3ea1d43, 0xbed01ef10d839bbc),
        (0x3b71cffb5ca02d28, 0xbed05375a588a72b),
        (0xbb03f71b7795fb19, 0x3e729afe7ea7ba43),
        (0x3af336f1328c096c, 0x3e66c8fe015e6610),
        (0x3aa95a6cae05afad, 0xbe0a2a01e7af0241),
        (0xba78518ab4937cc9, 0xbdf59b33050d2092),
        (0xba24b06fe0fbe4ff, 0x3d98c0a50e7852d3),
        (0x3a16de13327f4ad6, 0x3d7da3426f355690),
        (0xb9cf30f26c9fd872, 0xbd20d3b48429ce84),
        (0x399b0cd9583574ed, 0xbcfec46e14cd5d9d),
        (0xb948a60ceb0a4c6f, 0x3ca13c3da7405df7),
        (0x39055845c840358d, 0x3c79043b4a0f0164),
        (0xb8a1906c502f6088, 0xbc1b902f4b27239f),
        (0x3883b1fc8ff5cfb3, 0xbbf05dbe7502cf7d),
        (0x3824e300158ee2c9, 0x3b91b0f5dc9eb0ab),
        (0x3803c301aa3c55ff, 0x3b619b550a0380e8),
        (0x37a48585c1f9fef8, 0xbb02a3e12dd3a21e),
        (0xb755a96cd7834eba, 0xbacfb47bc6c98397),
        (0x3705d5f51aa77044, 0x3a706b51c73e4699),
        (0x36cba3931d96b5c4, 0x3a381b0b33df04c5),
        (0x367c6c6e733152da, 0xb9d8f4cc5f2cd78e),
    ],
    [
        (0xb6025bf8fd89fcf9, 0xb972fca96d67ab3c),
        (0x3c5f3474ffad3fda, 0x3fc40f8ffdf09a5f),
        (0x3c047810b9792fcf, 0xbf68c37a29c4586f),
        (0xbc348471cc77dfe6, 0xbf9aab099314b209),
        (0x3be6aa80fcd0444f, 0x3f406f735cc0f6a6),
        (0x3bebe02c39288577, 0x3f552d29a06802e0),
        (0x3b89fdf10253db0c, 0xbef9fc04c675c0ed),
        (0xbb62f9306dab6a7d, 0xbeffe48825ed3c8e),
        (0x3b4325167726e7d5, 0x3ea36bd2d58ff45b),
        (0xbb2af08b6d9e350f, 0x3e9be87e2cad2ce7),
        (0xbaec807c6aa50609, 0xbe40d1fdedc6cb17),
        (0xba7064ddbed15f1c, 0xbe2fdbeb107ddccf),
        (0x3a758934a4493d29, 0x3dd2f44d01321684),
        (0xba4232e1c56c35a5, 0x3db9921373ea0f26),
        (0xb9d9c42dee5eb68c, 0xbd5df6b83ac56981),
        (0x39db0125ce11e878, 0xbd3e6c005b62bffa),
        (0x3945c956b1073d49, 0x3ce1848c91de7161),
        (0xb95e0318e089dc41, 0x3cbbe5530d0fcf21),
        (0xb8fe3f75946d3513, 0xbc5f8540b8b05bd4),
        (0x38b0d6e492206c61, 0xbc3451eb9bf04528),
        (0xb875fecfd5f528a6, 0x3bd67f1edd3c6bd2),
        (0xb84a4a9a4b0746cb, 0x3ba8169212ae867e),
        (0xb7b418bb0baae011, 0xbb4a1a0ded4a699f),
        (0x378a8c57d3055718, 0xbb17b199d846adfb),
        (0xb72c6c0833446ef6, 0x3ab91bb0a0b6110c),
        (0xb701ddf674f848ee, 0x3a83a75f8b7df40d),
        (0xb6bc25ec9cd09377, 0xba243ac68cfbb83d),
        (0xb6618a00b084ab2f, 0xb9ebd3094ccdd46d),
    ],
    [
        (0x3c4997782859a00d, 0x3fc37aaceac987b9),
        (0x360c6c83fa2a3645, 0xb97062679197f9db),
        (0xbc3997782859a008, 0xbfb37aaceac987b9),
        (0xbbd05a6eb7d72466, 0x3f4e3fdbfd65014a),
        (0x3c1d91e80a0529b4, 0x3f79de7a33bc3a97),
        (0xbbb56ef10c1ebcd7, 0xbf1801d911fbd06c),
        (0x3bcce92f388f20fc, 0xbf2b605a5ade3a80),
        (0xbb539bcd81dc9893, 0x3ecb0a2608144a3d),
        (0xbb6b367577014662, 0x3eceeceb341ad833),
        (0xbaf6cb1030890b1a, 0xbe6f623fc7be9f9f),
        (0xbb0412babf35582b, 0xbe65a9ca94a0d7dd),
        (0x3a9025c74264c26e, 0x3e0638ba2f5f5e6b),
        (0xba745be766d5eb9d, 0x3df4a1b4217ef864),
        (0xba32b2b63b552f66, 0xbd9530c712e738d4),
        (0xba0b3a7b1b9435f0, 0xbd7c6e5208e89146),
        (0x39aef5cdd93013e3, 0x3d1d0eb5303a804a),
        (0x3990081346d33170, 0x3cfda6ae03deea13),
        (0x39354c124a1478d1, 0xbc9e05ef8aaeac88),
        (0x3913481334a524ae, 0xbc783796c23b10d8),
        (0xb898c79983099259, 0x3c183703be1e37be),
        (0x386b3b99444e2d90, 0x3befd20ef0b0aee1),
        (0x381f9397cf40686a, 0xbb8f58105cfa40b0),
        (0xb8049a8a70075841, 0xbb612ee751921afc),
        (0xb7875e58d602839d, 0x3b00a4ae3e504c17),
        (0x376491233ef907eb, 0x3acf0d6bf9e21094),
        (0xb6870c562302f707, 0xba6d8997e250f98a),
        (0x36a578a2cb0b00eb, 0xba37af38523f4d8b),
        (0x3679f669a4d364ef, 0x39d69507180a7fc3),
    ],
    [
        (0xb6079fabfeb11cd6, 0x39708ac287c0d455),
        (0xbc569479644686c3, 0xbfc2f206e49909c7),
        (0x3c0fab1b83d45c37, 0x3f64dbf6a9fb80d7),
        (0xbc32d9141794e6d9, 0x3f99336443318ed1),
        (0xbbd0d543bfdf046d, 0xbf3bb6aa3d4e9e78),
        (0x3bebe3f687043927, 0xbf540aaa5d94bd8d),
        (0x3b419d1c00b33367, 0x3ef5f61b666129a6),
        (0x3b7d88846cdeebc8, 0x3efe4158391f2c2b),
        (0x3b46369cf5ae0043, 0xbea07a7a0745f74a),
        (0xbaff9b3405275d12, 0xbe9a8ea97b670057),
        (0xbac6b3f282a770b3, 0x3e3cb10b3affeaff),
        (0x3a82ba290c76ba69, 0x3e2e6d78879f98a1),
        (0xba64a2f934330d70, 0xbdd044447904054e),
        (0xba47c3a2994a01a1, 0xbdb8850c99b76d50),
        (0x39f0a14b971e592d, 0x3d59e4fa8ff52f90),
        (0x39daa4eb91c06585, 0x3d3d4b2c1afd964f),
        (0x3939ae655dba1592, 0xbcde807e2b00279f),
        (0xb95a357647485d8d, 0xbcbaf942a59e1d56),
        (0x38d3683474a866fa, 0x3c5ba64c4108b1cf),
        (0x38d4c3ea257eaf58, 0x3c33ba614d4a472c),
        (0xb87c3f27ee7c8094, 0xbbd3e1ce58589614),
        (0xb8227a954e0b146b, 0xbba779ee6dc8cecf),
        (0x37e88af1866245a5, 0x3b473c7bb3ac4992),
        (0x37bd028f8f76417f, 0x3b172ca46f7c7bc4),
        (0x370731151b8a5503, 0xbab681c8b15fa806),
        (0xb7217a27a51bae97, 0xba83492f6b615d6b),
        (0xb6cf3fd3e1e70ee8, 0x3a22406d4c30a6aa),
        (0xb66fde7de35eca76, 0x39eb5d275267b9b5),
    ],
    [
        (0x3c6b7326e3fbaa70, 0xbfc2740819f1caaa),
        (0xb60fcb36d157f3a9, 0xb974f95783b2a4e6),
        (0xbc5b7326e3fbaa6e, 0x3fb2740819f1caaa),
        (0xbbe4b7a0f7878877, 0xbf49b6f37d0a00f9),
        (0xbc011249d3675d9f, 0xbf78868d7401bf2e),
        (0x3bb6d7fdfc21db19, 0x3f1470a7cbcb436a),
        (0xbbc5a6b64d367c53, 0x3f29fe94ce3d3e83),
        (0xbb5352a68b135b7b, 0xbec71660e61f0f0c),
        (0xbb68d34686b6e2aa, 0xbecd6dfcdb026028),
        (0x3af006603a221bb1, 0x3e6ae79c35d01bca),
        (0xbb0ef1a3e0a6b3b6, 0x3e64ac50be6ca02e),
        (0x3a8def2f028aea7c, 0xbe03258409210f45),
        (0x3a9043d06a02be21, 0xbdf3c09685c05e25),
        (0x3a14118389ed7283, 0x3d925d85992670e2),
        (0x3a183ee519f14ba8, 0x3d7b509288f452b8),
        (0x39b2d75ecdbfec21, 0xbd19586d0c8690d8),
        (0x395ab270216fde7c, 0xbcfc973a175ffbb5),
        (0x3937848946c30fb7, 0x3c9a5d61717a7048),
        (0xb90bfd5527d43039, 0x3c776fa9ee1a4030),
        (0x38b4a933130a967e, 0xbc156907cfeb9559),
        (0xb880e1c36e1b1a90, 0xbbeee74de5f3fe9d),
        (0xb824cf2a1363166f, 0x3b8be6f4a0dc6589),
        (0x37e62c8c1385fd2e, 0x3b60be8c55ae96fd),
        (0x37612da67cbdd550, 0xbafdd3c1d9b37f13),
        (0xb755b91662c8fbb1, 0xbace5ad7a8b76af2),
        (0x37022a0cf4345b80, 0x3a6aa28d0971fd8b),
        (0xb6d5974c96417b09, 0x3a37382bb4104861),
        (0xb67d94e7b14fbe9b, 0xb9d478c454226508),
    ],
    [
        (0x35ad2d12470026b7, 0xb961b6856d13f265),
        (0xbc6be2029a752b30, 0x3fc1ff5ebddd3c3a),
        (0xbc04bba52e05ed21, 0xbf61e2035324643c),
        (0xbc3a42d5bcad2050, 0xbf97f3506d4a1231),
        (0xbbd8ddfdbbecb4f0, 0x3f37c65c9302c53b),
        (0xbbff79e837d220e0, 0x3f53117816335151),
        (0x3b97f81238bbf14f, 0xbef2df9afa521294),
        (0xbb9270fec55a19f5, 0xbefcd5d4a9d78a14),
        (0xbb3a3cd8b565f901, 0x3e9c672d4d6f7766),
        (0xbb2736243a1ddf58, 0x3e995e4b102194f0),
        (0xbad74d875eae428a, 0xbe38d22636140e12),
        (0x3acf7d99ccb23385, 0xbe2d24fc9458bf7f),
        (0xba609b81368ae5f9, 0x3dcc462d9da63971),
        (0x3a5ec11516a5a336, 0x3db78eafb15964df),
        (0xb9d97eb9ab4c3802, 0xbd569f5d498b660b),
        (0x39c6683560d9facd, 0xbd3c3c4e5b0fab49),
        (0x39720c56827dbad1, 0x3cdaccaf11834f8b),
        (0x393a34dc5c675840, 0x3cba161c63abb183),
        (0x38da1df31f53fe59, 0xbc5870224af67b03),
        (0xb8dad0b4c075b658, 0xbc3324870c0ba14c),
        (0xb8797656c00d0ccf, 0x3bd1adca419ec564),
        (0x383d51df22546bde, 0x3ba6da83c474e675),
        (0x37e46d6d0e6bdfc9, 0xbb44c90643c60c0a),
        (0x37b79fc85a4600ba, 0xbb16a1256bbedf24),
        (0xb71139afc5c7d131, 0x3ab44023f1dfc483),
        (0x372a8761b55eada4, 0x3a82e31b2fbf1f0c),
        (0xb69918ba0175b2cd, 0xba208375ee71a5da),
        (0xb68573d1d2e5403d, 0xb9eadaddc4679dad),
    ],
    [
        (0xbc4081c2a50ad27b, 0x3fc192f2627a74e3),
        (0xb602cbb028d0be5e, 0xb96f19a82d415934),
        (0x3c3081c2a50ad27f, 0xbfb192f2627a74e3),
        (0xbbed88bdcc1e72ee, 0x3f4635b38affe698),
        (0x3bfbe434e30d7a5a, 0x3f775eceaabf7f86),
        (0x3b9ca8671aa555b7, 0xbf11ac9e0164f7c1),
        (0x3bc6ba93b83783b1, 0xbf28cc464a35b0af),
        (0xbb6979a9d3cf3231, 0x3ec4014d9bf389c3),
        (0xbb64740ce7aaeb6b, 0x3ecc1f05a2d85165),
        (0x3b05322df741e1c6, 0xbe6761d18ebb04af),
        (0xbb0eb4591bc8e09e, 0xbe63cb9af103e0f5),
        (0x3aa091626e94e457, 0x3e00b3cb55bbe5d2),
        (0xba9e3e57a5ff1ba9, 0x3df2f5c763b33667),
        (0x3a315b0c770078e5, 0xbd9017771db5a5e9),
        (0xba14aef80f7b4c8d, 0xbd7a4a5e79f39fb9),
        (0x399f2119d96b7934, 0x3d165128cc4d99cc),
        (0xb99044f60a617a04, 0x3cfb991575a9dafe),
        (0xb926aff2cdbba534, 0xbc9756473a57c945),
        (0xb8eddecc5081ac54, 0xbc76b070fece7fc3),
        (0x38aa9bf0f1b7e51f, 0x3c130e1f9f15895b),
        (0x387bf52f2b971db4, 0x3bee01b29beac9db),
        (0xb82b5d175030a5d2, 0xbb88f8681f6cb14f),
        (0xb802f353c2babb03, 0xbb604e16aece260c),
        (0x3796062f77cde739, 0x3afad6e7811caa58),
        (0xb769dd7aa9c09187, 0x3acda3c957b84f57),
        (0xb69ebf7c2909d637, 0xba681861ce3144a5),
        (0x36d6353c4ed8910a, 0xba36bb1e8cb840f7),
        (0xb6713ccdbc67576a, 0x39d29b2e1989c79c),
    ],
    [
        (0xb60827407277ec38, 0x3962a629f22fc940),
        (0xbc4e7aa4db2a788c, 0xbfc12dd55d4be2b3),
        (0xbbf6da5fa7c96d5a, 0x3f5f1aee31818d19),
        (0xbc319e43d5f0f1d3, 0x3f96de64242a8310),
        (0xbbd5e566736996f3, 0xbf34afdf89fca61a),
        (0x3bd951e5980d1ee4, 0xbf5238cfc13ac771),
        (0x3b9632bd2bd2da27, 0x3ef0719d13e00e52),
        (0x3b672b1f29067499, 0x3efb974781a526b4),
        (0xbb1d4aa2a14eec77, 0xbe98cc82a70d752f),
        (0xbb3bd18c494bed4b, 0xbe9850ae878c25bb),
        (0x3ad9b4f299f8cbdf, 0x3e35bba73e282ede),
        (0x3ac284476e69e80f, 0x3e2bfe1396e83644),
        (0x3a452a15bea62efd, 0xbdc8d7dbe4a241ab),
        (0xba562396ff096680, 0xbdb6adfbd773748d),
        (0x39fff36c9935638b, 0x3d53f51dcb398ad2),
        (0xb9d0c0a18775989a, 0x3d3b40d30be85f3d),
        (0xb968a32d9f779535, 0xbcd7bf544872d9cc),
        (0xb9300a2db3cd9c4c, 0xbcb93f163a09379e),
        (0xb8f5e58d33a695d3, 0x3c55c1ca7d35fae3),
        (0xb8cde16328522c93, 0x3c3293b43a786cbd),
        (0x3868b9d269703d27, 0xbbcfa1df021b9c1e),
        (0xb81fb6589e93977e, 0xbba63d0b33ef1315),
        (0xb7ebbf0e52f441d1, 0x3b42afdbaac319ef),
        (0x3763682a897a7957, 0x3b1614280111c2d3),
        (0x374088e1875cc18a, 0xbab24b9d3911b7e1),
        (0xb7203b62eacc40a5, 0xba82796c2afed41a),
        (0xb6bf609b433463fb, 0x3a1dfaf4709fa729),
        (0xb678488c980025d3, 0x39ea51c4a8e6f489),
    ],
    [
        (0x3c60c06e2860e868, 0xbfc0cf3ee98f769b),
        (0xb5f9feaabed5f911, 0xb969e1daf76ea4e3),
        (0xbc50c06e2860e867, 0x3fb0cf3ee98f769b),
        (0xbbb4697a99d5d86c, 0xbf436f451f6e14fb),
        (0x3c194225ed089993, 0xbf765d05948a946a),
        (0x3b91ed6777d1a61a, 0x3f0ef55c5a0d16cd),
        (0x3bcf058a2a3a8b03, 0x3f27bfec9d15d038),
        (0x3b563856fcc3c4f3, 0xbec18c549f28e528),
        (0x3b59158b454353b0, 0xbecaf7544eeac766),
        (0x3af5145c85d21cb0, 0x3e648e81edf38b1d),
        (0x3afbd8241e556e27, 0x3e630341e78d1014),
        (0xba51c1fc73bfe02c, 0xbdfd73d2c942e970),
        (0xba8a049a511a826c, 0xbdf23ec2729d3a6d),
        (0x3a22971a921d0af9, 0x3d8c78e44cdec8b3),
        (0x39d9c56e9b4dce66, 0x3d795a6f6107b4e6),
        (0xb9bb8e56874267da, 0xbd13d1c77de8a03d),
        (0x3998a5b51d6fc50e, 0xbcfaace944141c60),
        (0xb93cff4a30c305b7, 0x3c94cfcb5e849650),
        (0xb8e907aca70c354e, 0x3c75fbc3b0ec933d),
        (0xb8a0d1de4a1de392, 0xbc11115c48c08b34),
        (0x38719338bd822266, 0xbbed24eb2fafa2dc),
        (0xb7d3634e9413e271, 0x3b8677909f205b32),
        (0x37fb57fb5bb117fe, 0x3b5fbfecf15f1ecd),
        (0xb7822bf0c08e088c, 0xbaf8423bd6a1b4d1),
        (0xb76a00917d57eda7, 0xbacced1ef16333df),
        (0x36eddbee5e40116d, 0x3a65e06ebbfe2e14),
        (0xb6cf548c275b3e70, 0x3a363be97f954678),
        (0x3646fc15979b90b0, 0xb9d0f619585e1af0),
    ],
    [
        (0xb5eea501cb87ef99, 0x3972465ed4007bdd),
        (0xbc64353fd6c42f1e, 0x3fc0768257dad56a),
        (0x3bf3a4186dcff329, 0xbf5b602a7beaaa48),
        (0x3c16aaef36d826e2, 0xbf95ebc22efd092c),
        (0xbb76c80c9b064de4, 0x3f3236a604142e61),
        (0x3beddcf66c5978c8, 0x3f517a482faa8d85),
        (0x3b66ad649af92b73, 0xbeecfc00890787ae),
        (0x3b928f6dcddab559, 0xbefa7d7b128ac538),
        (0x3b25a6c98550b698, 0x3e95e419f1b57043),
        (0xbaf32b6c630a02f3, 0x3e97603cea235244),
        (0xbad9cee3035468eb, 0xbe333a0b436c8824),
        (0x3abe7890b1e58aeb, 0xbe2af46417845f40),
        (0x3a4f73757ba505ba, 0x3dc609ac8cd8177a),
        (0xba59cd5b34bda911, 0x3db5e11b87ad3903),
        (0x39f753770c6a00bd, 0xbd51c2756590391a),
        (0x39988f32e9922679, 0xbd3a586483dd891d),
        (0x3977131f1f9126e6, 0x3cd534d22c600962),
        (0x39500677438cdb27, 0x3cb8755ad322c506),
        (0xb8f45577511a2e70, 0xbc53809e528c5e70),
        (0x38d78081f89f0b4f, 0xbc3209a0a78d692e),
        (0xb8532441923d88b3, 0x3bcc77842d1425ee),
        (0x38002948000e0b89, 0x3ba5a4538094a38e),
        (0x37d2ec02968bd2b8, 0xbb40e2a860417e77),
        (0x37b26676c0acf6d9, 0xbb1588f3ef728c24),
        (0xb75b53bc7d88d00a, 0x3ab099700fe34bf3),
        (0x36e30a044384490d, 0x3a820f1c712673f5),
        (0xb6ab8f46c26e1b98, 0xba1b4fa75c261c78),
        (0x366effaba36280e2, 0xb9e9c5f163a1d50e),
    ],
    [
        (0x3c61166b7995967a, 0x3fc0230ba90f2871),
        (0xb5c8e0d3f893a50f, 0xb944e3fd95304f5b),
        (0xbc51166b7995967a, 0xbfb0230ba90f2871),
        (0x3be85d0df08ff932, 0x3f413164a0864cde),
        (0x3c175b99fcae0fa4, 0x3f7579c1bdbcfc99),
        (0xbbaa847d3261afc4, 0xbf0b67e1913c668b),
        (0xbbbdc3b0bfb76e55, 0xbf26d26de4fd8c5d),
        (0x3b509cffcaa70410, 0x3ebf1b520b063853),
        (0x3b56e93d7c50e034, 0x3ec9f01e7c1909a0),
        (0x3aeb43dee887c41e, 0xbe624071b1796027),
        (0x3a99b4c11c97bb23, 0xbe624f8e939ce43c),
        (0xba9ac0bb48385d05, 0x3dfa35663595463f),
        (0xba62567920c13146, 0x3df199120d49d507),
        (0x3a1bdcb5b3b180ed, 0xbd896771e1b15e44),
        (0x3a1adae6ea9dcc28, 0xbd787ede134ed7a0),
        (0x39ac92871627b22c, 0x3d11bccdfbafa1fd),
        (0xb98c58fbb742d0bf, 0x3cf9d22a6dc5a57b),
        (0xb91dd27f9db4339f, 0xbc92b00e367301ce),
        (0x391afb7d91b7ccfc, 0xbc755236722b2dc7),
        (0x38ab20732a5e1932, 0x3c0ec2ce49a984e5),
        (0xb878a4f67789acc8, 0x3bec52d2fb40494f),
        (0xb82efb139c9eabfa, 0xbb84522e39f45c23),
        (0xb7e18df511f3aea9, 0xbb5eeb3ed3be49c9),
        (0x3790a89c14bccf9f, 0x3af606528674e24f),
        (0x3762e9176a7f8941, 0x3acc3a09b96f5019),
        (0xb6d1b63a61a018b7, 0xba63f00d5dec7881),
        (0xb6d89c276f90a3f7, 0xba35bd3ff3734b8b),
        (0xb63f6e7b2ed1bd35, 0x39cf0615ab9d7cb3),
    ],
    [
        (0xb5d35e9581fc00c3, 0xb940bf81e7b6bf7f),
        (0x3c10417847765c1c, 0xbfbfa8b3f9ae4375),
        (0xbbcf334b36194e10, 0x3f5856073b7fa2cd),
        (0x3c3664c698368b18, 0x3f9514e652eb2e96),
        (0xbbb93b53ec8276c9, 0xbf3032298718ea1a),
        (0x3bfdfbf103edc8e4, 0xbf50d1153fde431b),
        (0x3b7c365adad42aa7, 0x3ee9cb455c29d6a7),
        (0xbb9fa9b04c310c1c, 0x3ef982167b1c52c3),
        (0x3b3466b70782cb7f, 0xbe938191ef68eab3),
        (0x3b31f4e70613c5d6, 0xbe968865345b3130),
        (0x3adaf50c9e003b40, 0x3e312960010da387),
        (0xbace36911be0e026, 0x3e2a0403679f6882),
        (0x3a41779eb4010bf6, 0xbdc3b67cc0b96c06),
        (0xba565f31bf6e70fc, 0xbdb526038b956efd),
        (0xb9d5ed9568f49fa3, 0x3d4fda06e840fa34),
        (0x39d1e175fdd2dad7, 0x3d3981de640d0e00),
        (0xb97d648a5890112c, 0xbcd311fc56c4b888),
        (0xb93dacd97e23c3ac, 0xbcb7b8f62899fc83),
        (0x38f158e1e7d5426e, 0x3c51977cb2740dc9),
        (0x38da9c4d3d78fa2e, 0x3c318706eb31dddd),
        (0x38690756a8b2b9b3, 0xbbc9c2ecf3513bc2),
        (0xb843d0ba97dc7b3a, 0xbba511e7341f19f5),
        (0xb7c70af900714fae, 0x3b3eaa18f0a1bc9d),
        (0xb7821dafb6b0a02f, 0x3b150190ffd95109),
        (0x372d7c9b2e2dbe3b, 0xbaae3fb72389f327),
        (0x372d26a5ec602bfe, 0xba81a629641ad013),
        (0xb6b095243f6598d9, 0x3a18f85627d46b37),
        (0xb679a0a342b383f9, 0x39e93a4bfaf23e38),
    ],
    [
        (0x3c50db2c50623ec0, 0xbfbf13fb0c0e6fcd),
        (0x35fba9ea35ce88dd, 0x396fbca69a15def7),
        (0xbc40db2c50623ec1, 0x3faf13fb0c0e6fcd),
        (0xbbddaa71693254ea, 0xbf3eb3e6fcc47c00),
        (0xbc1a7c91ef9a7d9f, 0xbf74af74cbd77bef),
        (0xbb8ca4e36c0e1909, 0x3f087bb1ebeaec75),
        (0x3bc0dc1e11076c53, 0x3f25fe629203150e),
        (0xbb5990b5b3489011, 0xbebbd0f2a6555e6b),
        (0x3b6af9c37c820e57, 0xbec9040de830649e),
        (0x3ae759c4b2a6504b, 0x3e6057f7a76993d7),
        (0xbaf1d65072f86b24, 0x3e61ad6dd5105c3e),
        (0x3a9d40eb328040f7, 0xbdf782a6f5738cdb),
        (0x3a8bf5e5277cbb15, 0xbdf1027dc06d4453),
        (0xba28e3867173de2a, 0x3d86d65b13b16f65),
        (0xb9fcb2787109df88, 0x3d77b59e9ed7367d),
        (0x39ae1b468c4215d5, 0xbd0ff800f2a49a01),
        (0xb999ff66e26a3daf, 0xbcf907be43054aeb),
        (0xb926a66c1da18e82, 0x3c90e2e1d8f5ad75),
        (0xb91450507fc3a22f, 0x3c74b3a8b0fb5013),
        (0x38960772f0ae799d, 0xbc0be06d96b7165d),
        (0xb880df93b3817817, 0xbbeb8c22733ba4a9),
        (0xb800279bf5e0f4bd, 0x3b8278dcce924024),
        (0x37f6405c8f354677, 0x3b5e1fb516b57c41),
        (0x379e9caf5e3a0d68, 0xbaf4159150b0936f),
        (0xb7554adeee798af2, 0xbacb8c841ae83900),
        (0x36e38beff289f2bd, 0x3a623d5f11f3220d),
        (0x36c4bb8b7542c332, 0x3a3540f31271ca17),
        (0xb6641ee382a94dd9, 0xb9cc774f548476e1),
    ],
    [
        (0xb5a8c0565f36d2df, 0x39138a74ee514d1c),
        (0x3c56a7c2ed8fa844, 0x3fbe8727c572a2c2),
        (0x3bfdee3ce04b3d12, 0xbf55d1ef092ab395),
        (0x3c206dda20eba87a, 0xbf9454e7a7395636),
        (0x3bb99eb39ffae527, 0x3f2d0cc3a7fa6d3d),
        (0xbbc9d1e5b54f9af7, 0x3f50398d2cbd02df),
        (0x3b8aa1f4bf33f006, 0xbee725b0909f9c66),
        (0xbb9b20d87a760073, 0xbef8a022f5fee447),
        (0xbb1fcb0c42b327cc, 0x3e9185b1ea97a54b),
        (0x3b3ea0aa1deb8165, 0x3e95c571167401ae),
        (0x3acd2759d32f612d, 0xbe2edff6514e1da9),
        (0x3ab2592f69cf47b5, 0xbe29298da3c4fa71),
        (0xba621593a3e29e00, 0x3dc1c34705eadfb1),
        (0xba424eda63e46424, 0x3db47ab8ec529c51),
        (0xb9e71b8905efe1df, 0xbd4cc24a2f007c3e),
        (0xb9c2a661b676a229, 0xbd38bbcc44e6c225),
        (0x396f64643e54c71a, 0x3cd1420df9bb4358),
        (0xb952c79ec699f349, 0x3cb7095f8b3266be),
        (0x38e9b99fa1cd4cdb, 0xbc4feb87e4e6508e),
        (0xb8d4b81d9d1d9436, 0xbc310c0caf4ba7ed),
        (0xb8635b1ba8939988, 0x3bc76f4ca2183df3),
        (0xb8470f94857758df, 0x3ba4868177e6a94b),
        (0xb7d51e8dd2ddba52, 0xbb3bf93dac9dd574),
        (0x37b66de2ce896857, 0xbb147f2c3ea4f41f),
        (0x37422b1ac15ef2d9, 0x3aabacb94681ac1b),
        (0x372b5172ed4e760e, 0x3a813fd96ffd162f),
        (0xb6bfc10cab672280, 0xba16e97ff088b7e8),
        (0x368ef32e3b7f65be, 0xb9e8b0d2637124f7),
    ],
    [
        (0xbc3b9f1d130797af, 0x3fbe018dac1c17e3),
        (0xb5e141aefdec9b98, 0xb96cc7fef8ec3e0d),
        (0x3c2b9f1d130797b4, 0xbfae018dac1c17e3),
        (0xbbc5edf62333ab91, 0x3f3ba21bd15d02a4),
        (0x3c138b4f0d3a410f, 0x3f73f9e0db07e7ef),
        (0xbba6f23ddcae3185, 0xbf060b77c5e27622),
        (0x3ba86d4e04faa24d, 0xbf253f9b1a5d228c),
        (0xbb5fddc005e37ab4, 0x3eb910b38812c253),
        (0xbb40935ad5366a7f, 0x3ec82ee6dfdfedeb),
        (0x3afa8bf30766210f, 0xbe5d7cc2a9a602ee),
        (0x3b0b507850a97457, 0xbe611a57d8645358),
        (0x3a8673cba0bc56e8, 0x3df53d401519442d),
        (0xba8c0365adf75127, 0x3df07915ef87408d),
        (0x3a20221e3beb4ce4, 0xbd84aabcb6141c76),
        (0xba1d282fee2792c3, 0xbd76fcb8e2eada15),
        (0xb9a71c1e6871ca1b, 0x3d0cfcfdb2faeb02),
        (0xb99d6b8f41ec2a2a, 0x3cf84c5369d2e33d),
        (0x39204114eb9c2f69, 0xbc8eb11d7e6581d1),
        (0xb9191341e21aab61, 0xbc741f9ab4632bff),
        (0x38a996918c6850e8, 0x3c0965159407e959),
        (0xb88a5f0c11aeb57c, 0x3bead0e3ccfe39e7),
        (0x382f54e660147ef3, 0xbb80dee0e993354d),
        (0xb7f1c5f5adc77455, 0xbb5d5e03476b81b0),
        (0x37898d6415c2c8f8, 0x3af264535f4853fa),
        (0x376323c802fd85b0, 0x3acae5afea45a1d0),
        (0xb7064e5aaa6a944b, 0xba60bfa00dd729a9),
        (0x36cb04c64064bf04, 0xba34c82d1f8992b1),
        (0xb64e911d15e3dd3d, 0x39ca34d3a49f024e),
    ],
    [
        (0x359edfac18b305f3, 0x39163cc00f5dbc0b),
        (0x3c555d27e18add4d, 0xbfbd82939ab62339),
        (0xbbe5f2055a671371, 0x3f53b5a54845670f),
        (0xbc331ee0af44ab91, 0x3f93a7ff1622def8),
        (0xbbb2001fd86915ca, 0xbf2a3ebc476a606d),
        (0xbbb7a5c728534a7e, 0xbf4f61adde3a8c61),
        (0xbb6721b027970cdc, 0x3ee4ec45da0478b0),
        (0x3b60295953d04e61, 0x3ef7d3b28159f23c),
        (0x3b21dd48381068cc, 0xbe8fb3f21bcd7fc4),
        (0x3b3047a478ead9ea, 0xbe95145a876d2273),
        (0xbac575cc2dd16f8c, 0x3e2bf64cdf7c0558),
        (0x3a9c7ea36a18f920, 0x3e2862204ac427d2),
        (0x3a25529af35b5dcf, 0xbdc01c0d48888811),
        (0x3a3766e366a3fc7a, 0xbdb3dd6cb1cbcd8f),
        (0x39e0fa707938a9eb, 0x3d4a2020c8b3c20d),
        (0x39bbbc58f9d26ec6, 0x3d3804a9d711292e),
        (0xb96e85f6cc946fb5, 0xbccf6a1ef9182e2a),
        (0x39591e00ef906d2f, 0xbcb665c82dfd3112),
        (0xb8eea5743e788fee, 0x3c4d1cbc474ae19a),
        (0x38c6e72f6eaa6ee4, 0x3c309883728ed044),
        (0xb85fe1b5b6914470, 0xbbc56bda19bbd907),
        (0xb83f2fa5959c5300, 0xbba4025c7e86cfc8),
        (0xb7d115ca11e0e05e, 0x3b39a17a25046119),
        (0xb7a514b6675a5e5f, 0x3b14026121051408),
        (0xb73ae1c02372f342, 0xbaa96b5d080744b6),
        (0xb6f3427ab8e0fc99, 0xba80dcf24b28c249),
        (0x36b101dd967ab100, 0x3a1519032b200dc8),
        (0x3675b89a8e98627b, 0x39e82ad2bc03a6a9),
    ],
    [
        (0xbc56edd809f4ec43, 0xbfbd09b21e36c0bd),
        (0x35e5d21c4590e23a, 0xb963ad3554cdbdc3),
        (0x3c46edd809f4ec44, 0x3fad09b21e36c0bd),
        (0x3bd1da06fbc18d60, 0xbf390b0e6a475e45),
        (0xbc140975d5184af7, 0xbf7355b904fbf7ee),
        (0x3b9e3b983f1177c4, 0x3f03fc459d1e25a4),
        (0xbbb494c77a177888, 0x3f2492cc61d19dfe),
        (0x3b46367d4d2995c5, 0xbeb6bcf110a02ad1),
        (0x3b6b6ea46d6df357, 0xbec76d44f6a83523),
        (0x3a803005742b014c, 0x3e5ac61efcb3c36c),
        (0x3afeb84629addc98, 0x3e609436fcaa3965),
        (0xba99e294d7264bfe, 0xbdf34eb6095f969b),
        (0xba88db7c11c38cf9, 0xbdeff662945eb368),
        (0xba237b9cbf75a8c0, 0x3d82d05bf8ff5181),
        (0xb9d4ca3e6003606c, 0x3d76525fbb563fe0),
        (0xb9a93e3cfc188270, 0xbd0a6eb937c7f6c0),
        (0xb98026c04e791e1c, 0xbcf79e90468589aa),
        (0x39252d8dea0a8e75, 0x3c8c0998b46c2a84),
        (0xb9065519f44afceb, 0x3c73955fe5fb10db),
        (0x38ac6a20d09a8309, 0xbc073ec89759ba44),
        (0x3868ea900cd0acee, 0xbbea20bd7b455493),
        (0xb7fd0a1ee99e85f5, 0x3b7ef3962cbda037),
        (0x37ea27d0e101e8a6, 0x3b5ca6506245824f),
        (0xb773dd8f476a0af7, 0xbaf0e8cae31c27f5),
        (0xb75f5c680a514474, 0xbaca461c76aba331),
        (0x36c4607ce2cc9a56, 0x3a5ede6da0b05cf5),
        (0x36b52adf7a554ae1, 0x3a3453a0c69c3020),
        (0xb66d21d6c5df4ac2, 0xb9c8341f8ff89500),
    ],
    [
        (0xb604e106c1a5d68b, 0x3960e1cf91ecbcd0),
        (0x3c4e1f19f10295a0, 0x3fbc96700031f601),
        (0x3bd8339df4a4988a, 0xbf51eb2a07d0f09e),
        (0xbc39bfe53ec12c89, 0xbf930b36eddaa234),
        (0xbbb206af4ba8cd76, 0x3f27dd2dde84b73c),
        (0xbbd1f3a6f0dd2e5a, 0x3f4e696553e0b8a5),
        (0x3b70bebcc672dde7, 0xbee3085aa35a1647),
        (0x3b964955e62a56d1, 0xbef7199f24bccae1),
        (0x3b29acbd06286b9c, 0x3e8cdbab6a766176),
        (0x3b336b635edd5e8a, 0x3e9472a7cffbf91e),
        (0xbaa919ffae68f01f, 0xbe2979f5211c4dae),
        (0xbacb3a83d6a4505a, 0xbe27ab4b01a902db),
        (0xba3d20efa9eb9b62, 0x3dbd63904e4c487c),
        (0x3a3a2266d5d6a8ef, 0x3db34c83fb1c2279),
        (0x39d10d542056a77f, 0xbd47dcfdf13e5ccc),
        (0x39b3859d37da710a, 0xbd375b01ddacf988),
        (0xb9565b371d7ee634, 0x3cccbd41ea6afd27),
        (0xb9581212fb798324, 0x3cb5cd46eaba1e98),
        (0x38e7f3fcdfe0798f, 0xbc4aaded22d1c26d),
        (0x38c46a63e955520e, 0xbc302c0feb2cc9ce),
        (0x385cf68aedadaeff, 0x3bc3ab0efb07293e),
        (0x384f8dd325a6cd09, 0x3ba38564ae72454e),
        (0xb7d09df47a273d60, 0xbb379494d9aa279b),
        (0x3794fe8c13f0d784, 0xbb138b6c661b3cba),
        (0xb7395294aeae0a19, 0x3aa76f5fd4aa472c),
        (0x3729b16a6763e36e, 0x3a807de116ccc64f),
        (0xb6b1c69123ace593, 0xba137e1e22a65820),
        (0x368881179d4440a6, 0xb9e7a919992ba54e),
    ],
    [
        (0xbc5ca34ef67ceca5, 0x3fbc2861347b1b39),
        (0xb5d9566d846e8e8c, 0x396cceb50dfed9ea),
        (0x3c4ca34ef67ceca4, 0xbfac2861347b1b39),
        (0x3bbf201ae22fff02, 0x3f36d57bffb37fd5),
        (0xbc16b4ecac2fa1c1, 0x3f72c060ef553f18),
        (0xbb9df91b90d34532, 0xbf023a407b722d2e),
        (0xbbca3432d200d5d1, 0xbf23f55581ec683d),
        (0xbb5e04bafba5db02, 0x3eb4bfb4daa2ff62),
        (0xbb42a29e46400df9, 0x3ec6bc69099af2f7),
        (0xbafb52717e31e7c3, 0xbe5873340987d276),
        (0xbadb973833d824fb, 0xbe601951f02d2b37),
        (0x3a8c2da9623b39af, 0x3df1a6174fe55fbe),
        (0xba72657ffe3c0d22, 0x3def0eca5bc0284a),
        (0xba1b8e7e494f244b, 0xbd8137cb1fcc55a3),
        (0x39c37c1a924f4abf, 0xbd75b4f7f2bb19ed),
        (0x3999da79a9d54cda, 0x3d083934ed797abc),
        (0xb989eab29e540769, 0x3cf6fd29fcbcf38a),
        (0x3920c6c752968451, 0xbc89bbe4a386b212),
        (0xb91a160d6358c9f0, 0xbc73143b8f6fa78f),
        (0xb8a480a782f588c3, 0x3c055f1ef8c6530d),
        (0xb887a3802525f474, 0x3be97b209a79f9f3),
        (0x37f474a28bfc52ff, 0xbb7c822a3d9869c8),
        (0x37f5fabb96a9e9b8, 0xbb5bf86d95cc8f12),
        (0xb78a0516ef829f54, 0x3aef35917d0b3f76),
        (0x37622045f0e24361, 0x3ac9adf870fd0a7b),
        (0xb6e0bdb6d552746a, 0xba5c8b4b32991b65),
        (0xb6ca53a583f641f6, 0xba33e3ad6afab22c),
        (0x3644ed17a9579532, 0x39c66c8a74f2a901),
    ],
    [
        (0x35d31fe5aa13389e, 0x39517eb126abaf87),
        (0xbc2986f8b543f27a, 0xbfbbbf246019c0d4),
        (0xbbf8243dc68fb135, 0x3f506224199140d8),
        (0xbc31c8ab4f070de8, 0x3f927c3416c09898),
        (0x3bc1400990dffef0, 0xbf25d279dc87cf69),
        (0xbba49f5dbb72a065, 0xbf4d86a5f5adbdac),
        (0x3b8c1a134d478c22, 0x3ee169041634c248),
        (0x3b908bd4df68b45a, 0x3ef66f5bd1bedae1),
        (0x3b2340bc492fa102, 0xbe8a69b2c717d54a),
        (0x3b2b327658083b81, 0xbe93de4eab456cbd),
        (0xba8a5251b6ada8e0, 0x3e2755eabbde6915),
        (0xbaceee9ff7faabed, 0x3e2702fe7ba9f8c3),
        (0xba380d3becf77482, 0xbdbaf2007eeae256),
        (0x3a5fcfcac159b57e, 0xbdb2c69746948e55),
        (0x39e98e02f368d7f4, 0x3d45e77ad9a9daf0),
        (0xb9d9fd1217de9157, 0x3d36bd7bc6045cd0),
        (0xb96832ead3387986, 0xbcca6a2a200a32c0),
        (0xb955d0d8052f28a5, 0xbcb53ef0001a8fc4),
        (0xb8bb1f311ea4c9b4, 0x3c488eb37fc1cfd3),
        (0x38cf9a896df080b1, 0x3c2f8c832f51b4aa),
        (0x37d4b18c2095b824, 0xbbc22200a0360453),
        (0xb829797fcaa4e047, 0xbba30f596dbfddae),
        (0x37a2d23a75e1c560, 0x3b35c6dd016030cd),
        (0xb734e085d594092d, 0x3b131a4ea769c7d3),
        (0x373bce3510c7cb6b, 0xbaa5ae7bd23b9081),
        (0x371fb7fbd6256541, 0xba8022d72980d3c8),
        (0xb6916a353736b148, 0x3a1211535635a157),
        (0xb68195f4662a5736, 0x39e72c1b69eda79f),
    ],
    [
        (0x3c497d2b9281abc8, 0xbfbb5a622198a72c),
        (0xb5e793f1c68299a8, 0xb94513e9f11c844a),
        (0xbc397d2b9281abc8, 0x3fab5a622198a72c),
        (0x3bd29fd72f6919a6, 0xbf34ee71cf67c243),
        (0x3bef69e3e89e3763, 0xbf7237c02b462f6f),
        (0xbbaa32f85df10cac, 0x3f00b67b6fdfec62),
        (0xbbcb53a20d4b2f6c, 0x3f2365167d8bc330),
        (0xbb5df27a7ce0ec5b, 0xbeb3086b7a296e69),
        (0xbb64ce30c488b9c8, 0xbec61a141425c2a6),
        (0xbaf5a586400cf358, 0x3e56712b23991e80),
        (0x3afc0839ad44e9db, 0x3e5f50736839e481),
        (0x3a894d2884c96f7c, 0xbdf0366d02b91b84),
        (0xba3b7d3083bff34e, 0xbdee38f9bda5f6df),
        (0x3a0f47f210ca9110, 0x3d7faa3d1cb01e65),
        (0xba1e53d61aed5735, 0x3d752317a2a5b9a2),
        (0x39a61437236ae98e, 0xbd064cd06847bd1d),
        (0xb98a88cfd6fbb36d, 0xbcf666eebfeb573c),
        (0xb921559c51711009, 0x3c87b8b54f1785d0),
        (0xb906f3154f9ab684, 0x3c729b70ca32ed3d),
        (0xb8ae6effa325d36e, 0xbc03ba848f793a6f),
        (0xb8862efc7da56047, 0xbbe8df655223f804),
        (0xb7f4102cd3bf6abf, 0x3b7a5b9007f4f46f),
        (0x37fe605a6b4873f8, 0x3b5b53fa1a9d0aeb),
        (0xb7858edfd4fa5bb7, 0xbaece6fce7a8c528),
        (0xb763c9762154c0c5, 0xbac91d34853abb67),
        (0xb6f1321ee8b9d4f4, 0x3a5a7ad6f4ddf512),
        (0x36c7e2ebffed7e36, 0x3a3378793c3201e1),
        (0x363afde2a34531f7, 0xb9c4d6575c0bc39d),
    ],
    [
        (0xb60d030c92133af6, 0x396cca2f2b0d4235),
        (0x3c56e5b3d654a3dd, 0x3fbaf9cb42cd08a7),
        (0x3bd4df22cc11675d, 0xbf4e1c66d7616e37),
        (0x3c32068ccdde2ecd, 0xbf91f90fd1013589),
        (0xbbb9426a5fb5b160, 0x3f240e3eb09b7d6d),
        (0x3bcf4131edb03e41, 0x3f4cb682ff471274),
        (0x3b8c9cd4cf70b12b, 0xbee0016819a61dd8),
        (0x3b8325d69b159807, 0xbef5d2d1c420646d),
        (0xbb157a8a155cbaaa, 0x3e884b0fe8a85355),
        (0x3b1d5bbe26d980b7, 0x3e93559d96bef598),
        (0x3ac3da55cd6ad83e, 0xbe257a0be74997ad),
        (0xbaac817776a96dfe, 0xbe26677c8f53b6a6),
        (0xba48a3c160c8aa22, 0x3db8d1ae21b756a1),
        (0xba55e41c05b333bc, 0x3db24a6e1ac65e87),
        (0x39d8b28dfb36d53c, 0xbd4432017034b94c),
        (0xb9b0c0b8d7240fb1, 0xbd362ae0447c7c88),
        (0xb96e042713d54739, 0x3cc86190e6a87083),
        (0xb940efa65c450767, 0x3cb4b9e1622f9586),
        (0x38cb94e9a0146b9b, 0xbc46b1f3e51e7332),
        (0x38c980a0b48ea2c9, 0xbc2ecd41315429fa),
        (0x386a6123111c55fe, 0x3bc0c7d6edd07b17),
        (0x3843ce4f1feb2ebc, 0x3ba29fe1dfade081),
        (0xb7b6d84618b5e516, 0xbb342eb57150f9e4),
        (0xb7a6474c8a6dcfb7, 0xbb12aee3771f508d),
        (0xb7333647a6c944af, 0x3aa4201b40c833df),
        (0xb71eb58df16b4736, 0x3a7f97bc8a144320),
        (0xb6aee477a8ff3709, 0xba10cc450b34befb),
        (0x368b6b8edb270aaf, 0xb9e6b4102ba6d220),
    ],
    [
        (0x3c49a6abbfd839f8, 0x3fba9d183bc04545),
        (0xb5f15a143d9b75a9, 0x395a77273a2f7cb1),
        (0xbc39a6abbfd839f9, 0xbfaa9d183bc04545),
        (0x3b97025ee34cb468, 0x3f334779874010dc),
        (0xbc01fefbfc4e4e9b, 0x3f71ba2299ab88a8),
        (0xbb55c0e2a1925dd2, 0xbefecb19f5bdc649),
        (0xbbcce75d4325824c, 0xbf22e052707cc859),
        (0xbb5a63ba682d624b, 0x3eb18a5da77edd83),
        (0xbb6dd00d5413ce37, 0x3ec5846b622f592b),
        (0xbae5e872e9afa841, 0xbe54b17e97b5884a),
        (0x3ac6a221aaf909f2, 0xbe5e7f76674dfaab),
        (0xba84b34281199881, 0x3dedeb504ed353c6),
        (0x3a86775826683d8c, 0x3ded72e25f190409),
        (0xba00d222c75ec8a5, 0xbd7d3e076b0201d6),
        (0x39c62484e2aaf790, 0xbd749b825302af88),
        (0x39ad85f9e975e5bb, 0x3d049d35475fc5c4),
        (0xb99ca04c877af6e6, 0x3cf5dac9623f7b3a),
        (0x39280969190bc2ec, 0xbc85f3d5130dc427),
        (0x390c3038a68f53f8, 0xbc722a4aeb6975db),
        (0x38930f7db32e8831, 0x3c02479e3ad0904f),
        (0x386d535dbc6c1f3b, 0x3be84cdbd491db42),
        (0x381f7a28d6ae78be, 0xbb78745ed68ebb0a),
        (0xb7eb97e422517d33, 0xbb5ab87a4b0f3210),
        (0x378e41504a923c16, 0x3aeada8753b4f2fa),
        (0x375ded58316b4b3b, 0x3ac8939aed0eb956),
        (0x36d24ca9391ef811, 0xba58a3c50700bb64),
        (0xb6c8cff6f49b7cc7, 0xba331204a4a28f43),
        (0x366456ce1c2efeb7, 0x39c36abd94cda9ea),
    ],
    [
        (0xb5d6aeec01447ebf, 0xb94fef3146f8bc8c),
        (0x3c5729f43724612e, 0xbfba4407dac72297),
        (0x3be5723d945cf783, 0x3f4bcba4dec1da44),
        (0xbc3fe775590d7dad, 0x3f91803c65cafdfb),
        (0x3bc90cfa57dbb4d4, 0xbf2283df2b3e4a6e),
        (0x3bc3e3e0391f1e1c, 0xbf4bf695e89259cd),
        (0x3b630f86ed6a9315, 0x3edd8f3759122d51),
        (0x3b9030fb5fd0d14f, 0x3ef54246c8e04218),
        (0xbb1b2983b9c2c402, 0xbe867111bc8e67a4),
        (0xbb3a3cef2b75d6f2, 0xbe92d72a1f62ec18),
        (0x3acbd165702624f6, 0x3e23d9ca3eb1e22f),
        (0xbacc01aeccf1babb, 0x3e25d74a6e225515),
        (0xba59e0b645683780, 0xbdb6f4b420b2f957),
        (0xba5cdfc243103d08, 0xbdb1d6f985dc3ec9),
        (0x39e7bf317b14f352, 0x3d42b1d81101dea8),
        (0xb9d83163c0057ca7, 0x3d35a2197089fbfb),
        (0x395c76bb39cce133, 0xbcc6974aff17c78e),
        (0x39598eb0a01efbec, 0xbcb43d4892ea1bcf),
        (0xb8ed15e01c0238c4, 0x3c450d21b45ff80d),
        (0xb84241f1dc4f6f34, 0x3c2e196b40872b31),
        (0x385c04a02280abca, 0xbbbf2abd4db5d9be),
        (0xb84959c23561aa48, 0xbba23699d666bfec),
        (0xb7d227fbae70ed91, 0x3b32c4313f19d90c),
        (0x37b6f63990f0594b, 0x3b1248f0a9284b37),
        (0x372ffeb55db91815, 0xbaa2bd11bf68d5b1),
        (0x371b75883a5be1b6, 0xba7ef1cd2cb739b8),
        (0xb6aa77e2877a5055, 0x3a0f531ca9868a0f),
        (0x368cea389ee72d20, 0x39e64106dd656961),
    ],
    [
        (0x3c4f5da9526c15aa, 0xbfb9ee5eee1a97c6),
        (0xb5f1d3382c1276a4, 0xb96402f67a0b5834),
        (0xbc3f5da9526c15a9, 0x3fa9ee5eee1a97c6),
        (0xbbc331009164145e, 0xbf31d55d93e59bd7),
        (0xbbb6887e944cd005, 0xbf7146219394a99c),
        (0x3b8ca8e50931e16c, 0x3efc7d32bb646e2a),
        (0x3b459561844c17b3, 0x3f226599aed4f1ca),
        (0x3b2f06633c8a4cba, 0xbeb03ba58ec85c12),
        (0xbb602686ed9edd3e, 0xbec4f9e3961bb9dd),
        (0x3aeea9ee2c6a2043, 0x3e5328d4dc334079),
        (0xbaef3a53acd5c6d4, 0x3e5dbda7ac70ff93),
        (0x3a77a849dd25fe0f, 0xbdebb7c2f187144e),
        (0xba64d006a8eb9c0b, 0xbdecbac24c760567),
        (0x3a16ef76e61a4831, 0x3d7b1c35b65eca4e),
        (0xba1dac5564e87883, 0x3d741d23f93de7a0),
        (0x39a9abd11b46d7e7, 0xbd03208d806698c2),
        (0x399718ab551976b2, 0xbcf557c166250d9c),
        (0xb92d6295443bb3ec, 0x3c84637130a9d7f8),
        (0x3916194c75a97d3e, 0x3c71c021934a59f1),
        (0xb89fd63bcb1b75d7, 0xbc00fed119c972a1),
        (0xb88c541b145b8b02, 0xbbe7c2d6451d6c10),
        (0x381dbc09437e925e, 0x3b76c3357c9d7057),
        (0x37fb570ed16daac3, 0x3b5a25664a0ac691),
        (0xb74761fb4819cca1, 0xbae906c5b9661a95),
        (0xb76d18cd574324fb, 0xbac810df45f48012),
        (0x36ef5b739f6495ca, 0x3a56fe3f66728fcb),
        (0xb6d5974f41aa1f59, 0x3a32b0377532218e),
        (0xb660f1fa91a8fb3e, 0xb9c22493f39e943f),
    ],
    [
        (0xb5e9353c25c13119, 0xb94016db9db1eb7c),
        (0x3c422d28ddb0663f, 0x3fb99be73fa3efcc),
        (0x3bc06623176e8b2d, 0xbf49c3248da75775),
        (0x3c24e67d5d6f01f7, 0xbf91107147eda800),
        (0xbbb9f85d6cd5a259, 0x3f212980be9d8520),
        (0xbbc6d7f4255305b5, 0x3f4b44e1221e6051),
        (0xbb75ffceb4091f98, 0xbedb67b78c1dd9c5),
        (0xbb952647474dbc63, 0xbef4bc49e9b5bcfd),
        (0xbb07d96f251b94dc, 0x3e84d02ffb8bcfd0),
        (0x3b3e32d27646b4d2, 0x3e9261c31456e43c),
        (0xbac761199eda3ccb, 0xbe226b400744def1),
        (0x3acd5778eedbc215, 0xbe2551251a2b4223),
        (0x3a46fe837cedeee3, 0x3db5500f9531112d),
        (0x3a083f723e083d02, 0x3db16b4e936f8a96),
        (0x39d95bf8d7caafb4, 0xbd415e70b28182e4),
        (0xb9d30dafff387703, 0xbd352230c03cb29f),
        (0x3945e77279062e4c, 0x3cc5019095879293),
        (0x395492014f9c999d, 0x3cb3c864dd8bd877),
        (0xb8c1fd1436671589, 0xbc4397af75e3218c),
        (0x38b351e168817cda, 0xbc2d701c9937c95b),
        (0xb855e93cc818d8e2, 0x3bbd0965d926054d),
        (0xb84897cb99dccaef, 0x3ba1d319c4fa1430),
        (0xb7dc0895bb1677aa, 0xbb3180c1e083f84d),
        (0x37be77bf0edd596f, 0xbb11e83053760c66),
        (0x374b4d74f0bce05a, 0x3aa17f5e91af9659),
        (0x371838fbe922ebee, 0x3a7e53a0935a1f43),
        (0x3684c2589850e2ae, 0xba0d4939983de020),
        (0x3684fdd4fb568c5f, 0xb9e5d2ed99505e47),
    ],
    [
        (0xbc55bbc298d062cd, 0x3fb94c6f5898708b),
        (0xb5f666b728efc40e, 0x396f720ac4a0590b),
        (0x3c45bbc298d062cc, 0xbfa94c6f5898708b),
        (0xbbc91fb79f2262de, 0x3f308f4f1b8fb0dc),
        (0x3c11274d5c28ac62, 0x3f70da931c776c71),
        (0x3b468ef3cfeee3c5, 0xbefa7550d2940f2a),
        (0x3bbf74462dfd6413, 0xbf21f3b978a47cf4),
        (0xbb46a542136ebda6, 0x3eae28ea02472df4),
        (0xbb6c80d173cac31c, 0x3ec47930aec78cb8),
        (0x3adf057ade22322a, 0xbe51ce32c4f34e51),
        (0x3afb45adc8973d4f, 0xbe5d0950e99c8b39),
        (0xba7a6617d572dc1b, 0x3de9c5a1603e79e8),
        (0xba8e2eff860f1411, 0x3dec0f1778cf9887),
        (0xba168f093dfd4cd6, 0xbd7938fe13097690),
        (0xba1ecefa9c9a3a82, 0xbd73a70bdaedabef),
        (0x39aed81642403d1e, 0x3d01ceeed28a9bbc),
        (0x3997da4ad363efc9, 0x3cf4dcf95c5ca949),
        (0xb91477913a3a55c0, 0xbc82ff92717657c8),
        (0x3910bab9b4893246, 0xbc715c5a3e02442f),
        (0x389d817a2ccfa8fd, 0x3bffb3c968d17246),
        (0xb87cd010a2041864, 0x3be740ae3a8d3fe3),
        (0xb7e3b9850dbb1fd0, 0xbb754054fa60c54d),
        (0x37fb925f8ad7d3fb, 0xbb599a3322fec174),
        (0x3784fd568b6ae4b0, 0x3ae763dc0015bdca),
        (0xb76b4c77e261251c, 0x3ac794a91384b8c1),
        (0x36e8d58063ff5bb1, 0xba5583a75fd37094),
        (0xb69a2c623fdcd0ef, 0xba3252e98814e1a2),
        (0x3663e86d6d6cf480, 0x39c0feb0ea2eacb6),
    ],
    [
        (0xb5e1d5aab91b0735, 0xb94d62f7652911ef),
        (0xbbb1d709b3e6a9e9, 0xbfb8ffc9b9a131f6),
        (0x3bdb4dd57b364903, 0x3f47f724314bb99d),
        (0xbc3086cbc396cba9, 0x3f90a89c5d1074ba),
        (0x3bb8da379d94d4d2, 0xbf1feeaeb33465e8),
        (0xbbef433944c1a759, 0xbf4a9fba36a7a08e),
        (0xbb7f256bdb98dce6, 0x3ed980034770f3b2),
        (0x3b91b4c8faeaae0e, 0x3ef43fa4bb8cba57),
        (0xbad948481030ddf3, 0xbe835f40f3a3902b),
        (0x3b3e2e9d940a676f, 0xbe91f465ba292c0d),
        (0xba98e1102f6772d7, 0x3e212688f74f8106),
        (0x3ac17030071395d4, 0x3e24d3f7e3fdde95),
        (0xba567e74d2a2034d, 0xbdb3daee9c38c74a),
        (0x3a51aeaf0688f02c, 0xbdb106a132ee69a7),
        (0x39ebe0f4bca30d89, 0x3d4030e7765d6c75),
        (0x39c14fc8399d63ea, 0x3d34aa4c24070ac4),
        (0x3933883151a4972a, 0xbcc3987402712b5f),
        (0xb942dee4f35f615b, 0xbcb35a87994f9f22),
        (0x38d71bf8bdfcbf64, 0x3c424aa138f72437),
        (0xb8ba56b686d5a4be, 0x3c2cd07eb7a4b5c9),
        (0xb8558c1bfdd348e3, 0xbbbb21f2d5bbbf78),
        (0xb83550a5585a2688, 0xbba174fb8120d7fa),
        (0x37b47b0dfc8fc083, 0x3b305ef4c6de1408),
        (0xb7bb881b76be83a0, 0x3b118c57455ac2ea),
        (0xb743465c91dd6734, 0xbaa061f86f5a1fc0),
        (0x3708e74190a8b10c, 0xba7dbce43544e4d4),
        (0xb68a21eaceaae3f4, 0x3a0b7321833d28bb),
        (0x36809484f141d2c7, 0x39e569a3e2d05d37),
    ],
    [
        (0x3c5827414357db53, 0xbfb8b5ccb03d459b),
        (0x35f22b8b60cf7290, 0xb96f103383ea761f),
        (0xbc4827414357db52, 0x3fa8b5ccb03d459b),
        (0x3bc829bf4eaf9cff, 0xbf2edc98b92fedf3),
        (0x3c0a89c9aabf283f, 0xbf70767d524cea88),
        (0xbb9b6aea43cf84ab, 0x3ef8a85a847c89c3),
        (0x3b9126e1254c1bf9, 0x3f2189afaa2bffcc),
        (0xbb239c5c72e2b4cb, 0xbeac1d231a69bb02),
        (0x3b6e89e6ffd35f3d, 0xbec40139bdbc9d36),
        (0xbac6599803b1efd4, 0x3e509a6859c8e374),
        (0x3aded4460b047b6f, 0x3e5c60fc5f9b505f),
        (0xba80f4ba4649d3f0, 0xbde80ad391f7224c),
        (0xba8175d1cbd50fa3, 0xbdeb6e954dd6c922),
        (0xba13a4fa3d4d4a86, 0x3d778adc76ca1d1b),
        (0x39e0b384d483dfc2, 0x3d733867c824f37c),
        (0xb993f2721a70fbeb, 0xbd00a1ec5f4dca90),
        (0xb99e890379206b1f, 0xbcf469ac872b68fc),
        (0xb9207c6b7625a1c3, 0x3c81c1b6937f11f5),
        (0xb901775a0f7e0f63, 0x3c70fe687033cdc5),
        (0xb89ff8b30b7fba36, 0xbbfda774598edc97),
        (0x3875c8039f93b5dd, 0xbbe6c5c790b36d34),
        (0x380ece04cbe1e4a9, 0x3b73e54f8cfb2b46),
        (0xb7d83bd27bce64ed, 0x3b59165852609f70),
        (0x377e5472e34f528f, 0xbae5eb353d0d715e),
        (0x376ef3c901224023, 0xbac71e9aa439eb8d),
        (0x36e5bd48680a81b7, 0x3a542e64e00789ab),
        (0xb6d03ab3c0296c8d, 0x3a31f9ea0deead8a),
        (0x364a1b07722309cf, 0xb9bfea22321266c2),
    ],
    [
        (0x35ef906f4164dc60, 0xb96927bfa8db52b2),
        (0x3c116404635b617f, 0x3fb86e51bb2ee24d),
        (0x3be40f7dbe05ef72, 0xbf465e3bad214eb8),
        (0xbc3502bddec37519, 0xbf9047d6ed159c31),
        (0xbbb68a41509da16e, 0x3f1dce4a381ce24e),
        (0x3be0a2f7a61834bd, 0x3f4a05b93842da9a),
        (0x3b384a0ade4543c9, 0xbed7ce3df60b0ea1),
        (0x3b8858267c36b09d, 0xbef3cb50138cce4b),
        (0x3b2cb60fbd506539, 0x3e8216e84a5dec74),
        (0x3b2d7337b289b637, 0x3e918e354c01533b),
        (0xbac53c3102b1edb8, 0xbe2005493039e5f0),
        (0x3ac1ef2cc0ca439b, 0xbe245ed4a3733e9d),
        (0xba3577e316826199, 0x3db28e2ed388f175),
        (0xba51361966067e0a, 0x3db0a83fb8aae5cc),
        (0x39d23d4aec4b7be8, 0xbd3e4745b1e410b2),
        (0xb9cc2b2efa2e5867, 0xbd3439aae872dae3),
        (0xb9550093ed38173e, 0x3cc2557a9d9a519f),
        (0xb95c225721c7ed22, 0x3cb2f313647cb524),
        (0xb8d9dd312bb7ea5b, 0xbc4120389f39a6c8),
        (0xb8cbc43aa5cd2092, 0xbc2c39cb315eea3a),
        (0xb84185a7cc406293, 0x3bb96c5a7748a20c),
        (0x38496ead43ea8719, 0x3ba11bdcff0ab3e8),
        (0xb7984affa3f0408a, 0xbb2eb47bf3d0d392),
        (0x37b23b859b45c934, 0xbb11351924a215f1),
        (0xb721ee8c0d2d3f36, 0x3a9ec1444b23676a),
        (0x371b235f8e2c7f49, 0x3a7d2d3b6879bb57),
        (0x36aa24c57ef1bc5e, 0xba09ca3fc26a353f),
        (0xb68eda4b8fd67c50, 0xb9e504fbf8d34ffc),
    ],
    [
        (0xbc5aa58d824fcbf0, 0x3fb829356c2fb67c),
        (0xb5fd77431283aa4a, 0x3960ecbe354c0a2c),
        (0x3c4aa58d824fcbf0, 0xbfa829356c2fb67c),
        (0x3bc03ee1c336a8c1, 0x3f2cd964e5e4caa0),
        (0x3c0adae94b604803, 0x3f70190ce66b97c3),
        (0xbb6093ba87cf1908, 0xbef70d5fd8bad7b5),
        (0xbbae25e1759c4268, 0xbf2126a150469b05),
        (0x3b4f93e54fe01c25, 0x3eaa49ea85815004),
        (0xbb4a798f7842e742, 0x3ec3910f5c159f54),
        (0x3ab455b6ae464798, 0xbe4f0f4bdbe20c90),
        (0xbaf74c179ba876ae, 0xbe5bc3697829b270),
        (0x3a88354a18f986f7, 0x3de67f2959cd47c2),
        (0x3a6dc6f645f5de18, 0x3dead81c085abf1e),
        (0xb9f0a5000d22a9e2, 0xbd760a10ab0898ac),
        (0x3a0d6591c74acc66, 0xbd72d07fd9984920),
        (0xb99f00218c9ca6f1, 0x3cff2887ab34ba4c),
        (0xb988be1c4afb3efb, 0x3cf3fd2c46f43bf9),
        (0xb8eadfedf70fa7fe, 0xbc80a48214b2d829),
        (0x3909f66df0bcce8b, 0xbc70a5cd24e7ae0a),
        (0xb899f61798b66d98, 0x3bfbd0274fe475d9),
        (0xb86046afc74d497f, 0x3be6519199343e6f),
        (0xb812000c63615941, 0xbb72acc84cb991fe),
        (0xb7e378e992bff237, 0xbb589952fc8a6a0a),
        (0x378b39a8bc2ecbfa, 0x3ae49748fe385729),
        (0xb76b443078ea68f5, 0x3ac6ae557e486745),
        (0x36f8b6a88bc9e19b, 0xba52f9ba63bb636e),
        (0xb6a8a49f117ff2cd, 0xba31a503187756c4),
        (0x36578837435086f9, 0x39be08f63a02804b),
    ],
    [
        (0xb5f43effe24b00dd, 0x3964e5a96d3cc648),
        (0xbc4ea89f3a432081, 0xbfb7e656ed57a0d1),
        (0x3bed3586339a9eb0, 0x3f44f0cfc62eb61c),
        (0x3c1736dd17549bf3, 0x3f8fdaba63e9c655),
        (0x3bb8a2f82c46dec6, 0xbf1be7bad1c50547),
        (0x3be9f603d64ca817, 0xbf4975ac0770c98d),
        (0xbb716dcdea217fca, 0x3ed64a6af83ba860),
        (0x3b95937b8720b7ef, 0x3ef35e6b41b6d77b),
        (0x3b1fd0d91a2464a1, 0xbe80f12d050df219),
        (0x3b0b801799b9f100, 0xbe912e744222ec5c),
        (0x3a882bc4d34bf5de, 0x3e1e04a8c0a89be6),
        (0x3ab7f6d69f6f7175, 0x3e23f0ed5e6a0d3b),
        (0x3a40221dd027ff69, 0xbdb163fd7157593b),
        (0x3a410e6fc017b151, 0xbdb04f8efe519654),
        (0xb9d61b861892425f, 0x3d3c6417f7a11d41),
        (0xb9df60b877c4e61e, 0x3d33cfa2affaf9e4),
        (0xb93fca12e22cd45c, 0xbcc1334e857adc92),
        (0xb940fcaa7e7a1f84, 0xbcb2917ae18b1995),
        (0x38d075a83e7685a6, 0x3c4013b4389ffbb8),
        (0x38b3ec31956d766e, 0x3c2bab4c1141e9b7),
        (0xb82612cb7e109475, 0xbbb7e1ea817e1f13),
        (0x3842297934775f91, 0xbba0c761c6922ab8),
        (0xb7a89c07cafc73fb, 0x3b2cdd9a9bd10cbd),
        (0xb7b8b565eb366f1b, 0x3b10e22afb97e007),
        (0xb73a476233fdd42f, 0xba9cef8ddac69711),
        (0xb716e797381cb799, 0xba7ca4454618f908),
        (0xb6a5522943296628, 0x3a0848f98ce0dc8b),
        (0x365472b048929931, 0x39e4a4c39def3d0d),
    ],
    [
        (0xbc52d2ff041ff2f6, 0xbfb7a597eb76a5e3),
        (0xb5d66723cbc9de74, 0x3962eb71e7008809),
        (0x3c42d2ff041ff2f6, 0x3fa7a597eb76a5e3),
        (0x3bb9dc564fefd757, 0xbf2b0bd9eb615315),
        (0xbc03cbe9e9bac1aa, 0xbf6f831b9629acd2),
        (0xbb95cc1ecba38581, 0x3ef59d1c40c95ba7),
        (0x3bad7b9ad1a3693d, 0x3f20c9d35d4be191),
        (0xbb39533f4678ca4e, 0xbea8a711bec6920b),
        (0xbb4180f2a560b483, 0xbec327e42f39066e),
        (0x3adb23dbc7dc1492, 0x3e4d225aa8ded6cd),
        (0xbaea1e7c5ffe5f6a, 0x3e5b2f83a946aed5),
        (0x3a7207083f443397, 0xbde51bef06eb5d9f),
        (0xba7f450be91eb6e9, 0xbdea4ab19859cce7),
        (0xba0907e63b078849, 0x3d74b03ce339d83a),
        (0x3a00617d77666243, 0x3d726eb2bbaa63c7),
        (0x3998200aa70ed135, 0xbcfd433d6a1a8fc4),
        (0xb95c36471111f47b, 0xbcf396dd9428b7d7),
        (0x391cfade411a585e, 0x3c7f470853f5d974),
        (0xb9104560036c71a2, 0x3c705215ddbe6f8c),
        (0xb89e9dfcc8c7b005, 0xbbfa26d0528541cb),
        (0x3877907f8f994b48, 0xbbe5e38754f217db),
        (0xb7a00f2e50ac7581, 0x3b7192400028c4b5),
        (0x37d0b5957a7cceb4, 0x3b5822a7a7e2efa6),
        (0x378c09ba2822432c, 0xbae3636f4f21f020),
        (0x3762ceed74e3669d, 0xbac6437cc6062eef),
        (0xb6e668ce43677d5d, 0x3a51e56360c22f9c),
        (0x36a1ef54ef26b01a, 0x3a3153d505a39c09),
        (0x36443c33037016fa, 0xb9bfaca1a975c527),
    ],
];
