/*
 * nUFRaw - new Unidentified Flying Raw converter for digital camera images
 *
 * nufraw.h - Common definitions for nUFRaw.
 * Copyright 2004-2016 by Udi Fuchs
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 */

#ifndef _NUFRAW_H
#define _NUFRAW_H

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include "uf_glib.h"
#include "ufobject.h"

#include "nikon_curve.h"
#include "uf_progress.h"

#ifndef HAVE_STRCASECMP
#define strcasecmp stricmp
#endif

#ifdef _WIN32
#define strcasecmp(a,b) _stricmp(a,b)
#define strncasecmp(a,b,c) _strnicmp(a,b,c)
#endif

// clamp a number between two values
#ifndef LIM
#define LIM(x,min,max) MAX((min),MIN((x),(max)))
#endif

// paths
#define UF_RC_FILE      ".nufrawrc"
#define UF_UI_RC_FILE   ".nufrawrc-ui"
#define UF_GTK_RC_FILE  ".nufrawrc-gtk"

// constant limits
#define UF_MAX_CURVES                 20
#define UF_CURVE_MAX_ANCHORS          20
#define UF_MAX_PROFILES               20
#define UF_MAX_PATH                   1024
#define UF_MAX_NAME                   256
#define UF_MAX_LIGHTNESS_ADJUSTMENTS  3
#define UF_MAX_EXPANDERS              32

/* An impossible value for conf float values */
#define NULLF -10000.0

#ifdef __cplusplus
extern "C" {
#endif // __cplusplus

/* Options, like auto-adjust buttons can be in 3 states. Enabled and disabled
 * are obvious. Apply means that the option was selected and some function
 * has to act accourdingly, before changing to one of the first two states */
enum {
    disabled_state, enabled_state, apply_state
};

extern const char uf_spot_wb[];
extern const char uf_manual_wb[];
extern const char uf_camera_wb[];
extern const char uf_auto_wb[];

/*
 * UFObject Definitions for nufraw_settings.cc
 */

extern UFName ufWB;
extern UFName ufPreset;
extern UFName ufWBFineTuning;
extern UFName ufTemperature;
extern UFName ufGreen;
extern UFName ufChannelMultipliers;
extern UFName ufLensfunAuto;
extern UFName ufLensfun;
extern UFName ufCameraModel;
extern UFName ufLensModel;
extern UFName ufFocalLength;
extern UFName ufAperture;
extern UFName ufDistance;
extern UFName ufTCA;
extern UFName ufVignetting;
extern UFName ufDistortion;
extern UFName ufModel;
extern UFName ufLensGeometry;
extern UFName ufTargetLensGeometry;
extern UFName ufRawImage;
extern UFName ufRawResources;
extern UFName ufCommandLine;

UFObject *nufraw_image_new();

#ifdef HAVE_LENSFUN
UFObject *nufraw_lensfun_new();
void nufraw_lensfun_init(UFObject *lensfun, UFBoolean reset);
struct lfDatabase *nufraw_lensfun_db(); /* mount/camera/lens database */
const struct lfCamera *nufraw_lensfun_camera(const UFObject *lensfun);
void nufraw_lensfun_set_camera(UFObject *lensfun, const struct lfCamera *camera);
const struct lfLens *nufraw_lensfun_interpolation_lens(const UFObject *lensfun);
void nufraw_lensfun_set_lens(UFObject *lensfun, const struct lfLens *lens);
#endif /* HAVE_LENSFUN */

struct nufraw_struct *nufraw_image_get_data(UFObject *obj);
void nufraw_image_set_data(UFObject *obj, struct nufraw_struct *uf);
UFObject *nufraw_resources_new();
UFObject *nufraw_command_line_new();

#ifdef __cplusplus
} // extern "C"
#endif // __cplusplus

enum { 
	rgb_histogram, r_g_b_histogram, luminosity_histogram, value_histogram, saturation_histogram
};

enum { linear_histogram, log_histogram };

/* The following enum should match the dcraw_interpolation enum
 * in dcraw_api.h. */
enum { 
	ahd_interpolation, vng_interpolation, four_color_interpolation,
	ppg_interpolation, bilinear_interpolation, xtrans_interpolation,
	none_interpolation, half_interpolation, obsolete_eahd_interpolation,
	num_interpolations
};
enum { no_id, also_id, only_id, send_id };
enum { manual_curve, linear_curve, custom_curve, camera_curve };
enum { in_profile, out_profile, display_profile, profile_types};
enum { 
	ppm_type, ppm16_deprecated_type, tiff_type, tiff16_deprecated_type,
	jpeg_type, png_type, png16_deprecated_type,
	embedded_jpeg_type, embedded_png_type, fits_type, num_types
};
enum { clip_details, restore_lch_details, restore_hsv_details, restore_types };
enum { digital_highlights, film_highlights, highlights_types };

// window mode
enum { nufraw_standalone, nufraw_gimp_plugin };

typedef enum { display_developer, file_developer, auto_developer } DeveloperMode;

typedef enum { 
	perceptual_intent, relative_intent, saturation_intent,
	absolute_intent, disable_intent
} Intent;

typedef enum { 
	nufraw_raw_phase, 
	nufraw_first_phase,
	nufraw_transform_phase,
	nufraw_develop_phase, 
	nufraw_display_phase, 
	nufraw_phases_num
} nUFRawPhase;

typedef enum { 
	grayscale_none, 
	grayscale_lightness,
	grayscale_luminance, 
	grayscale_value,
	grayscale_mixer, 
	grayscale_invalid
} GrayscaleMode;

typedef struct {
	const char *make;
	const char *model;
	const char *name;
	int tuning;
	double channel[4];
} wb_data;

typedef struct {
	double adjustment;
	double hue;
	double hueWidth;
} lightness_adjustment;

typedef struct {
	DeveloperMode mode;
	unsigned rgbMax, max, exposure, colors, useMatrix;
	int restoreDetails, clipHighlights;
	int rgbWB[4], colorMatrix[3][4];
	double gamma, linear;
	char profileFile[profile_types][UF_MAX_PATH];
	void *profile[profile_types];
	Intent intent[profile_types];
	gboolean updateTransform;
	void *colorTransform;
	void *working2displayTransform;
	void *rgbtolabTransform;
	double saturation;
#ifdef NUFRAW_CONTRAST
	double contrast;
#endif
	CurveData baseCurveData, luminosityCurveData;
	guint16 gammaCurve[0x10000];
	void *luminosityProfile;
	void *TransferFunction[3];
	void *saturationProfile;
	void *adjustmentProfile;
	GrayscaleMode grayscaleMode;
	double grayscaleMixer[3];
	lightness_adjustment lightnessAdjustment[UF_MAX_LIGHTNESS_ADJUSTMENTS];
} developer_data;

typedef guint16 nufraw_image_type[4];

typedef struct {
	char name[UF_MAX_NAME];
	char file[UF_MAX_PATH];
	char productName[UF_MAX_NAME];
	double gamma, linear;
	int BitDepth;
} profile_data;

typedef struct {
	gint x;
	gint y;
	gint width;
	gint height;
} UFRectangle;

typedef enum { 
	spot_cursor, crop_cursor, left_cursor, right_cursor, top_cursor, 
	bottom_cursor, top_left_cursor, top_right_cursor, 
	bottom_left_cursor, bottom_right_cursor, move_cursor, cursor_num
} CursorType;

// ui_conf holds the ui configuration
typedef struct {
   
	int expander_open[UF_MAX_EXPANDERS];
	int cursor_mode;

	double Zoom;

	int liveHistogramMode;
	int liveHistogramScale;
	int rawHistogramScale;

	int gridCount;

	gboolean showOverExp;
	gboolean showUnderExp;
	gboolean blinkOverUnder;

	// startup path for the file chooser dialog
	char inputPath[UF_MAX_PATH];

	// window size
	int win_width;
	int win_height;
	
	// size of vertical paned
	int left_size;
	
	// size of horizontal paned
	int up_size;
	
	char remoteGimpCommand[UF_MAX_PATH];
	gboolean lockAspect;
	gboolean rememberOutputPath;
	
	// startup path for the curve chooser dialog
	char curvePath[UF_MAX_PATH];
	
	// startup path for the profile chooser dialog
	char profilePath[UF_MAX_PATH];

	// save path
	char outputPath[UF_MAX_PATH];
	
	// show/hide preview in file open dialog
	gboolean showPreview;
	
	// show/hide hidden in file open dialog
	gboolean showHidden;
    
} ui_conf;

/* conf_data holds the configuration data of nUFRaw.
 * The data can be split into two groups: IMAGE manipulation ad SAVE options
 * The sources for this information are:
 * DEF: nUFRaw's defaults from conf_defaults.
 * RC: users defaults from ~/.nufrawrc. These options are set from the last
 *     interactive session.
 *     If saveConfiguration==disabled_state, IMAGE options are not saved.
 * ID: nUFRaw ID files used on their original image.
 * CONF: same ID files used as configuration for other raw images.
 * CMD: command line options.
 * UI: interactive input.
 * The options are set in the above order, therefore the last sources will
 * override the first ones with some subtelties:
 * * ID|CONF contains only data which is different from DEF, still it is
 *   assumed that IMAGE and SAVE options are included. Therefore missing
 *   options are set to DEF overwriting RC.
 * * if both CONF and ID are specified, only in/out filenames are taken from ID.
 * * in batch mode SAVE options from RC are ignored.
 * Some fields need special treatment:
 * RC|CONF: auto[Exposure|Black]==enable_state it is switched to apply_state.
 * RC|CONF: if !spot_wb reset chanMul[] to -1.0.
 * CONF|ID: curve/profile are added to the list from RC.
 * CONF: inputFilename, outputFilename are ignored.
 * outputPath can only be specified in CMD or guessed in interactive mode.
 * ID: createID==only_id is switched to no_id in case of nufraw-batch.
 * ID: chanMul[] override wb, green, temperature.
 */
typedef struct {

	int version;

	UFObject *ufobject;  // Eventually ufobject should replace conf_data.

	double wdThreshold;
	double hotpixel;
	
#ifdef NUFRAW_CONTRAST
	double contrast;
#endif
	double exposure;
	double saturation;
	double black; // black is only used in CMD
	
	int ExposureNorm;
	int restoreDetails;
	int clipHighlights;
	
	int autoExposure, autoBlack, fullCrop, autoCrop;
	
	// curves and profiles are used both for configuration and for adjustments
	
	int BaseCurveIndex;
	int BaseCurveCount;
	CurveData BaseCurve[UF_MAX_CURVES];
	
	int curveIndex; 
	int curveCount;
	CurveData curve[UF_MAX_CURVES];
	
	// TODO: display profile should go in ui_conf
	int profileIndex[profile_types];
	int profileCount[profile_types];
	profile_data profile[profile_types][UF_MAX_PROFILES];	
	Intent intent[profile_types];
	
	int interpolation;
	int smoothing;
	
	char darkframeFile[UF_MAX_PATH];
	struct nufraw_struct *darkframe;
	
	int CropX1;
	int CropY1;
	int CropX2;
	int CropY2;
	
	double aspectRatio;
	int orientation;
	double rotationAngle;
	
	// selective lightness
	int lightnessAdjustmentCount;
	lightness_adjustment lightnessAdjustment[UF_MAX_LIGHTNESS_ADJUSTMENTS];
	
	int grayscaleMode;
	double grayscaleMixer[3];
	int grayscaleMixerDefined;
	
	double despeckleWindow[4];
	double despeckleDecay[4];
	double despecklePasses[4];

	char inputFilename[UF_MAX_PATH];
	char outputFilename[UF_MAX_PATH];
	char outputPath[UF_MAX_PATH];
	char inputURI[UF_MAX_PATH];
	char inputModTime[UF_MAX_NAME];
	
	int type;
	int compression; 
	int createID;
	int embedExif;
	int progressiveJPEG;
	
	int shrink;     // shrink is used to resize the preview too
	int size;       // maximum size between width and height
	
	// TODO: overwrite should go in ui_conf but is used by batch mode too
	gboolean overwrite;
	gboolean losslessCompress;
	gboolean embeddedImage;
	gboolean rotate;
	
	// batch mode only
	gboolean silent;

	// EXIF data
	int CameraOrientation;
	float iso_speed;
	float shutter; 
	float aperture;
	float focal_len;
	float subject_distance;
	char exifSource[UF_MAX_NAME];
	char isoText[UF_MAX_NAME];
	char shutterText[UF_MAX_NAME];
	char apertureText[UF_MAX_NAME]; 
	char focalLenText[UF_MAX_NAME];
	char focalLen35Text[UF_MAX_NAME]; 
	char lensText[UF_MAX_NAME];
	char flashText[UF_MAX_NAME]; 
	char whiteBalanceText[UF_MAX_NAME];
	char timestampText[UF_MAX_NAME];
	char make[UF_MAX_NAME];
	char model[UF_MAX_NAME];
	time_t timestamp;
	
	// Unfortunately dcraw strips make and model, but we need originals too
	char real_make[UF_MAX_NAME], real_model[UF_MAX_NAME];
	
} conf_data;

typedef struct {
	guint8 *buffer;
	int height, width, depth, rowstride;
	/* This bit field marks valid pieces of the image with 1's.
	 The variable contains a fixed 4x8 matrix of bits, every bit containing
	 the validity of the respective subarea of the whole image. The subarea
	 sizes are determined by dividing the width by 4 and height by 8.
	 This field must always contain at least 32 bits. */
	guint32 valid;
	gboolean rgbg;
	gboolean invalidate_event;
} nufraw_image_data;

typedef struct nufraw_struct {
	int status;
	char *message;
	char filename[UF_MAX_PATH];
	int initialHeight, initialWidth, rgbMax, colors, raw_color, useMatrix;
	int rotatedHeight, rotatedWidth;
	int autoCropHeight, autoCropWidth;
	gboolean LoadingID; /* Indication that we are loading an ID file */
	gboolean WBDirty;
	float rgb_cam[3][4];
	nufraw_image_data Images[nufraw_phases_num];
	nufraw_image_data thumb;
	void *raw;
	gboolean HaveFilters;
	gboolean IsXTrans;
	void *unzippedBuf;
	gsize unzippedBufLen;
	developer_data *developer;
	developer_data *AutoDeveloper;
	guint8 *displayProfile;
	gint displayProfileSize;
	conf_data *conf;
	guchar *inputExifBuf;
	guint inputExifBufLen;
	guchar *outputExifBuf;
	guint outputExifBufLen;
	int gimpImage;
	int *RawHistogram;
	int RawChanMul[4];
	int RawCount;
#ifdef HAVE_LENSFUN
	int modFlags; /* postprocessing operations (LF_MODIFY_XXX) */
	struct lfModifier *TCAmodifier;
	struct lfModifier *modifier;
#endif /* HAVE_LENSFUN */
	int hotpixels;
	gboolean mark_hotpixels;
	unsigned raw_multiplier;
	gboolean wb_presets_make_model_match;
} nufraw_data;

extern const conf_data conf_default;
extern const wb_data wb_preset[];
extern const int wb_preset_count;
extern const char raw_ext[];
extern const char *file_type[];

/* nufraw_binary contains the name of the binary file for error messages.
 * It should be set in every nUFRaw main() */
extern char *nufraw_binary;

#ifdef __cplusplus
extern "C" {
#endif

// prototypes for functions in nufraw_interface.c *****************************

// open file and load data (not the image) from ID file (.nufraw), raw images or compressed raw
nufraw_data *nufraw_open(char *filename);

// merge rc, conf and command in uf->conf
// rc: user default from UF_RC_FILE
// conf: data from ID file
// cmd: data from commnad line options
int nufraw_config(nufraw_data *uf, conf_data *rc, conf_data *conf, conf_data *cmd);

// if uf->conf->embeddedImage load thumbnail from raw file
// else load raw mage
int nufraw_load_raw(nufraw_data *uf);

// load uf->conf->darkframeFile (if empty do nothing)
int nufraw_load_darkframe(nufraw_data *uf);

// prepare the developer for the required target (file or display)
int nufraw_developer_prepare(nufraw_data *uf, DeveloperMode mode);

int nufraw_convert_image(nufraw_data *uf);
nufraw_image_data *nufraw_get_image(nufraw_data *uf, nUFRawPhase phase, gboolean bufferok);
nufraw_image_data *nufraw_convert_image_area(nufraw_data *uf, unsigned saidx,nUFRawPhase phase);

// free darkframw 
void nufraw_close_darkframe(conf_data *uf);

// free uf data
void nufraw_close(nufraw_data *uf);

void nufraw_flip_orientation(nufraw_data *uf, int flip);
void nufraw_flip_image(nufraw_data *uf, int flip);

void nufraw_invalidate_layer(nufraw_data *uf, nUFRawPhase phase);
void nufraw_invalidate_tca_layer(nufraw_data *uf);
void nufraw_invalidate_hotpixel_layer(nufraw_data *uf);
void nufraw_invalidate_denoise_layer(nufraw_data *uf);
void nufraw_invalidate_darkframe_layer(nufraw_data *uf);
void nufraw_invalidate_despeckle_layer(nufraw_data *uf);
void nufraw_invalidate_whitebalance_layer(nufraw_data *uf);
void nufraw_invalidate_smoothing_layer(nufraw_data *uf);

int nufraw_set_wb(nufraw_data *uf);
void nufraw_auto_expose(nufraw_data *uf);
void nufraw_auto_black(nufraw_data *uf);
void nufraw_auto_curve(nufraw_data *uf);

void nufraw_normalize_rotation(nufraw_data *uf);
void nufraw_unnormalize_rotation(nufraw_data *uf);
void nufraw_get_image_dimensions(nufraw_data *uf);
// Get scaled crop coordinates in final image coordinates
void nufraw_get_scaled_crop(nufraw_data *uf, UFRectangle *crop);

UFRectangle nufraw_image_get_subarea_rectangle(nufraw_image_data *img, unsigned saidx);
unsigned nufraw_img_get_subarea_idx(nufraw_image_data *img, int x, int y);

// prototypes for functions in nufraw_message.c *********************************

char *nufraw_get_message(nufraw_data *uf);
/* The following functions should only be used internally */
void nufraw_message_init(nufraw_data *uf);
void nufraw_message_reset(nufraw_data *uf);
void nufraw_set_error(nufraw_data *uf, const char *format, ...);
void nufraw_set_warning(nufraw_data *uf, const char *format, ...);
void nufraw_set_info(nufraw_data *uf, const char *format, ...);
int nufraw_get_status(nufraw_data *uf);
int nufraw_is_error(nufraw_data *uf);
// Old error handling, should be removed after being fully implemented.
char *nufraw_message(int code, const char *format, ...);
void nufraw_batch_messenger(char *message);

// prototypes for functions in nufraw_ui.c **************************************

int nufraw_ui(nufraw_data *uf, ui_conf *pconf, int plugin, long(*save_func)());
void nufraw_focus(void *window, gboolean focus);
void nufraw_messenger(char *message, void *parentWindow);

// prototypes for functions in nufraw_routines.c ********************************

const char *uf_get_home_dir();
void uf_init_locale(const char *exename);
char *uf_file_set_type(const char *filename, const char *type);
char *uf_file_set_absolute(const char *filename);
/* Set locale of LC_NUMERIC to "C" to make sure that printf behaves correctly.*/
char *uf_set_locale_C();
void uf_reset_locale(char *locale);
char *uf_markup_buf(char *buffer, const char *format, ...);
double profile_default_linear(profile_data *p);
double profile_default_gamma(profile_data *p);
void Temperature_to_RGB(double T, double RGB[3]);
void RGB_to_Temperature(double RGB[3], double *T, double *Green);
int curve_load(CurveData *cp, char *filename);
int curve_save(CurveData *cp, char *filename);
char *curve_buffer(CurveData *cp);
/* Useful functions for handling the underappreciated Glib ptr arrays */
int ptr_array_insert_sorted(GPtrArray *array, const void *item, GCompareFunc compare);
int ptr_array_find_sorted(const GPtrArray *array, const void *item, GCompareFunc compare);
void ptr_array_insert_index(GPtrArray *array, const void *item, int index);

// prototypes for functions in nufraw_conf.c ************************************

// load/save ui configuration from UF_UI_RC_FILE
int ui_conf_load(ui_conf *c);
int ui_conf_save(ui_conf *c);

// load conf_data from IDFilename or .nufrawrc
// if IDFilename==NULL load conf from .nufrawrc ( without .nufrawrc load defaults)
// if IDFilename!=NULL load conf from IDFilename
int conf_load(conf_data *c, const char *confFilename);

// call conf_load and reset paths (to apply conf on another raw)
void conf_file_load(conf_data *conf, char *confFilename);

// save conf_data as XML on confbuffer or IDFilename or .nufrawrc
// if confBuffer!=NULL return configuration on *confbuffer
// if confbuffer==NULL && IDFilename!=NULL save configuration on IDFilename
// if confbuffer==NULL && IDFilename==NULL save configuration on .nufrawrc
int conf_save(conf_data *c, char *confFilename, char **confBuffer);

// copy default config to given instance and initialize non-const fields
void conf_init(conf_data *c);
// Copy the image manipulation options from *src to *dst (included curves and profiles)
void conf_copy_image(conf_data *dst, const conf_data *src);
// Copy the transformation options from *src to *dst
void conf_copy_transform(conf_data *dst, const conf_data *src);
// Copy the 'save options' from *src to *dst
void conf_copy_save(conf_data *dst, const conf_data *src);
// set conf_data from command line options
int conf_set_cmd(conf_data *conf, const conf_data *cmd);
int nufraw_process_args(int *argc, char ***argv, conf_data *cmd, conf_data *rc);

// prototype for functions in nufraw_developer.c ********************************

// Convert linear RGB to CIE-LCh
void uf_rgb_to_cielch(gint64 rgb[3], float lch[3]);
// Convert CIE-LCh to linear RGB
void uf_cielch_to_rgb(float lch[3], gint64 rgb[3]);
void uf_raw_to_cielch(const developer_data *d,
                      const guint16 raw[4], float lch[3]);
developer_data *developer_init();
void developer_destroy(developer_data *d);
void developer_profile(developer_data *d, int type, profile_data *p);
void developer_display_profile(developer_data *d,
                               unsigned char *profile, int size, char productName[]);
int developer_prepare(developer_data *d, conf_data *conf,
                       int rgbMax, float rgb_cam[3][4], int colors, int useMatrix,
                       DeveloperMode mode);
void develop(void *po, guint16 pix[4], developer_data *d, int mode, int count);
void develop_display(void *pout, void *pin, developer_data *d, int count);
void develop_linear(guint16 in[4], guint16 out[3], developer_data *d);

// prototype for functions in nufraw_writer.c ***********************************

int nufraw_write_image(nufraw_data *uf);
void nufraw_write_image_data(
    nufraw_data *uf, void * volatile out,
    const UFRectangle *Crop, int bitDepth, int grayscaleMode,
    int (*row_writer)(nufraw_data *, void * volatile, void *, int, int, int, int, int));

// prototype for functions in nufraw_embedded.c *********************************

int nufraw_read_embedded(nufraw_data *uf);
int nufraw_convert_embedded(nufraw_data *uf);
int nufraw_write_embedded(nufraw_data *uf);

// prototype for functions in nufraw_chooser.c **********************************

void nufraw_chooser(conf_data *conf, conf_data *rc, conf_data *cmd, ui_conf *pconf, const char *defPath);

// prototype for functions in nufraw_icons.c ************************************
void nufraw_icons_init();

// prototype for functions in nufraw_exiv2.cc ***********************************

int nufraw_exif_read_input(nufraw_data *uf);
int nufraw_exif_prepare_output(nufraw_data *uf);
int nufraw_exif_write(nufraw_data *uf);

#ifdef __cplusplus
} // extern "C"
#endif

/* status numbers from DCRaw and nUFRaw */
#define NUFRAW_SUCCESS 0
//#define NUFRAW_DCRAW_ERROR 1 /* General dcraw unrecoverable error */
//#define NUFRAW_DCRAW_UNSUPPORTED 2
//#define NUFRAW_DCRAW_NO_CAMERA_WB 3
//#define NUFRAW_DCRAW_VERBOSE 4
//#define NUFRAW_DCRAW_OPEN_ERROR 5
#define NUFRAW_DCRAW_SET_LOG 4 /* DCRAW_VERBOSE */
#define NUFRAW_ERROR 100
#define NUFRAW_CANCEL 101
#define NUFRAW_RC_VERSION 103 /* Mismatch in version from .nufrawrc */
#define NUFRAW_WARNING 104
#define NUFRAW_MESSAGE 105
#define NUFRAW_SET_ERROR 200
#define NUFRAW_SET_WARNING 201
#define NUFRAW_SET_LOG 202
#define NUFRAW_GET_ERROR 203 /* Return the warning buffer if an error occured */
#define NUFRAW_GET_WARNING 204 /* Return the warning buffer */
#define NUFRAW_GET_LOG 205 /* Return the log buffer */
#define NUFRAW_BATCH_MESSAGE 206
#define NUFRAW_INTERACTIVE_MESSAGE 207
#define NUFRAW_REPORT 208 /* Report previous messages */
#define NUFRAW_CLEAN 209 /* Clean all buffers */
#define NUFRAW_RESET 210 /* Reset warnings and errors */
#define NUFRAW_SET_PARENT 211 /* Set parent window for message dialog */

#endif /*_NUFRAW_H*/
