/*
 * nUFRaw - Unidentified Flying Raw converter for digital camera images
 *
 * nufraw_saver.c - The GUI file saver.
 * Copyright 2004-2016 by Udi Fuchs
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 */

#include "nufraw.h"
#include "uf_gtk.h"
#include <errno.h>
#include <string.h>
#include <glib/gi18n.h>

extern const ui_conf ui_conf_default;

static void set_fileext_from_typecombo(GtkComboBox *typeCombo, gpointer *user_data)
{
    GtkFileChooser *dialog = GTK_FILE_CHOOSER(user_data);
    char *f=gtk_file_chooser_get_filename(dialog);
    if (f==NULL) return;

    char* type = gtk_combo_box_get_active_text(typeCombo);
    char *file=g_path_get_basename(f);
    char *outfile = uf_file_set_type(file,type);
    gtk_file_chooser_set_current_name(dialog,outfile);

    g_free(f);
    g_free(file);
    g_free(type);
    g_free(outfile);
}

long nufraw_save_dialog(nufraw_data *uf, ui_conf *uiconf, void *widget)
{
	// build save window
	GtkFileChooser *dialog;
	GtkBox *hBox;
	GtkWidget* label, *button;

	dialog = GTK_FILE_CHOOSER(gtk_file_chooser_dialog_new(_("Save"),
		                                                  GTK_WINDOW(gtk_widget_get_toplevel(widget)),
		                                                  GTK_FILE_CHOOSER_ACTION_SAVE,
		                                                  _("_Cancel"),
		                                                  GTK_RESPONSE_CANCEL,
		                                                  _("_Save"),
		                                                  GTK_RESPONSE_ACCEPT,
		                                                  NULL));

	if (strlen(uiconf->outputPath)) gtk_file_chooser_set_current_folder(dialog,uiconf->outputPath);
	else gtk_file_chooser_set_filename(dialog,uf->conf->outputFilename);
	gtk_file_chooser_set_current_name(dialog,g_path_get_basename(uf->conf->outputFilename));

	GtkComboBox *typeCombo = GTK_COMBO_BOX(gtk_combo_box_new_text());
	int i = 0;
	gtk_combo_box_append_text(typeCombo, ".PPM");
	if (uf->conf->type == ppm_type) gtk_combo_box_set_active(typeCombo, i);
	i++;
#ifdef HAVE_LIBPNG
    gtk_combo_box_append_text(typeCombo, ".PNG");
    if (uf->conf->type == png_type) gtk_combo_box_set_active(typeCombo, i);
    i++;
#endif
#ifdef HAVE_LIBTIFF
    gtk_combo_box_append_text(typeCombo, ".TIF");
    if (uf->conf->type == tiff_type) gtk_combo_box_set_active(typeCombo, i);
    i++;
#endif
#ifdef HAVE_LIBJPEG
    gtk_combo_box_append_text(typeCombo, ".JPG");
    if (uf->conf->type == jpeg_type) gtk_combo_box_set_active(typeCombo, i);
    i++;
#endif
#ifdef HAVE_LIBCFITSIO
    gtk_combo_box_append_text(typeCombo, ".FITS");
    if (uf->conf->type == fits_type) gtk_combo_box_set_active(typeCombo, i);
    i++;
#endif
    g_signal_connect(G_OBJECT(typeCombo), "changed",G_CALLBACK(set_fileext_from_typecombo), dialog);
    gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dialog)->vbox),GTK_WIDGET(typeCombo), FALSE, FALSE, 0);

#ifdef HAVE_LIBJPEG
    // jpeg compression level
    hBox=GTK_BOX(gtk_hbox_new(FALSE,0));
    label = gtk_label_new(_("JPEG compression level"));
    gtk_box_pack_start(GTK_BOX(hBox), label, FALSE, FALSE, 0);
    GtkAdjustment *compressAdj =GTK_ADJUSTMENT(gtk_adjustment_new(uf->conf->compression,0, 100, 5, 10, 0));
    GtkWidget *scale = gtk_hscale_new(compressAdj);
    gtk_scale_set_draw_value(GTK_SCALE(scale), FALSE);
    gtk_box_pack_start(GTK_BOX(hBox), scale, TRUE, TRUE, 0);
    button = gtk_spin_button_new(compressAdj, 5, 0);
    g_object_set_data(G_OBJECT(compressAdj), "Parent-Widget", button);
    gtk_box_pack_start(GTK_BOX(hBox), button, FALSE, FALSE, 0);
    gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dialog)->vbox),GTK_WIDGET(hBox), FALSE, FALSE, 0);

    GtkWidget *jpg_button = gtk_check_button_new_with_label(_("JPEG progressive encoding"));
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(jpg_button),uf->conf->progressiveJPEG);
    gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dialog)->vbox),jpg_button, FALSE, FALSE, 0);
#endif // HAVE_LIBJPEG

#if defined(HAVE_LIBTIFF) && defined(HAVE_LIBZ)
    GtkWidget *tif_button = gtk_check_button_new_with_label(_("TIFF lossless Compress"));
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(tif_button),uf->conf->losslessCompress);
    gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dialog)->vbox), tif_button, FALSE, FALSE, 0);
#endif // HAVE_LIBTIFF && HAVE_LIBZ

#if defined(HAVE_EXIV2) && (defined(HAVE_LIBJPEG) || defined(HAVE_LIBPNG))
    GtkWidget *exif_button = gtk_check_button_new_with_label(_("Embed EXIF data in output"));
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(exif_button),uf->conf->embedExif);
    gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dialog)->vbox), exif_button, FALSE, FALSE, 0);
    gtk_widget_set_sensitive(exif_button, uf->inputExifBuf != NULL);
#endif // HAVE_EXIV2 && (HAVE_LIBJPEG || HAVE_LIBPNG)   // add tiff lossless check

    // id file
    hBox=GTK_BOX(gtk_hbox_new(FALSE,0));
    label = gtk_label_new(_("Create ID file "));
    gtk_box_pack_start(GTK_BOX(hBox),label, FALSE, FALSE, 0);
    GtkComboBox *idCombo = GTK_COMBO_BOX(gtk_combo_box_new_text());
    gtk_combo_box_append_text(idCombo, _("No"));
    gtk_combo_box_append_text(idCombo, _("Also"));
    gtk_combo_box_append_text(idCombo, _("Only"));
    uf_combo_box_set_data(idCombo, &uf->conf->createID);
    gtk_box_pack_start(GTK_BOX(hBox),GTK_WIDGET(idCombo), FALSE, FALSE, 0);
    gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dialog)->vbox),GTK_WIDGET(hBox), FALSE, FALSE, 0);

    // TODO: output icc profile, output intent, output bit depth

    // show save window
    gtk_widget_show_all(GTK_WIDGET(dialog));
    gint res = gtk_dialog_run(GTK_DIALOG(dialog));
    if (res==GTK_RESPONSE_ACCEPT){

        // add the right extension if not present
        set_fileext_from_typecombo(typeCombo, (void **)dialog);

        // retrieve dialog values
        char *f=gtk_file_chooser_get_filename(dialog);
        if (uiconf->rememberOutputPath) g_strlcpy(uiconf->outputPath, gtk_file_chooser_get_current_folder(dialog), UF_MAX_PATH);
        g_strlcpy(uf->conf->outputFilename, gtk_file_chooser_get_filename(dialog), UF_MAX_PATH);
        g_free(f);

        // set type
        char* type = gtk_combo_box_get_active_text(typeCombo);
        if (!strcasecmp(type,".PPM")) uf->conf->type=ppm_type;
        else if (!strcasecmp(type,".PNG")) uf->conf->type=png_type;
        else if (!strcasecmp(type,".TIF")) uf->conf->type=tiff_type;
        else if (!strcasecmp(type,".JPG")) uf->conf->type=jpeg_type;
        else if (!strcasecmp(type,".FITS")) uf->conf->type=fits_type;
        g_free(type);

#if defined(HAVE_EXIV2) && (defined(HAVE_LIBJPEG) || defined(HAVE_LIBPNG))
        uf->conf->embedExif=gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(exif_button));
 #endif
#if defined(HAVE_LIBTIFF) && defined(HAVE_LIBZ)
        uf->conf->losslessCompress=gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(tif_button));
#endif
#ifdef HAVE_LIBJPEG
        uf->conf->progressiveJPEG=gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(jpg_button));
        uf->conf->compression=gtk_adjustment_get_value(GTK_ADJUSTMENT(compressAdj));
#endif
    }

    gtk_widget_destroy(GTK_WIDGET(dialog));
    return res;
}

int nufraw_confirm_dialog(const char *message, const char* title, void *widget)
{
	int response;
	GtkWidget *dialog = gtk_dialog_new_with_buttons(title,
		                GTK_WINDOW(gtk_widget_get_toplevel(widget)),
		                GTK_DIALOG_DESTROY_WITH_PARENT | GTK_DIALOG_NO_SEPARATOR,
		                GTK_STOCK_NO, GTK_RESPONSE_NO,
		                GTK_STOCK_YES, GTK_RESPONSE_YES, NULL);
	GtkWidget *label = gtk_label_new(message);
	gtk_label_set_line_wrap(GTK_LABEL(label), TRUE);
	gtk_container_add(GTK_CONTAINER(GTK_DIALOG(dialog)->vbox), label);
	gtk_widget_show_all(dialog);
	response = gtk_dialog_run(GTK_DIALOG(dialog));
	gtk_widget_destroy(dialog);
	return response;
}

long nufraw_save_now(nufraw_data *uf, void *widget)
{
    char * volatile confFilename = uf_file_set_type(uf->conf->outputFilename, ".nufraw");

    if (!uf->conf->overwrite
         && uf->conf->createID != only_id
         && g_file_test(uf->conf->outputFilename, G_FILE_TEST_EXISTS)) {

        char message[UF_MAX_PATH];
        char *utf8 = g_filename_to_utf8(uf->conf->outputFilename, -1, NULL, NULL, NULL);
        if (utf8 == NULL) utf8 = g_strdup("Unknown file name");
        snprintf(message, UF_MAX_PATH,_("File '%s' already exists.\nOverwrite?"), utf8);
        g_free(utf8);
        g_free(confFilename);

        if (nufraw_confirm_dialog(message, _("File exists"), widget)==GTK_RESPONSE_NO)
            return NUFRAW_ERROR;

    }else if (!uf->conf->overwrite
              && uf->conf->createID != no_id
              && g_file_test(confFilename, G_FILE_TEST_EXISTS)) {

        // ask also for .nufraw overwiting

        char message[UF_MAX_PATH];
        char *utf8 = g_filename_to_utf8(confFilename, -1, NULL, NULL, NULL);
        if (utf8 == NULL) utf8 = g_strdup("Unknown file name");
        snprintf(message, UF_MAX_PATH,_("File '%s' already exists.\nOverwrite?"), utf8);
        g_free(utf8);
        g_free(confFilename);

        if (nufraw_confirm_dialog(message, _("File exists"), widget)==GTK_RESPONSE_NO)
            return NUFRAW_ERROR;
    }

    int status = nufraw_write_image(uf);
    if (status == NUFRAW_ERROR) {
        nufraw_message(status, nufraw_get_message(uf));
        return NUFRAW_ERROR;
    }
    if (status == NUFRAW_WARNING) {
        nufraw_message(status, nufraw_get_message(uf));
    }
    if (nufraw_get_message(uf) != NULL)
        nufraw_message(NUFRAW_SET_LOG, nufraw_get_message(uf));
    return NUFRAW_SUCCESS;
}

long nufraw_send_to_gimp(nufraw_data *uf, char* remoteGimpCommand)
{
	char *basename = g_path_get_basename(uf->conf->inputFilename);
	char *template = g_strconcat(basename, "_XXXXXX", NULL);
	g_free(basename);

	char *confFilename = NULL;
	GError *err = NULL;

	int fd = g_file_open_tmp(template, &confFilename, &err);
	g_free(template);
	if (fd == -1) {
		g_free(confFilename);
		nufraw_message(NUFRAW_ERROR, "%s\n%s", _("Error creating temporary file."), err->message);
		g_error_free(err);
		return NUFRAW_ERROR;
	}

	FILE *out = fdopen(fd, "w");
	if (out == NULL) {
		g_free(confFilename);
		nufraw_message(NUFRAW_ERROR, "%s\n%s",  _("Error creating temporary file."), g_strerror(errno));
		return NUFRAW_ERROR;
	}

	char *buffer;
	int saveCreateID = uf->conf->createID;
	uf->conf->createID = send_id;
	conf_save(uf->conf, confFilename, &buffer);
	uf->conf->createID = saveCreateID;
	if (fwrite(buffer, strlen(buffer), 1, out) != 1) {
		g_free(buffer);
		g_free(confFilename);
		nufraw_message(NUFRAW_ERROR, "%s\n%s", _("Error creating temporary file."), g_strerror(errno));
		return NUFRAW_ERROR;
	}
	g_free(buffer);

	if (fclose(out) != 0) {
		g_free(confFilename);
		nufraw_message(NUFRAW_ERROR, "%s\n%s",  _("Error creating temporary file."), g_strerror(errno));
		return NUFRAW_ERROR;
	}

	char *fullConfFilename = g_strconcat(confFilename, ".nufraw", NULL);
	if (g_rename(confFilename, fullConfFilename) == -1) {
		g_free(confFilename);
		g_free(fullConfFilename);
		nufraw_message(NUFRAW_ERROR, "%s\n%s", _("Error creating temporary file."), g_strerror(errno));
		return NUFRAW_ERROR;
	}
	g_free(confFilename);

	char *commandLine = g_strdup_printf("%s \"%s\"",remoteGimpCommand, fullConfFilename);

g_printerr("%s\n",commandLine);

	// gimp-remote starts the gimp in a fork() therefore we must call it asynchronously
	if (!g_spawn_command_line_async(commandLine, &err)) {

		g_free(commandLine);

#ifdef _WIN32
		if (strcmp(remoteGimpCommand, ui_conf_default.remoteGimpCommand) == 0) {

			//If the user didn't play with the remoteGimpCommand,try to run Gimp-2.6 instead of Gimp-2.8
			commandLine = g_strdup_printf("gimp-2.6.exe \"%s\"", fullConfFilename);
			g_error_free(err);
			err = NULL;

			// gimp-remote starts the gimp in a fork() therefore we must call it asynchronously
			if (!g_spawn_command_line_async(commandLine, &err)) {

				g_free(commandLine);
				g_free(fullConfFilename);
				nufraw_message(NUFRAW_ERROR, "%s\n%s", _("Error activating Gimp."), err->message);
				g_error_free(err);
				return NUFRAW_ERROR;
			}
		}else
#endif
		{
			g_free(fullConfFilename);
			nufraw_message(NUFRAW_ERROR, "%s\n%s", _("Error activating Gimp."), err->message);
			g_error_free(err);
			return NUFRAW_ERROR;
		}
	}
	g_free(fullConfFilename);
	g_free(commandLine);

	// Sleep for 0.2 seconds, giving time for gimp-remote to do the X query.
	g_usleep(200000);
	return NUFRAW_SUCCESS;
}
