/*
 * nUFRaw - Unidentified Flying Raw converter for digital camera images
 *
 * nufraw_ui_options.c - GUI for UI options
 * Copyright 2016 by Matteo Lucarelli
 *
 * based on the original ufraw sources by Udi Fuchs
 * based on the GIMP plug-in by Pawel T. Jochym jochym at ifj edu pl,
 *
 * based on the GIMP plug-in by Dave Coffin
 * http://www.cybercom.net/~dcoffin/
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 */
 
#include <string.h>
#include <math.h>
#include <glib/gi18n.h>

#include "nufraw.h"
#include "uf_gtk.h"
#include "nufraw_ui.h"

// from nufraw_ui.c
extern ui_data gUiData;
extern ui_conf *gUiConf;
extern colorLabels *color_labels_new(GtkTable *table, int x, int y, char *label, int format, gboolean forspot);
extern void color_labels_set(colorLabels *l, double data[]);
extern void render_preview();
extern gboolean ui_is_rendering();

void collect_raw_histogram_data()
{
	int i,c;

	nufraw_image_data *image = nufraw_get_image(gUiData.UF, nufraw_first_phase, TRUE);
	
	memset(gUiData.raw_his, 0, sizeof(gUiData.raw_his));
	
	for (i = 0; i < image->height * image->width; i++) {
	
		guint16 *buf = (guint16*)(image->buffer + i * image->depth);		
		for (c = 0; c < gUiData.UF->colors; c++){  // FIXME: gUiData.UF->colors max 4
			gUiData.raw_his[MIN(buf[c] *(UI_HIS_CURVE_MAXWIDTH - 1) / gUiData.UF->rgbMax, UI_HIS_CURVE_MAXWIDTH - 1)][c]++;
		}
	}
}

gboolean render_raw_histogram(gpointer unused)
{
	(void)unused;
	if (gUiData.FreezeDialog) return FALSE;

	guint8 pix[99], *p8, pen[4][3];
	nufraw_image_type p16;
	int x, c, cl, y, y0, y1;
	int raw_his[gUiData.LeftSpace][4], raw_his_max;

	int hisHeight = gUiData.RawHisto->allocation.height - 2;
	hisHeight = MAX(MIN(hisHeight, UI_HIS_MAXHEIGHT), UI_HIS_MINHEIGHT);

	GdkPixbuf *pixbuf = gtk_image_get_pixbuf(GTK_IMAGE(gUiData.RawHisto));
	if (pixbuf == NULL || 
	    gdk_pixbuf_get_height(pixbuf) != hisHeight + 2  || 
	    gdk_pixbuf_get_width(pixbuf) != gUiData.LeftSpace + 2) {
	    
		pixbuf = gdk_pixbuf_new(GDK_COLORSPACE_RGB, FALSE, 8, gUiData.LeftSpace + 2, hisHeight + 2);
		gtk_image_set_from_pixbuf(GTK_IMAGE(gUiData.RawHisto), pixbuf);
		g_object_unref(pixbuf);
	}
	
	int colors = gUiData.UF->colors;
	guint8 *pixies = gdk_pixbuf_get_pixels(pixbuf);
	int rowstride = gdk_pixbuf_get_rowstride(pixbuf);
	memset(pixies, 0, (gdk_pixbuf_get_height(pixbuf) - 1)*rowstride +
	     gdk_pixbuf_get_width(pixbuf)*gdk_pixbuf_get_n_channels(pixbuf));
	     
	// Normalize raw histogram data
	raw_his_max = 1;
	for (x = 0; x < gUiData.LeftSpace; x++) {
		for (c = 0, y = 0; c < colors; c++) {
			if (gUiConf->rawHistogramScale == log_histogram){
				raw_his[x][c] = log(1 + gUiData.raw_his[x*UI_HIS_CURVE_MAXWIDTH/gUiData.LeftSpace][c]) * 1000;
			}else{
			    raw_his[x][c] = gUiData.raw_his[x*UI_HIS_CURVE_MAXWIDTH/gUiData.LeftSpace][c];
			}
			y += raw_his[x][c];
		}
		raw_his_max = MAX(raw_his_max, y);
	}
	// Prepare pen color, which is not effected by exposure.
	// Use a small value to avoid highlights and then normalize.
	for (c = 0; c < colors; c++) {
		for (cl = 0; cl < colors; cl++) p16[cl] = 0;
		p16[c] = gUiData.UF->developer->max * 0x08000 / gUiData.UF->developer->rgbWB[c] *
		     0x10000 / gUiData.UF->developer->exposure;
		develop(pen[c], p16, gUiData.UF->developer, 8, 1);
		guint8 max = 1;
		for (cl = 0; cl < 3; cl++) max = MAX(pen[c][cl], max);
		for (cl = 0; cl < 3; cl++) pen[c][cl] = pen[c][cl] * 0xff / max;
	}
	// Calculate the curves
	p8 = pix;
	int grayCurve[gUiData.LeftSpace + 1][4]; // bottom of the conversion curves
	int pureCurve[gUiData.LeftSpace + 1][4]; // top of the conversion curves
	
	for (x = 0; x < gUiData.LeftSpace + 1; x++) {
		for (c = 0; c < colors; c++) {

			// Value for pixel x of color c in a gray pixel
			for (cl = 0; cl < colors; cl++){
			    p16[cl] = MIN((guint64)x * gUiData.UF->developer->rgbMax *
						gUiData.UF->developer->rgbWB[c] /
						gUiData.UF->developer->rgbWB[cl] / gUiData.LeftSpace, 0xFFFF);
			}
			develop(p8, p16, gUiData.UF->developer, 8, 1);
			grayCurve[x][c] = MAX(MAX(p8[0], p8[1]), p8[2]) * (hisHeight - 1) / 255;

			// Value for pixel x of pure color c
			p16[0] = p16[1] = p16[2] = p16[3] = 0;
			p16[c] = MIN((guint64)x * gUiData.UF->developer->rgbMax / gUiData.LeftSpace, 0xFFFF);
			develop(p8, p16, gUiData.UF->developer, 8, 1);
			pureCurve[x][c] = MAX(MAX(p8[0], p8[1]), p8[2]) * (hisHeight - 1) / 255;
		}
	}
	
	for (x = 0; x < gUiData.LeftSpace; x++) {
	
		// draw the raw histogram
		for (c = 0, y0 = 0; c < colors; c++) {
			for (y = 0; y < raw_his[x][c]*hisHeight / raw_his_max; y++){
				for (cl = 0; cl < 3; cl++){
			    		pixies[(hisHeight - y - y0)*rowstride + 3 * (x + 1) + cl] = pen[c][cl];
			    	}
			}
			y0 += y;
		}
		// draw curves on the raw histogram
		for (c = 0; c < colors; c++) {
			y = grayCurve[x][c];
			y1 = grayCurve[x + 1][c];
			for (; y <= y1; y++)
				for (cl = 0; cl < 3; cl++)
					pixies[(hisHeight - y)*rowstride + 3 * (x + 1) + cl] = pen[c][cl];
			y1 = pureCurve[x][c];
			for (; y < y1; y++)
				for (cl = 0; cl < 3; cl++)
					pixies[(hisHeight - y)*rowstride + 3 * (x + 1) + cl] = pen[c][cl] / 2;
			y1 = pureCurve[x + 1][c];
			for (; y <= y1; y++)
				for (cl = 0; cl < 3; cl++)
					pixies[(hisHeight - y)*rowstride + 3 * (x + 1) + cl] = pen[c][cl];
		}
	}
	gtk_widget_queue_draw(gUiData.RawHisto);
	return FALSE;
}

gboolean render_live_histogram(gpointer unused)
{
	(void)unused;    
	if (gUiData.FreezeDialog) return FALSE;

	int x, y, c, min, max;
	nufraw_image_data *img = nufraw_get_image(gUiData.UF,nufraw_develop_phase, TRUE);

	UFRectangle Crop;
	nufraw_get_scaled_crop(gUiData.UF, &Crop);

	double rgb[3];
	guint64 sum[3], sqr[3];
	int live_his[256][4];
	int vals[256][4];
	guint8 *p8;

	memset(live_his, 0, sizeof(live_his));
	memset(vals, 0, sizeof(vals));

	for (y = Crop.y; y < Crop.y + Crop.height; y++){
	
		for (x = Crop.x; x < Crop.x + Crop.width; x++) {

			p8 = img->buffer + y * img->rowstride + x * img->depth;
			max = 0;
			min = 0x100;      
			for (c = 0; c < 3; c++) {
				max = MAX(max, p8[c]);
				min = MIN(min, p8[c]);
				vals[p8[c]][c]++;
			}
			
			if (gUiConf->liveHistogramMode == luminosity_histogram){
				vals[(int)(0.3 * p8[0] + 0.59 * p8[1] + 0.11 * p8[2])][3]++;
			}else if (gUiConf->liveHistogramMode == value_histogram){
				vals[max][3]++;
			}else if (gUiConf->liveHistogramMode == saturation_histogram) {
				if (max == 0) vals[0][3]++;
				else vals[255 * (max - min) / max][3]++;
			}
		}
	}
	
	int hisHeight = MIN(gUiData.LiveHisto->allocation.height - 2, UI_HIS_MAXHEIGHT);
	hisHeight = MAX(hisHeight, UI_HIS_MINHEIGHT);

	GdkPixbuf *pixbuf = gtk_image_get_pixbuf(GTK_IMAGE(gUiData.LiveHisto));
	if (pixbuf == NULL || 
	    gdk_pixbuf_get_height(pixbuf) != hisHeight + 2 ||
	    gdk_pixbuf_get_width(pixbuf) != gUiData.LeftSpace + 2){
	    
		pixbuf = gdk_pixbuf_new(GDK_COLORSPACE_RGB, FALSE, 8, gUiData.LeftSpace + 2, hisHeight + 2);
		gtk_image_set_from_pixbuf(GTK_IMAGE(gUiData.LiveHisto), pixbuf);
		g_object_unref(pixbuf);
	}
	
	guint8 *pixies = gdk_pixbuf_get_pixels(pixbuf);
	int rowstride = gdk_pixbuf_get_rowstride(pixbuf);
	memset(pixies, 0, (gdk_pixbuf_get_height(pixbuf) - 1)*rowstride +
	     gdk_pixbuf_get_width(pixbuf)*gdk_pixbuf_get_n_channels(pixbuf));

	// set live_his_max
	int live_his_max=1;
	for (x = 0; x <= 255 ; x++) {

		if (gUiConf->liveHistogramScale == log_histogram){
			for (c = 0; c < 4; c++) live_his[x][c] = log(1 + vals[x][c]) * 1000;
		}else{
			for (c = 0; c < 4; c++) live_his[x][c] = vals[x][c];
		}

		if (gUiConf->liveHistogramMode == rgb_histogram){

			for (c = 0; c < 3; c++){
				live_his_max = MAX(live_his_max, live_his[x][c]);
			}
		    
		}else if (gUiConf->liveHistogramMode == r_g_b_histogram){

			live_his_max = MAX(live_his_max, live_his[x][0] + live_his[x][1] + live_his[x][2]);

		}else{
			live_his_max = MAX(live_his_max, live_his[x][3]);   
		}
	}

	// draw pixbuf
	int lhX;
	for (x = 0; x < gUiData.LeftSpace; x++){
		for (y = 0; y < hisHeight; y++){

			// convert x=0:gUiData.LeftSpace to live_his[0:255] index
			lhX=(int)((x/(float)(gUiData.LeftSpace))*255+0.5);

			if (gUiConf->liveHistogramMode == r_g_b_histogram) {

				if (y * live_his_max < live_his[lhX][0]*hisHeight)
					pixies[(hisHeight - y)*rowstride + 3 * (x + 1) + 0] = 255;
	
				else if (y * live_his_max <  (live_his[lhX][0] + live_his[lhX][1])*hisHeight)
					pixies[(hisHeight - y)*rowstride + 3 * (x + 1) + 1] = 255;
	
				else if (y * live_his_max < (live_his[lhX][0] + live_his[lhX][1] + live_his[lhX][2]) *hisHeight)
					pixies[(hisHeight - y)*rowstride + 3 * (x + 1) + 2] = 255;

			} else {

				for (c = 0; c < 3; c++){

					if (gUiConf->liveHistogramMode == rgb_histogram) {
						if (y * live_his_max < live_his[lhX][c]*hisHeight)
							pixies[(hisHeight - y)*rowstride + 3 * (x + 1) + c] = 255;
					} else {
						if (y * live_his_max < live_his[lhX][3]*hisHeight)
							pixies[(hisHeight - y)*rowstride + 3 * (x + 1) + c] = 255;
					}
				}
			}
		}
	}

	// draw vertical line at quarters "behind" the live histogram
	for (y = -1; y < hisHeight + 1; y++){
		for (x = 0; x < gUiData.LeftSpace; x += gUiData.LeftSpace/4 ) {
			guint8 *pix = pixies + (hisHeight - y) * rowstride + 3 * (x + 1);
			if (pix[0] == 0 && pix[1] == 0 && pix[2] == 0){
				for (c = 0; c < 3; c++) pix[c] = 96; // gray
			}
		}
	}
	gtk_widget_queue_draw(gUiData.LiveHisto);

	// set labels in value table
	
	for (c = 0; c < 3; c++) {
		sum[c] = 0;
		sqr[c] = 0;
		for (x = 0; x <= 255; x++) {
			sum[c] += x * vals[x][c];
			sqr[c] += (guint64)x * x * vals[x][c];
		}
	}

	float CropCount = Crop.width * Crop.height;
	if (CropCount == 0) CropCount = 1;	// Bug #387: Fix divide-by-zero crash.
	for (c = 0; c < 3; c++) rgb[c] = sum[c] / CropCount;
	color_labels_set(gUiData.AvrLabels, rgb);
	for (c = 0; c < 3; c++) rgb[c] = sqrt(sqr[c] / CropCount - rgb[c] * rgb[c]);
	color_labels_set(gUiData.DevLabels, rgb);
	for (c = 0; c < 3; c++) rgb[c] = 100.0 * vals[255][c] / CropCount;
	color_labels_set(gUiData.OverLabels, rgb);
	for (c = 0; c < 3; c++) rgb[c] = 100.0 * vals[0][c] / CropCount;
	color_labels_set(gUiData.UnderLabels, rgb);

	gchar buf[20];
	g_snprintf(buf, sizeof(buf), "%d", gUiData.UF->hotpixels);
	gtk_label_set_text(gUiData.HotpixelCount, buf);

	return FALSE;
}

static gboolean show_histogram_menu(GtkMenu *menu, GdkEventButton *event)
{
	if (gUiData.FreezeDialog) return FALSE;
	if (event->button != 3) return FALSE;
	gtk_menu_popup(menu, NULL, NULL, NULL, NULL, event->button, event->time);
	return TRUE;
}

static void radio_menu_update(GtkWidget *item, gint *valuep)
{
	if (gtk_check_menu_item_get_active(GTK_CHECK_MENU_ITEM(item))) {
		*valuep = GPOINTER_TO_INT(g_object_get_data(G_OBJECT(item), "Radio-Value"));
		render_preview();
	}
}

void histogram_expander(GtkWidget *expander, void *unused)
{
	(void)unused;

	GtkWidget *panel = gtk_widget_get_parent(expander);
	if (gtk_expander_get_expanded(GTK_EXPANDER(expander))) {

		g_object_set_data(G_OBJECT(expander), "expander-maximized", (gpointer)FALSE);
		gtk_widget_set_size_request(gUiData.RawHisto, -1, UI_HIS_MINHEIGHT);
		gtk_widget_set_size_request(gUiData.LiveHisto, -1, UI_HIS_MINHEIGHT);

		// FIXME: expander-maximized? why?
		gboolean expanderMaximized = GPOINTER_TO_INT(g_object_get_data(G_OBJECT(expander), "expander-maximized"));
		
		if (!expanderMaximized){
			gtk_box_set_child_packing(GTK_BOX(panel), GTK_WIDGET(expander), TRUE, TRUE, 0, GTK_PACK_START);
		}
	} else {
		gtk_box_set_child_packing(GTK_BOX(panel), GTK_WIDGET(expander), FALSE, FALSE, 0, GTK_PACK_START);
	}
}

void panel_size_allocate(GtkWidget *panel, GtkAllocation *allocation, gpointer unused)
{
    (void)unused;

    // Raw histogram expander status
    gboolean rawMaximized=FALSE;
    int rawHisHeight=0;
    gboolean rawExpanded=FALSE;
    GtkWidget *rawExpander = gtk_widget_get_ancestor(gUiData.RawHisto, GTK_TYPE_EXPANDER);
    if (rawExpander){
	rawMaximized = GPOINTER_TO_INT(g_object_get_data(G_OBJECT(rawExpander), "expander-maximized"));
	rawHisHeight = gUiData.RawHisto->allocation.height - 2;
	rawExpanded = gtk_expander_get_expanded(GTK_EXPANDER(rawExpander));
    }
        
    // Live histogram expander status
    gboolean liveMaximized=FALSE;
    int liveHisHeight=0;
    gboolean liveExpanded=FALSE;
    GtkWidget *liveExpander = gtk_widget_get_ancestor(gUiData.LiveHisto, GTK_TYPE_EXPANDER);
    if (liveExpander){
	liveMaximized = GPOINTER_TO_INT(g_object_get_data(G_OBJECT(liveExpander), "expander-maximized"));
	liveHisHeight = gUiData.LiveHisto->allocation.height - 2;
	liveExpanded = gtk_expander_get_expanded(GTK_EXPANDER(liveExpander));
    }

    // Stop allocating space to histogram after maximum height was reached
    gboolean maximizeAll = TRUE;
    if (rawExpanded && rawHisHeight <= UI_HIS_MAXHEIGHT) maximizeAll = FALSE;
    if (liveExpanded && liveHisHeight <= UI_HIS_MAXHEIGHT) maximizeAll = FALSE;

    if (maximizeAll && !rawMaximized) {
    
        gtk_box_set_child_packing(GTK_BOX(panel), rawExpander, FALSE, FALSE, 0, GTK_PACK_START);
        gtk_widget_set_size_request(gUiData.RawHisto, gUiData.RawHisto->allocation.width, UI_HIS_MAXHEIGHT + 2);
        g_object_set_data(G_OBJECT(rawExpander), "expander-maximized", (gpointer)TRUE);
    }
    
    if (maximizeAll && !liveMaximized) {
    
        gtk_box_set_child_packing(GTK_BOX(panel), liveExpander, FALSE, FALSE, 0, GTK_PACK_START);
        gtk_widget_set_size_request(gUiData.LiveHisto, gUiData.LiveHisto->allocation.width, UI_HIS_MAXHEIGHT + 2);
        g_object_set_data(G_OBJECT(liveExpander), "expander-maximized", (gpointer)TRUE);
    }

    GList *children = gtk_container_get_children(GTK_CONTAINER(panel));
    int childAlloc = 0;
    for (; children != NULL; children = g_list_next(children))
        childAlloc += GTK_WIDGET(children->data)->allocation.height;

    // If there is no extra space in the box expandable widgets must be shrinkable
    if (allocation->height == childAlloc) {
    
        if (rawExpanded && rawMaximized) {
        
            if (gUiData.RawHisto->requisition.height != UI_HIS_MINHEIGHT)
                gtk_widget_set_size_request(gUiData.RawHisto,
                                            gUiData.RawHisto->allocation.width, UI_HIS_MINHEIGHT);
            gboolean rawExpandable;
            gtk_box_query_child_packing(GTK_BOX(panel), rawExpander,
                                        &rawExpandable, NULL, NULL, NULL);
            if (!rawExpandable)
                gtk_box_set_child_packing(GTK_BOX(panel), rawExpander,
                                          TRUE, TRUE, 0, GTK_PACK_START);
            g_object_set_data(G_OBJECT(rawExpander),
                              "expander-maximized", (gpointer)FALSE);
        }
        
        if (liveExpanded && liveMaximized) {
            if (gUiData.LiveHisto->requisition.height != UI_HIS_MINHEIGHT)
                gtk_widget_set_size_request(gUiData.LiveHisto,
                                            gUiData.LiveHisto->allocation.width, UI_HIS_MINHEIGHT);
            gboolean liveExpandable;
            gtk_box_query_child_packing(GTK_BOX(panel), liveExpander,
                                        &liveExpandable, NULL, NULL, NULL);
            if (!liveExpandable)
                gtk_box_set_child_packing(GTK_BOX(panel), liveExpander,
                                          TRUE, TRUE, 0, GTK_PACK_START);
            g_object_set_data(G_OBJECT(liveExpander),
                              "expander-maximized", (gpointer)FALSE);
        }
    }

    if (!ui_is_rendering()) {
    
        // Redraw histograms if size allocation has changed
        GdkPixbuf *pixbuf = gtk_image_get_pixbuf(GTK_IMAGE(gUiData.RawHisto));
        rawHisHeight = gUiData.RawHisto->allocation.height;
        if (pixbuf == NULL || gdk_pixbuf_get_height(pixbuf) != rawHisHeight)
            if (rawExpanded)
                gdk_threads_add_idle_full(G_PRIORITY_DEFAULT_IDLE,(GSourceFunc)(render_raw_histogram),&gUiData, NULL);

        pixbuf = gtk_image_get_pixbuf(GTK_IMAGE(gUiData.LiveHisto));
        liveHisHeight = gUiData.LiveHisto->allocation.height;
        if (pixbuf == NULL || gdk_pixbuf_get_height(pixbuf) != liveHisHeight)
            if (liveExpanded)
                gdk_threads_add_idle_full(G_PRIORITY_DEFAULT_IDLE,(GSourceFunc)(render_live_histogram),&gUiData, NULL);
    }
}

void rawhistogram_fill_interface(GtkTable* table)
{
	GtkWidget *event_box;
	GdkPixbuf *pixbuf;
	guint8 *pixies;
	GtkWidget *menu;
	GSList *group;
	GtkWidget *menu_item;
	int rowstride;

	event_box = gtk_event_box_new();
	gtk_table_attach(table, event_box, 0, 1, 1, 2, GTK_EXPAND, GTK_EXPAND | GTK_FILL, 0, 0);

	pixbuf = gdk_pixbuf_new(GDK_COLORSPACE_RGB, FALSE, 8, gUiData.LeftSpace + 2, UI_HIS_MAXHEIGHT + 2);
	gUiData.RawHisto = gtk_image_new_from_pixbuf(pixbuf);
	gtk_widget_set_size_request(gUiData.RawHisto, gUiData.LeftSpace + 2, UI_HIS_MINHEIGHT + 2);
	g_object_unref(pixbuf);
	
	gtk_container_add(GTK_CONTAINER(event_box), gUiData.RawHisto);
	pixies = gdk_pixbuf_get_pixels(pixbuf);
	rowstride = gdk_pixbuf_get_rowstride(pixbuf);
	memset(pixies, 0, (gdk_pixbuf_get_height(pixbuf) - 1)* rowstride +
	     gdk_pixbuf_get_width(pixbuf)*gdk_pixbuf_get_n_channels(pixbuf));

	menu = gtk_menu_new();
	g_object_set_data(G_OBJECT(menu), "Parent-Widget", event_box);
	g_signal_connect_swapped(G_OBJECT(event_box), "button_press_event", G_CALLBACK(show_histogram_menu), menu);
	
	group = NULL;
	menu_item = gtk_radio_menu_item_new_with_label(group, _("Linear"));
	gtk_menu_attach(GTK_MENU(menu), menu_item, 0, 1, 0, 1);
	group = gtk_radio_menu_item_get_group(GTK_RADIO_MENU_ITEM(menu_item));
	gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(menu_item),gUiConf->rawHistogramScale == linear_histogram);
	g_object_set_data(G_OBJECT(menu_item), "Radio-Value",(gpointer)linear_histogram);
	g_signal_connect(G_OBJECT(menu_item), "toggled",G_CALLBACK(radio_menu_update), &gUiConf->rawHistogramScale);
	menu_item = gtk_radio_menu_item_new_with_label(group, _("Logarithmic"));
	gtk_menu_attach(GTK_MENU(menu), menu_item, 0, 1, 1, 2);
	group = gtk_radio_menu_item_get_group(GTK_RADIO_MENU_ITEM(menu_item));
	gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(menu_item),gUiConf->rawHistogramScale == log_histogram);
	g_object_set_data(G_OBJECT(menu_item), "Radio-Value",(gpointer)log_histogram);
	g_signal_connect(G_OBJECT(menu_item), "toggled",G_CALLBACK(radio_menu_update), &gUiConf->rawHistogramScale);
	gtk_widget_show_all(menu);
}

void livehistogram_fill_interface(GtkTable *table)
{
	GtkWidget *event_box;
	GdkPixbuf *pixbuf;
	guint8 *pixies;
	GtkWidget *menu;
	GSList *group;
	GtkWidget *menu_item;
	int rowstride;

	// histogram
	event_box = gtk_event_box_new();
	gtk_table_attach(table, event_box, 0, 7, 1, 2, GTK_EXPAND, GTK_EXPAND | GTK_FILL, 0, 0);
	pixbuf = gdk_pixbuf_new(GDK_COLORSPACE_RGB, FALSE, 8, gUiData.LeftSpace + 2, UI_HIS_MAXHEIGHT + 2);
	gUiData.LiveHisto = gtk_image_new_from_pixbuf(pixbuf);
	gtk_container_add(GTK_CONTAINER(event_box), gUiData.LiveHisto);
	gtk_widget_set_size_request(gUiData.LiveHisto, gUiData.LeftSpace + 2, UI_HIS_MINHEIGHT + 2);
	g_object_unref(pixbuf);
	pixies = gdk_pixbuf_get_pixels(pixbuf);
	rowstride = gdk_pixbuf_get_rowstride(pixbuf);
	memset(pixies, 0, (gdk_pixbuf_get_height(pixbuf) - 1)* rowstride + gdk_pixbuf_get_width(pixbuf)*gdk_pixbuf_get_n_channels(pixbuf));

	// context menu
	menu = gtk_menu_new();
	g_object_set_data(G_OBJECT(menu), "Parent-Widget", event_box);
	g_signal_connect_swapped(G_OBJECT(event_box), "button_press_event",G_CALLBACK(show_histogram_menu), menu);
	group = NULL;

	menu_item = gtk_radio_menu_item_new_with_label(group, _("RGB histogram"));
	gtk_menu_attach(GTK_MENU(menu), menu_item, 0, 1, 0, 1);
	group = gtk_radio_menu_item_get_group(GTK_RADIO_MENU_ITEM(menu_item));
	gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(menu_item),gUiConf->liveHistogramMode == rgb_histogram);
	g_object_set_data(G_OBJECT(menu_item), "Radio-Value",(gpointer)rgb_histogram);
	g_signal_connect(G_OBJECT(menu_item), "toggled",G_CALLBACK(radio_menu_update), &gUiConf->liveHistogramMode);

	menu_item = gtk_radio_menu_item_new_with_label(group, _("R+G+B histogram"));
	gtk_menu_attach(GTK_MENU(menu), menu_item, 0, 1, 1, 2);
	group = gtk_radio_menu_item_get_group(GTK_RADIO_MENU_ITEM(menu_item));
	gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(menu_item),gUiConf->liveHistogramMode == r_g_b_histogram);
	g_object_set_data(G_OBJECT(menu_item), "Radio-Value",(gpointer)r_g_b_histogram);
	g_signal_connect(G_OBJECT(menu_item), "toggled",G_CALLBACK(radio_menu_update), &gUiConf->liveHistogramMode);

	menu_item = gtk_radio_menu_item_new_with_label(group,_("Luminosity histogram"));
	gtk_menu_attach(GTK_MENU(menu), menu_item, 0, 1, 2, 3);
	group = gtk_radio_menu_item_get_group(GTK_RADIO_MENU_ITEM(menu_item));
	gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(menu_item),gUiConf->liveHistogramMode == luminosity_histogram);
	g_object_set_data(G_OBJECT(menu_item), "Radio-Value",(gpointer)luminosity_histogram);
	g_signal_connect(G_OBJECT(menu_item), "toggled",G_CALLBACK(radio_menu_update), &gUiConf->liveHistogramMode);

	menu_item = gtk_radio_menu_item_new_with_label(group, _("Value (maximum) histogram"));
	gtk_menu_attach(GTK_MENU(menu), menu_item, 0, 1, 3, 4);
	group = gtk_radio_menu_item_get_group(GTK_RADIO_MENU_ITEM(menu_item));
	gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(menu_item),gUiConf->liveHistogramMode == value_histogram);
	g_object_set_data(G_OBJECT(menu_item), "Radio-Value",(gpointer)value_histogram);
	g_signal_connect(G_OBJECT(menu_item), "toggled", G_CALLBACK(radio_menu_update), &gUiConf->liveHistogramMode);

	menu_item = gtk_radio_menu_item_new_with_label(group,_("Saturation histogram"));
	gtk_menu_attach(GTK_MENU(menu), menu_item, 0, 1, 4, 5);
	group = gtk_radio_menu_item_get_group(GTK_RADIO_MENU_ITEM(menu_item));
	gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(menu_item),gUiConf->liveHistogramMode == saturation_histogram);
	g_object_set_data(G_OBJECT(menu_item), "Radio-Value",(gpointer)saturation_histogram);
	g_signal_connect(G_OBJECT(menu_item), "toggled",G_CALLBACK(radio_menu_update), &gUiConf->liveHistogramMode);

	menu_item = gtk_separator_menu_item_new();
	gtk_menu_attach(GTK_MENU(menu), menu_item, 0, 1, 5, 6);
	group = NULL;

	menu_item = gtk_radio_menu_item_new_with_label(group, _("Linear"));
	gtk_menu_attach(GTK_MENU(menu), menu_item, 0, 1, 6, 7);
	group = gtk_radio_menu_item_get_group(GTK_RADIO_MENU_ITEM(menu_item));
	gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(menu_item), gUiConf->liveHistogramScale == linear_histogram);
	g_object_set_data(G_OBJECT(menu_item), "Radio-Value",(gpointer)linear_histogram);
	g_signal_connect(G_OBJECT(menu_item), "toggled", G_CALLBACK(radio_menu_update), &gUiConf->liveHistogramScale);

	menu_item = gtk_radio_menu_item_new_with_label(group, _("Logarithmic"));
	gtk_menu_attach(GTK_MENU(menu), menu_item, 0, 1, 7, 8);
	group = gtk_radio_menu_item_get_group(GTK_RADIO_MENU_ITEM(menu_item));
	gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(menu_item), gUiConf->liveHistogramScale == log_histogram);
	g_object_set_data(G_OBJECT(menu_item), "Radio-Value", (gpointer)log_histogram);
	g_signal_connect(G_OBJECT(menu_item), "toggled", G_CALLBACK(radio_menu_update), &gUiConf->liveHistogramScale);

	gtk_widget_show_all(menu);
}

void valuetable_fill_interface(GtkTable *table)
{
	GtkWidget *label;

	label = gtk_label_new(_("Red"));
	gtk_misc_set_alignment(GTK_MISC(label),0.5,0.5);
	gtk_table_attach(table, label, 1, 2, 0, 1, 0, 0, 0, 0);
	label = gtk_label_new(_("Green"));
	gtk_misc_set_alignment(GTK_MISC(label),0.5,0.5);
	gtk_table_attach(table, label, 2, 3, 0, 1, 0, 0, 0, 0);
	label = gtk_label_new(_("Blue"));
	gtk_misc_set_alignment(GTK_MISC(label),0.5,0.5);
	gtk_table_attach(table, label, 3, 4, 0, 1, 0, 0, 0, 0);

	gUiData.AvrLabels = color_labels_new(table, 0, 1, _("Average"), pixel_format, FALSE);                                       
	gUiData.DevLabels = color_labels_new(table, 0, 2, _("Std. deviation"), pixel_format, FALSE);                                      
	gUiData.OverLabels = color_labels_new(table, 0, 3, _("Overexposed"), percent_format, FALSE);
	gUiData.UnderLabels = color_labels_new(table, 0, 4, _("Underexposed"), percent_format, FALSE);
}

