/*
 * nUFRaw - Unidentified Flying Raw converter for digital camera images
 *
 * nufraw_chooser.c - A very simple file chooser for nUFRaw.
 * Copyright 2004-2015 by Udi Fuchs
 * Copyright 2016 by Matteo Lucarelli
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 */

#include "nufraw.h"
#include "uf_gtk.h"
#include <string.h>
#include <glib/gi18n.h>

#include "dcraw_api.h"

static gboolean preview_enabled;
static gboolean hidden_enabled;

static void update_preview_cb(GtkFileChooser *file_chooser, gpointer data)
{
	if (!preview_enabled){
		gtk_file_chooser_set_preview_widget_active(file_chooser, FALSE);
		return;
	}
	
	GtkWidget *preview;
	char *filename;
	GdkPixbuf *pixbuf=NULL;

	preview = GTK_WIDGET(data);
	filename = gtk_file_chooser_get_preview_filename(file_chooser);
		
	if (filename){
	
		nufraw_data *uf = nufraw_open(filename);
	
		if (uf){
				
			conf_data rc;
			
			// TODO: for .nufraw should use its conf
			//if (strcasecmp(filename + strlen(filename) - 6, ".nufraw") == 0) {
			
			//	conf_load(&rc, filename);
			//}else{
			
				// TO USE nufraw defaults
				conf_init(&rc);
				rc.ufobject = nufraw_resources_new();
			
				// TO USE user defaults
				//conf_load(&rc, NULL);
			//}
			
			nufraw_config(uf,&rc,NULL,NULL);
			uf->conf->embeddedImage=TRUE;
			
			if (nufraw_load_raw(uf) == NUFRAW_SUCCESS) {
			
				// TODO: thumb size from ui_conf
				uf->conf->shrink=MAX(uf->thumb.width/256,uf->thumb.height/256);
						
				if (nufraw_convert_embedded(uf) == NUFRAW_SUCCESS) {
					
					pixbuf = gdk_pixbuf_new_from_data(uf->thumb.buffer
						                            ,GDK_COLORSPACE_RGB
						                            ,FALSE
						                            ,8
						                            ,uf->thumb.width
						                            ,uf->thumb.height
						                            ,uf->thumb.width*3, NULL,NULL);
				}
			}else{
				// TODO: thumb size from ui_conf
				pixbuf = gdk_pixbuf_new_from_file_at_size(uf->filename, 256, 256, NULL);
			}
			g_free(uf);
		}
		
		if (pixbuf==NULL){
			// try open with gtk
			pixbuf = gdk_pixbuf_new_from_file_at_size(filename, 256, 256, NULL);
		}
		g_free (filename);
		
		if (pixbuf != NULL){
		
			gtk_image_set_from_pixbuf(GTK_IMAGE (preview), pixbuf);
			g_object_unref(pixbuf);
			gtk_file_chooser_set_preview_widget_active(file_chooser, TRUE);
			
		}else{
		
			gtk_file_chooser_set_preview_widget_active(file_chooser, FALSE);
		}
	}
}

void nufraw_hidden_chooser_toggle(GtkToggleButton *button, GtkFileChooser *fileChooser)
{
	hidden_enabled=gtk_toggle_button_get_active(button);
	gtk_file_chooser_set_show_hidden(fileChooser, gtk_toggle_button_get_active(button));
}

void nufraw_preview_chooser_toggle(GtkToggleButton *button, GtkFileChooser *fileChooser)
{
	if (gtk_toggle_button_get_active(button)){
		preview_enabled=TRUE;		
	}else{
		preview_enabled=FALSE;
	}
	g_signal_emit_by_name(fileChooser,"update-preview");
}

// Create a GtkFileChooser dialog for selecting raw files
GtkFileChooser *nufraw_raw_chooser(conf_data *conf,
                                   ui_conf* uiconf,
                                   const char *defPath,
                                   const gchar *label,
                                   GtkWindow *toplevel,
                                   const gchar *cancel,
                                   gboolean multiple,
                                   gboolean idonly)
{
	GtkFileChooser *fileChooser;
	GtkFileFilter *filter;
	char *cp;
	char **extList, **l, ext[UF_MAX_NAME];
	GtkWidget *button, *preview;

	fileChooser = GTK_FILE_CHOOSER(gtk_file_chooser_dialog_new(label, toplevel,
	                               GTK_FILE_CHOOSER_ACTION_OPEN,
	                               cancel, GTK_RESPONSE_CANCEL,
	                               GTK_STOCK_OPEN, GTK_RESPONSE_ACCEPT, NULL));
		                       
	if (toplevel == NULL)  gtk_window_set_type_hint(GTK_WINDOW(fileChooser),GDK_WINDOW_TYPE_HINT_NORMAL);
	else nufraw_focus(fileChooser, TRUE);

	gtk_window_set_icon_name(GTK_WINDOW(fileChooser), "nufraw");
	nufraw_message(NUFRAW_SET_PARENT, (char *)fileChooser);

	if (defPath != NULL) {
		char *fullPath = uf_file_set_absolute(defPath);
		gtk_file_chooser_set_current_folder(fileChooser, fullPath);
		g_free(fullPath);
	}

	filter = GTK_FILE_FILTER(gtk_file_filter_new());
	gtk_file_filter_set_name(filter, _("nUFRaw ID files"));
	gtk_file_filter_add_pattern(filter, "*.nufraw");
	gtk_file_filter_add_pattern(filter, "*.NUFRAW");
	gtk_file_filter_add_pattern(filter, "*.ufraw");
	gtk_file_filter_add_pattern(filter, "*.UFRAW");
	gtk_file_chooser_add_filter(fileChooser, filter);
    
	if (idonly==FALSE){
    
		filter = GTK_FILE_FILTER(gtk_file_filter_new());
		gtk_file_filter_set_name(filter, _("Raw images"));
		extList = g_strsplit(raw_ext, ",", 100);
		for (l = extList; *l != NULL; l++){
			if (strcmp(*l, "jpeg") && strcmp(*l, "jpg") && strcmp(*l, "tif") &&
			    strcmp(*l, "tiff") && strcmp(*l, "nufraw")) {
				snprintf(ext, UF_MAX_NAME, "*.%s", *l);
				gtk_file_filter_add_pattern(filter, ext);
				gtk_file_filter_add_pattern(filter, cp = g_ascii_strup(ext, -1));
				g_free(cp);
#ifdef HAVE_LIBZ
				snprintf(ext, UF_MAX_NAME, "*.%s.gz", *l);
				gtk_file_filter_add_pattern(filter, ext);
				snprintf(ext, UF_MAX_NAME, "*.%s.GZ", *l);
				gtk_file_filter_add_pattern(filter, ext);
				snprintf(ext, UF_MAX_NAME, "*.%s.gz", cp = g_ascii_strup(*l, -1));
				g_free(cp);
				gtk_file_filter_add_pattern(filter, ext);
				snprintf(ext, UF_MAX_NAME, "*.%s.GZ", cp = g_ascii_strup(*l, -1));
				g_free(cp);
				gtk_file_filter_add_pattern(filter, ext);
#endif
#ifdef HAVE_LIBBZ2
				snprintf(ext, UF_MAX_NAME, "*.%s.bz2", *l);
				gtk_file_filter_add_pattern(filter, ext);
				snprintf(ext, UF_MAX_NAME, "*.%s.BZ2", *l);
				gtk_file_filter_add_pattern(filter, ext);
				snprintf(ext, UF_MAX_NAME, "*.%s.bz2", cp = g_ascii_strup(*l, -1));
				g_free(cp);
				gtk_file_filter_add_pattern(filter, ext);
				snprintf(ext, UF_MAX_NAME, "*.%s.BZ2", cp = g_ascii_strup(*l, -1));
				g_free(cp);
				gtk_file_filter_add_pattern(filter, ext);
#endif
			}
		}
		g_strfreev(extList);
		gtk_file_chooser_add_filter(fileChooser, filter);
		gtk_file_chooser_set_filter (fileChooser, filter);

		filter = GTK_FILE_FILTER(gtk_file_filter_new());
		gtk_file_filter_set_name(filter, _("Jpeg"));
		// FIXME: gtk_file_filter_add_mime_type(filter,"image/jpeg"); 
		gtk_file_filter_add_pattern(filter, "*.jpeg");
		gtk_file_filter_add_pattern(filter, "*.jpg");
		gtk_file_filter_add_pattern(filter, "*.JPEG");
		gtk_file_filter_add_pattern(filter, "*.JPG");
		gtk_file_chooser_add_filter(fileChooser, filter);

		filter = GTK_FILE_FILTER(gtk_file_filter_new());
		// FIXME: gtk_file_filter_add_mime_type(filter,"image/tiff");
		gtk_file_filter_set_name(filter, _("Tiff"));
		gtk_file_filter_add_pattern(filter, "*.tif");
		gtk_file_filter_add_pattern(filter, "*.tiff");
		gtk_file_filter_add_pattern(filter, "*.TIF");
		gtk_file_filter_add_pattern(filter, "*.TIFF");
		gtk_file_chooser_add_filter(fileChooser, filter);

		filter = GTK_FILE_FILTER(gtk_file_filter_new());
		gtk_file_filter_set_name(filter, _("All files"));
		gtk_file_filter_add_pattern(filter, "*");
		gtk_file_chooser_add_filter(fileChooser, filter);

		gtk_file_chooser_set_show_hidden(fileChooser, uiconf->showHidden);
		button = gtk_check_button_new_with_label(_("Show hidden files"));
		
		gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(button), uiconf->showHidden);
		g_signal_connect(G_OBJECT(button), "toggled",  G_CALLBACK(nufraw_hidden_chooser_toggle), fileChooser);
		gtk_box_pack_start(GTK_BOX(GTK_DIALOG(fileChooser)->vbox),button, FALSE, FALSE, 0);

		button = gtk_check_button_new_with_label(_("Show preview"));
		gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(button), uiconf->showPreview);
		preview_enabled=uiconf->showPreview;
		g_signal_connect(G_OBJECT(button), "toggled",  G_CALLBACK(nufraw_preview_chooser_toggle), fileChooser);
		gtk_box_pack_start(GTK_BOX(GTK_DIALOG(fileChooser)->vbox),button, FALSE, FALSE, 0);

		if (multiple) gtk_file_chooser_set_select_multiple(fileChooser, TRUE);

		preview = gtk_image_new();
		gtk_file_chooser_set_preview_widget(fileChooser, preview);
		g_signal_connect(fileChooser, "update-preview", G_CALLBACK (update_preview_cb), preview);
		update_preview_cb(fileChooser, preview);
	}
        
	// Add shortcut to folder of last opened file
	if (strlen(conf->inputFilename) > 0) {
		char *cp = g_path_get_dirname(conf->inputFilename);
		gtk_file_chooser_add_shortcut_folder(fileChooser, cp, NULL);
		g_free(cp);
	}

	gtk_widget_show_all(GTK_WIDGET(fileChooser));
	return fileChooser;
}

void nufraw_chooser(conf_data *rc, conf_data *conf, conf_data *cmd, ui_conf* uiconf, const char *defPath)
{
	nufraw_data *uf;
	GtkFileChooser *fileChooser;
	GSList *list, *saveList;
	char *filename;

	if (defPath==NULL){
	  defPath=uiconf->inputPath;
	}

	fileChooser = nufraw_raw_chooser(rc, uiconf, defPath, "File Chooser - nUFRaw", NULL, GTK_STOCK_QUIT, FALSE, FALSE);

	while (gtk_dialog_run(GTK_DIALOG(fileChooser)) == GTK_RESPONSE_ACCEPT) {

		for (list = saveList = gtk_file_chooser_get_filenames(fileChooser); list != NULL; list = g_slist_next(list)) {

			filename = list->data;
			uf = nufraw_open(filename);
			if (uf == NULL) {
				nufraw_message(NUFRAW_REPORT, NULL);
				continue;
			}
			// TODO: start with user default or with previous image settings
			int status = nufraw_config(uf, rc, conf, cmd);
			if (status == NUFRAW_ERROR) {
			
				nufraw_close_darkframe(uf->conf);
				nufraw_close(uf);
				
			} else {

				// save file path in uiconf->inputPath
				gchar *p=g_path_get_dirname(filename);
				strncpy(uiconf->inputPath,p,UF_MAX_PATH);
				g_free(p);
				
				gtk_widget_hide(GTK_WIDGET(fileChooser));
				nufraw_ui(uf, uiconf, FALSE, NULL);
				gtk_widget_show(GTK_WIDGET(fileChooser));
				
				// TODO: use settings for the next image
			}
			g_free(uf);
			g_free(filename);
		}
		g_slist_free(saveList);
	}
	
	uiconf->showHidden=hidden_enabled;
	uiconf->showPreview=preview_enabled;
	
	if (rc->darkframe != NULL) nufraw_close_darkframe(rc);
	gtk_widget_destroy(GTK_WIDGET(fileChooser));
	nufraw_message(NUFRAW_SET_PARENT, NULL);
}

