#!/usr/bin/perl

use strict;
use warnings;
use Test::More;
use Test::Warnings;
use WebService::SonarQube;
use FindBin qw/$Bin/;
use lib "$Bin/lib";
use Mech;

my $mech  = init();
my $sonar = WebService::SonarQube->new(
    url  => 'http://localhost/sonar',
    mech => $mech,
);

setup();
good_commands();
bad_command();
error_command();
urls();

done_testing();

sub setup {
    # check that trailing url slashes are removed
    my $url = 'http://localhost/sonar';
    my $sonar = WebService::SonarQube->new(
        url  => "$url/",
        mech => init(),
    );
    is $url, $sonar->url, "Trailing slash (/) is removed";
}

sub good_commands {
    $mech->data([
        'http://localhost/sonar/api/qualitygates/show?name=My+Gate' => '{"id":1,"name":"SonarQube way","conditions":[{"id":1,"metric":"blocker_violations","op":"GT","error":"0"}]}',
    ]);
    my $ans = eval { $sonar->qualitygates_show( name => "My Gate" ) };
    my $error = $@;
    ok !$error, "No error show quality gate" or diag $error;
    is_deeply $ans, {id => 1, name => "SonarQube way", conditions => [{id => 1, metric => "blocker_violations", op => "GT", error => "0"}]};

    $mech->data([
        'http://localhost/sonar/api/qualitygates/show?name=My+Gate' => '',
    ]);
    $ans = eval { $sonar->qualitygates_show( name => "My Gate" ) };
    $error = $@;
    ok !$error, "No error show quality gate" or diag $error;
    is_deeply $ans, {};

    $mech->data([
        'http://localhost/sonar/api/qualitygates/select' => '',
    ]);
    $sonar->username('admin');
    $ans = eval { $sonar->qualitygates_select( gateId => 1, projectId => 1 ) };
    $error = $@;
    ok !$error, "No error show quality gate" or diag $error;
    is_deeply $ans, {};

    $mech->data([
        'http://admin:admin@localhost/sonar/api/qualitygates/select' => '{}',
    ]);
    $sonar->password('admin');
    $ans = eval { $sonar->qualitygates_select( gateId => 1, projectId => 1 ) };
    $error = $@;
    ok !$error, "No error show quality gate" or diag $error;
    is_deeply $ans, {};

}

sub bad_command {
    my $ans = eval { $sonar->non_existant() };
    my $error = $@;
    like $error, qr{Unknown command non/existant for SonarQube 4.5.4}, "Error when trying random method"
        or diag explain $ans, $error;
}

sub error_command {
    $mech->data([
        'http://admin:admin@localhost/sonar/api/qualitygates/select' => sub {
            die "Return error";
        },
    ]);
    my $ans = eval { $sonar->qualitygates_select() };
    my $error = $@;
    like $error, qr{Errored trying WebService::SonarQube::qualitygates_select[(][)]}, "Error from server"
        or diag explain $ans, $error;
}

sub urls {
    is WebService::SonarQube::_url_encode('@'), '%40', 'Encode @ correctly';
}

my $ws;
sub init {
    my $mech = Mech->new;
    if ( !$ws ) {
        $ws = <DATA>;
    }

    $mech->data([
        'http://localhost/sonar/api/webservices/list?include_internals=true' => $ws,
        'http://localhost/sonar/api/server/index' => '{"id":"20150512151348","version":"4.5.4","status":"UP"}',
    ]);
    return $mech;
}

# Test end points taken from SonarQube 4.5.4
__DATA__
{"webServices":[{"path":"api/action_plans","description":"Action plans management","actions":[{"key":"close","description":"Close an action plan. Requires Administer permission on project","since":"3.6","internal":false,"post":true,"hasResponseExample":false,"params":[{"key":"format","description":"Response format can be set through:<ul><li>Parameter format: xml | json</li><li>Or the 'Accept' property in the HTTP header:<ul><li>Accept:text/xml</li><li>Accept:application/json</li></ul></li></ul>If nothing is set, json is used","required":false,"possibleValues":["json","xml"]},{"key":"key","description":"Key of the action plan","required":true,"exampleValue":"3f19de90-1521-4482-a737-a311758ff513"}]},{"key":"create","description":"Create an action plan. Requires Administer permission on project","since":"3.6","internal":false,"post":true,"hasResponseExample":false,"params":[{"key":"deadLine","description":"Due date of the action plan. Format: YYYY-MM-DD","required":false,"exampleValue":"2013-12-31"},{"key":"description","description":"Description of the action plan","required":false,"exampleValue":"Version 3.6"},{"key":"format","description":"Response format can be set through:<ul><li>Parameter format: xml | json</li><li>Or the 'Accept' property in the HTTP header:<ul><li>Accept:text/xml</li><li>Accept:application/json</li></ul></li></ul>If nothing is set, json is used","required":false,"possibleValues":["json","xml"]},{"key":"name","description":"Name of the action plan","required":true,"exampleValue":"Version 3.6"},{"key":"project","description":"Project key","required":true,"exampleValue":"org.codehaus.sonar:sonar"}]},{"key":"delete","description":"Delete an action plan. Requires Administer permission on project","since":"3.6","internal":false,"post":true,"hasResponseExample":false,"params":[{"key":"format","description":"Response format can be set through:<ul><li>Parameter format: xml | json</li><li>Or the 'Accept' property in the HTTP header:<ul><li>Accept:text/xml</li><li>Accept:application/json</li></ul></li></ul>If nothing is set, json is used","required":false,"possibleValues":["json","xml"]},{"key":"key","description":"Key of the action plan","required":true,"exampleValue":"3f19de90-1521-4482-a737-a311758ff513"}]},{"key":"open","description":"Open an action plan. Requires Administer permission on project","since":"3.6","internal":false,"post":true,"hasResponseExample":false,"params":[{"key":"format","description":"Response format can be set through:<ul><li>Parameter format: xml | json</li><li>Or the 'Accept' property in the HTTP header:<ul><li>Accept:text/xml</li><li>Accept:application/json</li></ul></li></ul>If nothing is set, json is used","required":false,"possibleValues":["json","xml"]},{"key":"key","description":"Key of the action plan","required":true,"exampleValue":"3f19de90-1521-4482-a737-a311758ff513"}]},{"key":"search","description":"Get a list of action plans. Requires Browse permission on project","since":"3.6","internal":false,"post":false,"hasResponseExample":true,"params":[{"key":"format","description":"Response format can be set through:<ul><li>Parameter format: xml | json</li><li>Or the 'Accept' property in the HTTP header:<ul><li>Accept:text/xml</li><li>Accept:application/json</li></ul></li></ul>If nothing is set, json is used","required":false,"possibleValues":["json","xml"]},{"key":"project","description":"Project key","required":true,"exampleValue":"org.codehaus.sonar:sonar"}]},{"key":"update","description":"Update an action plan. Requires Administer permission on project","since":"3.6","internal":false,"post":true,"hasResponseExample":false,"params":[{"key":"deadLine","description":"Due date of the action plan. Format: YYYY-MM-DD","required":false,"exampleValue":"2013-12-31"},{"key":"description","description":"Description of the action plan","required":false,"exampleValue":"Version 3.6"},{"key":"format","description":"Response format can be set through:<ul><li>Parameter format: xml | json</li><li>Or the 'Accept' property in the HTTP header:<ul><li>Accept:text/xml</li><li>Accept:application/json</li></ul></li></ul>If nothing is set, json is used","required":false,"possibleValues":["json","xml"]},{"key":"key","description":"Key of the action plan","required":true,"exampleValue":"3f19de90-1521-4482-a737-a311758ff513"},{"key":"name","description":"Name of the action plan","required":true,"exampleValue":"Version 3.6"}]}]},{"path":"api/activities","description":"Tracking of activities","actions":[{"key":"search","description":"Search for activities","since":"4.4","internal":true,"post":false,"hasResponseExample":false,"params":[{"key":"f","description":"Comma-separated list of the fields to be returned in response. All the fields are returned by default.","required":false,"exampleValue":"type,action","possibleValues":["type","action","createdAt","login","message","details"]},{"key":"p","description":"1-based page number","required":false,"defaultValue":"1","exampleValue":"42"},{"key":"ps","description":"Page size. Must be greater than 0.","required":false,"defaultValue":"10","exampleValue":"20"},{"key":"type","description":"Types of activities to search","required":false,"defaultValue":"NONE,QPROFILE,SERVER","possibleValues":["NONE","QPROFILE","SERVER"]}]}]},{"path":"api/authentication","description":"Check authentication credentials","actions":[{"key":"validate","description":"Check credentials","since":"3.3","internal":false,"post":false,"hasResponseExample":true,"params":[{"key":"format","description":"Response format can be set through:<ul><li>Parameter format: xml | json</li><li>Or the 'Accept' property in the HTTP header:<ul><li>Accept:text/xml</li><li>Accept:application/json</li></ul></li></ul>If nothing is set, json is used","required":false,"possibleValues":["json","xml"]}]}]},{"path":"api/components","since":"4.2","description":"Components management","actions":[{"key":"app","description":"Coverage data required for rendering the component viewer","since":"4.4","internal":true,"post":false,"hasResponseExample":true,"params":[{"key":"key","description":"File key","required":true,"exampleValue":"org.codehaus.sonar:sonar-plugin-api:src/main/java/org/sonar/api/Plugin.java"},{"key":"period","description":"Period index in order to get differential measures","required":false,"possibleValues":["1","2","3","4","5"]}]},{"key":"suggestions","description":"Internal WS for the top-right search engine","since":"4.2","internal":true,"post":false,"hasResponseExample":true,"params":[{"key":"format","description":"Only json response format is available","required":false,"possibleValues":["json"]},{"key":"s","description":"Substring of project key (minimum 2 characters)","required":true,"exampleValue":"sonar"}]}]},{"path":"api/coverage","since":"4.4","description":"Display coverage information","actions":[{"key":"show","description":"Get code coverage. Require Browse permission on file's project<br/>Each element of the result array is composed of:<ol><li>Line number</li><li>Is the line covered?</li><li>Number of tests covering this line</li><li>Number of branches</li><li>Number of branches covered</li></ol>","since":"4.4","internal":false,"post":false,"hasResponseExample":true,"params":[{"key":"from","description":"First line to return. Starts at 1","required":false,"defaultValue":"1","exampleValue":"10"},{"key":"key","description":"File key","required":true,"exampleValue":"my_project:/src/foo/Bar.php"},{"key":"to","description":"Last line to return (inclusive)","required":false,"exampleValue":"20"},{"key":"type","description":"Type of coverage info to return :<ul><li>UT : Unit Tests</li><li>IT : Integration Tests</li><li>OVERALL : Unit and Integration Tests</li></ul>","required":false,"defaultValue":"UT","possibleValues":["UT","IT","OVERALL"]}]}]},{"path":"api/duplications","since":"4.4","description":"Display duplications information","actions":[{"key":"show","description":"Get duplications. Require Browse permission on file's project","since":"4.4","internal":false,"post":false,"hasResponseExample":true,"params":[{"key":"key","description":"File key","required":true,"exampleValue":"my_project:/src/foo/Bar.php"}]}]},{"path":"api/events","since":"2.6","description":"Project's events management","actions":[{"key":"index","description":"Documentation of this web service is available <a href=\"http://redirect.sonarsource.com/doc/old-web-service-api.html\">here</a>","since":"2.6","internal":false,"post":false,"hasResponseExample":false}]},{"path":"api/favorites","since":"2.6","description":"User's favorites management","actions":[{"key":"index","description":"Documentation of this web service is available <a href=\"http://redirect.sonarsource.com/doc/old-web-service-api.html\">here</a>","since":"2.6","internal":false,"post":false,"hasResponseExample":false}]},{"path":"api/issue_filters","since":"4.2","description":"Issue Filters management","actions":[{"key":"app","description":"Data required for rendering the page 'Issues'","since":"4.2","internal":true,"post":false,"hasResponseExample":true,"params":[{"key":"id","description":"Optionally, the ID of the current filter","required":false}]},{"key":"favorites","description":"The issue filters marked as favorite by request user","since":"4.2","internal":false,"post":false,"hasResponseExample":false},{"key":"show","description":"Get detail of an issue filter. Requires to be authenticated","since":"4.2","internal":false,"post":false,"hasResponseExample":true,"params":[{"key":"id","description":"ID of the issue filter","required":true}]}]},{"path":"api/issues","since":"3.6","description":"Coding rule issues","actions":[{"key":"add_comment","description":"Add a comment. Requires authentication and Browse permission on project","since":"3.6","internal":false,"post":true,"hasResponseExample":false,"params":[{"key":"format","description":"Response format can be set through:<ul><li>Parameter format: xml | json</li><li>Or the 'Accept' property in the HTTP header:<ul><li>Accept:text/xml</li><li>Accept:application/json</li></ul></li></ul>If nothing is set, json is used","required":false,"possibleValues":["json","xml"]},{"key":"issue","description":"Key of the issue","required":true,"exampleValue":"5bccd6e8-f525-43a2-8d76-fcb13dde79ef"},{"key":"text","description":"Comment","required":false,"exampleValue":"blabla..."}]},{"key":"assign","description":"Assign/Unassign an issue. Requires authentication and Browse permission on project","since":"3.6","internal":false,"post":true,"hasResponseExample":false,"params":[{"key":"assignee","description":"Login of the assignee","required":false,"exampleValue":"admin"},{"key":"format","description":"Response format can be set through:<ul><li>Parameter format: xml | json</li><li>Or the 'Accept' property in the HTTP header:<ul><li>Accept:text/xml</li><li>Accept:application/json</li></ul></li></ul>If nothing is set, json is used","required":false,"possibleValues":["json","xml"]},{"key":"issue","description":"Key of the issue","required":true,"exampleValue":"5bccd6e8-f525-43a2-8d76-fcb13dde79ef"}]},{"key":"bulk_change","description":"Bulk change on issues. Requires authentication and User role on project(s)","since":"3.7","internal":false,"post":true,"hasResponseExample":false,"params":[{"key":"actions","description":"Comma-separated list of actions to perform. Possible values: assign | set_severity | plan | do_transition","required":true,"exampleValue":"assign,plan"},{"key":"assign.assignee","description":"To assign the list of issues to a specific user (login), or unassign all the issues","required":false,"exampleValue":"sbrandhof"},{"key":"comment","description":"To add a comment to a list of issues","required":false,"exampleValue":"blabla..."},{"key":"do_transition.transition","description":"Transition","required":false,"exampleValue":"reopen","possibleValues":["confirm","unconfirm","reopen","resolve","falsepositive","close"]},{"key":"format","description":"Response format can be set through:<ul><li>Parameter format: xml | json</li><li>Or the 'Accept' property in the HTTP header:<ul><li>Accept:text/xml</li><li>Accept:application/json</li></ul></li></ul>If nothing is set, json is used","required":false,"possibleValues":["json","xml"]},{"key":"issues","description":"Comma-separated list of issue keys","required":true,"exampleValue":"01fc972e-2a3c-433e-bcae-0bd7f88f5123,01fc972e-2a3c-433e-bcae-0bd7f88f9999"},{"key":"plan.plan","description":"To plan the list of issues to a specific action plan (key), or unlink all the issues from an action plan","required":false,"exampleValue":"3f19de90-1521-4482-a737-a311758ff513"},{"key":"sendNotifications","description":"Available since version 4.0","required":false,"defaultValue":"false","possibleValues":["true","false"]},{"key":"set_severity.severity","description":"To change the severity of the list of issues","required":false,"exampleValue":"BLOCKER","possibleValues":["INFO","MINOR","MAJOR","CRITICAL","BLOCKER"]}]},{"key":"changelog","description":"Display changelog of an issue","since":"4.1","internal":false,"post":false,"hasResponseExample":true,"params":[{"key":"format","description":"Response format can be set through:<ul><li>Parameter format: xml | json</li><li>Or the 'Accept' property in the HTTP header:<ul><li>Accept:text/xml</li><li>Accept:application/json</li></ul></li></ul>If nothing is set, json is used","required":false,"possibleValues":["json","xml"]},{"key":"issue","description":"Key of the issue","required":true,"exampleValue":"5bccd6e8-f525-43a2-8d76-fcb13dde79ef"}]},{"key":"create","description":"Create a manual issue. Requires authentication and Browse permission on project","since":"3.6","internal":false,"post":true,"hasResponseExample":false,"params":[{"key":"component","description":"Key of the component on which to log the issue","required":true,"exampleValue":"org.apache.struts:struts:org.apache.struts.Action"},{"key":"format","description":"Response format can be set through:<ul><li>Parameter format: xml | json</li><li>Or the 'Accept' property in the HTTP header:<ul><li>Accept:text/xml</li><li>Accept:application/json</li></ul></li></ul>If nothing is set, json is used","required":false,"possibleValues":["json","xml"]},{"key":"line","description":"Line on which to log the issue. If no line is specified, the issue is attached to the component and not to a specific line","required":false,"exampleValue":"15"},{"key":"message","description":"Description of the issue","required":false,"exampleValue":"blabla..."},{"key":"rule","description":"Manual rule key","required":true,"exampleValue":"manual:performance"},{"key":"severity","description":"Severity of the issue","required":false,"exampleValue":"BLOCKER,CRITICAL","possibleValues":["INFO","MINOR","MAJOR","CRITICAL","BLOCKER"]}]},{"key":"delete_comment","description":"Delete a comment. Requires authentication and Browse permission on project","since":"3.6","internal":false,"post":true,"hasResponseExample":false,"params":[{"key":"key","description":"Key of the comment","required":true,"exampleValue":"392160d3-a4f2-4c52-a565-e4542cfa2096"}]},{"key":"do_action","description":"Do workflow transition on an issue. Requires authentication and Browse permission on project","since":"3.6","internal":false,"post":true,"hasResponseExample":false,"params":[{"key":"actionKey","description":"Action to perform","required":false,"exampleValue":"link-to-jira"},{"key":"format","description":"Response format can be set through:<ul><li>Parameter format: xml | json</li><li>Or the 'Accept' property in the HTTP header:<ul><li>Accept:text/xml</li><li>Accept:application/json</li></ul></li></ul>If nothing is set, json is used","required":false,"possibleValues":["json","xml"]},{"key":"issue","description":"Key of the issue","required":true,"exampleValue":"5bccd6e8-f525-43a2-8d76-fcb13dde79ef"}]},{"key":"do_transition","description":"Do workflow transition on an issue. Requires authentication and Browse permission on project","since":"3.6","internal":false,"post":true,"hasResponseExample":false,"params":[{"key":"format","description":"Response format can be set through:<ul><li>Parameter format: xml | json</li><li>Or the 'Accept' property in the HTTP header:<ul><li>Accept:text/xml</li><li>Accept:application/json</li></ul></li></ul>If nothing is set, json is used","required":false,"possibleValues":["json","xml"]},{"key":"issue","description":"Key of the issue","required":true,"exampleValue":"5bccd6e8-f525-43a2-8d76-fcb13dde79ef"},{"key":"transition","description":"Transition","required":false,"exampleValue":"reopen","possibleValues":["confirm","unconfirm","reopen","resolve","falsepositive","close"]}]},{"key":"edit_comment","description":"Edit a comment. Requires authentication and User role on project","since":"3.6","internal":false,"post":true,"hasResponseExample":false,"params":[{"key":"format","description":"Response format can be set through:<ul><li>Parameter format: xml | json</li><li>Or the 'Accept' property in the HTTP header:<ul><li>Accept:text/xml</li><li>Accept:application/json</li></ul></li></ul>If nothing is set, json is used","required":false,"possibleValues":["json","xml"]},{"key":"key","description":"Key of the comment","required":true,"exampleValue":"392160d3-a4f2-4c52-a565-e4542cfa2096"},{"key":"text","description":"New comment","required":false,"exampleValue":"blabla2..."}]},{"key":"plan","description":"Plan/Unplan an issue. Requires authentication and Browse permission on project","since":"3.6","internal":false,"post":true,"hasResponseExample":false,"params":[{"key":"format","description":"Response format can be set through:<ul><li>Parameter format: xml | json</li><li>Or the 'Accept' property in the HTTP header:<ul><li>Accept:text/xml</li><li>Accept:application/json</li></ul></li></ul>If nothing is set, json is used","required":false,"possibleValues":["json","xml"]},{"key":"issue","description":"Key of the issue","required":true,"exampleValue":"5bccd6e8-f525-43a2-8d76-fcb13dde79ef"},{"key":"plan","description":"Key of the action plan","required":false,"exampleValue":"3f19de90-1521-4482-a737-a311758ff513"}]},{"key":"search","description":"Get a list of issues. If the number of issues is greater than 10,000, only the first 10,000 ones are returned by the web service. Requires Browse permission on project(s)","since":"3.6","internal":false,"post":false,"hasResponseExample":true,"params":[{"key":"actionPlans","description":"Comma-separated list of action plan keys (not names)","required":false,"exampleValue":"3f19de90-1521-4482-a737-a311758ff513"},{"key":"asc","description":"Ascending sort","required":false,"possibleValues":["true","false"]},{"key":"assigned","description":"To retrieve assigned or unassigned issues","required":false,"possibleValues":["true","false"]},{"key":"assignees","description":"Comma-separated list of assignee logins","required":false,"exampleValue":"admin,usera"},{"key":"componentRoots","description":"To retrieve issues associated to a specific list of components and their sub-components (comma-separated list of component keys). Views are not supported","required":false,"exampleValue":"org.apache.struts:struts"},{"key":"components","description":"To retrieve issues associated to a specific list of components (comma-separated list of component keys). Note that if you set the value to a project key, only issues associated to this project are retrieved. Issues associated to its sub-components (such as files, packages, etc.) are not retrieved. See also componentRoots","required":false,"exampleValue":"org.apache.struts:struts:org.apache.struts.Action"},{"key":"createdAfter","description":"To retrieve issues created after the given date (inclusive). Format: date or datetime ISO formats","required":false,"exampleValue":"2013-05-01 (or 2013-05-01T13:00:00+0100)"},{"key":"createdAt","description":"To retrieve issues created at a given date. Format: date or datetime ISO formats","required":false,"exampleValue":"2013-05-01 (or 2013-05-01T13:00:00+0100)"},{"key":"createdBefore","description":"To retrieve issues created before the given date (exclusive). Format: date or datetime ISO formats","required":false,"exampleValue":"2013-05-01 (or 2013-05-01T13:00:00+0100)"},{"key":"extra_fields","description":"Add some extra fields on each issue. Available since 4.4","required":false,"possibleValues":["actions","transitions","assigneeName","reporterName","actionPlanName"]},{"key":"format","description":"Only json format is available. This parameter is kept only for backward compatibility and shouldn't be used anymore","required":false},{"key":"hideRules","description":"To not return rules","required":false,"possibleValues":["true","false"]},{"key":"issues","description":"Comma-separated list of issue keys","required":false,"exampleValue":"5bccd6e8-f525-43a2-8d76-fcb13dde79ef"},{"key":"languages","description":"Comma-separated list of languages. Available since 4.4","required":false,"exampleValue":"java,js"},{"key":"pageIndex","description":"Index of the selected page","required":false,"defaultValue":"1","exampleValue":"2"},{"key":"pageSize","description":"Maximum number of results per page. Default value: 100 (except when the 'components' parameter is set, value is set to \"-1\" in this case). If set to \"-1\", the max possible value is used","required":false,"exampleValue":"50"},{"key":"planned","description":"To retrieve issues associated to an action plan or not","required":false,"possibleValues":["true","false"]},{"key":"reporters","description":"Comma-separated list of reporter logins","required":false,"exampleValue":"admin"},{"key":"resolutions","description":"Comma-separated list of resolutions","required":false,"exampleValue":"FIXED,REMOVED","possibleValues":["FALSE-POSITIVE","FIXED","REMOVED"]},{"key":"resolved","description":"To match resolved or unresolved issues","required":false,"possibleValues":["true","false"]},{"key":"rules","description":"Comma-separated list of coding rule keys. Format is <repository>:<rule>","required":false,"exampleValue":"squid:AvoidCycles"},{"key":"severities","description":"Comma-separated list of severities","required":false,"exampleValue":"BLOCKER,CRITICAL","possibleValues":["INFO","MINOR","MAJOR","CRITICAL","BLOCKER"]},{"key":"sort","description":"Sort field","required":false,"possibleValues":["CREATION_DATE","UPDATE_DATE","CLOSE_DATE","ASSIGNEE","SEVERITY","STATUS"]},{"key":"statuses","description":"Comma-separated list of statuses","required":false,"exampleValue":"OPEN,REOPENED","possibleValues":["OPEN","CONFIRMED","REOPENED","RESOLVED","CLOSED"]}]},{"key":"set_severity","description":"Change severity. Requires authentication and Browse permission on project","since":"3.6","internal":false,"post":true,"hasResponseExample":false,"params":[{"key":"format","description":"Response format can be set through:<ul><li>Parameter format: xml | json</li><li>Or the 'Accept' property in the HTTP header:<ul><li>Accept:text/xml</li><li>Accept:application/json</li></ul></li></ul>If nothing is set, json is used","required":false,"possibleValues":["json","xml"]},{"key":"issue","description":"Key of the issue","required":true,"exampleValue":"5bccd6e8-f525-43a2-8d76-fcb13dde79ef"},{"key":"severity","description":"New severity","required":false,"exampleValue":"BLOCKER","possibleValues":["INFO","MINOR","MAJOR","CRITICAL","BLOCKER"]}]},{"key":"show","description":"Detail of issue","since":"4.2","internal":true,"post":false,"hasResponseExample":true,"params":[{"key":"key","description":"Issue key","required":false,"exampleValue":"5bccd6e8-f525-43a2-8d76-fcb13dde79ef"}]},{"key":"transitions","description":"Get Possible Workflow Transitions for an Issue. Requires Browse permission on project","since":"3.6","internal":false,"post":false,"hasResponseExample":true,"params":[{"key":"issue","description":"Key of the issue","required":true,"exampleValue":"5bccd6e8-f525-43a2-8d76-fcb13dde79ef"}]}]},{"path":"api/l10n","since":"4.4","description":"Localization","actions":[{"key":"index","description":"Get all localization messages for a given locale","since":"4.4","internal":true,"post":false,"hasResponseExample":false,"params":[{"key":"locale","description":"BCP47 language tag, used to override the browser Accept-Language header","required":false,"exampleValue":"fr-CH"},{"key":"ts","description":"UTC timestamp of the last cache update","required":false,"exampleValue":"2014-06-04T09:31:42Z"}]}]},{"path":"api/manual_measures","since":"2.10","description":"Manual measures management","actions":[{"key":"index","description":"Documentation of this web service is available <a href=\"http://redirect.sonarsource.com/doc/old-web-service-api.html\">here</a>","since":"2.10","internal":false,"post":false,"hasResponseExample":false}]},{"path":"api/metrics","since":"2.6","description":"Metrics management","actions":[{"key":"index","description":"Documentation of this web service is available <a href=\"http://redirect.sonarsource.com/doc/old-web-service-api.html\">here</a>","since":"2.6","internal":false,"post":false,"hasResponseExample":false}]},{"path":"api/permissions","since":"3.7","description":"Permissions","actions":[{"key":"add","description":"Add a global or a project permission. Requires Administer System permission for global permissions, requires Administer permission on project for project permissions","since":"3.7","internal":false,"post":true,"hasResponseExample":false,"params":[{"key":"component","description":"Key of the project. Required if a project permission is set. Available since version 4.0","required":false,"exampleValue":"org.codehaus.sonar"},{"key":"format","description":"Response format can be set through:<ul><li>Parameter format: xml | json</li><li>Or the 'Accept' property in the HTTP header:<ul><li>Accept:text/xml</li><li>Accept:application/json</li></ul></li></ul>If nothing is set, json is used","required":false,"possibleValues":["json","xml"]},{"key":"group","description":"Group name or \"Anyone\". Required if user is not set","required":false,"exampleValue":"sonar-administrators"},{"key":"permission","description":"Key of the permission to add","required":true,"exampleValue":"shareDashboard","possibleValues":["admin","profileadmin","shareDashboard","scan","dryRunScan","provisioning","user","issueadmin","codeviewer"]},{"key":"user","description":"User login. Required if group is not set","required":false,"exampleValue":"myuser"}]},{"key":"remove","description":"Remove a global or a project permission. Requires Administer System permission for global permissions, requires Administer permission on project for project permissions","since":"3.7","internal":false,"post":true,"hasResponseExample":false,"params":[{"key":"component","description":"Key of the project. Required if a project permission is set. Available since version 4.0","required":false,"exampleValue":"org.codehaus.sonar"},{"key":"format","description":"Response format can be set through:<ul><li>Parameter format: xml | json</li><li>Or the 'Accept' property in the HTTP header:<ul><li>Accept:text/xml</li><li>Accept:application/json</li></ul></li></ul>If nothing is set, json is used","required":false,"possibleValues":["json","xml"]},{"key":"group","description":"Group name or \"Anyone\". Required if user is not set","required":false,"exampleValue":"sonar-administrators"},{"key":"permission","description":"Key of the permission to remove","required":true,"exampleValue":"shareDashboard","possibleValues":["admin","profileadmin","shareDashboard","scan","dryRunScan","provisioning","user","issueadmin","codeviewer"]},{"key":"user","description":"User login. Required if group is not set","required":false,"exampleValue":"myuser"}]}]},{"path":"api/profiles","description":"Former quality profiles web service","actions":[{"key":"backup","description":"Backup a quality profile. Requires Administer Quality Profiles permission","since":"3.1","internal":false,"post":true,"hasResponseExample":false,"params":[{"key":"format","description":"Response format can be set through:<ul><li>Parameter format: xml | json</li><li>Or the 'Accept' property in the HTTP header:<ul><li>Accept:text/xml</li><li>Accept:application/json</li></ul></li></ul>If nothing is set, json is used","required":false,"possibleValues":["json","xml"]},{"key":"language","description":"Profile language","required":true,"exampleValue":"java"},{"key":"name","description":"Profile name. If not set, the default profile for the selected language is used","required":false,"exampleValue":"Sonar way"}]},{"key":"destroy","description":"Delete a quality profile. Requires Administer Quality Profiles permission","since":"3.3","internal":false,"post":true,"hasResponseExample":false,"params":[{"key":"language","description":"Profile language","required":true,"exampleValue":"java"},{"key":"name","description":"Profile name","required":true,"exampleValue":"Sonar way"}]},{"key":"index","description":"Get a profile","since":"3.3","internal":false,"post":false,"hasResponseExample":true,"params":[{"key":"format","description":"Response format can be set through:<ul><li>Parameter format: xml | json</li><li>Or the 'Accept' property in the HTTP header:<ul><li>Accept:text/xml</li><li>Accept:application/json</li></ul></li></ul>If nothing is set, json is used","required":false,"possibleValues":["json","xml"]},{"key":"language","description":"Profile language","required":true,"exampleValue":"java"},{"key":"name","description":"Profile name. If no profile name is given, default profile for the given language will be returned","required":false,"exampleValue":"Sonar way"}]},{"key":"list","description":"Get a list of profiles","since":"3.3","internal":false,"post":false,"hasResponseExample":true,"params":[{"key":"format","description":"Only json response format is available","required":false,"possibleValues":["json"]},{"key":"language","description":"Profile language","required":true,"exampleValue":"java"},{"key":"project","description":"Project key or id","required":false,"exampleValue":"org.codehaus.sonar:sonar"}]},{"key":"restore","description":"Restore a quality profile backup. Requires Administer Quality Profiles permission","since":"3.1","internal":false,"post":true,"hasResponseExample":false,"params":[{"key":"backup","description":"Path to the file containing the backup (HTML format)","required":true},{"key":"format","description":"Only json response format is available","required":false,"possibleValues":["json"]}]},{"key":"set_as_default","description":"Set a quality profile as Default. Requires Administer Quality Profiles permission","since":"3.3","internal":false,"post":true,"hasResponseExample":false,"params":[{"key":"language","description":"Profile language","required":true,"exampleValue":"java"},{"key":"name","description":"Profile name","required":true,"exampleValue":"Sonar way"}]}]},{"path":"api/projects","since":"2.10","description":"Projects management","actions":[{"key":"create","description":"Provision a project. Requires Provision Projects permission","since":"4.0","internal":false,"post":true,"hasResponseExample":true,"params":[{"key":"format","description":"Response format can be set through:<ul><li>Parameter format: xml | json</li><li>Or the 'Accept' property in the HTTP header:<ul><li>Accept:text/xml</li><li>Accept:application/json</li></ul></li></ul>If nothing is set, json is used","required":false,"possibleValues":["json","xml"]},{"key":"key","description":"Key of the project","required":true,"exampleValue":"org.codehaus.sonar:sonar"},{"key":"name","description":"Name of the project","required":true,"exampleValue":"SonarQube"}]},{"key":"destroy","description":"Delete a project. Requires Administer System permission","since":"2.11","internal":false,"post":true,"hasResponseExample":false,"params":[{"key":"id","description":"id or key of the resource (ie: component)","required":true,"exampleValue":"org.codehaus.sonar:sonar"}]},{"key":"index","description":"Search for projects","since":"2.10","internal":false,"post":false,"hasResponseExample":true,"params":[{"key":"desc","description":"Load project description","required":false,"defaultValue":"true","possibleValues":["true","false"]},{"key":"format","description":"Response format can be set through:<ul><li>Parameter format: xml | json</li><li>Or the 'Accept' property in the HTTP header:<ul><li>Accept:text/xml</li><li>Accept:application/json</li></ul></li></ul>If nothing is set, json is used","required":false,"possibleValues":["json","xml"]},{"key":"key","description":"id or key of the project","required":false,"exampleValue":"org.codehaus.sonar:sonar"},{"key":"libs","description":"Load libraries. Ignored if the parameter key is set","required":false,"defaultValue":"false","possibleValues":["true","false"]},{"key":"search","description":"Substring of project name, case insensitive","required":false,"exampleValue":"Sonar"},{"key":"subprojects","description":"Load sub-projects. Ignored if the parameter key is set","required":false,"defaultValue":"false","possibleValues":["true","false"]},{"key":"versions","description":"Load version","required":false,"defaultValue":"false","possibleValues":["true","false","last"]},{"key":"views","description":"Load views and sub-views. Ignored if the parameter key is set","required":false,"defaultValue":"false","possibleValues":["true","false"]}]}]},{"path":"api/properties","since":"2.6","description":"Properties management","actions":[{"key":"index","description":"Documentation of this web service is available <a href=\"http://redirect.sonarsource.com/doc/old-web-service-api.html\">here</a>","since":"2.6","internal":false,"post":false,"hasResponseExample":false}]},{"path":"api/qualitygates","since":"4.3","description":"This service manages quality gates, including conditions and project association","actions":[{"key":"app","description":"Get initialization items for the admin UI. For internal use","since":"4.3","internal":true,"post":false,"hasResponseExample":false},{"key":"copy","description":"Copy a Quality Gate. Require Administer Quality Profiles and Gates permission","since":"4.3","internal":false,"post":true,"hasResponseExample":false,"params":[{"key":"id","description":"The ID of the source quality gate","required":true,"exampleValue":"1"},{"key":"name","description":"The name of the quality gate to create","required":true,"exampleValue":"My Quality Gate"}]},{"key":"create","description":"Create a Quality Gate. Require Administer Quality Profiles and Gates permission","since":"4.3","internal":false,"post":true,"hasResponseExample":false,"params":[{"key":"name","description":"The name of the quality gate to create","required":true,"exampleValue":"My Quality Gate"}]},{"key":"create_condition","description":"Add a new condition to a quality gate. Require Administer Quality Profiles and Gates permission","since":"4.3","internal":false,"post":true,"hasResponseExample":false,"params":[{"key":"error","description":"Condition error threshold","required":false,"exampleValue":"10"},{"key":"gateId","description":"ID of the quality gate","required":true,"exampleValue":"1"},{"key":"metric","description":"Condition metric","required":true,"exampleValue":"blocker_violations"},{"key":"op","description":"Condition operator:<br/><ul><li>EQ = equals</li><li>NE = is not</li><li>LT = is lower than</li><li>GT = is greater than</li></ui>","required":false,"exampleValue":"EQ","possibleValues":["LT","GT","EQ","NE"]},{"key":"period","description":"Condition period","required":false,"exampleValue":"1"},{"key":"warning","description":"Condition warning threshold","required":false,"exampleValue":"5"}]},{"key":"delete_condition","description":"Delete a condition from a quality gate. Require Administer Quality Profiles and Gates permission","since":"4.3","internal":false,"post":true,"hasResponseExample":false,"params":[{"key":"id","description":"Condition ID","required":true,"exampleValue":"2"}]},{"key":"deselect","description":"Remove the association of a project from a quality gate. Require Administer Quality Profiles and Gates permission","since":"4.3","internal":false,"post":true,"hasResponseExample":false,"params":[{"key":"gateId","description":"Quality Gate ID","required":true,"exampleValue":"1"},{"key":"projectId","description":"Project ID","required":true,"exampleValue":"12"}]},{"key":"destroy","description":"Delete a Quality Gate. Require Administer Quality Profiles and Gates permission","since":"4.3","internal":false,"post":true,"hasResponseExample":false,"params":[{"key":"id","description":"ID of the quality gate to delete","required":true,"exampleValue":"1"}]},{"key":"list","description":"Get a list of quality gates","since":"4.3","internal":false,"post":false,"hasResponseExample":true},{"key":"rename","description":"Rename a Quality Gate. Require Administer Quality Profiles and Gates permission","since":"4.3","internal":false,"post":true,"hasResponseExample":false,"params":[{"key":"id","description":"ID of the quality gate to rename","required":true,"exampleValue":"1"},{"key":"name","description":"New name of the quality gate","required":true,"exampleValue":"My Quality Gate"}]},{"key":"search","description":"Search for projects associated (or not) to a quality gate","since":"4.3","internal":false,"post":false,"hasResponseExample":true,"params":[{"key":"gateId","description":"Quality Gate ID","required":true,"exampleValue":"1"},{"key":"page","description":"Page number","required":false,"defaultValue":"1","exampleValue":"2"},{"key":"pageSize","description":"Page size","required":false,"exampleValue":"10"},{"key":"query","description":"To search for projects containing this string. If this parameter is set, \"selected\" is set to \"all\".","required":false,"exampleValue":"abc"},{"key":"selected","description":"If \"selected\", search for projects associated to the quality gate","required":false,"defaultValue":"selected","exampleValue":"deselected","possibleValues":["all","selected","deselected"]}]},{"key":"select","description":"Associate a project to a quality gate. Require Administer Quality Profiles and Gates permission","since":"4.3","internal":false,"post":true,"hasResponseExample":false,"params":[{"key":"gateId","description":"Quality Gate ID","required":true,"exampleValue":"1"},{"key":"projectId","description":"Project ID","required":true,"exampleValue":"12"}]},{"key":"set_as_default","description":"Set a quality gate as the default quality gate. Require Administer Quality Profiles and Gates permission","since":"4.3","internal":false,"post":true,"hasResponseExample":false,"params":[{"key":"id","description":"ID of the quality gate to set as default","required":true,"exampleValue":"1"}]},{"key":"show","description":"Display the details of a quality gate","since":"4.3","internal":false,"post":false,"hasResponseExample":true,"params":[{"key":"id","description":"ID of the quality gate. Either id or name must be set","required":false,"exampleValue":"1"},{"key":"name","description":"Name of the quality gate. Either id or name must be set","required":false,"exampleValue":"My Quality Gate"}]},{"key":"unset_default","description":"Unset a quality gate as the default quality gate. Require Administer Quality Profiles and Gates permission","since":"4.3","internal":false,"post":true,"hasResponseExample":false,"params":[{"key":"id","description":"ID of the quality gate to unset as default","required":true,"exampleValue":"1"}]},{"key":"update_condition","description":"Update a condition attached to a quality gate. Require Administer Quality Profiles and Gates permission","since":"4.3","internal":false,"post":true,"hasResponseExample":false,"params":[{"key":"error","description":"Condition error threshold","required":false,"exampleValue":"10"},{"key":"id","description":"Condition ID","required":true,"exampleValue":"10"},{"key":"metric","description":"Condition metric","required":true,"exampleValue":"blocker_violations"},{"key":"op","description":"Condition operator:<br/><ul><li>EQ = equals</li><li>NE = is not</li><li>LT = is lower than</li><li>GT = is greater than</li></ui>","required":false,"exampleValue":"EQ","possibleValues":["LT","GT","EQ","NE"]},{"key":"period","description":"Condition period","required":false,"exampleValue":"1"},{"key":"warning","description":"Condition warning threshold","required":false,"exampleValue":"5"}]}]},{"path":"api/qualityprofiles","since":"4.4","description":"Quality Profiles","actions":[{"key":"activate_rule","description":"Activate a rule on a Quality profile","since":"4.4","internal":false,"post":true,"hasResponseExample":false,"params":[{"key":"params","description":"Parameters as semi-colon list of <key>=<value>, for example 'params=key1=v1;key2=v2'. Ignored if parameter reset is true.","required":false},{"key":"profile_key","description":"Key of Quality profile","required":true,"exampleValue":"Sonar way:java"},{"key":"reset","description":"Reset severity and parameters of activated rule. Set the values defined on parent profile or from rule default values.","required":false,"possibleValues":["true","false"]},{"key":"rule_key","description":"Key of the rule to activate","required":true,"exampleValue":"squid:AvoidCycles"},{"key":"severity","description":"Severity. Ignored if parameter reset is true.","required":false,"possibleValues":["INFO","MINOR","MAJOR","CRITICAL","BLOCKER"]}]},{"key":"activate_rules","description":"Bulk-activate rules on one or several Quality profiles","since":"4.4","internal":false,"post":true,"hasResponseExample":false,"params":[{"key":"activation","description":"Filter rules that are activated or deactivated on the selected Quality profile. Ignored if the parameter 'qprofile' is not set.","required":false,"possibleValues":["true","false"]},{"key":"activation_severity","description":"Optional severity of rules activated in bulk","required":false,"possibleValues":["INFO","MINOR","MAJOR","CRITICAL","BLOCKER"]},{"key":"active_severities","description":"Comma-separated list of activation severities, i.e the severity of rules in Quality profiles.","required":false,"exampleValue":"CRITICAL,BLOCKER","possibleValues":["INFO","MINOR","MAJOR","CRITICAL","BLOCKER"]},{"key":"asc","description":"Ascending sort","required":false,"defaultValue":"true","possibleValues":["true","false"]},{"key":"available_since","description":"Filters rules added since date. Format is yyyy-MM-dd","required":false,"exampleValue":"2014-06-22"},{"key":"debt_characteristics","description":"Comma-separated list of technical debt characteristics or sub-characteristics","required":false,"exampleValue":"RELIABILITY"},{"key":"has_debt_characteristic","description":"Filter rules that have a technical debt characteristic","required":false,"possibleValues":["true","false"]},{"key":"inheritance","description":"Value of inheritance for a rule within a quality profile Used only if the parameter 'activation' is set.","required":false,"exampleValue":"INHERITED,OVERRIDES","possibleValues":["NONE","INHERITED","OVERRIDES"]},{"key":"is_template","description":"Filter template rules","required":false,"possibleValues":["true","false"]},{"key":"languages","description":"Comma-separated list of languages","required":false,"exampleValue":"java,js"},{"key":"profile_key","description":"Quality Profile Key. To retrieve a profile key for a given language please see the api/qprofiles documentation","required":true,"exampleValue":"java:MyProfile"},{"key":"q","description":"UTF-8 search query","required":false,"exampleValue":"xpath"},{"key":"qprofile","description":"Key of Quality profile to filter on. Used only if the parameter 'activation' is set.","required":false,"exampleValue":"java:Sonar way"},{"key":"repositories","description":"Comma-separated list of repositories","required":false,"exampleValue":"checkstyle,findbugs"},{"key":"rule_key","description":"Key of rule to search for","required":false,"exampleValue":"squid:S001"},{"key":"s","description":"Sort field","required":false,"exampleValue":"name","possibleValues":["name","updatedAt","createdAt","key"]},{"key":"severities","description":"Comma-separated list of default severities. Not the same than severity of rules in Quality profiles.","required":false,"exampleValue":"CRITICAL,BLOCKER","possibleValues":["INFO","MINOR","MAJOR","CRITICAL","BLOCKER"]},{"key":"statuses","description":"Comma-separated list of status codes","required":false,"exampleValue":"READY","possibleValues":["BETA","DEPRECATED","READY","REMOVED"]},{"key":"tags","description":"Comma-separated list of tags. Returned rules match any of the tags (OR operator)","required":false,"exampleValue":"security,java8"},{"key":"template_key","description":"Key of the template rule to filter on. Used to search for the custom rules based on this template.","required":false,"exampleValue":"java:S001"}]},{"key":"deactivate_rule","description":"Deactivate a rule on a Quality profile","since":"4.4","internal":false,"post":true,"hasResponseExample":false,"params":[{"key":"profile_key","description":"Key of Quality profile","required":true,"exampleValue":"Sonar way:java"},{"key":"rule_key","description":"Key of the rule to activate","required":true,"exampleValue":"squid:AvoidCycles"}]},{"key":"deactivate_rules","description":"Bulk deactivate rules on Quality profiles","since":"4.4","internal":false,"post":true,"hasResponseExample":false,"params":[{"key":"activation","description":"Filter rules that are activated or deactivated on the selected Quality profile. Ignored if the parameter 'qprofile' is not set.","required":false,"possibleValues":["true","false"]},{"key":"active_severities","description":"Comma-separated list of activation severities, i.e the severity of rules in Quality profiles.","required":false,"exampleValue":"CRITICAL,BLOCKER","possibleValues":["INFO","MINOR","MAJOR","CRITICAL","BLOCKER"]},{"key":"asc","description":"Ascending sort","required":false,"defaultValue":"true","possibleValues":["true","false"]},{"key":"available_since","description":"Filters rules added since date. Format is yyyy-MM-dd","required":false,"exampleValue":"2014-06-22"},{"key":"debt_characteristics","description":"Comma-separated list of technical debt characteristics or sub-characteristics","required":false,"exampleValue":"RELIABILITY"},{"key":"has_debt_characteristic","description":"Filter rules that have a technical debt characteristic","required":false,"possibleValues":["true","false"]},{"key":"inheritance","description":"Value of inheritance for a rule within a quality profile Used only if the parameter 'activation' is set.","required":false,"exampleValue":"INHERITED,OVERRIDES","possibleValues":["NONE","INHERITED","OVERRIDES"]},{"key":"is_template","description":"Filter template rules","required":false,"possibleValues":["true","false"]},{"key":"languages","description":"Comma-separated list of languages","required":false,"exampleValue":"java,js"},{"key":"profile_key","description":"Quality Profile Key. To retrieve a profile key for a given language please see the api/qprofiles documentation","required":true,"exampleValue":"java:MyProfile"},{"key":"q","description":"UTF-8 search query","required":false,"exampleValue":"xpath"},{"key":"qprofile","description":"Key of Quality profile to filter on. Used only if the parameter 'activation' is set.","required":false,"exampleValue":"java:Sonar way"},{"key":"repositories","description":"Comma-separated list of repositories","required":false,"exampleValue":"checkstyle,findbugs"},{"key":"rule_key","description":"Key of rule to search for","required":false,"exampleValue":"squid:S001"},{"key":"s","description":"Sort field","required":false,"exampleValue":"name","possibleValues":["name","updatedAt","createdAt","key"]},{"key":"severities","description":"Comma-separated list of default severities. Not the same than severity of rules in Quality profiles.","required":false,"exampleValue":"CRITICAL,BLOCKER","possibleValues":["INFO","MINOR","MAJOR","CRITICAL","BLOCKER"]},{"key":"statuses","description":"Comma-separated list of status codes","required":false,"exampleValue":"READY","possibleValues":["BETA","DEPRECATED","READY","REMOVED"]},{"key":"tags","description":"Comma-separated list of tags. Returned rules match any of the tags (OR operator)","required":false,"exampleValue":"security,java8"},{"key":"template_key","description":"Key of the template rule to filter on. Used to search for the custom rules based on this template.","required":false,"exampleValue":"java:S001"}]},{"key":"restore_built_in","description":"Restore built-in profiles from the definitions declared by plugins. Missing profiles are created, existing ones are updated.","since":"4.4","internal":false,"post":true,"hasResponseExample":false,"params":[{"key":"language","description":"Restore the built-in profiles defined for this language","required":true,"exampleValue":"java"}]}]},{"path":"api/resources","since":"2.10","description":"Former components web service","actions":[{"key":"index","description":"Get a list of components. Requires Browse permission on resource","since":"2.10","internal":false,"post":false,"hasResponseExample":true,"params":[{"key":"depth","description":"Used only when resource is set:<br/><ul><li>0: only selected resource</li><li>-1: all children, including selected resource</li><li>>0: depth toward the selected resource</li></ul>","required":false,"defaultValue":"0","exampleValue":"-1"},{"key":"format","description":"Response format can be set through:<ul><li>Parameter format: xml | json</li><li>Or the 'Accept' property in the HTTP header:<ul><li>Accept:text/xml</li><li>Accept:application/json</li></ul></li></ul>If nothing is set, json is used","required":false,"possibleValues":["json","xml"]},{"key":"includealerts","description":"Include alerts data: add nodes &ltalert&gt (ERROR, WARN, OK) and &ltalert_text&gt","required":false,"defaultValue":"false","possibleValues":["true","false"]},{"key":"includetrends","description":"Include trends and period variations in response: add &lttrend&gt (1 if better, else worse), &ltvar&gt (1 if measure value increases) and nodes &ltp*&gt for period variations","required":false,"defaultValue":"false","possibleValues":["true","false"]},{"key":"limit","description":"Limit the number of results. Only used if one metric, and only one, is set","required":false,"exampleValue":"10"},{"key":"metrics","description":"Comma-separated list of <a href=\"http://redirect.sonarsource.com/doc/metric-definitions.html\">metric keys/ids</a>. Load measures on selected metrics. If only one metric is set, then measures are ordered by value","required":false,"exampleValue":"lines,blocker_violations"},{"key":"qualifiers","description":"Comma-separated list of qualifiers:<br/><ul><li>VW: view</li><li>SVW: sub-view</li><li>TRK: project</li><li>BRC: module</li><li>UTS: unit test</li><li>DIR: directory</li><li>FIL: file</li><li>DEV: developer</li></ul>","required":false,"exampleValue":"TRK,BRC"},{"key":"resource","description":"id or key of the resource","required":false,"exampleValue":"org.codehaus.sonar:sonar"},{"key":"rules","description":"Filter on rules: setting it to true will return rules id and rule name for measure having such info (such as 'blocker_violations', 'critical_violations', ..., 'new_blocker_violations', ...). Possible values: true | false | list of rule ids","required":false,"defaultValue":"false","exampleValue":"true"},{"key":"scopes","description":"Comma-separated list of scopes:<br/><ul><li>PRJ: project/module</li><li>DIR: directory (like Java package)</li><li>FIL: file</li></ul>","required":false,"exampleValue":"PRJ,DIR"},{"key":"verbose","description":"Add some data to response","required":false,"defaultValue":"false","possibleValues":["true","false"]}]},{"key":"search","description":"Search for components","since":"3.3","internal":true,"post":false,"hasResponseExample":true,"params":[{"key":"display_key","description":"Return the resource key instead of the resource id","required":false,"defaultValue":"false","possibleValues":["true","false"]},{"key":"f","description":"If 's2', then it will return a select2 compatible format","required":false,"exampleValue":"s2"},{"key":"format","description":"Only json response format is available","required":false,"possibleValues":["json"]},{"key":"p","description":"Page index","required":false,"defaultValue":"1","exampleValue":"2"},{"key":"ps","description":"Page size","required":false,"defaultValue":"10","exampleValue":"15"},{"key":"q","description":"Comma-separated list of qualifiers","required":false,"exampleValue":"TRK,BRC"},{"key":"qp","description":"Resource Property","required":false,"exampleValue":"supportsMeasureFilters"},{"key":"s","description":"To filter on resources containing a specified text in their key","required":false,"exampleValue":"sonar"}]}]},{"path":"api/rules","description":"Coding rules","actions":[{"key":"app","description":"Data required for rendering the page 'Coding Rules'","internal":true,"post":false,"hasResponseExample":false},{"key":"create","description":"Create a custom rule or a manual rule","since":"4.4","internal":false,"post":true,"hasResponseExample":false,"params":[{"key":"custom_key","description":"Key of the custom rule","required":false,"exampleValue":"Todo_should_not_be_used"},{"key":"manual_key","description":"Key of the manual rule","required":false,"exampleValue":"Error_handling"},{"key":"markdown_description","description":"Rule description","required":true,"exampleValue":"Description of my custom rule"},{"key":"name","description":"Rule name","required":true,"exampleValue":"My custom rule"},{"key":"params","description":"Parameters as semi-colon list of <key>=<value>, for example 'params=key1=v1;key2=v2' (Only for custom rule)","required":false},{"key":"prevent_reactivation","description":"If set to true and if the rule has been deactivated (status 'REMOVED'), a status 409 will be returned","required":false,"defaultValue":"false","possibleValues":["true","false"]},{"key":"severity","description":"Rule severity (Only for custom rule)","required":false,"possibleValues":["INFO","MINOR","MAJOR","CRITICAL","BLOCKER"]},{"key":"status","description":"Rule status (Only for custom rule)","required":false,"defaultValue":"READY","possibleValues":["BETA","DEPRECATED","READY","REMOVED"]},{"key":"template_key","description":"Key of the template rule in order to create a custom rule (mandatory for custom rule)","required":false,"exampleValue":"java:XPath"}]},{"key":"delete","description":"Delete custom rule or manual rule","since":"4.4","internal":false,"post":true,"hasResponseExample":false,"params":[{"key":"key","description":"Rule key","required":true,"exampleValue":"squid:XPath_1402065390816"}]},{"key":"search","description":"Search for a collection of relevant rules matching a specified query","since":"4.4","internal":false,"post":false,"hasResponseExample":true,"params":[{"key":"activation","description":"Filter rules that are activated or deactivated on the selected Quality profile. Ignored if the parameter 'qprofile' is not set.","required":false,"possibleValues":["true","false"]},{"key":"active_severities","description":"Comma-separated list of activation severities, i.e the severity of rules in Quality profiles.","required":false,"exampleValue":"CRITICAL,BLOCKER","possibleValues":["INFO","MINOR","MAJOR","CRITICAL","BLOCKER"]},{"key":"asc","description":"Ascending sort","required":false,"defaultValue":"true","possibleValues":["true","false"]},{"key":"available_since","description":"Filters rules added since date. Format is yyyy-MM-dd","required":false,"exampleValue":"2014-06-22"},{"key":"debt_characteristics","description":"Comma-separated list of technical debt characteristics or sub-characteristics","required":false,"exampleValue":"RELIABILITY"},{"key":"f","description":"Comma-separated list of the fields to be returned in response. All the fields are returned by default.","required":false,"exampleValue":"repo,name","possibleValues":["repo","name","createdAt","severity","status","internalKey","isTemplate","templateKey","tags","sysTags","lang","langName","htmlDesc","mdDesc","noteLogin","mdNote","htmlNote","defaultDebtChar","defaultDebtSubChar","debtChar","debtSubChar","debtCharName","debtSubCharName","defaultDebtRemFn","effortToFixDescription","debtOverloaded","debtRemFn","params","actives"]},{"key":"facets","description":"Compute predefined facets","required":false,"defaultValue":"false","possibleValues":["true","false"]},{"key":"has_debt_characteristic","description":"Filter rules that have a technical debt characteristic","required":false,"possibleValues":["true","false"]},{"key":"inheritance","description":"Value of inheritance for a rule within a quality profile Used only if the parameter 'activation' is set.","required":false,"exampleValue":"INHERITED,OVERRIDES","possibleValues":["NONE","INHERITED","OVERRIDES"]},{"key":"is_template","description":"Filter template rules","required":false,"possibleValues":["true","false"]},{"key":"languages","description":"Comma-separated list of languages","required":false,"exampleValue":"java,js"},{"key":"p","description":"1-based page number","required":false,"defaultValue":"1","exampleValue":"42"},{"key":"ps","description":"Page size. Must be greater than 0.","required":false,"defaultValue":"10","exampleValue":"20"},{"key":"q","description":"UTF-8 search query","required":false,"exampleValue":"xpath"},{"key":"qprofile","description":"Key of Quality profile to filter on. Used only if the parameter 'activation' is set.","required":false,"exampleValue":"java:Sonar way"},{"key":"repositories","description":"Comma-separated list of repositories","required":false,"exampleValue":"checkstyle,findbugs"},{"key":"rule_key","description":"Key of rule to search for","required":false,"exampleValue":"squid:S001"},{"key":"s","description":"Sort field","required":false,"exampleValue":"name","possibleValues":["name","updatedAt","createdAt","key"]},{"key":"severities","description":"Comma-separated list of default severities. Not the same than severity of rules in Quality profiles.","required":false,"exampleValue":"CRITICAL,BLOCKER","possibleValues":["INFO","MINOR","MAJOR","CRITICAL","BLOCKER"]},{"key":"statuses","description":"Comma-separated list of status codes","required":false,"exampleValue":"READY","possibleValues":["BETA","DEPRECATED","READY","REMOVED"]},{"key":"tags","description":"Comma-separated list of tags. Returned rules match any of the tags (OR operator)","required":false,"exampleValue":"security,java8"},{"key":"template_key","description":"Key of the template rule to filter on. Used to search for the custom rules based on this template.","required":false,"exampleValue":"java:S001"}]},{"key":"show","description":"Get detailed information about a rule","since":"4.2","internal":false,"post":false,"hasResponseExample":true,"params":[{"key":"actives","description":"Show rule's activations for all profiles (\"active rules\")","required":false,"defaultValue":"false","possibleValues":["true","false"]},{"key":"key","description":"Rule key","required":true,"exampleValue":"javascript:EmptyBlock"}]},{"key":"tags","description":"List all rule tags","since":"4.4","internal":false,"post":false,"hasResponseExample":true},{"key":"update","description":"Update an existing rule","since":"4.4","internal":false,"post":true,"hasResponseExample":false,"params":[{"key":"debt_remediation_fn_offset","required":false,"exampleValue":"1d"},{"key":"debt_remediation_fn_type","required":false,"possibleValues":["LINEAR","LINEAR_OFFSET","CONSTANT_ISSUE"]},{"key":"debt_remediation_fy_coeff","required":false,"exampleValue":"3min"},{"key":"debt_sub_characteristic","description":"Optional key of the debt sub-characteristic. Use empty value to unset (-> none) or '_default' to revert to default sub-characteristic.","required":false,"exampleValue":"FAULT_TOLERANCE"},{"key":"key","description":"Key of the rule to update","required":true,"exampleValue":"javascript:NullCheck"},{"key":"markdown_description","description":"Rule description (mandatory for custom rule and manual rule)","required":false,"exampleValue":"Description of my custom rule"},{"key":"markdown_note","description":"Optional note in markdown format. Use empty value to remove current note. Note is not changedif the parameter is not set.","required":false,"exampleValue":"my *note*"},{"key":"name","description":"Rule name (mandatory for custom rule and manual rule)","required":false,"exampleValue":"My custom rule"},{"key":"params","description":"Parameters as semi-colon list of <key>=<value>, for example 'params=key1=v1;key2=v2' (Only when updating a custom rule)","required":false},{"key":"severity","description":"Rule severity (Only when updating a custom rule)","required":false,"possibleValues":["INFO","MINOR","MAJOR","CRITICAL","BLOCKER"]},{"key":"status","description":"Rule status (Only when updating a custom rule)","required":false,"possibleValues":["BETA","DEPRECATED","READY","REMOVED"]},{"key":"tags","description":"Optional comma-separated list of tags to set. Use blank value to remove current tags. Tags are not changed if the parameter is not set.","required":false,"exampleValue":"java8,security"}]}]},{"path":"api/server","since":"2.10","description":"Get system properties and upgrade db","actions":[{"key":"index","description":"Get the server status:<ul><li>UP</li><li>DOWN (generally for database connection failures)</li><li>SETUP (if the server must be upgraded)</li><li>MIGRATION_RUNNING (the upgrade process is currently running)</li></ul>","since":"2.10","internal":false,"post":false,"hasResponseExample":true,"params":[{"key":"format","description":"Response format can be set through:<ul><li>Parameter format: xml | json</li><li>Or the 'Accept' property in the HTTP header:<ul><li>Accept:text/xml</li><li>Accept:application/json</li></ul></li></ul>If nothing is set, json is used","required":false,"possibleValues":["json","xml"]}]},{"key":"setup","description":"Upgrade the SonarQube database","since":"2.10","internal":false,"post":true,"hasResponseExample":true,"params":[{"key":"format","description":"Response format","required":false,"possibleValues":["json","csv","text"]}]},{"key":"system","description":"Get the system properties, server info (Java, OS), database configuration, JVM statistics and installed plugins. Requires Administer System permission","since":"2.10","internal":false,"post":false,"hasResponseExample":true,"params":[{"key":"format","description":"Response format can be set through:<ul><li>Parameter format: xml | json</li><li>Or the 'Accept' property in the HTTP header:<ul><li>Accept:text/xml</li><li>Accept:application/json</li></ul></li></ul>If nothing is set, json is used","required":false,"possibleValues":["json","xml"]}]}]},{"path":"api/sources","since":"4.2","description":"Display sources information","actions":[{"key":"scm","description":"Get SCM information of source files. Require Browse permission on file's project<br/>Each element of the result array is composed of:<ol><li>Line number</li><li>Author of the commit</li><li>Date of the commit</li></ol>","since":"4.4","internal":false,"post":false,"hasResponseExample":true,"params":[{"key":"commits_by_line","description":"Group lines by SCM commit if value is false, else display commits for each line, even if two consecutive lines relate to the same commit.","required":false,"defaultValue":"false","possibleValues":["true","false"]},{"key":"from","description":"First line to return. Starts at 1","required":false,"defaultValue":"1","exampleValue":"10"},{"key":"key","description":"File key","required":true,"exampleValue":"my_project:/src/foo/Bar.php"},{"key":"to","description":"Last line to return (inclusive)","required":false,"exampleValue":"20"}]},{"key":"show","description":"Get source code. Require Browse permission on file's project<br/>Each element of the result array is composed of:<ol><li>Line number</li><li>Content of the line</li></ol>","since":"4.4","internal":false,"post":false,"hasResponseExample":true,"params":[{"key":"from","description":"First line to return. Starts at 1","required":false,"defaultValue":"1","exampleValue":"10"},{"key":"key","description":"File key","required":true,"exampleValue":"my_project:/src/foo/Bar.php"},{"key":"to","description":"Last line to return (inclusive)","required":false,"exampleValue":"20"}]}]},{"path":"api/system","since":"4.3","description":"Restart server","actions":[{"key":"restart","description":"Restart server. Available only on development mode (sonar.web.dev=true), except when using Java 6 on MS Windows. Ruby on Rails extensions are not reloaded","since":"4.3","internal":false,"post":true,"hasResponseExample":false}]},{"path":"api/tests","since":"4.4","description":"Tests management","actions":[{"key":"covered_files","description":"Get the list of files covered by a test plan. Require Browse permission on file's project","since":"4.4","internal":false,"post":false,"hasResponseExample":true,"params":[{"key":"key","description":"Test plan key","required":true,"exampleValue":"my_project:/src/test/BarTest.java"},{"key":"test","description":"Test case used to list files covered by the test plan","required":true,"exampleValue":"my_test"}]},{"key":"show","description":"Get the list of test cases of a test plan. Require Browse permission on file's project","since":"4.4","internal":false,"post":false,"hasResponseExample":true,"params":[{"key":"key","description":"Test plan key","required":true,"exampleValue":"my_project:/src/test/BarTest.java"}]},{"key":"test_cases","description":"Get the list of test cases covering a given file and line. Require Browse permission on file's project","since":"4.4","internal":false,"post":false,"hasResponseExample":true,"params":[{"key":"key","description":"File key","required":true,"exampleValue":"my_project:/src/foo/Bar.php"},{"key":"line","description":"Line of the file used to get test cases","required":true,"exampleValue":"10"}]}]},{"path":"api/timemachine","since":"2.10","description":"Get a list of past measures","actions":[{"key":"index","description":"Get a list of past measures. Requires Browse permission on project","since":"2.10","internal":false,"post":false,"hasResponseExample":true,"params":[{"key":"format","description":"Response format","required":false,"possibleValues":["json","csv"]},{"key":"fromDateTime","description":"ISO-8601 datetime (inclusive)","required":false,"exampleValue":"2010-12-25T23:59:59+0100"},{"key":"metrics","description":"Comma-separated list of <a href=\"http://redirect.sonarsource.com/doc/metric-definitions.html\">metric keys/ids</a>","required":true,"exampleValue":"coverage,violations"},{"key":"resource","description":"id or key of the resource (ie: component)","required":true,"exampleValue":"org.codehaus.sonar:sonar"},{"key":"toDateTime","description":"ISO-8601 datetime (inclusive)","required":false,"exampleValue":"2010-12-25T23:59:59+0100"}]}]},{"path":"api/updatecenter","since":"2.10","description":"Get list of installed plugins","actions":[{"key":"installed_plugins","description":"Get the list of all the plugins installed on the SonarQube instance","since":"2.10","internal":false,"post":false,"hasResponseExample":true,"params":[{"key":"format","description":"Response format can be set through:<ul><li>Parameter format: xml | json</li><li>Or the 'Accept' property in the HTTP header:<ul><li>Accept:text/xml</li><li>Accept:application/json</li></ul></li></ul>If nothing is set, json is used","required":false,"possibleValues":["json","xml"]}]}]},{"path":"api/user_properties","since":"2.6","description":"User properties management","actions":[{"key":"index","description":"Documentation of this web service is available <a href=\"http://redirect.sonarsource.com/doc/old-web-service-api.html\">here</a>","since":"2.6","internal":false,"post":false,"hasResponseExample":false}]},{"path":"api/users","since":"3.6","description":"Users management","actions":[{"key":"create","description":"Create a user. Requires Administer System permission","since":"3.7","internal":false,"post":true,"hasResponseExample":false,"params":[{"key":"email","description":"User email","required":false,"exampleValue":"myname@email.com"},{"key":"format","description":"Response format can be set through:<ul><li>Parameter format: xml | json</li><li>Or the 'Accept' property in the HTTP header:<ul><li>Accept:text/xml</li><li>Accept:application/json</li></ul></li></ul>If nothing is set, json is used","required":false,"possibleValues":["json","xml"]},{"key":"login","description":"User login","required":true,"exampleValue":"myuser"},{"key":"name","description":"User name","required":true,"exampleValue":"My Name"},{"key":"password","description":"User password","required":true,"exampleValue":"mypassword"},{"key":"password_confirmation","description":"Must be the same value as \"password\"","required":true,"exampleValue":"mypassword"}]},{"key":"deactivate","description":"Deactivate a user. Requires Administer System permission","since":"3.7","internal":false,"post":true,"hasResponseExample":false,"params":[{"key":"format","description":"Response format can be set through:<ul><li>Parameter format: xml | json</li><li>Or the 'Accept' property in the HTTP header:<ul><li>Accept:text/xml</li><li>Accept:application/json</li></ul></li></ul>If nothing is set, json is used","required":false,"possibleValues":["json","xml"]},{"key":"login","description":"User login","required":true,"exampleValue":"myuser"}]},{"key":"search","description":"Get a list of users","since":"3.6","internal":false,"post":false,"hasResponseExample":true,"params":[{"key":"format","description":"Response format can be set through:<ul><li>Parameter format: xml | json</li><li>Or the 'Accept' property in the HTTP header:<ul><li>Accept:text/xml</li><li>Accept:application/json</li></ul></li></ul>If nothing is set, json is used","required":false,"possibleValues":["json","xml"]},{"key":"includeDeactivated","description":"Include deactivated users","required":false,"defaultValue":"false","possibleValues":["true","false"]},{"key":"logins","description":"Comma-separated list of user logins","required":false,"exampleValue":"admin,sbrandhof"}]},{"key":"update","description":"Update a user. Requires Administer System permission","since":"3.7","internal":false,"post":true,"hasResponseExample":false,"params":[{"key":"email","description":"User email","required":false,"exampleValue":"mynewname@email.com"},{"key":"format","description":"Response format can be set through:<ul><li>Parameter format: xml | json</li><li>Or the 'Accept' property in the HTTP header:<ul><li>Accept:text/xml</li><li>Accept:application/json</li></ul></li></ul>If nothing is set, json is used","required":false,"possibleValues":["json","xml"]},{"key":"login","description":"User login","required":true,"exampleValue":"myuser"},{"key":"name","description":"User name","required":false,"exampleValue":"My New Name"}]}]},{"path":"api/webservices","since":"4.2","description":"List web services","actions":[{"key":"list","description":"List web services","since":"4.2","internal":false,"post":false,"hasResponseExample":false,"params":[{"key":"include_internals","description":"Include web services that are implemented for internal use only. Their forward-compatibility is not assured","required":false,"defaultValue":"false","possibleValues":["true","false"]}]},{"key":"response_example","description":"Display web service response example","since":"4.4","internal":false,"post":false,"hasResponseExample":false,"params":[{"key":"action","description":"Action of the web service","required":true,"exampleValue":"search"},{"key":"controller","description":"Controller of the web service","required":true,"exampleValue":"api/issues"}]}]},{"path":"batch","since":"4.4","description":"Get JAR files and referentials for batch","actions":[{"key":"file","description":"Download a JAR file required by source analyzer","since":"4.4","internal":true,"post":false,"hasResponseExample":false,"params":[{"key":"name","description":"File name","required":false,"exampleValue":"batch-library-2.3.jar"}]},{"key":"global","description":"Return metrics and global properties","since":"4.5","internal":true,"post":false,"hasResponseExample":false},{"key":"index","description":"List the JAR files to be downloaded by source analyzer","since":"4.4","internal":true,"post":false,"hasResponseExample":false},{"key":"project","description":"Return project referentials","since":"4.5","internal":true,"post":false,"hasResponseExample":false,"params":[{"key":"key","description":"Project or module key","required":true,"exampleValue":"org.codehaus.sonar:sonar"},{"key":"preview","description":"Preview mode or not","required":false,"defaultValue":"false","possibleValues":["true","false"]},{"key":"profile","description":"Profile name","required":false,"exampleValue":"SonarQube Way"}]}]}]}
