#   - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
#
#   file: lib/Dist/Zilla/Plugin/Manifest/Read.pm
#
#   Copyright © 2015 Van de Bugger
#
#   This file is part of perl-Dist-Zilla-Plugin-Manifest-Read.
#
#   perl-Dist-Zilla-Plugin-Manifest-Read is free software: you can redistribute it and/or modify it
#   under the terms of the GNU General Public License as published by the Free Software Foundation,
#   either version 3 of the License, or (at your option) any later version.
#
#   perl-Dist-Zilla-Plugin-Manifest-Read is distributed in the hope that it will be useful, but
#   WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A
#   PARTICULAR PURPOSE. See the GNU General Public License for more details.
#
#   You should have received a copy of the GNU General Public License along with
#   perl-Dist-Zilla-Plugin-Manifest-Read. If not, see <http://www.gnu.org/licenses/>.
#
#   - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

#pod =for :this This is C<Dist::Zilla::Plugin::Manifest::Read> module documentation. Read this if you are going to hack or
#pod extend C<Manifest::Read>, or use it programmatically.
#pod
#pod =for :those If you want to have annotated source manifest, read the L<user manual|Dist::Zilla::Plugin::Manifest::Read::Manual>.
#pod General topics like getting source, building, installing, bug reporting and some others are covered
#pod in the F<README>.
#pod
#pod =for test_synopsis my $self;
#pod
#pod =head1 SYNOPSIS
#pod
#pod In your plugin:
#pod
#pod     # Iterate through the distribution files listed in MANIFEST
#pod     # (files not included into distrubution are not iterated):
#pod     my $files = $self->zilla->plugin_named( 'Manifest::Read' )->find_files();
#pod     for my $file ( @$files ) {
#pod         ...
#pod     };
#pod
#pod =head1 DESCRIPTION
#pod
#pod This class consumes L<Dist::Zilla::Role::FileGatherer> and C<Dist::Zilla::Role::FileFinder> role.
#pod In order to fulfill requirements, the class implements C<gather_files> and C<find_files> methods.
#pod Other methods are supporting.
#pod
#pod The class also consumes L<Dist::Zilla::Role::ErrorLogger> role. It allows the class not to stop
#pod at the first problem but continue and report multiple errors to user.
#pod
#pod =cut

# --------------------------------------------------------------------------------------------------

package Dist::Zilla::Plugin::Manifest::Read;

use Moose;
use namespace::autoclean;
use version 0.77;

# ABSTRACT: Read annotated source manifest
our $VERSION = 'v0.3.5'; # VERSION

with 'Dist::Zilla::Role::FileGatherer';
with 'Dist::Zilla::Role::FileFinder';
with 'Dist::Zilla::Role::ErrorLogger' => { -version => 0.006 }; # need log_errors_in_file

use Dist::Zilla::File::OnDisk;
use List::Util qw{ min max };
use Path::Tiny;
use Try::Tiny;

# --------------------------------------------------------------------------------------------------

#pod =attr manifest_name
#pod
#pod Name of manifest file to read.
#pod
#pod C<Str>, read-only, default value is C<MANIFEST>, C<init_arg> is C<manifest>.
#pod
#pod =cut

has manifest_name => (
    isa         => 'Str',
    is          => 'ro',
    default     => 'MANIFEST',
    init_arg    => 'manifest',
);

# --------------------------------------------------------------------------------------------------

#pod =attr manifest_file
#pod
#pod Manifest file as a C<Dist::Zilla> file object (C<Dist::Zilla::File::OnDisk>).
#pod
#pod C<Object>, read-only.
#pod
#pod =cut

has manifest_file => (
    isa         => 'Dist::Zilla::File::OnDisk',
    is          => 'ro',
    lazy        => 1,
    builder     => '_build_manifest_file',
    init_arg    => undef,
);

sub _build_manifest_file {
    my ( $self ) = @_;
    #   Straightforward appoach
    #       path( $self->manifest_name )
    #   is incorrect: it works only if the root directory is the current one, which is not always
    #   true. Following expression for manifest path is correct, but gives absolute (and often too
    #   long) name:
    #       path( $self->zilla->root )->child( $self->manifest )
    #   Let's try to shorten it. Hope this works:
    #       path( $self->zilla->root )->child( $self->manifest )->relative
    #   ...until someone changes the current directory...
    return Dist::Zilla::File::OnDisk->new( {
        name => path( $self->zilla->root )->child( $self->manifest_name )->relative . '',
    } );
};

# --------------------------------------------------------------------------------------------------

#pod =method gather_files
#pod
#pod This method fulfills L<Dist::Zilla::Role::FileGatherer> role requirement. It adds files listed in
#pod manifest to distribution (and to C<_files>). Files marked to exclude from distribution and
#pod directories are not added.
#pod
#pod =cut

sub gather_files {
    my ( $self ) = @_;
    for my $file ( @{ $self->_files } ) {
        $self->add_file( $file );
    };
    return;
};

# --------------------------------------------------------------------------------------------------

#pod =method find_files
#pod
#pod This method fulfills L<Dist::Zilla::Role::FileFinder> role requirement. It simply returns a copy of
#pod C<_files> attribute.
#pod
#pod This method can be called by other plugins to iterate through files added by C<Manifest::Read>,
#pod see L</"SYNOPSIS">.
#pod
#pod Note: The method always returns the same list of files. Plugins which remove files from
#pod distribution (i. e. plugins which do C<Dist::Zilla::Role::FilePruner> role) do not affect result of
#pod the method.
#pod
#pod =cut

sub find_files {
    my ( $self ) = @_;
    return [ @{ $self->_files } ];
};

# --------------------------------------------------------------------------------------------------

#pod =attr _files
#pod
#pod Array of files (object which do C<Dist::Zilla::Role::File> role) listed in the manifest I<and>
#pod marked for inclusion to the distribution.
#pod
#pod C<ArrayRef>, read-only, lazy, initialized with builder.
#pod
#pod =cut

has _files => (
    isa         => 'ArrayRef[Object]',
    is          => 'ro',
    lazy        => 1,
    builder     => '_build_files',
    init_arg    => undef,
);

sub _build_files {
    my ( $self ) = @_;
    my $files = [];
    my @errors;
    my $error = sub {
        my ( $item, $message ) = @_;
        my $err = sprintf(
            '%s %s at %s line %d.',
            $item->{ filename }, $message, $self->manifest_name, $item->{ line },
        );
        push( @errors, $item->{ line } => $err );
        return $self->log_error( $err );
    };
    foreach my $item ( $self->_parse_lines() ) {
        -e $item->{ filename } or $error->( $item, 'does not exist' ) and next;
        if ( $item->{ mark } eq '/' ) {
            -d _ or $error->( $item, 'is not a directory' ) and next;
        } else {
            -f _ or $error->( $item, 'is not a plain file' ) and next;
            if ( $item->{ mark } ne '-' ) {
                my $file = Dist::Zilla::File::OnDisk->new( { name => $item->{ filename } } );
                push( @$files, $file );
            };
        };
    };
    if ( @errors ) {
        $self->log_errors_in_file( $self->manifest_file, @errors );
    };
    $self->abort_if_error();
    return $files;
};

# --------------------------------------------------------------------------------------------------

#pod =attr _lines
#pod
#pod Array of chomped manifest lines, including comments and empty lines.
#pod
#pod C<ArrayRef[Str]>, read-only, lazy, initialized with builder.
#pod
#pod =cut

has _lines => (
    isa         => 'ArrayRef[Str]',
    is          => 'ro',
    lazy        => 1,
    init_arg    => undef,
    builder     => '_build_lines',
);

sub _build_lines {
    my ( $self ) = @_;
    my $lines = [];
    try {
        @$lines = split( "\n", $self->manifest_file->content );
    } catch {
        my $ex = $_;
        if ( blessed( $ex ) and $ex->isa( 'Path::Tiny::Error' ) ) {
            $self->abort( [ '%s: %s', $ex->{ file }, $ex->{ err } ] );
        } else {
            $self->abort( "$ex" );
        };
    };
    chomp( @$lines );
    return $lines;
};

# --------------------------------------------------------------------------------------------------

#pod =method _parse_lines
#pod
#pod This method parses manifest lines. Each line is parsed separately (there is no line continuation).
#pod
#pod If the method fails to parse a line, error is reported by calling method C<log_error> (implemented
#pod in L<Dist::Zilla::Role::ErrorLogger>). This means that parsing is not stopped at the first failure,
#pod but entire manifest will be parsed and all the found errors will be reported.
#pod
#pod The method returns list of hashrefs, a hash per file. Each hash has following keys and values:
#pod
#pod =for :list
#pod =   filename
#pod Parsed filename (single-quoted filenames are unquoted, escape sequences are evaluated, if any).
#pod =   mark
#pod Mark.
#pod =   comment
#pod File comment, leading and trailing whitespaces are stripped.
#pod =   line
#pod Number of manifest line the file listed in.
#pod
#pod =cut

my %RE = (
    filename => qr{ ' (*PRUNE) (?: [^'\\] ++ | \\ ['\\] ?+ ) ++ ' | \S ++ }x,
        # ^ TODO: Use Regexp::Common for quoted filename?
    mark     => qr{ [#/+-] }x,
    comment  => qr{ . *? }x,
);

sub _parse_lines {
    my ( $self ) = @_;
    my $manifest = $self->manifest_name;         # Shorter name.
    my ( %files, @files );
    my @errors;
    my $n = 0;
    for my $line ( @{ $self->_lines } ) {
        ++ $n;
        if ( $line =~ m{ \A \s * (?: \# | \z ) }x ) {   # Comment or empty line.
            next;
        };
        ## no critic ( ProhibitComplexRegexes )
        $line =~ m{
            \A
            \s *+                           # requires perl v5.10
            ( $RE{ filename } )
            (*PRUNE)                        # requires perl v5.10
            (?:
                \s ++
                ( $RE{ mark } )
                (*PRUNE)
                (?:
                    \s ++
                    ( $RE{ comment } )
                ) ?
            ) ?
            \s *
            \z
        }x and do {
            my ( $filename, $mark, $comment ) = ( $1, $2, $3 );
            if ( $filename =~ s{ \A ' ( . * ) ' \z }{ $1 }ex ) {
                $filename =~ s{  \\ ( ['\\] ) }{ $1 }gex;
            };
            if ( exists( $files{ $filename } ) ) {
                my $f = $files{ $filename };
                $self->log_error( [ '%s at %s line %d', $filename, $manifest, $n ] );
                $self->log_error( [ '    also listed at %s line %d.', $manifest, $f->{ line } ] );
                push( @errors,
                    $n           => 'The file also listed at line ' . $f->{ line } . '.',
                    $f->{ line } => 'The file also listed at line ' . $n . '.',
                );
                next;
            };
            my $file = {
                filename => $filename,
                mark     => $mark  // '+',     # requires perl v5.10
                comment  => $comment,
                line     => $n,
            };
            $files{ $filename } = $file;
            push( @files, $file );
            1;
        } or do {
            my $error = sprintf( 'Syntax error at %s line %d.', $manifest, $n );
            $self->log_error( $error );
            push( @errors, $n => $error );
            next;
        };
    };
    if ( @errors ) {
        $self->log_errors_in_file( $self->manifest_file, @errors );
        $self->abort();
    };
    return @files;
};

# --------------------------------------------------------------------------------------------------

__PACKAGE__->meta->make_immutable;

1;

# --------------------------------------------------------------------------------------------------

#pod =pod
#pod
#pod =encoding UTF-8
#pod
#pod =head1 WHAT?
#pod
#pod C<Dist-Zilla-Plugin-Manifest-Read> (or C<Manifest::Read> for brevity) is a C<Dist::Zilla> plugin. It reads
#pod I<annotated source> manifest, checks existence of all listed files and directories, and adds
#pod selected files to the distribution. C<Manifest::Read> also does C<FileFinder> role, providing the
#pod list of added files for other plugins.
#pod
#pod =cut


#pod =head1 SEE ALSO
#pod
#pod =for :list
#pod = L<Dist::Zilla>
#pod = L<Dist::Zilla::Role::FileGatherer>
#pod = L<Dist::Zilla::Role::ErrorLogger>
#pod
#pod =head1 COPYRIGHT AND LICENSE
#pod
#pod Copyright (C) 2015 Van de Bugger
#pod
#pod License GPLv3+: The GNU General Public License version 3 or later
#pod <http://www.gnu.org/licenses/gpl-3.0.txt>.
#pod
#pod This is free software: you are free to change and redistribute it. There is
#pod NO WARRANTY, to the extent permitted by law.
#pod
#pod
#pod =cut

# end of file #

__END__

=pod

=encoding UTF-8

=head1 NAME

Dist::Zilla::Plugin::Manifest::Read - Read annotated source manifest

=head1 VERSION

Version v0.3.5, released on 2015-10-29 22:42 UTC.

=head1 WHAT?

C<Dist-Zilla-Plugin-Manifest-Read> (or C<Manifest::Read> for brevity) is a C<Dist::Zilla> plugin. It reads
I<annotated source> manifest, checks existence of all listed files and directories, and adds
selected files to the distribution. C<Manifest::Read> also does C<FileFinder> role, providing the
list of added files for other plugins.

This is C<Dist::Zilla::Plugin::Manifest::Read> module documentation. Read this if you are going to hack or
extend C<Manifest::Read>, or use it programmatically.

If you want to have annotated source manifest, read the L<user manual|Dist::Zilla::Plugin::Manifest::Read::Manual>.
General topics like getting source, building, installing, bug reporting and some others are covered
in the F<README>.

=head1 SYNOPSIS

In your plugin:

    # Iterate through the distribution files listed in MANIFEST
    # (files not included into distrubution are not iterated):
    my $files = $self->zilla->plugin_named( 'Manifest::Read' )->find_files();
    for my $file ( @$files ) {
        ...
    };

=head1 DESCRIPTION

This class consumes L<Dist::Zilla::Role::FileGatherer> and C<Dist::Zilla::Role::FileFinder> role.
In order to fulfill requirements, the class implements C<gather_files> and C<find_files> methods.
Other methods are supporting.

The class also consumes L<Dist::Zilla::Role::ErrorLogger> role. It allows the class not to stop
at the first problem but continue and report multiple errors to user.

=head1 OBJECT ATTRIBUTES

=head2 manifest_name

Name of manifest file to read.

C<Str>, read-only, default value is C<MANIFEST>, C<init_arg> is C<manifest>.

=head2 manifest_file

Manifest file as a C<Dist::Zilla> file object (C<Dist::Zilla::File::OnDisk>).

C<Object>, read-only.

=head2 _files

Array of files (object which do C<Dist::Zilla::Role::File> role) listed in the manifest I<and>
marked for inclusion to the distribution.

C<ArrayRef>, read-only, lazy, initialized with builder.

=head2 _lines

Array of chomped manifest lines, including comments and empty lines.

C<ArrayRef[Str]>, read-only, lazy, initialized with builder.

=head1 OBJECT METHODS

=head2 gather_files

This method fulfills L<Dist::Zilla::Role::FileGatherer> role requirement. It adds files listed in
manifest to distribution (and to C<_files>). Files marked to exclude from distribution and
directories are not added.

=head2 find_files

This method fulfills L<Dist::Zilla::Role::FileFinder> role requirement. It simply returns a copy of
C<_files> attribute.

This method can be called by other plugins to iterate through files added by C<Manifest::Read>,
see L</"SYNOPSIS">.

Note: The method always returns the same list of files. Plugins which remove files from
distribution (i. e. plugins which do C<Dist::Zilla::Role::FilePruner> role) do not affect result of
the method.

=head2 _parse_lines

This method parses manifest lines. Each line is parsed separately (there is no line continuation).

If the method fails to parse a line, error is reported by calling method C<log_error> (implemented
in L<Dist::Zilla::Role::ErrorLogger>). This means that parsing is not stopped at the first failure,
but entire manifest will be parsed and all the found errors will be reported.

The method returns list of hashrefs, a hash per file. Each hash has following keys and values:

=over 4

=item filename

Parsed filename (single-quoted filenames are unquoted, escape sequences are evaluated, if any).

=item mark

Mark.

=item comment

File comment, leading and trailing whitespaces are stripped.

=item line

Number of manifest line the file listed in.

=back

=for test_synopsis my $self;

=head1 SEE ALSO

=over 4

=item L<Dist::Zilla>

=item L<Dist::Zilla::Role::FileGatherer>

=item L<Dist::Zilla::Role::ErrorLogger>

=back

=head1 AUTHOR

Van de Bugger <van.de.bugger@gmail.com>

=head1 COPYRIGHT AND LICENSE

Copyright (C) 2015 Van de Bugger

License GPLv3+: The GNU General Public License version 3 or later
<http://www.gnu.org/licenses/gpl-3.0.txt>.

This is free software: you are free to change and redistribute it. There is
NO WARRANTY, to the extent permitted by law.

=cut
