package Locale::Geocode;

use strict;

=head1 NAME

Locale::Geocode

=head1 DESCRIPTION

Locale::Geocode is a module that provides an
interface with which to find codes and information
on geographical locations and their administrative
subdivisions as defined primarily by ISO 3166-1
and ISO 3166-2.  It is the most complete ISO 3166
module available on CPAN.

Also included are, where applicable, FIPS codes.

=head1 SYNOPSIS

 my $lc     = new Locale::Geocode;

 # retrieve a Locale::Geocode::Territory object
 # for the ISO 3166-1 alpha-2 code 'US'
 my $lct    = $lc->lookup('US');

 # retrieve ISO 3166-1 information for US
 my $name   = $lct->name;   # United States
 my $alpha2 = $lct->alpha2; # US
 my $alpha3 = $lct->alpha3; # USA
 my $num    = $lct->num;    # 840

 # lookup a subdivision of US
 my $lcd    = $lct->lookup('TN');

 # retrieve ISO 3166-2 information for US-TN
 my $name   = $lcd->name;   # Tennessee
 my $code   = $lcd->code;   # TN

 # returns an array of Locale::Geocode::Division
 # objects representing all divisions of US
 my @divs   = $lct->divisions;

 # retrieve a Locale::Geocode::Division object
 # for the ISO 3166-1/ISO 3166-2 combo 'GB-ESS'
 my $lct    = $lc->lookup('GB-ESS');

 # retrieve ISO 3166-2 information for GB-ESS
 # as well as special regional information
 my $name   = $lct->name;   # Essex
 my $code   = $lct->name;   # ESS
 my $region = $lct->region; # ENG

=head1 SOURCES

 Wikipedia - http://en.wikipedia.org/wiki/ISO_3166
 Statoids - http://www.statoids.com

=head1 CONFORMING TO

 BS 6879
 ISO 3166-1
 ISO 3166-1 Newsletter V-1 (1998-02-05; Samoa)
 ISO 3166-1 Newsletter V-2 (1999-10-01; Occupied Palestinian Territory)
 ISO 3166-1 Newsletter V-3 (2002-02-01; Romania)
 ISO 3166-1 Newsletter V-4 (2002-05-20; Name changes)
 ISO 3166-1 Newsletter V-5 (2002-05-20; East Timor)
 ISO 3166-1 Newsletter V-6 (2002-11-15; Timor-Leste)
 ISO 3166-1 Newsletter V-7 (2003-01-14; Comoros)
 ISO 3166-1 Newsletter V-8 (2003-07-23; Serbia and Montenegro)
 ISO 3166-1 Newsletter V-9 (2004-02-13; &#xc5;land Islands)
 ISO 3166-1 Newsletter V-10 (2004-04-26; Name changes)
 ISO 3166-1 Newsletter V-11 (2006-03-29; Jersey, Guernsey, Isle of Man)
 ISO 3166-1 Newsletter V-12 (2006-09-26; Serbia, Montenegro)
 ISO 3166-2
 ISO 3166-2 Newsletter I-1 (2000-06-12)
 ISO 3166-2 Newsletter I-2 (2002-05-21)
 ISO 3166-2 Newsletter I-3 (2002-08-20)
 ISO 3166-2 Newsletter I-4 (2002-12-10)
 ISO 3166-2 Newsletter I-5 (2003-09-05)
 ISO 3166-2 Newsletter I-6 (2004-03-08)
 ISO 3166-2 Newsletter I-7 (2006-09-12)

=cut

our $VERSION = 1.20;

use Locale::Geocode::Territory;
use Locale::Geocode::Division;
use XML::Simple;

# XML::SAX::PurePerl barfs for some reason :/
$XML::Simple::PREFERRED_PARSER = 'XML::SAX::Expat';

# Locale::Geocode extensions.  the following recognized extensions
# are switchable flags that alter the behavior of the lookup methods.
# many of these extensions are part of the ISO 3166 standard as a
# courtesy to other international organizations (such as the UPU or
# ITU).  others are specific to Locale::Geocode for other practical
# reasons (such as the usm extension for US overseas military or
# usps for all US postal abbreviations).
my @exts = qw(upu wco itu uk fx eu usm usps ust);
my @defs = qw(ust);

# parse the XML data
my $opts = { ForceArray => [ 'division', 'ext', 'note' ], KeyAttr => [], SuppressEmpty => 1 };
my $data = { raw => XMLin(\*DATA, %$opts) };

# create lookup tables
foreach my $href (@{ $data->{raw}->{territory} }) {
	my $aref = $href->{division};

	$href->{divs_code} = { map { lc($_->{code}), $_ } grep { defined $_->{code} } @$aref };
	$href->{divs_fips} = { map { lc($_->{fips}), $_ } grep { defined $_->{fips} } @$aref };
	$href->{divs_name} = { map { lc($_->{name}), $_ } grep { defined $_->{name} } @$aref };
}

my $aref = $data->{raw}->{territory};

$data->{alpha2}	= { map { lc($_->{alpha2})	=> $_ } grep { defined $_->{alpha2} }	@$aref };
$data->{alpha3}	= { map { lc($_->{alpha3})	=> $_ } grep { defined $_->{alpha3} }	@$aref };
$data->{name}	= { map { lc($_->{name})	=> $_ } grep { defined $_->{name} }		@$aref };

=head1 METHODS

=over 4

=item new

=cut

sub new
{
	my $proto = shift;
	my $class = ref($proto) || $proto;

	my $args = { @_ };
	my $self = {};

	bless $self, $class;

	my @exts = @defs;

	if ($args->{ext}) {
		my $reftype = ref $args->{ext};

		die 'ext argument must be scalar or list reference'
			if $reftype ne '' && $reftype ne 'ARRAY';

		@exts = $reftype eq 'ARRAY' ? @{ $args->{ext} } : $args->{ext};
	}

	$self->ext(@exts);

	return $self;
}

=item lookup

=cut

sub lookup
{
	my $self	= shift;
	my $key		= shift;

	my $lct = new Locale::Geocode::Territory $key, $self;

	if (!$lct && $key =~ /-/) {
		my ($iso3166_1, $iso3166_2) = split '-', $key;

		$lct = new Locale::Geocode::Territory $iso3166_1, $self;

		return $lct->lookup($iso3166_2) if $lct;
	}

	return $lct;
}

=item territories

=cut

sub territories
{
	my $self = shift;

	return map { $self->lookup($_) || () } keys %{ $data->{alpha2} };
}

=item territories_sorted

=cut

sub territories_sorted
{
	my $self = shift;

	return sort { $a->name cmp $b->name } $self->territories;
}

=item territories_sorted_us

=cut

sub territories_sorted_us
{
	my $self = shift;

	sub sorted_us
	{
		return -1 if $a->alpha2 eq 'US';
		return 1  if $b->alpha2 eq 'US';
		return $a->name cmp $b->name;
	};

	return sort sorted_us $self->territories;
}

=item ext

=cut

sub ext
{
	my $self = shift;

	if (scalar @_ > 0) {
		$self->{ext} =
		{
			ust => 1, # 'ust' is always on unless explicitly disabled
			map {
				/^-(.*)$/
					? ($1 => 0)
					: $_ eq 'all'
						? map { $_ => 1 } @exts
						: ($_ => 1)
			} @_
		};
	}

	return grep { $self->{ext}->{$_} } keys %{ $self->{ext} };
}

=item ext_enable

=cut

sub ext_enable
{
	my $self = shift;

	foreach my $ext (@_) {
		$self->{ext}->{$ext} = 1 if grep { $ext eq $_ } @exts;
	}
}

=item ext_disable

=cut

sub ext_disable
{
	my $self = shift;

	delete $self->{ext}->{$_} foreach @_;
}

sub chkext
{
	my $self = shift;
	my $href = shift;

	return $href->{ext} ? grep { $self->{ext}->{$_} } @{ $href->{ext} } : 1;
}

sub data { return $data }

sub import { @defs = @_[1..$#_] }

=back

=cut

=head1 AUTHOR

 Mike Eldridge <diz@cpan.org>

=head1 CREDITS

 Kim Ryan

=head1 SEE ALSO

 L<Locale::Geocode::Territory>
 L<Locale::Geocode::Division>

=cut

1;

__DATA__
<?xml version="1.0" encoding="UTF-8" ?>
<iso3166>
	<territory>
		<name>Afghanistan</name>
		<alpha2>AF</alpha2>
		<alpha3>AFG</alpha3>
		<division>
			<name>Kandahar</name>
			<code>KAN</code>
			<fips>23</fips>
			<region></region>
		</division>
		<division>
			<name>Nangrahar [Nangarhar]</name>
			<code>NAN</code>
			<fips>18</fips>
			<region></region>
		</division>
		<division>
			<name>Balkh</name>
			<code>BAL</code>
			<fips>30</fips>
			<region></region>
		</division>
		<division>
			<name>Laghman</name>
			<code>LAG</code>
			<fips>16</fips>
			<region></region>
		</division>
		<division>
			<name>Jowzjan</name>
			<code>JOW</code>
			<fips>31</fips>
			<region></region>
		</division>
		<division>
			<name>Wardak [Wardag]</name>
			<code>WAR</code>
			<fips>27</fips>
			<region></region>
		</division>
		<division>
			<name>Paktika</name>
			<code>PKA</code>
			<fips>29</fips>
			<region></region>
		</division>
		<division>
			<name>Lowgar</name>
			<code>LOW</code>
			<fips>17</fips>
			<region></region>
		</division>
		<division>
			<name>Konar [Kunar]</name>
			<code>KNR</code>
			<fips>15</fips>
			<region></region>
		</division>
		<division>
			<name>Badakhshan</name>
			<code>BDS</code>
			<fips>01</fips>
			<region></region>
		</division>
		<division>
			<name>Zabol [Zabul]</name>
			<code>ZAB</code>
			<fips>28</fips>
			<region></region>
		</division>
		<division>
			<name>Helmand</name>
			<code>HEL</code>
			<fips>10</fips>
			<region></region>
		</division>
		<division>
			<name>Kapisa</name>
			<code>KAP</code>
			<fips>14</fips>
			<region></region>
		</division>
		<division>
			<name>Bamian</name>
			<code>BAM</code>
			<fips>05</fips>
			<region></region>
		</division>
		<division>
			<name>Oruzgan [Uruzgan]</name>
			<code>ORU</code>
			<fips>20</fips>
			<region></region>
		</division>
		<division>
			<name>Baghlan</name>
			<code>BGL</code>
			<fips>03</fips>
			<region></region>
		</division>
		<division>
			<name>Faryab</name>
			<code>FYB</code>
			<fips>07</fips>
			<region></region>
		</division>
		<division>
			<name>Nimruz</name>
			<code>NIM</code>
			<fips>19</fips>
			<region></region>
		</division>
		<division>
			<name>Farah</name>
			<code>FRA</code>
			<fips>06</fips>
			<region></region>
		</division>
		<division>
			<name>Sar-e Pol</name>
			<code>SAR</code>
			<fips>33</fips>
			<region></region>
		</division>
		<division>
			<name>Samangan</name>
			<code>SAM</code>
			<fips>32</fips>
			<region></region>
		</division>
		<division>
			<name>Kabul [Kabol]</name>
			<code>KAB</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Ghowr</name>
			<code>GHO</code>
			<fips>09</fips>
			<region></region>
		</division>
		<division>
			<name>Takhar</name>
			<code>TAK</code>
			<fips>26</fips>
			<region></region>
		</division>
		<division>
			<name>Ghazni</name>
			<code>GHA</code>
			<fips>08</fips>
			<region></region>
		</division>
		<division>
			<name>Herat</name>
			<code>HER</code>
			<fips>11</fips>
			<region></region>
		</division>
		<division>
			<name>Kondoz [Kunduz]</name>
			<code>KDZ</code>
			<fips>24</fips>
			<region></region>
		</division>
		<division>
			<name>Parwan</name>
			<code>PAR</code>
			<fips>22</fips>
			<region></region>
		</division>
		<division>
			<name>Badghis</name>
			<code>BDG</code>
			<fips>02</fips>
			<region></region>
		</division>
		<division>
			<name>Paktia</name>
			<code>PIA</code>
			<fips>21</fips>
			<region></region>
		</division>
		<num>004</num>
	</territory>
	<territory>
		<name>Albania</name>
		<alpha2>AL</alpha2>
		<alpha3>ALB</alpha3>
		<division>
			<name>P&#x3b;rmet</name>
			<code>PR</code>
			<fips>17</fips>
			<region>5</region>
		</division>
		<division>
			<name>Puk&#x3b;</name>
			<code>PU</code>
			<fips>19</fips>
			<region>10</region>
		</division>
		<division>
			<name>Pogradec</name>
			<code>PG</code>
			<fips>18</fips>
			<region>6</region>
		</division>
		<division>
			<name>Mirdit&#x3b;</name>
			<code>MR</code>
			<fips>16</fips>
			<region>8</region>
		</division>
		<division>
			<name>Peqin</name>
			<code>PQ</code>
			<fips>38</fips>
			<region>3</region>
		</division>
		<division>
			<name>Skrapar</name>
			<code>SK</code>
			<fips>22</fips>
			<region>1</region>
		</division>
		<division>
			<name>Mallakast&#x3b;r</name>
			<code>MK</code>
			<fips>37</fips>
			<region>4</region>
		</division>
		<division>
			<name>Mat</name>
			<code>MT</code>
			<fips>15</fips>
			<region>9</region>
		</division>
		<division>
			<name>Lushnj&#x3b;</name>
			<code>LU</code>
			<fips>14</fips>
			<region>4</region>
		</division>
		<division>
			<name>Tropoj&#x3b;</name>
			<code>TP</code>
			<fips>26</fips>
			<region>7</region>
		</division>
		<division>
			<name>Librazhd</name>
			<code>LB</code>
			<fips>13</fips>
			<region>3</region>
		</division>
		<division>
			<name>Tiran&#x3b;</name>
			<code>TR</code>
			<fips>39</fips>
			<region>11</region>
		</division>
		<division>
			<name>Tepelen&#x3b;</name>
			<code>TE</code>
			<fips>23</fips>
			<region>5</region>
		</division>
		<division>
			<name>Mal&#x3b;si e Madhe</name>
			<code>MM</code>
			<fips></fips>
			<region>10</region>
		</division>
		<division>
			<name>Vlor&#x3b;</name>
			<code>VL</code>
			<fips>27</fips>
			<region>12</region>
		</division>
		<division>
			<name>Shkod&#x3b;r</name>
			<code>SH</code>
			<fips>21</fips>
			<region>10</region>
		</division>
		<division>
			<name>Sarand&#x3b;</name>
			<code>SR</code>
			<fips>20</fips>
			<region>12</region>
		</division>
		<num>008</num>
	</territory>
	<territory>
		<name>Algeria</name>
		<alpha2>DZ</alpha2>
		<alpha3>DZA</alpha3>
		<division>
			<name>Tlemcen</name>
			<code>13</code>
			<fips>15</fips>
			<region></region>
		</division>
		<division>
			<name>Blida</name>
			<code>09</code>
			<fips>20</fips>
			<region></region>
		</division>
		<division>
			<name>Skikda</name>
			<code>21</code>
			<fips>31</fips>
			<region></region>
		</division>
		<division>
			<name>Tizi Ouzou</name>
			<code>15</code>
			<fips>14</fips>
			<region></region>
		</division>
		<division>
			<name>Msila</name>
			<code>28</code>
			<fips>27</fips>
			<region></region>
		</division>
		<division>
			<name>Bouira</name>
			<code>10</code>
			<fips>21</fips>
			<region></region>
		</division>
		<division>
			<name>Annaba</name>
			<code>23</code>
			<fips>37</fips>
			<region></region>
		</division>
		<division>
			<name>Chlef</name>
			<code>02</code>
			<fips>41</fips>
			<region></region>
		</division>
		<division>
			<name>Oum el Bouaghi</name>
			<code>04</code>
			<fips>29</fips>
			<region></region>
		</division>
		<division>
			<name>Naama</name>
			<code>45</code>
			<fips>49</fips>
			<region></region>
		</division>
		<division>
			<name>Adrar</name>
			<code>01</code>
			<fips>34</fips>
			<region></region>
		</division>
		<division>
			<name>Illizi</name>
			<code>33</code>
			<fips>46</fips>
			<region></region>
		</division>
		<division>
			<name>Gharda&#x3f;a</name>
			<code>47</code>
			<fips>45</fips>
			<region></region>
		</division>
		<division>
			<name>Tissemsilt</name>
			<code>38</code>
			<fips>56</fips>
			<region></region>
		</division>
		<division>
			<name>S&#xe9;tif</name>
			<code>19</code>
			<fips>12</fips>
			<region></region>
		</division>
		<division>
			<name>El Tarf</name>
			<code>36</code>
			<fips>44</fips>
			<region></region>
		</division>
		<division>
			<name>B&#xe9;ja&#x3f;a</name>
			<code>06</code>
			<fips>18</fips>
			<region></region>
		</division>
		<division>
			<name>Djelfa</name>
			<code>17</code>
			<fips>22</fips>
			<region></region>
		</division>
		<division>
			<name>Tiaret</name>
			<code>14</code>
			<fips>13</fips>
			<region></region>
		</division>
		<division>
			<name>Khenchela</name>
			<code>40</code>
			<fips>47</fips>
			<region></region>
		</division>
		<division>
			<name>Mostaganem</name>
			<code>27</code>
			<fips>07</fips>
			<region></region>
		</division>
		<division>
			<name>A&#x3f;n T&#xe9;mouchent</name>
			<code>46</code>
			<fips>36</fips>
			<region></region>
		</division>
		<division>
			<name>Laghouat</name>
			<code>03</code>
			<fips>25</fips>
			<region></region>
		</division>
		<division>
			<name>Batna</name>
			<code>05</code>
			<fips>03</fips>
			<region></region>
		</division>
		<division>
			<name>A&#x3f;n Defla</name>
			<code>44</code>
			<fips>35</fips>
			<region></region>
		</division>
		<division>
			<name>B&#xe9;char</name>
			<code>08</code>
			<fips>38</fips>
			<region></region>
		</division>
		<division>
			<name>Bordj Bou Arr&#xe9;ridj</name>
			<code>34</code>
			<fips>39</fips>
			<region></region>
		</division>
		<division>
			<name>El Oued</name>
			<code>39</code>
			<fips>43</fips>
			<region></region>
		</division>
		<division>
			<name>Tamanghasset</name>
			<code>11</code>
			<fips>53</fips>
			<region></region>
		</division>
		<division>
			<name>Souk Ahras</name>
			<code>41</code>
			<fips>52</fips>
			<region></region>
		</division>
		<division>
			<name>Sidi Bel Abb&#xe8;s</name>
			<code>22</code>
			<fips>30</fips>
			<region></region>
		</division>
		<division>
			<name>Jijel</name>
			<code>18</code>
			<fips>24</fips>
			<region></region>
		</division>
		<division>
			<name>Sa&#x3f;da</name>
			<code>20</code>
			<fips>10</fips>
			<region></region>
		</division>
		<division>
			<name>Constantine</name>
			<code>25</code>
			<fips>04</fips>
			<region></region>
		</division>
		<division>
			<name>Boumerd&#xe8;s</name>
			<code>35</code>
			<fips>40</fips>
			<region></region>
		</division>
		<division>
			<name>Guelma</name>
			<code>24</code>
			<fips>23</fips>
			<region></region>
		</division>
		<division>
			<name>Mascara</name>
			<code>29</code>
			<fips>26</fips>
			<region></region>
		</division>
		<division>
			<name>Mila</name>
			<code>43</code>
			<fips>48</fips>
			<region></region>
		</division>
		<division>
			<name>Relizane</name>
			<code>48</code>
			<fips>51</fips>
			<region></region>
		</division>
		<division>
			<name>Oran</name>
			<code>31</code>
			<fips>09</fips>
			<region></region>
		</division>
		<division>
			<name>Tindouf</name>
			<code>37</code>
			<fips>54</fips>
			<region></region>
		</division>
		<division>
			<name>M&#xe9;d&#xe9;a</name>
			<code>26</code>
			<fips>06</fips>
			<region></region>
		</division>
		<division>
			<name>El Bayadh</name>
			<code>32</code>
			<fips>42</fips>
			<region></region>
		</division>
		<division>
			<name>Tipaza</name>
			<code>42</code>
			<fips>55</fips>
			<region></region>
		</division>
		<division>
			<name>Ouargla</name>
			<code>30</code>
			<fips>50</fips>
			<region></region>
		</division>
		<division>
			<name>T&#xe9;bessa</name>
			<code>12</code>
			<fips>33</fips>
			<region></region>
		</division>
		<division>
			<name>Biskra</name>
			<code>07</code>
			<fips>19</fips>
			<region></region>
		</division>
		<division>
			<name>Alger</name>
			<code>16</code>
			<fips>01</fips>
			<region></region>
		</division>
		<num>012</num>
	</territory>
	<territory>
		<name>American Samoa</name>
		<alpha2>AS</alpha2>
		<alpha3>ASM</alpha3>
		<num>016</num>
	</territory>
	<territory>
		<name>Andorra</name>
		<alpha2>AD</alpha2>
		<alpha3>AND</alpha3>
		<division>
			<name>La Massana</name>
			<code>MA</code>
			<fips>04</fips>
			<region></region>
		</division>
		<division>
			<name>Ordino</name>
			<code>OR</code>
			<fips>05</fips>
			<region></region>
		</division>
		<division>
			<name>Sant Juli&#xe0; de L&#xf2;ria</name>
			<code>JL</code>
			<fips>06</fips>
			<region></region>
		</division>
		<division>
			<name>Encamp</name>
			<code>EN</code>
			<fips>03</fips>
			<region></region>
		</division>
		<division>
			<name>Escaldes-Engordany</name>
			<code>EE</code>
			<fips>08</fips>
			<region></region>
		</division>
		<division>
			<name>Canillo</name>
			<code>CA</code>
			<fips>02</fips>
			<region></region>
		</division>
		<division>
			<name>Andorra la Vella</name>
			<code>AN</code>
			<fips>07</fips>
			<region></region>
		</division>
		<num>020</num>
	</territory>
	<territory>
		<name>Angola</name>
		<alpha2>AO</alpha2>
		<alpha3>AGO</alpha3>
		<division>
			<name>Luanda</name>
			<code>LUA</code>
			<fips>10</fips>
			<region></region>
		</division>
		<division>
			<name>Cuando-Cubango</name>
			<code>CCU</code>
			<fips>04</fips>
			<region></region>
		</division>
		<division>
			<name>Bengo</name>
			<code>BGO</code>
			<fips>19</fips>
			<region></region>
		</division>
		<division>
			<name>Moxico</name>
			<code>MOX</code>
			<fips>14</fips>
			<region></region>
		</division>
		<division>
			<name>Huambo</name>
			<code>HUA</code>
			<fips>08</fips>
			<region></region>
		</division>
		<division>
			<name>Malange</name>
			<code>MAL</code>
			<fips>12</fips>
			<region></region>
		</division>
		<division>
			<name>Cuanza Sul</name>
			<code>CUS</code>
			<fips>06</fips>
			<region></region>
		</division>
		<division>
			<name>Cunene</name>
			<code>CNN</code>
			<fips>07</fips>
			<region></region>
		</division>
		<division>
			<name>Cuanza Norte</name>
			<code>CNO</code>
			<fips>05</fips>
			<region></region>
		</division>
		<division>
			<name>U&#xed;ge</name>
			<code>UIG</code>
			<fips>15</fips>
			<region></region>
		</division>
		<division>
			<name>Benguela</name>
			<code>BGU</code>
			<fips>01</fips>
			<region></region>
		</division>
		<division>
			<name>Lunda Norte</name>
			<code>LNO</code>
			<fips>17</fips>
			<region></region>
		</division>
		<division>
			<name>Bi&#xe9;</name>
			<code>BIE</code>
			<fips>02</fips>
			<region></region>
		</division>
		<division>
			<name>Lunda Sul</name>
			<code>LSU</code>
			<fips>18</fips>
			<region></region>
		</division>
		<division>
			<name>Zaire</name>
			<code>ZAI</code>
			<fips>16</fips>
			<region></region>
		</division>
		<division>
			<name>Cabinda</name>
			<code>CAB</code>
			<fips>03</fips>
			<region></region>
		</division>
		<division>
			<name>Namibe</name>
			<code>NAM</code>
			<fips>13</fips>
			<region></region>
		</division>
		<division>
			<name>Hu&#xed;la</name>
			<code>HUI</code>
			<fips>09</fips>
			<region></region>
		</division>
		<num>024</num>
	</territory>
	<territory>
		<name>Anguilla</name>
		<alpha2>AI</alpha2>
		<alpha3>AIA</alpha3>
		<num>660</num>
	</territory>
	<territory>
		<name>Antarctica</name>
		<alpha2>AQ</alpha2>
		<alpha3>ATA</alpha3>
		<num>010</num>
	</territory>
	<territory>
		<name>Antigua and Barbuda</name>
		<alpha2>AG</alpha2>
		<alpha3>ATG</alpha3>
		<num>028</num>
	</territory>
	<territory>
		<name>Argentina</name>
		<alpha2>AR</alpha2>
		<alpha3>ARG</alpha3>
		<division>
			<name>Misiones</name>
			<code>N</code>
			<fips>14</fips>
			<region></region>
		</division>
		<division>
			<name>Formosa</name>
			<code>P</code>
			<fips>09</fips>
			<region></region>
		</division>
		<division>
			<name>Catamarca</name>
			<code>K</code>
			<fips>02</fips>
			<region></region>
		</division>
		<division>
			<name>Chubut</name>
			<code>U</code>
			<fips>04</fips>
			<region></region>
		</division>
		<division>
			<name>Mendoza</name>
			<code>M</code>
			<fips>13</fips>
			<region></region>
		</division>
		<division>
			<name>Neuqu&#xe9;n</name>
			<code>Q</code>
			<fips>15</fips>
			<region></region>
		</division>
		<division>
			<name>Corrientes</name>
			<code>W</code>
			<fips>06</fips>
			<region></region>
		</division>
		<division>
			<name>Santiago del Estero</name>
			<code>G</code>
			<fips>22</fips>
			<region></region>
		</division>
		<division>
			<name>R&#xed;o Negro</name>
			<code>R</code>
			<fips>16</fips>
			<region></region>
		</division>
		<division>
			<name>Jujuy</name>
			<code>Y</code>
			<fips>10</fips>
			<region></region>
		</division>
		<division>
			<name>La Pampa</name>
			<code>L</code>
			<fips>11</fips>
			<region></region>
		</division>
		<division>
			<name>Tierra del Fuego</name>
			<code>V</code>
			<fips>23</fips>
			<region></region>
		</division>
		<division>
			<name>Salta</name>
			<code>A</code>
			<fips>17</fips>
			<region></region>
		</division>
		<division>
			<name>Chaco</name>
			<code>H</code>
			<fips>03</fips>
			<region></region>
		</division>
		<division>
			<name>San Luis</name>
			<code>D</code>
			<fips>19</fips>
			<region></region>
		</division>
		<division>
			<name>San Juan</name>
			<code>J</code>
			<fips>18</fips>
			<region></region>
		</division>
		<division>
			<name>Entre R&#xed;os</name>
			<code>E</code>
			<fips>08</fips>
			<region></region>
		</division>
		<division>
			<name>Santa Cruz</name>
			<code>Z</code>
			<fips>20</fips>
			<region></region>
		</division>
		<division>
			<name>Capital federal</name>
			<code>C</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>La Rioja</name>
			<code>F</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Tucum&#xe1;n</name>
			<code>T</code>
			<fips>24</fips>
			<region></region>
		</division>
		<division>
			<name>Buenos Aires</name>
			<code>B</code>
			<fips>01</fips>
			<region></region>
		</division>
		<division>
			<name>C&#xf3;rdoba</name>
			<code>X</code>
			<fips>05</fips>
			<region></region>
		</division>
		<division>
			<name>Santa Fe</name>
			<code>S</code>
			<fips>21</fips>
			<region></region>
		</division>
		<num>032</num>
	</territory>
	<territory>
		<name>Armenia</name>
		<alpha2>AM</alpha2>
		<alpha3>ARM</alpha3>
		<division>
			<name>Aragacotn</name>
			<code>AG</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Kotayk'</name>
			<code>KT</code>
			<fips>05</fips>
			<region></region>
		</division>
		<division>
			<name>Erevan</name>
			<code>ER</code>
			<fips>11</fips>
			<region></region>
		</division>
		<division>
			<name>Gegark'unik'</name>
			<code>GR</code>
			<fips>04</fips>
			<region></region>
		</division>
		<division>
			<name>Ararat</name>
			<code>AR</code>
			<fips>02</fips>
			<region></region>
		</division>
		<division>
			<name>Lory</name>
			<code>LO</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Syunik'</name>
			<code>SU</code>
			<fips>08</fips>
			<region></region>
		</division>
		<division>
			<name>Vayoc Jor</name>
			<code>VD</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Tavu&#x161;</name>
			<code>TV</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Armavir</name>
			<code>AV</code>
			<fips>03</fips>
			<region></region>
		</division>
		<division>
			<name>&#x160;irak</name>
			<code>SH</code>
			<fips></fips>
			<region></region>
		</division>
		<num>051</num>
	</territory>
	<territory>
		<name>Aruba</name>
		<alpha2>AW</alpha2>
		<alpha3>ABW</alpha3>
		<num>533</num>
	</territory>
	<territory>
		<name>Ascension Island</name>
		<alpha2>AC</alpha2>
		<alpha3></alpha3>
		<num></num>
		<note>reserved on request of UPU</note>
		<ext>upu</ext>
	</territory>
	<territory>
		<name>Australia</name>
		<alpha2>AU</alpha2>
		<alpha3>AUS</alpha3>
		<division>
			<name>New South Wales</name>
			<code>NSW</code>
			<fips>02</fips>
			<region></region>
		</division>
		<division>
			<name>Western Australia</name>
			<code>WA</code>
			<fips>08</fips>
			<region></region>
		</division>
		<division>
			<name>Tasmania</name>
			<code>TAS</code>
			<fips>06</fips>
			<region></region>
		</division>
		<division>
			<name>South Australia</name>
			<code>SA</code>
			<fips>05</fips>
			<region></region>
		</division>
		<division>
			<name>Australian Capital Territory</name>
			<code>ACT</code>
			<fips>01</fips>
			<region></region>
		</division>
		<division>
			<name>Northern Territory</name>
			<code>NT</code>
			<fips>03</fips>
			<region></region>
		</division>
		<division>
			<name>Victoria</name>
			<code>VIC</code>
			<fips>07</fips>
			<region></region>
		</division>
		<division>
			<name>Queensland</name>
			<code>QLD</code>
			<fips>04</fips>
			<region></region>
		</division>
		<num>036</num>
	</territory>
	<territory>
		<name>Austria</name>
		<alpha2>AT</alpha2>
		<alpha3>AUT</alpha3>
		<division>
			<name>K&#xe4;rnten</name>
			<code>2</code>
			<fips>02</fips>
			<region></region>
		</division>
		<division>
			<name>Wien</name>
			<code>9</code>
			<fips>09</fips>
			<region></region>
		</division>
		<division>
			<name>Steiermark</name>
			<code>6</code>
			<fips>06</fips>
			<region></region>
		</division>
		<division>
			<name>Vorarlberg</name>
			<code>8</code>
			<fips>08</fips>
			<region></region>
		</division>
		<division>
			<name>Salzburg</name>
			<code>5</code>
			<fips>05</fips>
			<region></region>
		</division>
		<division>
			<name>Tirol</name>
			<code>7</code>
			<fips>07</fips>
			<region></region>
		</division>
		<division>
			<name>Ober&#xf6;sterreich</name>
			<code>4</code>
			<fips>04</fips>
			<region></region>
		</division>
		<division>
			<name>Burgenland</name>
			<code>1</code>
			<fips>01</fips>
			<region></region>
		</division>
		<division>
			<name>Nieder&#xf6;sterreich</name>
			<code>3</code>
			<fips>03</fips>
			<region></region>
		</division>
		<num>040</num>
	</territory>
	<territory>
		<name>Azerbaijan</name>
		<alpha2>AZ</alpha2>
		<alpha3>AZE</alpha3>
		<division>
			<name>Agsu</name>
			<code>AGU</code>
			<fips>06</fips>
			<region></region>
		</division>
		<division>
			<name>K&#xe4;lb&#xe4;c&#xe4;r</name>
			<code>KAL</code>
			<fips>26</fips>
			<region></region>
		</division>
		<division>
			<name>Ismayilli</name>
			<code>ISM</code>
			<fips>25</fips>
			<region></region>
		</division>
		<division>
			<name>Xocav&#xe4;nd</name>
			<code>XVD</code>
			<fips>65</fips>
			<region></region>
		</division>
		<division>
			<name>Yevlax</name>
			<code>YEV</code>
			<fips>68</fips>
			<region></region>
		</division>
		<division>
			<name>Baki</name>
			<code>BA</code>
			<fips>09</fips>
			<region></region>
		</division>
		<division>
			<name>Zaqatala</name>
			<code>ZAQ</code>
			<fips>70</fips>
			<region></region>
		</division>
		<division>
			<name>B&#xe4;rd&#xe4;</name>
			<code>BAR</code>
			<fips>11</fips>
			<region></region>
		</division>
		<division>
			<name>Xank&#xe4;ndi</name>
			<code>XA</code>
			<fips>61</fips>
			<region></region>
		</division>
		<division>
			<name>Yardimli</name>
			<code>YAR</code>
			<fips>66</fips>
			<region></region>
		</division>
		<division>
			<name>Sahbuz</name>
			<code>SAH</code>
			<fips></fips>
			<region>NX</region>
		</division>
		<division>
			<name>Q&#xe4;b&#xe4;l&#xe4;</name>
			<code>QAB</code>
			<fips>38</fips>
			<region></region>
		</division>
		<division>
			<name>G&#xe4;nc&#xe4;</name>
			<code>GA</code>
			<fips>20</fips>
			<region></region>
		</division>
		<division>
			<name>Qubadli</name>
			<code>QBI</code>
			<fips>43</fips>
			<region></region>
		</division>
		<division>
			<name>Lerik</name>
			<code>LER</code>
			<fips>31</fips>
			<region></region>
		</division>
		<division>
			<name>&#xc4;li Bayramli</name>
			<code>AB</code>
			<fips>07</fips>
			<region></region>
		</division>
		<division>
			<name>Ordubad</name>
			<code>ORD</code>
			<fips></fips>
			<region>NX</region>
		</division>
		<division>
			<name>Samaxi</name>
			<code>SMI</code>
			<fips>50</fips>
			<region></region>
		</division>
		<division>
			<name>Abseron</name>
			<code>ABS</code>
			<fips>01</fips>
			<region></region>
		</division>
		<division>
			<name>S&#xe4;ki</name>
			<code>SAK</code>
			<fips>48</fips>
			<region></region>
		</division>
		<division>
			<name>Sabirabad</name>
			<code>SAB</code>
			<fips>46</fips>
			<region></region>
		</division>
		<division>
			<name>Agdam</name>
			<code>AGM</code>
			<fips>03</fips>
			<region></region>
		</division>
		<division>
			<name>Beyl&#xe4;qan</name>
			<code>BEY</code>
			<fips>12</fips>
			<region></region>
		</division>
		<division>
			<name>F&#xfc;zuli</name>
			<code>FUZ</code>
			<fips>18</fips>
			<region></region>
		</division>
		<division>
			<name>Saatli</name>
			<code>SAT</code>
			<fips>45</fips>
			<region></region>
		</division>
		<division>
			<name>Agcab&#xe4;di</name>
			<code>AGC</code>
			<fips>02</fips>
			<region></region>
		</division>
		<division>
			<name>Agdas</name>
			<code>AGS</code>
			<fips>04</fips>
			<region></region>
		</division>
		<division>
			<name>Dask&#xe4;s&#xe4;n</name>
			<code>DAS</code>
			<fips>16</fips>
			<region></region>
		</division>
		<division>
			<name>Quba</name>
			<code>QBA</code>
			<fips>42</fips>
			<region></region>
		</division>
		<division>
			<name>Neft&#xe7;ala</name>
			<code>NEF</code>
			<fips>36</fips>
			<region></region>
		</division>
		<division>
			<name>Haciqabul</name>
			<code>HAC</code>
			<fips>23</fips>
			<region></region>
		</division>
		<division>
			<name>G&#xf6;y&#xe7;ay</name>
			<code>GOY</code>
			<fips>22</fips>
			<region></region>
		</division>
		<division>
			<name>Qusar</name>
			<code>QUS</code>
			<fips>44</fips>
			<region></region>
		</division>
		<division>
			<name>Xanlar</name>
			<code>XAN</code>
			<fips>62</fips>
			<region></region>
		</division>
		<division>
			<name>Oguz</name>
			<code>OGU</code>
			<fips>37</fips>
			<region></region>
		</division>
		<division>
			<name>Tovuz</name>
			<code>TOV</code>
			<fips>58</fips>
			<region></region>
		</division>
		<division>
			<name>S&#xe4;rur</name>
			<code>SAR</code>
			<fips></fips>
			<region>NX</region>
		</division>
		<division>
			<name>Culfa</name>
			<code>CUL</code>
			<fips></fips>
			<region>NX</region>
		</division>
		<division>
			<name>S&#xe4;d&#xe4;r&#xe4;k</name>
			<code>SAD</code>
			<fips></fips>
			<region>NX</region>
		</division>
		<division>
			<name>Bil&#xe4;suvar</name>
			<code>BIL</code>
			<fips>13</fips>
			<region></region>
		</division>
		<division>
			<name>Ming&#xe4;&#xe7;evir</name>
			<code>MI</code>
			<fips>33</fips>
			<region></region>
		</division>
		<division>
			<name>D&#xe4;v&#xe4;&#xe7;i</name>
			<code>DAV</code>
			<fips>17</fips>
			<region></region>
		</division>
		<division>
			<name>Imisli</name>
			<code>IMI</code>
			<fips>24</fips>
			<region></region>
		</division>
		<division>
			<name>Siy&#xe4;z&#xe4;n</name>
			<code>SIY</code>
			<fips>53</fips>
			<region></region>
		</division>
		<division>
			<name>G&#xe4;d&#xe4;b&#xe4;y</name>
			<code>GAD</code>
			<fips>19</fips>
			<region></region>
		</division>
		<division>
			<name>Samux</name>
			<code>SMX</code>
			<fips>52</fips>
			<region></region>
		</division>
		<division>
			<name>C&#xe4;lilabab</name>
			<code>CAL</code>
			<fips>15</fips>
			<region></region>
		</division>
		<division>
			<name>Naftalan</name>
			<code>NA</code>
			<fips>34</fips>
			<region></region>
		</division>
		<division>
			<name>Goranboy</name>
			<code>GOR</code>
			<fips>21</fips>
			<region></region>
		</division>
		<division>
			<name>Qazax</name>
			<code>QAZ</code>
			<fips>40</fips>
			<region></region>
		</division>
		<division>
			<name>T&#xe4;rt&#xe4;r</name>
			<code>TAR</code>
			<fips>57</fips>
			<region></region>
		</division>
		<division>
			<name>Masalli</name>
			<code>MAS</code>
			<fips>32</fips>
			<region></region>
		</division>
		<division>
			<name>La&#xe7;in</name>
			<code>LAC</code>
			<fips>28</fips>
			<region></region>
		</division>
		<division>
			<name>L&#xe4;nk&#xe4;ran</name>
			<code>LAN</code>
			<fips>30</fips>
			<region></region>
		</division>
		<division>
			<name>C&#xe4;brayil</name>
			<code>CAB</code>
			<fips>14</fips>
			<region></region>
		</division>
		<division>
			<name>Z&#xe4;rdab</name>
			<code>ZAR</code>
			<fips>71</fips>
			<region></region>
		</division>
		<division>
			<name>Agstafa</name>
			<code>AGA</code>
			<fips>05</fips>
			<region></region>
		</division>
		<division>
			<name>Xizi</name>
			<code>XIZ</code>
			<fips>63</fips>
			<region></region>
		</division>
		<division>
			<name>Qax</name>
			<code>QAX</code>
			<fips>39</fips>
			<region></region>
		</division>
		<division>
			<name>Balak&#xe4;n</name>
			<code>BAL</code>
			<fips>10</fips>
			<region></region>
		</division>
		<division>
			<name>Xa&#xe7;maz</name>
			<code>XAC</code>
			<fips>60</fips>
			<region></region>
		</division>
		<division>
			<name>Astara</name>
			<code>AST</code>
			<fips>08</fips>
			<region></region>
		</division>
		<division>
			<name>Salyan</name>
			<code>SAL</code>
			<fips>49</fips>
			<region></region>
		</division>
		<division>
			<name>Bab&#xe4;k</name>
			<code>BAB</code>
			<fips></fips>
			<region>NX</region>
		</division>
		<division>
			<name>Sumqayit</name>
			<code>SM</code>
			<fips>54</fips>
			<region></region>
		</division>
		<division>
			<name>Ucar</name>
			<code>UCA</code>
			<fips>59</fips>
			<region></region>
		</division>
		<division>
			<name>Susa</name>
			<code>SUS</code>
			<fips>56</fips>
			<region></region>
		</division>
		<division>
			<name>Qobustan</name>
			<code>QOB</code>
			<fips>41</fips>
			<region></region>
		</division>
		<division>
			<name>K&#xfc;rd&#xe4;mir</name>
			<code>KUR</code>
			<fips>27</fips>
			<region></region>
		</division>
		<division>
			<name>Z&#xe4;ngilan</name>
			<code>ZAN</code>
			<fips>69</fips>
			<region></region>
		</division>
		<division>
			<name>Xocali</name>
			<code>XCI</code>
			<fips>64</fips>
			<region></region>
		</division>
		<division>
			<name>S&#xe4;mkir</name>
			<code>SKR</code>
			<fips>51</fips>
			<region></region>
		</division>
		<num>031</num>
	</territory>
	<territory>
		<name>Bahamas</name>
		<alpha2>BS</alpha2>
		<alpha3>BHS</alpha3>
		<division>
			<name>Sandy Point</name>
			<code>SP</code>
			<fips>34</fips>
			<region></region>
		</division>
		<division>
			<name>Marsh Harbour</name>
			<code>MH</code>
			<fips>31</fips>
			<region></region>
		</division>
		<division>
			<name>Fresh Creek</name>
			<code>FC</code>
			<fips>26</fips>
			<region></region>
		</division>
		<division>
			<name>Green Turtle Cay</name>
			<code>GT</code>
			<fips>28</fips>
			<region></region>
		</division>
		<division>
			<name>Harbour Island</name>
			<code>HI</code>
			<fips>22</fips>
			<region></region>
		</division>
		<division>
			<name>Acklins and Crooked Islands</name>
			<code>AC</code>
			<fips>24</fips>
			<region></region>
		</division>
		<division>
			<name>Cat Island</name>
			<code>CI</code>
			<fips>06</fips>
			<region></region>
		</division>
		<division>
			<name>Rock Sound</name>
			<code>RS</code>
			<fips>33</fips>
			<region></region>
		</division>
		<division>
			<name>Nicholls Town and Berry Islands</name>
			<code>NB</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Freeport</name>
			<code>FP</code>
			<fips>25</fips>
			<region></region>
		</division>
		<division>
			<name>High Rock</name>
			<code>HR</code>
			<fips>29</fips>
			<region></region>
		</division>
		<division>
			<name>Bimini</name>
			<code>BI</code>
			<fips>05</fips>
			<region></region>
		</division>
		<division>
			<name>Governor's Harbour</name>
			<code>GH</code>
			<fips>27</fips>
			<region></region>
		</division>
		<division>
			<name>Inagua</name>
			<code>IN</code>
			<fips>13</fips>
			<region></region>
		</division>
		<division>
			<name>Exuma</name>
			<code>EX</code>
			<fips>10</fips>
			<region></region>
		</division>
		<division>
			<name>San Salvador and Rum Cay</name>
			<code>SR</code>
			<fips>35</fips>
			<region></region>
		</division>
		<division>
			<name>Mayaguana</name>
			<code>MG</code>
			<fips>16</fips>
			<region></region>
		</division>
		<division>
			<name>Long Island</name>
			<code>LI</code>
			<fips>15</fips>
			<region></region>
		</division>
		<division>
			<name>Ragged Island</name>
			<code>RI</code>
			<fips>18</fips>
			<region></region>
		</division>
		<division>
			<name>Kemps Bay</name>
			<code>KB</code>
			<fips>30</fips>
			<region></region>
		</division>
		<division>
			<name>New Providence</name>
			<code>NP</code>
			<fips>23</fips>
			<region></region>
		</division>
		<num>044</num>
	</territory>
	<territory>
		<name>Bahrain</name>
		<alpha2>BH</alpha2>
		<alpha3>BHR</alpha3>
		<division>
			<name>Madinat `Is&#xe1;</name>
			<code>08</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Al Mintaqah ash Shamaliyah</name>
			<code>05</code>
			<fips>10</fips>
			<region></region>
		</division>
		<division>
			<name>Ar Rifa</name>
			<code>09</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Al Hadd</name>
			<code>01</code>
			<fips>01</fips>
			<region></region>
		</division>
		<division>
			<name>Madinat Hamad</name>
			<code>12</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Al Mintaqah al Wust&#xe1;</name>
			<code>07</code>
			<fips>11</fips>
			<region></region>
		</division>
		<division>
			<name>Al Manamah</name>
			<code>03</code>
			<fips>02</fips>
			<region></region>
		</division>
		<division>
			<name>Sitrah</name>
			<code>06</code>
			<fips>06</fips>
			<region></region>
		</division>
		<division>
			<name>Jidd Hafs</name>
			<code>04</code>
			<fips>05</fips>
			<region></region>
		</division>
		<division>
			<name>Al Mintaqah al Gharbiyah</name>
			<code>10</code>
			<fips>08</fips>
			<region></region>
		</division>
		<division>
			<name>Al Muharraq</name>
			<code>02</code>
			<fips>03</fips>
			<region></region>
		</division>
		<division>
			<name>Mintaqat Juzur Hawar</name>
			<code>11</code>
			<fips>09</fips>
			<region></region>
		</division>
		<num>048</num>
	</territory>
	<territory>
		<name>Bangladesh</name>
		<alpha2>BD</alpha2>
		<alpha3>BGD</alpha3>
		<division>
			<name>Gazipur zila</name>
			<code>18</code>
			<fips></fips>
			<region>3</region>
		</division>
		<division>
			<name>Patuakhali zila</name>
			<code>51</code>
			<fips></fips>
			<region>1</region>
		</division>
		<division>
			<name>Jhalakati zila</name>
			<code>25</code>
			<fips></fips>
			<region>1</region>
		</division>
		<division>
			<name>Mymensingh zila</name>
			<code>34</code>
			<fips></fips>
			<region>3</region>
		</division>
		<division>
			<name>Gaibandha zila</name>
			<code>19</code>
			<fips></fips>
			<region>5</region>
		</division>
		<division>
			<name>Naogaon zila</name>
			<code>48</code>
			<fips></fips>
			<region>5</region>
		</division>
		<division>
			<name>Lakshmipur zila</name>
			<code>31</code>
			<fips></fips>
			<region>2</region>
		</division>
		<division>
			<name>Chuadanga zila</name>
			<code>12</code>
			<fips></fips>
			<region>4</region>
		</division>
		<division>
			<name>Brahmanbaria zila</name>
			<code>04</code>
			<fips></fips>
			<region>2</region>
		</division>
		<division>
			<name>Moulvibazar zila</name>
			<code>38</code>
			<fips></fips>
			<region>6</region>
		</division>
		<division>
			<name>Sylhet zila</name>
			<code>60</code>
			<fips></fips>
			<region>6</region>
		</division>
		<division>
			<name>Dinajpur zila</name>
			<code>14</code>
			<fips></fips>
			<region>5</region>
		</division>
		<division>
			<name>Jhenaidah zila</name>
			<code>23</code>
			<fips></fips>
			<region>4</region>
		</division>
		<division>
			<name>Bogra zila</name>
			<code>03</code>
			<fips></fips>
			<region>5</region>
		</division>
		<division>
			<name>Satkhira zila</name>
			<code>58</code>
			<fips></fips>
			<region>4</region>
		</division>
		<division>
			<name>Dhaka zila</name>
			<code>13</code>
			<fips>81</fips>
			<region>3</region>
		</division>
		<division>
			<name>Sunamganj zila</name>
			<code>61</code>
			<fips></fips>
			<region>6</region>
		</division>
		<division>
			<name>Chandpur zila</name>
			<code>09</code>
			<fips></fips>
			<region>2</region>
		</division>
		<division>
			<name>Barisal zila</name>
			<code>06</code>
			<fips></fips>
			<region>1</region>
		</division>
		<division>
			<name>Bhola zila</name>
			<code>07</code>
			<fips></fips>
			<region>1</region>
		</division>
		<division>
			<name>Rajshahi zila</name>
			<code>54</code>
			<fips></fips>
			<region>5</region>
		</division>
		<division>
			<name>Kurigram zila</name>
			<code>28</code>
			<fips></fips>
			<region>5</region>
		</division>
		<division>
			<name>Cox's Bazar zila</name>
			<code>11</code>
			<fips></fips>
			<region>2</region>
		</division>
		<division>
			<name>Faridpur zila</name>
			<code>15</code>
			<fips></fips>
			<region>3</region>
		</division>
		<division>
			<name>Bagerhat zila</name>
			<code>05</code>
			<fips></fips>
			<region>4</region>
		</division>
		<division>
			<name>Gopalganj zila</name>
			<code>17</code>
			<fips></fips>
			<region>3</region>
		</division>
		<division>
			<name>Manikganj zila</name>
			<code>33</code>
			<fips></fips>
			<region>3</region>
		</division>
		<division>
			<name>Rangamati zila</name>
			<code>56</code>
			<fips></fips>
			<region>2</region>
		</division>
		<division>
			<name>Magura zila</name>
			<code>37</code>
			<fips></fips>
			<region>4</region>
		</division>
		<division>
			<name>Rajbari zila</name>
			<code>53</code>
			<fips></fips>
			<region>3</region>
		</division>
		<division>
			<name>Meherpur zila</name>
			<code>39</code>
			<fips></fips>
			<region>4</region>
		</division>
		<division>
			<name>Kushtia zila</name>
			<code>30</code>
			<fips></fips>
			<region>4</region>
		</division>
		<division>
			<name>Comilla zila</name>
			<code>08</code>
			<fips></fips>
			<region>2</region>
		</division>
		<division>
			<name>Feni zila</name>
			<code>16</code>
			<fips></fips>
			<region>2</region>
		</division>
		<division>
			<name>Jessore zila</name>
			<code>22</code>
			<fips></fips>
			<region>4</region>
		</division>
		<division>
			<name>Kishoreganj zila</name>
			<code>26</code>
			<fips></fips>
			<region>3</region>
		</division>
		<division>
			<name>Shariatpur zila</name>
			<code>62</code>
			<fips></fips>
			<region>3</region>
		</division>
		<division>
			<name>Nilphamari zila</name>
			<code>46</code>
			<fips></fips>
			<region>5</region>
		</division>
		<division>
			<name>Narsingdi zila</name>
			<code>42</code>
			<fips></fips>
			<region>3</region>
		</division>
		<division>
			<name>Barguna zila</name>
			<code>02</code>
			<fips></fips>
			<region>1</region>
		</division>
		<division>
			<name>Tangail zila</name>
			<code>63</code>
			<fips></fips>
			<region>3</region>
		</division>
		<division>
			<name>Narayanganj zila</name>
			<code>40</code>
			<fips></fips>
			<region>3</region>
		</division>
		<division>
			<name>Chittagong zila</name>
			<code>10</code>
			<fips></fips>
			<region>2</region>
		</division>
		<division>
			<name>Natore zila</name>
			<code>44</code>
			<fips></fips>
			<region>5</region>
		</division>
		<division>
			<name>Madaripur zila</name>
			<code>36</code>
			<fips></fips>
			<region>3</region>
		</division>
		<division>
			<name>Lalmonirhat zila</name>
			<code>32</code>
			<fips></fips>
			<region>5</region>
		</division>
		<division>
			<name>Pabna zila</name>
			<code>49</code>
			<fips></fips>
			<region>5</region>
		</division>
		<division>
			<name>Narail zila</name>
			<code>43</code>
			<fips></fips>
			<region>4</region>
		</division>
		<division>
			<name>Habiganj zila</name>
			<code>20</code>
			<fips></fips>
			<region>6</region>
		</division>
		<division>
			<name>Sherpur zila</name>
			<code>57</code>
			<fips></fips>
			<region>3</region>
		</division>
		<division>
			<name>Jaipurhat zila</name>
			<code>24</code>
			<fips></fips>
			<region>5</region>
		</division>
		<division>
			<name>Nawabganj zila</name>
			<code>45</code>
			<fips></fips>
			<region>5</region>
		</division>
		<division>
			<name>Bandarban zila</name>
			<code>01</code>
			<fips></fips>
			<region>2</region>
		</division>
		<division>
			<name>Khulna zila</name>
			<code>27</code>
			<fips></fips>
			<region>4</region>
		</division>
		<division>
			<name>Panchagarh zila</name>
			<code>52</code>
			<fips></fips>
			<region>5</region>
		</division>
		<division>
			<name>Netrakona zila</name>
			<code>41</code>
			<fips></fips>
			<region>3</region>
		</division>
		<division>
			<name>Noakhali zila</name>
			<code>47</code>
			<fips></fips>
			<region>2</region>
		</division>
		<division>
			<name>Jamalpur zila</name>
			<code>21</code>
			<fips></fips>
			<region>3</region>
		</division>
		<division>
			<name>Sirajganj zila</name>
			<code>59</code>
			<fips></fips>
			<region>5</region>
		</division>
		<division>
			<name>Thakurgaon zila</name>
			<code>64</code>
			<fips></fips>
			<region>5</region>
		</division>
		<division>
			<name>Pirojpur zila</name>
			<code>50</code>
			<fips></fips>
			<region>1</region>
		</division>
		<division>
			<name>Munshiganj zila</name>
			<code>35</code>
			<fips></fips>
			<region>3</region>
		</division>
		<division>
			<name>Khagrachari zila</name>
			<code>29</code>
			<fips></fips>
			<region>2</region>
		</division>
		<division>
			<name>Rangpur zila</name>
			<code>55</code>
			<fips></fips>
			<region>5</region>
		</division>
		<num>050</num>
	</territory>
	<territory>
		<name>Barbados</name>
		<alpha2>BB</alpha2>
		<alpha3>BRB</alpha3>
		<num>052</num>
	</territory>
	<territory>
		<name>Belarus</name>
		<alpha2>BY</alpha2>
		<alpha3>BLR</alpha3>
		<division>
			<name>Mahilyow</name>
			<code>MA</code>
			<fips>06</fips>
			<region></region>
		</division>
		<division>
			<name>Minsk</name>
			<code>MI</code>
			<fips>05</fips>
			<region></region>
		</division>
		<division>
			<name>Homyel'</name>
			<code>HO</code>
			<fips>02</fips>
			<region></region>
		</division>
		<division>
			<name>Vitsyebsk</name>
			<code>VI</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Brest</name>
			<code>BR</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Hrodna</name>
			<code>HR</code>
			<fips></fips>
			<region></region>
		</division>
		<num>112</num>
	</territory>
	<territory>
		<name>Belgium</name>
		<alpha2>BE</alpha2>
		<alpha3>BEL</alpha3>
		<division>
			<name>Limburg</name>
			<code>VLI</code>
			<fips>05</fips>
			<region>VLG</region>
		</division>
		<division>
			<name>Li&#xe8;ge</name>
			<code>WLG</code>
			<fips>04</fips>
			<region>WAL</region>
		</division>
		<division>
			<name>West Flanders </name>
			<code>VWV</code>
			<fips>09</fips>
			<region>VLG</region>
		</division>
		<division>
			<name>Wallon Brabant</name>
			<code>WBR</code>
			<fips>10</fips>
			<region>WAL</region>
		</division>
		<division>
			<name>Vlaams Brabant</name>
			<code>VBR</code>
			<fips>12</fips>
			<region>VLG</region>
		</division>
		<division>
			<name>East Flanders</name>
			<code>VOV</code>
			<fips>08</fips>
			<region>VLG</region>
		</division>
		<division>
			<name>Hainaut</name>
			<code>WHT</code>
			<fips>03</fips>
			<region>WAL</region>
		</division>
		<division>
			<name>Luxembourg</name>
			<code>WLX</code>
			<fips>06</fips>
			<region>WAL</region>
		</division>
		<division>
			<name>Namur</name>
			<code>WNA</code>
			<fips>07</fips>
			<region>WAL</region>
		</division>
		<division>
			<name>Antwerp</name>
			<code>VAN</code>
			<fips>01</fips>
			<region>VLG</region>
		</division>
		<num>056</num>
	</territory>
	<territory>
		<name>Belize</name>
		<alpha2>BZ</alpha2>
		<alpha3>BLZ</alpha3>
		<division>
			<name>Stann Creek</name>
			<code>SC</code>
			<fips>05</fips>
			<region></region>
		</division>
		<division>
			<name>Belize</name>
			<code>BZ</code>
			<fips>01</fips>
			<region></region>
		</division>
		<division>
			<name>Toledo</name>
			<code>TOL</code>
			<fips>06</fips>
			<region></region>
		</division>
		<division>
			<name>Orange Walk</name>
			<code>OW</code>
			<fips>04</fips>
			<region></region>
		</division>
		<division>
			<name>Corozal</name>
			<code>CZL</code>
			<fips>03</fips>
			<region></region>
		</division>
		<division>
			<name>Cayo</name>
			<code>CY</code>
			<fips>02</fips>
			<region></region>
		</division>
		<num>084</num>
	</territory>
	<territory>
		<name>Benin</name>
		<alpha2>BJ</alpha2>
		<alpha3>BEN</alpha3>
		<division>
			<name>Alibori</name>
			<code>AL</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Ou&#xe9;m&#xe9;</name>
			<code>OU</code>
			<fips>05</fips>
			<region></region>
		</division>
		<division>
			<name>Littoral</name>
			<code>LI</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Zou</name>
			<code>ZO</code>
			<fips>06</fips>
			<region></region>
		</division>
		<division>
			<name>Borgou</name>
			<code>BO</code>
			<fips>03</fips>
			<region></region>
		</division>
		<division>
			<name>Atlantique</name>
			<code>AQ</code>
			<fips>02</fips>
			<region></region>
		</division>
		<division>
			<name>Collines</name>
			<code>CO</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Kouffo</name>
			<code>KO</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Mono</name>
			<code>MO</code>
			<fips>04</fips>
			<region></region>
		</division>
		<division>
			<name>Plateau</name>
			<code>PL</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Atakora</name>
			<code>AK</code>
			<fips>01</fips>
			<region></region>
		</division>
		<division>
			<name>Donga</name>
			<code>DO</code>
			<fips></fips>
			<region></region>
		</division>
		<num>204</num>
	</territory>
	<territory>
		<name>Bermuda</name>
		<alpha2>BM</alpha2>
		<alpha3>BMU</alpha3>
		<num>060</num>
	</territory>
	<territory>
		<name>Bhutan</name>
		<alpha2>BT</alpha2>
		<alpha3>BTN</alpha3>
		<division>
			<name>Wangdue Phodrang</name>
			<code>24</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Gasa</name>
			<code>GA</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Paro</name>
			<code>11</code>
			<fips>13</fips>
			<region></region>
		</division>
		<division>
			<name>Ha</name>
			<code>13</code>
			<fips>10</fips>
			<region></region>
		</division>
		<division>
			<name>Sarpang</name>
			<code>31</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Tsirang</name>
			<code>21</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Punakha</name>
			<code>23</code>
			<fips>15</fips>
			<region></region>
		</division>
		<division>
			<name>Zhemgang</name>
			<code>34</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Samtse</name>
			<code>14</code>
			<fips>16</fips>
			<region></region>
		</division>
		<division>
			<name>Chhukha</name>
			<code>12</code>
			<fips>06</fips>
			<region></region>
		</division>
		<division>
			<name>Trashi Yangtse</name>
			<code>TY</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Lhuentse</name>
			<code>44</code>
			<fips>11</fips>
			<region></region>
		</division>
		<division>
			<name>Trongsa</name>
			<code>32</code>
			<fips>21</fips>
			<region></region>
		</division>
		<division>
			<name>Dagana</name>
			<code>22</code>
			<fips>08</fips>
			<region></region>
		</division>
		<division>
			<name>Thimphu</name>
			<code>15</code>
			<fips>20</fips>
			<region></region>
		</division>
		<division>
			<name>Monggar</name>
			<code>42</code>
			<fips>12</fips>
			<region></region>
		</division>
		<division>
			<name>Pemagatshel</name>
			<code>43</code>
			<fips>14</fips>
			<region></region>
		</division>
		<division>
			<name>Samdrup Jongkha</name>
			<code>45</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Trashigang</name>
			<code>41</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Bumthang</name>
			<code>33</code>
			<fips>05</fips>
			<region></region>
		</division>
		<num>064</num>
	</territory>
	<territory>
		<name>Bolivia</name>
		<alpha2>BO</alpha2>
		<alpha3>BOL</alpha3>
		<division>
			<name>Tarija</name>
			<code>T</code>
			<fips>09</fips>
			<region></region>
		</division>
		<division>
			<name>Cochabamba</name>
			<code>C</code>
			<fips>02</fips>
			<region></region>
		</division>
		<division>
			<name>Santa Cruz</name>
			<code>S</code>
			<fips>08</fips>
			<region></region>
		</division>
		<division>
			<name>Oruro</name>
			<code>O</code>
			<fips>05</fips>
			<region></region>
		</division>
		<division>
			<name>El Beni</name>
			<code>B</code>
			<fips>03</fips>
			<region></region>
		</division>
		<division>
			<name>Potos&#xed;</name>
			<code>P</code>
			<fips>07</fips>
			<region></region>
		</division>
		<division>
			<name>Chuquisaca</name>
			<code>H</code>
			<fips>01</fips>
			<region></region>
		</division>
		<division>
			<name>Pando</name>
			<code>N</code>
			<fips>06</fips>
			<region></region>
		</division>
		<division>
			<name>La Paz</name>
			<code>L</code>
			<fips>04</fips>
			<region></region>
		</division>
		<num>068</num>
	</territory>
	<territory>
		<name>Bosnia and Herzegovina</name>
		<alpha2>BA</alpha2>
		<alpha3>BIH</alpha3>
		<division>
			<name>Federacija Bosna i Hercegovina</name>
			<code>BIH</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Republika Srpska</name>
			<code>SRP</code>
			<fips>SS</fips>
			<region></region>
		</division>
		<num>070</num>
	</territory>
	<territory>
		<name>Botswana</name>
		<alpha2>BW</alpha2>
		<alpha3>BWA</alpha3>
		<division>
			<name>Central</name>
			<code>CE</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Ghanzi</name>
			<code>GH</code>
			<fips>03</fips>
			<region></region>
		</division>
		<division>
			<name>Kgalagadi</name>
			<code>KG</code>
			<fips>04</fips>
			<region></region>
		</division>
		<division>
			<name>Southern</name>
			<code>SO</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Kweneng</name>
			<code>KW</code>
			<fips>06</fips>
			<region></region>
		</division>
		<division>
			<name>North-West</name>
			<code>NW</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>South-East</name>
			<code>SE</code>
			<fips>09</fips>
			<region></region>
		</division>
		<division>
			<name>Kgatleng</name>
			<code>KL</code>
			<fips>05</fips>
			<region></region>
		</division>
		<division>
			<name>North-East</name>
			<code>NE</code>
			<fips></fips>
			<region></region>
		</division>
		<num>072</num>
	</territory>
	<territory>
		<name>Bouvet Island</name>
		<alpha2>BV</alpha2>
		<alpha3>BVT</alpha3>
		<num>074</num>
	</territory>
	<territory>
		<name>Brazil</name>
		<alpha2>BR</alpha2>
		<alpha3>BRA</alpha3>
		<division>
			<name>Goi&#xe1;s</name>
			<code>GO</code>
			<fips>10</fips>
			<region></region>
		</division>
		<division>
			<name>Bahia</name>
			<code>BA</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Rio Grande do Norte</name>
			<code>RN</code>
			<fips>22</fips>
			<region></region>
		</division>
		<division>
			<name>Maranh&#xe3;o</name>
			<code>MA</code>
			<fips>13</fips>
			<region></region>
		</division>
		<division>
			<name>Sergipe</name>
			<code>SE</code>
			<fips>28</fips>
			<region></region>
		</division>
		<division>
			<name>Rond&#xf4;nia</name>
			<code>RO</code>
			<fips>24</fips>
			<region></region>
		</division>
		<division>
			<name>Alagoas</name>
			<code>AL</code>
			<fips>02</fips>
			<region></region>
		</division>
		<division>
			<name>Santa Catarina</name>
			<code>SC</code>
			<fips>26</fips>
			<region></region>
		</division>
		<division>
			<name>Esp&#xed;rito Santo</name>
			<code>ES</code>
			<fips>08</fips>
			<region></region>
		</division>
		<division>
			<name>Cear&#xe1;</name>
			<code>CE</code>
			<fips>06</fips>
			<region></region>
		</division>
		<division>
			<name>S&#xe3;o Paulo</name>
			<code>SP</code>
			<fips>27</fips>
			<region></region>
		</division>
		<division>
			<name>Par&#xe1;</name>
			<code>PA</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Distrito Federal</name>
			<code>DF</code>
			<fips>07</fips>
			<region></region>
		</division>
		<division>
			<name>Amap&#xe1;</name>
			<code>AP</code>
			<fips>03</fips>
			<region></region>
		</division>
		<division>
			<name>Pernambuco</name>
			<code>PE</code>
			<fips>19</fips>
			<region></region>
		</division>
		<division>
			<name>Mato Grosso do Sul</name>
			<code>MS</code>
			<fips>11</fips>
			<region></region>
		</division>
		<division>
			<name>Rio Grande do Sul</name>
			<code>RS</code>
			<fips>23</fips>
			<region></region>
		</division>
		<division>
			<name>Acre</name>
			<code>AC</code>
			<fips>01</fips>
			<region></region>
		</division>
		<division>
			<name>Piau&#xed;</name>
			<code>PI</code>
			<fips>20</fips>
			<region></region>
		</division>
		<division>
			<name>Para&#xed;ba</name>
			<code>PB</code>
			<fips>17</fips>
			<region></region>
		</division>
		<division>
			<name>Paran&#xe1;</name>
			<code>PR</code>
			<fips>18</fips>
			<region></region>
		</division>
		<division>
			<name>Minas Gerais</name>
			<code>MG</code>
			<fips>15</fips>
			<region></region>
		</division>
		<division>
			<name>Tocantins</name>
			<code>TO</code>
			<fips>31</fips>
			<region></region>
		</division>
		<division>
			<name>Rio de Janeiro</name>
			<code>RJ</code>
			<fips>21</fips>
			<region></region>
		</division>
		<division>
			<name>Roraima</name>
			<code>RR</code>
			<fips>25</fips>
			<region></region>
		</division>
		<division>
			<name>Mato Grosso</name>
			<code>MT</code>
			<fips>14</fips>
			<region></region>
		</division>
		<division>
			<name>Amazonas</name>
			<code>AM</code>
			<fips>04</fips>
			<region></region>
		</division>
		<num>076</num>
	</territory>
	<territory>
		<name>British Indian Ocean Territory</name>
		<alpha2>IO</alpha2>
		<alpha3>IOT</alpha3>
		<num>086</num>
	</territory>
	<territory>
		<name>Brunei Darussalam</name>
		<alpha2>BN</alpha2>
		<alpha3>BRN</alpha3>
		<division>
			<name>Tutong</name>
			<code>TU</code>
			<fips>04</fips>
			<region></region>
		</division>
		<division>
			<name>Belait</name>
			<code>BE</code>
			<fips>01</fips>
			<region></region>
		</division>
		<division>
			<name>Temburong</name>
			<code>TE</code>
			<fips>03</fips>
			<region></region>
		</division>
		<division>
			<name>Brunei-Muara</name>
			<code>BM</code>
			<fips></fips>
			<region></region>
		</division>
		<num>096</num>
	</territory>
	<territory>
		<name>Bulgaria</name>
		<alpha2>BG</alpha2>
		<alpha3>BGR</alpha3>
		<division>
			<name>Silistra</name>
			<code>19</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Pernik</name>
			<code>14</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Ruse</name>
			<code>18</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Kjustendil</name>
			<code>10</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Vraca</name>
			<code>06</code>
			<fips>51</fips>
			<region></region>
		</division>
		<division>
			<name>Varna</name>
			<code>03</code>
			<fips>61</fips>
			<region></region>
		</division>
		<division>
			<name>&#x160;umen</name>
			<code>27</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Targovi&#x161;te</name>
			<code>25</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Sofija-Grad</name>
			<code>22</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Razgrad</name>
			<code>17</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Stara Zagora</name>
			<code>24</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Smoljan</name>
			<code>21</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Lovec</name>
			<code>11</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Montana</name>
			<code>12</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Pleven</name>
			<code>15</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Gabrovo</name>
			<code>07</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Sliven</name>
			<code>20</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Jambol</name>
			<code>28</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Pazard&#x17e;ik</name>
			<code>13</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Vidin</name>
			<code>05</code>
			<fips>47</fips>
			<region></region>
		</division>
		<division>
			<name>Kard&#x17e;ali</name>
			<code>09</code>
			<fips>43</fips>
			<region></region>
		</division>
		<division>
			<name>Haskovo</name>
			<code>26</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Plovdiv</name>
			<code>16</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Sofija</name>
			<code>23</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Veliko Tarnovo</name>
			<code>04</code>
			<fips></fips>
			<region></region>
		</division>
		<num>100</num>
	</territory>
	<territory>
		<name>Burkina Faso</name>
		<alpha2>BF</alpha2>
		<alpha3>BFA</alpha3>
		<division>
			<name>Gnagna</name>
			<code>GNA</code>
			<fips>21</fips>
			<region></region>
		</division>
		<division>
			<name>Banwa</name>
			<code>BAN</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Sanmatenga</name>
			<code>SMT</code>
			<fips>37</fips>
			<region></region>
		</division>
		<division>
			<name>Boulkiemd&#xe9;</name>
			<code>BLK</code>
			<fips>19</fips>
			<region></region>
		</division>
		<division>
			<name>Kouritenga</name>
			<code>KOT</code>
			<fips>28</fips>
			<region></region>
		</division>
		<division>
			<name>Yagha</name>
			<code>YAG</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Oudalan</name>
			<code>OUD</code>
			<fips>33</fips>
			<region></region>
		</division>
		<division>
			<name>L&#xe9;raba</name>
			<code>LER</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Kourw&#xe9;ogo</name>
			<code>KOW</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Como&#xe9;</name>
			<code>COM</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Zoundw&#xe9;ogo</name>
			<code>ZOU</code>
			<fips>44</fips>
			<region></region>
		</division>
		<division>
			<name>Tui</name>
			<code>TUI</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>K&#xe9;n&#xe9;dougou</name>
			<code>KEN</code>
			<fips>25</fips>
			<region></region>
		</division>
		<division>
			<name>Mouhoun</name>
			<code>MOU</code>
			<fips>29</fips>
			<region></region>
		</division>
		<division>
			<name>Nahouri</name>
			<code>NAO</code>
			<fips>31</fips>
			<region></region>
		</division>
		<division>
			<name>Ziro</name>
			<code>ZIR</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Passor&#xe9;</name>
			<code>PAS</code>
			<fips>34</fips>
			<region></region>
		</division>
		<division>
			<name>S&#xe9;no</name>
			<code>SEN</code>
			<fips>38</fips>
			<region></region>
		</division>
		<division>
			<name>Noumbiel</name>
			<code>NOU</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Sissili</name>
			<code>SIS</code>
			<fips>39</fips>
			<region></region>
		</division>
		<division>
			<name>Boulgou</name>
			<code>BLG</code>
			<fips>18</fips>
			<region></region>
		</division>
		<division>
			<name>Nayala</name>
			<code>NAY</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Loroum</name>
			<code>LOR</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Kompienga</name>
			<code>KMP</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Kadiogo</name>
			<code>KAD</code>
			<fips>24</fips>
			<region></region>
		</division>
		<division>
			<name>Poni</name>
			<code>PON</code>
			<fips>35</fips>
			<region></region>
		</division>
		<division>
			<name>Zondoma</name>
			<code>ZON</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Sourou</name>
			<code>SOR</code>
			<fips>41</fips>
			<region></region>
		</division>
		<division>
			<name>Bal&#xe9;</name>
			<code>BAL</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Koulp&#xe9;logo</name>
			<code>KOP</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Sangui&#xe9;</name>
			<code>SNG</code>
			<fips>36</fips>
			<region></region>
		</division>
		<division>
			<name>Bam</name>
			<code>BAM</code>
			<fips>15</fips>
			<region></region>
		</division>
		<division>
			<name>Kossi</name>
			<code>KOS</code>
			<fips>27</fips>
			<region></region>
		</division>
		<division>
			<name>Houet</name>
			<code>HOU</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Komondjari</name>
			<code>KMD</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Ioba</name>
			<code>IOB</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Yatenga</name>
			<code>YAT</code>
			<fips>43</fips>
			<region></region>
		</division>
		<division>
			<name>Ganzourgou</name>
			<code>GAN</code>
			<fips>20</fips>
			<region></region>
		</division>
		<division>
			<name>Gourma</name>
			<code>GOU</code>
			<fips>22</fips>
			<region></region>
		</division>
		<division>
			<name>Baz&#xe8;ga</name>
			<code>BAZ</code>
			<fips>16</fips>
			<region></region>
		</division>
		<division>
			<name>Soum</name>
			<code>SOM</code>
			<fips>40</fips>
			<region></region>
		</division>
		<division>
			<name>Namentenga</name>
			<code>NAM</code>
			<fips>30</fips>
			<region></region>
		</division>
		<division>
			<name>Tapoa</name>
			<code>TAP</code>
			<fips>42</fips>
			<region></region>
		</division>
		<division>
			<name>Bougouriba</name>
			<code>BGR</code>
			<fips>17</fips>
			<region></region>
		</division>
		<division>
			<name>Oubritenga</name>
			<code>OUB</code>
			<fips>32</fips>
			<region></region>
		</division>
		<num>854</num>
	</territory>
	<territory>
		<name>Burundi</name>
		<alpha2>BI</alpha2>
		<alpha3>BDI</alpha3>
		<division>
			<name>Mwaro</name>
			<code>MW</code>
			<fips></fips>
			<region></region>
		</division>
		<num>108</num>
	</territory>
	<territory>
		<name>Cambodia</name>
		<alpha2>KH</alpha2>
		<alpha3>KHM</alpha3>
		<division>
			<name>Krong Pailin [Krong Pailin]</name>
			<code>24</code>
			<fips></fips>
			<region></region>
		</division>
		<num>116</num>
	</territory>
	<territory>
		<name>Cameroon</name>
		<alpha2>CM</alpha2>
		<alpha3>CMR</alpha3>
		<division>
			<name>Centre</name>
			<code>CE</code>
			<fips>11</fips>
			<region></region>
		</division>
		<division>
			<name>Littoral</name>
			<code>LT</code>
			<fips>05</fips>
			<region></region>
		</division>
		<division>
			<name>South</name>
			<code>SU</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>North</name>
			<code>NO</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>West</name>
			<code>OU</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Adamaoua</name>
			<code>AD</code>
			<fips>10</fips>
			<region></region>
		</division>
		<division>
			<name>North-West</name>
			<code>NW</code>
			<fips>04</fips>
			<region></region>
		</division>
		<division>
			<name>South-West</name>
			<code>SW</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Far North</name>
			<code>EN</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>East</name>
			<code>ES</code>
			<fips></fips>
			<region></region>
		</division>
		<num>120</num>
	</territory>
	<territory>
		<name>Canada</name>
		<alpha2>CA</alpha2>
		<alpha3>CAN</alpha3>
		<division>
			<name>Saskatchewan</name>
			<code>SK</code>
			<fips>11</fips>
			<region></region>
		</division>
		<division>
			<name>Ontario</name>
			<code>ON</code>
			<fips>08</fips>
			<region></region>
		</division>
		<division>
			<name>Nunavut</name>
			<code>NU</code>
			<fips>14</fips>
			<region></region>
		</division>
		<division>
			<name>Alberta</name>
			<code>AB</code>
			<fips>01</fips>
			<region></region>
		</division>
		<division>
			<name>Quebec</name>
			<code>QC</code>
			<fips>10</fips>
			<region></region>
		</division>
		<division>
			<name>Manitoba</name>
			<code>MB</code>
			<fips>03</fips>
			<region></region>
		</division>
		<division>
			<name>British Columbia</name>
			<code>BC</code>
			<fips>02</fips>
			<region></region>
		</division>
		<division>
			<name>Nova Scotia</name>
			<code>NS</code>
			<fips>07</fips>
			<region></region>
		</division>
		<division>
			<name>Northwest Territories</name>
			<code>NT</code>
			<fips>13</fips>
			<region></region>
		</division>
		<division>
			<name>New Brunswick</name>
			<code>NB</code>
			<fips>04</fips>
			<region></region>
		</division>
		<division>
			<name>Yukon Territory</name>
			<code>YT</code>
			<fips>12</fips>
			<region></region>
		</division>
		<division>
			<name>Newfoundland and Labrador</name>
			<code>NL</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Prince Edward Island</name>
			<code>PE</code>
			<fips>09</fips>
			<region></region>
		</division>
		<num>124</num>
	</territory>
	<territory>
		<name>Canary Islands</name>
		<alpha2>IC</alpha2>
		<alpha3></alpha3>
		<num></num>
		<note>reserved on request of WCO to represent area outside EU customs territory</note>
		<ext>wco</ext>
	</territory>
	<territory>
		<name>Cape Verde</name>
		<alpha2>CV</alpha2>
		<alpha3>CPV</alpha3>
		<division>
			<name>Ribeira Grande</name>
			<code>RG</code>
			<fips>07</fips>
			<region>B</region>
		</division>
		<division>
			<name>Boa Vista</name>
			<code>BV</code>
			<fips>01</fips>
			<region>B</region>
		</division>
		<division>
			<name>Tarrafal</name>
			<code>TA</code>
			<fips>12</fips>
			<region>S</region>
		</division>
		<division>
			<name>Brava</name>
			<code>BR</code>
			<fips>02</fips>
			<region>S</region>
		</division>
		<division>
			<name>Mosteiros</name>
			<code>MO</code>
			<fips></fips>
			<region>S</region>
		</division>
		<division>
			<name>S&#xe3;o Vicente</name>
			<code>SV</code>
			<fips>11</fips>
			<region>B</region>
		</division>
		<division>
			<name>Santa Catarina</name>
			<code>CA</code>
			<fips>09</fips>
			<region>S</region>
		</division>
		<division>
			<name>Praia</name>
			<code>PR</code>
			<fips>06</fips>
			<region>S</region>
		</division>
		<division>
			<name>S&#xe3;o Nicolau</name>
			<code>SN</code>
			<fips>10</fips>
			<region>B</region>
		</division>
		<division>
			<name>S&#xe3;o Filipe</name>
			<code>SF</code>
			<fips></fips>
			<region>S</region>
		</division>
		<division>
			<name>Santa Cruz</name>
			<code>CR</code>
			<fips></fips>
			<region>S</region>
		</division>
		<division>
			<name>Maio</name>
			<code>MA</code>
			<fips>04</fips>
			<region>S</region>
		</division>
		<division>
			<name>Calheta de S&#xe3;o Miguel</name>
			<code>CS</code>
			<fips></fips>
			<region>S</region>
		</division>
		<division>
			<name>Porto Novo</name>
			<code>PN</code>
			<fips></fips>
			<region>B</region>
		</division>
		<division>
			<name>Sal</name>
			<code>SL</code>
			<fips>08</fips>
			<region>B</region>
		</division>
		<division>
			<name>Pa&#xfa;l</name>
			<code>PA</code>
			<fips>05</fips>
			<region>B</region>
		</division>
		<division>
			<name>S&#xe3;o Domingos</name>
			<code>SD</code>
			<fips></fips>
			<region>S</region>
		</division>
		<num>132</num>
	</territory>
	<territory>
		<name>Cayman Islands</name>
		<alpha2>KY</alpha2>
		<alpha3>CYM</alpha3>
		<num>136</num>
	</territory>
	<territory>
		<name>Central African Republic</name>
		<alpha2>CF</alpha2>
		<alpha3>CAF</alpha3>
		<division>
			<name>Ouaka</name>
			<code>UK</code>
			<fips>11</fips>
			<region></region>
		</division>
		<division>
			<name>Bangui</name>
			<code>BGF</code>
			<fips>18</fips>
			<region></region>
		</division>
		<division>
			<name>Bamingui-Bangoran</name>
			<code>BB</code>
			<fips>01</fips>
			<region></region>
		</division>
		<division>
			<name>Ouham-Pend&#xe9;</name>
			<code>OP</code>
			<fips>13</fips>
			<region></region>
		</division>
		<division>
			<name>Nana-Mamb&#xe9;r&#xe9;</name>
			<code>NM</code>
			<fips>09</fips>
			<region></region>
		</division>
		<division>
			<name>Lobaye</name>
			<code>LB</code>
			<fips>07</fips>
			<region></region>
		</division>
		<division>
			<name>Basse-Kotto</name>
			<code>BK</code>
			<fips>02</fips>
			<region></region>
		</division>
		<division>
			<name>Sangha-Mba&#xe9;r&#xe9;</name>
			<code>SE</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Nana-Gr&#xe9;bizi</name>
			<code>KB</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Mamb&#xe9;r&#xe9;-Kad&#xe9;&#x3f;</name>
			<code>HS</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Vakaga</name>
			<code>VK</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Haute-Kotto</name>
			<code>HK</code>
			<fips>03</fips>
			<region></region>
		</division>
		<division>
			<name>Ombella-Mpoko</name>
			<code>MP</code>
			<fips>17</fips>
			<region></region>
		</division>
		<division>
			<name>K&#xe9;mo</name>
			<code>KG</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Ouham</name>
			<code>AC</code>
			<fips>12</fips>
			<region></region>
		</division>
		<division>
			<name>Mbomou</name>
			<code>MB</code>
			<fips>08</fips>
			<region></region>
		</division>
		<division>
			<name>Haut-Mbomou</name>
			<code>HM</code>
			<fips>05</fips>
			<region></region>
		</division>
		<num>140</num>
	</territory>
	<territory>
		<name>Ceuta and Melilla</name>
		<alpha2>EA</alpha2>
		<alpha3></alpha3>
		<num></num>
		<note>reserved on request of WCO to represent area outside EU customs territory</note>
		<ext>wco</ext>
	</territory>
	<territory>
		<name>Chad</name>
		<alpha2>TD</alpha2>
		<alpha3>TCD</alpha3>
		<division>
			<name>Moyen-Chari</name>
			<code>MC</code>
			<fips>11</fips>
			<region></region>
		</division>
		<division>
			<name>Borkou-Ennedi-Tibesti</name>
			<code>BET</code>
			<fips>03</fips>
			<region></region>
		</division>
		<division>
			<name>Salamat</name>
			<code>SA</code>
			<fips>13</fips>
			<region></region>
		</division>
		<division>
			<name>Mayo-K&#xe9;bbi</name>
			<code>MK</code>
			<fips>10</fips>
			<region></region>
		</division>
		<division>
			<name>Logone-Oriental</name>
			<code>LR</code>
			<fips>09</fips>
			<region></region>
		</division>
		<division>
			<name>Logone-Occidental</name>
			<code>LO</code>
			<fips>08</fips>
			<region></region>
		</division>
		<division>
			<name>Kanem</name>
			<code>KA</code>
			<fips>06</fips>
			<region></region>
		</division>
		<division>
			<name>Biltine</name>
			<code>BI</code>
			<fips>02</fips>
			<region></region>
		</division>
		<division>
			<name>Batha</name>
			<code>BA</code>
			<fips>01</fips>
			<region></region>
		</division>
		<division>
			<name>Chari-Baguirmi</name>
			<code>CB</code>
			<fips>04</fips>
			<region></region>
		</division>
		<division>
			<name>Gu&#xe9;ra</name>
			<code>GR</code>
			<fips>05</fips>
			<region></region>
		</division>
		<division>
			<name>Tandjil&#xe9;</name>
			<code>TA</code>
			<fips>14</fips>
			<region></region>
		</division>
		<division>
			<name>Lac</name>
			<code>LC</code>
			<fips>07</fips>
			<region></region>
		</division>
		<division>
			<name>Ouadda&#x3f;</name>
			<code>OD</code>
			<fips>12</fips>
			<region></region>
		</division>
		<num>148</num>
	</territory>
	<territory>
		<name>Chile</name>
		<alpha2>CL</alpha2>
		<alpha3>CHL</alpha3>
		<division>
			<name>Regi&#xf3;n Metropolitana de Santiago</name>
			<code>RM</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Los Lagos</name>
			<code>LL</code>
			<fips>09</fips>
			<region></region>
		</division>
		<division>
			<name>Atacama</name>
			<code>AT</code>
			<fips>05</fips>
			<region></region>
		</division>
		<division>
			<name>Libertador General Bernardo O'Higgins</name>
			<code>LI</code>
			<fips>08</fips>
			<region></region>
		</division>
		<division>
			<name>Tarapac&#xe1;</name>
			<code>TA</code>
			<fips>13</fips>
			<region></region>
		</division>
		<division>
			<name>Maule</name>
			<code>ML</code>
			<fips>11</fips>
			<region></region>
		</division>
		<division>
			<name>B&#xed;o-B&#xed;o</name>
			<code>BI</code>
			<fips>06</fips>
			<region></region>
		</division>
		<division>
			<name>Valpara&#xed;so</name>
			<code>VS</code>
			<fips>01</fips>
			<region></region>
		</division>
		<division>
			<name>Araucan&#xed;a</name>
			<code>AR</code>
			<fips>04</fips>
			<region></region>
		</division>
		<division>
			<name>Coquimbo</name>
			<code>CO</code>
			<fips>07</fips>
			<region></region>
		</division>
		<division>
			<name>Antofagasta</name>
			<code>AN</code>
			<fips>03</fips>
			<region></region>
		</division>
		<division>
			<name>Magallanes</name>
			<code>MA</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Ais&#xe9;n del General Carlos Ib&#xe1;&#xf1;ez del Campo</name>
			<code>AI</code>
			<fips>02</fips>
			<region></region>
		</division>
		<num>152</num>
	</territory>
	<territory>
		<name>China</name>
		<alpha2>CN</alpha2>
		<alpha3>CHN</alpha3>
		<division>
			<name>Gansu</name>
			<code>62</code>
			<fips>15</fips>
			<region></region>
		</division>
		<division>
			<name>Guangxi</name>
			<code>45</code>
			<fips>16</fips>
			<region></region>
		</division>
		<division>
			<name>Ningxia</name>
			<code>64</code>
			<fips>21</fips>
			<region></region>
		</division>
		<division>
			<name>Shanghai</name>
			<code>31</code>
			<fips>23</fips>
			<region></region>
		</division>
		<division>
			<name>Hainan</name>
			<code>46</code>
			<fips>31</fips>
			<region></region>
		</division>
		<division>
			<name>Beijing</name>
			<code>11</code>
			<fips>22</fips>
			<region></region>
		</division>
		<division>
			<name>Sichuan</name>
			<code>51</code>
			<fips>32</fips>
			<region></region>
		</division>
		<division>
			<name>Shaanxi</name>
			<code>61</code>
			<fips>26</fips>
			<region></region>
		</division>
		<division>
			<name>Shandong</name>
			<code>37</code>
			<fips>25</fips>
			<region></region>
		</division>
		<division>
			<name>Hunan</name>
			<code>43</code>
			<fips>11</fips>
			<region></region>
		</division>
		<division>
			<name>Taiwan </name>
			<code>71</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Xizang</name>
			<code>54</code>
			<fips>14</fips>
			<region></region>
		</division>
		<division>
			<name>Aomen</name>
			<code>92</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Fujian</name>
			<code>35</code>
			<fips>07</fips>
			<region></region>
		</division>
		<division>
			<name>Hebei</name>
			<code>13</code>
			<fips>10</fips>
			<region></region>
		</division>
		<division>
			<name>Heilongjiang</name>
			<code>23</code>
			<fips>08</fips>
			<region></region>
		</division>
		<division>
			<name>Qinghai</name>
			<code>63</code>
			<fips>06</fips>
			<region></region>
		</division>
		<division>
			<name>Shanxi</name>
			<code>14</code>
			<fips>24</fips>
			<region></region>
		</division>
		<division>
			<name>Jiangxi</name>
			<code>36</code>
			<fips>03</fips>
			<region></region>
		</division>
		<division>
			<name>Jiangsu</name>
			<code>32</code>
			<fips>04</fips>
			<region></region>
		</division>
		<division>
			<name>Jilin</name>
			<code>22</code>
			<fips>05</fips>
			<region></region>
		</division>
		<division>
			<name>Henan</name>
			<code>41</code>
			<fips>09</fips>
			<region></region>
		</division>
		<division>
			<name>Guizhou</name>
			<code>52</code>
			<fips>18</fips>
			<region></region>
		</division>
		<division>
			<name>Liaoning</name>
			<code>21</code>
			<fips>19</fips>
			<region></region>
		</division>
		<division>
			<name>Anhui</name>
			<code>34</code>
			<fips>01</fips>
			<region></region>
		</division>
		<division>
			<name>Xinjiang</name>
			<code>65</code>
			<fips>13</fips>
			<region></region>
		</division>
		<division>
			<name>Chongqing</name>
			<code>50</code>
			<fips>33</fips>
			<region></region>
		</division>
		<division>
			<name>Yunnan</name>
			<code>53</code>
			<fips>29</fips>
			<region></region>
		</division>
		<division>
			<name>Hubei</name>
			<code>42</code>
			<fips>12</fips>
			<region></region>
		</division>
		<division>
			<name>Tianjin</name>
			<code>12</code>
			<fips>28</fips>
			<region></region>
		</division>
		<division>
			<name>Xianggang</name>
			<code>91</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Zhejiang</name>
			<code>33</code>
			<fips>02</fips>
			<region></region>
		</division>
		<division>
			<name>Neimenggu</name>
			<code>15</code>
			<fips>20</fips>
			<region></region>
		</division>
		<division>
			<name>Guangdong</name>
			<code>44</code>
			<fips>30</fips>
			<region></region>
		</division>
		<num>156</num>
	</territory>
	<territory>
		<name>Christmas Island</name>
		<alpha2>CX</alpha2>
		<alpha3>CXR</alpha3>
		<num>162</num>
	</territory>
	<territory>
		<name>Clipperton Island</name>
		<alpha2>CP</alpha2>
		<alpha3></alpha3>
		<num></num>
		<note>reserved on request of ITU</note>
		<ext>itu</ext>
	</territory>
	<territory>
		<name>Cocos (Keeling) Islands</name>
		<alpha2>CC</alpha2>
		<alpha3>CCK</alpha3>
		<num>166</num>
	</territory>
	<territory>
		<name>Colombia</name>
		<alpha2>CO</alpha2>
		<alpha3>COL</alpha3>
		<division>
			<name>Valle del Cauca</name>
			<code>VAC</code>
			<fips>29</fips>
			<region></region>
		</division>
		<division>
			<name>Magdalena</name>
			<code>MAG</code>
			<fips>38</fips>
			<region></region>
		</division>
		<division>
			<name>Putumayo</name>
			<code>PUT</code>
			<fips>22</fips>
			<region></region>
		</division>
		<division>
			<name>Cauca</name>
			<code>CAU</code>
			<fips>09</fips>
			<region></region>
		</division>
		<division>
			<name>Santander</name>
			<code>SAN</code>
			<fips>26</fips>
			<region></region>
		</division>
		<division>
			<name>Atl&#xe1;ntico</name>
			<code>ATL</code>
			<fips>04</fips>
			<region></region>
		</division>
		<division>
			<name>Sucre</name>
			<code>SUC</code>
			<fips>27</fips>
			<region></region>
		</division>
		<division>
			<name>Cundinamarca</name>
			<code>CUN</code>
			<fips>33</fips>
			<region></region>
		</division>
		<division>
			<name>Antioquia</name>
			<code>ANT</code>
			<fips>02</fips>
			<region></region>
		</division>
		<division>
			<name>Quind&#xed;o</name>
			<code>QUI</code>
			<fips>23</fips>
			<region></region>
		</division>
		<division>
			<name>Cesar</name>
			<code>CES</code>
			<fips>10</fips>
			<region></region>
		</division>
		<division>
			<name>Vichada</name>
			<code>VID</code>
			<fips>31</fips>
			<region></region>
		</division>
		<division>
			<name>Caquet&#xe1;</name>
			<code>CAQ</code>
			<fips>08</fips>
			<region></region>
		</division>
		<division>
			<name>Distrito Capital de Santa Fe de Bogot&#xe1;</name>
			<code>DC</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Norte de Santander</name>
			<code>NSA</code>
			<fips>21</fips>
			<region></region>
		</division>
		<division>
			<name>Risaralda</name>
			<code>RIS</code>
			<fips>24</fips>
			<region></region>
		</division>
		<division>
			<name>Bol&#xed;var</name>
			<code>BOL</code>
			<fips>35</fips>
			<region></region>
		</division>
		<division>
			<name>Nari&#xf1;o</name>
			<code>NAR</code>
			<fips>20</fips>
			<region></region>
		</division>
		<division>
			<name>Arauca</name>
			<code>ARA</code>
			<fips>03</fips>
			<region></region>
		</division>
		<division>
			<name>Meta</name>
			<code>MET</code>
			<fips>19</fips>
			<region></region>
		</division>
		<division>
			<name>Tolima</name>
			<code>TOL</code>
			<fips>28</fips>
			<region></region>
		</division>
		<division>
			<name>La Guajira</name>
			<code>LAG</code>
			<fips>17</fips>
			<region></region>
		</division>
		<division>
			<name>San Andr&#xe9;s, Providencia y Santa Catalina</name>
			<code>SAP</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Casanare</name>
			<code>CAS</code>
			<fips>32</fips>
			<region></region>
		</division>
		<division>
			<name>Choc&#xf3;</name>
			<code>CHO</code>
			<fips>11</fips>
			<region></region>
		</division>
		<division>
			<name>Vaup&#xe9;s</name>
			<code>VAU</code>
			<fips>30</fips>
			<region></region>
		</division>
		<division>
			<name>Caldas</name>
			<code>CAL</code>
			<fips>37</fips>
			<region></region>
		</division>
		<division>
			<name>Guaviare</name>
			<code>GUV</code>
			<fips>14</fips>
			<region></region>
		</division>
		<division>
			<name>Guain&#xed;a</name>
			<code>GUA</code>
			<fips>15</fips>
			<region></region>
		</division>
		<division>
			<name>Huila</name>
			<code>HUI</code>
			<fips>16</fips>
			<region></region>
		</division>
		<division>
			<name>C&#xf3;rdoba</name>
			<code>COR</code>
			<fips>12</fips>
			<region></region>
		</division>
		<division>
			<name>Amazonas</name>
			<code>AMA</code>
			<fips>01</fips>
			<region></region>
		</division>
		<division>
			<name>Boyac&#xe1;</name>
			<code>BOY</code>
			<fips>36</fips>
			<region></region>
		</division>
		<num>170</num>
	</territory>
	<territory>
		<name>Comoros</name>
		<alpha2>KM</alpha2>
		<alpha3>COM</alpha3>
		<division>
			<name>Grande Comore</name>
			<code>G</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Anjouan</name>
			<code>A</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Moh&#xe9;li</name>
			<code>M</code>
			<fips></fips>
			<region></region>
		</division>
		<num>174</num>
	</territory>
	<territory>
		<name>Congo (Brazzaville)</name>
		<alpha2>CG</alpha2>
		<alpha3>COG</alpha3>
		<division>
			<name>Likouala</name>
			<code>7</code>
			<fips>06</fips>
			<region></region>
		</division>
		<division>
			<name>Kouilou</name>
			<code>5</code>
			<fips>04</fips>
			<region></region>
		</division>
		<division>
			<name>Bouenza</name>
			<code>11</code>
			<fips>01</fips>
			<region></region>
		</division>
		<division>
			<name>Pool</name>
			<code>12</code>
			<fips>11</fips>
			<region></region>
		</division>
		<division>
			<name>Cuvette-Ouest</name>
			<code>15</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Brazzaville</name>
			<code>BZV</code>
			<fips>12</fips>
			<region></region>
		</division>
		<division>
			<name>Sangha</name>
			<code>13</code>
			<fips>10</fips>
			<region></region>
		</division>
		<division>
			<name>L&#xe9;koumou</name>
			<code>2</code>
			<fips>05</fips>
			<region></region>
		</division>
		<division>
			<name>Niari</name>
			<code>9</code>
			<fips>07</fips>
			<region></region>
		</division>
		<division>
			<name>Plateaux</name>
			<code>14</code>
			<fips>08</fips>
			<region></region>
		</division>
		<division>
			<name>Cuvette</name>
			<code>8</code>
			<fips>03</fips>
			<region></region>
		</division>
		<num>178</num>
	</territory>
	<territory>
		<name>Congo (Kinshasa)</name>
		<alpha2>CD</alpha2>
		<alpha3>COD</alpha3>
		<division>
			<name>Maniema</name>
			<code>MA</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Kasai-Occidental</name>
			<code>KW</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Orientale</name>
			<code>OR</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Bas-Congo</name>
			<code>BC</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>&#xc9;quateur</name>
			<code>EQ</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Kinshasa</name>
			<code>KN</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Bandundu</name>
			<code>BN</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Nord-Kivu</name>
			<code>NK</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Sud-Kivu</name>
			<code>SK</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Katanga</name>
			<code>KA</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Kasai-Oriental</name>
			<code>KE</code>
			<fips></fips>
			<region></region>
		</division>
		<num>180</num>
	</territory>
	<territory>
		<name>Cook Islands</name>
		<alpha2>CK</alpha2>
		<alpha3>COK</alpha3>
		<num>184</num>
	</territory>
	<territory>
		<name>Costa Rica</name>
		<alpha2>CR</alpha2>
		<alpha3>CRI</alpha3>
		<division>
			<name>Cartago</name>
			<code>C</code>
			<fips>02</fips>
			<region></region>
		</division>
		<division>
			<name>San Jos&#xe9;</name>
			<code>SJ</code>
			<fips>08</fips>
			<region></region>
		</division>
		<division>
			<name>Alajuela</name>
			<code>A</code>
			<fips>01</fips>
			<region></region>
		</division>
		<division>
			<name>Lim&#xf3;n</name>
			<code>L</code>
			<fips>06</fips>
			<region></region>
		</division>
		<division>
			<name>Heredia</name>
			<code>H</code>
			<fips>04</fips>
			<region></region>
		</division>
		<division>
			<name>Guanacaste</name>
			<code>G</code>
			<fips>03</fips>
			<region></region>
		</division>
		<division>
			<name>Puntarenas</name>
			<code>P</code>
			<fips>07</fips>
			<region></region>
		</division>
		<num>188</num>
	</territory>
	<territory>
		<name>Croatia</name>
		<alpha2>HR</alpha2>
		<alpha3>HRV</alpha3>
		<division>
			<name>Vukovarsko-srijemska</name>
			<code>16</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Po&#x17e;e&#x161;ko-slavonska</name>
			<code>11</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Grad Zagreb</name>
			<code>21</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Zadarska</name>
			<code>13</code>
			<fips>19</fips>
			<region></region>
		</division>
		<division>
			<name>Primorsko-goranska</name>
			<code>08</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>&#x160;ibensko-kninska</name>
			<code>15</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Istarska</name>
			<code>18</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Krapinsko-zagorska</name>
			<code>02</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Dubrovacko-neretvanska</name>
			<code>19</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Brodsko-posavska</name>
			<code>12</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Zagrebacka</name>
			<code>01</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Koprivnicko-kri&#x17e;evacka</name>
			<code>06</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Viroviticko-podravska</name>
			<code>10</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Bjelovarsko-bilogorska</name>
			<code>07</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Osjecko-baranjska</name>
			<code>14</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Sisacko-moslavacka</name>
			<code>03</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Vara&#x17e;dinska</name>
			<code>05</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Medimurska</name>
			<code>20</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Splitsko-dalmatinska</name>
			<code>17</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Licko-senjska</name>
			<code>09</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Karlovacka</name>
			<code>04</code>
			<fips></fips>
			<region></region>
		</division>
		<num>191</num>
	</territory>
	<territory>
		<name>Cuba</name>
		<alpha2>CU</alpha2>
		<alpha3>CUB</alpha3>
		<division>
			<name>Santiago de Cuba</name>
			<code>13</code>
			<fips>15</fips>
			<region></region>
		</division>
		<division>
			<name>Holgu&#xed;n</name>
			<code>11</code>
			<fips>12</fips>
			<region></region>
		</division>
		<division>
			<name>Ciego de &#xc1;vila</name>
			<code>08</code>
			<fips>07</fips>
			<region></region>
		</division>
		<division>
			<name>Isla de la Juventud</name>
			<code>99</code>
			<fips>04</fips>
			<region></region>
		</division>
		<division>
			<name>Pinar del R&#xed;o</name>
			<code>01</code>
			<fips>01</fips>
			<region></region>
		</division>
		<division>
			<name>La Habana</name>
			<code>02</code>
			<fips>11</fips>
			<region></region>
		</division>
		<division>
			<name>Ciudad de La Habana</name>
			<code>03</code>
			<fips>02</fips>
			<region></region>
		</division>
		<division>
			<name>Cienfuegos</name>
			<code>06</code>
			<fips>08</fips>
			<region></region>
		</division>
		<division>
			<name>Camag&#xfc;ey</name>
			<code>09</code>
			<fips>05</fips>
			<region></region>
		</division>
		<division>
			<name>Las Tunas</name>
			<code>10</code>
			<fips>13</fips>
			<region></region>
		</division>
		<division>
			<name>Guant&#xe1;namo</name>
			<code>14</code>
			<fips>10</fips>
			<region></region>
		</division>
		<division>
			<name>Granma</name>
			<code>12</code>
			<fips>09</fips>
			<region></region>
		</division>
		<division>
			<name>Matanzas</name>
			<code>04</code>
			<fips>03</fips>
			<region></region>
		</division>
		<division>
			<name>Villa Clara</name>
			<code>05</code>
			<fips>16</fips>
			<region></region>
		</division>
		<division>
			<name>Sancti Sp&#xed;ritus</name>
			<code>07</code>
			<fips>14</fips>
			<region></region>
		</division>
		<num>192</num>
	</territory>
	<territory>
		<name>Cyprus</name>
		<alpha2>CY</alpha2>
		<alpha3>CYP</alpha3>
		<division>
			<name>Pafos</name>
			<code>05</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Larnaka</name>
			<code>03</code>
			<fips>03</fips>
			<region></region>
		</division>
		<division>
			<name>Lefkosia</name>
			<code>01</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Keryneia</name>
			<code>06</code>
			<fips>02</fips>
			<region></region>
		</division>
		<division>
			<name>Ammochostos</name>
			<code>04</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Lemesos</name>
			<code>02</code>
			<fips>05</fips>
			<region></region>
		</division>
		<num>196</num>
	</territory>
	<territory>
		<name>Czech Republic</name>
		<alpha2>CZ</alpha2>
		<alpha3>CZE</alpha3>
		<division>
			<name>Moravskoslezsk&#xfd; kraj</name>
			<code>MO</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Karlovarsk&#xfd; kraj</name>
			<code>KA</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Jihomoravsk&#xfd; kraj </name>
			<code>JM</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Zl&#xed;nsk&#xfd; kraj</name>
			<code>ZL</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Praha, hlavn&#xed; mesto</name>
			<code>PR</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Jihocesk&#xfd; kraj</name>
			<code>JC</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Olomouck&#xfd; kraj</name>
			<code>OL</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Kr&#xe1;lov&#xe9;hradeck&#xfd; kraj</name>
			<code>KR</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Plzensk&#xfd; kraj</name>
			<code>PL</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Vysocina</name>
			<code>VY</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Stredocesk&#xfd; kraj</name>
			<code>ST</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Libereck&#xfd; kraj</name>
			<code>LI</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>&#xda;steck&#xfd; kraj </name>
			<code>US</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Pardubick&#xfd; kraj</name>
			<code>PA</code>
			<fips></fips>
			<region></region>
		</division>
		<num>203</num>
	</territory>
	<territory>
		<name>C&#xf4;te d'Ivoire</name>
		<alpha2>CI</alpha2>
		<alpha3>CIV</alpha3>
		<division>
			<name>Savanes</name>
			<code>03</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Dengu&#xe9;l&#xe9;</name>
			<code>10</code>
			<fips>33</fips>
			<region></region>
		</division>
		<division>
			<name>Sud-Como&#xe9;</name>
			<code>13</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Vall&#xe9;e du Bandama</name>
			<code>04</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Lagunes</name>
			<code>01</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Bas-Sassandra</name>
			<code>09</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Agn&#xe9;bi )</name>
			<code>16</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Zanzan</name>
			<code>08</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Sud-Bandama</name>
			<code>15</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Marahou&#xe9;</name>
			<code>12</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Dix-Huit Montagnes</name>
			<code>06</code>
			<fips>47</fips>
			<region></region>
		</division>
		<division>
			<name>Moyen-Como&#xe9;</name>
			<code>05</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Nzi-Como&#xe9;</name>
			<code>11</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Haut-Sassandra</name>
			<code>02</code>
			<fips>54</fips>
			<region></region>
		</division>
		<division>
			<name>Lacs</name>
			<code>07</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Worodougou</name>
			<code>14</code>
			<fips></fips>
			<region></region>
		</division>
		<num>384</num>
	</territory>
	<territory>
		<name>Denmark</name>
		<alpha2>DK</alpha2>
		<alpha3>DNK</alpha3>
		<division>
			<name>Viborg</name>
			<code>076</code>
			<fips>15</fips>
			<region></region>
		</division>
		<division>
			<name>Ribe</name>
			<code>055</code>
			<fips>08</fips>
			<region></region>
		</division>
		<division>
			<name>Frederiksborg</name>
			<code>020</code>
			<fips>03</fips>
			<region></region>
		</division>
		<division>
			<name>Frederiksberg</name>
			<code>147</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>&#xc5;rhus</name>
			<code>070</code>
			<fips>01</fips>
			<region></region>
		</division>
		<division>
			<name>Roskilde</name>
			<code>025</code>
			<fips>10</fips>
			<region></region>
		</division>
		<division>
			<name>Vestsj&#xe6;lland</name>
			<code>030</code>
			<fips>14</fips>
			<region></region>
		</division>
		<division>
			<name>S&#xf8;nderjylland</name>
			<code>050</code>
			<fips>11</fips>
			<region></region>
		</division>
		<division>
			<name>K&#xf8;benhavn</name>
			<code>015</code>
			<fips>06</fips>
			<region></region>
		</division>
		<division>
			<name>Storstr&#xf8;m</name>
			<code>035</code>
			<fips>12</fips>
			<region></region>
		</division>
		<division>
			<name>Bornholm</name>
			<code>040</code>
			<fips>02</fips>
			<region></region>
		</division>
		<division>
			<name>Nordjylland</name>
			<code>080</code>
			<fips>07</fips>
			<region></region>
		</division>
		<division>
			<name>Vejle</name>
			<code>060</code>
			<fips>13</fips>
			<region></region>
		</division>
		<division>
			<name>Ringk&#xf8;bing</name>
			<code>065</code>
			<fips>09</fips>
			<region></region>
		</division>
		<division>
			<name>Fyn</name>
			<code>042</code>
			<fips></fips>
			<region></region>
		</division>
		<num>208</num>
	</territory>
	<territory>
		<name>Diego Garcia</name>
		<alpha2>DG</alpha2>
		<alpha3></alpha3>
		<num></num>
		<note>reserved on request of ITU</note>
		<ext>itu</ext>
	</territory>
	<territory>
		<name>Djibouti</name>
		<alpha2>DJ</alpha2>
		<alpha3>DJI</alpha3>
		<division>
			<name>Obock</name>
			<code>OB</code>
			<fips>04</fips>
			<region></region>
		</division>
		<division>
			<name>Ali Sabieh</name>
			<code>AS</code>
			<fips>01</fips>
			<region></region>
		</division>
		<division>
			<name>Djibouti</name>
			<code>DJ</code>
			<fips>03</fips>
			<region></region>
		</division>
		<division>
			<name>Tadjoura</name>
			<code>TA</code>
			<fips>05</fips>
			<region></region>
		</division>
		<division>
			<name>Dikhil</name>
			<code>DI</code>
			<fips>02</fips>
			<region></region>
		</division>
		<num>262</num>
	</territory>
	<territory>
		<name>Dominica</name>
		<alpha2>DM</alpha2>
		<alpha3>DMA</alpha3>
		<num>212</num>
	</territory>
	<territory>
		<name>Dominican Republic</name>
		<alpha2>DO</alpha2>
		<alpha3>DOM</alpha3>
		<division>
			<name>Dajab&#xf3;n</name>
			<code>05</code>
			<fips>04</fips>
			<region></region>
		</division>
		<division>
			<name>Hato Mayor</name>
			<code>30</code>
			<fips>29</fips>
			<region></region>
		</division>
		<division>
			<name>Santiago Rodr&#xed;guez</name>
			<code>26</code>
			<fips>26</fips>
			<region></region>
		</division>
		<division>
			<name>San Crist&#xf3;bal</name>
			<code>21</code>
			<fips>33</fips>
			<region></region>
		</division>
		<division>
			<name>Barahona</name>
			<code>04</code>
			<fips>03</fips>
			<region></region>
		</division>
		<division>
			<name>Duarte</name>
			<code>06</code>
			<fips>06</fips>
			<region></region>
		</division>
		<division>
			<name>Valverde</name>
			<code>27</code>
			<fips>27</fips>
			<region></region>
		</division>
		<division>
			<name>San Pedro de Macor&#xed;s</name>
			<code>23</code>
			<fips>24</fips>
			<region></region>
		</division>
		<division>
			<name>S&#xe1;nchez Ram&#xed;rez</name>
			<code>24</code>
			<fips>21</fips>
			<region></region>
		</division>
		<division>
			<name>San Juan</name>
			<code>22</code>
			<fips>23</fips>
			<region></region>
		</division>
		<division>
			<name>Pedernales</name>
			<code>16</code>
			<fips>16</fips>
			<region></region>
		</division>
		<division>
			<name>Monte Plata</name>
			<code>29</code>
			<fips>32</fips>
			<region></region>
		</division>
		<division>
			<name>La Romana</name>
			<code>12</code>
			<fips>12</fips>
			<region></region>
		</division>
		<division>
			<name>Santiago</name>
			<code>25</code>
			<fips>25</fips>
			<region></region>
		</division>
		<division>
			<name>Saman&#xe1;</name>
			<code>20</code>
			<fips>20</fips>
			<region></region>
		</division>
		<division>
			<name>Monte Cristi</name>
			<code>15</code>
			<fips>15</fips>
			<region></region>
		</division>
		<division>
			<name>Distrito Nacional </name>
			<code>01</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Salcedo</name>
			<code>19</code>
			<fips>19</fips>
			<region></region>
		</division>
		<division>
			<name>El Seybo [El Seibo]</name>
			<code>08</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Mar&#xed;a Trinidad S&#xe1;nchez</name>
			<code>14</code>
			<fips>14</fips>
			<region></region>
		</division>
		<division>
			<name>Bahoruco</name>
			<code>03</code>
			<fips>02</fips>
			<region></region>
		</division>
		<division>
			<name>Puerto Plata</name>
			<code>18</code>
			<fips>18</fips>
			<region></region>
		</division>
		<division>
			<name>Independencia</name>
			<code>10</code>
			<fips>09</fips>
			<region></region>
		</division>
		<division>
			<name>La Estrelleta [El&#xed;as Pi&#xf1;a]</name>
			<code>07</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>La Altagracia</name>
			<code>11</code>
			<fips>10</fips>
			<region></region>
		</division>
		<division>
			<name>Espaillat</name>
			<code>09</code>
			<fips>08</fips>
			<region></region>
		</division>
		<division>
			<name>La Vega</name>
			<code>13</code>
			<fips>30</fips>
			<region></region>
		</division>
		<division>
			<name>Peravia</name>
			<code>17</code>
			<fips>17</fips>
			<region></region>
		</division>
		<division>
			<name>Azua</name>
			<code>02</code>
			<fips>01</fips>
			<region></region>
		</division>
		<division>
			<name>Monse&#xf1;or Nouel</name>
			<code>28</code>
			<fips>31</fips>
			<region></region>
		</division>
		<num>214</num>
	</territory>
	<territory>
		<name>Ecuador</name>
		<alpha2>EC</alpha2>
		<alpha3>ECU</alpha3>
		<division>
			<name>Orellana</name>
			<code>D</code>
			<fips></fips>
			<region></region>
		</division>
		<num>218</num>
	</territory>
	<territory>
		<name>Egypt</name>
		<alpha2>EG</alpha2>
		<alpha3>EGY</alpha3>
		<division>
			<name>Al Buhayrah</name>
			<code>BH</code>
			<fips>03</fips>
			<region></region>
		</division>
		<division>
			<name>Kafr ash Shaykh</name>
			<code>KFS</code>
			<fips>21</fips>
			<region></region>
		</division>
		<division>
			<name>Matruh</name>
			<code>MT</code>
			<fips>22</fips>
			<region></region>
		</division>
		<division>
			<name>Dumyat</name>
			<code>DT</code>
			<fips>20</fips>
			<region></region>
		</division>
		<division>
			<name>Al Qahirah</name>
			<code>C</code>
			<fips>11</fips>
			<region></region>
		</division>
		<division>
			<name>Shamal Sina'</name>
			<code>SIN</code>
			<fips>27</fips>
			<region></region>
		</division>
		<division>
			<name>Al Gharbiyah</name>
			<code>GH</code>
			<fips>05</fips>
			<region></region>
		</division>
		<division>
			<name>Suhaj</name>
			<code>SHG</code>
			<fips>24</fips>
			<region></region>
		</division>
		<division>
			<name>Al Minya</name>
			<code>MN</code>
			<fips>10</fips>
			<region></region>
		</division>
		<division>
			<name>Bur Sahid</name>
			<code>PTS</code>
			<fips>19</fips>
			<region></region>
		</division>
		<division>
			<name>Al Wadi al Jadid</name>
			<code>WAD</code>
			<fips>13</fips>
			<region></region>
		</division>
		<division>
			<name>Bani Suwayf</name>
			<code>BNS</code>
			<fips>18</fips>
			<region></region>
		</division>
		<division>
			<name>As Suways</name>
			<code>SUZ</code>
			<fips>15</fips>
			<region></region>
		</division>
		<division>
			<name>Qina</name>
			<code>KN</code>
			<fips>23</fips>
			<region></region>
		</division>
		<division>
			<name>Al Jizah</name>
			<code>GZ</code>
			<fips>08</fips>
			<region></region>
		</division>
		<division>
			<name>Janub Sina'</name>
			<code>JS</code>
			<fips>26</fips>
			<region></region>
		</division>
		<division>
			<name>Al Ismahiliyah</name>
			<code>IS</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Ad Daqahliyah</name>
			<code>DK</code>
			<fips>01</fips>
			<region></region>
		</division>
		<division>
			<name>Ash Sharqiyah</name>
			<code>SHR</code>
			<fips>14</fips>
			<region></region>
		</division>
		<division>
			<name>Al Bahr al Ahmar</name>
			<code>BA</code>
			<fips>02</fips>
			<region></region>
		</division>
		<division>
			<name>Al Minufiyah</name>
			<code>MNF</code>
			<fips>09</fips>
			<region></region>
		</division>
		<division>
			<name>Aswan</name>
			<code>ASN</code>
			<fips>16</fips>
			<region></region>
		</division>
		<division>
			<name>Al Iskandariyah</name>
			<code>ALX</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Al Qalyubiyah</name>
			<code>KB</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Al Fayyum</name>
			<code>FYM</code>
			<fips>04</fips>
			<region></region>
		</division>
		<division>
			<name>Asyut</name>
			<code>AST</code>
			<fips>17</fips>
			<region></region>
		</division>
		<num>818</num>
	</territory>
	<territory>
		<name>El Salvador</name>
		<alpha2>SV</alpha2>
		<alpha3>SLV</alpha3>
		<division>
			<name>La Libertad</name>
			<code>LI</code>
			<fips>05</fips>
			<region></region>
		</division>
		<division>
			<name>San Vicente</name>
			<code>SV</code>
			<fips>12</fips>
			<region></region>
		</division>
		<division>
			<name>Usulut&#xe1;n</name>
			<code>US</code>
			<fips>14</fips>
			<region></region>
		</division>
		<division>
			<name>Santa Ana</name>
			<code>SA</code>
			<fips>11</fips>
			<region></region>
		</division>
		<division>
			<name>San Salvador</name>
			<code>SS</code>
			<fips>10</fips>
			<region></region>
		</division>
		<division>
			<name>San Miguel</name>
			<code>SM</code>
			<fips>09</fips>
			<region></region>
		</division>
		<division>
			<name>Sonsonate</name>
			<code>SO</code>
			<fips>13</fips>
			<region></region>
		</division>
		<division>
			<name>La Paz</name>
			<code>PA</code>
			<fips>06</fips>
			<region></region>
		</division>
		<division>
			<name>Caba&#xf1;as</name>
			<code>CA</code>
			<fips>02</fips>
			<region></region>
		</division>
		<division>
			<name>Chalatenango</name>
			<code>CH</code>
			<fips>03</fips>
			<region></region>
		</division>
		<division>
			<name>Moraz&#xe1;n</name>
			<code>MO</code>
			<fips>08</fips>
			<region></region>
		</division>
		<division>
			<name>Cuscatl&#xe1;n</name>
			<code>CU</code>
			<fips>04</fips>
			<region></region>
		</division>
		<division>
			<name>Ahuachap&#xe1;n</name>
			<code>AH</code>
			<fips>01</fips>
			<region></region>
		</division>
		<division>
			<name>La Uni&#xf3;n</name>
			<code>UN</code>
			<fips>07</fips>
			<region></region>
		</division>
		<num>222</num>
	</territory>
	<territory>
		<name>Equatorial Guinea</name>
		<alpha2>GQ</alpha2>
		<alpha3>GNQ</alpha3>
		<division>
			<name>Centro Sur</name>
			<code>CS</code>
			<fips></fips>
			<region>C</region>
		</division>
		<division>
			<name>Bioko Norte</name>
			<code>BN</code>
			<fips></fips>
			<region>I</region>
		</division>
		<division>
			<name>Litoral</name>
			<code>LI</code>
			<fips></fips>
			<region>C</region>
		</division>
		<division>
			<name>Bioko Sur</name>
			<code>BS</code>
			<fips></fips>
			<region>I</region>
		</division>
		<division>
			<name>Regi&#xf3;n Continental</name>
			<code>C</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Annob&#xf3;n</name>
			<code>AN</code>
			<fips></fips>
			<region>I</region>
		</division>
		<division>
			<name>Wele-Nz&#xe1;s</name>
			<code>WN</code>
			<fips></fips>
			<region>C</region>
		</division>
		<division>
			<name>Kie-Ntem</name>
			<code>KN</code>
			<fips></fips>
			<region>C</region>
		</division>
		<division>
			<name>Regi&#xf3;n Insular</name>
			<code>I</code>
			<fips></fips>
			<region></region>
		</division>
		<num>226</num>
	</territory>
	<territory>
		<name>Eritrea</name>
		<alpha2>ER</alpha2>
		<alpha3>ERI</alpha3>
		<division>
			<name>Maakel [Maekel]</name>
			<code>MA</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Anseba</name>
			<code>AN</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Debubawi Keyih Bahri</name>
			<code>DK</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Debub</name>
			<code>DU</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Semenawi Keyih Bahri</name>
			<code>SK</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Gash-Barka</name>
			<code>GB</code>
			<fips></fips>
			<region></region>
		</division>
		<num>232</num>
	</territory>
	<territory>
		<name>Estonia</name>
		<alpha2>EE</alpha2>
		<alpha3>EST</alpha3>
		<division>
			<name>V&#xf5;rumaa</name>
			<code>86</code>
			<fips>21</fips>
			<region></region>
		</division>
		<division>
			<name>L&#xe4;&#xe4;nemaa</name>
			<code>57</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Viljandimaa</name>
			<code>84</code>
			<fips>20</fips>
			<region></region>
		</division>
		<division>
			<name>Valgamaa</name>
			<code>82</code>
			<fips>19</fips>
			<region></region>
		</division>
		<division>
			<name>P&#xe4;rnumaa</name>
			<code>67</code>
			<fips>11</fips>
			<region></region>
		</division>
		<division>
			<name>P&#xf5;lvamaa</name>
			<code>65</code>
			<fips>12</fips>
			<region></region>
		</division>
		<division>
			<name>J&#xf5;gevamaa</name>
			<code>49</code>
			<fips>05</fips>
			<region></region>
		</division>
		<division>
			<name>Harjumaa</name>
			<code>37</code>
			<fips>01</fips>
			<region></region>
		</division>
		<division>
			<name>Saaremaa</name>
			<code>74</code>
			<fips>14</fips>
			<region></region>
		</division>
		<division>
			<name>L&#xe4;&#xe4;ne-Virumaa</name>
			<code>59</code>
			<fips>08</fips>
			<region></region>
		</division>
		<division>
			<name>Tartumaa</name>
			<code>78</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Ida-Virumaa</name>
			<code>44</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>J&#xe4;rvamaa</name>
			<code>51</code>
			<fips>04</fips>
			<region></region>
		</division>
		<division>
			<name>Hiiumaa</name>
			<code>39</code>
			<fips>02</fips>
			<region></region>
		</division>
		<division>
			<name>Raplamaa</name>
			<code>70</code>
			<fips>13</fips>
			<region></region>
		</division>
		<num>233</num>
	</territory>
	<territory>
		<name>Ethiopia</name>
		<alpha2>ET</alpha2>
		<alpha3>ETH</alpha3>
		<division>
			<name>Dire Dawa</name>
			<code>DD</code>
			<fips></fips>
			<region></region>
		</division>
		<num>231</num>
	</territory>
	<territory>
		<name>European Union</name>
		<alpha2>EU</alpha2>
		<alpha3></alpha3>
		<num></num>
		<note>reserved on request of ISO-4217/MA for the euro</note>
		<ext>eu</ext>
	</territory>
	<territory>
		<name>Falkland Islands</name>
		<alpha2>FK</alpha2>
		<alpha3>FLK</alpha3>
		<num>238</num>
	</territory>
	<territory>
		<name>Faroe Islands</name>
		<alpha2>FO</alpha2>
		<alpha3>FRO</alpha3>
		<num>234</num>
	</territory>
	<territory>
		<name>Fiji</name>
		<alpha2>FJ</alpha2>
		<alpha3>FJI</alpha3>
		<division>
			<name>Central</name>
			<code>C</code>
			<fips>01</fips>
			<region></region>
		</division>
		<division>
			<name>Northern</name>
			<code>N</code>
			<fips>03</fips>
			<region></region>
		</division>
		<division>
			<name>Western</name>
			<code>W</code>
			<fips>05</fips>
			<region></region>
		</division>
		<division>
			<name>Rotuma</name>
			<code>R</code>
			<fips>04</fips>
			<region></region>
		</division>
		<division>
			<name>Eastern</name>
			<code>E</code>
			<fips>02</fips>
			<region></region>
		</division>
		<num>242</num>
	</territory>
	<territory>
		<name>Finland</name>
		<alpha2>FI</alpha2>
		<alpha3>FIN</alpha3>
		<division>
			<name>It&#xe4;-Suomen l&#xe4;&#xe4;ni</name>
			<code>IS</code>
			<fips>14</fips>
			<region></region>
		</division>
		<division>
			<name>Oulun l&#xe4;&#xe4;ni</name>
			<code>OL</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Lapin l&#xe4;&#xe4;ni</name>
			<code>LL</code>
			<fips>08</fips>
			<region></region>
		</division>
		<division>
			<name>L&#xe4;nsi-Suomen l&#xe4;&#xe4;ni</name>
			<code>LS</code>
			<fips>15</fips>
			<region></region>
		</division>
		<division>
			<name>Ahvenanmaan l&#xe4;&#xe4;ni</name>
			<code>AL</code>
			<fips>01</fips>
			<region></region>
		</division>
		<division>
			<name>Etel&#xe4;-Suomen l&#xe4;&#xe4;ni</name>
			<code>ES</code>
			<fips>13</fips>
			<region></region>
		</division>
		<num>246</num>
	</territory>
	<territory>
		<name>France, Metropolitan</name>
		<alpha2>FX</alpha2>
		<alpha3></alpha3>
		<num></num>
		<note>reserved on request of France</note>
		<ext>fx</ext>
		<division>
			<name>C&#xf4;te-d'Or</name>
			<code>21</code>
			<fips></fips>
			<region>D</region>
		</division>
		<division>
			<name>Gard</name>
			<code>30</code>
			<fips></fips>
			<region>K</region>
		</division>
		<division>
			<name>Is&#xe8;re</name>
			<code>38</code>
			<fips></fips>
			<region>V</region>
		</division>
		<division>
			<name>Calvados</name>
			<code>14</code>
			<fips></fips>
			<region>P</region>
		</division>
		<division>
			<name>Territoire de Belfort</name>
			<code>90</code>
			<fips></fips>
			<region>I</region>
		</division>
		<division>
			<name>Charente</name>
			<code>16</code>
			<fips></fips>
			<region>T</region>
		</division>
		<division>
			<name>Loir-et-Cher</name>
			<code>41</code>
			<fips></fips>
			<region>F</region>
		</division>
		<division>
			<name>Haute-Corse</name>
			<code>2B</code>
			<fips></fips>
			<region>H</region>
		</division>
		<division>
			<name>Dordogne</name>
			<code>24</code>
			<fips></fips>
			<region>B</region>
		</division>
		<division>
			<name>Wallis et Futuna</name>
			<note>this administrative division is also listed in ISO 3166-1 as WF</note>
			<code>WF</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Var</name>
			<code>83</code>
			<fips></fips>
			<region>U</region>
		</division>
		<division>
			<name>Nouvelle-Cal&#xe9;donie</name>
			<note>this administrative division is also listed in ISO 3166-1 as NC</note>
			<code>NC</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Seine-Saint-Denis</name>
			<code>93</code>
			<fips></fips>
			<region>J</region>
		</division>
		<division>
			<name>Sarthe</name>
			<code>72</code>
			<fips></fips>
			<region>R</region>
		</division>
		<division>
			<name>Creuse</name>
			<code>23</code>
			<fips></fips>
			<region>L</region>
		</division>
		<division>
			<name>Haute-Savoie</name>
			<code>74</code>
			<fips></fips>
			<region>V</region>
		</division>
		<division>
			<name>Indre-et-Loire</name>
			<code>37</code>
			<fips></fips>
			<region>F</region>
		</division>
		<division>
			<name>Haute-Garonne</name>
			<code>31</code>
			<fips></fips>
			<region>N</region>
		</division>
		<division>
			<name>Orne</name>
			<code>61</code>
			<fips></fips>
			<region>P</region>
		</division>
		<division>
			<name>Dr&#xf4;me</name>
			<code>26</code>
			<fips></fips>
			<region>V</region>
		</division>
		<division>
			<name>Aude</name>
			<code>11</code>
			<fips></fips>
			<region>K</region>
		</division>
		<division>
			<name>H&#xe9;rault</name>
			<code>34</code>
			<fips></fips>
			<region>K</region>
		</division>
		<division>
			<name>Doubs</name>
			<code>25</code>
			<fips></fips>
			<region>I</region>
		</division>
		<division>
			<name>Aveyron</name>
			<code>12</code>
			<fips>98</fips>
			<region>N</region>
		</division>
		<division>
			<name>Loire</name>
			<code>42</code>
			<fips></fips>
			<region>V</region>
		</division>
		<division>
			<name>Deux-S&#xe8;vres</name>
			<code>79</code>
			<fips></fips>
			<region>T</region>
		</division>
		<division>
			<name>Oise</name>
			<code>60</code>
			<fips></fips>
			<region>S</region>
		</division>
		<division>
			<name>Jura</name>
			<code>39</code>
			<fips></fips>
			<region>I</region>
		</division>
		<division>
			<name>Moselle</name>
			<code>57</code>
			<fips></fips>
			<region>M</region>
		</division>
		<division>
			<name>Manche</name>
			<code>50</code>
			<fips></fips>
			<region>P</region>
		</division>
		<division>
			<name>Terres Australes Fran&#xe7;aises</name>
			<note>this administrative division is also listed in ISO 3166-1 as TF</note>
			<code>TF</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Finist&#xe8;re</name>
			<code>29</code>
			<fips></fips>
			<region>E</region>
		</division>
		<division>
			<name>Pyr&#xe9;n&#xe9;es-Atlantiques</name>
			<code>64</code>
			<fips></fips>
			<region>B</region>
		</division>
		<division>
			<name>Ari&#xe8;ge</name>
			<code>09</code>
			<fips></fips>
			<region>N</region>
		</division>
		<division>
			<name>Rh&#xf4;ne</name>
			<code>69</code>
			<fips></fips>
			<region>V</region>
		</division>
		<division>
			<name>Eure</name>
			<code>27</code>
			<fips></fips>
			<region>Q</region>
		</division>
		<division>
			<name>Lot-et-Garonne</name>
			<code>47</code>
			<fips></fips>
			<region>B</region>
		</division>
		<division>
			<name>Haut-Rhin</name>
			<code>68</code>
			<fips></fips>
			<region>A</region>
		</division>
		<division>
			<name>Saint-Pierre-et-Miquelon</name>
			<note>this administrative division is also listed in ISO 3166-1 as PM</note>
			<code>PM</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Charente-Maritime</name>
			<code>17</code>
			<fips></fips>
			<region>T</region>
		</division>
		<division>
			<name>Tarn-et-Garonne</name>
			<code>82</code>
			<fips></fips>
			<region>N</region>
		</division>
		<division>
			<name>Ain</name>
			<code>01</code>
			<fips></fips>
			<region>V</region>
		</division>
		<division>
			<name>Gers</name>
			<code>32</code>
			<fips></fips>
			<region>N</region>
		</division>
		<division>
			<name>Pas-de-Calais</name>
			<code>62</code>
			<fips></fips>
			<region>O</region>
		</division>
		<division>
			<name>Hauts-de-Seine</name>
			<code>92</code>
			<fips></fips>
			<region>J</region>
		</division>
		<division>
			<name>Aube</name>
			<code>10</code>
			<fips></fips>
			<region>G</region>
		</division>
		<division>
			<name>Meurthe-et-Moselle</name>
			<code>54</code>
			<fips></fips>
			<region>M</region>
		</division>
		<division>
			<name>Cher</name>
			<code>18</code>
			<fips></fips>
			<region>F</region>
		</division>
		<division>
			<name>Marne</name>
			<code>51</code>
			<fips></fips>
			<region>G</region>
		</division>
		<division>
			<name>Loz&#xe8;re</name>
			<code>48</code>
			<fips></fips>
			<region>K</region>
		</division>
		<division>
			<name>Ardennes</name>
			<code>08</code>
			<fips></fips>
			<region>G</region>
		</division>
		<division>
			<name>Ni&#xe8;vre</name>
			<code>58</code>
			<fips></fips>
			<region>D</region>
		</division>
		<division>
			<name>Haute-Vienne</name>
			<code>87</code>
			<fips></fips>
			<region>L</region>
		</division>
		<division>
			<name>Haute-Sa&#xf4;ne</name>
			<code>70</code>
			<fips></fips>
			<region>I</region>
		</division>
		<division>
			<name>Cantal</name>
			<code>15</code>
			<fips></fips>
			<region>C</region>
		</division>
		<division>
			<name>Loire-Atlantique</name>
			<code>44</code>
			<fips></fips>
			<region>R</region>
		</division>
		<division>
			<name>Alpes-Maritimes</name>
			<code>06</code>
			<fips></fips>
			<region>U</region>
		</division>
		<division>
			<name>Mayenne</name>
			<code>53</code>
			<fips></fips>
			<region>R</region>
		</division>
		<division>
			<name>Seine-et-Marne</name>
			<code>77</code>
			<fips></fips>
			<region>J</region>
		</division>
		<division>
			<name>Lot</name>
			<code>46</code>
			<fips></fips>
			<region>N</region>
		</division>
		<division>
			<name>Meuse</name>
			<code>55</code>
			<fips></fips>
			<region>M</region>
		</division>
		<division>
			<name>Yvelines</name>
			<code>78</code>
			<fips></fips>
			<region>J</region>
		</division>
		<division>
			<name>Seine-Maritime</name>
			<code>76</code>
			<fips></fips>
			<region>Q</region>
		</division>
		<division>
			<name>Morbihan</name>
			<code>56</code>
			<fips></fips>
			<region>E</region>
		</division>
		<division>
			<name>Vaucluse</name>
			<code>84</code>
			<fips></fips>
			<region>U</region>
		</division>
		<division>
			<name>Yonne</name>
			<code>89</code>
			<fips></fips>
			<region>D</region>
		</division>
		<division>
			<name>Hautes-Pyr&#xe9;n&#xe9;es</name>
			<code>65</code>
			<fips></fips>
			<region>N</region>
		</division>
		<division>
			<name>Corr&#xe8;ze</name>
			<code>19</code>
			<fips></fips>
			<region>L</region>
		</division>
		<division>
			<name>Eure-et-Loir</name>
			<code>28</code>
			<fips></fips>
			<region>F</region>
		</division>
		<division>
			<name>Paris</name>
			<code>75</code>
			<fips></fips>
			<region>J</region>
		</division>
		<division>
			<name>Bouches-du-Rh&#xf4;ne</name>
			<code>13</code>
			<fips></fips>
			<region>U</region>
		</division>
		<division>
			<name>Vend&#xe9;e</name>
			<code>85</code>
			<fips></fips>
			<region>R</region>
		</division>
		<division>
			<name>Somme</name>
			<code>80</code>
			<fips></fips>
			<region>S</region>
		</division>
		<division>
			<name>C&#xf4;tes-d'Armor</name>
			<code>22</code>
			<fips></fips>
			<region>E</region>
		</division>
		<division>
			<name>Mayotte</name>
			<note>this administrative division is also listed in ISO 3166-1 as YT</note>
			<code>YT</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Sa&#xf4;ne-et-Loire</name>
			<code>71</code>
			<fips></fips>
			<region>D</region>
		</division>
		<division>
			<name>Ard&#xe8;che</name>
			<code>07</code>
			<fips></fips>
			<region>V</region>
		</division>
		<division>
			<name>Polyn&#xe9;sie fran&#xe7;aise</name>
			<note>this administrative division is also listed in ISO 3166-1 as PF</note>
			<code>PF</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Haute-Loire</name>
			<code>43</code>
			<fips></fips>
			<region>C</region>
		</division>
		<division>
			<name>Essonne</name>
			<code>91</code>
			<fips></fips>
			<region>J</region>
		</division>
		<division>
			<name>Ille-et-Vilaine</name>
			<code>35</code>
			<fips></fips>
			<region>E</region>
		</division>
		<division>
			<name>Allier</name>
			<code>03</code>
			<fips></fips>
			<region>C</region>
		</division>
		<division>
			<name>Gironde</name>
			<code>33</code>
			<fips></fips>
			<region>B</region>
		</division>
		<division>
			<name>Pyr&#xe9;n&#xe9;es-Orientales</name>
			<code>66</code>
			<fips></fips>
			<region>K</region>
		</division>
		<division>
			<name>Tarn</name>
			<code>81</code>
			<fips></fips>
			<region>N</region>
		</division>
		<division>
			<name>Corse-du-Sud</name>
			<code>2A</code>
			<fips></fips>
			<region>H</region>
		</division>
		<division>
			<name>Alpes-de-Haute-Provence</name>
			<code>04</code>
			<fips></fips>
			<region>U</region>
		</division>
		<division>
			<name>Vienne</name>
			<code>86</code>
			<fips></fips>
			<region>T</region>
		</division>
		<division>
			<name>Maine-et-Loire</name>
			<code>49</code>
			<fips></fips>
			<region>R</region>
		</division>
		<division>
			<name>Loiret</name>
			<code>45</code>
			<fips>B2</fips>
			<region>F</region>
		</division>
		<division>
			<name>Val-de-Marne</name>
			<code>94</code>
			<fips></fips>
			<region>J</region>
		</division>
		<division>
			<name>Haute-Marne</name>
			<code>52</code>
			<fips></fips>
			<region>G</region>
		</division>
		<division>
			<name>Val-d'Oise</name>
			<code>95</code>
			<fips></fips>
			<region>J</region>
		</division>
		<division>
			<name>Nord</name>
			<code>59</code>
			<fips></fips>
			<region>O</region>
		</division>
		<division>
			<name>Savoie</name>
			<code>73</code>
			<fips></fips>
			<region>V</region>
		</division>
		<division>
			<name>Landes</name>
			<code>40</code>
			<fips></fips>
			<region>B</region>
		</division>
		<division>
			<name>Indre</name>
			<code>36</code>
			<fips></fips>
			<region>F</region>
		</division>
		<division>
			<name>Puy-de-D&#xf4;me</name>
			<code>63</code>
			<fips></fips>
			<region>C</region>
		</division>
		<division>
			<name>Hautes-Alpes</name>
			<code>05</code>
			<fips></fips>
			<region>U</region>
		</division>
		<division>
			<name>Bas-Rhin</name>
			<code>67</code>
			<fips></fips>
			<region>A</region>
		</division>
		<division>
			<name>Aisne</name>
			<code>02</code>
			<fips></fips>
			<region>S</region>
		</division>
		<division>
			<name>Vosges</name>
			<code>88</code>
			<fips></fips>
			<region>M</region>
		</division>
		<num>250</num>
	</territory>
	<territory>
		<name>France</name>
		<alpha2>FR</alpha2>
		<alpha3>FRA</alpha3>
		<division>
			<name>C&#xf4;te-d'Or</name>
			<code>21</code>
			<fips></fips>
			<region>D</region>
		</division>
		<division>
			<name>Gard</name>
			<code>30</code>
			<fips></fips>
			<region>K</region>
		</division>
		<division>
			<name>Is&#xe8;re</name>
			<code>38</code>
			<fips></fips>
			<region>V</region>
		</division>
		<division>
			<name>Calvados</name>
			<code>14</code>
			<fips></fips>
			<region>P</region>
		</division>
		<division>
			<name>Territoire de Belfort</name>
			<code>90</code>
			<fips></fips>
			<region>I</region>
		</division>
		<division>
			<name>Charente</name>
			<code>16</code>
			<fips></fips>
			<region>T</region>
		</division>
		<division>
			<name>Loir-et-Cher</name>
			<code>41</code>
			<fips></fips>
			<region>F</region>
		</division>
		<division>
			<name>Haute-Corse</name>
			<code>2B</code>
			<fips></fips>
			<region>H</region>
		</division>
		<division>
			<name>Dordogne</name>
			<code>24</code>
			<fips></fips>
			<region>B</region>
		</division>
		<division>
			<name>Wallis et Futuna</name>
			<note>this administrative division is also listed in ISO 3166-1 as WF</note>
			<code>WF</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Var</name>
			<code>83</code>
			<fips></fips>
			<region>U</region>
		</division>
		<division>
			<name>Nouvelle-Cal&#xe9;donie</name>
			<note>this administrative division is also listed in ISO 3166-1 as NC</note>
			<code>NC</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Seine-Saint-Denis</name>
			<code>93</code>
			<fips></fips>
			<region>J</region>
		</division>
		<division>
			<name>Sarthe</name>
			<code>72</code>
			<fips></fips>
			<region>R</region>
		</division>
		<division>
			<name>Creuse</name>
			<code>23</code>
			<fips></fips>
			<region>L</region>
		</division>
		<division>
			<name>Haute-Savoie</name>
			<code>74</code>
			<fips></fips>
			<region>V</region>
		</division>
		<division>
			<name>Indre-et-Loire</name>
			<code>37</code>
			<fips></fips>
			<region>F</region>
		</division>
		<division>
			<name>Haute-Garonne</name>
			<code>31</code>
			<fips></fips>
			<region>N</region>
		</division>
		<division>
			<name>Orne</name>
			<code>61</code>
			<fips></fips>
			<region>P</region>
		</division>
		<division>
			<name>Dr&#xf4;me</name>
			<code>26</code>
			<fips></fips>
			<region>V</region>
		</division>
		<division>
			<name>Aude</name>
			<code>11</code>
			<fips></fips>
			<region>K</region>
		</division>
		<division>
			<name>H&#xe9;rault</name>
			<code>34</code>
			<fips></fips>
			<region>K</region>
		</division>
		<division>
			<name>Doubs</name>
			<code>25</code>
			<fips></fips>
			<region>I</region>
		</division>
		<division>
			<name>Aveyron</name>
			<code>12</code>
			<fips>98</fips>
			<region>N</region>
		</division>
		<division>
			<name>Loire</name>
			<code>42</code>
			<fips></fips>
			<region>V</region>
		</division>
		<division>
			<name>Deux-S&#xe8;vres</name>
			<code>79</code>
			<fips></fips>
			<region>T</region>
		</division>
		<division>
			<name>Oise</name>
			<code>60</code>
			<fips></fips>
			<region>S</region>
		</division>
		<division>
			<name>Jura</name>
			<code>39</code>
			<fips></fips>
			<region>I</region>
		</division>
		<division>
			<name>Moselle</name>
			<code>57</code>
			<fips></fips>
			<region>M</region>
		</division>
		<division>
			<name>Manche</name>
			<code>50</code>
			<fips></fips>
			<region>P</region>
		</division>
		<division>
			<name>Terres Australes Fran&#xe7;aises</name>
			<note>this administrative division is also listed in ISO 3166-1 as TF</note>
			<code>TF</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Finist&#xe8;re</name>
			<code>29</code>
			<fips></fips>
			<region>E</region>
		</division>
		<division>
			<name>Pyr&#xe9;n&#xe9;es-Atlantiques</name>
			<code>64</code>
			<fips></fips>
			<region>B</region>
		</division>
		<division>
			<name>Ari&#xe8;ge</name>
			<code>09</code>
			<fips></fips>
			<region>N</region>
		</division>
		<division>
			<name>Rh&#xf4;ne</name>
			<code>69</code>
			<fips></fips>
			<region>V</region>
		</division>
		<division>
			<name>Eure</name>
			<code>27</code>
			<fips></fips>
			<region>Q</region>
		</division>
		<division>
			<name>Lot-et-Garonne</name>
			<code>47</code>
			<fips></fips>
			<region>B</region>
		</division>
		<division>
			<name>Haut-Rhin</name>
			<code>68</code>
			<fips></fips>
			<region>A</region>
		</division>
		<division>
			<name>Saint-Pierre-et-Miquelon</name>
			<note>this administrative division is also listed in ISO 3166-1 as PM</note>
			<code>PM</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Charente-Maritime</name>
			<code>17</code>
			<fips></fips>
			<region>T</region>
		</division>
		<division>
			<name>Tarn-et-Garonne</name>
			<code>82</code>
			<fips></fips>
			<region>N</region>
		</division>
		<division>
			<name>Ain</name>
			<code>01</code>
			<fips></fips>
			<region>V</region>
		</division>
		<division>
			<name>Gers</name>
			<code>32</code>
			<fips></fips>
			<region>N</region>
		</division>
		<division>
			<name>Pas-de-Calais</name>
			<code>62</code>
			<fips></fips>
			<region>O</region>
		</division>
		<division>
			<name>Hauts-de-Seine</name>
			<code>92</code>
			<fips></fips>
			<region>J</region>
		</division>
		<division>
			<name>Aube</name>
			<code>10</code>
			<fips></fips>
			<region>G</region>
		</division>
		<division>
			<name>Meurthe-et-Moselle</name>
			<code>54</code>
			<fips></fips>
			<region>M</region>
		</division>
		<division>
			<name>Cher</name>
			<code>18</code>
			<fips></fips>
			<region>F</region>
		</division>
		<division>
			<name>Marne</name>
			<code>51</code>
			<fips></fips>
			<region>G</region>
		</division>
		<division>
			<name>Loz&#xe8;re</name>
			<code>48</code>
			<fips></fips>
			<region>K</region>
		</division>
		<division>
			<name>Ardennes</name>
			<code>08</code>
			<fips></fips>
			<region>G</region>
		</division>
		<division>
			<name>Ni&#xe8;vre</name>
			<code>58</code>
			<fips></fips>
			<region>D</region>
		</division>
		<division>
			<name>Haute-Vienne</name>
			<code>87</code>
			<fips></fips>
			<region>L</region>
		</division>
		<division>
			<name>Haute-Sa&#xf4;ne</name>
			<code>70</code>
			<fips></fips>
			<region>I</region>
		</division>
		<division>
			<name>Cantal</name>
			<code>15</code>
			<fips></fips>
			<region>C</region>
		</division>
		<division>
			<name>Loire-Atlantique</name>
			<code>44</code>
			<fips></fips>
			<region>R</region>
		</division>
		<division>
			<name>Alpes-Maritimes</name>
			<code>06</code>
			<fips></fips>
			<region>U</region>
		</division>
		<division>
			<name>Mayenne</name>
			<code>53</code>
			<fips></fips>
			<region>R</region>
		</division>
		<division>
			<name>Seine-et-Marne</name>
			<code>77</code>
			<fips></fips>
			<region>J</region>
		</division>
		<division>
			<name>Lot</name>
			<code>46</code>
			<fips></fips>
			<region>N</region>
		</division>
		<division>
			<name>Meuse</name>
			<code>55</code>
			<fips></fips>
			<region>M</region>
		</division>
		<division>
			<name>Yvelines</name>
			<code>78</code>
			<fips></fips>
			<region>J</region>
		</division>
		<division>
			<name>Seine-Maritime</name>
			<code>76</code>
			<fips></fips>
			<region>Q</region>
		</division>
		<division>
			<name>Morbihan</name>
			<code>56</code>
			<fips></fips>
			<region>E</region>
		</division>
		<division>
			<name>Vaucluse</name>
			<code>84</code>
			<fips></fips>
			<region>U</region>
		</division>
		<division>
			<name>Yonne</name>
			<code>89</code>
			<fips></fips>
			<region>D</region>
		</division>
		<division>
			<name>Hautes-Pyr&#xe9;n&#xe9;es</name>
			<code>65</code>
			<fips></fips>
			<region>N</region>
		</division>
		<division>
			<name>Corr&#xe8;ze</name>
			<code>19</code>
			<fips></fips>
			<region>L</region>
		</division>
		<division>
			<name>Eure-et-Loir</name>
			<code>28</code>
			<fips></fips>
			<region>F</region>
		</division>
		<division>
			<name>Paris</name>
			<code>75</code>
			<fips></fips>
			<region>J</region>
		</division>
		<division>
			<name>Bouches-du-Rh&#xf4;ne</name>
			<code>13</code>
			<fips></fips>
			<region>U</region>
		</division>
		<division>
			<name>Vend&#xe9;e</name>
			<code>85</code>
			<fips></fips>
			<region>R</region>
		</division>
		<division>
			<name>Somme</name>
			<code>80</code>
			<fips></fips>
			<region>S</region>
		</division>
		<division>
			<name>C&#xf4;tes-d'Armor</name>
			<code>22</code>
			<fips></fips>
			<region>E</region>
		</division>
		<division>
			<name>Mayotte</name>
			<note>this administrative division is also listed in ISO 3166-1 as YT</note>
			<code>YT</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Sa&#xf4;ne-et-Loire</name>
			<code>71</code>
			<fips></fips>
			<region>D</region>
		</division>
		<division>
			<name>Ard&#xe8;che</name>
			<code>07</code>
			<fips></fips>
			<region>V</region>
		</division>
		<division>
			<name>Polyn&#xe9;sie fran&#xe7;aise</name>
			<note>this administrative division is also listed in ISO 3166-1 as PF</note>
			<code>PF</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Haute-Loire</name>
			<code>43</code>
			<fips></fips>
			<region>C</region>
		</division>
		<division>
			<name>Essonne</name>
			<code>91</code>
			<fips></fips>
			<region>J</region>
		</division>
		<division>
			<name>Ille-et-Vilaine</name>
			<code>35</code>
			<fips></fips>
			<region>E</region>
		</division>
		<division>
			<name>Allier</name>
			<code>03</code>
			<fips></fips>
			<region>C</region>
		</division>
		<division>
			<name>Gironde</name>
			<code>33</code>
			<fips></fips>
			<region>B</region>
		</division>
		<division>
			<name>Pyr&#xe9;n&#xe9;es-Orientales</name>
			<code>66</code>
			<fips></fips>
			<region>K</region>
		</division>
		<division>
			<name>Tarn</name>
			<code>81</code>
			<fips></fips>
			<region>N</region>
		</division>
		<division>
			<name>Corse-du-Sud</name>
			<code>2A</code>
			<fips></fips>
			<region>H</region>
		</division>
		<division>
			<name>Alpes-de-Haute-Provence</name>
			<code>04</code>
			<fips></fips>
			<region>U</region>
		</division>
		<division>
			<name>Vienne</name>
			<code>86</code>
			<fips></fips>
			<region>T</region>
		</division>
		<division>
			<name>Maine-et-Loire</name>
			<code>49</code>
			<fips></fips>
			<region>R</region>
		</division>
		<division>
			<name>Loiret</name>
			<code>45</code>
			<fips>B2</fips>
			<region>F</region>
		</division>
		<division>
			<name>Val-de-Marne</name>
			<code>94</code>
			<fips></fips>
			<region>J</region>
		</division>
		<division>
			<name>Haute-Marne</name>
			<code>52</code>
			<fips></fips>
			<region>G</region>
		</division>
		<division>
			<name>Val-d'Oise</name>
			<code>95</code>
			<fips></fips>
			<region>J</region>
		</division>
		<division>
			<name>Nord</name>
			<code>59</code>
			<fips></fips>
			<region>O</region>
		</division>
		<division>
			<name>Savoie</name>
			<code>73</code>
			<fips></fips>
			<region>V</region>
		</division>
		<division>
			<name>Landes</name>
			<code>40</code>
			<fips></fips>
			<region>B</region>
		</division>
		<division>
			<name>Indre</name>
			<code>36</code>
			<fips></fips>
			<region>F</region>
		</division>
		<division>
			<name>Puy-de-D&#xf4;me</name>
			<code>63</code>
			<fips></fips>
			<region>C</region>
		</division>
		<division>
			<name>Hautes-Alpes</name>
			<code>05</code>
			<fips></fips>
			<region>U</region>
		</division>
		<division>
			<name>Bas-Rhin</name>
			<code>67</code>
			<fips></fips>
			<region>A</region>
		</division>
		<division>
			<name>Aisne</name>
			<code>02</code>
			<fips></fips>
			<region>S</region>
		</division>
		<division>
			<name>Vosges</name>
			<code>88</code>
			<fips></fips>
			<region>M</region>
		</division>
		<num>250</num>
	</territory>
	<territory>
		<name>France, Metropolitan</name>
		<alpha2>FX</alpha2>
		<alpha3></alpha3>
		<num></num>
		<note>reserved on request of France</note>
		<ext>fx</ext>
	</territory>
	<territory>
		<name>French Guiana</name>
		<alpha2>GF</alpha2>
		<alpha3>GUF</alpha3>
		<num>254</num>
	</territory>
	<territory>
		<name>French Polynesia</name>
		<alpha2>PF</alpha2>
		<alpha3>PYF</alpha3>
		<num>258</num>
	</territory>
	<territory>
		<name>French Southern Lands</name>
		<alpha2>TF</alpha2>
		<alpha3>ATF</alpha3>
		<num>260</num>
	</territory>
	<territory>
		<name>Gabon</name>
		<alpha2>GA</alpha2>
		<alpha3>GAB</alpha3>
		<division>
			<name>Ogoou&#xe9;-Lolo</name>
			<code>7</code>
			<fips>07</fips>
			<region></region>
		</division>
		<division>
			<name>Haut-Ogoou&#xe9;</name>
			<code>2</code>
			<fips>02</fips>
			<region></region>
		</division>
		<division>
			<name>Ogoou&#xe9;-Maritime</name>
			<code>8</code>
			<fips>08</fips>
			<region></region>
		</division>
		<division>
			<name>Woleu-Ntem</name>
			<code>9</code>
			<fips>09</fips>
			<region></region>
		</division>
		<division>
			<name>Estuaire</name>
			<code>1</code>
			<fips>01</fips>
			<region></region>
		</division>
		<division>
			<name>Moyen-Ogoou&#xe9;</name>
			<code>3</code>
			<fips>03</fips>
			<region></region>
		</division>
		<division>
			<name>Ngouni&#xe9;</name>
			<code>4</code>
			<fips>04</fips>
			<region></region>
		</division>
		<division>
			<name>Nyanga</name>
			<code>5</code>
			<fips>05</fips>
			<region></region>
		</division>
		<division>
			<name>Ogoou&#xe9;-Ivindo</name>
			<code>6</code>
			<fips>06</fips>
			<region></region>
		</division>
		<num>266</num>
	</territory>
	<territory>
		<name>Gambia</name>
		<alpha2>GM</alpha2>
		<alpha3>GMB</alpha3>
		<division>
			<name>MacCarthy Island</name>
			<code>M</code>
			<fips>03</fips>
			<region></region>
		</division>
		<division>
			<name>Lower River</name>
			<code>L</code>
			<fips>02</fips>
			<region></region>
		</division>
		<division>
			<name>Banjul</name>
			<code>B</code>
			<fips>01</fips>
			<region></region>
		</division>
		<division>
			<name>North Bank</name>
			<code>N</code>
			<fips>07</fips>
			<region></region>
		</division>
		<division>
			<name>Western</name>
			<code>W</code>
			<fips>05</fips>
			<region></region>
		</division>
		<division>
			<name>Upper River</name>
			<code>U</code>
			<fips>04</fips>
			<region></region>
		</division>
		<num>270</num>
	</territory>
	<territory>
		<name>Georgia</name>
		<alpha2>GE</alpha2>
		<alpha3>GEO</alpha3>
		<division>
			<name>Tbilisi</name>
			<code>TB</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Abkhazia</name>
			<code>AB</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Racha-Lechkhumi [and] Kvemo Svaneti</name>
			<code>RL</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Kakheti</name>
			<code>KA</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Mtskheta-Mtianeti</name>
			<code>MM</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Shida Kartli</name>
			<code>SK</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Imereti</name>
			<code>IM</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Ajaria</name>
			<code>AJ</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Samtskhe-Javakheti</name>
			<code>SJ</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Samegrelo-Zemo Svaneti</name>
			<code>SZ</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Guria</name>
			<code>GU</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Kvemo Kartli</name>
			<code>KK</code>
			<fips></fips>
			<region></region>
		</division>
		<num>268</num>
	</territory>
	<territory>
		<name>Germany</name>
		<alpha2>DE</alpha2>
		<alpha3>DEU</alpha3>
		<division>
			<name>Mecklenburg-Vorpommern</name>
			<code>MV</code>
			<fips>12</fips>
			<region></region>
		</division>
		<division>
			<name>Hessen</name>
			<code>HE</code>
			<fips>05</fips>
			<region></region>
		</division>
		<division>
			<name>Schleswig-Holstein</name>
			<code>SH</code>
			<fips>10</fips>
			<region></region>
		</division>
		<division>
			<name>Berlin</name>
			<code>BE</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Baden-W&#xfc;rttemberg</name>
			<code>BW</code>
			<fips>01</fips>
			<region></region>
		</division>
		<division>
			<name>Saarland</name>
			<code>SL</code>
			<fips>09</fips>
			<region></region>
		</division>
		<division>
			<name>Bremen</name>
			<code>HB</code>
			<fips>03</fips>
			<region></region>
		</division>
		<division>
			<name>Bayern</name>
			<code>BY</code>
			<fips>02</fips>
			<region></region>
		</division>
		<division>
			<name>Niedersachsen</name>
			<code>NI</code>
			<fips>06</fips>
			<region></region>
		</division>
		<division>
			<name>Nordrhein-Westfalen</name>
			<code>NW</code>
			<fips>07</fips>
			<region></region>
		</division>
		<division>
			<name>Sachsen</name>
			<code>SN</code>
			<fips>13</fips>
			<region></region>
		</division>
		<division>
			<name>Th&#xfc;ringen</name>
			<code>TH</code>
			<fips>15</fips>
			<region></region>
		</division>
		<division>
			<name>Hamburg</name>
			<code>HH</code>
			<fips>04</fips>
			<region></region>
		</division>
		<division>
			<name>Sachsen-Anhalt</name>
			<code>ST</code>
			<fips>14</fips>
			<region></region>
		</division>
		<division>
			<name>Rheinland-Pfalz</name>
			<code>RP</code>
			<fips>08</fips>
			<region></region>
		</division>
		<division>
			<name>Brandenburg</name>
			<code>BB</code>
			<fips>11</fips>
			<region></region>
		</division>
		<num>276</num>
	</territory>
	<territory>
		<name>Ghana</name>
		<alpha2>GH</alpha2>
		<alpha3>GHA</alpha3>
		<division>
			<name>Central</name>
			<code>CP</code>
			<fips>04</fips>
			<region></region>
		</division>
		<division>
			<name>Ashanti</name>
			<code>AH</code>
			<fips>02</fips>
			<region></region>
		</division>
		<division>
			<name>Volta</name>
			<code>TV</code>
			<fips>08</fips>
			<region></region>
		</division>
		<division>
			<name>Eastern</name>
			<code>EP</code>
			<fips>05</fips>
			<region></region>
		</division>
		<division>
			<name>Upper West</name>
			<code>UW</code>
			<fips>11</fips>
			<region></region>
		</division>
		<division>
			<name>Greater Accra</name>
			<code>AA</code>
			<fips>01</fips>
			<region></region>
		</division>
		<division>
			<name>Brong-Ahafo</name>
			<code>BA</code>
			<fips>03</fips>
			<region></region>
		</division>
		<division>
			<name>Northern</name>
			<code>NP</code>
			<fips>06</fips>
			<region></region>
		</division>
		<division>
			<name>Western</name>
			<code>WP</code>
			<fips>09</fips>
			<region></region>
		</division>
		<division>
			<name>Upper East</name>
			<code>UE</code>
			<fips>10</fips>
			<region></region>
		</division>
		<num>288</num>
	</territory>
	<territory>
		<name>Gibraltar</name>
		<alpha2>GI</alpha2>
		<alpha3>GIB</alpha3>
		<num>292</num>
	</territory>
	<territory>
		<name>Greece</name>
		<alpha2>GR</alpha2>
		<alpha3>GRC</alpha3>
		<division>
			<name>Rethymnon</name>
			<code>93</code>
			<fips>44</fips>
			<region>XIII</region>
		</division>
		<division>
			<name>Pieria</name>
			<code>61</code>
			<fips></fips>
			<region>II</region>
		</division>
		<division>
			<name>Florina</name>
			<code>63</code>
			<fips>08</fips>
			<region>III</region>
		</division>
		<division>
			<name>Imathia</name>
			<code>53</code>
			<fips>12</fips>
			<region>II</region>
		</division>
		<division>
			<name>Arta</name>
			<code>31</code>
			<fips>20</fips>
			<region>IV</region>
		</division>
		<division>
			<name>Samos</name>
			<code>84</code>
			<fips>48</fips>
			<region>XI</region>
		</division>
		<division>
			<name>Lefkas</name>
			<code>24</code>
			<fips>26</fips>
			<region>VI</region>
		</division>
		<division>
			<name>Drama</name>
			<code>52</code>
			<fips>04</fips>
			<region>I</region>
		</division>
		<division>
			<name>Evvoia</name>
			<code>04</code>
			<fips>34</fips>
			<region>VIII</region>
		</division>
		<division>
			<name>Evrytania</name>
			<code>05</code>
			<fips>30</fips>
			<region>VIII</region>
		</division>
		<division>
			<name>Attiki</name>
			<code>A1</code>
			<fips>35</fips>
			<region>IX</region>
		</division>
		<division>
			<name>Ileia</name>
			<code>14</code>
			<fips></fips>
			<region>VII</region>
		</division>
		<division>
			<name>Kozani</name>
			<code>58</code>
			<fips>11</fips>
			<region>III</region>
		</division>
		<division>
			<name>Lakonia</name>
			<code>16</code>
			<fips>42</fips>
			<region>X</region>
		</division>
		<division>
			<name>Fokis</name>
			<code>07</code>
			<fips>32</fips>
			<region>VIII</region>
		</division>
		<division>
			<name>Rodopi</name>
			<code>73</code>
			<fips>02</fips>
			<region>I</region>
		</division>
		<division>
			<name>Magnisia</name>
			<code>43</code>
			<fips>24</fips>
			<region>V</region>
		</division>
		<division>
			<name>Arkadia</name>
			<code>12</code>
			<fips>41</fips>
			<region>X</region>
		</division>
		<division>
			<name>Xanthi</name>
			<code>72</code>
			<fips>03</fips>
			<region>I</region>
		</division>
		<division>
			<name>Agio Oros</name>
			<code>69</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Kavalla</name>
			<code>55</code>
			<fips>14</fips>
			<region>I</region>
		</division>
		<division>
			<name>Karditsa</name>
			<code>41</code>
			<fips></fips>
			<region>V</region>
		</division>
		<division>
			<name>Zakynthos</name>
			<code>21</code>
			<fips>28</fips>
			<region>VI</region>
		</division>
		<division>
			<name>Serrai</name>
			<code>62</code>
			<fips>05</fips>
			<region>II</region>
		</division>
		<division>
			<name>Messinia</name>
			<code>17</code>
			<fips>40</fips>
			<region>X</region>
		</division>
		<division>
			<name>Fthiotis</name>
			<code>06</code>
			<fips>29</fips>
			<region>VIII</region>
		</division>
		<division>
			<name>Argolis</name>
			<code>11</code>
			<fips>36</fips>
			<region>X</region>
		</division>
		<division>
			<name>Kilkis</name>
			<code>57</code>
			<fips>06</fips>
			<region>II</region>
		</division>
		<division>
			<name>Lasithion</name>
			<code>92</code>
			<fips>46</fips>
			<region>XIII</region>
		</division>
		<division>
			<name>Aitolia-Akarnania</name>
			<code>01</code>
			<fips>39</fips>
			<region>VII</region>
		</division>
		<division>
			<name>Kefallinia</name>
			<code>23</code>
			<fips>27</fips>
			<region>VI</region>
		</division>
		<division>
			<name>Kyklades</name>
			<code>82</code>
			<fips>49</fips>
			<region>XII</region>
		</division>
		<division>
			<name>Thessaloniki</name>
			<code>54</code>
			<fips>13</fips>
			<region>II</region>
		</division>
		<division>
			<name>Voiotia</name>
			<code>03</code>
			<fips>33</fips>
			<region>VIII</region>
		</division>
		<division>
			<name>Larisa</name>
			<code>42</code>
			<fips>21</fips>
			<region>V</region>
		</division>
		<division>
			<name>Ioannina</name>
			<code>33</code>
			<fips>17</fips>
			<region>IV</region>
		</division>
		<division>
			<name>Dodekanisos</name>
			<code>81</code>
			<fips></fips>
			<region>XII</region>
		</division>
		<division>
			<name>Pella</name>
			<code>59</code>
			<fips>07</fips>
			<region>II</region>
		</division>
		<division>
			<name>Chalkidiki</name>
			<code>64</code>
			<fips></fips>
			<region>II</region>
		</division>
		<division>
			<name>Evros</name>
			<code>71</code>
			<fips>01</fips>
			<region>I</region>
		</division>
		<division>
			<name>Irakleion</name>
			<code>91</code>
			<fips>45</fips>
			<region>XIII</region>
		</division>
		<division>
			<name>Chios</name>
			<code>85</code>
			<fips></fips>
			<region>XI</region>
		</division>
		<division>
			<name>Preveza</name>
			<code>34</code>
			<fips>19</fips>
			<region>IV</region>
		</division>
		<division>
			<name>Kerkyra</name>
			<code>22</code>
			<fips>25</fips>
			<region>VI</region>
		</division>
		<division>
			<name>Trikala</name>
			<code>44</code>
			<fips>22</fips>
			<region>V</region>
		</division>
		<division>
			<name>Grevena</name>
			<code>51</code>
			<fips>10</fips>
			<region>III</region>
		</division>
		<division>
			<name>Chania</name>
			<code>94</code>
			<fips></fips>
			<region>XIII</region>
		</division>
		<division>
			<name>Acha&#x3f;a</name>
			<code>13</code>
			<fips></fips>
			<region>VII</region>
		</division>
		<division>
			<name>Korinthia</name>
			<code>15</code>
			<fips>37</fips>
			<region>X</region>
		</division>
		<division>
			<name>Thesprotia</name>
			<code>32</code>
			<fips>18</fips>
			<region>IV</region>
		</division>
		<division>
			<name>Kastoria</name>
			<code>56</code>
			<fips>09</fips>
			<region>III</region>
		</division>
		<division>
			<name>Lesvos</name>
			<code>83</code>
			<fips>51</fips>
			<region>XI</region>
		</division>
		<num>300</num>
	</territory>
	<territory>
		<name>Greenland</name>
		<alpha2>GL</alpha2>
		<alpha3>GRL</alpha3>
		<num>304</num>
	</territory>
	<territory>
		<name>Grenada</name>
		<alpha2>GD</alpha2>
		<alpha3>GRD</alpha3>
		<division>
			<name>Saint Andrew</name>
			<code>AN</code>
			<fips>1</fips>
			<region></region>
		</division>
		<division>
			<name>Carriacou</name>
			<code>CA</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Saint John</name>
			<code>JO</code>
			<fips>4</fips>
			<region></region>
		</division>
		<division>
			<name>Saint George</name>
			<code>GE</code>
			<fips>3</fips>
			<region></region>
		</division>
		<division>
			<name>Saint David</name>
			<code>DA</code>
			<fips>2</fips>
			<region></region>
		</division>
		<division>
			<name>Saint Patrick</name>
			<code>PA</code>
			<fips>6</fips>
			<region></region>
		</division>
		<division>
			<name>Saint Mark</name>
			<code>MA</code>
			<fips>5</fips>
			<region></region>
		</division>
		<num>308</num>
	</territory>
	<territory>
		<name>Guadeloupe</name>
		<alpha2>GP</alpha2>
		<alpha3>GLP</alpha3>
		<num>312</num>
	</territory>
	<territory>
		<name>Guam</name>
		<alpha2>GU</alpha2>
		<alpha3>GUM</alpha3>
		<num>316</num>
	</territory>
	<territory>
		<name>Guatemala</name>
		<alpha2>GT</alpha2>
		<alpha3>GTM</alpha3>
		<division>
			<name>Zacapa</name>
			<code>ZA</code>
			<fips>22</fips>
			<region></region>
		</division>
		<division>
			<name>Baja Verapaz</name>
			<code>BV</code>
			<fips>02</fips>
			<region></region>
		</division>
		<division>
			<name>San Marcos</name>
			<code>SM</code>
			<fips>17</fips>
			<region></region>
		</division>
		<division>
			<name>Izabal</name>
			<code>IZ</code>
			<fips>09</fips>
			<region></region>
		</division>
		<division>
			<name>Suchitep&#xe9;quez</name>
			<code>SU</code>
			<fips>20</fips>
			<region></region>
		</division>
		<division>
			<name>Escuintla</name>
			<code>ES</code>
			<fips>06</fips>
			<region></region>
		</division>
		<division>
			<name>Solol&#xe1;</name>
			<code>SO</code>
			<fips>19</fips>
			<region></region>
		</division>
		<division>
			<name>Totonicap&#xe1;n</name>
			<code>TO</code>
			<fips>21</fips>
			<region></region>
		</division>
		<division>
			<name>Pet&#xe9;n</name>
			<code>PE</code>
			<fips>12</fips>
			<region></region>
		</division>
		<division>
			<name>Alta Verapaz</name>
			<code>AV</code>
			<fips>01</fips>
			<region></region>
		</division>
		<division>
			<name>Quetzaltenango</name>
			<code>QZ</code>
			<fips>13</fips>
			<region></region>
		</division>
		<division>
			<name>Quich&#xe9;</name>
			<code>QC</code>
			<fips>14</fips>
			<region></region>
		</division>
		<division>
			<name>Guatemala</name>
			<code>GU</code>
			<fips>07</fips>
			<region></region>
		</division>
		<division>
			<name>Jalapa</name>
			<code>JA</code>
			<fips>10</fips>
			<region></region>
		</division>
		<division>
			<name>Retalhuleu</name>
			<code>RE</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>El Progreso</name>
			<code>PR</code>
			<fips>05</fips>
			<region></region>
		</division>
		<division>
			<name>Jutiapa</name>
			<code>JU</code>
			<fips>11</fips>
			<region></region>
		</division>
		<division>
			<name>Huehuetenango</name>
			<code>HU</code>
			<fips>08</fips>
			<region></region>
		</division>
		<division>
			<name>Chimaltenango</name>
			<code>CM</code>
			<fips>03</fips>
			<region></region>
		</division>
		<division>
			<name>Sacatep&#xe9;quez</name>
			<code>SA</code>
			<fips>16</fips>
			<region></region>
		</division>
		<division>
			<name>Chiquimula</name>
			<code>CQ</code>
			<fips>04</fips>
			<region></region>
		</division>
		<division>
			<name>Santa Rosa</name>
			<code>SR</code>
			<fips>18</fips>
			<region></region>
		</division>
		<num>320</num>
	</territory>
	<territory>
		<name>Guernsey</name>
		<alpha2>GG</alpha2>
		<alpha3>GGY</alpha3>
		<num>831</num>
	</territory>
	<territory>
		<name>Guinea</name>
		<alpha2>GN</alpha2>
		<alpha3>GIN</alpha3>
		<division>
			<name>Dabola</name>
			<code>DB</code>
			<fips>05</fips>
			<region>F</region>
		</division>
		<division>
			<name>Mamou</name>
			<code>MM</code>
			<fips>23</fips>
			<region>M</region>
		</division>
		<division>
			<name>Kindia</name>
			<code>KD</code>
			<fips>18</fips>
			<region>D</region>
		</division>
		<division>
			<name>Tougu&#xe9;</name>
			<code>TO</code>
			<fips>28</fips>
			<region>L</region>
		</division>
		<division>
			<name>Kankan</name>
			<code>KA</code>
			<fips>32</fips>
			<region>K</region>
		</division>
		<division>
			<name>For&#xe9;cariah</name>
			<code>FO</code>
			<fips>10</fips>
			<region>D</region>
		</division>
		<division>
			<name>L&#xe9;louma</name>
			<code>LE</code>
			<fips>35</fips>
			<region>L</region>
		</division>
		<division>
			<name>Gaoual</name>
			<code>GA</code>
			<fips>12</fips>
			<region>B</region>
		</division>
		<division>
			<name>Yomou</name>
			<code>YO</code>
			<fips>29</fips>
			<region>N</region>
		</division>
		<division>
			<name>Dubr&#xe9;ka</name>
			<code>DU</code>
			<fips>31</fips>
			<region>D</region>
		</division>
		<division>
			<name>Dalaba</name>
			<code>DL</code>
			<fips>06</fips>
			<region>M</region>
		</division>
		<division>
			<name>Fria</name>
			<code>FR</code>
			<fips>11</fips>
			<region>B</region>
		</division>
		<division>
			<name>Kissidougou</name>
			<code>KS</code>
			<fips>17</fips>
			<region>F</region>
		</division>
		<division>
			<name>Lab&#xe9;</name>
			<code>LA</code>
			<fips>34</fips>
			<region>L</region>
		</division>
		<division>
			<name>Nz&#xe9;r&#xe9;kor&#xe9;</name>
			<code>NZ</code>
			<fips>38</fips>
			<region>N</region>
		</division>
		<division>
			<name>Gu&#xe9;k&#xe9;dou</name>
			<code>GU</code>
			<fips>13</fips>
			<region>N</region>
		</division>
		<division>
			<name>Macenta</name>
			<code>MC</code>
			<fips>21</fips>
			<region>N</region>
		</division>
		<division>
			<name>Mandiana</name>
			<code>MD</code>
			<fips>37</fips>
			<region>K</region>
		</division>
		<division>
			<name>Coyah</name>
			<code>CO</code>
			<fips>30</fips>
			<region>D</region>
		</division>
		<division>
			<name>Koundara</name>
			<code>KN</code>
			<fips></fips>
			<region>B</region>
		</division>
		<division>
			<name>Faranah</name>
			<code>FA</code>
			<fips>09</fips>
			<region>F</region>
		</division>
		<division>
			<name>Pita</name>
			<code>PI</code>
			<fips>25</fips>
			<region>M</region>
		</division>
		<division>
			<name>Kouroussa</name>
			<code>KO</code>
			<fips>19</fips>
			<region>K</region>
		</division>
		<division>
			<name>Bok&#xe9;</name>
			<code>BK</code>
			<fips>03</fips>
			<region>B</region>
		</division>
		<division>
			<name>Koubia</name>
			<code>KB</code>
			<fips>33</fips>
			<region>L</region>
		</division>
		<division>
			<name>Boffa</name>
			<code>BF</code>
			<fips>02</fips>
			<region>B</region>
		</division>
		<division>
			<name>Siguiri</name>
			<code>SI</code>
			<fips>39</fips>
			<region>K</region>
		</division>
		<division>
			<name>Mali</name>
			<code>ML</code>
			<fips>22</fips>
			<region>L</region>
		</division>
		<division>
			<name>K&#xe9;rouan&#xe9;</name>
			<code>KE</code>
			<fips>15</fips>
			<region>K</region>
		</division>
		<division>
			<name>Dinguiraye</name>
			<code>DI</code>
			<fips>07</fips>
			<region>F</region>
		</division>
		<division>
			<name>Lola</name>
			<code>LO</code>
			<fips>36</fips>
			<region>N</region>
		</division>
		<division>
			<name>Beyla</name>
			<code>BE</code>
			<fips>01</fips>
			<region>N</region>
		</division>
		<division>
			<name>T&#xe9;lim&#xe9;l&#xe9;</name>
			<code>TE</code>
			<fips>27</fips>
			<region>D</region>
		</division>
		<num>324</num>
	</territory>
	<territory>
		<name>Guinea-Bissau</name>
		<alpha2>GW</alpha2>
		<alpha3>GNB</alpha3>
		<division>
			<name>Bolama</name>
			<code>BL</code>
			<fips>05</fips>
			<region>S</region>
		</division>
		<division>
			<name>Tombali</name>
			<code>TO</code>
			<fips>07</fips>
			<region>S</region>
		</division>
		<division>
			<name>Oio</name>
			<code>OI</code>
			<fips>04</fips>
			<region>N</region>
		</division>
		<division>
			<name>Bafat&#xe1;</name>
			<code>BA</code>
			<fips>01</fips>
			<region>L</region>
		</division>
		<division>
			<name>Biombo</name>
			<code>BM</code>
			<fips>12</fips>
			<region>N</region>
		</division>
		<division>
			<name>Bissau</name>
			<code>BS</code>
			<fips>11</fips>
			<region></region>
		</division>
		<division>
			<name>Cacheu</name>
			<code>CA</code>
			<fips>06</fips>
			<region>N</region>
		</division>
		<division>
			<name>Quinara</name>
			<code>QU</code>
			<fips>02</fips>
			<region>S</region>
		</division>
		<division>
			<name>Gab&#xfa;</name>
			<code>GA</code>
			<fips>10</fips>
			<region>L</region>
		</division>
		<num>624</num>
	</territory>
	<territory>
		<name>Guyana</name>
		<alpha2>GY</alpha2>
		<alpha3>GUY</alpha3>
		<division>
			<name>Upper Demerara-Berbice</name>
			<code>UD</code>
			<fips>18</fips>
			<region></region>
		</division>
		<division>
			<name>Upper Takutu-Upper Essequibo</name>
			<code>UT</code>
			<fips>19</fips>
			<region></region>
		</division>
		<division>
			<name>Pomeroon-Supenaam</name>
			<code>PM</code>
			<fips>16</fips>
			<region></region>
		</division>
		<division>
			<name>Potaro-Siparuni</name>
			<code>PT</code>
			<fips>17</fips>
			<region></region>
		</division>
		<division>
			<name>Essequibo Islands-West Demerara</name>
			<code>ES</code>
			<fips>14</fips>
			<region></region>
		</division>
		<division>
			<name>Demerara-Mahaica</name>
			<code>DE</code>
			<fips>12</fips>
			<region></region>
		</division>
		<division>
			<name>Mahaica-Berbice</name>
			<code>MA</code>
			<fips>15</fips>
			<region></region>
		</division>
		<division>
			<name>Cuyuni-Mazaruni</name>
			<code>CU</code>
			<fips>11</fips>
			<region></region>
		</division>
		<division>
			<name>East Berbice-Corentyne</name>
			<code>EB</code>
			<fips>13</fips>
			<region></region>
		</division>
		<num>328</num>
	</territory>
	<territory>
		<name>Haiti</name>
		<alpha2>HT</alpha2>
		<alpha3>HTI</alpha3>
		<division>
			<name>Ouest</name>
			<code>OU</code>
			<fips>11</fips>
			<region></region>
		</division>
		<division>
			<name>Centre</name>
			<code>CE</code>
			<fips>07</fips>
			<region></region>
		</division>
		<division>
			<name>Artibonite</name>
			<code>AR</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Sud-Est</name>
			<code>SE</code>
			<fips>13</fips>
			<region></region>
		</division>
		<division>
			<name>Grande-Anse</name>
			<code>GA</code>
			<fips>08</fips>
			<region></region>
		</division>
		<division>
			<name>Sud</name>
			<code>SD</code>
			<fips>12</fips>
			<region></region>
		</division>
		<division>
			<name>Nord-Ouest</name>
			<code>NO</code>
			<fips>03</fips>
			<region></region>
		</division>
		<division>
			<name>Nord</name>
			<code>ND</code>
			<fips>09</fips>
			<region></region>
		</division>
		<division>
			<name>Nord-Est</name>
			<code>NE</code>
			<fips></fips>
			<region></region>
		</division>
		<num>332</num>
	</territory>
	<territory>
		<name>Heard and McDonald Islands</name>
		<alpha2>HM</alpha2>
		<alpha3>HMD</alpha3>
		<num>334</num>
	</territory>
	<territory>
		<name>Vatican City</name>
		<alpha2>VA</alpha2>
		<alpha3>VAT</alpha3>
		<num>336</num>
	</territory>
	<territory>
		<name>Honduras</name>
		<alpha2>HN</alpha2>
		<alpha3>HND</alpha3>
		<division>
			<name>Ocotepeque</name>
			<code>OC</code>
			<fips>14</fips>
			<region></region>
		</division>
		<num>340</num>
	</territory>
	<territory>
		<name>Hong Kong</name>
		<alpha2>HK</alpha2>
		<alpha3>HKG</alpha3>
		<num>344</num>
		<note>special administrative region of China</note>
	</territory>
	<territory>
		<name>Hungary</name>
		<alpha2>HU</alpha2>
		<alpha3>HUN</alpha3>
		<division>
			<name>Hajd&#xfa;-Bihar</name>
			<code>HB</code>
			<fips>10</fips>
			<region></region>
		</division>
		<division>
			<name>Szeged</name>
			<code>SD</code>
			<fips>19</fips>
			<region></region>
		</division>
		<division>
			<name>Zala</name>
			<code>ZA</code>
			<fips>24</fips>
			<region></region>
		</division>
		<division>
			<name>Zalaegerszeg</name>
			<code>ZE</code>
			<fips>40</fips>
			<region></region>
		</division>
		<division>
			<name>B&#xe9;k&#xe9;s</name>
			<code>BE</code>
			<fips>03</fips>
			<region></region>
		</division>
		<division>
			<name>Csongr&#xe1;d</name>
			<code>CS</code>
			<fips>06</fips>
			<region></region>
		</division>
		<division>
			<name>Vas</name>
			<code>VA</code>
			<fips>22</fips>
			<region></region>
		</division>
		<division>
			<name>B&#xe1;cs-Kiskun</name>
			<code>BK</code>
			<fips>01</fips>
			<region></region>
		</division>
		<division>
			<name>Szombathely</name>
			<code>SH</code>
			<fips>37</fips>
			<region></region>
		</division>
		<division>
			<name>N&#xf3;gr&#xe1;d</name>
			<code>NO</code>
			<fips>14</fips>
			<region></region>
		</division>
		<division>
			<name>Sz&#xe9;kesfeh&#xe9;rv&#xe1;r</name>
			<code>SF</code>
			<fips>35</fips>
			<region></region>
		</division>
		<division>
			<name>Eger</name>
			<code>EG</code>
			<fips>28</fips>
			<region></region>
		</division>
		<division>
			<name>J&#xe1;sz-Nagykun-Szolnok</name>
			<code>JN</code>
			<fips>20</fips>
			<region></region>
		</division>
		<division>
			<name>Nagykanizsa</name>
			<code>NK</code>
			<fips>32</fips>
			<region></region>
		</division>
		<division>
			<name>Veszpr&#xe9;m</name>
			<code>VM</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Szabolcs-Szatm&#xe1;r-Bereg</name>
			<code>SZ</code>
			<fips>18</fips>
			<region></region>
		</division>
		<division>
			<name>Borsod-Aba&#xfa;j-Zempl&#xe9;n</name>
			<code>BZ</code>
			<fips>04</fips>
			<region></region>
		</division>
		<division>
			<name>Budapest</name>
			<code>BU</code>
			<fips>05</fips>
			<region></region>
		</division>
		<division>
			<name>Ny&#xed;regyh&#xe1;za</name>
			<code>NY</code>
			<fips>33</fips>
			<region></region>
		</division>
		<division>
			<name>Szeksz&#xe1;rd</name>
			<code>SS</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Baranya</name>
			<code>BA</code>
			<fips>02</fips>
			<region></region>
		</division>
		<division>
			<name>Gyor-Moson-Sopron</name>
			<code>GS</code>
			<fips>09</fips>
			<region></region>
		</division>
		<division>
			<name>Kom&#xe1;rom-Esztergom</name>
			<code>KE</code>
			<fips>12</fips>
			<region></region>
		</division>
		<division>
			<name>Fej&#xe9;r</name>
			<code>FE</code>
			<fips>08</fips>
			<region></region>
		</division>
		<division>
			<name>H&#xf3;dmezov&#xe1;s&#xe1;rhely</name>
			<code>HV</code>
			<fips>29</fips>
			<region></region>
		</division>
		<division>
			<name>Sopron</name>
			<code>SN</code>
			<fips>34</fips>
			<region></region>
		</division>
		<division>
			<name>Heves</name>
			<code>HE</code>
			<fips>11</fips>
			<region></region>
		</division>
		<division>
			<name>P&#xe9;cs</name>
			<code>PS</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Salg&#xf3;tarj&#xe1;n</name>
			<code>ST</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Szolnok</name>
			<code>SK</code>
			<fips>36</fips>
			<region></region>
		</division>
		<division>
			<name>Gyor</name>
			<code>GY</code>
			<fips>25</fips>
			<region></region>
		</division>
		<division>
			<name>Debrecen</name>
			<code>DE</code>
			<fips>07</fips>
			<region></region>
		</division>
		<division>
			<name>Kecskem&#xe9;t</name>
			<code>KM</code>
			<fips>31</fips>
			<region></region>
		</division>
		<division>
			<name>B&#xe9;k&#xe9;scsaba</name>
			<code>BC</code>
			<fips>26</fips>
			<region></region>
		</division>
		<division>
			<name>Pest</name>
			<code>PE</code>
			<fips>16</fips>
			<region></region>
		</division>
		<division>
			<name>Tolna</name>
			<code>TO</code>
			<fips>21</fips>
			<region></region>
		</division>
		<division>
			<name>Duna&#xfa;jv&#xe1;ros</name>
			<code>DU</code>
			<fips>27</fips>
			<region></region>
		</division>
		<division>
			<name>Tatab&#xe1;nya</name>
			<code>TB</code>
			<fips>38</fips>
			<region></region>
		</division>
		<division>
			<name>Miskolc</name>
			<code>MI</code>
			<fips>13</fips>
			<region></region>
		</division>
		<division>
			<name>Somogy</name>
			<code>SO</code>
			<fips>17</fips>
			<region></region>
		</division>
		<division>
			<name>Kaposv&#xe1;r</name>
			<code>KV</code>
			<fips>30</fips>
			<region></region>
		</division>
		<num>348</num>
	</territory>
	<territory>
		<name>Iceland</name>
		<alpha2>IS</alpha2>
		<alpha3>ISL</alpha3>
		<division>
			<name>Austurland</name>
			<code>7</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Su&#xf0;urnes</name>
			<code>2</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Vestfir&#xf0;ir</name>
			<code>4</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>H&#xf6;fu&#xf0;borgarsv&#xe6;&#xf0;i utan Reykjav&#xed;kur</name>
			<code>1</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Su&#xf0;urland</name>
			<code>8</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Nor&#xf0;urland eystra</name>
			<code>6</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Reykjav&#xed;k</name>
			<code>0</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Vesturland</name>
			<code>3</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Nor&#xf0;urland vestra</name>
			<code>5</code>
			<fips></fips>
			<region></region>
		</division>
		<num>352</num>
	</territory>
	<territory>
		<name>India</name>
		<alpha2>IN</alpha2>
		<alpha3>IND</alpha3>
		<division>
			<name>Daman and Diu</name>
			<code>DD</code>
			<fips>32</fips>
			<region></region>
		</division>
		<division>
			<name>Meghalaya</name>
			<code>ML</code>
			<fips>18</fips>
			<region></region>
		</division>
		<division>
			<name>Arunachal Pradesh</name>
			<code>AR</code>
			<fips>30</fips>
			<region></region>
		</division>
		<division>
			<name>Delhi</name>
			<code>DL</code>
			<fips>07</fips>
			<region></region>
		</division>
		<division>
			<name>Tripura</name>
			<code>TR</code>
			<fips>26</fips>
			<region></region>
		</division>
		<division>
			<name>Maharashtra</name>
			<code>MH</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Sikkim</name>
			<code>SK</code>
			<fips>29</fips>
			<region></region>
		</division>
		<division>
			<name>Tamil Nadu</name>
			<code>TN</code>
			<fips>25</fips>
			<region></region>
		</division>
		<division>
			<name>Karnataka</name>
			<code>KA</code>
			<fips>19</fips>
			<region></region>
		</division>
		<division>
			<name>Goa</name>
			<code>GA</code>
			<fips>33</fips>
			<region></region>
		</division>
		<division>
			<name>Assam</name>
			<code>AS</code>
			<fips>03</fips>
			<region></region>
		</division>
		<division>
			<name>Chhattisgarh</name>
			<code>CT</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Jharkhand</name>
			<code>JH</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Uttaranchal</name>
			<code>UL</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Jammu and Kashmir</name>
			<code>JK</code>
			<fips>12</fips>
			<region></region>
		</division>
		<division>
			<name>Andaman and Nicobar Islands</name>
			<code>AN</code>
			<fips>01</fips>
			<region></region>
		</division>
		<division>
			<name>Rajasthan</name>
			<code>RJ</code>
			<fips>24</fips>
			<region></region>
		</division>
		<division>
			<name>Pondicherry</name>
			<code>PY</code>
			<fips>22</fips>
			<region></region>
		</division>
		<division>
			<name>Uttar Pradesh</name>
			<code>UP</code>
			<fips>27</fips>
			<region></region>
		</division>
		<division>
			<name>Dadra and Nagar Haveli</name>
			<code>DN</code>
			<fips>06</fips>
			<region></region>
		</division>
		<division>
			<name>Kerala</name>
			<code>KL</code>
			<fips>13</fips>
			<region></region>
		</division>
		<division>
			<name>Himachal Pradesh</name>
			<code>HP</code>
			<fips>11</fips>
			<region></region>
		</division>
		<division>
			<name>Bihar</name>
			<code>BR</code>
			<fips>04</fips>
			<region></region>
		</division>
		<division>
			<name>Punjab</name>
			<code>PB</code>
			<fips>23</fips>
			<region></region>
		</division>
		<division>
			<name>Haryana</name>
			<code>HR</code>
			<fips>10</fips>
			<region></region>
		</division>
		<division>
			<name>Orissa</name>
			<code>OR</code>
			<fips>21</fips>
			<region></region>
		</division>
		<division>
			<name>Andhra Pradesh</name>
			<code>AP</code>
			<fips>02</fips>
			<region></region>
		</division>
		<division>
			<name>Nagaland</name>
			<code>NL</code>
			<fips>20</fips>
			<region></region>
		</division>
		<division>
			<name>Manipur</name>
			<code>MN</code>
			<fips>17</fips>
			<region></region>
		</division>
		<division>
			<name>Madhya Pradesh</name>
			<code>MP</code>
			<fips>15</fips>
			<region></region>
		</division>
		<division>
			<name>Chandigarh</name>
			<code>CH</code>
			<fips>05</fips>
			<region></region>
		</division>
		<division>
			<name>Mizoram</name>
			<code>MZ</code>
			<fips>31</fips>
			<region></region>
		</division>
		<division>
			<name>Lakshadweep</name>
			<code>LD</code>
			<fips>14</fips>
			<region></region>
		</division>
		<division>
			<name>West Bengal</name>
			<code>WB</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Gujarat</name>
			<code>GJ</code>
			<fips>09</fips>
			<region></region>
		</division>
		<num>356</num>
	</territory>
	<territory>
		<name>Indonesia</name>
		<alpha2>ID</alpha2>
		<alpha3>IDN</alpha3>
		<division>
			<name>Lampung</name>
			<code>LA</code>
			<fips>15</fips>
			<region>SM</region>
		</division>
		<division>
			<name>Jawa Tengah</name>
			<code>JT</code>
			<fips>07</fips>
			<region>JW</region>
		</division>
		<division>
			<name>Aceh</name>
			<code>AC</code>
			<fips>01</fips>
			<region>SM</region>
		</division>
		<division>
			<name>Sumatera Utara</name>
			<code>SU</code>
			<fips>26</fips>
			<region>SM</region>
		</division>
		<division>
			<name>Maluku Utara</name>
			<code>MU</code>
			<fips></fips>
			<region>MA</region>
		</division>
		<division>
			<name>Kalimantan Selatan</name>
			<code>KS</code>
			<fips>12</fips>
			<region>KA</region>
		</division>
		<division>
			<name>Yogyakarta</name>
			<code>YO</code>
			<fips>10</fips>
			<region>JW</region>
		</division>
		<division>
			<name>Sumatera Barat</name>
			<code>SB</code>
			<fips>24</fips>
			<region>SM</region>
		</division>
		<division>
			<name>Sulawesi Utara</name>
			<code>SA</code>
			<fips>31</fips>
			<region>SL</region>
		</division>
		<division>
			<name>Jawa Timur</name>
			<code>JI</code>
			<fips>08</fips>
			<region>JW</region>
		</division>
		<division>
			<name>Gorontalo</name>
			<code>GO</code>
			<fips></fips>
			<region>SL</region>
		</division>
		<division>
			<name>Kalimantan Tengah</name>
			<code>KT</code>
			<fips>14</fips>
			<region>KA</region>
		</division>
		<division>
			<name>Maluku</name>
			<code>MA</code>
			<fips>28</fips>
			<region>MA</region>
		</division>
		<division>
			<name>Jambi</name>
			<code>JA</code>
			<fips>05</fips>
			<region>SM</region>
		</division>
		<division>
			<name>Papua</name>
			<code>PA</code>
			<fips></fips>
			<region>IJ</region>
		</division>
		<division>
			<name>Nusa Tenggara Barat</name>
			<code>NB</code>
			<fips>17</fips>
			<region>NU</region>
		</division>
		<division>
			<name>Sulawesi Selatan</name>
			<code>SN</code>
			<fips>20</fips>
			<region>SL</region>
		</division>
		<division>
			<name>Banten</name>
			<code>BT</code>
			<fips></fips>
			<region>JW</region>
		</division>
		<division>
			<name>Jawa Barat</name>
			<code>JB</code>
			<fips>30</fips>
			<region>JW</region>
		</division>
		<division>
			<name>Nusa Tenggara Timur</name>
			<code>NT</code>
			<fips>18</fips>
			<region>NU</region>
		</division>
		<division>
			<name>Jakarta Raya</name>
			<code>JK</code>
			<fips>04</fips>
			<region>JW</region>
		</division>
		<division>
			<name>Kalimantan Barat</name>
			<code>KB</code>
			<fips>11</fips>
			<region>KA</region>
		</division>
		<division>
			<name>Bangka Belitung</name>
			<code>BB</code>
			<fips></fips>
			<region>SM</region>
		</division>
		<division>
			<name>Bengkulu</name>
			<code>BE</code>
			<fips>03</fips>
			<region>SM</region>
		</division>
		<division>
			<name>Sumatera Selatan</name>
			<code>SS</code>
			<fips>32</fips>
			<region>SM</region>
		</division>
		<division>
			<name>Sulawesi Tengah</name>
			<code>ST</code>
			<fips>21</fips>
			<region>SL</region>
		</division>
		<division>
			<name>Riau</name>
			<code>RI</code>
			<fips>19</fips>
			<region>SM</region>
		</division>
		<division>
			<name>Sulawesi Tenggara</name>
			<code>SG</code>
			<fips>22</fips>
			<region>SL</region>
		</division>
		<division>
			<name>Kalimantan Timur</name>
			<code>KI</code>
			<fips>14</fips>
			<region>KA</region>
		</division>
		<division>
			<name>Bali</name>
			<code>BA</code>
			<fips>02</fips>
			<region>NU</region>
		</division>
		<num>360</num>
	</territory>
	<territory>
		<name>Iran</name>
		<alpha2>IR</alpha2>
		<alpha3>IRN</alpha3>
		<division>
			<name>Yazd</name>
			<code>25</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Zanjan</name>
			<code>11</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Semnan</name>
			<code>12</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Sistan va Baluchestan</name>
			<code>13</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Qazvin</name>
			<code>28</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Qom</name>
			<code>26</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Tehran</name>
			<code>07</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Markazi</name>
			<code>22</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Lorestan</name>
			<code>20</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Mazandaran</name>
			<code>21</code>
			<fips></fips>
			<region></region>
		</division>
		<num>364</num>
	</territory>
	<territory>
		<name>Iraq</name>
		<alpha2>IQ</alpha2>
		<alpha3>IRQ</alpha3>
		<division>
			<name>Maysan</name>
			<code>MA</code>
			<fips>14</fips>
			<region></region>
		</division>
		<division>
			<name>Al Qadisiyah</name>
			<code>QA</code>
			<fips>04</fips>
			<region></region>
		</division>
		<division>
			<name>Al Anbar</name>
			<code>AN</code>
			<fips>01</fips>
			<region></region>
		</division>
		<division>
			<name>Dahuk</name>
			<code>DA</code>
			<fips>08</fips>
			<region></region>
		</division>
		<division>
			<name>Al Basrah</name>
			<code>BA</code>
			<fips>02</fips>
			<region></region>
		</division>
		<division>
			<name>Salah ad Din</name>
			<code>SD</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>An Najaf</name>
			<code>NA</code>
			<fips>17</fips>
			<region></region>
		</division>
		<division>
			<name>Arbil</name>
			<code>AR</code>
			<fips>11</fips>
			<region></region>
		</division>
		<division>
			<name>At Ta'mim</name>
			<code>TS</code>
			<fips>13</fips>
			<region></region>
		</division>
		<division>
			<name>Al Muthann&#xe1;</name>
			<code>MU</code>
			<fips>03</fips>
			<region></region>
		</division>
		<division>
			<name>Babil</name>
			<code>BB</code>
			<fips>06</fips>
			<region></region>
		</division>
		<division>
			<name>Baghdad</name>
			<code>BG</code>
			<fips>07</fips>
			<region></region>
		</division>
		<division>
			<name>As Sulaymaniyah</name>
			<code>SU</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Dhi Qar</name>
			<code>DQ</code>
			<fips>09</fips>
			<region></region>
		</division>
		<division>
			<name>Wasit</name>
			<code>WA</code>
			<fips>16</fips>
			<region></region>
		</division>
		<division>
			<name>Karbala'</name>
			<code>KA</code>
			<fips>12</fips>
			<region></region>
		</division>
		<division>
			<name>Diyal&#xe1;</name>
			<code>DI</code>
			<fips>10</fips>
			<region></region>
		</division>
		<division>
			<name>Ninaw&#xe1;</name>
			<code>NI</code>
			<fips>15</fips>
			<region></region>
		</division>
		<num>368</num>
	</territory>
	<territory>
		<name>Ireland</name>
		<alpha2>IE</alpha2>
		<alpha3>IRL</alpha3>
		<division>
			<name>Offaly</name>
			<code>OY</code>
			<fips>23</fips>
			<region>L</region>
		</division>
		<division>
			<name>Limerick</name>
			<code>LK</code>
			<fips>16</fips>
			<region>M</region>
		</division>
		<division>
			<name>Kerry</name>
			<code>KY</code>
			<fips></fips>
			<region>M</region>
		</division>
		<division>
			<name>Wicklow</name>
			<code>WW</code>
			<fips>31</fips>
			<region>L</region>
		</division>
		<division>
			<name>Tipperary</name>
			<code>TA</code>
			<fips>26</fips>
			<region>M</region>
		</division>
		<division>
			<name>Leitrim</name>
			<code>LM</code>
			<fips>14</fips>
			<region>C</region>
		</division>
		<division>
			<name>Donegal</name>
			<code>DL</code>
			<fips></fips>
			<region>U</region>
		</division>
		<division>
			<name>Wexford</name>
			<code>WX</code>
			<fips>30</fips>
			<region>L</region>
		</division>
		<division>
			<name>Kilkenny</name>
			<code>KK</code>
			<fips>13</fips>
			<region>L</region>
		</division>
		<division>
			<name>Carlow</name>
			<code>CW</code>
			<fips>01</fips>
			<region>L</region>
		</division>
		<division>
			<name>Cork</name>
			<code>C</code>
			<fips>04</fips>
			<region>M</region>
		</division>
		<division>
			<name>Mayo</name>
			<code>MO</code>
			<fips>20</fips>
			<region>C</region>
		</division>
		<division>
			<name>Waterford</name>
			<code>WD</code>
			<fips>27</fips>
			<region>M</region>
		</division>
		<division>
			<name>Meath</name>
			<code>MH</code>
			<fips>21</fips>
			<region>L</region>
		</division>
		<division>
			<name>Laois</name>
			<code>LS</code>
			<fips>15</fips>
			<region>L</region>
		</division>
		<division>
			<name>Sligo</name>
			<code>SO</code>
			<fips>25</fips>
			<region>C</region>
		</division>
		<division>
			<name>Westmeath</name>
			<code>WH</code>
			<fips>29</fips>
			<region>L</region>
		</division>
		<division>
			<name>Monaghan</name>
			<code>MN</code>
			<fips>22</fips>
			<region>U</region>
		</division>
		<division>
			<name>Kildare</name>
			<code>KE</code>
			<fips>12</fips>
			<region>L</region>
		</division>
		<division>
			<name>Longford</name>
			<code>LD</code>
			<fips>18</fips>
			<region>L</region>
		</division>
		<division>
			<name>Roscommon</name>
			<code>RN</code>
			<fips>24</fips>
			<region>C</region>
		</division>
		<division>
			<name>Clare</name>
			<code>CE</code>
			<fips>03</fips>
			<region>M</region>
		</division>
		<division>
			<name>Galway</name>
			<code>G</code>
			<fips>10</fips>
			<region>C</region>
		</division>
		<division>
			<name>Dublin</name>
			<code>D</code>
			<fips>07</fips>
			<region>L</region>
		</division>
		<division>
			<name>Cavan</name>
			<code>CN</code>
			<fips>02</fips>
			<region>U</region>
		</division>
		<division>
			<name>Louth</name>
			<code>LH</code>
			<fips>19</fips>
			<region>L</region>
		</division>
		<num>372</num>
	</territory>
	<territory>
		<name>Isle of Man</name>
		<alpha2>IM</alpha2>
		<alpha3>IMN</alpha3>
		<num>833</num>
	</territory>
	<territory>
		<name>Israel</name>
		<alpha2>IL</alpha2>
		<alpha3>ISR</alpha3>
		<division>
			<name>HaMerkaz</name>
			<code>M</code>
			<fips>02</fips>
			<region></region>
		</division>
		<division>
			<name>Tel-Aviv</name>
			<code>TA</code>
			<fips>05</fips>
			<region></region>
		</division>
		<division>
			<name>Yerushalayim</name>
			<code>JM</code>
			<fips>06</fips>
			<region></region>
		</division>
		<division>
			<name>HaDarom</name>
			<code>D</code>
			<fips>01</fips>
			<region></region>
		</division>
		<division>
			<name>Hefa</name>
			<code>HA</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Ha Z_afon</name>
			<code>Z</code>
			<fips>03</fips>
			<region></region>
		</division>
		<num>376</num>
	</territory>
	<territory>
		<name>Italy</name>
		<alpha2>IT</alpha2>
		<alpha3>ITA</alpha3>
		<division>
			<name>Oristano</name>
			<code>OR</code>
			<fips></fips>
			<region>88</region>
		</division>
		<division>
			<name>Genova</name>
			<code>GE</code>
			<fips></fips>
			<region>42</region>
		</division>
		<division>
			<name>Cremona</name>
			<code>CR</code>
			<fips></fips>
			<region>25</region>
		</division>
		<division>
			<name>Latina</name>
			<code>LT</code>
			<fips></fips>
			<region>62</region>
		</division>
		<division>
			<name>Isernia</name>
			<code>IS</code>
			<fips></fips>
			<region>67</region>
		</division>
		<division>
			<name>Torino</name>
			<code>TO</code>
			<fips></fips>
			<region>21</region>
		</division>
		<division>
			<name>Viterbo</name>
			<code>VT</code>
			<fips></fips>
			<region>62</region>
		</division>
		<division>
			<name>Matera</name>
			<code>MT</code>
			<fips></fips>
			<region>77</region>
		</division>
		<division>
			<name>Terni</name>
			<code>TR</code>
			<fips></fips>
			<region>55</region>
		</division>
		<division>
			<name>Nuoro</name>
			<code>NU</code>
			<fips></fips>
			<region>88</region>
		</division>
		<division>
			<name>Napoli</name>
			<code>NA</code>
			<fips></fips>
			<region>72</region>
		</division>
		<division>
			<name>Foggia</name>
			<code>FG</code>
			<fips></fips>
			<region>75</region>
		</division>
		<division>
			<name>Trento</name>
			<code>TN</code>
			<fips></fips>
			<region>32</region>
		</division>
		<division>
			<name>Verona</name>
			<code>VR</code>
			<fips></fips>
			<region>34</region>
		</division>
		<division>
			<name>Arezzo</name>
			<code>AR</code>
			<fips></fips>
			<region>52</region>
		</division>
		<division>
			<name>Pisa</name>
			<code>PI</code>
			<fips></fips>
			<region>52</region>
		</division>
		<division>
			<name>Ragusa</name>
			<code>RG</code>
			<fips></fips>
			<region>82</region>
		</division>
		<division>
			<name>Messina</name>
			<code>ME</code>
			<fips></fips>
			<region>82</region>
		</division>
		<division>
			<name>Catanzaro</name>
			<code>CZ</code>
			<fips></fips>
			<region>78</region>
		</division>
		<division>
			<name>Lecce</name>
			<code>LE</code>
			<fips></fips>
			<region>75</region>
		</division>
		<division>
			<name>Imperia</name>
			<code>IM</code>
			<fips></fips>
			<region>42</region>
		</division>
		<division>
			<name>Lecco</name>
			<code>LC</code>
			<fips></fips>
			<region>25</region>
		</division>
		<division>
			<name>Verbano-Cusio-Ossola</name>
			<code>VB</code>
			<fips></fips>
			<region>21</region>
		</division>
		<division>
			<name>Como</name>
			<code>CO</code>
			<fips></fips>
			<region>25</region>
		</division>
		<division>
			<name>Rovigo</name>
			<code>RO</code>
			<fips></fips>
			<region>34</region>
		</division>
		<division>
			<name>Mantova</name>
			<code>MN</code>
			<fips></fips>
			<region>25</region>
		</division>
		<division>
			<name>Grosseto</name>
			<code>GR</code>
			<fips></fips>
			<region>52</region>
		</division>
		<division>
			<name>Aosta</name>
			<code>AO</code>
			<fips></fips>
			<region>23</region>
		</division>
		<division>
			<name>Siracusa</name>
			<code>SR</code>
			<fips></fips>
			<region>82</region>
		</division>
		<division>
			<name>Varese</name>
			<code>VA</code>
			<fips></fips>
			<region>25</region>
		</division>
		<division>
			<name>Pistoia</name>
			<code>PT</code>
			<fips></fips>
			<region>52</region>
		</division>
		<division>
			<name>Avellino</name>
			<code>AV</code>
			<fips></fips>
			<region>72</region>
		</division>
		<division>
			<name>Bologna</name>
			<code>BO</code>
			<fips></fips>
			<region>45</region>
		</division>
		<division>
			<name>Trapani</name>
			<code>TP</code>
			<fips></fips>
			<region>82</region>
		</division>
		<division>
			<name>Treviso</name>
			<code>TV</code>
			<fips></fips>
			<region>34</region>
		</division>
		<division>
			<name>Venezia</name>
			<code>VE</code>
			<fips></fips>
			<region>34</region>
		</division>
		<division>
			<name>Bolzano</name>
			<code>BZ</code>
			<fips></fips>
			<region>32</region>
		</division>
		<division>
			<name>Prato</name>
			<code>PO</code>
			<fips></fips>
			<region>52</region>
		</division>
		<division>
			<name>L'Aquila</name>
			<code>AQ</code>
			<fips></fips>
			<region>65</region>
		</division>
		<division>
			<name>Pordenone</name>
			<code>PN</code>
			<fips></fips>
			<region>36</region>
		</division>
		<division>
			<name>Forl&#xec;</name>
			<code>FO</code>
			<fips></fips>
			<region>45</region>
		</division>
		<division>
			<name>Agrigento</name>
			<code>AG</code>
			<fips></fips>
			<region>82</region>
		</division>
		<division>
			<name>Chieti</name>
			<code>CH</code>
			<fips></fips>
			<region>65</region>
		</division>
		<division>
			<name>Ferrara</name>
			<code>FE</code>
			<fips></fips>
			<region>45</region>
		</division>
		<division>
			<name>Reggio Calabria</name>
			<code>RC</code>
			<fips></fips>
			<region>78</region>
		</division>
		<division>
			<name>Sondrio</name>
			<code>SO</code>
			<fips></fips>
			<region>25</region>
		</division>
		<division>
			<name>Ancona</name>
			<code>AN</code>
			<fips></fips>
			<region>57</region>
		</division>
		<division>
			<name>Enna</name>
			<code>EN</code>
			<fips></fips>
			<region>82</region>
		</division>
		<division>
			<name>Savona</name>
			<code>SV</code>
			<fips></fips>
			<region>42</region>
		</division>
		<division>
			<name>Frosinone </name>
			<code>FR</code>
			<fips></fips>
			<region>62</region>
		</division>
		<division>
			<name>Benevento</name>
			<code>BN</code>
			<fips></fips>
			<region>72</region>
		</division>
		<division>
			<name>Pescara</name>
			<code>PE</code>
			<fips></fips>
			<region>65</region>
		</division>
		<division>
			<name>Massa-Carrara</name>
			<code>MS</code>
			<fips></fips>
			<region>52</region>
		</division>
		<division>
			<name>Vibo Valentia</name>
			<code>VV</code>
			<fips></fips>
			<region>78</region>
		</division>
		<division>
			<name>Siena</name>
			<code>SI</code>
			<fips></fips>
			<region>52</region>
		</division>
		<division>
			<name>Rimini</name>
			<code>RN</code>
			<fips></fips>
			<region>45</region>
		</division>
		<division>
			<name>Pesaro e Urbino</name>
			<code>PS</code>
			<fips></fips>
			<region>57</region>
		</division>
		<division>
			<name>Macerata</name>
			<code>MC</code>
			<fips></fips>
			<region>57</region>
		</division>
		<division>
			<name>Parma</name>
			<code>PR</code>
			<fips></fips>
			<region>45</region>
		</division>
		<division>
			<name>Cagliari</name>
			<code>CA</code>
			<fips></fips>
			<region>88</region>
		</division>
		<division>
			<name>Milano</name>
			<code>MI</code>
			<fips></fips>
			<region>25</region>
		</division>
		<division>
			<name>Reggio Emilia</name>
			<code>RE</code>
			<fips></fips>
			<region>45</region>
		</division>
		<division>
			<name>Trieste</name>
			<code>TS</code>
			<fips></fips>
			<region>36</region>
		</division>
		<division>
			<name>Taranto</name>
			<code>TA</code>
			<fips></fips>
			<region>75</region>
		</division>
		<division>
			<name>Pavia</name>
			<code>PV</code>
			<fips></fips>
			<region>25</region>
		</division>
		<division>
			<name>Piacenza</name>
			<code>PC</code>
			<fips></fips>
			<region>45</region>
		</division>
		<division>
			<name>Firenze</name>
			<code>FI</code>
			<fips></fips>
			<region>52</region>
		</division>
		<division>
			<name>Modena</name>
			<code>MO</code>
			<fips></fips>
			<region>45</region>
		</division>
		<division>
			<name>Novara</name>
			<code>NO</code>
			<fips></fips>
			<region>21</region>
		</division>
		<division>
			<name>Brindisi</name>
			<code>BR</code>
			<fips></fips>
			<region>75</region>
		</division>
		<division>
			<name>Livorno</name>
			<code>LI</code>
			<fips></fips>
			<region>52</region>
		</division>
		<division>
			<name>Crotone</name>
			<code>KR</code>
			<fips></fips>
			<region>78</region>
		</division>
		<division>
			<name>Campobasso</name>
			<code>CB</code>
			<fips></fips>
			<region>67</region>
		</division>
		<division>
			<name>Caserta</name>
			<code>CE</code>
			<fips></fips>
			<region>72</region>
		</division>
		<division>
			<name>Bergamo</name>
			<code>BG</code>
			<fips></fips>
			<region>25</region>
		</division>
		<division>
			<name>Sassari</name>
			<code>SS</code>
			<fips></fips>
			<region>88</region>
		</division>
		<division>
			<name>Roma</name>
			<code>RM</code>
			<fips></fips>
			<region>62</region>
		</division>
		<division>
			<name>Caltanissetta</name>
			<code>CL</code>
			<fips></fips>
			<region>82</region>
		</division>
		<division>
			<name>Perugia</name>
			<code>PG</code>
			<fips></fips>
			<region>55</region>
		</division>
		<division>
			<name>Rieti</name>
			<code>RI</code>
			<fips></fips>
			<region>62</region>
		</division>
		<division>
			<name>Brescia</name>
			<code>BS</code>
			<fips></fips>
			<region>25</region>
		</division>
		<division>
			<name>Salerno</name>
			<code>SA</code>
			<fips></fips>
			<region>72</region>
		</division>
		<division>
			<name>Cuneo</name>
			<code>CN</code>
			<fips></fips>
			<region>21</region>
		</division>
		<division>
			<name>Vicenza</name>
			<code>VI</code>
			<fips></fips>
			<region>34</region>
		</division>
		<division>
			<name>Gorizia</name>
			<code>GO</code>
			<fips></fips>
			<region>36</region>
		</division>
		<division>
			<name>Biella</name>
			<code>BI</code>
			<fips></fips>
			<region>21</region>
		</division>
		<division>
			<name>Udine</name>
			<code>UD</code>
			<fips></fips>
			<region>36</region>
		</division>
		<division>
			<name>Lodi</name>
			<code>LO</code>
			<fips></fips>
			<region>25</region>
		</division>
		<division>
			<name>Alessandria</name>
			<code>AL</code>
			<fips></fips>
			<region>21</region>
		</division>
		<division>
			<name>Palermo</name>
			<code>PA</code>
			<fips></fips>
			<region>82</region>
		</division>
		<division>
			<name>Vercelli</name>
			<code>VC</code>
			<fips></fips>
			<region>21</region>
		</division>
		<division>
			<name>Ascoli Piceno</name>
			<code>AP</code>
			<fips></fips>
			<region>57</region>
		</division>
		<division>
			<name>Bari</name>
			<code>BA</code>
			<fips></fips>
			<region>75</region>
		</division>
		<division>
			<name>Teramo</name>
			<code>TE</code>
			<fips></fips>
			<region>65</region>
		</division>
		<division>
			<name>Cosenza</name>
			<code>CS</code>
			<fips></fips>
			<region>78</region>
		</division>
		<division>
			<name>Asti</name>
			<code>AT</code>
			<fips></fips>
			<region>21</region>
		</division>
		<division>
			<name>Belluno</name>
			<code>BL</code>
			<fips></fips>
			<region>34</region>
		</division>
		<division>
			<name>Potenza</name>
			<code>PZ</code>
			<fips></fips>
			<region>77</region>
		</division>
		<division>
			<name>Padova</name>
			<code>PD</code>
			<fips></fips>
			<region>34</region>
		</division>
		<division>
			<name>La Spezia</name>
			<code>SP</code>
			<fips></fips>
			<region>42</region>
		</division>
		<division>
			<name>Catania</name>
			<code>CT</code>
			<fips></fips>
			<region>82</region>
		</division>
		<division>
			<name>Lucca</name>
			<code>LU</code>
			<fips></fips>
			<region>52</region>
		</division>
		<division>
			<name>Ravenna</name>
			<code>RA</code>
			<fips></fips>
			<region>45</region>
		</division>
		<num>380</num>
	</territory>
	<territory>
		<name>Jamaica</name>
		<alpha2>JM</alpha2>
		<alpha3>JAM</alpha3>
		<division>
			<name>Saint Catherine</name>
			<code>14</code>
			<fips>10</fips>
			<region></region>
		</division>
		<division>
			<name>Saint Andrew</name>
			<code>02</code>
			<fips>08</fips>
			<region></region>
		</division>
		<division>
			<name>Westmoreland</name>
			<code>10</code>
			<fips>16</fips>
			<region></region>
		</division>
		<division>
			<name>Saint James</name>
			<code>08</code>
			<fips>12</fips>
			<region></region>
		</division>
		<division>
			<name>Saint Ann</name>
			<code>06</code>
			<fips>09</fips>
			<region></region>
		</division>
		<division>
			<name>Trelawny</name>
			<code>07</code>
			<fips>15</fips>
			<region></region>
		</division>
		<division>
			<name>Saint Thomas</name>
			<code>03</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Hanover</name>
			<code>09</code>
			<fips>02</fips>
			<region></region>
		</division>
		<division>
			<name>Saint Elizabeth</name>
			<code>11</code>
			<fips>11</fips>
			<region></region>
		</division>
		<division>
			<name>Saint Mary</name>
			<code>05</code>
			<fips>13</fips>
			<region></region>
		</division>
		<division>
			<name>Portland</name>
			<code>04</code>
			<fips>07</fips>
			<region></region>
		</division>
		<division>
			<name>Clarendon</name>
			<code>13</code>
			<fips>01</fips>
			<region></region>
		</division>
		<division>
			<name>Manchester</name>
			<code>12</code>
			<fips>04</fips>
			<region></region>
		</division>
		<division>
			<name>Kingston</name>
			<code>01</code>
			<fips></fips>
			<region></region>
		</division>
		<num>388</num>
	</territory>
	<territory>
		<name>Japan</name>
		<alpha2>JP</alpha2>
		<alpha3>JPN</alpha3>
		<division>
			<name>Yamaguti [Yamaguchi]</name>
			<code>35</code>
			<fips>45</fips>
			<region></region>
		</division>
		<division>
			<name>Okayama</name>
			<code>33</code>
			<fips>31</fips>
			<region></region>
		</division>
		<division>
			<name>Hukuoka [Fukuoka]</name>
			<code>40</code>
			<fips>07</fips>
			<region></region>
		</division>
		<division>
			<name>&#xd4;saka [Osaka]</name>
			<code>27</code>
			<fips>32</fips>
			<region></region>
		</division>
		<division>
			<name>Nagano</name>
			<code>20</code>
			<fips>26</fips>
			<region></region>
		</division>
		<division>
			<name>Saitama</name>
			<code>11</code>
			<fips>34</fips>
			<region></region>
		</division>
		<division>
			<name>Kumamoto</name>
			<code>43</code>
			<fips>21</fips>
			<region></region>
		</division>
		<division>
			<name>Miyazaki</name>
			<code>45</code>
			<fips>25</fips>
			<region></region>
		</division>
		<division>
			<name>K&#xf4;ti [Kochi]</name>
			<code>39</code>
			<fips>20</fips>
			<region></region>
		</division>
		<division>
			<name>Simane [Shimane]</name>
			<code>32</code>
			<fips>36</fips>
			<region></region>
		</division>
		<division>
			<name>Wakayama</name>
			<code>30</code>
			<fips>43</fips>
			<region></region>
		</division>
		<division>
			<name>Hy&#xf4;go [Hyogo]</name>
			<code>28</code>
			<fips>13</fips>
			<region></region>
		</division>
		<division>
			<name>Ehime</name>
			<code>38</code>
			<fips>05</fips>
			<region></region>
		</division>
		<division>
			<name>Nagasaki</name>
			<code>42</code>
			<fips>27</fips>
			<region></region>
		</division>
		<division>
			<name>Hukusima [Fukushima]</name>
			<code>07</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Ky&#xf4;to [Kyoto]</name>
			<code>26</code>
			<fips>22</fips>
			<region></region>
		</division>
		<division>
			<name>T&#xf4;ky&#xf4; [Tokyo]</name>
			<code>13</code>
			<fips>40</fips>
			<region></region>
		</division>
		<division>
			<name>Hokkaid&#xf4; [Hokkaido]</name>
			<code>01</code>
			<fips>12</fips>
			<region></region>
		</division>
		<division>
			<name>Totigi [Tochigi]</name>
			<code>09</code>
			<fips>38</fips>
			<region></region>
		</division>
		<division>
			<name>Tottori</name>
			<code>31</code>
			<fips>41</fips>
			<region></region>
		</division>
		<division>
			<name>Sizuoka [Shizuoka]</name>
			<code>22</code>
			<fips>37</fips>
			<region></region>
		</division>
		<division>
			<name>Mie</name>
			<code>24</code>
			<fips>23</fips>
			<region></region>
		</division>
		<division>
			<name>Siga [Shiga]</name>
			<code>25</code>
			<fips>35</fips>
			<region></region>
		</division>
		<division>
			<name>Nara</name>
			<code>29</code>
			<fips>28</fips>
			<region></region>
		</division>
		<division>
			<name>Yamanasi [Yamanashi]</name>
			<code>19</code>
			<fips>46</fips>
			<region></region>
		</division>
		<division>
			<name>Kagawa</name>
			<code>37</code>
			<fips>17</fips>
			<region></region>
		</division>
		<division>
			<name>Iwate</name>
			<code>03</code>
			<fips>16</fips>
			<region></region>
		</division>
		<division>
			<name>Tiba [Chiba]</name>
			<code>12</code>
			<fips>04</fips>
			<region></region>
		</division>
		<division>
			<name>Niigata</name>
			<code>15</code>
			<fips>29</fips>
			<region></region>
		</division>
		<division>
			<name>Hukui [Fukui]</name>
			<code>18</code>
			<fips>06</fips>
			<region></region>
		</division>
		<division>
			<name>Kagosima [Kagoshima]</name>
			<code>46</code>
			<fips>18</fips>
			<region></region>
		</division>
		<division>
			<name>Aichi</name>
			<code>23</code>
			<fips>01</fips>
			<region></region>
		</division>
		<division>
			<name>Hirosima [Hiroshima]</name>
			<code>34</code>
			<fips>11</fips>
			<region></region>
		</division>
		<division>
			<name>Gihu</name>
			<code>21</code>
			<fips>09</fips>
			<region></region>
		</division>
		<division>
			<name>Toyama</name>
			<code>16</code>
			<fips>42</fips>
			<region></region>
		</division>
		<division>
			<name>Saga</name>
			<code>41</code>
			<fips>33</fips>
			<region></region>
		</division>
		<division>
			<name>Akita</name>
			<code>05</code>
			<fips>02</fips>
			<region></region>
		</division>
		<division>
			<name>Yamagata</name>
			<code>06</code>
			<fips>44</fips>
			<region></region>
		</division>
		<division>
			<name>Ibaraki</name>
			<code>08</code>
			<fips>14</fips>
			<region></region>
		</division>
		<division>
			<name>Isikawa [Ishikawa]</name>
			<code>17</code>
			<fips>15</fips>
			<region></region>
		</division>
		<division>
			<name>Aomori</name>
			<code>02</code>
			<fips>03</fips>
			<region></region>
		</division>
		<division>
			<name>Okinawa</name>
			<code>47</code>
			<fips>47</fips>
			<region></region>
		</division>
		<division>
			<name>Tokusima [Tokushima]</name>
			<code>36</code>
			<fips>39</fips>
			<region></region>
		</division>
		<division>
			<name>Miyagi</name>
			<code>04</code>
			<fips>24</fips>
			<region></region>
		</division>
		<division>
			<name>Gunma</name>
			<code>10</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>&#xd4;ita [Oita]</name>
			<code>44</code>
			<fips>30</fips>
			<region></region>
		</division>
		<division>
			<name>Kanagawa</name>
			<code>14</code>
			<fips>19</fips>
			<region></region>
		</division>
		<num>392</num>
	</territory>
	<territory>
		<name>Jersey</name>
		<alpha2>JE</alpha2>
		<alpha3>JEY</alpha3>
		<num>832</num>
	</territory>
	<territory>
		<name>Jordan</name>
		<alpha2>JO</alpha2>
		<alpha3>JOR</alpha3>
		<division>
			<name>Ajlun</name>
			<code>AJ</code>
			<fips>20</fips>
			<region></region>
		</division>
		<division>
			<name>Az Zarqa'</name>
			<code>AZ</code>
			<fips>17</fips>
			<region></region>
		</division>
		<division>
			<name>Al Aqaba</name>
			<code>AQ</code>
			<fips>21</fips>
			<region></region>
		</division>
		<division>
			<name>Irbid</name>
			<code>IR</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Amman</name>
			<code>AM</code>
			<fips>16</fips>
			<region></region>
		</division>
		<division>
			<name>Al Karak</name>
			<code>KA</code>
			<fips>09</fips>
			<region></region>
		</division>
		<division>
			<name>At Tafilah</name>
			<code>AT</code>
			<fips>12</fips>
			<region></region>
		</division>
		<division>
			<name>Al Balqa'</name>
			<code>BA</code>
			<fips>02</fips>
			<region></region>
		</division>
		<division>
			<name>Jarash</name>
			<code>JA</code>
			<fips>22</fips>
			<region></region>
		</division>
		<division>
			<name>Ma`an</name>
			<code>MN</code>
			<fips>19</fips>
			<region></region>
		</division>
		<division>
			<name>Al Mafraq</name>
			<code>MA</code>
			<fips>15</fips>
			<region></region>
		</division>
		<division>
			<name>Madaba</name>
			<code>MD</code>
			<fips></fips>
			<region></region>
		</division>
		<num>400</num>
	</territory>
	<territory>
		<name>Kazakhstan</name>
		<alpha2>KZ</alpha2>
		<alpha3>KAZ</alpha3>
		<division>
			<name>Almaty oblysy</name>
			<code>ALM</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Aqmola oblysy</name>
			<code>AKM</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Mangghystau oblysy</name>
			<code>MAN</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Zhambyl oblysy</name>
			<code>ZHA</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Qostanay oblysy</name>
			<code>KUS</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Qyzylorda oblysy</name>
			<code>KZY</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Atyrau oblysy</name>
			<code>ATY</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Qaraghandy oblysy</name>
			<code>KAR</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Shyghys Qazaqstan oblysy</name>
			<code>VOS</code>
			<fips>15</fips>
			<region></region>
		</division>
		<division>
			<name>Aqt&#xf6;be oblysy</name>
			<code>AKT</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Ongt&#xfc;stik Qazaqstan oblysy</name>
			<code>YUZ</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Solt&#xfc;stik Qazaqstan oblysy</name>
			<code>SEV</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Batys Qazaqstan oblysy</name>
			<code>ZAP</code>
			<fips>07</fips>
			<region></region>
		</division>
		<division>
			<name>Pavlodar oblysy</name>
			<code>PAV</code>
			<fips></fips>
			<region></region>
		</division>
		<num>398</num>
	</territory>
	<territory>
		<name>Kenya</name>
		<alpha2>KE</alpha2>
		<alpha3>KEN</alpha3>
		<division>
			<name>Nairobi Municipality</name>
			<code>110</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Central</name>
			<code>200</code>
			<fips>01</fips>
			<region></region>
		</division>
		<division>
			<name>Eastern</name>
			<code>400</code>
			<fips>03</fips>
			<region></region>
		</division>
		<division>
			<name>Coast</name>
			<code>300</code>
			<fips>02</fips>
			<region></region>
		</division>
		<division>
			<name>North-Eastern</name>
			<code>500</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Nyanza</name>
			<code>600</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Western</name>
			<code>900</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Rift Valley</name>
			<code>700</code>
			<fips>08</fips>
			<region></region>
		</division>
		<num>404</num>
	</territory>
	<territory>
		<name>Kiribati</name>
		<alpha2>KI</alpha2>
		<alpha3>KIR</alpha3>
		<division>
			<name>Gilbert Islands</name>
			<code>G</code>
			<fips>01</fips>
			<region></region>
		</division>
		<division>
			<name>Line Islands</name>
			<code>L</code>
			<fips>02</fips>
			<region></region>
		</division>
		<division>
			<name>Phoenix Islands</name>
			<code>P</code>
			<fips>03</fips>
			<region></region>
		</division>
		<num>296</num>
	</territory>
	<territory>
		<name>Korea, North</name>
		<alpha2>KP</alpha2>
		<alpha3>PRK</alpha3>
		<division>
			<name>Hwanghaebuk-do</name>
			<code>HWB</code>
			<fips>07</fips>
			<region></region>
		</division>
		<division>
			<name>Hwanghaenam-do</name>
			<code>HWN</code>
			<fips>06</fips>
			<region></region>
		</division>
		<division>
			<name>Pyonganbuk-do</name>
			<code>PYB</code>
			<fips>15</fips>
			<region></region>
		</division>
		<division>
			<name>Pyongannam-do</name>
			<code>PYN</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Hamgyongnam-do</name>
			<code>HAN</code>
			<fips>03</fips>
			<region></region>
		</division>
		<division>
			<name>Nampo-si</name>
			<code>NAM</code>
			<fips>14</fips>
			<region></region>
		</division>
		<division>
			<name>Yanggang-do</name>
			<code>YAN</code>
			<fips>13</fips>
			<region></region>
		</division>
		<division>
			<name>Kangwon-do</name>
			<code>KAN</code>
			<fips>09</fips>
			<region></region>
		</division>
		<division>
			<name>Chagang-do</name>
			<code>CHA</code>
			<fips>01</fips>
			<region></region>
		</division>
		<division>
			<name>Najin Sonbong-si</name>
			<code>NAJ</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Pyongyang-si</name>
			<code>PYO</code>
			<fips>12</fips>
			<region></region>
		</division>
		<division>
			<name>Hamgyongbuk-do</name>
			<code>HAB</code>
			<fips>16</fips>
			<region></region>
		</division>
		<division>
			<name>Kaesong-si</name>
			<code>KAE</code>
			<fips>08</fips>
			<region></region>
		</division>
		<num>408</num>
	</territory>
	<territory>
		<name>Korea, South</name>
		<alpha2>KR</alpha2>
		<alpha3>KOR</alpha3>
		<division>
			<name>Ulsan Gwang'yeogsi [Ulsan-Kwangyokshi]</name>
			<code>31</code>
			<fips>21</fips>
			<region></region>
		</division>
		<division>
			<name>Seoul Teugbyeolsi [Seoul-T'ukpyolshi]</name>
			<code>11</code>
			<fips>11</fips>
			<region></region>
		</division>
		<division>
			<name>Gwangju Gwang'yeogsi [Kwangju-Kwangyokshi]</name>
			<code>29</code>
			<fips>18</fips>
			<region></region>
		</division>
		<division>
			<name>Daejeon Gwang'yeogsi [Taejon-Kwangyokshi]</name>
			<code>30</code>
			<fips>19</fips>
			<region></region>
		</division>
		<division>
			<name>Jeonranamdo [Chollanam-do]</name>
			<code>46</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Chungcheongnamdo [Ch'ungch'ongnam-do]</name>
			<code>44</code>
			<fips>17</fips>
			<region></region>
		</division>
		<division>
			<name>Gang'weondo [Kang-won-do]</name>
			<code>42</code>
			<fips>06</fips>
			<region></region>
		</division>
		<division>
			<name>Jejudo [Cheju-do]</name>
			<code>49</code>
			<fips>01</fips>
			<region></region>
		</division>
		<division>
			<name>Gyeongsangbugdo [Kyongsangbuk-do]</name>
			<code>47</code>
			<fips>14</fips>
			<region></region>
		</division>
		<division>
			<name>Incheon Gwang'yeogsi [Inch'n-Kwangyokshi]</name>
			<code>28</code>
			<fips>12</fips>
			<region></region>
		</division>
		<division>
			<name>Gyeonggido [Kyonggi-do]</name>
			<code>41</code>
			<fips>13</fips>
			<region></region>
		</division>
		<division>
			<name>Busan Gwang'yeogsi [Pusan-Kwangyokshi]</name>
			<code>26</code>
			<fips>10</fips>
			<region></region>
		</division>
		<division>
			<name>Gyeongsangnamdo [Kyongsangnam-do]</name>
			<code>48</code>
			<fips>20</fips>
			<region></region>
		</division>
		<division>
			<name>Daegu Gwang'yeogsi [Taegu-Kwangyokshi]</name>
			<code>27</code>
			<fips>15</fips>
			<region></region>
		</division>
		<division>
			<name>Jeonrabugdo[Chollabuk-do]</name>
			<code>45</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Chungcheongbugdo [Ch'ungch'ongbuk-do]</name>
			<code>43</code>
			<fips>05</fips>
			<region></region>
		</division>
		<num>410</num>
	</territory>
	<territory>
		<name>Kuwait</name>
		<alpha2>KW</alpha2>
		<alpha3>KWT</alpha3>
		<division>
			<name>Al Ahmadi</name>
			<code>AH</code>
			<fips>04</fips>
			<region></region>
		</division>
		<division>
			<name>Hawalli</name>
			<code>HA</code>
			<fips>03</fips>
			<region></region>
		</division>
		<division>
			<name>Al Kuwayt</name>
			<code>KU</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Al Farwaniyah</name>
			<code>FA</code>
			<fips>06</fips>
			<region></region>
		</division>
		<division>
			<name>Al Jahrah</name>
			<code>JA</code>
			<fips>05</fips>
			<region></region>
		</division>
		<num>414</num>
	</territory>
	<territory>
		<name>Kyrgyzstan</name>
		<alpha2>KG</alpha2>
		<alpha3>KGZ</alpha3>
		<division>
			<name>Bishkek</name>
			<code>GB</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Batken</name>
			<code>B</code>
			<fips></fips>
			<region></region>
		</division>
		<num>417</num>
	</territory>
	<territory>
		<name>Laos</name>
		<alpha2>LA</alpha2>
		<alpha3>LAO</alpha3>
		<division>
			<name>Xiangkhoang [Xieng Khouang]</name>
			<code>XI</code>
			<fips>14</fips>
			<region></region>
		</division>
		<division>
			<name>Khammouan</name>
			<code>KH</code>
			<fips>15</fips>
			<region></region>
		</division>
		<division>
			<name>Xaignabouli [Sayaboury]</name>
			<code>XA</code>
			<fips>13</fips>
			<region></region>
		</division>
		<division>
			<name>Xais&#xf4;mboun</name>
			<code>XN</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>X&#xe9;kong [S&#xe9;kong]</name>
			<code>XE</code>
			<fips>26</fips>
			<region></region>
		</division>
		<division>
			<name>Ph&#xf4;ngsali [Phong Saly]</name>
			<code>PH</code>
			<fips>18</fips>
			<region></region>
		</division>
		<division>
			<name>Vientiane</name>
			<code>VI</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Louang Namtha</name>
			<code>LM</code>
			<fips>16</fips>
			<region></region>
		</division>
		<division>
			<name>Savannakh&#xe9;t</name>
			<code>SV</code>
			<fips>20</fips>
			<region></region>
		</division>
		<division>
			<name>Bolikhamxai [Borikhane]</name>
			<code>BL</code>
			<fips>23</fips>
			<region></region>
		</division>
		<division>
			<name>Champasak [Champassak]</name>
			<code>CH</code>
			<fips>02</fips>
			<region></region>
		</division>
		<division>
			<name>Oud&#xf4;mxai [Oudomsai]</name>
			<code>OU</code>
			<fips>07</fips>
			<region></region>
		</division>
		<division>
			<name>Attapu [Attopeu]</name>
			<code>AT</code>
			<fips>01</fips>
			<region></region>
		</division>
		<division>
			<name>Houaphan</name>
			<code>HO</code>
			<fips>03</fips>
			<region></region>
		</division>
		<division>
			<name>Louangphabang [Louang Prabang]</name>
			<code>LP</code>
			<fips>17</fips>
			<region></region>
		</division>
		<division>
			<name>Bok&#xe8;o</name>
			<code>BK</code>
			<fips>22</fips>
			<region></region>
		</division>
		<division>
			<name>Salavan [Saravane]</name>
			<code>SL</code>
			<fips>19</fips>
			<region></region>
		</division>
		<num>418</num>
	</territory>
	<territory>
		<name>Latvia</name>
		<alpha2>LV</alpha2>
		<alpha3>LVA</alpha3>
		<division>
			<name>Valkas Aprinkis</name>
			<code>VK</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Daugavpils Aprinkis</name>
			<code>DA</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Jurmala</name>
			<code>JUR</code>
			<fips>13</fips>
			<region></region>
		</division>
		<division>
			<name>Ogres Aprinkis</name>
			<code>OG</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Limba&#x17e;u Aprinkis</name>
			<code>LM</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Valmieras Aprinkis</name>
			<code>VM</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Daugavpils</name>
			<code>DGV</code>
			<fips>06</fips>
			<region></region>
		</division>
		<division>
			<name>Jelgavas Aprinkis</name>
			<code>JL</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Tukuma Aprinkis</name>
			<code>TU</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Jelgava</name>
			<code>JEL</code>
			<fips>11</fips>
			<region></region>
		</division>
		<division>
			<name>Jekabpils Aprinkis</name>
			<code>JK</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Kuldigas Aprinkis</name>
			<code>KU</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Dobeles Aprinkis</name>
			<code>DO</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Rigas Aprinkis</name>
			<code>RI</code>
			<fips>25</fips>
			<region></region>
		</division>
		<division>
			<name>Saldus Aprinkis</name>
			<code>SA</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Talsu Aprinkis</name>
			<code>TA</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Madonas Aprinkis</name>
			<code>MA</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Riga</name>
			<code>RIX</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Kraslavas Aprinkis</name>
			<code>KR</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Liepaja</name>
			<code>LPX</code>
			<fips>16</fips>
			<region></region>
		</division>
		<division>
			<name>Gulbenes Aprinkis</name>
			<code>GU</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Aluksnes Aprinkis</name>
			<code>AL</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Cesu Aprinkis</name>
			<code>CE</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Rezeknes Aprinkis</name>
			<code>RE</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Liepajas Aprinkis</name>
			<code>LE</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Preilu Aprinkis</name>
			<code>PR</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Ludzas Aprinkis</name>
			<code>LU</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Balvu Aprinkis</name>
			<code>BL</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Rezekne</name>
			<code>REZ</code>
			<fips>23</fips>
			<region></region>
		</division>
		<division>
			<name>Aizkraukles Aprinkis</name>
			<code>AI</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Ventspils</name>
			<code>VEN</code>
			<fips>32</fips>
			<region></region>
		</division>
		<division>
			<name>Bauskas Aprinkis</name>
			<code>BU</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Ventspils Aprinkis</name>
			<code>VE</code>
			<fips></fips>
			<region></region>
		</division>
		<num>428</num>
	</territory>
	<territory>
		<name>Lebanon</name>
		<alpha2>LB</alpha2>
		<alpha3>LBN</alpha3>
		<division>
			<name>Jabal Loubn&#xe2;ne</name>
			<code>JL</code>
			<fips>05</fips>
			<region></region>
		</division>
		<division>
			<name>Nabat&#xee;y&#xe9;</name>
			<code>NA</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>El B&#xe9;qaa</name>
			<code>BI</code>
			<fips>01</fips>
			<region></region>
		</division>
		<division>
			<name>Beiro&#xfb;t</name>
			<code>BA</code>
			<fips>04</fips>
			<region></region>
		</division>
		<division>
			<name>Loubn&#xe2;ne ech Chem&#xe2;li</name>
			<code>AS</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Loubn&#xe2;ne ej Jno&#xfb;bi</name>
			<code>JA</code>
			<fips></fips>
			<region></region>
		</division>
		<num>422</num>
	</territory>
	<territory>
		<name>Lesotho</name>
		<alpha2>LS</alpha2>
		<alpha3>LSO</alpha3>
		<division>
			<name>Mokhotlong</name>
			<code>J</code>
			<fips>16</fips>
			<region></region>
		</division>
		<division>
			<name>Mafeteng</name>
			<code>E</code>
			<fips>13</fips>
			<region></region>
		</division>
		<division>
			<name>Berea</name>
			<code>D</code>
			<fips>10</fips>
			<region></region>
		</division>
		<division>
			<name>Butha-Buthe</name>
			<code>B</code>
			<fips>11</fips>
			<region></region>
		</division>
		<division>
			<name>Quthing</name>
			<code>G</code>
			<fips>18</fips>
			<region></region>
		</division>
		<division>
			<name>Thaba-Tseka</name>
			<code>K</code>
			<fips>19</fips>
			<region></region>
		</division>
		<division>
			<name>Maseru</name>
			<code>A</code>
			<fips>14</fips>
			<region></region>
		</division>
		<division>
			<name>Qacha's Nek</name>
			<code>H</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Leribe</name>
			<code>C</code>
			<fips>12</fips>
			<region></region>
		</division>
		<division>
			<name>Mohale's Hoek</name>
			<code>F</code>
			<fips>15</fips>
			<region></region>
		</division>
		<num>426</num>
	</territory>
	<territory>
		<name>Liberia</name>
		<alpha2>LR</alpha2>
		<alpha3>LBR</alpha3>
		<division>
			<name>Bomi</name>
			<code>BM</code>
			<fips>15</fips>
			<region></region>
		</division>
		<division>
			<name>Nimba</name>
			<code>NI</code>
			<fips>09</fips>
			<region></region>
		</division>
		<division>
			<name>Sinoe</name>
			<code>SI</code>
			<fips>10</fips>
			<region></region>
		</division>
		<division>
			<name>Maryland</name>
			<code>MY</code>
			<fips>13</fips>
			<region></region>
		</division>
		<division>
			<name>Grand Bassa</name>
			<code>GB</code>
			<fips>11</fips>
			<region></region>
		</division>
		<division>
			<name>Rivercess</name>
			<code>RI</code>
			<fips>18</fips>
			<region></region>
		</division>
		<division>
			<name>Grand Cape Mount</name>
			<code>CM</code>
			<fips>12</fips>
			<region></region>
		</division>
		<division>
			<name>Grand Kru</name>
			<code>GK</code>
			<fips>16</fips>
			<region></region>
		</division>
		<division>
			<name>Lofa</name>
			<code>LO</code>
			<fips>05</fips>
			<region></region>
		</division>
		<division>
			<name>Margibi</name>
			<code>MG</code>
			<fips>17</fips>
			<region></region>
		</division>
		<division>
			<name>Montserrado</name>
			<code>MO</code>
			<fips>14</fips>
			<region></region>
		</division>
		<division>
			<name>Bong</name>
			<code>BG</code>
			<fips>01</fips>
			<region></region>
		</division>
		<division>
			<name>Grand Gedeh</name>
			<code>GG</code>
			<fips>02</fips>
			<region></region>
		</division>
		<num>430</num>
	</territory>
	<territory>
		<name>Libya</name>
		<alpha2>LY</alpha2>
		<alpha3>LBY</alpha3>
		<division>
			<name>Yafran-Jadu</name>
			<code>YJ</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Ghat</name>
			<code>GT</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Al Hizam al Akhhar</name>
			<code>HZ</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Nalut</name>
			<code>NL</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Ash Shati'</name>
			<code>SH</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Surt</name>
			<code>SR</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Bani Walid</name>
			<code>BW</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Al Marqab</name>
			<code>MB</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Ghadamis</name>
			<code>GD</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Darnah</name>
			<code>DR</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Tajura' wa an Nawahi Arbah</name>
			<code>TN</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Murzuq</name>
			<code>MQ</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Al Jifarah</name>
			<code>JI</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Al Qatrun</name>
			<code>QT</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Sabratah Surman</name>
			<code>SS</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Tarhunah-Masallatah</name>
			<code>TM</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Jaghbub</name>
			<code>JB</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Sabha</name>
			<code>SB</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>An Nuqat al Khams</name>
			<code>NQ</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Al Marj</name>
			<code>MJ</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Wadi al Hayat</name>
			<code>WD</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Al Qubbah</name>
			<code>QB</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Mizdah</name>
			<code>MZ</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Al Kufrah</name>
			<code>KF</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Ajdabiya</name>
			<code>AJ</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Gharyan</name>
			<code>GR</code>
			<fips></fips>
			<region></region>
		</division>
		<num>434</num>
	</territory>
	<territory>
		<name>Liechtenstein</name>
		<alpha2>LI</alpha2>
		<alpha3>LIE</alpha3>
		<num>438</num>
	</territory>
	<territory>
		<name>Lithuania</name>
		<alpha2>LT</alpha2>
		<alpha3>LTU</alpha3>
		<division>
			<name>Klaipedos Apskritis</name>
			<code>KL</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Vilniaus Apskritis</name>
			<code>VL</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Paneve&#x17e;io Apskritis</name>
			<code>PN</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Taurages Apskritis</name>
			<code>TA</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Kauno Apskritis</name>
			<code>KU</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Marijampoles Apskritis</name>
			<code>MR</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Tel&#x161;iu Apskritis</name>
			<code>TE</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Utenos Apskritis</name>
			<code>UT</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Alytaus Apskritis</name>
			<code>AL</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>&#x160;iauliu Apskritis</name>
			<code>SA</code>
			<fips></fips>
			<region></region>
		</division>
		<num>440</num>
	</territory>
	<territory>
		<name>Luxembourg</name>
		<alpha2>LU</alpha2>
		<alpha3>LUX</alpha3>
		<division>
			<name>Diekirch</name>
			<code>D</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Luxembourg</name>
			<code>L</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Grevenmacher</name>
			<code>G</code>
			<fips></fips>
			<region></region>
		</division>
		<num>442</num>
	</territory>
	<territory>
		<name>Macau</name>
		<alpha2>MO</alpha2>
		<alpha3>MAC</alpha3>
		<num>446</num>
	</territory>
	<territory>
		<name>Macedonia</name>
		<alpha2>MK</alpha2>
		<alpha3>MKD</alpha3>
		<num>807</num>
	</territory>
	<territory>
		<name>Madagascar</name>
		<alpha2>MG</alpha2>
		<alpha3>MDG</alpha3>
		<division>
			<name>Toliara</name>
			<code>U</code>
			<fips>06</fips>
			<region></region>
		</division>
		<division>
			<name>Mahajanga</name>
			<code>M</code>
			<fips>03</fips>
			<region></region>
		</division>
		<division>
			<name>Antsiranana</name>
			<code>D</code>
			<fips>01</fips>
			<region></region>
		</division>
		<division>
			<name>Fianarantsoa</name>
			<code>F</code>
			<fips>02</fips>
			<region></region>
		</division>
		<division>
			<name>Antananarivo</name>
			<code>T</code>
			<fips>05</fips>
			<region></region>
		</division>
		<division>
			<name>Toamasina</name>
			<code>A</code>
			<fips>04</fips>
			<region></region>
		</division>
		<num>450</num>
	</territory>
	<territory>
		<name>Malawi</name>
		<alpha2>MW</alpha2>
		<alpha3>MWI</alpha3>
		<division>
			<name>Ntchisi</name>
			<code>NI</code>
			<fips>20</fips>
			<region>C</region>
		</division>
		<division>
			<name>Blantyre</name>
			<code>BL</code>
			<fips>24</fips>
			<region>S</region>
		</division>
		<division>
			<name>Phalombe</name>
			<code>PH</code>
			<fips></fips>
			<region>S</region>
		</division>
		<division>
			<name>Mulanje</name>
			<code>MU</code>
			<fips>14</fips>
			<region>S</region>
		</division>
		<division>
			<name>Zomba</name>
			<code>ZO</code>
			<fips>23</fips>
			<region>S</region>
		</division>
		<division>
			<name>Nkhata Bay</name>
			<code>NB</code>
			<fips>17</fips>
			<region>N</region>
		</division>
		<division>
			<name>Chikwawa</name>
			<code>CK</code>
			<fips>02</fips>
			<region>S</region>
		</division>
		<division>
			<name>Chiradzulu</name>
			<code>CR</code>
			<fips>03</fips>
			<region>S</region>
		</division>
		<division>
			<name>Ntcheu</name>
			<code>NU</code>
			<fips>16</fips>
			<region>C</region>
		</division>
		<division>
			<name>Chitipa</name>
			<code>CT</code>
			<fips>04</fips>
			<region>N</region>
		</division>
		<division>
			<name>Mangochi</name>
			<code>MG</code>
			<fips>12</fips>
			<region>S</region>
		</division>
		<division>
			<name>Balaka</name>
			<code>BA</code>
			<fips></fips>
			<region>S</region>
		</division>
		<division>
			<name>Likoma Island</name>
			<code>LK</code>
			<fips></fips>
			<region>N</region>
		</division>
		<division>
			<name>Salima</name>
			<code>SA</code>
			<fips>22</fips>
			<region>C</region>
		</division>
		<division>
			<name>Rumphi</name>
			<code>RU</code>
			<fips>21</fips>
			<region>N</region>
		</division>
		<division>
			<name>Nkhotakota</name>
			<code>NK</code>
			<fips>18</fips>
			<region>C</region>
		</division>
		<division>
			<name>Karonga</name>
			<code>KR</code>
			<fips>08</fips>
			<region>N</region>
		</division>
		<division>
			<name>Nsanje</name>
			<code>NS</code>
			<fips>19</fips>
			<region>S</region>
		</division>
		<division>
			<name>Kasungu</name>
			<code>KS</code>
			<fips>09</fips>
			<region>C</region>
		</division>
		<division>
			<name>Lilongwe</name>
			<code>LI</code>
			<fips>11</fips>
			<region>C</region>
		</division>
		<division>
			<name>Mchinji</name>
			<code>MC</code>
			<fips>13</fips>
			<region>C</region>
		</division>
		<division>
			<name>Thyolo</name>
			<code>TH</code>
			<fips>05</fips>
			<region>S</region>
		</division>
		<division>
			<name>Mwanza</name>
			<code>MW</code>
			<fips>25</fips>
			<region>S</region>
		</division>
		<division>
			<name>Mzimba</name>
			<code>MZ</code>
			<fips>15</fips>
			<region>N</region>
		</division>
		<division>
			<name>Dowa</name>
			<code>DO</code>
			<fips>07</fips>
			<region>C</region>
		</division>
		<division>
			<name>Dedza</name>
			<code>DE</code>
			<fips>06</fips>
			<region>C</region>
		</division>
		<division>
			<name>Machinga</name>
			<code>MH</code>
			<fips>10</fips>
			<region>S</region>
		</division>
		<num>454</num>
	</territory>
	<territory>
		<name>Malaysia</name>
		<alpha2>MY</alpha2>
		<alpha3>MYS</alpha3>
		<division>
			<name>Wilayah Persekutuan Putrajaya</name>
			<code>16</code>
			<fips></fips>
			<region></region>
		</division>
		<num>458</num>
	</territory>
	<territory>
		<name>Maldives</name>
		<alpha2>MV</alpha2>
		<alpha3>MDV</alpha3>
		<division>
			<name>Alif</name>
			<code>02</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Raa</name>
			<code>13</code>
			<fips>44</fips>
			<region></region>
		</division>
		<division>
			<name>Gaaf Alif</name>
			<code>27</code>
			<fips>34</fips>
			<region></region>
		</division>
		<division>
			<name>Haa Dhaalu</name>
			<code>23</code>
			<fips>37</fips>
			<region></region>
		</division>
		<division>
			<name>Seenu</name>
			<code>01</code>
			<fips>01</fips>
			<region></region>
		</division>
		<division>
			<name>Lhaviyani</name>
			<code>03</code>
			<fips>39</fips>
			<region></region>
		</division>
		<division>
			<name>Dhaalu</name>
			<code>17</code>
			<fips>32</fips>
			<region></region>
		</division>
		<division>
			<name>Gaafu Dhaalu</name>
			<code>28</code>
			<fips>35</fips>
			<region></region>
		</division>
		<division>
			<name>Male</name>
			<code>MLE</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Vaavu</name>
			<code>04</code>
			<fips>47</fips>
			<region></region>
		</division>
		<division>
			<name>Faafu</name>
			<code>14</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Gnaviyani</name>
			<code>29</code>
			<fips>42</fips>
			<region></region>
		</division>
		<division>
			<name>Noonu</name>
			<code>25</code>
			<fips>43</fips>
			<region></region>
		</division>
		<division>
			<name>Shaviyani</name>
			<code>24</code>
			<fips>45</fips>
			<region></region>
		</division>
		<division>
			<name>Haa Alif</name>
			<code>07</code>
			<fips>36</fips>
			<region></region>
		</division>
		<division>
			<name>Kaafu</name>
			<code>26</code>
			<fips>38</fips>
			<region></region>
		</division>
		<division>
			<name>Meemu</name>
			<code>12</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Thaa</name>
			<code>08</code>
			<fips>46</fips>
			<region></region>
		</division>
		<division>
			<name>Baa</name>
			<code>20</code>
			<fips>31</fips>
			<region></region>
		</division>
		<division>
			<name>Laamu</name>
			<code>05</code>
			<fips>05</fips>
			<region></region>
		</division>
		<num>462</num>
	</territory>
	<territory>
		<name>Mali</name>
		<alpha2>ML</alpha2>
		<alpha3>MLI</alpha3>
		<division>
			<name>Kidal</name>
			<code>8</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>S&#xe9;gou</name>
			<code>4</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Koulikoro</name>
			<code>2</code>
			<fips>07</fips>
			<region></region>
		</division>
		<division>
			<name>Mexti</name>
			<code>5</code>
			<fips>04</fips>
			<region></region>
		</division>
		<division>
			<name>Bamako</name>
			<code>BKO</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Gao</name>
			<code>7</code>
			<fips>02</fips>
			<region></region>
		</division>
		<division>
			<name>Kayes</name>
			<code>1</code>
			<fips>03</fips>
			<region></region>
		</division>
		<division>
			<name>Sikasso</name>
			<code>3</code>
			<fips>06</fips>
			<region></region>
		</division>
		<division>
			<name>Tombouctou</name>
			<code>6</code>
			<fips>08</fips>
			<region></region>
		</division>
		<num>466</num>
	</territory>
	<territory>
		<name>Malta</name>
		<alpha2>MT</alpha2>
		<alpha3>MLT</alpha3>
		<num>470</num>
	</territory>
	<territory>
		<name>Marshall Islands</name>
		<alpha2>MH</alpha2>
		<alpha3>MHL</alpha3>
		<division>
			<name>Lib</name>
			<code>LIB</code>
			<fips></fips>
			<region>L</region>
		</division>
		<division>
			<name>Ujae</name>
			<code>UJA</code>
			<fips></fips>
			<region>L</region>
		</division>
		<division>
			<name>Majuro</name>
			<code>MAJ</code>
			<fips></fips>
			<region>T</region>
		</division>
		<division>
			<name>Mejit</name>
			<code>MEJ</code>
			<fips></fips>
			<region>T</region>
		</division>
		<division>
			<name>Aur</name>
			<code>AUR</code>
			<fips></fips>
			<region>T</region>
		</division>
		<division>
			<name>Ailuk</name>
			<code>ALK</code>
			<fips></fips>
			<region>T</region>
		</division>
		<division>
			<name>Rongelap</name>
			<code>RON</code>
			<fips></fips>
			<region>L</region>
		</division>
		<division>
			<name>Wotho</name>
			<code>WTH</code>
			<fips></fips>
			<region>L</region>
		</division>
		<division>
			<name>Maloelap</name>
			<code>MAL</code>
			<fips></fips>
			<region>T</region>
		</division>
		<division>
			<name>Eniwetok</name>
			<code>ENI</code>
			<fips></fips>
			<region>L</region>
		</division>
		<division>
			<name>Ujelang</name>
			<code>UJL</code>
			<fips></fips>
			<region>L</region>
		</division>
		<division>
			<name>Namorik</name>
			<code>NMK</code>
			<fips></fips>
			<region>L</region>
		</division>
		<division>
			<name>Namu</name>
			<code>NMU</code>
			<fips></fips>
			<region>L</region>
		</division>
		<division>
			<name>Ailinglapalap</name>
			<code>ALL</code>
			<fips></fips>
			<region>L</region>
		</division>
		<division>
			<name>Utirik</name>
			<code>UTI</code>
			<fips></fips>
			<region>T</region>
		</division>
		<division>
			<name>Ebon</name>
			<code>EBO</code>
			<fips></fips>
			<region>L</region>
		</division>
		<division>
			<name>Jaluit</name>
			<code>JAL</code>
			<fips></fips>
			<region>L</region>
		</division>
		<division>
			<name>Kili</name>
			<code>KIL</code>
			<fips></fips>
			<region>L</region>
		</division>
		<division>
			<name>Kwajalein</name>
			<code>KWA</code>
			<fips></fips>
			<region>L</region>
		</division>
		<division>
			<name>Likiep</name>
			<code>LIK</code>
			<fips></fips>
			<region>T</region>
		</division>
		<division>
			<name>Mili</name>
			<code>MIL</code>
			<fips></fips>
			<region>T</region>
		</division>
		<division>
			<name>Arno</name>
			<code>ARN</code>
			<fips></fips>
			<region>T</region>
		</division>
		<division>
			<name>Lae</name>
			<code>LAE</code>
			<fips></fips>
			<region>L</region>
		</division>
		<division>
			<name>Wotje</name>
			<code>WTJ</code>
			<fips></fips>
			<region>T</region>
		</division>
		<num>584</num>
	</territory>
	<territory>
		<name>Martinique</name>
		<alpha2>MQ</alpha2>
		<alpha3>MTQ</alpha3>
		<num>474</num>
	</territory>
	<territory>
		<name>Mauritania</name>
		<alpha2>MR</alpha2>
		<alpha3>MRT</alpha3>
		<division>
			<name>Dakhlet Nou&#xe2;dhibou</name>
			<code>08</code>
			<fips>08</fips>
			<region></region>
		</division>
		<division>
			<name>Assaba</name>
			<code>03</code>
			<fips>03</fips>
			<region></region>
		</division>
		<division>
			<name>Adrar</name>
			<code>07</code>
			<fips>07</fips>
			<region></region>
		</division>
		<division>
			<name>Gorgol</name>
			<code>04</code>
			<fips>04</fips>
			<region></region>
		</division>
		<division>
			<name>Nouakchott</name>
			<code>NKC</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Hodh el Gharbi</name>
			<code>02</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Brakna</name>
			<code>05</code>
			<fips>05</fips>
			<region></region>
		</division>
		<division>
			<name>Hodh ech Chargui</name>
			<code>01</code>
			<fips>01</fips>
			<region></region>
		</division>
		<division>
			<name>Guidimaka</name>
			<code>10</code>
			<fips>10</fips>
			<region></region>
		</division>
		<division>
			<name>Tagant</name>
			<code>09</code>
			<fips>09</fips>
			<region></region>
		</division>
		<division>
			<name>Inchiri</name>
			<code>12</code>
			<fips>12</fips>
			<region></region>
		</division>
		<division>
			<name>Tiris Zemmour</name>
			<code>11</code>
			<fips>11</fips>
			<region></region>
		</division>
		<division>
			<name>Trarza</name>
			<code>06</code>
			<fips>06</fips>
			<region></region>
		</division>
		<num>478</num>
	</territory>
	<territory>
		<name>Mauritius</name>
		<alpha2>MU</alpha2>
		<alpha3>MUS</alpha3>
		<division>
			<name>Plaines Wilhems</name>
			<code>PW</code>
			<fips>17</fips>
			<region></region>
		</division>
		<division>
			<name>Beau Bassin-Rose Hill</name>
			<code>BR</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Port Louis</name>
			<code>PL</code>
			<fips>18</fips>
			<region></region>
		</division>
		<division>
			<name>Rodrigues Island</name>
			<code>RO</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Savanne</name>
			<code>SA</code>
			<fips>20</fips>
			<region></region>
		</division>
		<division>
			<name>Black River</name>
			<code>BL</code>
			<fips>12</fips>
			<region></region>
		</division>
		<division>
			<name>Rivi&#xe8;re du Rempart</name>
			<code>RR</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Agalega Islands</name>
			<code>AG</code>
			<fips>21</fips>
			<region></region>
		</division>
		<division>
			<name>Cargados Carajos</name>
			<code>CC</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Vacoas-Phoenix</name>
			<code>VP</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Quatre Bornes</name>
			<code>QB</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Moka</name>
			<code>MO</code>
			<fips>15</fips>
			<region></region>
		</division>
		<division>
			<name>Grand Port</name>
			<code>GP</code>
			<fips>14</fips>
			<region></region>
		</division>
		<division>
			<name>Pamplemousses</name>
			<code>PA</code>
			<fips>16</fips>
			<region></region>
		</division>
		<division>
			<name>Curepipe</name>
			<code>CU</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Flacq</name>
			<code>FL</code>
			<fips>13</fips>
			<region></region>
		</division>
		<num>480</num>
	</territory>
	<territory>
		<name>Mayotte</name>
		<alpha2>YT</alpha2>
		<alpha3>MYT</alpha3>
		<num>175</num>
	</territory>
	<territory>
		<name>Mexico</name>
		<alpha2>MX</alpha2>
		<alpha3>MEX</alpha3>
		<division>
			<name>Baja California Sur</name>
			<code>BCS</code>
			<fips>03</fips>
			<region></region>
		</division>
		<division>
			<name>Campeche</name>
			<code>CAM</code>
			<fips>04</fips>
			<region></region>
		</division>
		<division>
			<name>Nuevo Le&#xf3;n</name>
			<code>NLE</code>
			<fips>19</fips>
			<region></region>
		</division>
		<division>
			<name>Aguascalientes</name>
			<code>AGU</code>
			<fips>01</fips>
			<region></region>
		</division>
		<division>
			<name>Tamaulipas</name>
			<code>TAM</code>
			<fips>28</fips>
			<region></region>
		</division>
		<division>
			<name>Oaxaca</name>
			<code>OAX</code>
			<fips>20</fips>
			<region></region>
		</division>
		<division>
			<name>Morelos</name>
			<code>MOR</code>
			<fips>17</fips>
			<region></region>
		</division>
		<division>
			<name>Puebla</name>
			<code>PUE</code>
			<fips>21</fips>
			<region></region>
		</division>
		<division>
			<name>Distrito Federal</name>
			<code>DIF</code>
			<fips>09</fips>
			<region></region>
		</division>
		<division>
			<name>Sinaloa</name>
			<code>SIN</code>
			<fips>25</fips>
			<region></region>
		</division>
		<division>
			<name>Guerrero</name>
			<code>GRO</code>
			<fips>12</fips>
			<region></region>
		</division>
		<division>
			<name>Guanajuato</name>
			<code>GUA</code>
			<fips>11</fips>
			<region></region>
		</division>
		<division>
			<name>Quer&#xe9;taro</name>
			<code>QUE</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Tabasco</name>
			<code>TAB</code>
			<fips>27</fips>
			<region></region>
		</division>
		<division>
			<name>Veracruz</name>
			<code>VER</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Zacatecas</name>
			<code>ZAC</code>
			<fips>32</fips>
			<region></region>
		</division>
		<division>
			<name>Sonora</name>
			<code>SON</code>
			<fips>26</fips>
			<region></region>
		</division>
		<division>
			<name>Colima</name>
			<code>COL</code>
			<fips>08</fips>
			<region></region>
		</division>
		<division>
			<name>Chiapas</name>
			<code>CHP</code>
			<fips>05</fips>
			<region></region>
		</division>
		<division>
			<name>Tlaxcala</name>
			<code>TLA</code>
			<fips>29</fips>
			<region></region>
		</division>
		<division>
			<name>Baja California</name>
			<code>BCN</code>
			<fips>02</fips>
			<region></region>
		</division>
		<division>
			<name>Durango</name>
			<code>DUR</code>
			<fips>10</fips>
			<region></region>
		</division>
		<division>
			<name>Hidalgo</name>
			<code>HID</code>
			<fips>13</fips>
			<region></region>
		</division>
		<division>
			<name>Jalisco</name>
			<code>JAL</code>
			<fips>14</fips>
			<region></region>
		</division>
		<division>
			<name>San Luis Potos&#xed;</name>
			<code>SLP</code>
			<fips>24</fips>
			<region></region>
		</division>
		<division>
			<name>Chihuahua</name>
			<code>CHH</code>
			<fips>06</fips>
			<region></region>
		</division>
		<division>
			<name>Nayarit</name>
			<code>NAY</code>
			<fips>18</fips>
			<region></region>
		</division>
		<division>
			<name>Quintana Roo</name>
			<code>ROO</code>
			<fips>23</fips>
			<region></region>
		</division>
		<division>
			<name>Yucat&#xe1;n</name>
			<code>YUC</code>
			<fips>31</fips>
			<region></region>
		</division>
		<division>
			<name>Coahuila</name>
			<code>COA</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>M&#xe9;xico</name>
			<code>MEX</code>
			<fips>15</fips>
			<region></region>
		</division>
		<division>
			<name>Michoac&#xe1;n</name>
			<code>MIC</code>
			<fips></fips>
			<region></region>
		</division>
		<num>484</num>
	</territory>
	<territory>
		<name>Federated States of Micronesia</name>
		<alpha2>FM</alpha2>
		<alpha3>FSM</alpha3>
		<division>
			<name>Chuuk</name>
			<code>TRK</code>
			<fips>03</fips>
			<region></region>
		</division>
		<division>
			<name>Pohnpei</name>
			<code>PNI</code>
			<fips>02</fips>
			<region></region>
		</division>
		<division>
			<name>Kosrae</name>
			<code>KSA</code>
			<fips>01</fips>
			<region></region>
		</division>
		<division>
			<name>Yap</name>
			<code>YAP</code>
			<fips>04</fips>
			<region></region>
		</division>
		<num>583</num>
	</territory>
	<territory>
		<name>Moldova</name>
		<alpha2>MD</alpha2>
		<alpha3>MDA</alpha3>
		<division>
			<name>Cahul</name>
			<code>CA</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Edinet</name>
			<code>ED</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Gagauzia, Unitate Teritoriala Autonoma (UTAG)</name>
			<code>GA</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Ungheni</name>
			<code>UN</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Lapusna</name>
			<code>LA</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>St&#xee;nga Nistrului, unitatea teritoriala din</name>
			<code>SN</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Chisinau</name>
			<code>CH</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Orhei</name>
			<code>OR</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Balti</name>
			<code>BA</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Tighina [Bender]</name>
			<code>TI</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Soroca</name>
			<code>SO</code>
			<fips></fips>
			<region></region>
		</division>
		<num>498</num>
	</territory>
	<territory>
		<name>Monaco</name>
		<alpha2>MC</alpha2>
		<alpha3>MCO</alpha3>
		<num>492</num>
	</territory>
	<territory>
		<name>Mongolia</name>
		<alpha2>MN</alpha2>
		<alpha3>MNG</alpha3>
		<division>
			<name>&#xd6;mn&#xf6;govi</name>
			<code>053</code>
			<fips>14</fips>
			<region></region>
		</division>
		<division>
			<name>Dornogovi</name>
			<code>063</code>
			<fips>07</fips>
			<region></region>
		</division>
		<division>
			<name>Dundgovi</name>
			<code>059</code>
			<fips>08</fips>
			<region></region>
		</division>
		<division>
			<name>Bulgan</name>
			<code>067</code>
			<fips>21</fips>
			<region></region>
		</division>
		<division>
			<name>Bayanhongor</name>
			<code>069</code>
			<fips>02</fips>
			<region></region>
		</division>
		<division>
			<name>Hovd</name>
			<code>043</code>
			<fips>12</fips>
			<region></region>
		</division>
		<division>
			<name>Hentiy</name>
			<code>039</code>
			<fips>11</fips>
			<region></region>
		</division>
		<division>
			<name>Darhan uul</name>
			<code>037</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>H&#xf6;vsg&#xf6;l</name>
			<code>041</code>
			<fips>13</fips>
			<region></region>
		</division>
		<division>
			<name>S&#xfc;hbaatar</name>
			<code>051</code>
			<fips>17</fips>
			<region></region>
		</division>
		<division>
			<name>Selenge</name>
			<code>049</code>
			<fips>16</fips>
			<region></region>
		</division>
		<division>
			<name>Orhon</name>
			<code>035</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>T&#xf6;v</name>
			<code>047</code>
			<fips>18</fips>
			<region></region>
		</division>
		<division>
			<name>Ulaanbaatar</name>
			<code>1</code>
			<fips>20</fips>
			<region></region>
		</division>
		<division>
			<name>Dornod</name>
			<code>061</code>
			<fips>06</fips>
			<region></region>
		</division>
		<division>
			<name>&#xd6;v&#xf6;rhangay</name>
			<code>055</code>
			<fips>15</fips>
			<region></region>
		</division>
		<division>
			<name>Dzavhan</name>
			<code>057</code>
			<fips>09</fips>
			<region></region>
		</division>
		<division>
			<name>Arhangay</name>
			<code>073</code>
			<fips>01</fips>
			<region></region>
		</division>
		<division>
			<name>Uvs</name>
			<code>046</code>
			<fips>19</fips>
			<region></region>
		</division>
		<division>
			<name>Govi-Altay</name>
			<code>065</code>
			<fips>10</fips>
			<region></region>
		</division>
		<division>
			<name>Govi-S&#xfc;mber</name>
			<code>064</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Bayan-&#xd6;lgiy</name>
			<code>071</code>
			<fips>03</fips>
			<region></region>
		</division>
		<num>496</num>
	</territory>
	<territory>
		<name>Montenegro</name>
		<alpha2>ME</alpha2>
		<alpha3>MNE</alpha3>
		<num>499</num>
	</territory>
	<territory>
		<name>Montserrat</name>
		<alpha2>MS</alpha2>
		<alpha3>MSR</alpha3>
		<num>500</num>
	</territory>
	<territory>
		<name>Morocco</name>
		<alpha2>MA</alpha2>
		<alpha3>MAR</alpha3>
		<division>
			<name>Guelmim</name>
			<code>GUE</code>
			<fips>42</fips>
			<region>14</region>
		</division>
		<division>
			<name>T&#xe9;touan</name>
			<code>TET</code>
			<fips>40</fips>
			<region>01</region>
		</division>
		<division>
			<name>K&#xe9;nitra</name>
			<code>KEN</code>
			<fips>15</fips>
			<region>02</region>
		</division>
		<division>
			<name>Ifrane</name>
			<code>IFR</code>
			<fips>34</fips>
			<region>06</region>
		</division>
		<division>
			<name>Larache</name>
			<code>LAR</code>
			<fips></fips>
			<region>01</region>
		</division>
		<division>
			<name>Taounate</name>
			<code>TAO</code>
			<fips>37</fips>
			<region>03</region>
		</division>
		<division>
			<name>Tan-Tan</name>
			<code>TNT</code>
			<fips>36</fips>
			<region>14</region>
		</division>
		<division>
			<name>Tiznit</name>
			<code>TIZ</code>
			<fips>32</fips>
			<region>13</region>
		</division>
		<division>
			<name>Sidi Kacem</name>
			<code>SIK</code>
			<fips>38</fips>
			<region>02</region>
		</division>
		<division>
			<name>Khemisset</name>
			<code>KHE</code>
			<fips>16</fips>
			<region>07</region>
		</division>
		<division>
			<name>Taroudannt</name>
			<code>TAR</code>
			<fips>39</fips>
			<region>13</region>
		</division>
		<division>
			<name>Jerada</name>
			<code>JRA</code>
			<fips></fips>
			<region>04</region>
		</division>
		<division>
			<name>Khenifra</name>
			<code>KHN</code>
			<fips>17</fips>
			<region>06</region>
		</division>
		<division>
			<name>Oujda</name>
			<code>OUJ</code>
			<fips>23</fips>
			<region>04</region>
		</division>
		<division>
			<name>Settat</name>
			<code>SET</code>
			<fips>26</fips>
			<region>09</region>
		</division>
		<division>
			<name>Marrakech</name>
			<code>MAR</code>
			<fips>19</fips>
			<region>11</region>
		</division>
		<division>
			<name>Sefrou</name>
			<code>SEF</code>
			<fips></fips>
			<region>05</region>
		</division>
		<division>
			<name>Nador</name>
			<code>NAD</code>
			<fips>21</fips>
			<region>04</region>
		</division>
		<division>
			<name>Safi</name>
			<code>SAF</code>
			<fips>25</fips>
			<region>10</region>
		</division>
		<division>
			<name>Kelaat Sraghna</name>
			<code>KES</code>
			<fips></fips>
			<region>11</region>
		</division>
		<division>
			<name>Khouribga</name>
			<code>KHO</code>
			<fips>18</fips>
			<region>09</region>
		</division>
		<division>
			<name>Rabat-Sal&#xe9;</name>
			<code>RBA</code>
			<fips>24</fips>
			<region>07</region>
		</division>
		<division>
			<name>Laayoune (EH)</name>
			<code>LAA</code>
			<fips>35</fips>
			<region>15</region>
		</division>
		<division>
			<name>Tata</name>
			<code>TAT</code>
			<fips>29</fips>
			<region>14</region>
		</division>
		<division>
			<name>Ouarzazate</name>
			<code>OUA</code>
			<fips>22</fips>
			<region>13</region>
		</division>
		<division>
			<name>Oued ed Dahab (EH)</name>
			<code>OUD</code>
			<fips></fips>
			<region>16</region>
		</division>
		<division>
			<name>Tanger</name>
			<code>TNG</code>
			<fips>27</fips>
			<region>01</region>
		</division>
		<division>
			<name>Taza</name>
			<code>TAZ</code>
			<fips>30</fips>
			<region>03</region>
		</division>
		<division>
			<name>Figuig</name>
			<code>FIG</code>
			<fips>14</fips>
			<region>04</region>
		</division>
		<division>
			<name>Mekn&#xe8;s</name>
			<code>MEK</code>
			<fips>20</fips>
			<region>06</region>
		</division>
		<num>504</num>
	</territory>
	<territory>
		<name>Mozambique</name>
		<alpha2>MZ</alpha2>
		<alpha3>MOZ</alpha3>
		<division>
			<name>Tete</name>
			<code>T</code>
			<fips>08</fips>
			<region></region>
		</division>
		<division>
			<name>Sofala</name>
			<code>S</code>
			<fips>05</fips>
			<region></region>
		</division>
		<division>
			<name>Nampula</name>
			<code>N</code>
			<fips>06</fips>
			<region></region>
		</division>
		<division>
			<name>Manica</name>
			<code>B</code>
			<fips>10</fips>
			<region></region>
		</division>
		<division>
			<name>Gaza</name>
			<code>G</code>
			<fips>02</fips>
			<region></region>
		</division>
		<division>
			<name>Inhambane</name>
			<code>I</code>
			<fips>03</fips>
			<region></region>
		</division>
		<division>
			<name>Niassa</name>
			<code>A</code>
			<fips>07</fips>
			<region></region>
		</division>
		<division>
			<name>Maputo</name>
			<code>L</code>
			<fips>04</fips>
			<region></region>
		</division>
		<division>
			<name>Zamb&#xe9;zia</name>
			<code>Q</code>
			<fips>09</fips>
			<region></region>
		</division>
		<division>
			<name>Cabo Delgado</name>
			<code>P</code>
			<fips>01</fips>
			<region></region>
		</division>
		<num>508</num>
	</territory>
	<territory>
		<name>Myanmar</name>
		<alpha2>MM</alpha2>
		<alpha3>MMR</alpha3>
		<division>
			<name>Mon</name>
			<code>15</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Kachin</name>
			<code>11</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Sagaing</name>
			<code>01</code>
			<fips>10</fips>
			<region></region>
		</division>
		<division>
			<name>Shan</name>
			<code>17</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Rakhine</name>
			<code>16</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Bago</name>
			<code>02</code>
			<fips>16</fips>
			<region></region>
		</division>
		<division>
			<name>Kayin</name>
			<code>13</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Yangon</name>
			<code>06</code>
			<fips>17</fips>
			<region></region>
		</division>
		<division>
			<name>Chin</name>
			<code>14</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Mandalay</name>
			<code>04</code>
			<fips>08</fips>
			<region></region>
		</division>
		<division>
			<name>Tanintharyi</name>
			<code>05</code>
			<fips>12</fips>
			<region></region>
		</division>
		<division>
			<name>Kayah</name>
			<code>12</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Ayeyarwady</name>
			<code>07</code>
			<fips>03</fips>
			<region></region>
		</division>
		<division>
			<name>Magway</name>
			<code>03</code>
			<fips>15</fips>
			<region></region>
		</division>
		<num>104</num>
	</territory>
	<territory>
		<name>Namibia</name>
		<alpha2>NA</alpha2>
		<alpha3>NAM</alpha3>
		<division>
			<name>Karas</name>
			<code>KA</code>
			<fips>31</fips>
			<region></region>
		</division>
		<division>
			<name>Otjozondjupa</name>
			<code>OD</code>
			<fips>39</fips>
			<region></region>
		</division>
		<division>
			<name>Erongo</name>
			<code>ER</code>
			<fips>29</fips>
			<region></region>
		</division>
		<division>
			<name>Omaheke</name>
			<code>OH</code>
			<fips>35</fips>
			<region></region>
		</division>
		<division>
			<name>Ohangwena</name>
			<code>OW</code>
			<fips>33</fips>
			<region></region>
		</division>
		<division>
			<name>Khomas</name>
			<code>KH</code>
			<fips>21</fips>
			<region></region>
		</division>
		<division>
			<name>Hardap</name>
			<code>HA</code>
			<fips>30</fips>
			<region></region>
		</division>
		<division>
			<name>Oshikoto</name>
			<code>OT</code>
			<fips>38</fips>
			<region></region>
		</division>
		<division>
			<name>Oshana</name>
			<code>ON</code>
			<fips>37</fips>
			<region></region>
		</division>
		<division>
			<name>Omusati</name>
			<code>OS</code>
			<fips>36</fips>
			<region></region>
		</division>
		<division>
			<name>Kunene</name>
			<code>KU</code>
			<fips>32</fips>
			<region></region>
		</division>
		<division>
			<name>Caprivi</name>
			<code>CA</code>
			<fips>28</fips>
			<region></region>
		</division>
		<division>
			<name>Okavango</name>
			<code>OK</code>
			<fips>34</fips>
			<region></region>
		</division>
		<num>516</num>
	</territory>
	<territory>
		<name>Nauru</name>
		<alpha2>NR</alpha2>
		<alpha3>NRU</alpha3>
		<num>520</num>
	</territory>
	<territory>
		<name>Nepal</name>
		<alpha2>NP</alpha2>
		<alpha3>NPL</alpha3>
		<division>
			<name>Mahakali</name>
			<code>MA</code>
			<fips></fips>
			<region>5</region>
		</division>
		<division>
			<name>Karnali</name>
			<code>KA</code>
			<fips></fips>
			<region>2</region>
		</division>
		<division>
			<name>Kosi [Koshi]</name>
			<code>KO</code>
			<fips></fips>
			<region>4</region>
		</division>
		<division>
			<name>Dhawalagiri</name>
			<code>DH</code>
			<fips></fips>
			<region>3</region>
		</division>
		<division>
			<name>Bheri</name>
			<code>BH</code>
			<fips></fips>
			<region>2</region>
		</division>
		<division>
			<name>Bagmati</name>
			<code>BA</code>
			<fips></fips>
			<region>1</region>
		</division>
		<division>
			<name>Sagarmatha</name>
			<code>SA</code>
			<fips></fips>
			<region>4</region>
		</division>
		<division>
			<name>Rapti</name>
			<code>RA</code>
			<fips></fips>
			<region>2</region>
		</division>
		<division>
			<name>Narayani</name>
			<code>NA</code>
			<fips></fips>
			<region>1</region>
		</division>
		<division>
			<name>Mechi</name>
			<code>ME</code>
			<fips></fips>
			<region>4</region>
		</division>
		<division>
			<name>Gandaki</name>
			<code>GA</code>
			<fips></fips>
			<region>3</region>
		</division>
		<division>
			<name>Lumbini</name>
			<code>LU</code>
			<fips></fips>
			<region>3</region>
		</division>
		<division>
			<name>Seti</name>
			<code>SE</code>
			<fips></fips>
			<region>5</region>
		</division>
		<division>
			<name>Janakpur</name>
			<code>JA</code>
			<fips></fips>
			<region>1</region>
		</division>
		<num>524</num>
	</territory>
	<territory>
		<name>Netherlands</name>
		<alpha2>NL</alpha2>
		<alpha3>NLD</alpha3>
		<division>
			<name>Limburg</name>
			<code>LI</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Groningen</name>
			<code>GR</code>
			<fips>04</fips>
			<region></region>
		</division>
		<division>
			<name>Zeeland</name>
			<code>ZE</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Noord-Brabant</name>
			<code>NB</code>
			<fips>06</fips>
			<region></region>
		</division>
		<division>
			<name>Gelderland</name>
			<code>GE</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Utrecht</name>
			<code>UT</code>
			<fips>09</fips>
			<region></region>
		</division>
		<division>
			<name>Zuid-Holland</name>
			<code>ZH</code>
			<fips>11</fips>
			<region></region>
		</division>
		<division>
			<name>Drenthe</name>
			<code>DR</code>
			<fips>01</fips>
			<region></region>
		</division>
		<division>
			<name>Noord-Holland</name>
			<code>NH</code>
			<fips>07</fips>
			<region></region>
		</division>
		<division>
			<name>Friesland</name>
			<code>FR</code>
			<fips>02</fips>
			<region></region>
		</division>
		<division>
			<name>Overijssel</name>
			<code>OV</code>
			<fips>15</fips>
			<region></region>
		</division>
		<division>
			<name>Flevoland</name>
			<code>FL</code>
			<fips>16</fips>
			<region></region>
		</division>
		<num>528</num>
	</territory>
	<territory>
		<name>Netherlands Antilles</name>
		<alpha2>AN</alpha2>
		<alpha3>ANT</alpha3>
		<num>530</num>
	</territory>
	<territory>
		<name>New Zealand</name>
		<alpha2>NZ</alpha2>
		<alpha3>NZL</alpha3>
		<division>
			<name>Marlborough</name>
			<code>MBH</code>
			<fips>50</fips>
			<region>S</region>
		</division>
		<division>
			<name>West Coast</name>
			<code>WTC</code>
			<fips></fips>
			<region>S</region>
		</division>
		<division>
			<name>Otago</name>
			<code>OTA</code>
			<fips></fips>
			<region>S</region>
		</division>
		<division>
			<name>Nelson</name>
			<code>NSN</code>
			<fips></fips>
			<region>S</region>
		</division>
		<division>
			<name>Wellington</name>
			<code>WGN</code>
			<fips></fips>
			<region>N</region>
		</division>
		<division>
			<name>Bay of Plenty</name>
			<code>BOP</code>
			<fips></fips>
			<region>N</region>
		</division>
		<division>
			<name>Northland</name>
			<code>NTL</code>
			<fips></fips>
			<region>N</region>
		</division>
		<division>
			<name>Southland</name>
			<code>STL</code>
			<fips>72</fips>
			<region>S</region>
		</division>
		<division>
			<name>Auckland</name>
			<code>AUK</code>
			<fips>17</fips>
			<region>N</region>
		</division>
		<division>
			<name>Manawatu-Wanganui</name>
			<code>MWT</code>
			<fips></fips>
			<region>N</region>
		</division>
		<division>
			<name>Taranaki</name>
			<code>TKI</code>
			<fips>76</fips>
			<region>N</region>
		</division>
		<division>
			<name>Canterbury</name>
			<code>CAN</code>
			<fips></fips>
			<region>S</region>
		</division>
		<division>
			<name>Tasman</name>
			<code>TAS</code>
			<fips></fips>
			<region>S</region>
		</division>
		<division>
			<name>Hawkes's Bay</name>
			<code>HKB</code>
			<fips>31</fips>
			<region>N</region>
		</division>
		<division>
			<name>Gisborne</name>
			<code>GIS</code>
			<fips></fips>
			<region>N</region>
		</division>
		<division>
			<name>Waikato</name>
			<code>WKO</code>
			<fips>85</fips>
			<region>N</region>
		</division>
		<num>554</num>
	</territory>
	<territory>
		<name>Nicaragua</name>
		<alpha2>NI</alpha2>
		<alpha3>NIC</alpha3>
		<division>
			<name>R&#xed;o San Juan</name>
			<code>SJ</code>
			<fips>14</fips>
			<region></region>
		</division>
		<division>
			<name>Chontales</name>
			<code>CO</code>
			<fips>04</fips>
			<region></region>
		</division>
		<division>
			<name>Boaco</name>
			<code>BO</code>
			<fips>01</fips>
			<region></region>
		</division>
		<division>
			<name>Matagalpa</name>
			<code>MT</code>
			<fips>12</fips>
			<region></region>
		</division>
		<division>
			<name>Jinotega</name>
			<code>JI</code>
			<fips>07</fips>
			<region></region>
		</division>
		<division>
			<name>Madriz</name>
			<code>MD</code>
			<fips>09</fips>
			<region></region>
		</division>
		<division>
			<name>Granada</name>
			<code>GR</code>
			<fips>06</fips>
			<region></region>
		</division>
		<division>
			<name>Carazo</name>
			<code>CA</code>
			<fips>02</fips>
			<region></region>
		</division>
		<division>
			<name>Le&#xf3;n</name>
			<code>LE</code>
			<fips>08</fips>
			<region></region>
		</division>
		<division>
			<name>Nueva Segovia</name>
			<code>NS</code>
			<fips>13</fips>
			<region></region>
		</division>
		<division>
			<name>Atl&#xe1;ntico Sur</name>
			<code>AS</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Rivas</name>
			<code>RI</code>
			<fips>15</fips>
			<region></region>
		</division>
		<division>
			<name>Chinandega</name>
			<code>CI</code>
			<fips>03</fips>
			<region></region>
		</division>
		<division>
			<name>Estel&#xed;</name>
			<code>ES</code>
			<fips>05</fips>
			<region></region>
		</division>
		<division>
			<name>Masaya</name>
			<code>MS</code>
			<fips>11</fips>
			<region></region>
		</division>
		<division>
			<name>Atl&#xe1;ntico Norte</name>
			<code>AN</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Managua</name>
			<code>MN</code>
			<fips>10</fips>
			<region></region>
		</division>
		<num>558</num>
	</territory>
	<territory>
		<name>Niger</name>
		<alpha2>NE</alpha2>
		<alpha3>NER</alpha3>
		<division>
			<name>Zinder</name>
			<code>7</code>
			<fips>07</fips>
			<region></region>
		</division>
		<division>
			<name>Tillab&#xe9;ri</name>
			<code>6</code>
			<fips>09</fips>
			<region></region>
		</division>
		<division>
			<name>Agadez</name>
			<code>1</code>
			<fips>01</fips>
			<region></region>
		</division>
		<division>
			<name>Tahoua</name>
			<code>5</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Maradi</name>
			<code>4</code>
			<fips>04</fips>
			<region></region>
		</division>
		<division>
			<name>Diffa</name>
			<code>2</code>
			<fips>02</fips>
			<region></region>
		</division>
		<division>
			<name>Niamey</name>
			<code>8</code>
			<fips>08</fips>
			<region></region>
		</division>
		<division>
			<name>Dosso</name>
			<code>3</code>
			<fips>03</fips>
			<region></region>
		</division>
		<num>562</num>
	</territory>
	<territory>
		<name>Nigeria</name>
		<alpha2>NG</alpha2>
		<alpha3>NGA</alpha3>
		<division>
			<name>Katsina</name>
			<code>KT</code>
			<fips>24</fips>
			<region></region>
		</division>
		<division>
			<name>Yobe</name>
			<code>YO</code>
			<fips>44</fips>
			<region></region>
		</division>
		<division>
			<name>Adamawa</name>
			<code>AD</code>
			<fips>35</fips>
			<region></region>
		</division>
		<division>
			<name>Lagos</name>
			<code>LA</code>
			<fips>05</fips>
			<region></region>
		</division>
		<division>
			<name>Kano</name>
			<code>KN</code>
			<fips>29</fips>
			<region></region>
		</division>
		<division>
			<name>Bauchi</name>
			<code>BA</code>
			<fips>46</fips>
			<region></region>
		</division>
		<division>
			<name>Plateau</name>
			<code>PL</code>
			<fips>49</fips>
			<region></region>
		</division>
		<division>
			<name>Ogun</name>
			<code>OG</code>
			<fips>16</fips>
			<region></region>
		</division>
		<division>
			<name>Osun</name>
			<code>OS</code>
			<fips>42</fips>
			<region></region>
		</division>
		<division>
			<name>Niger</name>
			<code>NI</code>
			<fips>31</fips>
			<region></region>
		</division>
		<division>
			<name>Ekiti</name>
			<code>EK</code>
			<fips>54</fips>
			<region></region>
		</division>
		<division>
			<name>Edo</name>
			<code>ED</code>
			<fips>37</fips>
			<region></region>
		</division>
		<division>
			<name>Benue</name>
			<code>BE</code>
			<fips>26</fips>
			<region></region>
		</division>
		<division>
			<name>Kwara</name>
			<code>KW</code>
			<fips>30</fips>
			<region></region>
		</division>
		<division>
			<name>Kogi</name>
			<code>KO</code>
			<fips>41</fips>
			<region></region>
		</division>
		<division>
			<name>Oyo</name>
			<code>OY</code>
			<fips>32</fips>
			<region></region>
		</division>
		<division>
			<name>Ondo</name>
			<code>ON</code>
			<fips>48</fips>
			<region></region>
		</division>
		<division>
			<name>Anambra</name>
			<code>AN</code>
			<fips>25</fips>
			<region></region>
		</division>
		<division>
			<name>Jigawa</name>
			<code>JI</code>
			<fips>39</fips>
			<region></region>
		</division>
		<division>
			<name>Abia</name>
			<code>AB</code>
			<fips>45</fips>
			<region></region>
		</division>
		<division>
			<name>Borno</name>
			<code>BO</code>
			<fips>27</fips>
			<region></region>
		</division>
		<division>
			<name>Taraba</name>
			<code>TA</code>
			<fips>43</fips>
			<region></region>
		</division>
		<division>
			<name>Imo</name>
			<code>IM</code>
			<fips>28</fips>
			<region></region>
		</division>
		<division>
			<name>Akwa Ibom</name>
			<code>AK</code>
			<fips>21</fips>
			<region></region>
		</division>
		<division>
			<name>Zamfara</name>
			<code>ZA</code>
			<fips>57</fips>
			<region></region>
		</division>
		<division>
			<name>Kebbi</name>
			<code>KE</code>
			<fips>40</fips>
			<region></region>
		</division>
		<division>
			<name>Sokoto</name>
			<code>SO</code>
			<fips>51</fips>
			<region></region>
		</division>
		<division>
			<name>Gombe</name>
			<code>GO</code>
			<fips>55</fips>
			<region></region>
		</division>
		<division>
			<name>Bayelsa</name>
			<code>BY</code>
			<fips>52</fips>
			<region></region>
		</division>
		<division>
			<name>Kaduna</name>
			<code>KD</code>
			<fips>23</fips>
			<region></region>
		</division>
		<division>
			<name>Ebonyi</name>
			<code>EB</code>
			<fips>53</fips>
			<region></region>
		</division>
		<division>
			<name>Enugu</name>
			<code>EN</code>
			<fips>47</fips>
			<region></region>
		</division>
		<division>
			<name>Abuja Capital Territory</name>
			<code>FC</code>
			<fips>11</fips>
			<region></region>
		</division>
		<division>
			<name>Rivers</name>
			<code>RI</code>
			<fips>50</fips>
			<region></region>
		</division>
		<division>
			<name>Cross River</name>
			<code>CR</code>
			<fips>22</fips>
			<region></region>
		</division>
		<division>
			<name>Delta</name>
			<code>DE</code>
			<fips>36</fips>
			<region></region>
		</division>
		<division>
			<name>Nassarawa</name>
			<code>NA</code>
			<fips>56</fips>
			<region></region>
		</division>
		<num>566</num>
	</territory>
	<territory>
		<name>Niue</name>
		<alpha2>NU</alpha2>
		<alpha3>NIU</alpha3>
		<num>570</num>
	</territory>
	<territory>
		<name>Norfolk Island</name>
		<alpha2>NF</alpha2>
		<alpha3>NFK</alpha3>
		<num>574</num>
	</territory>
	<territory>
		<name>Northern Mariana Islands</name>
		<alpha2>MP</alpha2>
		<alpha3>MNP</alpha3>
		<num>580</num>
	</territory>
	<territory>
		<name>Norway</name>
		<alpha2>NO</alpha2>
		<alpha3>NOR</alpha3>
		<division>
			<name>Hordaland</name>
			<code>12</code>
			<fips>07</fips>
			<region></region>
		</division>
		<division>
			<name>Troms</name>
			<code>19</code>
			<fips>18</fips>
			<region></region>
		</division>
		<division>
			<name>Akershus</name>
			<code>02</code>
			<fips>01</fips>
			<region></region>
		</division>
		<division>
			<name>Vest-Agder</name>
			<code>10</code>
			<fips>19</fips>
			<region></region>
		</division>
		<division>
			<name>S&#xf8;r-Tr&#xf8;ndelag</name>
			<code>16</code>
			<fips>16</fips>
			<region></region>
		</division>
		<division>
			<name>Telemark</name>
			<code>08</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Aust-Agder</name>
			<code>09</code>
			<fips>02</fips>
			<region></region>
		</division>
		<division>
			<name>Hedmark</name>
			<code>04</code>
			<fips>06</fips>
			<region></region>
		</division>
		<division>
			<name>Finnmark</name>
			<code>20</code>
			<fips>05</fips>
			<region></region>
		</division>
		<division>
			<name>Nordland</name>
			<code>18</code>
			<fips>09</fips>
			<region></region>
		</division>
		<division>
			<name>Buskerud</name>
			<code>06</code>
			<fips>17</fips>
			<region></region>
		</division>
		<division>
			<name>Nord-Tr&#xf8;ndelag</name>
			<code>17</code>
			<fips>10</fips>
			<region></region>
		</division>
		<division>
			<name>Oppland</name>
			<code>05</code>
			<fips>11</fips>
			<region></region>
		</division>
		<division>
			<name>Svalbard (Arctic Region) (See also country code SJ)</name>
			<code>21</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>M&#xf8;re og Romsdal</name>
			<code>15</code>
			<fips>08</fips>
			<region></region>
		</division>
		<division>
			<name>Oslo</name>
			<code>03</code>
			<fips>12</fips>
			<region></region>
		</division>
		<division>
			<name>&#xd8;stfold</name>
			<code>01</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Rogaland</name>
			<code>11</code>
			<fips>14</fips>
			<region></region>
		</division>
		<division>
			<name>Jan Mayen (Arctic Region) (See also country code SJ)</name>
			<code>22</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Vestfold</name>
			<code>07</code>
			<fips>20</fips>
			<region></region>
		</division>
		<division>
			<name>Sogn og Fjordane</name>
			<code>14</code>
			<fips>15</fips>
			<region></region>
		</division>
		<num>578</num>
	</territory>
	<territory>
		<name>Oman</name>
		<alpha2>OM</alpha2>
		<alpha3>OMN</alpha3>
		<division>
			<name>Al Batinah</name>
			<code>BA</code>
			<fips>02</fips>
			<region></region>
		</division>
		<division>
			<name>Az Zahirah</name>
			<code>ZA</code>
			<fips>05</fips>
			<region></region>
		</division>
		<division>
			<name>Ad Dakhiliyah</name>
			<code>DA</code>
			<fips>01</fips>
			<region></region>
		</division>
		<division>
			<name>Al Wust&#xe1;</name>
			<code>WU</code>
			<fips>03</fips>
			<region></region>
		</division>
		<division>
			<name>Al Janubiyah [Zufar]</name>
			<code>JA</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Masqat</name>
			<code>MA</code>
			<fips>06</fips>
			<region></region>
		</division>
		<division>
			<name>Ash Sharqiyah</name>
			<code>SH</code>
			<fips>04</fips>
			<region></region>
		</division>
		<division>
			<name>Musandam</name>
			<code>MU</code>
			<fips>07</fips>
			<region></region>
		</division>
		<num>512</num>
	</territory>
	<territory>
		<name>Pakistan</name>
		<alpha2>PK</alpha2>
		<alpha3>PAK</alpha3>
		<division>
			<name>Northern Areas</name>
			<code>NA</code>
			<fips>07</fips>
			<region></region>
		</division>
		<division>
			<name>Azad Kashmir</name>
			<code>JK</code>
			<fips>06</fips>
			<region></region>
		</division>
		<division>
			<name>Federally Administered Tribal Areas</name>
			<code>TA</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Sind</name>
			<code>SD</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Punjab</name>
			<code>PB</code>
			<fips>04</fips>
			<region></region>
		</division>
		<division>
			<name>Baluchistan</name>
			<code>BA</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Islamabad</name>
			<code>IS</code>
			<fips>08</fips>
			<region></region>
		</division>
		<division>
			<name>North-West Frontier</name>
			<code>NW</code>
			<fips>03</fips>
			<region></region>
		</division>
		<num>586</num>
	</territory>
	<territory>
		<name>Palau</name>
		<alpha2>PW</alpha2>
		<alpha3>PLW</alpha3>
		<num>585</num>
		<fips>70</fips>
	</territory>
	<territory>
		<name>Palestine</name>
		<alpha2>PS</alpha2>
		<alpha3>PSE</alpha3>
		<num>275</num>
	</territory>
	<territory>
		<name>Panama</name>
		<alpha2>PA</alpha2>
		<alpha3>PAN</alpha3>
		<division>
			<name>Veraguas</name>
			<code>9</code>
			<fips>10</fips>
			<region></region>
		</division>
		<division>
			<name>Panam&#xe1;</name>
			<code>8</code>
			<fips>08</fips>
			<region></region>
		</division>
		<division>
			<name>Cocl&#xe9;</name>
			<code>2</code>
			<fips>03</fips>
			<region></region>
		</division>
		<division>
			<name>Dari&#xe9;n</name>
			<code>5</code>
			<fips>05</fips>
			<region></region>
		</division>
		<division>
			<name>Los Santos</name>
			<code>7</code>
			<fips>07</fips>
			<region></region>
		</division>
		<division>
			<name>Col&#xf3;n</name>
			<code>3</code>
			<fips>04</fips>
			<region></region>
		</division>
		<division>
			<name>Bocas del Toro</name>
			<code>1</code>
			<fips>01</fips>
			<region></region>
		</division>
		<division>
			<name>Chiriqu&#xed;</name>
			<code>4</code>
			<fips>02</fips>
			<region></region>
		</division>
		<division>
			<name>Comarca de San Blas</name>
			<code>0</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Herrera</name>
			<code>6</code>
			<fips>06</fips>
			<region></region>
		</division>
		<num>591</num>
	</territory>
	<territory>
		<name>Papua New Guinea</name>
		<alpha2>PG</alpha2>
		<alpha3>PNG</alpha3>
		<division>
			<name>West New Britain</name>
			<code>WBK</code>
			<fips>17</fips>
			<region></region>
		</division>
		<division>
			<name>Central</name>
			<code>CPM</code>
			<fips>01</fips>
			<region></region>
		</division>
		<division>
			<name>East New Britain</name>
			<code>EBR</code>
			<fips>10</fips>
			<region></region>
		</division>
		<division>
			<name>Madang</name>
			<code>MPM</code>
			<fips>12</fips>
			<region></region>
		</division>
		<division>
			<name>North Solomons</name>
			<code>NSA</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>East Sepik</name>
			<code>ESW</code>
			<fips>11</fips>
			<region></region>
		</division>
		<division>
			<name>Eastern Highlands</name>
			<code>EHG</code>
			<fips>09</fips>
			<region></region>
		</division>
		<division>
			<name>Western Highlands</name>
			<code>WHM</code>
			<fips>16</fips>
			<region></region>
		</division>
		<division>
			<name>Manus</name>
			<code>MRL</code>
			<fips>13</fips>
			<region></region>
		</division>
		<division>
			<name>National Capital District (Port Moresby)</name>
			<code>NCD</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Enga</name>
			<code>EPW</code>
			<fips>19</fips>
			<region></region>
		</division>
		<division>
			<name>Western</name>
			<code>WPD</code>
			<fips>06</fips>
			<region></region>
		</division>
		<division>
			<name>Chimbu</name>
			<code>CPK</code>
			<fips>08</fips>
			<region></region>
		</division>
		<division>
			<name>Sandaun [West Sepik]</name>
			<code>SAN</code>
			<fips>18</fips>
			<region></region>
		</division>
		<division>
			<name>Morobe</name>
			<code>MPL</code>
			<fips>14</fips>
			<region></region>
		</division>
		<division>
			<name>Gulf</name>
			<code>GPK</code>
			<fips>02</fips>
			<region></region>
		</division>
		<division>
			<name>New Ireland</name>
			<code>NIK</code>
			<fips>15</fips>
			<region></region>
		</division>
		<division>
			<name>Northern</name>
			<code>NPP</code>
			<fips>04</fips>
			<region></region>
		</division>
		<division>
			<name>Milne Bay</name>
			<code>MBA</code>
			<fips>03</fips>
			<region></region>
		</division>
		<division>
			<name>Southern Highlands</name>
			<code>SHM</code>
			<fips>05</fips>
			<region></region>
		</division>
		<num>598</num>
	</territory>
	<territory>
		<name>Paraguay</name>
		<alpha2>PY</alpha2>
		<alpha3>PRY</alpha3>
		<division>
			<name>Misiones</name>
			<code>8</code>
			<fips>12</fips>
			<region></region>
		</division>
		<division>
			<name>Central</name>
			<code>11</code>
			<fips>06</fips>
			<region></region>
		</division>
		<division>
			<name>Canindey&#xfa;</name>
			<code>14</code>
			<fips>19</fips>
			<region></region>
		</division>
		<division>
			<name>Amambay</name>
			<code>13</code>
			<fips>02</fips>
			<region></region>
		</division>
		<division>
			<name>Guair&#xe1;</name>
			<code>4</code>
			<fips>10</fips>
			<region></region>
		</division>
		<division>
			<name>Alto Paraguay</name>
			<code>16</code>
			<fips>23</fips>
			<region></region>
		</division>
		<division>
			<name>Asunci&#xf3;n</name>
			<code>ASU</code>
			<fips>22</fips>
			<region></region>
		</division>
		<division>
			<name>San Pedro</name>
			<code>2</code>
			<fips>17</fips>
			<region></region>
		</division>
		<division>
			<name>Concepci&#xf3;n</name>
			<code>1</code>
			<fips>07</fips>
			<region></region>
		</division>
		<division>
			<name>Cordillera</name>
			<code>3</code>
			<fips>08</fips>
			<region></region>
		</division>
		<division>
			<name>Caazap&#xe1;</name>
			<code>6</code>
			<fips>05</fips>
			<region></region>
		</division>
		<division>
			<name>Boquer&#xf3;n</name>
			<code>19</code>
			<fips>24</fips>
			<region></region>
		</division>
		<division>
			<name>Itap&#xfa;a</name>
			<code>7</code>
			<fips>11</fips>
			<region></region>
		</division>
		<division>
			<name>Presidente Hayes</name>
			<code>15</code>
			<fips>16</fips>
			<region></region>
		</division>
		<division>
			<name>Caaguaz&#xfa;</name>
			<code>5</code>
			<fips>04</fips>
			<region></region>
		</division>
		<division>
			<name>Paraguar&#xed;</name>
			<code>9</code>
			<fips>15</fips>
			<region></region>
		</division>
		<division>
			<name>&#xd1;eembuc&#xfa;</name>
			<code>12</code>
			<fips>13</fips>
			<region></region>
		</division>
		<division>
			<name>Alto Paran&#xe1;</name>
			<code>10</code>
			<fips>01</fips>
			<region></region>
		</division>
		<num>600</num>
	</territory>
	<territory>
		<name>Peru</name>
		<alpha2>PE</alpha2>
		<alpha3>PER</alpha3>
		<division>
			<name>La Libertad</name>
			<code>LAL</code>
			<fips>13</fips>
			<region></region>
		</division>
		<division>
			<name>Huancavelica</name>
			<code>HUV</code>
			<fips>09</fips>
			<region></region>
		</division>
		<division>
			<name>Ayacucho</name>
			<code>AYA</code>
			<fips>05</fips>
			<region></region>
		</division>
		<division>
			<name>Puno</name>
			<code>PUN</code>
			<fips>21</fips>
			<region></region>
		</division>
		<division>
			<name>Lambayeque</name>
			<code>LAM</code>
			<fips>14</fips>
			<region></region>
		</division>
		<division>
			<name>Loreto</name>
			<code>LOR</code>
			<fips>16</fips>
			<region></region>
		</division>
		<division>
			<name>San Mart&#xed;n</name>
			<code>SAM</code>
			<fips>22</fips>
			<region></region>
		</division>
		<division>
			<name>Madre de Dios</name>
			<code>MDD</code>
			<fips>17</fips>
			<region></region>
		</division>
		<division>
			<name>Lima</name>
			<code>LIM</code>
			<fips>15</fips>
			<region></region>
		</division>
		<division>
			<name>El Callao</name>
			<code>CAL</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Ica</name>
			<code>ICA</code>
			<fips>11</fips>
			<region></region>
		</division>
		<division>
			<name>Ancash</name>
			<code>ANC</code>
			<fips>02</fips>
			<region></region>
		</division>
		<division>
			<name>Hu&#xe1;nuco</name>
			<code>HUC</code>
			<fips>10</fips>
			<region></region>
		</division>
		<division>
			<name>Jun&#xed;n</name>
			<code>JUN</code>
			<fips>12</fips>
			<region></region>
		</division>
		<division>
			<name>Cuzco [Cusco]</name>
			<code>CUS</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Tumbes</name>
			<code>TUM</code>
			<fips>24</fips>
			<region></region>
		</division>
		<division>
			<name>Apur&#xed;mac</name>
			<code>APU</code>
			<fips>03</fips>
			<region></region>
		</division>
		<division>
			<name>Cajamarca</name>
			<code>CAJ</code>
			<fips>06</fips>
			<region></region>
		</division>
		<division>
			<name>Piura</name>
			<code>PIU</code>
			<fips>20</fips>
			<region></region>
		</division>
		<division>
			<name>Tacna</name>
			<code>TAC</code>
			<fips>23</fips>
			<region></region>
		</division>
		<division>
			<name>Arequipa</name>
			<code>ARE</code>
			<fips>04</fips>
			<region></region>
		</division>
		<division>
			<name>Moquegua</name>
			<code>MOQ</code>
			<fips>18</fips>
			<region></region>
		</division>
		<division>
			<name>Pasco</name>
			<code>PAS</code>
			<fips>19</fips>
			<region></region>
		</division>
		<division>
			<name>Ucayali</name>
			<code>UCA</code>
			<fips>25</fips>
			<region></region>
		</division>
		<division>
			<name>Amazonas</name>
			<code>AMA</code>
			<fips>01</fips>
			<region></region>
		</division>
		<num>604</num>
	</territory>
	<territory>
		<name>Philippines</name>
		<alpha2>PH</alpha2>
		<alpha3>PHL</alpha3>
		<division>
			<name>Sultan Kudarat</name>
			<code>SUK</code>
			<fips>71</fips>
			<region>12</region>
		</division>
		<division>
			<name>Shariff Kabunsuan</name>
			<code></code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Siquijor</name>
			<code>SIG</code>
			<fips>69</fips>
			<region>07</region>
		</division>
		<division>
			<name>Davao del Sur</name>
			<code>DAS</code>
			<fips>25</fips>
			<region>11</region>
		</division>
		<division>
			<name>Agusan del Sur</name>
			<code>AGS</code>
			<fips>03</fips>
			<region>13</region>
		</division>
		<division>
			<name>Surigao del Norte</name>
			<code>SUN</code>
			<fips>61</fips>
			<region>13</region>
		</division>
		<division>
			<name>Pampanga</name>
			<code>PAM</code>
			<fips>50</fips>
			<region>03</region>
		</division>
		<division>
			<name>Kalinga</name>
			<code>KAL</code>
			<fips>D6</fips>
			<region>15</region>
		</division>
		<division>
			<name>Quirino</name>
			<code>QUI</code>
			<fips>68</fips>
			<region>02</region>
		</division>
		<division>
			<name>Zamboanga Sibuguey [Zamboanga Sibugay]</name>
			<code>ZSI</code>
			<fips></fips>
			<region>09</region>
		</division>
		<division>
			<name>Zamboanga del Sur</name>
			<code>ZAS</code>
			<fips>66</fips>
			<region>09</region>
		</division>
		<division>
			<name>Leyte</name>
			<code>LEY</code>
			<fips>37</fips>
			<region>08</region>
		</division>
		<division>
			<name>Aklan</name>
			<code>AKL</code>
			<fips>04</fips>
			<region>06</region>
		</division>
		<division>
			<name>Bulacan</name>
			<code>BUL</code>
			<fips>13</fips>
			<region>03</region>
		</division>
		<division>
			<name>Bukidnon</name>
			<code>BUK</code>
			<fips>12</fips>
			<region>10</region>
		</division>
		<division>
			<name>Benguet</name>
			<code>BEN</code>
			<fips>10</fips>
			<region>15</region>
		</division>
		<division>
			<name>Laguna</name>
			<code>LAG</code>
			<fips>33</fips>
			<region>04</region>
		</division>
		<division>
			<name>Zamboanga del Norte</name>
			<code>ZAN</code>
			<fips>65</fips>
			<region>09</region>
		</division>
		<division>
			<name>Palawan</name>
			<code>PLW</code>
			<fips>49</fips>
			<region>04</region>
		</division>
		<division>
			<name>Iloilo</name>
			<code>ILI</code>
			<fips>30</fips>
			<region>06</region>
		</division>
		<division>
			<name>Ilocos Norte</name>
			<code>ILN</code>
			<fips>28</fips>
			<region>01</region>
		</division>
		<division>
			<name>Bataan</name>
			<code>BAN</code>
			<fips>07</fips>
			<region>03</region>
		</division>
		<division>
			<name>Cagayan</name>
			<code>CAG</code>
			<fips>14</fips>
			<region>02</region>
		</division>
		<division>
			<name>Lanao del Sur</name>
			<code>LAS</code>
			<fips>35</fips>
			<region>14</region>
		</division>
		<division>
			<name>La Union</name>
			<code>LUN</code>
			<fips>36</fips>
			<region>01</region>
		</division>
		<division>
			<name>Albay</name>
			<code>ALB</code>
			<fips>05</fips>
			<region>05</region>
		</division>
		<division>
			<name>South Cotabato</name>
			<code>SCO</code>
			<fips>70</fips>
			<region>11</region>
		</division>
		<division>
			<name>Misamis Oriental</name>
			<code>MSR</code>
			<fips>43</fips>
			<region>10</region>
		</division>
		<division>
			<name>Misamis Occidental</name>
			<code>MSC</code>
			<fips>42</fips>
			<region>10</region>
		</division>
		<division>
			<name>Davao del Norte</name>
			<code>DAV</code>
			<fips>24</fips>
			<region>11</region>
		</division>
		<division>
			<name>Nueva Vizcaya</name>
			<code>NUV</code>
			<fips>48</fips>
			<region>02</region>
		</division>
		<division>
			<name>Camarines Norte</name>
			<code>CAN</code>
			<fips>15</fips>
			<region>05</region>
		</division>
		<division>
			<name>North Cotabato</name>
			<code>NCO</code>
			<fips>57</fips>
			<region>12</region>
		</division>
		<division>
			<name>Zambales</name>
			<code>ZMB</code>
			<fips>64</fips>
			<region>03</region>
		</division>
		<division>
			<name>Mountain Province</name>
			<code>MOU</code>
			<fips></fips>
			<region>15</region>
		</division>
		<division>
			<name>Batanes</name>
			<code>BTN</code>
			<fips>08</fips>
			<region>02</region>
		</division>
		<division>
			<name>Maguindanao</name>
			<code>MAG</code>
			<fips>56</fips>
			<region>14</region>
		</division>
		<division>
			<name>Marinduque</name>
			<code>MAD</code>
			<fips>38</fips>
			<region>04</region>
		</division>
		<division>
			<name>Sorsogon</name>
			<code>SOR</code>
			<fips>58</fips>
			<region>05</region>
		</division>
		<division>
			<name>Basilan</name>
			<code>BAS</code>
			<fips>22</fips>
			<region>09</region>
		</division>
		<division>
			<name>Agusan del Norte</name>
			<code>AGN</code>
			<fips>02</fips>
			<region>13</region>
		</division>
		<division>
			<name>Camiguin</name>
			<code>CAM</code>
			<fips>17</fips>
			<region>10</region>
		</division>
		<division>
			<name>Mindoro Oriental</name>
			<code>MDR</code>
			<fips>41</fips>
			<region>04</region>
		</division>
		<division>
			<name>Masbate</name>
			<code>MAS</code>
			<fips>39</fips>
			<region>05</region>
		</division>
		<division>
			<name>Western Samar</name>
			<code>WSA</code>
			<fips>55</fips>
			<region>08</region>
		</division>
		<division>
			<name>Isabela</name>
			<code>ISA</code>
			<fips>31</fips>
			<region>02</region>
		</division>
		<division>
			<name>Sarangani</name>
			<code>SAR</code>
			<fips></fips>
			<region>11</region>
		</division>
		<division>
			<name>Apayao</name>
			<code>APA</code>
			<fips></fips>
			<region>15</region>
		</division>
		<division>
			<name>Bohol</name>
			<code>BOH</code>
			<fips>11</fips>
			<region>07</region>
		</division>
		<division>
			<name>Sulu</name>
			<code>SLU</code>
			<fips>60</fips>
			<region>14</region>
		</division>
		<division>
			<name>Capiz</name>
			<code>CAP</code>
			<fips>18</fips>
			<region>06</region>
		</division>
		<division>
			<name>Biliran</name>
			<code>BIL</code>
			<fips></fips>
			<region>08</region>
		</division>
		<division>
			<name>Antique</name>
			<code>ANT</code>
			<fips>06</fips>
			<region>06</region>
		</division>
		<division>
			<name>Mindoro Occidental</name>
			<code>MDC</code>
			<fips>40</fips>
			<region>04</region>
		</division>
		<division>
			<name>Rizal</name>
			<code>RIZ</code>
			<fips>53</fips>
			<region>04</region>
		</division>
		<division>
			<name>Ilocos Sur</name>
			<code>ILS</code>
			<fips>29</fips>
			<region>01</region>
		</division>
		<division>
			<name>Batangas</name>
			<code>BTG</code>
			<fips>09</fips>
			<region>04</region>
		</division>
		<division>
			<name>Tarlac</name>
			<code>TAR</code>
			<fips>63</fips>
			<region>03</region>
		</division>
		<division>
			<name>Negros Occidental</name>
			<code>NEC</code>
			<fips>H3</fips>
			<region>06</region>
		</division>
		<division>
			<name>Compostela Valley</name>
			<code>COM</code>
			<fips></fips>
			<region>11</region>
		</division>
		<division>
			<name>Negros Oriental</name>
			<code>NER</code>
			<fips>46</fips>
			<region>07</region>
		</division>
		<division>
			<name>Abra</name>
			<code>ABR</code>
			<fips>01</fips>
			<region>15</region>
		</division>
		<division>
			<name>Camarines Sur</name>
			<code>CAS</code>
			<fips>16</fips>
			<region>05</region>
		</division>
		<division>
			<name>Catanduanes</name>
			<code>CAT</code>
			<fips>19</fips>
			<region>05</region>
		</division>
		<division>
			<name>Davao Oriental</name>
			<code>DAO</code>
			<fips>26</fips>
			<region>11</region>
		</division>
		<division>
			<name>Northern Samar</name>
			<code>NSA</code>
			<fips>67</fips>
			<region>08</region>
		</division>
		<division>
			<name>Pangasinan</name>
			<code>PAN</code>
			<fips>51</fips>
			<region>01</region>
		</division>
		<division>
			<name>Aurora</name>
			<code>AUR</code>
			<fips>G8</fips>
			<region>04</region>
		</division>
		<division>
			<name>Tawi-Tawi</name>
			<code>TAW</code>
			<fips>72</fips>
			<region>14</region>
		</division>
		<division>
			<name>Cebu</name>
			<code>CEB</code>
			<fips>21</fips>
			<region>07</region>
		</division>
		<division>
			<name>Nueva Ecija</name>
			<code>NUE</code>
			<fips>47</fips>
			<region>03</region>
		</division>
		<division>
			<name>Ifugao</name>
			<code>IFU</code>
			<fips>27</fips>
			<region>15</region>
		</division>
		<division>
			<name>Guimaras</name>
			<code>GUI</code>
			<fips></fips>
			<region>06</region>
		</division>
		<division>
			<name>Eastern Samar</name>
			<code>EAS</code>
			<fips>23</fips>
			<region>08</region>
		</division>
		<division>
			<name>Lanao del Norte</name>
			<code>LAN</code>
			<fips>34</fips>
			<region>12</region>
		</division>
		<division>
			<name>Surigao del Sur</name>
			<code>SUR</code>
			<fips>62</fips>
			<region>13</region>
		</division>
		<division>
			<name>Romblon</name>
			<code>ROM</code>
			<fips>54</fips>
			<region>04</region>
		</division>
		<division>
			<name>Cavite</name>
			<code>CAV</code>
			<fips>20</fips>
			<region>04</region>
		</division>
		<division>
			<name>Quezon</name>
			<code>QUE</code>
			<fips>H2</fips>
			<region>04</region>
		</division>
		<division>
			<name>Southern Leyte</name>
			<code>SLE</code>
			<fips>59</fips>
			<region>08</region>
		</division>
		<num>608</num>
	</territory>
	<territory>
		<name>Pitcairn</name>
		<alpha2>PN</alpha2>
		<alpha3>PCN</alpha3>
		<num>612</num>
	</territory>
	<territory>
		<name>Poland</name>
		<alpha2>PL</alpha2>
		<alpha3>POL</alpha3>
		<division>
			<name>Lubelskie</name>
			<code>LU</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Podkarpackie</name>
			<code>PK</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Malopolskie</name>
			<code>MA</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Kujawsko-pomorskie</name>
			<code>KP</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Dolnoslaskie</name>
			<code>DS</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Wielkopolskie</name>
			<code>WP</code>
			<fips>86</fips>
			<region></region>
		</division>
		<division>
			<name>Lubuskie</name>
			<code>LB</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Podlaskie</name>
			<code>PD</code>
			<fips>81</fips>
			<region></region>
		</division>
		<division>
			<name>Opolskie</name>
			<code>OP</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Swietokrzyskie</name>
			<code>SK</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Zachodniopomorskie</name>
			<code>ZP</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Warminsko-mazurskie</name>
			<code>WN</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>L&#xf3;dzkie</name>
			<code>LD</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Slaskie</name>
			<code>SL</code>
			<fips>83</fips>
			<region></region>
		</division>
		<division>
			<name>Mazowieckie</name>
			<code>MZ</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Pomorskie</name>
			<code>PM</code>
			<fips></fips>
			<region></region>
		</division>
		<num>616</num>
	</territory>
	<territory>
		<name>Portugal</name>
		<alpha2>PT</alpha2>
		<alpha3>PRT</alpha3>
		<division>
			<name>Santar&#xe9;m</name>
			<code>14</code>
			<fips>18</fips>
			<region></region>
		</division>
		<division>
			<name>Lisboa</name>
			<code>11</code>
			<fips>14</fips>
			<region></region>
		</division>
		<division>
			<name>&#xc9;vora</name>
			<code>07</code>
			<fips>08</fips>
			<region></region>
		</division>
		<division>
			<name>Leiria</name>
			<code>10</code>
			<fips>13</fips>
			<region></region>
		</division>
		<division>
			<name>Aveiro</name>
			<code>01</code>
			<fips>02</fips>
			<region></region>
		</division>
		<division>
			<name>Bragan&#xe7;a</name>
			<code>04</code>
			<fips>05</fips>
			<region></region>
		</division>
		<division>
			<name>Viana do Castelo</name>
			<code>16</code>
			<fips>20</fips>
			<region></region>
		</division>
		<division>
			<name>Portalegre</name>
			<code>12</code>
			<fips>16</fips>
			<region></region>
		</division>
		<division>
			<name>Beja</name>
			<code>02</code>
			<fips>03</fips>
			<region></region>
		</division>
		<division>
			<name>Viseu</name>
			<code>18</code>
			<fips>22</fips>
			<region></region>
		</division>
		<division>
			<name>Faro</name>
			<code>08</code>
			<fips>09</fips>
			<region></region>
		</division>
		<division>
			<name>Coimbra</name>
			<code>06</code>
			<fips>07</fips>
			<region></region>
		</division>
		<division>
			<name>Vila Real</name>
			<code>17</code>
			<fips>21</fips>
			<region></region>
		</division>
		<division>
			<name>Braga</name>
			<code>03</code>
			<fips>04</fips>
			<region></region>
		</division>
		<division>
			<name>Set&#xfa;bal</name>
			<code>15</code>
			<fips>19</fips>
			<region></region>
		</division>
		<division>
			<name>Regi&#xe3;o Aut&#xf3;noma da Madeira</name>
			<code>30</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Porto</name>
			<code>13</code>
			<fips>17</fips>
			<region></region>
		</division>
		<division>
			<name>Regi&#xe3;o Aut&#xf3;noma dos A&#xe7;ores</name>
			<code>20</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Guarda</name>
			<code>09</code>
			<fips>11</fips>
			<region></region>
		</division>
		<division>
			<name>Castelo Branco</name>
			<code>05</code>
			<fips>06</fips>
			<region></region>
		</division>
		<num>620</num>
	</territory>
	<territory>
		<name>Puerto Rico</name>
		<alpha2>PR</alpha2>
		<alpha3>PRI</alpha3>
		<num>630</num>
	</territory>
	<territory>
		<name>Qatar</name>
		<alpha2>QA</alpha2>
		<alpha3>QAT</alpha3>
		<division>
			<name>Al Khawr</name>
			<code>KH</code>
			<fips>04</fips>
			<region></region>
		</division>
		<division>
			<name>Al Jumayliyah</name>
			<code>JU</code>
			<fips>03</fips>
			<region></region>
		</division>
		<division>
			<name>Umm Salal</name>
			<code>US</code>
			<fips>09</fips>
			<region></region>
		</division>
		<division>
			<name>Al Wakrah</name>
			<code>WA</code>
			<fips>05</fips>
			<region></region>
		</division>
		<division>
			<name>Al Ghuwayriyah</name>
			<code>GH</code>
			<fips>02</fips>
			<region></region>
		</division>
		<division>
			<name>Jariyan al Batnah</name>
			<code>JB</code>
			<fips>07</fips>
			<region></region>
		</division>
		<division>
			<name>Ar Rayyan</name>
			<code>RA</code>
			<fips>06</fips>
			<region></region>
		</division>
		<division>
			<name>Madinat ash Shamal</name>
			<code>MS</code>
			<fips>08</fips>
			<region></region>
		</division>
		<division>
			<name>Ad Dawhah</name>
			<code>DA</code>
			<fips>01</fips>
			<region></region>
		</division>
		<num>634</num>
	</territory>
	<territory>
		<name>R&#xe9;union</name>
		<alpha2>RE</alpha2>
		<alpha3>REU</alpha3>
		<num>638</num>
	</territory>
	<territory>
		<name>Romania</name>
		<alpha2>RO</alpha2>
		<alpha3>ROU</alpha3>
		<division>
			<name>Ilfov</name>
			<code>IF</code>
			<fips></fips>
			<region></region>
		</division>
		<num>642</num>
	</territory>
	<territory>
		<name>Russian Federation</name>
		<alpha2>RU</alpha2>
		<alpha3>RUS</alpha3>
		<division>
			<name>Kurganskaya oblast'</name>
			<code>KGN</code>
			<fips>40</fips>
			<region></region>
		</division>
		<division>
			<name>Kostromskaya oblast'</name>
			<code>KOS</code>
			<fips>37</fips>
			<region></region>
		</division>
		<division>
			<name>Koryakskiy avtonomnyy okrug</name>
			<code>KOR</code>
			<fips>36</fips>
			<region></region>
		</division>
		<division>
			<name>Permskaya oblast'</name>
			<code>PER</code>
			<fips>58</fips>
			<region></region>
		</division>
		<division>
			<name>Khanty-Mansiyskiy avtonomnyy okrug</name>
			<code>KHM</code>
			<fips>32</fips>
			<region></region>
		</division>
		<division>
			<name>Yamalo-Nenetskiy avtonomnyy okrug</name>
			<code>YAN</code>
			<fips>87</fips>
			<region></region>
		</division>
		<division>
			<name>Ryazanskaya oblast'</name>
			<code>RYA</code>
			<fips>62</fips>
			<region></region>
		</division>
		<division>
			<name>Khakasiya, Respublika</name>
			<code>KK</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Yaroslavskaya oblast'</name>
			<code>YAR</code>
			<fips>88</fips>
			<region></region>
		</division>
		<division>
			<name>Smolenskaya oblast'</name>
			<code>SMO</code>
			<fips>69</fips>
			<region></region>
		</division>
		<division>
			<name>Samarskaya oblast'</name>
			<code>SAM</code>
			<fips>65</fips>
			<region></region>
		</division>
		<division>
			<name>Adygeya, Respublika</name>
			<code>AD</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Moskva</name>
			<code>MOW</code>
			<fips>48</fips>
			<region></region>
		</division>
		<division>
			<name>Amurskaya oblast'</name>
			<code>AMU</code>
			<fips>05</fips>
			<region></region>
		</division>
		<division>
			<name>Omskaya oblast'</name>
			<code>OMS</code>
			<fips>54</fips>
			<region></region>
		</division>
		<division>
			<name>Stavropol'skiy kray</name>
			<code>STA</code>
			<fips>70</fips>
			<region></region>
		</division>
		<division>
			<name>Penzenskaya oblast'</name>
			<code>PNZ</code>
			<fips>57</fips>
			<region></region>
		</division>
		<division>
			<name>Rostovskaya oblast'</name>
			<code>ROS</code>
			<fips>61</fips>
			<region></region>
		</division>
		<division>
			<name>Volgogradskaya oblast'</name>
			<code>VGG</code>
			<fips>84</fips>
			<region></region>
		</division>
		<division>
			<name>Aginskiy Buryatskiy avtonomnyy okrug</name>
			<code>AGB</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Murmanskaya oblast'</name>
			<code>MUR</code>
			<fips>49</fips>
			<region></region>
		</division>
		<division>
			<name>Arkhangel'skaya oblast'</name>
			<code>ARK</code>
			<fips>06</fips>
			<region></region>
		</division>
		<division>
			<name>Tyva, Respublika [Tuva]</name>
			<code>TY</code>
			<fips>79</fips>
			<region></region>
		</division>
		<division>
			<name>Buryatiya, Respublika</name>
			<code>BU</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Tverskaya oblast'</name>
			<code>TVE</code>
			<fips>77</fips>
			<region></region>
		</division>
		<division>
			<name>Tyumenskaya oblast'</name>
			<code>TYU</code>
			<fips>78</fips>
			<region></region>
		</division>
		<division>
			<name>Sakha, Respublika [Yakutiya]</name>
			<code>SA</code>
			<fips>63</fips>
			<region></region>
		</division>
		<division>
			<name>Tatarstan, Respublika</name>
			<code>TA</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Komi, Respublika</name>
			<code>KO</code>
			<fips>34</fips>
			<region></region>
		</division>
		<division>
			<name>Severnaya Osetiya, Respublika [Alaniya] [Respublika Severnaya Osetiya-Alaniya]</name>
			<code>SE</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Chukotskiy avtonomnyy okrug</name>
			<code>CHU</code>
			<fips>15</fips>
			<region></region>
		</division>
		<division>
			<name>Ul'yanovskaya oblast'</name>
			<code>ULY</code>
			<fips>81</fips>
			<region></region>
		</division>
		<division>
			<name>Chitinskaya oblast'</name>
			<code>CHI</code>
			<fips>14</fips>
			<region></region>
		</division>
		<division>
			<name>Evenkiyskiy avtonomnyy okrug</name>
			<code>EVE</code>
			<fips>18</fips>
			<region></region>
		</division>
		<division>
			<name>Khabarovskiy kray</name>
			<code>KHA</code>
			<fips>30</fips>
			<region></region>
		</division>
		<division>
			<name>Yevreyskaya avtonomnaya oblast'</name>
			<code>YEV</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Kareliya, Respublika</name>
			<code>KR</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Ingushskaya Respublika [Respublika Ingushetiya]</name>
			<code>IN</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Chelyabinskaya oblast'</name>
			<code>CHE</code>
			<fips>13</fips>
			<region></region>
		</division>
		<division>
			<name>Sakhalinskaya oblast'</name>
			<code>SAK</code>
			<fips>64</fips>
			<region></region>
		</division>
		<division>
			<name>Dagestan, Respublika</name>
			<code>DA</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Nizhegorodskaya oblast'</name>
			<code>NIZ</code>
			<fips>51</fips>
			<region></region>
		</division>
		<division>
			<name>Pskovskaya oblast'</name>
			<code>PSK</code>
			<fips>60</fips>
			<region></region>
		</division>
		<division>
			<name>Kaliningradskaya oblast'</name>
			<code>KGD</code>
			<fips>23</fips>
			<region></region>
		</division>
		<division>
			<name>Bryanskaya oblast'</name>
			<code>BRY</code>
			<fips>10</fips>
			<region></region>
		</division>
		<division>
			<name>Magadanskaya oblast'</name>
			<code>MAG</code>
			<fips>44</fips>
			<region></region>
		</division>
		<division>
			<name>Komi-Permyatskiy avtonomnyy okrug</name>
			<code>KOP</code>
			<fips>35</fips>
			<region></region>
		</division>
		<division>
			<name>Moskovskaya oblast'</name>
			<code>MOS</code>
			<fips>47</fips>
			<region></region>
		</division>
		<division>
			<name>Tul'skaya oblast'</name>
			<code>TUL</code>
			<fips>76</fips>
			<region></region>
		</division>
		<division>
			<name>Taymyrskiy (Dolgano-Nenetskiy) avtonomnyy okrug</name>
			<code>TAY</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Chuvashskaya Respublika</name>
			<code>CU</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Vologodskaya oblast'</name>
			<code>VLG</code>
			<fips>85</fips>
			<region></region>
		</division>
		<division>
			<name>Novosibirskaya oblast'</name>
			<code>NVS</code>
			<fips>53</fips>
			<region></region>
		</division>
		<division>
			<name>Nenetskiy avtonomnyy okrug</name>
			<code>NEN</code>
			<fips>50</fips>
			<region></region>
		</division>
		<division>
			<name>Kaluzhskaya oblast'</name>
			<code>KLU</code>
			<fips>25</fips>
			<region></region>
		</division>
		<division>
			<name>Kalmykiya, Respublika</name>
			<code>KL</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Krasnoyarskiy kray</name>
			<code>KYA</code>
			<fips>39</fips>
			<region></region>
		</division>
		<division>
			<name>Tomskaya oblast'</name>
			<code>TOM</code>
			<fips>75</fips>
			<region></region>
		</division>
		<division>
			<name>Mariy El, Respublika</name>
			<code>ME</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Ust'-Ordynskiy Buryatskiy avtonomnyy okrug</name>
			<code>UOB</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Kemerovskaya oblast'</name>
			<code>KEM</code>
			<fips>29</fips>
			<region></region>
		</division>
		<division>
			<name>Vladimirskaya oblast'</name>
			<code>VLA</code>
			<fips>83</fips>
			<region></region>
		</division>
		<division>
			<name>Chechenskaya Respublika</name>
			<code>CE</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Kamchatskaya oblast'</name>
			<code>KAM</code>
			<fips>26</fips>
			<region></region>
		</division>
		<division>
			<name>Sverdlovskaya oblast'</name>
			<code>SVE</code>
			<fips>71</fips>
			<region></region>
		</division>
		<division>
			<name>Kirovskaya oblast'</name>
			<code>KIR</code>
			<fips>33</fips>
			<region></region>
		</division>
		<division>
			<name>Orenburgskaya oblast'</name>
			<code>ORE</code>
			<fips>55</fips>
			<region></region>
		</division>
		<division>
			<name>Leningradskaya oblast'</name>
			<code>LEN</code>
			<fips>42</fips>
			<region></region>
		</division>
		<division>
			<name>Udmurtskaya Respublika</name>
			<code>UD</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Astrakhanskaya oblast'</name>
			<code>AST</code>
			<fips>07</fips>
			<region></region>
		</division>
		<division>
			<name>Kabardino-Balkarskaya Respublika</name>
			<code>KB</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Ivanovskaya oblast'</name>
			<code>IVA</code>
			<fips>21</fips>
			<region></region>
		</division>
		<division>
			<name>Saratovskaya oblast'</name>
			<code>SAR</code>
			<fips>67</fips>
			<region></region>
		</division>
		<division>
			<name>Altay, Respublika</name>
			<code>AL</code>
			<fips>03</fips>
			<region></region>
		</division>
		<division>
			<name>Altayskiy kray</name>
			<code>ALT</code>
			<fips>04</fips>
			<region></region>
		</division>
		<division>
			<name>Lipetskaya oblast'</name>
			<code>LIP</code>
			<fips>43</fips>
			<region></region>
		</division>
		<division>
			<name>Belgorodskaya oblast'</name>
			<code>BEL</code>
			<fips>09</fips>
			<region></region>
		</division>
		<division>
			<name>Krasnodarskiy kray</name>
			<code>KDA</code>
			<fips>38</fips>
			<region></region>
		</division>
		<division>
			<name>Irkutskaya oblast'</name>
			<code>IRK</code>
			<fips>20</fips>
			<region></region>
		</division>
		<division>
			<name>Sankt-Peterburg</name>
			<code>SPE</code>
			<fips>66</fips>
			<region></region>
		</division>
		<division>
			<name>Bashkortostan, Respublika</name>
			<code>BA</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Primorskiy kray</name>
			<code>PRI</code>
			<fips>59</fips>
			<region></region>
		</division>
		<division>
			<name>Karachayevo-Cherkesskaya Respublika</name>
			<code>KC</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Novgorodskaya oblast'</name>
			<code>NGR</code>
			<fips>52</fips>
			<region></region>
		</division>
		<division>
			<name>Kurskaya oblast'</name>
			<code>KRS</code>
			<fips>41</fips>
			<region></region>
		</division>
		<division>
			<name>Orlovskaya oblast'</name>
			<code>ORL</code>
			<fips>56</fips>
			<region></region>
		</division>
		<division>
			<name>Tambovskaya oblast'</name>
			<code>TAM</code>
			<fips>72</fips>
			<region></region>
		</division>
		<division>
			<name>Mordoviya, Respublika</name>
			<code>MO</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Voronezhskaya oblast'</name>
			<code>VOR</code>
			<fips>86</fips>
			<region></region>
		</division>
		<num>643</num>
	</territory>
	<territory>
		<name>Rwanda</name>
		<alpha2>RW</alpha2>
		<alpha3>RWA</alpha3>
		<division>
			<name>Cyangugu</name>
			<code>E</code>
			<fips>03</fips>
			<region></region>
		</division>
		<division>
			<name>Mutara</name>
			<code>M</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Kigali-Ville</name>
			<code>L</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Ruhengeri</name>
			<code>H</code>
			<fips>10</fips>
			<region></region>
		</division>
		<division>
			<name>Kigali-Rural</name>
			<code>K</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Kibuye</name>
			<code>F</code>
			<fips>08</fips>
			<region></region>
		</division>
		<division>
			<name>Gitarama</name>
			<code>B</code>
			<fips>06</fips>
			<region></region>
		</division>
		<division>
			<name>Butare</name>
			<code>C</code>
			<fips>01</fips>
			<region></region>
		</division>
		<division>
			<name>Gisenyi</name>
			<code>G</code>
			<fips>05</fips>
			<region></region>
		</division>
		<division>
			<name>Kibungo</name>
			<code>J</code>
			<fips>07</fips>
			<region></region>
		</division>
		<division>
			<name>Gikongoro</name>
			<code>D</code>
			<fips>04</fips>
			<region></region>
		</division>
		<division>
			<name>Byumba</name>
			<code>I</code>
			<fips>02</fips>
			<region></region>
		</division>
		<num>646</num>
	</territory>
	<territory>
		<name>Saint Helena</name>
		<alpha2>SH</alpha2>
		<alpha3>SHN</alpha3>
		<division>
			<name>Saint Helena</name>
			<code>SH</code>
			<fips>02</fips>
			<region></region>
		</division>
		<division>
			<name>Ascension</name>
			<code>AC</code>
			<fips>01</fips>
			<region></region>
		</division>
		<division>
			<name>Tristan da Cunha</name>
			<code>TA</code>
			<fips>03</fips>
			<region></region>
		</division>
		<num>654</num>
	</territory>
	<territory>
		<name>Saint Kitts and Nevis</name>
		<alpha2>KN</alpha2>
		<alpha3>KNA</alpha3>
		<num>659</num>
	</territory>
	<territory>
		<name>Saint Lucia</name>
		<alpha2>LC</alpha2>
		<alpha3>LCA</alpha3>
		<num>662</num>
	</territory>
	<territory>
		<name>Saint Pierre and Miquelon</name>
		<alpha2>PM</alpha2>
		<alpha3>SPM</alpha3>
		<num>666</num>
	</territory>
	<territory>
		<name>Saint Vincent and the Grenadines</name>
		<alpha2>VC</alpha2>
		<alpha3>VCT</alpha3>
		<num>670</num>
	</territory>
	<territory>
		<name>Samoa</name>
		<alpha2>WS</alpha2>
		<alpha3>WSM</alpha3>
		<division>
			<name>Palauli</name>
			<code>PA</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Atua</name>
			<code>AT</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Aiga-i-le-Tai</name>
			<code>AL</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Tuamasaga</name>
			<code>TU</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>A'ana</name>
			<code>AA</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Satupa'itea</name>
			<code>SA</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Gagaifomauga</name>
			<code>GI</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Vaisigano</name>
			<code>VS</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Va'a-o-Fonoti</name>
			<code>VF</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Gaga'emauga</name>
			<code>GE</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Fa'asaleleaga</name>
			<code>FA</code>
			<fips></fips>
			<region></region>
		</division>
		<num>882</num>
	</territory>
	<territory>
		<name>San Marino</name>
		<alpha2>SM</alpha2>
		<alpha3>SMR</alpha3>
		<num>674</num>
		<division>
			<name>Acquaviva</name>
			<code>AC</code>
			<fips>01</fips>
		</division>
		<division>
			<name>Borgo Maggiore</name>
			<code>BM</code>
			<fips>06</fips>
		</division>
		<division>
			<name>Chiesanuova</name>
			<code>CH</code>
			<fips>02</fips>
		</division>
		<division>
			<name>Domagnano</name>
			<code>DO</code>
			<fips>03</fips>
		</division>
		<division>
			<name>Faetano</name>
			<code>FA</code>
			<fips>04</fips>
		</division>
		<division>
			<name>Fiorentino</name>
			<code>FI</code>
			<fips>05</fips>
		</division>
		<division>
			<name>Montegiardino</name>
			<code>MG</code>
			<fips>08</fips>
		</division>
		<division>
			<name>San Marino</name>
			<code>SM</code>
			<fips>07</fips>
		</division>
		<division>
			<name>Serravalle</name>
			<code>SE</code>
			<fips>09</fips>
		</division>
	</territory>
	<territory>
		<name>Sao Tome and Principe</name>
		<alpha2>ST</alpha2>
		<alpha3>STP</alpha3>
		<division>
			<name>Pr&#xed;ncipe</name>
			<code>P</code>
			<fips>01</fips>
			<region></region>
		</division>
		<division>
			<name>S&#xe3;o Tom&#xe9;</name>
			<code>S</code>
			<fips>02</fips>
			<region></region>
		</division>
		<num>678</num>
	</territory>
	<territory>
		<name>Saudi Arabia</name>
		<alpha2>SA</alpha2>
		<alpha3>SAU</alpha3>
		<division>
			<name>Jizan</name>
			<code>09</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Asir</name>
			<code>14</code>
			<fips>11</fips>
			<region></region>
		</division>
		<division>
			<name>Al Jawf</name>
			<code>12</code>
			<fips>20</fips>
			<region></region>
		</division>
		<division>
			<name>Ha'il</name>
			<code>06</code>
			<fips>13</fips>
			<region></region>
		</division>
		<division>
			<name>Ash Sharqiyah</name>
			<code>04</code>
			<fips>06</fips>
			<region></region>
		</division>
		<division>
			<name>Ar Riyah</name>
			<code>01</code>
			<fips>10</fips>
			<region></region>
		</division>
		<division>
			<name>Al Qasim</name>
			<code>05</code>
			<fips>08</fips>
			<region></region>
		</division>
		<division>
			<name>Al Madinah</name>
			<code>03</code>
			<fips>05</fips>
			<region></region>
		</division>
		<division>
			<name>Tabuk</name>
			<code>07</code>
			<fips>19</fips>
			<region></region>
		</division>
		<division>
			<name>Najran</name>
			<code>10</code>
			<fips>16</fips>
			<region></region>
		</division>
		<division>
			<name>Al Bahah</name>
			<code>11</code>
			<fips>02</fips>
			<region></region>
		</division>
		<division>
			<name>Makkah</name>
			<code>02</code>
			<fips>14</fips>
			<region></region>
		</division>
		<division>
			<name>Al Hudud ash Shamaliyah</name>
			<code>08</code>
			<fips></fips>
			<region></region>
		</division>
		<num>682</num>
	</territory>
	<territory>
		<name>Senegal</name>
		<alpha2>SN</alpha2>
		<alpha3>SEN</alpha3>
		<division>
			<name>Thi&#xe8;s</name>
			<code>TH</code>
			<fips>07</fips>
			<region></region>
		</division>
		<division>
			<name>Ziguinchor</name>
			<code>ZG</code>
			<fips>12</fips>
			<region></region>
		</division>
		<division>
			<name>Kolda</name>
			<code>KD</code>
			<fips>11</fips>
			<region></region>
		</division>
		<division>
			<name>Kaolack</name>
			<code>KL</code>
			<fips>10</fips>
			<region></region>
		</division>
		<division>
			<name>Louga</name>
			<code>LG</code>
			<fips>08</fips>
			<region></region>
		</division>
		<division>
			<name>Dakar</name>
			<code>DK</code>
			<fips>01</fips>
			<region></region>
		</division>
		<division>
			<name>Fatick</name>
			<code>FK</code>
			<fips>09</fips>
			<region></region>
		</division>
		<division>
			<name>Saint-Louis</name>
			<code>SL</code>
			<fips>04</fips>
			<region></region>
		</division>
		<division>
			<name>Diourbel</name>
			<code>DB</code>
			<fips>03</fips>
			<region></region>
		</division>
		<division>
			<name>Tambacounda</name>
			<code>TC</code>
			<fips>05</fips>
			<region></region>
		</division>
		<num>686</num>
	</territory>
	<territory>
		<name>Serbia</name>
		<alpha2>RS</alpha2>
		<alpha3>SRB</alpha3>
		<division>
			<name>Vojvodina</name>
			<code>VO</code>
			<fips></fips>
			<region>CS-SR</region>
		</division>
		<division>
			<name>Srbija</name>
			<code>SR</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Kosovo-Metohija</name>
			<code>KM</code>
			<fips></fips>
			<region>CS-SR</region>
		</division>
		<num>688</num>
	</territory>
	<territory>
		<name>Seychelles</name>
		<alpha2>SC</alpha2>
		<alpha3>SYC</alpha3>
		<num>690</num>
	</territory>
	<territory>
		<name>Sierra Leone</name>
		<alpha2>SL</alpha2>
		<alpha3>SLE</alpha3>
		<division>
			<name>Western Area (Freetown)</name>
			<code>W</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Southern</name>
			<code>S</code>
			<fips>03</fips>
			<region></region>
		</division>
		<division>
			<name>Northern</name>
			<code>N</code>
			<fips>02</fips>
			<region></region>
		</division>
		<division>
			<name>Eastern</name>
			<code>E</code>
			<fips>01</fips>
			<region></region>
		</division>
		<num>694</num>
	</territory>
	<territory>
		<name>Singapore</name>
		<alpha2>SG</alpha2>
		<alpha3>SGP</alpha3>
		<num>702</num>
	</territory>
	<territory>
		<name>Slovakia</name>
		<alpha2>SK</alpha2>
		<alpha3>SVK</alpha3>
		<division>
			<name>Banskobystrick&#xfd; kraj</name>
			<code>BC</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>&#x17d;ilinsk&#xfd; kraj</name>
			<code>ZI</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Trnavsk&#xfd; kraj</name>
			<code>TA</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Ko&#x161;ick&#xfd; kraj</name>
			<code>KI</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Bratislavsk&#xfd; kraj</name>
			<code>BL</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Nitriansky kraj</name>
			<code>NI</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Pre&#x161;ovsk&#xfd; kraj</name>
			<code>PV</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Trenciansky kraj</name>
			<code>TC</code>
			<fips></fips>
			<region></region>
		</division>
		<num>703</num>
	</territory>
	<territory>
		<name>Slovenia</name>
		<alpha2>SI</alpha2>
		<alpha3>SVN</alpha3>
		<division>
			<name>Race-Fram</name>
			<code>098</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Hodo&#x161;</name>
			<code>161</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Lo&#x161;ki Potok</name>
			<code>066</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Bohinj</name>
			<code>004</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Puconci</name>
			<code>097</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Cankova</name>
			<code>152</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>&#x17d;u&#x17e;emberk</name>
			<code>193</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Kobarid</name>
			<code>046</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Naklo</name>
			<code>082</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Ptuj</name>
			<code>096</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>&#x17d;alec</name>
			<code>190</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Vipava</name>
			<code>136</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Ti&#x161;ina</name>
			<code>010</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Cerkvenjak</name>
			<code>153</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Medvode</name>
			<code>071</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Me&#x17e;ica</name>
			<code>074</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Ljutomer</name>
			<code>063</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Dobrovnik</name>
			<code>156</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Sodra&#x17e;ica</name>
			<code>179</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Crnomelj</name>
			<code>017</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Mirna Pec</name>
			<code>170</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Pivka</name>
			<code>091</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Murska Sobota</name>
			<code>080</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Kostel</name>
			<code>165</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Horjul</name>
			<code>162</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Ru&#x161;e</name>
			<code>108</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>&#x160;kofljica</name>
			<code>123</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>&#x160;entilj</name>
			<code>118</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Trbovlje</name>
			<code>129</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Ig</name>
			<code>037</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Trebnje</name>
			<code>130</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Brezovica</name>
			<code>008</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>&#x160;kofja Loka</name>
			<code>122</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Ormo&#x17e;</name>
			<code>087</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Moravske Toplice</name>
			<code>078</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Velike La&#x161;ce</name>
			<code>134</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Luce</name>
			<code>067</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Duplek</name>
			<code>026</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Cerknica</name>
			<code>013</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Moravce</name>
			<code>077</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Logatec</name>
			<code>064</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Dornava</name>
			<code>024</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Videm</name>
			<code>135</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Miklav&#x17e; na Dravskem polju</name>
			<code>169</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Slovenske Konjice</name>
			<code>114</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Nova Gorica</name>
			<code>084</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Lendava</name>
			<code>059</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Ivancna Gorica</name>
			<code>039</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Borovnica</name>
			<code>005</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Hrastnik</name>
			<code>034</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Podlehnik</name>
			<code>172</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Litija</name>
			<code>060</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Kri&#x17e;evci</name>
			<code>166</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Kranjska Gora</name>
			<code>053</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Solcava</name>
			<code>180</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Kungota</name>
			<code>055</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Benedikt</name>
			<code>148</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>&#x17d;irovnica</name>
			<code>192</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Sveta Ana</name>
			<code>181</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Slovenska Bistrica</name>
			<code>113</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Celje</name>
			<code>011</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Zavrc</name>
			<code>143</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Oplotnica</name>
			<code>171</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Ljubljana</name>
			<code>061</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Ribnica na Pohorju</name>
			<code>177</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Ribnica</name>
			<code>104</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Se&#x17e;ana</name>
			<code>111</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Bloke</name>
			<code>150</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Metlika</name>
			<code>073</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Koper</name>
			<code>050</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Radenci</name>
			<code>100</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Dobje</name>
			<code>154</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Vransko</name>
			<code>189</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>&#x17d;etale</name>
			<code>191</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Ljubno</name>
			<code>062</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Tr&#x17e;ic</name>
			<code>131</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Dravograd</name>
			<code>025</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Lo&#x161;ka dolina</name>
			<code>065</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Slovenj Gradec</name>
			<code>112</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Trzin</name>
			<code>186</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Razkri&#x17e;je</name>
			<code>176</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Ajdov&#x161;cina</name>
			<code>001</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Vitanje</name>
			<code>137</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Nazarje</name>
			<code>083</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Hoce-Slivnica</name>
			<code>160</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>&#x17d;elezniki</name>
			<code>146</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>&#x160;martno pri Litiji</name>
			<code>194</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Idrija</name>
			<code>036</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Bistrica ob Sotli</name>
			<code>149</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Komenda</name>
			<code>164</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Divaca</name>
			<code>019</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Gornja Radgona</name>
			<code>029</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Odranci</name>
			<code>086</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Cerkno</name>
			<code>014</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Vrhnika</name>
			<code>140</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Prebold</name>
			<code>174</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>&#x160;martno ob Paki</name>
			<code>125</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>&#x160;alovci</name>
			<code>033</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Hajdina</name>
			<code>159</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Menge&#x161;</name>
			<code>072</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Vodice</name>
			<code>138</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Kuzma</name>
			<code>056</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Beltinci</name>
			<code>002</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Destrnik</name>
			<code>018</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Kranj</name>
			<code>052</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>&#x160;marje pri Jel&#x161;ah</name>
			<code>124</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Gorenja vas-Poljane</name>
			<code>027</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Radlje ob Dravi</name>
			<code>101</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Radece</name>
			<code>099</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Osilnica</name>
			<code>088</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>&#x160;entjernej</name>
			<code>119</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Preddvor</name>
			<code>095</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Dol pri Ljubljani</name>
			<code>022</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Grad</name>
			<code>158</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Pesnica</name>
			<code>089</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Dolenjske Toplice</name>
			<code>157</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Ilirska Bistrica</name>
			<code>038</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Dom&#x17e;ale</name>
			<code>023</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Podvelka</name>
			<code>093</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Jur&#x161;inci</name>
			<code>042</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Vojnik</name>
			<code>139</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Velika Polana</name>
			<code>187</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Trnovska vas</name>
			<code>185</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Sveti Andra&#x17e; v Slovenskih goricah</name>
			<code>182</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Brda</name>
			<code>007</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Kozje</name>
			<code>051</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Hrpelje-Kozina</name>
			<code>035</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Turni&#x161;ce</name>
			<code>132</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>&#x160;entjur pri Celju</name>
			<code>120</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Kidricevo</name>
			<code>045</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Zagorje ob Savi</name>
			<code>142</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Gornji Petrovci</name>
			<code>031</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Zrece</name>
			<code>144</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Cren&#x161;ovci</name>
			<code>015</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Maj&#x161;perk</name>
			<code>069</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Sevnica</name>
			<code>110</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Maribor</name>
			<code>070</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Muta</name>
			<code>081</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Novo mesto</name>
			<code>085</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>&#x160;kocjan</name>
			<code>121</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Crna na Koro&#x161;kem</name>
			<code>016</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>&#x160;tore</name>
			<code>127</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Prevalje</name>
			<code>175</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Braslovce</name>
			<code>151</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Radovljica</name>
			<code>102</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Velenje</name>
			<code>133</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Ravne na Koro&#x161;kem</name>
			<code>103</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Dobrna</name>
			<code>155</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Podcetrtek</name>
			<code>092</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>&#x160;empeter-Vrtojba</name>
			<code>183</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Star&#x161;e</name>
			<code>115</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Polzela</name>
			<code>173</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Sveti Jurij</name>
			<code>116</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Kamnik</name>
			<code>043</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Jesenice</name>
			<code>041</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Cerklje na Gorenjskem</name>
			<code>012</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Bovec</name>
			<code>006</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Tolmin</name>
			<code>128</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Roga&#x161;ka Slatina</name>
			<code>106</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Komen</name>
			<code>049</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>&#x17d;iri</name>
			<code>147</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Miren-Kostanjevica</name>
			<code>075</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>La&#x161;ko</name>
			<code>057</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Piran</name>
			<code>090</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Gori&#x161;nica</name>
			<code>028</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Bre&#x17e;ice</name>
			<code>009</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Kr&#x161;ko</name>
			<code>054</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Mislinja</name>
			<code>076</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Mozirje</name>
			<code>079</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Roga&#x161;ovci</name>
			<code>105</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Lukovica</name>
			<code>068</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Lovrenc na Pohorju</name>
			<code>167</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Kanal</name>
			<code>044</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Kocevje</name>
			<code>048</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Jezersko</name>
			<code>163</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Bled</name>
			<code>003</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>&#x160;o&#x161;tanj</name>
			<code>126</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Tabor</name>
			<code>184</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Izola</name>
			<code>040</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Dobrova-Polhov Gradec</name>
			<code>021</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Kobilje</name>
			<code>047</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Semic</name>
			<code>109</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Vuzenica</name>
			<code>141</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Grosuplje</name>
			<code>032</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Gornji Grad</name>
			<code>030</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Lenart</name>
			<code>058</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>&#x160;encur</name>
			<code>117</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Dobrepolje</name>
			<code>020</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Selnica ob Dravi</name>
			<code>178</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Postojna</name>
			<code>094</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Ver&#x17e;ej</name>
			<code>188</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Rogatec</name>
			<code>107</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Markovci</name>
			<code>168</code>
			<fips></fips>
			<region></region>
		</division>
		<num>705</num>
	</territory>
	<territory>
		<name>Solomon Islands</name>
		<alpha2>SB</alpha2>
		<alpha3>SLB</alpha3>
		<division>
			<name>Guadalcanal</name>
			<code>GU</code>
			<fips>06</fips>
			<region></region>
		</division>
		<division>
			<name>Central</name>
			<code>CE</code>
			<fips>05</fips>
			<region></region>
		</division>
		<division>
			<name>Makira</name>
			<code>MK</code>
			<fips>08</fips>
			<region></region>
		</division>
		<division>
			<name>Malaita</name>
			<code>ML</code>
			<fips>03</fips>
			<region></region>
		</division>
		<division>
			<name>Temotu</name>
			<code>TE</code>
			<fips>09</fips>
			<region></region>
		</division>
		<division>
			<name>Capital Territory (Honiara)</name>
			<code>CT</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Isabel</name>
			<code>IS</code>
			<fips>07</fips>
			<region></region>
		</division>
		<division>
			<name>Western</name>
			<code>WE</code>
			<fips>04</fips>
			<region></region>
		</division>
		<num>090</num>
	</territory>
	<territory>
		<name>Somalia</name>
		<alpha2>SO</alpha2>
		<alpha3>SOM</alpha3>
		<division>
			<name>Gedo</name>
			<code>GE</code>
			<fips>06</fips>
			<region></region>
		</division>
		<division>
			<name>Jubbada Dhexe</name>
			<code>JD</code>
			<fips>08</fips>
			<region></region>
		</division>
		<division>
			<name>Woqooyi Galbeed</name>
			<code>WO</code>
			<fips>16</fips>
			<region></region>
		</division>
		<division>
			<name>Mudug</name>
			<code>MU</code>
			<fips>10</fips>
			<region></region>
		</division>
		<division>
			<name>Bakool</name>
			<code>BK</code>
			<fips>01</fips>
			<region></region>
		</division>
		<division>
			<name>Togdheer</name>
			<code>TO</code>
			<fips>15</fips>
			<region></region>
		</division>
		<division>
			<name>Sool</name>
			<code>SO</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Nugaal</name>
			<code>NU</code>
			<fips>11</fips>
			<region></region>
		</division>
		<division>
			<name>Awdal</name>
			<code>AW</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Banaadir</name>
			<code>BN</code>
			<fips>02</fips>
			<region></region>
		</division>
		<division>
			<name>Sanaag</name>
			<code>SA</code>
			<fips>12</fips>
			<region></region>
		</division>
		<division>
			<name>Bay</name>
			<code>BY</code>
			<fips>04</fips>
			<region></region>
		</division>
		<division>
			<name>Jubbada Hoose</name>
			<code>JH</code>
			<fips>09</fips>
			<region></region>
		</division>
		<division>
			<name>Bari</name>
			<code>BR</code>
			<fips>03</fips>
			<region></region>
		</division>
		<division>
			<name>Galguduud</name>
			<code>GA</code>
			<fips>05</fips>
			<region></region>
		</division>
		<division>
			<name>Hiiraan</name>
			<code>HI</code>
			<fips>07</fips>
			<region></region>
		</division>
		<division>
			<name>Shabeellaha Hoose</name>
			<code>SH</code>
			<fips>14</fips>
			<region></region>
		</division>
		<division>
			<name>Shabeellaha Dhexe</name>
			<code>SD</code>
			<fips>13</fips>
			<region></region>
		</division>
		<num>706</num>
	</territory>
	<territory>
		<name>South Africa</name>
		<alpha2>ZA</alpha2>
		<alpha3>ZAF</alpha3>
		<division>
			<name>Eastern Cape</name>
			<code>EC</code>
			<fips>05</fips>
			<region></region>
		</division>
		<division>
			<name>Free State</name>
			<code>FS</code>
			<fips>03</fips>
			<region></region>
		</division>
		<division>
			<name>Gauteng</name>
			<code>GT</code>
			<fips>06</fips>
			<region></region>
		</division>
		<division>
			<name>Kwazulu-Natal</name>
			<code>NL</code>
			<fips>02</fips>
			<region></region>
		</division>
		<division>
			<name>Western Cape</name>
			<code>WC</code>
			<fips>11</fips>
			<region></region>
		</division>
		<division>
			<name>North-West</name>
			<code>NW</code>
			<fips>10</fips>
			<region></region>
		</division>
		<division>
			<name>Mpumalanga</name>
			<code>MP</code>
			<fips>07</fips>
			<region></region>
		</division>
		<division>
			<name>Northern Cape</name>
			<code>NC</code>
			<fips>08</fips>
			<region></region>
		</division>
		<division>
			<name>Northern Province</name>
			<code>NP</code>
			<fips>09</fips>
			<region></region>
		</division>
		<num>710</num>
	</territory>
	<territory>
		<name>South Georgia and South Sandwich Islands</name>
		<alpha2>GS</alpha2>
		<alpha3>SGS</alpha3>
		<num>239</num>
	</territory>
	<territory>
		<name>Spain</name>
		<alpha2>ES</alpha2>
		<alpha3>ESP</alpha3>
		<division>
			<name>Valladolid</name>
			<code>VA</code>
			<fips></fips>
			<region>CL</region>
		</division>
		<division>
			<name>Badajoz</name>
			<code>BA</code>
			<fips></fips>
			<region>EX</region>
		</division>
		<division>
			<name>Huelva</name>
			<code>H</code>
			<fips></fips>
			<region>AN</region>
		</division>
		<division>
			<name>Girona</name>
			<code>GI</code>
			<fips></fips>
			<region>CT</region>
		</division>
		<division>
			<name>Cuenca</name>
			<code>CU</code>
			<fips></fips>
			<region>CM</region>
		</division>
		<division>
			<name>Zaragoza</name>
			<code>Z</code>
			<fips></fips>
			<region>AR</region>
		</division>
		<division>
			<name>Cantabria</name>
			<code>S</code>
			<fips>39</fips>
			<region>S</region>
		</division>
		<division>
			<name>Ciudad Real</name>
			<code>CR</code>
			<fips></fips>
			<region>CM</region>
		</division>
		<division>
			<name>Burgos</name>
			<code>BU</code>
			<fips></fips>
			<region>CL</region>
		</division>
		<division>
			<name>Granada</name>
			<code>GR</code>
			<fips></fips>
			<region>AN</region>
		</division>
		<division>
			<name>Segovia</name>
			<code>SG</code>
			<fips></fips>
			<region>CL</region>
		</division>
		<division>
			<name>Palencia</name>
			<code>P</code>
			<fips></fips>
			<region>CL</region>
		</division>
		<division>
			<name>Alicante</name>
			<code>A</code>
			<fips></fips>
			<region>VC</region>
		</division>
		<division>
			<name>Le&#xf3;n</name>
			<code>LE</code>
			<fips></fips>
			<region>CL</region>
		</division>
		<division>
			<name>Asturias</name>
			<code>O</code>
			<fips>34</fips>
			<region>O</region>
		</division>
		<division>
			<name>Baleares</name>
			<code>PM</code>
			<fips>07</fips>
			<region>IB</region>
		</division>
		<division>
			<name>Santa Cruz de Tenerife</name>
			<code>TF</code>
			<fips></fips>
			<region>CN</region>
		</division>
		<division>
			<name>Ourense</name>
			<code>OR</code>
			<fips></fips>
			<region>GA</region>
		</division>
		<division>
			<name>Lleida</name>
			<code>L</code>
			<fips></fips>
			<region>CT</region>
		</division>
		<division>
			<name>Madrid</name>
			<code>M</code>
			<fips>29</fips>
			<region>M</region>
		</division>
		<division>
			<name>Salamanca</name>
			<code>SA</code>
			<fips></fips>
			<region>CL</region>
		</division>
		<division>
			<name>Tarragona</name>
			<code>T</code>
			<fips></fips>
			<region>CT</region>
		</division>
		<division>
			<name>Melilla</name>
			<code>ML</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>La Rioja</name>
			<code>LO</code>
			<fips>27</fips>
			<region>LO</region>
		</division>
		<division>
			<name>&#xc1;vila</name>
			<code>AV</code>
			<fips></fips>
			<region>CL</region>
		</division>
		<division>
			<name>Toledo</name>
			<code>TO</code>
			<fips></fips>
			<region>CM</region>
		</division>
		<division>
			<name>C&#xe1;ceres</name>
			<code>CC</code>
			<fips></fips>
			<region>EX</region>
		</division>
		<division>
			<name>Barcelona</name>
			<code>B</code>
			<fips></fips>
			<region>CT</region>
		</division>
		<division>
			<name>Sevilla</name>
			<code>SE</code>
			<fips></fips>
			<region>AN</region>
		</division>
		<division>
			<name>Las Palmas</name>
			<code>GC</code>
			<fips></fips>
			<region>CN</region>
		</division>
		<division>
			<name>Teruel</name>
			<code>TE</code>
			<fips></fips>
			<region>AR</region>
		</division>
		<division>
			<name>Guip&#xfa;zcoa</name>
			<code>SS</code>
			<fips></fips>
			<region>PV</region>
		</division>
		<division>
			<name>Vizcaya</name>
			<code>BI</code>
			<fips></fips>
			<region>PV</region>
		</division>
		<division>
			<name>A Coru&#xf1;a</name>
			<code>C</code>
			<fips></fips>
			<region>GA</region>
		</division>
		<division>
			<name>Almer&#xed;a</name>
			<code>AL</code>
			<fips></fips>
			<region>AN</region>
		</division>
		<division>
			<name>Castell&#xf3;n</name>
			<code>CS</code>
			<fips></fips>
			<region>VC</region>
		</division>
		<division>
			<name>Ja&#xe9;n</name>
			<code>J</code>
			<fips></fips>
			<region>AN</region>
		</division>
		<division>
			<name>Albacete</name>
			<code>AB</code>
			<fips></fips>
			<region>CM</region>
		</division>
		<division>
			<name>Murcia</name>
			<code>MU</code>
			<fips>31</fips>
			<region>MU</region>
		</division>
		<division>
			<name>Soria</name>
			<code>SO</code>
			<fips></fips>
			<region>CL</region>
		</division>
		<division>
			<name>Ceuta</name>
			<code>CE</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Pontevedra</name>
			<code>PO</code>
			<fips></fips>
			<region>GA</region>
		</division>
		<division>
			<name>&#xc1;lava</name>
			<code>VI</code>
			<fips></fips>
			<region>PV</region>
		</division>
		<division>
			<name>Navarra</name>
			<code>NA</code>
			<fips>32</fips>
			<region>NA</region>
		</division>
		<division>
			<name>M&#xe1;laga</name>
			<code>MA</code>
			<fips></fips>
			<region>AN</region>
		</division>
		<division>
			<name>Zamora</name>
			<code>ZA</code>
			<fips></fips>
			<region>CL</region>
		</division>
		<division>
			<name>C&#xe1;diz</name>
			<code>CA</code>
			<fips></fips>
			<region>AN</region>
		</division>
		<division>
			<name>Lugo</name>
			<code>LU</code>
			<fips></fips>
			<region>GA</region>
		</division>
		<division>
			<name>Valencia</name>
			<code>V</code>
			<fips>60</fips>
			<region>VC</region>
		</division>
		<division>
			<name>Huesca</name>
			<code>HU</code>
			<fips></fips>
			<region>AR</region>
		</division>
		<division>
			<name>Guadalajara</name>
			<code>GU</code>
			<fips></fips>
			<region>CM</region>
		</division>
		<division>
			<name>C&#xf3;rdoba</name>
			<code>CO</code>
			<fips></fips>
			<region>AN</region>
		</division>
		<num>724</num>
	</territory>
	<territory>
		<name>Sri Lanka</name>
		<alpha2>LK</alpha2>
		<alpha3>LKA</alpha3>
		<division>
			<name>Anuradhapura</name>
			<code>71</code>
			<fips></fips>
			<region>7</region>
		</division>
		<division>
			<name>Mullaittivu</name>
			<code>45</code>
			<fips></fips>
			<region>4</region>
		</division>
		<division>
			<name>Mannar</name>
			<code>43</code>
			<fips></fips>
			<region>4</region>
		</division>
		<division>
			<name>Kandy</name>
			<code>21</code>
			<fips></fips>
			<region>2</region>
		</division>
		<division>
			<name>Trincomalee</name>
			<code>53</code>
			<fips></fips>
			<region>5</region>
		</division>
		<division>
			<name>Jaffna</name>
			<code>41</code>
			<fips></fips>
			<region>4</region>
		</division>
		<division>
			<name>Galle</name>
			<code>31</code>
			<fips></fips>
			<region>3</region>
		</division>
		<division>
			<name>Matale</name>
			<code>22</code>
			<fips></fips>
			<region>2</region>
		</division>
		<division>
			<name>Monaragala</name>
			<code>82</code>
			<fips></fips>
			<region>8</region>
		</division>
		<division>
			<name>Ampara</name>
			<code>52</code>
			<fips></fips>
			<region>5</region>
		</division>
		<division>
			<name>Nuwara Eliya</name>
			<code>23</code>
			<fips></fips>
			<region>2</region>
		</division>
		<division>
			<name>Vavuniya</name>
			<code>44</code>
			<fips></fips>
			<region>4</region>
		</division>
		<division>
			<name>Polonnaruwa</name>
			<code>72</code>
			<fips></fips>
			<region>7</region>
		</division>
		<division>
			<name>Badulla</name>
			<code>81</code>
			<fips></fips>
			<region>8</region>
		</division>
		<division>
			<name>Batticaloa</name>
			<code>51</code>
			<fips></fips>
			<region>5</region>
		</division>
		<division>
			<name>Kalutara</name>
			<code>13</code>
			<fips></fips>
			<region>1</region>
		</division>
		<division>
			<name>Matara</name>
			<code>32</code>
			<fips></fips>
			<region>3</region>
		</division>
		<division>
			<name>Puttalam</name>
			<code>62</code>
			<fips></fips>
			<region>6</region>
		</division>
		<division>
			<name>Colombo</name>
			<code>11</code>
			<fips></fips>
			<region>1</region>
		</division>
		<division>
			<name>Kilinochchi</name>
			<code>42</code>
			<fips></fips>
			<region>4</region>
		</division>
		<division>
			<name>Kurunegala</name>
			<code>61</code>
			<fips></fips>
			<region>6</region>
		</division>
		<division>
			<name>Kegalla</name>
			<code>92</code>
			<fips></fips>
			<region>9</region>
		</division>
		<division>
			<name>Ratnapura</name>
			<code>91</code>
			<fips></fips>
			<region>9</region>
		</division>
		<division>
			<name>Hambantota</name>
			<code>33</code>
			<fips></fips>
			<region>3</region>
		</division>
		<division>
			<name>Gampaha</name>
			<code>12</code>
			<fips></fips>
			<region>1</region>
		</division>
		<num>144</num>
	</territory>
	<territory>
		<name>Sudan</name>
		<alpha2>SD</alpha2>
		<alpha3>SDN</alpha3>
		<division>
			<name>Ash Shamaliyah</name>
			<code>01</code>
			<fips>43</fips>
			<region></region>
		</division>
		<division>
			<name>Ahali an Nil</name>
			<code>23</code>
			<fips>35</fips>
			<region></region>
		</division>
		<division>
			<name>Janub Kurdufan</name>
			<code>13</code>
			<fips>50</fips>
			<region></region>
		</division>
		<division>
			<name>Kassala</name>
			<code>05</code>
			<fips>52</fips>
			<region></region>
		</division>
		<division>
			<name>Shamal Kurdufan</name>
			<code>09</code>
			<fips>56</fips>
			<region></region>
		</division>
		<division>
			<name>An Nil al Abyah</name>
			<code>08</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Warab</name>
			<code>21</code>
			<fips>59</fips>
			<region></region>
		</division>
		<division>
			<name>Sinnar</name>
			<code>25</code>
			<fips>58</fips>
			<region></region>
		</division>
		<division>
			<name>Shamal Bahr al Ghazal</name>
			<code>15</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Junqali</name>
			<code>20</code>
			<fips>51</fips>
			<region></region>
		</division>
		<division>
			<name>Al Jazirah</name>
			<code>07</code>
			<fips>38</fips>
			<region></region>
		</division>
		<division>
			<name>Al Khartum</name>
			<code>03</code>
			<fips>29</fips>
			<region></region>
		</division>
		<division>
			<name>Gharb Darfur</name>
			<code>12</code>
			<fips>47</fips>
			<region></region>
		</division>
		<division>
			<name>Al Wahdah</name>
			<code>22</code>
			<fips>40</fips>
			<region></region>
		</division>
		<division>
			<name>Gharb Bahr al Ghazal</name>
			<code>14</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Gharb Kurdufan</name>
			<code>10</code>
			<fips>48</fips>
			<region></region>
		</division>
		<division>
			<name>Al Bahr al Ahmar</name>
			<code>26</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Al Qaharif</name>
			<code>06</code>
			<fips>39</fips>
			<region></region>
		</division>
		<division>
			<name>Central Equatoria</name>
			<code>17</code>
			<fips>44</fips>
			<region></region>
		</division>
		<division>
			<name>Gharb al Istiwa'iyah</name>
			<code>16</code>
			<fips>45</fips>
			<region></region>
		</division>
		<division>
			<name>Shamal Darfur</name>
			<code>02</code>
			<fips>55</fips>
			<region></region>
		</division>
		<division>
			<name>An Nil</name>
			<code>04</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Janub Darfur</name>
			<code>11</code>
			<fips>49</fips>
			<region></region>
		</division>
		<division>
			<name>Al Buhayrat</name>
			<code>18</code>
			<fips>37</fips>
			<region></region>
		</division>
		<division>
			<name>An Nil al Azraq</name>
			<code>24</code>
			<fips>42</fips>
			<region></region>
		</division>
		<division>
			<name>Sharq al Istiwa'iyah</name>
			<code>19</code>
			<fips>57</fips>
			<region></region>
		</division>
		<num>736</num>
	</territory>
	<territory>
		<name>Suriname</name>
		<alpha2>SR</alpha2>
		<alpha3>SUR</alpha3>
		<division>
			<name>Saramacca</name>
			<code>SA</code>
			<fips>17</fips>
			<region></region>
		</division>
		<division>
			<name>Commewijne</name>
			<code>CM</code>
			<fips>11</fips>
			<region></region>
		</division>
		<division>
			<name>Paramaribo</name>
			<code>PM</code>
			<fips>16</fips>
			<region></region>
		</division>
		<division>
			<name>Para</name>
			<code>PR</code>
			<fips>15</fips>
			<region></region>
		</division>
		<division>
			<name>Coronie</name>
			<code>CR</code>
			<fips>12</fips>
			<region></region>
		</division>
		<division>
			<name>Nickerie</name>
			<code>NI</code>
			<fips>14</fips>
			<region></region>
		</division>
		<division>
			<name>Marowijne</name>
			<code>MA</code>
			<fips>13</fips>
			<region></region>
		</division>
		<division>
			<name>Wanica</name>
			<code>WA</code>
			<fips>19</fips>
			<region></region>
		</division>
		<division>
			<name>Brokopondo</name>
			<code>BR</code>
			<fips>10</fips>
			<region></region>
		</division>
		<division>
			<name>Sipaliwini</name>
			<code>SI</code>
			<fips>18</fips>
			<region></region>
		</division>
		<num>740</num>
	</territory>
	<territory>
		<name>Svalbard and Jan Mayen Islands</name>
		<alpha2>SJ</alpha2>
		<alpha3>SJM</alpha3>
		<num>744</num>
	</territory>
	<territory>
		<name>Swaziland</name>
		<alpha2>SZ</alpha2>
		<alpha3>SWZ</alpha3>
		<division>
			<name>Manzini</name>
			<code>MA</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Hhohho</name>
			<code>HH</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Lubombo</name>
			<code>LU</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Shiselweni</name>
			<code>SH</code>
			<fips></fips>
			<region></region>
		</division>
		<num>748</num>
	</territory>
	<territory>
		<name>Sweden</name>
		<alpha2>SE</alpha2>
		<alpha3>SWE</alpha3>
		<division>
			<name>Uppsala l&#xe4;n</name>
			<code>C</code>
			<fips>21</fips>
			<region>&#xd6;stra Mellansverige</region>
		</division>
		<division>
			<name>Kronobergs l&#xe4;n</name>
			<code>G</code>
			<fips>12</fips>
			<region>Sm&#xe5;land med &#xf6;arna</region>
		</division>
		<division>
			<name>G&#xe4;vleborgs l&#xe4;n</name>
			<code>X</code>
			<fips>03</fips>
			<region>Norra Mellansverige</region>
		</division>
		<division>
			<name>Dalarnas l&#xe4;n</name>
			<code>W</code>
			<fips>10</fips>
			<region>Norra Mellansverige</region>
		</division>
		<division>
			<name>&#xd6;sterg&#xf6;tlands l&#xe4;n</name>
			<code>E</code>
			<fips>16</fips>
			<region>&#xd6;stra Mellansverige</region>
		</division>
		<division>
			<name>Gotlands l&#xe4;n</name>
			<code>I</code>
			<fips>05</fips>
			<region>Sm&#xe5;land med &#xf6;arna</region>
		</division>
		<division>
			<name>Stockholms l&#xe4;n</name>
			<code>AB</code>
			<fips>26</fips>
			<region>Stockholm</region>
		</division>
		<division>
			<name>V&#xe4;stmanlands l&#xe4;n</name>
			<code>U</code>
			<fips>25</fips>
			<region>&#xd6;stra Mellansverige</region>
		</division>
		<division>
			<name>Sk&#xe5;ne l&#xe4;n</name>
			<code>M</code>
			<fips>27</fips>
			<region>Sydsverige</region>
		</division>
		<division>
			<name>V&#xe4;sternorrlands l&#xe4;n</name>
			<code>Y</code>
			<fips>24</fips>
			<region>Mellersta Norrland</region>
		</division>
		<division>
			<name>Norrbottens l&#xe4;n</name>
			<code>BD</code>
			<fips>14</fips>
			<region>&#xd6;vre Norrland</region>
		</division>
		<division>
			<name>V&#xe4;rmlands l&#xe4;n</name>
			<code>S</code>
			<fips>22</fips>
			<region>Norra Mellansverige</region>
		</division>
		<division>
			<name>V&#xe4;sterbottens l&#xe4;n</name>
			<code>AC</code>
			<fips>23</fips>
			<region>&#xd6;vre Norrland</region>
		</division>
		<division>
			<name>&#xd6;rebro l&#xe4;n</name>
			<code>T</code>
			<fips>15</fips>
			<region>&#xd6;stra Mellansverige</region>
		</division>
		<division>
			<name>Blekinge l&#xe4;n</name>
			<code>K</code>
			<fips>02</fips>
			<region>Sydsverige</region>
		</division>
		<division>
			<name>Hallands l&#xe4;n</name>
			<code>N</code>
			<fips>06</fips>
			<region>V&#xe4;stsverige</region>
		</division>
		<division>
			<name>J&#xe4;mtlands l&#xe4;n</name>
			<code>Z</code>
			<fips>07</fips>
			<region>Mellersta Norrland</region>
		</division>
		<division>
			<name>Kalmar l&#xe4;n</name>
			<code>H</code>
			<fips>09</fips>
			<region>Sm&#xe5;land med &#xf6;arna</region>
		</division>
		<division>
			<name>J&#xf6;nk&#xf6;pings l&#xe4;n</name>
			<code>F</code>
			<fips>08</fips>
			<region>Sm&#xe5;land med &#xf6;arna</region>
		</division>
		<division>
			<name>V&#xe4;stra G&#xf6;talands l&#xe4;n</name>
			<code>O</code>
			<fips></fips>
			<region>V&#xe4;stsverige</region>
		</division>
		<division>
			<name>S&#xf6;dermanlands l&#xe4;n</name>
			<code>D</code>
			<fips>18</fips>
			<region>&#xd6;stra Mellansverige</region>
		</division>
		<num>752</num>
	</territory>
	<territory>
		<name>Switzerland</name>
		<alpha2>CH</alpha2>
		<alpha3>CHE</alpha3>
		<division>
			<name>Schaffhausen</name>
			<code>SH</code>
			<fips>16</fips>
			<region></region>
		</division>
		<division>
			<name>Z&#xfc;rich</name>
			<code>ZH</code>
			<fips>25</fips>
			<region></region>
		</division>
		<division>
			<name>Appenzell Ausserrhoden</name>
			<code>AR</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Basel-Landschaft</name>
			<code>BL</code>
			<fips>03</fips>
			<region></region>
		</division>
		<division>
			<name>Jura</name>
			<code>JU</code>
			<fips>26</fips>
			<region></region>
		</division>
		<division>
			<name>Nidwalden</name>
			<code>NW</code>
			<fips>13</fips>
			<region></region>
		</division>
		<division>
			<name>Gen&#xe8;ve</name>
			<code>GE</code>
			<fips>07</fips>
			<region></region>
		</division>
		<division>
			<name>Zug</name>
			<code>ZG</code>
			<fips>24</fips>
			<region></region>
		</division>
		<division>
			<name>Thurgau</name>
			<code>TG</code>
			<fips>19</fips>
			<region></region>
		</division>
		<division>
			<name>Schwyz</name>
			<code>SZ</code>
			<fips>17</fips>
			<region></region>
		</division>
		<division>
			<name>Fribourg</name>
			<code>FR</code>
			<fips>06</fips>
			<region></region>
		</division>
		<division>
			<name>Basel-Stadt</name>
			<code>BS</code>
			<fips>04</fips>
			<region></region>
		</division>
		<division>
			<name>Valais</name>
			<code>VS</code>
			<fips>22</fips>
			<region></region>
		</division>
		<division>
			<name>Graub&#xfc;nden</name>
			<code>GR</code>
			<fips>09</fips>
			<region></region>
		</division>
		<division>
			<name>Luzern</name>
			<code>LU</code>
			<fips>11</fips>
			<region></region>
		</division>
		<division>
			<name>Sankt Gallen</name>
			<code>SG</code>
			<fips>15</fips>
			<region></region>
		</division>
		<division>
			<name>Vaud</name>
			<code>VD</code>
			<fips>23</fips>
			<region></region>
		</division>
		<division>
			<name>Appenzell Innerrhoden</name>
			<code>AI</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Glarus</name>
			<code>GL</code>
			<fips>08</fips>
			<region></region>
		</division>
		<division>
			<name>Ticino</name>
			<code>TI</code>
			<fips>20</fips>
			<region></region>
		</division>
		<division>
			<name>Neuch&#xe2;tel</name>
			<code>NE</code>
			<fips>12</fips>
			<region></region>
		</division>
		<division>
			<name>Aargau</name>
			<code>AG</code>
			<fips>01</fips>
			<region></region>
		</division>
		<division>
			<name>Obwalden</name>
			<code>OW</code>
			<fips>14</fips>
			<region></region>
		</division>
		<division>
			<name>Uri</name>
			<code>UR</code>
			<fips>21</fips>
			<region></region>
		</division>
		<division>
			<name>Solothurn</name>
			<code>SO</code>
			<fips>18</fips>
			<region></region>
		</division>
		<division>
			<name>Bern</name>
			<code>BE</code>
			<fips>05</fips>
			<region></region>
		</division>
		<num>756</num>
	</territory>
	<territory>
		<name>Syria</name>
		<alpha2>SY</alpha2>
		<alpha3>SYR</alpha3>
		<division>
			<name>Dimashq</name>
			<code>DI</code>
			<fips>13</fips>
			<region></region>
		</division>
		<division>
			<name>Al Hasakah</name>
			<code>HA</code>
			<fips>01</fips>
			<region></region>
		</division>
		<division>
			<name>Rif Dimashq</name>
			<code>RD</code>
			<fips>08</fips>
			<region></region>
		</division>
		<division>
			<name>Dayr az Zawr</name>
			<code>DY</code>
			<fips>07</fips>
			<region></region>
		</division>
		<division>
			<name>Al Ladhiqiyah</name>
			<code>LA</code>
			<fips>02</fips>
			<region></region>
		</division>
		<division>
			<name>Hamah</name>
			<code>HM</code>
			<fips>10</fips>
			<region></region>
		</division>
		<division>
			<name>Darha</name>
			<code>DR</code>
			<fips>06</fips>
			<region></region>
		</division>
		<division>
			<name>Idlib</name>
			<code>ID</code>
			<fips>12</fips>
			<region></region>
		</division>
		<division>
			<name>Ar Raqqah</name>
			<code>RA</code>
			<fips>04</fips>
			<region></region>
		</division>
		<division>
			<name>Halab</name>
			<code>HL</code>
			<fips>09</fips>
			<region></region>
		</division>
		<division>
			<name>As Suwayda'</name>
			<code>SU</code>
			<fips>05</fips>
			<region></region>
		</division>
		<division>
			<name>Hims</name>
			<code>HI</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Al Qunaytirah</name>
			<code>QU</code>
			<fips>03</fips>
			<region></region>
		</division>
		<division>
			<name>Tartus</name>
			<code>TA</code>
			<fips>14</fips>
			<region></region>
		</division>
		<num>760</num>
	</territory>
	<territory>
		<name>Taiwan</name>
		<alpha2>TW</alpha2>
		<alpha3>TWN</alpha3>
		<division>
			<name>Miaoli</name>
			<code>MIA</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Kaohsiung</name>
			<code>KHQ</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Hualien</name>
			<code>HUA</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Ilan</name>
			<code>ILA</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Taichung</name>
			<code>TXQ</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Penghu</name>
			<code>PEN</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Changhua</name>
			<code>CHA</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Yunlin</name>
			<code>YUN</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Pingtung</name>
			<code>PIF</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Chiayi</name>
			<code>CYQ</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Tainan</name>
			<code>TNQ</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Taitung</name>
			<code>TTT</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Nantou</name>
			<code>NAN</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Taoyuan</name>
			<code>TAO</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Keelung</name>
			<code>KEE</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Taipei</name>
			<code>TPQ</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Hsinchu</name>
			<code>HSQ</code>
			<fips></fips>
			<region></region>
		</division>
		<num>158</num>
	</territory>
	<territory>
		<name>Tajikistan</name>
		<alpha2>TJ</alpha2>
		<alpha3>TJK</alpha3>
		<division>
			<name>Sughd</name>
			<code>SU</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Khatlon</name>
			<code>KT</code>
			<fips>02</fips>
			<region></region>
		</division>
		<division>
			<name>Gorno-Badakhshan</name>
			<code>GB</code>
			<fips></fips>
			<region></region>
		</division>
		<num>762</num>
	</territory>
	<territory>
		<name>Tanzania</name>
		<alpha2>TZ</alpha2>
		<alpha3>TZA</alpha3>
		<division>
			<name>Manyara</name>
			<code>26</code>
			<fips></fips>
			<region></region>
		</division>
		<num>834</num>
	</territory>
	<territory>
		<name>Thailand</name>
		<alpha2>TH</alpha2>
		<alpha3>THA</alpha3>
		<division>
			<name>Nonthaburi</name>
			<code>12</code>
			<fips>38</fips>
			<region></region>
		</division>
		<division>
			<name>Phra Nakhon Si Ayutthaya</name>
			<code>14</code>
			<fips>36</fips>
			<region></region>
		</division>
		<division>
			<name>Phuket</name>
			<code>83</code>
			<fips>62</fips>
			<region></region>
		</division>
		<division>
			<name>Songkhla</name>
			<code>90</code>
			<fips>68</fips>
			<region></region>
		</division>
		<division>
			<name>Prachin Buri</name>
			<code>25</code>
			<fips>74</fips>
			<region></region>
		</division>
		<division>
			<name>Phangnga</name>
			<code>82</code>
			<fips>61</fips>
			<region></region>
		</division>
		<division>
			<name>Saraburi</name>
			<code>19</code>
			<fips>37</fips>
			<region></region>
		</division>
		<division>
			<name>Ratchaburi</name>
			<code>70</code>
			<fips>52</fips>
			<region></region>
		</division>
		<division>
			<name>Chai Nat</name>
			<code>18</code>
			<fips>32</fips>
			<region></region>
		</division>
		<division>
			<name>Roi Et</name>
			<code>45</code>
			<fips>25</fips>
			<region></region>
		</division>
		<division>
			<name>Surat Thani</name>
			<code>84</code>
			<fips>60</fips>
			<region></region>
		</division>
		<division>
			<name>Kanchanaburi</name>
			<code>71</code>
			<fips>50</fips>
			<region></region>
		</division>
		<division>
			<name>Mukdahan</name>
			<code>49</code>
			<fips>78</fips>
			<region></region>
		</division>
		<division>
			<name>Nong Khai</name>
			<code>43</code>
			<fips>17</fips>
			<region></region>
		</division>
		<division>
			<name>Chumphon</name>
			<code>86</code>
			<fips>58</fips>
			<region></region>
		</division>
		<division>
			<name>Pathum Thani</name>
			<code>13</code>
			<fips>39</fips>
			<region></region>
		</division>
		<division>
			<name>Udon Thani</name>
			<code>41</code>
			<fips>76</fips>
			<region></region>
		</division>
		<division>
			<name>Mae Hong Son</name>
			<code>58</code>
			<fips>01</fips>
			<region></region>
		</division>
		<division>
			<name>Maha Sarakham</name>
			<code>44</code>
			<fips>24</fips>
			<region></region>
		</division>
		<division>
			<name>Chiang Rai</name>
			<code>57</code>
			<fips>03</fips>
			<region></region>
		</division>
		<division>
			<name>Yasothon</name>
			<code>35</code>
			<fips>72</fips>
			<region></region>
		</division>
		<division>
			<name>Nakhon Nayok</name>
			<code>26</code>
			<fips>43</fips>
			<region></region>
		</division>
		<division>
			<name>Sakon Nakhon</name>
			<code>47</code>
			<fips>20</fips>
			<region></region>
		</division>
		<division>
			<name>Uttaradit</name>
			<code>53</code>
			<fips>10</fips>
			<region></region>
		</division>
		<division>
			<name>Nong Bua Lam Phu</name>
			<code>39</code>
			<fips>79</fips>
			<region></region>
		</division>
		<division>
			<name>Pattani</name>
			<code>94</code>
			<fips>69</fips>
			<region></region>
		</division>
		<division>
			<name>Phrae</name>
			<code>54</code>
			<fips>07</fips>
			<region></region>
		</division>
		<division>
			<name>Nakhon Phanom</name>
			<code>48</code>
			<fips>73</fips>
			<region></region>
		</division>
		<division>
			<name>Uthai Thani</name>
			<code>61</code>
			<fips>15</fips>
			<region></region>
		</division>
		<division>
			<name>Buri Ram</name>
			<code>31</code>
			<fips>28</fips>
			<region></region>
		</division>
		<division>
			<name>Phichit</name>
			<code>66</code>
			<fips>13</fips>
			<region></region>
		</division>
		<division>
			<name>Yala</name>
			<code>95</code>
			<fips>70</fips>
			<region></region>
		</division>
		<division>
			<name>Lamphun</name>
			<code>51</code>
			<fips>05</fips>
			<region></region>
		</division>
		<division>
			<name>Krabi</name>
			<code>81</code>
			<fips>63</fips>
			<region></region>
		</division>
		<division>
			<name>Surin</name>
			<code>32</code>
			<fips>29</fips>
			<region></region>
		</division>
		<division>
			<name>Khon Kaen</name>
			<code>40</code>
			<fips>22</fips>
			<region></region>
		</division>
		<division>
			<name>Phatthalung</name>
			<code>93</code>
			<fips>66</fips>
			<region></region>
		</division>
		<division>
			<name>Trat</name>
			<code>23</code>
			<fips>49</fips>
			<region></region>
		</division>
		<division>
			<name>Sing Buri</name>
			<code>17</code>
			<fips>33</fips>
			<region></region>
		</division>
		<division>
			<name>Chanthaburi</name>
			<code>22</code>
			<fips>48</fips>
			<region></region>
		</division>
		<division>
			<name>Narathiwat</name>
			<code>96</code>
			<fips>31</fips>
			<region></region>
		</division>
		<division>
			<name>Samut Prakan</name>
			<code>11</code>
			<fips>42</fips>
			<region></region>
		</division>
		<division>
			<name>Chachoengsao</name>
			<code>24</code>
			<fips>44</fips>
			<region></region>
		</division>
		<division>
			<name>Samut Songkhram</name>
			<code>75</code>
			<fips>54</fips>
			<region></region>
		</division>
		<division>
			<name>Tak</name>
			<code>63</code>
			<fips>08</fips>
			<region></region>
		</division>
		<division>
			<name>Satun</name>
			<code>91</code>
			<fips>67</fips>
			<region></region>
		</division>
		<division>
			<name>Phayao</name>
			<code>56</code>
			<fips>41</fips>
			<region></region>
		</division>
		<division>
			<name>Nan</name>
			<code>55</code>
			<fips>04</fips>
			<region></region>
		</division>
		<division>
			<name>Phatthaya</name>
			<code>S</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Ranong</name>
			<code>85</code>
			<fips>59</fips>
			<region></region>
		</division>
		<division>
			<name>Nakhon Ratchasima</name>
			<code>30</code>
			<fips>27</fips>
			<region></region>
		</division>
		<division>
			<name>Nakhon Pathom</name>
			<code>73</code>
			<fips>53</fips>
			<region></region>
		</division>
		<division>
			<name>Suphan Buri</name>
			<code>72</code>
			<fips>51</fips>
			<region></region>
		</division>
		<division>
			<name>Phitsanulok</name>
			<code>65</code>
			<fips>12</fips>
			<region></region>
		</division>
		<division>
			<name>Samut Sakhon</name>
			<code>74</code>
			<fips>55</fips>
			<region></region>
		</division>
		<division>
			<name>Sukhothai</name>
			<code>64</code>
			<fips>09</fips>
			<region></region>
		</division>
		<division>
			<name>Si Sa Ket</name>
			<code>33</code>
			<fips>30</fips>
			<region></region>
		</division>
		<division>
			<name>Ubon Ratchathani</name>
			<code>34</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Chiang Mai</name>
			<code>50</code>
			<fips>02</fips>
			<region></region>
		</division>
		<division>
			<name>Trang</name>
			<code>92</code>
			<fips>65</fips>
			<region></region>
		</division>
		<division>
			<name>Ang Thong</name>
			<code>15</code>
			<fips>35</fips>
			<region></region>
		</division>
		<division>
			<name>Phetchaburi</name>
			<code>76</code>
			<fips>56</fips>
			<region></region>
		</division>
		<division>
			<name>Nakhon Si Thammarat</name>
			<code>80</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Lampang</name>
			<code>52</code>
			<fips>06</fips>
			<region></region>
		</division>
		<division>
			<name>Chon Buri</name>
			<code>20</code>
			<fips>46</fips>
			<region></region>
		</division>
		<division>
			<name>Phetchabun</name>
			<code>67</code>
			<fips>14</fips>
			<region></region>
		</division>
		<division>
			<name>Kalasin</name>
			<code>46</code>
			<fips>23</fips>
			<region></region>
		</division>
		<division>
			<name>Krung Thep Maha Nakhon [Bangkok]</name>
			<code>10</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Chaiyaphum</name>
			<code>36</code>
			<fips>26</fips>
			<region></region>
		</division>
		<division>
			<name>Prachuap Khiri Khan</name>
			<code>77</code>
			<fips>57</fips>
			<region></region>
		</division>
		<division>
			<name>Rayong</name>
			<code>21</code>
			<fips>47</fips>
			<region></region>
		</division>
		<division>
			<name>Nakhon Sawan</name>
			<code>60</code>
			<fips>16</fips>
			<region></region>
		</division>
		<division>
			<name>Sa Kaeo</name>
			<code>27</code>
			<fips>80</fips>
			<region></region>
		</division>
		<division>
			<name>Lop Buri</name>
			<code>16</code>
			<fips>34</fips>
			<region></region>
		</division>
		<division>
			<name>Loei</name>
			<code>42</code>
			<fips>18</fips>
			<region></region>
		</division>
		<division>
			<name>Amnat Charoen</name>
			<code>37</code>
			<fips>77</fips>
			<region></region>
		</division>
		<division>
			<name>Kamphaeng Phet</name>
			<code>62</code>
			<fips>11</fips>
			<region></region>
		</division>
		<num>764</num>
	</territory>
	<territory>
		<name>Timor-Leste</name>
		<alpha2>TL</alpha2>
		<alpha3>TLS</alpha3>
		<division>
			<name>Cova Lima</name>
			<code>CO</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Bobonaro</name>
			<code>BO</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Ermera</name>
			<code>ER</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Aileu</name>
			<code>AL</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Manatuto</name>
			<code>MT</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Manufahi</name>
			<code>MF</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Liqui&#xe7;a</name>
			<code>LI</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Lautem</name>
			<code>LA</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Ainaro</name>
			<code>AN</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Oecussi</name>
			<code>OE</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Dili</name>
			<code>DI</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Viqueque</name>
			<code>VI</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Baucau</name>
			<code>BA</code>
			<fips></fips>
			<region></region>
		</division>
		<num>626</num>
	</territory>
	<territory>
		<name>Togo</name>
		<alpha2>TG</alpha2>
		<alpha3>TGO</alpha3>
		<division>
			<name>Savannes</name>
			<code>S</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Maritime (R&#xe9;gion)</name>
			<code>M</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Kara</name>
			<code>K</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Centre</name>
			<code>C</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Plateaux</name>
			<code>P</code>
			<fips></fips>
			<region></region>
		</division>
		<num>768</num>
	</territory>
	<territory>
		<name>Tokelau</name>
		<alpha2>TK</alpha2>
		<alpha3>TKL</alpha3>
		<num>772</num>
	</territory>
	<territory>
		<name>Tonga</name>
		<alpha2>TO</alpha2>
		<alpha3>TON</alpha3>
		<num>776</num>
	</territory>
	<territory>
		<name>Trinidad and Tobago</name>
		<alpha2>TT</alpha2>
		<alpha3>TTO</alpha3>
		<division>
			<name>Penal-Debe</name>
			<code>PED</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Port of Spain</name>
			<code>POS</code>
			<fips>05</fips>
			<region></region>
		</division>
		<division>
			<name>Chaguanas</name>
			<code>CHA</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Diego Martin</name>
			<code>DMN</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>San Fernando</name>
			<code>SFO</code>
			<fips>10</fips>
			<region></region>
		</division>
		<division>
			<name>Rio Claro-Mayaro</name>
			<code>RCM</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Eastern Tobago</name>
			<code>ETO</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Sangre Grande</name>
			<code>SGE</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Princes Town</name>
			<code>PRT</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>San Juan-Laventille</name>
			<code>SJL</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Couva-Tabaquite-Talparo</name>
			<code>CTT</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Siparia</name>
			<code>SIP</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Tunapuna-Piarco</name>
			<code>TUP</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Point Fortin</name>
			<code>PTF</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Western Tobago</name>
			<code>WTO</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Arima</name>
			<code>ARI</code>
			<fips>01</fips>
			<region></region>
		</division>
		<num>780</num>
	</territory>
	<territory>
		<name>Tristan da Cunha</name>
		<alpha2>TA</alpha2>
		<alpha3></alpha3>
		<num></num>
		<note>reserved on request of UPU</note>
		<ext>upu</ext>
	</territory>
	<territory>
		<name>Tunisia</name>
		<alpha2>TN</alpha2>
		<alpha3>TUN</alpha3>
		<division>
			<name>Manouba</name>
			<code>14</code>
			<fips></fips>
			<region></region>
		</division>
		<num>788</num>
	</territory>
	<territory>
		<name>Turkey</name>
		<alpha2>TR</alpha2>
		<alpha3>TUR</alpha3>
		<division>
			<name>Afyon</name>
			<code>03</code>
			<fips>03</fips>
			<region></region>
		</division>
		<division>
			<name>Sinop</name>
			<code>57</code>
			<fips>57</fips>
			<region></region>
		</division>
		<division>
			<name>Kirikkale</name>
			<code>71</code>
			<fips>79</fips>
			<region></region>
		</division>
		<division>
			<name>Mugla</name>
			<code>48</code>
			<fips>48</fips>
			<region></region>
		</division>
		<division>
			<name>Hakk&#xe2;ri</name>
			<code>30</code>
			<fips>70</fips>
			<region></region>
		</division>
		<division>
			<name>Sirnak</name>
			<code>73</code>
			<fips>80</fips>
			<region></region>
		</division>
		<division>
			<name>Kastamonu</name>
			<code>37</code>
			<fips>37</fips>
			<region></region>
		</division>
		<division>
			<name>Bursa</name>
			<code>16</code>
			<fips>16</fips>
			<region></region>
		</division>
		<division>
			<name>Izmir</name>
			<code>35</code>
			<fips>35</fips>
			<region></region>
		</division>
		<division>
			<name>Samsun</name>
			<code>55</code>
			<fips>55</fips>
			<region></region>
		</division>
		<division>
			<name>Antalya</name>
			<code>07</code>
			<fips>07</fips>
			<region></region>
		</division>
		<division>
			<name>Ardahan</name>
			<code>75</code>
			<fips>86</fips>
			<region></region>
		</division>
		<division>
			<name>Eskisehir</name>
			<code>26</code>
			<fips>26</fips>
			<region></region>
		</division>
		<division>
			<name>&#x43;ankiri</name>
			<code>18</code>
			<fips>82</fips>
			<region></region>
		</division>
		<division>
			<name>Kirsehir</name>
			<code>40</code>
			<fips>40</fips>
			<region></region>
		</division>
		<division>
			<name>Bing&#xf6;l</name>
			<code>12</code>
			<fips>12</fips>
			<region></region>
		</division>
		<division>
			<name>K&#xfc;tahya</name>
			<code>43</code>
			<fips>43</fips>
			<region></region>
		</division>
		<division>
			<name>Kahramanmaras</name>
			<code>46</code>
			<fips>46</fips>
			<region></region>
		</division>
		<division>
			<name>Sakarya</name>
			<code>54</code>
			<fips>54</fips>
			<region></region>
		</division>
		<division>
			<name>Tokat</name>
			<code>60</code>
			<fips>60</fips>
			<region></region>
		</division>
		<division>
			<name>Burdur</name>
			<code>15</code>
			<fips>15</fips>
			<region></region>
		</division>
		<division>
			<name>Balikesir</name>
			<code>10</code>
			<fips>10</fips>
			<region></region>
		</division>
		<division>
			<name>Karaman</name>
			<code>70</code>
			<fips>78</fips>
			<region></region>
		</division>
		<division>
			<name>Bitlis</name>
			<code>13</code>
			<fips>13</fips>
			<region></region>
		</division>
		<division>
			<name>Karab&#xfc;k</name>
			<code>78</code>
			<fips>89</fips>
			<region></region>
		</division>
		<division>
			<name>Kilis</name>
			<code>79</code>
			<fips>90</fips>
			<region></region>
		</division>
		<division>
			<name>Kirklareli</name>
			<code>39</code>
			<fips>39</fips>
			<region></region>
		</division>
		<division>
			<name>Isparta</name>
			<code>32</code>
			<fips>33</fips>
			<region></region>
		</division>
		<division>
			<name>Zonguldak</name>
			<code>67</code>
			<fips>85</fips>
			<region></region>
		</division>
		<division>
			<name>Sivas</name>
			<code>58</code>
			<fips>58</fips>
			<region></region>
		</division>
		<division>
			<name>Erzincan</name>
			<code>24</code>
			<fips>24</fips>
			<region></region>
		</division>
		<division>
			<name>G&#xfc;m&#xfc;shane</name>
			<code>29</code>
			<fips>69</fips>
			<region></region>
		</division>
		<division>
			<name>Ankara</name>
			<code>06</code>
			<fips>68</fips>
			<region></region>
		</division>
		<division>
			<name>Bayburt</name>
			<code>69</code>
			<fips>77</fips>
			<region></region>
		</division>
		<division>
			<name>Edirne</name>
			<code>22</code>
			<fips>22</fips>
			<region></region>
		</division>
		<division>
			<name>Erzurum</name>
			<code>25</code>
			<fips>25</fips>
			<region></region>
		</division>
		<division>
			<name>Konya</name>
			<code>42</code>
			<fips>71</fips>
			<region></region>
		</division>
		<division>
			<name>Adiyaman</name>
			<code>02</code>
			<fips>02</fips>
			<region></region>
		</division>
		<division>
			<name>Trabzon</name>
			<code>61</code>
			<fips>61</fips>
			<region></region>
		</division>
		<division>
			<name>Aksaray</name>
			<code>68</code>
			<fips>75</fips>
			<region></region>
		</division>
		<division>
			<name>Mardin</name>
			<code>47</code>
			<fips>72</fips>
			<region></region>
		</division>
		<division>
			<name>Bolu</name>
			<code>14</code>
			<fips>14</fips>
			<region></region>
		</division>
		<division>
			<name>Malatya</name>
			<code>44</code>
			<fips>44</fips>
			<region></region>
		</division>
		<division>
			<name>Bilecik</name>
			<code>11</code>
			<fips>11</fips>
			<region></region>
		</division>
		<division>
			<name>Rize</name>
			<code>53</code>
			<fips>53</fips>
			<region></region>
		</division>
		<division>
			<name>&#x43;anakkale</name>
			<code>17</code>
			<fips>17</fips>
			<region></region>
		</division>
		<division>
			<name>Osmaniye</name>
			<code>80</code>
			<fips>91</fips>
			<region></region>
		</division>
		<division>
			<name>Kars</name>
			<code>36</code>
			<fips>84</fips>
			<region></region>
		</division>
		<division>
			<name>Nigde</name>
			<code>51</code>
			<fips>73</fips>
			<region></region>
		</division>
		<division>
			<name>Elazig</name>
			<code>23</code>
			<fips>23</fips>
			<region></region>
		</division>
		<division>
			<name>Denizli</name>
			<code>20</code>
			<fips>20</fips>
			<region></region>
		</division>
		<division>
			<name>Istanbul</name>
			<code>34</code>
			<fips>34</fips>
			<region></region>
		</division>
		<division>
			<name>Yalova</name>
			<code>77</code>
			<fips>92</fips>
			<region></region>
		</division>
		<division>
			<name>Tunceli</name>
			<code>62</code>
			<fips>62</fips>
			<region></region>
		</division>
		<division>
			<name>Adana</name>
			<code>01</code>
			<fips>81</fips>
			<region></region>
		</division>
		<division>
			<name>&#x43;orum</name>
			<code>19</code>
			<fips>19</fips>
			<region></region>
		</division>
		<division>
			<name>Ordu</name>
			<code>52</code>
			<fips>52</fips>
			<region></region>
		</division>
		<division>
			<name>I&#xe7;el</name>
			<code>33</code>
			<fips>32</fips>
			<region></region>
		</division>
		<division>
			<name>Gaziantep</name>
			<code>27</code>
			<fips>83</fips>
			<region></region>
		</division>
		<division>
			<name>Bartin</name>
			<code>74</code>
			<fips>87</fips>
			<region></region>
		</division>
		<division>
			<name>Agri</name>
			<code>04</code>
			<fips>04</fips>
			<region></region>
		</division>
		<division>
			<name>D&#xfc;zce</name>
			<code>81</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Tekirdag</name>
			<code>59</code>
			<fips>59</fips>
			<region></region>
		</division>
		<division>
			<name>Giresun</name>
			<code>28</code>
			<fips>28</fips>
			<region></region>
		</division>
		<division>
			<name>Hatay</name>
			<code>31</code>
			<fips>31</fips>
			<region></region>
		</division>
		<division>
			<name>Artvin</name>
			<code>08</code>
			<fips>08</fips>
			<region></region>
		</division>
		<division>
			<name>Siirt</name>
			<code>56</code>
			<fips>74</fips>
			<region></region>
		</division>
		<division>
			<name>Amasya</name>
			<code>05</code>
			<fips>05</fips>
			<region></region>
		</division>
		<division>
			<name>Batman</name>
			<code>72</code>
			<fips>76</fips>
			<region></region>
		</division>
		<division>
			<name>Mus</name>
			<code>49</code>
			<fips>49</fips>
			<region></region>
		</division>
		<division>
			<name>Kocaeli</name>
			<code>41</code>
			<fips>41</fips>
			<region></region>
		</division>
		<division>
			<name>Usak</name>
			<code>64</code>
			<fips>64</fips>
			<region></region>
		</division>
		<division>
			<name>Nevsehir</name>
			<code>50</code>
			<fips>50</fips>
			<region></region>
		</division>
		<division>
			<name>Yozgat</name>
			<code>66</code>
			<fips>66</fips>
			<region></region>
		</division>
		<division>
			<name>Manisa</name>
			<code>45</code>
			<fips>45</fips>
			<region></region>
		</division>
		<division>
			<name>Aydin</name>
			<code>09</code>
			<fips>09</fips>
			<region></region>
		</division>
		<division>
			<name>Kayseri</name>
			<code>38</code>
			<fips>38</fips>
			<region></region>
		</division>
		<division>
			<name>Van</name>
			<code>65</code>
			<fips>65</fips>
			<region></region>
		</division>
		<division>
			<name>Diyarbakir</name>
			<code>21</code>
			<fips>21</fips>
			<region></region>
		</division>
		<division>
			<name>Igdir</name>
			<code>76</code>
			<fips>88</fips>
			<region></region>
		</division>
		<division>
			<name>Sanliurfa</name>
			<code>63</code>
			<fips>63</fips>
			<region></region>
		</division>
		<num>792</num>
	</territory>
	<territory>
		<name>Turkmenistan</name>
		<alpha2>TM</alpha2>
		<alpha3>TKM</alpha3>
		<division>
			<name>Ahal</name>
			<code>A</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Dasoguz</name>
			<code>D</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Mary</name>
			<code>M</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Lebap</name>
			<code>L</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Balkan</name>
			<code>B</code>
			<fips></fips>
			<region></region>
		</division>
		<num>795</num>
	</territory>
	<territory>
		<name>Turks and Caicos Islands</name>
		<alpha2>TC</alpha2>
		<alpha3>TCA</alpha3>
		<num>796</num>
	</territory>
	<territory>
		<name>Tuvalu</name>
		<alpha2>TV</alpha2>
		<alpha3>TUV</alpha3>
		<num>798</num>
	</territory>
	<territory>
		<name>Uganda</name>
		<alpha2>UG</alpha2>
		<alpha3>UGA</alpha3>
		<division>
			<name>Kanungu</name>
			<code>414</code>
			<fips></fips>
			<region>W</region>
		</division>
		<division>
			<name>Wakiso</name>
			<code>113</code>
			<fips></fips>
			<region>C</region>
		</division>
		<division>
			<name>Nakapiripirit</name>
			<code>311</code>
			<fips></fips>
			<region>N</region>
		</division>
		<division>
			<name>Yumbe</name>
			<code>313</code>
			<fips></fips>
			<region>N</region>
		</division>
		<division>
			<name>Pader</name>
			<code>312</code>
			<fips></fips>
			<region>N</region>
		</division>
		<division>
			<name>Kyenjojo</name>
			<code>415</code>
			<fips></fips>
			<region>W</region>
		</division>
		<division>
			<name>Kaberamaido</name>
			<code>213</code>
			<fips></fips>
			<region>E</region>
		</division>
		<division>
			<name>Kayunga</name>
			<code>112</code>
			<fips></fips>
			<region>C</region>
		</division>
		<division>
			<name>Sironko</name>
			<code>215</code>
			<fips></fips>
			<region>E</region>
		</division>
		<division>
			<name>Kamwenge</name>
			<code>413</code>
			<fips></fips>
			<region>W</region>
		</division>
		<division>
			<name>Mayuge</name>
			<code>214</code>
			<fips></fips>
			<region>E</region>
		</division>
		<num>800</num>
	</territory>
	<territory>
		<name>Ukraine</name>
		<alpha2>UA</alpha2>
		<alpha3>UKR</alpha3>
		<division>
			<name>Volyns'ka Oblast'</name>
			<code>07</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Dnipropetrovs'ka Oblast'</name>
			<code>12</code>
			<fips>04</fips>
			<region></region>
		</division>
		<division>
			<name>Luhans'ka Oblast'</name>
			<code>09</code>
			<fips>14</fips>
			<region></region>
		</division>
		<division>
			<name>Kharkivs'ka Oblast'</name>
			<code>63</code>
			<fips>07</fips>
			<region></region>
		</division>
		<division>
			<name>Ivano-Frankivs'ka Oblast'</name>
			<code>26</code>
			<fips>06</fips>
			<region></region>
		</division>
		<division>
			<name>Ternopil's'ka Oblast'</name>
			<code>61</code>
			<fips>22</fips>
			<region></region>
		</division>
		<division>
			<name>Ky&#x3f;vs'ka Oblast'</name>
			<code>32</code>
			<fips>13</fips>
			<region></region>
		</division>
		<division>
			<name>Odes'ka Oblast'</name>
			<code>51</code>
			<fips>17</fips>
			<region></region>
		</division>
		<division>
			<name>Khersons'ka Oblast'</name>
			<code>65</code>
			<fips>08</fips>
			<region></region>
		</division>
		<division>
			<name>Cherkas'ka Oblast'</name>
			<code>71</code>
			<fips>01</fips>
			<region></region>
		</division>
		<division>
			<name>Vinnyts'ka Oblast'</name>
			<code>05</code>
			<fips>23</fips>
			<region></region>
		</division>
		<division>
			<name>L'vivs'ka Oblast'</name>
			<code>46</code>
			<fips>15</fips>
			<region></region>
		</division>
		<division>
			<name>Ky&#x3f;v</name>
			<code>30</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Poltavs'ka Oblast'</name>
			<code>53</code>
			<fips>18</fips>
			<region></region>
		</division>
		<division>
			<name>Respublika Krym</name>
			<code>43</code>
			<fips>11</fips>
			<region></region>
		</division>
		<division>
			<name>Zaporiz'ka Oblast'</name>
			<code>23</code>
			<fips>26</fips>
			<region></region>
		</division>
		<division>
			<name>Sums'ka Oblast'</name>
			<code>59</code>
			<fips>21</fips>
			<region></region>
		</division>
		<division>
			<name>Zhytomyrs'ka Oblast'</name>
			<code>18</code>
			<fips>27</fips>
			<region></region>
		</division>
		<division>
			<name>Donets'ka Oblast'</name>
			<code>14</code>
			<fips>05</fips>
			<region></region>
		</division>
		<division>
			<name>Mykola&#x3f;vs'ka Oblast'</name>
			<code>48</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Zakarpats'ka Oblast'</name>
			<code>21</code>
			<fips>25</fips>
			<region></region>
		</division>
		<division>
			<name>Chernihivs'ka Oblast'</name>
			<code>74</code>
			<fips>02</fips>
			<region></region>
		</division>
		<division>
			<name>Kirovohrads'ka Oblast'</name>
			<code>35</code>
			<fips>10</fips>
			<region></region>
		</division>
		<division>
			<name>Khmel'nyts'ka Oblast'</name>
			<code>68</code>
			<fips>09</fips>
			<region></region>
		</division>
		<division>
			<name>Rivnens'ka Oblast'</name>
			<code>56</code>
			<fips>19</fips>
			<region></region>
		</division>
		<division>
			<name>Sevastopol'</name>
			<code>40</code>
			<fips>20</fips>
			<region></region>
		</division>
		<division>
			<name>Chernivets'ka Oblast'</name>
			<code>77</code>
			<fips>03</fips>
			<region></region>
		</division>
		<num>804</num>
	</territory>
	<territory>
		<name>United Arab Emirates</name>
		<alpha2>AE</alpha2>
		<alpha3>ARE</alpha3>
		<division>
			<name>Umm al Qaywayn</name>
			<code>UQ</code>
			<fips>07</fips>
			<region></region>
		</division>
		<division>
			<name>Abu Dhabi</name>
			<code>AZ</code>
			<fips>01</fips>
			<region></region>
		</division>
		<division>
			<name>Ras al Khaymah</name>
			<code>RK</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Sharjah</name>
			<code>SH</code>
			<fips>06</fips>
			<region></region>
		</division>
		<division>
			<name>Al Fujayrah</name>
			<code>FU</code>
			<fips>04</fips>
			<region></region>
		</division>
		<division>
			<name>Ajman</name>
			<code>AJ</code>
			<fips>02</fips>
			<region></region>
		</division>
		<division>
			<name>Dubay</name>
			<code>DU</code>
			<fips>03</fips>
			<region></region>
		</division>
		<num>784</num>
	</territory>
	<territory>
		<name>United Kingdom</name>
		<alpha2>UK</alpha2>
		<alpha3></alpha3>
		<ext>uk</ext>
		<division>
			<name>Gloucestershire</name>
			<code>GLS</code>
			<fips></fips>
			<region>ENG</region>
		</division>
		<division>
			<name>Bournemouth</name>
			<code>BMH</code>
			<fips></fips>
			<region>ENG</region>
		</division>
		<division>
			<name>Suffolk</name>
			<code>SFK</code>
			<fips></fips>
			<region>ENG</region>
		</division>
		<division>
			<name>Shetland Islands</name>
			<code>ZET</code>
			<fips></fips>
			<region>SCT</region>
		</division>
		<division>
			<name>North Lanarkshire</name>
			<code>NLK</code>
			<fips></fips>
			<region>SCT</region>
		</division>
		<division>
			<name>Leicestershire</name>
			<code>LEC</code>
			<fips></fips>
			<region>ENG</region>
		</division>
		<division>
			<name>Cambridgeshire</name>
			<code>CAM</code>
			<fips></fips>
			<region>ENG</region>
		</division>
		<division>
			<name>Magherafelt</name>
			<code>MFT</code>
			<fips></fips>
			<region>NIR</region>
		</division>
		<division>
			<name>Haringey</name>
			<code>HRY</code>
			<fips></fips>
			<region>ENG</region>
		</division>
		<division>
			<name>Middlesbrough</name>
			<code>MDB</code>
			<fips></fips>
			<region>ENG</region>
		</division>
		<division>
			<name>Torfaen</name>
			<code>TOF</code>
			<fips></fips>
			<region>WLS</region>
		</division>
		<division>
			<name>Hounslow</name>
			<code>HNS</code>
			<fips></fips>
			<region>ENG</region>
		</division>
		<division>
			<name>Newport</name>
			<code>NWP</code>
			<fips></fips>
			<region>WLS</region>
		</division>
		<division>
			<name>Newham</name>
			<code>NWM</code>
			<fips></fips>
			<region>ENG</region>
		</division>
		<division>
			<name>Stockport</name>
			<code>SKP</code>
			<fips></fips>
			<region>ENG</region>
		</division>
		<division>
			<name>Warwickshire</name>
			<code>WAR</code>
			<fips></fips>
			<region>ENG</region>
		</division>
		<division>
			<name>Midlothian</name>
			<code>MLN</code>
			<fips></fips>
			<region>SCT</region>
		</division>
		<division>
			<name>Harrow</name>
			<code>HRW</code>
			<fips></fips>
			<region>ENG</region>
		</division>
		<division>
			<name>Hackney</name>
			<code>HCK</code>
			<fips></fips>
			<region>ENG</region>
		</division>
		<division>
			<name>Bristol, City of</name>
			<code>BST</code>
			<fips></fips>
			<region>ENG</region>
		</division>
		<division>
			<name>Moyle</name>
			<code>MYL</code>
			<fips></fips>
			<region>NIR</region>
		</division>
		<division>
			<name>Aberdeen City</name>
			<code>ABE</code>
			<fips></fips>
			<region>SCT</region>
		</division>
		<division>
			<name>West Dunbartonshire</name>
			<code>WDU</code>
			<fips></fips>
			<region>SCT</region>
		</division>
		<division>
			<name>Torbay</name>
			<code>TOB</code>
			<fips></fips>
			<region>ENG</region>
		</division>
		<division>
			<name>Knowsley</name>
			<code>KWL</code>
			<fips></fips>
			<region>ENG</region>
		</division>
		<division>
			<name>Cornwall</name>
			<code>CON</code>
			<fips></fips>
			<region>ENG</region>
		</division>
		<division>
			<name>Brent</name>
			<code>BEN</code>
			<fips></fips>
			<region>ENG</region>
		</division>
		<division>
			<name>Stirling</name>
			<code>STG</code>
			<fips></fips>
			<region>SCT</region>
		</division>
		<division>
			<name>North Tyneside</name>
			<code>NTY</code>
			<fips></fips>
			<region>ENG</region>
		</division>
		<division>
			<name>Norfolk</name>
			<code>NFK</code>
			<fips></fips>
			<region>ENG</region>
		</division>
		<division>
			<name>Blaenau Gwent</name>
			<code>BGW</code>
			<fips></fips>
			<region>WLS</region>
		</division>
		<division>
			<name>East Ayrshire</name>
			<code>EAY</code>
			<fips></fips>
			<region>SCT</region>
		</division>
		<division>
			<name>Buckinghamshire</name>
			<code>BKM</code>
			<fips></fips>
			<region>ENG</region>
		</division>
		<division>
			<name>Rutland</name>
			<code>RUT</code>
			<fips></fips>
			<region>ENG</region>
		</division>
		<division>
			<name>Islington</name>
			<code>ISL</code>
			<fips></fips>
			<region>ENG</region>
		</division>
		<division>
			<name>Merthyr Tydfil</name>
			<code>MTY</code>
			<fips></fips>
			<region>WLS</region>
		</division>
		<division>
			<name>Larne</name>
			<code>LRN</code>
			<fips></fips>
			<region>NIR</region>
		</division>
		<division>
			<name>Conwy</name>
			<code>CWY</code>
			<fips></fips>
			<region>WLS</region>
		</division>
		<division>
			<name>Bracknell Forest</name>
			<code>BRC</code>
			<fips></fips>
			<region>ENG</region>
		</division>
		<division>
			<name>Bedfordshire</name>
			<code>BDF</code>
			<fips></fips>
			<region>ENG</region>
		</division>
		<division>
			<name>Solihull</name>
			<code>SOL</code>
			<fips></fips>
			<region>ENG</region>
		</division>
		<division>
			<name>Medway</name>
			<code>MDW</code>
			<fips></fips>
			<region>ENG</region>
		</division>
		<division>
			<name>Lewisham</name>
			<code>LEW</code>
			<fips></fips>
			<region>ENG</region>
		</division>
		<division>
			<name>Bexley</name>
			<code>BEX</code>
			<fips></fips>
			<region>ENG</region>
		</division>
		<division>
			<name>Blackpool</name>
			<code>BPL</code>
			<fips></fips>
			<region>ENG</region>
		</division>
		<division>
			<name>North Ayrshire</name>
			<code>NAY</code>
			<fips></fips>
			<region>SCT</region>
		</division>
		<division>
			<name>London, City of</name>
			<code>LND</code>
			<fips></fips>
			<region>ENG</region>
		</division>
		<division>
			<name>North Lincolnshire</name>
			<code>NLN</code>
			<fips></fips>
			<region>ENG</region>
		</division>
		<division>
			<name>St. Helens</name>
			<code>SHN</code>
			<fips></fips>
			<region>ENG</region>
		</division>
		<division>
			<name>Strabane</name>
			<code>STB</code>
			<fips></fips>
			<region>NIR</region>
		</division>
		<division>
			<name>Redcar and Cleveland</name>
			<code>RCC</code>
			<fips></fips>
			<region>ENG</region>
		</division>
		<division>
			<name>Shropshire</name>
			<code>SHR</code>
			<fips></fips>
			<region>ENG</region>
		</division>
		<division>
			<name>Derbyshire</name>
			<code>DBY</code>
			<fips></fips>
			<region>ENG</region>
		</division>
		<division>
			<name>Surrey</name>
			<code>SRY</code>
			<fips></fips>
			<region>ENG</region>
		</division>
		<division>
			<name>Oxfordshire</name>
			<code>OXF</code>
			<fips></fips>
			<region>ENG</region>
		</division>
		<division>
			<name>Richmond upon Thames</name>
			<code>RIC</code>
			<fips></fips>
			<region>ENG</region>
		</division>
		<division>
			<name>Craigavon</name>
			<code>CGV</code>
			<fips></fips>
			<region>NIR</region>
		</division>
		<division>
			<name>Wirral</name>
			<code>WRL</code>
			<fips></fips>
			<region>ENG</region>
		</division>
		<division>
			<name>Greenwich</name>
			<code>GRE</code>
			<fips></fips>
			<region>ENG</region>
		</division>
		<division>
			<name>North Somerset</name>
			<code>NSM</code>
			<fips></fips>
			<region>ENG</region>
		</division>
		<division>
			<name>Dungannon</name>
			<code>DGN</code>
			<fips></fips>
			<region>NIR</region>
		</division>
		<division>
			<name>Hammersmith and Fulham</name>
			<code>HMF</code>
			<fips></fips>
			<region>ENG</region>
		</division>
		<division>
			<name>Barking and Dagenham</name>
			<code>BDG</code>
			<fips></fips>
			<region>ENG</region>
		</division>
		<division>
			<name>Dorset</name>
			<code>DOR</code>
			<fips></fips>
			<region>ENG</region>
		</division>
		<division>
			<name>Derry</name>
			<code>DRY</code>
			<fips></fips>
			<region>NIR</region>
		</division>
		<division>
			<name>Milton Keynes</name>
			<code>MIK</code>
			<fips></fips>
			<region>ENG</region>
		</division>
		<division>
			<name>Reading</name>
			<code>RDG</code>
			<fips></fips>
			<region>ENG</region>
		</division>
		<division>
			<name>Scottish Borders, The</name>
			<code>SCB</code>
			<fips></fips>
			<region>SCT</region>
		</division>
		<division>
			<name>Inverclyde</name>
			<code>IVC</code>
			<fips></fips>
			<region>SCT</region>
		</division>
		<division>
			<name>Bath and North East Somerset</name>
			<code>BAS</code>
			<fips></fips>
			<region>ENG</region>
		</division>
		<division>
			<name>Bury</name>
			<code>BUR</code>
			<fips></fips>
			<region>ENG</region>
		</division>
		<division>
			<name>Darlington</name>
			<code>DAL</code>
			<fips></fips>
			<region>ENG</region>
		</division>
		<division>
			<name>Belfast</name>
			<code>BFS</code>
			<fips></fips>
			<region>NIR</region>
		</division>
		<division>
			<name>Oldham</name>
			<code>OLD</code>
			<fips></fips>
			<region>ENG</region>
		</division>
		<division>
			<name>North Down</name>
			<code>NDN</code>
			<fips></fips>
			<region>NIR</region>
		</division>
		<division>
			<name>East Renfrewshire</name>
			<code>ERW</code>
			<fips></fips>
			<region>SCT</region>
		</division>
		<division>
			<name>East Riding of Yorkshire</name>
			<code>ERY</code>
			<fips></fips>
			<region>ENG</region>
		</division>
		<division>
			<name>Guernsey</name>
			<code>GSY</code>
			<fips></fips>
			<region>CHA</region>
		</division>
		<division>
			<name>Banbridge</name>
			<code>BNB</code>
			<fips></fips>
			<region>NIR</region>
		</division>
		<division>
			<name>Worcestershire</name>
			<code>WOR</code>
			<fips></fips>
			<region>ENG</region>
		</division>
		<division>
			<name>Salford</name>
			<code>SLF</code>
			<fips></fips>
			<region>ENG</region>
		</division>
		<division>
			<name>Somerset</name>
			<code>SOM</code>
			<fips>M3</fips>
			<region>ENG</region>
		</division>
		<division>
			<name>Northumberland</name>
			<code>NBL</code>
			<fips></fips>
			<region>ENG</region>
		</division>
		<division>
			<name>Brighton and Hove</name>
			<code>BNH</code>
			<fips></fips>
			<region>ENG</region>
		</division>
		<division>
			<name>Devon</name>
			<code>DEV</code>
			<fips></fips>
			<region>ENG</region>
		</division>
		<division>
			<name>Ballymoney</name>
			<code>BLY</code>
			<fips></fips>
			<region>NIR</region>
		</division>
		<division>
			<name>Leeds</name>
			<code>LDS</code>
			<fips></fips>
			<region>ENG</region>
		</division>
		<division>
			<name>Fermanagh</name>
			<code>FER</code>
			<fips></fips>
			<region>NIR</region>
		</division>
		<division>
			<name>Wigan</name>
			<code>WGN</code>
			<fips></fips>
			<region>ENG</region>
		</division>
		<division>
			<name>South Gloucestershire</name>
			<code>SGC</code>
			<fips></fips>
			<region>ENG</region>
		</division>
		<division>
			<name>Lincolnshire</name>
			<code>LIN</code>
			<fips></fips>
			<region>ENG</region>
		</division>
		<division>
			<name>Sutton</name>
			<code>STN</code>
			<fips></fips>
			<region>ENG</region>
		</division>
		<division>
			<name>North Yorkshire</name>
			<code>NYK</code>
			<fips></fips>
			<region>ENG</region>
		</division>
		<division>
			<name>Wakefield</name>
			<code>WKF</code>
			<fips></fips>
			<region>ENG</region>
		</division>
		<division>
			<name>Coventry</name>
			<code>COV</code>
			<fips></fips>
			<region>ENG</region>
		</division>
		<division>
			<name>Nottingham</name>
			<code>NGM</code>
			<fips></fips>
			<region>ENG</region>
		</division>
		<division>
			<name>Tower Hamlets</name>
			<code>TWH</code>
			<fips></fips>
			<region>ENG</region>
		</division>
		<division>
			<name>Orkney Islands</name>
			<code>ORK</code>
			<fips></fips>
			<region>SCT</region>
		</division>
		<division>
			<name>East Sussex</name>
			<code>ESX</code>
			<fips></fips>
			<region>ENG</region>
		</division>
		<division>
			<name>Fife</name>
			<code>FIF</code>
			<fips></fips>
			<region>SCT</region>
		</division>
		<division>
			<name>Birmingham</name>
			<code>BIR</code>
			<fips></fips>
			<region>ENG</region>
		</division>
		<division>
			<name>Sandwell</name>
			<code>SAW</code>
			<fips></fips>
			<region>ENG</region>
		</division>
		<division>
			<name>York</name>
			<code>YOR</code>
			<fips></fips>
			<region>ENG</region>
		</division>
		<division>
			<name>Kensington and Chelsea</name>
			<code>KEC</code>
			<fips></fips>
			<region>ENG</region>
		</division>
		<division>
			<name>Gateshead</name>
			<code>GAT</code>
			<fips></fips>
			<region>ENG</region>
		</division>
		<division>
			<name>West Sussex</name>
			<code>WSX</code>
			<fips></fips>
			<region>ENG</region>
		</division>
		<division>
			<name>Isle of Wight</name>
			<code>IOW</code>
			<fips></fips>
			<region>ENG</region>
		</division>
		<division>
			<name>Kingston upon Thames</name>
			<code>KTT</code>
			<fips></fips>
			<region>ENG</region>
		</division>
		<division>
			<name>Ballymena</name>
			<code>BLA</code>
			<fips></fips>
			<region>NIR</region>
		</division>
		<division>
			<name>Hillingdon</name>
			<code>HIL</code>
			<fips></fips>
			<region>ENG</region>
		</division>
		<division>
			<name>Isle of Anglesey</name>
			<code>AGY</code>
			<fips></fips>
			<region>WLS</region>
		</division>
		<division>
			<name>Castlereagh</name>
			<code>CSR</code>
			<fips></fips>
			<region>NIR</region>
		</division>
		<division>
			<name>Swindon</name>
			<code>SWD</code>
			<fips></fips>
			<region>ENG</region>
		</division>
		<division>
			<name>Glasgow City</name>
			<code>GLG</code>
			<fips></fips>
			<region>SCT</region>
		</division>
		<division>
			<name>Redbridge</name>
			<code>RDB</code>
			<fips></fips>
			<region>ENG</region>
		</division>
		<division>
			<name>Newcastle upon Tyne</name>
			<code>NET</code>
			<fips></fips>
			<region>ENG</region>
		</division>
		<division>
			<name>Westminster</name>
			<code>WSM</code>
			<fips></fips>
			<region>ENG</region>
		</division>
		<division>
			<name>Lancashire</name>
			<code>LAN</code>
			<fips></fips>
			<region>ENG</region>
		</division>
		<division>
			<name>Highland</name>
			<code>HLD</code>
			<fips></fips>
			<region>SCT</region>
		</division>
		<division>
			<name>South Lanarkshire</name>
			<code>SLK</code>
			<fips></fips>
			<region>SCT</region>
		</division>
		<division>
			<name>Rhondda, Cynon, Taff</name>
			<code>RCT</code>
			<fips></fips>
			<region>WLS</region>
		</division>
		<division>
			<name>Hartlepool</name>
			<code>HPL</code>
			<fips></fips>
			<region>ENG</region>
		</division>
		<division>
			<name>Northamptonshire</name>
			<code>NTH</code>
			<fips></fips>
			<region>ENG</region>
		</division>
		<division>
			<name>Herefordshire, County of</name>
			<code>HEF</code>
			<fips></fips>
			<region>ENG</region>
		</division>
		<division>
			<name>Tameside</name>
			<code>TAM</code>
			<fips></fips>
			<region>ENG</region>
		</division>
		<division>
			<name>Luton</name>
			<code>LUT</code>
			<fips></fips>
			<region>ENG</region>
		</division>
		<division>
			<name>Ards</name>
			<code>ARD</code>
			<fips></fips>
			<region>NIR</region>
		</division>
		<division>
			<name>Aberdeenshire</name>
			<code>ABD</code>
			<fips></fips>
			<region>SCT</region>
		</division>
		<division>
			<name>Halton</name>
			<code>HAL</code>
			<fips></fips>
			<region>ENG</region>
		</division>
		<division>
			<name>Caerphilly</name>
			<code>CAY</code>
			<fips></fips>
			<region>WLS</region>
		</division>
		<division>
			<name>Leicester</name>
			<code>LCE</code>
			<fips></fips>
			<region>ENG</region>
		</division>
		<division>
			<name>Denbighshire</name>
			<code>DEN</code>
			<fips></fips>
			<region>WLS</region>
		</division>
		<division>
			<name>Newry and Mourne</name>
			<code>NYM</code>
			<fips></fips>
			<region>NIR</region>
		</division>
		<division>
			<name>Wokingham</name>
			<code>WOK</code>
			<fips></fips>
			<region>ENG</region>
		</division>
		<division>
			<name>West Lothian</name>
			<code>WLN</code>
			<fips></fips>
			<region>SCT</region>
		</division>
		<division>
			<name>Dumfries and Galloway</name>
			<code>DGY</code>
			<fips></fips>
			<region>SCT</region>
		</division>
		<division>
			<name>Plymouth</name>
			<code>PLY</code>
			<fips></fips>
			<region>ENG</region>
		</division>
		<division>
			<name>Blackburn with Darwen</name>
			<code>BBD</code>
			<fips></fips>
			<region>ENG</region>
		</division>
		<division>
			<name>Wiltshire</name>
			<code>WIL</code>
			<fips></fips>
			<region>ENG</region>
		</division>
		<division>
			<name>Warrington</name>
			<code>WRT</code>
			<fips></fips>
			<region>ENG</region>
		</division>
		<division>
			<name>Moray</name>
			<code>MRY</code>
			<fips></fips>
			<region>SCT</region>
		</division>
		<division>
			<name>Croydon</name>
			<code>CRY</code>
			<fips></fips>
			<region>ENG</region>
		</division>
		<division>
			<name>Doncaster</name>
			<code>DNC</code>
			<fips></fips>
			<region>ENG</region>
		</division>
		<division>
			<name>Lambeth</name>
			<code>LBH</code>
			<fips></fips>
			<region>ENG</region>
		</division>
		<division>
			<name>Poole</name>
			<code>POL</code>
			<fips></fips>
			<region>ENG</region>
		</division>
		<division>
			<name>Nottinghamshire</name>
			<code>NTT</code>
			<fips></fips>
			<region>ENG</region>
		</division>
		<division>
			<name>Eilean Siar</name>
			<code>ELS</code>
			<fips></fips>
			<region>SCT</region>
		</division>
		<division>
			<name>Kingston upon Hull, City of</name>
			<code>KHL</code>
			<fips></fips>
			<region>ENG</region>
		</division>
		<division>
			<name>Gwynedd</name>
			<code>GWN</code>
			<fips></fips>
			<region>WLS</region>
		</division>
		<division>
			<name>Newtownabbey</name>
			<code>NTA</code>
			<fips></fips>
			<region>NIR</region>
		</division>
		<division>
			<name>Carrickfergus</name>
			<code>CKF</code>
			<fips></fips>
			<region>NIR</region>
		</division>
		<division>
			<name>Dundee City</name>
			<code>DND</code>
			<fips></fips>
			<region>SCT</region>
		</division>
		<division>
			<name>West Berkshire</name>
			<code>WBK</code>
			<fips></fips>
			<region>ENG</region>
		</division>
		<division>
			<name>Waltham Forest</name>
			<code>WFT</code>
			<fips></fips>
			<region>ENG</region>
		</division>
		<division>
			<name>South Ayrshire</name>
			<code>SAY</code>
			<fips></fips>
			<region>SCT</region>
		</division>
		<division>
			<name>Havering</name>
			<code>HAV</code>
			<fips></fips>
			<region>ENG</region>
		</division>
		<division>
			<name>Slough</name>
			<code>SLG</code>
			<fips></fips>
			<region>ENG</region>
		</division>
		<division>
			<name>Angus</name>
			<code>ANS</code>
			<fips></fips>
			<region>SCT</region>
		</division>
		<division>
			<name>Neath Port Talbot</name>
			<code>NTL</code>
			<fips></fips>
			<region>WLS</region>
		</division>
		<division>
			<name>Clackmannanshire</name>
			<code>CLK</code>
			<fips></fips>
			<region>SCT</region>
		</division>
		<division>
			<name>Edinburgh, City of</name>
			<code>EDH</code>
			<fips></fips>
			<region>SCT</region>
		</division>
		<division>
			<name>Flintshire</name>
			<code>FLN</code>
			<fips></fips>
			<region>WLS</region>
		</division>
		<division>
			<name>Liverpool</name>
			<code>LIV</code>
			<fips></fips>
			<region>ENG</region>
		</division>
		<division>
			<name>Wandsworth</name>
			<code>WND</code>
			<fips></fips>
			<region>ENG</region>
		</division>
		<division>
			<name>Jersey</name>
			<code>JSY</code>
			<fips></fips>
			<region>CHA</region>
		</division>
		<division>
			<name>Kirklees</name>
			<code>KIR</code>
			<fips></fips>
			<region>ENG</region>
		</division>
		<division>
			<name>Southend-on-Sea</name>
			<code>SOS</code>
			<fips></fips>
			<region>ENG</region>
		</division>
		<division>
			<name>Isles of Scilly</name>
			<code>IOS</code>
			<fips></fips>
			<region>ENG</region>
		</division>
		<division>
			<name>Southampton</name>
			<code>STH</code>
			<fips></fips>
			<region>ENG</region>
		</division>
		<division>
			<name>Perth and Kinross</name>
			<code>PKN</code>
			<fips></fips>
			<region>SCT</region>
		</division>
		<division>
			<name>East Lothian</name>
			<code>ELN</code>
			<fips></fips>
			<region>SCT</region>
		</division>
		<division>
			<name>Thurrock</name>
			<code>THR</code>
			<fips></fips>
			<region>ENG</region>
		</division>
		<division>
			<name>Cardiff</name>
			<code>CRF</code>
			<fips></fips>
			<region>WLS</region>
		</division>
		<division>
			<name>Manchester</name>
			<code>MAN</code>
			<fips></fips>
			<region>ENG</region>
		</division>
		<division>
			<name>Pembrokeshire</name>
			<code>PEM</code>
			<fips></fips>
			<region>WLS</region>
		</division>
		<division>
			<name>Lisburn</name>
			<code>LSB</code>
			<fips></fips>
			<region>NIR</region>
		</division>
		<division>
			<name>Stoke-on-Trent</name>
			<code>STE</code>
			<fips></fips>
			<region>ENG</region>
		</division>
		<division>
			<name>North East Lincolnshire</name>
			<code>NEL</code>
			<fips></fips>
			<region>ENG</region>
		</division>
		<division>
			<name>Swansea</name>
			<code>SWA</code>
			<fips></fips>
			<region>WLS</region>
		</division>
		<division>
			<name>Ceredigion</name>
			<code>CGN</code>
			<fips></fips>
			<region>WLS</region>
		</division>
		<division>
			<name>Enfield</name>
			<code>ENF</code>
			<fips></fips>
			<region>ENG</region>
		</division>
		<division>
			<name>Hampshire</name>
			<code>HAM</code>
			<fips></fips>
			<region>ENG</region>
		</division>
		<division>
			<name>Limavady</name>
			<code>LMV</code>
			<fips></fips>
			<region>NIR</region>
		</division>
		<division>
			<name>Staffordshire</name>
			<code>STS</code>
			<fips></fips>
			<region>ENG</region>
		</division>
		<division>
			<name>Portsmouth</name>
			<code>POR</code>
			<fips></fips>
			<region>ENG</region>
		</division>
		<division>
			<name>Cheshire</name>
			<code>CHS</code>
			<fips></fips>
			<region>ENG</region>
		</division>
		<division>
			<name>Southwark</name>
			<code>SWK</code>
			<fips></fips>
			<region>ENG</region>
		</division>
		<division>
			<name>Renfrewshire</name>
			<code>RFW</code>
			<fips></fips>
			<region>SCT</region>
		</division>
		<division>
			<name>Barnsley</name>
			<code>BNS</code>
			<fips></fips>
			<region>ENG</region>
		</division>
		<division>
			<name>Vale of Glamorgan</name>
			<code>VGL</code>
			<fips></fips>
			<region>WLS</region>
		</division>
		<division>
			<name>Camden</name>
			<code>CMD</code>
			<fips></fips>
			<region>ENG</region>
		</division>
		<division>
			<name>Armagh</name>
			<code>ARM</code>
			<fips></fips>
			<region>NIR</region>
		</division>
		<division>
			<name>Bolton</name>
			<code>BOL</code>
			<fips></fips>
			<region>ENG</region>
		</division>
		<division>
			<name>Wrexham</name>
			<code>WRX</code>
			<fips></fips>
			<region>WLS</region>
		</division>
		<division>
			<name>Durham</name>
			<code>DUR</code>
			<fips></fips>
			<region>ENG</region>
		</division>
		<division>
			<name>Trafford</name>
			<code>TRF</code>
			<fips></fips>
			<region>ENG</region>
		</division>
		<division>
			<name>Bridgend</name>
			<code>BGE</code>
			<fips></fips>
			<region>WLS</region>
		</division>
		<division>
			<name>Sheffield</name>
			<code>SHF</code>
			<fips></fips>
			<region>ENG</region>
		</division>
		<division>
			<name>Cookstown</name>
			<code>CKT</code>
			<fips></fips>
			<region>NIR</region>
		</division>
		<division>
			<name>Peterborough</name>
			<code>PTE</code>
			<fips></fips>
			<region>ENG</region>
		</division>
		<division>
			<name>Rochdale</name>
			<code>RCH</code>
			<fips></fips>
			<region>ENG</region>
		</division>
		<division>
			<name>Ealing</name>
			<code>EAL</code>
			<fips></fips>
			<region>ENG</region>
		</division>
		<division>
			<name>Down</name>
			<code>DOW</code>
			<fips></fips>
			<region>NIR</region>
		</division>
		<division>
			<name>Calderdale</name>
			<code>CLD</code>
			<fips></fips>
			<region>ENG</region>
		</division>
		<division>
			<name>Sunderland</name>
			<code>SND</code>
			<fips></fips>
			<region>ENG</region>
		</division>
		<division>
			<name>Omagh</name>
			<code>OMH</code>
			<fips></fips>
			<region>NIR</region>
		</division>
		<division>
			<name>Falkirk</name>
			<code>FAL</code>
			<fips></fips>
			<region>SCT</region>
		</division>
		<division>
			<name>Kent</name>
			<code>KEN</code>
			<fips></fips>
			<region>ENG</region>
		</division>
		<division>
			<name>Carmarthenshire</name>
			<code>CMN</code>
			<fips></fips>
			<region>WLS</region>
		</division>
		<division>
			<name>East Dunbartonshire</name>
			<code>EDU</code>
			<fips></fips>
			<region>SCT</region>
		</division>
		<division>
			<name>Dudley</name>
			<code>DUD</code>
			<fips></fips>
			<region>ENG</region>
		</division>
		<division>
			<name>Bromley</name>
			<code>BRY</code>
			<fips></fips>
			<region>ENG</region>
		</division>
		<division>
			<name>Rotherham</name>
			<code>ROT</code>
			<fips></fips>
			<region>ENG</region>
		</division>
		<division>
			<name>Argyll and Bute</name>
			<code>AGB</code>
			<fips></fips>
			<region>SCT</region>
		</division>
		<division>
			<name>Antrim</name>
			<code>ANT</code>
			<fips></fips>
			<region>NIR</region>
		</division>
		<division>
			<name>Wolverhampton</name>
			<code>WLV</code>
			<fips></fips>
			<region>ENG</region>
		</division>
		<division>
			<name>Essex</name>
			<code>ESS</code>
			<fips></fips>
			<region>ENG</region>
		</division>
		<division>
			<name>Coleraine</name>
			<code>CLR</code>
			<fips></fips>
			<region>NIR</region>
		</division>
		<division>
			<name>Hertfordshire</name>
			<code>HRT</code>
			<fips></fips>
			<region>ENG</region>
		</division>
		<division>
			<name>Windsor and Maidenhead</name>
			<code>WNM</code>
			<fips></fips>
			<region>ENG</region>
		</division>
		<division>
			<name>Powys</name>
			<code>POW</code>
			<fips></fips>
			<region>WLS</region>
		</division>
		<division>
			<name>Monmouthshire [Sir Fynwy GB-FYN]</name>
			<code>MON</code>
			<fips></fips>
			<region>WLS</region>
		</division>
		<division>
			<name>Derby</name>
			<code>DER</code>
			<fips></fips>
			<region>ENG</region>
		</division>
		<division>
			<name>Barnet</name>
			<code>BNE</code>
			<fips></fips>
			<region>ENG</region>
		</division>
		<division>
			<name>Cumbria</name>
			<code>CMA</code>
			<fips></fips>
			<region>ENG</region>
		</division>
		<division>
			<name>South Tyneside</name>
			<code>STY</code>
			<fips></fips>
			<region>ENG</region>
		</division>
		<division>
			<name>Telford and Wrekin</name>
			<code>TFW</code>
			<fips></fips>
			<region>ENG</region>
		</division>
		<division>
			<name>Stockton-on-Tees</name>
			<code>STT</code>
			<fips></fips>
			<region>ENG</region>
		</division>
		<division>
			<name>Merton</name>
			<code>MRT</code>
			<fips></fips>
			<region>ENG</region>
		</division>
		<division>
			<name>Sefton</name>
			<code>SFT</code>
			<fips></fips>
			<region>ENG</region>
		</division>
		<division>
			<name>Bradford</name>
			<code>BRD</code>
			<fips></fips>
			<region>ENG</region>
		</division>
		<division>
			<name>Walsall</name>
			<code>WLL</code>
			<fips></fips>
			<region>ENG</region>
		</division>
		<num></num>
	</territory>
	<territory>
		<name>United Kingdom</name>
		<alpha2>GB</alpha2>
		<alpha3>GBR</alpha3>
		<division>
			<name>Gloucestershire</name>
			<code>GLS</code>
			<fips></fips>
			<region>ENG</region>
		</division>
		<division>
			<name>Bournemouth</name>
			<code>BMH</code>
			<fips></fips>
			<region>ENG</region>
		</division>
		<division>
			<name>Suffolk</name>
			<code>SFK</code>
			<fips></fips>
			<region>ENG</region>
		</division>
		<division>
			<name>Shetland Islands</name>
			<code>ZET</code>
			<fips></fips>
			<region>SCT</region>
		</division>
		<division>
			<name>North Lanarkshire</name>
			<code>NLK</code>
			<fips></fips>
			<region>SCT</region>
		</division>
		<division>
			<name>Leicestershire</name>
			<code>LEC</code>
			<fips></fips>
			<region>ENG</region>
		</division>
		<division>
			<name>Cambridgeshire</name>
			<code>CAM</code>
			<fips></fips>
			<region>ENG</region>
		</division>
		<division>
			<name>Magherafelt</name>
			<code>MFT</code>
			<fips></fips>
			<region>NIR</region>
		</division>
		<division>
			<name>Haringey</name>
			<code>HRY</code>
			<fips></fips>
			<region>ENG</region>
		</division>
		<division>
			<name>Middlesbrough</name>
			<code>MDB</code>
			<fips></fips>
			<region>ENG</region>
		</division>
		<division>
			<name>Torfaen</name>
			<code>TOF</code>
			<fips></fips>
			<region>WLS</region>
		</division>
		<division>
			<name>Hounslow</name>
			<code>HNS</code>
			<fips></fips>
			<region>ENG</region>
		</division>
		<division>
			<name>Newport</name>
			<code>NWP</code>
			<fips></fips>
			<region>WLS</region>
		</division>
		<division>
			<name>Newham</name>
			<code>NWM</code>
			<fips></fips>
			<region>ENG</region>
		</division>
		<division>
			<name>Stockport</name>
			<code>SKP</code>
			<fips></fips>
			<region>ENG</region>
		</division>
		<division>
			<name>Warwickshire</name>
			<code>WAR</code>
			<fips></fips>
			<region>ENG</region>
		</division>
		<division>
			<name>Midlothian</name>
			<code>MLN</code>
			<fips></fips>
			<region>SCT</region>
		</division>
		<division>
			<name>Harrow</name>
			<code>HRW</code>
			<fips></fips>
			<region>ENG</region>
		</division>
		<division>
			<name>Hackney</name>
			<code>HCK</code>
			<fips></fips>
			<region>ENG</region>
		</division>
		<division>
			<name>Bristol, City of</name>
			<code>BST</code>
			<fips></fips>
			<region>ENG</region>
		</division>
		<division>
			<name>Moyle</name>
			<code>MYL</code>
			<fips></fips>
			<region>NIR</region>
		</division>
		<division>
			<name>Aberdeen City</name>
			<code>ABE</code>
			<fips></fips>
			<region>SCT</region>
		</division>
		<division>
			<name>West Dunbartonshire</name>
			<code>WDU</code>
			<fips></fips>
			<region>SCT</region>
		</division>
		<division>
			<name>Torbay</name>
			<code>TOB</code>
			<fips></fips>
			<region>ENG</region>
		</division>
		<division>
			<name>Knowsley</name>
			<code>KWL</code>
			<fips></fips>
			<region>ENG</region>
		</division>
		<division>
			<name>Cornwall</name>
			<code>CON</code>
			<fips></fips>
			<region>ENG</region>
		</division>
		<division>
			<name>Brent</name>
			<code>BEN</code>
			<fips></fips>
			<region>ENG</region>
		</division>
		<division>
			<name>Stirling</name>
			<code>STG</code>
			<fips></fips>
			<region>SCT</region>
		</division>
		<division>
			<name>North Tyneside</name>
			<code>NTY</code>
			<fips></fips>
			<region>ENG</region>
		</division>
		<division>
			<name>Norfolk</name>
			<code>NFK</code>
			<fips></fips>
			<region>ENG</region>
		</division>
		<division>
			<name>Blaenau Gwent</name>
			<code>BGW</code>
			<fips></fips>
			<region>WLS</region>
		</division>
		<division>
			<name>East Ayrshire</name>
			<code>EAY</code>
			<fips></fips>
			<region>SCT</region>
		</division>
		<division>
			<name>Buckinghamshire</name>
			<code>BKM</code>
			<fips></fips>
			<region>ENG</region>
		</division>
		<division>
			<name>Rutland</name>
			<code>RUT</code>
			<fips></fips>
			<region>ENG</region>
		</division>
		<division>
			<name>Islington</name>
			<code>ISL</code>
			<fips></fips>
			<region>ENG</region>
		</division>
		<division>
			<name>Merthyr Tydfil</name>
			<code>MTY</code>
			<fips></fips>
			<region>WLS</region>
		</division>
		<division>
			<name>Larne</name>
			<code>LRN</code>
			<fips></fips>
			<region>NIR</region>
		</division>
		<division>
			<name>Conwy</name>
			<code>CWY</code>
			<fips></fips>
			<region>WLS</region>
		</division>
		<division>
			<name>Bracknell Forest</name>
			<code>BRC</code>
			<fips></fips>
			<region>ENG</region>
		</division>
		<division>
			<name>Bedfordshire</name>
			<code>BDF</code>
			<fips></fips>
			<region>ENG</region>
		</division>
		<division>
			<name>Solihull</name>
			<code>SOL</code>
			<fips></fips>
			<region>ENG</region>
		</division>
		<division>
			<name>Medway</name>
			<code>MDW</code>
			<fips></fips>
			<region>ENG</region>
		</division>
		<division>
			<name>Lewisham</name>
			<code>LEW</code>
			<fips></fips>
			<region>ENG</region>
		</division>
		<division>
			<name>Bexley</name>
			<code>BEX</code>
			<fips></fips>
			<region>ENG</region>
		</division>
		<division>
			<name>Blackpool</name>
			<code>BPL</code>
			<fips></fips>
			<region>ENG</region>
		</division>
		<division>
			<name>North Ayrshire</name>
			<code>NAY</code>
			<fips></fips>
			<region>SCT</region>
		</division>
		<division>
			<name>London, City of</name>
			<code>LND</code>
			<fips></fips>
			<region>ENG</region>
		</division>
		<division>
			<name>North Lincolnshire</name>
			<code>NLN</code>
			<fips></fips>
			<region>ENG</region>
		</division>
		<division>
			<name>St. Helens</name>
			<code>SHN</code>
			<fips></fips>
			<region>ENG</region>
		</division>
		<division>
			<name>Strabane</name>
			<code>STB</code>
			<fips></fips>
			<region>NIR</region>
		</division>
		<division>
			<name>Redcar and Cleveland</name>
			<code>RCC</code>
			<fips></fips>
			<region>ENG</region>
		</division>
		<division>
			<name>Shropshire</name>
			<code>SHR</code>
			<fips></fips>
			<region>ENG</region>
		</division>
		<division>
			<name>Derbyshire</name>
			<code>DBY</code>
			<fips></fips>
			<region>ENG</region>
		</division>
		<division>
			<name>Surrey</name>
			<code>SRY</code>
			<fips></fips>
			<region>ENG</region>
		</division>
		<division>
			<name>Oxfordshire</name>
			<code>OXF</code>
			<fips></fips>
			<region>ENG</region>
		</division>
		<division>
			<name>Richmond upon Thames</name>
			<code>RIC</code>
			<fips></fips>
			<region>ENG</region>
		</division>
		<division>
			<name>Craigavon</name>
			<code>CGV</code>
			<fips></fips>
			<region>NIR</region>
		</division>
		<division>
			<name>Wirral</name>
			<code>WRL</code>
			<fips></fips>
			<region>ENG</region>
		</division>
		<division>
			<name>Greenwich</name>
			<code>GRE</code>
			<fips></fips>
			<region>ENG</region>
		</division>
		<division>
			<name>North Somerset</name>
			<code>NSM</code>
			<fips></fips>
			<region>ENG</region>
		</division>
		<division>
			<name>Dungannon</name>
			<code>DGN</code>
			<fips></fips>
			<region>NIR</region>
		</division>
		<division>
			<name>Hammersmith and Fulham</name>
			<code>HMF</code>
			<fips></fips>
			<region>ENG</region>
		</division>
		<division>
			<name>Barking and Dagenham</name>
			<code>BDG</code>
			<fips></fips>
			<region>ENG</region>
		</division>
		<division>
			<name>Dorset</name>
			<code>DOR</code>
			<fips></fips>
			<region>ENG</region>
		</division>
		<division>
			<name>Derry</name>
			<code>DRY</code>
			<fips></fips>
			<region>NIR</region>
		</division>
		<division>
			<name>Milton Keynes</name>
			<code>MIK</code>
			<fips></fips>
			<region>ENG</region>
		</division>
		<division>
			<name>Reading</name>
			<code>RDG</code>
			<fips></fips>
			<region>ENG</region>
		</division>
		<division>
			<name>Scottish Borders, The</name>
			<code>SCB</code>
			<fips></fips>
			<region>SCT</region>
		</division>
		<division>
			<name>Inverclyde</name>
			<code>IVC</code>
			<fips></fips>
			<region>SCT</region>
		</division>
		<division>
			<name>Bath and North East Somerset</name>
			<code>BAS</code>
			<fips></fips>
			<region>ENG</region>
		</division>
		<division>
			<name>Bury</name>
			<code>BUR</code>
			<fips></fips>
			<region>ENG</region>
		</division>
		<division>
			<name>Darlington</name>
			<code>DAL</code>
			<fips></fips>
			<region>ENG</region>
		</division>
		<division>
			<name>Belfast</name>
			<code>BFS</code>
			<fips></fips>
			<region>NIR</region>
		</division>
		<division>
			<name>Oldham</name>
			<code>OLD</code>
			<fips></fips>
			<region>ENG</region>
		</division>
		<division>
			<name>North Down</name>
			<code>NDN</code>
			<fips></fips>
			<region>NIR</region>
		</division>
		<division>
			<name>East Renfrewshire</name>
			<code>ERW</code>
			<fips></fips>
			<region>SCT</region>
		</division>
		<division>
			<name>East Riding of Yorkshire</name>
			<code>ERY</code>
			<fips></fips>
			<region>ENG</region>
		</division>
		<division>
			<name>Guernsey</name>
			<code>GSY</code>
			<fips></fips>
			<region>CHA</region>
		</division>
		<division>
			<name>Banbridge</name>
			<code>BNB</code>
			<fips></fips>
			<region>NIR</region>
		</division>
		<division>
			<name>Worcestershire</name>
			<code>WOR</code>
			<fips></fips>
			<region>ENG</region>
		</division>
		<division>
			<name>Salford</name>
			<code>SLF</code>
			<fips></fips>
			<region>ENG</region>
		</division>
		<division>
			<name>Somerset</name>
			<code>SOM</code>
			<fips>M3</fips>
			<region>ENG</region>
		</division>
		<division>
			<name>Northumberland</name>
			<code>NBL</code>
			<fips></fips>
			<region>ENG</region>
		</division>
		<division>
			<name>Brighton and Hove</name>
			<code>BNH</code>
			<fips></fips>
			<region>ENG</region>
		</division>
		<division>
			<name>Devon</name>
			<code>DEV</code>
			<fips></fips>
			<region>ENG</region>
		</division>
		<division>
			<name>Ballymoney</name>
			<code>BLY</code>
			<fips></fips>
			<region>NIR</region>
		</division>
		<division>
			<name>Leeds</name>
			<code>LDS</code>
			<fips></fips>
			<region>ENG</region>
		</division>
		<division>
			<name>Fermanagh</name>
			<code>FER</code>
			<fips></fips>
			<region>NIR</region>
		</division>
		<division>
			<name>Wigan</name>
			<code>WGN</code>
			<fips></fips>
			<region>ENG</region>
		</division>
		<division>
			<name>South Gloucestershire</name>
			<code>SGC</code>
			<fips></fips>
			<region>ENG</region>
		</division>
		<division>
			<name>Lincolnshire</name>
			<code>LIN</code>
			<fips></fips>
			<region>ENG</region>
		</division>
		<division>
			<name>Sutton</name>
			<code>STN</code>
			<fips></fips>
			<region>ENG</region>
		</division>
		<division>
			<name>North Yorkshire</name>
			<code>NYK</code>
			<fips></fips>
			<region>ENG</region>
		</division>
		<division>
			<name>Wakefield</name>
			<code>WKF</code>
			<fips></fips>
			<region>ENG</region>
		</division>
		<division>
			<name>Coventry</name>
			<code>COV</code>
			<fips></fips>
			<region>ENG</region>
		</division>
		<division>
			<name>Nottingham</name>
			<code>NGM</code>
			<fips></fips>
			<region>ENG</region>
		</division>
		<division>
			<name>Tower Hamlets</name>
			<code>TWH</code>
			<fips></fips>
			<region>ENG</region>
		</division>
		<division>
			<name>Orkney Islands</name>
			<code>ORK</code>
			<fips></fips>
			<region>SCT</region>
		</division>
		<division>
			<name>East Sussex</name>
			<code>ESX</code>
			<fips></fips>
			<region>ENG</region>
		</division>
		<division>
			<name>Fife</name>
			<code>FIF</code>
			<fips></fips>
			<region>SCT</region>
		</division>
		<division>
			<name>Birmingham</name>
			<code>BIR</code>
			<fips></fips>
			<region>ENG</region>
		</division>
		<division>
			<name>Sandwell</name>
			<code>SAW</code>
			<fips></fips>
			<region>ENG</region>
		</division>
		<division>
			<name>York</name>
			<code>YOR</code>
			<fips></fips>
			<region>ENG</region>
		</division>
		<division>
			<name>Kensington and Chelsea</name>
			<code>KEC</code>
			<fips></fips>
			<region>ENG</region>
		</division>
		<division>
			<name>Gateshead</name>
			<code>GAT</code>
			<fips></fips>
			<region>ENG</region>
		</division>
		<division>
			<name>West Sussex</name>
			<code>WSX</code>
			<fips></fips>
			<region>ENG</region>
		</division>
		<division>
			<name>Isle of Wight</name>
			<code>IOW</code>
			<fips></fips>
			<region>ENG</region>
		</division>
		<division>
			<name>Kingston upon Thames</name>
			<code>KTT</code>
			<fips></fips>
			<region>ENG</region>
		</division>
		<division>
			<name>Ballymena</name>
			<code>BLA</code>
			<fips></fips>
			<region>NIR</region>
		</division>
		<division>
			<name>Hillingdon</name>
			<code>HIL</code>
			<fips></fips>
			<region>ENG</region>
		</division>
		<division>
			<name>Isle of Anglesey</name>
			<code>AGY</code>
			<fips></fips>
			<region>WLS</region>
		</division>
		<division>
			<name>Castlereagh</name>
			<code>CSR</code>
			<fips></fips>
			<region>NIR</region>
		</division>
		<division>
			<name>Swindon</name>
			<code>SWD</code>
			<fips></fips>
			<region>ENG</region>
		</division>
		<division>
			<name>Glasgow City</name>
			<code>GLG</code>
			<fips></fips>
			<region>SCT</region>
		</division>
		<division>
			<name>Redbridge</name>
			<code>RDB</code>
			<fips></fips>
			<region>ENG</region>
		</division>
		<division>
			<name>Newcastle upon Tyne</name>
			<code>NET</code>
			<fips></fips>
			<region>ENG</region>
		</division>
		<division>
			<name>Westminster</name>
			<code>WSM</code>
			<fips></fips>
			<region>ENG</region>
		</division>
		<division>
			<name>Lancashire</name>
			<code>LAN</code>
			<fips></fips>
			<region>ENG</region>
		</division>
		<division>
			<name>Highland</name>
			<code>HLD</code>
			<fips></fips>
			<region>SCT</region>
		</division>
		<division>
			<name>South Lanarkshire</name>
			<code>SLK</code>
			<fips></fips>
			<region>SCT</region>
		</division>
		<division>
			<name>Rhondda, Cynon, Taff</name>
			<code>RCT</code>
			<fips></fips>
			<region>WLS</region>
		</division>
		<division>
			<name>Hartlepool</name>
			<code>HPL</code>
			<fips></fips>
			<region>ENG</region>
		</division>
		<division>
			<name>Northamptonshire</name>
			<code>NTH</code>
			<fips></fips>
			<region>ENG</region>
		</division>
		<division>
			<name>Herefordshire, County of</name>
			<code>HEF</code>
			<fips></fips>
			<region>ENG</region>
		</division>
		<division>
			<name>Tameside</name>
			<code>TAM</code>
			<fips></fips>
			<region>ENG</region>
		</division>
		<division>
			<name>Luton</name>
			<code>LUT</code>
			<fips></fips>
			<region>ENG</region>
		</division>
		<division>
			<name>Ards</name>
			<code>ARD</code>
			<fips></fips>
			<region>NIR</region>
		</division>
		<division>
			<name>Aberdeenshire</name>
			<code>ABD</code>
			<fips></fips>
			<region>SCT</region>
		</division>
		<division>
			<name>Halton</name>
			<code>HAL</code>
			<fips></fips>
			<region>ENG</region>
		</division>
		<division>
			<name>Caerphilly</name>
			<code>CAY</code>
			<fips></fips>
			<region>WLS</region>
		</division>
		<division>
			<name>Leicester</name>
			<code>LCE</code>
			<fips></fips>
			<region>ENG</region>
		</division>
		<division>
			<name>Denbighshire</name>
			<code>DEN</code>
			<fips></fips>
			<region>WLS</region>
		</division>
		<division>
			<name>Newry and Mourne</name>
			<code>NYM</code>
			<fips></fips>
			<region>NIR</region>
		</division>
		<division>
			<name>Wokingham</name>
			<code>WOK</code>
			<fips></fips>
			<region>ENG</region>
		</division>
		<division>
			<name>West Lothian</name>
			<code>WLN</code>
			<fips></fips>
			<region>SCT</region>
		</division>
		<division>
			<name>Dumfries and Galloway</name>
			<code>DGY</code>
			<fips></fips>
			<region>SCT</region>
		</division>
		<division>
			<name>Plymouth</name>
			<code>PLY</code>
			<fips></fips>
			<region>ENG</region>
		</division>
		<division>
			<name>Blackburn with Darwen</name>
			<code>BBD</code>
			<fips></fips>
			<region>ENG</region>
		</division>
		<division>
			<name>Wiltshire</name>
			<code>WIL</code>
			<fips></fips>
			<region>ENG</region>
		</division>
		<division>
			<name>Warrington</name>
			<code>WRT</code>
			<fips></fips>
			<region>ENG</region>
		</division>
		<division>
			<name>Moray</name>
			<code>MRY</code>
			<fips></fips>
			<region>SCT</region>
		</division>
		<division>
			<name>Croydon</name>
			<code>CRY</code>
			<fips></fips>
			<region>ENG</region>
		</division>
		<division>
			<name>Doncaster</name>
			<code>DNC</code>
			<fips></fips>
			<region>ENG</region>
		</division>
		<division>
			<name>Lambeth</name>
			<code>LBH</code>
			<fips></fips>
			<region>ENG</region>
		</division>
		<division>
			<name>Poole</name>
			<code>POL</code>
			<fips></fips>
			<region>ENG</region>
		</division>
		<division>
			<name>Nottinghamshire</name>
			<code>NTT</code>
			<fips></fips>
			<region>ENG</region>
		</division>
		<division>
			<name>Eilean Siar</name>
			<code>ELS</code>
			<fips></fips>
			<region>SCT</region>
		</division>
		<division>
			<name>Kingston upon Hull, City of</name>
			<code>KHL</code>
			<fips></fips>
			<region>ENG</region>
		</division>
		<division>
			<name>Gwynedd</name>
			<code>GWN</code>
			<fips></fips>
			<region>WLS</region>
		</division>
		<division>
			<name>Newtownabbey</name>
			<code>NTA</code>
			<fips></fips>
			<region>NIR</region>
		</division>
		<division>
			<name>Carrickfergus</name>
			<code>CKF</code>
			<fips></fips>
			<region>NIR</region>
		</division>
		<division>
			<name>Dundee City</name>
			<code>DND</code>
			<fips></fips>
			<region>SCT</region>
		</division>
		<division>
			<name>West Berkshire</name>
			<code>WBK</code>
			<fips></fips>
			<region>ENG</region>
		</division>
		<division>
			<name>Waltham Forest</name>
			<code>WFT</code>
			<fips></fips>
			<region>ENG</region>
		</division>
		<division>
			<name>South Ayrshire</name>
			<code>SAY</code>
			<fips></fips>
			<region>SCT</region>
		</division>
		<division>
			<name>Havering</name>
			<code>HAV</code>
			<fips></fips>
			<region>ENG</region>
		</division>
		<division>
			<name>Slough</name>
			<code>SLG</code>
			<fips></fips>
			<region>ENG</region>
		</division>
		<division>
			<name>Angus</name>
			<code>ANS</code>
			<fips></fips>
			<region>SCT</region>
		</division>
		<division>
			<name>Neath Port Talbot</name>
			<code>NTL</code>
			<fips></fips>
			<region>WLS</region>
		</division>
		<division>
			<name>Clackmannanshire</name>
			<code>CLK</code>
			<fips></fips>
			<region>SCT</region>
		</division>
		<division>
			<name>Edinburgh, City of</name>
			<code>EDH</code>
			<fips></fips>
			<region>SCT</region>
		</division>
		<division>
			<name>Flintshire</name>
			<code>FLN</code>
			<fips></fips>
			<region>WLS</region>
		</division>
		<division>
			<name>Liverpool</name>
			<code>LIV</code>
			<fips></fips>
			<region>ENG</region>
		</division>
		<division>
			<name>Wandsworth</name>
			<code>WND</code>
			<fips></fips>
			<region>ENG</region>
		</division>
		<division>
			<name>Jersey</name>
			<code>JSY</code>
			<fips></fips>
			<region>CHA</region>
		</division>
		<division>
			<name>Kirklees</name>
			<code>KIR</code>
			<fips></fips>
			<region>ENG</region>
		</division>
		<division>
			<name>Southend-on-Sea</name>
			<code>SOS</code>
			<fips></fips>
			<region>ENG</region>
		</division>
		<division>
			<name>Isles of Scilly</name>
			<code>IOS</code>
			<fips></fips>
			<region>ENG</region>
		</division>
		<division>
			<name>Southampton</name>
			<code>STH</code>
			<fips></fips>
			<region>ENG</region>
		</division>
		<division>
			<name>Perth and Kinross</name>
			<code>PKN</code>
			<fips></fips>
			<region>SCT</region>
		</division>
		<division>
			<name>East Lothian</name>
			<code>ELN</code>
			<fips></fips>
			<region>SCT</region>
		</division>
		<division>
			<name>Thurrock</name>
			<code>THR</code>
			<fips></fips>
			<region>ENG</region>
		</division>
		<division>
			<name>Cardiff</name>
			<code>CRF</code>
			<fips></fips>
			<region>WLS</region>
		</division>
		<division>
			<name>Manchester</name>
			<code>MAN</code>
			<fips></fips>
			<region>ENG</region>
		</division>
		<division>
			<name>Pembrokeshire</name>
			<code>PEM</code>
			<fips></fips>
			<region>WLS</region>
		</division>
		<division>
			<name>Lisburn</name>
			<code>LSB</code>
			<fips></fips>
			<region>NIR</region>
		</division>
		<division>
			<name>Stoke-on-Trent</name>
			<code>STE</code>
			<fips></fips>
			<region>ENG</region>
		</division>
		<division>
			<name>North East Lincolnshire</name>
			<code>NEL</code>
			<fips></fips>
			<region>ENG</region>
		</division>
		<division>
			<name>Swansea</name>
			<code>SWA</code>
			<fips></fips>
			<region>WLS</region>
		</division>
		<division>
			<name>Ceredigion</name>
			<code>CGN</code>
			<fips></fips>
			<region>WLS</region>
		</division>
		<division>
			<name>Enfield</name>
			<code>ENF</code>
			<fips></fips>
			<region>ENG</region>
		</division>
		<division>
			<name>Hampshire</name>
			<code>HAM</code>
			<fips></fips>
			<region>ENG</region>
		</division>
		<division>
			<name>Limavady</name>
			<code>LMV</code>
			<fips></fips>
			<region>NIR</region>
		</division>
		<division>
			<name>Staffordshire</name>
			<code>STS</code>
			<fips></fips>
			<region>ENG</region>
		</division>
		<division>
			<name>Portsmouth</name>
			<code>POR</code>
			<fips></fips>
			<region>ENG</region>
		</division>
		<division>
			<name>Cheshire</name>
			<code>CHS</code>
			<fips></fips>
			<region>ENG</region>
		</division>
		<division>
			<name>Southwark</name>
			<code>SWK</code>
			<fips></fips>
			<region>ENG</region>
		</division>
		<division>
			<name>Renfrewshire</name>
			<code>RFW</code>
			<fips></fips>
			<region>SCT</region>
		</division>
		<division>
			<name>Barnsley</name>
			<code>BNS</code>
			<fips></fips>
			<region>ENG</region>
		</division>
		<division>
			<name>Vale of Glamorgan</name>
			<code>VGL</code>
			<fips></fips>
			<region>WLS</region>
		</division>
		<division>
			<name>Camden</name>
			<code>CMD</code>
			<fips></fips>
			<region>ENG</region>
		</division>
		<division>
			<name>Armagh</name>
			<code>ARM</code>
			<fips></fips>
			<region>NIR</region>
		</division>
		<division>
			<name>Bolton</name>
			<code>BOL</code>
			<fips></fips>
			<region>ENG</region>
		</division>
		<division>
			<name>Wrexham</name>
			<code>WRX</code>
			<fips></fips>
			<region>WLS</region>
		</division>
		<division>
			<name>Durham</name>
			<code>DUR</code>
			<fips></fips>
			<region>ENG</region>
		</division>
		<division>
			<name>Trafford</name>
			<code>TRF</code>
			<fips></fips>
			<region>ENG</region>
		</division>
		<division>
			<name>Bridgend</name>
			<code>BGE</code>
			<fips></fips>
			<region>WLS</region>
		</division>
		<division>
			<name>Sheffield</name>
			<code>SHF</code>
			<fips></fips>
			<region>ENG</region>
		</division>
		<division>
			<name>Cookstown</name>
			<code>CKT</code>
			<fips></fips>
			<region>NIR</region>
		</division>
		<division>
			<name>Peterborough</name>
			<code>PTE</code>
			<fips></fips>
			<region>ENG</region>
		</division>
		<division>
			<name>Rochdale</name>
			<code>RCH</code>
			<fips></fips>
			<region>ENG</region>
		</division>
		<division>
			<name>Ealing</name>
			<code>EAL</code>
			<fips></fips>
			<region>ENG</region>
		</division>
		<division>
			<name>Down</name>
			<code>DOW</code>
			<fips></fips>
			<region>NIR</region>
		</division>
		<division>
			<name>Calderdale</name>
			<code>CLD</code>
			<fips></fips>
			<region>ENG</region>
		</division>
		<division>
			<name>Sunderland</name>
			<code>SND</code>
			<fips></fips>
			<region>ENG</region>
		</division>
		<division>
			<name>Omagh</name>
			<code>OMH</code>
			<fips></fips>
			<region>NIR</region>
		</division>
		<division>
			<name>Falkirk</name>
			<code>FAL</code>
			<fips></fips>
			<region>SCT</region>
		</division>
		<division>
			<name>Kent</name>
			<code>KEN</code>
			<fips></fips>
			<region>ENG</region>
		</division>
		<division>
			<name>Carmarthenshire</name>
			<code>CMN</code>
			<fips></fips>
			<region>WLS</region>
		</division>
		<division>
			<name>East Dunbartonshire</name>
			<code>EDU</code>
			<fips></fips>
			<region>SCT</region>
		</division>
		<division>
			<name>Dudley</name>
			<code>DUD</code>
			<fips></fips>
			<region>ENG</region>
		</division>
		<division>
			<name>Bromley</name>
			<code>BRY</code>
			<fips></fips>
			<region>ENG</region>
		</division>
		<division>
			<name>Rotherham</name>
			<code>ROT</code>
			<fips></fips>
			<region>ENG</region>
		</division>
		<division>
			<name>Argyll and Bute</name>
			<code>AGB</code>
			<fips></fips>
			<region>SCT</region>
		</division>
		<division>
			<name>Antrim</name>
			<code>ANT</code>
			<fips></fips>
			<region>NIR</region>
		</division>
		<division>
			<name>Wolverhampton</name>
			<code>WLV</code>
			<fips></fips>
			<region>ENG</region>
		</division>
		<division>
			<name>Essex</name>
			<code>ESS</code>
			<fips></fips>
			<region>ENG</region>
		</division>
		<division>
			<name>Coleraine</name>
			<code>CLR</code>
			<fips></fips>
			<region>NIR</region>
		</division>
		<division>
			<name>Hertfordshire</name>
			<code>HRT</code>
			<fips></fips>
			<region>ENG</region>
		</division>
		<division>
			<name>Windsor and Maidenhead</name>
			<code>WNM</code>
			<fips></fips>
			<region>ENG</region>
		</division>
		<division>
			<name>Powys</name>
			<code>POW</code>
			<fips></fips>
			<region>WLS</region>
		</division>
		<division>
			<name>Monmouthshire [Sir Fynwy GB-FYN]</name>
			<code>MON</code>
			<fips></fips>
			<region>WLS</region>
		</division>
		<division>
			<name>Derby</name>
			<code>DER</code>
			<fips></fips>
			<region>ENG</region>
		</division>
		<division>
			<name>Barnet</name>
			<code>BNE</code>
			<fips></fips>
			<region>ENG</region>
		</division>
		<division>
			<name>Cumbria</name>
			<code>CMA</code>
			<fips></fips>
			<region>ENG</region>
		</division>
		<division>
			<name>South Tyneside</name>
			<code>STY</code>
			<fips></fips>
			<region>ENG</region>
		</division>
		<division>
			<name>Telford and Wrekin</name>
			<code>TFW</code>
			<fips></fips>
			<region>ENG</region>
		</division>
		<division>
			<name>Stockton-on-Tees</name>
			<code>STT</code>
			<fips></fips>
			<region>ENG</region>
		</division>
		<division>
			<name>Merton</name>
			<code>MRT</code>
			<fips></fips>
			<region>ENG</region>
		</division>
		<division>
			<name>Sefton</name>
			<code>SFT</code>
			<fips></fips>
			<region>ENG</region>
		</division>
		<division>
			<name>Bradford</name>
			<code>BRD</code>
			<fips></fips>
			<region>ENG</region>
		</division>
		<division>
			<name>Walsall</name>
			<code>WLL</code>
			<fips></fips>
			<region>ENG</region>
		</division>
		<num>826</num>
	</territory>
	<territory>
		<name>United States</name>
		<alpha2>US</alpha2>
		<alpha3>USA</alpha3>
		<division>
			<name>Hawaii</name>
			<code>HI</code>
			<fips>15</fips>
			<region></region>
		</division>
		<division>
			<name>North Dakota</name>
			<code>ND</code>
			<fips>38</fips>
			<region></region>
		</division>
		<division>
			<name>Nebraska</name>
			<code>NE</code>
			<fips>31</fips>
			<region></region>
		</division>
		<division>
			<name>Delaware</name>
			<code>DE</code>
			<fips>10</fips>
			<region></region>
		</division>
		<division>
			<name>District of Columbia</name>
			<code>DC</code>
			<fips>11</fips>
			<region></region>
		</division>
		<division>
			<name>Washington</name>
			<code>WA</code>
			<fips>53</fips>
			<region></region>
		</division>
		<division>
			<name>Maryland</name>
			<code>MD</code>
			<fips>24</fips>
			<region></region>
		</division>
		<division>
			<name>West Virginia</name>
			<code>WV</code>
			<fips>54</fips>
			<region></region>
		</division>
		<division>
			<name>Vermont</name>
			<code>VT</code>
			<fips>50</fips>
			<region></region>
		</division>
		<division>
			<name>Oregon</name>
			<code>OR</code>
			<fips>41</fips>
			<region></region>
		</division>
		<division>
			<name>Connecticut</name>
			<code>CT</code>
			<fips>09</fips>
			<region></region>
		</division>
		<division>
			<name>Colorado</name>
			<code>CO</code>
			<fips>08</fips>
			<region></region>
		</division>
		<division>
			<name>Arizona</name>
			<code>AZ</code>
			<fips>04</fips>
			<region></region>
		</division>
		<division>
			<name>South Dakota</name>
			<code>SD</code>
			<fips>46</fips>
			<region></region>
		</division>
		<division>
			<name>Missouri</name>
			<code>MO</code>
			<fips>29</fips>
			<region></region>
		</division>
		<division>
			<name>California</name>
			<code>CA</code>
			<fips>06</fips>
			<region></region>
		</division>
		<division>
			<name>Nevada</name>
			<code>NV</code>
			<fips>32</fips>
			<region></region>
		</division>
		<division>
			<name>American Samoa</name>
			<note>this administrative division is also listed in ISO 3166-1 as AS</note>
			<code>AS</code>
			<fips>60</fips>
			<region></region>
			<ext>ust</ext>
		</division>
		<division>
			<name>Virginia</name>
			<code>VA</code>
			<fips>51</fips>
			<region></region>
		</division>
		<division>
			<name>Montana</name>
			<code>MT</code>
			<fips>30</fips>
			<region></region>
		</division>
		<division>
			<name>Puerto Rico</name>
			<note>this administrative division is also listed in ISO 3166-1 as PR</note>
			<code>PR</code>
			<fips>72</fips>
			<region></region>
			<ext>ust</ext>
		</division>
		<division>
			<name>New Mexico</name>
			<code>NM</code>
			<fips>35</fips>
			<region></region>
		</division>
		<division>
			<name>New Jersey</name>
			<code>NJ</code>
			<fips>34</fips>
			<region></region>
		</division>
		<division>
			<name>Oklahoma</name>
			<code>OK</code>
			<fips>40</fips>
			<region></region>
		</division>
		<division>
			<name>Maine</name>
			<code>ME</code>
			<fips>23</fips>
			<region></region>
		</division>
		<division>
			<name>Ohio</name>
			<code>OH</code>
			<fips>39</fips>
			<region></region>
		</division>
		<division>
			<name>New Hampshire</name>
			<code>NH</code>
			<fips>33</fips>
			<region></region>
		</division>
		<division>
			<name>Rhode Island</name>
			<code>RI</code>
			<fips>44</fips>
			<region></region>
		</division>
		<division>
			<name>Mississippi</name>
			<code>MS</code>
			<fips>28</fips>
			<region></region>
		</division>
		<division>
			<name>North Carolina</name>
			<code>NC</code>
			<fips>37</fips>
			<region></region>
		</division>
		<division>
			<name>Illinois</name>
			<code>IL</code>
			<fips>17</fips>
			<region></region>
		</division>
		<division>
			<name>Kansas</name>
			<code>KS</code>
			<fips>20</fips>
			<region></region>
		</division>
		<division>
			<name>Alaska</name>
			<code>AK</code>
			<fips>02</fips>
			<region></region>
		</division>
		<division>
			<name>Florida</name>
			<code>FL</code>
			<fips>12</fips>
			<region></region>
		</division>
		<division>
			<name>Louisiana</name>
			<code>LA</code>
			<fips>22</fips>
			<region></region>
		</division>
		<division>
			<name>Alabama</name>
			<code>AL</code>
			<fips>01</fips>
			<region></region>
		</division>
		<division>
			<name>Kentucky</name>
			<code>KY</code>
			<fips>21</fips>
			<region></region>
		</division>
		<division>
			<name>Massachusetts</name>
			<code>MA</code>
			<fips>25</fips>
			<region></region>
		</division>
		<division>
			<name>Texas</name>
			<code>TX</code>
			<fips>48</fips>
			<region></region>
		</division>
		<division>
			<name>Pennsylvania</name>
			<code>PA</code>
			<fips>42</fips>
			<region></region>
		</division>
		<division>
			<name>Minnesota</name>
			<code>MN</code>
			<fips>27</fips>
			<region></region>
		</division>
		<division>
			<name>Idaho</name>
			<code>ID</code>
			<fips>16</fips>
			<region></region>
		</division>
		<division>
			<name>New York</name>
			<code>NY</code>
			<fips>36</fips>
			<region></region>
		</division>
		<division>
			<name>Northern Mariana Islands</name>
			<note>this administrative division is also listed in ISO 3166-1 as MP</note>
			<code>MP</code>
			<fips>69</fips>
			<region></region>
			<ext>ust</ext>
		</division>
		<division>
			<name>Wisconsin</name>
			<code>WI</code>
			<fips>55</fips>
			<region></region>
		</division>
		<division>
			<name>Wyoming</name>
			<code>WY</code>
			<fips>56</fips>
			<region></region>
		</division>
		<division>
			<name>Arkansas</name>
			<code>AR</code>
			<fips>05</fips>
			<region></region>
		</division>
		<division>
			<name>Michigan</name>
			<code>MI</code>
			<fips>26</fips>
			<region></region>
		</division>
		<division>
			<name>Virgin Islands, U.S.</name>
			<note>this administrative division is also listed in ISO 3166-1 as VI</note>
			<code>VI</code>
			<fips>78</fips>
			<region></region>
			<ext>ust</ext>
		</division>
		<division>
			<name>Guam</name>
			<note>this administrative division is also listed in ISO 3166-1 as GU</note>
			<code>GU</code>
			<fips>66</fips>
			<region></region>
			<ext>ust</ext>
		</division>
		<division>
			<name>South Carolina</name>
			<code>SC</code>
			<fips>45</fips>
			<region></region>
		</division>
		<division>
			<name>Iowa</name>
			<code>IA</code>
			<fips>19</fips>
			<region></region>
		</division>
		<division>
			<name>Georgia</name>
			<code>GA</code>
			<fips>13</fips>
			<region></region>
		</division>
		<division>
			<name>United States Minor Outlying Islands</name>
			<note>this administrative division is also listed in ISO 3166-1 as UM</note>
			<code>UM</code>
			<fips></fips>
			<region></region>
			<ext>ust</ext>
		</division>
		<division>
			<name>Tennessee</name>
			<code>TN</code>
			<fips>47</fips>
			<region></region>
		</division>
		<division>
			<name>Utah</name>
			<code>UT</code>
			<fips>49</fips>
			<region></region>
		</division>
		<division>
			<name>Indiana</name>
			<code>IN</code>
			<fips>18</fips>
			<region></region>
		</division>
		<division>
			<name>Armed Forces Americas (except Canada)</name>
			<code>AA</code>
			<ext>usm</ext>
			<ext>usps</ext>
		</division>
		<division>
			<name>Armed Forces Europe / Canada / Middle East / Africa</name>
			<code>AE</code>
			<ext>usm</ext>
			<ext>usps</ext>
		</division>
		<division>
			<name>Armed Forces Pacific</name>
			<code>AP</code>
			<ext>usm</ext>
			<ext>usps</ext>
		</division>
		<division>
			<name>Federated States of Micronesia</name>
			<code>FM</code>
			<note>this soverign nation (ISO 3166-1: FM) is recognized as a domestic postal destination by the United States Postal Service using the associated abbreviation</note>
			<ext>usps</ext>
		</division>
		<division>
			<name>Marshall Islands</name>
			<code>MH</code>
			<note>this soverign nation (ISO 3166-1: MH) is recognized as a domestic postal destination by the United States Postal Service using the associated abbreviation</note>
			<ext>usps</ext>
		</division>
		<division>
			<name>Palau</name>
			<code>PW</code>
			<note>this soverign nation (ISO 3166-1: PW) is recognized as a domestic postal destination by the United States Postal Service using the associated abbreviation</note>
			<ext>usps</ext>
		</division>
		<num>840</num>
	</territory>
	<territory>
		<name>United States Minor Outlying Islands</name>
		<alpha2>UM</alpha2>
		<alpha3>UMI</alpha3>
		<division>
			<name>Johnston Atoll</name>
			<code>67</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Howland Island</name>
			<code>84</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Kingman Reef</name>
			<code>89</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Palmyra Atoll</name>
			<code>95</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Navassa Island</name>
			<code>76</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Jarvis Island</name>
			<code>86</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Wake Island</name>
			<code>79</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Baker Island</name>
			<code>81</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Midway Islands</name>
			<code>71</code>
			<fips></fips>
			<region></region>
		</division>
		<num>581</num>
	</territory>
	<territory>
		<name>Uruguay</name>
		<alpha2>UY</alpha2>
		<alpha3>URY</alpha3>
		<division>
			<name>Soriano</name>
			<code>SO</code>
			<fips>17</fips>
			<region></region>
		</division>
		<division>
			<name>Florida</name>
			<code>FD</code>
			<fips>07</fips>
			<region></region>
		</division>
		<division>
			<name>Rocha</name>
			<code>RO</code>
			<fips>14</fips>
			<region></region>
		</division>
		<division>
			<name>Colonia</name>
			<code>CO</code>
			<fips>04</fips>
			<region></region>
		</division>
		<division>
			<name>Durazno</name>
			<code>DU</code>
			<fips>05</fips>
			<region></region>
		</division>
		<division>
			<name>Canelones</name>
			<code>CA</code>
			<fips>02</fips>
			<region></region>
		</division>
		<division>
			<name>R&#xed;o Negro</name>
			<code>RN</code>
			<fips>12</fips>
			<region></region>
		</division>
		<division>
			<name>Cerro Largo</name>
			<code>CL</code>
			<fips>03</fips>
			<region></region>
		</division>
		<division>
			<name>Maldonado</name>
			<code>MA</code>
			<fips>09</fips>
			<region></region>
		</division>
		<division>
			<name>Montevideo</name>
			<code>MO</code>
			<fips>10</fips>
			<region></region>
		</division>
		<division>
			<name>Artigas</name>
			<code>AR</code>
			<fips>01</fips>
			<region></region>
		</division>
		<division>
			<name>Rivera</name>
			<code>RV</code>
			<fips>13</fips>
			<region></region>
		</division>
		<division>
			<name>Salto</name>
			<code>SA</code>
			<fips>15</fips>
			<region></region>
		</division>
		<division>
			<name>Lavalleja</name>
			<code>LA</code>
			<fips>08</fips>
			<region></region>
		</division>
		<division>
			<name>Flores</name>
			<code>FS</code>
			<fips>06</fips>
			<region></region>
		</division>
		<division>
			<name>San Jos&#xe9;</name>
			<code>SJ</code>
			<fips>16</fips>
			<region></region>
		</division>
		<division>
			<name>Treinta y Tres</name>
			<code>TT</code>
			<fips>19</fips>
			<region></region>
		</division>
		<division>
			<name>Tacuaremb&#xf3;</name>
			<code>TA</code>
			<fips>18</fips>
			<region></region>
		</division>
		<division>
			<name>Paysand&#xfa;</name>
			<code>PA</code>
			<fips>11</fips>
			<region></region>
		</division>
		<num>858</num>
	</territory>
	<territory>
		<name>Uzbekistan</name>
		<alpha2>UZ</alpha2>
		<alpha3>UZB</alpha3>
		<division>
			<name>Toshkent</name>
			<code>TK</code>
			<fips></fips>
			<region></region>
		</division>
		<num>860</num>
	</territory>
	<territory>
		<name>Vanuatu</name>
		<alpha2>VU</alpha2>
		<alpha3>VUT</alpha3>
		<division>
			<name>Torba</name>
			<code>TOB</code>
			<fips>07</fips>
			<region></region>
		</division>
		<division>
			<name>P&#xe9;nama</name>
			<code>PAM</code>
			<fips>17</fips>
			<region></region>
		</division>
		<division>
			<name>Taf&#xe9;a</name>
			<code>TAE</code>
			<fips>15</fips>
			<region></region>
		</division>
		<division>
			<name>Sh&#xe9;fa</name>
			<code>SEE</code>
			<fips>18</fips>
			<region></region>
		</division>
		<division>
			<name>Sanma</name>
			<code>SAM</code>
			<fips>13</fips>
			<region></region>
		</division>
		<division>
			<name>Malampa</name>
			<code>MAP</code>
			<fips>16</fips>
			<region></region>
		</division>
		<num>548</num>
	</territory>
	<territory>
		<name>Venezuela</name>
		<alpha2>VE</alpha2>
		<alpha3>VEN</alpha3>
		<division>
			<name>Vargas</name>
			<code>X</code>
			<fips></fips>
			<region></region>
		</division>
		<num>862</num>
	</territory>
	<territory>
		<name>Vietnam</name>
		<alpha2>VN</alpha2>
		<alpha3>VNM</alpha3>
		<division>
			<name>Quang Tri</name>
			<code>25</code>
			<fips>64</fips>
			<region></region>
		</division>
		<division>
			<name>Thua Thien-Hue</name>
			<code>26</code>
			<fips>66</fips>
			<region></region>
		</division>
		<division>
			<name>Kien Giang</name>
			<code>47</code>
			<fips>21</fips>
			<region></region>
		</division>
		<division>
			<name>Son La</name>
			<code>05</code>
			<fips>32</fips>
			<region></region>
		</division>
		<division>
			<name>Binh Dinh</name>
			<code>31</code>
			<fips>46</fips>
			<region></region>
		</division>
		<division>
			<name>Bac Giang</name>
			<code>54</code>
			<fips>71</fips>
			<region></region>
		</division>
		<division>
			<name>Ninh Binh</name>
			<code>18</code>
			<fips>59</fips>
			<region></region>
		</division>
		<division>
			<name>Dac Lac</name>
			<code>33</code>
			<fips>07</fips>
			<region></region>
		</division>
		<division>
			<name>Quang Nam</name>
			<code>27</code>
			<fips>84</fips>
			<region></region>
		</division>
		<division>
			<name>Yen Bai</name>
			<code>06</code>
			<fips>70</fips>
			<region></region>
		</division>
		<division>
			<name>Soc Trang</name>
			<code>52</code>
			<fips>65</fips>
			<region></region>
		</division>
		<division>
			<name>Binh Phuoc</name>
			<code>58</code>
			<fips>76</fips>
			<region></region>
		</division>
		<division>
			<name>Binh Duong</name>
			<code>57</code>
			<fips>75</fips>
			<region></region>
		</division>
		<division>
			<name>Quang Binh</name>
			<code>24</code>
			<fips>62</fips>
			<region></region>
		</division>
		<division>
			<name>Lang Son</name>
			<code>09</code>
			<fips>39</fips>
			<region></region>
		</division>
		<division>
			<name>Gia Lai</name>
			<code>30</code>
			<fips>49</fips>
			<region></region>
		</division>
		<division>
			<name>Dong Thap</name>
			<code>45</code>
			<fips>09</fips>
			<region></region>
		</division>
		<division>
			<name>Hung Yen</name>
			<code>66</code>
			<fips>81</fips>
			<region></region>
		</division>
		<division>
			<name>Ho Chi Minh, thanh pho [Sai Gon]</name>
			<code>65</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Ha Tinh</name>
			<code>23</code>
			<fips>52</fips>
			<region></region>
		</division>
		<division>
			<name>Quang Ninh</name>
			<code>13</code>
			<fips>30</fips>
			<region></region>
		</division>
		<division>
			<name>Ninh Thuan</name>
			<code>36</code>
			<fips>60</fips>
			<region></region>
		</division>
		<division>
			<name>Ben Tre</name>
			<code>50</code>
			<fips>03</fips>
			<region></region>
		</division>
		<division>
			<name>Da Nang, thanh pho</name>
			<code>60</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Hai Duong</name>
			<code>61</code>
			<fips>79</fips>
			<region></region>
		</division>
		<division>
			<name>Dong Nai</name>
			<code>39</code>
			<fips>43</fips>
			<region></region>
		</division>
		<division>
			<name>Ha Nam</name>
			<code>63</code>
			<fips>80</fips>
			<region></region>
		</division>
		<division>
			<name>Lam Dong</name>
			<code>35</code>
			<fips>23</fips>
			<region></region>
		</division>
		<division>
			<name>Thai Binh</name>
			<code>20</code>
			<fips>35</fips>
			<region></region>
		</division>
		<division>
			<name>Hoa Binh</name>
			<code>14</code>
			<fips>53</fips>
			<region></region>
		</division>
		<division>
			<name>Kon Tum</name>
			<code>28</code>
			<fips>55</fips>
			<region></region>
		</division>
		<division>
			<name>Bac Ninh</name>
			<code>56</code>
			<fips>74</fips>
			<region></region>
		</division>
		<division>
			<name>Tien Giang</name>
			<code>46</code>
			<fips>37</fips>
			<region></region>
		</division>
		<division>
			<name>Quang Ngai</name>
			<code>29</code>
			<fips>63</fips>
			<region></region>
		</division>
		<division>
			<name>Lai Chau</name>
			<code>01</code>
			<fips>22</fips>
			<region></region>
		</division>
		<division>
			<name>Bac Lieu</name>
			<code>55</code>
			<fips>73</fips>
			<region></region>
		</division>
		<division>
			<name>Nghe An</name>
			<code>22</code>
			<fips>58</fips>
			<region></region>
		</division>
		<division>
			<name>Ha Giang</name>
			<code>03</code>
			<fips>50</fips>
			<region></region>
		</division>
		<division>
			<name>Ha Noi, thu do</name>
			<code>64</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Khanh Hoa</name>
			<code>34</code>
			<fips>54</fips>
			<region></region>
		</division>
		<division>
			<name>Tuyen Quang</name>
			<code>07</code>
			<fips>68</fips>
			<region></region>
		</division>
		<division>
			<name>Thanh Hoa</name>
			<code>21</code>
			<fips>34</fips>
			<region></region>
		</division>
		<division>
			<name>Lao Cai</name>
			<code>02</code>
			<fips>56</fips>
			<region></region>
		</division>
		<division>
			<name>Bac Can</name>
			<code>53</code>
			<fips>72</fips>
			<region></region>
		</division>
		<division>
			<name>Vinh Phuc</name>
			<code>70</code>
			<fips>86</fips>
			<region></region>
		</division>
		<division>
			<name>Tay Ninh</name>
			<code>37</code>
			<fips>33</fips>
			<region></region>
		</division>
		<division>
			<name>Long An</name>
			<code>41</code>
			<fips>24</fips>
			<region></region>
		</division>
		<division>
			<name>Vinh Long</name>
			<code>49</code>
			<fips>69</fips>
			<region></region>
		</division>
		<division>
			<name>Tra Vinh</name>
			<code>51</code>
			<fips>67</fips>
			<region></region>
		</division>
		<division>
			<name>Cao Bang</name>
			<code>04</code>
			<fips>05</fips>
			<region></region>
		</division>
		<division>
			<name>Hai Phong, thanh pho</name>
			<code>62</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Thai Nguyen</name>
			<code>69</code>
			<fips>85</fips>
			<region></region>
		</division>
		<division>
			<name>Ba Ria - Vung Tau</name>
			<code>43</code>
			<fips>45</fips>
			<region></region>
		</division>
		<division>
			<name>An Giang</name>
			<code>44</code>
			<fips>01</fips>
			<region></region>
		</division>
		<division>
			<name>Phu Tho</name>
			<code>68</code>
			<fips>83</fips>
			<region></region>
		</division>
		<division>
			<name>Ha Tay</name>
			<code>15</code>
			<fips>51</fips>
			<region></region>
		</division>
		<division>
			<name>Phu Yen</name>
			<code>32</code>
			<fips>61</fips>
			<region></region>
		</division>
		<division>
			<name>Nam Dinh</name>
			<code>67</code>
			<fips>82</fips>
			<region></region>
		</division>
		<division>
			<name>Can Tho</name>
			<code>48</code>
			<fips>48</fips>
			<region></region>
		</division>
		<division>
			<name>Ca Mau</name>
			<code>59</code>
			<fips>77</fips>
			<region></region>
		</division>
		<division>
			<name>Binh Thuan</name>
			<code>40</code>
			<fips>47</fips>
			<region></region>
		</division>
		<num>704</num>
	</territory>
	<territory>
		<name>Virgin Islands, British</name>
		<alpha2>VG</alpha2>
		<alpha3>VGB</alpha3>
		<num>092</num>
	</territory>
	<territory>
		<name>Virgin Islands, U.S.</name>
		<alpha2>VI</alpha2>
		<alpha3>VIR</alpha3>
		<num>850</num>
	</territory>
	<territory>
		<name>Wallis and Futuna Islands</name>
		<alpha2>WF</alpha2>
		<alpha3>WLF</alpha3>
		<num>876</num>
	</territory>
	<territory>
		<name>Western Sahara</name>
		<alpha2>EH</alpha2>
		<alpha3>ESH</alpha3>
		<num>732</num>
	</territory>
	<territory>
		<name>Yemen</name>
		<alpha2>YE</alpha2>
		<alpha3>YEM</alpha3>
		<division>
			<name>Adan</name>
			<code>AD</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Al Hudaydah</name>
			<code>HU</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Al Bayha'</name>
			<code>BA</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Hahramawt</name>
			<code>HD</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Abyan</name>
			<code>AB</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Sahdah</name>
			<code>SD</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Hajjah</name>
			<code>HJ</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Al Mahwit</name>
			<code>MW</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Tahizz</name>
			<code>TA</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Shabwah</name>
			<code>SH</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Al Jawf</name>
			<code>JA</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Al Mahrah</name>
			<code>MR</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Sanha'</name>
			<code>SN</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Dhamar</name>
			<code>DH</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Ma'rib</name>
			<code>MA</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Lahij</name>
			<code>LA</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Ibb</name>
			<code>IB</code>
			<fips></fips>
			<region></region>
		</division>
		<num>887</num>
	</territory>
	<territory>
		<name>Zambia</name>
		<alpha2>ZM</alpha2>
		<alpha3>ZMB</alpha3>
		<division>
			<name>Central</name>
			<code>02</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>North-Western</name>
			<code>06</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Luapula</name>
			<code>04</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Lusaka</name>
			<code>09</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Copperbelt</name>
			<code>08</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Eastern</name>
			<code>03</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Southern</name>
			<code>07</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Northern</name>
			<code>05</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Western</name>
			<code>01</code>
			<fips></fips>
			<region></region>
		</division>
		<num>894</num>
	</territory>
	<territory>
		<name>Zimbabwe</name>
		<alpha2>ZW</alpha2>
		<alpha3>ZWE</alpha3>
		<division>
			<name>Masvingo</name>
			<code>MV</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Bulawayo</name>
			<code>BU</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Matabeleland South</name>
			<code>MS</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Manicaland</name>
			<code>MA</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Mashonaland East</name>
			<code>ME</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Harare</name>
			<code>HA</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Midlands</name>
			<code>MI</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Mashonaland Central</name>
			<code>MC</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Mashonaland West</name>
			<code>MW</code>
			<fips></fips>
			<region></region>
		</division>
		<division>
			<name>Matabeleland North</name>
			<code>MN</code>
			<fips></fips>
			<region></region>
		</division>
		<num>716</num>
	</territory>
</iso3166>
