/*
 * Decompiled with CFR 0.152.
 */
package org.openqa.selenium.remote.server;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Throwables;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.ImmutableMap;
import com.google.common.net.MediaType;
import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import java.util.concurrent.TimeUnit;
import java.util.function.Supplier;
import java.util.logging.Handler;
import java.util.logging.Logger;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Stream;
import javax.servlet.ServletException;
import javax.servlet.ServletInputStream;
import javax.servlet.http.HttpServlet;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletRequestWrapper;
import javax.servlet.http.HttpServletResponse;
import org.openqa.selenium.Platform;
import org.openqa.selenium.TimeoutException;
import org.openqa.selenium.logging.LoggingHandler;
import org.openqa.selenium.remote.BeanToJsonConverter;
import org.openqa.selenium.remote.ErrorCodes;
import org.openqa.selenium.remote.SessionId;
import org.openqa.selenium.remote.server.DefaultDriverFactory;
import org.openqa.selenium.remote.server.DefaultDriverSessions;
import org.openqa.selenium.remote.server.DriverSessions;
import org.openqa.selenium.remote.server.InputStreamWrappingServletInputStream;
import org.openqa.selenium.remote.server.JsonHttpCommandHandler;
import org.openqa.selenium.remote.server.ServletRequestWrappingHttpRequest;
import org.openqa.selenium.remote.server.ServletResponseWrappingHttpResponse;
import org.openqa.selenium.remote.server.StaticResourceHandler;
import org.openqa.selenium.remote.server.log.LoggingManager;
import org.openqa.selenium.remote.server.log.PerSessionLogHandler;
import org.openqa.selenium.remote.server.xdrpc.CrossDomainRpc;
import org.openqa.selenium.remote.server.xdrpc.CrossDomainRpcLoader;

public class DriverServlet
extends HttpServlet {
    public static final String SESSIONS_KEY = DriverServlet.class.getName() + ".sessions";
    public static final String SESSION_TIMEOUT_PARAMETER = "webdriver.server.session.timeout";
    public static final String BROWSER_TIMEOUT_PARAMETER = "webdriver.server.browser.timeout";
    private static final String CROSS_DOMAIN_RPC_PATH = "/xdrpc";
    private final StaticResourceHandler staticResourceHandler = new StaticResourceHandler();
    private final ExecutorService executor = Executors.newCachedThreadPool();
    private final Supplier<DriverSessions> sessionsSupplier;
    private final ErrorCodes errorCodes = new ErrorCodes();
    private JsonHttpCommandHandler commandHandler;
    private long individualCommandTimeoutMs;
    private long inactiveSessionTimeoutMs;

    public DriverServlet() {
        this.sessionsSupplier = new DriverSessionsSupplier();
    }

    @VisibleForTesting
    DriverServlet(Supplier<DriverSessions> sessionsSupplier) {
        this.sessionsSupplier = sessionsSupplier;
    }

    @Override
    public void init() throws ServletException {
        super.init();
        Logger logger = this.configureLogging();
        DriverSessions driverSessions = this.sessionsSupplier.get();
        this.commandHandler = new JsonHttpCommandHandler(driverSessions, logger);
        this.inactiveSessionTimeoutMs = this.getValueToUseInMs(SESSION_TIMEOUT_PARAMETER, 1800L);
        this.individualCommandTimeoutMs = this.getValueToUseInMs(BROWSER_TIMEOUT_PARAMETER, 0L);
        if (this.individualCommandTimeoutMs == 0L) {
            this.individualCommandTimeoutMs = Math.min(this.inactiveSessionTimeoutMs, Long.MAX_VALUE);
        }
    }

    @VisibleForTesting
    long getInactiveSessionTimeout() {
        return this.inactiveSessionTimeoutMs;
    }

    @VisibleForTesting
    long getIndividualCommandTimeoutMs() {
        return this.individualCommandTimeoutMs;
    }

    private synchronized Logger configureLogging() {
        Logger logger = this.getLogger();
        logger.addHandler(LoggingHandler.getInstance());
        Logger rootLogger = Logger.getLogger("");
        boolean sessionLoggerAttached = false;
        for (Handler handler : rootLogger.getHandlers()) {
            sessionLoggerAttached |= handler instanceof PerSessionLogHandler;
        }
        if (!sessionLoggerAttached) {
            rootLogger.addHandler(LoggingManager.perSessionLogHandler());
        }
        return logger;
    }

    private long getValueToUseInMs(String propertyName, long defaultValue) {
        long time = defaultValue;
        String property = this.getServletContext().getInitParameter(propertyName);
        if (property != null) {
            time = Long.parseLong(property);
        }
        return TimeUnit.SECONDS.toMillis(time);
    }

    @Override
    public void destroy() {
        this.getLogger().removeHandler(LoggingHandler.getInstance());
    }

    protected Logger getLogger() {
        return Logger.getLogger(this.getClass().getName());
    }

    @Override
    protected void service(HttpServletRequest request, HttpServletResponse response) throws IOException, ServletException {
        if (request.getHeader("Origin") != null) {
            this.setAccessControlHeaders(response);
        }
        response.setHeader("Expires", "Thu, 01 Jan 1970 00:00:00 GMT");
        response.setHeader("Cache-Control", "no-cache");
        super.service(request, response);
    }

    private void setAccessControlHeaders(HttpServletResponse response) {
        response.setHeader("Access-Control-Allow-Origin", "*");
        response.setHeader("Access-Control-Allow-Methods", "DELETE,GET,HEAD,POST");
        response.setHeader("Access-Control-Allow-Headers", "Accept,Content-Type");
    }

    @Override
    protected void doGet(HttpServletRequest request, HttpServletResponse response) throws ServletException, IOException {
        if (request.getPathInfo() == null || "/".equals(request.getPathInfo())) {
            this.staticResourceHandler.redirectToHub(request, response);
        } else if (this.staticResourceHandler.isStaticResourceRequest(request)) {
            this.staticResourceHandler.service(request, response);
        } else {
            this.handleRequest(request, response);
        }
    }

    @Override
    protected void doPost(HttpServletRequest request, HttpServletResponse response) throws ServletException, IOException {
        if (CROSS_DOMAIN_RPC_PATH.equalsIgnoreCase(request.getPathInfo())) {
            this.handleCrossDomainRpc(request, response);
        } else {
            this.handleRequest(request, response);
        }
    }

    @Override
    protected void doDelete(HttpServletRequest request, HttpServletResponse response) throws ServletException, IOException {
        this.handleRequest(request, response);
    }

    private void handleCrossDomainRpc(HttpServletRequest servletRequest, HttpServletResponse servletResponse) throws ServletException, IOException {
        CrossDomainRpc rpc;
        try {
            rpc = new CrossDomainRpcLoader().loadRpc(servletRequest);
        }
        catch (IllegalArgumentException e) {
            servletResponse.setStatus(400);
            servletResponse.getOutputStream().println(e.getMessage());
            servletResponse.getOutputStream().flush();
            return;
        }
        servletRequest.setAttribute("Content-Type", MediaType.JSON_UTF_8.toString());
        HttpServletRequestWrapper wrapper = new HttpServletRequestWrapper(servletRequest){

            @Override
            public String getMethod() {
                return rpc.getMethod();
            }

            @Override
            public String getPathInfo() {
                return rpc.getPath();
            }

            @Override
            public ServletInputStream getInputStream() throws IOException {
                return new InputStreamWrappingServletInputStream(new ByteArrayInputStream(rpc.getContent()));
            }
        };
        this.handleRequest(wrapper, servletResponse);
    }

    protected void handleRequest(HttpServletRequest servletRequest, HttpServletResponse servletResponse) throws ServletException, IOException {
        String sessionId = "unknown";
        String info = servletRequest.getPathInfo();
        Matcher matcher = Pattern.compile("^.*/session/([^/]+)").matcher(info == null ? "" : info);
        if (matcher.find()) {
            sessionId = matcher.group(1);
        }
        Future<?> future = this.executor.submit(() -> {
            String originalThreadName = Thread.currentThread().getName();
            Thread.currentThread().setName("Selenium Server handling " + servletRequest.getPathInfo());
            try {
                this.commandHandler.handleRequest(new ServletRequestWrappingHttpRequest(servletRequest), new ServletResponseWrappingHttpResponse(servletResponse));
            }
            catch (IOException e) {
                servletResponse.reset();
                throw new RuntimeException(e);
            }
            catch (Throwable e) {
                this.writeThrowable(servletResponse, e);
            }
            finally {
                Thread.currentThread().setName(originalThreadName);
            }
        });
        try {
            future.get(this.getIndividualCommandTimeoutMs(), TimeUnit.MILLISECONDS);
        }
        catch (InterruptedException e) {
            this.writeThrowable(servletResponse, e);
        }
        catch (ExecutionException e) {
            this.writeThrowable(servletResponse, Throwables.getRootCause(e));
        }
        catch (java.util.concurrent.TimeoutException e) {
            this.writeThrowable(servletResponse, new TimeoutException("Command timed out in client when executing: " + servletRequest.getPathInfo()));
            this.sessionsSupplier.get().deleteSession(new SessionId(sessionId));
        }
    }

    private void writeThrowable(HttpServletResponse resp, Throwable e) {
        int errorCode = this.errorCodes.toStatusCode(e);
        String error = this.errorCodes.toState(errorCode);
        ImmutableMap value = ImmutableMap.of("status", errorCode, "value", ImmutableMap.of("error", error, "message", e.getMessage() == null ? "" : e.getMessage(), "stacktrace", Throwables.getStackTraceAsString(e), "stackTrace", Stream.of(e.getStackTrace()).map(element -> ImmutableMap.of("fileName", element.getFileName(), "className", element.getClassName(), "methodName", element.getMethodName(), "lineNumber", element.getLineNumber())).collect(ImmutableList.toImmutableList())));
        byte[] bytes = new BeanToJsonConverter().convert(value).getBytes(StandardCharsets.UTF_8);
        try {
            resp.setStatus(500);
            resp.setHeader("Content-Type", MediaType.JSON_UTF_8.toString());
            resp.setHeader("Content-Length", String.valueOf(bytes.length));
            resp.getOutputStream().write(bytes);
        }
        catch (IOException | RuntimeException e2) {
            this.log("Unable to send response", e2);
        }
    }

    private class DriverSessionsSupplier
    implements Supplier<DriverSessions> {
        private DriverSessionsSupplier() {
        }

        @Override
        public DriverSessions get() {
            Object attribute = DriverServlet.this.getServletContext().getAttribute(SESSIONS_KEY);
            if (attribute == null) {
                attribute = new DefaultDriverSessions(new DefaultDriverFactory(Platform.getCurrent()), DriverServlet.this.getInactiveSessionTimeout());
            }
            return (DriverSessions)attribute;
        }
    }
}

