/*
 * Copyright 2009- ECMWF.
 *
 * This software is licensed under the terms of the Apache Licence version 2.0
 * which can be obtained at http://www.apache.org/licenses/LICENSE-2.0.
 * In applying this licence, ECMWF does not waive the privileges and immunities
 * granted to it by virtue of its status as an intergovernmental organisation
 * nor does it submit to any jurisdiction.
 */

#ifndef ecflow_viewer_ServerFilter_HPP
#define ecflow_viewer_ServerFilter_HPP

#include <vector>

#include "ServerItem.hpp"
#include "ecflow/node/Node.hpp"

class VSettings;

#include <boost/property_tree/ptree.hpp>

class ServerFilterObserver {
public:
    virtual ~ServerFilterObserver()                     = default;
    virtual void notifyServerFilterAdded(ServerItem*)   = 0;
    virtual void notifyServerFilterRemoved(ServerItem*) = 0;
    virtual void notifyServerFilterChanged(ServerItem*) = 0;
    virtual void notifyServerFilterDelete()             = 0;
};

class ServerFilter : public ServerItemObserver {
public:
    ServerFilter();
    ~ServerFilter() override;

    enum ChangeAspect { Reset, Added, Removed };

    const std::vector<ServerItem*>& items() const { return items_; }
    int itemCount() const { return static_cast<int>(items_.size()); }
    void serverNames(std::vector<std::string>&) const;

    void addServer(ServerItem*, bool broadcast = true);
    void removeServer(ServerItem*);
    bool isFiltered(ServerItem*) const;
    bool isFiltered(ServerHandler*) const;
    bool isFiltered(const std::string& serverName) const;

    void writeSettings(VSettings*) const;
    void readSettings(VSettings*);

    void addObserver(ServerFilterObserver*);
    void removeObserver(ServerFilterObserver*);

    // From ServerItemObserver
    void notifyServerItemChanged(ServerItem*) override;
    void notifyServerItemDeletion(ServerItem*) override;

protected:
    void broadcastAdd(ServerItem*);
    void broadcastRemove(ServerItem*);
    void broadcastChange(ServerItem*);

private:
    std::vector<ServerItem*> items_;
    std::vector<ServerFilterObserver*> observers_;
};

#endif /* ecflow_viewer_ServerFilter_HPP */
