/*
 * GnomeZclockPreview - A preview and format selection widget for GtkZclock
 * Copyright (C) 2000  Zack Hobson <zack@malfunktion.net>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */

#include <config.h>
#include "gnome_zclock_preview.h"

static GnomeDialogClass* parent_class;

static void gnome_zclock_preview_class_init (GnomeZclockPreviewClass*);
static void gnome_zclock_preview_init (GnomeZclockPreview*);
static void gnome_zclock_preview_destroy (GtkObject*);
static gboolean zp_clicked_cb (GnomeZclockPreview*, gint, gpointer);
static void entry_changed_cb (GtkEditable*, gpointer);
static void list_select_cb (GtkCList*, gint, gint, GdkEventButton*, gpointer);
static void list_unselect_cb (GtkCList*, gint, gint, GdkEventButton*, gpointer);
static void add_button_clicked_cb (GtkWidget*, gpointer);
static void rem_button_clicked_cb (GtkWidget*, gpointer);

guint
gnome_zclock_preview_get_type (void)
{
	static guint zclock_preview_type = 0;

	if (!zclock_preview_type)
	{
		static const GtkTypeInfo zclock_preview_info =
		{
			"GnomeZclockPreview",
			sizeof (GnomeZclockPreview),
			sizeof (GnomeZclockPreviewClass),
			(GtkClassInitFunc) gnome_zclock_preview_class_init,
			(GtkObjectInitFunc) gnome_zclock_preview_init,
			NULL,
			NULL,
		};

		zclock_preview_type = gtk_type_unique(gnome_dialog_get_type(),
				&zclock_preview_info);
	}

	return zclock_preview_type;
}

static void
gnome_zclock_preview_class_init (GnomeZclockPreviewClass* class)
{
	GtkObjectClass*		object_class;
	GtkWidgetClass*		widget_class;
	GtkContainerClass*	container_class;

	object_class	= (GtkObjectClass *) class;
	widget_class	= (GtkWidgetClass *) class;
	container_class	= (GtkContainerClass *) class;

	parent_class = gtk_type_class(gnome_dialog_get_type());

	object_class->destroy = gnome_zclock_preview_destroy;
}

static void
gnome_zclock_preview_init (GnomeZclockPreview* zp)
{
	g_assert(zp != NULL);
	zp->glist = NULL;
	zp->format = NULL;
	zp->selected = -1;
}

GtkWidget*
gnome_zclock_preview_new (GList** list, gchar** format)
{
	GnomeZclockPreview* zp = gtk_type_new(gnome_zclock_preview_get_type());
	GList *tmp = NULL;
	gint i = 0;

	/* save pointers to the glist and format */
	zp->pglist = list;
	zp->format = format;
	zp->glist = g_list_copy(*(list));

	/* Set my dialog properties */
	gnome_dialog_append_buttons(GNOME_DIALOG(zp),
			GNOME_STOCK_BUTTON_OK, GNOME_STOCK_BUTTON_CANCEL, NULL);
	gnome_dialog_set_close(GNOME_DIALOG(zp), TRUE);
	gtk_signal_connect(GTK_OBJECT(zp), "clicked",
			GTK_SIGNAL_FUNC(zp_clicked_cb), NULL);

	/* Create and add widgets */

	/* table to hold the everything except the preview */
	zp->table = gtk_table_new(3, 2, FALSE);
	gtk_container_set_border_width(GTK_CONTAINER ( GNOME_DIALOG (zp)->vbox),
			GNOME_PAD);
	gtk_box_pack_start_defaults(GTK_BOX (GNOME_DIALOG (zp)->vbox), zp->table);
	gtk_widget_show(zp->table);

	/* format entry box */
	zp->entry = gtk_entry_new();
	gtk_entry_set_text(GTK_ENTRY (zp->entry), *(zp->format));
	gtk_object_set_user_data(GTK_OBJECT (zp->entry), zp);
	gtk_signal_connect(GTK_OBJECT(zp->entry), "changed",
			GTK_SIGNAL_FUNC(entry_changed_cb), NULL);
	gtk_table_attach(GTK_TABLE (zp->table), zp->entry, 0, 1, 0, 1,
			GTK_EXPAND | GTK_FILL, 0, 2, 2);
	gtk_widget_show(zp->entry);

	/* Add button */
	zp->add_button = gtk_button_new_with_label (_("Add"));
	gtk_widget_set_sensitive(zp->add_button,
			strlen(*zp->format) ? TRUE : FALSE);
	gtk_table_attach(GTK_TABLE (zp->table), zp->add_button, 1, 2, 0, 1,
			0, 0, 2, 2);
	gtk_object_set_user_data(GTK_OBJECT (zp->add_button), zp);
	gtk_signal_connect(GTK_OBJECT(zp->add_button), "clicked",
			GTK_SIGNAL_FUNC (add_button_clicked_cb), NULL);
	gtk_widget_show(zp->add_button);

	/* Remove button */
	zp->rem_button = gtk_button_new_with_label(_("Remove"));
	gtk_widget_set_sensitive(zp->rem_button, FALSE);
	gtk_table_attach(GTK_TABLE (zp->table), zp->rem_button, 2, 3, 0, 1,
			0, 0, 2, 2);
	gtk_object_set_user_data(GTK_OBJECT (zp->rem_button), zp);
	gtk_signal_connect(GTK_OBJECT(zp->rem_button), "clicked",
			GTK_SIGNAL_FUNC (rem_button_clicked_cb), NULL);
	gtk_widget_show(zp->rem_button);

	/* list of formats */
	zp->list = gtk_clist_new(1);
	gtk_clist_set_shadow_type(GTK_CLIST(zp->list), GTK_SHADOW_IN);
	gtk_clist_set_selection_mode(GTK_CLIST(zp->list), GTK_SELECTION_SINGLE);
	gtk_table_attach(GTK_TABLE (zp->table), zp->list, 0, 3, 1, 2,
			GTK_EXPAND | GTK_FILL, GTK_EXPAND | GTK_FILL,
			2, 2);
	gtk_widget_show(zp->list);

	/* fill the list */
	tmp = zp->glist;
	while (tmp != NULL)
	{
		gchar* s = (gchar *)tmp->data;

		gtk_clist_append(GTK_CLIST (zp->list), &s);
		gtk_clist_set_row_data(GTK_CLIST (zp->list), i, s);

		tmp = g_list_next(tmp);
		i++;
	}
	gtk_signal_connect(GTK_OBJECT (zp->list), "select-row",
			GTK_SIGNAL_FUNC (list_select_cb), zp);
	gtk_signal_connect(GTK_OBJECT (zp->list), "unselect-row",
			GTK_SIGNAL_FUNC (list_unselect_cb), zp);

	/* Zclock preview frame */
	zp->frame = gtk_frame_new(_("Preview"));
	gtk_container_set_border_width(GTK_CONTAINER (zp->frame), GNOME_PAD);
	gtk_box_pack_start_defaults(GTK_BOX (GNOME_DIALOG (zp)->vbox), zp->frame);
	gtk_widget_show(zp->frame);

	/* Box to hold the zclock. I'm not putting the clock directly in the frame
	 * because the clock expands to fill the frame, which makes it pretty
	 * inaccurate as a preview. If anyone knows a better way around this, let
	 * me know. */
	zp->hbox_clk = gtk_hbox_new(FALSE, GNOME_PAD);
	gtk_container_add(GTK_CONTAINER (zp->frame), zp->hbox_clk);
	gtk_widget_show(zp->hbox_clk);

	/* GtkZclock widget */
	zp->zclock = gtk_zclock_new(*(zp->format));
	gtk_frame_set_shadow_type(GTK_FRAME (zp->zclock), GTK_SHADOW_NONE);
	gtk_zclock_set_justify(GTK_ZCLOCK (zp->zclock), GTK_JUSTIFY_CENTER);
	gtk_container_set_border_width(GTK_CONTAINER (zp->zclock), GNOME_PAD);
	gtk_box_pack_start(GTK_BOX (zp->hbox_clk), zp->zclock,
			TRUE, FALSE, GNOME_PAD);
	gtk_widget_show(zp->zclock);

	return GTK_WIDGET(zp);
}

static gboolean
zp_clicked_cb (GnomeZclockPreview* zp, gint button, gpointer data)
{
	if (button == 0)
	{
		g_free(*(zp->format));
		*(zp->format) = gtk_editable_get_chars(
				GTK_EDITABLE (zp->entry), 0, -1);

		g_list_free(*(zp->pglist));
		*(zp->pglist) = g_list_copy(zp->glist);
	}

	return TRUE;
}

static void
entry_changed_cb (GtkEditable* e, gpointer data)
{
	GnomeZclockPreview* zp = gtk_object_get_user_data (GTK_OBJECT (e));
	gchar* format = gtk_editable_get_chars (e, 0, -1);
	gtk_zclock_set_format (GTK_ZCLOCK (zp->zclock), format);

	gtk_widget_set_sensitive(zp->add_button, strlen(format) ? TRUE : FALSE);

	g_free (format);
}

static void
list_select_cb (GtkCList* list, gint row, gint col,
		GdkEventButton* e, gpointer data)
{
	GnomeZclockPreview* zp = data;
	gchar* format = gtk_clist_get_row_data(list, row);

	zp->selected = row;
	gtk_widget_set_sensitive(zp->rem_button, TRUE);

	gtk_entry_set_text(GTK_ENTRY (zp->entry), format);
	gtk_editable_changed(GTK_EDITABLE (zp->entry));
}

static void
list_unselect_cb (GtkCList* list, gint row, gint col,
		GdkEventButton* e, gpointer data)
{
	GnomeZclockPreview* zp = data;
	zp->selected = -1;
	gtk_widget_set_sensitive(zp->rem_button, FALSE);
}

static void
add_button_clicked_cb (GtkWidget* w, gpointer data)
{
	GnomeZclockPreview* zp = gtk_object_get_user_data(GTK_OBJECT (w));
	gchar* format = gtk_editable_get_chars(GTK_EDITABLE (zp->entry), 0, -1);
	gint row = zp->selected == -1 ? 0 : zp->selected; 

	gtk_clist_insert(GTK_CLIST (zp->list), row, &format);
	gtk_clist_set_row_data(GTK_CLIST (zp->list), row, format);
	zp->glist = g_list_insert(zp->glist, format, row);
}

static void
rem_button_clicked_cb (GtkWidget* w, gpointer data)
{
	GnomeZclockPreview* zp = gtk_object_get_user_data(GTK_OBJECT (w));
	GList* format;
	gint selected = zp->selected;

	g_assert (selected >= 0);

	gtk_clist_remove(GTK_CLIST (zp->list), selected);

	/* Yay! It finally works!! */
	format = g_list_nth(zp->glist, selected);
	g_assert(format != NULL);
	zp->glist = g_list_remove_link(zp->glist, format);
	g_list_free_1(format);
}

static void 
gnome_zclock_preview_destroy (GtkObject *object)
{
	GnomeZclockPreview* zp = GNOME_ZCLOCK_PREVIEW(object);

	g_list_free(zp->glist);
	zp->glist  = NULL;
	zp->pglist = NULL;
	zp->format = NULL;

	/* Chain up */
	if (GTK_OBJECT_CLASS(parent_class)->destroy)
		(* GTK_OBJECT_CLASS(parent_class)->destroy) (object);
}

