/* -*-c-*- */
/*
 * WallFire -- a comprehensive firewall administration tool.
 * 
 * Copyright (C) 2001 Herv Eychenne <rv@wallfire.org>
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 * 
 */

%option prefix="snort"
%option outfile="snort.cc"
%option noyywrap

%{
using namespace std;

#define YY_NO_UNPUT

#define IPLEN 16

#include <unistd.h>
#include <string.h>
#include <ctype.h>

#include "snort.h"
#include "common.h"
#include "defs.h"

#define SNORT_OPT_SRC 1
#define SNORT_OPT_DST 2
#define SNORT_OPT_PORT 4

static void snort_parse_branch(char* input);
static void snort_parse_ip(char* input, unsigned char mode);

static wf_logentry* logentry;
static bool nohit = false;
%}

MONTH	"Jan"|"Feb"|"Mar"|"Apr"|"May"|"Jun"|"Jul"|"Aug"|"Sep"|"Oct"|"Nov"|"Dec"
STRING	[a-zA-Z0-9._-]*
STRING2	[ -Z\\^-~]*
STRING3	[a-zA-Z(][ a-zA-Z0-9()/._-]*[a-zA-Z)]
LOGHOST	[0-9.a-zA-Z()_:-]*
DIGIT	[0-9]
NUMBER	{DIGIT}+
BYTE	{DIGIT}{1,3}
IPADDR	{BYTE}\.{BYTE}\.{BYTE}\.{BYTE}
PORT	{DIGIT}{1,5}
PROTO	"TCP"|"UDP"|"ICMP"

%%

{MONTH}[ ]{1,2}{DIGIT}{1,2}[ ]{DIGIT}{2}:{DIGIT}{2}:{DIGIT}{2}[ ]{LOGHOST}	parse_date(snorttext, logentry);
"snort: "				/* ignore */
"snort["{NUMBER}"]: "			/* ignore */
"["{NUMBER}":"{NUMBER}":"{DIGIT}"]"	/* ignore */
{STRING3}		logentry->chainlabel = snorttext; // |SNORT_CHAIN;
"[Classification: "{STRING2}"]"		snort_parse_branch(snorttext + 17);
"[Priority: "{DIGIT}"]:"		/* ignore */
"{"{PROTO}"}"[ ]{IPADDR}	snort_parse_ip(snorttext + 1, SNORT_OPT_SRC);
"{"{PROTO}"}"[ ]{IPADDR}":"{PORT}	snort_parse_ip(snorttext + 1, SNORT_OPT_SRC|SNORT_OPT_PORT);
"-> "{IPADDR}			snort_parse_ip(snorttext, SNORT_OPT_DST);
"-> "{IPADDR}":"{PORT}		snort_parse_ip(snorttext, SNORT_OPT_DST|SNORT_OPT_PORT);
"spp_portscan: ".*		nohit = true;
"spp_stream4: ".*		nohit = true;
[ ]+		/* ignore whitespace */
[\n]		/* ignore */
{STRING}	if (verbose) fprintf(stderr, _("Unrecognized token: %s\n"), snorttext);
.		if (verbose) fprintf(stderr, _("Unrecognized character: %s\n"), snorttext);

%%

static void
snort_parse_branch(char* input) {
  char* ptr = strchr(input, ']');
  *ptr = '\0';
  logentry->branchname = input;
  // |SNORT_BRANCH;
}

static void
snort_parse_ip(char* input, unsigned char mode) {
  char ip[IPLEN];
  int host1, host2, host3, host4;

  if (mode & SNORT_OPT_SRC) {
    char proto[9];
    if (mode & SNORT_OPT_PORT) {
      int sport;
      if (sscanf(input, "%8s %3d.%3d.%3d.%3d:%5d",
		 proto, &host1, &host2, &host3, &host4, &sport) != 6)
	return;
      logentry->sport = sport;
    }
    else if (sscanf(input, "%8s %3d.%3d.%3d.%3d",
		    proto, &host1, &host2, &host3, &host4) != 5)
      return;

    if (!strncmp(proto, "TCP", 3)) logentry->protocol = IPPROTO_TCP;
    else if (!strncmp(proto, "UDP", 3)) logentry->protocol = IPPROTO_UDP;
    else if (!strncmp(proto, "ICMP", 4)) logentry->protocol = IPPROTO_ICMP;
    
    // |SNORT_PROTO;
  }
  else if (mode & SNORT_OPT_DST) {
    if (mode & SNORT_OPT_PORT) {
      int dport;
      if (sscanf(input, "-> %3d.%3d.%3d.%3d:%5d",
		 &host1, &host2, &host3, &host4, &dport) != 5)
	return;
      logentry->dport = dport;
    }
    else if (sscanf(input, "-> %3d.%3d.%3d.%3d",
		    &host1, &host2, &host3, &host4) != 4)
      return;
  }
  else
    return;
  
  snprintf(ip, IPLEN, "%d.%d.%d.%d", host1, host2, host3, host4);

  if (mode & SNORT_OPT_SRC) {
    if (logentry->sipaddr.set(ip) == false)
      return;
    // |SNORT_SRC;
  }
  else if (mode & SNORT_OPT_DST) {
    if (logentry->dipaddr.set(ip) == false)
      return;
    // |SNORT_DST;
  }
}


wf_inmodule_snort::wf_inmodule_snort() {
}

bool
wf_inmodule_snort::match(const string& line) const {
  return (line.find(" snort") != string::npos);
}

enum wf_logentry_parsing_result
wf_inmodule_snort::parse(wf_logentry** retlogentry,
			 const string& line, int linenum) {
  logentry = new wf_logentry();
  if (logentry == NULL)
    return WF_LOGENTRY_PARSING_ERROR;

  snort_scan_string(line.c_str());
  snortlex();

  logentry->count = 1;
  logentry->format = "snort";

  if (nohit)
    return WF_LOGENTRY_PARSING_NOHIT;

  /*
  if (parser == (SNORT_DATE|SNORT_SRC|SNORT_DST))
  */
    *retlogentry = logentry;
    return WF_LOGENTRY_PARSING_OK;

  if (verbose)
    fprintf(stderr, _("snort parse error in line %d, ignoring.\n"),
	    linenum);

  delete logentry;
  return WF_LOGENTRY_PARSING_ERROR;
}

extern "C" wf_inmodule*
wf_inmodule_snort_init() {
  return new wf_inmodule_snort();
}
