/**************************************************************************
* This file is part of the WebIssues program
* Copyright (C) 2006 Michał Męciński
* Copyright (C) 2007-2009 WebIssues Team
*
* This program is free software; you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation; either version 2 of the License, or
* (at your option) any later version.
**************************************************************************/

#include "columnsettingsdialog.h"

#include <QLayout>
#include <QLabel>
#include <QToolButton>
#include <QHeaderView>
#include <QPushButton>
#include <QComboBox>
#include <QListWidget>
#include <QDialogButtonBox>

#include "data/datamanager.h"
#include "models/treeviewsettings.h"
#include "models/tablemodelshelper.h"
#include "widgets/gradientwidget.h"
#include "iconloader.h"

ColumnSettingsDialog::ColumnSettingsDialog( QWidget* parent ) : QDialog( parent )
{
    setWindowTitle( tr( "Column Settings" ) );

    QVBoxLayout* topLayout = new QVBoxLayout( this );
    topLayout->setMargin( 0 );
    topLayout->setSpacing( 0 );

    QWidget* promptWidget = new GradientWidget( this );
    topLayout->addWidget( promptWidget );

    QHBoxLayout* promptLayout = new QHBoxLayout( promptWidget );
    promptLayout->setSpacing( 10 );

    QLabel* promptPixmap = new QLabel( promptWidget );
    promptPixmap->setPixmap( IconLoader::pixmap( "configure-columns", 22 ) );
    promptLayout->addWidget( promptPixmap, 0 );

    m_promptLabel = new QLabel( promptWidget );
    m_promptLabel->setWordWrap( true );
    m_promptLabel->setMinimumWidth( 350 );
    promptLayout->addWidget( m_promptLabel, 1 );

    QFrame* separator = new QFrame( this );
    separator->setFrameStyle( QFrame::HLine | QFrame::Sunken );
    topLayout->addWidget( separator );

    QVBoxLayout* mainLayout = new QVBoxLayout();
    mainLayout->setMargin( 9 );
    mainLayout->setSpacing( 6 );
    topLayout->addLayout( mainLayout );

    QGridLayout* gridLayout = new QGridLayout();
    mainLayout->addLayout( gridLayout );

    QLabel* availableLabel = new QLabel( tr( "A&valiable columns:" ), this );
    gridLayout->addWidget( availableLabel, 0, 0 );

    QLabel* selectedLabel = new QLabel( tr( "&Selected columns:" ), this );
    gridLayout->addWidget( selectedLabel, 0, 2 );

    m_availableList = new QListWidget( this );
    gridLayout->addWidget( m_availableList, 1, 0 );

    m_currentList = new QListWidget( this );
    gridLayout->addWidget( m_currentList, 1, 2 );

    availableLabel->setBuddy( m_availableList );
    selectedLabel->setBuddy( m_currentList );

    connect( m_availableList, SIGNAL( currentItemChanged( QListWidgetItem*, QListWidgetItem* ) ), this, SLOT( availableSelected() ) );
    connect( m_currentList, SIGNAL( currentItemChanged( QListWidgetItem*, QListWidgetItem* ) ), this, SLOT( currentSelected() ) );

    connect( m_availableList, SIGNAL( itemActivated( QListWidgetItem* ) ), this, SLOT( addClicked() ) );
    connect( m_currentList, SIGNAL( itemActivated( QListWidgetItem* ) ), this, SLOT( removeClicked() ) );

    QVBoxLayout* addRemoveLayout = new QVBoxLayout();
    gridLayout->addLayout( addRemoveLayout, 1, 1 );

    addRemoveLayout->addStretch( 1 );

    m_addButton = new QToolButton( this );
    m_addButton->setIcon( IconLoader::icon( "move-right" ) );
    addRemoveLayout->addWidget( m_addButton );

    m_removeButton = new QToolButton( this );
    m_removeButton->setIcon( IconLoader::icon( "move-left" ) );
    addRemoveLayout->addWidget( m_removeButton );

    addRemoveLayout->addStretch( 1 );

    QVBoxLayout* upDownLayout = new QVBoxLayout();
    gridLayout->addLayout( upDownLayout, 1, 3 );

    upDownLayout->addStretch( 1 );

    m_upButton = new QToolButton( this );
    m_upButton->setIcon( IconLoader::icon( "move-up" ) );
    m_upButton->setAutoRepeat( true );
    upDownLayout->addWidget( m_upButton );

    m_downButton = new QToolButton( this );
    m_downButton->setIcon( IconLoader::icon( "move-down" ) );
    m_downButton->setAutoRepeat( true );
    upDownLayout->addWidget( m_downButton );

    upDownLayout->addStretch( 1 );

    connect( m_addButton, SIGNAL( clicked() ), this, SLOT( addClicked() ) );
    connect( m_removeButton, SIGNAL( clicked() ), this, SLOT( removeClicked() ) );
    connect( m_upButton, SIGNAL( clicked() ), this, SLOT( upClicked() ) );
    connect( m_downButton, SIGNAL( clicked() ), this, SLOT( downClicked() ) );

    mainLayout->addSpacing( 10 );

    QHBoxLayout* sortLayout = new QHBoxLayout();
    mainLayout->addLayout( sortLayout );

    sortLayout->addStretch( 1 );

    QLabel* sortLabel = new QLabel( tr( "&Default sort column:" ), this );
    sortLayout->addWidget( sortLabel );

    m_sortCombo = new QComboBox( this );
    m_sortCombo->setFixedWidth( 150 );
    sortLayout->addWidget( m_sortCombo );

    sortLabel->setBuddy( m_sortCombo );

    sortLayout->addSpacing( 10 );

    QLabel* orderLabel = new QLabel( tr( "Sort &order:" ), this );
    sortLayout->addWidget( orderLabel );

    m_orderCombo = new QComboBox( this );
    m_orderCombo->addItem( tr ( "Ascending" ) );
    m_orderCombo->addItem( tr( "Descending" ) );
    sortLayout->addWidget( m_orderCombo );

    orderLabel->setBuddy( m_orderCombo );

    sortLayout->addStretch( 1 );

    mainLayout->addSpacing( 10 );

    m_buttonBox = new QDialogButtonBox( QDialogButtonBox::RestoreDefaults | QDialogButtonBox::Ok
        | QDialogButtonBox::Apply | QDialogButtonBox::Cancel, Qt::Horizontal, this );
    mainLayout->addWidget( m_buttonBox );

    m_buttonBox->button( QDialogButtonBox::RestoreDefaults )->setText( tr( "&Reset" ) );
    m_buttonBox->button( QDialogButtonBox::Ok )->setText( tr( "&OK" ) );
    m_buttonBox->button( QDialogButtonBox::Apply )->setText( tr( "&Apply" ) );
    m_buttonBox->button( QDialogButtonBox::Cancel )->setText( tr( "&Cancel" ) );

    connect( m_buttonBox, SIGNAL( accepted() ), this, SLOT( accept() ) );
    connect( m_buttonBox, SIGNAL( rejected() ), this, SLOT( reject() ) );
    connect( m_buttonBox->button( QDialogButtonBox::RestoreDefaults ), SIGNAL( clicked() ), this, SLOT( restoreDefaults() ) );
    connect( m_buttonBox->button( QDialogButtonBox::Apply ), SIGNAL( clicked() ), this, SIGNAL( settingsApplied() ) );

    resize( 500, 350 );
}

ColumnSettingsDialog::~ColumnSettingsDialog()
{
}

void ColumnSettingsDialog::setPrompt( const QString& text )
{
    m_promptLabel->setText( text );

    m_promptLabel->setFixedHeight( m_promptLabel->heightForWidth( m_promptLabel->minimumWidth() ) );
}

void ColumnSettingsDialog::setAvailableColumns( const QList<int>& columns )
{
    m_availableColumns = columns;
}

void ColumnSettingsDialog::setDefaultColumns( const QList<int>& columns )
{
    m_defaultColumns = columns;
}

void ColumnSettingsDialog::setFixedColumns( const QList<int>& columns )
{
    m_fixedColumns = columns;
}

void ColumnSettingsDialog::setColumns( const QList<int>& columns )
{
    m_availableList->clear();

    for ( int i = 0; i < m_availableColumns.count(); i++ ) {
        int column = m_availableColumns.at( i );
        if ( !columns.contains( column ) ) {
            QListWidgetItem* item = new QListWidgetItem( m_availableList );
            item->setText( TableModelsHelper::columnName( column ) );
            item->setData( Qt::DecorationRole, IconLoader::pixmap( "attribute" ) );
            item->setData( Qt::UserRole, QVariant( (int)column ) );
        }
    }

    m_availableList->sortItems( Qt::AscendingOrder );

    m_currentList->clear();
    m_sortCombo->clear();

    for ( int i = 0; i < columns.count(); i++ ) {
        int column = columns.at( i );
        QListWidgetItem* item = new QListWidgetItem( m_currentList );
        item->setText( TableModelsHelper::columnName( column ) );
        item->setData( Qt::DecorationRole, IconLoader::pixmap( "attribute" ) );
        item->setData( Qt::UserRole, QVariant( (int)column ) );
        if ( m_fixedColumns.contains( column ) )
            item->setForeground( palette().color( QPalette::Disabled, QPalette::Text ) );
        m_sortCombo->addItem( item->text() );
    }

    m_addButton->setEnabled( false );
    m_removeButton->setEnabled( false );
    m_upButton->setEnabled( false );
    m_downButton->setEnabled( false );
}

QList<int> ColumnSettingsDialog::columns() const
{
    QList<int> columns;

    for ( int i = 0; i < m_currentList->count(); i++ ) {
        QListWidgetItem* item = m_currentList->item( i );
        int column = item->data( Qt::UserRole ).toInt();
        columns.append( column );
    }

    return columns;
}

void ColumnSettingsDialog::setDefaultSortOrder( const QPair<int, Qt::SortOrder>& order )
{
    m_defaultOrder = order;
}

void ColumnSettingsDialog::setSortOrder( const QPair<int, Qt::SortOrder>& order )
{
    for ( int i = 0; i < m_currentList->count(); i++ ) {
        QListWidgetItem* item = m_currentList->item( i );
        int column = item->data( Qt::UserRole ).toInt();
        if ( order.first == column ) {
            m_sortCombo->setCurrentIndex( i );
            break;
        }
    }

    m_orderCombo->setCurrentIndex( order.second == Qt::DescendingOrder ? 1 : 0 );
}

QPair<int, Qt::SortOrder> ColumnSettingsDialog::sortOrder()
{
    QPair<int, Qt::SortOrder> order;

    order.first = m_currentList->item( m_sortCombo->currentIndex() )->data( Qt::UserRole ).toInt();
    order.second = m_orderCombo->currentIndex() == 1 ? Qt::DescendingOrder : Qt::AscendingOrder;

    return order;
}

void ColumnSettingsDialog::availableSelected()
{
    if ( m_availableList->currentItem() )
        m_addButton->setEnabled( true );
    else
        m_addButton->setEnabled( false );
}

void ColumnSettingsDialog::currentSelected()
{
    QListWidgetItem* item = m_currentList->currentItem();
    if ( item ) {
        int column = item->data( Qt::UserRole ).toInt();
        int row = m_currentList->row( item );

        if ( !m_fixedColumns.contains( column ) )
            m_removeButton->setEnabled( true );
        else
            m_removeButton->setEnabled( false );

        if ( row > 0 )
            m_upButton->setEnabled( true );
        else
            m_upButton->setEnabled( false );

        if ( row < m_currentList->count() - 1 )
            m_downButton->setEnabled( true );
        else
            m_downButton->setEnabled( false );
    } else {
        m_removeButton->setEnabled( false );
        m_upButton->setEnabled( false );
        m_downButton->setEnabled( false );
    }
}

void ColumnSettingsDialog::addClicked()
{
    QListWidgetItem* item = m_availableList->currentItem();

    m_availableList->takeItem( m_availableList->row( item ) );

    QListWidgetItem* afterItem = m_currentList->currentItem();
    if ( afterItem ) {
        int position = m_currentList->row( afterItem ) + 1;
        m_currentList->insertItem( position, item );
        m_sortCombo->insertItem( position, item->text() );
    } else {
        m_currentList->addItem( item );
        m_sortCombo->addItem( item->text() );
    }

    m_currentList->setCurrentItem( item );

    m_buttonBox->button( QDialogButtonBox::Ok )->setEnabled( true );
    m_buttonBox->button( QDialogButtonBox::Apply )->setEnabled( true );
}

void ColumnSettingsDialog::removeClicked()
{
    QListWidgetItem* item = m_currentList->currentItem();

    int column = item->data( Qt::UserRole ).toInt();

    if ( m_fixedColumns.contains( column ) )
        return;

    int position = m_currentList->row( item );
    m_currentList->takeItem( position );
    m_sortCombo->removeItem( position );

    m_availableList->addItem( item );
    m_availableList->sortItems( Qt::AscendingOrder );

    currentSelected();

    if ( m_currentList->count() > 0 ) {
        m_buttonBox->button( QDialogButtonBox::Ok )->setEnabled( true );
        m_buttonBox->button( QDialogButtonBox::Apply )->setEnabled( true );
    } else {
        m_buttonBox->button( QDialogButtonBox::Ok )->setEnabled( false );
        m_buttonBox->button( QDialogButtonBox::Apply )->setEnabled( false );
    }
}

void ColumnSettingsDialog::upClicked()
{
    moveItem( -1 );
}

void ColumnSettingsDialog::downClicked()
{
    moveItem( 1 );
}

void ColumnSettingsDialog::moveItem( int offset )
{
    QListWidgetItem* item = m_currentList->currentItem();

    int row = m_currentList->row( item );
    if ( row + offset < 0 || row + offset > m_currentList->count() - 1 )
        return;

    m_currentList->takeItem( row );
    m_currentList->insertItem( row + offset, item );

    bool selected = m_sortCombo->currentIndex() == row;
    m_sortCombo->removeItem( row );
    m_sortCombo->insertItem( row + offset, item->text() );
    if ( selected )
        m_sortCombo->setCurrentIndex( row + offset );

    m_currentList->setCurrentItem( item );

    currentSelected();
}

void ColumnSettingsDialog::accept()
{
    emit settingsApplied();

    QDialog::accept();
}

void ColumnSettingsDialog::restoreDefaults()
{
    setColumns( m_defaultColumns );
    setSortOrder( m_defaultOrder );
}
