/**************************************************************************
* This file is part of the WebIssues program
* Copyright (C) 2006 Michał Męciński
* Copyright (C) 2007-2008 WebIssues Team
*
* This program is free software; you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation; either version 2 of the License, or
* (at your option) any later version.
**************************************************************************/

#include "commanddialog.h"

#include <QLayout>
#include <QLabel>
#include <QProgressBar>
#include <QDialogButtonBox>
#include <QPushButton>

#include "commands/abstractbatch.h"
#include "commands/commandmanager.h"
#include "widgets/gradientwidget.h"
#include "iconloader.h"

using namespace WebIssues;

/* TRANSLATOR WebIssues::CommandDialog */

CommandDialog::CommandDialog( QWidget* parent ) : QDialog( parent ),
    m_promptPixmap( NULL ),
    m_contentLayout( NULL ),
    m_progressBar( NULL ),
    m_progressLabel( NULL ),
    m_batch( NULL )
{
    QVBoxLayout* topLayout = new QVBoxLayout( this );
    topLayout->setMargin( 0 );
    topLayout->setSpacing( 0 );

    m_promptWidget = new GradientWidget( this );
    topLayout->addWidget( m_promptWidget );

    m_promptLayout = new QHBoxLayout( m_promptWidget );
    m_promptLayout->setSpacing( 10 );

    m_promptLabel = new QLabel( m_promptWidget );
    m_promptLabel->setWordWrap( true );
    m_promptLayout->addWidget( m_promptLabel, 1 );

    QFrame* separator = new QFrame( this );
    separator->setFrameStyle( QFrame::HLine | QFrame::Sunken );
    topLayout->addWidget( separator );

    m_mainLayout = new QVBoxLayout();
    m_mainLayout->setMargin( 9 );
    m_mainLayout->setSpacing( 6 );
    topLayout->addLayout( m_mainLayout );

    QHBoxLayout* statusLayout = new QHBoxLayout();
    m_mainLayout->addLayout( statusLayout );

    m_statusPixmap = new QLabel( this );
    statusLayout->addWidget( m_statusPixmap, 0 );

    m_statusLabel = new QLabel( this );
    statusLayout->addWidget( m_statusLabel, 1 );

    m_buttonBox = new QDialogButtonBox( QDialogButtonBox::Ok | QDialogButtonBox::Cancel,
        Qt::Horizontal, this );
    m_mainLayout->addWidget( m_buttonBox );

    m_buttonBox->button( QDialogButtonBox::Ok )->setText( tr( "&OK" ) );
    m_buttonBox->button( QDialogButtonBox::Cancel )->setText( tr( "&Cancel" ) );

    connect( m_buttonBox, SIGNAL( accepted() ), this, SLOT( accept() ) );
    connect( m_buttonBox, SIGNAL( rejected() ), this, SLOT( reject() ) );

    showInfo( tr( "Enter command arguments" ) );
}

CommandDialog::~CommandDialog()
{
}

void CommandDialog::setPrompt( const QString& text )
{
    m_promptLabel->setText( text );

    m_promptLabel->setMinimumWidth( 350 );
    m_promptLabel->setFixedHeight( m_promptLabel->heightForWidth( 350 ) );
}

void CommandDialog::setPromptPixmap( const QPixmap& pixmap )
{
    if ( !m_promptPixmap ) {
        m_promptPixmap = new QLabel( m_promptWidget );
        m_promptLayout->insertWidget( 0, m_promptPixmap, 0 );
    }

    m_promptPixmap->setPixmap( pixmap );
}

void CommandDialog::setContentLayout( QLayout* layout, bool fixed )
{
    if ( layout ) {
        m_contentLayout = layout;

        m_mainLayout->insertLayout( 0, layout );

        QFrame* separator = new QFrame( this );
        separator->setFrameStyle( QFrame::HLine | QFrame::Sunken );
        m_mainLayout->insertWidget( 1, separator );
    }

    if ( fixed )
        setMaximumHeight( sizeHint().height() );
}

void CommandDialog::createProgressPanel( int total, const QString& text )
{
    m_progressBar = new QProgressBar( this );
    m_progressBar->setRange( 0, total );
    m_progressBar->setTextVisible( false );
    m_progressBar->setEnabled( false );
    m_mainLayout->insertWidget( 2, m_progressBar );

    m_progressLabel = new QLabel( text, this );
    m_mainLayout->insertWidget( 3, m_progressLabel );

    QFrame* separator = new QFrame( this );
    separator->setFrameStyle( QFrame::HLine | QFrame::Sunken );
    m_mainLayout->insertWidget( 4, separator );

    m_progressText = text;
}

void CommandDialog::setProgress( int done, const QString& text )
{
    m_progressBar->setValue( done );
    m_progressLabel->setText( text );
}

void CommandDialog::showInfo( const QString& text )
{
    m_statusPixmap->setPixmap( IconLoader::pixmap( "status-info" ) );
    m_statusLabel->setText( text );
}

void CommandDialog::showWarning( const QString& text )
{
    m_statusPixmap->setPixmap( IconLoader::pixmap( "status-warning" ) );
    m_statusLabel->setText( text );
}

void CommandDialog::executeBatch( AbstractBatch* batch )
{
    if ( m_contentLayout )
        setWidgetsEnabled( m_contentLayout, false );

    if ( m_progressBar )
        m_progressBar->setEnabled( true );

    m_buttonBox->button( QDialogButtonBox::Ok )->hide();
    m_buttonBox->button( QDialogButtonBox::Cancel )->setText( tr( "&Abort" ) );

    m_statusPixmap->setPixmap( IconLoader::pixmap( "status-busy" ) );
    m_statusLabel->setText( tr( "Executing command..." ) );

    connect( batch, SIGNAL( completed( bool ) ), this, SLOT( batchCompleted( bool ) ) );

    m_batch = batch;

    commandManager->execute( batch );
}

void CommandDialog::reject()
{
    if ( m_batch ) {
        commandManager->abort( m_batch );
        return;
    }

    QDialog::reject();
}

void CommandDialog::batchCompleted( bool successful )
{
    if ( successful ) {
        if ( batchSuccessful( m_batch ) ) {
            QDialog::accept();
            m_batch = NULL;
            return;
        }
    } else {
        showWarning( commandManager->errorMessage( tr( "Command could not be executed" ) ) );
    }

    if ( m_contentLayout )
        setWidgetsEnabled( m_contentLayout, true );

    if ( m_progressBar ) {
        m_progressBar->setEnabled( false );
        setProgress( 0, m_progressText );
    }

    m_buttonBox->button( QDialogButtonBox::Ok )->show();
    m_buttonBox->button( QDialogButtonBox::Cancel )->setText( tr( "&Cancel" ) );

    m_batch = NULL;
}

bool CommandDialog::batchSuccessful( AbstractBatch* batch )
{
    return true;
}

void CommandDialog::setWidgetsEnabled( QLayoutItem* item, bool enabled )
{
    QLayout* layout = item->layout();
    if ( layout ) {
        for ( int i = 0; i < layout->count(); i++ )
            setWidgetsEnabled( layout->itemAt( i ), enabled );
    }

    QWidget* widget = item->widget();
    if ( widget )
        widget->setEnabled( enabled );
}
