/**************************************************************************
* This file is part of the WebIssues program
* Copyright (C) 2006 Michał Męciński
* Copyright (C) 2007-2008 WebIssues Team
*
* This program is free software; you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation; either version 2 of the License, or
* (at your option) any later version.
**************************************************************************/

#include "connectionmanager.h"

// fix for GCC 4.3-snapshot
#include <cstdlib>

#include <QSettings>
#include <QDir>

#if defined( Q_WS_WIN )
#define _WIN32_IE 0x0400
#include <shlobj.h>
#endif

#include "commands/commandmanager.h"
#include "data/datamanager.h"
#include "data/connectioninfo.h"
#include "data/attachmentscache.h"
#include "viewmanager.h"
#include "configdata.h"

using namespace WebIssues;

ConnectionManager* WebIssues::connectionManager = NULL;

ConnectionManager::ConnectionManager() :
    m_info( NULL ),
    m_settings( NULL ),
    m_attachmentsCache( NULL )
{
    QString homePath = QDir::homePath();

#if defined( Q_WS_WIN )
    char appDataPath[ MAX_PATH ];
    if ( SHGetSpecialFolderPathA( 0, appDataPath, CSIDL_APPDATA, FALSE ) ) {
        m_dataPath = QDir::fromNativeSeparators( QString::fromLocal8Bit( appDataPath ) );
        m_dataPath += "/WebIssues";
    } else {
        m_dataPath = homePath + "/WebIssues";
    }

    char localAppDataPath[ MAX_PATH ];
    if ( SHGetSpecialFolderPathA( 0, localAppDataPath, CSIDL_LOCAL_APPDATA, FALSE ) ) {
        m_cachePath = QDir::fromNativeSeparators( QString::fromLocal8Bit( localAppDataPath ) );
        m_cachePath += "/WebIssues";
    } else {
        m_cachePath = m_dataPath + "/.cache";
    }
#else
    char* dataHome = getenv( "XDG_DATA_HOME" );
    if ( dataHome ) {
        if ( dataHome[ 0 ] == '/' )
            m_dataPath = dataHome;
        else
            m_dataPath = homePath + '/' + dataHome;
    } else {
        m_dataPath = homePath + "/.local/share";
    }

    m_dataPath += "/webissues";

    char* cacheHome = getenv( "XDG_CACHE_HOME" );
    if ( cacheHome ) {
        if ( cacheHome[ 0 ] == '/' )
            m_cachePath = cacheHome;
        else
            m_cachePath = homePath + '/' + cacheHome;
    } else {
        m_cachePath = homePath + "/.cache";
    }

    m_cachePath += "/webissues";
#endif
}

ConnectionManager::~ConnectionManager()
{
    closeConnection();
}

void ConnectionManager::createCommandManager()
{
    QNetworkProxy::setApplicationProxy( configData->networkProxy() );

    commandManager = new CommandManager();
}

void ConnectionManager::openConnection( ConnectionInfo* info )
{
    m_info = info;

    QString path = locateDataFile( "server.conf" );

    m_settings = new QSettings( path, QSettings::IniFormat );

    m_attachmentsCache = new AttachmentsCache();

    dataManager = new DataManager();
    viewManager = new ViewManager();
}

void ConnectionManager::delayedCloseConnection()
{
    QMetaObject::invokeMethod( this, "closeConnection", Qt::QueuedConnection );
}

void ConnectionManager::closeConnection()
{
    if ( commandManager != NULL )
        commandManager->abortAll();

    delete viewManager;
    viewManager = NULL;

    delete dataManager;
    dataManager = NULL;

    if ( commandManager != NULL ) {
        commandManager->deleteLater();
        commandManager = NULL;
    }

    delete m_info;
    m_info = NULL;

    delete m_settings;
    m_settings = NULL;

    delete m_attachmentsCache;
    m_attachmentsCache = NULL;
}

QString ConnectionManager::locateDataFile( const QString& name )
{
    QString path = m_dataPath + '/' + m_info->serverUuid() + '/' + name;

    checkAccess( path );

    return path;
}

QString ConnectionManager::locateCacheFile( const QString& name )
{
    QString path = m_cachePath + '/' + m_info->serverUuid() + '/' + name;

    checkAccess( path );

    return path;
}

bool ConnectionManager::checkAccess( const QString& path )
{
    QFileInfo fileInfo( path );

    if ( fileInfo.exists() )
        return fileInfo.isReadable();

    QDir dir = QDir::root();

    QStringList pathParts = path.split( '/', QString::SkipEmptyParts );

    for ( int i = 0; i < pathParts.size() - 1; i++ ) {
        const QString& part = pathParts.at( i );

        if ( dir.cd( part ) )
            continue;

        if ( dir.mkdir( part ) && dir.cd( part ) )
            continue;

        return false;
    }

    return true;
}

void ConnectionManager::exportSettings( QSettings* settings )
{
    copyAllKeys( settings, m_settings );

    settings->setValue( "ServerUuid", m_info->serverUuid() );
}

bool ConnectionManager::canImportSettings( const QSettings* settings )
{
    return settings->value( "ServerUuid", QString() ).toString() == m_info->serverUuid();
}

void ConnectionManager::importSettings( const QSettings* settings )
{
    m_settings->clear();

    copyAllKeys( m_settings, settings );
}

void ConnectionManager::copyAllKeys( QSettings* dest, const QSettings* src )
{
    QStringList keys = src->allKeys();

    for ( int i = 0; i < keys.count(); i++ ) {
        if ( keys[ i ] == "ServerUuid" )
            continue;
        dest->setValue( keys[ i ], src->value( keys[ i ] ) );
    }
}
