"""Helpers that produce simple HTML tags.

Most helpers have an ``**attrs`` argument to specify additional HTML
attributes.  A trailing underscore in the name will be deleted; this is 
especially important for attributes that are identical to Python keywords;
e.g., ``class_``.  Some helpers handle certain keywords specially; these are
noted in the helpers' docstrings.

To create your own custom tags, see ``webhelpers.html.builder``.

A set of CSS styles complementing these helpers is in
``webhelpers/public/stylesheets/webhelpers.css``.
"""

import datetime
import logging
import os
import re
import urllib
import urlparse

from webhelpers import containers
from webhelpers.html import escape, HTML, literal, url_escape
import webhelpers.media as media
from webhelpers.misc import NotGiven

__all__ = [
           # Form tags
           "form", "end_form", 
           "text", "textarea", "hidden", "file", "password", 
           "checkbox", "radio", "submit",
           "select", "Options", "Option", "OptGroup",
           "ModelTags", "title", "required_legend",
           # hyperlinks
           "link_to", "link_to_if", "link_to_unless",
           # Table tags
           "th_sortable",
           # Other non-form tags
           "ol", "ul", "image", "BR",
           # Head tags and document type
           "stylesheet_link", "javascript_link", "auto_discovery_link",
           "Doctype", "xml_declaration",
           # Utility functions
           "css_classes", "convert_boolean_attrs",
           ]

log = logging.getLogger(__name__)

NL = literal("\n")
BR = literal("<br />\n")

def form(url, method="post", multipart=False, hidden_fields=None, **attrs):
    """An open tag for a form that will submit to ``url``.

    You must close the form yourself by calling ``end_form()`` or outputting
    </form>.
    
    Options:

    ``method``
        The method to use when submitting the form, usually either 
        "GET" or "POST". If "PUT", "DELETE", or another verb is used, a
        hidden input with name _method is added to simulate the verb
        over POST.
    
    ``multipart``
        If set to True, the enctype is set to "multipart/form-data".
        You must set it to true when uploading files, or the browser will
        submit the filename rather than the file.

    ``hidden_fields``
        Additional hidden fields to add to the beginning of the form.  It may
        be a dict or an iterable of key-value tuples. This is implemented by
        calling the object's ``.items()`` method if it has one, or just
        iterating the object.  (This will successfuly get multiple values for
        the same key in WebOb MultiDict objects.)

    Because input tags must be placed in a block tag rather than directly
    inside the form, all hidden fields will be put in a 
    '<div style="display:none">'.  The style prevents the <div> from being
    displayed or affecting the layout.

    Examples:

    >>> form("/submit")
    literal(u'<form action="/submit" method="post">')
    >>> form("/submit", method="get")
    literal(u'<form action="/submit" method="get">')
    >>> form("/submit", method="put")
    literal(u'<form action="/submit" method="post"><div style="display:none">\\n<input id="_method" name="_method" type="hidden" value="put" />\\n</div>\\n')
    >>> form("/submit", "post", multipart=True) 
    literal(u'<form action="/submit" enctype="multipart/form-data" method="post">')

    Changed in WebHelpers 2.0b2: add <div> and ``hidden_fields`` arg.
    """
    fields = []
    attrs["action"] = url
    if multipart:
        attrs["enctype"] = "multipart/form-data"
    if method.lower() in ['post', 'get']:
        attrs['method'] = method
    else:
        attrs['method'] = "post"
        field = hidden("_method", method)
        fields.append(field)
    if hidden_fields is not None:
        try:
            it = hidden_fields.items()
        except AttributeError:
            it = hidden_fields
        for name, value in it:
            field = hidden(name, value)
            fields.append(field)
    if fields:
        div = HTML.div(style="display:none", _nl=True, *fields)
    else:
        div = None
    return HTML.form(div, _closed=False, **attrs)


def end_form():
    """Output "</form>".
    
    Example::

        >>> end_form()
        literal(u'</form>')
    """
    return literal("</form>")


def text(name, value=None, id=NotGiven, **attrs):
    """Create a standard text field.
    
    ``value`` is a string, the content of the text field.

    ``id`` is the HTML ID attribute, and should be passed as a keyword
    argument.  By default the ID is the same as the name filtered through
    ``_make_safe_id_component()``.  Pass None to suppress the
    ID attribute entirely.

    
    Options:
    
    * ``disabled`` - If set to True, the user will not be able to use
        this input.
    * ``size`` - The number of visible characters that will fit in the
        input.
    * ``maxlength`` - The maximum number of characters that the browser
        will allow the user to enter.
    
    The remaining keyword args will be standard HTML attributes for the tag.
    
    """
    _set_input_attrs(attrs, "text", name, value)
    _set_id_attr(attrs, id, name)
    convert_boolean_attrs(attrs, ["disabled"])
    return HTML.input(**attrs)


def hidden(name, value=None, id=NotGiven, **attrs):
    """Create a hidden field.
    """
    _set_input_attrs(attrs, "hidden", name, value)
    _set_id_attr(attrs, id, name)
    return HTML.input(**attrs)


def file(name, value=None, id=NotGiven, **attrs):
    """Create a file upload field.
    
    If you are using file uploads then you will also need to set the 
    multipart option for the form.

    Example::

        >>> file('myfile')
        literal(u'<input id="myfile" name="myfile" type="file" />')
    
    """
    _set_input_attrs(attrs, "file", name, value)
    _set_id_attr(attrs, id, name)
    return HTML.input(**attrs)


def password(name, value=None, id=NotGiven, **attrs):
    """Create a password field.
    
    Takes the same options as ``text()``.
    
    """
    _set_input_attrs(attrs, "password", name, value)
    _set_id_attr(attrs, id, name)
    return HTML.input(**attrs)


def textarea(name, content="", id=NotGiven, **attrs):
    """Create a text input area.
    
    Example::
    
        >>> textarea("body", "", cols=25, rows=10)
        literal(u'<textarea cols="25" id="body" name="body" rows="10"></textarea>')
    
    """
    attrs["name"] = name
    _set_id_attr(attrs, id, name)
    return HTML.textarea(content, **attrs)


def checkbox(name, value="1", checked=False, label=None, id=NotGiven, **attrs):
    """Create a check box.

    Arguments:
    ``name`` -- the widget's name.

    ``value`` -- the value to return to the application if the box is checked.

    ``checked`` -- true if the box should be initially checked.

    ``label`` -- a text label to display to the right of the box.

    ``id`` is the HTML ID attribute, and should be passed as a keyword
    argument.  By default the ID is the same as the name filtered through
    ``_make_safe_id_component()``.  Pass None to suppress the
    ID attribute entirely.

    The following HTML attributes may be set by keyword argument:

    * ``disabled`` - If true, checkbox will be grayed out.

    * ``readonly`` - If true, the user will not be able to modify the checkbox.

    To arrange multiple checkboxes in a group, see
    webhelpers.containers.distribute().

    Example::
    
        >>> checkbox("hi")
        literal(u'<input id="hi" name="hi" type="checkbox" value="1" />')
    """
    _set_input_attrs(attrs, "checkbox", name, value)
    _set_id_attr(attrs, id, name)
    if checked:
        attrs["checked"] = "checked"
    convert_boolean_attrs(attrs, ["disabled", "readonly"])
    widget = HTML.input(**attrs)
    if label:
        widget = HTML.label(widget, label)
    return widget

def radio(name, value, checked=False, label=None, **attrs):
    """Create a radio button.

    Arguments:
    ``name`` -- the field's name.

    ``value`` -- the value returned to the application if the button is
    pressed.

    ``checked`` -- true if the button should be initially pressed.

    ``label`` -- a text label to display to the right of the button.
    
    The id of the radio button will be set to the name + '_' + value to 
    ensure its uniqueness.  An ``id`` keyword arg overrides this.  (Note
    that this behavior is unique to the ``radio()`` helper.)
    
    To arrange multiple radio buttons in a group, see
    webhelpers.containers.distribute().
    """
    _set_input_attrs(attrs, "radio", name, value)
    if checked:
        attrs["checked"] = "checked"
    if not "id" in attrs:
        attrs["id"] = '%s_%s' % (name, _make_safe_id_component(value))
    widget = HTML.input(**attrs)
    if label:
        widget = HTML.label(widget, label)
    return widget


def submit(name, value, id=NotGiven, **attrs):
    """Create a submit button with the text ``value`` as the caption."""
    _set_input_attrs(attrs, "submit", name, value)
    _set_id_attr(attrs, id, name)
    return HTML.input(**attrs)


def select(name, selected_values, options, id=NotGiven, **attrs):
    """Create a dropdown selection box.

    * ``name`` -- the name of this control.

    * ``selected_values`` -- a string or list of strings or integers giving
      the value(s) that should be preselected.

    * ``options`` -- an ``Options`` object or iterable of ``(value, label)``
      pairs.  The label will be shown on the form; the option will be returned
      to the application if that option is chosen.  If you pass a string or int
      instead of a 2-tuple, it will be used for both the value and the label.
      If the `value` is a tuple or a list, it will be added as an optgroup,
      with `label` as label.

    ``id`` is the HTML ID attribute, and should be passed as a keyword
    argument.  By default the ID is the same as the name.  filtered through
    ``_make_safe_id_component()``.  Pass None to suppress the
    ID attribute entirely.


      CAUTION: the old rails helper ``options_for_select`` had the label first.
      The order was reversed because most real-life collections have the value
      first, including dicts of the form ``{value: label}``.  For those dicts
      you can simply pass ``D.items()`` as this argument.

      HINT: You can sort options alphabetically by label via:
      ``sorted(my_options, key=lambda x: x[1])``

    The following options may only be keyword arguments:

    * ``multiple`` -- if true, this control will allow multiple
       selections.

    * ``prompt`` -- if specified, an extra option will be prepended to the 
      list: ("", ``prompt``).  This is intended for those "Please choose ..."
      pseudo-options.  Its value is "", equivalent to not making a selection.

    Any other keyword args will become HTML attributes for the <select>.

    Examples (call, result)::
    
        >>> select("currency", "$", [["$", "Dollar"], ["DKK", "Kroner"]])
        literal(u'<select id="currency" name="currency">\\n<option selected="selected" value="$">Dollar</option>\\n<option value="DKK">Kroner</option>\\n</select>')
        >>> select("cc", "MasterCard", [ "VISA", "MasterCard" ], id="cc", class_="blue")
        literal(u'<select class="blue" id="cc" name="cc">\\n<option value="VISA">VISA</option>\\n<option selected="selected" value="MasterCard">MasterCard</option>\\n</select>')
        >>> select("cc", ["VISA", "Discover"], [ "VISA", "MasterCard", "Discover" ])
        literal(u'<select id="cc" name="cc">\\n<option selected="selected" value="VISA">VISA</option>\\n<option value="MasterCard">MasterCard</option>\\n<option selected="selected" value="Discover">Discover</option>\\n</select>')
        >>> select("currency", None, [["$", "Dollar"], ["DKK", "Kroner"]], prompt="Please choose ...")
        literal(u'<select id="currency" name="currency">\\n<option selected="selected" value="">Please choose ...</option>\\n<option value="$">Dollar</option>\\n<option value="DKK">Kroner</option>\\n</select>')
        >>> select("privacy", 3L, [(1, "Private"), (2, "Semi-public"), (3, "Public")])
        literal(u'<select id="privacy" name="privacy">\\n<option value="1">Private</option>\\n<option value="2">Semi-public</option>\\n<option selected="selected" value="3">Public</option>\\n</select>')
        >>> select("recipients", None, [([("u1", "User1"), ("u2", "User2")], "Users"), ([("g1", "Group1"), ("g2", "Group2")], "Groups")])
        literal(u'<select id="recipients" name="recipients">\\n<optgroup label="Users">\\n<option value="u1">User1</option>\\n<option value="u2">User2</option>\\n</optgroup>\\n<optgroup label="Groups">\\n<option value="g1">Group1</option>\\n<option value="g2">Group2</option>\\n</optgroup>\\n</select>')
        
    """
    _set_id_attr(attrs, id, name)
    attrs["name"] = name
    convert_boolean_attrs(attrs, ["multiple"])
    # Accept None as selected_values meaning that no option is selected
    if selected_values is None:
        selected_values = ('',)
    # Turn a single string or integer into a list
    elif isinstance(selected_values, (basestring, int, long)):
        selected_values = (selected_values,)
    # Cast integer values to strings
    selected_values = map(unicode, selected_values)
    # Prepend the prompt
    prompt = attrs.pop("prompt", None)
    if prompt:
        options = [Option("", prompt)] + list(options)
    # Canonicalize the options and make the HTML options.
    if not isinstance(options, Options):
        options = Options(options)
    html_options = []
    # Create the options structure
    def gen_opt(val, label):
        if val in selected_values:
            return HTML.option(label, value=val, selected="selected")
        else:
            return HTML.option(label, value=val)
    # Loop options and create tree (if optgroups presents)
    for opt in options:
        if isinstance(opt, OptGroup):
            optgroup_options = []
            for subopt in opt.options:
                optgroup_options.append(gen_opt(subopt.value, subopt.label))
            optgroup = HTML.optgroup(NL, NL.join(optgroup_options), NL, label=opt.label)
            html_options.append(optgroup)
        else:
            html_options.append(gen_opt(opt.value, opt.label))
    return HTML.select(NL, NL.join(html_options), NL, **attrs)


class ModelTags(object):
    """A nice way to build a form for a database record.
    
    ModelTags allows you to build a create/update form easily.  (This is the
    C and U in CRUD.)  The constructor takes a database record, which can be
    a SQLAlchemy mapped class, or any object with attributes or keys for the
    field values.  Its methods shadow the the form field helpers, but it
    automatically fills in the value attribute based on the current value in
    the record.  (It also knows about the 'checked' and 'selected' attributes
    for certain tags.)

    You can also use the same form  to input a new record.  Pass ``None`` or
    ``""`` instead of a record, and it will set all the current values to a
    default value, which is either the `default` keyword arg to the method, or
    `""` if not specified.

    (Hint: in Pylons you can put ``mt = ModelTags(c.record)`` in your template,
    and then if the record doesn't exist you can either set ``c.record = None``
    or not set it at all.  That's because nonexistent ``c`` attributes resolve
    to `""` unless you've set ``config["pylons.strict_c"] = True``. However,
    having a ``c`` attribute that's sometimes set and sometimes not is
    arguably bad programming style.)
    """

    undefined_values = set([None, ""])

    def __init__(self, record, use_keys=False, date_format="%m/%d/%Y", 
        id_format=None):
        """Create a ``ModelTags`` object.

        ``record`` is the database record to lookup values in.  It may be
        any object with attributes or keys, including a SQLAlchemy mapped
        instance.  It may also be ``None`` or ``""`` to indicate that a new
        record is being created.  (The class attribute ``undefined_values``
        tells which values indicate a new record.)

        If ``use_keys`` is true, values will be looked up by key.  If false
        (default), values will be looked up by attribute.

        ``date_format`` is a strftime-compatible string used by the ``.date``
        method.  The default is American format (MM/DD/YYYY), which is
        most often seen in text fields paired with popup calendars.
        European format (DD/MM/YYYY) is "%d/%m/%Y".  ISO format (YYYY-MM-DD)
        is "%Y-%m-%d".

        ``id_format`` is a formatting-operator format for the HTML 'id' attribute.
        It should contain one "%s" where the tag's name will be embedded.
        """
        self.record = record
        self.use_keys = use_keys
        self.date_format = date_format
        self.id_format = id_format
    
    def checkbox(self, name, value='1', label=None, **kw):
        """Build a checkbox field.
        
        The box will be initially checked if the value of the corresponding
        database field is true.

        The submitted form value will be "1" if the box was checked. If the
        box is unchecked, no value will be submitted. (This is a downside of
        the standard checkbox tag.)

        To display multiple checkboxes in a group, see
        webhelper.containers.distribute().
        """
        self._update_id(name, kw)
        value = kw.pop("value", "1")
        checked = bool(self._get_value(name, kw))
        return checkbox(name, value, checked, label, **kw)

    def date(self, name, **kw):
        """Same as text but format a date value into a date string.

        The value can be a `datetime.date`, `datetime.datetime`, `None`,
        or `""`.  The former two are converted to a string using the
        date format passed to the constructor.  The latter two are converted
        to "".

        If there's no database record, consult keyword arg `default`. It it's
        the string "today", use todays's date. Otherwise it can be any of the
        values allowed above.  If no default is specified, the text field is
        initialized to "".

        Hint: you may wish to attach a Javascript calendar to the field.
        """
        self._update_id(name, kw)
        value = self._get_value(name, kw)
        if isinstance(value, datetime.date):
            value = value.strftime(self.date_format)
        elif value == "today":
            value = datetime.date.today().strftime(self.date_format)
        else:
            value = ""
        return text(name, value, **kw)

    def file(self, name, **kw):
        """Build a file upload field.
        
        User agents may or may not respect the contents of the 'value' attribute."""
        self._update_id(name, kw)
        value = self._get_value(name, kw)
        return file(name, value, **kw)

    def hidden(self, name, **kw):
        """Build a hidden HTML field."""
        self._update_id(name, kw)
        value = self._get_value(name, kw)
        return hidden(name, value, **kw)

    def password(self, name, **kw):
        """Build a password field.
        
        This is the same as a text box but the value will not be shown on the
        screen as the user types.
        """
        self._update_id(name, kw)
        value = self._get_value(name, kw)
        return password(name, value, **kw)

    def radio(self, name, checked_value, label=None, **kw):
        """Build a radio button.

        The radio button will initially be selected if the database value 
        equals ``checked_value``.  On form submission the value will be 
        ``checked_value`` if the button was selected, or ``""`` otherwise.

        In case of a ModelTags object that is created from scratch
        (e.g. ``new_employee=ModelTags(None)``) the option that should
        be checked can be set by the 'default' parameter. As in:
        ``new_employee.radio('status', checked_value=7, default=7)``

        The control's 'id' attribute will be modified as follows:

        1. If not specified but an 'id_format' was given to the constructor,
           generate an ID based on the format.
        2. If an ID was passed in or was generated by step (1), append an
           underscore and the checked value.  Before appending the checked
           value, lowercase it, change any spaces to ``"_"``, and remove any
           non-alphanumeric characters except underscores and hyphens.
        3. If no ID was passed or generated by step (1), the radio button 
           will not have an 'id' attribute.

        To display multiple radio buttons in a group, see
        webhelper.containers.distribute().
        """
        self._update_id(name, kw)
        value = self._get_value(name, kw)
        if 'id' in kw:
            kw["id"] = '%s_%s' % (kw['id'], _make_safe_id_component(checked_value))
        checked = (value == checked_value)
        return radio(name, checked_value, checked, label, **kw)

    def select(self, name, options, **kw):
        """Build a dropdown select box or list box.

        See the ``select()`` function for the meaning of the arguments.

        If the corresponding database value is not a list or tuple, it's
        wrapped in a one-element list.  But if it's "" or ``None``, an empty
        list is substituted.  This is to accommodate multiselect lists, which
        may have multiple values selected. """
        self._update_id(name, kw)
        selected_values = self._get_value(name, kw)
        if not isinstance(selected_values, (list, tuple)):
            if selected_values in ["", None]:
                selected_values = []
            else:
                selected_values = [selected_values]
        return select(name, selected_values, options, **kw)

    def text(self, name, **kw):
        """Build a text box."""
        self._update_id(name, kw)
        value = self._get_value(name, kw)
        return text(name, value, **kw)

    def textarea(self, name, **kw):
        """Build a rectangular text area."""
        self._update_id(name, kw)
        content = self._get_value(name, kw)
        return textarea(name, content, **kw)

    # Private methods.
    def _get_value(self, name, kw):
        """Get the current value of a field from the database record.

        ``name``: The field to look up.

        ``kw``: The keyword args passed to the original method.  This is
        _not_ a "\*\*" argument!  It's a dict that will be modified in place!

        ``kw["default"]`` will be popped from the dict in all cases for
        possible use as a default value.  If the record doesn't exist, this
        default is returned, or ``""`` if no default was passed.
        """
        default = kw.pop("default", "")
        # This used to be ``self.record in self.undefined_values``, but this
        # fails if the record is a dict because dicts aren't hashable.
        for undefined_value in self.undefined_values:
            if self.record == undefined_value:
                return default
        if self.use_keys:
            return self.record[name]    # Raises KeyError.
        else:
            return getattr(self.record, name)   # Raises AttributeError.

    def _update_id(self, name, kw):
        """Apply the 'id' attribute algorithm.

        ``name``: The name of the HTML field.

        ``kw``: The keyword args passed to the original method.  This is
        _not_ a "\*\*" argument!  It's a dict that will be modified in place!

        If an ID format was specified but no 'id' keyword was passed, 
        set the 'id' attribute to a value generated from the format and name.
        Otherwise do nothing.
        """
        if self.id_format is not None and 'id' not in kw:
            kw['id'] = self.id_format % name


class Option(object):
    """An option for an HTML select.
    
    A simple container with two attributes, ``.value`` and ``.label``.
    """
    __slots__ = ("value", "label")


    def __repr__(self):
        return str((self.value, self.label))
    def __init__(self, value, label):
        self.value = value
        self.label = label

class OptGroup(object):
    """A container for Options"""
    __slots__ = ('options', 'label')
    def __init__(self, label, options):
        self.options = Options(options)
        self.label = label
    def __repr__(self):
        classname = self.__class__.__name__
        data = [x for x in self.options]
        return "%s(%s, %s)" % (classname, data, repr(self.label))

class Options(tuple):
    """A tuple of ``Option`` objects for the ``select()`` helper.

    ``select()`` calls this automatically so you don't have to.  However,
    you may find it useful for organizing your code, and its methods can be
    convenient.

    This class has multiple jobs:

    - Canonicalize the options given to ``select()`` into a consistent format.
    - Avoid canonicalizing the same data multiple times.  It subclasses tuple
      rather than a list to guarantee that nonconformant elements won't be 
      added after canonicalization.
    - Provide convenience methods to iterate the values and labels separately.

    >>> opts = Options(["A", 1, ("b", "B")])
    >>> opts
    Options([(u'A', u'A'), (u'1', u'1'), (u'b', u'B')])
    >>> list(opts.values())
    [u'A', u'1', u'b']
    >>> list(opts.labels())
    [u'A', u'1', u'B']
    >>> opts[2].value
    u'b'
    >>> opts[2].label
    u'B'
    """
    def __new__(class_, options):
        opts = []
        for opt in options:
            if isinstance(opt, (Option, OptGroup)):
                opts.append(opt)
                continue
            if isinstance(opt, (list, tuple)):
                value, label = opt[:2]
                if isinstance(value, (list, tuple)):  # It's an optgroup
                    opts.append(OptGroup(label, value))
                    continue
            else:
                value = label = opt
            if not isinstance(value, unicode):
                value = unicode(value)
            if not isinstance(label, unicode):  # Preserves literal.
                label = unicode(label)
            opt = Option(value, label)
            opts.append(opt)
        return super(Options, class_).__new__(class_, opts)

    def __repr__(self):
        classname = self.__class__.__name__
        data = [x for x in self]
        return "%s(%s)" % (classname, data)
        
    def values(self):
        """Iterate the value element of each pair."""
        return (x.value for x in self)

    def labels(self):
        """Iterate the label element of each pair."""
        return (x.label for x in self)

def title(title, required=False, label_for=None):
    """Format the user-visible title for a form field.

    Use this for forms that have a text title above or next to each
    field.

    ``title`` -- the name of the field; e.g., "First Name".

    ``required`` -- if true, append a \*" to the title and use the
    'required' HTML format (see example); otherwise use the 'not
    required' format.

    ``label_for`` -- if provided, put ``<label for="ID">`` around the
    title.  The value should be the HTML ID of the input field related
    to this title.  Per the HTML standard, the ID should point to a
    single control (input, select, textarea), not to multiple controls
    (fieldset, group of checkboxes, group of radio buttons).  ID's are
    set by passing the keyword arg ``id`` to the appropriate helper.
    
    Note that checkboxes and radio buttions typically have their own
    individual labels in addition to the title.  You can set these with
    the ``label`` argument to ``checkbox()`` and ``radio()``.

    This helper does not accept other keyword arguments.

    See webhepers/public/stylesheets/webhelpers.css for suggested styles.

    >>> title("First Name")
    literal(u'<span class="not-required">First Name</span>')
    >>> title("Last Name", True)
    literal(u'<span class="required">Last Name <span class="required-symbol">*</span></span>')
    >>> title("First Name", False, "fname")
    literal(u'<span class="not-required"><label for="fname">First Name</label></span>')
    >>> title("Last Name", True, label_for="lname")
    literal(u'<span class="required"><label for="lname">Last Name</label> <span class="required-symbol">*</span></span>')
    """
    title_html = title
    required_html = literal("")
    if label_for:
        title_html = HTML.label(title_html, for_=label_for)
    if required:
        required_symbol = HTML.span("*", class_="required-symbol")
        return HTML.span(
            title_html, 
            " ",
            required_symbol,
            class_="required")
    else:
        return HTML.span(title_html, class_="not-required")

def required_legend():
    """Return an inline HTML snippet explaining which fields are required.
    
    See webhepers/public/stylesheets/webhelpers.css for suggested styles.

    >>> required_legend()
    literal(u'<span class="required required-symbol">*</span> = required')
    """
    return HTML(
        HTML.span("*", class_="required required-symbol"), 
        " = required",
        )
        

#### Hyperlink tags

def link_to(label, url='', **attrs):
    """Create a hyperlink with the given text pointing to the URL.
    
    If the label is ``None`` or empty, the URL will be used as the label.

    This function does not modify the URL in any way.  The label will be
    escaped if it contains HTML markup.  To prevent escaping, wrap the label
    in a ``webhelpers.html.literal()``.
    """
    attrs['href'] = url
    if label == '' or label is None:
        label = url
    return HTML.a(label, **attrs)


def link_to_if(condition, label, url='', **attrs):
    """Same as ``link_to`` but return just the label if the condition is false.
    
    This is useful in a menu when you don't want the current option to be a
    link.  The condition will be something like:
    ``actual_value != value_of_this_menu_item``.
    """
    if condition:
        return link_to(label, url, **attrs)
    else:
        return label

def link_to_unless(condition, label, url='', **attrs):
    """The opposite of ``link_to``. Return just the label if the condition is 
    true.
    """
    if not condition:
        return link_to(label, url, **attrs)
    else:
        return label

#### Table tags
def th_sortable(current_order, column_order, label, url,
    class_if_sort_column="sort", class_if_not_sort_column=None, 
    link_attrs=None, name="th", **attrs):
    """<th> for a "click-to-sort-by" column.

    Convenience function for a sortable column.  If this is the current sort
    column, just display the label and set the cell's class to
    ``class_if_sort_column``.
    
    ``current_order`` is the table's current sort order.  ``column_order`` is
    the value pertaining to this column.  In other words, if the two are equal,
    the table is currently sorted by this column.

    If this is the sort column, display the label and set the <th>'s class to
    ``class_if_sort_column``.

    If this is not the sort column, display an <a> hyperlink based on
    ``label``, ``url``, and ``link_attrs`` (a dict), and set the <th>'s class
    to ``class_if_not_sort_column``.  
    
    ``url`` is the literal href= value for the link.  Pylons users would
    typically pass something like ``url=h.url_for("mypage", sort="date")``.

    ``**attrs`` are additional attributes for the <th> tag.

    If you prefer a <td> tag instead of <th>, pass ``name="td"``.

    To change the sort order via client-side Javascript, pass ``url=None`` and
    the appropriate Javascript attributes in ``link_attrs``.

    Examples:

    >>> sort = "name"
    >>> th_sortable(sort, "name", "Name", "?sort=name")
    literal(u'<th class="sort">Name</th>')
    >>> th_sortable(sort, "date", "Date", "?sort=date")
    literal(u'<th><a href="?sort=date">Date</a></th>')
    >>> th_sortable(sort, "date", "Date", None, link_attrs={"onclick": "myfunc()"})
    literal(u'<th><a onclick="myfunc()">Date</a></th>')
    """
    from webhelpers.html import HTML
    if current_order == column_order:
        content = label
        class_ = class_if_sort_column
    else:
        link_attrs = link_attrs or {}
        content = HTML.a(label, href=url, **link_attrs)
        class_ = class_if_not_sort_column
    return HTML.th(content, class_=class_, **attrs)



#### Other non-form tags

def ul(items, default=None, li_attrs=None, **attrs):
    R"""Return an unordered list with each item wrapped in <li>.

    ``items``
        list of strings.

    ``default``
        value returned _instead of the <ul>_ if there are no items in the list.
        If ``None``, return an empty <ul>.

    ``li_attrs``
        dict of attributes for the <li> tags.

    Examples:

    >>> ul(["foo", "bar"])
    literal(u'<ul>\n<li>foo</li>\n<li>bar</li>\n</ul>')
    >>> ul(["A", "B"], li_attrs={"class_": "myli"}, class_="mylist") 
    literal(u'<ul class="mylist">\n<li class="myli">A</li>\n<li class="myli">B</li>\n</ul>')
    >>> ul([])
    literal(u'<ul></ul>')
    >>> ul([], default="")
    ''
    >>> ul([], default=literal('<span class="no-data">No data</span>'))
    literal(u'<span class="no-data">No data</span>')
    >>> ul(["A"], default="NOTHING")
    literal(u'<ul>\n<li>A</li>\n</ul>')
    """
    li_attrs = li_attrs or {}
    return _list("ul", items, default, attrs, li_attrs)

def ol(items, default=literal(""), li_attrs=None, **attrs):
    R"""Return an ordered list with each item wrapped in <li>.

    ``items``
        list of strings.

    ``default``
        value returned _instead of the <ol>_ if there are no items in the list.
        If ``None``, return an empty <ol>.

    ``li_attrs``
        dict of attributes for the <li> tags.

    Examples:

    >>> ol(["foo", "bar"])
    literal(u'<ol>\n<li>foo</li>\n<li>bar</li>\n</ol>')
    >>> ol(["A", "B"], li_attrs={"class_": "myli"}, class_="mylist") 
    literal(u'<ol class="mylist">\n<li class="myli">A</li>\n<li class="myli">B</li>\n</ol>')
    >>> ol([])
    literal(u'')
    """
    li_attrs = li_attrs or {}
    return _list("ol", items, default, attrs, li_attrs)

def _list(tag, items, default, attrs, li_attrs):
    content = [HTML.li(x, **li_attrs) for x in items]
    if content:
        content = [""] + content + [""]
    elif default is not None:
        return default
    content = literal("\n").join(content)
    return getattr(HTML, tag)(content, **attrs)
    

def image(url, alt, width=None, height=None, path=None, use_pil=False, 
    **attrs):
    """Return an image tag for the specified ``source``.

    ``url``
        The URL of the image.  (This must be the exact URL desired.  A
        previous version of this helper added magic prefixes; this is
        no longer the case.)
    
    ``alt``
        The img's alt tag. Non-graphical browsers and screen readers will
        output this instead of the image.  If the image is pure decoration
        and uninteresting to non-graphical users, pass "".  To omit the
        alt tag completely, pass None.

    ``width``
        The width of the image, default is not included.

    ``height``
        The height of the image, default is not included.

    ``path``
        Calculate the width and height based on the image file at ``path`` if
        possible. May not be specified if ``width`` or ``height`` is 
        specified. The results are also written to the debug log for
        troubleshooting.

    ``use_pil``
        If true, calcuate the image dimensions using the Python Imaging 
        Library, which must be installed. Otherwise use a pure Python
        algorithm which understands fewer image formats and may be less
        accurate. This flag controls whether
        ``webhelpers.media.get_dimensions_pil`` or
        ``webhelpers.media.get_dimensions`` is called. It has no effect if
        ``path`` is not specified.
        
    Examples::

        >>> image('/images/rss.png', 'rss syndication')
        literal(u'<img alt="rss syndication" src="/images/rss.png" />')

        >>> image('/images/xml.png', "")
        literal(u'<img alt="" src="/images/xml.png" />')

        >>> image("/images/icon.png", height=16, width=10, alt="Edit Entry")
        literal(u'<img alt="Edit Entry" height="16" src="/images/icon.png" width="10" />')

        >>> image("/icons/icon.gif", alt="Icon", width=16, height=16)
        literal(u'<img alt="Icon" height="16" src="/icons/icon.gif" width="16" />')

        >>> image("/icons/icon.gif", None, width=16)
        literal(u'<img alt="" src="/icons/icon.gif" width="16" />')
    """
    if not alt:
        alt = ""
    if width is not None or height is not None:
        attrs['width'] = width
        attrs['height'] = height
        if path:
            raise TypeError(
                "can't specify path if width and height are specified")
    elif path:
        if use_pil:
            result = media.get_dimensions_pil(path)
            msg = "using PIL"
        else:
            result = media.get_dimensions(path)
            msg = "not using PIL"
        abspath = os.path.abspath(path)
        log.debug("image size is %s for '%s' (%s)", result, abspath, msg)
        attrs['width'] = result[0]
        attrs['height'] = result[1]
    return HTML.img(src=url, alt=alt, **attrs)

#### Non-tag utilities

def css_classes(value_condition_pairs):
    """Add CSS classes to a tag programmatically.

    This helper is meant to be used as the ``class_`` argument to a tag helper.

    The argument is an iterable of ``(class, condition)`` pairs, where each
    ``class`` is a string and ``condition`` is a boolean.  The function 
    returns a space-separated list of classes whose conditions were true.

    If all conditions are false, return ``None``.  This tells the caller to
    suppress the "class" attribute entirely.

    Examples::

        >>> arg = [("first", False), ("even", True)]
        >>> HTML.td("My content.", class_=css_classes(arg))
        literal(u'<td class="even">My content.</td>')
        >>> arg = [("first", True), ("even", True)]
        >>> HTML.td("My content.", class_=css_classes(arg))
        literal(u'<td class="first even">My content.</td>')
        >>> arg = [("first", False), ("even", False)]
        >>> HTML.td("My content.", class_=css_classes(arg))
        literal(u'<td>My content.</td>')

 
    """
    classes = []
    for value, condition in value_condition_pairs:
        if condition:    # Or ``if condition and value``
            classes.append(value)
    if not classes:
        return None
    return " ".join(classes)


#### Tags for the HTML head

    #### Private methods
    def _make_doctype(self, type, uri, dtd):
        return literal('<!DOCTYPE %s PUBLIC "%s" "%s">') % (type, uri, dtd)

def javascript_link(*urls, **attrs):
    """Return script include tags for the specified javascript URLs.
    
    ``urls`` should be the exact URLs desired.  A previous version of this
    helper added magic prefixes; this is no longer the case.

    Specify the keyword argument ``defer=True`` to enable the script 
    defer attribute.

    Examples::
    
        >>> print javascript_link('/javascripts/prototype.js', '/other-javascripts/util.js')
        <script src="/javascripts/prototype.js" type="text/javascript"></script>
        <script src="/other-javascripts/util.js" type="text/javascript"></script>

        >>> print javascript_link('/app.js', '/test/test.1.js')
        <script src="/app.js" type="text/javascript"></script>
        <script src="/test/test.1.js" type="text/javascript"></script>
        
    """
    convert_boolean_attrs(attrs, ["defer"])
    tags = []
    for url in urls:
        tag = HTML.script("", type="text/javascript", src=url, **attrs)
        tags.append(tag)
    return literal("\n").join(tags)


def stylesheet_link(*urls, **attrs):
    """Return CSS link tags for the specified stylesheet URLs.

    ``urls`` should be the exact URLs desired.  A previous version of this
    helper added magic prefixes; this is no longer the case.

    Examples::

        >>> stylesheet_link('/stylesheets/style.css')
        literal(u'<link href="/stylesheets/style.css" media="screen" rel="stylesheet" type="text/css" />')

        >>> stylesheet_link('/stylesheets/dir/file.css', media='all')
        literal(u'<link href="/stylesheets/dir/file.css" media="all" rel="stylesheet" type="text/css" />')

    """
    if "href" in attrs:
        raise TypeError("keyword arg 'href' not allowed")
    attrs.setdefault("rel", "stylesheet")
    attrs.setdefault("type", "text/css")
    attrs.setdefault("media", "screen")
    tags = []
    for url in urls:
        tag = HTML.link(href=url, **attrs)
        tags.append(tag)
    return literal('\n').join(tags)


def auto_discovery_link(url, feed_type="rss", **attrs):
    """Return a link tag allowing auto-detecting of RSS or ATOM feed.
    
    The auto-detection of feed for the current page is only for
    browsers and news readers that support it.

    ``url``
        The URL of the feed.  (This should be the exact URLs desired.  A
        previous version of this helper added magic prefixes; this is no longer
        the case.)

    ``feed_type``
        The type of feed. Specifying 'rss' or 'atom' automatically 
        translates to a type of 'application/rss+xml' or 
        'application/atom+xml', respectively. Otherwise the type is
        used as specified. Defaults to 'rss'.
        
    Examples::

        >>> auto_discovery_link('http://feed.com/feed.xml')
        literal(u'<link href="http://feed.com/feed.xml" rel="alternate" title="RSS" type="application/rss+xml" />')

        >>> auto_discovery_link('http://feed.com/feed.xml', feed_type='atom')
        literal(u'<link href="http://feed.com/feed.xml" rel="alternate" title="ATOM" type="application/atom+xml" />')

        >>> auto_discovery_link('app.rss', feed_type='atom', title='atom feed')
        literal(u'<link href="app.rss" rel="alternate" title="atom feed" type="application/atom+xml" />')

        >>> auto_discovery_link('/app.html', feed_type='text/html')
        literal(u'<link href="/app.html" rel="alternate" title="" type="text/html" />')
        
    """
    if "href" in attrs:
        raise TypeError("keyword arg 'href' is not allowed")
    if "type" in attrs:
        raise TypeError("keyword arg 'type' is not allowed")
    title = ""
    if feed_type.lower() in ('rss', 'atom'):
        title = feed_type.upper()
        feed_type = 'application/%s+xml' % feed_type.lower()
    attrs.setdefault("title", title)
    return HTML.link(rel="alternate", type=feed_type, href=url, **attrs)

#### Document type and XML declaration

class Doctype(object):
    """Document type declarations for HTML and XHTML."""

    def html5(self):
        """Create a <!DOCTYPE> for HTML 5.
        
        Usage::

            >>> Doctype().html5()
            literal(u'<!doctype html>')
        """
        return literal("<!doctype html>")

    def xhtml1(self, subtype="transitional", version="1.0"):
        """Create a <!DOCTYPE> for XHTML 1.

           Usage::

               >>> Doctype().xhtml1()
               literal(u'<!DOCTYPE html PUBLIC "-//W3C//DTD XHTML 1.0 Transitional//EN" "http://www.w3.org/TR/xhtml1/DTD/xhtml1-transitional.dtd">')
               >>> Doctype().xhtml1("strict")
               literal(u'<!DOCTYPE html PUBLIC "-//W3C//DTD XHTML 1.0 Strict//EN" "http://www.w3.org/TR/xhtml1/DTD/xhtml1-strict.dtd">')
               >>> Doctype().xhtml1("frameset")
               literal(u'<!DOCTYPE html PUBLIC "-//W3C//DTD XHTML 1.0 Frameset//EN" "http://www.w3.org/TR/xhtml1/DTD/xhtml1-frameset.dtd">')
        """
        if subtype in ["transitional", "loose"]:
            name = "Transitional"
            dtd = "transitional"
        else:
            name = subtype.capitalize()
            dtd = subtype.lower()
        uri = "-//W3C//DTD XHTML %s %s//EN" % (version, name)
        dtd = "http://www.w3.org/TR/xhtml1/DTD/xhtml1-%s.dtd" % dtd
        return self._make_doctype("html", uri, dtd)

    def html4(self, subtype="transitional", version="4.01"):
        """Create a <!DOCTYPE> for HTML 4.

           Usage:

               >>> Doctype().html4()
               literal(u'<!DOCTYPE HTML PUBLIC "-//W3C//DTD HTML 4.01 Transitional//EN" "http://www.w3.org/TR/html4/loose.dtd">')
               >>> Doctype().html4("strict")
               literal(u'<!DOCTYPE HTML PUBLIC "-//W3C//DTD HTML 4.01 Strict//EN" "http://www.w3.org/TR/html4/strict.dtd">')
               >>> Doctype().html4("frameset")
               literal(u'<!DOCTYPE HTML PUBLIC "-//W3C//DTD HTML 4.01 Frameset//EN" "http://www.w3.org/TR/html4/frameset.dtd">')
        """
        if subtype in ["transitional", "loose"]:
            name = "Transitional"
            dtd = "loose"
        else:
            name = subtype.capitalize()
            dtd = subtype.lower()
        uri = "-//W3C//DTD HTML %s %s//EN" % (version, name)
        dtd = "http://www.w3.org/TR/html4/%s.dtd" % dtd
        return self._make_doctype("HTML", uri, dtd)

    #### Private methods
    def _make_doctype(self, type, uri, dtd):
        return literal('<!DOCTYPE %s PUBLIC "%s" "%s">') % (type, uri, dtd)


def xml_declaration(version="1.0", encoding="utf-8"):
    """Create an XML declaration.

       Usage::

           >>> xml_declaration()
           literal(u'<?xml version="1.0" encoding="utf-8" ?>')
    """
    return literal('<?xml version="%s" encoding="%s" ?>') % (version, encoding)



########## INTERNAL FUNCTIONS ##########

def convert_boolean_attrs(attrs, bool_attrs):
    """Convert boolean values into proper HTML attributes.

    ``attrs`` is a dict of HTML attributes, which will be modified in
    place.

    ``bool_attrs`` is a list of attribute names.

    For every element in ``bool_attrs``, I look for a corresponding key in
    ``attrs``.  If its value is true, I change the value to match the key.
    For example, I convert ``selected=True`` into ``selected="selected"``.  If
    the value is false, I delete the key.
    
    """
    for a in bool_attrs:
        if attrs.has_key(a) and attrs[a]:
            attrs[a] = a
        elif attrs.has_key(a):
            del attrs[a]

def _set_input_attrs(attrs, type, name, value):
    attrs["type"] = type
    attrs["name"] = name
    attrs["value"] = value

def _set_id_attr(attrs, id_arg, name):
    if "id_" in attrs:
        if id_arg is not NotGiven:
            raise TypeError("can't pass both 'id' and 'id_' args to helper")
        attrs["id"] = attrs.pop("id_")
    elif id_arg is NotGiven:
        attrs["id"] = _make_safe_id_component(name)
    elif id_arg is not None and id_arg != "":
        attrs["id"] = id_arg
    # Else id_arg is None or "", so do nothing.

def _make_safe_id_component(idstring):
    """Make a string safe for including in an id attribute.
    
    The HTML spec says that id attributes 'must begin with 
    a letter ([A-Za-z]) and may be followed by any number 
    of letters, digits ([0-9]), hyphens ("-"), underscores 
    ("_"), colons (":"), and periods (".")'. These regexps
    are slightly over-zealous, in that they remove colons
    and periods unnecessarily.
    
    Whitespace is transformed into underscores, and then
    anything which is not a hyphen or a character that 
    matches \w (alphanumerics and underscore) is removed.
    
    """
    # Transform all whitespace to underscore
    idstring = re.sub(r'\s', "_", '%s' % idstring)
    # Remove everything that is not a hyphen or a member of \w
    idstring = re.sub(r'(?!-)\W', "", idstring).lower()
    return idstring


if __name__ == "__main__":
    import doctest
    doctest.testmod()
