// Adds colors from the specified property of the element to the given list
function webdeveloper_addColor(element, property, colorList)
{
    // If the element, property and color list are set
    if(element && property && colorList)
    {
        var computedStyle = element.ownerDocument.defaultView.getComputedStyle(element, null);
        var color         = computedStyle.getPropertyCSSValue(property);

        // If the color is set and it is a color
        if(color && color.primitiveType == CSSPrimitiveValue.CSS_RGBCOLOR)
        {
            color = color.getRGBColorValue();

            var blue  = color.blue.getFloatValue(CSSPrimitiveValue.CSS_NUMBER).toString(16);
            var green = color.green.getFloatValue(CSSPrimitiveValue.CSS_NUMBER).toString(16);
            var red   = color.red.getFloatValue(CSSPrimitiveValue.CSS_NUMBER).toString(16);

            // If blue is set to 0
            if(blue == "0")
            {
                blue = "00";
            }

            // If green is set to 0
            if(green == "0")
            {
                green = "00";
            }

            // If red is set to 0
            if(red == "0")
            {
                red = "00";
            }

            colorList.push("#" + red + green + blue);
        }
    }
}

// Filter for display access key tree walker
function webdeveloper_accessKeyFilter(node)
{
    // If the node has an accesskey attribute
    if(node.hasAttribute("accesskey"))
    {
        return NodeFilter.FILTER_ACCEPT;
    }

    return NodeFilter.FILTER_SKIP;
}

// Filter for display anchors tree walker
function webdeveloper_anchorFilter(node)
{
    // If the node has an id attribute or is an a tag with a name attribute
    if(node.hasAttribute("id") || (node.nodeName && node.nodeName.toLowerCase() == "a" && node.hasAttribute("name")))
    {
        return NodeFilter.FILTER_ACCEPT;
    }

    return NodeFilter.FILTER_SKIP;
}

// Filter for class or id attributes tree walker
function webdeveloper_classIdFilter(node)
{
    // If the node has a class or id attribute
    if(node.hasAttribute("class") || node.hasAttribute("id"))
    {
        return NodeFilter.FILTER_ACCEPT;
    }

    return NodeFilter.FILTER_SKIP;
}

// Displays all access keys
function webdeveloper_displayAccessKeys(element, applyStyle)
{
    var accessKeyElement        = null;
    var accessKeyElementList    = null;
    var accessKeyElementsLength = null;
    var checked                 = element.getAttribute("checked");
    var documentList            = webdeveloper_getDocuments(webdeveloper_getContentWindow());
    var documentLength          = documentList.length;
    var pageDocument            = null;
    var spanElement             = null;
    var spanElementList         = null;
    var spanElementsLength      = null;
    var treeWalker              = null;

    // Loop through the documents
    for(var i = 0; i < documentLength; i++)
    {
        pageDocument = documentList[i];

        // If the element is checked
        if(checked)
        {
            accessKeyElementList = new Array();
            treeWalker           = pageDocument.createTreeWalker(pageDocument.body, NodeFilter.SHOW_ELEMENT, webdeveloper_accessKeyFilter, false);

            // Loop through the access key elements
            while((accessKeyElement = treeWalker.nextNode()) != null)
            {
                accessKeyElementList.push(accessKeyElement);
            }

            accessKeyElementsLength = accessKeyElementList.length;

            // Loop through all the access key elements
            for(var j = 0; j < accessKeyElementsLength; j++)
            {
                accessKeyElement = accessKeyElementList[j];
                spanElement      = pageDocument.createElement("span");

                spanElement.setAttribute("class", "webdeveloper-access-key");
                spanElement.appendChild(pageDocument.createTextNode("Accesskey=" + accessKeyElement.getAttribute("accesskey")));
                accessKeyElement.parentNode.insertBefore(spanElement, accessKeyElement);
            }
        }
        else
        {
            spanElementList    = pageDocument.getElementsByTagName("span");
            spanElementsLength = spanElementList.length;

            // Loop through all the span tags
            for(j = 0; j < spanElementsLength; j++)
            {
                spanElement = spanElementList[j];

                // If the span element is set and the class exists and is set to webdeveloper-access-key
                if(spanElement && spanElement.hasAttribute("class") && spanElement.getAttribute("class") == "webdeveloper-access-key")
                {
                    spanElement.parentNode.removeChild(spanElement);
                    j--;
                }
            }
        }
    }

    webdeveloper_toggleStyleSheet(element, "chrome://webdeveloper/content/stylesheets/imports/tooltips.css", "webdeveloper-display-access-keys", applyStyle);
    webdeveloper_toggleFeatureTooltipStyles(element, "webdeveloper-display-access-keys-tooltips", "span.webdeveloper-access-key");
}

// Displays all anchors
function webdeveloper_displayAnchors(element, applyStyle)
{
    var anchorElement        = null;
    var anchorElementList    = null;
    var anchorElementsLength = null;
    var anchorText           = null;
    var checked              = element.getAttribute("checked");
    var documentList         = webdeveloper_getDocuments(webdeveloper_getContentWindow());
    var documentLength       = documentList.length;
    var pageDocument         = null;
    var spanElement          = null;
    var spanElementList      = null;
    var spanElementsLength   = null;
    var treeWalker           = null;

    // Loop through the documents
    for(var i = 0; i < documentLength; i++)
    {
        pageDocument = documentList[i];

        // If the element is checked
        if(checked)
        {
            anchorElementList = new Array();
            treeWalker        = pageDocument.createTreeWalker(pageDocument.body, NodeFilter.SHOW_ELEMENT, webdeveloper_anchorFilter, false);

            // Loop through the anchor elements
            while((anchorElement = treeWalker.nextNode()) != null)
            {
                anchorElementList.push(anchorElement);
            }

            anchorElementsLength = anchorElementList.length;

            // Loop through all the anchor elements
            for(var j = 0; j < anchorElementsLength; j++)
            {
                anchorElement = anchorElementList[j];
                spanElement   = pageDocument.createElement("span");

                // If the anchor element has an id attribute
                if(anchorElement.hasAttribute("id"))
                {
                    anchorText = "#" + anchorElement.getAttribute("id");
                }
                else if(anchorElement.hasAttribute("name"))
                {
                    anchorText = "#" + anchorElement.getAttribute("name");
                }

                spanElement.setAttribute("class", "webdeveloper-anchor");
                spanElement.appendChild(pageDocument.createTextNode(anchorText));
                anchorElement.parentNode.insertBefore(spanElement, anchorElement);
            }
        }
        else
        {
            spanElementList    = pageDocument.getElementsByTagName("span");
            spanElementsLength = spanElementList.length;

            // Loop through all the span tags
            for(j = 0; j < spanElementsLength; j++)
            {
                spanElement = spanElementList[j];

                // If the span element is set and the class exists and is set to webdeveloper-anchor
                if(spanElement && spanElement.hasAttribute("class") && spanElement.getAttribute("class") == "webdeveloper-anchor")
                {
                    spanElement.parentNode.removeChild(spanElement);
                    j--;
                }
            }
        }
    }

    webdeveloper_toggleStyleSheet(element, "chrome://webdeveloper/content/stylesheets/imports/tooltips.css", "webdeveloper-display-anchors", applyStyle);
    webdeveloper_toggleFeatureTooltipStyles(element, "webdeveloper-display-anchors-tooltips", "span.webdeveloper-anchor");
}

// Displays all block sizes
function webdeveloper_displayBlockSize(element, applyStyle)
{
    var blockSize                   = null;
    var blockSizeElement            = null;
    var blockSizeElementList        = null;
    var blockSizeElementsLength     = null;
    var blockSizeElementTypes       = new Array("div", "form", "table");
    var blockSizeElementTypesLength = blockSizeElementTypes.length;
    var display                     = element.getAttribute("checked");
    var documentList                = webdeveloper_getDocuments(webdeveloper_getContentWindow());
    var documentLength              = documentList.length;
    var height                      = null;
    var left                        = null;
    var pageDocument                = null;
    var top                         = null;
    var treeWalker                  = null;
    var width                       = null;

    // Loop through the documents
    for(var i = 0; i < documentLength; i++)
    {
        pageDocument = documentList[i];

        // If displaying
        if(display)
        {
            blockSizeElementTypes = new Array("div", "form", "table");

            // Loop through the block size elements
            for(var j = 0; j < blockSizeElementTypesLength; j++)
            {
                blockSizeElementList    = pageDocument.getElementsByTagName(blockSizeElementTypes[j]);
                blockSizeElementsLength = blockSizeElementList.length;

                // Loop through the block size elements
                for(var k = 0; k < blockSizeElementsLength; k++)
                {
                    blockSize        = pageDocument.createElement("span");
                    blockSizeElement = blockSizeElementList[k];
                    height           = blockSizeElement.offsetHeight;
                    left             = blockSizeElement.offsetLeft;
                    top              = blockSizeElement.offsetTop;
                    width            = blockSizeElement.offsetWidth;

                    blockSize.style.left     = left + "px";
                    blockSize.style.position = "absolute";
                    blockSize.style.top      = top + "px";

                    blockSize.appendChild(pageDocument.createTextNode(webdeveloper_getElementDescription(blockSizeElement) + " " + width + "x" + height));
                    blockSize.setAttribute("class", "webdeveloper-block-size");

                    webdeveloper_insertAsFirstChild(blockSizeElement, blockSize);
                }
            }
        }
        else
        {
            blockSizeElementList    = pageDocument.getElementsByTagName("span");
            blockSizeElementsLength = blockSizeElementList.length;

            // Loop through the block size elements
            for(j = 0; j < blockSizeElementsLength; j++)
            {
                blockSizeElement = blockSizeElementList[j];

                // If the block size element is set and the class exists and is set to webdeveloper-block-size
                if(blockSizeElement && blockSizeElement.hasAttribute("class") && blockSizeElement.getAttribute("class") == "webdeveloper-block-size")
                {
                    blockSizeElement.parentNode.removeChild(blockSizeElement);
                    j--;
                }
            }
        }
    }

    webdeveloper_toggleStyleSheet(element, "chrome://webdeveloper/content/stylesheets/display_block_size.css", "webdeveloper-display-block-size", applyStyle);
    webdeveloper_toggleFeatureTooltipStyles(element, "webdeveloper-display-block-size-tooltips", "span.webdeveloper-block-size");
}

// Filter for display block size tree walker
function webdeveloper_displayBlockSizeFilter(node)
{
    // If the class of the node is webdeveloper block size
    if(node.hasAttribute("class") && node.getAttribute("class") == "webdeveloper-block-size")
    {
        return NodeFilter.FILTER_ACCEPT;
    }

    return NodeFilter.FILTER_SKIP;
}

// Displays the order of the divs on the page
function webdeveloper_displayDivOrder(element, applyStyle)
{
    var display            = element.getAttribute("checked");
    var divElement         = null;
    var divElementList     = null;
    var divElementsLength  = null;
    var documentList       = webdeveloper_getDocuments(webdeveloper_getContentWindow());
    var documentLength     = documentList.length;
    var pageDocument       = null;
    var spanElement        = null;
    var spanElementList    = null;
    var spanElementsLength = null;

    // Loop through the documents
    for(var i = 0; i < documentLength; i++)
    {
        pageDocument = documentList[i];

        // If displaying
        if(display)
        {
            divElementList    = pageDocument.getElementsByTagName("div");
            divElementsLength = divElementList.length;

            // Loop through the div elements
            for(var j = 0; j < divElementsLength; j++)
            {
                divElement  = divElementList[j];
                spanElement = pageDocument.createElement("span");

                spanElement.appendChild(pageDocument.createTextNode(webdeveloper_getElementDescription(divElement) + " " + (j + 1)));
                spanElement.setAttribute("class", "webdeveloper-div-order");

                webdeveloper_insertAsFirstChild(divElement, spanElement);
            }
        }
        else
        {
            spanElementList    = pageDocument.getElementsByTagName("span");
            spanElementsLength = spanElementList.length;

            // Loop through the span elements
            for(j = 0; j < spanElementsLength; j++)
            {
                spanElement = spanElementList[j];

                // If the span element is set and the class exists and is set to webdeveloper-div-order
                if(spanElement && spanElement.hasAttribute("class") && spanElement.getAttribute("class") == "webdeveloper-div-order")
                {
                    spanElement.parentNode.removeChild(spanElement);
                    j--;
                }
            }
        }
    }

    webdeveloper_toggleStyleSheet(element, "chrome://webdeveloper/content/stylesheets/display_div_order.css", "webdeveloper-display-div-order", applyStyle);
    webdeveloper_toggleFeatureTooltipStyles(element, "webdeveloper-display-div-order-tooltips", "span.webdeveloper-div-order");
}

// Displays id and class details for the page
function webdeveloper_displayIdClassDetails(element, applyStyle)
{
    var documentList          = webdeveloper_getDocuments(webdeveloper_getContentWindow());
    var documentLength        = documentList.length;
    var idClassElement        = null;
    var idClassElementList    = null;
    var idClassElementsLength = null;
    var idClassElementText    = null;
    var pageDocument          = null;
    var spanElement           = null;
    var spanElementList       = null;
    var spanElementsLength    = null;
    var treeWalker            = null;

    // Loop through the documents
    for(var i = 0; i < documentLength; i++)
    {
        pageDocument = documentList[i];

        // If the menu is checked
        if(element.getAttribute("checked"))
        {
            idClassElementList = new Array();
            treeWalker         = pageDocument.createTreeWalker(pageDocument.body, NodeFilter.SHOW_ELEMENT, webdeveloper_classIdFilter, false);

            // Loop through the id class elements
            while((idClassElement = treeWalker.nextNode()) != null)
            {
                idClassElementList.push(idClassElement);
            }

            idClassElementsLength = idClassElementList.length;

            // Loop through all the id class elements
            for(var j = 0; j < idClassElementsLength; j++)
            {
                idClassElement     = idClassElementList[j];
                idClassElementText = "";
                spanElement        = pageDocument.createElement("span");

                // If the id class element has an id attribute
                if(idClassElement.hasAttribute("id"))
                {
                    idClassElementText += " #" + idClassElement.getAttribute("id");
                }

                // If the id class element has a class attribute
                if(idClassElement.hasAttribute("class"))
                {
                    idClassElementText += " ." + idClassElement.getAttribute("class");
                }

                spanElement.setAttribute("class", "webdeveloper-id-class-details");
                spanElement.appendChild(pageDocument.createTextNode(idClassElementText));
                idClassElement.parentNode.insertBefore(spanElement, idClassElement);
            }
        }
        else
        {
            spanElementList    = pageDocument.getElementsByTagName("span");
            spanElementsLength = spanElementList.length;

            // Loop through all the span tags
            for(j = 0; j < spanElementsLength; j++)
            {
                spanElement = spanElementList[j];

                // If the span element is set and the class exists and is set to webdeveloper-form
                if(spanElement && spanElement.hasAttribute("class") && spanElement.getAttribute("class") == "webdeveloper-id-class-details")
                {
                    spanElement.parentNode.removeChild(spanElement);
                    j--;
                }
            }
        }
    }

    webdeveloper_toggleStyleSheet(element, "chrome://webdeveloper/content/stylesheets/imports/tooltips.css", "webdeveloper-display-id-class-details", applyStyle);
    webdeveloper_toggleFeatureTooltipStyles(element, "webdeveloper-display-id-class-details-tooltips", "span.webdeveloper-id-class-details");
}

// Displays all link details
function webdeveloper_displayLinkDetails(element, applyStyle)
{
    webdeveloper_toggleStyleSheet(element, "chrome://webdeveloper/content/stylesheets/display_link_details.css", "webdeveloper-display-link-details", applyStyle);
    webdeveloper_toggleFeatureTooltipStyles(element, "webdeveloper-display-link-details-tooltips", "*:before");
}

// Displays object information
function webdeveloper_displayObjectInformation(element, applyStyle)
{
    var checked              = element.getAttribute("checked");
    var divElement           = null;
    var divElementList       = null;
    var divElementsLength    = null;
    var documentList         = webdeveloper_getDocuments(webdeveloper_getContentWindow());
    var documentLength       = documentList.length;
    var objectAttributes     = null;
    var objectElement        = null;
    var objectElementList    = null;
    var objectElementsLength = null;
    var pageDocument         = null;
    var paramAttributes      = null;
    var paramElement         = null;
    var paramElementList     = null;
    var paramElementsLength  = null;
    var pElement             = null;
    var treeWalker           = null;

    // Loop through the documents
    for(var i = 0; i < documentLength; i++)
    {
        pageDocument = documentList[i];

        // If the element is checked
        if(checked)
        {
            objectElementList    = pageDocument.getElementsByTagName("object");
            objectElementsLength = objectElementList.length;

            // Loop through all the object elements
            for(var j = 0; j < objectElementsLength; j++)
            {
                divElement          = pageDocument.createElement("div");
                objectAttributes    = "";
                objectElement       = objectElementList[j];
                paramElementList    = objectElement.getElementsByTagName("param");
                paramElementsLength = paramElementList.length;
                pElement            = pageDocument.createElement("p");

                // If the object has an width attribute
                if(objectElement.hasAttribute("width"))
                {
                    objectAttributes += ' width="' + objectElement.getAttribute("width") + '"';
                }

                // If the object has an height attribute
                if(objectElement.hasAttribute("height"))
                {
                    objectAttributes += ' height="' + objectElement.getAttribute("height") + '"';
                }

                // If the object has an archive attribute
                if(objectElement.hasAttribute("archive"))
                {
                    objectAttributes += ' archive="' + objectElement.getAttribute("archive") + '"';
                }

                // If the object has an classid attribute
                if(objectElement.hasAttribute("classid"))
                {
                    objectAttributes += ' classid="' + objectElement.getAttribute("classid") + '"';
                }

                // If the object has an codebase attribute
                if(objectElement.hasAttribute("codebase"))
                {
                    objectAttributes += ' codebase="' + objectElement.getAttribute("codebase") + '"';
                }

                // If the object has an codetype attribute
                if(objectElement.hasAttribute("codetype"))
                {
                    objectAttributes += ' codetype="' + objectElement.getAttribute("codetype") + '"';
                }

                // If the object has an data attribute
                if(objectElement.hasAttribute("data"))
                {
                    objectAttributes += ' data="' + objectElement.getAttribute("data") + '"';
                }

                // If the object has an standby attribute
                if(objectElement.hasAttribute("standby"))
                {
                    objectAttributes += ' standby="' + objectElement.getAttribute("standby") + '"';
                }

                // If the object has an type attribute
                if(objectElement.hasAttribute("type"))
                {
                    objectAttributes += ' type="' + objectElement.getAttribute("type") + '"';
                }

                pElement.appendChild(document.createTextNode("<object" + objectAttributes + ">"));
                divElement.appendChild(pElement);

                // Loop through all the param elements
                for(var k = 0; k < paramElementsLength; k++)
                {
                    paramAttributes = "";
                    paramElement    = paramElementList[k];
                    pElement        = pageDocument.createElement("p");

                    // If the param has a name attribute
                    if(paramElement.hasAttribute("name"))
                    {
                        paramAttributes += ' name="' + paramElement.getAttribute("name") + '"';
                    }

                    // If the param has a value attribute
                    if(paramElement.hasAttribute("value"))
                    {
                        paramAttributes += ' value="' + paramElement.getAttribute("value") + '"';
                    }

                    pElement.appendChild(document.createTextNode("<param" + paramAttributes + " />"));
                    pElement.setAttribute("class", "param");
                    divElement.appendChild(pElement);
                }

                pElement = pageDocument.createElement("p");
                pElement.appendChild(document.createTextNode("</object>"));
                divElement.appendChild(pElement);

                divElement.setAttribute("class", "webdeveloper-object-information");
                objectElement.parentNode.insertBefore(divElement, objectElement);
            }
        }
        else
        {
            divElementList    = pageDocument.getElementsByTagName("div");
            divElementsLength = divElementList.length;

            // Loop through all the div tags
            for(j = 0; j < divElementsLength; j++)
            {
                divElement = divElementList[j];

                // If the div element is set and the class exists and is set to webdeveloper-object-information
                if(divElement && divElement.hasAttribute("class") && divElement.getAttribute("class") == "webdeveloper-object-information")
                {
                    divElement.parentNode.removeChild(divElement);
                    j--;
                }
            }
        }
    }

    webdeveloper_toggleStyleSheet(element, "chrome://webdeveloper/content/stylesheets/display_object_information.css", "webdeveloper-display-object-information", applyStyle);
    webdeveloper_toggleFeatureTooltipStyles(element, "webdeveloper-display-object-information-tooltips", "div.webdeveloper-object-information");
}

// Displays the stack level of all elements on the page
function webdeveloper_displayStackLevels(element, applyStyle)
{
    var display            = element.getAttribute("checked");
    var documentList       = webdeveloper_getDocuments(webdeveloper_getContentWindow());
    var documentLength     = documentList.length;
    var pageDocument       = null;
    var pageElement        = null;
    var spanElement        = null;
    var spanElementList    = null;
    var spanElementsLength = null;
    var treeWalker         = null;

    // Loop through the documents
    for(var i = 0; i < documentLength; i++)
    {
        pageDocument = documentList[i];

        // If displaying
        if(display)
        {
            treeWalker = pageDocument.createTreeWalker(pageDocument, NodeFilter.SHOW_ELEMENT, webdeveloper_stackLevelFilter, false);

            // While the tree walker has more nodes
            while((pageElement = treeWalker.nextNode()) != null)
            {
                spanElement = pageDocument.createElement("span");

                spanElement.appendChild(pageDocument.createTextNode(webdeveloper_getElementDescription(pageElement) + " Z-index=" + pageElement.ownerDocument.defaultView.getComputedStyle(pageElement, null).getPropertyCSSValue("z-index").cssText));
                spanElement.setAttribute("class", "webdeveloper-stack-level");

                webdeveloper_insertAsFirstChild(pageElement, spanElement);
            }
        }
        else
        {
            spanElementList    = pageDocument.getElementsByTagName("span");
            spanElementsLength = spanElementList.length;

            // Loop through the span elements
            for(var j = 0; j < spanElementsLength; j++)
            {
                spanElement = spanElementList[j];

                // If the span element is set and the class exists and is set to webdeveloper-stack-level
                if(spanElement && spanElement.hasAttribute("class") && spanElement.getAttribute("class") == "webdeveloper-stack-level")
                {
                    spanElement.parentNode.removeChild(spanElement);
                    j--;
                }
            }
        }
    }

    webdeveloper_toggleStyleSheet(element, "chrome://webdeveloper/content/stylesheets/imports/tooltips.css", "webdeveloper-display-stack-levels", applyStyle);
    webdeveloper_toggleFeatureTooltipStyles(element, "webdeveloper-display-stack-levels-tooltips", "span.webdeveloper-stack-level");
}

// Displays all tab indices
function webdeveloper_displayTabIndex(element, applyStyle)
{
    var checked                = element.getAttribute("checked");
    var documentList           = webdeveloper_getDocuments(webdeveloper_getContentWindow());
    var documentLength         = documentList.length;
    var pageDocument           = null;
    var spanElement            = null;
    var spanElementList        = null;
    var spanElementsLength     = null;
    var tabIndexElement        = null;
    var tabIndexElementList    = null;
    var tabIndexElementsLength = null;
    var treeWalker             = null;

    // Loop through the documents
    for(var i = 0; i < documentLength; i++)
    {
        pageDocument = documentList[i];

        // If the element is checked
        if(checked)
        {
            tabIndexElementList = new Array();
            treeWalker          = pageDocument.createTreeWalker(pageDocument.body, NodeFilter.SHOW_ELEMENT, webdeveloper_tabIndexFilter, false);

            // Loop through the tab index elements
            while((tabIndexElement = treeWalker.nextNode()) != null)
            {
                tabIndexElementList.push(tabIndexElement);
            }

            tabIndexElementsLength = tabIndexElementList.length;

            // Loop through all the tab index elements
            for(var j = 0; j < tabIndexElementsLength; j++)
            {
                spanElement     = pageDocument.createElement("span");
                tabIndexElement = tabIndexElementList[j];

                spanElement.setAttribute("class", "webdeveloper-tab-index");
                spanElement.appendChild(document.createTextNode("Tabindex=" + tabIndexElement.getAttribute("tabindex")));
                tabIndexElement.parentNode.insertBefore(spanElement, tabIndexElement);
            }
        }
        else
        {
            spanElementList    = pageDocument.getElementsByTagName("span");
            spanElementsLength = spanElementList.length;

            // Loop through all the span tags
            for(j = 0; j < spanElementsLength; j++)
            {
                spanElement = spanElementList[j];

                // If the span element is set and the class exists and is set to webdeveloper-tab-index
                if(spanElement && spanElement.hasAttribute("class") && spanElement.getAttribute("class") == "webdeveloper-tab-index")
                {
                    spanElement.parentNode.removeChild(spanElement);
                    j--;
                }
            }
        }
    }

    webdeveloper_toggleStyleSheet(element, "chrome://webdeveloper/content/stylesheets/imports/tooltips.css", "webdeveloper-display-tab-index", applyStyle);
    webdeveloper_toggleFeatureTooltipStyles(element, "webdeveloper-display-tab-index-tooltips", "span.webdeveloper-tab-index");
}

// Displays the depth of all tables
function webdeveloper_displayTableDepth(element, applyStyle)
{
    var checked             = element.getAttribute("checked");
    var documentList        = webdeveloper_getDocuments(webdeveloper_getContentWindow());
    var documentLength      = documentList.length;
    var pageDocument        = null;
    var spanElement         = null;
    var spanElementList     = null;
    var spanElementsLength  = null;
    var stringBundle        = document.getElementById("webdeveloper-string-bundle");
    var tableElement        = null;
    var tableElementList    = null;
    var tableElementsLength = null;

    // Loop through the documents
    for(var i = 0; i < documentLength; i++)
    {
        pageDocument = documentList[i];

        // If the element is checked
        if(checked)
        {
            tableElementList    = pageDocument.getElementsByTagName("table");
            tableElementsLength = tableElementList.length;

            // Loop through all the table elements
            for(var j = 0; j < tableElementsLength; j++)
            {
                spanElement  = pageDocument.createElement("span");
                tableElement = tableElementList[j];

                spanElement.setAttribute("class", "webdeveloper-table-depth");
                spanElement.appendChild(document.createTextNode(stringBundle.getString("webdeveloper_depth") + "=" + webdeveloper_getTableDepth(tableElement)));
                tableElement.parentNode.insertBefore(spanElement, tableElement);
            }
        }
        else
        {
            spanElementList    = pageDocument.getElementsByTagName("span");
            spanElementsLength = spanElementList.length;

            // Loop through all the span tags
            for(j = 0; j < spanElementsLength; j++)
            {
                spanElement = spanElementList[j];

                // If the span element is set and the class exists and is set to webdeveloper-table-depth
                if(spanElement && spanElement.hasAttribute("class") && spanElement.getAttribute("class") == "webdeveloper-table-depth")
                {
                    spanElement.parentNode.removeChild(spanElement);
                    j--;
                }
            }
        }
    }

    webdeveloper_toggleStyleSheet(element, "chrome://webdeveloper/content/stylesheets/imports/tooltips.css", "webdeveloper-display-table-depth", applyStyle);
    webdeveloper_toggleFeatureTooltipStyles(element, "webdeveloper-display-table-depth-tooltips", "span.webdeveloper-table-depth");
}

// Displays table information
function webdeveloper_displayTableInformation(element, applyStyle)
{
    var checked                 = element.getAttribute("checked");
    var divElement              = null;
    var divElementList          = null;
    var divElementsLength       = null;
    var documentList            = webdeveloper_getDocuments(webdeveloper_getContentWindow());
    var documentLength          = documentList.length;
    var pageDocument            = null;
    var tableCellElementList    = null;
    var tableCellElementsLength = null;
    var tableElement            = null;
    var tableElementList        = null;
    var tableElementsLength     = null;

    // Loop through the documents
    for(var i = 0; i < documentLength; i++)
    {
        pageDocument = documentList[i];

        // If the element is checked
        if(checked)
        {
            tableElementList    = pageDocument.getElementsByTagName("table");
            tableElementsLength = tableElementList.length;

            // Loop through all the table elements
            for(var j = 0; j < tableElementsLength; j++)
            {
                tableElement            = tableElementList[j];
                tableCellElementList    = tableElement.getElementsByTagName("th");
                tableCellElementsLength = tableCellElementList.length;

                // If the table has a summary attribute
                if(tableElement.hasAttribute("summary"))
                {
                    divElement = pageDocument.createElement("div");

                    divElement.setAttribute("class", "webdeveloper-table-information");
                    divElement.appendChild(pageDocument.createTextNode("Summary=" + tableElement.getAttribute("summary")));
                    tableElement.parentNode.insertBefore(divElement, tableElement);
                }

                // Loop through the cell elements
                for(var k = 0; k < tableCellElementsLength; k++)
                {
                    webdeveloper_outputTableCellInformation(tableCellElementList[k], pageDocument);
                }

                tableCellElementList    = tableElement.getElementsByTagName("td");
                tableCellElementsLength = tableCellElementList.length;

                // Loop through the cell elements
                for(k = 0; k < tableCellElementsLength; k++)
                {
                    webdeveloper_outputTableCellInformation(tableCellElementList[k], pageDocument);
                }
            }
        }
        else
        {
            divElementList    = pageDocument.getElementsByTagName("div");
            divElementsLength = divElementList.length;

            // Loop through all the div tags
            for(j = 0; j < divElementsLength; j++)
            {
                divElement = divElementList[j];

                // If the div element is set and the class exists and is set to webdeveloper-table-information
                if(divElement && divElement.hasAttribute("class") && divElement.getAttribute("class") == "webdeveloper-table-information")
                {
                    divElement.parentNode.removeChild(divElement);
                    j--;
                }
            }
        }
    }

    webdeveloper_toggleStyleSheet(element, "chrome://webdeveloper/content/stylesheets/display_table_information.css", "webdeveloper-display-table-information", applyStyle);
    webdeveloper_toggleFeatureTooltipStyles(element, "webdeveloper-display-table-information-tooltips", "div.webdeveloper-table-information");
}

// Displays all title attributes
function webdeveloper_displayTitleAttributes(element, applyStyle)
{
    var checked                      = element.getAttribute("checked");
    var documentList                 = webdeveloper_getDocuments(webdeveloper_getContentWindow());
    var documentLength               = documentList.length;
    var pageDocument                 = null;
    var spanElement                  = null;
    var spanElementList              = null;
    var spanElementsLength           = null;
    var titleAttributeElement        = null;
    var titleAttributeElementList    = null;
    var titleAttributeElementsLength = null;
    var treeWalker                   = null;

    // Loop through the documents
    for(var i = 0; i < documentLength; i++)
    {
        pageDocument = documentList[i];

        // If the element is checked
        if(checked)
        {
            titleAttributeElementList = new Array();
            treeWalker                = pageDocument.createTreeWalker(pageDocument.body, NodeFilter.SHOW_ELEMENT, webdeveloper_titleAttributeFilter, false);

            // Loop through the title attribute elements
            while((titleAttributeElement = treeWalker.nextNode()) != null)
            {
                titleAttributeElementList.push(titleAttributeElement);
            }

            titleAttributeElementsLength = titleAttributeElementList.length;

            // Loop through all the title attribute elements
            for(var j = 0; j < titleAttributeElementsLength; j++)
            {
                spanElement           = pageDocument.createElement("span");
                titleAttributeElement = titleAttributeElementList[j];

                spanElement.setAttribute("class", "webdeveloper-title-attribute");
                spanElement.appendChild(pageDocument.createTextNode("Title=" + titleAttributeElement.getAttribute("title")));
                titleAttributeElement.parentNode.insertBefore(spanElement, titleAttributeElement);
            }
        }
        else
        {
            spanElementList    = pageDocument.getElementsByTagName("span");
            spanElementsLength = spanElementList.length;

            // Loop through all the span tags
            for(j = 0; j < spanElementsLength; j++)
            {
                spanElement = spanElementList[j];

                // If the span element is set and the class exists and is set to webdeveloper-title-attribute
                if(spanElement && spanElement.hasAttribute("class") && spanElement.getAttribute("class") == "webdeveloper-title-attribute")
                {
                    spanElement.parentNode.removeChild(spanElement);
                    j--;
                }
            }
        }
    }

    webdeveloper_toggleStyleSheet(element, "chrome://webdeveloper/content/stylesheets/imports/tooltips.css", "webdeveloper-display-title-attributes", applyStyle);
    webdeveloper_toggleFeatureTooltipStyles(element, "webdeveloper-display-title-attributes-tooltips", "span.webdeveloper-title-attribute");
}

// Displays the topographic information for all elements
function webdeveloper_displayTopographicInformation(element, applyStyle)
{
    webdeveloper_toggleStyleSheet(element, "chrome://webdeveloper/content/stylesheets/display_topographic_information.css", "webdeveloper-display-topographic-information", applyStyle);
}

// Returns the depth of the table
function webdeveloper_getTableDepth(table)
{
    var depth = 1;

    // If the table is set
    if(table)
    {
        var ancestorList   = webdeveloper_getElementAncestors(table);
        var ancestorLength = ancestorList.length;
        var tagName        = null;

        // Loop through the ancestors
        for(var i = 0; i < ancestorLength; i++)
        {
            tagName = ancestorList[i].tagName;

            // If the tag name is set and equals table
            if(tagName && tagName.toLowerCase() == "table")
            {
                depth++;
            }
        }
    }

    return depth;
}

// Returns the text from a table heading
function webdeveloper_getTableHeadingText(headingNode, headingText)
{
    // If the heading node is set
    if(headingNode)
    {
        var childNode       = null;
        var childNodeList   = headingNode.childNodes;
        var childNodeLength = childNodeList.length;
        var childNodeType   = null;

        // Loop through the child nodes
        for(var i = 0; i < childNodeLength; i++)
        {
            childNode     = childNodeList[i];
            childNodeType = childNode.nodeType;

            // If the child node type is an element and it does not have a webdeveloper-table-information class
            if(childNodeType == Node.ELEMENT_NODE && (!childNode.hasAttribute("class") || childNode.getAttribute("class") != "webdeveloper-table-information"))
            {
                headingText = webdeveloper_getTableHeadingText(childNode, headingText);
            }
            else if(childNodeType == Node.TEXT_NODE)
            {
                headingText += childNode.nodeValue + " ";
            }
        }
    }

    return headingText.trim();
}

// Filter for headings tree walker
function webdeveloper_headingFilter(node)
{
    // If the node has an id attribute or is an a tag with a name attribute
    if(node.nodeName && webdeveloper_contains(new Array("h1", "h2", "h3", "h4", "h5", "h6"), node.nodeName.toLowerCase()))
    {
        return NodeFilter.FILTER_ACCEPT;
    }

    return NodeFilter.FILTER_SKIP;
}

// Outputs the information for a table cell
function webdeveloper_outputTableCellInformation(tableCellElement, pageDocument)
{
    // If the table cell element is set
    if(tableCellElement)
    {
        var divElement   = null;
        var pElement     = null;
        var stringBundle = document.getElementById("webdeveloper-string-bundle");

        // If the table cell has a headers attribute
        if(tableCellElement.hasAttribute("headers"))
        {
            var definitionElement = pageDocument.createElement("dd");
            var headerList        = tableCellElement.getAttribute("headers").split(" ");
            var headersLength     = headerList.length;

            // Loop through the headers
            for(var i = 0; i < headersLength; i++)
            {
                pElement = pageDocument.createElement("p");
                pElement.appendChild(pageDocument.createTextNode(webdeveloper_getTableHeadingText(pageDocument.getElementById(headerList[i]), "")));
                definitionElement.appendChild(pElement);
            }

            // If the definition element has child nodes
            if(definitionElement.childNodes.length > 0)
            {
                var listElement = pageDocument.createElement("dl");
                var termElement = pageDocument.createElement("dt");

                divElement = pageDocument.createElement("div");

                termElement.appendChild(pageDocument.createTextNode(stringBundle.getString("webdeveloper_headers")));
                listElement.appendChild(termElement);
                listElement.appendChild(definitionElement);

                divElement.setAttribute("class", "webdeveloper-table-information");
                divElement.appendChild(listElement);
                tableCellElement.appendChild(divElement);
            }
        }

        divElement = pageDocument.createElement("div");

        // If the table cell has a scope attribute
        if(tableCellElement.hasAttribute("scope"))
        {
            var scope = tableCellElement.getAttribute("scope");

            pElement = pageDocument.createElement("p");

            // If the scope is col
            if(scope == "col")
            {
                pElement.appendChild(pageDocument.createTextNode(stringBundle.getString("webdeveloper_tableInformationHeadingColumn")));
            }
            else if(scope == "colgroup")
            {
                pElement.appendChild(pageDocument.createTextNode(stringBundle.getString("webdeveloper_tableInformationHeadingColumnGroup")));
            }
            else if(scope == "row")
            {
                pElement.appendChild(pageDocument.createTextNode(stringBundle.getString("webdeveloper_tableInformationHeadingRow")));
            }
            else if(scope == "rowgroup")
            {
                pElement.appendChild(pageDocument.createTextNode(stringBundle.getString("webdeveloper_tableInformationHeadingRowGroup")));
            }

            divElement.appendChild(pElement);
        }

        // If the table cell has an abbr attribute
        if(tableCellElement.hasAttribute("abbr"))
        {
            pElement = pageDocument.createElement("p");
            pElement.appendChild(pageDocument.createTextNode(stringBundle.getString("webdeveloper_abbreviation") + "=" + tableCellElement.getAttribute("abbr")));
            divElement.appendChild(pElement);
        }

        // If the table cell has an axis attribute
        if(tableCellElement.hasAttribute("axis"))
        {
            pElement = pageDocument.createElement("p");
            pElement.appendChild(pageDocument.createTextNode(stringBundle.getString("webdeveloper_axis") + "=" + tableCellElement.getAttribute("axis")));
            divElement.appendChild(pElement);
        }

        // If the div element has child nodes
        if(divElement.childNodes.length > 0)
        {
            divElement.setAttribute("class", "webdeveloper-table-information");
            tableCellElement.appendChild(divElement);
        }
    }
}

// Sets the image file sizes for a list of images
function webdeveloper_setImageFileSizes(imageList)
{
    var image           = null;
    var imageListLength = imageList.length;
    var newImageList    = new Array();

    // Loop through the images
    for(var i = 0; i < imageListLength; i++)
    {
        image          = imageList[i];
        image.fileSize = webdeveloper_getFileSize(image.src);

        newImageList.push(image);
    }

    return newImageList;
}

// Sets the script file sizes for a list of scripts
function webdeveloper_setScriptFileSizes(scriptList)
{
    var script           = null;
    var scriptListLength = scriptList.length;
    var newScriptList    = new Array();

    // Loop through the images
    for(var i = 0; i < scriptListLength; i++)
    {
        script          = scriptList[i];
        script.fileSize = webdeveloper_getFileSize(script.src);

        newScriptList.push(script);
    }

    return newScriptList;
}

// Sets the style sheet file sizes for a list of style sheets
function webdeveloper_setStyleSheetFileSizes(styleSheetList)
{
    var styleSheet           = null;
    var styleSheetListLength = styleSheetList.length;
    var newStyleSheetList    = new Array();

    // Loop through the style sheets
    for(var i = 0; i < styleSheetListLength; i++)
    {
        styleSheet          = styleSheetList[i];
        styleSheet.fileSize = webdeveloper_getFileSize(styleSheet.href);

        newStyleSheetList.push(styleSheet);
    }

    return newStyleSheetList;
}

// Sorts two anchors
function webdeveloper_sortAnchors(anchorOne, anchorTwo)
{
    // If both anchors are set
    if(anchorOne && anchorTwo)
    {
        var anchorOneLink = null;
        var anchorTwoLink = null;

        // If the first anchor has an id attribute
        if(anchorOne.hasAttribute("id"))
        {
            anchorOneLink = anchorOne.getAttribute("id");
        }
        else if(anchorOne.hasAttribute("name"))
        {
            anchorOneLink = anchorOne.getAttribute("name");
        }

        // If the second anchor has an id attribute
        if(anchorTwo.hasAttribute("id"))
        {
            anchorTwoLink = anchorTwo.getAttribute("id");
        }
        else if(anchorTwo.hasAttribute("name"))
        {
            anchorTwoLink = anchorTwo.getAttribute("name");
        }

        // If the anchor links are equal
        if(anchorOneLink == anchorTwoLink)
        {
            return 0;
        }
        else if(anchorOneLink < anchorTwoLink)
        {
            return -1;
        }
    }

    return 1;
}

// Sorts two documents
function webdeveloper_sortDocuments(documentOne, documentTwo)
{
    // If both documents are set
    if(documentOne && documentTwo)
    {
        var documentOneURL = documentOne.documentURI;
        var documentTwoURL = documentTwo.documentURI;

        // If the documents are equal
        if(documentOneURL == documentTwoURL)
        {
            return 0;
        }
        else if(documentOneURL < documentTwoURL)
        {
            return -1;
        }
    }

    return 1;
}

// Sorts two documents by file size
function webdeveloper_sortDocumentsByFileSize(documentOne, documentTwo)
{
    // If both documents and their URLs are set
    if(documentOne && documentTwo && documentOne.documentURI && documentTwo.documentURI)
    {
        var documentOneFileSize = webdeveloper_getFileSize(documentOne.documentURI);
        var documentTwoFileSize = webdeveloper_getFileSize(documentTwo.documentURI);

        // If the document file sizes are equal
        if(documentOneFileSize == documentTwoFileSize)
        {
            return webdeveloper_sortDocuments(documentOne, documentTwo);
        }
        else if(documentOneFileSize > documentTwoFileSize)
        {
            return -1;
        }
    }

    return 1;
}

// Sorts two images by file size
function webdeveloper_sortImagesByFileSize(imageOne, imageTwo)
{
    // If both images and their file sizes are set
    if(imageOne && imageTwo && imageOne.fileSize && imageTwo.fileSize)
    {
        var imageOneFileSize = imageOne.fileSize;
        var imageTwoFileSize = imageTwo.fileSize;

        // If the image file sizes are equal
        if(imageOneFileSize == imageTwoFileSize)
        {
            return webdeveloper_sortImages(imageOne, imageTwo);
        }
        else if(imageOneFileSize > imageTwoFileSize)
        {
            return -1;
        }
    }

    return 1;
}

// Sorts two links
function webdeveloper_sortLinks(linkOne, linkTwo)
{
    // If both links are set
    if(linkOne && linkTwo)
    {
        var linkOneHref = linkOne.href;
        var linkTwoHref = linkTwo.href;

        // If the links are equal
        if(linkOneHref == linkTwoHref)
        {
            return 0;
        }
        else if(linkOneHref < linkTwoHref)
        {
            return -1;
        }
    }

    return 1;
}

// Sorts two scripts
function webdeveloper_sortScripts(scriptOne, scriptTwo)
{
    // If both scripts and src are set
    if(scriptOne && scriptTwo && scriptOne.src && scriptTwo.src)
    {
        var scriptOneSrc = scriptOne.src;
        var scriptTwoSrc = scriptTwo.src;

        // If the scripts are equal
        if(scriptOneSrc == scriptTwoSrc)
        {
            return 0;
        }
        else if(scriptOneSrc < scriptTwoSrc)
        {
            return -1;
        }
    }

    return 1;
}

// Sorts two scripts by file size
function webdeveloper_sortScriptsByFileSize(scriptOne, scriptTwo)
{
    // If both scripts and src are set
    if(scriptOne && scriptTwo && scriptOne.src && scriptTwo.src)
    {
        var scriptOneFileSize = webdeveloper_getFileSize(scriptOne.src);
        var scriptTwoFileSize = webdeveloper_getFileSize(scriptTwo.src);

        // If the script file sizes are equal
        if(scriptOneFileSize == scriptTwoFileSize)
        {
            return webdeveloper_sortScripts(scriptOne, scriptTwo);
        }
        else if(scriptOneFileSize > scriptTwoFileSize)
        {
            return -1;
        }
    }

    return 1;
}

// Sorts two style sheets
function webdeveloper_sortStyleSheets(styleSheetOne, styleSheetTwo)
{
    // If both style sheets and href are set
    if(styleSheetOne && styleSheetTwo && styleSheetOne.href && styleSheetTwo.href)
    {
        var styleSheetOneSrc = styleSheetOne.href;
        var styleSheetTwoSrc = styleSheetTwo.href;

        // If the style sheets are equal
        if(styleSheetOneSrc == styleSheetTwoSrc)
        {
            return 0;
        }
        else if(styleSheetOneSrc < styleSheetTwoSrc)
        {
            return -1;
        }
    }

    return 1;
}

// Sorts two style sheets by file size
function webdeveloper_sortStyleSheetsByFileSize(styleSheetOne, styleSheetTwo)
{
    // If both style sheets and href are set
    if(styleSheetOne && styleSheetTwo && styleSheetOne.href && styleSheetTwo.href)
    {
        var styleSheetOneFileSize = webdeveloper_getFileSize(styleSheetOne.href);
        var styleSheetTwoFileSize = webdeveloper_getFileSize(styleSheetTwo.href);

        // If the style sheet file sizes are equal
        if(styleSheetOneFileSize == styleSheetTwoFileSize)
        {
            return webdeveloper_sortStyleSheets(styleSheetOne, styleSheetTwo);
        }
        else if(styleSheetOneFileSize > styleSheetTwoFileSize)
        {
            return -1;
        }
    }

    return 1;
}

// Filter for stack level tree walker
function webdeveloper_stackLevelFilter(node)
{
    // If the node has a class attribute and it starts with webdeveloper
    if(node && (!node.hasAttribute("class") || node.getAttribute("class").indexOf("webdeveloper-") != 0))
    {
        var zIndex = node.ownerDocument.defaultView.getComputedStyle(node, null).getPropertyCSSValue("z-index").cssText;

        // If the node has a z-index and it is not set to auto
        if(zIndex && zIndex != "auto")
        {
            return NodeFilter.FILTER_ACCEPT;
        }
    }

    return NodeFilter.FILTER_SKIP;
}

// Filter for display tab index tree walker
function webdeveloper_tabIndexFilter(node)
{
    // If the node has an tabindex attribute
    if(node.hasAttribute("tabindex"))
    {
        return NodeFilter.FILTER_ACCEPT;
    }

    return NodeFilter.FILTER_SKIP;
}

// Tidies a list of anchors by removing duplicates and sorting
function webdeveloper_tidyAnchors(anchorList)
{
    var anchor              = null;
    var anchorLink          = null;
    var anchorListLength    = anchorList.length;
    var newAnchorList       = new Array();
    var newAnchorListLength = null;
    var nextAnchor          = null;
    var nextAnchorLink      = null;
    var tidiedAnchorList    = new Array();

    // Loop through the anchors
    for(var i = 0; i < anchorListLength; i++)
    {
        newAnchorList.push(anchorList[i]);
    }

    newAnchorList.sort(webdeveloper_sortAnchors);

    newAnchorListLength = newAnchorList.length;

    // Loop through the anchors
    for(i = 0; i < newAnchorListLength; i++)
    {
        anchor = newAnchorList[i];

        // If the anchor has an id attribute
        if(anchor.hasAttribute("id"))
        {
            anchorLink = anchor.getAttribute("id");
        }
        else
        {
            anchorLink = anchor.getAttribute("name");
        }

        // If this is not the last anchor
        if(i + 1 < newAnchorListLength)
        {
            nextAnchor = newAnchorList[i + 1];

            // If the next anchor has an id attribute
            if(nextAnchor.hasAttribute("id"))
            {
                nextAnchorLink = nextAnchor.getAttribute("id");
            }
            else
            {
                nextAnchorLink = nextAnchor.getAttribute("name");
            }

            // If the anchor link is the same as the next anchor link
            if(anchorLink == nextAnchorLink)
            {
                continue;
            }
        }

        tidiedAnchorList.push(anchor);
    }

    return tidiedAnchorList;
}

// Tidies a list of colors by removing duplicates and sorting
function webdeveloper_tidyColors(colorList)
{
    var color              = null;
    var colorListLength    = colorList.length;
    var newColorList       = new Array();
    var newColorListLength = null;
    var tidiedColorList    = new Array();

    // Loop through the colors
    for(var i = 0; i < colorListLength; i++)
    {
        newColorList.push(colorList[i]);
    }

    newColorList.sort();

    newColorListLength = newColorList.length;

    // Loop through the colors
    for(i = 0; i < newColorListLength; i++)
    {
        color = newColorList[i];

        // If this is not the last color and the color is the same as the next color
        if(i + 1 < newColorListLength && color == newColorList[i + 1])
        {
            continue;
        }

        tidiedColorList.push(color);
    }

    return tidiedColorList;
}

// Tidies a list of documents by removing duplicates and sorting
function webdeveloper_tidyDocuments(documentList)
{
    var documentListLength    = documentList.length;
    var documentURL           = null;
    var newDocumentList       = new Array();
    var newDocumentListLength = null;
    var pageDocument          = null;
    var tidiedDocumentList    = new Array();

    // Loop through the documents
    for(var i = 0; i < documentListLength; i++)
    {
        newDocumentList.push(documentList[i]);
    }

    newDocumentList.sort(webdeveloper_sortDocuments);

    newDocumentListLength = newDocumentList.length;

    // Loop through the documents
    for(i = 0; i < newDocumentListLength; i++)
    {
        pageDocument = newDocumentList[i];

        // If this is not the last document and the document is the same as the next document
        if(i + 1 < newDocumentListLength && pageDocument.documentURI == newDocumentList[i + 1].documentURI)
        {
            continue;
        }

        tidiedDocumentList.push(pageDocument);
    }

    return tidiedDocumentList;
}

// Tidies a list of links by removing duplicates and sorting
function webdeveloper_tidyLinks(linkList)
{
    var link              = null;
    var linkListLength    = linkList.length;
    var newLinkList       = new Array();
    var newLinkListLength = null;
    var tidiedLinkList    = new Array();

    // Loop through the links
    for(var i = 0; i < linkListLength; i++)
    {
        link = linkList[i];

        // If this link is set
        if(link.href)
        {
            newLinkList.push(link);
        }
    }

    newLinkList.sort(webdeveloper_sortLinks);

    newLinkListLength = newLinkList.length;

    // Loop through the links
    for(i = 0; i < newLinkListLength; i++)
    {
        link = newLinkList[i];

        // If this is not the last link and the link is the same as the next link
        if(i + 1 < newLinkListLength && link.href == newLinkList[i + 1].href)
        {
            continue;
        }

        tidiedLinkList.push(link);
    }

    return tidiedLinkList;
}

// Tidies a list of scripts by removing duplicates and sorting
function webdeveloper_tidyScripts(scriptList)
{
    var script              = null;
    var scriptListLength    = scriptList.length;
    var newScriptList       = new Array();
    var newScriptListLength = null;
    var tidiedScriptList    = new Array();

    // Loop through the scripts
    for(var i = 0; i < scriptListLength; i++)
    {
        script = scriptList[i];

        // If this script and the src is set
        if(script && script.src)
        {
            newScriptList.push(script);
        }
    }

    newScriptList.sort(webdeveloper_sortScripts);

    newScriptListLength = newScriptList.length;

    // Loop through the scripts
    for(i = 0; i < newScriptListLength; i++)
    {
        script = newScriptList[i];

        // If this is not the last script and the script is the same as the next script
        if(i + 1 < newScriptListLength && script.src == newScriptList[i + 1].src)
        {
            continue;
        }

        tidiedScriptList.push(script);
    }

    return tidiedScriptList;
}

// Tidies a list of style sheets by removing duplicates and sorting
function webdeveloper_tidyStyleSheets(styleSheetList)
{
    var styleSheet              = null;
    var styleSheetListLength    = styleSheetList.length;
    var newStyleSheetList       = new Array();
    var newStyleSheetListLength = null;
    var tidiedStyleSheetList    = new Array();

    // Loop through the style sheets
    for(var i = 0; i < styleSheetListLength; i++)
    {
        styleSheet = styleSheetList[i];

        // If this style sheet and the href is set
        if(styleSheet && styleSheet.href)
        {
            newStyleSheetList.push(styleSheet);
        }
    }

    newStyleSheetList.sort(webdeveloper_sortStyleSheets);

    newStyleSheetListLength = newStyleSheetList.length;

    // Loop through the style sheets
    for(i = 0; i < newStyleSheetListLength; i++)
    {
        styleSheet = newStyleSheetList[i];

        // If this is not the last style sheet and the style sheet is the same as the next style sheet
        if(i + 1 < newStyleSheetListLength && styleSheet.href == newStyleSheetList[i + 1].href)
        {
            continue;
        }

        tidiedStyleSheetList.push(styleSheet);
    }

    return tidiedStyleSheetList;
}

// Filter for display title attributes tree walker
function webdeveloper_titleAttributeFilter(node)
{
    // If the node has a title attribute
    if(node.hasAttribute("title"))
    {
        return NodeFilter.FILTER_ACCEPT;
    }

    return NodeFilter.FILTER_SKIP;
}

// Updates the information menu
function webdeveloper_updateInformationMenu(suffix)
{
    webdeveloper_configureElementByElement(document.getElementById("webdeveloper-display-access-keys-" + suffix), "checked", "webdeveloper-display-access-keys");
    webdeveloper_configureElementByElement(document.getElementById("webdeveloper-display-anchors-" + suffix), "checked", "webdeveloper-display-anchors");
    webdeveloper_configureElementByElement(document.getElementById("webdeveloper-display-block-size-" + suffix), "checked", "webdeveloper-display-block-size");
    webdeveloper_configureElementByElement(document.getElementById("webdeveloper-display-div-order-" + suffix), "checked", "webdeveloper-display-div-order");
    webdeveloper_configureElementByElement(document.getElementById("webdeveloper-display-element-information-" + suffix), "checked", "webdeveloper-display-element-information");
    webdeveloper_configureElementByElement(document.getElementById("webdeveloper-display-id-class-details-" + suffix), "checked", "webdeveloper-display-id-class-details");
    webdeveloper_configureElementByElement(document.getElementById("webdeveloper-display-link-details-" + suffix), "checked", "webdeveloper-display-link-details");
    webdeveloper_configureElementByElement(document.getElementById("webdeveloper-display-object-information-" + suffix), "checked", "webdeveloper-display-object-information");
    webdeveloper_configureElementByElement(document.getElementById("webdeveloper-display-stack-levels-" + suffix), "checked", "webdeveloper-display-stack-levels");
    webdeveloper_configureElementByElement(document.getElementById("webdeveloper-display-tab-index-" + suffix), "checked", "webdeveloper-display-tab-index");
    webdeveloper_configureElementByElement(document.getElementById("webdeveloper-display-table-depth-" + suffix), "checked", "webdeveloper-display-table-depth");
    webdeveloper_configureElementByElement(document.getElementById("webdeveloper-display-table-information-" + suffix), "checked", "webdeveloper-display-table-information");
    webdeveloper_configureElementByElement(document.getElementById("webdeveloper-display-title-attributes-" + suffix), "checked", "webdeveloper-display-title-attributes");
    webdeveloper_configureElementByElement(document.getElementById("webdeveloper-display-topographic-information-" + suffix), "checked", "webdeveloper-display-topographic-information");
}

// Displays all the anchors for the page
function webdeveloper_viewAnchorInformation()
{
    var anchor            = null;
    var anchorLength      = null;
    var anchorLink        = null;
    var anchorList        = null;
    var documentList      = webdeveloper_getDocuments(webdeveloper_getContentWindow());
    var documentLength    = documentList.length;
    var documentURL       = null;
    var linkElement       = null;
    var linkHref          = null;
    var linkLength        = null;
    var listElement       = null;
    var listItemElement   = null;
    var oldTab            = getBrowser().selectedTab;
    var oldURL            = getBrowser().currentURI.spec;
    var generatedDocument = webdeveloper_generateDocument("");
    var headerElement     = generatedDocument.createElement("h1");
    var pageDocument      = null;
    var pElement          = null;
    var stringBundle      = document.getElementById("webdeveloper-string-bundle");
    var title             = stringBundle.getFormattedString("webdeveloper_viewAnchorInformationTitle", [oldURL]);
    var treeWalker        = null;

    generatedDocument.title = title;

    webdeveloper_addGeneratedStyles(generatedDocument);

    headerElement.appendChild(generatedDocument.createTextNode(title));
    generatedDocument.body.appendChild(headerElement);

    // Loop through the documents
    for(var i = 0; i < documentLength; i++)
    {
        anchorList    = new Array();
        headerElement = generatedDocument.createElement("h2");
        linkElement   = generatedDocument.createElement("a");
        pageDocument  = documentList[i];
        documentURL   = pageDocument.documentURI;
        treeWalker    = pageDocument.createTreeWalker(pageDocument.body, NodeFilter.SHOW_ELEMENT, webdeveloper_anchorFilter, false);

        // Loop through the anchor elements
        while((anchor = treeWalker.nextNode()) != null)
        {
            anchorList.push(anchor);
        }

        // If the tidy information preference is set
        if(webdeveloper_getBooleanPreference("webdeveloper.information.tidy", true))
        {
            anchorList = webdeveloper_tidyAnchors(anchorList);
        }

        anchorLength = anchorList.length;

        linkElement.setAttribute("href", documentURL);
        linkElement.appendChild(generatedDocument.createTextNode(documentURL));
        headerElement.appendChild(linkElement);
        generatedDocument.body.appendChild(headerElement);

        // If there are no anchors
        if(anchorLength == 0)
        {
            pElement = generatedDocument.createElement("p");

            pElement.appendChild(generatedDocument.createTextNode(stringBundle.getString("webdeveloper_noAnchors")));
            generatedDocument.body.appendChild(pElement);
        }
        else
        {
            listElement = generatedDocument.createElement("ol");

            // Loop through the links
            for(var j = 0; j < anchorLength; j++)
            {
                anchor = anchorList[j];

                // If the anchor has an id attribute
                if(anchor.hasAttribute("id"))
                {
                    anchorLink = "#" + anchor.getAttribute("id");
                }
                else
                {
                    anchorLink = "#" + anchor.getAttribute("name");
                }

                linkElement     = generatedDocument.createElement("a");
                listItemElement = generatedDocument.createElement("li");

                linkElement.setAttribute("href", documentURL + anchorLink);
                linkElement.appendChild(generatedDocument.createTextNode(anchorLink));
                listItemElement.appendChild(linkElement);
                listElement.appendChild(listItemElement);
            }

            generatedDocument.body.appendChild(listElement);
        }
    }

    // If the open tabs in background preference is set to true
    if(webdeveloper_getBooleanPreference("webdeveloper.open.tabs.background", true))
    {
        getBrowser().selectedTab = oldTab;
    }
}

// Displays all the colors on the page
function webdeveloper_viewColorInformation()
{
    var color             = null;
    var colorList         = null;
    var colorsLength      = null;
    var divElement        = null;
    var documentList      = webdeveloper_getDocuments(webdeveloper_getContentWindow());
    var documentLength    = documentList.length;
    var documentURL       = null;
    var element           = null;
    var oldTab            = getBrowser().selectedTab;
    var oldURL            = getBrowser().currentURI.spec;
    var generatedDocument = webdeveloper_generateDocument("");
    var headerElement     = generatedDocument.createElement("h1");
    var linkElement       = generatedDocument.createElement("link");
    var pageDocument      = null;
    var pElement          = null;
    var title             = document.getElementById("webdeveloper-string-bundle").getFormattedString("webdeveloper_viewColorInformationTitle", [oldURL]);
    var treeWalker        = null;

    generatedDocument.title = title;

    webdeveloper_addGeneratedStyles(generatedDocument);

    linkElement.setAttribute("href", "chrome://webdeveloper/content/stylesheets/generated/view_color_information.css");
    linkElement.setAttribute("rel", "stylesheet");
    linkElement.setAttribute("type", "text/css");
    generatedDocument.getElementsByTagName("head")[0].appendChild(linkElement);

    headerElement.appendChild(generatedDocument.createTextNode(title));
    generatedDocument.body.appendChild(headerElement);

    // Loop through the documents
    for(var i = 0; i < documentLength; i++)
    {
        colorList     = new Array();
        headerElement = generatedDocument.createElement("h2");
        linkElement   = generatedDocument.createElement("a");
        pageDocument  = documentList[i];
        documentURL   = pageDocument.documentURI;
        treeWalker    = pageDocument.createTreeWalker(pageDocument.body, NodeFilter.SHOW_ELEMENT, null, false);

        // Loop through the anchor elements
        while((element = treeWalker.nextNode()) != null)
        {
            webdeveloper_addColor(element, "background-color", colorList);
            webdeveloper_addColor(element, "border-bottom-color", colorList);
            webdeveloper_addColor(element, "border-left-color", colorList);
            webdeveloper_addColor(element, "border-right-color", colorList);
            webdeveloper_addColor(element, "border-top-color", colorList);
            webdeveloper_addColor(element, "color", colorList);
        }

        colorList    = webdeveloper_tidyColors(colorList);
        colorsLength = colorList.length;

        linkElement.setAttribute("href", documentURL);
        linkElement.appendChild(generatedDocument.createTextNode(documentURL));
        headerElement.appendChild(linkElement);
        generatedDocument.body.appendChild(headerElement);

        // Loop through the colors
        for(var j = 0; j < colorsLength; j++)
        {
            color      = colorList[j];
            divElement = generatedDocument.createElement("div");
            pElement   = generatedDocument.createElement("p");

            pElement.appendChild(generatedDocument.createTextNode(color));
            generatedDocument.body.appendChild(pElement);

            divElement.setAttribute("style", "background-color: " + color);
            generatedDocument.body.appendChild(divElement);
        }
    }

    // If the open tabs in background preference is set to true
    if(webdeveloper_getBooleanPreference("webdeveloper.open.tabs.background", true))
    {
        getBrowser().selectedTab = oldTab;
    }
}

// Displays an outline of the page
function webdeveloper_viewDocumentOutline()
{
    var documentList        = webdeveloper_getDocuments(webdeveloper_getContentWindow());
    var documentLength      = documentList.length;
    var documentURL         = null;
    var headerLevel         = null;
    var headerTagName       = null;
    var headerText          = null;
    var oldTab              = getBrowser().selectedTab;
    var oldURL              = getBrowser().currentURI.spec;
    var generatedDocument   = webdeveloper_generateDocument("");
    var headerElement       = generatedDocument.createElement("h1");
    var linkElement         = generatedDocument.createElement("link");
    var outlineElement      = null;
    var pageDocument        = null;
    var previousHeaderLevel = null;
    var stringBundle        = document.getElementById("webdeveloper-string-bundle");
    var title               = stringBundle.getFormattedString("webdeveloper_viewDocumentOutlineTitle", [oldURL]);
    var treeWalker          = null;

    generatedDocument.title = title;

    webdeveloper_addGeneratedStyles(generatedDocument);

    linkElement.setAttribute("href", "chrome://webdeveloper/content/stylesheets/generated/view_document_outline.css");
    linkElement.setAttribute("rel", "stylesheet");
    linkElement.setAttribute("type", "text/css");
    generatedDocument.getElementsByTagName("head")[0].appendChild(linkElement);

    headerElement.appendChild(generatedDocument.createTextNode(title));
    generatedDocument.body.appendChild(headerElement);

    // Loop through the documents
    for(var i = 0; i < documentLength; i++)
    {
        headerElement       = generatedDocument.createElement("h2");
        linkElement         = generatedDocument.createElement("a");
        pageDocument        = documentList[i];
        documentURL         = pageDocument.documentURI;
        previousHeaderLevel = 0;
        treeWalker          = pageDocument.createTreeWalker(pageDocument.body, NodeFilter.SHOW_ELEMENT, webdeveloper_headingFilter, false);

        linkElement.setAttribute("href", documentURL);
        linkElement.appendChild(generatedDocument.createTextNode(documentURL));
        headerElement.appendChild(linkElement);
        generatedDocument.body.appendChild(headerElement);

        // Loop through the heading elements
        while((headerElement = treeWalker.nextNode()) != null)
        {
            headerTagName = headerElement.tagName.toLowerCase();
            headerText    = webdeveloper_getElementText(headerElement).trim();
            headerLevel   = parseInt(headerTagName.substring(1));

            // Loop through any missing headers
            for(var j = previousHeaderLevel + 1; j < headerLevel; j++)
            {
                outlineElement = generatedDocument.createElement("h" + j);

                outlineElement.setAttribute("class", "webdeveloper-document-outline webdeveloper-document-outline-missing");
                outlineElement.appendChild(generatedDocument.createTextNode(stringBundle.getString("webdeveloper_viewDocumentOutlineMissingHeading")));
                generatedDocument.body.appendChild(outlineElement);
            }

            // If there is no header text
            if(!headerText)
            {
                headerText = stringBundle.getString("webdeveloper_viewDocumentOutlineEmptyHeading");
            }

            outlineElement = generatedDocument.createElement(headerTagName);

            outlineElement.setAttribute("class", "webdeveloper-document-outline");
            outlineElement.appendChild(generatedDocument.createTextNode(headerText));
            generatedDocument.body.appendChild(outlineElement);

            previousHeaderLevel = headerLevel;
        }
    }

    // If the open tabs in background preference is set to true
    if(webdeveloper_getBooleanPreference("webdeveloper.open.tabs.background", true))
    {
        getBrowser().selectedTab = oldTab;
    }
}

// Displays the size of the page
function webdeveloper_viewDocumentSize()
{
    var calculatedFileSize   = 0;
    var documentList         = webdeveloper_tidyDocuments(webdeveloper_getDocuments(webdeveloper_getContentWindow())).sort(webdeveloper_sortDocumentsByFileSize);
    var documentLength       = documentList.length;
    var documentURL          = null;
    var fileSize             = null;
    var image                = null;
    var imageLength          = null;
    var imageList            = new Array();
    var oldTab               = getBrowser().selectedTab;
    var oldURL               = getBrowser().currentURI.spec;
    var generatedDocument    = webdeveloper_generateDocument("");
    var cellDataElement      = generatedDocument.createElement("td");
    var headElement          = generatedDocument.getElementsByTagName("head")[0];
    var headerElement        = generatedDocument.createElement("h1");
    var linkElement          = generatedDocument.createElement("link");
    var pageDocument         = null;
    var scriptElement        = null;
    var scriptLength         = null;
    var scriptList           = new Array();
    var src                  = null;
    var stringBundle         = document.getElementById("webdeveloper-string-bundle");
    var styleSheet           = null;
    var styleSheetLength     = null;
    var styleSheetList       = new Array();
    var tableElement         = generatedDocument.createElement("table");
    var tableRowElement      = generatedDocument.createElement("tr");
    var title                = stringBundle.getFormattedString("webdeveloper_viewDocumentSizeTitle", [oldURL]);
    var totalCellDataElement = null;
    var totalFileSize        = 0;

    generatedDocument.title = title;

    webdeveloper_addGeneratedStyles(generatedDocument);

    linkElement.setAttribute("href", "chrome://webdeveloper/content/stylesheets/generated/view_document_size.css");
    linkElement.setAttribute("rel", "stylesheet");
    linkElement.setAttribute("type", "text/css");
    headElement.appendChild(linkElement);

    headerElement.appendChild(generatedDocument.createTextNode(title));
    generatedDocument.body.appendChild(headerElement);

    cellDataElement.setAttribute("class", "pivot");
    tableRowElement.appendChild(cellDataElement);

    cellDataElement = generatedDocument.createElement("td");

    // If there is one document
    if(documentLength == 1)
    {
        cellDataElement.appendChild(generatedDocument.createTextNode(stringBundle.getString("webdeveloper_documents") + " (" + documentLength + " " + stringBundle.getString("webdeveloper_file").toLowerCase() + ")"));
    }
    else
    {
        cellDataElement.appendChild(generatedDocument.createTextNode(stringBundle.getString("webdeveloper_documents") + " (" + documentLength + " " + stringBundle.getString("webdeveloper_files").toLowerCase() + ")"));
    }

    tableRowElement.appendChild(cellDataElement);

    totalCellDataElement = generatedDocument.createElement("td");

    totalCellDataElement.setAttribute("class", "size");
    tableRowElement.appendChild(totalCellDataElement);
    tableRowElement.setAttribute("class", "collapsed");
    tableElement.appendChild(tableRowElement);

    // Loop through the documents
    for(var i = 0; i < documentLength; i++)
    {
        cellDataElement = generatedDocument.createElement("td");
        pageDocument    = documentList[i];
        documentURL     = pageDocument.documentURI;
        fileSize        = webdeveloper_getFileSize(documentURL);
        imageList       = imageList.concat(webdeveloper_getImagesForDocument(pageDocument, true, true));
        linkElement     = generatedDocument.createElement("a");
        scriptList      = scriptList.concat(webdeveloper_getScriptsForDocument(pageDocument, false));
        styleSheetList  = styleSheetList.concat(webdeveloper_getStyleSheetsForDocument(pageDocument, false, false));
        tableRowElement = generatedDocument.createElement("tr");

        // If the file size is set
        if(fileSize)
        {
            calculatedFileSize += fileSize;
        }

        // If this is an even row
        if(i % 2 == 0)
        {
            tableRowElement.setAttribute("class", "hidden shaded");
        }
        else
        {
            tableRowElement.setAttribute("class", "hidden");
        }

        cellDataElement.setAttribute("class", "indent");
        tableRowElement.appendChild(cellDataElement);

        cellDataElement = generatedDocument.createElement("td");

        linkElement.setAttribute("href", documentURL);
        linkElement.appendChild(generatedDocument.createTextNode(documentURL));
        cellDataElement.appendChild(linkElement);
        tableRowElement.appendChild(cellDataElement);

        cellDataElement = generatedDocument.createElement("td");

        cellDataElement.setAttribute("class", "size");
        cellDataElement.appendChild(generatedDocument.createTextNode(webdeveloper_formatFileSize(fileSize)));
        tableRowElement.appendChild(cellDataElement);
        tableElement.appendChild(tableRowElement);
    }

    totalCellDataElement.appendChild(generatedDocument.createTextNode(webdeveloper_formatFileSize(calculatedFileSize)));

    totalFileSize += calculatedFileSize;

    calculatedFileSize = 0;
    cellDataElement    = generatedDocument.createElement("td");
    imageList          = webdeveloper_setImageFileSizes(webdeveloper_tidyImages(imageList)).sort(webdeveloper_sortImagesByFileSize);
    imageLength        = imageList.length;
    tableRowElement    = generatedDocument.createElement("tr");

    cellDataElement.setAttribute("class", "pivot");
    tableRowElement.appendChild(cellDataElement);

    cellDataElement = generatedDocument.createElement("td");

    // If there is one image
    if(imageLength == 1)
    {
        cellDataElement.appendChild(generatedDocument.createTextNode(stringBundle.getString("webdeveloper_images") + " (" + imageLength + " " + stringBundle.getString("webdeveloper_file").toLowerCase() + ")"));
    }
    else
    {
        cellDataElement.appendChild(generatedDocument.createTextNode(stringBundle.getString("webdeveloper_images") + " (" + imageLength + " " + stringBundle.getString("webdeveloper_files").toLowerCase() + ")"));
    }

    tableRowElement.appendChild(cellDataElement);

    totalCellDataElement = generatedDocument.createElement("td");

    totalCellDataElement.setAttribute("class", "size");
    tableRowElement.appendChild(totalCellDataElement);
    tableRowElement.setAttribute("class", "collapsed");
    tableElement.appendChild(tableRowElement);

    // Loop through the images
    for(i = 0; i < imageLength; i++)
    {
        cellDataElement = generatedDocument.createElement("td");
        image           = imageList[i];
        linkElement     = generatedDocument.createElement("a");
        src             = image.src;
        fileSize        = image.fileSize;
        tableRowElement = generatedDocument.createElement("tr");

        // If the file size is set
        if(fileSize)
        {
            calculatedFileSize += fileSize;
        }

        // If this is an even row
        if(i % 2 == 0)
        {
            tableRowElement.setAttribute("class", "hidden shaded");
        }
        else
        {
            tableRowElement.setAttribute("class", "hidden");
        }

        cellDataElement.setAttribute("class", "indent");
        tableRowElement.appendChild(cellDataElement);

        cellDataElement = generatedDocument.createElement("td");

        linkElement.setAttribute("href", src);
        linkElement.appendChild(generatedDocument.createTextNode(src));
        cellDataElement.appendChild(linkElement);
        tableRowElement.appendChild(cellDataElement);

        cellDataElement = generatedDocument.createElement("td");

        cellDataElement.setAttribute("class", "size");
        cellDataElement.appendChild(generatedDocument.createTextNode(webdeveloper_formatFileSize(fileSize)));
        tableRowElement.appendChild(cellDataElement);
        tableElement.appendChild(tableRowElement);
    }

    totalCellDataElement.appendChild(generatedDocument.createTextNode(webdeveloper_formatFileSize(calculatedFileSize)));

    totalFileSize += calculatedFileSize;

    calculatedFileSize = 0;
    cellDataElement    = generatedDocument.createElement("td");
    styleSheetList     = webdeveloper_setStyleSheetFileSizes(webdeveloper_tidyStyleSheets(styleSheetList)).sort(webdeveloper_sortStyleSheetsByFileSize);
    styleSheetLength   = styleSheetList.length;
    tableRowElement    = generatedDocument.createElement("tr");

    cellDataElement.setAttribute("class", "pivot");
    tableRowElement.appendChild(cellDataElement);

    cellDataElement = generatedDocument.createElement("td");

    // If there is one style sheet
    if(styleSheetLength == 1)
    {
        cellDataElement.appendChild(generatedDocument.createTextNode(stringBundle.getString("webdeveloper_styleSheets") + " (" + styleSheetLength + " " + stringBundle.getString("webdeveloper_file").toLowerCase() + ")"));
    }
    else
    {
        cellDataElement.appendChild(generatedDocument.createTextNode(stringBundle.getString("webdeveloper_styleSheets") + " (" + styleSheetLength + " " + stringBundle.getString("webdeveloper_files").toLowerCase() + ")"));
    }

    tableRowElement.appendChild(cellDataElement);

    totalCellDataElement = generatedDocument.createElement("td");

    totalCellDataElement.setAttribute("class", "size");
    tableRowElement.appendChild(totalCellDataElement);
    tableRowElement.setAttribute("class", "collapsed");
    tableElement.appendChild(tableRowElement);

    // Loop through the style sheets
    for(i = 0; i < styleSheetLength; i++)
    {
        cellDataElement = generatedDocument.createElement("td");
        linkElement     = generatedDocument.createElement("a");
        styleSheet      = styleSheetList[i];
        src             = styleSheet.href;
        fileSize        = styleSheet.fileSize;
        tableRowElement = generatedDocument.createElement("tr");

        // If the file size is set
        if(fileSize)
        {
            calculatedFileSize += fileSize;
        }

        // If this is an even row
        if(i % 2 == 0)
        {
            tableRowElement.setAttribute("class", "hidden shaded");
        }
        else
        {
            tableRowElement.setAttribute("class", "hidden");
        }

        cellDataElement.setAttribute("class", "indent");
        tableRowElement.appendChild(cellDataElement);

        cellDataElement = generatedDocument.createElement("td");

        linkElement.setAttribute("href", src);
        linkElement.appendChild(generatedDocument.createTextNode(src));
        cellDataElement.appendChild(linkElement);
        tableRowElement.appendChild(cellDataElement);

        cellDataElement = generatedDocument.createElement("td");

        cellDataElement.setAttribute("class", "size");
        cellDataElement.appendChild(generatedDocument.createTextNode(webdeveloper_formatFileSize(fileSize)));
        tableRowElement.appendChild(cellDataElement);
        tableElement.appendChild(tableRowElement);
    }

    totalCellDataElement.appendChild(generatedDocument.createTextNode(webdeveloper_formatFileSize(calculatedFileSize)));

    totalFileSize += calculatedFileSize;

    calculatedFileSize = 0;
    cellDataElement    = generatedDocument.createElement("td");
    scriptList         = webdeveloper_setScriptFileSizes(webdeveloper_tidyScripts(scriptList)).sort(webdeveloper_sortScriptsByFileSize);
    scriptLength       = scriptList.length;
    tableRowElement    = generatedDocument.createElement("tr");

    cellDataElement.setAttribute("class", "pivot");
    tableRowElement.appendChild(cellDataElement);

    cellDataElement = generatedDocument.createElement("td");

    // If there is one script
    if(scriptLength == 1)
    {
        cellDataElement.appendChild(generatedDocument.createTextNode(stringBundle.getString("webdeveloper_scripts") + " (" + scriptLength + " " + stringBundle.getString("webdeveloper_file").toLowerCase() + ")"));
    }
    else
    {
        cellDataElement.appendChild(generatedDocument.createTextNode(stringBundle.getString("webdeveloper_scripts") + " (" + scriptLength + " " + stringBundle.getString("webdeveloper_files").toLowerCase() + ")"));
    }

    tableRowElement.appendChild(cellDataElement);

    totalCellDataElement = generatedDocument.createElement("td");

    totalCellDataElement.setAttribute("class", "size");
    tableRowElement.appendChild(totalCellDataElement);
    tableRowElement.setAttribute("class", "collapsed");
    tableElement.appendChild(tableRowElement);

    // Loop through the scripts
    for(i = 0; i < scriptLength; i++)
    {
        cellDataElement = generatedDocument.createElement("td");
        linkElement     = generatedDocument.createElement("a");
        scriptElement   = scriptList[i];
        src             = scriptElement.src;
        fileSize        = scriptElement.fileSize;
        tableRowElement = generatedDocument.createElement("tr");

        // If the file size is set
        if(fileSize)
        {
            calculatedFileSize += fileSize;
        }

        // If this is an even row
        if(i % 2 == 0)
        {
            tableRowElement.setAttribute("class", "hidden shaded");
        }
        else
        {
            tableRowElement.setAttribute("class", "hidden");
        }

        cellDataElement.setAttribute("class", "indent");
        tableRowElement.appendChild(cellDataElement);

        cellDataElement = generatedDocument.createElement("td");

        linkElement.setAttribute("href", src);
        linkElement.appendChild(generatedDocument.createTextNode(src));
        cellDataElement.appendChild(linkElement);
        tableRowElement.appendChild(cellDataElement);

        cellDataElement = generatedDocument.createElement("td");

        cellDataElement.setAttribute("class", "size");
        cellDataElement.appendChild(generatedDocument.createTextNode(webdeveloper_formatFileSize(fileSize)));
        tableRowElement.appendChild(cellDataElement);
        tableElement.appendChild(tableRowElement);
    }

    totalCellDataElement.appendChild(generatedDocument.createTextNode(webdeveloper_formatFileSize(calculatedFileSize)));

    totalFileSize += calculatedFileSize;

    cellDataElement = generatedDocument.createElement("td");
    tableRowElement = generatedDocument.createElement("tr");

    cellDataElement.setAttribute("class", "total-indent");
    tableRowElement.setAttribute("class", "total");
    tableRowElement.appendChild(cellDataElement);

    cellDataElement = generatedDocument.createElement("td");

    cellDataElement.appendChild(generatedDocument.createTextNode(stringBundle.getString("webdeveloper_total")));
    tableRowElement.appendChild(cellDataElement);

    cellDataElement = generatedDocument.createElement("td");

    cellDataElement.setAttribute("class", "size");
    cellDataElement.appendChild(generatedDocument.createTextNode(webdeveloper_formatFileSize(totalFileSize)));
    tableRowElement.appendChild(cellDataElement);
    tableElement.appendChild(tableRowElement);
    generatedDocument.body.appendChild(tableElement);

    scriptElement = generatedDocument.createElement("script");

    scriptElement.setAttribute("defer", "defer");
    scriptElement.setAttribute("src", "chrome://webdeveloper/content/generated/table_pivot.js");
    scriptElement.setAttribute("type", "text/javascript");
    headElement.appendChild(scriptElement);

    // If the open tabs in background preference is set to true
    if(webdeveloper_getBooleanPreference("webdeveloper.open.tabs.background", true))
    {
        getBrowser().selectedTab = oldTab;
    }
}

// View JavaScript
function webdeveloper_viewJavaScript()
{
    var documentList      = webdeveloper_getDocuments(webdeveloper_getContentWindow());
    var documentLength    = documentList.length;
    var documentURL       = null;
    var linkElement       = null;
    var oldTab            = getBrowser().selectedTab;
    var oldURL            = getBrowser().currentURI.spec;
    var generatedDocument = webdeveloper_generateDocument("");
    var headerElement     = generatedDocument.createElement("h1");
    var pageDocument      = null;
    var preElement        = null;
    var scriptElement     = null;
    var scriptLength      = null;
    var scriptList        = new Array();
    var scriptSource      = null;
    var stringBundle      = document.getElementById("webdeveloper-string-bundle");
    var title             = stringBundle.getFormattedString("webdeveloper_viewJavaScriptTitle", [oldURL]);

    generatedDocument.title = title;

    webdeveloper_addGeneratedStyles(generatedDocument);

    headerElement.appendChild(generatedDocument.createTextNode(title));
    generatedDocument.body.appendChild(headerElement);

    // Loop through the documents
    for(var i = 0; i < documentLength; i++)
    {
        headerElement = generatedDocument.createElement("h2");
        linkElement   = generatedDocument.createElement("a");
        pageDocument  = documentList[i];
        documentURL   = pageDocument.documentURI;
        scriptList    = pageDocument.getElementsByTagName("script");
        scriptLength  = scriptList.length;

        linkElement.setAttribute("href", pageDocument.documentURI);
        linkElement.appendChild(generatedDocument.createTextNode(documentURL));
        headerElement.appendChild(linkElement);
        generatedDocument.body.appendChild(headerElement);

        // Loop through the scripts
        for(var j = 0; j < scriptLength; j++)
        {
            scriptElement = scriptList[j];
            scriptSource  = scriptElement.src;

            // If the script is external
            if(scriptSource)
            {
                // If this is a not chrome script
                if(scriptSource.indexOf("chrome://") != 0)
                {
                    headerElement = generatedDocument.createElement("h2");
                    linkElement   = generatedDocument.createElement("a");
                    preElement    = generatedDocument.createElement("pre");

                    linkElement.setAttribute("href", scriptSource);
                    linkElement.appendChild(generatedDocument.createTextNode(scriptSource));
                    headerElement.appendChild(linkElement);
                    generatedDocument.body.appendChild(headerElement);

                    preElement.appendChild(generatedDocument.createTextNode(webdeveloper_retrieveSource(scriptSource).replace(new RegExp("\r", "gi"), "\n")));
                    generatedDocument.body.appendChild(preElement);
                }
            }
            else
            {
                headerElement = generatedDocument.createElement("h2");
                preElement    = generatedDocument.createElement("pre");

                headerElement.appendChild(generatedDocument.createTextNode(stringBundle.getFormattedString("webdeveloper_inlineScript", [documentURL])));
                generatedDocument.body.appendChild(headerElement);

                preElement.appendChild(generatedDocument.createTextNode(scriptElement.innerHTML));
                generatedDocument.body.appendChild(preElement);
            }
        }
    }

    // If the open tabs in background preference is set to true
    if(webdeveloper_getBooleanPreference("webdeveloper.open.tabs.background", true))
    {
        getBrowser().selectedTab = oldTab;
    }
}

// Displays all the links for the page
function webdeveloper_viewLinkInformation()
{
    var documentList      = webdeveloper_getDocuments(webdeveloper_getContentWindow());
    var documentLength    = documentList.length;
    var documentURL       = null;
    var link              = null;
    var linkElement       = null;
    var linkHref          = null;
    var linkLength        = null;
    var links             = null;
    var listElement       = null;
    var listItemElement   = null;
    var oldTab            = getBrowser().selectedTab;
    var oldURL            = getBrowser().currentURI.spec;
    var generatedDocument = webdeveloper_generateDocument("");
    var headerElement     = generatedDocument.createElement("h1");
    var pageDocument      = null;
    var pElement          = null;
    var stringBundle      = document.getElementById("webdeveloper-string-bundle");
    var title             = stringBundle.getFormattedString("webdeveloper_viewLinkInformationTitle", [oldURL]);

    generatedDocument.title = title;

    webdeveloper_addGeneratedStyles(generatedDocument);

    headerElement.appendChild(generatedDocument.createTextNode(title));
    generatedDocument.body.appendChild(headerElement);

    // Loop through the documents
    for(var i = 0; i < documentLength; i++)
    {
        headerElement = generatedDocument.createElement("h2");
        linkElement   = generatedDocument.createElement("a");
        pageDocument  = documentList[i];
        documentURL   = pageDocument.documentURI;
        links         = pageDocument.getElementsByTagName("a");

        // If the tidy information preference is set
        if(webdeveloper_getBooleanPreference("webdeveloper.information.tidy", true))
        {
            links = webdeveloper_tidyLinks(links);
        }

        linkLength = links.length;

        linkElement.setAttribute("href", documentURL);
        linkElement.appendChild(generatedDocument.createTextNode(documentURL));
        headerElement.appendChild(linkElement);
        generatedDocument.body.appendChild(headerElement);

        // If there are no links
        if(linkLength == 0)
        {
            pElement = generatedDocument.createElement("p");

            pElement.appendChild(generatedDocument.createTextNode(stringBundle.getString("webdeveloper_noLinks")));
            generatedDocument.body.appendChild(pElement);
        }
        else
        {
            listElement = generatedDocument.createElement("ol");

            // Loop through the links
            for(var j = 0; j < linkLength; j++)
            {
                link     = links[j];
                linkHref = link.href;

                // If the link is set
                if(linkHref)
                {
                    linkElement     = generatedDocument.createElement("a");
                    listItemElement = generatedDocument.createElement("li");

                    linkElement.setAttribute("href", linkHref);
                    linkElement.appendChild(generatedDocument.createTextNode(linkHref));
                    listItemElement.appendChild(linkElement);
                    listElement.appendChild(listItemElement);
                }
            }

            generatedDocument.body.appendChild(listElement);
        }
    }

    // If the open tabs in background preference is set to true
    if(webdeveloper_getBooleanPreference("webdeveloper.open.tabs.background", true))
    {
        getBrowser().selectedTab = oldTab;
    }
}

// Displays all the meta tags for the page
function webdeveloper_viewMetaTagInformation()
{
    var cellDataElement   = null;
    var cellHeaderElement = null;
    var documentList      = webdeveloper_getDocuments(webdeveloper_getContentWindow());
    var documentLength    = documentList.length;
    var documentURL       = null;
    var linkElement       = null;
    var metaTag           = null;
    var metaTags          = null;
    var metaTagsLength    = null;
    var oldTab            = getBrowser().selectedTab;
    var oldURL            = getBrowser().currentURI.spec;
    var generatedDocument = webdeveloper_generateDocument("");
    var headElement       = generatedDocument.getElementsByTagName("head")[0];
    var headerElement     = generatedDocument.createElement("h1");
    var pageDocument      = null;
    var pElement          = null;
    var scriptElement     = generatedDocument.createElement("script");
    var stringBundle      = document.getElementById("webdeveloper-string-bundle");
    var tableElement      = null;
    var tableRowElement   = null;
    var title             = stringBundle.getFormattedString("webdeveloper_viewMetaTagInformationTitle", [oldURL]);

    generatedDocument.title = title;

    webdeveloper_addGeneratedStyles(generatedDocument);

    scriptElement.setAttribute("defer", "defer");
    scriptElement.setAttribute("src", "chrome://webdeveloper/content/common/dom.js");
    scriptElement.setAttribute("type", "text/javascript");
    headElement.appendChild(scriptElement);

    headerElement.appendChild(generatedDocument.createTextNode(title));
    generatedDocument.body.appendChild(headerElement);

    // Loop through the documents
    for(var i = 0; i < documentLength; i++)
    {
        headerElement  = generatedDocument.createElement("h2");
        linkElement    = generatedDocument.createElement("a");
        pageDocument   = documentList[i];
        documentURL    = pageDocument.documentURI;
        metaTags       = pageDocument.getElementsByTagName("meta");
        metaTagsLength = metaTags.length;

        linkElement.setAttribute("href", documentURL);
        linkElement.appendChild(generatedDocument.createTextNode(documentURL));
        headerElement.appendChild(linkElement);
        generatedDocument.body.appendChild(headerElement);

        // If there are no meta tags
        if(metaTagsLength == 0)
        {
            pElement = generatedDocument.createElement("p");

            pElement.appendChild(generatedDocument.createTextNode(stringBundle.getString("webdeveloper_noMetaTags")));
            generatedDocument.body.appendChild(pElement);
        }
        else
        {
            tableElement    = generatedDocument.createElement("table");
            tableRowElement = generatedDocument.createElement("tr");

            //  Name heading
            cellHeaderElement = generatedDocument.createElement("th");

            cellHeaderElement.appendChild(generatedDocument.createTextNode(stringBundle.getString("webdeveloper_name")));
            tableRowElement.appendChild(cellHeaderElement);

            //  Content heading
            cellHeaderElement = generatedDocument.createElement("th");

            cellHeaderElement.appendChild(generatedDocument.createTextNode(stringBundle.getString("webdeveloper_content")));
            tableRowElement.appendChild(cellHeaderElement);
            tableElement.appendChild(tableRowElement);

            // Loop through the meta tags
            for(var j = 0; j < metaTagsLength; j++)
            {
                metaTag         = metaTags[j];
                tableRowElement = generatedDocument.createElement("tr");

                // If this is an even row
                if(j % 2 != 0)
                {
                    tableRowElement.setAttribute("class", "shaded");
                }

                cellDataElement = generatedDocument.createElement("td");

                // If the meta tag has a name attribute
                if(metaTag.hasAttribute("name"))
                {
                    cellDataElement.appendChild(generatedDocument.createTextNode(metaTag.getAttribute("name")));
                }
                else if(metaTag.hasAttribute("http-equiv"))
                {
                    cellDataElement.appendChild(generatedDocument.createTextNode(metaTag.getAttribute("http-equiv")));
                }

                tableRowElement.appendChild(cellDataElement);

                //  Content
                cellDataElement = generatedDocument.createElement("td");

                cellDataElement.appendChild(generatedDocument.createTextNode(metaTag.getAttribute("content")));
                tableRowElement.appendChild(cellDataElement);
                tableElement.appendChild(tableRowElement);
            }

            tableElement.setAttribute("class", "sortable");
            generatedDocument.body.appendChild(tableElement);
        }
    }

    scriptElement = generatedDocument.createElement("script");

    scriptElement.setAttribute("defer", "defer");
    scriptElement.setAttribute("src", "chrome://webdeveloper/content/generated/table_sort.js");
    scriptElement.setAttribute("type", "text/javascript");
    headElement.appendChild(scriptElement);

    // If the open tabs in background preference is set to true
    if(webdeveloper_getBooleanPreference("webdeveloper.open.tabs.background", true))
    {
        getBrowser().selectedTab = oldTab;
    }
}

// View page information
function webdeveloper_viewPageInformation(chromeLocation)
{
    window.openDialog(chromeLocation, "webdeveloper-page-information-dialog", "chrome,dialog=no,resizable");
}
