/* valaccodefunction.vala
 *
 * Copyright (C) 2006-2007  Jürg Billeter
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.

 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.

 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301  USA
 *
 * Author:
 * 	Jürg Billeter <j@bitron.ch>
 */

#include <ccode/valaccodefunction.h>
#include <gee/arraylist.h>
#include <gee/list.h>
#include <gee/collection.h>
#include <ccode/valaccodelinedirective.h>




struct _ValaCCodeFunctionPrivate {
	char* _name;
	ValaCCodeModifiers _modifiers;
	char* _return_type;
	ValaCCodeBlock* _block;
	GeeList* parameters;
};

#define VALA_CCODE_FUNCTION_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), VALA_TYPE_CCODE_FUNCTION, ValaCCodeFunctionPrivate))
enum  {
	VALA_CCODE_FUNCTION_DUMMY_PROPERTY,
	VALA_CCODE_FUNCTION_NAME,
	VALA_CCODE_FUNCTION_MODIFIERS,
	VALA_CCODE_FUNCTION_RETURN_TYPE,
	VALA_CCODE_FUNCTION_BLOCK
};
static void vala_ccode_function_real_write (ValaCCodeNode* base, ValaCCodeWriter* writer);
static gpointer vala_ccode_function_parent_class = NULL;
static void vala_ccode_function_dispose (GObject * obj);



ValaCCodeFunction* vala_ccode_function_new (const char* name, const char* return_type) {
	ValaCCodeFunction * self;
	g_return_val_if_fail (name != NULL, NULL);
	g_return_val_if_fail (return_type != NULL, NULL);
	self = g_object_newv (VALA_TYPE_CCODE_FUNCTION, 0, NULL);
	vala_ccode_function_set_return_type (self, return_type);
	vala_ccode_function_set_name (self, name);
	return self;
}


/**
 * Appends the specified parameter to the list of function parameters.
 *
 * @param param a formal parameter
 */
void vala_ccode_function_add_parameter (ValaCCodeFunction* self, ValaCCodeFormalParameter* param) {
	g_return_if_fail (VALA_IS_CCODE_FUNCTION (self));
	g_return_if_fail (VALA_IS_CCODE_FORMAL_PARAMETER (param));
	gee_collection_add (GEE_COLLECTION (self->priv->parameters), param);
}


/**
 * Returns a copy of this function.
 *
 * @return copied function
 */
ValaCCodeFunction* vala_ccode_function_copy (ValaCCodeFunction* self) {
	ValaCCodeFunction* func;
	g_return_val_if_fail (VALA_IS_CCODE_FUNCTION (self), NULL);
	func = vala_ccode_function_new (self->priv->_name, self->priv->_return_type);
	vala_ccode_function_set_modifiers (func, self->priv->_modifiers);
	/* no deep copy for lists available yet
	 * func.parameters = parameters.copy ();
	 */
	{
		GeeList* param_collection;
		int param_it;
		param_collection = self->priv->parameters;
		for (param_it = 0; param_it < gee_collection_get_size (GEE_COLLECTION (param_collection)); param_it = param_it + 1) {
			ValaCCodeFormalParameter* param;
			param = ((ValaCCodeFormalParameter*) (gee_list_get (GEE_LIST (param_collection), param_it)));
			{
				gee_collection_add (GEE_COLLECTION (func->priv->parameters), param);
				(param == NULL ? NULL : (param = (g_object_unref (param), NULL)));
			}
		}
	}
	vala_ccode_function_set_block (func, self->priv->_block);
	return func;
}


static void vala_ccode_function_real_write (ValaCCodeNode* base, ValaCCodeWriter* writer) {
	ValaCCodeFunction * self;
	gboolean first;
	self = VALA_CCODE_FUNCTION (base);
	g_return_if_fail (VALA_IS_CCODE_WRITER (writer));
	vala_ccode_writer_write_indent (writer, vala_ccode_node_get_line (VALA_CCODE_NODE (self)));
	if (((self->priv->_modifiers) & (VALA_CCODE_MODIFIERS_STATIC)) == (VALA_CCODE_MODIFIERS_STATIC)) {
		vala_ccode_writer_write_string (writer, "static ");
	}
	if (((self->priv->_modifiers) & (VALA_CCODE_MODIFIERS_INLINE)) == (VALA_CCODE_MODIFIERS_INLINE)) {
		vala_ccode_writer_write_string (writer, "inline ");
	}
	vala_ccode_writer_write_string (writer, self->priv->_return_type);
	vala_ccode_writer_write_string (writer, " ");
	vala_ccode_writer_write_string (writer, self->priv->_name);
	vala_ccode_writer_write_string (writer, " (");
	first = TRUE;
	{
		GeeList* param_collection;
		int param_it;
		param_collection = self->priv->parameters;
		for (param_it = 0; param_it < gee_collection_get_size (GEE_COLLECTION (param_collection)); param_it = param_it + 1) {
			ValaCCodeFormalParameter* param;
			param = ((ValaCCodeFormalParameter*) (gee_list_get (GEE_LIST (param_collection), param_it)));
			{
				if (!first) {
					vala_ccode_writer_write_string (writer, ", ");
				} else {
					first = FALSE;
				}
				vala_ccode_node_write (VALA_CCODE_NODE (param), writer);
				(param == NULL ? NULL : (param = (g_object_unref (param), NULL)));
			}
		}
	}
	if (first) {
		vala_ccode_writer_write_string (writer, "void");
	}
	vala_ccode_writer_write_string (writer, ")");
	if (self->priv->_block == NULL) {
		vala_ccode_writer_write_string (writer, ";");
	} else {
		vala_ccode_node_write (VALA_CCODE_NODE (self->priv->_block), writer);
		vala_ccode_writer_write_newline (writer);
	}
	vala_ccode_writer_write_newline (writer);
}


const char* vala_ccode_function_get_name (ValaCCodeFunction* self) {
	g_return_val_if_fail (VALA_IS_CCODE_FUNCTION (self), NULL);
	return self->priv->_name;
}


void vala_ccode_function_set_name (ValaCCodeFunction* self, const char* value) {
	char* _tmp2;
	const char* _tmp1;
	g_return_if_fail (VALA_IS_CCODE_FUNCTION (self));
	_tmp2 = NULL;
	_tmp1 = NULL;
	self->priv->_name = (_tmp2 = (_tmp1 = value, (_tmp1 == NULL ? NULL : g_strdup (_tmp1))), (self->priv->_name = (g_free (self->priv->_name), NULL)), _tmp2);
}


ValaCCodeModifiers vala_ccode_function_get_modifiers (ValaCCodeFunction* self) {
	g_return_val_if_fail (VALA_IS_CCODE_FUNCTION (self), 0);
	return self->priv->_modifiers;
}


void vala_ccode_function_set_modifiers (ValaCCodeFunction* self, ValaCCodeModifiers value) {
	g_return_if_fail (VALA_IS_CCODE_FUNCTION (self));
	self->priv->_modifiers = value;
}


const char* vala_ccode_function_get_return_type (ValaCCodeFunction* self) {
	g_return_val_if_fail (VALA_IS_CCODE_FUNCTION (self), NULL);
	return self->priv->_return_type;
}


void vala_ccode_function_set_return_type (ValaCCodeFunction* self, const char* value) {
	char* _tmp2;
	const char* _tmp1;
	g_return_if_fail (VALA_IS_CCODE_FUNCTION (self));
	_tmp2 = NULL;
	_tmp1 = NULL;
	self->priv->_return_type = (_tmp2 = (_tmp1 = value, (_tmp1 == NULL ? NULL : g_strdup (_tmp1))), (self->priv->_return_type = (g_free (self->priv->_return_type), NULL)), _tmp2);
}


ValaCCodeBlock* vala_ccode_function_get_block (ValaCCodeFunction* self) {
	g_return_val_if_fail (VALA_IS_CCODE_FUNCTION (self), NULL);
	return self->priv->_block;
}


void vala_ccode_function_set_block (ValaCCodeFunction* self, ValaCCodeBlock* value) {
	ValaCCodeBlock* _tmp2;
	ValaCCodeBlock* _tmp1;
	g_return_if_fail (VALA_IS_CCODE_FUNCTION (self));
	_tmp2 = NULL;
	_tmp1 = NULL;
	self->priv->_block = (_tmp2 = (_tmp1 = value, (_tmp1 == NULL ? NULL : g_object_ref (_tmp1))), (self->priv->_block == NULL ? NULL : (self->priv->_block = (g_object_unref (self->priv->_block), NULL))), _tmp2);
}


static void vala_ccode_function_get_property (GObject * object, guint property_id, GValue * value, GParamSpec * pspec) {
	ValaCCodeFunction * self;
	self = VALA_CCODE_FUNCTION (object);
	switch (property_id) {
		case VALA_CCODE_FUNCTION_NAME:
		g_value_set_string (value, vala_ccode_function_get_name (self));
		break;
		case VALA_CCODE_FUNCTION_MODIFIERS:
		g_value_set_flags (value, vala_ccode_function_get_modifiers (self));
		break;
		case VALA_CCODE_FUNCTION_RETURN_TYPE:
		g_value_set_string (value, vala_ccode_function_get_return_type (self));
		break;
		case VALA_CCODE_FUNCTION_BLOCK:
		g_value_set_object (value, vala_ccode_function_get_block (self));
		break;
		default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property_id, pspec);
		break;
	}
}


static void vala_ccode_function_set_property (GObject * object, guint property_id, const GValue * value, GParamSpec * pspec) {
	ValaCCodeFunction * self;
	self = VALA_CCODE_FUNCTION (object);
	switch (property_id) {
		case VALA_CCODE_FUNCTION_NAME:
		vala_ccode_function_set_name (self, g_value_get_string (value));
		break;
		case VALA_CCODE_FUNCTION_MODIFIERS:
		vala_ccode_function_set_modifiers (self, g_value_get_flags (value));
		break;
		case VALA_CCODE_FUNCTION_RETURN_TYPE:
		vala_ccode_function_set_return_type (self, g_value_get_string (value));
		break;
		case VALA_CCODE_FUNCTION_BLOCK:
		vala_ccode_function_set_block (self, g_value_get_object (value));
		break;
		default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property_id, pspec);
		break;
	}
}


static void vala_ccode_function_class_init (ValaCCodeFunctionClass * klass) {
	vala_ccode_function_parent_class = g_type_class_peek_parent (klass);
	g_type_class_add_private (klass, sizeof (ValaCCodeFunctionPrivate));
	G_OBJECT_CLASS (klass)->get_property = vala_ccode_function_get_property;
	G_OBJECT_CLASS (klass)->set_property = vala_ccode_function_set_property;
	G_OBJECT_CLASS (klass)->dispose = vala_ccode_function_dispose;
	VALA_CCODE_NODE_CLASS (klass)->write = vala_ccode_function_real_write;
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_CCODE_FUNCTION_NAME, g_param_spec_string ("name", "name", "name", NULL, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_CCODE_FUNCTION_MODIFIERS, g_param_spec_flags ("modifiers", "modifiers", "modifiers", VALA_TYPE_CCODE_MODIFIERS, 0, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_CCODE_FUNCTION_RETURN_TYPE, g_param_spec_string ("return-type", "return-type", "return-type", NULL, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_CCODE_FUNCTION_BLOCK, g_param_spec_object ("block", "block", "block", VALA_TYPE_CCODE_BLOCK, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
}


static void vala_ccode_function_instance_init (ValaCCodeFunction * self) {
	self->priv = VALA_CCODE_FUNCTION_GET_PRIVATE (self);
	self->priv->parameters = GEE_LIST (gee_array_list_new (VALA_TYPE_CCODE_FORMAL_PARAMETER, ((GBoxedCopyFunc) (g_object_ref)), g_object_unref, g_direct_equal));
}


static void vala_ccode_function_dispose (GObject * obj) {
	ValaCCodeFunction * self;
	self = VALA_CCODE_FUNCTION (obj);
	self->priv->_name = (g_free (self->priv->_name), NULL);
	self->priv->_return_type = (g_free (self->priv->_return_type), NULL);
	(self->priv->_block == NULL ? NULL : (self->priv->_block = (g_object_unref (self->priv->_block), NULL)));
	(self->priv->parameters == NULL ? NULL : (self->priv->parameters = (g_object_unref (self->priv->parameters), NULL)));
	G_OBJECT_CLASS (vala_ccode_function_parent_class)->dispose (obj);
}


GType vala_ccode_function_get_type (void) {
	static GType vala_ccode_function_type_id = 0;
	if (G_UNLIKELY (vala_ccode_function_type_id == 0)) {
		static const GTypeInfo g_define_type_info = { sizeof (ValaCCodeFunctionClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) vala_ccode_function_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (ValaCCodeFunction), 0, (GInstanceInitFunc) vala_ccode_function_instance_init };
		vala_ccode_function_type_id = g_type_register_static (VALA_TYPE_CCODE_NODE, "ValaCCodeFunction", &g_define_type_info, 0);
	}
	return vala_ccode_function_type_id;
}




