/*   EXTRAITS DE LA LICENCE
	Copyright CEA, contributeurs : Luc BILLARD et Damien
	CALISTE, laboratoire L_Sim, (2001-2005)
  
	Adresse ml :
	BILLARD, non joignable par ml ;
	CALISTE, damien P caliste AT cea P fr.

	Ce logiciel est un programme informatique servant  visualiser des
	structures atomiques dans un rendu pseudo-3D. 

	Ce logiciel est rgi par la licence CeCILL soumise au droit franais et
	respectant les principes de diffusion des logiciels libres. Vous pouvez
	utiliser, modifier et/ou redistribuer ce programme sous les conditions
	de la licence CeCILL telle que diffuse par le CEA, le CNRS et l'INRIA 
	sur le site "http://www.cecill.info".

	Le fait que vous puissiez accder  cet en-tte signifie que vous avez 
	pris connaissance de la licence CeCILL, et que vous en avez accept les
	termes (cf. le fichier Documentation/licence.fr.txt fourni avec ce logiciel).
*/

/*   LICENCE SUM UP
	Copyright CEA, contributors : Luc BILLARD et Damien
	CALISTE, laboratoire L_Sim, (2001-2005)

	E-mail address:
	BILLARD, not reachable any more ;
	CALISTE, damien P caliste AT cea P fr.

	This software is a computer program whose purpose is to visualize atomic
	configurations in 3D.

	This software is governed by the CeCILL  license under French law and
	abiding by the rules of distribution of free software.  You can  use, 
	modify and/ or redistribute the software under the terms of the CeCILL
	license as circulated by CEA, CNRS and INRIA at the following URL
	"http://www.cecill.info". 

	The fact that you are presently reading this means that you have had
	knowledge of the CeCILL license and that you accept its terms. You can
	find a copy of this licence shipped with this software at Documentation/licence.en.txt.
*/
#include "visu_elements.h"
#include "visu_object.h"
#include "visu_configFile.h"

#include <stdlib.h>
#include <string.h>

#include "openGLFunctions/objectList.h"
#include "opengl.h"
#include "coreTools/toolColor.h"
#include "coreTools/toolConfigFile.h"

#include <GL/gl.h>
#include <GL/glu.h>

/**
 * SECTION:visu_elements
 * @short_description: defines methods to create and acccess to
 * #VisuElement.
 *
 * <para>V_Sim is used to rendered at given position several object of
 * the same kind. The #VisuElement object is used to control that
 * kind. Typically, it corresponds to chemical element. It can
 * represent the silicon, the iron...</para>
 *
 * <para>#VisuElement are defined by their name and have some
 * characteristic like their color or if they are rendered or not. The
 * color characteristic is defined by an RGBA array and different
 * value for the behavior of the light, as defined in OpenGL :
 * diffusivity, shiningness, emissivity, specular and ambient. These
 * values can be controlled with the following methods :
 * visu_element_setAllColorValues(), visu_element_setAllRGBValues(),
 * visu_element_setRGBValue(), visu_element_setAllMaterialValues() and
 * visu_element_setMaterialValue().</para>
 *
 * <para>If the OpenGL representation of one element is not dependent
 * of its position, it is recommended to use the OpenGL list
 * associated to each #VisuElement that can be accessed by a call to
 * visu_element_getIdentifierMaterial().</para>
 */

enum {
  ELEMENT_NEW_SIGNAL,
  ELEMENT_RENDERED_CHANGED_SIGNAL,
  ELEMENT_MATERIAL_CHANGED_SIGNAL,
  ELEMENT_PLANE_CHANGED_SIGNAL,
  LAST_SIGNAL
};

struct _VisuElementClass
{
  GObjectClass parent;

  int identifierMaterials;
  guint idEle;

  /* This hashtable contains a list of all different
     elements loaded in visu. ntype is _not_ the number of items
     in this hashtable. */
  GHashTable *allElements_table;
  GList *allElements_list;

  /* This int stores how many user functions have needs
     to recreate the nodes of an element when its material
     has been changed. */
  int flagCreateNodesAfterMaterialChange;
};

#define FLAG_ELEMENT_COLOR      "element_color"
#define DESC_ELEMENT_COLOR      "Codes the main color in RedGreenBlueAlpha format" \
  "and the light effects on material, nine floats between 0. and 1."
#define FLAG_ELEMENT_PROPERTIES "element_properties"
#define DESC_ELEMENT_PROPERTIES "Define some properties ; rendered (0 or 1) masked" \
  "(0 or 1)."

/* These functions detail how to read the RGB resource and
   the material resource introduced by this part.
   These routines are obsolete but kept for backward compatibility. */
static gboolean readMaterial(gchar **lines, int nbLines, int position,
			     VisuData *dataObj, GError **error);
static gboolean readRendered(gchar **lines, int nbLines, int position,
			     VisuData *dataObj, GError **error);

/* Read routines for the config file. */
static gboolean readElementColor(gchar **lines, int nbLines, int position,
				 VisuData *dataObj, GError **error);
static gboolean readElementProperties(gchar **lines, int nbLines, int position,
				      VisuData *dataObj, GError **error);


/* These functions write all the element list to export there associated resources. */
static void exportResourcesRenderingBase(GString *data, VisuData *dataObj);
static void visu_element_dispose(GObject* obj);
static void visu_element_finalize(GObject* obj);

static VisuElementClass *my_class = (VisuElementClass*)0;
static guint visu_element_signals[LAST_SIGNAL] = { 0 };

G_DEFINE_TYPE(VisuElement, visu_element, G_TYPE_OBJECT)

static void visu_element_class_init(VisuElementClass *klass)
{
  VisuConfigFileEntry *resourceEntry, *oldEntry;

  DBG_fprintf(stderr, "Visu Element: creating the class of the object.\n");
  DBG_fprintf(stderr, "                - adding new signals ;\n");
  /**
   * VisuElement::ElementRenderedChanged:
   * @element: the object which emit the signal ;
   *
   * Gets emitted when one of the rendered #VisuElement is made
   * visible or hidden.
   *
   * Since: 3.6
   */
  visu_element_signals[ELEMENT_RENDERED_CHANGED_SIGNAL] =
    g_signal_new("ElementRenderedChanged",
                 G_TYPE_FROM_CLASS(klass),
                 G_SIGNAL_RUN_LAST | G_SIGNAL_NO_RECURSE,
                 0, NULL, NULL, g_cclosure_marshal_VOID__VOID,
                 G_TYPE_NONE, 0);

  /**
   * VisuElement::ElementMaterialChanged:
   * @element: the object which emit the signal ;
   *
   * Gets emitted when the material or color properties of a VisuElement
   * are modified.
   *
   * Since: 3.6
   */
  visu_element_signals[ELEMENT_MATERIAL_CHANGED_SIGNAL] =
    g_signal_new("ElementMaterialChanged",
                 G_TYPE_FROM_CLASS(klass),
                 G_SIGNAL_RUN_LAST | G_SIGNAL_NO_RECURSE,
                 0, NULL, NULL, g_cclosure_marshal_VOID__VOID,
                 G_TYPE_NONE, 0);

  /**
   * VisuElement::ElementPlaneChanged:
   * @element: the object which emit the signal ;
   *
   * Gets emitted when the property of masking by plane is modified.
   *
   * Since: 3.6
   */
  visu_element_signals[ELEMENT_PLANE_CHANGED_SIGNAL] =
    g_signal_new("ElementPlaneChanged",
                 G_TYPE_FROM_CLASS(klass),
                 G_SIGNAL_RUN_LAST | G_SIGNAL_NO_RECURSE,
                 0, NULL, NULL, g_cclosure_marshal_VOID__VOID,
                 G_TYPE_NONE, 0);

  /**
   * VisuElement::ElementNew:
   * @element: the object emitting the signal.
   *
   * A new element is available.
   *
   * Since: 3.6
   */
  visu_element_signals[ELEMENT_NEW_SIGNAL] = 
    g_signal_new("ElementNew",
                 G_TYPE_FROM_CLASS (klass),
                 G_SIGNAL_RUN_LAST | G_SIGNAL_NO_RECURSE,
                 0, NULL, NULL, g_cclosure_marshal_VOID__VOID,
                 G_TYPE_NONE, 0);


  /* Connect the overloading methods. */
  G_OBJECT_CLASS(klass)->dispose  = visu_element_dispose;
  G_OBJECT_CLASS(klass)->finalize = visu_element_finalize;

  /* Set internal parameters. */
  klass->allElements_table =
    g_hash_table_new_full(g_str_hash, g_str_equal,
			  NULL, (GDestroyNotify)g_object_unref);
  klass->allElements_list = (GList*)0;

  /* Create a VisuModule to registered the new resources as
     rgb and material. */
  oldEntry = visu_configFile_addEntry(VISU_CONFIGFILE_RESOURCE,
				     "material",
				     "Obsolete entry for element_color",
				     1, readMaterial);
  resourceEntry = visu_configFile_addEntry(VISU_CONFIGFILE_RESOURCE,
					  FLAG_ELEMENT_COLOR,
					  DESC_ELEMENT_COLOR,
					  1, readElementColor);
  visu_configFile_entry_setVersion(resourceEntry, 3.4f);
  visu_configFile_entry_setReplace(resourceEntry, oldEntry);
  oldEntry = visu_configFile_addEntry(VISU_CONFIGFILE_RESOURCE,
				     "element_is_rendered",
				     "Obsolete entry included in element_properties",
				     1, readRendered);
  visu_configFile_entry_setVersion(resourceEntry, 3.1f);
  resourceEntry = visu_configFile_addEntry(VISU_CONFIGFILE_RESOURCE,
					  FLAG_ELEMENT_PROPERTIES,
					  DESC_ELEMENT_PROPERTIES,
					  1, readElementProperties);
  visu_configFile_entry_setVersion(resourceEntry, 3.4f);
  visu_configFile_entry_setReplace(resourceEntry, oldEntry);
  visu_configFile_addExportFunction(VISU_CONFIGFILE_RESOURCE,
				   exportResourcesRenderingBase);


  /* Get an OpenGL identifier to store all the materials. */
  klass->identifierMaterials = visu_openGL_objectList_new(VISU_ELEMENT_MAX_NUMBER);
  klass->flagCreateNodesAfterMaterialChange = 0;
  klass->idEle = 0;

  my_class = klass;
}

static void visu_element_init(VisuElement *ele)
{
  int i;

  DBG_fprintf(stderr, "Visu Element: initializing a new object (%p).\n",
	      (gpointer)ele);
  ele->materialIsUpToDate       = FALSE;
  ele->rendered                 = TRUE;
  ele->sensitiveToMaskingPlanes = TRUE;

  for (i = 0; i < 4; i++)
    ele->rgb[i] = 1.;
  for (i = 0; i < 5; i++)
    ele->material[i] = 0.25;
}

/* This method can be called several times.
   It should unref all of its reference to
   GObjects. */
static void visu_element_dispose(GObject* obj)
{
  DBG_fprintf(stderr, "Visu Element: dispose object %p.\n", (gpointer)obj);

  if (VISU_ELEMENT(obj)->dispose_has_run)
    return;
  VISU_ELEMENT(obj)->dispose_has_run = TRUE;

  /* Chain up to the parent class */
  G_OBJECT_CLASS(visu_element_parent_class)->dispose(obj);
}
/* This method is called once only. */
static void visu_element_finalize(GObject* obj)
{
  VisuElement *ele;
  VisuElementClass *klass;

  g_return_if_fail(obj);
  DBG_fprintf(stderr, "Visu Element: finalize object %p.\n", (gpointer)obj);

  ele = VISU_ELEMENT(obj);
  klass = VISU_ELEMENT_GET_CLASS(ele);
  g_free(ele->name);
  g_hash_table_steal(klass->allElements_table, ele);
  klass->allElements_list = g_list_remove(klass->allElements_list, ele);

  /* Chain up to the parent class */
  DBG_fprintf(stderr, "Visu Element: chain to parent.\n");
  G_OBJECT_CLASS(visu_element_parent_class)->finalize(obj);
  DBG_fprintf(stderr, "Visu Element: freeing ... OK.\n");
}

/**
 * visu_element_new:
 * @key: the name of the new element to create.
 *
 * Allocate a new visuElement with the specified name. Remember
 * that names must be unique since they identify the element.
 *
 * Returns: (transfer none): the newly created VisuElement or 0 if something goes
 * wrong in the process (if the name already exist for example).
 */
VisuElement *visu_element_new(const char *key)
{
  VisuElement *ele;

  if (!my_class)
    g_type_class_ref(VISU_ELEMENT_TYPE);

  ele = g_hash_table_lookup(my_class->allElements_table, (gpointer)key);
  if (ele)
    {
      g_warning("Element '%s' already exists.", key);
      return ele;
    }

  ele = VISU_ELEMENT(g_object_new(VISU_ELEMENT_TYPE, NULL));
  my_class->idEle += 1;
  if (my_class->idEle == VISU_ELEMENT_MAX_NUMBER)
    {
      g_error("The system requires more than %d elements. This value is"
              " set at compilation time with the variable VISU_ELEMENT_MAX_NUMBER. If you"
              " need more, ask the maintainer to raise this value.\n", VISU_ELEMENT_MAX_NUMBER);
    }
  ele->name       = g_strdup(key);
  ele->typeNumber = my_class->idEle;
  g_hash_table_insert(my_class->allElements_table,
                      (gpointer)ele->name, (gpointer)ele);
  my_class->allElements_list = g_list_append(my_class->allElements_list, (gpointer)ele);


  DBG_fprintf(stderr, "Visu Elements: create a new VisuElement '%s' -> %p.\n",
	      key, (gpointer)ele);  
  g_signal_emit(G_OBJECT(ele), visu_element_signals[ELEMENT_NEW_SIGNAL], 0, NULL);
  DBG_fprintf(stderr, "Visu Elements: new element signal OK.\n");

  return ele;
}

/**
 * visu_element_getAllElements:
 *
 * This method returns a list of all the registered #VisuElement.
 * The returned list is read-only.
 *
 * Returns: (element-type VisuElement) (transfer none): the list of
 * all known #VisuElement.
 */
const GList *visu_element_getAllElements(void)
{
  if (!my_class)
    g_type_class_ref(VISU_ELEMENT_TYPE);

  return my_class->allElements_list;
}

/**
 * visu_element_retrieveFromName:
 * @name: a string that identify the #VisuElement (in UTF8) ;
 * @nw: (out caller-allocates): a location to store a boolean.
 *
 * Try to find a #VisuElement already associated to that @name or
 * create a new one if none has been found. If @nw is not NULL it is
 * set to FALSE if @name was found.
 *
 * Returns: (transfer none): a #VisuElement associated to this @name.
 */
VisuElement *visu_element_retrieveFromName(const gchar *name, gboolean *nw)
{
  VisuElement *ele;

  DBG_fprintf(stderr, "Visu Element: retrieve '%s'.\n", name);

  if (!my_class)
    g_type_class_ref(VISU_ELEMENT_TYPE);

  if (nw)
    *nw = FALSE;

  ele = g_hash_table_lookup(my_class->allElements_table, (gpointer)name);
  if (ele)
    return ele;

  if (nw)
    *nw = TRUE;

  return visu_element_new(name);
}

/**
 * visu_element_lookup:
 * @name: a string.
 *
 * Lookup for element @name in the base. Do not create it if not
 * found. To do this, use visu_element_retrieveFromName().
 *
 * Since: 3.6
 *
 * Returns: the found #VisuElement or NULL.
 */
VisuElement *visu_element_lookup(const gchar *name)
{
  if (!my_class)
    g_type_class_ref(VISU_ELEMENT_TYPE);

  return g_hash_table_lookup(my_class->allElements_table, (gpointer)name);
}


/**
 * visu_element_setAllColorValues:
 * @ele: the element of which the color must be changed ;
 * @rgb:  (in) (array fixed-size=4): the new color given by a {red, green, blue} array ;
 * @material:  (in) (array fixed-size=5): the new values to define the lighting.
 *
 * This method is used to set all the values that define the color
 * and the lighting of the given element.
 *
 * Returns: > 0 if visu_data_createNodes() should be called and then
 *          redraw is need if the value is not null.
 */
gint visu_element_setAllColorValues(VisuElement* ele, float rgb[4], float material[5])
{
  int chgt;

  chgt = (ele->rgb[0] != rgb[0]) || (ele->rgb[1] != rgb[1]) ||
    (ele->rgb[2] != rgb[2]) || (ele->rgb[3] != rgb[3]);
  ele->rgb[0] = rgb[0];
  ele->rgb[1] = rgb[1];
  ele->rgb[2] = rgb[2];
  ele->rgb[3] = rgb[3];
  chgt = chgt || (ele->material[0] != material[0]) || (ele->material[1] != material[1]) ||
    (ele->material[2] != material[2]) || (ele->material[3] != material[3]) ||
    (ele->material[4] != material[4]);
  ele->material[0] = material[0];
  ele->material[1] = material[1];
  ele->material[2] = material[2];
  ele->material[3] = material[3];
  ele->material[4] = material[4];

  if (chgt)
    {
      visu_element_createMaterial(ele);
      DBG_fprintf(stderr, "Visu Element: %p(%s) emit the 'ElementMaterialChanged'.\n",
                  (gpointer)ele, ele->name);
      g_signal_emit(ele, visu_element_signals[ELEMENT_MATERIAL_CHANGED_SIGNAL],
                    0, NULL);
    }

  if (chgt && ele->rendered)
    return 2 * VISU_ELEMENT_GET_CLASS(ele)->flagCreateNodesAfterMaterialChange - 1;
  else
    return 0;
}

/**
 * visu_element_setAllRGBValues:
 * @ele: the element of which the color must be changed,
 * @rgb:  (in) (array fixed-size=4): the new color given by a {red, green, blue} array.
 *
 * It saves the values of rgb in the specified VisuElement.
 *
 * Returns: > 0 if visu_data_createNodes() should be called and then
 *          redraw is need if the value is not null.
 */
gint visu_element_setAllRGBValues(VisuElement* ele, float rgb[4])
{
  g_return_val_if_fail(ele, 0);

  DBG_fprintf(stderr, "Visu Element: '%s' set color (%f, %f, %f, %f)\n",
	      ele->name, ele->rgb[0], ele->rgb[1], ele->rgb[2], ele->rgb[3]);

  if (ele->rgb[0] == rgb[0] &&
      ele->rgb[1] == rgb[1] &&
      ele->rgb[2] == rgb[2] &&
      ele->rgb[3] == rgb[3])
    return 0;

  DBG_fprintf(stderr, "                      -> (%f, %f, %f, %f).\n",
	      rgb[0], rgb[1], rgb[2], rgb[3]);

  ele->rgb[0] = rgb[0];
  ele->rgb[1] = rgb[1];
  ele->rgb[2] = rgb[2];
  ele->rgb[3] = rgb[3];

  visu_element_createMaterial(ele);

  DBG_fprintf(stderr, "Visu Element: %p(%s) emit the 'ElementMaterialChanged'.\n",
              (gpointer)ele, ele->name);
  g_signal_emit(ele, visu_element_signals[ELEMENT_MATERIAL_CHANGED_SIGNAL],
                0, NULL);

  if (ele->rendered)
    return 2 * VISU_ELEMENT_GET_CLASS(ele)->flagCreateNodesAfterMaterialChange - 1;
  else
    return 0;
}

/**
 * visu_element_setRGBValue:
 * @ele: the element of which the color must be changed,
 * @rgb: the component to change, 0 for red, 1 for green and 2
 * for blue,
 * @value: the value for one of the red, green or blue component.
 *
 * It saves the specific value of rgb (0 for red, 1 for green and 2
 * for bluein the specified VisuElement.
 *
 * Returns: > 0 if visu_data_createNodes() should be called and then
 *          redraw is need if the value is not null.
 */
gint visu_element_setRGBValue(VisuElement* ele, int rgb, float value)
{
  if (rgb < 0 || rgb > 3)
    return 0;

  if (value == ele->rgb[rgb])
    return 0;

  ele->rgb[rgb] = value;
  visu_element_createMaterial(ele);

  DBG_fprintf(stderr, "Visu Element: %p(%s) emit the 'ElementMaterialChanged'.\n",
              (gpointer)ele, ele->name);
  g_signal_emit(ele, visu_element_signals[ELEMENT_MATERIAL_CHANGED_SIGNAL],
                0, NULL);

  if (ele->rendered)
    return 2 * VISU_ELEMENT_GET_CLASS(ele)->flagCreateNodesAfterMaterialChange - 1;
  else
    return 0;
}

/**
 * visu_element_setAllMaterialValues:
 * @ele: the element of which the color must be changed,
 * @material:  (in) (array fixed-size=5): the new values to define the lighting.
 *
 * It saves the values of material in the specified VisuElement.
 *
 * Returns: > 0 if visu_data_createNodes() should be called and then
 *          redraw is need if the value is not null.
 */
gint visu_element_setAllMaterialValues(VisuElement* ele, float material[5])
{
  if (!ele)
    return 0;

  ele->material[0] = material[0];
  ele->material[1] = material[1];
  ele->material[2] = material[2];
  ele->material[3] = material[3];
  ele->material[4] = material[4];

  visu_element_createMaterial(ele);

  DBG_fprintf(stderr, "Visu Element: %p(%s) emit the 'ElementMaterialChanged'.\n",
              (gpointer)ele, ele->name);
  g_signal_emit(ele, visu_element_signals[ELEMENT_MATERIAL_CHANGED_SIGNAL],
                0, NULL);

  if (ele->rendered)
    return 2 * VISU_ELEMENT_GET_CLASS(ele)->flagCreateNodesAfterMaterialChange - 1;
  else
    return 0;
}

/**
 * visu_element_setMaterialValue:
 * @ele: the element of which the color must be changed,
 * @material: the component to change,
 * @value: the new value of the specified component.
 *
 * It saves the specific value of material (use the enum as the parameter
 * material) in the specified VisuElement.
 *
 * Returns: > 0 if visu_data_createNodes() should be called and then
 *          redraw is need if the value is not null.
 */
gboolean visu_element_setMaterialValue(VisuElement* ele, int material, float value)
{
  if (material < 0 || material > 4)
    return 0;

  if (value == ele->material[material])
    return 0;

  ele->material[material] = value;
  visu_element_createMaterial(ele);

  DBG_fprintf(stderr, "Visu Element: %p(%s) emit the 'ElementMaterialChanged'.\n",
              (gpointer)ele, ele->name);
  g_signal_emit(ele, visu_element_signals[ELEMENT_MATERIAL_CHANGED_SIGNAL],
                0, NULL);

  if (ele->rendered)
    return 2 * VISU_ELEMENT_GET_CLASS(ele)->flagCreateNodesAfterMaterialChange - 1;
  else
    return 0;
}

/**
 * visu_element_setSensitiveToPlanes:
 * @element: a #VisuElement object ;
 * @status: TRUE or FALSE.
 *
 * This method sets the private attribute 'sensitiveToMaskingPlanes' to TRUE or FALSE
 * for the specified #visuElement. If TRUE, all the nodes of that #VisuElement
 * are not sensitive to the masking property of planes.
 *
 * Returns: 1 if the calling method should call visu_data_createNodes() for the given
 *          #VisuElement and then should add visu_object_redraw() to the Gloop.
 */
gboolean visu_element_setSensitiveToPlanes(VisuElement *element, gboolean status)
{
  g_return_val_if_fail(element, FALSE);

  DBG_fprintf(stderr, "Visu Element: '%s' set sensitiveToMaskingPlanes %d\n",
	      element->name, element->sensitiveToMaskingPlanes);

  if (element->sensitiveToMaskingPlanes == status)
    return FALSE;

  DBG_fprintf(stderr, " -> %d\n", status);

  DBG_fprintf(stderr, "Visu Element: %p(%s) emit the 'ElementPlaneChanged'.\n",
              (gpointer)element, element->name);
  g_signal_emit(element, visu_element_signals[ELEMENT_PLANE_CHANGED_SIGNAL],
                0, NULL);

  element->sensitiveToMaskingPlanes = status;
  return TRUE;
}

/**
 * visu_element_getSensitiveToPlanes:
 * @element: a #VisuElement object.
 *
 * This method is used to retrieve if nodes of the element are sensitive to
 * the masking property of planes, or not.
 *
 * Returns: TRUE if nodes are masked by planes.
 */
gboolean visu_element_getSensitiveToPlanes(VisuElement *element)
{
  g_return_val_if_fail(element, TRUE);
  
  return element->sensitiveToMaskingPlanes;
}

/**
 * visu_element_createMaterial:
 * @ele: a pointer to a valid %VisuElement.
 *
 * Create a list whose number is defined by #identifierMaterials + @ele->typeNumber
 * that stores the definition of light and color for this #VisuElement. The previous
 * OpenGL object list with the same identifier is deleted.
 */
void visu_element_createMaterial(VisuElement *ele)
{
  g_return_if_fail(ele);

  DBG_fprintf(stderr, "Visu Element: create material for %s (ele id %d - OpenGL id %d)\n",
              ele->name, ele->typeNumber,
              VISU_ELEMENT_GET_CLASS(ele)->identifierMaterials + ele->typeNumber);
/*   fprintf(stderr, "%f %f %f - %f %f %f %f %f\n", ele->rgb[0], ele->rgb[1], ele->rgb[2], ele->material[0], ele->material[1], ele->material[2], ele->material[3], ele->material[4]); */

  glDeleteLists(VISU_ELEMENT_GET_CLASS(ele)->identifierMaterials + ele->typeNumber, 1);
  glNewList(VISU_ELEMENT_GET_CLASS(ele)->identifierMaterials + ele->typeNumber, GL_COMPILE);
  glColor4fv(ele->rgb);
  openGLSet_color(ele->material, ele->rgb);
  glEndList();
  ele->materialIsUpToDate = TRUE;
}

/**
 * visu_element_getIdentifierMaterial:
 * @ele: an element.
 *
 * This method is useful for the rendering method to get the
 * OpenGl identifier of the material of the specified element.
 *
 * Returns: the OpenGL identifier of the specified VisuElement.
 */
int visu_element_getIdentifierMaterial(VisuElement *ele)
{
  if (!ele)
    return 0;
/*   if (!ele->materialIsUpToDate) */
/*     visu_element_createMaterial(ele); */

  return VISU_ELEMENT_GET_CLASS(ele)->identifierMaterials + ele->typeNumber;
}

/**
 * visu_element_setRendered:
 * @element: a #VisuElement object ;
 * @rendered: TRUE or FALSE.
 *
 * This method sets the private attribute 'rendered' to TRUE or FALSE
 * for the specified #visuElement. If FALSE, all the nodes of that #VisuElement
 * are not included in the nodes OpenGL list.
 *
 * Returns: 1 if the calling method should call visu_data_createNodes() for the given
 *          #VisuElement and then should emit the 'OpenGLAskForReDraw' signal.
 */

gboolean visu_element_setRendered(VisuElement *element, gboolean rendered)
{
  g_return_val_if_fail(element, FALSE);

  if (element->rendered == rendered)
    return FALSE;

  element->rendered = rendered;
  DBG_fprintf(stderr, "Visu Element: %p(%s) emit the 'ElementRenderedChanged'.\n",
              (gpointer)element, element->name);
  g_signal_emit(element, visu_element_signals[ELEMENT_RENDERED_CHANGED_SIGNAL],
		0, NULL);
  return TRUE;
}

/**
 * visu_element_getRendered:
 * @element: a #VisuElement object.
 *
 * This gets the value of the private attribute 'rendered' for
 * the specified #VisuElement.
 *
 * Returns: the value of attribute 'rendered'.
 */
gboolean visu_element_getRendered(VisuElement *element)
{
  g_return_val_if_fail(element, FALSE);

  return element->rendered;
}

/**
 * visu_element_setUpdateNodesOnMaterialChange:
 *
 * If this method is called, whenever a change occurs to a #VisuElement material
 * description (color and light) the corresponding OpenGl list of nodes of the
 * currentVisuData is rebuilt. This is usefull if the nodes can't use the list material
 * because the color is node dependant for example. Use
 * visu_element_unsetUpdateNodesOnMaterialChange() to return to the normal behavior.
 */
void visu_element_setUpdateNodesOnMaterialChange(void)
{
  if (!my_class)
    g_type_class_ref(VISU_ELEMENT_TYPE);

  my_class->flagCreateNodesAfterMaterialChange += 1;
  DBG_fprintf(stderr, "Visu Element : set flagCreateNodesAfterMaterialChange to %d.\n",
	      my_class->flagCreateNodesAfterMaterialChange);
}

/**
 * visu_element_unsetUpdateNodesOnMaterialChange:
 *
 * If this method is called (and no other method has used
 * visu_element_setUpdateNodesOnMaterialChange(), a changement
 * in the material description only change the list of the material.
 */
void visu_element_unsetUpdateNodesOnMaterialChange(void)
{
  if (!my_class)
    g_type_class_ref(VISU_ELEMENT_TYPE);

  my_class->flagCreateNodesAfterMaterialChange -= 1;
  if (my_class->flagCreateNodesAfterMaterialChange < 0)
    my_class->flagCreateNodesAfterMaterialChange = 0;
  DBG_fprintf(stderr, "Visu Element : set flagCreateNodesAfterMaterialChange to %d.\n",
	      my_class->flagCreateNodesAfterMaterialChange);
}

/**
 * visu_element_getUpdateNodesOnMaterialChange:
 *
 * Retrieve if one the #VisuElement is sensitive to the material
 * values. See visu_element_setUpdateNodesOnMaterialChange().
 *
 * Returns: TRUE if visu_data_createNodes() should be called when the
 * material values are changed.
 */
gboolean visu_element_getUpdateNodesOnMaterialChange(void)
{
  if (!my_class)
    g_type_class_ref(VISU_ELEMENT_TYPE);

  return (my_class->flagCreateNodesAfterMaterialChange > 0);
}

/******************/
/* Resources part */
/******************/
static gboolean readElementColor(gchar **lines, int nbLines, int position,
				 VisuData *dataObj _U_, GError **error)
{
  VisuElement* ele;
  float all[9];

  g_return_val_if_fail(nbLines == 1, FALSE);

  if (!tool_configFile_readFloatWithElement(lines[0], position, all, 9, &ele, error))
    return FALSE;
  ele->rgb[0] = CLAMP(all[0], 0., 1.);
  ele->rgb[1] = CLAMP(all[1], 0., 1.);
  ele->rgb[2] = CLAMP(all[2], 0., 1.);
  ele->rgb[3] = CLAMP(all[3], 0., 1.);
  ele->material[0] = CLAMP(all[4], 0., 1.);
  ele->material[1] = CLAMP(all[5], 0., 1.);
  ele->material[2] = CLAMP(all[6], 0., 1.);
  ele->material[3] = CLAMP(all[7], 0., 1.);
  ele->material[4] = CLAMP(all[8], 0., 1.);
  tool_color_addFloatRGBA(ele->rgb, (int*)0);
  
  return TRUE;
}
static gboolean readElementProperties(gchar **lines, int nbLines, int position,
				 VisuData *dataObj _U_, GError **error)
{
  VisuElement* ele;
  float values[2];

  g_return_val_if_fail(nbLines == 1, FALSE);

  if (!tool_configFile_readFloatWithElement(lines[0], position, values, 2, &ele, error))
    return FALSE;
  visu_element_setRendered(ele, (gboolean)values[0]);
  visu_element_setSensitiveToPlanes(ele, (gboolean)values[1]);
  
  return TRUE;
}

static gboolean readMaterial(gchar **lines, int nbLines, int position,
			     VisuData *dataObj, GError **error)
{
  return readElementColor(lines, nbLines, position, dataObj, error);
}
static gboolean readRendered(gchar **lines, int nbLines, int position,
			     VisuData *dataObj _U_, GError **error)
{
  VisuElement* ele;
  float value;

  g_return_val_if_fail(nbLines == 1, FALSE);

  if (!tool_configFile_readFloatWithElement(lines[0], position, &value, 1, &ele, error))
    return FALSE;
  visu_element_setRendered(ele, (gboolean)value);
  
  return TRUE;
}

/* These functions read all the element list to export there associated resources. */
static void exportResourcesRenderingBase(GString *data, VisuData *dataObj)
{
  GList *pos, *eleList;
  unsigned int i;

  g_string_append_printf(data, "# %s\n", DESC_ELEMENT_COLOR);
  /* We create a list of elements, or get the whole list. */
  eleList = (GList*)0;
  if (dataObj)
    {
      for (i = 0; i < dataObj->ntype; i++)
	eleList = g_list_prepend(eleList, (gpointer)dataObj->fromIntToVisuElement[i]);
      pos = eleList;
    }
  else
    pos = (GList*)visu_element_getAllElements();
  while(pos)
    {
      g_string_append_printf(data, "%s:\n", FLAG_ELEMENT_COLOR);
      g_string_append_printf(data, "    %s %4.3f %4.3f %4.3f %4.3f"
			     "   %4.2f %4.2f %4.2f %4.2f %4.2f\n",
			     ((VisuElement*)pos->data)->name,
			     ((VisuElement*)pos->data)->rgb[0],
			     ((VisuElement*)pos->data)->rgb[1],
			     ((VisuElement*)pos->data)->rgb[2],
			     ((VisuElement*)pos->data)->rgb[3],
			     ((VisuElement*)pos->data)->material[0],
			     ((VisuElement*)pos->data)->material[1],
			     ((VisuElement*)pos->data)->material[2],
			     ((VisuElement*)pos->data)->material[3],
			     ((VisuElement*)pos->data)->material[4]);
      pos = g_list_next(pos);
    }
  g_string_append_printf(data, "# %s\n", DESC_ELEMENT_PROPERTIES);
  if (dataObj)
    pos = eleList;
  else
    pos = (GList*)visu_element_getAllElements();
  while(pos)
    {
      g_string_append_printf(data, "%s:\n", FLAG_ELEMENT_PROPERTIES);
      g_string_append_printf(data, "    %s %d %d\n",
			     ((VisuElement*)pos->data)->name,
			     ((VisuElement*)pos->data)->rendered,
			     ((VisuElement*)pos->data)->sensitiveToMaskingPlanes);
      pos = g_list_next(pos);
    }
  g_string_append_printf(data, "\n");
  if (eleList)
    g_list_free(eleList);
}
