/*   EXTRAITS DE LA LICENCE
	Copyright CEA, contributeurs : Luc BILLARD et Damien
	CALISTE, laboratoire L_Sim, (2001-2005)
  
	Adresse ml :
	BILLARD, non joignable par ml ;
	CALISTE, damien P caliste AT cea P fr.

	Ce logiciel est un programme informatique servant  visualiser des
	structures atomiques dans un rendu pseudo-3D. 

	Ce logiciel est rgi par la licence CeCILL soumise au droit franais et
	respectant les principes de diffusion des logiciels libres. Vous pouvez
	utiliser, modifier et/ou redistribuer ce programme sous les conditions
	de la licence CeCILL telle que diffuse par le CEA, le CNRS et l'INRIA 
	sur le site "http://www.cecill.info".

	Le fait que vous puissiez accder  cet en-tte signifie que vous avez 
	pris connaissance de la licence CeCILL, et que vous en avez accept les
	termes (cf. le fichier Documentation/licence.fr.txt fourni avec ce logiciel).
*/

/*   LICENCE SUM UP
	Copyright CEA, contributors : Luc BILLARD et Damien
	CALISTE, laboratoire L_Sim, (2001-2005)

	E-mail address:
	BILLARD, not reachable any more ;
	CALISTE, damien P caliste AT cea P fr.

	This software is a computer program whose purpose is to visualize atomic
	configurations in 3D.

	This software is governed by the CeCILL  license under French law and
	abiding by the rules of distribution of free software.  You can  use, 
	modify and/ or redistribute the software under the terms of the CeCILL
	license as circulated by CEA, CNRS and INRIA at the following URL
	"http://www.cecill.info". 

	The fact that you are presently reading this means that you have had
	knowledge of the CeCILL license and that you accept its terms. You can
	find a copy of this licence shipped with this software at Documentation/licence.en.txt.
*/
#include "visu_data.h"


#include <string.h>
#include <stdlib.h>
#include <stdio.h>
#include <math.h>

#include <GL/gl.h>
#include <GL/glu.h> 

#include "visu_object.h"
#include "visu_nodes.h"
#include "visu_extension.h"
#include "visu_rendering.h"
#include "openGLFunctions/objectList.h"
#include "openGLFunctions/text.h"
#include "openGLFunctions/view.h"
#include "extraFunctions/dataNode.h"
#include "extraFunctions/geometry.h"
#include "coreTools/toolMatrix.h"

/**
 * SECTION:visu_data
 * @short_description: Give methods to store and manage data from
 * input file(s).
 * @include: visu_nodes.h
 * 
 * <para>The main goal of V_Sim is to draw lists of elements. For
 * example, when used to render atoms, a box that contains 24 silicon
 * atoms and 46 germanium atoms is a box with two elements (silicon
 * and germanium) where the silicon element has 24 nodes and the
 * germanium element has 46 nodes. This module gives then methods to
 * create nodes (see #VisuElement to create and managed
 * elements).</para>
 *
 * <para>All nodes are stored in a structure called #VisuNodes and
 * #VisuNodes is encapsulated in a #VisuData for all not-node related
 * information. V_Sim uses one #VisuData per input file(s). This
 * structure contains a list of pointers on all the #VisuElement used
 * in this file.</para>
 *
 * <para>To iterate on nodes, one should use the provided iterators
 * (see #VisuDataIter) methods, like visu_data_iter_next().</para>
 */

struct FileDescription_struct
{
  /* Identity. */
  int kind;

  /* The name of a file. */
  gchar *name;

  /* The format of the file.
     This format can be null, if the file has not been
     parsed yet and the format is unknown.
     If the file file is not parsed but the format is
     set, then it is just an indication and may not
     be the right format. */
  ToolFileFormat *format;
};

enum {
  BOX_SIZE_CHANGED_SIGNAL,
  UNIT_CHANGED_SIGNAL,
  NODE_RENDERED_CHANGED_SIGNAL,
  NODE_MATERIAL_CHANGED_SIGNAL,
  NODE_POSITION_CHANGED_SIGNAL,
  NODE_ASK_FOR_SHOW_HIDE_SIGNAL,
  NODE_POPULATION_INCREASE_SIGNAL,
  NODE_POPULATION_DECREASE_SIGNAL,
  NODE_POPULATION_DEFINED_SIGNAL,
  OPENGL_THETA_PHI_OMEGA_CHANGED_SIGNAL,
  OPENGL_XS_YS_CHANGED_SIGNAL,
  OPENGL_GROSS_CHANGED_SIGNAL,
  OPENGL_PERSP_CHANGED_SIGNAL,
  OPENGL_NEAR_FAR_CHANGED_SIGNAL,
  OPENGL_WIDTH_HEIGHT_CHANGED_SIGNAL,
  OPENGL_FACETTES_CHANGED_SIGNAL,
  VISU_DATA_FREED_SIGNAL,
  LAST_SIGNAL
};

enum
  {
    PROP_0,
    TOTAL_ENERGY_PROP
  };

/* Define a key for a VisuDataNode value. */
#define COORDINATES_ID     "visu_data_coordinates"

struct _VisuDataClass
{
  GObjectClass parent;

  int identifierAllNodes;
};

/* Local variables. */
static VisuDataNode *dataNodeCoord;


struct _VisuDataPrivate
{
  gboolean dispose_has_run;

  /*************/
  /* The nodes */
  /*************/
  VisuNodeArray *nodeArray;

  /*******************/
  /* Drawing Methods */
  /*******************/
  /* This function is used to scale the nodes before drawing them. */
  VisuDataScalingFunc scaling;
  /* Define a method to set the color of each node.
     If this method is NULL, the color of the element is used. */
  VisuDataColorFunc setColor;

  /********************/
  /* Files attributes */
  /********************/
  /* Files that correspond to that VisuData.
     They are identified by a kind which is an integer.
     The list is pointers to FileDescription_struct. */
  GList *files;
  /* Commentary associated to the rendered data, this commentary can
     be different for each set of nodes. */
  gchar** commentary;
  /* The number of set of nodes in one file, and the currently loaded one. */
  int nSets, iSet;

  /********************/
  /* Box   attributes */
  /********************/
  /* This is the geometry of the box. Array position 1 to 6
     correspond to xx, xy, yy, zx, zy and zz. */
  double boxGeometry[6];
  gboolean cartesiansAreFull;
  /* This is the unitary vector corresponding to boxGeometry. */
  double boxNorm[6];
  /* The longest length in the box (with or without extension), and
     the margin to add. */
  float extens[2], margin;
  /* The translation to draw in the center of the box. */
  float dxxs2, dyys2, dzzs2;
  /* The periodicity. */
  VisuDataBoxBoundaries bc;
  /* This is the matrix that transform cartesian coordinates to
     coordinates in the box geometry. Use visu_data_convertXYZtoBoxCoordinates()
     to access this matrix. */
  double fromXYZtoBoxD0[3][3];
  /* This is the matrix that transform box coordinates to
     cartesian coordinates. Use visu_data_convertBoxCoordinatestoXYZ()
     to access this matrix. */
  double fromBoxtoXYZD0[3][3];
  /* This matrix is set up if the box was given in full [3][3]
     matrix and that cartesian coordinates need rotation before
     storing them. */
  float fromFullToLocal[3][3];
  
  /* Translation applied to all nodes when rendered. */
  gboolean translationApply;
  float translation[3];
  /* The extension from the box. */
  float extension[3];

  /********************/
  /* Misc. attributes */
  /********************/
  /* This list contains pointers on source id, to be removed when the object
     is finalized. */
  GList *timeoutList;
  /* The total energy of the system in eV. */
  gdouble totalEnergy;
  /* Listeners on VisuElement signals. */
  gulong *elementRendered, *elementMaterial;

  /* This object is used to describe the geometry of the OpenGL
     context. */
  VisuOpenGLView *attachedView;
};


static void visu_data_dispose     (GObject* obj);
static void visu_data_finalize    (GObject* obj);
static void visu_data_get_property(GObject* obj, guint property_id,
				   GValue *value, GParamSpec *pspec);
static void visu_data_set_property(GObject* obj, guint property_id,
				   const GValue *value, GParamSpec *pspec);

/* Local callbacks. */
static void onAskForShowHideSignal(VisuData *visuData, gboolean *redraw, gpointer data);
static void onElementRenderChanged(VisuData *data, VisuElement *element);
static void onElementMaterialChanged(VisuData *data, VisuElement *element);
static void visu_data_rebuildList(VisuData *dataObj);

/* Local routines. */
static gboolean setCoordFromString(VisuDataNode *dataNode, VisuData *dataObj,
				   VisuNode *node, gchar* labelIn,
				   gchar **labelOut, gboolean *modify);
static gchar* getCoordFromString(VisuDataNode *dataNode, VisuData *dataObj,
				 VisuNode *node);
static int* shrinkNodeList(VisuData *data, int coord, float valueTo);
static int* extendNodeList(VisuData *data, int coord, float valueTo);
static void applyBox(VisuData *data, float margin);
static float getAllNodeExtens(VisuData *dataObj);
static float defaultScaling(VisuData *data, VisuNode *node);


static guint visu_data_signals[LAST_SIGNAL] = { 0 };
static GList* allObjects;

G_DEFINE_TYPE(VisuData, visu_data, G_TYPE_OBJECT)

static void visu_data_class_init(VisuDataClass *klass)
{
  GType paramPointer[1] = {G_TYPE_POINTER};
  char *nameI18n    = _("All elements");
  char *description = _("Draw all the nodes.");
  VisuExtension *extensionData;

  DBG_fprintf(stderr, "Visu Data: creating the class of the object.\n");
  DBG_fprintf(stderr, "                - adding new signals ;\n");
  /**
   * VisuData::BoxSizeChanged:
   * @dataObj: the object which received the signal ;
   *
   * Gets emitted when the box size is changed (because of box
   * duplication for instance).
   *
   * Since: 3.4
   */
  visu_data_signals[BOX_SIZE_CHANGED_SIGNAL] =
    g_signal_newv("BoxSizeChanged",
		  G_TYPE_FROM_CLASS(klass),
		  G_SIGNAL_RUN_LAST | G_SIGNAL_NO_RECURSE | G_SIGNAL_NO_HOOKS,
		  NULL, NULL, NULL, g_cclosure_marshal_VOID__VOID,
		  G_TYPE_NONE, 0, NULL);
  /**
   * VisuData::UnitChanged:
   * @dataObj: the object which received the signal ;
   * @factor: the factor that has been applied.
   *
   * Gets emitted when the units are changed.
   *
   * Since: 3.6
   */
  visu_data_signals[UNIT_CHANGED_SIGNAL] =
    g_signal_new("UnitChanged", G_TYPE_FROM_CLASS(klass),
                 G_SIGNAL_RUN_LAST | G_SIGNAL_NO_RECURSE | G_SIGNAL_NO_HOOKS,
		 0, NULL, NULL, g_cclosure_marshal_VOID__FLOAT,
                 G_TYPE_NONE, 1, G_TYPE_FLOAT, NULL);
  /**
   * VisuData::NodeRenderedChanged:
   * @dataObj: the object which received the signal ;
   *
   * Gets emitted when one or more nodes have changed of
   * visibility. Some may have appeared, some may have disappeared.
   *
   * Since: 3.2
   */
  visu_data_signals[NODE_RENDERED_CHANGED_SIGNAL] =
    g_signal_newv("NodeRenderedChanged",
		  G_TYPE_FROM_CLASS(klass),
		  G_SIGNAL_RUN_LAST | G_SIGNAL_NO_RECURSE | G_SIGNAL_NO_HOOKS,
		  NULL, NULL, NULL, g_cclosure_marshal_VOID__VOID,
		  G_TYPE_NONE, 0, NULL);

  /**
   * VisuData::NodeMaterialChanged:
   * @dataObj: the object which received the signal ;
   *
   * Gets emitted when one or more nodes have changed of
   * color or material.
   *
   * Since: 3.6
   */
  visu_data_signals[NODE_MATERIAL_CHANGED_SIGNAL] =
    g_signal_newv("NodeMaterialChanged",
		  G_TYPE_FROM_CLASS(klass),
		  G_SIGNAL_RUN_LAST | G_SIGNAL_NO_RECURSE | G_SIGNAL_NO_HOOKS,
		  NULL, NULL, NULL, g_cclosure_marshal_VOID__VOID,
		  G_TYPE_NONE, 0, NULL);

  /**
   * VisuData::NodePositionChanged:
   * @dataObj: the object which received the signal ;
   *
   * Gets emitted when one or more nodes have moved, because of
   * translations or because the user has moved them manually.
   *
   * Since: 3.2
   */
  visu_data_signals[NODE_POSITION_CHANGED_SIGNAL] =
    g_signal_newv ("NodePositionChanged",
		   G_TYPE_FROM_CLASS (klass),
		   G_SIGNAL_RUN_LAST | G_SIGNAL_NO_RECURSE | G_SIGNAL_NO_HOOKS,
		   NULL , NULL, NULL, g_cclosure_marshal_VOID__VOID,
		   G_TYPE_NONE, 0, NULL);

  /**
   * VisuData::NodePopulationIncrease:
   * @dataObj: the object which received the signal ;
   * @ids: an array of #VisuNode ids.
   *
   * Gets emitted when the number of nodes has changed,
   * increasing. @ids contains all new ids and is -1 terminated.
   *
   * Since: 3.4
   */
  visu_data_signals[NODE_POPULATION_INCREASE_SIGNAL] =
    g_signal_newv ("NodePopulationIncrease",
		   G_TYPE_FROM_CLASS (klass),
		   G_SIGNAL_RUN_LAST | G_SIGNAL_NO_RECURSE | G_SIGNAL_NO_HOOKS,
		   NULL , NULL, NULL, g_cclosure_marshal_VOID__POINTER,
		   G_TYPE_NONE, 1, paramPointer);

  /**
   * VisuData::NodePopulationDecrease:
   * @dataObj: the object which received the signal ;
   * @ids: an array of #VisuNode ids.
   *
   * Gets emitted when the number of nodes has changed,
   * decreasing. @ids contains all removed ids and is -1 terminated.
   * When emitted, nodes have already been removed, so no external
   * routines should keep pointers on these nodes.
   *
   * Since: 3.4
   */
  visu_data_signals[NODE_POPULATION_DECREASE_SIGNAL] =
    g_signal_newv ("NodePopulationDecrease",
		   G_TYPE_FROM_CLASS (klass),
		   G_SIGNAL_RUN_LAST | G_SIGNAL_NO_RECURSE | G_SIGNAL_NO_HOOKS,
		   NULL , NULL, NULL, g_cclosure_marshal_VOID__POINTER,
		   G_TYPE_NONE, 1, paramPointer);

  /**
   * VisuData::NodePopulationDefined:
   * @dataObj: the object which received the signal ;
   * @nodes: a #VisuNodeArray object or NULL.
   *
   * Gets emitted when the population of nodes is created or
   * destroyed. It is possible then to associate new #VisuNodeProperty
   * for instance.
   *
   * Since: 3.5
   */
  visu_data_signals[NODE_POPULATION_DEFINED_SIGNAL] =
    g_signal_newv ("NodePopulationDefined",
		   G_TYPE_FROM_CLASS (klass),
		   G_SIGNAL_RUN_LAST | G_SIGNAL_NO_RECURSE | G_SIGNAL_NO_HOOKS,
		   NULL , NULL, NULL, g_cclosure_marshal_VOID__POINTER,
		   G_TYPE_NONE, 1, paramPointer);

  /**
   * VisuData::NodeAskForShowHide:
   * @dataObj: the object which received the signal ;
   * @redraw: a location on a boolean.
   *
   * Gets emitted when external modules should recompute their masking
   * effect on nodes. Location pointed by @redraw must be set to TRUE
   * if the visibility of at least one node is changed.
   *
   * Since: 3.2
   */
  visu_data_signals[NODE_ASK_FOR_SHOW_HIDE_SIGNAL] =
    g_signal_newv ("NodeAskForShowHide",
		   G_TYPE_FROM_CLASS (klass),
		   G_SIGNAL_RUN_LAST | G_SIGNAL_NO_RECURSE | G_SIGNAL_NO_HOOKS,
		   NULL , NULL, NULL, g_cclosure_marshal_VOID__POINTER,
		   G_TYPE_NONE, 1, paramPointer);

  /**
   * VisuData::OpenGLThetaPhiOmega:
   * @dataObj: the object which received the signal ;
   * @view: the new #VisuOpenGLView.
   *
   * Gets emitted when the camera angles have been changed.
   *
   * Since: 3.2
   */
  visu_data_signals[OPENGL_THETA_PHI_OMEGA_CHANGED_SIGNAL] =
    g_signal_newv ("OpenGLThetaPhiOmega", G_TYPE_FROM_CLASS (klass),
		   G_SIGNAL_RUN_LAST | G_SIGNAL_NO_RECURSE | G_SIGNAL_NO_HOOKS,
		   NULL , NULL, NULL, g_cclosure_marshal_VOID__POINTER,
		   G_TYPE_NONE, 1, paramPointer);

  /**
   * VisuData::OpenGLXsYs:
   * @dataObj: the object which received the signal ;
   * @view: the new #VisuOpenGLView.
   *
   * Gets emitted when the camera position has been changed.
   *
   * Since: 3.2
   */
  visu_data_signals[OPENGL_XS_YS_CHANGED_SIGNAL] =
    g_signal_newv ("OpenGLXsYs", G_TYPE_FROM_CLASS (klass),
		   G_SIGNAL_RUN_LAST | G_SIGNAL_NO_RECURSE | G_SIGNAL_NO_HOOKS,
		   NULL , NULL, NULL, g_cclosure_marshal_VOID__POINTER,
		   G_TYPE_NONE, 1, paramPointer);

  /**
   * VisuData::OpenGLGross:
   * @dataObj: the object which received the signal ;
   * @view: the new #VisuOpenGLView.
   *
   * Gets emitted when the camera zoom factor has been changed.
   *
   * Since: 3.2
   */
  visu_data_signals[OPENGL_GROSS_CHANGED_SIGNAL] =
    g_signal_newv ("OpenGLGross", G_TYPE_FROM_CLASS (klass),
		   G_SIGNAL_RUN_LAST | G_SIGNAL_NO_RECURSE | G_SIGNAL_NO_HOOKS,
		   NULL , NULL, NULL, g_cclosure_marshal_VOID__POINTER,
		   G_TYPE_NONE, 1, paramPointer);

  /**
   * VisuData::OpenGLPersp:
   * @dataObj: the object which received the signal ;
   * @view: the new #VisuOpenGLView.
   *
   * Gets emitted when the camera perspective factor has been changed.
   *
   * Since: 3.2
   */
  visu_data_signals[OPENGL_PERSP_CHANGED_SIGNAL] =
    g_signal_newv ("OpenGLPersp", G_TYPE_FROM_CLASS (klass),
		   G_SIGNAL_RUN_LAST | G_SIGNAL_NO_RECURSE | G_SIGNAL_NO_HOOKS,
		   NULL , NULL, NULL, g_cclosure_marshal_VOID__POINTER,
		   G_TYPE_NONE, 1, paramPointer);

  /**
   * VisuData::OpenGLNearFar:
   * @dataObj: the object which received the signal ;
   * @view: the new #VisuOpenGLView.
   *
   * Gets emitted when the viewing field has been changed.
   *
   * Since: 3.2
   */
  visu_data_signals[OPENGL_NEAR_FAR_CHANGED_SIGNAL] =
    g_signal_newv ("OpenGLNearFar", G_TYPE_FROM_CLASS (klass),
		   G_SIGNAL_RUN_LAST | G_SIGNAL_NO_RECURSE | G_SIGNAL_NO_HOOKS,
		   NULL , NULL, NULL, g_cclosure_marshal_VOID__POINTER,
		   G_TYPE_NONE, 1, paramPointer);

  /**
   * VisuData::OpenGLWidthHeight:
   * @dataObj: the object which received the signal ;
   * @view: the new #VisuOpenGLView.
   *
   * Gets emitted when the viewing frame has been changed.
   *
   * Since: 3.2
   */
  visu_data_signals[OPENGL_WIDTH_HEIGHT_CHANGED_SIGNAL] =
    g_signal_newv ("OpenGLWidthHeight", G_TYPE_FROM_CLASS (klass),
		   G_SIGNAL_RUN_LAST | G_SIGNAL_NO_RECURSE | G_SIGNAL_NO_HOOKS,
		   NULL , NULL, NULL, g_cclosure_marshal_VOID__POINTER,
		   G_TYPE_NONE, 1, paramPointer);

  /**
   * VisuData::OpenGLFacetteChanged:
   * @dataObj: the object which received the signal ;
   *
   * Gets emitted when precision of the drawn object has been changed.
   *
   * Since: 3.2
   */
  visu_data_signals[OPENGL_FACETTES_CHANGED_SIGNAL] =
    g_signal_newv ("OpenGLFacetteChanged", G_TYPE_FROM_CLASS (klass),
		   G_SIGNAL_RUN_LAST | G_SIGNAL_NO_RECURSE | G_SIGNAL_NO_HOOKS,
		   NULL , NULL, NULL, g_cclosure_marshal_VOID__VOID,
		   G_TYPE_NONE, 0, NULL);

  /**
   * VisuData::objectFreed:
   * @dataObj: the object which received the signal ;
   *
   * Gets emitted when the object is been destroyed. All external
   * objects having a reference on this #VisuData should clean it.
   *
   * Since: 3.3
   */
  visu_data_signals[VISU_DATA_FREED_SIGNAL] =
    g_signal_newv ("objectFreed", G_TYPE_FROM_CLASS (klass),
		   G_SIGNAL_RUN_LAST | G_SIGNAL_NO_RECURSE | G_SIGNAL_NO_HOOKS,
		   NULL , NULL, NULL, g_cclosure_marshal_VOID__VOID,
		   G_TYPE_NONE, 0, NULL);

  /* Connect the overloading methods. */
  G_OBJECT_CLASS(klass)->dispose      = visu_data_dispose;
  G_OBJECT_CLASS(klass)->finalize     = visu_data_finalize;
  G_OBJECT_CLASS(klass)->set_property = visu_data_set_property;
  G_OBJECT_CLASS(klass)->get_property = visu_data_get_property;

  /**
   * VisuData::totalEnergy:
   *
   * Store the total energy of the system in eV.
   *
   * Since: 3.6
   */
  g_object_class_install_property(G_OBJECT_CLASS(klass), TOTAL_ENERGY_PROP,
				  g_param_spec_double("totalEnergy", "Total energy",
						      "Total energy of"
						      " the system (eV)",
						      -G_MAXFLOAT, G_MAXFLOAT,
						      G_MAXFLOAT,
						      G_PARAM_CONSTRUCT |
						      G_PARAM_READWRITE));

  /* Initialise internal variables. */
  /* Get an OpenGL identifier to store all the materials. */
  klass->identifierAllNodes     = visu_openGL_objectList_new(VISU_ELEMENT_MAX_NUMBER + 1);
  extensionData = visu_extension_new(EXT_VISU_DATA_ID, nameI18n, description,
				      klass->identifierAllNodes,
				      visu_data_rebuildList);
  visu_extension_setPriority(extensionData, VISU_EXTENSION_PRIORITY_NODES);
  visu_extension_setSensitiveToRenderingMode(extensionData, TRUE);
  extensionData->used = 1;
  visuExtensions_add(extensionData);

  allObjects = (GList*)0;

  /* Register a new NodeData. */
  dataNodeCoord = VISU_DATA_NODE(visu_data_node_newWithCallbacks
                                 (COORDINATES_ID,
                                  setCoordFromString, getCoordFromString));
  visu_data_node_setLabel(dataNodeCoord, _("Coord. (x, y, z)"));
  visu_data_node_setEditable(dataNodeCoord, TRUE);
}

static void visu_data_init(VisuData *obj)
{
  int i;

  DBG_fprintf(stderr, "Visu Data: initializing a new object (%p).\n",
	      (gpointer)obj);
  
  /* Connect a method to make all nodes rendered attribute to TRUE
     when the NodeAskForShowHide signal is emited. */
  g_signal_connect(G_OBJECT(obj), "NodeAskForShowHide",
		   G_CALLBACK(onAskForShowHideSignal), (gpointer)0);

  obj->privateDt = g_malloc(sizeof(VisuDataPrivate));
  obj->privateDt->dispose_has_run = FALSE;

  /* Public data. */
  obj->ntype= 0;
  obj->fromVisuElementToInt = (GHashTable*)0;
  obj->fromIntToVisuElement = (VisuElement**)0;

  /* Private data. */
  obj->privateDt->nodeArray        = (VisuNodeArray*)0;
  obj->privateDt->files            = (GList*)0;
  obj->privateDt->commentary       = g_malloc(sizeof(gchar*) * 2);
  obj->privateDt->commentary[0]    = g_strdup("");
  obj->privateDt->commentary[1]    = (gchar*)0;
  obj->privateDt->translationApply = FALSE;
  for (i = 0; i < 3; i++)
    {
      obj->privateDt->translation[i] = 0.;
      obj->privateDt->extension[i]   = 0.;
    }
  for (i = 0; i < 6; i++)
    obj->privateDt->boxGeometry[i] = 0.;
  obj->privateDt->cartesiansAreFull = FALSE;
  obj->privateDt->bc               = VISU_DATA_BOX_FREE;
  obj->privateDt->extens[0]        = 0.f;
  obj->privateDt->extens[1]        = 0.f;
  obj->privateDt->margin           = 0.f;
  obj->privateDt->nSets            = 1;
  obj->privateDt->iSet             = -1;
  obj->privateDt->timeoutList      = (GList*)0;  
  obj->privateDt->attachedView     = (VisuOpenGLView*)0;
  obj->privateDt->scaling          = defaultScaling;
  obj->privateDt->setColor         = (VisuDataColorFunc)0;
  obj->privateDt->elementRendered  = (gulong*)0;
  obj->privateDt->elementMaterial  = (gulong*)0;

  /* Initial allocations. */
  obj->fromVisuElementToInt = g_hash_table_new_full(g_direct_hash, g_direct_equal,
						     NULL, g_free);
  g_return_if_fail(obj->fromVisuElementToInt);

  visu_data_node_setUsed(dataNodeCoord, obj, 3);

  /* Add object from allObjects list. */
  allObjects = g_list_append(allObjects, (gpointer)obj);

  DBG_fprintf(stderr, "Visu Data: emit the 'DataNew' signal.\n");
  g_signal_emit(VISU_INSTANCE, VISU_SIGNALS[DATANEW_SIGNAL],
		0 /* details */, G_OBJECT(obj), NULL);
  DBG_fprintf(stderr, "Visu Data: emission OK of the 'DataNew' signal.\n");
}

/* This method can be called several times.
   It should unref all of its reference to
   GObjects. */
static void visu_data_dispose(GObject* obj)
{
  VisuData *data;
  guint i;

  DBG_fprintf(stderr, "Visu Data: dispose object %p.\n", (gpointer)obj);

  data = VISU_DATA(obj);
  if (data->privateDt->dispose_has_run)
    return;
  data->privateDt->dispose_has_run = TRUE;

#if DEBUG == 1
  g_mem_profile();
#endif

  DBG_fprintf(stderr, "Visu Data: emit a 'objectFreed' signal.\n");
  g_signal_emit(obj, visu_data_signals[VISU_DATA_FREED_SIGNAL],
		0, NULL);

  visu_data_node_setUsed(dataNodeCoord, data, 0);
  for (i = 0; i < data->ntype; i++)
    {
      g_signal_handler_disconnect(G_OBJECT(data->fromIntToVisuElement[i]),
                                  data->privateDt->elementRendered[i]);
      g_signal_handler_disconnect(G_OBJECT(data->fromIntToVisuElement[i]),
                                  data->privateDt->elementMaterial[i]);
    }

  /* Chain up to the parent class */
  G_OBJECT_CLASS(visu_data_parent_class)->dispose(obj);
}
/* This method is called once only. */
static void visu_data_finalize(GObject* obj)
{
  VisuData *data;
  GList *lst;

  g_return_if_fail(obj);

  DBG_fprintf(stderr, "Visu Data: finalize object %p.\n", (gpointer)obj);

  /* Remove object from allObjects list. */
  allObjects = g_list_remove(allObjects, (gpointer)obj);

  data = VISU_DATA(obj);

  /* Free public elements. */
  if (data->fromVisuElementToInt)
    g_hash_table_destroy(data->fromVisuElementToInt);
  if (data->fromIntToVisuElement)
    g_free(data->fromIntToVisuElement);

  /* Free privateDts elements. */
  if (data->privateDt)
    {
      DBG_fprintf(stderr, "Visu data: free private data.\n");
      if (data->privateDt->nodeArray)
	visu_node_array_freeNodes(data->privateDt->nodeArray);
      if (data->privateDt->files)
	{
	  lst = data->privateDt->files;
	  while (lst)
	    {
	      g_free(((struct FileDescription_struct*)lst->data)->name);
	      g_free(lst->data);
	      lst = g_list_next(lst);
	    }
	  g_list_free(data->privateDt->files);
	}
      if (data->privateDt->commentary)
	g_strfreev(data->privateDt->commentary);
      if (data->privateDt->timeoutList)
	{
	  lst = data->privateDt->timeoutList;
	  while (lst)
	    {
	      g_source_remove(*(guint*)lst->data);
	      g_free(lst->data);
	      lst = g_list_next(lst);
	    }
	  g_list_free(data->privateDt->timeoutList);
	}
      if (data->privateDt->attachedView)
	VisuOpenGLViewFree(data->privateDt->attachedView);
      if (data->privateDt->elementRendered)
        g_free(data->privateDt->elementRendered);
      if (data->privateDt->elementMaterial)
        g_free(data->privateDt->elementMaterial);
      g_free(data->privateDt);
    }
  /* The free is called by g_type_free_instance... */
/*   g_free(data); */

  /* Chain up to the parent class */
  DBG_fprintf(stderr, "Visu data: chain to parent.\n");
  G_OBJECT_CLASS(visu_data_parent_class)->finalize(obj);

  DBG_fprintf(stderr, "Visu data: freeing ... OK.\n");

#if DEBUG == 1
  g_mem_profile();
#endif
}
static void visu_data_get_property(GObject* obj, guint property_id,
				   GValue *value, GParamSpec *pspec)
{
  VisuData *self = VISU_DATA(obj);

  DBG_fprintf(stderr, "Visu Data: get property '%s' -> ",
	      g_param_spec_get_name(pspec));
  switch (property_id)
    {
    case TOTAL_ENERGY_PROP:
      g_value_set_double(value, self->privateDt->totalEnergy);
      DBG_fprintf(stderr, "%geV.\n", self->privateDt->totalEnergy);
      break;
    default:
      /* We don't have any other property... */
      G_OBJECT_WARN_INVALID_PROPERTY_ID(obj, property_id, pspec);
      break;
    }
}
static void visu_data_set_property(GObject* obj, guint property_id,
				   const GValue *value, GParamSpec *pspec)
{
  VisuData *self = VISU_DATA(obj);

  DBG_fprintf(stderr, "Visu Data: set property '%s' -> ",
	      g_param_spec_get_name(pspec));
  switch (property_id)
    {
    case TOTAL_ENERGY_PROP:
      self->privateDt->totalEnergy = g_value_get_double(value);
      DBG_fprintf(stderr, "%geV.\n", self->privateDt->totalEnergy);
      break;
    default:
      /* We don't have any other property... */
      G_OBJECT_WARN_INVALID_PROPERTY_ID(obj, property_id, pspec);
      break;
    }
}


/**
 * visu_data_new:
 *
 * This creates an empty #VisuData object.
 *
 * Returns: a newly created #VisuData object (its ref count is set to 1).
 */
VisuData* visu_data_new(void)
{
  VisuData *data;

#if DEBUG == 1
  g_mem_profile();
#endif

  data = VISU_DATA(g_object_new(VISU_DATA_TYPE, NULL));

  if (!data)
    return (VisuData*)0;

  data->privateDt->attachedView = VisuOpenGLViewNew();

#if DEBUG == 1
  g_mem_profile();
#endif

  return data;
}

/**
 * visu_data_newWithSize:
 * @w: the width size ;
 * @h: the height size.
 *
 * Create a new #VisuData object with a given size for the view.
 *
 * Returns: (transfer full): a newly created object.
 */
VisuData* visu_data_newWithSize(guint w, guint h)
{
  VisuData *data;

#if DEBUG == 1
  g_mem_profile();
#endif

  data = VISU_DATA(g_object_new(VISU_DATA_TYPE, NULL));

  if (!data)
    return (VisuData*)0;

  data->privateDt->attachedView = VisuOpenGLViewNew();
  data->privateDt->attachedView->window->width = w;
  data->privateDt->attachedView->window->height = h;

#if DEBUG == 1
  g_mem_profile();
#endif

  return data;
}

/**
 * visu_data_newWithView:
 * @view: an #VisuOpenGLView object.
 * @autoAdjust: a boolean.
 *
 * This creates an empty #VisuData object with values for its view taken
 * from the given @view argument (the argument is indeed copied). If @autoAdjust
 * is TRUE, the zoom level is automaticaly adjusted for the object to be
 * full size at zoom level 1.
 *
 * Returns: (transfer full): a newly created #VisuData object (its ref count is set to 1).
 */
VisuData* visu_data_newWithView(VisuOpenGLView *view, gboolean autoAdjust)
{
  VisuData *data;
  
  g_return_val_if_fail(view, (VisuData*)0);

#if DEBUG == 1
  g_mem_profile();
#endif

  data = VISU_DATA(g_object_new(VISU_DATA_TYPE, NULL));

  if (!data)
    return (VisuData*)0;

  data->privateDt->attachedView = VisuOpenGLViewCopy(view);
  if (autoAdjust)
    openGLCameraSet_refLength(data->privateDt->attachedView->camera, -1.f, unit_undefined);
  DBG_fprintf(stderr, "Visu Data: create a new object with ref length : %g.\n",
              data->privateDt->attachedView->camera->length0);

#if DEBUG == 1
  g_mem_profile();
#endif

  return data;
}


static gboolean setCoordFromString(VisuDataNode *dataNode, VisuData *dataObj,
				   VisuNode *node, gchar* labelIn,
				   gchar **labelOut, gboolean *modify)
{
  int res, ln, i;
  gchar **datas;
  gboolean error;
  float valueFloat;

  g_return_val_if_fail(IS_VISU_DATA_NODE_TYPE(dataNode) &&
		       IS_VISU_DATA_TYPE(dataObj) && node, FALSE);
  g_return_val_if_fail(labelIn && labelOut && modify, FALSE);

  /* Parse the given labelIn.
     remove first and last parenthesis. */
  if (labelIn[0] == '(')
    labelIn += 1;
  ln = strlen(labelIn);
  if (labelIn[ln - 1] == ')')
    labelIn[ln - 1] = '\0';
  datas = g_strsplit(labelIn, ";", 3);
  *modify = FALSE;
  for (i = 0; datas[i]; i++)
    {
      error = FALSE;
      res = sscanf(datas[i], "%f", &valueFloat);
      if (res != 1)
	error = TRUE;
      else
	if (node->xyz[i] != valueFloat)
	  {
	    node->xyz[i] = valueFloat;
	    *modify = TRUE;
	  }
      if (error)
	{
	  *labelOut = getCoordFromString(dataNode, dataObj, node);
	  g_strfreev(datas);
	  return FALSE;
	}
    }
  if (i != 3)
    error = TRUE;
  else
    error = FALSE;

  *labelOut = getCoordFromString(dataNode, dataObj, node);
  g_strfreev(datas);

  if (*modify)
    {
      visu_data_createNodes(dataObj, dataObj->fromIntToVisuElement[node->posElement]);
      visu_data_emitNodePositionChanged(dataObj);
      VISU_ADD_REDRAW;
    }

  return !error;
}
static gchar* getCoordFromString(VisuDataNode *dataNode, VisuData *dataObj,
				 VisuNode *node)
{
  GString *str;
  gchar *value;

  g_return_val_if_fail(IS_VISU_DATA_NODE_TYPE(dataNode) &&
		       IS_VISU_DATA_TYPE(dataObj) && node, (gchar*)0);

  /* Check if the given property has an association with the given VisuData. */
  DBG_fprintf(stderr, "Visu Data: get label for node coordinates.\n");

  /* Set the format attribute. */
  str = g_string_new("");
  g_string_append_printf(str, "( %g ; %g ; %g )",
			 node->xyz[0], node->xyz[1], node->xyz[2]);
  value = str->str;
  g_string_free(str, FALSE);

  DBG_fprintf(stderr, "Visu Data: get values '%s'.\n", value);
  return value;
}
static void onAskForShowHideSignal(VisuData *visuData, gboolean *redraw,
				   gpointer data _U_)
{
  unsigned int i, j;

  DBG_fprintf(stderr, "Visu Data: caught the 'NodeAskForShowHide' signal,"
	      " setting all node rendered attribute to TRUE.\n");

  for (i = 0; i < visuData->privateDt->nodeArray->ntype; i++)
    for(j = 0; j < visuData->privateDt->nodeArray->numberOfStoredNodes[i]; j++)
      *redraw = visu_node_setVisibility
	(&visuData->privateDt->nodeArray->nodes[i][j], TRUE) || *redraw;

  DBG_fprintf(stderr, "             - returned redraw value : %d.\n", *redraw);
}
static void onElementRenderChanged(VisuData *data, VisuElement *element _U_)
{
  DBG_fprintf(stderr, "Visu Data: caught the 'ElementRenderedChanged' signal,"
	      " emitting node render signal.\n");
  g_signal_emit(data, visu_data_signals[NODE_RENDERED_CHANGED_SIGNAL],
		0 , NULL);
}
static void onElementMaterialChanged(VisuData *data, VisuElement *element _U_)
{
  DBG_fprintf(stderr, "Visu Data: caught the 'ElementMaterialChanged' signal,"
	      " emitting node material signal.\n");
  g_signal_emit(data, visu_data_signals[NODE_MATERIAL_CHANGED_SIGNAL],
		0 , NULL);
}




/***************/
/* OpenGL Part */
/***************/

/**
 * visu_data_createNodes: 
 * @data: a #VisuData object ;
 * @ele: a #VisuElement object.
 *
 * This creates the glObjectList that contains all the nodes of the given #VisuElement.
 */
void visu_data_createNodes(VisuData *data, VisuElement *ele)
{
  VisuRendering *method;
  VisuNode *node;
  int *nEle;
  unsigned int j;
  int mat;

  g_return_if_fail(data && ele);

  nEle = (int*)g_hash_table_lookup(data->fromVisuElementToInt, (gpointer)ele);
  g_return_if_fail(nEle);
  method = visu_object_getRendering(VISU_INSTANCE);
  g_return_if_fail(method);

  if (!ele->materialIsUpToDate)
    visu_element_createMaterial(ele);

  glNewList(VISU_DATA_GET_CLASS(data)->identifierAllNodes + *nEle + 1, GL_COMPILE);
  if (ele->rendered)
    {
      /* Set the material by default if available. */
      mat = visu_element_getIdentifierMaterial(ele);
      if (!mat)
	g_warning("Can't find the material identifier for the"
		  " element '%s'.\n", ele->name);
      else
	glCallList(mat);
      DBG_fprintf(stderr, "Visu Data: creating glObjectList of nodes for '%s' - %d.\n",
		  ele->name, mat);
      for(j = 0; j < data->privateDt->nodeArray->numberOfStoredNodes[*nEle]; j++)
	{
	  node = data->privateDt->nodeArray->nodes[*nEle] + j;
	  if (node->rendered)
	    {
	      glLoadName((GLuint)node->number);
	      visu_rendering_createNode(method, data, node, ele);
	    }
	}
    }
  else
    DBG_fprintf(stderr, "Visu Data: skipping glObjectList of nodes for '%s' - %d.\n",
		ele->name, ele->openGLIdentifier);
  glEndList();
}

/**
 * visu_data_createNode: 
 * @data: a #VisuData object ;
 * @node: a #VisuNode object.
 *
 * This method call the createNode method of the current rendering method
 * for the given node. It also calls the list of the material associated
 * to the given node.
 */
void visu_data_createNode(VisuData *data, VisuNode *node)
{
/* DBG_fprintf(stderr, "OK\n"); */
  VisuRendering *method;
  int mat;
  VisuElement *ele;

  method = visu_object_getRendering(VISU_INSTANCE);
  g_return_if_fail(method);

  ele = data->fromIntToVisuElement[node->posElement];
  g_return_if_fail(ele);
  /* Set the material by default if available. */
  mat = visu_element_getIdentifierMaterial(ele);
  if (!mat)
    g_warning("Can't find the material identifier for the"
	      " element '%s'.\n", ele->name);
  else
    glCallList(mat);

  glLoadName((GLuint)node->number);
  visu_rendering_createNode(method, data, node, ele);
}

/**
 * visu_data_createAllNodes: 
 * @data: a #VisuData object.
 *
 * This create the glObjectList registered that contains all the
 * nodes. This glObjectList is made of all nodes of all element whose
 * attribute rendered is TRUE and translated to their own positions.
 *
 * This method doesn't ask for redraw and #VISU_ADD_REDRAW macro
 * should be called by caller.
 */
void visu_data_createAllNodes(VisuData *data)
{
  unsigned int i;
#if DEBUG == 1
  GTimer *timer;
  gulong fractionTimer;
#endif

  g_return_if_fail(IS_VISU_DATA_TYPE(data));

  DBG_fprintf(stderr, "Visu Data: 'visu_data_createAllNodes' called.\n");

#if DEBUG == 1
  timer = g_timer_new();
  g_timer_start(timer);
#endif

  DBG_fprintf(stderr, "Visu Data: loop on elements.\n");
  for (i = 0; i < data->ntype; i++)
    visu_data_createNodes(data, data->fromIntToVisuElement[i]);
  DBG_fprintf(stderr, "Visu Data: OK.\n");

  glNewList(VISU_DATA_GET_CLASS(data)->identifierAllNodes, GL_COMPILE);
  glLineWidth(1);
  for (i = 0; i < data->ntype; i++)
    {
      DBG_fprintf(stderr, "Visu Data: add list %d.\n",
		  VISU_DATA_GET_CLASS(data)->identifierAllNodes + i + 1);
/*       if (currentVisuData->fromIntToVisuElement[i]->rendered) */
      glCallList(VISU_DATA_GET_CLASS(data)->identifierAllNodes + i + 1);
    }
  glEndList();

#if DEBUG == 1
  g_timer_stop(timer);
  fprintf(stderr, "Visu Data: lists built in %g micro-s.\n", g_timer_elapsed(timer, &fractionTimer)/1e-6);
  g_timer_destroy(timer);
#endif
}

/**
 * visu_data_createAllElements:
 * @data: a #VisuData object.
 *
 * This method will call the createOpenGLElementFunc() method of the current #RenderingMethod
 * on all the nodes of the given #VisuData.
 */
void visu_data_createAllElements(VisuData *data)
{
  unsigned int i;
  VisuRendering *method;

  method = visu_object_getRendering(VISU_INSTANCE);
  g_return_if_fail(method);

  g_return_if_fail(IS_VISU_DATA_TYPE(data));

  DBG_fprintf(stderr, "Visu Data: create OpenGl elements for"
	      " all VisuElement used in given VisuData %p.\n", (gpointer)data);
  for (i = 0; i < data->ntype; i++)
    {
      DBG_fprintf(stderr, "Visu Data: element '%s' %p.\n",
		  data->fromIntToVisuElement[i]->name,
                  (gpointer)data->fromIntToVisuElement[i]);
      data->fromIntToVisuElement[i]->openGLIdentifier =
        visu_rendering_createElement(method, data->fromIntToVisuElement[i],
                                     data->privateDt->attachedView->camera,
                                     MIN(data->privateDt->attachedView->window->width,
                                         data->privateDt->attachedView->window->height));
    }
  DBG_fprintf(stderr, "Visu Data: creation done for all elements.\n");
}

/**
 * visu_data_getObjectList:
 * @data: a valid #VisuObject ;
 *
 * All the drawned nodes are stored in an OpenGL list. The identifier of
 * this list can be access with the present method.
 *
 * Returns: an integer that identifies the OpenGL list of all drawn nodes.
 */
int visu_data_getObjectList(VisuData *data)
{
  return VISU_DATA_GET_CLASS(data)->identifierAllNodes;
}

static void visu_data_rebuildList(VisuData *dataObj)
{
  unsigned int i;

  DBG_fprintf(stderr, "Visu Data: rebuilding object list for visuData %p.\n", (gpointer)dataObj);
  g_return_if_fail(dataObj);

  for (i = 0; i < dataObj->ntype; i++)
    visu_element_createMaterial(dataObj->fromIntToVisuElement[i]);
  visu_data_createAllElements(dataObj);
  visu_data_createAllNodes(dataObj);
}

/**
 * visu_data_setPopulation: (skip)
 * @data: a #VisuData object;
 * @nbOfTypes: number of #VisuElement to defined.
 * @nbOfNodesPerVisuElement: an array with the number of nodes per #VisuElement.
 * @visu_element_Used: an array of the actual #VisuElement.
 *
 * Allocate the storage arrays for @data, with @nbOfTypes #VisuElement
 * and corresponding numbers of nodes from @nbOfNodesPerVisuElement.
 *
 * Returns: 1 on success.
 */
int visu_data_setPopulation(VisuData *data, unsigned int nbOfTypes,
			     unsigned int* nbOfNodesPerVisuElement,
			     VisuElement **visu_element_Used)
{
  GArray *nodes, *elements;
  guint i;
  gboolean ret;

  nodes = g_array_new(FALSE, FALSE, sizeof(guint));
  elements = g_array_new(FALSE, FALSE, sizeof(VisuElement*));
  for (i = 0; i < nbOfTypes; i++)
    {
      g_array_append_val(nodes, nbOfNodesPerVisuElement[i]);
      g_array_append_val(elements, visu_element_Used[i]);
    }
  ret = visu_data_allocatePopulation(data, nodes, elements);
  g_array_free(nodes, TRUE);
  g_array_free(elements, TRUE);

  return ret;
}
/**
 * visu_data_allocatePopulation:
 * @data: a #VisuData object;
 * @nNodesPerElement: (in) (element-type guint): number of #VisuNode per VisuElement;
 * @elements: (in) (element-type VisuElement*): pointers to #VisuElement in the same order that @nNodesPerElement;
 *
 * This method allocates the storing part of the given #VisuData structure and
 * store all the #VisuNodes.
 *
 * Returns: TRUE if everything goes right.
 */
gboolean visu_data_allocatePopulation(VisuData *data,
                                      GArray *nNodesPerElement,
                                      GArray *elements)
{
  unsigned int i;
  int *pos;

  g_return_val_if_fail(IS_VISU_DATA_TYPE(data), FALSE);
  g_return_val_if_fail(elements && nNodesPerElement, FALSE);
  g_return_val_if_fail(nNodesPerElement->len == elements->len, FALSE);

  DBG_fprintf(stderr, "Visu Data: allocating VisuData %p.\n", (gpointer)data);

  data->fromIntToVisuElement       = g_malloc(sizeof(VisuElement*) * elements->len);
  data->ntype                      = elements->len;
  data->privateDt->elementRendered = g_malloc(sizeof(gulong) * elements->len);
  data->privateDt->elementMaterial = g_malloc(sizeof(gulong) * elements->len);
  
  for(i = 0; i < data->ntype; i++)
    {
      pos = g_malloc(sizeof(int));
      *pos = i;
      g_hash_table_insert(data->fromVisuElementToInt,
			  g_array_index(elements, gpointer, i), (gpointer)pos);
      DBG_fprintf(stderr, " | add VisuElement '%p' -> '%s' to hashtable.\n",
                  g_array_index(elements, gpointer, i),
                  g_array_index(elements, VisuElement*, i)->name);
      data->fromIntToVisuElement[i] = g_array_index(elements, VisuElement*, i);
      data->privateDt->elementRendered[i] =
        g_signal_connect_swapped(G_OBJECT(data->fromIntToVisuElement[i]),
                                 "ElementRenderedChanged",
                                 G_CALLBACK(onElementRenderChanged),
                                 data);
      data->privateDt->elementMaterial[i] =
        g_signal_connect_swapped(G_OBJECT(data->fromIntToVisuElement[i]),
                                 "ElementMaterialChanged",
                                 G_CALLBACK(onElementMaterialChanged),
                                 data);
    }
  DBG_fprintf(stderr, " ... OK\n");

  data->privateDt->nodeArray =
    visu_node_array_newNodes(nNodesPerElement->len,
                             (unsigned int*)nNodesPerElement->data);
  g_return_val_if_fail(data->privateDt->nodeArray, 0);

  DBG_fprintf(stderr, "VisuData: emit NodePopulationDefined.\n");
  g_signal_emit(data, visu_data_signals[NODE_POPULATION_DEFINED_SIGNAL],
		0, data->privateDt->nodeArray, NULL);
  DBG_fprintf(stderr, "VisuData: emission done (NodePopulationDefined).\n");

  return TRUE;
}
/**
 * visu_data_allocatePopulationByNames:
 * @data: a #VisuData object;
 * @nNodesPerElement: (in) (element-type guint): number of #VisuNode per VisuElement;
 * @elementNames: (in) (element-type utf8): names of elements;
 *
 * This method allocates the storing part of the given #VisuData structure and
 * store all the #VisuNodes.
 *
 * Returns: TRUE if everything goes right.
 */
gboolean visu_data_allocatePopulationByNames(VisuData *data,
                                             GArray *nNodesPerElement,
                                             GArray *elementNames)
{
  GArray *elements;
  guint i;
  gboolean ret;
  VisuElement *ele;

  DBG_fprintf(stderr, "Visu Data: allocating VisuData %p from names.\n", (gpointer)data);

  elements = g_array_new(FALSE, FALSE, sizeof(VisuElement*));
  for (i = 0; i < elementNames->len; i++)
    {
      ele = visu_element_retrieveFromName(g_array_index(elementNames, gchar*, i),
                                          (gboolean*)0);
      g_array_append_val(elements, ele);
    }
  ret = visu_data_allocatePopulation(data, nNodesPerElement, elements);
  g_array_free(elements, TRUE);

  return ret;
}

/**
 * visu_data_freePopulation:
 * @data: a VisuData to be freed.
 *
 * This method frees only the allocated memory that deals with
 * the nodes (i.e. everything except the data of the files,
 * the properties and the setColor method.
 */
void visu_data_freePopulation(VisuData *data)
{
  if (!data)
    return;

  DBG_fprintf(stderr, "Visu Data: freeing the population of VisuData %p ...\n", (gpointer)data);

  if (data->fromVisuElementToInt)
    g_hash_table_destroy(data->fromVisuElementToInt);
  data->fromVisuElementToInt = g_hash_table_new_full(g_direct_hash, g_direct_equal,
						     NULL, g_free);
  if (data->fromIntToVisuElement)
    g_free(data->fromIntToVisuElement);
  data->fromIntToVisuElement = (VisuElement**)0;

  if (data->privateDt->nodeArray)
    visu_node_array_freeNodes(data->privateDt->nodeArray);
  data->privateDt->nodeArray = (VisuNodeArray*)0;

  data->privateDt->iSet = -1;
  g_object_set_data(G_OBJECT(data), "unit", (gpointer)0);
  data->privateDt->extension[0]   = 0.f;
  data->privateDt->extension[1]   = 0.f;
  data->privateDt->extension[2]   = 0.f;
  data->privateDt->translation[0] = 0.f;
  data->privateDt->translation[1] = 0.f;
  data->privateDt->translation[2] = 0.f;

  DBG_fprintf(stderr, "Visu Data: freeing ... OK.\n");

  DBG_fprintf(stderr, "VisuData: emit NodePopulationDefined.\n");
  g_signal_emit(data, visu_data_signals[NODE_POPULATION_DEFINED_SIGNAL],
		0, data->privateDt->nodeArray, NULL);
  DBG_fprintf(stderr, "VisuData: emission done (NodePopulationDefined).\n");
}

/**
 * visu_data_addFile:
 * @data: a #VisuData object ;
 * @file: a string that points to a file ;
 * @kind: an integer to qualify the file to add ;
 * @format: (allow-none): a file format.
 *
 * This method is used to add files
 * of type @kind to the @data. The @file
 * attribute is copied. The @format argument can be null.
 */
void visu_data_addFile(VisuData *data, gchar* file, int kind, ToolFileFormat *format)
{
  struct FileDescription_struct *dt;

  g_return_if_fail(data && file);

  dt = g_malloc(sizeof(struct FileDescription_struct));
  dt->kind = kind;
  dt->name = g_strdup(file);
  dt->format = format;
  data->privateDt->files = g_list_prepend(data->privateDt->files, (gpointer)dt);
  DBG_fprintf(stderr, "Visu Data: adding '%s' filename (key: %d) to the VisuData %p.\n",
	      file, kind, (gpointer)data);
}

/**
 * visu_data_removeAllFiles:
 * @data: a #VisuData object.
 *
 * This method is used to empty the list of
 * known file from the given @data.
 */

void visu_data_removeAllFiles(VisuData *data)
{
  GList *lst;

  g_return_if_fail(IS_VISU_DATA_TYPE(data));

  lst = data->privateDt->files;
  while (lst)
    {
      g_free(((struct FileDescription_struct*)lst->data)->name);
      g_free(lst->data);
      lst = g_list_next(lst);
    }
  g_list_free(data->privateDt->files);
  data->privateDt->files = (GList*)0;
  visu_data_setNSubset(data, 1);
}

/**
 * visu_data_getFile:
 * @data: a #VisuData object.
 * @kind: an integer to qualify the required file ;
 * @format: a location for a file format (can be NULL).
 *
 * This prototype is used to retrieve stored
 * files identify by their @kind.
 *
 * Returns: the name of a file (it should not be deleted).
 */
gchar* visu_data_getFile(VisuData *data, int kind, ToolFileFormat **format)
{
  GList *lst;

  g_return_val_if_fail(IS_VISU_DATA_TYPE(data), (gchar*)0);

  DBG_fprintf(stderr, "Visu Data: retrieve filename with key %d"
	      " from the VisuData %p.\n", kind, (gpointer)data);
  if (format)
    *format = (ToolFileFormat*)0;
  for (lst = data->privateDt->files; lst; lst = g_list_next(lst))
    if (((struct FileDescription_struct*)lst->data)->kind == kind)
      {
	if (format)
	  *format = ((struct FileDescription_struct*)lst->data)->format;
	return ((struct FileDescription_struct*)lst->data)->name;
      }
  return (gchar*)0;
}

/**
 * visu_data_getFilesAsLabel:
 * @data: a #VisuData object.
 *
 * Creates a label using the list of files used to defined this @data
 * separated by dashes.
 *
 * Since: 3.6
 *
 * Returns: a newly created string with the filenames. If no filename
 * were used to defined @data, the function returns NULL.
 */
gchar* visu_data_getFilesAsLabel(VisuData *data)
{
  GList *lst;
  GString *str;
  gchar *s;

  g_return_val_if_fail(IS_VISU_DATA_TYPE(data), (gchar*)0);

  if (!data->privateDt->files)
    return (gchar*)0;

  DBG_fprintf(stderr, "Visu Data: retrieve all filenames"
	      " from the VisuData %p.\n", (gpointer)data);
  str = g_string_new("");
  for (lst = data->privateDt->files->next; lst; lst = g_list_next(lst))
    {
      s = g_path_get_basename(((struct FileDescription_struct*)lst->data)->name);
      g_string_append_printf(str, "\302\240\342\200\224 %s", s);
      g_free(s);
    }
  s = g_path_get_basename(((struct FileDescription_struct*)data->privateDt->files->data)->name);
  g_string_prepend(str, s);
  g_free(s);
  return g_string_free(str, FALSE);
}

/**
 * visu_data_setFileCommentary:
 * @data: a #VisuData object ;
 * @commentary: the message to be stored (null terminated) ;
 * @iSet: an integer.
 *
 * This method is used to store a description of the given @data. This
 * string is copied and @commentary can be freed. Before using this
 * method, the number of possible node sets must have been defined
 * using visu_data_setNSubset(), if not, only iSet == 0 is allowed.
 */
void visu_data_setFileCommentary(VisuData *data, gchar* commentary, gint iSet)
{
  g_return_if_fail(IS_VISU_DATA_TYPE(data) &&
		   iSet >= 0 && iSet < data->privateDt->nSets);
  
  g_free(data->privateDt->commentary[iSet]);
  data->privateDt->commentary[iSet] = g_strdup(commentary);
}

/**
 * visu_data_getFileCommentary:
 * @data: a #VisuData object ;
 * @iSet: an integer (>= 0).
 *
 * Get the commentary associated to the given @data, for the given
 * node set.
 *
 * Returns: a string description (possibly empty). This string is own by V_Sim
 *          and should not be freed.
 */
gchar* visu_data_getFileCommentary(VisuData *data, gint iSet)
{
  g_return_val_if_fail(IS_VISU_DATA_TYPE(data) &&
		       iSet >= 0 && iSet < data->privateDt->nSets, (gchar*)0);
  
  return data->privateDt->commentary[iSet];
}

/**
 * visu_data_setNSubset:
 * @data: a #VisuData object ;
 * @nSet: an integer.
 *
 * Change the number of available sets of nodes for this
 * #VisuData. This has a side effect to delete all previously saved
 * file commentaries (see visu_data_setFileCommentary()).
 */
void visu_data_setNSubset(VisuData *data, int nSet)
{
  int i;

  g_return_if_fail(IS_VISU_DATA_TYPE(data) && nSet > 0);

  data->privateDt->nSets = nSet;
  if (data->privateDt->commentary)
    g_strfreev(data->privateDt->commentary);
  data->privateDt->commentary = g_malloc(sizeof(gchar*) * (nSet + 1));
  for (i = 0; i < nSet; i++)
    data->privateDt->commentary[i] = g_strdup("");
  data->privateDt->commentary[nSet] = (gchar*)0;
}

/**
 * visu_data_getNSubset:
 * @data: a #VisuData object.
 *
 * Retrieve the number of available sets of nodes for this #VisuData,
 * see visu_data_setNSubset().
 *
 * Returns: the number of set of nodes (1 is default).
 */

int visu_data_getNSubset(VisuData *data)
{
  g_return_val_if_fail(IS_VISU_DATA_TYPE(data), 1);

  return data->privateDt->nSets;
}

/**
 * visu_data_setISubset:
 * @data: a #VisuData object ;
 * @iSet: an integer.
 *
 * Change the current id of the set of data (ordered as in C,
 * beginning at 0).
 */
void visu_data_setISubset(VisuData *data, int iSet)
{
  g_return_if_fail(IS_VISU_DATA_TYPE(data));
  DBG_fprintf(stderr, "Visu Data: set id to %d (%d).\n",
	      iSet, data->privateDt->nSets);

  g_return_if_fail(iSet >= 0 && iSet < data->privateDt->nSets);

  data->privateDt->iSet = iSet;
}

/**
 * visu_data_getISubset:
 * @data: a #VisuData object.
 *
 * Retrieve the id of the current set of data (ordered as in C,
 * beginning at 0).
 *
 * Returns: the id of the set of nodes currently loaded, -1 if none.
 */
int visu_data_getISubset(VisuData *data)
{
  g_return_val_if_fail(IS_VISU_DATA_TYPE(data), -1);

  DBG_fprintf(stderr, "Visu Data: get current set id %d.\n", data->privateDt->iSet);
  return data->privateDt->iSet;
}

/**
 * visu_data_compareElements:
 * @data1: a #VisuData object ;
 * @data2: an other #VisuData object.
 *
 * This method is used to compare the composition of the given two #VisuData objects.
 * The test is only done on #VisuElement lists.
 *
 * Returns: TRUE if the two objects contains exactly the same #VisuElement objects (not
 *          one more or one less or one different), FALSE otherwise.
 */
gboolean visu_data_compareElements(VisuData *data1, VisuData *data2)
{
  unsigned int i;

  g_return_val_if_fail(data1 && data2, FALSE);

  DBG_fprintf(stderr, "Visu Data: comparing composition of '%p' and '%p'.\n",
	      (gpointer)data1, (gpointer)data2);
  if (data1 == data2)
    return FALSE;

  if (data1->ntype != data2->ntype)
    return FALSE;

  for (i = 0; i< data1->ntype; i++)
    if (!g_hash_table_lookup(data2->fromVisuElementToInt,
			     (gpointer)data1->fromIntToVisuElement[i]))
      return FALSE;

  for (i = 0; i< data2->ntype; i++)
    if (!g_hash_table_lookup(data1->fromVisuElementToInt,
			     (gpointer)data2->fromIntToVisuElement[i]))
      return FALSE;

  return TRUE;
}

/**
 * visu_data_setChangeElementFlag:
 * @data: a #VisuData object ;
 * @changeElement: a boolean.
 *
 * This method is mainly used by internal gears to set a flag. This flag control
 * if the @data object has the same #VisuElement objects than the previously rendered one.
 */
void visu_data_setChangeElementFlag(VisuData *data, gboolean changeElement)
{
  gboolean *val;

  g_return_if_fail(data);

  /* Test if already exists */
  val = (gboolean*)g_object_get_data(G_OBJECT(data), "changeElementListFlag");
  if (!val)
    {
      val = g_malloc(sizeof(gboolean));
      g_object_set_data_full(G_OBJECT(data), "changeElementListFlag",
			     val, g_free);
    }
  *val = changeElement;
}

/**
 * visu_data_getChangeElementFlag:
 * @data: a #VisuData object.
 *
 * V_Sim can use a flag set on @data object to know if @data has exactly the same
 * #VisuElement list than the previously rendered one.
 *
 * Returns: TRUE if the previously rendered #VisuData object has had the same
 *          #VisuElement list than the given one, FALSE otherwise.
 */
gboolean visu_data_getChangeElementFlag(VisuData *data)
{
  gboolean *val;

  g_return_val_if_fail(data, FALSE);

  val = (gboolean*)g_object_get_data(G_OBJECT(data), "changeElementListFlag");
  if (val)
    return *val;
  else
    return FALSE;
}


/*************************/
/* The geometry routines */
/*************************/

/**
 * visu_data_getXYZtranslation:
 * @data: a #VisuData object.
 *
 * The nodes are rendered at thier coordinates plus a translation. This method
 * allows to retrieve that translation.
 *
 * Returns: (array fixed-size=3): a newly allocated array of 3 floats. It should be freed with a call
 *          to free() after use.
 */
float* visu_data_getXYZtranslation(VisuData* data)
{
  float *trans;
  int i;

  g_return_val_if_fail(IS_VISU_DATA_TYPE(data), (float*)0);

  trans = g_malloc(sizeof(float) * 3);
  for (i = 0; i < 3; i++)
    trans[i] = data->privateDt->translation[i];
  return trans;
}

/**
 * visu_data_setXYZtranslation:
 * @data: a #VisuData object ;
 * @xyz: (in) (array fixed-size=3): an array of floating point values.
 *
 * This set the translations of the specified #VisuData whatever previous values.
 * The translation is done in the orthonormal referential, not the referential of
 * the box.
 *
 * Returns: if returns TRUE, visu_data_createAllNodes() should be called and
 *          the 'NodePositionChanged' should be emitted.
 */
gboolean visu_data_setXYZtranslation(VisuData* data, float xyz[3])
{
  gboolean res;

  g_return_val_if_fail(IS_VISU_DATA_TYPE(data) && xyz, 0);

  res = FALSE;
  if (data->privateDt->translation[0] != xyz[0] &&
      data->privateDt->bc != VISU_DATA_BOX_FREE &&
      data->privateDt->bc != VISU_DATA_BOX_SURFACE_YZ)
    {
      data->privateDt->translation[0] = xyz[0];
      res = TRUE;
    }
  if (data->privateDt->translation[1] != xyz[1] &&
      data->privateDt->bc != VISU_DATA_BOX_FREE &&
      data->privateDt->bc != VISU_DATA_BOX_SURFACE_ZX)
    {
      data->privateDt->translation[1] = xyz[1];
      res = TRUE;
    }
  if (data->privateDt->translation[2] != xyz[2] &&
      data->privateDt->bc != VISU_DATA_BOX_FREE &&
      data->privateDt->bc != VISU_DATA_BOX_SURFACE_ZX)
    {
      data->privateDt->translation[2] = xyz[2];
      res = TRUE;
    }
  DBG_fprintf(stderr, "Visu Data: set translation to: %f %f %f\n",
	      data->privateDt->translation[0], data->privateDt->translation[1],
	      data->privateDt->translation[2]);

  return res;
}

/**
 * visu_data_getBoxMatrix:
 * @data: a #VisuData object ;
 * @matrix: (inout) (array fixed-size=3) (transfer full): an area to store the matrix.
 *
 * This method is used when the box matrix is required. This matrix can transform
 * a vector given in box coordinates into a cartesian vector. If a simple vector
 * multication is required, then the use of visu_data_convertBoxCoordinatestoXYZ()
 * should be prefered.
 */
void visu_data_getBoxMatrix(VisuData *data, float matrix[3][3])
{
  int i, j;

  g_return_if_fail(IS_VISU_DATA_TYPE(data) && matrix);

  for (i = 0; i < 3; i++)
    for (j = 0; j < 3; j++)
      matrix[i][j] = (float)data->privateDt->fromBoxtoXYZD0[i][j];
}
/**
 * visu_data_getBoxMatrixD0:
 * @data: a #VisuData object ;
 * @matrix: (inout) (array fixed-size=3) (transfer full): an area to
 * store the matrix.
 *
 * Same as visu_data_getBoxMatrix() but with a double precision
 * matrix.
 */
void visu_data_getBoxMatrixD0(VisuData *data, double matrix[3][3])
{
  int i, j;

  g_return_if_fail(IS_VISU_DATA_TYPE(data) && matrix);

  for (i = 0; i < 3; i++)
    for (j = 0; j < 3; j++)
      matrix[i][j] = data->privateDt->fromBoxtoXYZD0[i][j];
}

/**
 * visu_data_getBoxVertices:
 * @data: a #VisuData object.
 * @v: (in) (array): the position of the eight vertices of the bounding box.
 * @withExtension: a boolean.
 *
 * All nodes are rendered inside a bounding box, this method can be used to retrieve
 * it. This box is not the drawn box but the box containing all the
 * nodes, included possible extension. To get the box itself, use
 * visu_data_getBoxMatrix() instead. One can also get the vertices of
 * the box itself using FALSE as @withExtension argument.
 */
void visu_data_getBoxVertices(VisuData *data, float v[8][3],
                              gboolean withExtension)
{
  float transX[3], transY[3], transZ[3], ext[3];
  double *boxGeometry;

  g_return_if_fail(IS_VISU_DATA_TYPE(data));

  if (withExtension)
    {
      ext[0] = data->privateDt->extension[0];
      ext[1] = data->privateDt->extension[1];
      ext[2] = data->privateDt->extension[2];
    }
  else
    {
      ext[0] = 0.f;
      ext[1] = 0.f;
      ext[2] = 0.f;
    }
  DBG_fprintf(stderr, "Visu Data: get the box vertices with"
	      " extension (%g;%g;%g).\n", ext[0], ext[1], ext[2]);

  boxGeometry = data->privateDt->boxGeometry;
  transX[0] = ext[0] * boxGeometry[0];
  transX[1] = 0.f;
  transX[2] = 0.f;
  transY[0] = ext[1] * boxGeometry[1];
  transY[1] = ext[1] * boxGeometry[2];
  transY[2] = 0.f;
  transZ[0] = ext[2] * boxGeometry[3];
  transZ[1] = ext[2] * boxGeometry[4];
  transZ[2] = ext[2] * boxGeometry[5];

  /* [0;0;0] */
  v[0][0] = 0.f - transX[0] - transY[0] - transZ[0];
  v[0][1] = 0.f - transX[1] - transY[1] - transZ[1];
  v[0][2] = 0.f - transX[2] - transY[2] - transZ[2];

  /* [1;0;0] */
  v[1][0] = boxGeometry[0] + transX[0] - transY[0] - transZ[0];
  v[1][1] = 0.f            + transX[1] - transY[1] - transZ[1];
  v[1][2] = 0.f            + transX[2] - transY[2] - transZ[2];

  /* [0;1;0] */
  v[3][0] = boxGeometry[1] - transX[0] + transY[0] - transZ[0];
  v[3][1] = boxGeometry[2] - transX[1] + transY[1] - transZ[1];
  v[3][2] = 0.f            - transX[2] + transY[2] - transZ[2];

  /* [0;0;1] */
  v[4][0] = boxGeometry[3] - transX[0] - transY[0] + transZ[0];
  v[4][1] = boxGeometry[4] - transX[1] - transY[1] + transZ[1];
  v[4][2] = boxGeometry[5] - transX[2] - transY[2] + transZ[2];

  /* [1;1;0] */
  v[2][0] = boxGeometry[0] + boxGeometry[1] + transX[0] + transY[0] - transZ[0];
  v[2][1] = 0.f            + boxGeometry[2] + transX[1] + transY[1] - transZ[1];
  v[2][2] = 0.f            + 0.f            + transX[2] + transY[2] - transZ[2];
      
  /* [1;0;1] */
  v[5][0] = boxGeometry[3] + boxGeometry[0] + transX[0] - transY[0] + transZ[0];
  v[5][1] = boxGeometry[4] + 0.f            + transX[1] - transY[1] + transZ[1];
  v[5][2] = boxGeometry[5] + 0.f            + transX[2] - transY[2] + transZ[2];
      
  /* [1;1;1] */
  v[6][0] = boxGeometry[3] + boxGeometry[0] + boxGeometry[1] + transX[0] + transY[0] + transZ[0];
  v[6][1] = boxGeometry[4] + boxGeometry[2] + transX[1] + transY[1] + transZ[1];
  v[6][2] = boxGeometry[5] + 0.f            + transX[2] + transY[2] + transZ[2];
      
  /* [0;1;1] */
  v[7][0] = boxGeometry[3] + boxGeometry[1] - transX[0] + transY[0] + transZ[0];
  v[7][1] = boxGeometry[4] + boxGeometry[2] - transX[1] + transY[1] + transZ[1];
  v[7][2] = boxGeometry[5] + 0.f            - transX[2] + transY[2] + transZ[2];
}

/**
 * visu_data_getBoxSpan:
 * @data: an #VisuData object;
 * @span: (in) (array fixed-size=2): a location to store two floats.
 *
 * Compute the two most far distance from origin in the box. The first
 * is the negative one and the second, the positive one.
 */
void visu_data_getBoxSpan(VisuData *data, float span[2])
{
  float diag[3], ext[3];

  g_return_if_fail(IS_VISU_DATA_TYPE(data));

  ext[0]  = data->privateDt->extension[0] * data->privateDt->extension[0];
  ext[1]  = data->privateDt->extension[1] * data->privateDt->extension[1];
  ext[2]  = data->privateDt->extension[2] * data->privateDt->extension[2];
  diag[0] = data->privateDt->boxGeometry[0] +
    data->privateDt->boxGeometry[1] + data->privateDt->boxGeometry[3];
  diag[0] *= diag[0];
  diag[1] = data->privateDt->boxGeometry[2] + data->privateDt->boxGeometry[4];
  diag[1] *= diag[1];
  diag[2] = data->privateDt->boxGeometry[5];
  diag[2] *= diag[2];
  span[0] = -sqrt(ext[0] * diag[0] + ext[1] * diag[1] + ext[2] * diag[2]);
  span[1] = sqrt((data->privateDt->extension[0] + 1.f) *
		 (data->privateDt->extension[0] + 1.f) * diag[0] +
		 (data->privateDt->extension[1] + 1.f) *
		 (data->privateDt->extension[1] + 1.f) * diag[1] +
		 (data->privateDt->extension[2] + 1.f) *
		 (data->privateDt->extension[2] + 1.f) * diag[2]);
}

/**
 * visu_data_convertFullCartesianToSpherical:
 * @data: a #VisuData object.
 * @sph: (out caller-allocates) (array fixed-size=3):
 * @xyz: (in) (array fixed-size=3):
 *
 * Convert given cartesian coordinates to spherical ones. If the box has been rotated
 * (see visu_data_setBoxGeometryFull()), then the cartesian coordinates should
 * be the one before the rotation.
 *
 * Since: 3.6
 */
void visu_data_convertFullCartesianToSpherical(VisuData *data, float sph[3], float xyz[3])
{
  float coord[3];

  g_return_if_fail(IS_VISU_DATA_TYPE(data));

  if (data->privateDt->cartesiansAreFull)
    tool_matrix_productVector(coord, data->privateDt->fromFullToLocal, xyz);
  else
    {
      coord[0] = xyz[0];
      coord[1] = xyz[1];
      coord[2] = xyz[2];
    }
  tool_matrix_cartesianToSpherical(sph, coord);
}

/**
 * visu_data_convertXYZtoBoxCoordinates: (skip)
 * @data: a #VisuData object ;
 * @boxCoord: (in) (array fixed-size=3):an array of floating point values to store the result ;
 * @xyz: (in) (array fixed-size=3): an array of floating point values describing coordinates in cartesian.
 *
 * Use this method to transform cartesian coordinates to the box coordinates.
 */
void visu_data_convertXYZtoBoxCoordinates(VisuData *data, float boxCoord[3], float xyz[3])
{
  int i, j;

  g_return_if_fail(IS_VISU_DATA_TYPE(data) && boxCoord && xyz);

  for (i = 0; i < 3; i++)
    {
      boxCoord[i] = 0.;
      for (j = 0; j < 3; j++)
	boxCoord[i] += (float)data->privateDt->fromXYZtoBoxD0[i][j] * xyz[j];
    }
}
/**
 * visu_data_convertXYZToReduced:
 * @data: a #VisuData object ;
 * @xyz: (in) (array fixed-size=3) (element-type gfloat): floating
 * point values that describes the cartesian coordinates.
 * @u: (out caller-allocates): the x coordinate.
 * @v: (out caller-allocates): the y coordinate.
 * @w: (out caller-allocates): the z coordinate.
 *
 * Use this method to transform cartesian into box coordinates.
 *
 * Since: 3.6
 */
void visu_data_convertXYZToReduced(VisuData *data, GArray *xyz,
				   float *u, float *v, float *w)
{
  float red_[3];

  g_return_if_fail(xyz && u && v && w && xyz->len == 3);

  visu_data_convertXYZtoBoxCoordinates(data, red_, (float*)xyz->data);
  *u = red_[0]; *v = red_[1]; *w = red_[2];
}

/**
 * visu_data_convertBoxCoordinatestoXYZ:
 * @data: a #VisuData object ;
 * @xyz: (out caller-allocates) (array fixed-size=3): an array of floating point values to store the result ;
 * @boxCoord: (in) (array fixed-size=3): an array of floating point values that describes the box coordinates.
 *
 * Use this method to transform box coordinates into cartesian.
 */
void visu_data_convertBoxCoordinatestoXYZ(VisuData *data, float xyz[3], float boxCoord[3])
{
  int i, j;

  g_return_if_fail(IS_VISU_DATA_TYPE(data) && boxCoord && xyz);

  for (i = 0; i < 3; i++)
    {
      xyz[i] = 0.;
      for (j = 0; j < 3; j++)
	xyz[i] += (float)data->privateDt->fromBoxtoXYZD0[i][j] * boxCoord[j];
    }
}
/**
 * visu_data_convertReducedToXYZ:
 * @data: a #VisuData object ;
 * @red: (in) (array fixed-size=3) (element-type gfloat): floating
 * point values that describes the cartesian coordinates.
 * @x: (out caller-allocates): the x coordinate.
 * @y: (out caller-allocates): the y coordinate.
 * @z: (out caller-allocates): the z coordinate.
 *
 * Use this method to transform box coordinates into cartesian.
 *
 * Since: 3.6
 */
void visu_data_convertReducedToXYZ(VisuData *data, GArray *red,
				   float *x, float *y, float *z)
{
  float xyz_[3];

  g_return_if_fail(red && x && y && z && red->len == 3);

  visu_data_convertBoxCoordinatestoXYZ(data, xyz_, (float*)red->data);
  *x = xyz_[0]; *y = xyz_[1]; *z = xyz_[2];
}

/**
 * visu_data_getBoundaryConditions:
 * @data: a #VisuData object.
 *
 * Get the boundary conditions defined for @data.
 *
 * Since: 3.5
 *
 * Returns: a #VisuDataBoxBoundaries flag.
 */
VisuDataBoxBoundaries visu_data_getBoundaryConditions(VisuData *data)
{
  g_return_val_if_fail(IS_VISU_DATA_TYPE(data), VISU_DATA_BOX_PERIODIC);

  return data->privateDt->bc;
}

/**
 * visu_data_setBoxGeometry:
 * @data: a #VisuData object ;
 * @geometry:  (in) (array fixed-size=6):a 6 floating point array ;
 * @bc: define the boundary conditions.
 *
 * This methods set the size of the box. The boundary conditions can
 * be specified thanks to @bc.
 */
void visu_data_setBoxGeometry(VisuData *data, double geometry[6],
			     VisuDataBoxBoundaries bc)
{
  int i;
  double normY, normZ;

  g_return_if_fail(IS_VISU_DATA_TYPE(data) && geometry);

  data->privateDt->bc = bc;
  switch (bc)
    {
    case (VISU_DATA_BOX_PERIODIC):
    case (VISU_DATA_BOX_FREE):
    case (VISU_DATA_BOX_SURFACE_XY):
      openGLViewSet_upAxis(data->privateDt->attachedView, VIEW_Z);
      break;
    case (VISU_DATA_BOX_SURFACE_YZ):
      openGLViewSet_upAxis(data->privateDt->attachedView, VIEW_X);
      break;
    case (VISU_DATA_BOX_SURFACE_ZX):
      openGLViewSet_upAxis(data->privateDt->attachedView, VIEW_Y);
      break;
    }
  for (i = 0; i < 6; i++)
    data->privateDt->boxGeometry[i] = geometry[i];
  normY = sqrt(geometry[VISU_DATA_BOX_DYX] * geometry[VISU_DATA_BOX_DYX] +
	       geometry[VISU_DATA_BOX_DYY] * geometry[VISU_DATA_BOX_DYY]);
  normZ = sqrt(geometry[VISU_DATA_BOX_DZX] * geometry[VISU_DATA_BOX_DZX] +
	       geometry[VISU_DATA_BOX_DZY] * geometry[VISU_DATA_BOX_DZY] +
	       geometry[VISU_DATA_BOX_DZZ] * geometry[VISU_DATA_BOX_DZZ]);
  data->privateDt->boxNorm[0] = 1.;
  for (i = 1; i < 3; i++)
    data->privateDt->boxNorm[i] = geometry[i] / normY;
  for (i = 3; i < 6; i++)
    data->privateDt->boxNorm[i] = geometry[i] / normZ;
  data->privateDt->dxxs2 = 0.5 * (geometry[VISU_DATA_BOX_DXX] +
				  geometry[VISU_DATA_BOX_DYX] +
				  geometry[VISU_DATA_BOX_DZX]);
  data->privateDt->dyys2 = 0.5 * (geometry[VISU_DATA_BOX_DYY] +
				  geometry[VISU_DATA_BOX_DZY]);
  data->privateDt->dzzs2 = 0.5 * (geometry[VISU_DATA_BOX_DZZ]);

  DBG_fprintf(stderr, "Visu Data: the bounding box is set to:\n %f %f %f\n %f %f %f\n",
	      geometry[0], geometry[1], geometry[2], geometry[3],
	      geometry[4], geometry[5]);

  /* Create the transformation matrix. */
  data->privateDt->fromXYZtoBoxD0[0][0] =
    1. / data->privateDt->boxGeometry[VISU_DATA_BOX_DXX];
  data->privateDt->fromXYZtoBoxD0[0][1] =
    - data->privateDt->boxGeometry[VISU_DATA_BOX_DYX] /
    data->privateDt->boxGeometry[VISU_DATA_BOX_DXX] /
    data->privateDt->boxGeometry[VISU_DATA_BOX_DYY];
  data->privateDt->fromXYZtoBoxD0[0][2] =
    - (data->privateDt->boxGeometry[VISU_DATA_BOX_DZX] /
       data->privateDt->boxGeometry[VISU_DATA_BOX_DXX] -
       data->privateDt->boxGeometry[VISU_DATA_BOX_DYX] *
       data->privateDt->boxGeometry[VISU_DATA_BOX_DZY] / 
       data->privateDt->boxGeometry[VISU_DATA_BOX_DXX] / 
       data->privateDt->boxGeometry[VISU_DATA_BOX_DYY] ) /
    data->privateDt->boxGeometry[VISU_DATA_BOX_DZZ];
  data->privateDt->fromXYZtoBoxD0[1][0] = 0.;
  data->privateDt->fromXYZtoBoxD0[1][1] =
    1. / data->privateDt->boxGeometry[VISU_DATA_BOX_DYY];
  data->privateDt->fromXYZtoBoxD0[1][2] =
    - data->privateDt->boxGeometry[VISU_DATA_BOX_DZY] /
    data->privateDt->boxGeometry[VISU_DATA_BOX_DYY] /
    data->privateDt->boxGeometry[VISU_DATA_BOX_DZZ];
  data->privateDt->fromXYZtoBoxD0[2][0] = 0.;
  data->privateDt->fromXYZtoBoxD0[2][1] = 0.;
  data->privateDt->fromXYZtoBoxD0[2][2] = 1. /
    data->privateDt->boxGeometry[VISU_DATA_BOX_DZZ];

  data->privateDt->fromBoxtoXYZD0[0][0] =
    data->privateDt->boxGeometry[VISU_DATA_BOX_DXX];
  data->privateDt->fromBoxtoXYZD0[0][1] =
    data->privateDt->boxGeometry[VISU_DATA_BOX_DYX];
  data->privateDt->fromBoxtoXYZD0[0][2] =
    data->privateDt->boxGeometry[VISU_DATA_BOX_DZX];
  data->privateDt->fromBoxtoXYZD0[1][0] = 0.;
  data->privateDt->fromBoxtoXYZD0[1][1] =
    data->privateDt->boxGeometry[VISU_DATA_BOX_DYY];
  data->privateDt->fromBoxtoXYZD0[1][2] =
    data->privateDt->boxGeometry[VISU_DATA_BOX_DZY];
  data->privateDt->fromBoxtoXYZD0[2][0] = 0.;
  data->privateDt->fromBoxtoXYZD0[2][1] = 0.;
  data->privateDt->fromBoxtoXYZD0[2][2] =
    data->privateDt->boxGeometry[VISU_DATA_BOX_DZZ];

  DBG_fprintf(stderr, "Visu Data: set box geometry done.\n");
}
/**
 * visu_data_setBoxGeometryFull:
 * @data: a #VisuData object ;
 * @geometry: (in) (array fixed-size=9): a 9 floating point array ;
 * @bc: define the boundary conditions.
 *
 * This methods set the size of the box. The boundary conditions can
 * be specified thanks to @bc. The basis set will be rotated by V_Sim
 * to align the given first axis with X, and put the second axis in
 * the XY plane. Further inclusion of nodes with visu_data_addNodeFromElement()
 * and friends will automatically apply the rotation if coordinates are
 * given in cartesian. This rotation will stop after any call to
 * visu_data_applyBoxGeometry().
 *
 * Returns: FALSE if the given box is planar.
 * 
 * Since: 3.6
 */
gboolean visu_data_setBoxGeometryFull(VisuData *data, double geometry[3][3],
                                      VisuDataBoxBoundaries bc)
{
  double box[6];

  g_return_val_if_fail(IS_VISU_DATA_TYPE(data), FALSE);

  if (!tool_matrix_reducePrimitiveVectors(box, geometry))
    return FALSE;
  visu_data_setBoxGeometry(data, box, bc);
  data->privateDt->cartesiansAreFull = TRUE;
  if (!tool_matrix_getRotationFromFull(data->privateDt->fromFullToLocal, geometry, box))
    return FALSE;

  return TRUE;
}
/**
 * visu_data_applyBoxGeometry:
 * @data: a #VisuData object.
 *
 * The box expansion is calculated. From this call, all cartesian
 * coordinates are treated in V_Sim box definition and not a possible
 * full definition.
 */
void visu_data_applyBoxGeometry(VisuData *data)
{
  g_return_if_fail(IS_VISU_DATA_TYPE(data));
  /* From here, all cartesian coordinates are treated in V_Sim
     box definition and not a possible full definition. */
  data->privateDt->cartesiansAreFull = FALSE;
  /* Set the box definition and raise change signals. */
  applyBox(data, visu_data_getAllElementExtens(data) + getAllNodeExtens(data));
}
static void applyBox(VisuData *data, float margin)
{
  float dz2, dy2, dx, dy, dz, su, sc, ext[3];

  g_return_if_fail(IS_VISU_DATA_TYPE(data));

  /* We update the extension of the OpenGL box. */
  DBG_fprintf(stderr, "Visu Data: change box (margin = %f).\n",
	      margin);
  /* Calculate box length and margin. */
  if (margin >= 0.f)
    data->privateDt->margin = margin;

  /* calculate bare = 1/2 radius of centered sample */
  dz = data->privateDt->boxGeometry[VISU_DATA_BOX_DZZ];
  dz2 = dz * dz;
  dy = (data->privateDt->boxGeometry[VISU_DATA_BOX_DYY] +
	data->privateDt->boxGeometry[VISU_DATA_BOX_DZY]);
  dy2 = dy * dy;
  dx = (data->privateDt->boxGeometry[VISU_DATA_BOX_DXX] +
	data->privateDt->boxGeometry[VISU_DATA_BOX_DYX] +
	data->privateDt->boxGeometry[VISU_DATA_BOX_DZX]);
  su = dx * dx + dy2 + dz2;
  dx = (-data->privateDt->boxGeometry[VISU_DATA_BOX_DXX] +
	data->privateDt->boxGeometry[VISU_DATA_BOX_DYX] +
	data->privateDt->boxGeometry[VISU_DATA_BOX_DZX]);
  sc = dx * dx + dy2 + dz2;
  if (sc > su) su = sc;
  dx = (data->privateDt->boxGeometry[VISU_DATA_BOX_DXX] -
	data->privateDt->boxGeometry[VISU_DATA_BOX_DYX] +
	data->privateDt->boxGeometry[VISU_DATA_BOX_DZX]);
  dy = (data->privateDt->boxGeometry[VISU_DATA_BOX_DYY] -
	data->privateDt->boxGeometry[VISU_DATA_BOX_DZY]);
  dy2 = dy * dy;
  sc = dx * dx + dy2 + dz2;
  if (sc > su) su = sc;
  dx = (data->privateDt->boxGeometry[VISU_DATA_BOX_DXX] +
	data->privateDt->boxGeometry[VISU_DATA_BOX_DYX] -
	data->privateDt->boxGeometry[VISU_DATA_BOX_DZX]);
  sc = dx * dx + dy2 + dz2;
  if (sc > su) su = sc;
  data->privateDt->extens[0] = sqrt(su) * 0.5f;
  ext[0] = (1. + 2. * data->privateDt->extension[0]);
  ext[1] = (1. + 2. * data->privateDt->extension[1]);
  ext[2] = (1. + 2. * data->privateDt->extension[2]);
  dz = data->privateDt->boxGeometry[VISU_DATA_BOX_DZZ] * ext[2];
  dz2 = dz * dz;
  dy = (data->privateDt->boxGeometry[VISU_DATA_BOX_DYY] * ext[1] +
	data->privateDt->boxGeometry[VISU_DATA_BOX_DZY] * ext[2]);
  dy2 = dy * dy;
  dx = (data->privateDt->boxGeometry[VISU_DATA_BOX_DXX] * ext[0] +
	data->privateDt->boxGeometry[VISU_DATA_BOX_DYX] * ext[1] +
	data->privateDt->boxGeometry[VISU_DATA_BOX_DZX] * ext[2]);
  su = dx * dx + dy2 + dz2;
  dx = (-data->privateDt->boxGeometry[VISU_DATA_BOX_DXX] * ext[0] +
	data->privateDt->boxGeometry[VISU_DATA_BOX_DYX] * ext[1] +
	data->privateDt->boxGeometry[VISU_DATA_BOX_DZX] * ext[2]);
  sc = dx * dx + dy2 + dz2;
  if (sc > su) su = sc;
  dx = (data->privateDt->boxGeometry[VISU_DATA_BOX_DXX] * ext[0] -
	data->privateDt->boxGeometry[VISU_DATA_BOX_DYX] * ext[1] +
	data->privateDt->boxGeometry[VISU_DATA_BOX_DZX] * ext[2]);
  dy = (data->privateDt->boxGeometry[VISU_DATA_BOX_DYY] * ext[1] -
	data->privateDt->boxGeometry[VISU_DATA_BOX_DZY] * ext[2]);
  dy2 = dy * dy;
  sc = dx * dx + dy2 + dz2;
  if (sc > su) su = sc;
  dx = (data->privateDt->boxGeometry[VISU_DATA_BOX_DXX] * ext[0] +
	data->privateDt->boxGeometry[VISU_DATA_BOX_DYX] * ext[1] -
	data->privateDt->boxGeometry[VISU_DATA_BOX_DZX] * ext[2]);
  sc = dx * dx + dy2 + dz2;
  if (sc > su) su = sc;
  data->privateDt->extens[1] = sqrt(su) * 0.5f + data->privateDt->margin;
  DBG_fprintf(stderr, " | extens = %g / %g.\n", data->privateDt->extens[0],
	      data->privateDt->extens[1]);

  DBG_fprintf(stderr, "Visu Data: emit BoxSizeChanged.\n");
  g_signal_emit(data, visu_data_signals[BOX_SIZE_CHANGED_SIGNAL],
		0 , NULL);
  DBG_fprintf(stderr, "Visu Data: emission done (BoxSizeChanged).\n");
}

/**
 * visu_data_getBoxGeometry:
 * @data: a #VisuData object ;
 * @vector: an int corresponding to a vector of the box.
 *
 * Retrieve the value of a vector defining the bounding box. The vector
 * is chosen with an int, see the #visu_data_boxVector enum for more
 * details.
 *
 * Returns: the value of the required vector (always a positive value for vector = 0, 2 or 5 !).
 */
float visu_data_getBoxGeometry(VisuData *data, int vector)
{
  g_return_val_if_fail(IS_VISU_DATA_TYPE(data) && vector >= 0 && vector < 6, -1);

  return (float)data->privateDt->boxGeometry[vector];
}

/**
 * visu_data_getNodeBoxFromNumber:
 * @data: a #VisuData object.
 * @nodeId: the index of the node considered.
 * @nodeBox: (in) (array fixed-size=3): the array to store the box of the node.
 *
 * This method retrieves the value of the box associated to a node (with respect to the unit cell).
 * 
 * Returns: TRUE if everything went well, FALSE otherwise. The box is stored in the nodeBox array.
 */
gboolean visu_data_getNodeBoxFromNumber(VisuData *data, guint nodeId, int nodeBox[3])
{
  float xred[3];
  float xcart[3];
  VisuNodeArray *visu_node_Array = visu_data_getNodeArray(data);

  g_return_val_if_fail(IS_VISU_DATA_TYPE(data) && (nodeId < visu_node_Array->idCounter), FALSE);
  visu_data_getNodePosition(data, visu_data_getNodeFromNumber(data,nodeId), xcart);
  visu_data_convertXYZtoBoxCoordinates(data, xred, xcart);
  nodeBox[0] = floor(xred[0]);
  nodeBox[1] = floor(xred[1]);
  nodeBox[2] = floor(xred[2]);
  DBG_fprintf(stderr, "Visu Data: nodeBox found for atom %d : %d %d %d.\n", nodeId, nodeBox[0], nodeBox[1], nodeBox[2]);
  return TRUE;
}

/**
 * visu_data_getNodeBoxFromCoord:
 * @data: a #VisuData object.
 * @xcart: (in) (array fixed-size=3): the coordinates of a node.
 * @nodeBox: (in) (array fixed-size=3): the array to store the box of the node.
 *
 * This method retrieves the value of the box associated to the coordinates of the node (with respect to the unit cell).
 * 
 * Returns: TRUE if everything went well, FALSE otherwise. The box is stored in the nodeBox array.
 */
gboolean visu_data_getNodeBoxFromCoord(VisuData *data, float xcart[3], int nodeBox[3])
{
  float xred[3];
  visu_data_convertXYZtoBoxCoordinates(data, xred, xcart);
  nodeBox[0] = floor(xred[0]);
  nodeBox[1] = floor(xred[1]);
  nodeBox[2] = floor(xred[2]);
  DBG_fprintf(stderr, "Visu Data: nodeBox found for atom at %f %f %f : %d %d %d.\n",
		xcart[0], xcart[1], xcart[2], nodeBox[0], nodeBox[1], nodeBox[2]);
  return TRUE;
}
/**
 * visu_data_getUnit:
 * @data: a #VisuData object.
 *
 * The lengths of @data may be given in a certain unit using
 * visu_data_setUnit().
 *
 * Since: 3.5
 *
 * Returns: the #ToolUnits of @data or #unit_undefined.
 */
ToolUnits visu_data_getUnit(VisuData *data)
{
  gpointer unit_;

  unit_ = g_object_get_data(G_OBJECT(data), "unit");
  if (unit_)
    return GPOINTER_TO_INT(unit_);
  else
    return unit_undefined;
}
/**
 * visu_data_setUnit:
 * @data: a #VisuData object.
 * @unit: a #ToolUnits flag.
 *
 * The lengths of @data may be given in a certain unit by calling this
 * routine. If the unit is different from the previously defined, the
 * coordinate are scaled accordingly.
 *
 * Since: 3.5
 *
 * Returns: TRUE if the unit has been changed.
 */
gboolean visu_data_setUnit(VisuData *data, ToolUnits unit)
{
  ToolUnits unit_, unitLength0;
  double fact;
  double box[6];
  VisuDataIter iter;
  float length0;

  unit_ = visu_data_getUnit(data);
  DBG_fprintf(stderr, "Visu Data: set unit to %d (%d).\n", unit, unit_);
  if (unit_ == unit)
    return FALSE;

  g_return_val_if_fail(IS_VISU_DATA_TYPE(data), FALSE);

  g_object_set_data(G_OBJECT(data), "unit", GINT_TO_POINTER(unit));
  if (unit_ == unit_undefined || unit == unit_undefined)
    return TRUE;

  fact = (double)tool_physic_getUnitValueInMeter(unit_) /
    tool_physic_getUnitValueInMeter(unit);
  DBG_fprintf(stderr, "Visu Data: multiplying factor is %g.\n", fact);

  /* We do an homothety on the box. */
  box[0] = data->privateDt->boxGeometry[0] * fact;
  box[1] = data->privateDt->boxGeometry[1] * fact;
  box[2] = data->privateDt->boxGeometry[2] * fact;
  box[3] = data->privateDt->boxGeometry[3] * fact;
  box[4] = data->privateDt->boxGeometry[4] * fact;
  box[5] = data->privateDt->boxGeometry[5] * fact;
  visu_data_setBoxGeometry(data, box, data->privateDt->bc);

  /* We do an homothety on the nodes. */
  data->privateDt->translation[0] *= fact;
  data->privateDt->translation[1] *= fact;
  data->privateDt->translation[2] *= fact;
  visu_data_iterNew(data, &iter);
  for( visu_data_iterStart(data, &iter); iter.node;
       visu_data_iterNext(data, &iter))
    {
      iter.node->xyz[0] *= fact;
      iter.node->xyz[1] *= fact;
      iter.node->xyz[2] *= fact;
      iter.node->translation[0] *= fact;
      iter.node->translation[1] *= fact;
      iter.node->translation[2] *= fact;
    }

  visu_data_applyBoxGeometry(data);

  DBG_fprintf(stderr, "Visu Data: emit UnitChanged.\n");
  g_signal_emit(data, visu_data_signals[UNIT_CHANGED_SIGNAL],
		0, fact, NULL);
  DBG_fprintf(stderr, "Visu Data: emission done (UnitChanged).\n");

  /* TODO: To be put elsewhere */
  length0 = openGLCameraGet_refLength(data->privateDt->attachedView->camera, &unitLength0);
  if (unitLength0 != unit_undefined)
    fact = (double)tool_physic_getUnitValueInMeter(unitLength0) /
      tool_physic_getUnitValueInMeter(unit);
  openGLCameraSet_refLength(data->privateDt->attachedView->camera, length0 * (float)fact, unit);
  openGLProject(data->privateDt->attachedView->window,
		data->privateDt->attachedView->camera,
		data->privateDt->extens[1]);
  DBG_fprintf(stderr, "Visu Data: emit OpenGLNearFar.\n");
  g_signal_emit(data, visu_data_signals[OPENGL_NEAR_FAR_CHANGED_SIGNAL],
		0, data->privateDt->attachedView, NULL);
  DBG_fprintf(stderr, "Visu Data: emission done (OpenGLNearFar).\n");

  /* We raise the signals. */
  visu_data_emitNodePositionChanged(data);

  return TRUE;
}

/**
 * visu_data_constrainedElementInTheBox:
 * @data: a #VisuData object ;
 * @element: a #VisuElement object.
 *
 * Check all the nodes of the specified @element and change their coordinates if they are out
 * of the bounding box. The position of each node is the result of the
 * sum of their own position and of the box translation.
 *
 * Returns: TRUE if visu_data_createAllNodes() should be called and
 *          the 'NodePositionChanged' should be emitted.
 */
gboolean visu_data_constrainedElementInTheBox(VisuData *data, VisuElement *element)
{
  int *indexEle, i;
  unsigned int j;
  gboolean changed;
  float cartCoord[3], t[3];
  VisuNode *node;

  g_return_val_if_fail(IS_VISU_DATA_TYPE(data) && element, FALSE);

  indexEle = (int*)g_hash_table_lookup(data->fromVisuElementToInt, (gpointer)element);
  g_return_val_if_fail(indexEle, 0);

  i = *indexEle;

  if (!data->fromIntToVisuElement[i]->rendered)
    return FALSE;

  DBG_fprintf(stderr, "Visu Data: Checking for nodes of element '%s'"
	      " to be in the box.\n", element->name);
  changed = FALSE;
  for(j = 0; j < data->privateDt->nodeArray->numberOfStoredNodes[i]; j++)
    {
      node = data->privateDt->nodeArray->nodes[i] + j;
      visu_data_getNodePosition(data, node, cartCoord);
      if (tool_matrix_constrainInBox(t, cartCoord,
				data->privateDt->extension,
				data->privateDt->fromXYZtoBoxD0,
				data->privateDt->fromBoxtoXYZD0))
	{
	  changed = TRUE;
	  node->translation[0] += t[0];
	  node->translation[1] += t[1];
	  node->translation[2] += t[2];
	}
    }
  data->privateDt->translationApply = TRUE;
  return changed;
}

/**
 * visu_data_constrainedInTheBox:
 * @data: a #VisuData object.
 *
 * It does the same things that visu_data_constrainedElementInTheBox() but for all
 * the #VisuElement of the given @data. I.e. it checks all the nodes and changes
 * their coordinates if they are out of the bounding box.
 * The position of each node is the result of the
 * sum of their own position and of the box translation.
 *
 * Returns: TRUE if visu_data_createAllNodes() should be called and
 *          the 'NodePositionChanged' should be emitted.
 */
gboolean visu_data_constrainedInTheBox(VisuData *data)
{
  unsigned int i;
  gboolean changed;

  g_return_val_if_fail(IS_VISU_DATA_TYPE(data), 0);

  changed = FALSE;
  for (i = 0; i < data->ntype; i++)
    changed = visu_data_constrainedElementInTheBox(data, data->fromIntToVisuElement[i]) || changed;
  data->privateDt->translationApply = TRUE;
  return changed;
}

/**
 * visu_data_constrainedFree:
 * @data: a #VisuData object.
 *
 * Return all the nodes to their original position, except for the
 * global translation.
 *
 * Returns: TRUE if visu_data_createAllNodes() should be called and
 *          the 'NodePositionChanged' should be emitted.
 */
gboolean visu_data_constrainedFree(VisuData *data)
{
  unsigned int i, j;

  g_return_val_if_fail(IS_VISU_DATA_TYPE(data), 0);

  for (i = 0; i < data->privateDt->nodeArray->ntype; i++)
    for(j = 0; j < data->privateDt->nodeArray->numberOfStoredNodes[i]; j++)
      {
	data->privateDt->nodeArray->nodes[i][j].translation[0] = 0.;
	data->privateDt->nodeArray->nodes[i][j].translation[1] = 0.;
	data->privateDt->nodeArray->nodes[i][j].translation[2] = 0.;
      }
  data->privateDt->translationApply = FALSE;
  return TRUE;
}

/**
 * visu_data_getTranslationStatus:
 * @data: a #VisuData object.
 *
 * When a translation is applied (even with a [0,0,0] vector), the
 * nodes are shifted to be in the box. This routine returns the
 * translation status of all nodes. If one of them is translated, then
 * return value is TRUE.
 *
 * Returns: if one of the nodes is shifted.
 */
gboolean visu_data_getTranslationStatus(VisuData *data)
{
  g_return_val_if_fail(IS_VISU_DATA_TYPE(data), FALSE);
  
  DBG_fprintf(stderr, "Visu Data: retrieve translation status (%d).\n",
	      data->privateDt->translationApply);
  return data->privateDt->translationApply;
}
/**
 * visu_data_setTightBox:
 * @data: a #VisuData object.
 *
 * Calculate the box geometry to have a tight box in directions that
 * are not periodic. If some directions are still periodic, the box
 * size in these directions should be setup first with
 * visu_data_setBoxGeometry().
 */
void visu_data_setTightBox(VisuData *data)
{
  double xMin, yMin, zMin, xMax, yMax, zMax, xFree, yFree, zFree;
  double boxGeometry[6];
  unsigned int i, j;
  
  g_return_if_fail(IS_VISU_DATA_TYPE(data));

  if (data->privateDt->bc == VISU_DATA_BOX_PERIODIC)
    return;

  /* Store the coordinates */
  xMin = 1e5;
  yMin = 1e5;
  zMin = 1e5;
  xMax = -1e5;
  yMax = -1e5;
  zMax = -1e5;
  xFree = (data->privateDt->bc == VISU_DATA_BOX_FREE ||
           data->privateDt->bc == VISU_DATA_BOX_SURFACE_YZ)?1.:0.;
  yFree = (data->privateDt->bc == VISU_DATA_BOX_FREE ||
           data->privateDt->bc == VISU_DATA_BOX_SURFACE_ZX)?1.:0.;
  zFree = (data->privateDt->bc == VISU_DATA_BOX_FREE ||
           data->privateDt->bc == VISU_DATA_BOX_SURFACE_XY)?1.:0.;
  
  for (i = 0; i < data->privateDt->nodeArray->ntype; i++)
    for(j = 0; j < data->privateDt->nodeArray->numberOfStoredNodes[i]; j++)
      {
	xMin = MIN(xMin, data->privateDt->nodeArray->nodes[i][j].xyz[0]);
	yMin = MIN(yMin, data->privateDt->nodeArray->nodes[i][j].xyz[1]);
	zMin = MIN(zMin, data->privateDt->nodeArray->nodes[i][j].xyz[2]);
	xMax = MAX(xMax, data->privateDt->nodeArray->nodes[i][j].xyz[0]);
	yMax = MAX(yMax, data->privateDt->nodeArray->nodes[i][j].xyz[1]);
	zMax = MAX(zMax, data->privateDt->nodeArray->nodes[i][j].xyz[2]);
      }

  DBG_fprintf(stderr, "Visu Data: the elements are in [%f, %f]x[%f, %f]x[%f, %f].\n",
	      xMin, xMax, yMin, yMax, zMin, zMax);
  boxGeometry[0] = (xMax - xMin + 1e-5) * xFree + (1. - xFree) * data->privateDt->boxGeometry[0];
  boxGeometry[1] = 0.                           + (1. - yFree) * data->privateDt->boxGeometry[1];
  boxGeometry[2] = (yMax - yMin + 1e-5) * yFree + (1. - yFree) * data->privateDt->boxGeometry[2];
  boxGeometry[3] = 0.                           + (1. - zFree) * data->privateDt->boxGeometry[3];
  boxGeometry[4] = 0.                           + (1. - zFree) * data->privateDt->boxGeometry[4];
  boxGeometry[5] = (zMax - zMin + 1e-5) * zFree + (1. - zFree) * data->privateDt->boxGeometry[5];
  visu_data_setBoxGeometry(data, boxGeometry, data->privateDt->bc);

  data->privateDt->translation[0] = -xMin * xFree;
  data->privateDt->translation[1] = -yMin * yFree;
  data->privateDt->translation[2] = -zMin * zFree;
  data->privateDt->translationApply = TRUE;
}

/**
 * visu_data_replicate:
 * @data: a #VisuData object ;
 * @extension: (in) (array fixed-size=3): three floating point values ;
 * @rebuild: a location to store a flag.
 *
 * This routine will create (or remove) nodes to expand the initial box to
 * the required size. An extension of 0 means no extension, i.e. the initial
 * box. The extension is done symmetrically in each direction toward negative
 * and positive direction.
 *
 * If the @rebuild argument is TRUE, the nodes should be rebuilt (with
 * visu_data_createAllNodes()).
 *
 * To remove added nodes, see visu_data_restore().
 *
 * returns: TRUE if the redraw should be done.
 */
gboolean visu_data_replicate(VisuData *data, float extension[3], gboolean *rebuild)
{
  int i;
  gboolean rebuild_, changed;
  /* Will store the index of removed or added nodes, always finish by -1.
     The two first values are specific and used to store:
     - the current number of index ;
     - the allocated size of the array (not including these two
       specific values and the -1 at the end. */
  int *index;

  g_return_val_if_fail(IS_VISU_DATA_TYPE(data), FALSE);
  g_return_val_if_fail(extension[0] >= 0. &&
		       extension[1] >= 0. &&
		       extension[2] >= 0., FALSE);

  /* Keep only three digits for the extension to avoid rounding
     troubles. */
  extension[0] = (float)((int)(extension[0] * 1000)) / 1000;
  extension[1] = (float)((int)(extension[1] * 1000)) / 1000;
  extension[2] = (float)((int)(extension[2] * 1000)) / 1000;
  DBG_fprintf(stderr, "Visu Data: modify extension (%g, %g, %g).\n",
	      extension[0], extension[1], extension[2]);

  rebuild_ = changed = FALSE;
  for (i = 0; i < 3; i++)
    {
      if (data->privateDt->extension[i] > extension[i])
	{
	  index = shrinkNodeList(data, i, extension[i]);
	  if (index)
	    {
	      /* We finish the list with -1. */
	      index[index[0] + 2] = -1;
	      visu_data_removeNodes(data, index + 2);
	      rebuild_ = TRUE;
	      g_free(index);
	    }
	  changed = TRUE;
	}
      else if (data->privateDt->extension[i] < extension[i])
	{
	  index = extendNodeList(data, i, extension[i]);
	  if (index)
	    {
	      /* We finish the list with -1. */
	      index[index[0] + 2] = -1;
	      DBG_fprintf(stderr, "Visu Data: emit a "
			  "'NodePopulationIncrease' signal.\n");
	      g_signal_emit(G_OBJECT(data),
			    visu_data_signals[NODE_POPULATION_INCREASE_SIGNAL],
			    0, (gpointer)index, NULL);
	      rebuild_ = TRUE;
	      g_free(index);
	    }
	  changed = TRUE;
	}
      data->privateDt->extension[i] = extension[i];
    }
  if (changed)
    {
      applyBox(data, -1.f);
      DBG_fprintf(stderr, "Visu Data: emit OpenGLNearFar.\n");
      g_signal_emit(data, visu_data_signals[OPENGL_NEAR_FAR_CHANGED_SIGNAL],
		    0, data->privateDt->attachedView, NULL);
      DBG_fprintf(stderr, "Visu Data: emission done (OpenGLNearFar).\n");

      DBG_fprintf(stderr, "VisuData: call 'visu_data_createAllElements'.\n");
      visu_data_createAllElements(data);
      DBG_fprintf(stderr, "VisuData: call 'visu_data_createAllElements' done.\n");
    }
  if (rebuild)
    *rebuild = rebuild_;
  if (DEBUG)
    visu_node_array_traceProperty(data->privateDt->nodeArray, "originalId");
  return changed;
}

/**
 * visu_data_restore:
 * @data: a #VisuData object.
 *
 * Remove all nodes that have been added by a visu_data_replicate()
 * call.
 *
 * Returns: TRUE if some nodes has been indeed removed.
 */
gboolean visu_data_restore(VisuData *data)
{
  gboolean rebuild;
  int *nodeNumbers;

  g_return_val_if_fail(IS_VISU_DATA_TYPE(data), FALSE);

  data->privateDt->extension[0] = 0.f;
  data->privateDt->extension[1] = 0.f;
  data->privateDt->extension[2] = 0.f;
  applyBox(data, -1.f);
  DBG_fprintf(stderr, "Visu Data: emit OpenGLNearFar.\n");
  g_signal_emit(data, visu_data_signals[OPENGL_NEAR_FAR_CHANGED_SIGNAL],
		0, data->privateDt->attachedView, NULL);
  DBG_fprintf(stderr, "Visu Data: emission done (OpenGLNearFar).\n");

  DBG_fprintf(stderr, "VisuData: call 'visu_data_createAllElements'.\n");
  visu_data_createAllElements(data);
  DBG_fprintf(stderr, "VisuData: call 'visu_data_createAllElements' done.\n");

  nodeNumbers = (int*)0;
  rebuild = visu_node_array_removeAllDuplicateNodes(data->privateDt->nodeArray,
					     &nodeNumbers);

  if (rebuild)
    {
      DBG_fprintf(stderr, "Visu Data: emit a 'NodePopulationDecrease' signal.\n");
      g_signal_emit(G_OBJECT(data),
		    visu_data_signals[NODE_POPULATION_DECREASE_SIGNAL],
		    0, (gpointer)nodeNumbers, NULL);
      g_free(nodeNumbers);
      return TRUE;
    }
  return FALSE;
}
static int* reallocIndexList(int *index, int size)
{
  int currentSize;
  
  if (index)
    currentSize = index[1];
  else
    currentSize = 0;

  index = g_realloc(index, sizeof(int) * (currentSize + size + 3));
  index[0] = currentSize;
  index[1] = currentSize + size;

  DBG_fprintf(stderr, "Visu Data: realloc internal index list (%d %d).\n",
	      index[0], index[1]);
  return index;
}
static int* addIndexList(int *index, int value, int size)
{
  if (!index || index[0] == index[1])
    index = reallocIndexList(index, size);
  index[index[0] + 2]  = value;
  index[0]            += 1;
  return index;
}
static int* shrinkNodeList(VisuData *data, int coord, float valueTo)
{
  float cartCoord[3], boxCoord[3];
  int *index;
  VisuDataIter iter;

  g_return_val_if_fail(coord == 0 || coord == 1 || coord == 2, FALSE);
  g_return_val_if_fail(valueTo < data->privateDt->extension[coord], FALSE);

  DBG_fprintf(stderr, "Visu Data: shrink from (%g, %g, %g).\n",
	      data->privateDt->extension[0], data->privateDt->extension[1],
	      data->privateDt->extension[2]);
  index = (int*)0;
  visu_data_iterNew(data, &iter);
  for (visu_data_iterStart(data, &iter); iter.node;
       visu_data_iterNext(data, &iter))
    {
      cartCoord[0] = iter.node->xyz[0] + data->privateDt->translation[0] +
	iter.node->translation[0];
      cartCoord[1] = iter.node->xyz[1] + data->privateDt->translation[1] +
	iter.node->translation[1];
      cartCoord[2] = iter.node->xyz[2] + data->privateDt->translation[2] +
	iter.node->translation[2];
      visu_data_convertXYZtoBoxCoordinates(data, boxCoord, cartCoord);
      if ((boxCoord[coord] < - valueTo - 1e-6 ||  /* We are out on the
						     low coord. */
	   boxCoord[coord] >= 1.f + valueTo -1e-6) && /* We are out on
							 the high
							 coord. */
	  visu_node_array_getOriginal(data->privateDt->nodeArray, iter.node->number) >= 0)
	/* We remove the element. */
	index = addIndexList(index, iter.node->number,
			     data->privateDt->nodeArray->nbOfAllStoredNodes);
      DBG_fprintf(stderr, "Visu Data: test shrink for %d: %d %15.12fx%15.12fx%15.12f.\n",
		  iter.node->number, (index)?index[0]:0,
		  boxCoord[0], boxCoord[1], boxCoord[2]);
    }
  return index;
}
static int* extendNodeList(VisuData *data, int coord, float valueTo)
{
  int k, id;
  unsigned nbInit, nb;
  VisuNode *newNode;
  float cartCoord[3], boxCoord[3], ratio;
  int *index;
  VisuDataIter iter;

  g_return_val_if_fail(coord == 0 || coord == 1 || coord == 2, FALSE);
  g_return_val_if_fail(valueTo > data->privateDt->extension[coord], FALSE);

  DBG_fprintf(stderr, "Visu Data: expand in %d direction to %g.\n",
	      coord, valueTo);
  DBG_fprintf(stderr, " | k runs in [%d %d[ ]%d %d].\n", (int)floor(-valueTo),
	      -(int)data->privateDt->extension[coord],
	      (int)data->privateDt->extension[coord],
	      (int)ceil(valueTo));
  DBG_fprintf(stderr, " | keeps new ele in [%g %g] [%g %g].\n", -valueTo,
	      -data->privateDt->extension[coord],
	      data->privateDt->extension[coord] + 1.f,
	      valueTo + 1.f);
  index = (int*)0;

  /* We estimate the number of data to be added and we call a realloc of
     this amount now to avoid to much small reallocations.
     The slab of box to be extend is 2*(valueTo-extension[coord]).
     So the volume in box coordinates is the same value and since
     the volume in box coordinates is product(1+2*extension),
     the ratio of new space is the fraction.
     So we realloc all elements on this ratio. */
  ratio = (2.f * (valueTo - data->privateDt->extension[coord])) /
    (1.f + 2.f * data->privateDt->extension[coord]);
  visu_data_iterNew(data, &iter);
  for (visu_data_iterStart(data, &iter); iter.element;
       visu_data_iterNextElement(data, &iter))
    {
      nb = (int)ceil((float)iter.nStoredNodes[iter.iElement] * ratio);
      if (iter.nStoredNodes[iter.iElement] + nb >
	  data->privateDt->nodeArray->numberOfNodes[iter.iElement])
	visu_node_array_allocateNewNodes(data->privateDt->nodeArray, iter.iElement, nb);
    }
  
  /* All node with an id higher than nbInit are considered as new
     nodes. */
  nbInit = data->privateDt->nodeArray->idCounter;
  for (visu_data_iterStartNumber(data, &iter); iter.node;
       visu_data_iterNextNodeNumber(data, &iter))
    {
      /* Do not duplicate the new nodes. */
      if (iter.node->number >= nbInit)
	continue;

      cartCoord[0] = iter.node->xyz[0] + data->privateDt->translation[0] +
	iter.node->translation[0];
      cartCoord[1] = iter.node->xyz[1] + data->privateDt->translation[1] +
	iter.node->translation[1];
      cartCoord[2] = iter.node->xyz[2] + data->privateDt->translation[2] +
	iter.node->translation[2];
      visu_data_convertXYZtoBoxCoordinates(data, boxCoord, cartCoord);
      for (k = (int)floor(-valueTo); k < (int)ceil(valueTo) + 1; k++)
	{
	  if (k >= -(int)data->privateDt->extension[coord] &&
	      k <   (int)data->privateDt->extension[coord] + 1)
	    continue;
	  boxCoord[coord] += (float)k;
	  if ((boxCoord[coord] >= -valueTo &&
	       boxCoord[coord] < -data->privateDt->extension[coord] ) ||
	      (boxCoord[coord] < valueTo + 1.f  &&
	       boxCoord[coord] >= data->privateDt->extension[coord] + 1.f))
	    {
	      DBG_fprintf(stderr, "Visu Data: replicating node %d, (%d)"
			  " (%15.12fx%15.12fx%15.12f).\n", iter.node->number, coord,
			  boxCoord[0], boxCoord[1], boxCoord[2]);
	      /* We save the current node id, because the pointer may be
		 relocated by the visu_node_array_getCopyNode() call. */
	      id = iter.node->number;
	      /* We create and add a new element. */
	      newNode = visu_node_array_getCopyNode(data->privateDt->nodeArray, iter.node);
	      index = addIndexList(index, newNode->number,
				   data->privateDt->nodeArray->nbOfAllStoredNodes);
	      visu_data_convertBoxCoordinatestoXYZ(data, newNode->xyz, boxCoord);
	      newNode->xyz[0] -= data->privateDt->translation[0] +
		newNode->translation[0];
	      newNode->xyz[1] -= data->privateDt->translation[1] +
		newNode->translation[1];
	      newNode->xyz[2] -= data->privateDt->translation[2] +
		newNode->translation[2];
	      /* We reset the iter.node pointer. */
	      iter.node = data->privateDt->nodeArray->nodeTable[id];
	    }
	  boxCoord[coord] -= (float)k;
	}
    }
  return index;
}

/**
 * visu_data_getExtension:
 * @dataObj: a #VisuData object ;
 * @extension: (in) (array fixed-size=3): an allocated array to store the values.
 *
 * Using visu_data_replicate(), it is possible to duplicate the primitive box
 * in each directions. Use this method to know the current extension. Returned
 * values are positive floating point values. An extension of 0. means that
 * only the primitive box exists, while a value of one means a duplication of
 * one box in each direction of the coordinate.
 */
void visu_data_getExtension(VisuData *dataObj, float extension[3])
{
  g_return_if_fail(IS_VISU_DATA_TYPE(dataObj));

  extension[0] = dataObj->privateDt->extension[0];
  extension[1] = dataObj->privateDt->extension[1];
  extension[2] = dataObj->privateDt->extension[2];
}
/**
 * visu_data_getAllElementExtens:
 * @dataObj: a #VisuData object.
 *
 * Calculate the maximum size of all #VisuElement used in this
 * @dataObj.
 *
 * Returns: a positive size.
 */
float visu_data_getAllElementExtens(VisuData *dataObj)
{
  float ext;
  guint i;
  VisuRendering *method;

  g_return_val_if_fail(IS_VISU_DATA_TYPE(dataObj), 0.f);
  
  /* Compute the max size of a drawn node. */
  method = visu_object_getRendering(VISU_INSTANCE);
  ext = 0.f;
  for (i = 0; i < dataObj->ntype; i++)
    ext = MAX(ext, visu_rendering_getSizeOfElement
	      (method, dataObj->fromIntToVisuElement[i]));
  return ext;
}
/**
 * visu_data_getBoxExtens:
 * @dataObj: a #VisuData object.
 *
 * This routines returns two float values giving the longest length of
 * the bounding box, with or without extension applied.
 *
 * Returns: two floating point values.
 */
float* visu_data_getBoxExtens(VisuData *dataObj)
{
  g_return_val_if_fail(IS_VISU_DATA_TYPE(dataObj), (float*)0);

  return dataObj->privateDt->extens;
}
/**
 * visu_data_getBoxLengths:
 * @dataObj: a #VisuData object ;
 * @ext1: (out caller-allocates): the length scale ;
 * @ext2: (out caller-allocates): ...
 *
 * Equivalent function to visu_data_getBoxExtens().
 *
 * Since: 3.6
 */
void visu_data_getBoxLengths(VisuData *dataObj, float *ext1, float *ext2)
{
  float *ext;

  ext = visu_data_getBoxExtens(dataObj);
  g_return_if_fail(ext);

  *ext1 = ext[0];
  *ext2 = ext[1];
}

/**
 * visu_data_getBoxCentre:
 * @dataObj: a #VisuData object ;
 * @centre: (in) (array fixed-size=3): coordinates of the centre.
 *
 * @centre contains on output the cartesian coordinates of the centre
 * of the bounding box.
 */
void visu_data_getBoxCentre(VisuData *dataObj, float centre[3])
{
  g_return_if_fail(IS_VISU_DATA_TYPE(dataObj));

  centre[0] = dataObj->privateDt->dxxs2;
  centre[1] = dataObj->privateDt->dyys2;
  centre[2] = dataObj->privateDt->dzzs2;
}

static float getAllNodeExtens(VisuData *dataObj)
{
  guint i, j;
  VisuNodeArray *array;
  float xyz[2][3], t[3], lg[2], coord[3];

  g_return_val_if_fail(IS_VISU_DATA_TYPE(dataObj), 0.f);

  t[0] = (float)(dataObj->privateDt->boxGeometry[VISU_DATA_BOX_DXX] +
		 dataObj->privateDt->boxGeometry[VISU_DATA_BOX_DYX] +
		 dataObj->privateDt->boxGeometry[VISU_DATA_BOX_DZX]);
  t[1] = (float)(dataObj->privateDt->boxGeometry[VISU_DATA_BOX_DYY] +
		 dataObj->privateDt->boxGeometry[VISU_DATA_BOX_DZY]);
  t[2] = (float)(dataObj->privateDt->boxGeometry[VISU_DATA_BOX_DZZ]);
  xyz[0][0] = xyz[0][1] = xyz[0][2] = 0.f;
  xyz[1][0] = xyz[1][1] = xyz[1][2] = 0.f;

  array = dataObj->privateDt->nodeArray;
  if (!array)
    return 0.f;
  for (i = 0; i < array->ntype; i++)
    for (j = 0; j < array->numberOfStoredNodes[i]; j++)
      {
	visu_data_getNodePosition(dataObj, array->nodes[i] + j, coord);
	xyz[0][0] = MIN(xyz[0][0], coord[0]);
	xyz[0][1] = MIN(xyz[0][1], coord[1]);
	xyz[0][2] = MIN(xyz[0][2], coord[2]);

	xyz[1][0] = MAX(xyz[1][0], coord[0] - t[0]);
	xyz[1][1] = MAX(xyz[1][1], coord[1] - t[1]);
	xyz[1][2] = MAX(xyz[1][2], coord[2] - t[2]);
      }
  /* Compute the longest vector out of the box. */
  lg[0] = sqrt(xyz[0][0] * xyz[0][0] + 
	       xyz[0][1] * xyz[0][1] + 
	       xyz[0][2] * xyz[0][2]);
  lg[1] = sqrt(xyz[1][0] * xyz[1][0] + 
	       xyz[1][1] * xyz[1][1] + 
	       xyz[1][2] * xyz[1][2]);
  DBG_fprintf(stderr, "VisuData: vectors outside of the box %g %g.\n", lg[0], lg[1]);
  return MAX(lg[0], lg[1]);
}

/**
 * visu_data_getDistanceList:
 * @data: a #VisuData object ;
 * @nodeId: a node id.
 * @minVal: a location for a float.
 *
 * This routine creates an array of #VisuNodeInfo, storing for each
 * node its node id and its distance to @nodeId. The periodicity is
 * NOT taken into account. The array is not distance sorted, but if
 * @minVal is provided, it will contain the minimal distance between
 * @nodeId and the other nodes.
 *
 * Since: 3.5
 *
 * Returns: an array of #VisuNodeInfo of size the number of nodes. It
 * is terminated by @nodeId value itself.
 */
VisuNodeInfo* visu_data_getDistanceList(VisuData *data, guint nodeId, float *minVal)
{
  VisuNodeInfo *infos;
  int nNodes;
  VisuDataIter iter;
  VisuNode *nodeRef;
  float xyz[3], xyzRef[3], min;

  g_return_val_if_fail(IS_VISU_DATA_TYPE(data), (VisuNodeInfo*)0);

  DBG_fprintf(stderr, "Visu Data: get distance list for node %d.\n", nodeId);
  nodeRef = visu_data_getNodeFromNumber(data, nodeId);
  g_return_val_if_fail(nodeRef, (VisuNodeInfo*)0);

  nNodes = data->privateDt->nodeArray->nbOfAllStoredNodes;
  infos = g_malloc(sizeof(VisuNodeInfo) * nNodes);
  
  visu_data_getNodePosition(data, nodeRef, xyzRef);

  min = G_MAXFLOAT;
  nNodes = 0;
  visu_data_iterNew(data, &iter);
  for (visu_data_iterStart(data, &iter); iter.node;
       visu_data_iterNextVisible(data, &iter))
    {
      infos[nNodes].id = iter.node->number;
      visu_data_getNodePosition(data, iter.node, xyz);
      infos[nNodes].dist =
	(xyz[0] - xyzRef[0]) * (xyz[0] - xyzRef[0]) +
	(xyz[1] - xyzRef[1]) * (xyz[1] - xyzRef[1]) +
	(xyz[2] - xyzRef[2]) * (xyz[2] - xyzRef[2]);
      if (infos[nNodes].dist > 0.0001f)
	{
	  min = MIN(min, infos[nNodes].dist);
	  nNodes += 1;
	}
    }
  infos[nNodes].id = nodeId;

  DBG_fprintf(stderr, " | min value = %g.\n", min);
  if (minVal)
    *minVal = min;

  return infos;
}
/**
 * visu_data_setNewBasisFromNodes:
 * @data: a #VisuData object.
 * @nO: the index of node as origin.
 * @nA: the index of node on X axis.
 * @nB: the index of node as Y axis.
 * @nC: the index of node as Z axis.
 *
 * Change the basis set by providing the new basis set from a list of
 * nodes. See also visu_data_setNewBasis(). Nodes outside the new box
 * are killed.
 *
 * Since: 3.6
 *
 * Returns: TRUE if the new basis set is valid.
 */
gboolean visu_data_setNewBasisFromNodes(VisuData *data, guint nO, guint nA, guint nB, guint nC)
{
  VisuNode *orig, *nodeA, *nodeB, *nodeC;
  float matA[3][3], O[3], xyz[3];

  orig = visu_data_getNodeFromNumber(data, nO);
  DBG_fprintf(stderr, " orig = %p\n", (gpointer)orig);
  nodeA = visu_data_getNodeFromNumber(data, nA);
  DBG_fprintf(stderr, " nodeA = %p\n", (gpointer)nodeA);
  nodeB = visu_data_getNodeFromNumber(data, nB);
  DBG_fprintf(stderr, " nodeB = %p\n", (gpointer)nodeB);
  nodeC = visu_data_getNodeFromNumber(data, nC);
  DBG_fprintf(stderr, " nodeC = %p\n", (gpointer)nodeC);
  g_return_val_if_fail(orig && nodeA && nodeB && nodeC, FALSE);

  visu_data_getNodePosition(data, orig, O);
  visu_data_getNodePosition(data, nodeA, xyz);
  matA[0][0] = xyz[0] - O[0];
  matA[1][0] = xyz[1] - O[1];
  matA[2][0] = xyz[2] - O[2];
  visu_data_getNodePosition(data, nodeB, xyz);
  matA[0][1] = xyz[0] - O[0];
  matA[1][1] = xyz[1] - O[1];
  matA[2][1] = xyz[2] - O[2];
  visu_data_getNodePosition(data, nodeC, xyz);
  matA[0][2] = xyz[0] - O[0];
  matA[1][2] = xyz[1] - O[1];
  matA[2][2] = xyz[2] - O[2];

  return visu_data_setNewBasis(data, matA, O);
}
/**
 * visu_data_setNewBasis:
 * @data: a #VisuData object.
 * @matA: a basis set definition.
 * @O: the origin cartesian coordinates.
 *
 * Change the basis set of @data according to the new definition given
 * by @matA and @O. Nodes outside the new box are killed. See also
 * visu_data_setNewBasisFromNodes() for a convenient function using
 * nodes as basis set definition.
 * 
 * Since: 3.6
 *
 * Returns: TRUE if the new basis set is valid.
 */
gboolean visu_data_setNewBasis(VisuData *data, float matA[3][3], float O[3])
{
  double mat_[3][3];
  float inv[3][3], vect[3], xred[3];
  double box[6];
  float vectEps[3], deltaEps[3];
  VisuDataIter iter;
  int *rmNodes, i;
#define EPS 1.e-5

  DBG_fprintf(stderr, "Visu Data: basis matrice:\n");
  DBG_fprintf(stderr, "  (%10.5f  %10.5f  %10.5f)\n",
	      matA[0][0], matA[0][1], matA[0][2]);
  DBG_fprintf(stderr, "  (%10.5f  %10.5f  %10.5f)\n",
	      matA[1][0], matA[1][1], matA[1][2]);
  DBG_fprintf(stderr, "  (%10.5f  %10.5f  %10.5f)\n",
	      matA[2][0], matA[2][1], matA[2][2]);
  if (!tool_matrix_invert(inv, matA))
    return FALSE;
  DBG_fprintf(stderr, "Visu Data: transformation matrice:\n");
  DBG_fprintf(stderr, "  (%10.5f  %10.5f  %10.5f)\n",
	      inv[0][0], inv[0][1], inv[0][2]);
  DBG_fprintf(stderr, "  (%10.5f  %10.5f  %10.5f)\n",
	      inv[1][0], inv[1][1], inv[1][2]);
  DBG_fprintf(stderr, "  (%10.5f  %10.5f  %10.5f)\n",
	      inv[2][0], inv[2][1], inv[2][2]);

  mat_[0][0] = (double)matA[0][0];
  mat_[1][0] = (double)matA[0][1];
  mat_[2][0] = (double)matA[0][2];
  mat_[0][1] = (double)matA[1][0];
  mat_[1][1] = (double)matA[1][1];
  mat_[2][1] = (double)matA[1][2];
  mat_[0][2] = (double)matA[2][0];
  mat_[1][2] = (double)matA[2][1];
  mat_[2][2] = (double)matA[2][2];
  if (!tool_matrix_reducePrimitiveVectors(box, mat_))
    return FALSE;
  DBG_fprintf(stderr, "Visu Data: new box:\n");
  DBG_fprintf(stderr, "  (%10.5f  %10.5f  %10.5f)\n",
	      box[0], box[1], box[2]);
  DBG_fprintf(stderr, "  (%10.5f  %10.5f  %10.5f)\n",
	      box[3], box[4], box[5]);

  visu_data_setBoxGeometry(data, box, VISU_DATA_BOX_PERIODIC);

  /* We need to move all the atoms of (eps, eps, eps) in the new box
     to avoid rounding problems. */
  xred[0] = 1.f;
  xred[1] = 1.f;
  xred[2] = 1.f;
  tool_matrix_productVector(vect, matA, xred);
  vectEps[0] = (vect[0] >= 0.f)?EPS:-EPS;
  vectEps[1] = (vect[1] >= 0.f)?EPS:-EPS;
  vectEps[2] = (vect[2] >= 0.f)?EPS:-EPS;
  tool_matrix_productVector(xred, inv, vectEps);
  visu_data_convertBoxCoordinatestoXYZ(data, deltaEps, xred);
  DBG_fprintf(stderr, "Visu Data: applied epsilon (%10.5f  %10.5f  %10.5f)\n",
	      vectEps[0], vectEps[1], vectEps[2]);

  /* Transform each atomic coordinates using this matrice. */
  DBG_fprintf(stderr, "Visu Data: reset the coordinates for all nodes.\n");
  visu_data_iterNew(data, &iter);
  rmNodes = g_malloc(sizeof(int) * iter.nAllStoredNodes);
  i = 0;
  for (visu_data_iterStart(data, &iter); iter.node;
       visu_data_iterNext(data, &iter))
    {
      visu_data_getNodePosition(data, iter.node, vect);
      vect[0] += - O[0] + vectEps[0];
      vect[1] += - O[1] + vectEps[1];
      vect[2] += - O[2] + vectEps[2];
      tool_matrix_productVector(xred, inv, vect);
      if (xred[0] < 0.f || xred[0] >= 1.f ||
	  xred[1] < 0.f || xred[1] >= 1.f ||
	  xred[2] < 0.f || xred[2] >= 1.f)
	{
	  rmNodes[i] = (int)iter.node->number;
	  DBG_fprintf(stderr, " | %d  (%6.1f %6.1f %6.1f)"
		      " %10.5f %10.5f %10.5f -> removed\n",
		      iter.node->number, vect[0], vect[1], vect[2],
		      xred[0], xred[1], xred[2]);
	  i+= 1;
	}
      else
	{
	  visu_data_convertBoxCoordinatestoXYZ(data, iter.node->xyz, xred);
	  iter.node->xyz[0] -= deltaEps[0];
	  iter.node->xyz[1] -= deltaEps[1];
	  iter.node->xyz[2] -= deltaEps[2];
	  iter.node->translation[0] = 0.f;
	  iter.node->translation[1] = 0.f;
	  iter.node->translation[2] = 0.f;
	  visu_node_array_setOriginal(data->privateDt->nodeArray, iter.node->number);
	  DBG_fprintf(stderr, " | %d  (%6.1f %6.1f %6.1f)"
		      " %10.5f %10.5f %10.5f -> %10.5f %10.5f %10.5f\n",
		      iter.node->number, vect[0], vect[1], vect[2],
		      xred[0], xred[1], xred[2], iter.node->xyz[0],
		      iter.node->xyz[1], iter.node->xyz[2]);
	}
    }
  rmNodes[i] = -1;

  visu_data_removeNodes(data, rmNodes);

  /* Remove possible translation. */
  data->privateDt->translation[0] = 0.f;
  data->privateDt->translation[1] = 0.f;
  data->privateDt->translation[2] = 0.f;
  data->privateDt->translationApply = FALSE;
  visu_data_createAllNodes(data);

  g_free(rmNodes);

  /* Remove possible extension. */
  data->privateDt->extension[0] = 0.f;
  data->privateDt->extension[1] = 0.f;
  data->privateDt->extension[2] = 0.f;
  visu_data_applyBoxGeometry(data);

  visu_data_emitNodePositionChanged(data);

  VISU_FORCE_REDRAW;

  return TRUE;
}
/**
 * visu_data_reorder:
 * @data: a #VisuData object, to reorder.
 * @dataRef: a #VisuData object, to take the order from.
 *
 * This routine modifies the node ordering of @data using the order in
 * @dataRef. The association is done by nearest neigbours conditions.
 *
 * Since: 3.6
 *
 * Returns: TRUE is the reordering is successfull (i.e. all nodes of
 * @data correspond to one of @dataRef).
 */
gboolean visu_data_reorder(VisuData *data, VisuData *dataRef)
{
  VisuNodeArray *nodesRef, *nodes;
  VisuDataIter iter, iterRef;
  float d, diff[3], dMin;
  guint id;

  g_return_val_if_fail(IS_VISU_DATA_TYPE(dataRef), FALSE);
  g_return_val_if_fail(IS_VISU_DATA_TYPE(data), FALSE);

  DBG_fprintf(stderr, "Geometry: reorder between %p and %p.\n",
	      (gpointer)dataRef, (gpointer)data);

  nodesRef = visu_data_getNodeArray(dataRef);
  nodes    = visu_data_getNodeArray(data);

  DBG_fprintf(stderr, " | %d - %d.\n", nodes->nbOfAllStoredNodes, nodesRef->nbOfAllStoredNodes);
  if (nodes->nbOfAllStoredNodes != nodesRef->nbOfAllStoredNodes)
    return FALSE;

  visu_data_iterNew(data, &iter);
  for (visu_data_iterStart(data, &iter); iter.node;
       visu_data_iterNext(data, &iter))
    {
      id = 0;
      dMin = G_MAXFLOAT;
      visu_data_iterNew(dataRef, &iterRef);
      for (visu_data_iterStart(dataRef, &iterRef); iterRef.node;
           visu_data_iterNext(dataRef, &iterRef))
        {
          visu_geodiff_getPeriodicDistance(diff, data, iter.node, iterRef.node);
          d = diff[0] * diff[0] + diff[1] * diff[1] + diff[2] * diff[2];
          if (d < dMin)
            {
              id = iterRef.node->number;
              dMin = d;
            }
        }
      visu_node_array_switchNumber(nodes, iter.node->number, id);
    }
  return TRUE;
}


/*****************************/
/* The node related routines */
/*****************************/

/**
 * visu_data_removeNodes:
 * @data: a #VisuData object ;
 * @nodeNumbers: (in) (array): an allocated int array with number identifying nodes.
 *
 * Delete some nodes using this method. The nodes are identified by their
 * number and have not to be of the same element. This routine raises
 * the "NodePopulationDecrease" signal. The @nodeNumbers argument must
 * be terminated with a negative value.
 */
void visu_data_removeNodes(VisuData *data, int *nodeNumbers)
{
  visu_node_array_removeNodes(data->privateDt->nodeArray, nodeNumbers);

  DBG_fprintf(stderr, "Visu Data: emit a 'NodePopulationDecrease' signal.\n");
  g_signal_emit(G_OBJECT(data), visu_data_signals[NODE_POPULATION_DECREASE_SIGNAL],
		0, (gpointer)nodeNumbers, NULL);
}

/**
 * visu_data_addNodeFromIndex:
 * @data: the #VisuData where to add the new #VisuNode ;
 * @position: a integer corresponding to the position of
 *            a #VisuElement in the array **nodes in the structure;
 * @xyz: (in) (array fixed-size=3): its coordinates ;
 * @reduced: coordinates are in reduced coordinates ;
 * @emitSignal: a boolean.
 *
 * This method adds a new #VisuNode to the specified #VisuData. Position must be
 * chosen between 0 and (ntype - 1) and corresponds to the position of the array
 * of #VisuNodes of a #VisuElement. If @emitSignal is TRUE, then
 * NodePopulationIncrease signal is triggered.
 *
 * Returns: (transfer none): a pointer to the newly created node.
 */
VisuNode* visu_data_addNodeFromIndex(VisuData *data, unsigned int position,
                                     float xyz[3], gboolean reduced, gboolean emitSignal)
{
  VisuNode *node;
  int index[4];
  float coord[3];

  g_return_val_if_fail(IS_VISU_DATA_TYPE(data) && position < data->ntype,
		       (VisuNode*)0);

  node = visu_node_array_getNewNode(data->privateDt->nodeArray, position);
  g_return_val_if_fail(node, (VisuNode*)0);

  /* If coordinates are reduced, we expand them. */
  DBG_fprintf(stderr, "Visu Data: set node coordinates from (%g;%g;%g).\n",
              xyz[0], xyz[1], xyz[2]);
  if (reduced)
    visu_data_convertBoxCoordinatestoXYZ(data, coord, xyz);
  else if (data->privateDt->cartesiansAreFull)
    tool_matrix_productVector(coord, data->privateDt->fromFullToLocal, xyz);
  else
    {
      coord[0] = xyz[0];
      coord[1] = xyz[1];
      coord[2] = xyz[2];
    }

  visu_node_newValues(node, coord);

  if (emitSignal)
    {
      index[0] = 1;
      index[1] = 2;
      index[2] = node->number;
      index[3] = -1;
      DBG_fprintf(stderr, "Visu Data: emit a "
		  "'NodePopulationIncrease' signal.\n");
      g_signal_emit(G_OBJECT(data),
		    visu_data_signals[NODE_POPULATION_INCREASE_SIGNAL],
		    0, (gpointer)(&index), NULL);
    }
  return node;
}

/**
 * visu_data_addNodeFromElement:
 * @data: the #VisuData where to add the new #VisuNode ;
 * @ele: the #VisuElement kind of the new #VisuNode ;
 * @xyz: (in) (array fixed-size=3): its coordinates ;
 * @reduced: coordinates are in reduced coordinates ;
 * @emitSignal: a boolean.
 *
 * This method adds a new #VisuNode to the specified #VisuData. If
 * @emitSignal is TRUE, then NodePopulationIncrease signal is
 * triggered.
 *
 * Returns: (transfer none): a pointer to the newly created node.
 */
VisuNode* visu_data_addNodeFromElement(VisuData *data, VisuElement *ele,
                                       float xyz[3], gboolean reduced,
                                       gboolean emitSignal)
{
  unsigned int *pos;

  g_return_val_if_fail(IS_VISU_DATA_TYPE(data) && ele, (VisuNode*)0);

  pos = (unsigned int*)g_hash_table_lookup(data->fromVisuElementToInt, (gpointer)ele);
  g_return_val_if_fail(pos && *pos <= data->ntype, (VisuNode*)0);

  return visu_data_addNodeFromIndex(data, *pos, xyz, reduced, emitSignal);
}
/**
 * visu_data_addNodeFromElementName:
 * @data: the #VisuData where to add the new #VisuNode ;
 * @name: the name of the element ;
 * @xyz: (in) (array fixed-size=3): its coordinates ;
 * @reduced: coordinates are in reduced coordinates ;
 * @emitSignal: a boolean.
 *
 * This method adds a new #VisuNode to the specified #VisuData. If
 * @emitSignal is TRUE, then NodePopulationIncrease signal is
 * triggered.
 *
 * Returns: (transfer none): a pointer to the newly created node.
 *
 * Since: 3.6
 */
VisuNode* visu_data_addNodeFromElementName(VisuData *data, gchar *name,
                                           float xyz[3], gboolean reduced,
                                           gboolean emitSignal)
{
  unsigned int *pos;
  VisuElement *ele;
  gboolean new;

  g_return_val_if_fail(IS_VISU_DATA_TYPE(data) && name, (VisuNode*)0);

  ele = visu_element_retrieveFromName(name, &new);
  g_return_val_if_fail(!new, (VisuNode*)0);

  pos = (unsigned int*)g_hash_table_lookup(data->fromVisuElementToInt, (gpointer)ele);
  g_return_val_if_fail(pos && *pos <= data->ntype, (VisuNode*)0);

  return visu_data_addNodeFromIndex(data, *pos, xyz, reduced, emitSignal);
}


/**
 * visu_data_getNodeCoordinates:
 * @data: a #VisuData object ;
 * @node: a #VisuNode object ;
 * @x: (out caller-allocates): the x coordinate.
 * @y: (out caller-allocates): the y coordinate.
 * @z: (out caller-allocates): the z coordinate.
 *
 * Wrapper for the function visu_data_getNodePosition() in case of call
 * from python.
 *
 * Since: 3.6
 */
void visu_data_getNodeCoordinates(VisuData *data, VisuNode *node,
				  float *x, float *y, float *z)
{
  float xyz[3];

  g_return_if_fail(x && y && z);

  visu_data_getNodePosition(data, node, xyz);
  *x = xyz[0];
  *y = xyz[1];
  *z = xyz[2];
}


/**
 * visu_data_getNodePosition: (skip)
 * @data: a #VisuData object ;
 * @node: a #VisuNode object ;
 * @coord: (array fixed-size=3) (out caller-allocates): an array of 3
 * floating point values to store the position.
 *
 * Position of nodes are subject to various translations and different transformations.
 * Their coordinates should not be access directly through node.[xyz]. This method
 * is used to retrieve the given node position.
 *
 */
void visu_data_getNodePosition(VisuData *data, VisuNode *node, float coord[3])
{
  g_return_if_fail(IS_VISU_DATA_TYPE(data) && node && coord);

  coord[0] = node->xyz[0] + node->translation[0] + data->privateDt->translation[0];
  coord[1] = node->xyz[1] + node->translation[1] + data->privateDt->translation[1];
  coord[2] = node->xyz[2] + node->translation[2] + data->privateDt->translation[2];
}


/**
 * visu_data_getReducedNodePosition: (skip)
 * @data: a #VisuData object ;
 * @node: a #VisuNode object ;
 * @red: (array fixed-size=3) (out caller-allocates): coordinates of the reduced node
 * 
 * This function is equivalent to visu_data_getNodePosition() but
 * gives reduced coordinates instead.
 */
void visu_data_getReducedNodePosition(VisuData *data, VisuNode *node, float red[3])
{
  float coord[3];

  g_return_if_fail(IS_VISU_DATA_TYPE(data) && node && red);

  coord[0] = node->xyz[0] + node->translation[0] + data->privateDt->translation[0];
  coord[1] = node->xyz[1] + node->translation[1] + data->privateDt->translation[1];
  coord[2] = node->xyz[2] + node->translation[2] + data->privateDt->translation[2];
  red[0] =
    (float)data->privateDt->fromXYZtoBoxD0[0][0] * coord[0] +
    (float)data->privateDt->fromXYZtoBoxD0[0][1] * coord[1] +
    (float)data->privateDt->fromXYZtoBoxD0[0][2] * coord[2];
  red[1] =
    (float)data->privateDt->fromXYZtoBoxD0[1][0] * coord[0] +
    (float)data->privateDt->fromXYZtoBoxD0[1][1] * coord[1] +
    (float)data->privateDt->fromXYZtoBoxD0[1][2] * coord[2];
  red[2] =
    (float)data->privateDt->fromXYZtoBoxD0[2][0] * coord[0] +
    (float)data->privateDt->fromXYZtoBoxD0[2][1] * coord[1] +
    (float)data->privateDt->fromXYZtoBoxD0[2][2] * coord[2];
}

/**
 * visu_data_getNodeFromNumber:
 * @data: a %VisuData structure which stores the nodes.
 * @number: an integer.
 *
 * This methods retrieves the #VisuNode identified by the integer @number.
 * The number must be strictly positive. No eror is raised if no node corresponds
 * to the given number.
 *
 * Returns: (transfer none): the found #VisuNode or NULL if none corresponds to number.
 */
VisuNode* visu_data_getNodeFromNumber(VisuData *data, unsigned int number)
{
  g_return_val_if_fail(IS_VISU_DATA_TYPE(data), (VisuNode*)0);

  DBG_fprintf(stderr, "Visu Data: get VisuNode from number %d.\n", number);
  if (data->privateDt->nodeArray && number < data->privateDt->nodeArray->idCounter)
    return data->privateDt->nodeArray->nodeTable[number];
  else
    return (VisuNode*)0;
}


/*********************/
/* The emit routines */
/*********************/

/**
 * visu_data_emitAskForShowHideNodes:
 * @data: a valid #VisuData object ;
 * @redraw: a pointer to a location to store if redraw
 *          is needed after all listeners have modified the
 *          nodes of the given @data.
 *
 * This methods is used to emit the 'NodeAskForShowHide' signal. This
 * signal asks all modules that may hide nodes to recompute their
 * hiding scheme and put in @redraw if they have changed something. Since
 * all listeners write in @redraw, they should modify it with an union.
 * @redraw is initialised at FALSE before the signal is emitted.
 */
void visu_data_emitAskForShowHideNodes(VisuData *data, gboolean *redraw)
{
  g_return_if_fail(data && redraw);

  *redraw = FALSE;
  DBG_fprintf(stderr, "Visu Data: %p emit the 'NodeAskForShowHide' signal.\n", (gpointer)data);
  g_signal_emit(data, visu_data_signals[NODE_ASK_FOR_SHOW_HIDE_SIGNAL],
		0 , (gpointer)redraw, NULL);
  DBG_fprintf(stderr, "Visu Data: emission done (NodeAskForShowHide).\n");
}

/**
 * visu_data_emitNodePositionChanged:
 * @data: a valid #VisuData object.
 *
 * This method is used to emit the 'NodePositionChanged' signal. This signal asks
 * all modules that are dependent of the nodes' positions to recompute their
 * OpenGL lists.
 */

void visu_data_emitNodePositionChanged(VisuData *data)
{
  g_return_if_fail(data);

  DBG_fprintf(stderr, "Visu Data: %p emit the 'NodePositionChanged' signal.\n", (gpointer)data);
  g_signal_emit(data, visu_data_signals[NODE_POSITION_CHANGED_SIGNAL],
		0 , NULL);
  DBG_fprintf(stderr, "Visu Data: emission done (NodePositionChanged).\n");
}

/**
 * visu_data_emitNodeRenderedChange:
 * @data: a valid #VisuData object.
 *
 * This method is used to emit the 'NodeRenderedChanged' signal. This signal asks
 * all modules that are dependent of the nodes' visibility to recompute their
 * OpenGL lists. This signal is usualy emitted after a call to
 * visu_data_emitAskForShowHideNodes().
 */

void visu_data_emitNodeRenderedChange(VisuData *data)
{
  g_return_if_fail(data);

  DBG_fprintf(stderr, "Visu Data: %p emit the 'NodeRenderedChanged' signal.\n", (gpointer)data);
  g_signal_emit(data, visu_data_signals[NODE_RENDERED_CHANGED_SIGNAL],
		0 , NULL);
}

/**
 * visu_data_emitFacettesChanged:
 * @data: a valid #VisuData object.
 *
 * Emit the 'OpenGLFacetteChanged' signal.
 */

void visu_data_emitFacettesChanged(VisuData *data)
{
  g_return_if_fail(IS_VISU_DATA_TYPE(data));
  
  DBG_fprintf(stderr, "Visu Data: %p emit the 'OpenGLFacetteChanged' signal.\n", (gpointer)data);
  g_signal_emit(data, visu_data_signals[OPENGL_FACETTES_CHANGED_SIGNAL],
		0 , NULL);
}



/****************/
/* The timeouts */
/****************/

/**
 * visu_data_addTimeout:
 * @data: a valid #VisuData object ;
 * @time: the period of call in milliseconds ;
 * @func: (scope call): the callback function to be called ;
 * @user_data: (closure): a pointer to some user defined informations.
 *
 * This method is used to add the @func method to be called regularly at the period
 * @time. This methos calls in fact g_timeout_add() with the given arguments. But
 * the source id is stored internaly and the timeout function is removed automatically
 * when the object @data is destroyed. It is convienient to add a method working
 * on the #VisuData object that is called periodically during the life of te object.
 *
 * Returns: the source id if the calling method need to work with it. To remove
 *          the callback, don't use g_source_remove() but visu_data_removeTimeout()
 *          to inform the #VisuData object that this source has been removed and
 *          not to remove it when the object will be destroyed.
 */
guint visu_data_addTimeout(VisuData *data, guint time,
			   GSourceFunc func, gpointer user_data)
{
  guint *id;

  g_return_val_if_fail(IS_VISU_DATA_TYPE(data) && func, (guint)0);

  id = g_malloc(sizeof(guint));
  *id = g_timeout_add(time, func, user_data);
  data->privateDt->timeoutList = g_list_prepend(data->privateDt->timeoutList, (gpointer)id);
  DBG_fprintf(stderr, "Visu Data: create a new timeout callback : %d.\n", *id);
  return *id;
}

/**
 * visu_data_removeTimeout:
 * @data: a valid #VisuData object ;
 * @timeoutId: a source id.
 *
 * This method is used to remove a timeout that has been associated to the given
 * @data (see visu_data_addTimeout()).
 *
 * Returns: TRUE if the source has been found and removed.
 */
gboolean visu_data_removeTimeout(VisuData *data, guint timeoutId)
{
  GList *lst;
  gboolean found;

  g_return_val_if_fail(IS_VISU_DATA_TYPE(data), FALSE);

  DBG_fprintf(stderr, "Visu Data: trying to remove a timeout callback (%d) ... ", timeoutId);
  found = FALSE;
  lst = data->privateDt->timeoutList;
  while(lst)
    {
      if (*(guint*)lst->data == timeoutId)
	{
	  found = g_source_remove(timeoutId);
	  data->privateDt->timeoutList =
	    g_list_delete_link(data->privateDt->timeoutList, lst);
	  DBG_fprintf(stderr, "OK (%d).\n", found);
	  return found;
	}
      lst = g_list_next(lst);
    }
  DBG_fprintf(stderr, "not found.\n");
  return found;
}


/***********************/
/* The view properties */
/***********************/

/**
 * visu_data_getOpenGLView:
 * @data: a #VisuData object.
 *
 * Once the object @data has been initialised, an #VisuOpenGLView object
 * is automattically attached and this method can be used to retrieve
 * it.
 *
 * Returns: (transfer none): the #VisuOpenGLView attached to the given @data.
 */
VisuOpenGLView* visu_data_getOpenGLView(VisuData *data)
{
  g_return_val_if_fail(IS_VISU_DATA_TYPE(data), (VisuOpenGLView*)0);

  return data->privateDt->attachedView;
}

/**
 * visu_data_setAngleOfView:
 * @data: a #VisuData object ;
 * @valueTheta: a floatinf point value in degrees ;
 * @valuePhi: a floating point value in degrees ;
 * @valueOmega: a floating point value in degrees ;
 * @mask: to specified what values will be changed.
 *
 * This method is used to change the camera orientation for the given @data.
 * If necessary, this method will emit the 'OpenGLThetaPhiOmega' signal.
 *
 * Returns: 1 if the 'OpenGLAskForReDraw' signal should be emitted.
 */

gboolean visu_data_setAngleOfView(VisuData *data, float valueTheta,
				  float valuePhi, float valueOmega, int mask)
{
  gboolean res;

  g_return_val_if_fail(IS_VISU_DATA_TYPE(data), FALSE);
  
  res = openGLCameraSet_thetaPhiOmega(data->privateDt->attachedView->camera,
				      valueTheta, valuePhi, valueOmega, mask);
  if (res)
    {
      openGLModelize(data->privateDt->attachedView->camera);

      g_signal_emit(data, visu_data_signals[OPENGL_THETA_PHI_OMEGA_CHANGED_SIGNAL],
		    0, data->privateDt->attachedView, NULL);
    }
    
  return res;
}
/**
 * visu_data_setPositionOfView:
 * @data: a #VisuData object ;
 * @valueX: a floatinf point value in the bounding box scale
 *          (1 is the size of the bounding box) ;
 * @valueY: a floating point value in bounding box scale ;
 * @mask: to specified what values will be changed.
 *
 * This method is used to change the camera position for the given @data.
 * If necessary, this method will emit the 'OpenGLXsYs' signal.
 *
 * Returns: 1 if the 'OpenGLAskForReDraw' signal should be emitted.
 */
gboolean visu_data_setPositionOfView(VisuData *data,
				    float valueX, float valueY, int mask)
{
  gboolean res;

  g_return_val_if_fail(IS_VISU_DATA_TYPE(data), FALSE);
  
  res = openGLCameraSet_XsYs(data->privateDt->attachedView->camera, valueX,
			     valueY, mask);
  if (res)
    {

      openGLProject(data->privateDt->attachedView->window,
		    data->privateDt->attachedView->camera,
		    data->privateDt->extens[1]);

      g_signal_emit(data, visu_data_signals[OPENGL_XS_YS_CHANGED_SIGNAL],
		    0, data->privateDt->attachedView, NULL);
    }
    
  return res;
}

/**
 * visu_data_setZoomOfView:
 * @data: a #VisuData object ;
 * @value: a positive floating point value.
 *
 * This method is used to change the camera zoom for the given @data.
 * If necessary, this method will emit the 'OpenGLGross' signal and
 * the 'OpenGLFacetteChanged' signal.
 *
 * Returns: 1 if the 'OpenGLAskForReDraw' signal should be emitted.
 */
gboolean visu_data_setZoomOfView(VisuData *data, float value)
{
  gboolean res;

  g_return_val_if_fail(IS_VISU_DATA_TYPE(data), FALSE);

  res = openGLCameraSet_gross(data->privateDt->attachedView->camera, value);
  if (res)
    {

      openGLProject(data->privateDt->attachedView->window,
		    data->privateDt->attachedView->camera,
		    data->privateDt->extens[1]);
      g_signal_emit(data, visu_data_signals[OPENGL_NEAR_FAR_CHANGED_SIGNAL],
		    0, data->privateDt->attachedView, NULL);
      g_signal_emit(data, visu_data_signals[OPENGL_GROSS_CHANGED_SIGNAL],
		    0 , data->privateDt->attachedView, NULL);
      g_signal_emit(data, visu_data_signals[OPENGL_FACETTES_CHANGED_SIGNAL],
		    0 , NULL);
      visu_data_createAllElements(data);
    }
  return res;
}

/**
 * visu_data_setPerspectiveOfView:
 * @data: a #VisuData object ;
 * @value: a positive floating point value (> 1.1).
 *
 * This method is used to change the camera perspective for the given @data.
 * If necessary, this method will emit the 'OpenGLPersp' signal and
 * the 'OpenGLFacetteChanged' signal.
 *
 * Returns: 1 if the 'OpenGLAskForReDraw' signal should be emitted.
 */
gboolean visu_data_setPerspectiveOfView(VisuData *data, float value)
{
  gboolean res;

  g_return_val_if_fail(IS_VISU_DATA_TYPE(data), FALSE);
  
  res = openGLCameraSet_persp(data->privateDt->attachedView->camera, value);
  if (res)
    {
      openGLModelize(data->privateDt->attachedView->camera);

      openGLProject(data->privateDt->attachedView->window,
		    data->privateDt->attachedView->camera,
		    data->privateDt->extens[1]);

      g_signal_emit(data, visu_data_signals[OPENGL_NEAR_FAR_CHANGED_SIGNAL],
		    0, data->privateDt->attachedView, NULL);
      g_signal_emit(data, visu_data_signals[OPENGL_PERSP_CHANGED_SIGNAL],
		    0 , data->privateDt->attachedView, NULL);
      g_signal_emit(data, visu_data_signals[OPENGL_FACETTES_CHANGED_SIGNAL],
		    0 , NULL);
      visu_data_createAllElements(data);
    }
    
  return res;
}

/**
 * visu_data_setSizeOfView:
 * @data: a valid #VisuData object ;
 * @width: the new horizontal size ;
 * @height: the new vertical size.
 *
 * It changes the size of the OpenGl area and reccompute the OpenGL viewport.
 * Warning : it doesn't change the size of the window.
 *
 * Returns: 1 if the 'OpenGLAskForReDraw' signal should be emitted.
 */
int visu_data_setSizeOfView(VisuData *data, guint width, guint height)
{
  gboolean res;

  g_return_val_if_fail(IS_VISU_DATA_TYPE(data), 0);
  
  res = openGLWindowSet_viewport(data->privateDt->attachedView->window,
				 width, height);
  if (res)
    {
      if(!(data->privateDt->extens[1] > 0.f))
        visu_data_applyBoxGeometry(data);
      openGLProject(data->privateDt->attachedView->window,
		    data->privateDt->attachedView->camera,
		    data->privateDt->extens[1]);

      g_signal_emit(data, visu_data_signals[OPENGL_NEAR_FAR_CHANGED_SIGNAL],
		    0, data->privateDt->attachedView, NULL);
      g_signal_emit(data, visu_data_signals[OPENGL_WIDTH_HEIGHT_CHANGED_SIGNAL],
		    0 , data->privateDt->attachedView, NULL);
      g_signal_emit(data, visu_data_signals[OPENGL_FACETTES_CHANGED_SIGNAL],
		    0 , NULL);
      visu_data_createAllElements(data);
    }
    
  return res;
}



/*****************/
/* Miscellaneous */
/*****************/

/**
 * visu_data_getAllObjects:
 *
 * This methods is used to retrieve all #VisuObject currently allocated
 * in V_Sim. It is usefull to apply some changes on all objects (resources
 * for example).
 *
 * Returns: (transfer none): a list of V_Sim own #VisuData objects.
 */
GList* visu_data_getAllObjects(void)
{
  DBG_fprintf(stderr, "Visu Data: get the allObjects list.\n");
  return allObjects;
}
/**
 * visu_data_Trace:
 * @data: the #VisuData to trace.
 *
 * This method gives informations about the argument.
 */
void visu_data_Trace(VisuData *data)
{
  VisuElement *ele;
  unsigned int i;

  if (!data)
    return;

  fprintf(stderr, "Visu Data: data %d.\n", GPOINTER_TO_INT(data));
  fprintf(stderr, " | nb VisuElements : %d\n", data->ntype);
  for (i = 0; i < data->ntype; i++)
    {
      ele = data->fromIntToVisuElement[i];
      fprintf(stderr, " | VisuElements '%s' (%p), %d allocated VisuNodes"
	      " and %d stored.\n", data->fromIntToVisuElement[i]->name,
	      (gpointer)data->fromIntToVisuElement[i],
	      data->privateDt->nodeArray->numberOfNodes[i],
	      data->privateDt->nodeArray->numberOfStoredNodes[i]);
    }
}

/**
 * visu_data_setColorFunc:
 * @data: a #VisuData object ;
 * @func: (scope call): a method that colorize the nodes.
 *
 * This is a little trick to colorized the nodes. It should not be used since it
 * will probably be different in future release.
 */
void visu_data_setColorFunc(VisuData *data, VisuDataColorFunc func)
{
  g_return_if_fail(data);

  DBG_fprintf(stderr, "Visu Data: set the color method to %d.\n",
	      GPOINTER_TO_INT(func));
  data->privateDt->setColor = func;
}
/**
 * visu_data_getUserColor:
 * @data: a #VisuData object ;
 * @ele: a #VisuElement object ;
 * @node: a #VisuNode object ;
 * @rgba: (array fixed-size=4): a location to store the color.
 *
 * If a user defined color has been set (see
 * visu_data_setColorFunc()), then call this method to obtain a color
 * for the given node.
 *
 * Returns: TRUE if a user color has been defined.
 *
 * Since: 3.6
 */
gboolean visu_data_getUserColor(VisuData *data, VisuElement *ele,
				VisuNode *node, float rgba[4])
{
  if (!data->privateDt->setColor)
    return FALSE;

  data->privateDt->setColor(data, rgba, ele, node);
  return TRUE;
}
/**
 * visu_data_hasUserColorFunc:
 * @data: a #VisuData object.
 * 
 * Test the existence of a user defined colourisation function.
 *
 * Returns: TRUE if a user color function has been defined with
 * visu_data_setColorFunc().
 *
 * Since: 3.6
 */
gboolean visu_data_hasUserColorFunc(VisuData *data)
{
  return (data->privateDt->setColor)?TRUE:FALSE;
}

/**
 * visu_data_setNodeScalingFunc:
 * @data: a #VisuData object ;
 * @scaling: (scope call): a scaling routine.
 *
 * Change the scaling routine when nodes are drawn.
 *
 * Since: 3.5
 */
void visu_data_setNodeScalingFunc(VisuData *data, VisuDataScalingFunc scaling)
{
  g_return_if_fail(IS_VISU_DATA_TYPE(data));

  if (scaling)
    data->privateDt->scaling = scaling;
  else
    data->privateDt->scaling = defaultScaling;
}
/**
 * visu_data_getNodeScalingFactor:
 * @data: a #VisuData object.
 * @node: a #VisuNode object.
 *
 * One can modify the size of a given node using a routine set by
 * visu_data_setNodeScalingFunc(). By default the scaling is 1.
 *
 * Since: 3.5
 *
 * Returns: the scaling factor to be applied to node @node.
 */
float visu_data_getNodeScalingFactor(VisuData *data, VisuNode *node)
{
  g_return_val_if_fail(IS_VISU_DATA_TYPE(data), 0.f);

  return data->privateDt->scaling(data, node);
}
static float defaultScaling(VisuData *data _U_, VisuNode *node _U_)
{
  return 1.f;
}



/***********************/
/* The node properties */
/***********************/

/**
 * visu_data_getNodeArray:
 * @data: a #VisuData object.
 *
 * This method retrieve the #VisuNodeArray associated to the given @data.
 *
 * Returns: (transfer none): the associated #VisuNodeArray.
 */
VisuNodeArray* visu_data_getNodeArray(VisuData *data)
{
  g_return_val_if_fail(IS_VISU_DATA_TYPE(data), (VisuNodeArray*)0);
  return (data->privateDt->nodeArray);
}


/****************/
/* The iterator */
/****************/

/**
 * visu_data_iterNew:
 * @data: a #VisuData object ;
 * @iter: (out caller-allocates) (transfer full): an alocated iterator.
 *
 * Set values to a #VisuDataIter object to iterate over nodes.
 * Its contain is initialised with the data size (number of elements,
 * number of nodes per element...).
 */
void visu_data_iterNew(VisuData *data, VisuDataIter *iter)
{
  g_return_if_fail(iter);

  iter->nAllStoredNodes = 0;
  iter->nElements       = 0;
  iter->nStoredNodes    = 0;

  g_return_if_fail(IS_VISU_DATA_TYPE(data) && data->privateDt->nodeArray);

  iter->data            = data;
  iter->idMax           = data->privateDt->nodeArray->idCounter - 1;
  iter->nAllStoredNodes = data->privateDt->nodeArray->nbOfAllStoredNodes;
  iter->nElements       = data->privateDt->nodeArray->ntype;
  iter->nStoredNodes    = data->privateDt->nodeArray->numberOfStoredNodes;
  iter->iElement        = -1;
  iter->node            = (VisuNode*)0;
  iter->element         = (VisuElement*)0;
  iter->type            = ITER_NODES_BY_TYPE;
  iter->init            = FALSE;
}

/**
 * visu_data_iterStart:
 * @data: a #VisuData object ;
 * @iter: a #VisuDataIter object.
 *
 * Initialise the node and element internal pointers for a run over the nodes.
 */
void visu_data_iterStart(VisuData *data, VisuDataIter *iter)
{
  g_return_if_fail(IS_VISU_DATA_TYPE(data) && iter && data == iter->data);

  iter->init = TRUE;

  iter->iElement = -1;
  iter->node     = (VisuNode*)0;
  iter->element  = (VisuElement*)0;
  if (data->ntype == 0)
    return;

  iter->iElement = 0;
  iter->element  = data->fromIntToVisuElement[0];
  /* We look for an element with stored nodes. */
  while (data->privateDt->nodeArray->numberOfStoredNodes[iter->iElement] == 0)
    {
      iter->iElement += 1;
      if (iter->iElement >= data->privateDt->nodeArray->ntype)
	{
	  /* We found nothing. */
	  iter->iElement = -1;
	  iter->element  = (VisuElement*)0;
	  return;
	}
      iter->element   = data->fromIntToVisuElement[iter->iElement];
    }

  iter->node     = data->privateDt->nodeArray->nodes[iter->iElement];
}

/**
 * visu_data_iterStartNumber:
 * @data: a #VisuData object ;
 * @iter: a #VisuDataIter object.
 *
 * Initialise the node and element internal pointers for a run
 * following the node oder.
 */
void visu_data_iterStartNumber(VisuData *data, VisuDataIter *iter)
{
  guint i;

  g_return_if_fail(IS_VISU_DATA_TYPE(data) && iter && data == iter->data);

  iter->init = TRUE;

  iter->iElement = -1;
  iter->node     = (VisuNode*)0;
  iter->element  = (VisuElement*)0;
  if (data->ntype == 0)
    return;

  i = 0;
  iter->node = (VisuNode*)0;
  do
    {
      iter->node = visu_data_getNodeFromNumber(data, i);
      i += 1;
    }
  while (!iter->node && i < data->privateDt->nodeArray->nodeTableSize);
  if (!iter->node)
    return;
  iter->iElement = iter->node->posElement;
  iter->element  = data->fromIntToVisuElement[iter->iElement];
}

/**
 * visu_data_iterStartVisible:
 * @data: a #VisuData object ;
 * @iter: a #VisuDataIter object.
 *
 * Initialise the node and element internal pointers for a run over the 
 * visible nodes (see visu_data_iterNextVisible).
 */
void visu_data_iterStartVisible(VisuData *data, VisuDataIter *iter)
{
  visu_data_iterStart(data, iter);
  if (iter->node && iter->node->rendered && iter->element->rendered)
    /* Ok, first is good. */
    return;

  /* First was not visible, we go next. */
  visu_data_iterNextVisible(data, iter);
}

/**
 * visu_data_iterRestartNode:
 * @data: a #VisuData object ;
 * @iter: a #VisuDataIter object.
 *
 * The element internal pointer must be associated. Then, it returns the
 * node pointer to the first node for this element.
 */
void visu_data_iterRestartNode(VisuData *data, VisuDataIter *iter)
{
  g_return_if_fail(IS_VISU_DATA_TYPE(data) && iter && data == iter->data);
  g_return_if_fail(iter->iElement < data->ntype);

  iter->init = TRUE;

  iter->node = data->privateDt->nodeArray->nodes[iter->iElement];
}

/**
 * visu_data_iterNext:
 * @data: a #VisuData object ;
 * @iter: a #VisuDataIter object.
 *
 * Modify node and element internal pointers to the next node, or NULL if
 * none remains.
 */
void visu_data_iterNext(VisuData *data, VisuDataIter *iter)
{
  unsigned int iNode;
  VisuNodeArray *array;

  g_return_if_fail(IS_VISU_DATA_TYPE(data) && iter && data == iter->data);
  g_return_if_fail(iter->init && iter->node && iter->iElement == iter->node->posElement);
  
  array = data->privateDt->nodeArray;
  iNode = iter->node->posNode + 1;
  if (iNode < array->numberOfStoredNodes[iter->iElement])
    iter->node = array->nodes[iter->iElement] + iNode;
  else
    {
      iter->iElement += 1;
      if (iter->iElement >= array->ntype)
	{
	  iter->iElement = -1;
	  iter->node     = (VisuNode*)0;
	  iter->element  = (VisuElement*)0;
	}
      else
	{
	  iter->node     = array->nodes[iter->iElement];
	  iter->element  = data->fromIntToVisuElement[iter->iElement];
	}
    }
}

/**
 * visu_data_iterNextVisible:
 * @data: a #VisuData object ;
 * @iter: a #VisuDataIter object.
 *
 * Go to the next rendered node (changing element if required).
 */
void visu_data_iterNextVisible(VisuData *data, VisuDataIter *iter)
{
  g_return_if_fail(IS_VISU_DATA_TYPE(data) && iter && data == iter->data);

  /* Get the next node, and test if it is rendered. */
  visu_data_iterNext(data, iter);
  if (!iter->node || (iter->element->rendered && iter->node->rendered))
    return;

  /* From the current node, we go next to find one that is rendred. */
  for (; iter->element; visu_data_iterNextElement(data, iter))
    if (iter->element->rendered)
      for (; iter->node; visu_data_iterNextNode(data, iter))
	if (iter->node->rendered)
	  return;
}

/**
 * visu_data_iterNextNode:
 * @data: a #VisuData object ;
 * @iter: a #VisuDataIter object.
 *
 * Modify node internal pointer to the next node, or NULL if
 * none remains. Contrary to visu_data_iterNext() it does not go to the
 * next element if one exists.
 */
void visu_data_iterNextNode(VisuData *data, VisuDataIter *iter)
{
  g_return_if_fail(IS_VISU_DATA_TYPE(data) && iter && data == iter->data);
  g_return_if_fail(iter->init && iter->node);

  if (iter->node->posNode + 1 <
      data->privateDt->nodeArray->numberOfStoredNodes[iter->node->posElement])
    iter->node = iter->node + 1;
  else
    iter->node = (VisuNode*)0;
}
/**
 * visu_data_iterNextNodeOriginal:
 * @data: a #VisuData object ;
 * @iter: a #VisuDataIter object.
 *
 * Modify node internal pointer to the next original node, or NULL if
 * none remains. Contrary to visu_data_iterNext() it does not go to the
 * next element if one exists.
 *
 * Since: 3.6
 */
void visu_data_iterNextNodeOriginal(VisuData *data, VisuDataIter *iter)
{
  g_return_if_fail(IS_VISU_DATA_TYPE(data) && iter && data == iter->data);
  g_return_if_fail(iter->init && iter->node);

  do
    {
      if (iter->node->posNode + 1 <
	  data->privateDt->nodeArray->numberOfStoredNodes[iter->node->posElement])
	iter->node = iter->node + 1;
      else
	iter->node = (VisuNode*)0;
    }
  while (iter->node &&
	 visu_node_array_getOriginal(data->privateDt->nodeArray, iter->node->number) >= 0);
}

/**
 * visu_data_iterNextNodeNumber:
 * @data: a #VisuData object ;
 * @iter: a #VisuDataIter object.
 *
 * Modify node internal pointer to the next node, increasing the id of
 * the current node. The element internal pointer is also updated
 * accordingly. If no more nodes exist after the given one, node and
 * element internal pointers are set to NULL.
 */
void visu_data_iterNextNodeNumber(VisuData *data, VisuDataIter *iter)
{
  unsigned int i;

  g_return_if_fail(IS_VISU_DATA_TYPE(data) && iter && data == iter->data);
  g_return_if_fail(iter->init && iter->node);

  for (i = iter->node->number + 1;
       !(iter->node = visu_data_getNodeFromNumber(data, i)) &&
	 (i < data->privateDt->nodeArray->idCounter) ; i++);

  if (iter->node)
    {
      iter->iElement = iter->node->posElement;
      iter->element  = data->fromIntToVisuElement[iter->iElement];
    }
  else
    iter->element = (VisuElement*)0;
}

/**
 * visu_data_iterNextElement:
 * @data: a #VisuData object ;
 * @iter: a #VisuDataIter object.
 *
 * Modify element internal pointer to the next element and set node
 * to the first one, or NULL if none remains.
 */
void visu_data_iterNextElement(VisuData *data, VisuDataIter *iter)
{
  g_return_if_fail(IS_VISU_DATA_TYPE(data) && iter && data == iter->data);
  g_return_if_fail(iter->init && iter->iElement < data->ntype);

  do
    iter->iElement += 1;
  while(iter->iElement < data->privateDt->nodeArray->ntype &&
	iter->nStoredNodes[iter->iElement] == 0);

  if (iter->iElement == data->privateDt->nodeArray->ntype)
    {
      iter->iElement = -1;
      iter->node     = (VisuNode*)0;
      iter->element  = (VisuElement*)0;
    }
  else
    {
      iter->node     = data->privateDt->nodeArray->nodes[iter->iElement];
      iter->element  = data->fromIntToVisuElement[iter->iElement];
    }
}


/*************************************/
/* Additionnal routines for bindings */
/*************************************/
/**
 * visu_data_iter_next:
 * @dataIter: a #VisuDataIter object.
 *
 * Run the iterator to go to next item.
 *
 * Since: 3.6
 * 
 * Returns: TRUE if any item is found, FALSE otherwise.
 */
gboolean visu_data_iter_next(VisuDataIter *iter)
{
  if (!iter->init)
    switch (iter->type)
      {
      case ITER_NODES_BY_TYPE:
      case ITER_ELEMENTS:
	visu_data_iterStart(iter->data, iter);
	break;
      case ITER_NODES_BY_NUMBER:
      case ITER_NODES_ORIGINAL:
	visu_data_iterStartNumber(iter->data, iter);
	break;
      case ITER_NODES_VISIBLE:
	visu_data_iterStartVisible(iter->data, iter);
	break;
      }
  else
    switch (iter->type)
      {
      case ITER_NODES_BY_TYPE:
	visu_data_iterNext(iter->data, iter);
	break;
      case ITER_NODES_BY_NUMBER:
	visu_data_iterNextNodeNumber(iter->data, iter);
	break;
      case ITER_NODES_VISIBLE:
	visu_data_iterNextVisible(iter->data, iter);
	break;
      case ITER_NODES_ORIGINAL:
	visu_data_iterNextNodeOriginal(iter->data, iter);
	break;
      case ITER_ELEMENTS:
	visu_data_iterNextElement(iter->data, iter);
	break;
      }
  
  if (iter->node)
    return TRUE;
  else
    return FALSE;
}
/**
 * visu_data_iter_next2:
 * @iter1: a #VisuDataIter object.
 * @iter2: a #VisuDataIter object.
 *
 * Iterator to run on a pair of different nodes.
 *
 * Returns: TRUE if any item is found, FALSE otherwise.
 *
 * Since: 3.6
 */
gboolean visu_data_iter_next2(VisuDataIter *iter1, VisuDataIter *iter2)
{
  if (!iter1->init)
    {
      visu_data_iterStart(iter1->data, iter1);
      visu_data_iterStart(iter1->data, iter2);
    }
  else
    {
      if (!iter1->node)
        return FALSE;

      /* DBG_fprintf(stderr, "go next %p-%p ->", (gpointer)iter1->node, (gpointer)iter2->node); */
      visu_data_iterNext(iter1->data, iter2);
      if (!iter2->node ||
	  iter2->node->posElement > iter1->node->posElement ||
	  (iter2->node->posElement == iter1->node->posElement &&
	   iter2->node->posNode    >= iter1->node->posNode))
	{
	  visu_data_iterNext(iter1->data, iter1);
	  if (iter1->node)
	    visu_data_iterStart(iter1->data, iter2);
	  else
	    iter2->node = (VisuNode*)0;
	}
      /* DBG_fprintf(stderr, " %p-%p\n", (gpointer)iter1->node, (gpointer)iter2->node); */
    }

  if (!iter1->node && !iter2->node)
    return FALSE;
  else
    return TRUE;
}
