/*   EXTRAITS DE LA LICENCE
	Copyright CEA, contributeurs : Luc BILLARD et Damien
	CALISTE, laboratoire L_Sim, (2001-2005)
  
	Adresse ml :
	BILLARD, non joignable par ml ;
	CALISTE, damien P caliste AT cea P fr.

	Ce logiciel est un programme informatique servant  visualiser des
	structures atomiques dans un rendu pseudo-3D. 

	Ce logiciel est rgi par la licence CeCILL soumise au droit franais et
	respectant les principes de diffusion des logiciels libres. Vous pouvez
	utiliser, modifier et/ou redistribuer ce programme sous les conditions
	de la licence CeCILL telle que diffuse par le CEA, le CNRS et l'INRIA 
	sur le site "http://www.cecill.info".

	Le fait que vous puissiez accder  cet en-tte signifie que vous avez 
	pris connaissance de la licence CeCILL, et que vous en avez accept les
	termes (cf. le fichier Documentation/licence.fr.txt fourni avec ce logiciel).
*/

/*   LICENCE SUM UP
	Copyright CEA, contributors : Luc BILLARD et Damien
	CALISTE, laboratoire L_Sim, (2001-2005)

	E-mail address:
	BILLARD, not reachable any more ;
	CALISTE, damien P caliste AT cea P fr.

	This software is a computer program whose purpose is to visualize atomic
	configurations in 3D.

	This software is governed by the CeCILL  license under French law and
	abiding by the rules of distribution of free software.  You can  use, 
	modify and/ or redistribute the software under the terms of the CeCILL
	license as circulated by CEA, CNRS and INRIA at the following URL
	"http://www.cecill.info". 

	The fact that you are presently reading this means that you have had
	knowledge of the CeCILL license and that you accept its terms. You can
	find a copy of this licence shipped with this software at Documentation/licence.en.txt.
*/
#include "box.h"

#include <GL/gl.h>
#include <GL/glu.h> 

#include <math.h>

#include <opengl.h>
#include <visu_object.h>
#include <visu_tools.h>
#include <visu_configFile.h>
#include <openGLFunctions/objectList.h>
#include <openGLFunctions/text.h>
#include <coreTools/toolColor.h>
#include <coreTools/toolConfigFile.h>

/**
 * SECTION:box
 * @short_description: Draw a bounding box around nodes.
 *
 * <para>This extension allows V_Sim to draw a box around the
 * nodes. The box is defined in the #VisuData structure and can be
 * retrieved with visu_data_getBoxGeometry(). This box is not necessary
 * orthogonal.</para>
 * <para>It has several properties, namely, its colour, its line width
 * and its line pattern. It is represented in OpenGL with simple lines
 * and is affected by the antialiasing property. Defined resources:</para>
 * <itemizedlist>
 *  <listitem>
 *   <para><emphasis>box_is_on</emphasis> (boolean): controls if a box
 *   is drawn around the rendering area (since 3.0).</para>
 *  </listitem>
 *  <listitem>
 *   <para><emphasis>box_color</emphasis> (RGB in [0;1]): defines the
 *   color of the box(since 3.0).</para>
 *  </listitem>
 *  <listitem>
 *   <para><emphasis>box_line_width</emphasis> (integer in [1;10]):
 *   defines the width of the lines of the box (since 3.0).</para>
 *  </listitem>
 *  <listitem>
 *   <para><emphasis>box_line_stipple</emphasis> (2 integers in
 *   ]0;65535]): dot scheme detail for the lines of the box. The first
 *   value is the pattern for the line of the main box and the second
 *   is the pattern for the lines of the expanded areas (since 3.4).</para>
 *  </listitem>
 * </itemizedlist>
 */




static VisuExtension* extensionBox, *extBoxLegend;
static float boxRGBDefault[3] = {1.0, 0.5, 0.1};
static gboolean boxHasBeenBuilt, legendHasBeenBuilt;

/* Parameters & resources*/
/* This is a boolean to control is the box is render or not. */
#define FLAG_RESOURCE_BOX_USED   "box_is_on"
#define DESC_RESOURCE_BOX_USED   "Control if a box is drawn around the rendering area ; boolean (0 or 1)"
#define RESOURCE_BOX_USED_DEFAULT 0
static gboolean readBoxIsOn(gchar **lines, int nbLines,
			    int position, VisuData *dataObj, GError **error);
/* A resource to control the color used to render the lines of the box. */
#define FLAG_RESOURCE_BOX_COLOR   "box_color"
#define DESC_RESOURCE_BOX_COLOR   "Define the color of the box ; three floating point values (0. <= v <= 1.)"
static float boxRGB[3];
static gboolean readBoxColor(gchar **lines, int nbLines,
		       int position, VisuData *dataObj, GError **error);
/* A resource to control the width to render the lines of the box. */
#define FLAG_RESOURCE_BOX_LINE   "box_line_width"
#define DESC_RESOURCE_BOX_LINE   "Define the width of the lines of the box ; one integer (1. <= v <= 10.)"
#define RESOURCE_BOX_LINE_DEFAULT 1.
static float boxLineWidth;
static gboolean readBoxLineWidth(gchar **lines, int nbLines,
				 int position, VisuData *dataObj, GError **error);
/* A resource to control the stipple to render the lines of the box. */
#define FLAG_RESOURCE_BOX_STIPPLE   "box_line_stipple"
#define DESC_RESOURCE_BOX_STIPPLE   "Dot scheme detail for the lines of the box (main and expanded) ; 0 < 2 integers < 2^16"
#define RESOURCE_BOX_STIPPLE_DEFAULT 65535
#define RESOURCE_EXPAND_STIPPLE_DEFAULT 65280
static guint16 boxLineStipple[2];
static gboolean readBoxLineStipple(gchar **lines, int nbLines,
				   int position, VisuData *dataObj, GError **error);

/* A resource to control if the box lengths are printed. */
#define FLAG_RESOURCE_BOX_LENGTHS   "box_show_lengths"
#define DESC_RESOURCE_BOX_LENGTHS   "Print the box lengths ; boolean (0 or 1)"
#define RESOURCE_WITH_LENGTHS_DEFAULT FALSE
static gboolean readBoxLengths(gchar **lines, int nbLines,
			       int position, VisuData *dataObj, GError **error);

#define RESOURCE_WITH_BASIS_DEFAULT FALSE
static gboolean withBasis;
static float basisLength = 2.5f;

/* Export function that is called by visu_module to write the
   values of resources to a file. */
static void exportResourcesBox(GString *data, VisuData *dataObj);

static void rebuildBox(VisuData *dataObj);

/* Callbacks. */
static void onDataNewSignal(GObject *visu, VisuData *dataObj, gpointer data);
static void rebuildBoxOnDataReady(GObject *obj, VisuData *dataObj, gpointer data);
static void rebuildBoxOnResources(GObject *obj, VisuData *dataObj, gpointer data);
static void onBoxSizeChanged(VisuData *data, gpointer user_data);

VisuExtension* initExtensionBox()
{
  char *description = _("Draw a box representing the limit of the area.");
  int listBox;
  VisuConfigFileEntry *resourceEntry;

  DBG_fprintf(stderr,"Initialising the box OpenGL extension...\n");
  listBox = visu_openGL_objectList_new(1);
  extensionBox = visu_extension_new(VISU_GLEXT_BOX_ID, _("Box"),
				     description, listBox, rebuildBox);
  visu_extension_setPriority(extensionBox, VISU_EXTENSION_PRIORITY_LOW);

  extBoxLegend = visu_extension_new("boxLegend", _("Box legend"),
				     _("Draw informations related to the box."),
				     visu_openGL_objectList_new(1), rebuildBox);
  visu_extension_setPriority(extBoxLegend, VISU_EXTENSION_PRIORITY_LAST);
  visu_extension_setSaveOpenGLState(extBoxLegend, TRUE);
  extBoxLegend->used = RESOURCE_WITH_LENGTHS_DEFAULT;
  visuExtensions_add(extBoxLegend);

  g_signal_connect(VISU_INSTANCE, "dataNew",
		   G_CALLBACK(onDataNewSignal), (gpointer)0);
  g_signal_connect(VISU_INSTANCE, "dataReadyForRendering",
		   G_CALLBACK(rebuildBoxOnDataReady), (gpointer)0);
  g_signal_connect(VISU_INSTANCE, "resourcesLoaded",
 		   G_CALLBACK(rebuildBoxOnResources), (gpointer)0);

  resourceEntry = visu_configFile_addEntry(VISU_CONFIGFILE_RESOURCE,
					  FLAG_RESOURCE_BOX_USED,
					  DESC_RESOURCE_BOX_USED,
					  1, readBoxIsOn);
  resourceEntry = visu_configFile_addEntry(VISU_CONFIGFILE_RESOURCE,
					  FLAG_RESOURCE_BOX_COLOR,
					  DESC_RESOURCE_BOX_COLOR,
					  1, readBoxColor);
  resourceEntry = visu_configFile_addEntry(VISU_CONFIGFILE_RESOURCE,
					  FLAG_RESOURCE_BOX_LINE,
					  DESC_RESOURCE_BOX_LINE,
					  1, readBoxLineWidth);
  resourceEntry = visu_configFile_addEntry(VISU_CONFIGFILE_RESOURCE,
					  FLAG_RESOURCE_BOX_STIPPLE,
					  DESC_RESOURCE_BOX_STIPPLE,
					  1, readBoxLineStipple);
  visu_configFile_entry_setVersion(resourceEntry, 3.4f);
  resourceEntry = visu_configFile_addEntry(VISU_CONFIGFILE_RESOURCE,
					  FLAG_RESOURCE_BOX_LENGTHS,
					  DESC_RESOURCE_BOX_LENGTHS,
					  1, readBoxLengths);
  visu_configFile_entry_setVersion(resourceEntry, 3.6f);
  visu_configFile_addExportFunction(VISU_CONFIGFILE_RESOURCE,
				   exportResourcesBox);

  /* Initialisation des valeurs par dfaut. */
  visu_glExt_box_setRGBValues(boxRGBDefault, TOOL_COLOR_MASK_RGBA);
  extensionBox->used = RESOURCE_BOX_USED_DEFAULT;
  boxLineWidth       = RESOURCE_BOX_LINE_DEFAULT;
  boxHasBeenBuilt    = FALSE;
  legendHasBeenBuilt = FALSE;
  boxLineStipple[0]  = RESOURCE_BOX_STIPPLE_DEFAULT;
  boxLineStipple[1]  = RESOURCE_EXPAND_STIPPLE_DEFAULT;
  withBasis          = RESOURCE_WITH_BASIS_DEFAULT;

  return extensionBox;
}

/* Method used to change the value of the parameter box_color. */
gboolean visu_glExt_box_setRGBValues(float rgb[3], int mask)
{
  int diff = 0;
  
  if (mask & TOOL_COLOR_MASK_R && boxRGB[0] != rgb[0])
    {
      boxRGB[0] = rgb[0];
      diff = 1;
    }
  if (mask & TOOL_COLOR_MASK_G && boxRGB[1] != rgb[1])
    {
      boxRGB[1] = rgb[1];
      diff = 1;
    }
  if (mask & TOOL_COLOR_MASK_B && boxRGB[2] != rgb[2])
    {
      boxRGB[2] = rgb[2];
      diff = 1;
    }
  if (!diff)
    return FALSE;

  /* Il faut recrer la bote puisque des param ont changs. */
  boxHasBeenBuilt = FALSE;
/*   visu_glExt_box_draw(); */
  return (gboolean)extensionBox->used;
}
/* Method used to change the value of the parameter box_line_width. */
gboolean visu_glExt_box_setLineWidth(float width)
{
  if (width < 1. || width > 10. || width == boxLineWidth)
    return FALSE;

  boxLineWidth = width;
  /* Il faut recrer la bote puisque des param ont changs. */
  boxHasBeenBuilt = FALSE;
/*   visu_glExt_box_draw(); */
  return (gboolean)extensionBox->used;
}
/* Method used to change the value of the parameter box_is_on. */
gboolean visu_glExt_box_setOn(int value)
{
  if (value == extensionBox->used)
    return FALSE;

  extensionBox->used = value;
  return (value && (!boxHasBeenBuilt || !legendHasBeenBuilt));
}
gboolean visu_glExt_box_setLineStipple(guint16 stipple)
{
  if (stipple == boxLineStipple[0])
    return FALSE;

  boxLineStipple[0] = stipple;
  boxHasBeenBuilt = FALSE;
  return (gboolean)extensionBox->used;
}
gboolean visu_glExt_box_setExpandStipple(guint16 stipple)
{
  if (stipple == boxLineStipple[1])
    return FALSE;

  boxLineStipple[1] = stipple;
  boxHasBeenBuilt = FALSE;
  return (gboolean)extensionBox->used;
}
gboolean visu_glExt_box_setShowLegend(gboolean value)
{
  if (value == extBoxLegend->used)
    return FALSE;

  extBoxLegend->used = value;
  return (value && (!boxHasBeenBuilt || !legendHasBeenBuilt));
}
/* Get methods. */
float* visu_glExt_box_getRGBValues()
{
  return boxRGB;
}
int visu_glExt_box_getOn()
{
  if (extensionBox)
    return extensionBox->used;
  else
    return 0;
}
float visu_glExt_box_getLineWidth()
{
  return boxLineWidth;
}
guint16 visu_glExt_box_getLineStipple()
{
  return boxLineStipple[0];
}
guint16 visu_glExt_box_getExpandStipple()
{
  return boxLineStipple[1];
}
/**
 * visu_glExt_box_getShowLegend:
 *
 * The box extension can draw a frame showing the lengths of the box.
 *
 * Since: 3.6
 *
 * Returns: TRUE if the frame is displayed.
 */
gboolean visu_glExt_box_getShowLegend()
{
  return extBoxLegend->used;
}


/****************/
/* Private part */
/****************/

static void onDataNewSignal(GObject *visu _U_, VisuData *dataObj, gpointer data _U_)
{
  if (dataObj)
    g_signal_connect(G_OBJECT(dataObj), "BoxSizeChanged",
		     G_CALLBACK(onBoxSizeChanged), (gpointer)0);
}
static void rebuildBox(VisuData *dataObj)
{
  /* Force redraw. */
  boxHasBeenBuilt = FALSE;
  visu_glExt_box_draw(dataObj);
  legendHasBeenBuilt = FALSE;
  visu_glExt_box_draw_legend(dataObj);
}
static void rebuildBoxOnDataReady(GObject *obj _U_, VisuData *dataObj,
				  gpointer data _U_)
{
  if (dataObj)
    {
      DBG_fprintf(stderr, "Extension box: caught the 'dataReadyForRendering' signal.\n");
      boxHasBeenBuilt = FALSE;
      visu_glExt_box_draw(dataObj);
      legendHasBeenBuilt = FALSE;
      visu_glExt_box_draw_legend(dataObj);
    }
}
static void rebuildBoxOnResources(GObject *obj _U_, VisuData *dataObj _U_,
				  gpointer data _U_)
{
  boxHasBeenBuilt = FALSE;
  legendHasBeenBuilt = FALSE;
}
static void onBoxSizeChanged(VisuData *data, gpointer user_data _U_)
{
  DBG_fprintf(stderr, "Extension Box: caught the 'BoxSizeChanged' signal.\n");

  boxHasBeenBuilt = FALSE;
  visu_glExt_box_draw(data);
  legendHasBeenBuilt = FALSE;
  visu_glExt_box_draw_legend(data);
}

void visu_glExt_box_draw(VisuData *data)
{
  int i, j, k;
  float v[8][3];
  float ext[3], centre[3];
  float material[5] = {.5f, .5f, .2f, .5f, .0f};
  GLUquadricObj *obj;
  gchar strLg[64];

  if (boxHasBeenBuilt)
    return;

  DBG_fprintf(stderr, "Extension box: creating box for"
	      " VisuData %p.\n", (gpointer)data);
  boxHasBeenBuilt = TRUE;
  visu_data_getBoxVertices(data, v, FALSE);
  glDeleteLists(extensionBox->objectListId, 1);
  glNewList(extensionBox->objectListId, GL_COMPILE);
  glLineWidth(boxLineWidth);
  glColor3f(boxRGB[0], boxRGB[1], boxRGB[2]);
  glDisable(GL_LIGHTING);
  glDisable(GL_DITHER);
  /* Draw the basic lines. */
  if (boxLineStipple[0] != 65535)
    {
      glEnable(GL_LINE_STIPPLE);
      glLineStipple(1, boxLineStipple[0]);
    }
  glBegin(GL_LINES);
  glVertex3fv(v[0]);
  glVertex3fv(v[1]);
  glVertex3fv(v[1]);
  glVertex3fv(v[2]);
  glVertex3fv(v[2]);
  glVertex3fv(v[3]);
  glVertex3fv(v[3]);
  glVertex3fv(v[0]);
  glVertex3fv(v[4]);
  glVertex3fv(v[5]);
  glVertex3fv(v[5]);
  glVertex3fv(v[6]);
  glVertex3fv(v[6]);
  glVertex3fv(v[7]);
  glVertex3fv(v[7]);
  glVertex3fv(v[4]);
  glVertex3fv(v[0]);
  glVertex3fv(v[4]);
  glVertex3fv(v[1]);
  glVertex3fv(v[5]);
  glVertex3fv(v[2]);
  glVertex3fv(v[6]);
  glVertex3fv(v[3]);
  glVertex3fv(v[7]);
  glEnd();
  if (boxLineStipple[0] != 65535)
    glDisable(GL_LINE_STIPPLE);
  /* Draw the extension lines. */
  visu_data_getExtension(data, ext);
  if (ext[0] > 0. || ext[1] > 0. || ext[2] > 0.)
    {
      if (boxLineStipple[1] != 65535)
	{
	  glEnable(GL_LINE_STIPPLE);
	  glLineStipple(1, boxLineStipple[1]);
	}
      glBegin(GL_LINES);
      /* X coordinate. */
      for (j = -(int)ext[1]; j < 2 + (int)ext[1]; j++)
	for (k = -(int)ext[2]; k < 2 + (int)ext[2]; k++)
	  {
	    glVertex3f(-ext[0] * v[1][0] +  v[3][0] * j + v[4][0] * k,
		       -ext[0] * v[1][1] +  v[3][1] * j + v[4][1] * k,
		       -ext[0] * v[1][2] +  v[3][2] * j + v[4][2] * k);
	    if ((j == 0 || j == 1) && (k == 0 || k == 1))
	      {
		glVertex3f(v[3][0] * j + v[4][0] * k,
			   v[3][1] * j + v[4][1] * k,
			   v[3][2] * j + v[4][2] * k);
		glVertex3f(v[1][0] +  v[3][0] * j + v[4][0] * k,
			   v[1][1] +  v[3][1] * j + v[4][1] * k,
			   v[1][2] +  v[3][2] * j + v[4][2] * k);
	      }
	    glVertex3f((1. + ext[0]) * v[1][0] +  v[3][0] * j + v[4][0] * k,
		       (1. + ext[0]) * v[1][1] +  v[3][1] * j + v[4][1] * k,
		       (1. + ext[0]) * v[1][2] +  v[3][2] * j + v[4][2] * k);
	  }
      /* Y coordinate. */
      for (i = -(int)ext[0]; i < 2 + (int)ext[0]; i++)
	for (k = -(int)ext[2]; k < 2 + (int)ext[2]; k++)
	  {
	    glVertex3f(-ext[1] * v[3][0] +  v[1][0] * i + v[4][0] * k,
		       -ext[1] * v[3][1] +  v[1][1] * i + v[4][1] * k,
		       -ext[1] * v[3][2] +  v[1][2] * i + v[4][2] * k);
	    if ((i == 0 || i == 1) && (k == 0 || k == 1))
	      {
		glVertex3f(v[1][0] * i + v[4][0] * k,
			   v[1][1] * i + v[4][1] * k,
			   v[1][2] * i + v[4][2] * k);
		glVertex3f(v[3][0] +  v[1][0] * i + v[4][0] * k,
			   v[3][1] +  v[1][1] * i + v[4][1] * k,
			   v[3][2] +  v[1][2] * i + v[4][2] * k);
	      }
	    glVertex3f((1. + ext[1]) * v[3][0] +  v[1][0] * i + v[4][0] * k,
		       (1. + ext[1]) * v[3][1] +  v[1][1] * i + v[4][1] * k,
		       (1. + ext[1]) * v[3][2] +  v[1][2] * i + v[4][2] * k);
	  }
      /* Z coordinate. */
      for (i = -(int)ext[0]; i < 2 + (int)ext[0]; i++)
	for (j = -(int)ext[1]; j < 2 + (int)ext[1]; j++)
	  {
	    glVertex3f(-ext[2] * v[4][0] +  v[1][0] * i + v[3][0] * j,
		       -ext[2] * v[4][1] +  v[1][1] * i + v[3][1] * j,
		       -ext[2] * v[4][2] +  v[1][2] * i + v[3][2] * j);
	    if ((j == 0 || j == 1) && (i == 0 || i == 1))
	      {
		glVertex3f(v[1][0] * i + v[3][0] * j,
			   v[1][1] * i + v[3][1] * j,
			   v[1][2] * i + v[3][2] * j);
		glVertex3f(v[4][0] +  v[1][0] * i + v[3][0] * j,
			   v[4][1] +  v[1][1] * i + v[3][1] * j,
			   v[4][2] +  v[1][2] * i + v[3][2] * j);
	      }
	    glVertex3f((1. + ext[2]) * v[4][0] +  v[1][0] * i + v[3][0] * j,
		       (1. + ext[2]) * v[4][1] +  v[1][1] * i + v[3][1] * j,
		       (1. + ext[2]) * v[4][2] +  v[1][2] * i + v[3][2] * j);
	  }
      glEnd();
      if (boxLineStipple[1] != 65535)
	glDisable(GL_LINE_STIPPLE);
    }
  glEnable(GL_LIGHTING);
  glEnable(GL_DITHER); /* WARNING: it is the default! */
  /* Draw the basis set if needed. */
  if (withBasis)
    {
      visu_data_getBoxCentre(data, centre);

      obj = gluNewQuadric();
      openGLSet_highlightColor(material, boxRGB, 1.f);

      /* Draw the basis set. */
      glPushMatrix();
      glTranslated(0., 0., 0.);
      glRotated(90., 0, 1, 0);  
      visu_openGL_drawSmoothArrow(obj, -1, VISU_OPENGL_ARROW_BOTTOM_CENTERED,
				       basisLength - 0.3f, 0.1f, 10, FALSE,
				       0.3f, 0.2f, 10, FALSE);
      glRasterPos3f(0.0f, 0.0f, basisLength);
      sprintf(strLg, _("x: %7.3f"), centre[0]);
      openGLText_drawChars(strLg, TEXT_SMALL); 
      glPopMatrix();
      glPushMatrix();
      glTranslated(0., 0., 0.);
      glRotated(-90., 1, 0, 0);  
      visu_openGL_drawSmoothArrow(obj, -1, VISU_OPENGL_ARROW_BOTTOM_CENTERED,
				       basisLength - 0.3f, 0.1f, 10, FALSE,
				       0.3f, 0.2f, 10, FALSE);
      glRasterPos3f(0.0f, 0.0f, basisLength);
      sprintf(strLg, _("y: %7.3f"), centre[1]);
      openGLText_drawChars(strLg, TEXT_SMALL); 
      glPopMatrix();
      glPushMatrix();
      glTranslated(0., 0., 0.);
      visu_openGL_drawSmoothArrow(obj, -1, VISU_OPENGL_ARROW_BOTTOM_CENTERED,
				       basisLength - 0.3f, 0.1f, 10, FALSE,
				       0.3f, 0.2f, 10, FALSE);
      glRasterPos3f(0.0f, 0.0f, basisLength);
      sprintf(strLg, _("z: %7.3f"), centre[2]);
      openGLText_drawChars(strLg, TEXT_SMALL); 
      glPopMatrix();

      gluDeleteQuadric(obj);
    }
  glLineWidth(1.);
  glEndList();
}
void visu_glExt_box_draw_legend(VisuData *dataObj)
{
  VisuOpenGLView *view;
  float buf, wLg, w;
  gchar strLg[64];
#define BOX_PADDING 10
  float vertices[8][3], box[3];

  if (legendHasBeenBuilt)
    return;
  legendHasBeenBuilt = TRUE;

  view = visu_data_getOpenGLView(dataObj);
  w = 0.16f * MIN(view->window->width, view->window->height);

  buf = 70.f + BOX_PADDING;
  wLg = MAX(w, 90);

  openGLText_initFontList();

  glDeleteLists(extBoxLegend->objectListId, 1);
  glNewList(extBoxLegend->objectListId, GL_COMPILE);

  glViewport((float)view->window->width / 2. - wLg / 2.,
	     BOX_PADDING, wLg, 70.f);

  /* Dsactivation de la lumire et du brouillard et activation du culling. */
  glDisable(GL_LIGHTING);
  glDisable(GL_FOG);

  /* Resetting the depth buffer. */
  glDisable(GL_DEPTH_TEST);

  glEnable(GL_BLEND);
  glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);

  glMatrixMode(GL_PROJECTION);
  glPushMatrix();
  glLoadIdentity();
  glOrtho(0.0, wLg, 0., 70.f, -50., +50.);
  glMatrixMode(GL_MODELVIEW);
  glPushMatrix();
  glLoadIdentity();

  /* We draw a big transparent square to do the back. */
  glColor4f(1., 1., 1., 0.4);
  glRecti(0, 0, wLg, 75.f);

  glColor4f(0.f, 0.f, 0.f, 1.f);

  visu_data_getBoxVertices(dataObj, vertices, FALSE);
  box[0] = sqrt((vertices[1][0] - vertices[0][0]) *
		(vertices[1][0] - vertices[0][0]) +
		(vertices[1][1] - vertices[0][1]) *
		(vertices[1][1] - vertices[0][1]) +
		(vertices[1][2] - vertices[0][2]) *
		(vertices[1][2] - vertices[0][2]));
  box[1] = sqrt((vertices[3][0] - vertices[0][0]) *
		(vertices[3][0] - vertices[0][0]) +
		(vertices[3][1] - vertices[0][1]) *
		(vertices[3][1] - vertices[0][1]) +
		(vertices[3][2] - vertices[0][2]) *
		(vertices[3][2] - vertices[0][2]));
  box[2] = sqrt((vertices[4][0] - vertices[0][0]) *
		(vertices[4][0] - vertices[0][0]) +
		(vertices[4][1] - vertices[0][1]) *
		(vertices[4][1] - vertices[0][1]) +
		(vertices[4][2] - vertices[0][2]) *
		(vertices[4][2] - vertices[0][2]));

  glRasterPos2f(BOX_PADDING / 2.f, 52.5f);
  openGLText_drawChars(_("Box lengths"), TEXT_SMALL); 

  glRasterPos2f(BOX_PADDING / 2.f, 35.0f);
  sprintf(strLg, _("  x: %7.3f"), box[0]);
  openGLText_drawChars(strLg, TEXT_SMALL); 

  glRasterPos2f(BOX_PADDING / 2.f, 20.0f);
  sprintf(strLg, _("  y: %7.3f"), box[1]);
  openGLText_drawChars(strLg, TEXT_SMALL); 

  glRasterPos2f(BOX_PADDING / 2.f, 5.0f);
  sprintf(strLg, _("  z: %7.3f"), box[2]);
  openGLText_drawChars(strLg, TEXT_SMALL); 

  glPopMatrix();
  glMatrixMode(GL_PROJECTION);
  glPopMatrix();
  glMatrixMode(GL_MODELVIEW);

  glEndList();
}

/* Parameters & resources*/

/* This is a boolean to control is the box is render or not. */
static gboolean readBoxIsOn(gchar **lines, int nbLines,
			    int position, VisuData *dataObj _U_, GError **error)
{
  gboolean val;
  
  g_return_val_if_fail(nbLines == 1, FALSE);

  if (!tool_configFile_readBoolean(lines[0], position, &val, 1, error))
    return FALSE;
  visu_glExt_box_setOn(val);
  return TRUE;
}
/* A resource to control the color used to render the lines of the box. */
static gboolean readBoxColor(gchar **lines, int nbLines,
			     int position, VisuData *dataObj _U_, GError **error)
{
  float rgb[3];
  
  g_return_val_if_fail(nbLines == 1, FALSE);

  if (!tool_configFile_readFloat(lines[0], position, rgb, 3, error))
    return FALSE;

  if (tool_configFile_clampFloat(&rgb[0], rgb[0], 0., 1.) ||
      tool_configFile_clampFloat(&rgb[1], rgb[1], 0., 1.) ||
      tool_configFile_clampFloat(&rgb[2], rgb[2], 0., 1.))
    {
      *error = g_error_new(TOOL_CONFIGFILE_ERROR, TOOL_CONFIGFILE_ERROR_VALUE,
			   _("Parse error at line %d: 3 floating points"
			     "(0 <= v <= 1) must appear after the %s markup.\n"),
			   position, FLAG_RESOURCE_BOX_COLOR);
      visu_glExt_box_setRGBValues(boxRGBDefault, TOOL_COLOR_MASK_RGBA);
      return FALSE;
    }
  visu_glExt_box_setRGBValues(rgb, TOOL_COLOR_MASK_RGBA);

  return TRUE;
}
/* A resource to control the width to render the lines of the box. */
static gboolean readBoxLineWidth(gchar **lines, int nbLines,
				 int position, VisuData *dataObj _U_, GError **error)
{
  float width;
  
  g_return_val_if_fail(nbLines == 1, FALSE);

  if (!tool_configFile_readFloat(lines[0], position, &width, 1, error))
    return FALSE;
  if (tool_configFile_clampFloat(&width, width, 1., 10.))
    {
      *error = g_error_new(TOOL_CONFIGFILE_ERROR, TOOL_CONFIGFILE_ERROR_VALUE,
			   _("Parse error at line %d: 1 floating point"
			     "(1 <= v <= 10) must appear after the %s markup.\n"),
			   position, FLAG_RESOURCE_BOX_LINE);
      visu_glExt_box_setLineWidth(RESOURCE_BOX_LINE_DEFAULT);
      return FALSE;
    }
  visu_glExt_box_setLineWidth(width);

  return TRUE;
}
static gboolean readBoxLineStipple(gchar **lines, int nbLines, int position,
				   VisuData *dataObj _U_, GError **error)
{
  int stipple[2];

  g_return_val_if_fail(nbLines == 1, FALSE);
  
  if (!tool_configFile_readInteger(lines[0], position, stipple, 2, error))
    return FALSE;
  visu_glExt_box_setLineStipple((guint16)stipple[0]);
  visu_glExt_box_setExpandStipple((guint16)stipple[1]);

  return TRUE;
}
static gboolean readBoxLengths(gchar **lines, int nbLines,
			       int position, VisuData *dataObj _U_, GError **error)
{
  gboolean val;

  g_return_val_if_fail(nbLines == 1, FALSE);

  if (!tool_configFile_readBoolean(lines[0], position, &val, 1, error))
    return FALSE;
  extBoxLegend->used = val;
  return TRUE;
}

/* Export function that is called by visu_module to write the
   values of resources to a file. */
static void exportResourcesBox(GString *data, VisuData *dataObj _U_)
{
  g_string_append_printf(data, "# %s\n", DESC_RESOURCE_BOX_USED);
  g_string_append_printf(data, "%s:\n", FLAG_RESOURCE_BOX_USED);
  g_string_append_printf(data, "    %d\n", extensionBox->used);
  g_string_append_printf(data, "# %s\n", DESC_RESOURCE_BOX_COLOR);
  g_string_append_printf(data, "%s:\n", FLAG_RESOURCE_BOX_COLOR);
  g_string_append_printf(data, "    %4.3f %4.3f %4.3f\n",
	  boxRGB[0], boxRGB[1], boxRGB[2]);
  g_string_append_printf(data, "# %s\n", DESC_RESOURCE_BOX_LINE);
  g_string_append_printf(data, "%s:\n", FLAG_RESOURCE_BOX_LINE);
  g_string_append_printf(data, "    %4.0f\n", boxLineWidth);
  g_string_append_printf(data, "# %s\n", DESC_RESOURCE_BOX_STIPPLE);
  g_string_append_printf(data, "%s:\n", FLAG_RESOURCE_BOX_STIPPLE);
  g_string_append_printf(data, "    %d %d\n", boxLineStipple[0], boxLineStipple[1]);
  g_string_append_printf(data, "# %s\n", DESC_RESOURCE_BOX_LENGTHS);
  g_string_append_printf(data, "%s:\n", FLAG_RESOURCE_BOX_LENGTHS);
  g_string_append_printf(data, "    %d\n", extBoxLegend->used);
  g_string_append_printf(data, "\n");
}
