/*   EXTRAITS DE LA LICENCE
	Copyright CEA, contributeurs : Luc BILLARD et Damien
	CALISTE, laboratoire L_Sim, (2001-2005)
  
	Adresse ml :
	BILLARD, non joignable par ml ;
	CALISTE, damien P caliste AT cea P fr.

	Ce logiciel est un programme informatique servant  visualiser des
	structures atomiques dans un rendu pseudo-3D. 

	Ce logiciel est rgi par la licence CeCILL soumise au droit franais et
	respectant les principes de diffusion des logiciels libres. Vous pouvez
	utiliser, modifier et/ou redistribuer ce programme sous les conditions
	de la licence CeCILL telle que diffuse par le CEA, le CNRS et l'INRIA 
	sur le site "http://www.cecill.info".

	Le fait que vous puissiez accder  cet en-tte signifie que vous avez 
	pris connaissance de la licence CeCILL, et que vous en avez accept les
	termes (cf. le fichier Documentation/licence.fr.txt fourni avec ce logiciel).
*/

/*   LICENCE SUM UP
	Copyright CEA, contributors : Luc BILLARD et Damien
	CALISTE, laboratoire L_Sim, (2001-2005)

	E-mail address:
	BILLARD, not reachable any more ;
	CALISTE, damien P caliste AT cea P fr.

	This software is a computer program whose purpose is to visualize atomic
	configurations in 3D.

	This software is governed by the CeCILL  license under French law and
	abiding by the rules of distribution of free software.  You can  use, 
	modify and/ or redistribute the software under the terms of the CeCILL
	license as circulated by CEA, CNRS and INRIA at the following URL
	"http://www.cecill.info". 

	The fact that you are presently reading this means that you have had
	knowledge of the CeCILL license and that you accept its terms. You can
	find a copy of this licence shipped with this software at Documentation/licence.en.txt.
*/
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "visu_rendering.h"
#include "visu_data.h"
#include "renderingMethods/externalRenderingMethods.h"

#include "visu_object.h"
#include "visu_tools.h"
#include "visu_configFile.h"
#include "coreTools/toolConfigFile.h"

#include <glib.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

/**
 * SECTION:visu_rendering
 * @short_description: Methods to create and add new rendering
 * methods.
 *
 * <para> The way visu renders its data is done by modules. They are
 * called rendering methods and they describes how data are drawn on
 * the screen. Many can be defined but only one is used at a time to
 * render the data.</para>
 *
 * <para>One or more file type are associated with a rendering
 * method. And a rendering method must specify the way to load the
 * data it needs. Taking the example of a spin system representation,
 * there are two kinds of file. The first kind describes the position
 * the spin and the second contains their orientations.</para>
 *
 * <para>To create a new rendering method, use
 * renderingMethod_new(). The name is mandatory and must be
 * unique. The description is not compulsory. The number of file kinds
 * is also required. Use renderingMethodSet_fileType() to associated a
 * #GList of #FileFormat. In our example of spin system, the first
 * kind of file is about positions, and the associated file formats
 * are *.ascii, *.d3 and *.xyz.</para>
 *
 * <para>The #RenderingMethod_struct has to two pointers on methods
 * that are important. The first, createOpenGLElementFunc() is called
 * when V_Sim needs to create an OpenGL list corresponding to the
 * #VisuElement given as argument. This list then can be used to
 * render each node and thus accelerating the rendering
 * operations. The second method is createOpenGLNodeFunc() and is
 * called by V_Sim for each node of the system when the main OpenGL
 * list (the one for the nodes) is created. Thus, in the contrary of
 * the first method, thios one should not create an OpenGL list but
 * directly call OpenGL routines to draw the node. This method is also
 * responsible to put the node at the right position. Use
 * visuDataGet_nodePosition() to retrieve the position and translate
 * the node accordingly.</para>
 */

#define FLAG_FAV_RENDER    "rendering_favoriteMethod"
#define DESC_FAV_RENDER    "Favorite method used to render files ; chain"
#define DEFAULT_RENDER_FAV ""
static gboolean readFavRenderingMethod(gchar **lines, int nbLines, int position,
					    VisuData *dataObj, GError **error);
static gboolean exportFavRenderingMethod(GString *data, int *nbLinesWritten,
					 VisuData *dataObj);


/* A GHashTable to store all the available rendering methods
   in the system. The keys are the name of each method. */
GHashTable *availableRenderingMethod;

/* The list of the available rendering method. */
GList *listOfAvailableRenderingMethod;

/* A pointer to the in use rendering method. */
RenderingMethod *renderingMethodInUse;

/* Local callbacks. */
static void onResourcesLoaded(VisuObject *visu, VisuData *dataObj, gpointer data);


/***************/
/* Public area */
/***************/

/* A method used by user to registered a new method. */
void registerRenderingMethod(RenderingMethod *method)
{
  DBG_fprintf(stderr, "Registering a new rendering method ... ");
  g_return_if_fail(method && method->name && method->name[0]);

  g_hash_table_insert(availableRenderingMethod, (gpointer)method->name, (gpointer)method);
  listOfAvailableRenderingMethod = g_list_append(listOfAvailableRenderingMethod,
						 method);
  
  DBG_fprintf(stderr, "'%s'\n", method->name);
}

GList* renderingMethodGet_AllMethods()
{
  return listOfAvailableRenderingMethod;
}

/* Free all the allocated attributes of the specified method. */
void renderingMethod_free(RenderingMethod* method)
{
  GList *tmpLst;
  int i;

  if (!method)
    return;
  if (method->name)
    g_free(method->name);
  if (method->printName)
    g_free(method->printName);
  if (method->description)
    g_free(method->description);
  if (method->fileType)
    {
      for (i = 0; i < method->nbFilesType; i++)
	{
	  tmpLst = method->fileType[i];
	  while(tmpLst)
	    {
	      fileFormatFree((FileFormat*)tmpLst->data);
	      tmpLst = g_list_next(tmpLst);
	    }
	}
      g_free(method->fileType);
    }
  g_free(method);
}

/* Allocate the structure and give values to all the attributes. */
RenderingMethod* renderingMethod_new(char* name, char* printName, char* description,
				     int nbFileType, renderingMethodLoadFunc loadMethod)
{
  RenderingMethod *method;
  int i;

  g_return_val_if_fail(name && description && nbFileType >= 0, (RenderingMethod*)0);

  method = g_malloc(sizeof(RenderingMethod));
  method->name = g_strdup(name);
  method->printName = g_strdup(printName);
  method->description = g_strdup(description);
  method->icon = (char*)0;
  method->nbFilesType = nbFileType;
  method->fileTypeLabel = (gchar**)0;
  method->fileType = (GList**)0;
  method->loadMethod = (renderingMethodLoadFunc)0;
  method->createElement = (createOpenGLElementFunc)0;
  method->createNode = (createOpenGLNodeFunc)0;
  method->getExtensForElement = (getExtensOfNodeFunc)0;

  method->loadMethod = loadMethod;
  method->fileType = g_malloc(sizeof(GList*) * nbFileType);
  for (i = 0; i < nbFileType; i++)
    method->fileType[i] = (GList*)0;
  method->fileTypeLabel = g_malloc(sizeof(GList*) * nbFileType);

  return method;
}
int renderingMethodGet_nbFileType(RenderingMethod *method)
{
  g_return_val_if_fail(method, -1);

  return method->nbFilesType;
}

void renderingMethodSet_fileType(RenderingMethod *method, GList *fileTypeList,
				 int fileType, gchar* name)
{
  g_return_if_fail(method && name);
  g_return_if_fail(fileType >=0 && fileType < method->nbFilesType);

  method->fileType[fileType] = fileTypeList;
  method->fileTypeLabel[fileType] = g_strdup(name);
}
void renderingMethodAdd_fileFormat(RenderingMethod *method, FileFormat *fmt,
				   int fileType)
{
  g_return_if_fail(method && fmt);
  g_return_if_fail(fileType >=0 && fileType < method->nbFilesType);

  method->fileType[fileType] = g_list_prepend(method->fileType[fileType], fmt);
}
GList* renderingMethodGet_fileType(RenderingMethod *method, int fileType)
{
  g_return_val_if_fail(method, (GList*)0);
  g_return_val_if_fail(fileType >=0 && fileType < method->nbFilesType, (GList*)0);

  return method->fileType[fileType];
}
gchar* renderingMethodGet_fileTypeName(RenderingMethod *method, int fileType)
{
  g_return_val_if_fail(method, (gchar*)0);
  g_return_val_if_fail(fileType >=0 && fileType < method->nbFilesType, (gchar*)0);

  return method->fileTypeLabel[fileType];
}
float renderingMethodGet_sizeOfElement(RenderingMethod *method, VisuElement *ele)
{
  g_return_val_if_fail(method && method->getExtensForElement && ele, 0.);

  return method->getExtensForElement(ele);
}


void renderingMethodSet_icon(RenderingMethod *method, char *path)
{
  if (!method)
    return;

  if (method->icon)
    {
      g_free(method->icon);
      method->icon = (char*)0;
    }
  if (!path)
    return;

  method->icon = g_strdup(path);
}

/* Choose the method used to render the data. */
void setRenderingMethodInUse(RenderingMethod* method)
{
  if (method == renderingMethodInUse)
    return;

  if (DEBUG)
    {
      if (method)
	fprintf(stderr, "Visu Rendering : set the rendering method to '%s' (%d).\n",
		method->name, GPOINTER_TO_INT(method));
      else
	fprintf(stderr, "Visu Rendering : set the rendering method to 'None' (%d).\n",
		      GPOINTER_TO_INT(method));
    }
  
  renderingMethodInUse = method;
  g_signal_emit(visu, VISU_GET_CLASS(visu)->renderingChanged_signal_id,
		0, (gpointer)method, NULL);
}
int setRenderingMethodByName(char* name)
{
  RenderingMethod* method;

  method = g_hash_table_lookup(availableRenderingMethod, (gpointer)name);
  setRenderingMethodInUse(method);
  if (method)
    return 1;
  else
    return 0;
}
/* Get the current method used to render the data. */
RenderingMethod* getRenderingMethodInUse()
{
  return renderingMethodInUse;
}
/* This method allows to initiate the method
   to deal with the OpenGL methods. */
void setOpenGLMethods(RenderingMethod* method,
		      createOpenGLElementFunc createElement,
		      createOpenGLNodeFunc createNode,
		      getExtensOfNodeFunc getSize)
{
  if (!method)
    return;

  method->createElement = createElement;
  method->createNode = createNode;
  method->getExtensForElement = getSize;
}

gint renderingMethodCompare_priority(gconstpointer a, gconstpointer b)
{
  if (((RenderingFormatLoad*)a)->priority < ((RenderingFormatLoad*)b)->priority)
    return (gint)-1;
  else if (((RenderingFormatLoad*)a)->priority > ((RenderingFormatLoad*)b)->priority)
    return (gint)+1;
  else
    return (gint)0;
}



/****************/
/* Private area */
/****************/

/* Initialise all the variable of this part. */
int initRenderingMethods()
{
  int i, res;
  RenderingMethod *meth;
  VisuConfigFileEntry *resourceEntry;

  resourceEntry = visuConfigFileAdd_entry(VISU_CONFIGFILE_PARAMETER,
					  FLAG_FAV_RENDER, DESC_FAV_RENDER,
					  1, readFavRenderingMethod);
  visuConfigFileAdd_exportFunction(VISU_CONFIGFILE_PARAMETER,
				   exportFavRenderingMethod);

  renderingMethodInUse = (RenderingMethod*)0;
  availableRenderingMethod = g_hash_table_new(g_str_hash, g_str_equal);
  listOfAvailableRenderingMethod = (GList*)0;

  g_return_val_if_fail(availableRenderingMethod, 0);

  VISU_ERROR_RENDERING =  g_quark_from_string("visu_rendering");

  res = 1;
  for (i = 0; listInitRendenringFunc[i]; i++)
    {
      meth = listInitRendenringFunc[i]();
      if (!meth)
	res = 0;
      registerRenderingMethod(meth);
    }
  
  if (!res)
    g_warning("Some rendering methods can't initialse.\n");

  DBG_fprintf(stderr, " %d valid rendering method(s) found.\n",
	      g_hash_table_size(availableRenderingMethod));

  g_signal_connect(G_OBJECT(visu), "resourcesLoaded",
 		   G_CALLBACK(onResourcesLoaded), (gpointer)0);

  return res;
}

static void onResourcesLoaded(VisuObject *visu _U_, VisuData *dataObj,
			      gpointer data _U_)
{
  VisuDataIter iter;

  DBG_fprintf(stderr, "Visu Rendering: caught the 'resourcesLoaded' signal.\n");

  if (!dataObj || !renderingMethodInUse)
    return;

  visuDataIter_new(dataObj, &iter);
  for(visuDataIter_start(dataObj, &iter); iter.element;
      visuDataIter_nextElement(dataObj, &iter))
    {
      visuElement_createMaterial(iter.element);
      renderingMethodInUse->createElement(dataObj, iter.element);
      if (visuElementGet_updateNodesOnMaterialChange())
	visuData_createNodes(dataObj, iter.element);
    }
}

/**************/
/* Parameters */
/**************/

static gboolean readFavRenderingMethod(gchar **lines, int nbLines, int position,
				       VisuData *dataObj _U_, GError **error)
{
  int err;

  g_return_val_if_fail(nbLines == 1, FALSE);

  lines[0] = g_strstrip(lines[0]);

  if (!lines[0][0])
    setRenderingMethodInUse((RenderingMethod*)0);
  else
    {
      err = setRenderingMethodByName(lines[0]);
      if (!err)
	{
	  *error = g_error_new(CONFIG_FILE_ERROR, CONFIG_FILE_ERROR_VALUE,
			       _("Parse error at line %d:"
				 " the specified method (%s) is unknown.\n"),
			       position, lines[0]);
	  return FALSE;
	}
    }

  return TRUE;
}
static gboolean exportFavRenderingMethod(GString *data, int *nbLinesWritten,
					 VisuData *dataObj _U_)
{
  g_string_append_printf(data, "# %s\n", DESC_FAV_RENDER);
  if (renderingMethodInUse)
    g_string_append_printf(data, "%s: %s\n\n", FLAG_FAV_RENDER, renderingMethodInUse->name);
  else
    g_string_append_printf(data, "%s:\n\n", FLAG_FAV_RENDER);

  *nbLinesWritten = 3;
  return TRUE;
}
