/*   EXTRAITS DE LA LICENCE
	Copyright CEA, contributeurs : Luc BILLARD et Damien
	CALISTE, laboratoire L_Sim, (2001-2005)
  
	Adresse ml :
	BILLARD, non joignable par ml ;
	CALISTE, damien P caliste AT cea P fr.

	Ce logiciel est un programme informatique servant  visualiser des
	structures atomiques dans un rendu pseudo-3D. 

	Ce logiciel est rgi par la licence CeCILL soumise au droit franais et
	respectant les principes de diffusion des logiciels libres. Vous pouvez
	utiliser, modifier et/ou redistribuer ce programme sous les conditions
	de la licence CeCILL telle que diffuse par le CEA, le CNRS et l'INRIA 
	sur le site "http://www.cecill.info".

	Le fait que vous puissiez accder  cet en-tte signifie que vous avez 
	pris connaissance de la licence CeCILL, et que vous en avez accept les
	termes (cf. le fichier Documentation/licence.fr.txt fourni avec ce logiciel).
*/

/*   LICENCE SUM UP
	Copyright CEA, contributors : Luc BILLARD et Damien
	CALISTE, laboratoire L_Sim, (2001-2005)

	E-mail address:
	BILLARD, not reachable any more ;
	CALISTE, damien P caliste AT cea P fr.

	This software is a computer program whose purpose is to visualize atomic
	configurations in 3D.

	This software is governed by the CeCILL  license under French law and
	abiding by the rules of distribution of free software.  You can  use, 
	modify and/ or redistribute the software under the terms of the CeCILL
	license as circulated by CEA, CNRS and INRIA at the following URL
	"http://www.cecill.info". 

	The fact that you are presently reading this means that you have had
	knowledge of the CeCILL license and that you accept its terms. You can
	find a copy of this licence shipped with this software at Documentation/licence.en.txt.
*/

#include "toolFileFormat.h"

#include <visu_tools.h>
#include <string.h>
#include <stdlib.h>

/**
 * SECTION:toolFileFormat
 * @short_description: Describes file format objects (name patterns,
 * description, properties...).
 *
 * <para>When dealing with files, it is convenient to class then by
 * formats, one format for JPEG files, one other for
 * postscript... Such file formats are characterized by their
 * description, explaining what they are, one or more name pattern
 * (e.g. "*.jpg") and some properties (e.g. compression level for JPEG
 * file).</para>
 * <para>This module describes objects that can store all these
 * informations and deals with them. To create a new file format, use
 * fileFormatNew(). A #FileFormat object can be used in a
 * GtkFileChooser object, using a GtkFileFilter. fileFormatGet_label()
 * will return directly the label to give to the GtkFileFilter and the
 * name patterns can be passed to it also.</para>
 * <para>A file format property can be an integer, a boolean or a
 * string. For instance, use fileFormatAdd_propertyBoolean() to add a
 * boolean property to a given #FileFormat object. Then the stored
 * value can be changed using fileFormatSet_propertyBoolean() and
 * retrieve with fileFormatGet_propertyBoolean().</para>
 */

/* Private structure to store some properties associated to FileFormat objects. */
struct FileFormatProperty_struct
{
  /* Label that can be used by an interface or print to the console,
     to describe this property (must be in UTF8. */
  gchar* labelUTF8;

  /* String used by saving method to identify the property. */
  gchar* name;
  
  /* Use the FileFormatType_enum to index this variable. */
  int type;

  gpointer value;
};
void fileFormatFree_property(FileFormatProperty* propValue);

FileFormat* fileFormatNew(gchar* descr, gchar** patterns)
{
  int nbCharacterMax = 45;
  FileFormat *res;
  int lg, n;
  GString *tmpStr;
  gchar* tmpPattern;

  g_return_val_if_fail(descr && descr[0] && patterns, (FileFormat*)0);

  lg = strlen(descr);
  if (lg > nbCharacterMax)
    {
      g_warning("The descr parameter given to 'fileFormatNew' is bigger than"
		" %d characters and it will be truncated.\n", nbCharacterMax);
      lg = nbCharacterMax;
    }

  res = g_malloc(sizeof(FileFormat));
  res->description = (gchar*)0;
  res->fileType = (GList*)0;
  res->labelString = (gchar*)0;
  res->properties = (GList*)0;
  
  res->description = g_strndup(descr, lg);

  tmpStr = g_string_new(res->description);
  g_string_append_printf(tmpStr, " (");
  n = 0;
  while (patterns[n])
    {
      g_string_append_printf(tmpStr, "%s", patterns[n]);
      tmpPattern = g_strdup(patterns[n]);
      res->fileType = g_list_append(res->fileType, tmpPattern);
      n++;
      if (patterns[n])
	g_string_append_printf(tmpStr, ", ");
    }
  if (n == 0)
    g_string_append_printf(tmpStr, "no filter");
  g_string_append_printf(tmpStr, ")");
  res->labelString = tmpStr->str;
  g_string_free(tmpStr, FALSE);

  return res;
}
void fileFormatFree(FileFormat *format)
{
  GList *tmpLst;

  if (!format)
    return;

  if (format->description)
    g_free(format->description);
  if (format->labelString)
    g_free(format->labelString);
  tmpLst = format->fileType;
  while(tmpLst)
    {
      g_free(tmpLst->data);
      tmpLst = g_list_next(tmpLst);
    }
  g_list_free(format->fileType);
  tmpLst = format->properties;
  while(tmpLst)
    {
      fileFormatFree_property((FileFormatProperty*)tmpLst->data);
      tmpLst = g_list_next(tmpLst);
    }
  g_list_free(format->properties);

  g_free(format);
}
gchar* fileFormatGet_label(FileFormat *format)
{
  g_return_val_if_fail(format, (gchar*)0);
  return format->labelString;
}
gboolean fileFormatUse_match(FileFormat *format, gchar *filename)
{
  GList *tmpLst;
  gboolean match;

  g_return_val_if_fail(format, FALSE);
  match = FALSE;
  tmpLst = format->fileType;
  while (tmpLst)
    {
      match = match || g_pattern_match_simple(tmpLst->data, filename);
      tmpLst = g_list_next(tmpLst);
    }
  return match;
}

void fileFormatFree_property(FileFormatProperty* propValue)
{
  g_return_if_fail(propValue);
  g_free(propValue->value);
  g_free(propValue->labelUTF8);
  g_free(propValue->name);
  g_free(propValue);
}
GList* fileFormatGet_propertiesList(FileFormat *format)
{
  g_return_val_if_fail(format, (GList*)0);
  return format->properties;
}
gchar* fileFormatGet_propertyLabelUTF8(FileFormatProperty *property)
{
  g_return_val_if_fail(property, (gchar*)0);
  return property->labelUTF8;
}
FileFormatPropertyTypeId fileFormatGet_propertyType(FileFormatProperty *property)
{
  g_return_val_if_fail(property, -1);
  return property->type;
}
gchar* fileFormatGet_propertyName(FileFormatProperty *property)
{
  g_return_val_if_fail(property, (gchar*)0);
  return property->name;
}


void fileFormatAdd_propertyInteger(FileFormat *format, const gchar* name,
				   const gchar* labelUTF8, int value)
{
  FileFormatProperty *propValue;
  int *valInt;

  g_return_if_fail(format && name && name[0] && labelUTF8 && labelUTF8[0]);
  propValue = g_malloc(sizeof(FileFormatProperty));
  propValue->labelUTF8 = g_strdup(labelUTF8);
  propValue->name = g_strdup(name);
  propValue->type = fileFormat_type_integer;
  valInt = g_malloc(sizeof(int));
  *valInt = value;
  propValue->value = (gpointer*)valInt;
  format->properties = g_list_append(format->properties, (gpointer)propValue);
  DBG_fprintf(stderr, "Tool FileFormat : adding %p property to object"
	      " %p (integer %d).\n", (gpointer)propValue, (gpointer)format, value);
}
gboolean fileFormatSet_propertyInteger(FileFormatProperty *property, int value)
{
  g_return_val_if_fail(property && property->type == fileFormat_type_integer, FALSE);
  DBG_fprintf(stderr, "Tool FileFormat : setting %p property value"
	      " to %d.\n", (gpointer)property, value);
  
  if (*(int*)(property->value) == value)
    return FALSE;
  *(int*)(property->value) = value;
  return TRUE;
}
int fileFormatGet_propertyInteger(FileFormatProperty *property)
{
  g_return_val_if_fail(property && property->type == fileFormat_type_integer, -1);
  DBG_fprintf(stderr, "Tool FileFormat : getting %p property value"
	      " : %d.\n", (gpointer)property, *(int*)(property->value));

  return *(int*)(property->value);
}
void fileFormatAdd_propertyBoolean(FileFormat *format, const gchar* name,
				   const gchar* labelUTF8, gboolean value)
{
  FileFormatProperty *propValue;
  int *valBool;

  g_return_if_fail(format && name && name[0] && labelUTF8 && labelUTF8[0]);
  propValue = g_malloc(sizeof(FileFormatProperty));
  propValue->labelUTF8 = g_strdup(labelUTF8);
  propValue->name = g_strdup(name);
  propValue->type = fileFormat_type_boolean;
  valBool = g_malloc(sizeof(int));
  *valBool = value;
  propValue->value = (gpointer*)valBool;
  format->properties = g_list_append(format->properties, (gpointer)propValue);
  DBG_fprintf(stderr, "Tool FileFormat : adding %p property to object"
	      " %p (boolean %d).\n", (gpointer)propValue, (gpointer)format, (int)value);
}
gboolean fileFormatSet_propertyBoolean(FileFormatProperty *property, gboolean value)
{
  g_return_val_if_fail(property && property->type == fileFormat_type_boolean, FALSE);
  DBG_fprintf(stderr, "Tool FileFormat : setting %p boolean property value"
	      " to %d.\n", (gpointer)property, (int)value);
  
  if (*(gboolean*)(property->value) == value)
    return FALSE;
  *(gboolean*)(property->value) = value;
  return TRUE;
}
gboolean fileFormatGet_propertyBoolean(FileFormatProperty *property)
{
  g_return_val_if_fail(property && property->type == fileFormat_type_boolean, FALSE);
  DBG_fprintf(stderr, "Tool FileFormat : getting %p boolean property value"
	      " : %d.\n", (gpointer)property, (int)(*(gboolean*)(property->value)));

  return *(gboolean*)(property->value);
}
