/*   EXTRAITS DE LA LICENCE
	Copyright CEA, contributeurs : Luc BILLARD et Damien
	CALISTE, laboratoire L_Sim, (2001-2005)
  
	Adresse ml :
	BILLARD, non joignable par ml ;
	CALISTE, damien P caliste AT cea P fr.

	Ce logiciel est un programme informatique servant  visualiser des
	structures atomiques dans un rendu pseudo-3D. 

	Ce logiciel est rgi par la licence CeCILL soumise au droit franais et
	respectant les principes de diffusion des logiciels libres. Vous pouvez
	utiliser, modifier et/ou redistribuer ce programme sous les conditions
	de la licence CeCILL telle que diffuse par le CEA, le CNRS et l'INRIA 
	sur le site "http://www.cecill.info".

	Le fait que vous puissiez accder  cet en-tte signifie que vous avez 
	pris connaissance de la licence CeCILL, et que vous en avez accept les
	termes (cf. le fichier Documentation/licence.fr.txt fourni avec ce logiciel).
*/

/*   LICENCE SUM UP
	Copyright CEA, contributors : Luc BILLARD et Damien
	CALISTE, laboratoire L_Sim, (2001-2005)

	E-mail address:
	BILLARD, not reachable any more ;
	CALISTE, damien P caliste AT cea P fr.

	This software is a computer program whose purpose is to visualize atomic
	configurations in 3D.

	This software is governed by the CeCILL  license under French law and
	abiding by the rules of distribution of free software.  You can  use, 
	modify and/ or redistribute the software under the terms of the CeCILL
	license as circulated by CEA, CNRS and INRIA at the following URL
	"http://www.cecill.info". 

	The fact that you are presently reading this means that you have had
	knowledge of the CeCILL license and that you accept its terms. You can
	find a copy of this licence shipped with this software at Documentation/licence.en.txt.
*/

#include "toolColor.h"

#include <glib.h>

#include <visu_tools.h>
#include <visu_object.h>

/**
 * SECTION:toolColor
 * @short_description: Simple handling and storage of RGBA colours.
 *
 * <para>This file defines a basic structure to store colours (not
 * using the GDK one beca use V_Sim core should not rely on GDK and
 * GTK): #Color. Several transformations are possible on a colour,
 * going from and to RGB encoding. Use color_HSVtoRGB() and
 * color_HSLtoRGB() to do that.</para>
 *
 * <para>This file gives also the capability to store known colours in
 * a list. Use methods such as colorAdd_color() or
 * colorAdd_floatRGBA(). Them, one can access to stored colours, using
 * colorGet_byValues() or colorGet_byColor().</para>
 */

/******************/
/* Storing colors */
/******************/
#define COLOR_NB_MAX 10
GList *color_storageArray;

/******************/
/* Storing colors */
/******************/
void initColorStorage()
{
  color_storageArray = (GList*)0;
}
Color* colorNew_floatRGBA(float rgba[4])
{
  Color *color;
  int i;

  color = g_malloc(sizeof(Color));
  for (i = 0; i< 4; i++)
    {
      if (rgba[i] < 0.)
	color->rgba[i] = 0.;
      else if (rgba[i] > 1.)
	color->rgba[i] = 1.;
      else
	color->rgba[i] = rgba[i];
    }
  color->userData = (gpointer)0;
  return color;
}
gboolean colorEqual_color(Color *color1, Color *color2)
{
  return (color1->rgba[0] == color2->rgba[0] &&
	  color1->rgba[1] == color2->rgba[1] &&
	  color1->rgba[2] == color2->rgba[2]);
}
Color* colorGet_byId(int num)
{
  GList *tmpLst;

  tmpLst = g_list_nth(color_storageArray, num);
  if (tmpLst)
    return (Color*)tmpLst->data;
  else
    return (Color*)0;
}
int colorGet_byColor(Color *color)
{
  int pos;

  if (color)
    colorGet_byValues(&pos, color->rgba[0], color->rgba[1], color->rgba[2], color->rgba[3]);
  else
    pos = -1;
  return pos;

}
Color* colorGet_byValues(int *pos, float red, float green, float blue, float alpha)
{
  GList *tmpLst;
  int found;

  tmpLst = color_storageArray;
  found = 0;
  if (pos)
    *pos = -1;
  while (tmpLst && !found)
    {
      found = (red == ((Color*)tmpLst->data)->rgba[0]) &&
	(green == ((Color*)tmpLst->data)->rgba[1]) &&
	(blue == ((Color*)tmpLst->data)->rgba[2]) &&
	(alpha == ((Color*)tmpLst->data)->rgba[3]);
      if (pos)
	*pos += 1;
      if (!found)
	tmpLst = g_list_next(tmpLst);
    }
  if (found)
    return (Color*)tmpLst->data;
  else
    {
      if (pos)
	*pos = -1;
      return (Color*)0;
    }
}
GList* colorGet_storedColors()
{
  return color_storageArray;
}
void colorCopy_color(Color *color, Color *color_old)
{
  g_return_if_fail(color && color_old);

  memcpy(color->rgba, color_old->rgba, sizeof(float) * 4);
  color->userData = color_old->userData;
}
Color* colorAdd_color(Color* color)
{
  int pos;

  g_return_val_if_fail(color, (Color*)0);

  if (!colorGet_byValues(&pos, color->rgba[0], color->rgba[1], color->rgba[2], color->rgba[3]))
    {
      color_storageArray = g_list_append(color_storageArray, (gpointer)color);
      DBG_fprintf(stderr, "Visu Tools : adding a new color at %p (%d).\n",
		  (gpointer)color_storageArray, g_list_length(color_storageArray));
      g_signal_emit (visu, VISU_GET_CLASS(visu)->colorNewAvailable_signal_id,
		     0 , (gpointer)color, NULL);
    }
  return color;
}
Color* colorAdd_floatRGBA(float rgba[4], int *position)
{
  Color *color;
  int i;

  for (i = 0; i < 4; i++)
    g_return_val_if_fail(rgba[i] >= 0. && rgba[i] <= 1., (Color*)0);

  color = colorGet_byValues(position, rgba[0], rgba[1], rgba[2], rgba[3]);
  if (color)
    {
      DBG_fprintf(stderr, "Visu Tools : color already exist, no color added.\n");
      return color;
    }

  color = colorNew_floatRGBA(rgba);
  color_storageArray = g_list_append(color_storageArray, (gpointer)color);
  if (position)
    *position = g_list_length(color_storageArray);
  DBG_fprintf(stderr, "Visu Tools : adding a new color at %p (%d).\n",
	      (gpointer)color_storageArray, g_list_length(color_storageArray));
  g_signal_emit (visu, VISU_GET_CLASS (visu)->colorNewAvailable_signal_id,
		 0 , (gpointer)color, NULL);
  return color;
}
Color* colorAdd_intRGBA(int rgba[4])
{
  Color *color;
  float rgbaf[4];
  int pos, i;

  for (i = 0; i < 4; i++)
    g_return_val_if_fail(rgba[i] >= 0 && rgba[i] < 256, (Color*)0);

  for (i = 0; i < 4; i++)
    rgbaf[i] = (float)rgba[i] / 255.;
  color = colorGet_byValues(&pos, rgbaf[0], rgbaf[1], rgbaf[2], rgbaf[3]);
  if (color)
    {
      DBG_fprintf(stderr, "Visu Tools : color already exist, no color added.\n");
      return color;
    }

  color = colorNew_floatRGBA(rgbaf);
  color_storageArray = g_list_append(color_storageArray, (gpointer)color);
  DBG_fprintf(stderr, "Visu Tools : adding a new color at %p (%d).\n",
	      (gpointer)color_storageArray, g_list_length(color_storageArray));
  g_signal_emit (visu, VISU_GET_CLASS (visu)->colorNewAvailable_signal_id,
		 0 , (gpointer)color, NULL);
  return color;
}
Color* colorGet_lastStored()
{
  return (Color*)g_list_last(color_storageArray)->data;
}

void color_HSVtoRGB(float* rgb, float* hsv)
{
  float var_h, var_0, var_1, var_2, var_3;
  int var_i;
  
  g_return_if_fail(rgb && hsv);

  if (hsv[1] == 0)
    {
      rgb[0] = hsv[2];
      rgb[1] = hsv[2];
      rgb[2] = hsv[2];
    }
  else
    {
      var_h = hsv[0] * 6.;
      var_i = (int)var_h;
      var_0 = hsv[2];
      var_1 = hsv[2] * (1. - hsv[1]);
      var_2 = hsv[2] * (1. - hsv[1] * (var_h - (float)var_i));
      var_3 = hsv[2] * (1. - hsv[1] * (1. - (var_h - (float)var_i)));
      switch (var_i % 6)
	{
	case 0:
	  rgb[0] = var_0;
	  rgb[1] = var_3;
	  rgb[2] = var_1;
	  break;
	case 1:
	  rgb[0] = var_2;
	  rgb[1] = var_0;
	  rgb[2] = var_1;
	  break;
	case 2:
	  rgb[0] = var_1;
	  rgb[1] = var_0;
	  rgb[2] = var_3;
	  break;
	case 3:
	  rgb[0] = var_1;
	  rgb[1] = var_2;
	  rgb[2] = var_0;
	  break;
	case 4:
	  rgb[0] = var_3;
	  rgb[1] = var_1;
	  rgb[2] = var_0;
	  break;
	case 5:
	  rgb[0] = var_0;
	  rgb[1] = var_1;
	  rgb[2] = var_2;
	  break;
	}
    }
}

float Hue_2_RGB(float v1, float v2, float vH)
{
  /* from www.easyrgb.com */
  if (vH < 0) vH += 1;
  if (vH > 1) vH -= 1;
  if ((6*vH) < 1) 
    return (v1+(v2-v1)*6*vH);
  if ((2*vH) < 1) 
    return v2;
  if ((3*vH) < 2) 
    return (v1+(v2-v1)*((2./3.)-vH)*6);
  return v1;
}

/* hsl[0] = hue
   hsl[1] = saturation
   hsl[2] = lightness */
void color_HSLtoRGB(float *rgb, float *hsl)
{
  /* from www.easyrgb.com */
  if (hsl[1] == 0)                       /*HSL values = 0  1*/
    {
      rgb[0] = hsl[2];                      /*RGB results = 0  255*/
      rgb[1] = hsl[2];
      rgb[2] = hsl[2];
    }

  else
    {
      float var_1, var_2;
      if (hsl[2] < 0.5) 
	var_2 = hsl[2] * (1 + hsl[1]);		       
      else           
	var_2 = (hsl[2] + hsl[1]) - (hsl[1] * hsl[2]);
					
      var_1 = 2 * hsl[2] - var_2;

      rgb[0] = Hue_2_RGB( var_1, var_2, hsl[0] + ( 1. / 3. ) );
      rgb[1] = Hue_2_RGB( var_1, var_2, hsl[0] );
      rgb[2] = Hue_2_RGB( var_1, var_2, hsl[0] - ( 1. / 3. ) );
    }  
}
