/*
 *       Parser for TTCN-3
 *
 * Copyright (C) 2001 Institute for Telematics
 *
 *    Roman Koch <rkoch@itm.mu-luebeck.de>
 *    Michael Schmitt <schmitt@itm.mu-luebeck.de>
 *
 *    Medical University of Luebeck,
 *    Ratzeburger Allee 160,
 *    23538 Luebeck,
 *    Germany
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 * $Id: TTCNParser.g,v 1.123 2001/08/07 18:12:45 schmitt Exp $
 *
 * This file is intended to be used with ANTLR 2.7.1 or higher. ANTLR is a parser generator that
 * is based on 'linear approximate' lookahead analysis (a kind of weak LL(k) analysis).
 * ANTLR is free software provided by jGuru.com (MageLang Institute). For further information
 * please refer to http://www.antlr.org. My special thanks go to Terence Parr from jGuru.com.
 *
 * If you make use of this TTCN-3 grammar specification, please send a little email in which
 * you tell us what you are using it for and whether you consider it useful. Any grammar
 * corrections and extensions (e.g. for semantics checks and abstract syntax tree transformation)
 * are welcome. For the latest release of all TTCN-3 grammar files please have a look at
 * http://www.itm.mu-luebeck.de/english/research/specification/ttcn3parser. We also invite you to
 * visit the WWW pages of the Institute for Telematics at http://www.itm.mu-luebeck.de.
 *
 *
 * This grammar is based on the final draft of the TTCN-3 standard (version 1.0.11 of May 2001)
 *
 *
 * General considerations:
 *
 *   - the original TTCN-3 grammar rules have been adopted as much as possible since the TTCN-3
 *     standard is likely to undergo further changes in the future which should be easy to
 *     integrate
 *   - the TTCN-3 parser constructs an abstract syntax tree (AST) based on the token stream
 *     delivered by 'TTCNLexer'. However, as no type information about names is available at this
 *     stage of processing, several decisions have to be postponed. E.g., the parser cannot always
 *     determine whether a statement represents a 'functionRef' or a 'testcaseRef'. Such
 *     ambiguities are resolved by a succeeding tree parser that operates on the intermediate AST.
 *     For a full list of unresolvable nondeterminisms please see below
 *
 * Unsolved problems:
 *
 *   - check for comments on syntactical restrictions in the textual part of the TTCN-3 standard
 *   - build symbol tables
 *
 * Inconsistencies and potential enhancements of the TTCN-3 standard:
 *
 *  1. Precedence of bitwise operators in 'singleExpression' is not expressed in the
 *     grammar
 *  2. There is no notion of a global timer that is defined outside any test module. As a
 *     consequence, it is impossible to retrieve the current time ('now') or to state when
 *     a test case is actually executed (e.g. periodically every night at 2AM)
 *  3. Remove the macro-like 'group' concept and substitute it by a generalized 'module' concept.
 *     (Would it be possible/make sense to have nested modules?)
 *  4. Allow to define nested structured data types in one single statement
 *     (e.g. a record of record-of).
 *  5. 'timerValue' should derive to 'expression' in order to initialize timer arrays (fixed)
 */

header "pre_include_hpp"
{
  // $Id: TTCNParser.g,v 1.123 2001/08/07 18:12:45 schmitt Exp $

  #include <string>
  #include "EnvAST.hpp"

  using std::string;
}


header "post_include_cpp"
{
  #include <iostream>
  #include "InfoAST.hpp"
  #include "ScopeAST.hpp"

  using std::cerr;
  using std::endl;

  namespace TTCNGrammar
  {
    void TTCNParser::reportError( const string &errorMessage )
    {
      reportMessage( errorMessage );
      noOfErrors++;
    }

    void TTCNParser::reportError( const antlr::RecognitionException &ex )
    {
      cerr << ex.toString().c_str() << endl;
      noOfErrors++;
    }

    void TTCNParser::reportMessage( const string &message )
    {
      if ( getFilename().length() > 0 )
        cerr << getFilename();
      else
        cerr << "<stdin>";

      cerr << ":" << LT( 1 ) -> getLine() << ":" << LT( 1 ) -> getColumn() << ": error: " << message << endl;
    }

    unsigned int TTCNParser::numberOfErrors() const
    {
      return noOfErrors;
    }

    void TTCNParser::resetErrors()
    {
      noOfErrors = 0;
    }

    const string &TTCNParser::versionInfo()
    {
      static string   info = "$Id: TTCNParser.g,v 1.123 2001/08/07 18:12:45 schmitt Exp $";

      return info;
    }

    void TTCNParser::consume()
    {
      lastToken = LA( 1 ); LLkParser::consume();
    }

    bool TTCNParser::optionalSemicolon()
    {
      return lastToken == EndChar || LA( 1 ) == EndChar;
    }

    bool TTCNParser::checkModuleControlPart( string statementName )
    {
      if( isModuleControlPart )
      {
        reportError( statementName + " are not allowed in module control parts." );
        return false;
      }
      else
        return true;
    }
  }

  #define ADDVOIDCHILD(tokenast)     astFactory.addASTChild( currentAST, tokenast )
  #define ADDCHILD(tokenAST,token) { const antlr::RefAST &tmpAST = tokenAST; \
                                     SETLINECOLUMN( tmpAST, token ); \
                                     astFactory.addASTChild( currentAST, tmpAST ); }
  #define MAKEROOT(tokenAST,token) { const antlr::RefAST &tmpAST = tokenAST; \
                                     SETLINECOLUMN( tmpAST, token ) \
                                     astFactory.makeASTRoot( currentAST, tmpAST ); }
  #define SETLINECOLUMN(AST,token) { antlr::RefInfoAST( AST ) -> setLine( token -> getLine() ); \
                                     antlr::RefInfoAST( AST ) -> setColumn( token -> getColumn() ); }

  // Create new EnvAST and set parent environment of it
  #define NEWENVIRONMENT_BEGIN  { antlr::RefEnvAST  oldEnv = currentEnv; \
                                  currentEnv = antlr::RefEnvAST( new antlr::EnvAST() ); \
                                  currentEnv -> setParentEnv( oldEnv ); }

  // Create new ScopeAST (vertex in AST with reference to associated EnvAST)
  #define SCOPEAST(var,tokenType)  antlr::RefAST  var = antlr::RefAST( new antlr::ScopeAST( currentEnv ) ); \
                                   var -> initialize(tokenType,"");

  // Restore current environment
  #define NEWENVIRONMENT_END    { currentEnv = currentEnv -> getParentEnv(); }

}


options
{
  language  = "Cpp";
  namespace = "TTCNGrammar";
  namespaceStd = "std";
  namespaceAntlr = "antlr";
  genHashLines = true;
}


class TTCNParser extends Parser;

options
{
  k = 3;
  importVocab = TTCNLexer;
  exportVocab = TTCNParser;
  defaultErrorHandler = true;
  buildAST = true;
  codeGenMakeSwitchThreshold = 10;
}

  //
  // Tokens that are not allowed in general due to nondeterminisms in the rules below:
  //
  // In rule 'fieldReference':
  //
  //   recordRef - parRef
  //     - signatureParIdentifier
  //
  // In rule 'singleValueOrAttrib':
  //
  //   singleExpression - templateRefWithParList
  //     simpleExpression -
  //       subExpression -
  //         product -
  //           term -
  //             factor -
  //               primary -
  //                 value -
  //                   referencedValue -
  //                     valueReference -
  //                   predefinedValue -
  //                     enumeratedValue -
  //                 opCall -
  //                   functionInstance -
  //                     functionRef -
  //                       * functionActualParList - templateActualParList
  //                           - ... singleValueOrAttrib !!!
  //
  // In rule 'functionActualPar':
  //
  //   timerRef - componentRef - templateInstance - port
  //     - componentIdentifier - inLineTemplate -
  //       - variableRef -
  //
  // In rule 'opCall':
  //
  //   configurationOps - timerOps
  //     runningOp - runningTimerOp
  //       componentIdentifier - timerRef (-> above)
  //
  // In rule 'defOrFieldRef':
  //
  //   definitionRef - fieldReference
  //
  // In rule 'procOrType'/'baseTemplate'/'inLineTemplate':
  //
  //   type - signature
  //     referencedType -
  //
  // In rule 'functionFormalPar':
  //
  //   formalPortPar - formalValuePar
  //
  // In rule 'behaviourStatements':
  //
  //   functionInstance - teststepInstance
  //     functionRef - teststepRef
  //
  // In rule 'referencedType'/'referencedValue'/'valueReference':
  //
  //   globalModuleId - typeReference - extendedFieldReference
  //
  // In rule 'functionStatement':
  //
  //   timerStatements - communicationStatements // no problem with
  //     startTimerStatement - startStatement    // 'startTCStatement'
  //       timerRef - portOrAll
  //         - port
  //     stopTimerStatement - stopStatement
  //       timerRef - portOrAll
  //         - port
  //
  // In rule 'typeActualPar':
  //
  //   singleConstExpression - type
  //     ...
  //       primary - referencedType
  //         value -
  //           referencedValue -
  //             valueReference -
  //           predefinedValue -
  //             enumeratedValue -
  //
  // In rule 'value':
  //
  //   predefinedValue - referencedValue
  //     enumeratedValue - valueReference
  //

  tokens
  {
    ActivateOp;                  Address;                     All;
    AllComponent;                AllowedValues;               AllPort;
    AllTimer;                    AltConstruct;                AltGuardChar;
    AnyComponent;                AnyOrOmit;                   AnyPort;
    AnyTimer;                    AnyValue;                    ArrayConstExpression;
    ArrayDef;                    ArrayDefList;                ArrayExpression;
    ArrayOrBitRef;               ArrayValueOrAttrib;          Assignment;
    AssignmentList;              AttribQualifier;             BaseTemplate;
    CallBodyGuard;               CallBodyStatement;           CallParameters;
    CallStatement;               CatchOpParameter;            CatchStatement;
    CharStringMatch;             CheckParameter;              CheckStatement;
    ClearStatement;              Complement;                  ComponentDef;
    ComponentType;               ConditionalConstruct;        ConfigSpec;
    ConnectStatement;            ConstDef;                    CreateOp;
    DeactivateStatement;         DefConst;
    DefFunction;                 DefSignature;                DefTemplate;
    DefTestcase;                 DefTeststep;                 DefType;
    DefinitiveIdentifier;        DefinitiveNameAndNumberForm; DerivedDef;
    DisconnectStatement;         Display;                     DoneStatement;
    DoWhileStatement;            DummyToken;                  ElseClause;
    ElseIfClause;                ElseStatement;
    Encode;                      EnumDef;                     Error;
    ExceptionSpec;               ExtConstDef;                 Extension;
    ExtFunctionDef;              ExtModuleActualParList;      Fail;
    False;                       FieldConstExpressionList;    FieldConstExpressionSpec;
    FieldExpressionList;         FieldExpressionSpec;         FieldSpec;
    FieldSpecList;               FormalTemplatePar;           FormalTimerPar;
    FormalTypePar;               ForStatement;                FromClause;
    FunctionBody;                FunctionDef;                 FunctionFormalParList;
    GetCallStatement;            GetLocalVerdict;             GetReplyStatement;
    GotoStatement;               GroupDef;
    GuardStatement;              IfPresentMatch;              ImportAllSpec;
    ImportConstSpec;             ImportFromSpec;              ImportFunctionSpec;
    ImportGroupSpec;             ImportSignatureSpec;         ImportTemplateSpec;
    ImportTestcaseSpec;          ImportTeststepSpec;          ImportTypeDefSpec;
    In;                          Inconc;                      Infinity;
    InOut;                       IntegerRange;
    IntegerRangeDef;             InterleavedConstruct;        InterleavedGuardElement;
    InterleavedGuard;            Label;                       LanguageSpec;
    LogStatement;                MapStatement;                MatchOp;
    MessageAttribs;              MessageList;                 MinusInfinity;
    MixedAttribs;                MixedList;                   ModuleControlBody;
    ModuleControlPart;           ModuleDefinition;            ModuleDefinitionsPart;
    ModuleId;                    ModulePar;                   ModuleParList;
    MTCOp;                       NameAndNumberForm;           NamedValue;
    Noblock;                     None;
    Nonrecursive;                NotUsed;                     Nowait;
    Null;                        ObjectIdentifierValue;       Omit;

    OpBitwiseAnd;                OpBitwiseXor;                OpBitwiseOr;
    OpLogicalAnd;                OpLogicalXor;                OpLogicalOr;
    OpConcatenate;               OpEqual;                     OpLess;
    OpGreater;                   OpNotEqual;                  OpGreaterOrEqual;
    OpLessOrEqual;               OpShiftLeft;                 OpShiftRight;
    OpRotateLeft;                OpRotateRight;               OpBinaryPlus;
    OpBinaryMinus;               OpTimes;                     OpDivide;
    OpMod;                       OpRem;                       OpUnaryPlus;
    OpUnaryMinus;                OpLogicalNot;                OpBitwiseNot;

    Optional;                    Out;                         Override;
    ParaAssignmentList;          ParaSpec;                    ParExpressionSpec;
    Pass;                        PortCallBody;                PortCatchOp;
    PortDef;                     PortElement;                 PortGetCallOp;
    PortGetReplyOp;              PortInstance;                PortReceiveOp;
    PortRedirect;                PortRedirectWithParam;       PortRef;
    PortType;                    ProcedureAttribs;            ProcedureList;
    Quadruple;                   RaiseStatement;              ReadTimerOp;
    ReceiveStatement;            RecordDef;                   RecordOfDef;
    RedirectSpec;                RepeatStatement;             ReplyStatement;
    ReplyValue;                  ReturnStatement;             ReturnType;
    RunsOnSpec;                  SenderSpec;                  SendStatement;
    SelfOp;                      SetDef;                      SetLocalVerdict;
    SetOfDef;                    SignatureDef;                SignatureFormalParList;
    SingleConstDef;              SingleValueOrAttrib;         SingleVarInstance;
    SingleWithAttrib;            SomeActualParList;           SomeFormalPar;
    SomeInstance;                SomeRunningOp;               SomeStartStatement;
    SomeStopStatement;           StartTCStatement;            StatementBlock;
    StopTCStatement;             StringLength;                StructDefFormalParList;
    StructFieldDef;              SubTypeDef;                  SUTStatements;
    SystemOp;                    SystemSpec;                  TemplateDef;
    TemplateFormalParList;       TestcaseActualParList;       TestcaseDef;
    TestcaseFormalParList;       TestcaseInstance;            TestcaseRef;
    TeststepDef;                 Timeout;                     TimeoutStatement;
    TimerInstance;               ToClause;                    TriggerStatement;
    True;                        TTCN3Document;               TTCN3Module;
    TTCN3ModuleId;               TypeActualParList;           TypeAddress;
    TypeBitstring;               TypeBoolean;                 TypeChar;
    TypeCharstring;              TypeDefault;
    TypeFloat;                   TypeHexstring;
    TypeInteger;                 TypeObjId;                   TypeOctetstring;
    TypeUniversalChar;           TypeUniversalCharstring;     TypeVerdictType;
    UnionDef;                    UnionFieldDef;               UnmapStatement;
    ValueList;                   ValueMatchSpec;              ValueofOp;
    ValueSpec;                   VariableAssignment;          VariableList;
    VarInstance;                 WhileStatement;              WithStatement;
  }

{
  public:

    unsigned int numberOfErrors() const;
    void resetErrors();

    static const string &versionInfo();

    antlr::RefEnvAST getEnvAST() { return currentEnv; }

  protected:

    void reportError( const string &errorMessage );
    void reportError( const antlr::RecognitionException& ex );
    void reportMessage( const string &message );

  private:

    void consume();
    bool optionalSemicolon();
    bool checkModuleControlPart( string statementName );

    antlr::RefEnvAST currentEnv;
    int              lastToken;
    unsigned int     noOfErrors;
    bool             isModuleControlPart;
}

ttcn3Document
  {
    isModuleControlPart = false;

    currentEnv = antlr::RefEnvAST( new antlr::EnvAST() );
    NEWENVIRONMENT_BEGIN
    currentEnv -> setType( TTCN3Document );
  } :

  ( ttcn3Module )+

  {
    SCOPEAST( scopeAST, TTCN3Document );
    ## = #( scopeAST, ## );

    NEWENVIRONMENT_END
  }

;


/*****  SECTION A.1.6.1 - TTCN Module *****/

ttcn3Module
  {
    NEWENVIRONMENT_BEGIN
    currentEnv -> setType( TTCN3Module );
  } :

  start:MODULE! id:ttcn3ModuleId ( mpl:moduleParList )?
  BeginChar! ( moduleDefinitionsPart )? ( moduleControlPart )? EndChar!
  ( ws:withStatement )? ( SemiColon! | { optionalSemicolon() }? )

  {
    antlr::RefAST  newAST = astFactory.create( TTCN3Module );
    antlr::EnvAST::addASTList( newAST, 3, &astFactory, &#id, &#mpl, &#ws );
    currentEnv -> getParentEnv() -> addChild( newAST );

    SCOPEAST( scopeAST, TTCN3Module );
    ## = #( scopeAST, ## );

    SETLINECOLUMN( ##, start );

    NEWENVIRONMENT_END
  }

;


ttcn3ModuleId :

  /*module*/ Identifier
  ( options { greedy = true; } : definitiveIdentifier )?

  {
    ## = #( [TTCN3ModuleId], ## );
  }

;


definitiveIdentifier :

  start:Dot! OBJID! "{"! definitiveObjIdComponentList "}"!

  { ## = #( [DefinitiveIdentifier], ## ); SETLINECOLUMN( ##, start ); }

;


definitiveObjIdComponentList :   // root node already added in 'definitiveIdentifier'

  ( definitiveObjIdComponent )+

;


definitiveObjIdComponent :

    nameForm
  | definitiveNumberForm
  | definitiveNameAndNumberForm

;


definitiveNumberForm :

  Number

;


definitiveNameAndNumberForm :

  Identifier "("! definitiveNumberForm ")"!

  { ## = #( [DefinitiveNameAndNumberForm], ## ); }

;


moduleParList :

  start:"("! modulePar ( ","! modulePar )* ")"!

  {
    ## = #( [ModuleParList], ## ); SETLINECOLUMN( ##, start );

    currentEnv -> addChild( astFactory.dupTree( ## ) );
  }

;


modulePar { bool in = false; } :

  ( start:IN! { in = true; } )? /*modulePar*/ type /*modulePar*/ Identifier
  ( AssignmentChar constantExpression )?

  { ## = #( [ModulePar], ## ); if ( in ) SETLINECOLUMN( ##, start ); }

;


/*****  SECTION A.1.6.2 - Module Definitions Part *****/

moduleDefinitionsPart :

  moduleDefinitionsList

  { ## = #( [ModuleDefinitionsPart], ## ); }

;


moduleDefinitionsList :   // root node already added in 'moduleDefinitionsPart'

  ( moduleDefinition ( SemiColon! | { optionalSemicolon() }? ) )+

;


moduleDefinition :

  (   typeDef
    | constDef
    | templateDef
    | functionDef
    | signatureDef
    | testcaseDef
    | teststepDef
    | importDef
    | groupDef
    | extFunctionDef
    | extConstDef )
  ( withStatement )?

  { ## = #( [ModuleDefinition], ## ); }

;


/*****  SECTION A.1.6.2.1 - Typedef Definitions *****/

typeDef :   // no root node; nodes will be introduced in children of 'typeDefBody'

  start:TYPE! body:typeDefBody

  { SETLINECOLUMN( #body, start ); }

;


typeDefBody :

    structuredTypeDef
  | subTypeDef

;


structuredTypeDef :

    recordDef
  | unionDef
  | setDef
  | recordOfDef
  | setOfDef
  | enumDef
  | portDef
  | componentDef

;


recordDef :

  start:RECORD! structDefBody

  {
    ## = #( [RecordDef], ## ); SETLINECOLUMN( ##, start );

    currentEnv -> addChild( astFactory.dupTree( ## ) );
  }

;


structDefBody :   // root node already added in 'recordDef' and 'setDef'

  (   /*structType*/ Identifier
      ( structDefFormalParList )?
    | ch:ADDRESS! { ADDCHILD(#[Address],ch); }
  )

  BeginChar!( structFieldDef ( ","! structFieldDef )* )? EndChar!

;


structDefFormalParList :

  start:"("! structDefFormalPar ( ","! structDefFormalPar )* ")"!

  { ## = #( [StructDefFormalParList], ## ); SETLINECOLUMN( ##, start ); }

;


structDefFormalPar :

    ( formalValuePar ) =>   // Reason: LL(k)-approx is insufficient
    formalValuePar
  | formalTypePar

;


structFieldDef :

  type /*structField*/ Identifier ( arrayDef )? ( subTypeSpec )?
  ( ch:OPTIONAL! { ADDCHILD(#[Optional],ch); } )?

  { ## = #( [StructFieldDef], ## ); }

;


unionDef :

  start:UNION! unionDefBody

  { ## = #( [UnionDef], ## ); SETLINECOLUMN( ##, start ); }

;


unionDefBody :   // root node already added in 'unionDef'

  (   /*structType*/ Identifier ( structDefFormalParList )?
    | ch:ADDRESS! { ADDCHILD(#[Address],ch); } )
  BeginChar! unionFieldDef ( ","! unionFieldDef )* EndChar!

;


unionFieldDef :

  type /*structField*/ Identifier ( arrayDef )? ( subTypeSpec )?

  { ## = #( [UnionFieldDef], ## ); }

;


setDef :

  start:SET! structDefBody

  {
    ## = #( [SetDef], ## ); SETLINECOLUMN( ##, start );

    currentEnv -> addChild( astFactory.dupTree( ## ) );
  }

;


recordOfDef :

  start:RECORD! OF! ( stringLength )? structOfDefBody

  { ## = #( [RecordOfDef], ## ); SETLINECOLUMN( ##, start ); }

;


structOfDefBody :   // root node already added in 'recordOfDef' and 'setOfDef'

  type
  (   /*structType*/ Identifier
    | ch:ADDRESS! { ADDCHILD(#[Address],ch); } )
  ( subTypeSpec )?

;


setOfDef :

  start:SET! OF! ( stringLength )? structOfDefBody

  { ## = #( [SetOfDef], ## ); SETLINECOLUMN( ##, start ); }

;


enumDef :

  start:ENUMERATED!
  (   /*enumType*/ Identifier
    | ch:ADDRESS! { ADDCHILD(#[Address],ch); } )
  BeginChar! namedValueList EndChar!

  { ## = #( [EnumDef], ## ); SETLINECOLUMN( ##, start ); }

;


namedValueList :   // no root node as the rule itself does not provide additional information

  namedValue ( ","! namedValue )*

;


namedValue :

  /*namedValue*/ Identifier ( "("! Number ")"! )?

  { ## = #( [NamedValue], ## ); }

;


subTypeDef :

  type
  (   /*subType*/ Identifier
    | ch:ADDRESS! { ADDCHILD(#[Address],ch); } )
  ( arrayDef )? ( subTypeSpec )?

  { ## = #( [SubTypeDef], ## ); }

;


subTypeSpec :

    allowedValues
  | stringLength

;


allowedValues :

  start:"("! valueOrRange ( ","! valueOrRange )* ")"!

  { ## = #( [AllowedValues], ## ); SETLINECOLUMN( ##, start ); }

;


valueOrRange :   // no root node due to nondeterminism

    ( lowerBound ".." ) =>
    integerRangeDef
  | singleConstExpression

;


integerRangeDef :

  lowerBound ".."! upperBound

  { ## = #( [IntegerRangeDef], ## ); }

;


stringLength :

  start:LENGTH! "("! singleConstExpression ( ".."! upperBound )? ")"!

  { ## = #( [StringLength], ## ); SETLINECOLUMN( ##, start ); }

;


portType :

  ( globalModuleId Dot! )? /*portType*/ Identifier

  { ## = #( [PortType], ## ); }

;


portDef :

  start:PORT! portDefBody

  { ## = #( [PortDef], ## ); SETLINECOLUMN( ##, start ); }

;


portDefBody :   // root node already added in 'portDef'

  /*portType*/ Identifier portDefAttribs

;


portDefAttribs :

    messageAttribs
  | procedureAttribs
  | mixedAttribs

;


messageAttribs :

  start:MESSAGE! BeginChar! ( messageList ( SemiColon! | { optionalSemicolon() }? ) )+ EndChar!

  { ## = #( [MessageAttribs], ## ); SETLINECOLUMN( ##, start ); }

;


messageList :

  direction allOrTypeList

  { ## = #( [MessageList], ## ); }

;


direction! :

    start1:IN!      { ## = #[In]; SETLINECOLUMN( ##, start1 ); }
  | start2:OUT!     { ## = #[Out]; SETLINECOLUMN( ##, start2 ); }
  | start3:INOUT!   { ## = #[InOut]; SETLINECOLUMN( ##, start3 ); }

;


allOrTypeList :

    start:ALL!   { ## = #[All]; SETLINECOLUMN( ##, start ); }
  | typeList

;


typeList :   // no root node as the rule itself does not provide additional information

  type ( ","! type )*

;


procedureAttribs :

  start:PROCEDURE! BeginChar! ( procedureList ( SemiColon! | { optionalSemicolon() }? ) )+ EndChar!

  { ## = #( [ProcedureAttribs], ## ); SETLINECOLUMN( ##, start ); }

;


procedureList :

  direction allOrSignatureList

  { ## = #( [ProcedureList], ## ); }

;


allOrSignatureList :

    start:ALL!      { ## = #[All]; SETLINECOLUMN( ##, start ); }
  | signatureList

;


signatureList :   // no root node as the rule itself does not provide additional information

  signature ( ","! signature )*

;


mixedAttribs :

  start:MIXED! BeginChar! ( mixedList ( SemiColon! | { optionalSemicolon() }? ) )+ EndChar!

  { ## = #( [MixedAttribs], ## ); SETLINECOLUMN( ##, start ); }

;


mixedList :

  direction procOrTypeList

  { ## = #( [MixedList], ## ); }

;


procOrTypeList :   // no root node as the rule itself does not provide additional information

    start:ALL!  { ## = #[All]; SETLINECOLUMN( ##, start ); }
  | procOrType ( ","! procOrType )*

;


procOrType :

    type
  // | signature   // subsumed by 'type.referencedType'

;


componentDef :

  start:COMPONENT! /*componentType*/ Identifier
  BeginChar! ( componentDefList )? EndChar!

  { ## = #( [ComponentDef], ## ); SETLINECOLUMN( ##, start ); }

;


componentType :

  ( ( globalModuleId Dot Identifier ) =>   // Reason: LL(3) is insufficient
    globalModuleId Dot!
  )? /*componentType*/ Identifier

  { ## = #( [ComponentType], ## ); }

;


componentDefList :   // no root node as the rule itself does not provide additional information

  ( componentElementDef ( SemiColon! | { optionalSemicolon() }? ) )+

;


componentElementDef :

    portInstance
  | varInstance
  | timerInstance
  | constDef

;


portInstance :

  start:PORT! portType portElement ( ","! portElement )*

  { ## = #( [PortInstance], ## ); SETLINECOLUMN( ##, start ); }

;


portElement :

  /*port*/ Identifier ( arrayDef )?

  { ## = #( [PortElement], ## ); }

;


/*****  SECTION A.1.6.2.2 - Constant Definitions  *****/

constDef :

  start:CONST! type constList

  {
    ## = #( [ConstDef], ## ); SETLINECOLUMN( ##, start );

    currentEnv -> addChild( astFactory.dupTree( ## ) );
  }

;


constList :   //  no root node as the rule itself does not provide additional information

  singleConstDef ( ","! singleConstDef )*

;


singleConstDef :

  /*const*/ Identifier ( arrayDef )? AssignmentChar! constantExpression

  { ## = #( [SingleConstDef], ## ); }

;


/*****  SECTION A.1.6.2.3 - Template Definitions  *****/

templateDef :

  start:TEMPLATE! baseTemplate ( derivedDef )? AssignmentChar! templateBody

  { ## = #( [TemplateDef], ## ); SETLINECOLUMN( ##, start ); }

;


baseTemplate :

  (   type
    // | signature   // subsumed by 'type.referencedType'
  )
  /*template*/ Identifier ( "("! templateFormalParList ")"! )?

  { ## = #( [BaseTemplate], ## ); }

;


derivedDef :

  start:MODIFIES! templateRef

  { ## = #( [DerivedDef], ## ); SETLINECOLUMN( ##, start ); }

;


templateFormalParList :

  templateFormalPar ( ","! templateFormalPar )*

  { ## = #( [TemplateFormalParList], ## ); }

;


templateFormalPar :

    ( ( IN )? TEMPLATE ) =>
    formalTemplatePar
  | formalValuePar

;


templateBody :

    simpleSpec
  | ( BeginChar ( fieldReference AssignmentChar | EndChar ) ) =>
    fieldSpecList
  | arrayValueOrAttrib

;


simpleSpec :

  singleValueOrAttrib

;


fieldSpecList :

  start:BeginChar! ( fieldSpec ( ","! fieldSpec )* )? EndChar!

  { ## = #( [FieldSpecList], ## ); SETLINECOLUMN( ##, start ); }

;


fieldSpec :

  fieldReference AssignmentChar! templateBody

  { ## = #( [FieldSpec], ## ); }

;


fieldReference :   // no root node due to nondeterminism

    recordRef
  | arrayOrBitRef
  // | parRef     // subsumed by 'recordRef'

;


recordRef :   // no root node due to nondeterminism

  /*structField*/ Identifier

;


parRef :   // no root node due to nondeterminism

  signatureParIdentifier

;


signatureParIdentifier :   // no root node due to nondeterminism

  /*valuePar*/ Identifier

;


arrayOrBitRef :

  start:"["! fieldOrBitNumber "]"!

  { ## = #( [ArrayOrBitRef], ## ); SETLINECOLUMN( ##, start ); }

;


fieldOrBitNumber :

  singleExpression

;


singleValueOrAttrib :

    ( ( globalModuleId Dot )? Identifier templateActualParList ) =>   // conflicts with singleExpression
    templateRefWithParList
  | ( ~LParenthesis |   // LL(k) insufficient; !!! ANTLR first checks LL(k), then the predicate !!!
      ( integerRange ) => integerRange | valueList ) =>
    matchingSymbol ( extraMatchingAttributes )?
  | singleExpression ( extraMatchingAttributes )?

;


arrayValueOrAttrib :

  start:BeginChar! arrayElementSpecList EndChar!

  { ## = #( [ArrayValueOrAttrib], ## ); SETLINECOLUMN( ##, start ); }

;


arrayElementSpecList :   // root node already added in 'arrayValueOrAttrib'

  arrayElementSpec ( Comma! arrayElementSpec )*

;


arrayElementSpec :

    ( Dash ( Comma | EndChar ) ) =>   // Reason: LL(3) is insufficient
    notUsedSymbol
  | templateBody

;


notUsedSymbol! :

  start:Dash!

  { ## = #[NotUsed]; SETLINECOLUMN( ##, start ); }

;


matchingSymbol :

    complement
  | omit
  | anyValue
  | anyOrOmit
  | ( LParenthesis lowerBound ".." ) =>   // Reason: LL(3) is insufficient
    integerRange
  | ( LParenthesis singleConstExpression Comma ) =>   // Reason: LL(3) is insufficient
    valueList
  | bitStringMatch
  | hexStringMatch
  | octetStringMatch
  | charStringMatch

;


extraMatchingAttributes :

    lengthMatch
  | ifPresentMatch

;


bitStringMatch :   // 'BString' is covered by 'singleExpression'

  BStringMatch

;


hexStringMatch :   // 'HString' is covered by 'singleExpression'

  HStringMatch

;


octetStringMatch :   // 'OString' is covered by 'singleExpression'

  OStringMatch

;


charStringMatch :

  start:PATTERN! charStringValue

  { ## = #( [CharStringMatch], ## ); SETLINECOLUMN( ##, start ); }

;


complement :

  start:COMPLEMENT!
  (   ( LParenthesis singleConstExpression Comma ) =>   // Reason: LL(3) is insufficient
      valueList
    | singleConstExpression
  )

  { ## = #( [Complement], ## ); SETLINECOLUMN( ##, start ); }

;


omit! :

  start:OMIT!

  { ## = #[Omit]; SETLINECOLUMN( ##, start ); }

;


anyValue! :

  start:"?"!

  { ## = #[AnyValue]; SETLINECOLUMN( ##, start ); }

;


anyOrOmit! :

  start:"*"!

  { ## = #[AnyOrOmit]; SETLINECOLUMN( ##, start ); }

;


valueList :

  start:"("! singleConstExpression ( ","! singleConstExpression )+ ")"!

  { ## = #( [ValueList], ## ); SETLINECOLUMN( ##, start ); }

;


lengthMatch :

  stringLength

;


ifPresentMatch :

  start:IFPRESENT!

  { ## = #[IfPresentMatch]; SETLINECOLUMN( ##, start ); }

;


integerRange :

  start:LParenthesis! lowerBound ".."! upperBound RParenthesis!

  { ## = #( [IntegerRange], ## ); SETLINECOLUMN( ##, start ); }

;


lowerBound :

    singleConstExpression
  | start:Dash! INFINITY!   { ## = #[MinusInfinity]; SETLINECOLUMN( ##, start ); }

;


upperBound :

    singleConstExpression
  | start:INFINITY!   { ## = #[Infinity]; SETLINECOLUMN( ##, start ); }

;


templateInstance :   // no root node due to nondeterminism

  inLineTemplate

;


templateRefWithParList :   // no root node due to nondeterminism

    ( globalModuleId Dot! )? /*template*/ Identifier ( templateActualParList )?

    { ## = #( [SomeInstance], ## ); }

  // | /*templatePar*/ Identifier   // subsumed by first alternative

;


templateRef :   // root node already added in 'derivedDef'

    ( globalModuleId Dot! )? /*template*/ Identifier

  // | /*templatePar*/ Identifier   // subsumed by first alternative

;


inLineTemplate :   // no root node due to nondeterminism

  ( ( type Colon ) =>   // Reason: LL(3) is insufficient
    (   type
      // | signature   // subsumed by 'type.referencedType'
    )
    Colon! )?
  ( derivedDef AssignmentChar! )? templateBody

;


templateActualParList :   // no specific root node due to nondeterminism

  start:"("! templateActualPar ( ","! templateActualPar )* ")"!

  { ## = #( [SomeActualParList], ## ); SETLINECOLUMN( ##, start ); }

;


templateActualPar :

  templateInstance

;


templateOps :

    matchOp
  | valueofOp

;


matchOp :

  start:MATCH! "("! expression ","! templateInstance ")"!

  { ## = #( [MatchOp], ## ); SETLINECOLUMN( ##, start ); }

;


valueofOp :

  start:VALUEOF! "("! templateInstance ")"!

  { ## = #( [ValueofOp], ## ); SETLINECOLUMN( ##, start ); }

;


/*****  SECTION A.1.6.2.4 - Function Definitions  *****/

functionDef
  {
    NEWENVIRONMENT_BEGIN
    currentEnv -> setType( FunctionDef );
  } :

  start:FUNCTION! /*function*/ id:Identifier
  "("! ( ffpl:functionFormalParList )? ")"! ( ros:runsOnSpec )? ( rt:returnType )?
  BeginChar! functionBody EndChar!

  {
    antlr::RefAST  newAST = astFactory.create( FunctionDef );
    antlr::EnvAST::addASTList( newAST, 4, &astFactory, &#id, &#ffpl, &#ros, &#rt );	
    currentEnv -> getParentEnv() -> addChild( newAST );

    SCOPEAST( ScopeAST, FunctionDef );
    ## = #( ScopeAST, ## );

    SETLINECOLUMN( ##, start );

    NEWENVIRONMENT_END
  }

;


functionFormalParList :

  functionFormalPar ( ","! functionFormalPar )*

  {
    ## = #( [FunctionFormalParList], ## );

    currentEnv -> addChild( astFactory.dupTree( ## ) );
  }

;


functionFormalPar :

    ( ( IN )? TIMER ) =>
    formalTimerPar
  | ( ( IN )? TEMPLATE ) =>
    formalTemplatePar
  // | formalPortPar        // subsumed by 'formalValuePar'
  | formalValuePar

;


returnType :

  start:RETURN! type

  {
    ## = #( [ReturnType], ## ); SETLINECOLUMN( ##, start );

    currentEnv -> addChild( astFactory.dupTree( ## ) );
  }

;


runsOnSpec :

  start:RUNS! ON!
  (   componentType
    | ch:MTC!   { ADDCHILD(#[MTCOp],ch); } )

  {
    ## = #( [RunsOnSpec], ## ); SETLINECOLUMN( ##, start );

    currentEnv -> addChild( astFactory.dupTree( ## ) );
  }

;


functionBody :

  ( functionStatementOrDefList )?

  { ## = #( [FunctionBody], ## ); }

;


functionStatementOrDefList :   // root node already added in 'functionBody' and 'statementBlock'

  ( functionStatementOrDef ( SemiColon! | { optionalSemicolon() }? ) )+

;


functionStatementOrDef :

    functionLocalDef
  | functionLocalInst
  | functionStatement

;


functionLocalInst :

    varInstance
  | timerInstance

;


functionLocalDef :

  constDef

;


functionStatement :

    (   CONNECT | DISCONNECT | MAP | UNMAP | ( doneStatement ) => doneStatement /*sorry, inevitable*/
      | startTCStatement | STOP ) =>      // Attention: 'startTCStatement' overlaps
    configurationStatements               // with 'timerStatements.startTimerStatement'
  | ( timerRef Dot ( START | STOP | TIMEOUT ) | ANY TIMER | ALL TIMER ) =>
    timerStatements                 // Attention: 'startTimerStatement'/'stopTimerStatement'
                                    // overlaps with 'communicationStatements'!!!
  | ( variableRef AssignmentChar | LOG | FOR | WHILE | DO | IF ) =>   // LL(3) is insufficient
    basicStatements
  | (   port Dot (   SEND | CALL | REPLY | RAISE | RECEIVE
                   | TRIGGER | GETCALL | GETREPLY | CATCH | CHECK | CLEAR | START | STOP )
      | ANY PORT | ALL PORT ) =>
    communicationStatements
  | behaviourStatements
  | verdictStatements
  | sutStatements

;


functionInstance :   // no root node due to nondeterminism

  functionRef "("! ( functionActualParList )? ")"!

  { ## = #( [SomeInstance], ## ); }

;


functionRef :   // no root node due to nondeterminism

  ( globalModuleId Dot! )? /*function*/ Identifier

;


functionActualParList :   // no specific root node due to nondeterminism

  functionActualPar ( ","! functionActualPar )*

  { ## = #( [SomeActualParList], ## ); }

;


functionActualPar :   // no root node due to nondeterminism

  // timerRef             // subsumed by 'componentRef'
    ( componentRef ) =>   // Reason: Grammar induced nondeterminisms
    componentRef
  | templateInstance
  // | port               // subsumed by 'componentRef'

;


/*****  SECTION A.1.6.2.5 - Signature Definitions  *****/

signatureDef :

  start:SIGNATURE! /*signature*/ Identifier
  "("! ( signatureFormalParList )? ")"!
  (   returnType
    | ch:NOBLOCK! { ADDCHILD(#[Noblock],ch) } )?
  ( exceptionSpec )?

  { ## = #( [SignatureDef], ## ); SETLINECOLUMN( ##, start ); }

;


signatureFormalParList :

  signatureFormalPar ( ","! signatureFormalPar )*

  { ## = #( [SignatureFormalParList], ## ); }

;


signatureFormalPar :

  formalValuePar

;


exceptionSpec :

  start:EXCEPTION! "("! exceptionTypeList ")"!

  { ## = #( [ExceptionSpec], ## ); SETLINECOLUMN( ##, start ); }

;


exceptionTypeList :   // root node already added in 'exceptionSpec'

  type ( ","! type )*

;


signature :   // no root node due to nondeterminism

  ( globalModuleId Dot! )? /*signature*/ Identifier

;


/*****  SECTION A.1.6.2.6 - Testcase Definitions  *****/

testcaseDef
  {
    NEWENVIRONMENT_BEGIN
    currentEnv -> setType( TestcaseDef );
  } :

  start:TESTCASE! /*testcase*/ id:Identifier
  "("! ( tfpl:testcaseFormalParList )? ")"! cs:configSpec
  BeginChar! functionBody EndChar!

  {
    antlr::RefAST  newAST = astFactory.create( TestcaseDef );
    antlr::EnvAST::addASTList( newAST, 3, &astFactory, &#id, &#tfpl, &#cs );
    currentEnv -> getParentEnv() -> addChild( newAST );

    SCOPEAST( ScopeAST, TestcaseDef );
    ## = #( ScopeAST, ## );

    SETLINECOLUMN( ##, start );

    NEWENVIRONMENT_END
  }

;


testcaseFormalParList :

  testcaseFormalPar ( ","! testcaseFormalPar )*

  {
    ## = #( [TestcaseFormalParList], ## );

    currentEnv -> addChild( astFactory.dupTree( ## ) );
  }

;


testcaseFormalPar :

    ( ( IN )? TEMPLATE ) =>
    formalTemplatePar
  | formalValuePar

;


configSpec :

  runsOnSpec ( systemSpec )?

  { ## = #( [ConfigSpec], ## ); }

;


systemSpec :

  start:SYSTEM! componentType

  { ## = #( [SystemSpec], ## ); SETLINECOLUMN( ##, start ); }

;


testcaseInstance :

  start:EXECUTE! "("! testcaseRef "("! ( testcaseActualParList )? ")"! ( ","! timerValue )? ")"!

  { ## = #( [TestcaseInstance], ## ); SETLINECOLUMN( ##, start ); }

;


testcaseRef :

  ( globalModuleId Dot! )? /*testcase*/ Identifier

  { ## = #( [TestcaseRef], ## ); }

;


testcaseActualParList :

  testcaseActualPar ( ","! testcaseActualPar )*

  { ## = #( [TestcaseActualParList], ## ); }

;


testcaseActualPar :

  templateInstance

;


/*****  SECTION A.1.6.2.7 - Teststep Definitions  *****/

teststepDef
  {
    NEWENVIRONMENT_BEGIN
    currentEnv -> setType( TeststepDef );
  } :

  start:TESTSTEP! /*teststep*/ id:Identifier
  "("! ( tfpl:teststepFormalParList )? ")"! ( ros:RunsOnSpec )?
  BeginChar! altGuardList EndChar!

  {
    antlr::RefAST  newAST = astFactory.create( TeststepDef );
    antlr::EnvAST::addASTList( newAST, 3, &astFactory, &#id, &#tfpl, &#ros );
    currentEnv -> getParentEnv() -> addChild( newAST );

    SCOPEAST( ScopeAST, TeststepDef );
    ## = #( ScopeAST, ## );

    SETLINECOLUMN( ##, start );

    NEWENVIRONMENT_END
  }

;


teststepFormalParList :

  functionFormalParList

;


teststepInstance :   // no root node due to nondeterminism

  teststepRef "("! ( functionActualParList )? ")"!

  { ## = #( [SomeInstance], ## ); }

;


teststepRef :   // no root node due to nondeterminism

  ( globalModuleId Dot! )? /*teststep*/ Identifier

;


/*****  SECTION A.1.6.2.8 - Import Definitions  *****/

importDef :   // no root node; nodes will be introduced in children of 'importSpec'

  start:IMPORT! body:importSpec

  { SETLINECOLUMN( #body, start ); }

;


importSpec :

    importAllSpec
  | importGroupSpec
  | importTypeDefSpec
  | importTemplateSpec
  | importConstSpec
  | importTestcaseSpec
  | importTeststepSpec
  | importFunctionSpec
  | importSignatureSpec

;


importAllSpec :

  start:ALL! ( defKeyword )? importFromSpec

  { ## = #( [ImportAllSpec], ## ); SETLINECOLUMN( ##, start ); }

;


importFromSpec :

  start:FROM! moduleId ( ch:NONRECURSIVE! { ADDCHILD(#[Nonrecursive],ch); } )?

  { ## = #( [ImportFromSpec], ## ); SETLINECOLUMN( ##, start ); }

;


moduleId :

  globalModuleId ( "(" extModuleActualParList ")" )? ( languageSpec )?

  { ## = #( [ModuleId], ## ); }

;


languageSpec :

  start:LANGUAGE! freeText

  { ## = #( [LanguageSpec], ## ); SETLINECOLUMN( ##, start ); }

;


globalModuleId :   // no root node due to nondeterminism

  /*module*/ Identifier ( options { greedy = true; } : Dot! objectIdentifierValue )?   // reason of ambiguity unknown

;


extModuleActualParList :

  (   extModuleParExpressionList
    | extModuleParArrayExpression )

  { ## = #( [ExtModuleActualParList], ## ); }

;


extModuleParExpressionList :

  parExpressionSpec ( "," parExpressionSpec )*

;


parExpressionSpec :

  /*extModulePar*/ Identifier AssignmentChar /*modulePar*/ expression

  { ## = #( [ParExpressionSpec], ## ); }

;


extModuleParArrayExpression :

  notUsedOrModuleParExpression ( "," notUsedOrModuleParExpression )*

;


notUsedOrModuleParExpression :

    ( Dash ( ")" | Comma ) ) =>   // Reason: LL(3)-approx is insufficient
    notUsedSymbol
  | /*modulePar*/ expression

;


defKeyword :

    start1:TYPE!        { ## = #[DefType];      SETLINECOLUMN( ##, start1 ); }
  | start2:CONST!       { ## = #[DefConst];     SETLINECOLUMN( ##, start2 ); }
  | start3:TEMPLATE!    { ## = #[DefTemplate];  SETLINECOLUMN( ##, start3 ); }
  | start4:TESTCASE!    { ## = #[DefTestcase];  SETLINECOLUMN( ##, start4 ); }
  | start5:FUNCTION!    { ## = #[DefFunction];  SETLINECOLUMN( ##, start5 ); }
  | start6:SIGNATURE!   { ## = #[DefSignature]; SETLINECOLUMN( ##, start6 ); }
  | start7:TESTSTEP!    { ## = #[DefTeststep];  SETLINECOLUMN( ##, start7 ); }

;


importGroupSpec :

  start:GROUP! /*group*/ Identifier ( ","! /*group*/ Identifier )* importFromSpec

  { ## = #( [ImportGroupSpec], ## ); SETLINECOLUMN( ##, start ); }

;


importTypeDefSpec :

  start:TYPE! typeDefIdentifier ( ","! typeDefIdentifier )* importFromSpec

  { ## = #( [ImportTypeDefSpec], ## ); SETLINECOLUMN( ##, start ); }

;


typeDefIdentifier :

  /*structType,enumType,portType,componentType,subType*/ Identifier

;


importTemplateSpec :

  start:TEMPLATE! /*template*/ Identifier ( ","! /*template*/ Identifier )* importFromSpec

  { ## = #( [ImportTemplateSpec], ## ); SETLINECOLUMN( ##, start ); }

;


importConstSpec :

  start:CONST! /*const*/ Identifier ( ","! /*const*/ Identifier )* importFromSpec

  { ## = #( [ImportConstSpec], ## ); SETLINECOLUMN( ##, start ); }

;


importTestcaseSpec :

  start:TESTCASE! /*testcase*/ Identifier ( ","! /*testcase*/ Identifier )* importFromSpec

  { ## = #( [ImportTestcaseSpec], ## ); SETLINECOLUMN( ##, start ); }

;


importFunctionSpec :

  start:FUNCTION! /*function*/ Identifier ( ","! /*function*/ Identifier )* importFromSpec

  { ## = #( [ImportFunctionSpec], ## ); SETLINECOLUMN( ##, start ); }

;


importSignatureSpec :

  start:SIGNATURE! /*signature*/ Identifier ( ","! /*signature*/ Identifier )* importFromSpec

  { ## = #( [ImportSignatureSpec], ## ); SETLINECOLUMN( ##, start ); }

;


importTeststepSpec :

  start:TESTSTEP! /*teststep*/ Identifier ( ","! /*teststep*/ Identifier )* importFromSpec

  { ## = #( [ImportTeststepSpec], ## ); SETLINECOLUMN( ##, start ); }

;


/*****  SECTION A.1.6.2.9 - Group Definitions  *****/

groupDef :

  start:GROUP! /*group*/ Identifier BeginChar! ( moduleDefinitionsPart )? EndChar!

  { ## = #( [GroupDef], ## ); SETLINECOLUMN( ##, start ); }

;


/*****  SECTION A.1.6.2.10 - External Function Definitions  *****/

extFunctionDef :

  start:EXTERNAL! FUNCTION! /*extFunction*/ Identifier
  "("! ( functionFormalParList )? ")"! ( returnType )?

  {
    ## = #( [ExtFunctionDef], ## ); SETLINECOLUMN( ##, start );

    currentEnv -> addChild( astFactory.dupTree( ## ) );
  }

;


/*****  SECTION A.1.6.2.11 - External Constant Definitions  *****/

extConstDef :

  start:EXTERNAL! CONST! type /*extConst*/ Identifier

  {
    ## = #( [ExtConstDef], ## ); SETLINECOLUMN( ##, start );

    currentEnv -> addChild( astFactory.dupTree( ## ) );
  }

;


/*****  SECTION A.1.6.3 - Control Part  *****/

moduleControlPart
  {
    NEWENVIRONMENT_BEGIN
    currentEnv -> setType( ModuleControlPart );
  } :

  start:CONTROL!
  BeginChar! { isModuleControlPart = true; } moduleControlBody { isModuleControlPart = false; } EndChar!
  ( ws:withStatement )? ( SemiColon! | { optionalSemicolon() }? )

  {
    antlr::RefAST  newAST = astFactory.create( ModuleControlPart );
    antlr::EnvAST::addASTList( newAST, 1, &astFactory, &#ws );
    currentEnv -> getParentEnv() -> addChild( newAST );

    SCOPEAST( ScopeAST, ModuleControlPart );
    ## = #( ScopeAST, ## );

    SETLINECOLUMN( ##, start );

    NEWENVIRONMENT_END
  }

;


moduleControlBody :

  ( controlStatementOrDefList )?

  { ## = #( [ModuleControlBody], ## ); }

;


controlStatementOrDefList :   // root node already added in 'moduleControlBody' and 'controlStatementBlock'

  ( controlStatementOrDef ( SemiColon! | { optionalSemicolon() }? ) )+

;


controlStatementOrDef :

    functionLocalInst
  | controlStatement
  | functionLocalDef

;


controlStatement :   // no root node due to nondeterminism

    ( timerRef Dot ( START | STOP | TIMEOUT ) | ANY TIMER | ALL TIMER ) =>
    timerStatements          // Reason: LL(3) is insufficient
  | ( variableRef AssignmentChar | LOG | FOR | WHILE | DO | IF ) =>
    basicStatements   // Reason: LL(3) is insufficient
  | behaviourStatements
  | sutStatements

;


/*****  SECTION A.1.6.3.1 - Variable Instantiation  *****/

varInstance :

  start:VAR! type varList

  {
    ## = #( [VarInstance], ## ); SETLINECOLUMN( ##, start );

    currentEnv -> addChild( astFactory.dupTree( ## ) );
  }

;


varList :   // no root node as the rule itself does not provide additional information

  singleVarInstance ( ","! singleVarInstance )*

;


singleVarInstance :

  /*var*/ Identifier ( arrayDef )? ( AssignmentChar! varInitialValue )?

  { ## = #( [SingleVarInstance], ## ); }

;


varInitialValue :

  expression

;


variableRef :   // no root node due to nondeterminism

  /*var,valuePar*/ Identifier
  (
    ( Dot Identifier | "[" ) =>   // Reason: LL(3)-approx is insufficient
    extendedFieldReference
  )?

;


/*****  SECTION A.1.6.3.2 - Timer Instantiation  *****/

timerInstance :

  start:TIMER! /*timer*/ Identifier ( arrayDef )? ( AssignmentChar! timerValue )?

  { ## = #( [TimerInstance], ## ); SETLINECOLUMN( ##, start ); }

;


timerValue :

  expression

;


timerRef :   // no root node due to nondeterminism

  /*timer,timerPar*/ Identifier ( arrayOrBitRef )?

;


/*****  SECTION A.1.6.3.3 - Component Operations  *****/

configurationStatements :

    connectStatement
  | mapStatement
  | disconnectStatement
  | unmapStatement
  | ( doneStatement ) =>   // Reason: LL(3)-approx is insufficient
    doneStatement
  | startTCStatement
  | stopTCStatement

;


configurationOps :   // no root node due to nondeterminism

    ( createOp ) =>   // Reason: LL(3) is insufficient
    createOp
  | selfOp
  | systemOp
  | mtcOp
  | runningOp

;


createOp :

  componentType Dot! CREATE!

  { ## = #( [CreateOp], ## ); }

;


systemOp! :

  start:SYSTEM!

  { ## = #[SystemOp]; SETLINECOLUMN( ##, start ); }

;


selfOp! :

  start:SELF!

  { ## = #[SelfOp]; SETLINECOLUMN( ##, start ); }

;


mtcOp! :

  start:MTC!

  { ## = #[MTCOp]; SETLINECOLUMN( ##, start ); }

;


doneStatement :

  componentId Dot! DONE!

  { ## = #( [DoneStatement], ## ); }

;


componentId :

    componentIdentifier
  | (   start1:ANY! { ## = #[AnyComponent]; SETLINECOLUMN( ##, start1 ); }
      | start2:ALL! { ## = #[AllComponent]; SETLINECOLUMN( ##, start2 ); } )
    COMPONENT!

;


runningOp :   // no specific root node due to nondeterminism

  componentId Dot! RUNNING!

  { ## = #( [SomeRunningOp], ## ); }

;


connectStatement :

  start:CONNECT! portSpec

  { ## = #( [ConnectStatement], ## ); SETLINECOLUMN( ##, start ); }

;


portSpec :   // root node already added in 'connectStatement'/'disconnectStatement'/
             // 'mapStatement'/'unmapStatement'
  "("! portRef ","! portRef ")"!

;


portRef :

  componentRef Colon! port

  { ## = #( [PortRef], ## ); }

;


componentRef :   // no root node due to nondeterminism

    componentIdentifier
  | systemOp
  | selfOp
  | mtcOp

;


disconnectStatement :

  start:DISCONNECT! portSpec

  { ## = #( [DisconnectStatement], ## ); SETLINECOLUMN( ##, start ); }

;


mapStatement :

  start:MAP! portSpec

  { ## = #( [MapStatement], ## ); SETLINECOLUMN( ##, start ); }

;


unmapStatement :

  start:UNMAP! portSpec

  { ## = #( [UnmapStatement], ## ); SETLINECOLUMN( ##, start ); }

;


startTCStatement :

  componentIdentifier Dot! START! "("! functionInstance ")"!

  { ## = #( [StartTCStatement], ## ); }

;


stopTCStatement! :

  start:STOP!

  { ## = #[StopTCStatement]; SETLINECOLUMN( ##, start ); }

;


componentIdentifier :   // no root node due to nondeterminism

    ( functionInstance ) =>   // Reason: LL(3) is insufficient
    functionInstance
  | variableRef

;


/*****  SECTION A.1.6.3.4 - Port Operations  *****/

port :   // no root node due to nondeterminism

  /*port,portPar*/ Identifier ( arrayOrBitRef )?

;


communicationStatements :   // no root node due to nondeterminism

    ( port Dot SEND  ) =>           // Reason: LL(3) is insufficient
    sendStatement
  | ( port Dot CALL ) =>            // Reason: LL(3) is insufficient
    callStatement
  | ( port Dot REPLY ) =>           // Reason: LL(3) is insufficient
    replyStatement
  | ( port Dot RAISE ) =>           // Reason: LL(3) is insufficient
    raiseStatement
  | ( portOrAny Dot RECEIVE ) =>    // Reason: LL(3) is insufficient
    receiveStatement
  | ( portOrAny Dot TRIGGER ) =>    // Reason: LL(3) is insufficient
    triggerStatement
  | ( portOrAny Dot GETCALL ) =>    // Reason: LL(3) is insufficient
    getCallStatement
  | ( portOrAny Dot GETREPLY ) =>   // Reason: LL(3) is insufficient
    getReplyStatement
  | ( portOrAny Dot CATCH ) =>      // Reason: LL(3) is insufficient
    catchStatement
  | ( portOrAny Dot CHECK ) =>      // Reason: LL(3) is insufficient
    checkStatement
  | ( portOrAll Dot CLEAR ) =>      // Reason: LL(3) is insufficient
    clearStatement
  | ( portOrAll Dot START ) =>      // Reason: LL(3) is insufficient
    startStatement
  | stopStatement

;


sendStatement :

  port Dot! portSendOp

  { ## = #( [SendStatement], ## ); }

;


portSendOp :

  start:SEND! "("! sendParameter ")"! ( toClause )?

;


sendParameter :

  templateInstance

;


toClause :

  start:TO! addressRef

  { ## = #( [ToClause], ## ); SETLINECOLUMN( ##, start ); }

;


addressRef :

    ( functionInstance ) =>   // Reason: LL(3) is insufficient
    functionInstance
  | variableRef

;


callStatement :

  port Dot! portCallOp ( portCallBody )?

  { ## = #( [CallStatement], ## ); }

;


portCallOp :

  start:CALL! "("! callParameters ")"! ( toClause )?

;


callParameters :

  templateInstance ( ","! callTimerValue )?

  { ## = #( [CallParameters], ## ); }

;


callTimerValue :

    timerValue
  | start:NOWAIT!   { ## = #[Nowait]; SETLINECOLUMN( ##, start ); }

;


portCallBody :

  start:BeginChar! callBodyStatementList EndChar!

  { ## = #( [PortCallBody], ## ); SETLINECOLUMN( ##, start ); }

;


callBodyStatementList :   // root node already added in 'portCallBody'

  ( callBodyStatement ( SemiColon! | { optionalSemicolon() }? ) )+

;


callBodyStatement :

  callBodyGuard statementBlock

  { ## = #( [CallBodyStatement], ## ); }

;


callBodyGuard :

  altGuardChar callBodyOps

  { ## = #( [CallBodyGuard], ## ); }

;


callBodyOps :

    ( portOrAny Dot GETREPLY ) =>   // Reason: LL(3) is insufficient
    getReplyStatement
  | catchStatement

;


replyStatement :

  port Dot! portReplyOp

  { ## = #( [ReplyStatement], ## ); }

;


portReplyOp :

  start:REPLY! "("! templateInstance ( replyValue )? ")"! ( toClause )?

;


replyValue :

  start:VALUE! expression

  { ## = #( [ReplyValue], ## ); SETLINECOLUMN( ##, start ); }

;


raiseStatement :

  port Dot! portRaiseOp

  { ## = #( [RaiseStatement], ## ); }

;


portRaiseOp :

  start:RAISE! "("! signature ","! templateInstance ")"! ( toClause )?

;


receiveStatement :

  portOrAny Dot! portReceiveOp

  { ## = #( [ReceiveStatement], ## ); }

;


portOrAny :

    port
  | start:ANY! PORT!   { ## = #[AnyPort]; SETLINECOLUMN( ##, start ); }

;


portReceiveOp :

  start:RECEIVE! ( "("! receiveParameter ")"! )? ( fromClause )? ( portRedirect )?

  { ## = #( [PortReceiveOp], ## ); SETLINECOLUMN( ##, start ); }

;


receiveParameter :

  templateInstance

;


fromClause :

  start:FROM! addressRef

  { ## = #( [FromClause], ## ); SETLINECOLUMN( ##, start ); }

;


portRedirect :

  start:PortRedirectSymbol!
  (   valueSpec ( senderSpec )?
    | senderSpec )

  { ## = #( [PortRedirect], ## ); SETLINECOLUMN( ##, start ); }

;


valueSpec :

  start:VALUE! variableRef

  { ## = #( [ValueSpec], ## ); SETLINECOLUMN( ##, start ); }

;


senderSpec :

  start:SENDER! variableRef

  { ## = #( [SenderSpec], ## ); SETLINECOLUMN( ##, start ); }

;


triggerStatement :

  portOrAny Dot! portTriggerOp

  { ## = #( [TriggerStatement], ## ); }

;


portTriggerOp :

  start:TRIGGER! ( "("! receiveParameter ")"! )? ( fromClause )? ( portRedirect )?

;


getCallStatement :

  portOrAny Dot! portGetCallOp

  { ## = #( [GetCallStatement], ## ); }

;


portGetCallOp :

  start:GETCALL! ( "("! receiveParameter ")"! )? ( fromClause )? ( portRedirectWithParam )?

  { ## = #( [PortGetCallOp], ## ); SETLINECOLUMN( ##, start ); }

;


portRedirectWithParam :

  start:PortRedirectSymbol! redirectSpec

  { ## = #( [PortRedirectWithParam], ## ); SETLINECOLUMN( ##, start ); }

;


redirectSpec :

  (   valueSpec ( paraSpec )? ( senderSpec )?
    | paraSpec ( senderSpec )?
    | senderSpec )

  { ## = #( [RedirectSpec], ## ); }

;


paraSpec :

  start:PARAM! paraAssignmentList

  { ## = #( [ParaSpec], ## ); SETLINECOLUMN( ##, start ); }

;


paraAssignmentList :

  start:"("!
  (   ( variableRef AssignmentChar ) =>   // Reason: LL(3) is insufficient
      assignmentList
    | variableList )
  ")"!

  { ## = #( [ParaAssignmentList], ## ); SETLINECOLUMN( ##, start ); }

;


assignmentList :

  variableAssignment ( ","! variableAssignment )*

  { ## = #( [AssignmentList], ## ); }

;


variableAssignment :

  variableRef AssignmentChar! parameterIdentifier

  { ## = #( [VariableAssignment], ## ); }

;


parameterIdentifier :

  /*valuePar,timerPar,templatePar,portPar*/ Identifier

;


variableList :

  variableEntry ( ","! variableEntry )*

  { ## = #( [VariableList], ## ); }

;


variableEntry :

    variableRef
  | notUsedSymbol

;



getReplyStatement :

  portOrAny Dot! portGetReplyOp

  { ## = #( [GetReplyStatement], ## ); }

;


portGetReplyOp :

  start:GETREPLY! ( "("! receiveParameter ( valueMatchSpec )? ")"! )?
  ( fromClause )? ( portRedirectWithParam )?

  { ## = #( [PortGetReplyOp], ## ); SETLINECOLUMN( ##, start ); }

;


valueMatchSpec :

  start:VALUE! templateInstance

  { ## = #( [ValueMatchSpec], ## ); SETLINECOLUMN( ##, start ); }

;


checkStatement :

  portOrAny Dot! portCheckOp

  { ## = #( [CheckStatement], ## ); }

;


portCheckOp :

  start:CHECK! ( "("! checkParameter ")"! )?

;


checkParameter :

    portReceiveOp
  | portGetCallOp
  | portGetReplyOp
  | portCatchOp
  | ( fromClause )? ( PortRedirectSymbol! senderSpec )?
    { ## = #( [CheckParameter], ## ); }

;


catchStatement :

  portOrAny Dot! portCatchOp

  { ## = #( [CatchStatement], ## ); }

;


portCatchOp :

  start:CATCH! ( "("! catchOpParameter ")"! )? ( fromClause )? ( portRedirect )?

  { ## = #( [PortCatchOp], ## ); SETLINECOLUMN( ##, start ); }

;


catchOpParameter :

    signature ","! templateInstance

    { ## = #( [CatchOpParameter], ## ); }

  | start:TIMEOUT!

    { ## = #[Timeout]; SETLINECOLUMN( ##, start ); }

;


clearStatement :

  portOrAll Dot! CLEAR!

  { ## = #( [ClearStatement], ## ); }

;


portOrAll :   // no root node due to nondeterminism

    port
  | start:ALL! PORT!   { ## = #[AllPort]; SETLINECOLUMN( ##, start ); }

;


startStatement :   // no specific root node due to nondeterminism

  portOrAll Dot! START!

  { ## = #( [SomeStartStatement], ## ); }

;


stopStatement :   // no specific root node due to nondeterminism

  portOrAll Dot! STOP!

  { ## = #( [SomeStopStatement], ## ); }

;


/*****  SECTION A.1.6.3.5 - Timer Operations  *****/

timerStatements :   // no root node due to nondeterminism

    ( timerRef Dot START ) =>
    startTimerStatement
  | ( timerRef Dot STOP | ALL ) =>
    stopTimerStatement
  | timeoutStatement

;


timerOps :   // no root node due to nondeterminism

    ( readTimerOp ) =>   // Reason: LL(3) is insufficient
    readTimerOp
  | runningTimerOp

;


startTimerStatement :   // no specific root node due to nondeterminism

  timerRef Dot! START! ( "("! timerValue ")"! )?

  { ## = #( [SomeStartStatement], ## ); }

;


stopTimerStatement :   // no specific root node due to nondeterminism

  timerRefOrAll Dot! STOP!

  { ## = #( [SomeStopStatement], ## ); }

;


timerRefOrAll :

    timerRef
  | start:ALL! TIMER!   { ## = #[AllTimer]; SETLINECOLUMN( ##, start ); }

;


readTimerOp :

  timerRef Dot! READ!

  { ## = #( [ReadTimerOp], ## ); }

;


runningTimerOp :   // no specific root node due to nondeterminism

  timerRefOrAny Dot! RUNNING!

  { ## = #( [SomeRunningOp], ## ); }

;


timeoutStatement :

  timerRefOrAny Dot! TIMEOUT!

  { ## = #( [TimeoutStatement], ## ); }

;


timerRefOrAny :

    timerRef
  | start:ANY! TIMER!   { ## = #[AnyTimer]; SETLINECOLUMN( ##, start ); }

;


/*****  SECTION A.1.6.4 - Type  *****/

type :   // no root node due to nondeterminism

    predefinedType
  | referencedType

;


predefinedType! :

    start01:BITSTRING!               { ## = #[TypeBitstring]; SETLINECOLUMN( ##, start01 );}
  | start02:BOOLEAN!                 { ## = #[TypeBoolean]; SETLINECOLUMN( ##, start02 ); }
  | start03:CHARSTRING!              { ## = #[TypeCharstring]; SETLINECOLUMN( ##, start03 ); }
  | start04:UNIVERSAL! CHARSTRING!   { ## = #[TypeUniversalCharstring]; SETLINECOLUMN( ##, start04 ); }
  | start05:CHAR!                    { ## = #[TypeChar]; SETLINECOLUMN( ##, start05 ); }
  | start06:UNIVERSAL! CHAR!         { ## = #[TypeUniversalChar]; SETLINECOLUMN( ##, start06 ); }
  | start07:INTEGER!                 { ## = #[TypeInteger]; SETLINECOLUMN( ##, start07 ); }
  | start08:OCTETSTRING!             { ## = #[TypeOctetstring]; SETLINECOLUMN( ##, start08 ); }
  | start09:OBJID!                   { ## = #[TypeObjId]; SETLINECOLUMN( ##, start09 ); }
  | start10:HEXSTRING!               { ## = #[TypeHexstring]; SETLINECOLUMN( ##, start10 ); }
  | start11:VERDICTTYPE!             { ## = #[TypeVerdictType]; SETLINECOLUMN( ##, start11 ); }
  | start12:FLOAT!                   { ## = #[TypeFloat]; SETLINECOLUMN( ##, start12 ); }
  | start13:ADDRESS!                 { ## = #[TypeAddress]; SETLINECOLUMN( ##, start13 ); }
  | start14:DEFAULT!                 { ## = #[TypeDefault]; SETLINECOLUMN( ##, start14 ); }

;


referencedType :   // no root node due to nondeterminism

  (
    ( globalModuleId Dot typeReference ) =>   // Reason: Grammar induced nondeterminism
    globalModuleId Dot!
  )?
  typeReference ( extendedFieldReference )?

;


typeReference :   // no root node due to nondeterminism

    /*structType*/ Identifier ( typeActualParList )?
  // | /*enumType,subType,typePar,componentType*/ Identifier   // subsumed by first alternative

;


typeActualParList :

  start:"("! typeActualPar ( ","! typeActualPar )* ")"!

  { ## = #( [TypeActualParList], ## ); SETLINECOLUMN( ##, start ); }

;


typeActualPar :

    ( singleConstExpression ) =>   // Reason: Grammar induced nondeterminism
    singleConstExpression
  | type

;


/*****  SECTION A.1.6.4.1 - Array Types  *****/

arrayDef :   // Attention: Rule 'arrayDef' had to be split in order to construct a nonambiguous AST

  ( arrayDef2 )+

  { ## = #( [ArrayDefList], ## ); }

;


arrayDef2 :   // Attention: New rule used by 'arrayDef'

  start:"["! arrayBounds ( ".."! arrayBounds )? "]"!

  { ## = #( [ArrayDef], ## ); SETLINECOLUMN( ##, start ); }

;


arrayBounds :

  singleConstExpression

;


/*****  SECTION A.1.6.5 - Value  *****/

value :   // no root node due to nondeterminism

    predefinedValue
  | referencedValue

;


predefinedValue :   // no root node due to nondeterminism

    bitStringValue
  | booleanValue
  | charStringValue
  | integerValue
  | octetStringValue
  | objectIdentifierValue
  | hexstringValue
  | verdictTypeValue
  // | enumeratedValue   // subsumed by 'referencedValue' in 'value'
  | FloatValue
  | addressValue

;


bitStringValue :

  BString

;


booleanValue! :

    start1:TRUE!    { ## = #[True]; SETLINECOLUMN( ##, start1 ); }
  | start2:FALSE!   { ## = #[False]; SETLINECOLUMN( ##, start2 ); }

;


integerValue :

  Number

;


octetStringValue :

  OString

;


objectIdentifierValue :

  start:OBJID! "{"! objIdComponentList "}"

  { ## = #( [ObjectIdentifierValue], ## ); SETLINECOLUMN( ##, start ); }

;


objIdComponentList :   // no root node due to nondeterminism

  ( objIdComponent )+

;


objIdComponent :   // no root node due to nondeterminism

  // nameForm  // subsumed by numberForm.referencedValue
  numberForm
  // | nameAndNumberForm  // subsumed by numberForm.referencedValue + numberForm

;


numberForm :   // no root node due to nondeterminism

    Number
  | referencedValue

;


nameAndNumberForm :

  Identifier numberForm

  { ## = #( [NameAndNumberForm], ## ); }

;


nameForm :

  Identifier

;


hexstringValue :

  HString

;


verdictTypeValue! :

    start1:PASS!     { ## = #[Pass];   SETLINECOLUMN( ##, start1 ); }
  | start2:FAIL!     { ## = #[Fail];   SETLINECOLUMN( ##, start2 ); }
  | start3:INCONC!   { ## = #[Inconc]; SETLINECOLUMN( ##, start3 ); }
  | start4:NONE!     { ## = #[None];   SETLINECOLUMN( ##, start4 ); }
  | start5:ERROR!    { ## = #[Error];  SETLINECOLUMN( ##, start5 ); }

;


enumeratedValue :   // no root node due to nondeterminism

  /*namedValue*/ Identifier

;


charStringValue :

    CString
  | quadruple

;


quadruple :

  start:"("! /*group*/ Number ","! /*plane*/ Number ","! /*row*/ Number ","! /*cell*/ Number ")"!

  { ## = #( [Quadruple], ## ); SETLINECOLUMN( ##, start ); }

;


referencedValue :   // no root node due to nondeterminism

  valueReference ( extendedFieldReference )?

;


valueReference :   // no root node due to nondeterminism

    ( ( globalModuleId Dot Identifier ) =>   // Reason: Grammar induced nondeterminism
      globalModuleId Dot!
    )?
    /*const*/ Identifier

  // | /*extConst,valuePar,modulePar,var*/ Identifier    // subsumed by first alternative

;


freeText :

  CString   // Attention: Free text is restricted to a common character string!

;


addressValue! :

  start:NULL_!   { ## = #[Null]; SETLINECOLUMN( ##, start ); }

;


/*****  SECTION A.1.6.6 - Parameterisation  *****/

formalValuePar { bool in = false; } :   // no specific root node due to nondeterminism

  (   start1:IN!    { in = true; }
    | start2:INOUT! { ADDCHILD(#[InOut],start2); }
    | start3:OUT!   { ADDCHILD(#[Out],start3); } )?
  type /*valuePar*/ Identifier

  { ## = #( [SomeFormalPar], ## ); if ( in ) SETLINECOLUMN( ##, start1 ); }

;


formalTypePar { bool in = false; } :

  ( start:IN! { in = true; } )? /*typePar*/ Identifier

  { ## = #( [FormalTypePar], ## ); if ( in ) SETLINECOLUMN( ##, start ); }

;


formalPortPar { bool inout = false; } :   // no specific root node due to nondeterminism

  ( start:INOUT! { inout = true; } )?
  /*portType*/ Identifier /*portPar*/ Identifier

  { ## = #( [SomeFormalPar], ## ); if ( inout ) SETLINECOLUMN( ##, start ); }

;


formalTimerPar { bool inout = false; } :

  ( start1:INOUT! { inout = true; } )? start2:TIMER! /*timerPar*/ Identifier

  { ## = #( [FormalTimerPar], ## );
    if ( inout ) { SETLINECOLUMN( ##, start1 ); } else { SETLINECOLUMN( ##, start2 ); } }

;


formalTemplatePar { bool in = false; } :

  ( start1:IN! { in = true; } )? start2:TEMPLATE!
  type /*templatePar*/ Identifier

  { ## = #( [FormalTemplatePar], ## );
    if ( in ) { SETLINECOLUMN( ##, start1 ); } else { SETLINECOLUMN( ##, start2 ); } }

;


/*****  SECTION A.1.6.7 - With Statement  *****/

withStatement :

  start:WITH! withAttribList

  { ## = #( [WithStatement], ## ); SETLINECOLUMN( ##, start ); }

;


withAttribList :   // no root node as the rule itself does not provide additional information

  BeginChar! multiWithAttrib EndChar!

;


multiWithAttrib :   // no root node as the rule itself does not provide additional information

  ( singleWithAttrib ( SemiColon! | { optionalSemicolon() }? ) )+

;


singleWithAttrib :

  attribKeyword ( ch:OVERRIDE! { ADDCHILD(#[Override],ch); } )? ( attribQualifier )? attribSpec

  { ## = #( [SingleWithAttrib], ## ); }

;


attribKeyword :

    start1:ENCODE!      { ## = #[Encode];    SETLINECOLUMN( ##, start1 ); }
  | start2:DISPLAY!     { ## = #[Display];   SETLINECOLUMN( ##, start2 ); }
  | start3:EXTENSION!   { ## = #[Extension]; SETLINECOLUMN( ##, start3 ); }

;


attribQualifier :

  start:"("! defOrFieldRefList ")"!

  { ## = #( [AttribQualifier], ## ); SETLINECOLUMN( ##, start ); }

;


defOrFieldRefList :   // root node already added in 'attribQualifier'

  defOrFieldRef ( ","! defOrFieldRef )*

;



defOrFieldRef :

  // definitionRef     // subsumed by 'fieldReference'
    fieldReference

;


definitionRef :   // no root node due to nondeterminism

  /*structType,enumType,portType,componentType,subType,
    const,template,teststep,testcase,function,signature*/ Identifier

;


attribSpec :

  freeText

;


/*****  SECTION A.1.6.8 - Behaviour Statements  *****/

behaviourStatements :

    testcaseInstance
  | functionInstance
  | returnStatement
  | altConstruct
  | interleavedConstruct
  | labelStatement
  | gotoStatement
  | repeatStatement
  | deactivateStatement
  // | teststepInstance    // subsumed by 'functionInstance'

;


verdictStatements :

  setLocalVerdict

;


verdictOps :

  getLocalVerdict

;


setLocalVerdict :

  start:setVerdictKeyword! "("! singleExpression ")"!

  { ## = #( [SetLocalVerdict], ## ); SETLINECOLUMN( ##, antlr::RefInfoAST( #start ) ); }

;


setVerdictKeyword! :   // root node just needed for tracking line/column information

  start:VERDICT! Dot! SET!

  { ## = #[DummyToken]; SETLINECOLUMN( ##, start ); }

;


getLocalVerdict :

  start:VERDICT! Dot! GET!

  { ## = #( [GetLocalVerdict], ## ); SETLINECOLUMN( ##, start ); }

;


sutStatements :

  start:sutAction! "("! ( freeText | templateRefWithParList ) ")"!

  { ## = #( [SUTStatements], ## ); SETLINECOLUMN( ##, antlr::RefInfoAST( #start ) ); }

;


sutAction! :   // root node just needed for tracking line/column information

  start:SUT! Dot! ACTION!

  { ## = #[DummyToken]; SETLINECOLUMN( ##, start ); }

;


returnStatement :

  start:RETURN! ( options { greedy = true; } : expression )?

  { ## = #( [ReturnStatement], ## ); SETLINECOLUMN( ##, start ); }

;


altConstruct :

  start:ALT! BeginChar! altGuardList EndChar!

  { ## = #( [AltConstruct], ## ); SETLINECOLUMN( ##, start ); }

;


altGuardList :   // root node already added in 'altConstruct' and 'teststepDef'

  altGuardList_loop ( elseStatement ( SemiColon! | { optionalSemicolon() }? ) )?

;


altGuardList_loop :

  guardStatement ( SemiColon! | { optionalSemicolon() }? )
  (   ( "[" ~ELSE ) =>   // Reason: LL(3)-approx is insufficient
      altGuardList_loop )?

;


guardStatement :

  altGuardChar
  (   ( guardOp ) =>
      guardOp statementBlock
    | teststepInstance )

  { ## = #( [GuardStatement], ## ); }

;


elseStatement :

  start:"["! ELSE! "]"! statementBlock

  { ## = #( [ElseStatement], ## ); SETLINECOLUMN( ##, start ); }

;


altGuardChar :

  start:"["! ( booleanExpression )? "]"!

  { ## = #( [AltGuardChar], ## ); SETLINECOLUMN( ##, start ); }

;


guardOp :

    ( timerRef Dot TIMEOUT | ANY TIMER ) =>   // Reason: LL(3) is insufficient
    timeoutStatement
  | ( portOrAny Dot RECEIVE ) =>              // Reason: LL(3) is insufficient
    { checkModuleControlPart( "'receive' statements" ); }
    receiveStatement
  | ( portOrAny Dot TRIGGER ) =>              // Reason: LL(3) is insufficient
    { checkModuleControlPart( "'trigger' statements" ); }
    triggerStatement
  | ( portOrAny Dot GETCALL ) =>              // Reason: LL(3) is insufficient
    { checkModuleControlPart( "'getcall' statements" ); }
    getCallStatement
  | ( portOrAny Dot CATCH ) =>                // Reason: LL(3) is insufficient
    { checkModuleControlPart( "'catch' statements" ); }
    catchStatement
  | ( portOrAny Dot CHECK ) =>                // Reason: LL(3) is insufficient
    { checkModuleControlPart( "'check' statements" ); }
    checkStatement
  | ( portOrAny Dot GETREPLY ) =>             // Reason: LL(3) is insufficient
    { checkModuleControlPart( "'getreply' statements" ); }
    getReplyStatement
  | { checkModuleControlPart( "'done' statements" ); }
    doneStatement

;


statementBlock :

  start:BeginChar! ( functionStatementOrDefList )? EndChar!

  { ## = #( [StatementBlock], ## ); SETLINECOLUMN( ##, start ); }

;


interleavedConstruct :

  start:INTERLEAVE! BeginChar! interleavedGuardList EndChar!

  { ## = #( [InterleavedConstruct], ## ); SETLINECOLUMN( ##, start ); }

;


interleavedGuardList :   // root node already added in 'interleavedConstruct'

  ( interleavedGuardElement ( SemiColon! | { optionalSemicolon() }? ) )+

;


interleavedGuardElement :

  interleavedGuard interleavedAction

  { ## = #( [InterleavedGuardElement], ## ); }

;



interleavedGuard :

  start:"["! "]"! guardOp

  { ## = #( [InterleavedGuard], ## ); SETLINECOLUMN( ##, start ); }

;


interleavedAction :

  statementBlock

;


labelStatement :

  start:LABEL! /*label*/ Identifier

  { ## = #( [Label], ## ); SETLINECOLUMN( ##, start ); }

;


gotoStatement :

  start:GOTO! /*label*/ Identifier

  { ## = #( [GotoStatement], ## ); SETLINECOLUMN( ##, start ); }

;


repeatStatement! :

  start:REPEAT!

  { ## = #[RepeatStatement]; SETLINECOLUMN( ##, start ); }

;


activateOp :

  start:ACTIVATE! "("! teststepInstance ")"!

  { ## = #( [ActivateOp], ## ); SETLINECOLUMN( ##, start ); }

;


deactivateStatement :

  start:DEACTIVATE! ( "("! expression ")"! )?

  { ## = #( [DeactivateStatement], ## ); SETLINECOLUMN( ##, start ); }

;


/*****  SECTION A.1.6.9 - Basic Statements  *****/

basicStatements :

    assignment
  | logStatement
  | loopConstruct
  | conditionalConstruct

;


expression :

    singleExpression
  | compoundExpression

;


compoundExpression :

    ( BeginChar fieldReference AssignmentChar ) =>   // Reason: LL(3) is insufficient
    fieldExpressionList
  | arrayExpression

;


fieldExpressionList :

  start:BeginChar! fieldExpressionSpec ( ","! fieldExpressionSpec )* EndChar!

  { ## = #( [FieldExpressionList], ## ); SETLINECOLUMN( ##, start ); }

;


fieldExpressionSpec :

  fieldReference AssignmentChar! expression

  { ## = #( [FieldExpressionSpec], ## ); }

;


arrayExpression :

  start:BeginChar! ( arrayElementExpressionList )? EndChar!

  { ## = #( [ArrayExpression], ## ); SETLINECOLUMN( ##, start ); }

;



arrayElementExpressionList :   // root node already added in 'arrayExpression'

  notUsedOrExpression ( ","! notUsedOrExpression )*

;


notUsedOrExpression :

    ( Dash ( EndChar | Comma ) ) =>   // Reason: LL(3)-approx is insufficient
    notUsedSymbol
  | expression

;


constantExpression :   // no root node due to nondeterminism

    singleConstExpression
  | compoundConstExpression

;


singleConstExpression :   // no root node due to nondeterminism

  singleExpression

;


booleanExpression :

  singleExpression

;


compoundConstExpression :

    ( BeginChar fieldReference AssignmentChar ) =>
    fieldConstExpressionList
  | arrayConstExpression

;


fieldConstExpressionList :

  start:BeginChar! fieldConstExpressionSpec ( ","! fieldConstExpressionSpec )* EndChar!

  { ## = #( [FieldConstExpressionList], ## ); SETLINECOLUMN( ##, start ); }

;


fieldConstExpressionSpec :

  fieldReference AssignmentChar! constantExpression

  { ## = #( [FieldConstExpressionSpec], ## ); }

;


arrayConstExpression :

  start:BeginChar! ( arrayElementConstExpressionList )? EndChar!

  { ## = #( [ArrayConstExpression], ## ); SETLINECOLUMN( ##, start ); }

;


arrayElementConstExpressionList :   // root node already added in 'arrayConstExpression'

  constantExpression ( ","! constantExpression )*

;


assignment :

  variableRef AssignmentChar! expression

  { ## = #( [Assignment], ## ); }

;


singleExpression :

  simpleExpression
  ( (   start1:AND4B!         { ## = #( [OpBitwiseAnd], ## );  SETLINECOLUMN( ##, start1 ); }
      | start2:XOR4B!         { ## = #( [OpBitwiseXor], ## );  SETLINECOLUMN( ##, start2 ); }
      | start3:OR4B!          { ## = #( [OpBitwiseOr], ## );   SETLINECOLUMN( ##, start3 ); }
      | start4:AND!           { ## = #( [OpLogicalAnd], ## );  SETLINECOLUMN( ##, start4 ); }
      | start5:XOR!           { ## = #( [OpLogicalXor], ## );  SETLINECOLUMN( ##, start5 ); }
      | start6:OR!            { ## = #( [OpLogicalOr], ## );   SETLINECOLUMN( ##, start6 ); }
      | start7:Ampersand!     { ## = #( [OpConcatenate], ## ); SETLINECOLUMN( ##, start7 ); } )
    simpleExpression )*

;


simpleExpression :

  subExpression
  ( (   start1:"=="!   { ## = #( [OpEqual], ## );          SETLINECOLUMN( ##, start1 ); }
      | start2:"<"!    { ## = #( [OpLess], ## );           SETLINECOLUMN( ##, start2 ); }
      | start3:">"!    { ## = #( [OpGreater], ## );        SETLINECOLUMN( ##, start3 ); }
      | start4:"!="!   { ## = #( [OpNotEqual], ## );       SETLINECOLUMN( ##, start4 ); }
      | start5:">="!   { ## = #( [OpGreaterOrEqual], ## ); SETLINECOLUMN( ##, start5 ); }
      | start6:"<="!   { ## = #( [OpLessOrEqual], ## );    SETLINECOLUMN( ##, start6 ); } )
    subExpression )?

;


subExpression :

  product
  ( (   start1:ShiftLeft!     { ## = #( [OpShiftLeft], ## );   SETLINECOLUMN( ##, start1 ); }
      | start2:ShiftRight!    { ## = #( [OpShiftRight], ## );  SETLINECOLUMN( ##, start2 ); }
      | start3:RotateLeft!    { ## = #( [OpRotateLeft], ## );  SETLINECOLUMN( ##, start3 ); }
      | start4:RotateRight!   { ## = #( [OpRotateRight], ## ); SETLINECOLUMN( ##, start4 ); } )
    product )?

;


product :

  term
  ( options { greedy = true; } :   // necessary due to optional semicolons
    (   start1:"+"!   { ## = #( [OpBinaryPlus], ## );  SETLINECOLUMN( ##, start1 ); }
      | start2:"-"!   { ## = #( [OpBinaryMinus], ## ); SETLINECOLUMN( ##, start2 ); } )
    term )*

;


term :

  factor
  ( (   start1:"*"!   { ## = #( [OpTimes], ## );  SETLINECOLUMN( ##, start1 ); }
      | start2:"/"!   { ## = #( [OpDivide], ## ); SETLINECOLUMN( ##, start2 ); }
      | start3:MOD!   { ## = #( [OpMod], ## );    SETLINECOLUMN( ##, start3 ); }
      | start4:REM!   { ## = #( [OpRem], ## );    SETLINECOLUMN( ##, start4 ); } )
    factor )*

;


factor :

  (   start1:"+"!   { MAKEROOT( #[OpUnaryPlus],  start1 ); }
    | start2:"-"!   { MAKEROOT( #[OpUnaryMinus], start2 ); }
    | start3:NOT!   { MAKEROOT( #[OpLogicalNot], start3 ); }
    | start4:NOT4B! { MAKEROOT( #[OpBitwiseNot], start4 ); } )?
  primary

;


primary :   // no root node due to nondeterminism

    ( opCall ) =>
    opCall
  | ( ~LParenthesis | quadruple ) =>
    value
  | "("! singleExpression ")"!

;


extendedFieldReference :   // no root node due to nondeterminism

  // Originally: ( Dot! /*structField*/ Identifier | arrayOrBitRef )+

  (   Dot! /*structField*/ Identifier
    | arrayOrBitRef )
  (   ( Dot Identifier | "[" ) =>   // Reason: LL(3)-approx is insufficient
      extendedFieldReference
    | /* empty alternative */ )

;


opCall :   // no root node due to nondeterminism

    ( configurationOps ) =>   // Reason: LL(k)-approx is insufficient
    { checkModuleControlPart( "configuration operations" ); }
    configurationOps
  | ( timerOps ) =>           // Reason: LL(k)-approx is insufficient
    timerOps
  | { checkModuleControlPart( "verdict operations" ); }
    verdictOps
  | testcaseInstance
  | functionInstance
  | templateOps
  | { checkModuleControlPart( "activate operation" ); }
    activateOp

;


logStatement :

  start:LOG! "("! ( freeText )? ")"!

  { ## = #( [LogStatement], ## ); SETLINECOLUMN( ##, start ); }

;


loopConstruct :

    forStatement
  | whileStatement
  | doWhileStatement

;


forStatement :

  start:FOR! "("! initial ( SemiColon! | { optionalSemicolon() }? )
                  final   ( SemiColon! | { optionalSemicolon() }? )
                  step ")"! statementBlock

  { ## = #( [ForStatement], ## ); SETLINECOLUMN( ##, start ); }

;


initial :

  (   varInstance
    | assignment )

;


final :

  booleanExpression

;


step :

  assignment

;


whileStatement :

  start:WHILE! "("! booleanExpression ")"! statementBlock

  { ## = #( [WhileStatement], ## ); SETLINECOLUMN( ##, start ); }

;


doWhileStatement :

  start:DO! statementBlock WHILE! "("! booleanExpression ")"!

  { ## = #( [DoWhileStatement], ## ); SETLINECOLUMN( ##, start ); }

;


conditionalConstruct :

  start:IF! "("! booleanExpression ")"!
  statementBlock ( options { greedy = true; } : elseIfClause )* ( elseClause )?

  { ## = #( [ConditionalConstruct], ## ); SETLINECOLUMN( ##, start ); }

;


elseIfClause :

  start:ELSE! IF! "("! booleanExpression ")"! statementBlock

  { ## = #( [ElseIfClause], ## ); SETLINECOLUMN( ##, start ); }

;


elseClause :

  start:ELSE! statementBlock

  { ## = #( [ElseClause], ## ); SETLINECOLUMN( ##, start ); }

;
