/* machine.c -- draw the Machine Settings dialog of TuxTime
 *
 * Copyright (c) 1996-98  Jonathan A. Buzzard (jonathan@buzzard.org.uk)
 *
 * $Log: machine.c,v $
 * Revision 1.4  2001/02/04 22:40:20  jab
 * fixed the option menu bug on the battery options page
 *
 * Revision 1.3  2000/02/11 00:00:23  jab
 * fixed ambiguously nested if-elses
 * super bright LCD settings fixed courtesy of Tom May <tom@you-bastards.com>
 *
 * Revision 1.2  1999/12/04 13:43:21  jab
 * changed for the new tuxtime-conf program
 *
 * Revision 1.1  1999/04/09 17:59:08  jab
 * Initial revision
 *
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software 
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 */

static const char rcsid[]="$Id: machine.c,v 1.4 2001/02/04 22:40:20 jab Exp jab $";


#include<stdio.h>
#include<stdlib.h>
#include<paths.h>
#include<gtk/gtk.h>

#include"sci.h"
#include"misc.h"
#include"config.h"
#include"machine.h"

#include"batterycharged.xpm"
#include"plus.xpm"
#include"minus.xpm"
#include"plusdisabled.xpm"
#include"minusdisabled.xpm"


/*
 * Structure to hold the battery options settings
 */
struct BatterySettings {
	unsigned short save;
	unsigned short hdd;
	unsigned short display;
	unsigned short sleep;
	unsigned short speed;
	unsigned short lcd;
	unsigned short autooff;
	unsigned short cooling;
	unsigned short standby;
};

/*
 * Structure to hold all other settings for the machine settings dialog
 */
struct MachineSettings {
	unsigned short volume;
	unsigned short panelalarm;
	unsigned short lowalarm;
	unsigned short beep;
	unsigned short powerup;
	unsigned short panelpower;
	unsigned short alarm;
};


static struct BatterySettings external;
static struct BatterySettings internal;
static struct MachineSettings machine;


static int powermode,source;
static int hours,minutes,alarmformat;
static unsigned short alarmon;
static int lcdtype,savetype,cooltype;

GtkWidget *battery,*hdd,*display,*autooff;
GtkWidget *speed,*sleep,*lcd,*cooling,*standby;
GtkWidget *panel,*low,*power,*level,*beep;
GtkWidget *resume,*disable;
GtkWidget *hourplus,*hourminus,*minuteplus,*minuteminus;
GtkWidget *alarmtime,*alarmam,*alarmpm,*alarm24;
GtkObject *slider;

GdkPixmap *plus,*minus,*plusdisabled,*minusdisabled,*chargedpixmap;
GdkBitmap *plusmask,*minusmask,*chargedmask;

GtkWidget *pluspix,*minuspix,*plusdisabledpix,*minusdisabledpix;
GtkWidget *hourpluspix,*hourminuspix,*minutepluspix,*minuteminuspix;
GtkWidget *hourplusdisabledpix,*hourminusdisabledpix;
GtkWidget *minuteplusdisabledpix,*minuteminusdisabledpix;


/*
 * function declatrations for the option menu callbacks
 */
void BatteryCallback(GtkWidget *widget, int data);
void HDDCallback(GtkWidget *widget, int data);
void DisplayCallback(GtkWidget *widget, int data);
void AutoOffCallback(GtkWidget *widget, int data);
void SpeedCallback(GtkWidget *widget, int data);
void SleepCallback(GtkWidget *widget, int data);
void LCDCallback(GtkWidget *widget, int data);
void CoolingCallback(GtkWidget *widget, int data);
void StandbyCallback(GtkWidget *widget, int data);
void SetSystemOptions(void);


enum { STOCKHOLM, ATLANTA, MAUI, LIBRETTO };
enum { HIGHLOW, BRIGHT, SUPER, BACKLIGHT };

#define RESUME_DEPENDENCY \
"The System Auto Off feature is only functional\n\
when Resume Mode is enabled\n\
Resume mode is currently disabled\n\
\n\
Do you wish to enable Resume Mode?"


typedef void (*MenuItemCallback) (GtkWidget *widget, int data);

typedef struct
{
	char *label;
	MenuItemCallback callback;
	gpointer data;
	GtkWidget *widget;
} MenuItem;


/*
 * The structures holding the option menu items for the battery options page
 */
static MenuItem batterymaui[] =
{
	{"Full Power", BatteryCallback, (gpointer) SCI_FULL_POWER, NULL},
	{"Low Power", BatteryCallback, (gpointer) SCI_LOW_POWER, NULL},
	{"User Settings", BatteryCallback, (gpointer) SCI_USER_SETTINGS, NULL},
	{NULL, NULL, NULL, NULL}
};

static MenuItem batterystockholm[] =
{
	{"Full Power", BatteryCallback, (gpointer) SCI_FULL_LIFE, NULL},
	{"Economy", BatteryCallback, (gpointer) SCI_ECONOMY, NULL},
	{"Low Power", BatteryCallback, (gpointer) SCI_LOW_POWER, NULL},
	{"User Settings", BatteryCallback, (gpointer) SCI_USER_SETTINGS, NULL},
	{NULL, NULL, NULL, NULL}
};

static MenuItem batteryatlanta[] =
{
	{"Normal Life", BatteryCallback, (gpointer) SCI_NORMAL_LIFE, NULL},
	{"Long Life", BatteryCallback, (gpointer) SCI_LONG_LIFE, NULL},
	{"User Settings", BatteryCallback, (gpointer) SCI_USER_SETTINGS, NULL},
	{NULL, NULL, NULL, NULL}
};

static MenuItem batterylibretto[] =
{
	{"Full Power", BatteryCallback, (gpointer) SCI_FULL_POWER, NULL},
	{"High Power", BatteryCallback, (gpointer) SCI_LOW_POWER, NULL},
	{"Medium Power", BatteryCallback, (gpointer) SCI_LOW_POWER, NULL},
	{"Low Power", BatteryCallback, (gpointer) SCI_LOW_POWER, NULL},
	{"User Settings", BatteryCallback, (gpointer) SCI_USER_SETTINGS, NULL},
	{NULL, NULL, NULL, NULL}
};


static MenuItem hddmenu[] =
{
	{"Disabled", HDDCallback, (gpointer) SCI_TIME_DISABLED, NULL},
	{"1 Minute", HDDCallback, (gpointer) SCI_TIME_01, NULL},
	{"3 Minutes", HDDCallback, (gpointer) SCI_TIME_03, NULL},
	{"5 Minutes", HDDCallback, (gpointer) SCI_TIME_05, NULL},
	{"10 Minutes", HDDCallback, (gpointer) SCI_TIME_10, NULL},
	{"15 Minutes", HDDCallback, (gpointer) SCI_TIME_15, NULL},
	{"20 Minutes", HDDCallback, (gpointer) SCI_TIME_20, NULL},
	{"30 Minutes", HDDCallback, (gpointer) SCI_TIME_30, NULL},
	{NULL, NULL, NULL, NULL}
};

static MenuItem displaymenu[] =
{
	{"Disabled", DisplayCallback, (gpointer) SCI_TIME_DISABLED, NULL},
	{"1 Minute", DisplayCallback, (gpointer) SCI_TIME_01, NULL},
	{"3 Minutes", DisplayCallback, (gpointer) SCI_TIME_03, NULL},
	{"5 Minutes", DisplayCallback, (gpointer) SCI_TIME_05, NULL},
	{"10 Minutes", DisplayCallback, (gpointer) SCI_TIME_10, NULL},
	{"15 Minutes", DisplayCallback, (gpointer) SCI_TIME_15, NULL},
	{"20 Minutes", DisplayCallback, (gpointer) SCI_TIME_20, NULL},
	{"30 Minutes", DisplayCallback, (gpointer) SCI_TIME_30, NULL},
	{NULL, NULL, NULL, NULL}
};

static MenuItem autooffmenu[] =
{
	{"Disabled", AutoOffCallback, (gpointer) SCI_TIME_DISABLED, NULL},
	{"10 Minutes", AutoOffCallback, (gpointer) SCI_TIME_10, NULL},
	{"20 Minutes", AutoOffCallback, (gpointer) SCI_TIME_20, NULL},
	{"30 Minutes", AutoOffCallback, (gpointer) SCI_TIME_30, NULL},
	{"40 Minutes", AutoOffCallback, (gpointer) SCI_TIME_40, NULL},
	{"50 Minutes", AutoOffCallback, (gpointer) SCI_TIME_50, NULL},
	{"60 Minutes", AutoOffCallback, (gpointer) SCI_TIME_60, NULL},
	{NULL, NULL, NULL, NULL}
};

static MenuItem standbymenu[] =
{
	{"Disabled", StandbyCallback, (gpointer) SCI_TIME_DISABLED, NULL},
	{"5 Minutes", StandbyCallback, (gpointer) SCI_TIME_05, NULL},
	{"10 Minutes", StandbyCallback, (gpointer) SCI_TIME_10, NULL},
	{"15 Minutes", StandbyCallback, (gpointer) SCI_TIME_15, NULL},
	{"30 Minutes", StandbyCallback, (gpointer) SCI_TIME_30, NULL},
	{"45 Minutes", StandbyCallback, (gpointer) SCI_TIME_45, NULL},
	{"60 Minutes", StandbyCallback, (gpointer) SCI_TIME_60, NULL},
	{"Unlimited", StandbyCallback, (gpointer) SCI_TIME_00, NULL},
	{NULL, NULL, NULL, NULL}
};


static MenuItem speedmenu[] =
{
	{"Low", SpeedCallback, (gpointer) SCI_LOW, NULL},
	{"High", SpeedCallback, (gpointer) SCI_HIGH, NULL},
	{NULL, NULL, NULL, NULL}
};

static MenuItem sleepmenu[] =
{
	{"Disabled", SleepCallback, (gpointer) SCI_DISABLED, NULL},
	{"Enabled", SleepCallback, (gpointer) SCI_ENABLED, NULL},
	{NULL, NULL, NULL, NULL}
};


static MenuItem lcdhighlowmenu[] =
{
	{"Low", LCDCallback, (gpointer) SCI_LOW, NULL},
	{"High", LCDCallback, (gpointer) SCI_HIGH, NULL},
	{NULL, NULL, NULL, NULL}
};


static MenuItem lcdbrightmenu[] =
{
	{"Bright", LCDCallback, (gpointer) SCI_BRIGHT, NULL},
	{"Semi-Bright", LCDCallback, (gpointer) SCI_SEMI_BRIGHT, NULL},
	{NULL, NULL, NULL, NULL}
};

static MenuItem lcdsupermenu[] =
{
	{"Super-Bright", LCDCallback, (gpointer) SCI_SUPER_BRIGHT, NULL},
	{"Bright", LCDCallback, (gpointer) SCI_BRIGHT, NULL},
	{"Semi-Bright", LCDCallback, (gpointer) SCI_SEMI_BRIGHT, NULL},
	{NULL, NULL, NULL, NULL}
};

static MenuItem lcdbackmenu[] =
{
	{"Bright", LCDCallback, (gpointer) SCI_BACK_BRIGHT, NULL},
	{"Semi-Bright", LCDCallback, (gpointer) SCI_BACK_SEMI, NULL},
	{"Dim", LCDCallback, (gpointer) SCI_BACK_DIM, NULL},
	{"Dark", LCDCallback, (gpointer) SCI_BACK_DARK, NULL},
	{NULL, NULL, NULL, NULL}
};

static MenuItem coolingmenu[] =
{
	{"Performance", CoolingCallback, (gpointer) SCI_PERFORMANCE, NULL},
	{"Quiet", CoolingCallback, (gpointer) SCI_QUIET, NULL},
	{NULL, NULL, NULL, NULL}
};


/*
 * close the resume dependency dialog and enable resume mode
 */
void ResumeYesCallback(GtkWidget *widget, gpointer *data)
{
	SetSetting(SCI_POWER_UP, SCI_RESUME, ERROR_GUI);

	SetSystemOptions();

	gtk_widget_destroy(GTK_WIDGET(data));
	return;
}


/*
 * close the resume dependency dialog
 */
void ResumeNoCallback(GtkWidget *widget, gpointer *data)
{
	gtk_widget_destroy(GTK_WIDGET(data));
	return;
}


/*
 * Pop up a dialog box to query the user about the resume mode dependency
 * of the system auto off setting
 */
void ResumeDependency(void)
{
	GtkWidget *window,*button,*label,*box,*icon;

	window = gtk_dialog_new();

	/* handle the window manager close signal */

	gtk_signal_connect(GTK_OBJECT(window), "destroy",
		GTK_SIGNAL_FUNC(gtk_widget_destroyed), &window);

	gtk_window_set_title(GTK_WINDOW(window), "Resume Mode Dependency");

	/* add the Yes/No/Help buttons */

	button = gtk_button_new_with_label("Yes");
	gtk_widget_set_usize(button, 60, 20);
	gtk_signal_connect(GTK_OBJECT(button), "clicked",
		GTK_SIGNAL_FUNC(ResumeYesCallback), GTK_OBJECT(window));
	gtk_box_pack_start(GTK_BOX(GTK_DIALOG(window)->action_area), button,
		FALSE, FALSE, 0);

	button = gtk_button_new_with_label("No");
	gtk_widget_set_usize(button, 60, 20);
	gtk_signal_connect(GTK_OBJECT(button), "clicked",
		GTK_SIGNAL_FUNC(ResumeNoCallback), GTK_OBJECT(window));
	gtk_box_pack_start(GTK_BOX(GTK_DIALOG(window)->action_area), button,
		FALSE, FALSE, 0);

	button = gtk_button_new_with_label("Help");
	gtk_widget_set_usize(button, 60, 20);
	gtk_widget_set_sensitive(GTK_WIDGET(button), FALSE);
	gtk_box_pack_start(GTK_BOX(GTK_DIALOG(window)->action_area), button,
		FALSE, FALSE, 0);

	/* draw the message area */

	box = gtk_hbox_new(FALSE, 4);
	gtk_box_pack_start(GTK_BOX(GTK_DIALOG(window)->vbox), box,
		TRUE, TRUE, 0);

	icon = gtk_pixmap_new(chargedpixmap, chargedmask);
	gtk_box_pack_start(GTK_BOX(box), icon, FALSE, FALSE, 4);

	label = gtk_label_new(RESUME_DEPENDENCY);
	gtk_misc_set_padding(GTK_MISC(label), 10, 10);
	gtk_box_pack_start(GTK_BOX(box), label, FALSE, TRUE, 0);

	gtk_widget_show_all(window);

	return;
}


/*
 * Build an option menu
 */
GtkWidget* BuildMenu(MenuItem *list)
{
	GtkWidget *menu,*item;

	menu = gtk_menu_new();
	
	while (list->label) {
		item = gtk_menu_item_new_with_label(list->label);
		gtk_menu_append(GTK_MENU(menu), item);

		if (list->callback) {
			gtk_signal_connect(GTK_OBJECT(item), "activate",
				GTK_SIGNAL_FUNC(list->callback), list->data);
		}

		/* a bug in Gtk keeps gtk_widget_show_all(window) from handling
		   these labels properly, so show them manually for now */

		gtk_widget_show(item);
		list->widget = item;
		list++;
	}

	return menu;
}


/*
 * Get the number of the menu item corresponding to the setting else return -1
 */
int GetMenuItemNumber(MenuItem *menu, int setting)
{
	int i;

	for (i=0;menu[i].label!=NULL;i++) {
		if ((int) menu[i].data==setting)
			return i;
	}

	return -1;
}


/*
 * Set sensitivity of the battery option menus depending on battery save mode
 */
void SetBatterySensitivity(int save)
{

	if (save==SCI_USER_SETTINGS) {
		gtk_widget_set_sensitive(GTK_WIDGET(hdd), TRUE);
		gtk_widget_set_sensitive(GTK_WIDGET(display), TRUE);
		gtk_widget_set_sensitive(GTK_WIDGET(autooff), TRUE);
		gtk_widget_set_sensitive(GTK_WIDGET(speed), TRUE);
		gtk_widget_set_sensitive(GTK_WIDGET(sleep), TRUE);
		gtk_widget_set_sensitive(GTK_WIDGET(lcd), TRUE);
		if (cooltype==TRUE)
			gtk_widget_set_sensitive(GTK_WIDGET(cooling), TRUE);
	} else {
		gtk_widget_set_sensitive(GTK_WIDGET(hdd), FALSE);
		gtk_widget_set_sensitive(GTK_WIDGET(display), FALSE);
		gtk_widget_set_sensitive(GTK_WIDGET(autooff), FALSE);
		gtk_widget_set_sensitive(GTK_WIDGET(speed), FALSE);
		gtk_widget_set_sensitive(GTK_WIDGET(sleep), FALSE);
		gtk_widget_set_sensitive(GTK_WIDGET(lcd), FALSE);
		if (cooltype==TRUE)
			gtk_widget_set_sensitive(GTK_WIDGET(cooling), FALSE);
	}

	return;
}


/*
 * Set the battery option option menus to reflect the stored settings
 */
void SetBatteryOptions(int mode)
{
	int i = 0;
	struct BatterySettings settings;


	/* make the option menus sensitive to avoid some remaining greyed */

	SetBatterySensitivity(SCI_USER_SETTINGS);

	if (mode==SCI_MAINS) {
		settings.save = external.save;
		settings.hdd = external.hdd;
		settings.display = external.display;
		settings.sleep = external.sleep;
		settings.speed = external.speed;
		settings.lcd = external.lcd;
		settings.autooff = external.autooff;
		settings.cooling = external.cooling;	
	} else {
		settings.save = internal.save;
		settings.hdd = internal.hdd;
		settings.display = internal.display;
		settings.sleep = internal.sleep;
		settings.speed = internal.speed;
		settings.lcd = internal.lcd;
		settings.autooff = internal.autooff;
		settings.cooling = internal.cooling;	
	};

	if (savetype==STOCKHOLM) {
		if ((i = GetMenuItemNumber(batterystockholm, settings.save))!=-1)
			gtk_option_menu_set_history(GTK_OPTION_MENU(battery), i);
	}
	else if (savetype==ATLANTA) {
		if ((i = GetMenuItemNumber(batteryatlanta, settings.save))!=-1)
			gtk_option_menu_set_history(GTK_OPTION_MENU(battery), i);
	}
	else if (savetype==MAUI) {
		if ((i = GetMenuItemNumber(batterymaui, settings.save))!=-1)
			gtk_option_menu_set_history(GTK_OPTION_MENU(battery), i);
	}
	else {
		if ((i = GetMenuItemNumber(batterylibretto, settings.save))!=-1)
			gtk_option_menu_set_history(GTK_OPTION_MENU(battery), i);
	}

	if ((i = GetMenuItemNumber(hddmenu, settings.hdd))!=-1)
		gtk_option_menu_set_history(GTK_OPTION_MENU(hdd), i);

	if ((i = GetMenuItemNumber(displaymenu, settings.display))!=-1)
		gtk_option_menu_set_history(GTK_OPTION_MENU(display), i);

	if ((i = GetMenuItemNumber(autooffmenu, settings.autooff))!=-1)
		gtk_option_menu_set_history(GTK_OPTION_MENU(autooff), i);

	if ((i = GetMenuItemNumber(speedmenu, settings.speed))!=-1)
		gtk_option_menu_set_history(GTK_OPTION_MENU(speed), i);

	if ((i = GetMenuItemNumber(sleepmenu, settings.sleep))!=-1)
		gtk_option_menu_set_history(GTK_OPTION_MENU(sleep), i);

	if (lcdtype==BRIGHT) {
		if ((i = GetMenuItemNumber(lcdbrightmenu, settings.lcd))!=-1)
			gtk_option_menu_set_history(GTK_OPTION_MENU(lcd), i);
	} else if (lcdtype==SUPER) {
		if ((i = GetMenuItemNumber(lcdsupermenu, settings.lcd))!=-1)
			gtk_option_menu_set_history(GTK_OPTION_MENU(lcd), i);
	} else {
		if ((i = GetMenuItemNumber(lcdbackmenu, settings.lcd))!=-1)
			gtk_option_menu_set_history(GTK_OPTION_MENU(lcd), i);
	}

	if (cooltype==TRUE) {
		if ((i = GetMenuItemNumber(coolingmenu, settings.cooling))!=-1)
			gtk_option_menu_set_history(GTK_OPTION_MENU(cooling), i);
	}


	/* we are done so set the enable/disable the option menus as needed */

	SetBatterySensitivity(settings.save);

	return;
}


/*
 * Update the displayed sound settings so they reflect their actual settings
 */
void SetSoundOptions(void)
{
	SMMRegisters reg;

	reg.ebx = SCI_PANEL_ALARM;	
 	if (GetSetting(&reg, ERROR_GUI)==SCI_SUCCESS) {
		if ((reg.ecx & 0xffff)==SCI_ENABLED)
			gtk_toggle_button_set_state(GTK_TOGGLE_BUTTON(panel),
				TRUE);
		else
			gtk_toggle_button_set_state(GTK_TOGGLE_BUTTON(panel),
				FALSE);
	}

	reg.ebx = SCI_BATTERY_ALARM;
	if (GetSetting(&reg, ERROR_GUI)==SCI_SUCCESS) {
		if ((reg.ecx & 0xffff)==SCI_ENABLED)
			gtk_toggle_button_set_state(GTK_TOGGLE_BUTTON(low),
				TRUE);
		else
			gtk_toggle_button_set_state(GTK_TOGGLE_BUTTON(low),
				FALSE);
	}

	reg.ebx = SCI_SYSTEM_BEEP;
	if (GetSetting(&reg, ERROR_GUI)==SCI_SUCCESS) {
		if ((reg.ecx & 0xffff)==SCI_ENABLED)
			gtk_toggle_button_set_state(GTK_TOGGLE_BUTTON(beep),
				TRUE);
		else
			gtk_toggle_button_set_state(GTK_TOGGLE_BUTTON(beep),
				FALSE);
	}

	reg.ebx = SCI_SPEAKER_VOLUME;
	if (GetSetting(&reg, ERROR_GUI)==SCI_SUCCESS) {
		gtk_adjustment_set_value(GTK_ADJUSTMENT(slider),
			(gfloat) (reg.ecx & 0xffff)+1);
	}

	return;
}


/*
 * Update the displayed system settings so they reflect their actual settings
 */
void SetSystemOptions(void)
{
	SMMRegisters reg;

	reg.ebx = SCI_ALARM_POWER;	
 	if (GetSetting(&reg, ERROR_GUI)==SCI_SUCCESS) {
		if ((reg.ecx & 0xffff)==SCI_ALARM_DISABLED) {
			gtk_widget_set_sensitive(GTK_WIDGET(hourplus), FALSE);
			gtk_widget_set_sensitive(GTK_WIDGET(hourminus), FALSE);
			gtk_widget_set_sensitive(GTK_WIDGET(minuteplus), FALSE);
			gtk_widget_set_sensitive(GTK_WIDGET(minuteminus),
				FALSE);
			gtk_widget_set_sensitive(GTK_WIDGET(alarmam), FALSE);
			gtk_widget_set_sensitive(GTK_WIDGET(alarmpm), FALSE);
			gtk_widget_set_sensitive(GTK_WIDGET(alarm24), FALSE);
			gtk_toggle_button_set_state(GTK_TOGGLE_BUTTON(disable),
				TRUE);
		} else {
			gtk_widget_set_sensitive(GTK_WIDGET(hourplus), TRUE);
			gtk_widget_set_sensitive(GTK_WIDGET(hourminus), TRUE);
			gtk_widget_set_sensitive(GTK_WIDGET(minuteplus), TRUE);
			gtk_widget_set_sensitive(GTK_WIDGET(minuteminus), TRUE);
			gtk_widget_set_sensitive(GTK_WIDGET(alarmam), TRUE);
			gtk_widget_set_sensitive(GTK_WIDGET(alarmpm), TRUE);
			gtk_widget_set_sensitive(GTK_WIDGET(alarm24), TRUE);
			gtk_toggle_button_set_state(GTK_TOGGLE_BUTTON(disable),
				FALSE);
		}
	}

	reg.ebx = SCI_POWER_UP;
	if (GetSetting(&reg, ERROR_GUI)==SCI_SUCCESS) {
		if ((reg.ecx & 0xffff)==SCI_RESUME)
			gtk_toggle_button_set_state(GTK_TOGGLE_BUTTON(resume),
				TRUE);
		else
			gtk_toggle_button_set_state(GTK_TOGGLE_BUTTON(resume),
				FALSE);
	}

	reg.ebx = SCI_PANEL_POWER;
	if (GetSetting(&reg, ERROR_GUI)==SCI_SUCCESS) {
		if ((reg.ecx & 0xffff)==SCI_ENABLED)
			gtk_toggle_button_set_state(GTK_TOGGLE_BUTTON(power),
				TRUE);
		else
			gtk_toggle_button_set_state(GTK_TOGGLE_BUTTON(power),
				FALSE);
	}
	if (GTK_TOGGLE_BUTTON(resume)->active==FALSE)
		gtk_widget_set_sensitive(GTK_WIDGET(power), FALSE);


	return;
}


/*
 * Change the option menus when the power source is changed by the radio buttons
 */
void PowerSourceCallback(GtkWidget *widget, int data)
{
	if (source!=(int) data) {
	        source = (int) data;
		SetBatteryOptions(source);
	}

	return;
}


/*
 * Set the option menus to reflect the currently selected battery save option
 */
void BatteryCallback(GtkWidget *widget, int data)
{
	if (source==SCI_MAINS)
		external.save = data;
	else
		internal.save = data;

	SetBatteryOptions(source);

	return;
}


void HDDCallback(GtkWidget *widget, int data)
{
	if (source==SCI_MAINS)
		external.hdd = data;
	else
		internal.hdd = data;

	return;
}


void DisplayCallback(GtkWidget *widget, int data)
{
	if (source==SCI_MAINS)
		external.display = data;
	else
		internal.display = data;

	return;
}


void AutoOffCallback(GtkWidget *widget, int data)
{
	SMMRegisters reg;

	reg.ebx = SCI_POWER_UP;
 	if (GetSetting(&reg, ERROR_GUI)==SCI_SUCCESS) {
		if ((reg.ecx & 0xffff)==SCI_BOOT)
			ResumeDependency();
	}

	if (source==SCI_MAINS)
		external.autooff = data;
	else
		internal.autooff = data;

	return;
}


void SpeedCallback(GtkWidget *widget, int data)
{
	if (source==SCI_MAINS)
		external.speed = data;
	else
		internal.speed = data;

	return;
}


void SleepCallback(GtkWidget *widget, int data)
{
	if (source==SCI_MAINS)
		external.sleep = data;
	else
		internal.sleep = data;

	return;
}


void LCDCallback(GtkWidget *widget, int data)
{
	if (source==SCI_MAINS)
		external.lcd = data;
	else
		internal.lcd = data;

	return;
}


void CoolingCallback(GtkWidget *widget, int data)
{
	if (source==SCI_MAINS)
		external.cooling = data;
	else
		internal.cooling = data;

	return;
}


void StandbyCallback(GtkWidget *widget, int data)
{
	if (source==SCI_MAINS)
		external.standby = data;
	else
		internal.standby = data;

	return;
}


/*
 * Save the selected settings to the configuration file, and set them
 * appropriately for the current power source when closing the Machine
 * Settings dialog
 */
void MachineOkCallback(GtkWidget *widget, GtkWidget *dialog)
{
	char buffer[16];

	/* delete the dialog first, makes program feel more responsive */

	gtk_widget_destroy(dialog);
	dialog = NULL;

	/* save the battery power source settings */

	sprintf(buffer, "%d", internal.hdd);
	WriteConfigString("BatPowerSettings", "HDDAutoOff", buffer, config);
	sprintf(buffer, "%d", internal.display);
	WriteConfigString("BatPowerSettings", "DisplayAutoOff", buffer, config);
	sprintf(buffer, "%d", internal.sleep);
	WriteConfigString("BatPowerSettings", "SleepMode", buffer, config);
	sprintf(buffer, "%d", internal.speed);
	WriteConfigString("BatPowerSettings", "CPUSpeed", buffer, config);
	sprintf(buffer, "%d", internal.lcd);
	WriteConfigString("BatPowerSettings", "LCDBrightness", buffer, config);
	sprintf(buffer, "%d", internal.save);
	WriteConfigString("BatPowerSettings", "BatterySaveMode", buffer,
		config);
	sprintf(buffer, "%d", internal.autooff);
	WriteConfigString("BatPowerSettings", "SystemAutoOff", buffer, config);
	sprintf(buffer, "%d", internal.cooling);
	WriteConfigString("BatPowerSettings", "CoolingMethod", buffer, config);

	/* save the mains power source settings */

	sprintf(buffer, "%d", external.hdd);
	WriteConfigString("ExtPowerSettings", "HDDAutoOff", buffer, config);
	sprintf(buffer, "%d", external.display);
	WriteConfigString("ExtPowerSettings", "DisplayAutoOff", buffer, config);
	sprintf(buffer, "%d", external.sleep);
	WriteConfigString("ExtPowerSettings", "SleepMode", buffer, config);
	sprintf(buffer, "%d", external.speed);
	WriteConfigString("ExtPowerSettings", "CPUSpeed", buffer, config);
	sprintf(buffer, "%d", external.lcd);
	WriteConfigString("ExtPowerSettings", "LCDBrightness", buffer, config);
	sprintf(buffer, "%d", external.save);
	WriteConfigString("ExtPowerSettings", "BatterySaveMode", buffer,
		config);
	sprintf(buffer, "%d", external.autooff);
	WriteConfigString("ExtPowerSettings", "SystemAutoOff", buffer, config);
	sprintf(buffer, "%d", external.cooling);
	WriteConfigString("ExtPowerSettings", "CoolingMethod", buffer, config);

	/* delete the pixmaps */

	gdk_pixmap_unref(plus);
	gdk_pixmap_unref(minus);
	gdk_pixmap_unref(plusdisabled);
	gdk_pixmap_unref(minusdisabled);
	gdk_pixmap_unref(chargedpixmap);
	gdk_bitmap_unref(plusmask);
	gdk_bitmap_unref(minusmask);

	/* exit deleting pid file */

	ExitTuxTimeConfig(0x20);

	return;
}


/*
 * Quit the Machine Settings dialog returning everything to it original state
 */
void MachineCancelCallback(GtkWidget *widget, GtkWidget *dialog)
{
	gtk_widget_destroy(dialog);
	dialog = NULL;

	/* delete the pixmaps */

	gdk_pixmap_unref(plus);
	gdk_pixmap_unref(minus);
	gdk_pixmap_unref(plusdisabled);
	gdk_pixmap_unref(minusdisabled);
	gdk_pixmap_unref(chargedpixmap);
	gdk_bitmap_unref(plusmask);
	gdk_bitmap_unref(minusmask);

	/* return the sound settings to their entry values */

	SetSetting(SCI_PANEL_ALARM, machine.panelalarm, ERROR_GUI);
	SetSetting(SCI_BATTERY_ALARM, machine.lowalarm, ERROR_GUI);
	SetSetting(SCI_SYSTEM_BEEP, machine.beep, ERROR_GUI);
	SetSetting(SCI_SPEAKER_VOLUME, machine.volume, ERROR_GUI);

	/* return the system settings to their entry values */

	SetSetting(SCI_POWER_UP, machine.powerup, ERROR_GUI);
	SetSetting(SCI_PANEL_POWER, machine.panelpower, ERROR_GUI);

/*	printf("FIXME: need to set alarm power to it's entry value "
		"on canceling the machine settings dialog\n");*/

	/* exit deleting pid file */

	ExitTuxTimeConfig(0);

	return;
}


/*
 * Return the settings to their default values
 */
void MachineDefaultsCallback(GtkWidget *widget, gpointer *data)
{

	/* return the settings to their defaults */

	internal.save = SCI_LOW_POWER;
	internal.hdd = SCI_TIME_03;
	internal.display = SCI_TIME_03;
	internal.autooff = SCI_TIME_30;
	internal.speed = SCI_LOW;
	internal.sleep = SCI_ENABLED;
	internal.lcd = SCI_SEMI_BRIGHT;
	internal.cooling = SCI_QUIET;
	internal.standby = 0;

	external.save = SCI_FULL_POWER;
	external.hdd = SCI_TIME_30;
	external.display = SCI_TIME_DISABLED;
	external.autooff = SCI_TIME_DISABLED;
	external.speed = SCI_HIGH;
	external.sleep = SCI_DISABLED;
	external.lcd = SCI_BRIGHT;
	external.cooling = SCI_PERFORMANCE;
	external.standby = 0;

	machine.volume = SCI_VOLUME_HIGH;
	machine.panelalarm = SCI_ENABLED;
	machine.lowalarm = SCI_ENABLED;
	machine.beep = SCI_ENABLED;
	machine.alarm = SCI_ALARM_DISABLED;
	machine.powerup = SCI_BOOT;
	machine.panelpower = SCI_DISABLED;

	/* set the default values for the sound and system pages*/

	SetSetting(SCI_SPEAKER_VOLUME, SCI_VOLUME_HIGH, ERROR_GUI);
	SetSetting(SCI_PANEL_ALARM, SCI_ENABLED, ERROR_GUI);
	SetSetting(SCI_BATTERY_ALARM, SCI_ENABLED, ERROR_GUI);
	SetSetting(SCI_SYSTEM_BEEP, SCI_ENABLED, ERROR_GUI);
	SetSetting(SCI_POWER_UP, SCI_BOOT, ERROR_GUI);
	SetSetting(SCI_PANEL_POWER, SCI_DISABLED, ERROR_GUI);
	SetSetting(SCI_ALARM_POWER, SCI_ALARM_DISABLED, ERROR_GUI);

	/* update the displayed settings */

	SetBatteryOptions(powermode);
	SetSoundOptions();
	SetSystemOptions();

	return;
}


/*
 * Build the Battery Options page of the notebook
 */
void AddBatteryPage(GtkWidget *notebook, GtkWidget *window)
{
	SMMRegisters reg;
	int placement,size;
	GtkWidget *box,*frame,*buttons,*options;
	GtkWidget *label,*operation,*button,*table;
	GtkWidget *menu;
	GSList *radio;

	/* get the settings from the configuration file */

	internal.hdd = GetConfigInt("BatPowerSettings", "HDDAutoOff", 512,
		config);
	internal.display = GetConfigInt("BatPowerSettings", "DisplayAutoOff",
		1, config);
	internal.sleep = GetConfigInt("BatPowerSettings", "SleepMode", 0,
		config);
	internal.speed = GetConfigInt("BatPowerSettings", "CPUSpeed", 1,
		config);
	internal.lcd = GetConfigInt("BatPowerSettings", "LCDBrightness", 0,
		config);
	internal.save = GetConfigInt("BatPowerSettings", "BatterySaveMode", 2,
		config);
	internal.autooff = GetConfigInt("BatPowerSettings", "SystemAutoOff", 1,
		config);
	internal.cooling = GetConfigInt("BatPowerSettings", "CoolingMethod", 0,
		config);

	external.hdd = GetConfigInt("ExtPowerSettings", "HDDAutoOff", 512,
		config);
	external.display = GetConfigInt("ExtPowerSettings", "DisplayAutoOff",
		1, config);
	external.sleep = GetConfigInt("ExtPowerSettings", "SleepMode", 0,
		config);
	external.speed = GetConfigInt("ExtPowerSettings", "CPUSpeed", 1,
		config);
	external.lcd = GetConfigInt("ExtPowerSettings", "LCDBrightness", 0,
		config);
	external.save = GetConfigInt("ExtPowerSettings", "BatterySaveMode", 2,
		config);
	external.autooff = GetConfigInt("ExtPowerSettings", "SystemAutoOff", 1,
		config);
	external.cooling = GetConfigInt("ExtPowerSettings", "CoolingMethod", 0,
		config);

	/* determine the battery save mode and LCD types for the machine.
	   need info on machine ID's for Atlanta and Derby machines */

	reg.ebx = SCI_BATTERY_SAVE;
	GetSetting(&reg, ERROR_NONE);
	if ((reg.edx & 0xffff)==3)
		savetype = STOCKHOLM;
	else
		savetype = MAUI;

	lcdtype = BACKLIGHT;
	reg.ebx = SCI_LCD_BRIGHTNESS;
	if (GetSetting(&reg, ERROR_NONE)==SCI_SUCCESS) {
		if ((reg.ebx & 0xffff)==0x8001)
			lcdtype = SUPER;
		if ((reg.ebx & 0xffff)==0x0000)
			lcdtype = BRIGHT;
	}
	reg.ebx = SCI_LCD_MAXBRIGHT;
	if (GetSetting(&reg, ERROR_NONE)==SCI_SUCCESS) {
		lcdtype = HIGHLOW;
		if (external.lcd>1)
			external.lcd = 1;
		if (internal.lcd>1)
			internal.lcd = 1;
	}

	/* check to make sure that cooling method is supported */
	reg.ebx = SCI_COOLING_METHOD;
	if (GetSetting(&reg, ERROR_NONE)==SCI_SUCCESS) {
		cooltype = TRUE;
	} else {
		cooltype = FALSE;
	}

	
	/* now create the widgets for the page */

	box = gtk_hbox_new(FALSE, 4);

	frame = gtk_frame_new("Battery Options");
	gtk_box_pack_start(GTK_BOX(box), frame, TRUE, TRUE, 0);
	gtk_container_border_width(GTK_CONTAINER(frame), 4);

	buttons = gtk_vbox_new(FALSE, 4);
	gtk_container_border_width(GTK_CONTAINER(buttons), 4);
	gtk_box_pack_start(GTK_BOX(box), buttons, FALSE, TRUE, 4);

	AddButtons(buttons, window, BUTTONS_SETTING);
	
	options = gtk_vbox_new(FALSE, 4);
	gtk_container_border_width(GTK_CONTAINER(options), 4);
	gtk_container_add(GTK_CONTAINER(frame), options);

	frame = gtk_frame_new("Show Settings For");
	gtk_box_pack_start(GTK_BOX(options), frame, FALSE, FALSE, 0);
	gtk_container_border_width(GTK_CONTAINER(frame), 4);

	/* get the current power source and set radio buttons accordingly */

	source = SciACPower();

	/* now pack the power source settings frame with radio buttons */

	operation = gtk_vbox_new(FALSE, 4);
	gtk_container_border_width(GTK_CONTAINER(operation), 4);
	gtk_container_add(GTK_CONTAINER(frame), operation);

	button = gtk_radio_button_new_with_label(NULL, "Battery Power Operation");
	gtk_box_pack_start(GTK_BOX(operation), button, FALSE, TRUE, 0);
	radio = gtk_radio_button_group(GTK_RADIO_BUTTON(button));
	gtk_signal_connect(GTK_OBJECT(button), "pressed",
		GTK_SIGNAL_FUNC(PowerSourceCallback), (gpointer) SCI_BATTERY);
  
	button = gtk_radio_button_new_with_label(radio, "External Power Operation");
	gtk_box_pack_start(GTK_BOX(operation), button, FALSE, TRUE, 0);
	if (source==SCI_MAINS)
		gtk_toggle_button_set_state(GTK_TOGGLE_BUTTON(button), TRUE);
	gtk_signal_connect(GTK_OBJECT(button), "pressed",
		GTK_SIGNAL_FUNC(PowerSourceCallback), (gpointer) SCI_MAINS);


	/* determine the number of options going to be added */
	size = 8;
	if (cooltype==FALSE)
		size--;

	/* create a table to hold all the option buttons and labels */

	table = gtk_table_new(size, 2, FALSE);
	gtk_container_border_width(GTK_CONTAINER(table), 2);
	gtk_box_pack_start(GTK_BOX(options), table, TRUE, TRUE, 0);

	/* add all the labels */

	placement = 0;
	label = gtk_label_new("Battery Save Mode");
	gtk_misc_set_alignment(GTK_MISC(label), 0.0, 1.0);
	gtk_table_attach(GTK_TABLE(table), label, 0, 1, placement, placement+1,
		GTK_SHRINK | GTK_FILL | GTK_EXPAND,
		GTK_SHRINK | GTK_FILL | GTK_EXPAND, 0, 2);
	placement++;

	label = gtk_label_new("HDD Auto Off");
	gtk_misc_set_alignment(GTK_MISC(label), 0.0, 1.0);
	gtk_table_attach(GTK_TABLE(table), label, 0, 1, placement, placement+1,
		GTK_SHRINK | GTK_FILL | GTK_EXPAND,
		GTK_SHRINK | GTK_FILL | GTK_EXPAND, 0, 2);
	placement++;

	label = gtk_label_new("Display Auto Off");
	gtk_misc_set_alignment(GTK_MISC(label), 0.0, 1.0);
	gtk_table_attach(GTK_TABLE(table), label, 0, 1, placement, placement+1,
		GTK_SHRINK | GTK_FILL | GTK_EXPAND,
		GTK_SHRINK | GTK_FILL | GTK_EXPAND, 0, 2);
	placement++;

	label = gtk_label_new("System Auto Off");
	gtk_misc_set_alignment(GTK_MISC(label), 0.0, 1.0);
	gtk_table_attach(GTK_TABLE(table), label, 0, 1, placement, placement+1,
		GTK_SHRINK | GTK_FILL | GTK_EXPAND,
		GTK_SHRINK | GTK_FILL | GTK_EXPAND, 0, 2);
	placement++;

	label = gtk_label_new("Processing Speed");
	gtk_misc_set_alignment(GTK_MISC(label), 0.0, 1.0);
	gtk_table_attach (GTK_TABLE(table), label, 0, 1, placement, placement+1,
		GTK_SHRINK | GTK_FILL | GTK_EXPAND,
		GTK_SHRINK | GTK_FILL | GTK_EXPAND, 0, 2);
	placement++;

	label = gtk_label_new("Sleep Mode");
	gtk_misc_set_alignment(GTK_MISC(label), 0.0, 1.0);
	gtk_table_attach(GTK_TABLE(table), label, 0, 1, placement, placement+1,
		GTK_SHRINK | GTK_FILL | GTK_EXPAND,
		GTK_SHRINK | GTK_FILL | GTK_EXPAND, 0, 2);
	placement++;

	label = gtk_label_new("LCD Brightness");
	gtk_misc_set_alignment(GTK_MISC(label), 0.0, 1.0);
	gtk_table_attach (GTK_TABLE(table), label, 0, 1, placement, placement+1,
		GTK_SHRINK | GTK_FILL | GTK_EXPAND,
		GTK_SHRINK | GTK_FILL | GTK_EXPAND, 0, 2);
	placement++;

	if (cooltype==TRUE) {
		label = gtk_label_new("Cooling Method");
		gtk_misc_set_alignment(GTK_MISC(label), 0.0, 1.0);
		gtk_table_attach(GTK_TABLE(table), label, 0, 1, placement,
			placement+1, GTK_SHRINK | GTK_FILL | GTK_EXPAND,
			GTK_SHRINK | GTK_FILL | GTK_EXPAND, 0, 2);
		placement++;
	}

	/* now the option buttons */

	placement = 0;
	if (savetype==STOCKHOLM)
		menu = BuildMenu(batterystockholm);
	else if (savetype==ATLANTA)
		menu = BuildMenu(batteryatlanta);	
	else if (savetype==MAUI)
		menu = BuildMenu(batterymaui);	
	else
		menu = BuildMenu(batterylibretto);

	battery = gtk_option_menu_new();
	gtk_table_attach(GTK_TABLE(table), battery, 1, 2, placement,
		placement+1, GTK_SHRINK | GTK_FILL | GTK_EXPAND,
		GTK_SHRINK | GTK_FILL | GTK_EXPAND, 4, 2);
	gtk_option_menu_set_menu(GTK_OPTION_MENU(battery), menu);
	placement++;

	switch (source) {
		case SCI_MAINS:
			powermode = external.save;
			break;
		case SCI_BATTERY:
			powermode = internal.save;
			break;
	}

	menu = BuildMenu(hddmenu);
	hdd = gtk_option_menu_new();
	gtk_table_attach(GTK_TABLE(table), hdd, 1, 2, placement, placement+1,
		GTK_SHRINK | GTK_FILL | GTK_EXPAND,
		GTK_SHRINK | GTK_FILL | GTK_EXPAND, 4, 2);
	gtk_option_menu_set_menu(GTK_OPTION_MENU(hdd), menu);
	placement++;

	menu = BuildMenu(displaymenu);
	display = gtk_option_menu_new();
	gtk_table_attach(GTK_TABLE(table), display, 1, 2, placement,
		placement+1, GTK_SHRINK | GTK_FILL | GTK_EXPAND,
		GTK_SHRINK | GTK_FILL | GTK_EXPAND, 4, 2);
	gtk_option_menu_set_menu(GTK_OPTION_MENU(display), menu);
	placement++;

	menu = BuildMenu(autooffmenu);
	autooff = gtk_option_menu_new();
	gtk_table_attach(GTK_TABLE(table), autooff, 1, 2, placement,
		placement+1, GTK_SHRINK | GTK_FILL | GTK_EXPAND,
		GTK_SHRINK | GTK_FILL | GTK_EXPAND, 4, 2);
	gtk_option_menu_set_menu(GTK_OPTION_MENU(autooff), menu);
	placement++;

	menu = BuildMenu(speedmenu);
	speed = gtk_option_menu_new();
	gtk_table_attach(GTK_TABLE(table), speed, 1, 2, placement, placement+1,
		GTK_SHRINK | GTK_FILL | GTK_EXPAND,
		GTK_SHRINK | GTK_FILL | GTK_EXPAND, 4, 2);
	gtk_option_menu_set_menu(GTK_OPTION_MENU(speed), menu);
	placement++;

	menu = BuildMenu(sleepmenu);
	sleep = gtk_option_menu_new();
	gtk_table_attach(GTK_TABLE(table), sleep, 1, 2, placement, placement+1,
		GTK_SHRINK | GTK_FILL | GTK_EXPAND,
		GTK_SHRINK | GTK_FILL | GTK_EXPAND, 4, 2);
	gtk_option_menu_set_menu(GTK_OPTION_MENU(sleep), menu);
	placement++;

	if (lcdtype==BRIGHT)
		menu = BuildMenu(lcdbrightmenu);
	else if (lcdtype==SUPER)
		menu = BuildMenu(lcdsupermenu);
	else if (lcdtype==HIGHLOW)
		menu = BuildMenu(lcdhighlowmenu);
	else
		menu = BuildMenu(lcdbackmenu);

	lcd = gtk_option_menu_new();
	gtk_table_attach(GTK_TABLE(table), lcd, 1, 2, placement, placement+1,
		GTK_SHRINK | GTK_FILL | GTK_EXPAND,
		GTK_SHRINK | GTK_FILL | GTK_EXPAND, 4, 2);
	gtk_option_menu_set_menu(GTK_OPTION_MENU(lcd), menu);
	placement++;

	if (cooltype==TRUE) {
		menu = BuildMenu(coolingmenu);
		cooling = gtk_option_menu_new();
		gtk_table_attach(GTK_TABLE(table), cooling, 1, 2, placement,
			placement+1, GTK_SHRINK | GTK_FILL | GTK_EXPAND,
			GTK_SHRINK | GTK_FILL | GTK_EXPAND, 4, 2);
		gtk_option_menu_set_menu(GTK_OPTION_MENU(cooling), menu);
		placement++;
	}

	/* set the option menus to match the current battery save mode */

	SetBatteryOptions(source);

	/* finally add the page to the notebook */

	label = gtk_label_new("Battery Options");
	gtk_notebook_append_page(GTK_NOTEBOOK (notebook), box, label);

	return;
}


/*
 * Set the panel power alarm on/off
 */
void PanelAlarmCallback(GtkWidget *widget, GdkEvent *event, gpointer data)
{
	unsigned short status;	

	if (GTK_TOGGLE_BUTTON(widget)->active==TRUE)
		status = SCI_ENABLED;
	else
		status = SCI_DISABLED;

	SetSetting(SCI_PANEL_ALARM, status, ERROR_GUI);

	return;
}


/*
 * Set the low battery alarm on/off
 */
void LowCallback(GtkWidget *widget, GdkEvent *event, gpointer data)
{
	unsigned short status;	

	if (GTK_TOGGLE_BUTTON(widget)->active==TRUE)
		status = SCI_ENABLED;
	else
		status = SCI_DISABLED;

	SetSetting(SCI_BATTERY_ALARM, status, ERROR_GUI);

	return;
}


/*
 * Set the system beep enabled/disabled
 */
void BeepCallback(GtkWidget *widget, GdkEvent *event, gpointer data)
{
	unsigned short status;	

	if (GTK_TOGGLE_BUTTON(widget)->active==TRUE)
		status = SCI_ENABLED;
	else
		status = SCI_DISABLED;

	SetSetting(SCI_SYSTEM_BEEP, status, ERROR_GUI);

	return;
}


/*
 * Set the label on the volume slider, disable the toggle buttons if volume
 * is off, and set the volume
 */
void VolumeUpdate(GtkAdjustment *adjustment, int *junk)
{
	unsigned short loudness;
	static unsigned short old = 0;

	loudness = (adjustment->value)-1;

	if (loudness==0) {
		gtk_widget_set_sensitive(GTK_WIDGET(panel), FALSE);
		gtk_widget_set_sensitive(GTK_WIDGET(low), FALSE);
	}
	else if (old==0) {
		gtk_widget_set_sensitive(GTK_WIDGET(panel), TRUE);
		gtk_widget_set_sensitive(GTK_WIDGET(low), TRUE);
	}

	switch (loudness) {
		case 0:
			gtk_label_set(GTK_LABEL(level), "0%");
			break;
		case 1:
			gtk_label_set(GTK_LABEL(level), "33%");
			break;
		case 2:
			gtk_label_set(GTK_LABEL(level), "66%");
			break;
		case 3:
			gtk_label_set(GTK_LABEL(level), "100%");
			break;
		default:
			gtk_label_set(GTK_LABEL(level), "100%");
			break;
	}

	old = loudness;
	
	SetSetting(SCI_SPEAKER_VOLUME, loudness, ERROR_GUI);
	
	return;
}


/*
 * Build the Sound Configuration page of the notebook
 */
void AddSoundPage(GtkWidget *notebook, GtkWidget *window)
{
	SMMRegisters reg;
	GtkWidget *frame,*sound,*speaker;
	GtkWidget *box,*buttons;
	GtkWidget *label,*volume;

	/* get the sound settings from the BIOS and store for cancel */

	reg.ebx = SCI_SPEAKER_VOLUME;
 	if (GetSetting(&reg, ERROR_GUI)==SCI_SUCCESS)
		machine.volume = (reg.ecx & 0xffff);
	else
		machine.volume = SCI_VOLUME_HIGH;

	reg.ebx = SCI_PANEL_ALARM;
 	if (GetSetting(&reg, ERROR_GUI)==SCI_SUCCESS)
		machine.panelalarm = (reg.ecx & 0xffff);
	else
		machine.panelalarm = SCI_ENABLED;

	reg.ebx = SCI_BATTERY_ALARM;
 	if (GetSetting(&reg, ERROR_GUI)==SCI_SUCCESS)
		machine.lowalarm = (reg.ecx & 0xffff);
	else
		machine.lowalarm = SCI_ENABLED;

	reg.ebx = SCI_SYSTEM_BEEP;
 	if (GetSetting(&reg, ERROR_GUI)==SCI_SUCCESS)
		machine.beep = (reg.ecx & 0xffff);
	else
		machine.beep = SCI_ENABLED;


	box = gtk_hbox_new(FALSE, 4);

	frame = gtk_frame_new("Sound Control");
	gtk_box_pack_start(GTK_BOX(box), frame, TRUE, TRUE, 0);
	gtk_container_border_width(GTK_CONTAINER(frame), 4);

	buttons = gtk_vbox_new(FALSE, 4);
	gtk_container_border_width(GTK_CONTAINER(buttons), 4);
	gtk_box_pack_start(GTK_BOX(box), buttons, FALSE, TRUE, 4);

	AddButtons(buttons, window, BUTTONS_SETTING);
	
	sound = gtk_vbox_new(FALSE, 4);
	gtk_container_border_width(GTK_CONTAINER(sound), 4);
	gtk_container_add(GTK_CONTAINER(frame), sound);

	frame = gtk_frame_new("Speaker Control");
	gtk_box_pack_start(GTK_BOX(sound), frame, FALSE, FALSE, 0);
	gtk_container_border_width(GTK_CONTAINER(frame), 4);

	speaker = gtk_vbox_new(FALSE, 0);
	gtk_container_border_width(GTK_CONTAINER(speaker), 4);
	gtk_container_add(GTK_CONTAINER(frame), speaker);

	label = gtk_label_new("Speaker Volume:");
	gtk_box_pack_start(GTK_BOX(speaker), label, FALSE, FALSE, 4);
	gtk_misc_set_alignment(GTK_MISC(label), 0.0, 1.0);

	level = gtk_label_new("100%");
	gtk_box_pack_start(GTK_BOX(speaker), level, FALSE, FALSE, 4);

	slider = gtk_adjustment_new(4, 1, 5, 1, 1, 1);
	volume = gtk_hscale_new(GTK_ADJUSTMENT(slider));
	gtk_scale_set_digits(GTK_SCALE(volume), 0);
	gtk_scale_set_draw_value(GTK_SCALE(volume), FALSE);
	gtk_signal_connect(GTK_OBJECT(slider), "value_changed",
		GTK_SIGNAL_FUNC(VolumeUpdate), NULL);
	gtk_box_pack_start(GTK_BOX(speaker), volume, FALSE,FALSE,4);
	
	panel = gtk_check_button_new_with_label("Panel Close Alarm");
	gtk_box_pack_start(GTK_BOX(speaker), panel, FALSE, FALSE, 4);
	if (machine.panelalarm==SCI_ENABLED) {
		gtk_toggle_button_set_state(GTK_TOGGLE_BUTTON(panel), TRUE);
	}
	gtk_signal_connect(GTK_OBJECT(panel), "toggled",
		GTK_SIGNAL_FUNC(PanelAlarmCallback), NULL);

	low = gtk_check_button_new_with_label("Audiable Low Battery Alarm");
	gtk_box_pack_start(GTK_BOX(speaker), low, FALSE, FALSE, 4);
	if (machine.lowalarm==SCI_ENABLED) {
		gtk_toggle_button_set_state(GTK_TOGGLE_BUTTON(low), TRUE);
	}
	gtk_signal_connect(GTK_OBJECT(low), "toggled",
		GTK_SIGNAL_FUNC(LowCallback), NULL);
	
	/* now the panel and low widgets are created set the volume slider */

	gtk_adjustment_set_value(GTK_ADJUSTMENT(slider),
		(gfloat) machine.volume+1);

	frame = gtk_frame_new("System Beep");
	gtk_container_border_width(GTK_CONTAINER(frame), 4);
	gtk_box_pack_start(GTK_BOX(sound), frame, FALSE, FALSE, 0);

	beep = gtk_check_button_new_with_label("Enable System Beep");
	gtk_container_border_width(GTK_CONTAINER(beep), 4);
	gtk_container_add(GTK_CONTAINER(frame), beep);
	if (machine.beep==SCI_ENABLED) {
		gtk_toggle_button_set_state(GTK_TOGGLE_BUTTON(beep), TRUE);
	}
	gtk_signal_connect(GTK_OBJECT(beep), "toggled",
		GTK_SIGNAL_FUNC(BeepCallback), NULL);

	/* finally add the page to the notebook */

	label = gtk_label_new("Sound Control");
	gtk_notebook_append_page(GTK_NOTEBOOK (notebook), box, label);

	return;
}


void ResumeCallback(GtkWidget *widget, GdkEvent *event, gpointer data)
{
	unsigned short status;	

	if (GTK_TOGGLE_BUTTON(widget)->active==TRUE) {
		status = SCI_RESUME;
		gtk_widget_set_sensitive(GTK_WIDGET(power), TRUE);
	}
	else {
		status = SCI_BOOT;
		gtk_widget_set_sensitive(GTK_WIDGET(power), FALSE);
	}

	SetSetting(SCI_POWER_UP, status, ERROR_GUI);

	return;
}

void PowerCallback(GtkWidget *widget, GdkEvent *event, gpointer data)
{
	unsigned short status;	

	if (GTK_TOGGLE_BUTTON(widget)->active==TRUE)
		status = SCI_ENABLED;
	else
		status = SCI_DISABLED;

	SetSetting(SCI_PANEL_POWER, status, ERROR_GUI);

	return;
}


void HourCallback(GtkWidget *widget, gpointer data)
{
	char buffer[8];

	if ((int) data==1) {
		hours++;
		if (hours==24)
			hours = 0;
		if ((hours>12) && (alarmformat!=3))
			hours = 12;
	}
	else {
		hours--;
		if (hours==-1) {
			if (alarmformat==3)
				hours = 23;
			else
				hours = 12;
		}
	}

	sprintf(buffer, "%02d:%02d", hours, minutes);
	gtk_label_set(GTK_LABEL(alarmtime), buffer);

	SetSetting(SCI_ALARM_POWER,
		SCI_ALARM_ENABLED | SCI_TIME(hours, minutes), ERROR_GUI);

	return;
}

void MinuteCallback(GtkWidget *widget, gpointer data)
{
	char buffer[8];

	if ((int) data==1) {
		minutes++;
		if (minutes==60) minutes = 0;
	}
	else {
		minutes--;
		if (minutes==-1) minutes = 59;
	}

	sprintf(buffer, "%02d:%02d", hours, minutes);
	gtk_label_set(GTK_LABEL(alarmtime), buffer);

	SetSetting(SCI_ALARM_POWER,
		SCI_ALARM_ENABLED | SCI_TIME(hours, minutes), ERROR_GUI);

	return;
}

void FormatCallback(GtkWidget *widget, gpointer data)
{
	char buffer[8];

	alarmformat = (int) data;

	if ((alarmformat!=3) && (hours>12)) {
		hours = 12;
		sprintf(buffer, "%02d:%02d", hours, minutes);
		gtk_label_set(GTK_LABEL(alarmtime), buffer);
	}

	return;
}

void AlarmCallback(GtkWidget *widget, GdkEvent *event, gpointer data)
{
	unsigned short status;	

	if (GTK_TOGGLE_BUTTON(widget)->active==TRUE) {
		gtk_widget_set_sensitive(GTK_WIDGET(hourplus), FALSE);
		gtk_widget_set_sensitive(GTK_WIDGET(hourminus), FALSE);
		gtk_widget_set_sensitive(GTK_WIDGET(minuteplus), FALSE);
		gtk_widget_set_sensitive(GTK_WIDGET(minuteminus), FALSE);
		gtk_widget_set_sensitive(GTK_WIDGET(alarmam), FALSE);
		gtk_widget_set_sensitive(GTK_WIDGET(alarmpm), FALSE);
		gtk_widget_set_sensitive(GTK_WIDGET(alarm24), FALSE);
		status = SCI_ALARM_DISABLED;
		gtk_pixmap_set(GTK_PIXMAP(hourpluspix), plusdisabled, plusmask);
		gtk_pixmap_set(GTK_PIXMAP(hourminuspix), minusdisabled, minusmask);
		gtk_pixmap_set(GTK_PIXMAP(minutepluspix), plusdisabled, plusmask);
		gtk_pixmap_set(GTK_PIXMAP(minuteminuspix), minusdisabled, minusmask);
	}
	else {
		gtk_widget_set_sensitive(GTK_WIDGET(hourplus), TRUE);
		gtk_widget_set_sensitive(GTK_WIDGET(hourminus), TRUE);
		gtk_widget_set_sensitive(GTK_WIDGET(minuteplus), TRUE);
		gtk_widget_set_sensitive(GTK_WIDGET(minuteminus), TRUE);
		gtk_widget_set_sensitive(GTK_WIDGET(alarmam), TRUE);
		gtk_widget_set_sensitive(GTK_WIDGET(alarmpm), TRUE);
		gtk_widget_set_sensitive(GTK_WIDGET(alarm24), TRUE);
		status = SCI_ALARM_ENABLED | SCI_TIME(hours, minutes);
		gtk_pixmap_set(GTK_PIXMAP(hourpluspix), plus, plusmask);
		gtk_pixmap_set(GTK_PIXMAP(hourminuspix), minus, minusmask);
		gtk_pixmap_set(GTK_PIXMAP(minutepluspix), plus, plusmask);
		gtk_pixmap_set(GTK_PIXMAP(minuteminuspix), minus, minusmask);
	}

	SetSetting(SCI_ALARM_POWER, status, ERROR_GUI);

	return;
}


/*
 * Build the System Configuration page of the notebook
 *
 */
void AddSystemPage(GtkWidget *notebook, GtkWidget *window)
{
	GtkWidget *box,*frame,*label;
	GtkWidget *time,*format;
	GtkWidget *buttons,*system,*alarm,*timebox;
	GSList *radio;
	SMMRegisters reg;
	char buffer[16];

	/* get the system settings from the BIOS and store for cancel */

	reg.ebx = SCI_ALARM_POWER;
 	if (GetSetting(&reg, ERROR_GUI)==SCI_SUCCESS)
		machine.alarm = (reg.ecx & 0xffff);
	else
		machine.alarm = SCI_ALARM_DISABLED;

	reg.ebx = SCI_POWER_UP;
 	if (GetSetting(&reg, ERROR_GUI)==SCI_SUCCESS)
		machine.powerup = (reg.ecx & 0xffff);
	else
		machine.powerup = SCI_BOOT;

	reg.ebx = SCI_PANEL_POWER;
 	if (GetSetting(&reg, ERROR_GUI)==SCI_SUCCESS)
		machine.panelpower = (reg.ecx & 0xffff);
	else
		machine.panelpower = SCI_DISABLED;

	/* now setup the notebook page */

	box = gtk_hbox_new(FALSE, 0);

	frame = gtk_frame_new("System Configuration");
	gtk_container_border_width(GTK_CONTAINER(frame), 4);
	gtk_box_pack_start(GTK_BOX(box), frame, TRUE, TRUE, 0);

	buttons = gtk_vbox_new(FALSE, 4);
	gtk_container_border_width(GTK_CONTAINER(buttons), 4);
	gtk_box_pack_start(GTK_BOX(box), buttons, FALSE, TRUE, 4);

	AddButtons(buttons, window, BUTTONS_SETTING);

	system = gtk_vbox_new(FALSE, 4);
	gtk_container_border_width(GTK_CONTAINER(system), 4);
	gtk_container_add(GTK_CONTAINER (frame), system);

	/* add main frame for the page */

	frame = gtk_frame_new("Alarm Power On");
	gtk_container_border_width(GTK_CONTAINER(frame), 4);
	gtk_box_pack_start(GTK_BOX(system), frame, FALSE, TRUE, 0);

	alarm = gtk_vbox_new(FALSE, 4);
	gtk_container_border_width(GTK_CONTAINER(alarm), 4);
	gtk_container_add(GTK_CONTAINER(frame), alarm);


	time = gtk_hbox_new(FALSE, 4);
	gtk_container_border_width(GTK_CONTAINER(time), 4);
	gtk_box_pack_start(GTK_BOX(alarm), time, TRUE, TRUE, 0);

	frame = gtk_frame_new("Time");
	gtk_container_border_width(GTK_CONTAINER(frame), 4);
	gtk_box_pack_start(GTK_BOX(time), frame, TRUE, TRUE, 0);

	/* fill out the time frame */

	format = gtk_vbox_new(FALSE, 4);
	gtk_container_border_width(GTK_CONTAINER(format), 4);
	gtk_container_add(GTK_CONTAINER(frame), format);

	/* Get the alarm time and set the label */

	GetConfigString("TuxTime", "AlarmPowerOn", "12:00H", buffer, 16, config);
	alarmon = SCI_TIME_ON(machine.alarm);
	if (ConvertTime(buffer, &hours, &minutes)==0) {
		hours = 12;
		minutes = 0;
	}
	sprintf(buffer, "%02d:%02d", hours, minutes);

	alarmtime = gtk_label_new(buffer);
	gtk_box_pack_start(GTK_BOX(format), alarmtime, FALSE, TRUE, 0);

	timebox = gtk_hbox_new(FALSE, 4);
	gtk_container_border_width(GTK_CONTAINER(timebox), 4);
	gtk_box_pack_start(GTK_BOX(format), timebox, FALSE, TRUE, 0);

	/* buttons for adjusting the alarm time hour */

	hourplus = gtk_button_new();
	gtk_widget_set_usize(hourplus, 22, 22);
	gtk_container_add(GTK_CONTAINER(hourplus), hourpluspix);
	gtk_box_pack_start(GTK_BOX(timebox), hourplus, FALSE, FALSE, 0);
	gtk_signal_connect(GTK_OBJECT(hourplus), "clicked",
		GTK_SIGNAL_FUNC(HourCallback), (gpointer) 1);

	hourminus = gtk_button_new();
	gtk_widget_set_usize(hourminus, 22, 22);
	gtk_container_add(GTK_CONTAINER(hourminus), hourminuspix);
	gtk_box_pack_start(GTK_BOX(timebox), hourminus, FALSE, FALSE, 0);
	gtk_signal_connect(GTK_OBJECT(hourminus), "clicked",
		GTK_SIGNAL_FUNC(HourCallback), (gpointer) -1);

	/* buttons for adjusting the alarm time minutes */

	minuteminus = gtk_button_new();
	gtk_widget_set_usize(minuteminus, 22, 22);
	gtk_container_add(GTK_CONTAINER(minuteminus), minuteminuspix);
	gtk_box_pack_end(GTK_BOX(timebox), minuteminus, FALSE, FALSE, 0);
	gtk_signal_connect(GTK_OBJECT(minuteminus), "clicked",
		GTK_SIGNAL_FUNC(MinuteCallback), (gpointer) -1);

	minuteplus = gtk_button_new();
	gtk_widget_set_usize(minuteplus, 22, 22);
	gtk_container_add(GTK_CONTAINER(minuteplus), minutepluspix);
	gtk_box_pack_end(GTK_BOX(timebox), minuteplus, FALSE, FALSE, 0);
	gtk_signal_connect(GTK_OBJECT(minuteplus), "clicked",
		GTK_SIGNAL_FUNC(MinuteCallback), (gpointer) 1);

	timebox = gtk_hbox_new(FALSE, 4);
	gtk_container_border_width(GTK_CONTAINER(timebox), 4);
	gtk_box_pack_start(GTK_BOX(format), timebox, FALSE, TRUE, 0);

	label = gtk_label_new("Hours");
	gtk_box_pack_start(GTK_BOX(timebox), label, TRUE, TRUE, 0);
	label = gtk_label_new("Minutes");
	gtk_box_pack_start(GTK_BOX(timebox), label, TRUE, TRUE, 0);

	frame = gtk_frame_new("Time Format");
	gtk_container_border_width(GTK_CONTAINER(frame), 4);
	gtk_box_pack_start(GTK_BOX(time), frame, TRUE, TRUE, 0);


	/* now pack the time format frame with radio buttons */

	format = gtk_vbox_new(FALSE, 4);
	gtk_container_border_width(GTK_CONTAINER(format), 4);
	gtk_container_add(GTK_CONTAINER(frame), format);

	alarmam = gtk_radio_button_new_with_label(NULL, "AM");
	gtk_box_pack_start(GTK_BOX(format), alarmam, FALSE, TRUE, 0);
	radio = gtk_radio_button_group(GTK_RADIO_BUTTON(alarmam));
	gtk_signal_connect(GTK_OBJECT(alarmam), "pressed",
		GTK_SIGNAL_FUNC(FormatCallback), (gpointer) 1);

	alarmpm = gtk_radio_button_new_with_label(radio, "PM");
	gtk_box_pack_start(GTK_BOX(format), alarmpm, FALSE, TRUE, 0);
	radio = gtk_radio_button_group(GTK_RADIO_BUTTON(alarmpm));
	gtk_signal_connect(GTK_OBJECT(alarmpm), "pressed",
		GTK_SIGNAL_FUNC(FormatCallback), (gpointer) 2);

	alarm24 = gtk_radio_button_new_with_label(radio, "24 Hour");
	gtk_box_pack_start(GTK_BOX(format), alarm24, FALSE, TRUE, 0);
	gtk_signal_connect(GTK_OBJECT(alarm24), "pressed",
		GTK_SIGNAL_FUNC(FormatCallback), (gpointer) 3);


	disable = gtk_check_button_new_with_label("Disable Alarm Power On");
	gtk_box_pack_start(GTK_BOX(alarm), disable, FALSE, FALSE, 2);
	gtk_signal_connect(GTK_OBJECT(disable), "toggled",
		GTK_SIGNAL_FUNC(AlarmCallback), NULL);

	/* now set the sensitivity of the alarm power on widgets */

	if (alarmon==SCI_ALARM_DISABLED) {
		gtk_widget_set_sensitive(GTK_WIDGET(hourplus), FALSE);
		gtk_widget_set_sensitive(GTK_WIDGET(hourminus), FALSE);
		gtk_widget_set_sensitive(GTK_WIDGET(minuteplus), FALSE);
		gtk_widget_set_sensitive(GTK_WIDGET(minuteminus), FALSE);
		gtk_widget_set_sensitive(GTK_WIDGET(alarmam), FALSE);
		gtk_widget_set_sensitive(GTK_WIDGET(alarmpm), FALSE);
		gtk_widget_set_sensitive(GTK_WIDGET(alarm24), FALSE);
		gtk_toggle_button_set_state(GTK_TOGGLE_BUTTON(disable), TRUE);
	}

	/* switch radio buttons to 24 hour if time is after 12 noon */

	if (hours>12)
		gtk_toggle_button_set_state(GTK_TOGGLE_BUTTON(alarm24), TRUE);

	resume = gtk_check_button_new_with_label("Resume Mode");
	gtk_box_pack_start(GTK_BOX(system), resume, FALSE, FALSE, 2);
	if (machine.powerup==SCI_RESUME)
		gtk_toggle_button_set_state(GTK_TOGGLE_BUTTON(resume), TRUE);
	gtk_signal_connect(GTK_OBJECT(resume), "toggled",
		GTK_SIGNAL_FUNC(ResumeCallback), NULL);

	power = gtk_check_button_new_with_label("Panel Power On/Off");
	gtk_box_pack_start(GTK_BOX(system), power, FALSE, FALSE, 2);

	if (machine.panelpower==SCI_ENABLED)
		gtk_toggle_button_set_state(GTK_TOGGLE_BUTTON(power), TRUE);
	if (GTK_TOGGLE_BUTTON(resume)->active==FALSE)
		gtk_widget_set_sensitive(GTK_WIDGET(power), FALSE);
	gtk_signal_connect(GTK_OBJECT(power), "toggled",
		GTK_SIGNAL_FUNC(PowerCallback), NULL);

	label = gtk_label_new("System Configuration");
	gtk_notebook_append_page(GTK_NOTEBOOK (notebook), box, label);

	return;
}


/*
 * Create the Machine Settings dialog
 */
void MachineDialog(GtkWidget *window)
{
	GtkWidget *notebook;
	GdkColor *transparent = NULL;

	/* create a new notebook, place the position of the tabs */

	notebook = gtk_notebook_new();
	gtk_notebook_set_tab_pos(GTK_NOTEBOOK(notebook), GTK_POS_TOP);
	gtk_container_add(GTK_CONTAINER(window), notebook);

	/* create the pixmaps used by the dialog */

	plus = gdk_pixmap_create_from_xpm_d(window->window,
		&plusmask, transparent, plus_xpm);
	minus = gdk_pixmap_create_from_xpm_d(window->window,
		&minusmask, transparent, minus_xpm);
	plusdisabled = gdk_pixmap_create_from_xpm_d(window->window,
		&plusmask, transparent, plusdisabled_xpm);
	minusdisabled = gdk_pixmap_create_from_xpm_d(window->window,
		&minusmask, transparent, minusdisabled_xpm);
	chargedpixmap = gdk_pixmap_create_from_xpm_d(window->window,
		&chargedmask, transparent, batterycharged_xpm);
	hourpluspix = gtk_pixmap_new(plus, plusmask);
	hourplusdisabledpix = gtk_pixmap_new(plusdisabled, plusmask);
	hourminuspix = gtk_pixmap_new(minus, minusmask);
	hourminusdisabledpix = gtk_pixmap_new(minusdisabled, minusmask);
	minutepluspix = gtk_pixmap_new(plus, plusmask);
	minuteplusdisabledpix = gtk_pixmap_new(plusdisabled, plusmask);
	minuteminuspix = gtk_pixmap_new(minus, minusmask);
	minuteminusdisabledpix = gtk_pixmap_new(minusdisabled, minusmask);

	/* add the three pages to the notebook */

	AddBatteryPage(notebook, window);
	AddSoundPage(notebook, window);
	AddSystemPage(notebook, window);

	/* set to start at first page */

	gtk_notebook_set_page(GTK_NOTEBOOK(notebook), 0);

	return;
}
